/* siggenPSS_3_3072_testfc.c */
/*
 * Copyright (c) 2017 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

/*
 * This test vector was created based on SigGen15_186-3.txt from
 * https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Algorithm-Validation-Program/documents/dss/186-3rsatestvectors.zip
 *
 * The link to the archive file exists in the following URL:
 * https://csrc.nist.gov/projects/cryptographic-algorithm-validation-program/digital-signatures/
 */
/*
 * https://csrc.nist.gov/projects/cryptographic-algorithm-validation-program/digital-signatures/
 * https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Algorithm-Validation-Program/documents/dss/186-3rsatestvectors.zip
 * SigGenPSS_186-3.txt
 */ 
#include "siggenPSStest.h"

static const unsigned short sn1[] = {
	0xa7a1, 0x882a, 0x7fb8, 0x9678, 0x6034, 0xd07f, 0xb1b9, 0xf632,
	0x7c27, 0xbdd7, 0xce6f, 0xe39c, 0x285a, 0xe3b6, 0xc342, 0x59ad,
	0xc0dc, 0x4f7b, 0x9c7d, 0xec3c, 0xa4a2, 0x0d34, 0x0733, 0x9eed,
	0xd7a1, 0x2a42, 0x1da1, 0x8f59, 0x5467, 0x3cac, 0x2ff0, 0x5915,
	0x6ecc, 0x73c6, 0x861e, 0xc761, 0xe6a0, 0xf2a5, 0xa033, 0xa676,
	0x8c6a, 0x42d8, 0xb459, 0xe1b4, 0x9323, 0x49e8, 0x4efd, 0x92df,
	0x59b4, 0x5935, 0xf3d0, 0xe308, 0x17c6, 0x6201, 0xaa99, 0xd07a,
	0xe36c, 0x5d74, 0xf408, 0xd69c, 0xc08f, 0x0441, 0x51ff, 0x4960,
	0xe531, 0x360c, 0xb190, 0x7783, 0x3adf, 0x7bce, 0x77ec, 0xfaa1,
	0x33c0, 0xccc6, 0x3c93, 0xb856, 0x8145, 0x69e0, 0xb988, 0x4ee5,
	0x5406, 0x1b9a, 0x20ab, 0x46c3, 0x8263, 0xc094, 0xdae7, 0x91aa,
	0x61a1, 0x7f8d, 0x16f0, 0xe85b, 0x7e5c, 0xe3b0, 0x67ec, 0xe89e,
	0x20bc, 0x4e8f, 0x1ae8, 0x14b2, 0x76d2, 0x34e0, 0x4f4e, 0x766f,
	0x501d, 0xa74e, 0xa7e3, 0x817c, 0x24ea, 0x35d0, 0x1667, 0x6cec,
	0xe652, 0xb823, 0xb051, 0x6255, 0x73ca, 0x9275, 0x7fc7, 0x20d2,
	0x54ec, 0xf1dc, 0xbbfd, 0x21d9, 0x8307, 0x561e, 0xcaab, 0x5454,
	0x80c7, 0xc52a, 0xd7e9, 0xfa6b, 0x597f, 0x5fe5, 0x5055, 0x9c2f,
	0xe923, 0x205a, 0xc176, 0x1a99, 0x737c, 0xa02d, 0x7b19, 0x822e,
	0x008a, 0x8969, 0x349c, 0x87fb, 0x874c, 0x8162, 0x0e38, 0xf613,
	0xc852, 0x1f03, 0x81fe, 0x5ba5, 0x5b74, 0x827d, 0xad3e, 0x1cf2,
	0xaa29, 0xc693, 0x3629, 0xf2b2, 0x86ad, 0x11be, 0x88fa, 0x6436,
	0xe7e3, 0xf64a, 0x75e3, 0x5952, 0x90dc, 0x0d1c, 0xd5ee, 0xe7aa,
	0xac54, 0x959c, 0xc53b, 0xd5a9, 0x34a3, 0x65e7, 0x2dd8, 0x1a2b,
	0xd4fb, 0x9a67, 0x821b, 0xffed, 0xf2ef, 0x2bd9, 0x4913, 0xde8b,
};

static const unsigned short se1[] = {
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0014, 0x15a7,
};

static const unsigned short sd1[] = {
	0x073a, 0x5fc4, 0xcd64, 0x2f61, 0x13df, 0xfc4f, 0x8403, 0x5cee,
	0x3a2b, 0x8acc, 0x5497, 0x0375, 0x1a1d, 0x6a5e, 0xaa13, 0x4872,
	0x29a5, 0x8ef7, 0xd7a5, 0x22bb, 0x9f4f, 0x2551, 0x0f1a, 0xa0f7,
	0x4c6a, 0x8fc8, 0xa5c5, 0xbe8b, 0x91a6, 0x74ed, 0xe50e, 0x92f7,
	0xe34a, 0x90a3, 0xc9da, 0x999f, 0xffb1, 0xd695, 0xe458, 0x8f45,
	0x1256, 0xc163, 0x484c, 0x1513, 0x50cb, 0x9c78, 0x25a7, 0xd910,
	0x845e, 0xe5cf, 0x826f, 0xecf9, 0xa7c0, 0xfbbb, 0xba22, 0xbb4a,
	0x531c, 0x131d, 0x2e77, 0x61ba, 0x898f, 0x002e, 0xbef8, 0xab87,
	0x2185, 0x11f8, 0x1d32, 0x66e1, 0xec07, 0xa7ca, 0x8622, 0x514c,
	0x6dfd, 0xc86c, 0x6767, 0x9a2c, 0x8f5f, 0x031d, 0xe9a0, 0xc22b,
	0x5a88, 0x060b, 0x46ee, 0x0c64, 0xd3b9, 0xaf3c, 0x0a37, 0x9bcd,
	0x9c6a, 0x1b51, 0xcf64, 0x8045, 0x6d3f, 0xd6de, 0xf94c, 0xd2a6,
	0xc171, 0xdd3f, 0x010e, 0x3c9d, 0x662b, 0xc857, 0x2082, 0x48c9,
	0x4ebc, 0xb9fd, 0x997b, 0x9ff4, 0xa7e5, 0xfd95, 0x5585, 0x6990,
	0x6525, 0xe741, 0xd783, 0x44f6, 0xf6cf, 0xdbd5, 0x9d4f, 0xaa52,
	0xee3f, 0xa964, 0xfb7c, 0xccb2, 0xd6be, 0x1935, 0xd211, 0xfe14,
	0x9821, 0x7716, 0x2739, 0x39a9, 0x4608, 0x1fd8, 0x5099, 0x13fd,
	0x4774, 0x7c5c, 0x2f03, 0xefd4, 0xd6fc, 0x9c6f, 0xcfd8, 0x402e,
	0x9f40, 0xa0a5, 0xb3de, 0x3ca2, 0xb3c0, 0xfac9, 0x4569, 0x38fa,
	0xa6cf, 0x2c20, 0xe391, 0x2e59, 0x81c9, 0x876d, 0x8ca1, 0xff29,
	0xb87a, 0x15ee, 0xae0c, 0xcce3, 0xf8a8, 0xf1e4, 0x0509, 0x1c08,
	0x3b98, 0xbcc5, 0xfe0d, 0x0dea, 0xae33, 0xc67c, 0x0394, 0x437f,
	0x0ecc, 0xb385, 0xb7ef, 0xb17a, 0xeebb, 0xa8af, 0xaecc, 0xa30a,
	0x2f63, 0xeac8, 0xf0ac, 0x8f1e, 0xacad, 0x85bb, 0xcaf3, 0x960b,
};

#define HAS_SHA224_TESTVECTOR

#define HAS_SHA256_TESTVECTOR

#define HAS_SHA384_TESTVECTOR

#define HAS_SHA512_TESTVECTOR

/* 10 elements */
static mod3072_t SHA224[] = {
	/* SHA224 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0xc8, 0xed, 0x14, 0x89, 0x5c, 0x80, 0xa9, 0x1f,
			0xda, 0x83, 0x67, 0xcf, 0x4a, 0xee, 0x38, 0x6b,
			0x8a, 0x37, 0x86, 0x45, 0xf0, 0x6a, 0xfe, 0xe7,
			0x2f, 0x7c, 0x94, 0x04, 0x7f, 0xdd, 0xc7, 0xae,
			0xf8, 0x4c, 0x26, 0xc8, 0x3f, 0xef, 0x13, 0xbf,
			0x65, 0xa3, 0xc7, 0x75, 0x0c, 0x91, 0x96, 0x7e,
			0xcc, 0x02, 0x74, 0x8f, 0xd5, 0x74, 0xb9, 0x33,
			0xd5, 0xec, 0x21, 0xc0, 0x1c, 0x8f, 0x17, 0x8a,
			0xfe, 0x6c, 0x33, 0x56, 0x78, 0x9d, 0x01, 0x12,
			0x17, 0x8e, 0x04, 0xc3, 0x16, 0x9c, 0xfa, 0xbe,
			0xc6, 0xe2, 0x62, 0x1b, 0x33, 0x4f, 0x3c, 0x67,
			0x05, 0xfc, 0x10, 0x99, 0xa4, 0xbd, 0x31, 0x47,
			0xa0, 0xf7, 0x43, 0x1a, 0x4f, 0xb1, 0xfb, 0x80,
			0xb8, 0xed, 0x26, 0xa0, 0xaf, 0x38, 0xed, 0x93,
			0x42, 0x80, 0x57, 0xd1, 0x54, 0x26, 0x0f, 0xe9,
			0x88, 0x54, 0x68, 0x76, 0x61, 0x91, 0x9e, 0x4e,
		},
		/* S0 : 384 byte */
		{
			0x27, 0xb4, 0xf0, 0xaa, 0x13, 0x95, 0x65, 0xfb,
			0xd7, 0x86, 0x07, 0x60, 0x61, 0x0f, 0x68, 0x66,
			0xd5, 0xb5, 0xf0, 0xd7, 0x77, 0x92, 0x1f, 0x06,
			0xf5, 0x05, 0x32, 0x91, 0x12, 0x3e, 0x3b, 0x25,
			0x9d, 0x67, 0x29, 0x4c, 0xcb, 0x8c, 0x0d, 0x06,
			0x8b, 0x8d, 0xae, 0x36, 0x0a, 0xad, 0x2c, 0xf7,
			0xd0, 0x72, 0x96, 0xb5, 0x39, 0xe4, 0xd2, 0xe9,
			0xb0, 0x8c, 0x34, 0x32, 0x86, 0xd5, 0x22, 0xf7,
			0xdd, 0x63, 0xc6, 0x62, 0x0e, 0x86, 0x72, 0xbe,
			0x49, 0x2f, 0x3b, 0x03, 0x9f, 0x73, 0xd8, 0x8a,
			0xb9, 0xd2, 0x2a, 0x54, 0x63, 0xcd, 0x1f, 0x07,
			0xd6, 0x88, 0xe8, 0xba, 0x3f, 0xba, 0xd5, 0x31,
			0xb0, 0xc3, 0x87, 0x0c, 0xcb, 0xfe, 0xbb, 0x59,
			0x6c, 0xe4, 0xec, 0x64, 0x3d, 0x30, 0x97, 0x44,
			0xbd, 0xbd, 0x67, 0x5d, 0x58, 0x41, 0x28, 0x4c,
			0xba, 0xc9, 0x02, 0xcf, 0xb7, 0x0a, 0xde, 0x6d,
			0x33, 0x94, 0x6d, 0x8d, 0xc6, 0x10, 0x9b, 0xbb,
			0xc4, 0x24, 0x12, 0xdb, 0x25, 0xb8, 0xc6, 0x22,
			0x22, 0xc5, 0xff, 0x94, 0xf8, 0xeb, 0x86, 0x89,
			0x82, 0x26, 0x53, 0x92, 0xa4, 0x4e, 0x80, 0x74,
			0x74, 0x91, 0x0b, 0x4b, 0x39, 0x55, 0x8b, 0xbe,
			0xf3, 0x31, 0x97, 0x90, 0x71, 0x78, 0xce, 0x14,
			0x6f, 0xdd, 0x7e, 0x94, 0x09, 0x2a, 0xd5, 0x8b,
			0xf4, 0x1a, 0x47, 0x4e, 0x62, 0x61, 0x36, 0x78,
			0x9f, 0xc2, 0xfe, 0x63, 0x74, 0xa1, 0xb5, 0xfe,
			0xfd, 0xdd, 0x5f, 0xec, 0xb7, 0xf8, 0xca, 0x58,
			0x93, 0x22, 0x0d, 0x1a, 0xb9, 0xe8, 0x22, 0xc3,
			0xae, 0x8a, 0xdd, 0xa1, 0xeb, 0xad, 0xdb, 0x18,
			0xa6, 0xa1, 0x2b, 0xfc, 0x16, 0x5d, 0x12, 0x07,
			0x14, 0x41, 0xa9, 0x91, 0x37, 0x7c, 0xee, 0x6d,
			0xc8, 0xe5, 0x08, 0x39, 0x49, 0x73, 0x46, 0xfe,
			0xe1, 0x3f, 0x12, 0xc5, 0xb7, 0xb6, 0xd0, 0x24,
			0xb8, 0xec, 0xfd, 0xad, 0x80, 0xd5, 0xef, 0x6e,
			0x9e, 0x49, 0x96, 0xac, 0x21, 0xc4, 0xeb, 0x60,
			0x36, 0xbb, 0x51, 0xf5, 0xbe, 0x5e, 0x38, 0xf2,
			0x65, 0x18, 0x11, 0x54, 0x00, 0x08, 0x24, 0xe3,
			0xc1, 0xf2, 0x31, 0xd1, 0x85, 0x89, 0xcc, 0xda,
			0xee, 0x90, 0xfe, 0x30, 0x7b, 0xa5, 0x63, 0x24,
			0x31, 0x8b, 0x53, 0x58, 0x46, 0x8e, 0x9f, 0x39,
			0x13, 0xb8, 0x3a, 0xb8, 0xb3, 0x4d, 0x94, 0x96,
			0x29, 0xed, 0x78, 0x39, 0xf8, 0xda, 0x85, 0xbd,
			0xcd, 0xa5, 0x2f, 0x3d, 0xa5, 0xa4, 0x19, 0xf7,
			0x77, 0xb3, 0x86, 0x0d, 0xbf, 0x2f, 0xfe, 0x28,
			0xd9, 0x62, 0x44, 0x31, 0x25, 0x49, 0x52, 0x8a,
			0x20, 0xcc, 0x73, 0x99, 0xfc, 0x01, 0x08, 0x44,
			0x36, 0x58, 0x06, 0x16, 0x7f, 0xe4, 0x32, 0x35,
			0x52, 0x1c, 0x90, 0x95, 0x87, 0xc2, 0xc7, 0xb8,
			0xdb, 0x4e, 0x29, 0x6d, 0xad, 0x2a, 0xef, 0xa2,
		},
		28, /* salt length */
		/* V0 : 28 byte */
		{
			0x3f, 0x80, 0x50, 0x57, 0x47, 0x1a, 0xab, 0x0a,
			0x28, 0xcf, 0xc8, 0x43, 0x0d, 0xab, 0xcf, 0x99,
			0x06, 0x12, 0xe8, 0xa9, 0x08, 0xb1, 0x58, 0xae,
			0x36, 0xb4, 0xed, 0x53, 
		},
	},
	/* SHA224 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0xd0, 0x4b, 0xe7, 0x58, 0xe9, 0x76, 0x44, 0xee,
			0x60, 0xa9, 0x21, 0x2e, 0x5e, 0xb8, 0x1a, 0x10,
			0x88, 0x04, 0x1a, 0xab, 0x31, 0xe4, 0x28, 0xb0,
			0xcd, 0x4a, 0x84, 0x37, 0xa9, 0xa3, 0xf3, 0xbe,
			0xda, 0xfe, 0x57, 0x6e, 0x74, 0x71, 0x82, 0xa1,
			0xfc, 0xb8, 0x4c, 0xa2, 0x1f, 0x20, 0xe3, 0xb3,
			0xa3, 0xa4, 0x63, 0x55, 0x9f, 0x55, 0xa7, 0xc3,
			0xe7, 0xff, 0x5e, 0xc0, 0xcb, 0x09, 0x61, 0x92,
			0x01, 0x9d, 0x44, 0x4f, 0xdf, 0x09, 0x2a, 0x57,
			0xcd, 0x65, 0xde, 0x22, 0xfb, 0x76, 0x20, 0x3c,
			0x4f, 0xd3, 0x3d, 0x8d, 0xa2, 0x46, 0xe3, 0xde,
			0x2b, 0x75, 0x32, 0x99, 0x3b, 0xc2, 0x16, 0xd0,
			0x2b, 0x6f, 0xd5, 0x81, 0x93, 0x06, 0xe4, 0x19,
			0xbd, 0xf8, 0xff, 0x36, 0x5a, 0x84, 0x78, 0xb1,
			0x73, 0xda, 0xd0, 0xdc, 0xa2, 0x81, 0x84, 0x08,
			0x81, 0xf6, 0x29, 0x4b, 0x63, 0x96, 0xbb, 0x80,
		},
		/* S1 : 384 byte */
		{
			0x4a, 0xba, 0x73, 0x2c, 0x62, 0x55, 0xf0, 0xbc,
			0x44, 0x39, 0x39, 0xc1, 0x31, 0xdd, 0x4c, 0xe6,
			0x44, 0x78, 0xd4, 0xf5, 0x8d, 0xcb, 0xf1, 0xd7,
			0x3f, 0x5f, 0x0e, 0x66, 0x0c, 0x49, 0x23, 0x15,
			0xe9, 0x87, 0xca, 0xfb, 0xc8, 0x3a, 0x1a, 0x0b,
			0xe3, 0xd3, 0x59, 0xa9, 0x60, 0x78, 0x3d, 0x29,
			0x3d, 0x37, 0x5c, 0xcc, 0x3e, 0xc0, 0xd8, 0x2c,
			0x72, 0xab, 0xca, 0xcc, 0x33, 0x9f, 0x1b, 0x42,
			0x20, 0x7a, 0x03, 0x79, 0x5b, 0xe6, 0x80, 0x8b,
			0xa0, 0x6a, 0x89, 0x1e, 0x3b, 0x42, 0x51, 0xe1,
			0xb3, 0x00, 0x1d, 0xfb, 0x53, 0x72, 0x52, 0x57,
			0x2a, 0x33, 0xb4, 0xc5, 0x28, 0x46, 0xda, 0xfe,
			0xfb, 0x24, 0xac, 0xa5, 0x3f, 0xc0, 0x8e, 0x63,
			0xc3, 0x9d, 0xa0, 0x2c, 0x41, 0x38, 0xb3, 0xde,
			0x95, 0x10, 0xfb, 0x79, 0x0f, 0x87, 0x56, 0x6c,
			0xd1, 0x43, 0x80, 0xb1, 0x38, 0xc7, 0x28, 0xc2,
			0x43, 0x54, 0x3b, 0x89, 0xd1, 0xf9, 0x16, 0xce,
			0x27, 0xca, 0xda, 0x85, 0xfa, 0x32, 0xd8, 0x18,
			0x5d, 0xee, 0xfa, 0x25, 0xc3, 0x23, 0xc6, 0x5c,
			0x7e, 0xd5, 0x78, 0xca, 0x57, 0x27, 0x6b, 0x66,
			0x74, 0x4a, 0x7a, 0x1a, 0x78, 0xe6, 0x6d, 0x4e,
			0x57, 0x09, 0x99, 0xd1, 0x70, 0x15, 0xbd, 0xbd,
			0xd8, 0xd3, 0xd6, 0x18, 0x5a, 0x3e, 0xb1, 0xde,
			0xc8, 0xbc, 0x3a, 0x12, 0x87, 0xa2, 0xe2, 0x35,
			0xe4, 0xf1, 0x16, 0xa8, 0xb9, 0x1d, 0x06, 0x12,
			0x8d, 0x36, 0xb5, 0x8e, 0xd4, 0xc9, 0xa6, 0xed,
			0x84, 0x77, 0x3d, 0xc4, 0x9f, 0x75, 0x5e, 0x2e,
			0x27, 0xa6, 0xf1, 0xae, 0xa3, 0x14, 0x17, 0x06,
			0x9b, 0xd0, 0x66, 0xb8, 0x48, 0x09, 0x5c, 0x00,
			0x2f, 0x22, 0xdd, 0x6c, 0xaa, 0x72, 0x95, 0x7e,
			0x21, 0xa1, 0xe6, 0x40, 0xf9, 0xab, 0x9b, 0x91,
			0x80, 0xdf, 0x8e, 0xf8, 0x96, 0x3e, 0x36, 0x11,
			0xdf, 0x26, 0x93, 0xa7, 0xed, 0x06, 0x4f, 0x34,
			0x82, 0x21, 0xe7, 0xed, 0xb1, 0xa5, 0xa8, 0x1a,
			0xcc, 0xe2, 0x4a, 0xcc, 0x33, 0x5c, 0x6e, 0xe7,
			0xd4, 0xf1, 0xaf, 0x6d, 0x68, 0xac, 0xaf, 0x15,
			0xd7, 0x7e, 0x12, 0x81, 0x42, 0xca, 0x9b, 0xfc,
			0x55, 0xa1, 0x21, 0xb1, 0xb1, 0x3f, 0xe5, 0xba,
			0xfe, 0x2e, 0x4d, 0x6a, 0x55, 0x46, 0xb8, 0xcc,
			0x63, 0x1b, 0xb9, 0xd3, 0x04, 0xc0, 0xe9, 0xf3,
			0xd6, 0xd5, 0xdf, 0xe8, 0x33, 0xc3, 0x46, 0x96,
			0x5f, 0x01, 0x03, 0x69, 0x8d, 0x34, 0xa5, 0x1b,
			0xca, 0x5d, 0xb2, 0x66, 0xaf, 0xde, 0xd2, 0x71,
			0xd8, 0x49, 0x06, 0x45, 0xb3, 0xf6, 0x3e, 0xfc,
			0x99, 0x1e, 0x01, 0x68, 0x32, 0x11, 0xf9, 0x48,
			0x2d, 0x21, 0x4c, 0xfa, 0x92, 0x20, 0xf7, 0xbc,
			0x81, 0xe8, 0xcb, 0xb4, 0xd1, 0x18, 0xa2, 0xc3,
			0x06, 0x70, 0x98, 0x07, 0xc0, 0x70, 0xc6, 0x0d,
		},
		28, /* salt length */
		/* V1 : 28 byte */
		{
			0x3f, 0x80, 0x50, 0x57, 0x47, 0x1a, 0xab, 0x0a,
			0x28, 0xcf, 0xc8, 0x43, 0x0d, 0xab, 0xcf, 0x99,
			0x06, 0x12, 0xe8, 0xa9, 0x08, 0xb1, 0x58, 0xae,
			0x36, 0xb4, 0xed, 0x53, 
		},
	},
	/* SHA224 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x39, 0xd8, 0xec, 0x48, 0x16, 0xfa, 0x93, 0x65,
			0xcd, 0xf2, 0x99, 0xce, 0x60, 0x05, 0x3b, 0x9c,
			0x1e, 0x99, 0x54, 0x0e, 0xd2, 0x9d, 0x2d, 0x16,
			0x3a, 0x24, 0x97, 0x18, 0xba, 0x53, 0x37, 0xee,
			0x52, 0x7e, 0x22, 0x2f, 0xce, 0x8e, 0xaa, 0xb1,
			0x3c, 0xa6, 0x77, 0x4c, 0xa3, 0x06, 0xd9, 0xe1,
			0xf2, 0x2f, 0x5c, 0x9b, 0x37, 0x47, 0x9d, 0x75,
			0x11, 0xc0, 0x5d, 0xfd, 0x68, 0x35, 0xd4, 0x57,
			0x5b, 0x94, 0x47, 0x84, 0x7a, 0x82, 0xdd, 0xe5,
			0x36, 0xfb, 0xaf, 0xfa, 0x95, 0x39, 0x1e, 0x70,
			0x2b, 0xd8, 0x69, 0x5b, 0x45, 0x37, 0x7f, 0xc0,
			0x67, 0x21, 0x11, 0x56, 0xf9, 0xad, 0xec, 0x8d,
			0x3d, 0x62, 0x86, 0xd0, 0x84, 0x9f, 0xd6, 0x07,
			0xa2, 0x3a, 0x69, 0x61, 0x9f, 0x68, 0xb3, 0x50,
			0xaf, 0xdd, 0xa3, 0xd5, 0x64, 0x34, 0x7a, 0xfd,
			0x23, 0x90, 0xdc, 0xac, 0xd5, 0x84, 0x27, 0x99,
		},
		/* S2 : 384 byte */
		{
			0x0d, 0xf8, 0x1e, 0xc6, 0xe9, 0xc2, 0xf0, 0xeb,
			0xe8, 0x24, 0xc4, 0x45, 0x00, 0x99, 0x02, 0xcd,
			0x55, 0xe2, 0x71, 0x85, 0x23, 0x54, 0x6f, 0x08,
			0xed, 0x13, 0xfa, 0xf8, 0x11, 0xec, 0x4e, 0x57,
			0xe6, 0xf5, 0x77, 0x20, 0x37, 0xe0, 0x70, 0x25,
			0xc3, 0xc0, 0xc9, 0x9c, 0xd9, 0xd6, 0xc8, 0x85,
			0x68, 0x2e, 0x0e, 0xb9, 0x04, 0xa3, 0x31, 0x4b,
			0x82, 0x59, 0x48, 0x81, 0x9a, 0xce, 0xcd, 0x19,
			0x5c, 0x84, 0x5a, 0x81, 0xe2, 0x2a, 0xe6, 0x2c,
			0x13, 0x25, 0x18, 0x23, 0xd6, 0xee, 0x38, 0x6e,
			0x0b, 0xe1, 0x7a, 0x60, 0x4b, 0xaf, 0xc6, 0x49,
			0x7b, 0x7a, 0x6c, 0xda, 0xad, 0x1a, 0x33, 0xcd,
			0x5a, 0xe3, 0x3b, 0xdd, 0x50, 0xe6, 0x20, 0x63,
			0xbd, 0xdf, 0x6d, 0x12, 0xb8, 0x78, 0xb3, 0x1d,
			0x3b, 0x7d, 0x49, 0x0c, 0xe8, 0x68, 0x10, 0xf9,
			0xd4, 0x56, 0x73, 0x9b, 0xce, 0xbd, 0xe5, 0x92,
			0xb0, 0x78, 0x08, 0x35, 0x0a, 0xee, 0x54, 0x24,
			0x55, 0xd1, 0x76, 0x11, 0x54, 0x18, 0x8e, 0x6e,
			0x02, 0xcb, 0xda, 0x79, 0x5e, 0x48, 0xe4, 0xf2,
			0x8a, 0xcb, 0x81, 0x94, 0x40, 0xbc, 0xd8, 0xda,
			0x53, 0xfd, 0xf1, 0x98, 0x08, 0x45, 0x68, 0x98,
			0xa1, 0x8f, 0xba, 0x51, 0x7a, 0xf0, 0x6b, 0x51,
			0x15, 0x61, 0x29, 0xb0, 0xb8, 0x02, 0x95, 0x47,
			0xca, 0x9b, 0xd9, 0x43, 0x6a, 0x06, 0x73, 0xe5,
			0xb5, 0xcb, 0x99, 0x53, 0x40, 0xfc, 0x42, 0x5f,
			0xec, 0xc5, 0x66, 0xac, 0xc9, 0x98, 0x84, 0xe0,
			0xb4, 0xfc, 0x87, 0x24, 0x8f, 0x5b, 0x35, 0xbb,
			0xf0, 0x8b, 0x0d, 0xfd, 0x0b, 0x9e, 0xad, 0x06,
			0x73, 0x7b, 0x67, 0xc8, 0x5f, 0x94, 0xe1, 0xea,
			0xc8, 0x80, 0x2f, 0xea, 0x1b, 0x1d, 0xce, 0xa4,
			0x46, 0xb7, 0xca, 0xb8, 0xa4, 0x5b, 0x25, 0x42,
			0x97, 0x50, 0x94, 0x6b, 0xc8, 0xb2, 0x2e, 0x07,
			0x68, 0x28, 0xa0, 0xa9, 0x71, 0x82, 0x77, 0x56,
			0x8b, 0x9b, 0x72, 0x02, 0xa8, 0xcc, 0x36, 0x88,
			0xd4, 0x41, 0x94, 0xe8, 0x34, 0xe0, 0xa4, 0x05,
			0xfb, 0x9e, 0xea, 0x46, 0xbc, 0x7e, 0x94, 0x25,
			0x5d, 0x60, 0x0f, 0xf6, 0xc9, 0x5a, 0x46, 0xeb,
			0xf4, 0x64, 0x49, 0x51, 0x0f, 0xdb, 0x39, 0xb6,
			0xce, 0x05, 0xa2, 0x0a, 0xc1, 0x83, 0x29, 0x38,
			0xb6, 0x59, 0x31, 0x87, 0x64, 0xdc, 0x0b, 0x7e,
			0x4a, 0x02, 0x15, 0xfd, 0x25, 0x3f, 0x52, 0x19,
			0x29, 0x6f, 0xbc, 0x82, 0xf0, 0x3a, 0x7b, 0x95,
			0xa1, 0x26, 0x28, 0xd2, 0x19, 0x09, 0x3e, 0x2c,
			0xda, 0xc4, 0x2e, 0x20, 0xeb, 0xa3, 0xdd, 0x5a,
			0xee, 0xb9, 0xdd, 0x7b, 0xef, 0x5d, 0x64, 0x7f,
			0x15, 0x1b, 0x04, 0xab, 0x85, 0xc4, 0x89, 0x70,
			0xcf, 0xe7, 0x3e, 0xf9, 0xfc, 0x3e, 0x7d, 0x1d,
			0x8a, 0x13, 0x8d, 0xec, 0x3f, 0x5d, 0x5f, 0xb5,
		},
		28, /* salt length */
		/* V2 : 28 byte */
		{
			0x3f, 0x80, 0x50, 0x57, 0x47, 0x1a, 0xab, 0x0a,
			0x28, 0xcf, 0xc8, 0x43, 0x0d, 0xab, 0xcf, 0x99,
			0x06, 0x12, 0xe8, 0xa9, 0x08, 0xb1, 0x58, 0xae,
			0x36, 0xb4, 0xed, 0x53, 
		},
	},
	/* SHA224 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0xf7, 0xb2, 0x2d, 0xe3, 0xbe, 0xe8, 0x29, 0x5c,
			0x4d, 0x8c, 0x8a, 0x94, 0xda, 0x8c, 0xd7, 0x04,
			0xc5, 0x54, 0x1c, 0x97, 0x21, 0x43, 0x90, 0xbc,
			0x6f, 0x5c, 0x75, 0xba, 0xac, 0x3f, 0x40, 0x45,
			0x8f, 0x57, 0xfa, 0x4e, 0x0c, 0x54, 0xe6, 0x1f,
			0x1c, 0xdc, 0x64, 0xa6, 0xc0, 0x7d, 0x15, 0x11,
			0x43, 0xe7, 0x40, 0x9c, 0xc0, 0x58, 0x74, 0xa7,
			0xe5, 0x57, 0x6f, 0x0c, 0xf6, 0xa5, 0x3f, 0xaf,
			0x15, 0x71, 0xa7, 0x57, 0xc0, 0xcb, 0xc4, 0xbc,
			0x9b, 0x5b, 0xf0, 0xe1, 0x70, 0x53, 0xe7, 0xa3,
			0x74, 0xa2, 0x29, 0x92, 0xcc, 0x6b, 0x9f, 0x01,
			0x4f, 0xb5, 0x80, 0x59, 0x8e, 0x64, 0x76, 0xb3,
			0x11, 0x68, 0xfd, 0xa5, 0xe4, 0x34, 0x0c, 0x5b,
			0x53, 0x71, 0xf8, 0xea, 0xf1, 0xf4, 0x95, 0xe2,
			0xdf, 0xee, 0x9e, 0x22, 0x4a, 0x63, 0x57, 0xf1,
			0x36, 0xde, 0x70, 0x4a, 0x7a, 0x62, 0x2d, 0x76,
		},
		/* S3 : 384 byte */
		{
			0x72, 0x76, 0x69, 0xab, 0xeb, 0x6b, 0xcc, 0x95,
			0x02, 0xd7, 0xe8, 0x81, 0x62, 0xf4, 0xa6, 0xc1,
			0xdf, 0xe1, 0xa0, 0xf5, 0x14, 0x1d, 0x37, 0x63,
			0xe0, 0xf7, 0xe1, 0x67, 0x44, 0xf9, 0x06, 0x38,
			0x74, 0xf1, 0x53, 0xcc, 0x2d, 0xe4, 0x87, 0x84,
			0xde, 0x84, 0x42, 0x6b, 0x54, 0x8b, 0x03, 0xe0,
			0x5a, 0x90, 0x74, 0xce, 0xf6, 0xa9, 0x51, 0x64,
			0x0e, 0xaf, 0x9b, 0x32, 0x01, 0x4d, 0x97, 0xcd,
			0x9f, 0x3a, 0x82, 0x8b, 0x45, 0xde, 0xf1, 0x35,
			0x27, 0xf7, 0x2a, 0x3e, 0x5e, 0x5a, 0xdc, 0xca,
			0xec, 0xe8, 0x22, 0x12, 0xc0, 0x16, 0xc2, 0x8f,
			0x9f, 0x33, 0x12, 0x85, 0x3b, 0xf5, 0x20, 0x62,
			0xe7, 0x19, 0x08, 0x1b, 0xc0, 0x28, 0xf7, 0x08,
			0x31, 0xf9, 0xfc, 0x91, 0x32, 0xe8, 0xb6, 0x38,
			0x24, 0xe3, 0x7c, 0x7c, 0xde, 0xba, 0x46, 0x3f,
			0x90, 0x34, 0xd8, 0x15, 0x68, 0x3e, 0x27, 0x75,
			0x0c, 0xb9, 0xb3, 0x83, 0xc3, 0x42, 0x0f, 0x12,
			0x2a, 0x3b, 0x7f, 0xc6, 0xe9, 0x44, 0x09, 0x25,
			0xa7, 0x7d, 0x76, 0x6f, 0x93, 0xd5, 0x86, 0x16,
			0x1e, 0x96, 0x07, 0xbe, 0xb8, 0xa6, 0xe4, 0xac,
			0x72, 0xc3, 0x2e, 0xf7, 0xb6, 0x9e, 0xd5, 0x2f,
			0x50, 0x77, 0xa8, 0x81, 0xdd, 0x0e, 0x49, 0x45,
			0x91, 0xe2, 0xba, 0x55, 0x2b, 0x74, 0x73, 0x1c,
			0x18, 0xce, 0xce, 0x99, 0x05, 0x56, 0x14, 0x59,
			0xf4, 0x55, 0x3d, 0x49, 0xac, 0xfd, 0x6c, 0xc6,
			0xbe, 0x02, 0x78, 0x33, 0xa2, 0x20, 0x42, 0x9d,
			0x46, 0xbc, 0xb8, 0x8d, 0xfc, 0xff, 0x0d, 0x2c,
			0x5c, 0xb5, 0x67, 0x37, 0x15, 0x63, 0xb4, 0x85,
			0x2b, 0x7e, 0x62, 0x8c, 0x4a, 0x64, 0x32, 0xaf,
			0x96, 0x7e, 0x8e, 0xd6, 0x9c, 0x9b, 0x64, 0x28,
			0xac, 0x55, 0x2c, 0xd3, 0x70, 0x92, 0x2a, 0x0a,
			0x4b, 0x01, 0xef, 0x1b, 0xdf, 0xdc, 0xbc, 0x90,
			0x88, 0xcd, 0xfb, 0x6d, 0x9f, 0xe3, 0x26, 0xbd,
			0x6b, 0x2b, 0xb1, 0xfc, 0x2a, 0xcf, 0xea, 0x3b,
			0xcf, 0x60, 0xd1, 0xfa, 0xc5, 0x88, 0x0b, 0x05,
			0x10, 0x73, 0x6b, 0x7e, 0x20, 0x1e, 0xe8, 0xf6,
			0xbc, 0x63, 0x32, 0xc0, 0x75, 0x63, 0x15, 0x78,
			0x97, 0x00, 0x35, 0x0f, 0xa5, 0x49, 0x00, 0x9d,
			0x16, 0xe0, 0xba, 0xc0, 0x84, 0xbf, 0x6a, 0xa3,
			0x49, 0x2f, 0x63, 0x36, 0x78, 0x19, 0x50, 0x6b,
			0xf0, 0xc4, 0xf9, 0xc2, 0x32, 0xfb, 0xd7, 0xc4,
			0xd4, 0xad, 0x66, 0x3a, 0x75, 0x66, 0x10, 0x82,
			0x38, 0xc3, 0x1f, 0xed, 0x88, 0x7f, 0x36, 0x86,
			0x66, 0xdc, 0x75, 0xa6, 0x23, 0xf2, 0x22, 0xd3,
			0x57, 0xf8, 0xe5, 0x23, 0xff, 0x08, 0x41, 0x11,
			0xbe, 0x4d, 0xb6, 0xba, 0xf4, 0x44, 0xf1, 0x91,
			0xad, 0x14, 0x68, 0xd0, 0x77, 0x34, 0x9f, 0xef,
			0x8a, 0x22, 0xf3, 0xfa, 0x56, 0x08, 0x59, 0x75,
		},
		28, /* salt length */
		/* V3 : 28 byte */
		{
			0x3f, 0x80, 0x50, 0x57, 0x47, 0x1a, 0xab, 0x0a,
			0x28, 0xcf, 0xc8, 0x43, 0x0d, 0xab, 0xcf, 0x99,
			0x06, 0x12, 0xe8, 0xa9, 0x08, 0xb1, 0x58, 0xae,
			0x36, 0xb4, 0xed, 0x53, 
		},
	},
	/* SHA224 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0x8d, 0x48, 0xfd, 0xdf, 0x28, 0xb0, 0x5b, 0x42,
			0xc9, 0xb4, 0xdf, 0x47, 0x42, 0xed, 0x8e, 0x73,
			0x5a, 0x14, 0x0a, 0x69, 0x72, 0x16, 0x5c, 0xa6,
			0x69, 0x6b, 0xf0, 0x6e, 0xbe, 0xa4, 0xe1, 0x06,
			0xf4, 0x44, 0x78, 0x24, 0x3b, 0xd1, 0xef, 0xa4,
			0x4c, 0x2b, 0x7a, 0x7c, 0x95, 0x1c, 0x88, 0xf2,
			0x96, 0x2f, 0x45, 0x0d, 0x8b, 0xc6, 0x64, 0x49,
			0x4b, 0x67, 0x1d, 0x8e, 0x70, 0x57, 0x71, 0x63,
			0xb8, 0x6a, 0xb5, 0x60, 0xab, 0x19, 0x4e, 0xe1,
			0x7e, 0xd5, 0xba, 0x02, 0x38, 0x9b, 0xd0, 0xc7,
			0x13, 0xc9, 0x48, 0x9a, 0x25, 0x30, 0x7d, 0xfb,
			0x3f, 0x6a, 0x72, 0x73, 0x16, 0x6d, 0x13, 0xc9,
			0xa0, 0x61, 0xbe, 0x79, 0xc1, 0xaf, 0x02, 0x62,
			0x27, 0x5b, 0xa7, 0xbf, 0x73, 0x93, 0xee, 0x58,
			0x99, 0x88, 0x19, 0xfa, 0x89, 0x7c, 0x2e, 0x24,
			0x0f, 0x1c, 0xf9, 0x03, 0xf7, 0x11, 0x50, 0xa0,
		},
		/* S4 : 384 byte */
		{
			0xa1, 0xa4, 0xd1, 0x69, 0x56, 0xd7, 0x18, 0x83,
			0x0f, 0x62, 0x5f, 0x06, 0xc4, 0x2e, 0x99, 0x18,
			0x9e, 0x36, 0xa8, 0x05, 0x23, 0xb2, 0x5f, 0x0c,
			0x9a, 0x7b, 0xb8, 0x55, 0x68, 0xce, 0x76, 0xd1,
			0xe8, 0x5e, 0x43, 0x7d, 0xb0, 0xa7, 0x72, 0x8b,
			0x8a, 0x9c, 0x90, 0xd2, 0x5e, 0x6f, 0x38, 0x15,
			0x02, 0x08, 0xde, 0xbe, 0x54, 0xe1, 0xe3, 0xf6,
			0x48, 0xff, 0x01, 0x79, 0x8a, 0x8c, 0xe1, 0x32,
			0xe4, 0xb3, 0x3f, 0x3d, 0x26, 0xfa, 0x89, 0x63,
			0x77, 0x14, 0x40, 0xfd, 0xc4, 0xf5, 0xd8, 0x52,
			0x11, 0x7b, 0x3c, 0xce, 0xa9, 0x75, 0xda, 0x10,
			0xe5, 0xd4, 0xf2, 0x7a, 0xf1, 0xbe, 0xc1, 0xb8,
			0x53, 0xb7, 0xb5, 0xc9, 0xb4, 0x20, 0x01, 0x23,
			0x17, 0xa6, 0xc3, 0x3b, 0x25, 0x96, 0xdb, 0xdc,
			0xeb, 0xf9, 0x7b, 0xef, 0x82, 0x1b, 0x30, 0x76,
			0xce, 0x86, 0x34, 0x53, 0x09, 0xb6, 0xbd, 0xf2,
			0x9a, 0x4a, 0xcd, 0x39, 0x1d, 0x3b, 0x2e, 0x5c,
			0x4a, 0x68, 0x66, 0x13, 0x62, 0x87, 0xd1, 0x7c,
			0xb0, 0xe2, 0xd4, 0xd6, 0xa6, 0xcf, 0x89, 0xd6,
			0x42, 0x72, 0xd5, 0xc0, 0x18, 0x49, 0xed, 0x57,
			0xfa, 0x28, 0x42, 0x07, 0x4d, 0x3b, 0x77, 0x34,
			0xc4, 0xc9, 0x2b, 0xe5, 0x0a, 0x92, 0x2d, 0x05,
			0x17, 0xeb, 0xb9, 0x89, 0x10, 0x72, 0xb1, 0xb4,
			0x7a, 0x71, 0x08, 0x87, 0x00, 0x4b, 0x23, 0x8f,
			0x90, 0x07, 0x9d, 0x10, 0xfb, 0x2c, 0xad, 0x7f,
			0x50, 0x13, 0xe7, 0x24, 0x30, 0x89, 0xf3, 0xc6,
			0x01, 0x86, 0x5c, 0x6b, 0xce, 0x1c, 0xb8, 0xd0,
			0xd6, 0x69, 0xf2, 0xbb, 0x70, 0x92, 0x53, 0xe3,
			0xf1, 0xe4, 0x21, 0x93, 0x6f, 0x6a, 0x16, 0x43,
			0xbb, 0xbb, 0x7d, 0x50, 0x3b, 0x06, 0x31, 0xf7,
			0xe1, 0x66, 0x03, 0x82, 0xba, 0xcf, 0x46, 0x80,
			0xde, 0x8d, 0x70, 0xe2, 0x4a, 0xbf, 0x44, 0x50,
			0x51, 0x0e, 0x6b, 0x40, 0x47, 0x5b, 0xfc, 0x9f,
			0xe5, 0x47, 0x75, 0x2d, 0x0d, 0x5f, 0x63, 0xf4,
			0x0f, 0x62, 0xf4, 0xdc, 0xc9, 0x03, 0xfe, 0x6d,
			0x26, 0x0f, 0xa4, 0x5a, 0x1b, 0x85, 0xa7, 0x50,
			0x10, 0x65, 0xaa, 0x19, 0x00, 0xa3, 0xf8, 0x41,
			0xe5, 0x4c, 0x13, 0x6d, 0x68, 0x6f, 0xad, 0xbb,
			0x33, 0xb2, 0x25, 0xd1, 0x5a, 0xe6, 0xfc, 0x34,
			0x8b, 0xe5, 0x7f, 0xc9, 0xcc, 0xbf, 0xde, 0xb5,
			0x7d, 0x5c, 0xbf, 0x53, 0xe3, 0x47, 0x9d, 0x9b,
			0xae, 0x9f, 0x4f, 0xf8, 0x59, 0xcb, 0xd3, 0xfb,
			0x07, 0x60, 0x73, 0xca, 0x01, 0x6a, 0xd9, 0x40,
			0x86, 0x70, 0x0c, 0xc8, 0x5a, 0xce, 0xd8, 0x3a,
			0xeb, 0xb4, 0x25, 0x4b, 0x0c, 0xfc, 0x81, 0x45,
			0x85, 0xf9, 0x30, 0xdc, 0x62, 0x3c, 0x7f, 0x85,
			0xe8, 0x9d, 0xe6, 0xa5, 0x54, 0xb9, 0x89, 0x89,
			0x18, 0xd7, 0xcb, 0xb4, 0xcd, 0x2d, 0xb0, 0x75,
		},
		28, /* salt length */
		/* V4 : 28 byte */
		{
			0x3f, 0x80, 0x50, 0x57, 0x47, 0x1a, 0xab, 0x0a,
			0x28, 0xcf, 0xc8, 0x43, 0x0d, 0xab, 0xcf, 0x99,
			0x06, 0x12, 0xe8, 0xa9, 0x08, 0xb1, 0x58, 0xae,
			0x36, 0xb4, 0xed, 0x53, 
		},
	},
	/* SHA224 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0x47, 0x53, 0x18, 0x3c, 0xe5, 0x60, 0x7f, 0xa0,
			0x36, 0x36, 0xdb, 0x2f, 0xdc, 0x84, 0x72, 0x2a,
			0xeb, 0x9d, 0x98, 0xa6, 0xed, 0x70, 0xd0, 0x28,
			0x2a, 0xba, 0x35, 0x71, 0x26, 0x7a, 0x18, 0x9b,
			0x6a, 0xa6, 0xeb, 0x65, 0x87, 0x1c, 0x5d, 0xcc,
			0x59, 0xdb, 0xc7, 0xdb, 0x89, 0x73, 0xc7, 0xc3,
			0x55, 0xba, 0x2a, 0x2e, 0x94, 0xc1, 0x10, 0xd1,
			0xf4, 0x06, 0x4a, 0x40, 0x87, 0xeb, 0x07, 0x07,
			0x7e, 0x67, 0xb0, 0xf6, 0x34, 0xfc, 0x10, 0xbc,
			0x6e, 0xe9, 0xb8, 0xb8, 0xe1, 0xa0, 0xa2, 0x0b,
			0xf4, 0x7a, 0x14, 0xf2, 0xc8, 0xaa, 0xc7, 0x53,
			0x75, 0x70, 0x49, 0x95, 0x97, 0x8f, 0xa0, 0xb5,
			0x0a, 0x00, 0x30, 0x96, 0xf1, 0xe8, 0xdf, 0x99,
			0xfd, 0xc8, 0x76, 0x6e, 0xec, 0xf3, 0x4a, 0x2a,
			0x4f, 0x46, 0x1d, 0x99, 0x91, 0x13, 0x3f, 0xd5,
			0x35, 0x5e, 0xf8, 0x17, 0x5f, 0x4c, 0x2b, 0xce,
		},
		/* S5 : 384 byte */
		{
			0x2e, 0x07, 0x8b, 0x29, 0xb5, 0x28, 0x8a, 0x77,
			0xed, 0x25, 0xec, 0xec, 0xec, 0xa6, 0x45, 0xf6,
			0xd9, 0x29, 0x8e, 0x42, 0x94, 0xe3, 0xef, 0x08,
			0x17, 0x3c, 0xc3, 0x7c, 0xcb, 0xf7, 0x27, 0xac,
			0x9b, 0x09, 0x2c, 0xd2, 0x7d, 0x6f, 0xbd, 0x37,
			0x8f, 0xff, 0x7b, 0x10, 0x61, 0xb5, 0x6e, 0xd5,
			0xcf, 0x07, 0x7f, 0xd1, 0xa2, 0x27, 0x77, 0x1f,
			0x58, 0xcb, 0xb2, 0xc1, 0x19, 0x5a, 0x01, 0xf8,
			0x30, 0xf0, 0x36, 0x6f, 0x98, 0x9a, 0xa2, 0xd0,
			0xc4, 0x86, 0xd4, 0x41, 0xe1, 0x12, 0xda, 0xea,
			0xf8, 0x3e, 0x85, 0x95, 0x8f, 0x65, 0xa9, 0xe6,
			0x0a, 0x19, 0x37, 0xd2, 0xa7, 0x02, 0x27, 0x81,
			0xfc, 0xd1, 0xa8, 0x3b, 0x3f, 0x76, 0x41, 0xa7,
			0x43, 0x00, 0x1e, 0xba, 0xd5, 0x3a, 0x46, 0x69,
			0x40, 0x56, 0x03, 0xba, 0x03, 0x93, 0xbc, 0xd9,
			0x4f, 0x64, 0x32, 0x4f, 0x1b, 0x77, 0x70, 0x68,
			0xa3, 0xab, 0x10, 0x1a, 0x08, 0x6a, 0x69, 0x72,
			0xb2, 0xc1, 0x13, 0x76, 0x30, 0x7e, 0x7d, 0x24,
			0x85, 0xfb, 0xfa, 0xd8, 0x5b, 0xe7, 0x17, 0x1d,
			0x20, 0xa5, 0x25, 0x1c, 0xf9, 0xa5, 0xf0, 0x04,
			0x84, 0x7d, 0x17, 0x2c, 0x77, 0xbd, 0x80, 0xfb,
			0xac, 0x08, 0x70, 0xa0, 0xb6, 0xbb, 0x97, 0x33,
			0x53, 0x7c, 0xa7, 0x2b, 0xb6, 0xea, 0xc3, 0x51,
			0xc2, 0x15, 0x88, 0x28, 0x7c, 0x31, 0x76, 0x25,
			0xa2, 0x5f, 0x41, 0x61, 0x29, 0xe6, 0xf5, 0x3c,
			0x60, 0x7a, 0xe0, 0x8f, 0x43, 0xe5, 0xe0, 0x33,
			0x97, 0x40, 0x77, 0x5a, 0x53, 0x1c, 0x72, 0x0f,
			0x3f, 0x73, 0x18, 0x40, 0x18, 0x4a, 0xc7, 0xcd,
			0x3b, 0x1f, 0x7b, 0xb8, 0x20, 0xff, 0x30, 0xba,
			0x7b, 0xb1, 0x20, 0xb2, 0x1b, 0x4b, 0xae, 0x7f,
			0x9d, 0x7f, 0xc3, 0x4d, 0x74, 0x18, 0xf7, 0x00,
			0xb1, 0x42, 0xcf, 0x8f, 0xff, 0x43, 0xd8, 0x15,
			0x99, 0x23, 0x6e, 0xba, 0xbe, 0x93, 0xd2, 0xe8,
			0x9f, 0x47, 0x02, 0xfa, 0xda, 0x87, 0x42, 0xdc,
			0x3b, 0xb4, 0xbc, 0x8f, 0xc5, 0xe5, 0x5b, 0x4f,
			0x87, 0x4a, 0xe5, 0x9f, 0x5d, 0xc9, 0x63, 0x68,
			0x68, 0x82, 0x8e, 0xfb, 0xe1, 0x02, 0x5a, 0x8c,
			0xa5, 0xc6, 0x1e, 0xd8, 0xcc, 0x83, 0x26, 0x86,
			0xd5, 0xd0, 0x0c, 0x08, 0x77, 0x55, 0x90, 0xb3,
			0x16, 0x06, 0x02, 0x85, 0xdc, 0x5b, 0xb9, 0xd3,
			0x2c, 0x90, 0xa4, 0x74, 0xa7, 0x27, 0xdd, 0xba,
			0x9e, 0x7a, 0x8b, 0x7d, 0x69, 0xba, 0xe5, 0x55,
			0x60, 0x4a, 0xdd, 0x9d, 0xe0, 0xda, 0xb0, 0xeb,
			0x0d, 0x55, 0x1b, 0xac, 0x06, 0x7c, 0x00, 0x88,
			0x52, 0x3d, 0x13, 0x4b, 0x2e, 0x50, 0xdf, 0xe3,
			0xff, 0x73, 0xee, 0xfe, 0xd9, 0x34, 0xc0, 0x98,
			0x4a, 0xa4, 0xa5, 0xc5, 0x63, 0xb8, 0x62, 0xd4,
			0x6e, 0xd9, 0x57, 0xec, 0x34, 0x46, 0xfd, 0x24,
		},
		28, /* salt length */
		/* V5 : 28 byte */
		{
			0x3f, 0x80, 0x50, 0x57, 0x47, 0x1a, 0xab, 0x0a,
			0x28, 0xcf, 0xc8, 0x43, 0x0d, 0xab, 0xcf, 0x99,
			0x06, 0x12, 0xe8, 0xa9, 0x08, 0xb1, 0x58, 0xae,
			0x36, 0xb4, 0xed, 0x53, 
		},
	},
	/* SHA224 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0xaa, 0xd0, 0x3f, 0x3a, 0xa4, 0xcb, 0xd2, 0x36,
			0xd3, 0x0f, 0xcf, 0x23, 0x9c, 0x40, 0xda, 0x68,
			0xde, 0x8e, 0xf5, 0x4d, 0xcb, 0x36, 0xf5, 0xa6,
			0xf6, 0x4b, 0x32, 0xb6, 0xac, 0xb6, 0x83, 0x4e,
			0x88, 0x7c, 0x6a, 0x35, 0x42, 0x3f, 0x8b, 0xcc,
			0xc8, 0x08, 0x63, 0xf2, 0x90, 0x43, 0x36, 0x26,
			0x2c, 0x0b, 0x49, 0xeb, 0x1f, 0xa8, 0x52, 0x71,
			0xef, 0x56, 0x2d, 0x71, 0x7b, 0x48, 0xd0, 0x59,
			0x8f, 0xed, 0x81, 0xa9, 0xb6, 0x72, 0x47, 0x9d,
			0x4f, 0x88, 0x9e, 0x0c, 0xe3, 0x67, 0x6e, 0x90,
			0xb6, 0x13, 0x3e, 0xe7, 0x9c, 0xde, 0xa5, 0x99,
			0x0e, 0x2e, 0x02, 0xdb, 0x7d, 0x80, 0x6d, 0xb4,
			0xe6, 0xad, 0xee, 0x5e, 0xa7, 0x6c, 0xec, 0xef,
			0x91, 0x19, 0xe8, 0x39, 0x3e, 0xb5, 0x6b, 0xee,
			0xa5, 0x2d, 0x3c, 0x08, 0xeb, 0xdf, 0xd7, 0x67,
			0x7d, 0x5a, 0x1b, 0xbc, 0x5b, 0x65, 0x43, 0xa7,
		},
		/* S6 : 384 byte */
		{
			0x1b, 0xc3, 0x25, 0x41, 0x2c, 0xc9, 0x52, 0xa8,
			0xdd, 0x69, 0x18, 0xdb, 0x8f, 0xb0, 0x81, 0x92,
			0xcd, 0xf8, 0x1b, 0xf4, 0x11, 0x1c, 0xb5, 0xf0,
			0xa5, 0x80, 0xa8, 0x2d, 0x4d, 0xd2, 0xe1, 0x4d,
			0x74, 0x45, 0xeb, 0x7c, 0xb9, 0x4c, 0xca, 0x6d,
			0xa0, 0x6d, 0x2b, 0x5c, 0xc4, 0x3e, 0x6e, 0xc2,
			0x2a, 0x5c, 0x9c, 0x84, 0x5d, 0x99, 0xac, 0x03,
			0x53, 0x05, 0x0c, 0x13, 0x74, 0x86, 0x6b, 0xef,
			0xd9, 0xb6, 0xb8, 0x49, 0xcf, 0x3b, 0x0e, 0xfc,
			0xc6, 0x44, 0xce, 0x17, 0xcc, 0xa0, 0xda, 0xfc,
			0xf7, 0x70, 0x0c, 0x9c, 0x7d, 0x87, 0x0c, 0x1e,
			0x14, 0x51, 0x16, 0x51, 0xb1, 0xd0, 0x3a, 0x53,
			0x51, 0x10, 0x13, 0x9c, 0x53, 0xb5, 0x59, 0x38,
			0xcc, 0x4a, 0x47, 0x1d, 0x75, 0x6a, 0x55, 0xb5,
			0x0d, 0x1b, 0xd2, 0x80, 0xc3, 0x24, 0xac, 0x4d,
			0xba, 0xf5, 0x26, 0x59, 0x0c, 0x48, 0xc1, 0x97,
			0x57, 0x3f, 0x3a, 0x91, 0xc7, 0x03, 0x73, 0xec,
			0x62, 0xbd, 0x16, 0x82, 0x88, 0xb0, 0xd1, 0x63,
			0xa0, 0x9e, 0x62, 0x35, 0x89, 0xd1, 0xca, 0x5a,
			0x70, 0xd1, 0x7a, 0xa5, 0x4c, 0x86, 0x27, 0xc7,
			0xa6, 0x4d, 0x92, 0x1a, 0xad, 0x12, 0x62, 0x6f,
			0x7d, 0x32, 0xd6, 0x1e, 0x8f, 0x14, 0xd0, 0xaa,
			0x97, 0xc2, 0xd6, 0x50, 0x20, 0x21, 0xe7, 0x08,
			0x55, 0x58, 0x1f, 0x5e, 0x35, 0x3e, 0x27, 0xf9,
			0x6e, 0xfe, 0x1b, 0xc7, 0x8c, 0x7f, 0xba, 0xec,
			0xe6, 0x6a, 0x56, 0x0b, 0x93, 0xc0, 0xe7, 0x36,
			0x5d, 0x97, 0xdc, 0x4c, 0x72, 0x92, 0x35, 0x48,
			0x4a, 0xbe, 0x10, 0xbc, 0xca, 0xe9, 0x9f, 0xa8,
			0xdb, 0x94, 0x25, 0x61, 0x4b, 0x67, 0x3d, 0x5b,
			0xbc, 0x18, 0x8e, 0xa8, 0xf4, 0x65, 0x42, 0x4f,
			0x76, 0x8d, 0x80, 0x31, 0xf7, 0xee, 0xfb, 0xb6,
			0x98, 0xf0, 0x58, 0xe1, 0x57, 0x8a, 0xc4, 0x14,
			0x26, 0x73, 0x94, 0x10, 0xaa, 0x7e, 0xac, 0xf7,
			0x96, 0xf4, 0x3a, 0x4e, 0x4b, 0x2b, 0x4a, 0x46,
			0x39, 0x84, 0xd3, 0xd1, 0x7d, 0x6d, 0x66, 0x7c,
			0xd1, 0x5b, 0xf2, 0xe2, 0xb4, 0x87, 0xae, 0xc3,
			0x49, 0x34, 0x40, 0x79, 0x4c, 0x09, 0x90, 0x85,
			0x45, 0xf4, 0x16, 0xb7, 0x01, 0xa1, 0x30, 0xf0,
			0x80, 0x27, 0xb8, 0xbc, 0xab, 0x4d, 0xc4, 0xa7,
			0x8c, 0xf4, 0xa5, 0x5a, 0x68, 0x8b, 0x2e, 0x1c,
			0xa3, 0xa7, 0x3a, 0x08, 0xff, 0x0e, 0xd8, 0x90,
			0xbe, 0xe4, 0xa0, 0xfa, 0x85, 0x8c, 0xf6, 0x91,
			0x42, 0xf2, 0xf7, 0x65, 0x40, 0x0e, 0x7c, 0x29,
			0xc4, 0xb5, 0x40, 0x53, 0x0a, 0x05, 0x46, 0x41,
			0x96, 0x14, 0x99, 0xc7, 0x09, 0xdb, 0xb4, 0xf3,
			0x6e, 0x7e, 0x75, 0xa5, 0x99, 0x3c, 0xb3, 0xab,
			0x8c, 0xd4, 0xc8, 0x86, 0xf6, 0xa3, 0xf5, 0xe3,
			0xbd, 0xd3, 0xd6, 0x8e, 0xf0, 0xa7, 0x77, 0x50,
		},
		28, /* salt length */
		/* V6 : 28 byte */
		{
			0x3f, 0x80, 0x50, 0x57, 0x47, 0x1a, 0xab, 0x0a,
			0x28, 0xcf, 0xc8, 0x43, 0x0d, 0xab, 0xcf, 0x99,
			0x06, 0x12, 0xe8, 0xa9, 0x08, 0xb1, 0x58, 0xae,
			0x36, 0xb4, 0xed, 0x53, 
		},
	},
	/* SHA224 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0xc8, 0x28, 0xec, 0xa4, 0x60, 0xb3, 0x97, 0x03,
			0x69, 0x67, 0x50, 0x99, 0x9e, 0x23, 0x48, 0x6a,
			0x43, 0x2d, 0x80, 0x00, 0x08, 0x82, 0xd0, 0x61,
			0x31, 0x6b, 0x2e, 0x3e, 0xf4, 0x51, 0x2d, 0x6d,
			0x22, 0xd2, 0xc4, 0x9a, 0x0a, 0x15, 0x51, 0x39,
			0x9b, 0x5a, 0xdd, 0xbe, 0xc8, 0xd5, 0xa2, 0x11,
			0x31, 0xbc, 0xca, 0x3c, 0xff, 0x9f, 0x7a, 0x67,
			0x0f, 0xf8, 0x0f, 0x07, 0x54, 0x03, 0xa8, 0x52,
			0x76, 0xcf, 0xe4, 0xf6, 0xbf, 0x95, 0xed, 0x0a,
			0x38, 0x4a, 0xb5, 0x45, 0x0f, 0x70, 0x7f, 0x6e,
			0x3c, 0x31, 0xa2, 0x13, 0x64, 0xae, 0x89, 0x7e,
			0xfe, 0x95, 0xff, 0xe5, 0xb4, 0xf1, 0xa9, 0xe1,
			0x0c, 0x47, 0xd4, 0x21, 0x47, 0xde, 0x72, 0x60,
			0x8a, 0x5e, 0x5e, 0x94, 0x3b, 0x9d, 0xe8, 0x69,
			0xae, 0xb5, 0x8d, 0xed, 0x01, 0x5a, 0x06, 0x8d,
			0x44, 0x6a, 0x85, 0x40, 0xdd, 0xc6, 0x3b, 0x02,
		},
		/* S7 : 384 byte */
		{
			0x79, 0x94, 0x50, 0xa1, 0x25, 0x6d, 0x24, 0x5d,
			0xf0, 0xbb, 0x7d, 0x52, 0x90, 0xab, 0xce, 0xfe,
			0x69, 0xd3, 0xb0, 0xe3, 0xb9, 0x49, 0x24, 0x07,
			0x2f, 0x2d, 0x67, 0xd5, 0x3a, 0x96, 0x65, 0x13,
			0x95, 0x5f, 0xa7, 0xa0, 0x1b, 0x83, 0x0b, 0xa2,
			0xcb, 0xbb, 0x05, 0x67, 0x16, 0xfd, 0x60, 0x5a,
			0x0c, 0xfd, 0xc0, 0x5f, 0x8f, 0xf5, 0x8d, 0x88,
			0xcb, 0x1b, 0xf3, 0x22, 0x48, 0xf1, 0x17, 0xde,
			0x41, 0xdd, 0xfd, 0xc4, 0x66, 0x21, 0x5f, 0xa4,
			0xe7, 0x04, 0x09, 0x69, 0x47, 0xa2, 0xdb, 0xe8,
			0x36, 0xa9, 0x90, 0x71, 0xea, 0x73, 0x44, 0xbe,
			0x0f, 0xfc, 0x78, 0x2d, 0x14, 0xf9, 0x95, 0xe4,
			0xbf, 0xc7, 0x4d, 0xc3, 0xab, 0x1f, 0xa9, 0x6d,
			0x72, 0x23, 0xec, 0x45, 0x64, 0x97, 0xa2, 0xf5,
			0x1e, 0x1e, 0xb1, 0x99, 0xf0, 0x46, 0x4d, 0x41,
			0x5a, 0xef, 0x00, 0xf8, 0x41, 0xe3, 0x9f, 0x45,
			0x78, 0xa0, 0xc2, 0x6d, 0x72, 0x6f, 0x30, 0x65,
			0xee, 0x68, 0x7a, 0xdb, 0xe4, 0x02, 0x07, 0x80,
			0x18, 0x57, 0x16, 0x0d, 0x44, 0x01, 0x51, 0xfa,
			0x37, 0x42, 0x57, 0xea, 0xa3, 0xf7, 0x77, 0x33,
			0x7d, 0x12, 0x9d, 0xc8, 0xb8, 0xc7, 0x01, 0xee,
			0xd5, 0x6a, 0x27, 0x6e, 0xc9, 0x0c, 0x03, 0xdf,
			0x54, 0x30, 0x5f, 0x30, 0x0e, 0xf8, 0xc5, 0x11,
			0x55, 0xdb, 0x30, 0xb6, 0x8c, 0x0b, 0x06, 0xda,
			0xe4, 0xc4, 0xaa, 0x07, 0xe7, 0x5e, 0xf0, 0xfb,
			0x09, 0x29, 0x9b, 0x2b, 0x04, 0xd7, 0x3d, 0x0b,
			0x3e, 0x87, 0x4e, 0xa1, 0xb6, 0xac, 0x4e, 0x16,
			0xf1, 0xbe, 0xd0, 0xcd, 0x8d, 0xd3, 0xcf, 0x95,
			0x8a, 0x27, 0xe1, 0x4e, 0x09, 0x70, 0x5d, 0x4f,
			0x0e, 0x10, 0xf8, 0xd4, 0x6c, 0x75, 0xa1, 0x95,
			0x38, 0x01, 0x26, 0xb4, 0x37, 0xc6, 0x81, 0x83,
			0xe6, 0xbd, 0x39, 0x09, 0x7e, 0x2f, 0x45, 0xb1,
			0x18, 0x4f, 0x51, 0x9b, 0x2e, 0xb1, 0x01, 0x11,
			0x0d, 0xb7, 0x45, 0x19, 0x01, 0x62, 0x97, 0x68,
			0x3a, 0xca, 0x4b, 0x46, 0x1c, 0xec, 0x1d, 0x92,
			0xa7, 0xe6, 0x8c, 0xbf, 0x30, 0xc2, 0xbb, 0x0d,
			0x96, 0xc3, 0xb3, 0x3d, 0xc6, 0x2d, 0x27, 0x8b,
			0x9a, 0x64, 0x04, 0x78, 0x25, 0x8c, 0x34, 0x05,
			0xa6, 0xab, 0x5f, 0xce, 0xf5, 0x28, 0x04, 0x08,
			0xd4, 0x57, 0x3b, 0x7a, 0xe4, 0x24, 0x08, 0xb9,
			0xc4, 0x04, 0x83, 0x76, 0x8f, 0x16, 0xa0, 0x1c,
			0x9e, 0xe4, 0x16, 0x3b, 0x32, 0x5b, 0xbb, 0x8e,
			0x37, 0x70, 0x34, 0xfd, 0x31, 0xc7, 0x87, 0xcc,
			0x0d, 0xb8, 0xa5, 0x3f, 0x6c, 0x0c, 0xe9, 0x3e,
			0x7d, 0x85, 0x44, 0x11, 0xa1, 0x36, 0xe1, 0x01,
			0x3d, 0x69, 0xfd, 0x03, 0xa0, 0x17, 0x11, 0x76,
			0xdc, 0x07, 0x12, 0x64, 0x0e, 0xf2, 0xf7, 0x92,
			0xc3, 0x40, 0xee, 0xdd, 0x0d, 0x07, 0xa8, 0xe6,
		},
		28, /* salt length */
		/* V7 : 28 byte */
		{
			0x3f, 0x80, 0x50, 0x57, 0x47, 0x1a, 0xab, 0x0a,
			0x28, 0xcf, 0xc8, 0x43, 0x0d, 0xab, 0xcf, 0x99,
			0x06, 0x12, 0xe8, 0xa9, 0x08, 0xb1, 0x58, 0xae,
			0x36, 0xb4, 0xed, 0x53, 
		},
	},
	/* SHA224 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0x87, 0xed, 0xd9, 0x71, 0x82, 0xf3, 0x22, 0xc2,
			0x4e, 0x93, 0x76, 0x64, 0xc9, 0x44, 0x43, 0xa2,
			0x5d, 0xd4, 0xeb, 0xe5, 0x28, 0xfe, 0x0c, 0xdf,
			0x5a, 0x3e, 0x05, 0x0a, 0xdf, 0xe4, 0xb6, 0x51,
			0x3f, 0x68, 0x87, 0x0c, 0xc2, 0xfd, 0xab, 0x32,
			0xd7, 0x68, 0xa6, 0xca, 0xb6, 0x13, 0x0c, 0xa3,
			0x45, 0x5d, 0x8c, 0x45, 0x38, 0x35, 0x2e, 0x27,
			0x7d, 0xe7, 0xd9, 0x23, 0xd7, 0x35, 0x18, 0x26,
			0xc9, 0xaa, 0x1d, 0x2c, 0xb5, 0x2b, 0x07, 0x6c,
			0x45, 0xcf, 0x60, 0xcf, 0x0a, 0xf1, 0xea, 0xa7,
			0x63, 0x83, 0x9b, 0x9e, 0xa1, 0xa4, 0xe6, 0xec,
			0x68, 0x75, 0x3c, 0xce, 0x58, 0x29, 0xd3, 0x33,
			0xed, 0x5c, 0xa6, 0xb8, 0xa4, 0xa6, 0xbd, 0xd6,
			0x60, 0x6f, 0xae, 0x5a, 0x0b, 0x05, 0x64, 0x16,
			0x80, 0xeb, 0x1f, 0xd7, 0xa9, 0x75, 0xbc, 0x97,
			0xe4, 0x91, 0x37, 0xf3, 0xac, 0xe8, 0x6e, 0xdf,
		},
		/* S8 : 384 byte */
		{
			0x9c, 0xba, 0x01, 0xf7, 0x9f, 0x35, 0x51, 0xac,
			0xfc, 0xcf, 0x56, 0xe7, 0x44, 0x28, 0xe2, 0x70,
			0x94, 0x9f, 0x78, 0xa0, 0x0b, 0x4f, 0xf3, 0x50,
			0x7e, 0xf1, 0x80, 0xce, 0x4c, 0x78, 0xef, 0x4c,
			0x53, 0xf3, 0xb7, 0x34, 0x7e, 0xe3, 0x76, 0x33,
			0xc6, 0x53, 0xaa, 0xec, 0xa8, 0x34, 0xfc, 0x00,
			0x43, 0x85, 0xf8, 0x77, 0x98, 0x92, 0x2c, 0x53,
			0xf8, 0xfd, 0x74, 0x1c, 0xbc, 0xe1, 0x5d, 0xe8,
			0xdc, 0xae, 0x8b, 0xb0, 0x4c, 0x7d, 0x48, 0x1a,
			0x82, 0x3e, 0xad, 0xac, 0x7d, 0x4d, 0x45, 0x46,
			0xfa, 0x4b, 0x0c, 0xc7, 0xe2, 0x5e, 0x67, 0xb1,
			0x66, 0xed, 0xde, 0x4b, 0x6f, 0x66, 0x74, 0x80,
			0x17, 0xa4, 0xdc, 0xef, 0x85, 0x95, 0x2c, 0xbf,
			0x37, 0xe8, 0x02, 0xfe, 0x53, 0x4e, 0xcb, 0x98,
			0x4c, 0xb3, 0x2f, 0x44, 0x6c, 0x02, 0xcc, 0xb6,
			0x0e, 0x25, 0x7a, 0x18, 0xac, 0x36, 0x8c, 0x2d,
			0x2e, 0xd2, 0x19, 0x75, 0x09, 0x34, 0x99, 0xe3,
			0x58, 0x80, 0x93, 0x0f, 0x85, 0x29, 0x79, 0x0c,
			0x1c, 0x77, 0x62, 0xae, 0x11, 0x52, 0x6e, 0x82,
			0x9d, 0xc0, 0x62, 0x1a, 0xc9, 0x04, 0xb8, 0x22,
			0xba, 0x48, 0x15, 0xd8, 0xf8, 0x3a, 0xc8, 0xf0,
			0xfb, 0x0f, 0x8f, 0xc1, 0x1b, 0xd3, 0x3b, 0x02,
			0xaf, 0xf4, 0xe4, 0x06, 0xf8, 0xfd, 0xa5, 0xef,
			0xab, 0xf3, 0x9e, 0x66, 0x41, 0xa7, 0x91, 0xcf,
			0x82, 0x41, 0xb0, 0x94, 0x6b, 0x67, 0x5f, 0xa4,
			0x8d, 0x07, 0xe4, 0x86, 0x39, 0xcc, 0x1e, 0xcf,
			0x42, 0x03, 0x80, 0xb8, 0x58, 0x1a, 0x53, 0x9a,
			0x4d, 0xe6, 0x0a, 0xdb, 0x0d, 0xa2, 0x2e, 0x10,
			0xad, 0x41, 0xf8, 0xba, 0x6a, 0xf4, 0x0d, 0x11,
			0xe2, 0x72, 0x00, 0x86, 0xa6, 0x3d, 0xb7, 0x2a,
			0x5d, 0x7f, 0xbe, 0x97, 0x92, 0x9a, 0xb2, 0x3c,
			0xae, 0x1d, 0x75, 0xc4, 0x85, 0xd6, 0x14, 0xca,
			0x38, 0x09, 0x4b, 0xac, 0xa6, 0x99, 0xe4, 0x72,
			0x00, 0xf7, 0xa7, 0x92, 0x29, 0x2b, 0x5c, 0x7a,
			0xb9, 0x5b, 0x96, 0x0d, 0x69, 0x21, 0xf8, 0xbe,
			0xab, 0x94, 0xd2, 0x6f, 0x96, 0x29, 0xd8, 0x70,
			0x2c, 0x40, 0xdf, 0x69, 0x67, 0x87, 0xa6, 0xfb,
			0x6a, 0xb9, 0xd6, 0xf3, 0xc1, 0x24, 0x0c, 0x2f,
			0xe5, 0x8c, 0x56, 0x5c, 0x93, 0x28, 0xdc, 0xab,
			0x60, 0x38, 0x97, 0x69, 0x3d, 0x9d, 0xc7, 0xdc,
			0xda, 0xf5, 0x00, 0x85, 0x07, 0x11, 0xe6, 0xf3,
			0x0b, 0x5d, 0x84, 0x98, 0xa3, 0x8e, 0x34, 0x84,
			0x69, 0xdf, 0x79, 0xc3, 0x62, 0x8f, 0xe1, 0x40,
			0x3a, 0x76, 0x49, 0xe8, 0x2f, 0x06, 0x16, 0x1e,
			0x0e, 0xce, 0x42, 0x47, 0x9a, 0x56, 0xea, 0xa8,
			0x45, 0xf0, 0x58, 0x2c, 0xbf, 0x81, 0x7d, 0x4b,
			0xa7, 0xdc, 0xed, 0x36, 0xe9, 0x3a, 0x6d, 0xc7,
			0xdc, 0x73, 0x62, 0xf6, 0x58, 0xf0, 0x64, 0x61,
		},
		28, /* salt length */
		/* V8 : 28 byte */
		{
			0x3f, 0x80, 0x50, 0x57, 0x47, 0x1a, 0xab, 0x0a,
			0x28, 0xcf, 0xc8, 0x43, 0x0d, 0xab, 0xcf, 0x99,
			0x06, 0x12, 0xe8, 0xa9, 0x08, 0xb1, 0x58, 0xae,
			0x36, 0xb4, 0xed, 0x53, 
		},
	},
	/* SHA224 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0x02, 0xa1, 0xa6, 0x5f, 0x8a, 0xf9, 0x0a, 0x29,
			0x86, 0x36, 0xfe, 0x8f, 0xd3, 0x11, 0x64, 0xb6,
			0x90, 0x7d, 0x74, 0xc8, 0xd3, 0x8a, 0x0e, 0xf5,
			0x9a, 0x8a, 0x4e, 0xb8, 0x05, 0x72, 0x62, 0x5c,
			0xc2, 0x83, 0x98, 0xbe, 0xc8, 0x29, 0xbb, 0x54,
			0x48, 0x23, 0xa0, 0x6e, 0xe0, 0xe4, 0xfc, 0xbc,
			0x13, 0x39, 0x78, 0x11, 0xf6, 0x2d, 0x08, 0x66,
			0x2b, 0x2a, 0x78, 0x22, 0x13, 0x60, 0x48, 0x99,
			0x40, 0x6a, 0xb9, 0xd2, 0x29, 0x2f, 0x28, 0x8d,
			0x22, 0x07, 0x9b, 0x84, 0x8b, 0x20, 0x9a, 0xf2,
			0x47, 0x1f, 0x40, 0x52, 0x70, 0x0a, 0x91, 0x69,
			0x48, 0x65, 0x0e, 0x86, 0x73, 0x9b, 0x87, 0x09,
			0x64, 0xa0, 0x31, 0x22, 0x16, 0xd5, 0xf8, 0xdb,
			0xfc, 0x2c, 0x16, 0x59, 0x3a, 0x8c, 0xe5, 0x5e,
			0x15, 0x77, 0xf1, 0x13, 0xa8, 0xea, 0x52, 0x05,
			0xd9, 0x84, 0x39, 0x6d, 0x8c, 0xeb, 0xc8, 0xb4,
		},
		/* S9 : 384 byte */
		{
			0x74, 0x0e, 0xeb, 0x1c, 0x71, 0x94, 0x0c, 0xcb,
			0xc0, 0x41, 0xcf, 0x20, 0x44, 0x69, 0xbd, 0x2d,
			0x6a, 0x46, 0x15, 0x58, 0xb1, 0xd1, 0x5c, 0x9e,
			0xb2, 0x33, 0x61, 0xcd, 0x55, 0xe1, 0xad, 0x41,
			0x8a, 0x7d, 0x28, 0x51, 0xed, 0x3d, 0x44, 0xf9,
			0xc0, 0x28, 0x81, 0xa2, 0x2f, 0x9e, 0x4b, 0xe0,
			0x42, 0xd4, 0x51, 0x99, 0x8b, 0xc1, 0x81, 0x88,
			0x79, 0x50, 0xda, 0x38, 0x24, 0x6d, 0xc1, 0x65,
			0x62, 0x43, 0xdb, 0x15, 0xfe, 0xf3, 0x59, 0xfe,
			0x50, 0xd2, 0xaf, 0x87, 0x11, 0xb3, 0x97, 0x3a,
			0x57, 0x76, 0x3b, 0xfc, 0x39, 0x64, 0xcf, 0xe3,
			0xc9, 0x11, 0xb9, 0x37, 0x57, 0x2e, 0x63, 0x9a,
			0xee, 0x53, 0xa9, 0x87, 0x52, 0x59, 0x8c, 0x4b,
			0x15, 0xdd, 0x53, 0xdd, 0x93, 0x55, 0xae, 0xe8,
			0x66, 0xd5, 0xf1, 0xe4, 0x81, 0x37, 0xc1, 0x2c,
			0x34, 0x2e, 0x8f, 0x27, 0x46, 0x90, 0xb7, 0xb2,
			0x77, 0xac, 0xd0, 0x87, 0xf2, 0x93, 0xcb, 0x8b,
			0x8c, 0x9a, 0x3e, 0x4b, 0x3f, 0x02, 0x77, 0xe8,
			0x31, 0xa6, 0x86, 0x4e, 0x50, 0x3f, 0x92, 0x55,
			0x57, 0x51, 0x1e, 0x57, 0xb5, 0x28, 0x52, 0x21,
			0x42, 0x18, 0x79, 0x69, 0x68, 0x02, 0x06, 0x65,
			0x87, 0xce, 0x6f, 0x99, 0x3a, 0xac, 0xb7, 0x0d,
			0xaf, 0xd3, 0x9f, 0x63, 0xf0, 0x9c, 0xb3, 0xdc,
			0xc2, 0x8e, 0x56, 0x78, 0x2d, 0xbf, 0xb8, 0xb4,
			0xcc, 0xb1, 0xb1, 0x98, 0x76, 0x10, 0x15, 0x73,
			0xee, 0x96, 0x78, 0xa5, 0xf6, 0x26, 0x5f, 0x80,
			0x8f, 0x75, 0xe7, 0x71, 0x19, 0x46, 0xc2, 0x7c,
			0x7a, 0x22, 0xdc, 0xe9, 0xf5, 0x92, 0xac, 0xdd,
			0xac, 0x81, 0xc6, 0x7a, 0xfa, 0x17, 0xbf, 0xfb,
			0x76, 0x60, 0x58, 0xe2, 0x31, 0x8a, 0x12, 0x11,
			0x07, 0x98, 0x42, 0xbd, 0x5f, 0xc5, 0x8f, 0x9c,
			0xef, 0x4b, 0x50, 0xff, 0x0e, 0xe1, 0xa2, 0x93,
			0xf8, 0x0a, 0xc1, 0xbf, 0x2e, 0xb6, 0x4c, 0xe4,
			0xe1, 0x05, 0x1e, 0x1a, 0xbe, 0x55, 0xee, 0x06,
			0x7d, 0xb6, 0xc2, 0x41, 0x30, 0xf0, 0xbf, 0x4c,
			0x13, 0x4b, 0x0a, 0xbf, 0x1e, 0x2f, 0x44, 0x65,
			0xdc, 0x50, 0xfd, 0x37, 0x99, 0xf6, 0xdc, 0x20,
			0x6b, 0x9a, 0x7d, 0x2f, 0xe3, 0x4b, 0x4f, 0x42,
			0x57, 0x06, 0x5d, 0x74, 0x94, 0xae, 0x73, 0x3c,
			0x28, 0xd7, 0x0a, 0xad, 0xb0, 0x57, 0xce, 0x1b,
			0xcf, 0xf3, 0x6e, 0xdf, 0x9f, 0x9c, 0xa6, 0x90,
			0x8c, 0xac, 0x21, 0x41, 0x84, 0x53, 0x10, 0x66,
			0x0a, 0xb7, 0x59, 0xd1, 0xf3, 0xe6, 0x51, 0xdd,
			0x9f, 0xa8, 0x05, 0x6a, 0x62, 0x4e, 0xfc, 0x71,
			0x4f, 0x51, 0xf3, 0xa4, 0xf8, 0x5a, 0xdc, 0xba,
			0x68, 0xf4, 0xa5, 0x8e, 0x3a, 0x95, 0x6a, 0xf9,
			0x3a, 0x5a, 0x52, 0xf2, 0xb8, 0x9f, 0x9c, 0x91,
			0x4b, 0x48, 0xe8, 0xdf, 0xb9, 0x19, 0xcf, 0xc6,
		},
		28, /* salt length */
		/* V9 : 28 byte */
		{
			0x3f, 0x80, 0x50, 0x57, 0x47, 0x1a, 0xab, 0x0a,
			0x28, 0xcf, 0xc8, 0x43, 0x0d, 0xab, 0xcf, 0x99,
			0x06, 0x12, 0xe8, 0xa9, 0x08, 0xb1, 0x58, 0xae,
			0x36, 0xb4, 0xed, 0x53, 
		},
	},
};

/* 10 elements */
static mod3072_t SHA256[] = {
	/* SHA256 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0xc1, 0x64, 0x99, 0x11, 0x0e, 0xd5, 0x77, 0x20,
			0x2a, 0xed, 0x2d, 0x3e, 0x4d, 0x51, 0xde, 0xd6,
			0xc6, 0x63, 0x73, 0xfa, 0xef, 0x65, 0x33, 0xa8,
			0x60, 0xe1, 0x93, 0x4c, 0x63, 0x48, 0x4f, 0x87,
			0xa8, 0xd9, 0xb9, 0x2f, 0x3a, 0xc4, 0x51, 0x97,
			0xb2, 0x90, 0x97, 0x10, 0xab, 0xba, 0x1d, 0xaf,
			0x75, 0x9f, 0xe0, 0x51, 0x0e, 0x9b, 0xd8, 0xdd,
			0x4d, 0x73, 0xce, 0xc9, 0x61, 0xf0, 0x6e, 0xe0,
			0x7a, 0xcd, 0x9d, 0x42, 0xc6, 0xd4, 0x0d, 0xac,
			0x9f, 0x43, 0x0e, 0xf9, 0x03, 0x74, 0xa7, 0xe9,
			0x44, 0xbd, 0xe5, 0x22, 0x00, 0x96, 0x73, 0x74,
			0x54, 0xf9, 0x6b, 0x61, 0x4d, 0x0f, 0x6c, 0xdd,
			0x9f, 0x08, 0xed, 0x52, 0x9a, 0x4a, 0xd0, 0xe7,
			0x59, 0xcf, 0x3a, 0x02, 0x3d, 0xc8, 0xa3, 0x0b,
			0x9a, 0x87, 0x29, 0x74, 0xaf, 0x9b, 0x2a, 0xf6,
			0xdc, 0x3d, 0x11, 0x1d, 0x0f, 0xeb, 0x70, 0x06,
		},
		/* S0 : 384 byte */
		{
			0x43, 0x35, 0x70, 0x7d, 0xa7, 0x35, 0xcf, 0xd1,
			0x04, 0x11, 0xc9, 0xc0, 0x48, 0xca, 0x9b, 0x60,
			0xbb, 0x46, 0xe2, 0xfe, 0x36, 0x1e, 0x51, 0xfb,
			0xe3, 0x36, 0xf9, 0x50, 0x8d, 0xc9, 0x45, 0xaf,
			0xe0, 0x75, 0x50, 0x3d, 0x24, 0xf8, 0x36, 0x61,
			0x0f, 0x21, 0x78, 0x99, 0x6b, 0x52, 0xc4, 0x11,
			0x69, 0x30, 0x52, 0xd5, 0xd7, 0xae, 0xd9, 0x76,
			0x54, 0xa4, 0x00, 0x74, 0xed, 0x20, 0xed, 0x66,
			0x89, 0xc0, 0x50, 0x1b, 0x7f, 0xba, 0xc2, 0x1d,
			0xc4, 0x6b, 0x66, 0x5a, 0xc0, 0x79, 0x76, 0x00,
			0x86, 0x41, 0x44, 0x06, 0xcd, 0x66, 0xf8, 0x53,
			0x7d, 0x1e, 0xbf, 0x0d, 0xce, 0x4c, 0xf0, 0xc9,
			0x8d, 0x4c, 0x30, 0xc7, 0x1d, 0xa3, 0x59, 0xe9,
			0xcd, 0x40, 0x1f, 0xf4, 0x97, 0x18, 0xfd, 0xd4,
			0xd0, 0xf9, 0x9e, 0xfe, 0x70, 0xad, 0x8d, 0xd8,
			0xba, 0x13, 0x04, 0xce, 0xfb, 0x88, 0xf2, 0x4b,
			0x0e, 0xed, 0xf7, 0x01, 0x16, 0xda, 0x15, 0x93,
			0x2c, 0x76, 0xf0, 0x06, 0x95, 0x51, 0xa2, 0x45,
			0xb5, 0xfc, 0x3b, 0x91, 0xec, 0x10, 0x1f, 0x1d,
			0x63, 0xb9, 0x85, 0x3b, 0x59, 0x8c, 0x6f, 0xa1,
			0xc1, 0xac, 0xdb, 0xac, 0xf9, 0x62, 0x63, 0x56,
			0xc7, 0x60, 0x11, 0x9b, 0xe0, 0x95, 0x56, 0x44,
			0x30, 0x18, 0x96, 0xd9, 0xd0, 0xd3, 0xea, 0x5e,
			0x64, 0x43, 0xcb, 0x72, 0xca, 0x29, 0xf4, 0xd4,
			0x52, 0x46, 0xd1, 0x6d, 0x74, 0xd0, 0x05, 0x68,
			0xc2, 0x19, 0x18, 0x2f, 0xeb, 0x19, 0x11, 0x79,
			0xe4, 0x59, 0x3d, 0xc1, 0x52, 0xc6, 0x08, 0xfd,
			0x80, 0x53, 0x63, 0x29, 0xa5, 0x33, 0xb3, 0xa6,
			0x31, 0x56, 0x68, 0x14, 0xcd, 0x65, 0x4f, 0x58,
			0x7c, 0x2d, 0x8c, 0xe6, 0x96, 0x08, 0x5e, 0x6e,
			0xd1, 0xb0, 0xb0, 0x27, 0x8e, 0x60, 0xa0, 0x49,
			0xec, 0x7a, 0x39, 0x9f, 0x94, 0xfc, 0xca, 0xe6,
			0x46, 0x23, 0x71, 0xa6, 0x96, 0x95, 0xef, 0x52,
			0x5e, 0x00, 0x93, 0x6f, 0xa7, 0xd9, 0x78, 0x1f,
			0x9e, 0xe2, 0x89, 0xd4, 0x10, 0x5e, 0xe8, 0x27,
			0xa2, 0x79, 0x96, 0x58, 0x30, 0x33, 0xce, 0xdb,
			0x2f, 0x29, 0x7e, 0x7b, 0x49, 0x26, 0xd9, 0x06,
			0xce, 0x0d, 0x09, 0xd8, 0x41, 0x28, 0x40, 0x6a,
			0xb3, 0x3d, 0x7d, 0xa0, 0xf8, 0xa1, 0xd4, 0xd2,
			0xf6, 0x66, 0x56, 0x86, 0x86, 0xc3, 0x94, 0xd1,
			0x39, 0xb0, 0xe5, 0xe9, 0x93, 0x37, 0x75, 0x8d,
			0xe8, 0x59, 0x10, 0xa5, 0xfa, 0x25, 0xca, 0x2a,
			0xa6, 0xd8, 0xfb, 0x1c, 0x77, 0x72, 0x44, 0xe7,
			0xd9, 0x8d, 0xe4, 0xc7, 0x9b, 0xbd, 0x42, 0x6a,
			0x5e, 0x6f, 0x65, 0x7e, 0x37, 0x47, 0x7e, 0x01,
			0x24, 0x74, 0x32, 0xf8, 0x37, 0x97, 0xfb, 0xf3,
			0x1b, 0x50, 0xd0, 0x2b, 0x83, 0xf6, 0x9d, 0xed,
			0x26, 0xd4, 0x94, 0x5b, 0x2b, 0xc3, 0xf8, 0x6e,
		},
		32, /* salt length */
		/* V0 : 32 byte */
		{
			0x3e, 0x07, 0xad, 0xe7, 0x2a, 0x3f, 0x52, 0x53,
			0x0f, 0x53, 0x13, 0x5a, 0x5d, 0x7d, 0x93, 0x21,
			0x74, 0x35, 0xba, 0x00, 0x1e, 0xa5, 0x5a, 0x8f,
			0x5d, 0x5d, 0x13, 0x04, 0x68, 0x48, 0x74, 0xbc,
		},
	},
	/* SHA256 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0x60, 0x40, 0x2d, 0xed, 0x89, 0xd0, 0x97, 0x9a,
			0xfb, 0x49, 0xf8, 0x50, 0x8e, 0xb9, 0x78, 0xa8,
			0x41, 0xab, 0xc2, 0xae, 0xc5, 0x9c, 0xac, 0xef,
			0x40, 0xb3, 0x1a, 0xd3, 0x4b, 0xac, 0x1f, 0x2d,
			0x3c, 0x16, 0x66, 0x11, 0xab, 0xbe, 0xd1, 0xe6,
			0x2f, 0x6b, 0x5f, 0xbb, 0x69, 0xcb, 0x53, 0xdf,
			0x44, 0xae, 0x93, 0xab, 0x7a, 0x72, 0x4e, 0xa3,
			0x5b, 0xbe, 0xe1, 0xbe, 0xca, 0x74, 0xfc, 0x01,
			0x88, 0xe0, 0x00, 0x52, 0xb5, 0x36, 0xac, 0x8c,
			0x93, 0x3b, 0xf9, 0xcf, 0x8e, 0x42, 0x42, 0x1a,
			0x79, 0x5a, 0xa8, 0x1b, 0x1b, 0xc6, 0xb5, 0x45,
			0xea, 0xad, 0x40, 0x24, 0x16, 0x13, 0x90, 0xed,
			0xc9, 0x08, 0xc4, 0x5a, 0xae, 0x1f, 0x71, 0xb4,
			0xb0, 0x22, 0x8e, 0x31, 0x04, 0x04, 0x8d, 0x81,
			0x69, 0x17, 0xcb, 0xa4, 0xae, 0x7f, 0x2a, 0xfe,
			0x75, 0xe7, 0xfc, 0xad, 0x38, 0x73, 0x24, 0x1a,
		},
		/* S1 : 384 byte */
		{
			0x5f, 0x18, 0x30, 0x09, 0x70, 0x8b, 0x37, 0x96,
			0x37, 0xda, 0xc2, 0xb1, 0x42, 0x93, 0x70, 0x9a,
			0xa6, 0xd7, 0xe8, 0x6c, 0x26, 0x7a, 0x0b, 0x69,
			0x0a, 0x3c, 0x27, 0x50, 0x31, 0x13, 0x98, 0x91,
			0x26, 0x7c, 0x64, 0xe5, 0xed, 0xec, 0xdf, 0xf1,
			0x4c, 0x2c, 0xc2, 0xf2, 0xd9, 0x85, 0xb6, 0x2f,
			0x90, 0x0a, 0xee, 0x6e, 0x04, 0xca, 0x51, 0xa7,
			0x0a, 0x5f, 0x94, 0x64, 0x63, 0x69, 0x1c, 0xf1,
			0x6c, 0x2d, 0x45, 0x54, 0x7c, 0x53, 0x74, 0xf1,
			0x5b, 0xdb, 0x88, 0x81, 0x64, 0x1d, 0x30, 0x40,
			0xef, 0x57, 0x80, 0x75, 0x32, 0xcf, 0x5b, 0x2c,
			0xed, 0x07, 0x62, 0x3d, 0x0f, 0x63, 0x8b, 0x39,
			0xeb, 0xc2, 0xf2, 0xce, 0x28, 0x3e, 0xea, 0x22,
			0x47, 0xe1, 0xdf, 0x3a, 0xf5, 0x43, 0x05, 0x54,
			0xd1, 0xd4, 0xb8, 0x8b, 0x7b, 0x21, 0x62, 0x29,
			0x93, 0x41, 0x99, 0x71, 0xb7, 0xd0, 0xd5, 0x44,
			0x91, 0x22, 0xa1, 0x0f, 0xc3, 0x1b, 0x2d, 0xdc,
			0xc5, 0x3f, 0xf7, 0x51, 0xff, 0x4b, 0xf4, 0xd3,
			0x36, 0xfa, 0xc6, 0x67, 0xb6, 0x46, 0x78, 0x02,
			0x72, 0xdb, 0x89, 0xa3, 0xea, 0x42, 0x26, 0xaf,
			0xa2, 0x08, 0x77, 0xbf, 0xb8, 0x6b, 0xa3, 0xff,
			0x42, 0x04, 0xe5, 0xcd, 0x56, 0xe1, 0x3a, 0x1d,
			0xc9, 0xd5, 0x3f, 0x5c, 0x94, 0x65, 0xb9, 0x7a,
			0x18, 0x2b, 0x2b, 0xf6, 0x71, 0x51, 0x2e, 0xf8,
			0x9e, 0x6c, 0x39, 0x69, 0xf9, 0x73, 0x07, 0xa3,
			0xe4, 0xbe, 0xba, 0x39, 0xa7, 0x8e, 0x0a, 0xd1,
			0xbb, 0x97, 0x99, 0xcd, 0xa9, 0x29, 0x76, 0xca,
			0x39, 0xd9, 0x9d, 0xb4, 0xac, 0x14, 0x9c, 0x84,
			0xbb, 0x9b, 0xc8, 0x99, 0x7e, 0x8d, 0x5e, 0x05,
			0x6d, 0x67, 0xca, 0x23, 0xfe, 0x4b, 0xe2, 0x8e,
			0x66, 0xc4, 0xbc, 0x00, 0xa2, 0x5d, 0x65, 0xbb,
			0x9d, 0x7d, 0x62, 0x3f, 0xea, 0x2d, 0x3b, 0x9c,
			0xf8, 0x59, 0xdf, 0xd9, 0xef, 0xa9, 0xe5, 0x22,
			0x68, 0xbf, 0xa2, 0x97, 0xaf, 0xb1, 0xcc, 0x28,
			0x83, 0xdb, 0x0c, 0x9c, 0x42, 0xfc, 0x04, 0x18,
			0x0e, 0x2e, 0xc6, 0xf4, 0x96, 0x57, 0xc7, 0x00,
			0x8e, 0x40, 0x25, 0x06, 0x1f, 0x89, 0x68, 0x86,
			0x61, 0x38, 0x95, 0xa3, 0x5b, 0xc2, 0xd3, 0x65,
			0x5a, 0x8f, 0x50, 0xa9, 0xfc, 0xa2, 0xac, 0x64,
			0x8f, 0x35, 0x2e, 0xb0, 0x6b, 0xfb, 0xa2, 0xfc,
			0x34, 0x0a, 0xae, 0xea, 0xd4, 0xa8, 0x45, 0x7c,
			0x65, 0xe2, 0xe8, 0xfd, 0xba, 0x56, 0x8c, 0x60,
			0xa6, 0xd8, 0xd3, 0x81, 0xf5, 0xd9, 0xca, 0xa3,
			0x01, 0x27, 0x77, 0x1f, 0x4a, 0x94, 0xfd, 0xb8,
			0xcd, 0xe7, 0xbe, 0x4f, 0xa7, 0xb4, 0xf8, 0x9f,
			0xe3, 0x79, 0xdd, 0x3e, 0x1c, 0xa6, 0x6a, 0xe1,
			0xfd, 0xd6, 0x3b, 0xeb, 0xdc, 0x00, 0x15, 0x44,
			0x8e, 0x61, 0xef, 0x16, 0x66, 0x59, 0x4b, 0x8f,
		},
		32, /* salt length */
		/* V1 : 32 byte */
		{
			0x3e, 0x07, 0xad, 0xe7, 0x2a, 0x3f, 0x52, 0x53,
			0x0f, 0x53, 0x13, 0x5a, 0x5d, 0x7d, 0x93, 0x21,
			0x74, 0x35, 0xba, 0x00, 0x1e, 0xa5, 0x5a, 0x8f,
			0x5d, 0x5d, 0x13, 0x04, 0x68, 0x48, 0x74, 0xbc,
		},
	},
	/* SHA256 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x2f, 0x03, 0x70, 0x1c, 0x2f, 0xe0, 0x7d, 0x47,
			0xf5, 0xfa, 0x2c, 0x83, 0xa8, 0xea, 0x82, 0x4f,
			0x1d, 0x42, 0x9c, 0xe4, 0xfa, 0x1d, 0xf2, 0x67,
			0x1b, 0xfa, 0xdd, 0x62, 0x34, 0xca, 0x57, 0x75,
			0xb8, 0x47, 0x02, 0x49, 0xfa, 0x88, 0x6d, 0xc6,
			0x93, 0xd2, 0x92, 0x86, 0x03, 0xb2, 0xa3, 0x89,
			0x9b, 0x48, 0x06, 0x2a, 0x9a, 0xe6, 0x9e, 0x51,
			0x96, 0xda, 0x4c, 0xeb, 0x1d, 0x87, 0xb5, 0x97,
			0x9d, 0xbb, 0x46, 0xa2, 0x81, 0x3c, 0x76, 0x36,
			0x9d, 0xa4, 0x4b, 0xce, 0xcc, 0x6f, 0x20, 0xed,
			0xd7, 0x53, 0xa5, 0x10, 0x99, 0xd0, 0x27, 0xe1,
			0x61, 0x07, 0x12, 0xad, 0x98, 0xcf, 0xb4, 0x18,
			0xa4, 0x06, 0x43, 0x10, 0x0b, 0x25, 0x22, 0xff,
			0xdc, 0x17, 0x60, 0x45, 0x4b, 0x4c, 0x82, 0xe5,
			0x9b, 0x09, 0x82, 0x7e, 0x41, 0x02, 0x17, 0x7e,
			0x46, 0x2a, 0x37, 0x92, 0xed, 0xca, 0xda, 0x61,
		},
		/* S2 : 384 byte */
		{
			0x82, 0x91, 0xbc, 0x1b, 0xe9, 0xc9, 0x81, 0x66,
			0x31, 0x56, 0xec, 0x80, 0xc1, 0xed, 0x16, 0x75,
			0x76, 0x3d, 0xe0, 0x61, 0x99, 0xb9, 0xf2, 0x76,
			0x0c, 0xaa, 0xed, 0x52, 0x07, 0xfb, 0x4b, 0x3d,
			0x60, 0x37, 0xbd, 0x08, 0x46, 0x2b, 0x10, 0x0b,
			0xb1, 0x76, 0x7e, 0x33, 0x40, 0x10, 0x5b, 0x1a,
			0x68, 0x72, 0x8b, 0xc4, 0x5c, 0x7d, 0x6f, 0xd0,
			0x78, 0xdc, 0x1b, 0x5e, 0x7c, 0xbf, 0xa1, 0x93,
			0x00, 0x6d, 0x52, 0xf6, 0x7e, 0x77, 0xfc, 0xf8,
			0x09, 0xcf, 0x26, 0x17, 0x2a, 0x46, 0xdb, 0x38,
			0x4e, 0xaf, 0x55, 0x2a, 0x5f, 0xb8, 0xe3, 0x38,
			0x40, 0xfa, 0x3e, 0xf3, 0xd6, 0xb2, 0x0c, 0x7b,
			0x46, 0xc3, 0x2e, 0xf0, 0x19, 0xe8, 0xd1, 0x5d,
			0xd3, 0x8e, 0xab, 0x66, 0xf6, 0xe4, 0x03, 0x99,
			0xad, 0x0b, 0xbb, 0x07, 0xf9, 0x4b, 0x8c, 0x55,
			0x51, 0x96, 0x90, 0x1c, 0x27, 0xe2, 0xd4, 0x57,
			0x39, 0x58, 0xf5, 0x30, 0x60, 0xd8, 0x00, 0xcf,
			0xff, 0x40, 0xc6, 0x02, 0x30, 0x80, 0x44, 0xb7,
			0x5d, 0x64, 0x51, 0x80, 0x1c, 0x68, 0x8d, 0x27,
			0x65, 0x25, 0xc3, 0xfe, 0xe1, 0x7a, 0x67, 0x92,
			0x88, 0x2a, 0x07, 0x4c, 0x8a, 0x41, 0x42, 0x01,
			0x09, 0xe2, 0x51, 0x14, 0x18, 0xc9, 0xee, 0xaf,
			0x3a, 0xb4, 0x73, 0x50, 0xdd, 0x8c, 0x2d, 0x3e,
			0x06, 0x6a, 0xbe, 0xb7, 0x91, 0x3e, 0x08, 0xf0,
			0xa4, 0x0a, 0xbe, 0x71, 0xd3, 0x97, 0xc3, 0xdd,
			0xda, 0xfc, 0x41, 0xfb, 0xd0, 0x4c, 0xc8, 0xfa,
			0x3b, 0x06, 0x41, 0xbf, 0x53, 0xa9, 0x00, 0x31,
			0xb6, 0x1a, 0x2a, 0x9b, 0x63, 0xd8, 0xed, 0x8a,
			0xac, 0xc9, 0xb3, 0x01, 0x59, 0x3c, 0x9f, 0x42,
			0x51, 0x05, 0x49, 0x8c, 0xc4, 0xf8, 0x46, 0x27,
			0xf4, 0x95, 0x07, 0x58, 0xe0, 0x1a, 0x29, 0x1b,
			0x9b, 0x1a, 0x33, 0xba, 0x91, 0x8a, 0xac, 0xc1,
			0x72, 0xb6, 0x8c, 0x9f, 0xb2, 0xc7, 0x67, 0xc6,
			0x59, 0x10, 0x81, 0x69, 0x21, 0x28, 0x1a, 0xa8,
			0xe5, 0x48, 0x25, 0x12, 0xce, 0xe6, 0x86, 0xe5,
			0x1c, 0xab, 0xe8, 0x8e, 0x18, 0xf9, 0x23, 0xfd,
			0xe1, 0x70, 0xa5, 0x06, 0xba, 0x3c, 0x34, 0x0f,
			0xd1, 0xd6, 0x82, 0x61, 0x98, 0x63, 0x47, 0xd3,
			0x0d, 0x12, 0x49, 0x31, 0xdb, 0x2c, 0xe1, 0x76,
			0x02, 0x15, 0x00, 0x00, 0xb7, 0x94, 0xc0, 0x50,
			0xe1, 0x37, 0xf4, 0xeb, 0xd4, 0x5c, 0xc4, 0x1f,
			0x70, 0xef, 0x3d, 0xf1, 0x65, 0x62, 0x18, 0xff,
			0x76, 0xf2, 0xe7, 0x5a, 0xd9, 0x6e, 0x41, 0x67,
			0xee, 0xd5, 0x24, 0xfa, 0x2e, 0xd9, 0xfd, 0x1a,
			0x0c, 0xf7, 0x69, 0x26, 0xf3, 0x82, 0xff, 0xb1,
			0x61, 0x24, 0xdf, 0xc8, 0x7b, 0xb1, 0xa4, 0x11,
			0x09, 0x28, 0xd5, 0xb1, 0xcd, 0x3b, 0x16, 0x20,
			0x4c, 0xee, 0xec, 0xcb, 0x7d, 0xb8, 0x8f, 0xce,
		},
		32, /* salt length */
		/* V2 : 32 byte */
		{
			0x3e, 0x07, 0xad, 0xe7, 0x2a, 0x3f, 0x52, 0x53,
			0x0f, 0x53, 0x13, 0x5a, 0x5d, 0x7d, 0x93, 0x21,
			0x74, 0x35, 0xba, 0x00, 0x1e, 0xa5, 0x5a, 0x8f,
			0x5d, 0x5d, 0x13, 0x04, 0x68, 0x48, 0x74, 0xbc,
		},
	},
	/* SHA256 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0xaf, 0x90, 0xf1, 0x31, 0xf9, 0xfc, 0x13, 0xdb,
			0x0b, 0xce, 0xbf, 0xae, 0x4a, 0x2e, 0x90, 0xad,
			0x39, 0xdc, 0x53, 0x3f, 0x34, 0x16, 0x5e, 0x32,
			0x62, 0xbc, 0x23, 0xff, 0xe5, 0xb2, 0x04, 0x50,
			0x53, 0x86, 0x69, 0xbf, 0x6a, 0x52, 0x10, 0xe1,
			0xff, 0xe4, 0xa5, 0x83, 0x38, 0x1d, 0x93, 0x33,
			0xfb, 0x97, 0x19, 0x03, 0xa6, 0x8a, 0xa0, 0x89,
			0x01, 0xf1, 0x4c, 0x2a, 0x71, 0xe8, 0xd1, 0x99,
			0x6e, 0x59, 0x88, 0x9a, 0x36, 0xd7, 0xc2, 0x0c,
			0xc3, 0xca, 0x5c, 0x26, 0xfb, 0xcd, 0x93, 0x01,
			0x28, 0x54, 0x1a, 0x56, 0xa7, 0x92, 0x6a, 0x8a,
			0xe4, 0x9a, 0x5a, 0xe7, 0x86, 0xc4, 0xef, 0x2d,
			0xe6, 0x52, 0x75, 0x49, 0xc6, 0x53, 0xce, 0x64,
			0x40, 0xc8, 0x0b, 0x1f, 0xfc, 0x06, 0x39, 0x1d,
			0xa6, 0x5b, 0x7d, 0xc3, 0x9f, 0xf4, 0x64, 0x3b,
			0xf3, 0xfe, 0x74, 0xbf, 0x8c, 0x0c, 0x07, 0x14,
		},
		/* S3 : 384 byte */
		{
			0x8c, 0x45, 0xe3, 0x8e, 0xaf, 0xaa, 0xf1, 0x0a,
			0x71, 0x0e, 0x13, 0x1b, 0xec, 0x63, 0xe5, 0x1e,
			0x67, 0x74, 0x17, 0x74, 0xa9, 0xdd, 0xbf, 0xcc,
			0xdd, 0x13, 0x1a, 0x12, 0x3a, 0xe2, 0xa0, 0x30,
			0x67, 0xe7, 0xa6, 0xa9, 0x2e, 0x65, 0x3a, 0x25,
			0x17, 0x8b, 0xf5, 0x27, 0xb9, 0x3d, 0x6a, 0xa8,
			0x3f, 0xa3, 0x66, 0xa2, 0xbd, 0x44, 0x89, 0x6b,
			0xaa, 0x8b, 0x7f, 0x3f, 0x54, 0x83, 0x0e, 0x4d,
			0x9f, 0x56, 0x32, 0xc2, 0xd1, 0xbc, 0xae, 0x2a,
			0xaa, 0xe8, 0xc5, 0x57, 0x82, 0x13, 0x2a, 0xa7,
			0x27, 0x9c, 0xf1, 0xcb, 0xb6, 0xb7, 0xa8, 0x1e,
			0x49, 0x65, 0xff, 0x84, 0x63, 0x5c, 0x29, 0x6c,
			0x5a, 0xc2, 0x06, 0xa0, 0x46, 0x80, 0xe9, 0x1e,
			0x7b, 0x1e, 0xe7, 0xe5, 0x79, 0x37, 0x01, 0xb1,
			0xfe, 0xb8, 0x32, 0x25, 0x00, 0x10, 0xd4, 0xad,
			0x40, 0x17, 0xc1, 0x60, 0x8d, 0xe8, 0xf4, 0x05,
			0x01, 0x4c, 0xa7, 0x3c, 0x39, 0xad, 0xae, 0x7c,
			0x4a, 0xdc, 0xba, 0xee, 0x35, 0xfb, 0xbc, 0x71,
			0x15, 0x1c, 0xf9, 0x55, 0xac, 0xec, 0xd8, 0x08,
			0x36, 0x77, 0xfe, 0x49, 0xec, 0xec, 0xcb, 0x62,
			0x35, 0x3c, 0x0a, 0x89, 0xc9, 0xdc, 0xb9, 0xc5,
			0x07, 0x97, 0x9b, 0x56, 0xbf, 0xe0, 0x60, 0xfe,
			0xc4, 0x55, 0x67, 0x51, 0x7c, 0x05, 0xf2, 0x9e,
			0x26, 0x2d, 0xf5, 0x07, 0x67, 0xdf, 0x75, 0x47,
			0x63, 0x0d, 0x8a, 0x7b, 0x32, 0x48, 0x3b, 0x92,
			0x3b, 0xb1, 0xe3, 0xd5, 0x10, 0x42, 0x2d, 0xd4,
			0xcc, 0x2d, 0x61, 0xa6, 0x47, 0xe4, 0xf9, 0x63,
			0x6a, 0xa7, 0x58, 0x7d, 0x4f, 0x8e, 0xd8, 0x4b,
			0x61, 0x74, 0xc1, 0xfd, 0xca, 0x9a, 0x21, 0x7d,
			0x9b, 0x90, 0x79, 0x72, 0xa6, 0x6c, 0x1f, 0x5a,
			0x2e, 0xc2, 0xda, 0xdb, 0x60, 0xb9, 0x3b, 0x51,
			0x5b, 0xf7, 0x40, 0x72, 0xd3, 0x15, 0xd1, 0x7d,
			0x54, 0xd5, 0x7d, 0x72, 0x1c, 0x8f, 0x4c, 0xe1,
			0xa4, 0x3e, 0xed, 0xf2, 0x02, 0x5e, 0x51, 0xa4,
			0x8e, 0x9e, 0xa2, 0x81, 0x60, 0xcf, 0x30, 0x0d,
			0x7a, 0x26, 0x01, 0x03, 0x83, 0xc3, 0x28, 0x0a,
			0x18, 0x6c, 0x44, 0xa5, 0x3b, 0x71, 0x88, 0xe6,
			0xca, 0xa3, 0x64, 0xbf, 0x4d, 0xbe, 0x0b, 0xaf,
			0x4d, 0xcb, 0xe3, 0x7d, 0x70, 0xe3, 0xa4, 0x75,
			0xcf, 0xda, 0xe3, 0x39, 0x38, 0x65, 0x58, 0xcc,
			0xbc, 0x11, 0x98, 0x73, 0xb1, 0x86, 0x39, 0x75,
			0xe2, 0x30, 0x0e, 0xde, 0x1e, 0x42, 0x00, 0x31,
			0xb4, 0xcd, 0xac, 0x56, 0x7e, 0x7b, 0x9c, 0x5d,
			0x57, 0x5c, 0x8b, 0xae, 0x27, 0xee, 0xbb, 0x37,
			0x09, 0x70, 0x50, 0xac, 0xdc, 0x87, 0x00, 0x8c,
			0xa2, 0x38, 0x0f, 0x56, 0x31, 0xd1, 0x90, 0x02,
			0x9a, 0x1d, 0x71, 0x2a, 0xcd, 0xa1, 0x47, 0xc5,
			0xc4, 0x37, 0x8c, 0xb6, 0xea, 0xc8, 0x17, 0x31,
		},
		32, /* salt length */
		/* V3 : 32 byte */
		{
			0x3e, 0x07, 0xad, 0xe7, 0x2a, 0x3f, 0x52, 0x53,
			0x0f, 0x53, 0x13, 0x5a, 0x5d, 0x7d, 0x93, 0x21,
			0x74, 0x35, 0xba, 0x00, 0x1e, 0xa5, 0x5a, 0x8f,
			0x5d, 0x5d, 0x13, 0x04, 0x68, 0x48, 0x74, 0xbc,
		},
	},
	/* SHA256 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0xe5, 0x7d, 0xeb, 0xad, 0x35, 0x63, 0xfa, 0x81,
			0xf4, 0xb9, 0x81, 0x94, 0x05, 0xe4, 0x1f, 0x98,
			0xa5, 0x40, 0x96, 0xd4, 0x4f, 0x6e, 0xd1, 0x19,
			0xdc, 0xeb, 0x25, 0xf8, 0xef, 0xe7, 0xd7, 0x32,
			0x90, 0x54, 0xde, 0x70, 0x17, 0x3d, 0xeb, 0x34,
			0x4c, 0x59, 0xa7, 0x10, 0xcc, 0xe0, 0x3b, 0x16,
			0xaf, 0x9d, 0x16, 0x8f, 0x67, 0x45, 0xea, 0xf0,
			0xeb, 0x07, 0xf8, 0x09, 0x16, 0x64, 0x8e, 0x80,
			0x49, 0x41, 0xce, 0x7e, 0x58, 0x3a, 0xb0, 0xa8,
			0xa4, 0x3a, 0x4b, 0x51, 0x84, 0x48, 0x50, 0xed,
			0xea, 0xa4, 0xd7, 0xc9, 0x43, 0x13, 0x5e, 0xfa,
			0x9e, 0x77, 0x0e, 0x94, 0x11, 0xa2, 0x41, 0x1c,
			0x58, 0x6c, 0x42, 0x3f, 0xc0, 0x03, 0x53, 0xc3,
			0x44, 0x83, 0xf5, 0xbf, 0xf5, 0xc7, 0x63, 0x07,
			0x9f, 0x7e, 0x60, 0xeb, 0xa9, 0x81, 0x32, 0x21,
			0x3d, 0x64, 0xef, 0xff, 0xa9, 0x4a, 0xf7, 0xed,
		},
		/* S4 : 384 byte */
		{
			0x85, 0x1d, 0xcd, 0x2d, 0x4e, 0x1d, 0x34, 0xda,
			0xe0, 0xfd, 0x58, 0x5a, 0xf1, 0x26, 0xbe, 0x44,
			0x8d, 0x61, 0x1a, 0xca, 0xea, 0xcf, 0xa3, 0x4f,
			0x14, 0x92, 0xaa, 0x7d, 0x1c, 0xaf, 0xf6, 0x16,
			0x70, 0x7d, 0xc3, 0x1b, 0x05, 0x18, 0x6c, 0xdb,
			0xef, 0x76, 0x94, 0x79, 0x24, 0x3a, 0xfb, 0x34,
			0x15, 0x77, 0x80, 0x3b, 0x57, 0x9e, 0x10, 0x50,
			0x70, 0xad, 0x54, 0x06, 0xa6, 0x74, 0x4f, 0x56,
			0xe5, 0x5f, 0x56, 0x93, 0x70, 0xb9, 0xfc, 0xf6,
			0xab, 0x10, 0xe1, 0xaa, 0x03, 0x83, 0xf9, 0x18,
			0x2d, 0x45, 0x1a, 0xfb, 0x41, 0x35, 0x8a, 0x2f,
			0x8c, 0x29, 0xd1, 0xa5, 0x71, 0xe1, 0x1c, 0x40,
			0x4e, 0x68, 0x70, 0xcb, 0xb0, 0x4f, 0x6e, 0xf3,
			0x04, 0x14, 0xd9, 0xb6, 0xd7, 0xf1, 0x41, 0x6b,
			0xac, 0xab, 0x01, 0x84, 0xee, 0xbd, 0x8d, 0xea,
			0xe7, 0x2f, 0x2a, 0x48, 0xbe, 0xa3, 0xa7, 0x84,
			0x4a, 0x8b, 0xf4, 0x72, 0xa5, 0xf8, 0xd3, 0x49,
			0xd5, 0x97, 0x3f, 0xfd, 0xe3, 0xb1, 0xc4, 0x06,
			0x23, 0xdb, 0xaa, 0xbd, 0x6f, 0x68, 0x14, 0x85,
			0xa9, 0x69, 0x1c, 0x9b, 0xe1, 0x26, 0x18, 0xbb,
			0xa3, 0x93, 0xb3, 0x96, 0xf4, 0x1c, 0xfe, 0xb8,
			0x9e, 0x18, 0xe3, 0x78, 0xc5, 0x1f, 0x14, 0x7c,
			0x7b, 0x0e, 0xde, 0xdb, 0xc4, 0x03, 0xbb, 0x13,
			0x06, 0x45, 0x48, 0x48, 0xc9, 0xbd, 0xb8, 0x9f,
			0x94, 0x78, 0x43, 0xd0, 0xae, 0xaa, 0xdc, 0xdf,
			0x09, 0xba, 0xd9, 0x9e, 0xfb, 0x76, 0xe7, 0x42,
			0x32, 0x25, 0x21, 0x92, 0x9f, 0x03, 0x4d, 0xad,
			0xff, 0xa4, 0x83, 0x95, 0x8d, 0xf5, 0x8a, 0x71,
			0xaf, 0x7d, 0xa4, 0x54, 0x61, 0xfc, 0x40, 0x8c,
			0x7c, 0x45, 0x97, 0x3f, 0xc6, 0x0c, 0x37, 0xa6,
			0x35, 0x87, 0x43, 0x31, 0x51, 0x69, 0xb3, 0x10,
			0x0d, 0x4c, 0xd5, 0x4f, 0x81, 0x0d, 0x6e, 0x03,
			0x69, 0xb9, 0x84, 0x7e, 0xe3, 0x87, 0x95, 0xcf,
			0xe5, 0x84, 0x43, 0x01, 0x95, 0x23, 0xc3, 0xc9,
			0x00, 0x3e, 0xde, 0xc4, 0xcd, 0xaa, 0x70, 0xde,
			0x31, 0xd0, 0x09, 0x58, 0x65, 0x30, 0x58, 0xd8,
			0x50, 0x99, 0x07, 0xa5, 0x14, 0x9a, 0x9f, 0x81,
			0xbe, 0x0e, 0xd0, 0x28, 0x72, 0x4f, 0x72, 0x32,
			0xb5, 0x7f, 0x93, 0xdc, 0x62, 0xcc, 0xf0, 0x93,
			0xa2, 0x63, 0x5e, 0xe1, 0xe5, 0xbf, 0xe6, 0xca,
			0x9e, 0xa0, 0x17, 0xff, 0xab, 0x79, 0x18, 0x2e,
			0xef, 0xff, 0x54, 0x2d, 0x27, 0x8c, 0x47, 0x1e,
			0x1a, 0x2b, 0x34, 0x23, 0x17, 0x00, 0x42, 0x3b,
			0xd0, 0xe7, 0x57, 0xf6, 0xa5, 0x72, 0xa1, 0x4a,
			0x99, 0xc9, 0x03, 0x29, 0xdd, 0x07, 0x01, 0xf3,
			0x47, 0xd8, 0xa6, 0x79, 0xcf, 0xf2, 0x5f, 0xd6,
			0xb0, 0xd3, 0x80, 0xee, 0x5d, 0xc3, 0x30, 0xd6,
			0xff, 0x1b, 0x4b, 0x1a, 0x34, 0x7f, 0xc9, 0x8d,
		},
		32, /* salt length */
		/* V4 : 32 byte */
		{
			0x3e, 0x07, 0xad, 0xe7, 0x2a, 0x3f, 0x52, 0x53,
			0x0f, 0x53, 0x13, 0x5a, 0x5d, 0x7d, 0x93, 0x21,
			0x74, 0x35, 0xba, 0x00, 0x1e, 0xa5, 0x5a, 0x8f,
			0x5d, 0x5d, 0x13, 0x04, 0x68, 0x48, 0x74, 0xbc,
		},
	},
	/* SHA256 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0x28, 0xdb, 0x8f, 0xfa, 0x55, 0xe1, 0x15, 0xdf,
			0x7f, 0x18, 0x8d, 0x62, 0x7c, 0xd2, 0x91, 0xfd,
			0xec, 0xfb, 0xee, 0xa1, 0x10, 0x9e, 0x11, 0x55,
			0xe0, 0xaa, 0xbc, 0x21, 0x57, 0xf7, 0xfe, 0x2a,
			0x12, 0x84, 0x61, 0x1e, 0x19, 0x03, 0x65, 0xd2,
			0xfd, 0x97, 0x2d, 0x2a, 0x23, 0xdc, 0x79, 0x3a,
			0x5f, 0x28, 0xd4, 0xaa, 0xc4, 0x10, 0x0f, 0x5f,
			0xbb, 0x2e, 0xed, 0x57, 0x53, 0x22, 0x20, 0xd5,
			0xd8, 0xd7, 0x74, 0xbf, 0xa7, 0x08, 0x4b, 0x44,
			0x40, 0x02, 0x49, 0xc1, 0x9d, 0xab, 0x50, 0xe6,
			0xc3, 0xc3, 0xaf, 0x15, 0x96, 0x6a, 0x96, 0x0a,
			0xf1, 0xe2, 0xce, 0xc1, 0xf6, 0x97, 0xa6, 0x94,
			0xa3, 0x5c, 0x31, 0xa5, 0xa6, 0xf8, 0xae, 0x7b,
			0x73, 0xe1, 0x48, 0xf0, 0x93, 0x47, 0x00, 0x4a,
			0x3f, 0x54, 0xe7, 0xa8, 0x2d, 0xb3, 0x90, 0xa0,
			0xaa, 0x4f, 0xc5, 0x26, 0xe9, 0x5d, 0x79, 0xaf,
		},
		/* S5 : 384 byte */
		{
			0x72, 0xc5, 0x55, 0x51, 0x11, 0xea, 0xef, 0x95,
			0x42, 0x36, 0x16, 0x37, 0x53, 0x67, 0x4a, 0x6f,
			0xf8, 0x1f, 0x18, 0x2c, 0xbb, 0x37, 0x9b, 0xfc,
			0x6b, 0x54, 0x8a, 0x52, 0xf9, 0xa5, 0xf2, 0x60,
			0xa0, 0xed, 0x58, 0xf5, 0x62, 0xa6, 0x08, 0x6c,
			0xf5, 0xed, 0x00, 0xed, 0x30, 0xad, 0xb0, 0x23,
			0xe9, 0x00, 0x76, 0xa8, 0xad, 0xfa, 0x17, 0xcf,
			0xd7, 0xd7, 0x4f, 0x1e, 0x7b, 0x19, 0x78, 0xb2,
			0x10, 0xda, 0x84, 0x7e, 0xda, 0x6b, 0x49, 0x89,
			0x1e, 0x6b, 0xd3, 0xfc, 0x6c, 0xd4, 0xc8, 0x7b,
			0x93, 0x26, 0xe8, 0x48, 0x1a, 0x16, 0xc6, 0x6e,
			0x40, 0x02, 0x1e, 0x5f, 0x87, 0x8c, 0x30, 0x3d,
			0x3d, 0x85, 0x32, 0xbd, 0x7d, 0x96, 0x65, 0x13,
			0x71, 0x7d, 0x54, 0x99, 0x86, 0x5b, 0x2d, 0x03,
			0xe3, 0x78, 0xe7, 0x6f, 0x79, 0x40, 0xf0, 0x44,
			0x8a, 0xb4, 0xd1, 0x12, 0xe3, 0xc5, 0x2c, 0xb3,
			0x32, 0xd3, 0x40, 0xaf, 0x12, 0x2d, 0xe3, 0xee,
			0x84, 0x9f, 0x2e, 0x25, 0x44, 0xa4, 0x06, 0x91,
			0xdd, 0xf7, 0x01, 0xd9, 0x02, 0xbf, 0xe6, 0x29,
			0x76, 0x6b, 0x36, 0xd8, 0x24, 0x49, 0x28, 0x6f,
			0xd0, 0x3f, 0x75, 0xbb, 0x26, 0x32, 0xdd, 0x61,
			0xd6, 0xb3, 0xc6, 0xce, 0x1c, 0x9e, 0xa8, 0xe5,
			0xaf, 0xf9, 0x2a, 0xd2, 0xca, 0x95, 0xa9, 0x50,
			0xee, 0xcd, 0x99, 0x8e, 0x49, 0x5e, 0x90, 0xe1,
			0xf0, 0x96, 0x6f, 0x92, 0x2b, 0x7f, 0xb3, 0xf0,
			0x33, 0x80, 0x38, 0x5f, 0x3b, 0x14, 0x3a, 0xc1,
			0x96, 0x0c, 0x3b, 0xb6, 0x88, 0xad, 0xbf, 0xd9,
			0x1d, 0x8f, 0xe1, 0xa1, 0xc3, 0x21, 0x60, 0x24,
			0x3d, 0x3b, 0xd2, 0x31, 0xa3, 0x1c, 0x95, 0xdd,
			0x78, 0xb6, 0x64, 0x8c, 0x11, 0x75, 0xfa, 0x9c,
			0x3c, 0x12, 0x44, 0xb1, 0xfa, 0x34, 0xd7, 0xc6,
			0xf3, 0x25, 0x58, 0x53, 0xeb, 0xac, 0xf5, 0xb3,
			0xec, 0x19, 0xb8, 0x64, 0xe0, 0xa4, 0xea, 0xee,
			0x63, 0xfd, 0x71, 0x9c, 0x21, 0xa7, 0x2f, 0xc2,
			0x5b, 0x30, 0xb0, 0x32, 0x07, 0xcf, 0x2a, 0xa4,
			0x5f, 0xd1, 0x5d, 0x71, 0x02, 0xe5, 0xba, 0xe9,
			0x08, 0x82, 0xd0, 0x0a, 0x81, 0x29, 0x59, 0x59,
			0x30, 0x31, 0xea, 0x3a, 0x43, 0x68, 0x98, 0x58,
			0x2c, 0xae, 0x5e, 0xde, 0xd5, 0xc7, 0xce, 0x43,
			0xde, 0x3d, 0xca, 0xc3, 0x0b, 0x86, 0x90, 0x63,
			0x1e, 0x8d, 0xb9, 0xf7, 0xa0, 0xa7, 0xf3, 0xf6,
			0x7b, 0x75, 0x24, 0xdb, 0x27, 0x5a, 0xaf, 0xe0,
			0x24, 0x48, 0x72, 0x7f, 0xf6, 0x29, 0xd1, 0x3a,
			0xfa, 0x94, 0x80, 0x1d, 0x37, 0x52, 0x6f, 0xbd,
			0x91, 0x76, 0xfc, 0x4c, 0x21, 0x62, 0x11, 0x03,
			0x7f, 0x8e, 0xc2, 0x6b, 0x4f, 0x26, 0x72, 0x97,
			0x58, 0x87, 0xd7, 0x0b, 0xcd, 0xbe, 0xef, 0x1e,
			0x6a, 0xe9, 0x9e, 0xdb, 0xfb, 0x6c, 0x9a, 0x9c,
		},
		32, /* salt length */
		/* V5 : 32 byte */
		{
			0x3e, 0x07, 0xad, 0xe7, 0x2a, 0x3f, 0x52, 0x53,
			0x0f, 0x53, 0x13, 0x5a, 0x5d, 0x7d, 0x93, 0x21,
			0x74, 0x35, 0xba, 0x00, 0x1e, 0xa5, 0x5a, 0x8f,
			0x5d, 0x5d, 0x13, 0x04, 0x68, 0x48, 0x74, 0xbc,
		},
	},
	/* SHA256 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0x48, 0x39, 0xd7, 0x1a, 0xab, 0xda, 0xd8, 0xb1,
			0x5d, 0x9f, 0x37, 0xc3, 0xd3, 0x7a, 0x34, 0x67,
			0x58, 0xd8, 0x94, 0x1b, 0x01, 0xc8, 0x39, 0x09,
			0xe4, 0x60, 0xf5, 0x89, 0x85, 0x5c, 0xa0, 0xe6,
			0x91, 0x09, 0x68, 0x65, 0xcf, 0x62, 0x69, 0x83,
			0x53, 0x78, 0x7e, 0x7f, 0xf5, 0x17, 0x56, 0x18,
			0x01, 0xa6, 0xca, 0x98, 0x30, 0x4f, 0x6d, 0x11,
			0xd7, 0x60, 0x65, 0xe7, 0x5f, 0xf1, 0x7a, 0x8e,
			0xf5, 0xc8, 0x6d, 0x95, 0x82, 0x79, 0x8b, 0xe4,
			0xde, 0xd1, 0x81, 0x42, 0x41, 0x75, 0x72, 0x1a,
			0xfa, 0xc7, 0x47, 0x7e, 0x63, 0x09, 0x47, 0x6c,
			0x14, 0xc5, 0xe7, 0x50, 0x57, 0x6c, 0xe3, 0xcb,
			0xdc, 0x3d, 0x8d, 0xb3, 0xae, 0x68, 0x65, 0x5b,
			0x66, 0x74, 0xeb, 0x14, 0x9f, 0xde, 0xb1, 0xf3,
			0xa9, 0x03, 0xb4, 0xd5, 0x82, 0x3f, 0xec, 0xa1,
			0x01, 0x57, 0x22, 0xcd, 0x55, 0x14, 0x02, 0x24,
		},
		/* S6 : 384 byte */
		{
			0x79, 0x6a, 0xc3, 0xf6, 0xad, 0xf4, 0xea, 0xbc,
			0xb7, 0xa5, 0x28, 0xca, 0x63, 0xa6, 0x16, 0x8c,
			0xa6, 0xd3, 0x1d, 0x5e, 0x35, 0x7a, 0xd7, 0xa3,
			0xfd, 0x18, 0x03, 0x34, 0xa9, 0x0d, 0x22, 0xba,
			0xb2, 0x0b, 0x76, 0x2d, 0x76, 0x7a, 0x6e, 0x30,
			0x77, 0xc2, 0xcc, 0x87, 0x32, 0x78, 0x4e, 0x81,
			0x33, 0x00, 0x41, 0xdc, 0x79, 0x06, 0x8d, 0x50,
			0x75, 0x3b, 0xd4, 0x10, 0x9c, 0x9c, 0x6f, 0x9b,
			0xa0, 0x3b, 0x5a, 0xc4, 0x4e, 0xfb, 0xcc, 0x23,
			0xec, 0xda, 0x27, 0x94, 0x85, 0x11, 0x64, 0x5f,
			0xa1, 0x78, 0x97, 0xda, 0xd7, 0xc1, 0x22, 0x95,
			0x7a, 0xe5, 0x6b, 0xf4, 0xff, 0xe3, 0xd7, 0xbe,
			0xf8, 0x50, 0x10, 0xb3, 0x3d, 0x3b, 0x91, 0x78,
			0x5b, 0x04, 0x27, 0x41, 0x7d, 0x94, 0xb1, 0x1f,
			0x73, 0xfd, 0xa9, 0x0e, 0x6a, 0x87, 0x48, 0xe6,
			0xac, 0xc1, 0xd2, 0xd5, 0x82, 0xe8, 0x83, 0x6b,
			0xc7, 0xdb, 0xe1, 0x96, 0x87, 0x6a, 0x95, 0x45,
			0xb2, 0xa3, 0x20, 0x7c, 0x1d, 0x4e, 0xc2, 0x8a,
			0xcf, 0x8f, 0xe6, 0xf2, 0x4c, 0x24, 0x0b, 0x56,
			0xab, 0x3b, 0x4e, 0x43, 0x13, 0xa3, 0xd9, 0x51,
			0xaa, 0x1a, 0x55, 0x82, 0x30, 0xe5, 0xf1, 0xea,
			0xf3, 0x8c, 0xd7, 0xfd, 0x9b, 0x39, 0x3d, 0x58,
			0xd3, 0x59, 0xf5, 0x8f, 0x4a, 0xe5, 0x1d, 0xd3,
			0x97, 0x1b, 0x41, 0x8c, 0x5b, 0x81, 0xd0, 0x70,
			0x7c, 0xd9, 0xe2, 0xc3, 0x3a, 0x14, 0x8e, 0x49,
			0x2e, 0x74, 0xbf, 0xdd, 0x56, 0x5e, 0xba, 0x8b,
			0x1f, 0x39, 0x35, 0xe3, 0x7a, 0x9d, 0x1a, 0x87,
			0x64, 0xcd, 0x30, 0x49, 0x70, 0x66, 0xe3, 0xc4,
			0x62, 0x26, 0x11, 0xfc, 0x14, 0xc4, 0x5b, 0xf4,
			0x6f, 0xc8, 0x5b, 0x3e, 0xd3, 0xf6, 0xc9, 0xd4,
			0xd6, 0x5e, 0x99, 0x25, 0xfe, 0x4b, 0x85, 0xed,
			0x30, 0xec, 0x35, 0xff, 0xc6, 0x9c, 0x5f, 0xdc,
			0x2b, 0xfa, 0x35, 0xd1, 0xbb, 0xdc, 0xb2, 0x0e,
			0x39, 0x9c, 0xf9, 0x34, 0xfe, 0x93, 0x8f, 0x4c,
			0x57, 0x98, 0xcf, 0x09, 0x1d, 0x51, 0x10, 0x0b,
			0x4d, 0xb4, 0xbe, 0x42, 0xe8, 0x19, 0x01, 0xe5,
			0xdc, 0x79, 0xa9, 0x80, 0x74, 0x11, 0x9b, 0x79,
			0x80, 0xb0, 0x28, 0x21, 0xf4, 0xc3, 0xff, 0x8e,
			0xa0, 0x7a, 0x2f, 0xc0, 0x9a, 0x70, 0x19, 0x78,
			0x36, 0x4b, 0xbd, 0x00, 0xce, 0x4c, 0x5e, 0x2e,
			0x45, 0x62, 0x95, 0x26, 0xe3, 0x4a, 0x36, 0x52,
			0x71, 0x9d, 0x27, 0xa4, 0x73, 0x71, 0x48, 0x0d,
			0xaf, 0x52, 0xfa, 0x49, 0x84, 0x4f, 0x64, 0x95,
			0xf3, 0x5e, 0x6f, 0x5e, 0x31, 0x16, 0xc0, 0x0b,
			0x27, 0x04, 0x2b, 0x3c, 0xea, 0xd2, 0x83, 0xbf,
			0xc5, 0x77, 0x90, 0x5f, 0x8b, 0xe8, 0x7f, 0x0d,
			0x5d, 0xaa, 0x13, 0xd1, 0xca, 0x74, 0x20, 0x3a,
			0x9e, 0x0d, 0x91, 0x99, 0xe8, 0x85, 0xf4, 0xfb,
		},
		32, /* salt length */
		/* V6 : 32 byte */
		{
			0x3e, 0x07, 0xad, 0xe7, 0x2a, 0x3f, 0x52, 0x53,
			0x0f, 0x53, 0x13, 0x5a, 0x5d, 0x7d, 0x93, 0x21,
			0x74, 0x35, 0xba, 0x00, 0x1e, 0xa5, 0x5a, 0x8f,
			0x5d, 0x5d, 0x13, 0x04, 0x68, 0x48, 0x74, 0xbc,
		},
	},
	/* SHA256 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0xc0, 0xb8, 0xb2, 0x4f, 0x4b, 0x8e, 0x0b, 0xf2,
			0x91, 0x68, 0xba, 0x73, 0xaa, 0x91, 0x2c, 0x97,
			0x12, 0x1f, 0x71, 0x40, 0xf3, 0x25, 0x9c, 0x40,
			0xa7, 0x2a, 0x6d, 0x6f, 0x78, 0xda, 0x2d, 0xfc,
			0xab, 0xfc, 0xda, 0x00, 0xbe, 0xa4, 0x84, 0x59,
			0xed, 0xaa, 0xf7, 0xb5, 0xfb, 0x5a, 0x9a, 0xed,
			0x2e, 0x6d, 0x97, 0x95, 0x9c, 0x39, 0x3c, 0xd1,
			0xa5, 0x24, 0xa2, 0x69, 0xc1, 0x5e, 0x8c, 0x20,
			0x7c, 0xd0, 0x91, 0x42, 0xbe, 0x4f, 0x7e, 0x7d,
			0x50, 0x16, 0xf6, 0xf1, 0x9c, 0x73, 0x5b, 0x8a,
			0xb4, 0xc0, 0xf2, 0x8e, 0x96, 0x95, 0x41, 0x72,
			0xaf, 0x3c, 0xbc, 0xf2, 0x9d, 0x65, 0xa1, 0x61,
			0x39, 0x1b, 0x21, 0x3d, 0xd5, 0xf7, 0xc0, 0x06,
			0xc2, 0x94, 0xfe, 0x50, 0x16, 0x42, 0x37, 0x18,
			0xab, 0xff, 0xc8, 0x54, 0x6b, 0xa3, 0x73, 0xcd,
			0xcb, 0x5a, 0x05, 0x31, 0x96, 0x57, 0x35, 0x64,
		},
		/* S7 : 384 byte */
		{
			0x85, 0x03, 0xb8, 0x5d, 0xbd, 0x9e, 0xba, 0x8d,
			0x6f, 0xc5, 0x7c, 0x6a, 0xe2, 0x10, 0x3a, 0x78,
			0xdf, 0x1f, 0xff, 0x36, 0x00, 0x58, 0x5e, 0x3e,
			0x18, 0xf6, 0xba, 0x64, 0x36, 0xa3, 0xac, 0xaf,
			0x8e, 0x49, 0xfd, 0x12, 0xdc, 0xbb, 0x37, 0xc2,
			0x5b, 0x4b, 0x76, 0x50, 0x37, 0xf5, 0x45, 0xc3,
			0xda, 0x8c, 0x39, 0xef, 0x68, 0x42, 0xbc, 0x9e,
			0xc2, 0x64, 0xaf, 0x6f, 0x51, 0x92, 0x72, 0xf3,
			0xd8, 0x69, 0x8e, 0xf2, 0xce, 0xac, 0x55, 0x39,
			0x3b, 0xaa, 0x98, 0x46, 0xa7, 0x96, 0x1b, 0x73,
			0x8e, 0x41, 0xf6, 0x36, 0x00, 0x53, 0xd8, 0x66,
			0x76, 0x3c, 0x82, 0x4b, 0xc5, 0x87, 0x3d, 0xa1,
			0x4a, 0x28, 0xeb, 0x47, 0xd6, 0x8d, 0x67, 0xf0,
			0xca, 0xd7, 0x88, 0x08, 0x53, 0xae, 0xb5, 0x61,
			0x04, 0x5f, 0x75, 0x7a, 0x31, 0xd9, 0xf5, 0xc7,
			0x56, 0xf5, 0x4d, 0x79, 0x36, 0x37, 0xd7, 0x21,
			0xc8, 0x8f, 0xb1, 0xf6, 0x01, 0x26, 0xd3, 0xd1,
			0x64, 0x78, 0xf1, 0xfc, 0x15, 0xe0, 0xc4, 0xed,
			0xbb, 0x53, 0x1c, 0x2c, 0xa2, 0xe2, 0xfd, 0x9e,
			0x8d, 0xab, 0xe1, 0xdf, 0x2c, 0x09, 0xfd, 0x55,
			0xbb, 0xc7, 0x24, 0xeb, 0xeb, 0xa2, 0x90, 0xa7,
			0x64, 0x62, 0x49, 0xcd, 0x77, 0x9f, 0xa1, 0xa9,
			0x23, 0x90, 0x9b, 0x29, 0x34, 0x5e, 0x54, 0xa2,
			0xe2, 0x5d, 0xd9, 0x35, 0xbf, 0x06, 0x12, 0xa5,
			0x58, 0x00, 0x18, 0xb2, 0x33, 0xd7, 0x65, 0xa6,
			0xfa, 0xe3, 0xb4, 0x6e, 0xf5, 0x1b, 0xd8, 0x32,
			0x59, 0x12, 0xf4, 0x39, 0xa7, 0xdc, 0x40, 0x14,
			0x8f, 0xdb, 0x75, 0x4e, 0x2d, 0x86, 0x6f, 0x35,
			0x7b, 0x8f, 0x0e, 0xbf, 0xf6, 0xf1, 0x8a, 0x65,
			0x04, 0xba, 0x31, 0xd1, 0x0f, 0xe4, 0x52, 0x26,
			0xc8, 0x8c, 0x92, 0x07, 0xb9, 0xbe, 0x3c, 0x63,
			0x26, 0x1d, 0x75, 0x27, 0x04, 0x66, 0xb4, 0x3c,
			0x27, 0x1f, 0x75, 0xb1, 0xab, 0x3c, 0x1d, 0x6b,
			0x5a, 0x00, 0xdd, 0xa8, 0x45, 0x7b, 0x4d, 0x5c,
			0x21, 0x95, 0xf3, 0x20, 0xb0, 0xbd, 0x54, 0x5f,
			0xdd, 0x06, 0x79, 0xc8, 0x44, 0x83, 0xc1, 0x4a,
			0x46, 0xb4, 0xd4, 0x3c, 0x84, 0x52, 0x87, 0x97,
			0x25, 0xaa, 0x91, 0xd0, 0x1f, 0xcc, 0x2c, 0x38,
			0x67, 0x39, 0x1c, 0x72, 0x20, 0x0c, 0xa5, 0xd6,
			0x28, 0xed, 0x9b, 0x56, 0x63, 0x89, 0xf0, 0x2f,
			0xe7, 0x4b, 0xa2, 0xa4, 0x28, 0xa7, 0xba, 0x31,
			0xc0, 0x0e, 0xf6, 0xb8, 0xd3, 0x8c, 0x6b, 0x82,
			0xb7, 0x37, 0x9d, 0x2f, 0xeb, 0x11, 0x03, 0x18,
			0x48, 0xfe, 0xc0, 0xfa, 0xc5, 0xb6, 0x09, 0x1e,
			0xb7, 0x60, 0x71, 0x38, 0xbf, 0x0b, 0x96, 0xc3,
			0xd2, 0xc1, 0x74, 0xb5, 0x71, 0x3d, 0x0d, 0xc8,
			0x47, 0x0b, 0x53, 0x2e, 0xee, 0x6e, 0xa0, 0xca,
			0x1e, 0x8f, 0xfa, 0x3b, 0x15, 0xcb, 0xe0, 0xbb,
		},
		32, /* salt length */
		/* V7 : 32 byte */
		{
			0x3e, 0x07, 0xad, 0xe7, 0x2a, 0x3f, 0x52, 0x53,
			0x0f, 0x53, 0x13, 0x5a, 0x5d, 0x7d, 0x93, 0x21,
			0x74, 0x35, 0xba, 0x00, 0x1e, 0xa5, 0x5a, 0x8f,
			0x5d, 0x5d, 0x13, 0x04, 0x68, 0x48, 0x74, 0xbc,
		},
	},
	/* SHA256 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0x49, 0x35, 0xea, 0xcc, 0xd2, 0xaf, 0x7c, 0x5b,
			0x99, 0x40, 0x54, 0x71, 0xbe, 0xd9, 0xb2, 0x1d,
			0xa8, 0x96, 0x50, 0x04, 0xf5, 0xe6, 0xf2, 0xa6,
			0xb7, 0xed, 0x3e, 0xe2, 0xdd, 0x26, 0xce, 0xbc,
			0xef, 0x4d, 0x84, 0x5f, 0xff, 0x7c, 0x1d, 0x5e,
			0xdc, 0x94, 0x09, 0x3f, 0x88, 0xde, 0x7a, 0x3a,
			0xec, 0xf2, 0xbc, 0x3e, 0xcb, 0xd8, 0xc4, 0x35,
			0xf5, 0x6e, 0x0b, 0x89, 0xbd, 0x09, 0x9d, 0xe7,
			0xac, 0x5f, 0x6c, 0x43, 0x77, 0xa5, 0xeb, 0x1c,
			0x2f, 0xf4, 0xd8, 0x01, 0xb8, 0xf1, 0x59, 0x54,
			0x7c, 0xad, 0x4b, 0x4e, 0x60, 0xca, 0xd7, 0x43,
			0xf8, 0xe0, 0x46, 0x27, 0xf6, 0x1e, 0x16, 0x52,
			0xe9, 0x35, 0x4d, 0x80, 0x24, 0x71, 0x0d, 0x1c,
			0xfb, 0x29, 0x69, 0xbe, 0x36, 0x5a, 0x77, 0xf2,
			0xbf, 0x8f, 0xa6, 0x3b, 0x9e, 0x04, 0x52, 0x57,
			0x27, 0x0a, 0x96, 0xc5, 0x72, 0xad, 0x62, 0x85,
		},
		/* S8 : 384 byte */
		{
			0x66, 0xd1, 0xce, 0xa9, 0x4b, 0x96, 0x03, 0xef,
			0xad, 0x92, 0xb6, 0xca, 0x8a, 0x1f, 0xbe, 0x0c,
			0x6c, 0x4b, 0x9d, 0xc6, 0x0e, 0xc0, 0xab, 0x2c,
			0x33, 0xbb, 0x62, 0xd2, 0x7a, 0x10, 0x0e, 0x83,
			0x93, 0x78, 0xa3, 0x92, 0x08, 0x71, 0x5d, 0xe2,
			0x10, 0x2e, 0xae, 0x38, 0x4c, 0xa4, 0x07, 0xe9,
			0x27, 0x87, 0xce, 0x11, 0x18, 0xf9, 0x1a, 0x0c,
			0xa2, 0x64, 0x0a, 0x5c, 0x93, 0xfd, 0xb7, 0x86,
			0x35, 0xbc, 0x91, 0x08, 0x2c, 0x99, 0x96, 0x8c,
			0xea, 0xb2, 0x89, 0x89, 0x0b, 0x3e, 0xc2, 0x10,
			0xd6, 0xcc, 0x6f, 0x1c, 0xf7, 0xe0, 0xfb, 0xe2,
			0xda, 0xe8, 0x81, 0x55, 0xe8, 0x8f, 0x2f, 0xb7,
			0xb3, 0x25, 0xab, 0x5e, 0x52, 0x9e, 0x4b, 0x63,
			0x49, 0x3e, 0x55, 0x1c, 0x53, 0xae, 0x38, 0xc3,
			0xfb, 0xfa, 0xe4, 0x98, 0x10, 0x05, 0x0a, 0x81,
			0xcd, 0xce, 0xa6, 0x27, 0xda, 0x21, 0xb6, 0x32,
			0x24, 0x61, 0x2d, 0x43, 0x61, 0xb9, 0xdf, 0x19,
			0x76, 0x1d, 0x6e, 0xad, 0x44, 0x48, 0x8d, 0xca,
			0xbb, 0x50, 0x12, 0x71, 0x49, 0xf0, 0x77, 0xc2,
			0x96, 0x3a, 0xfc, 0x04, 0x9a, 0xc8, 0x83, 0x7f,
			0xf2, 0xc2, 0x9e, 0x6a, 0x35, 0x59, 0x3e, 0x22,
			0x53, 0x1e, 0xcc, 0x2e, 0x9e, 0xf8, 0xbc, 0xba,
			0xae, 0x43, 0x49, 0xbd, 0x72, 0x27, 0xff, 0x3e,
			0x13, 0xb3, 0x1b, 0xb9, 0x29, 0xbb, 0xd4, 0x9e,
			0x50, 0x05, 0x9f, 0x28, 0xfd, 0x9f, 0xfe, 0x8c,
			0x29, 0x6a, 0x05, 0x6c, 0x27, 0x60, 0xe5, 0xf6,
			0xd8, 0xda, 0xb4, 0x3e, 0x9b, 0xd5, 0x57, 0x79,
			0x3f, 0x07, 0x59, 0xad, 0x8e, 0x08, 0xb5, 0xc3,
			0x77, 0x3a, 0x30, 0x5a, 0x0d, 0x31, 0x6f, 0xf9,
			0xbd, 0x07, 0xb4, 0x31, 0x06, 0x33, 0x59, 0x42,
			0x05, 0x5a, 0xdc, 0x46, 0x1a, 0x43, 0x46, 0xf0,
			0x5a, 0xb4, 0x55, 0x78, 0x0f, 0x32, 0x02, 0x7d,
			0xe8, 0xb8, 0xbb, 0x6d, 0x48, 0x45, 0xbb, 0x24,
			0xd0, 0xc5, 0xa2, 0x1c, 0x29, 0x3d, 0x2b, 0x07,
			0x40, 0xe8, 0xd0, 0x6e, 0xf5, 0xfb, 0x9d, 0xbd,
			0xac, 0xb4, 0xfa, 0x1c, 0x62, 0x25, 0xfd, 0x4e,
			0x19, 0xda, 0xe6, 0x9a, 0x8e, 0x2c, 0xbf, 0xdf,
			0xf1, 0xef, 0x8b, 0x7f, 0x21, 0x80, 0x4e, 0xad,
			0x0a, 0x45, 0x27, 0x4c, 0x73, 0x5f, 0xcc, 0xbf,
			0xa1, 0xd6, 0x0b, 0xf4, 0x97, 0xa3, 0xaa, 0x93,
			0x1b, 0xeb, 0xac, 0x2e, 0x0c, 0x8b, 0xed, 0xa9,
			0xaf, 0x59, 0x6d, 0xff, 0x0c, 0xbe, 0x11, 0xe8,
			0xd4, 0x60, 0x2d, 0x36, 0xb2, 0xf6, 0xc6, 0xf5,
			0xbb, 0x80, 0xf1, 0x2f, 0x4b, 0x9d, 0xaf, 0x2c,
			0x07, 0x48, 0xf5, 0x91, 0x09, 0x8e, 0xa6, 0x3d,
			0x31, 0x93, 0xf5, 0x0a, 0x1f, 0x47, 0x37, 0xef,
			0xac, 0xb6, 0x2e, 0xa8, 0x5f, 0xb6, 0xfb, 0x21,
			0x2b, 0x3e, 0xc8, 0xef, 0xfe, 0x78, 0x8e, 0x55,
		},
		32, /* salt length */
		/* V8 : 32 byte */
		{
			0x3e, 0x07, 0xad, 0xe7, 0x2a, 0x3f, 0x52, 0x53,
			0x0f, 0x53, 0x13, 0x5a, 0x5d, 0x7d, 0x93, 0x21,
			0x74, 0x35, 0xba, 0x00, 0x1e, 0xa5, 0x5a, 0x8f,
			0x5d, 0x5d, 0x13, 0x04, 0x68, 0x48, 0x74, 0xbc,
		},
	},
	/* SHA256 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0x3b, 0x8a, 0x68, 0xda, 0x11, 0xb6, 0x1b, 0x5f,
			0xee, 0x1c, 0x2c, 0xa0, 0x0a, 0x6a, 0xa3, 0x5b,
			0xbf, 0xdb, 0xdd, 0x42, 0x85, 0x5b, 0x28, 0x43,
			0x20, 0xec, 0x8d, 0x0c, 0x18, 0x48, 0xed, 0xcf,
			0x6a, 0xc8, 0x50, 0x42, 0x7d, 0x84, 0x79, 0xeb,
			0x57, 0xbc, 0xbe, 0x9a, 0x11, 0x77, 0x16, 0x37,
			0x88, 0x69, 0x74, 0xbd, 0x56, 0x1a, 0x53, 0x87,
			0x01, 0x45, 0x92, 0xcb, 0x71, 0x7e, 0x83, 0x64,
			0xa8, 0x18, 0x3f, 0xd4, 0xad, 0x46, 0x3c, 0x89,
			0xc9, 0x80, 0x21, 0x5f, 0xf6, 0x29, 0xd8, 0x67,
			0x95, 0x6e, 0xe5, 0xe7, 0x5f, 0x71, 0xf7, 0xa1,
			0x9e, 0xa7, 0xbd, 0x58, 0x9d, 0x7e, 0xfb, 0x91,
			0x5d, 0x44, 0xdd, 0x97, 0x89, 0x44, 0x8b, 0xc1,
			0xac, 0x32, 0xfd, 0xf7, 0xa2, 0xc9, 0x11, 0x73,
			0x4d, 0xb2, 0xdb, 0xc5, 0x89, 0xa8, 0x3c, 0x1a,
			0x61, 0xda, 0xb6, 0xbd, 0x83, 0x90, 0x7e, 0xde,
		},
		/* S9 : 384 byte */
		{
			0x79, 0x00, 0x58, 0x35, 0x5d, 0x7a, 0xb9, 0xec,
			0xcb, 0x46, 0xea, 0x12, 0x36, 0x8f, 0x3b, 0xe9,
			0xcf, 0x6b, 0x89, 0x5e, 0x17, 0x34, 0xeb, 0x20,
			0xa1, 0x3c, 0x74, 0x95, 0x57, 0xb9, 0xfe, 0xcf,
			0x92, 0xb3, 0x16, 0x87, 0x0f, 0x0f, 0x76, 0x58,
			0x64, 0xb6, 0x07, 0x43, 0x9e, 0xe5, 0xf7, 0xe5,
			0x10, 0xe2, 0xc8, 0x3b, 0x27, 0x56, 0xa0, 0xd9,
			0x87, 0x7b, 0x48, 0xe0, 0xcf, 0x25, 0x7b, 0x13,
			0xc9, 0x97, 0xb9, 0xdc, 0x70, 0x42, 0x1d, 0x2d,
			0x87, 0xc9, 0xb9, 0xe5, 0x62, 0x5c, 0x36, 0xa1,
			0x7e, 0x21, 0xe2, 0x0e, 0xd3, 0x89, 0x65, 0x7a,
			0x3e, 0x54, 0x4c, 0x67, 0x74, 0x64, 0xee, 0xff,
			0xf0, 0x8a, 0x9e, 0xe4, 0xad, 0xb0, 0x91, 0xa9,
			0xfb, 0xce, 0x76, 0x26, 0xcd, 0xc1, 0x27, 0xb5,
			0xcf, 0x81, 0x7c, 0x2a, 0x5f, 0x06, 0x9e, 0x32,
			0xc7, 0x20, 0xbc, 0x20, 0x41, 0xcd, 0x21, 0xa6,
			0xba, 0xe8, 0x16, 0xdb, 0xbb, 0xe2, 0x85, 0x52,
			0xd0, 0x22, 0xb7, 0xb6, 0x08, 0xfa, 0x99, 0xda,
			0x4d, 0x21, 0x7d, 0xae, 0x8a, 0x69, 0xf5, 0x40,
			0x04, 0xfa, 0x3c, 0x00, 0x4d, 0x50, 0x54, 0x09,
			0x57, 0x64, 0x82, 0x96, 0xe1, 0x4c, 0xca, 0x72,
			0x9f, 0x79, 0x1b, 0x38, 0xe3, 0x64, 0x52, 0x04,
			0xc2, 0xc6, 0xd4, 0xcb, 0x67, 0x8b, 0x0d, 0xb6,
			0x3a, 0x18, 0x1b, 0x40, 0xcd, 0x98, 0x51, 0xbe,
			0x84, 0x62, 0x9a, 0x06, 0x84, 0x15, 0xd5, 0x4c,
			0xab, 0x5c, 0xb5, 0x24, 0x4c, 0x8d, 0xac, 0x8d,
			0xc9, 0x79, 0x9a, 0x0d, 0xf1, 0xb5, 0x8c, 0xeb,
			0xfb, 0xcd, 0x83, 0x77, 0xa3, 0x91, 0x77, 0x88,
			0x69, 0xdd, 0x27, 0x5e, 0x0d, 0xc8, 0x30, 0x5e,
			0xb0, 0x35, 0x1d, 0x81, 0xe3, 0xaf, 0xa4, 0x67,
			0x19, 0x35, 0x5e, 0xee, 0x4f, 0x90, 0x89, 0x4f,
			0x7f, 0xed, 0x66, 0x2d, 0xd3, 0xb0, 0x32, 0x70,
			0x66, 0x0a, 0xdf, 0xf6, 0x37, 0xb9, 0x1e, 0x18,
			0x33, 0x0a, 0x4f, 0x3a, 0x62, 0xc9, 0x14, 0xf0,
			0xd3, 0x2b, 0x4e, 0xb6, 0xa3, 0x0b, 0x79, 0x37,
			0x1a, 0xb5, 0x51, 0x90, 0x57, 0x8a, 0x1e, 0x7d,
			0x43, 0x29, 0x4b, 0xb0, 0xa7, 0x21, 0xde, 0xf7,
			0xda, 0xe3, 0xe0, 0x21, 0x98, 0x17, 0x07, 0x93,
			0x0b, 0xd9, 0xb5, 0xcb, 0x58, 0x67, 0x58, 0x51,
			0xc8, 0x3a, 0xcf, 0x33, 0x0c, 0x6b, 0xa3, 0xae,
			0xcb, 0x3a, 0x89, 0x0a, 0xd3, 0xc1, 0x51, 0xa1,
			0xe2, 0xb5, 0x83, 0xa7, 0xdc, 0xcb, 0xf2, 0x04,
			0x85, 0x0d, 0xaa, 0x9f, 0x46, 0x79, 0xe7, 0x59,
			0xec, 0x05, 0x6a, 0xbe, 0xf7, 0xba, 0x4d, 0x6e,
			0x0b, 0xdf, 0xa5, 0x7a, 0x5c, 0x5a, 0xfb, 0x63,
			0x68, 0xb0, 0x48, 0xa2, 0xb7, 0x4e, 0x35, 0x30,
			0xbf, 0xa8, 0x99, 0x1c, 0x55, 0xde, 0x7c, 0xc8,
			0xbb, 0xfa, 0x99, 0x0d, 0x11, 0x8a, 0xda, 0x80,
		},
		32, /* salt length */
		/* V9 : 32 byte */
		{
			0x3e, 0x07, 0xad, 0xe7, 0x2a, 0x3f, 0x52, 0x53,
			0x0f, 0x53, 0x13, 0x5a, 0x5d, 0x7d, 0x93, 0x21,
			0x74, 0x35, 0xba, 0x00, 0x1e, 0xa5, 0x5a, 0x8f,
			0x5d, 0x5d, 0x13, 0x04, 0x68, 0x48, 0x74, 0xbc,
		},
	},
};

/* 10 elements */
static mod3072_t SHA384[] = {
	/* SHA384 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0x92, 0x21, 0xf0, 0xfe, 0x91, 0x15, 0x84, 0x35,
			0x54, 0xd5, 0x68, 0x5d, 0x9f, 0xe6, 0x9d, 0xc4,
			0x9e, 0x95, 0xce, 0xb5, 0x79, 0x39, 0x86, 0xe4,
			0x28, 0xb8, 0xa1, 0x0b, 0x89, 0x4c, 0x01, 0xd6,
			0xaf, 0x87, 0x82, 0xfd, 0x7d, 0x95, 0x2f, 0xaf,
			0x74, 0xc2, 0xb6, 0x37, 0xca, 0x3b, 0x19, 0xda,
			0xbc, 0x19, 0xa7, 0xfe, 0x25, 0x9b, 0x2b, 0x92,
			0x4e, 0xb3, 0x63, 0xa9, 0x08, 0xc5, 0xb3, 0x68,
			0xf8, 0xab, 0x1b, 0x23, 0x33, 0xfc, 0x67, 0xc3,
			0x0b, 0x8e, 0xa5, 0x6b, 0x28, 0x39, 0xdc, 0x5b,
			0xda, 0xde, 0xfb, 0x14, 0xad, 0xa8, 0x10, 0xbc,
			0x3e, 0x92, 0xba, 0xc5, 0x4e, 0x2a, 0xe1, 0xca,
			0x15, 0x94, 0xa4, 0xb9, 0xd8, 0xd1, 0x93, 0x37,
			0xbe, 0x42, 0x1f, 0x40, 0xe0, 0x67, 0x4e, 0x0e,
			0x9f, 0xed, 0xb4, 0x3d, 0x3a, 0xe8, 0x9e, 0x2c,
			0xa0, 0x5d, 0x90, 0xa6, 0x82, 0x03, 0xf2, 0xc2,
		},
		/* S0 : 384 byte */
		{
			0x96, 0x87, 0x11, 0x5b, 0xe4, 0x78, 0xe4, 0xb6,
			0x42, 0xcd, 0x36, 0x93, 0x92, 0xb9, 0xdd, 0x0f,
			0x35, 0x76, 0xe7, 0x04, 0xaf, 0x72, 0x18, 0xb1,
			0xf9, 0x4d, 0x7f, 0x8f, 0xe7, 0xf0, 0x70, 0x73,
			0xe3, 0xe8, 0xe1, 0x18, 0x6f, 0xa7, 0x68, 0x97,
			0x7d, 0x6b, 0x51, 0x4e, 0x51, 0x34, 0x59, 0xf2,
			0x37, 0x3d, 0xf6, 0xec, 0x52, 0xe3, 0xde, 0x9b,
			0xd8, 0x3f, 0xcc, 0x5c, 0xc3, 0xe6, 0xb9, 0x7f,
			0x8b, 0x3f, 0xb5, 0x34, 0x16, 0x3c, 0x64, 0xf5,
			0x26, 0x76, 0x20, 0x70, 0x0e, 0x9d, 0x8c, 0x52,
			0xb3, 0xdf, 0x61, 0xa7, 0xc3, 0x74, 0x8e, 0xf1,
			0x59, 0xd6, 0xb3, 0x90, 0x89, 0x5a, 0xfa, 0x3a,
			0xf5, 0x91, 0x09, 0xa5, 0x47, 0x8d, 0x01, 0x6d,
			0x96, 0xc4, 0x9f, 0x68, 0xdf, 0xc7, 0x35, 0xba,
			0x2a, 0xaf, 0xd5, 0x01, 0x2c, 0x13, 0x51, 0x5e,
			0xd6, 0x64, 0x4f, 0x0d, 0x41, 0x09, 0xc4, 0x55,
			0x56, 0xe1, 0x4a, 0x38, 0x21, 0xe1, 0xaa, 0x24,
			0xbe, 0xb8, 0xa8, 0x1a, 0x48, 0xda, 0x27, 0xf1,
			0x31, 0xde, 0x84, 0xf7, 0xba, 0x51, 0x58, 0x1d,
			0x81, 0xb8, 0xff, 0x31, 0xba, 0x92, 0xb8, 0xa1,
			0xfd, 0xe8, 0x67, 0xf0, 0x7e, 0x32, 0xe6, 0xc2,
			0x70, 0x92, 0x53, 0x44, 0x81, 0x74, 0xdd, 0x31,
			0x32, 0x4d, 0xbc, 0x32, 0xb0, 0x5f, 0x07, 0x58,
			0x7f, 0x76, 0xa9, 0x99, 0x7d, 0xec, 0xb8, 0x0f,
			0x38, 0xd8, 0xc1, 0x3d, 0x0f, 0x6e, 0xb3, 0xc1,
			0x0e, 0x3d, 0x96, 0xa2, 0x29, 0x3f, 0x74, 0x64,
			0xf1, 0xe0, 0x46, 0x02, 0xef, 0x6e, 0x84, 0xc2,
			0xd0, 0x24, 0x5d, 0x7d, 0xb2, 0x56, 0xa6, 0x7d,
			0x13, 0x2a, 0x47, 0xca, 0xe9, 0xab, 0xe0, 0x6b,
			0x61, 0xa8, 0x96, 0x8f, 0x50, 0xa1, 0x74, 0x99,
			0x95, 0xdc, 0x15, 0xef, 0x0d, 0xcb, 0x1d, 0x5f,
			0x59, 0x59, 0xe4, 0xd4, 0x54, 0xc8, 0x54, 0x7b,
			0xbb, 0x4d, 0x19, 0x56, 0x98, 0xf4, 0x84, 0x61,
			0x7b, 0xfd, 0x12, 0x2a, 0xca, 0xae, 0x2d, 0x0e,
			0x8c, 0x76, 0xd2, 0x8b, 0x24, 0x00, 0x5a, 0xb0,
			0x3c, 0xaa, 0x78, 0x1e, 0xa9, 0x7b, 0x1c, 0x4d,
			0x93, 0x96, 0xa1, 0x6f, 0x79, 0x98, 0xee, 0xe7,
			0xdd, 0xd9, 0xde, 0x4c, 0xab, 0xe5, 0x70, 0x32,
			0xd9, 0x43, 0x8a, 0x5d, 0x99, 0xc6, 0xb3, 0x4a,
			0x95, 0x61, 0x22, 0x35, 0x02, 0x63, 0xc7, 0xe9,
			0x98, 0xbc, 0x61, 0xde, 0xc9, 0x13, 0x81, 0x01,
			0x2e, 0x68, 0x6d, 0x07, 0x9e, 0x39, 0xe9, 0x6b,
			0x1e, 0xa4, 0xbf, 0xdb, 0x7c, 0xdf, 0x63, 0x0d,
			0xdb, 0x42, 0x2c, 0x6b, 0x58, 0x0e, 0x55, 0x06,
			0xc9, 0xcc, 0x3d, 0x6c, 0x10, 0x0f, 0x20, 0x41,
			0xd1, 0x7c, 0xea, 0xaa, 0xa5, 0x45, 0x89, 0x24,
			0x9f, 0x04, 0xa1, 0x37, 0x0f, 0xfa, 0x3b, 0xf3,
			0xff, 0x1a, 0xde, 0xb8, 0x90, 0x68, 0x86, 0x98,
		},
		48, /* salt length */
		/* V0 : 48 byte */
		{
			0x61, 0xa7, 0x62, 0xf8, 0x96, 0x8d, 0x5f, 0x36,
			0x7e, 0x2d, 0xbc, 0xac, 0xb4, 0x02, 0x16, 0x53,
			0xdc, 0x75, 0x43, 0x7d, 0x90, 0x00, 0xe3, 0x16,
			0x9d, 0x94, 0x37, 0x29, 0x70, 0x38, 0x37, 0xa5,
			0xcb, 0xf4, 0xde, 0x62, 0xbd, 0xed, 0xc9, 0x5f,
			0xd0, 0xd1, 0x00, 0x4e, 0x84, 0x75, 0x14, 0x52,
		},
	},
	/* SHA384 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0x75, 0x2a, 0x99, 0x16, 0xf4, 0x49, 0xae, 0xbf,
			0x81, 0x4c, 0xe5, 0x9c, 0xa6, 0xe8, 0x2f, 0xa8,
			0x03, 0x8e, 0x46, 0x85, 0x41, 0x92, 0x41, 0xc1,
			0x48, 0x8c, 0x66, 0x59, 0xb2, 0xff, 0x3f, 0x7b,
			0x7f, 0x38, 0xf0, 0x90, 0x0a, 0x79, 0xc7, 0x7a,
			0x3b, 0x57, 0x15, 0x1a, 0xff, 0x61, 0x3c, 0x16,
			0xf5, 0x02, 0x0a, 0xd9, 0x6b, 0xa9, 0x45, 0xdb,
			0x88, 0x26, 0x87, 0x22, 0xca, 0x58, 0x4c, 0x09,
			0xb4, 0x05, 0x4a, 0x40, 0xc0, 0x09, 0x01, 0x14,
			0x9b, 0xb3, 0x92, 0xf0, 0x91, 0x6c, 0xd4, 0x24,
			0x46, 0x99, 0xa5, 0xe6, 0xa8, 0xc3, 0x7e, 0x96,
			0x21, 0xf5, 0x4b, 0x47, 0x11, 0x66, 0x79, 0x7a,
			0x7b, 0x58, 0x50, 0x2c, 0xff, 0x40, 0x83, 0x14,
			0x08, 0x27, 0x05, 0x26, 0x46, 0x50, 0x1f, 0x5b,
			0x5f, 0x1b, 0xc0, 0xb4, 0xe1, 0x29, 0x14, 0x7d,
			0x7c, 0xc1, 0x57, 0xcf, 0x6e, 0x73, 0xec, 0x58,
		},
		/* S1 : 384 byte */
		{
			0x66, 0x46, 0xa8, 0x8e, 0xe4, 0xb8, 0x45, 0xda,
			0x49, 0x31, 0x27, 0x4c, 0x23, 0x84, 0x0d, 0xad,
			0xa6, 0x14, 0x5f, 0xe0, 0xaf, 0x95, 0x48, 0x29,
			0xd1, 0xd5, 0x66, 0x61, 0x54, 0x6a, 0x25, 0xe4,
			0x63, 0x16, 0xe2, 0x16, 0xbb, 0x6b, 0x94, 0x46,
			0xb3, 0x68, 0x88, 0x4b, 0xa1, 0x49, 0x69, 0xa6,
			0xf6, 0x8c, 0xcb, 0xc1, 0xcf, 0x5b, 0x4e, 0x7a,
			0x6d, 0x3a, 0xab, 0xec, 0x67, 0xf6, 0x49, 0x63,
			0xf6, 0x3b, 0x08, 0x8f, 0xa8, 0x17, 0xc8, 0x55,
			0xd7, 0x76, 0xdd, 0xca, 0xda, 0x57, 0xe5, 0xda,
			0xa5, 0x0f, 0xc1, 0xc8, 0x77, 0x38, 0x9c, 0x3c,
			0xb9, 0xd9, 0x90, 0x95, 0xa8, 0x69, 0xa9, 0x63,
			0xbc, 0x91, 0xec, 0x24, 0xb2, 0x42, 0x2e, 0xf6,
			0xb8, 0xdd, 0x18, 0xfd, 0x20, 0xd2, 0xb2, 0x15,
			0xfe, 0xe6, 0xe9, 0x8c, 0xda, 0x41, 0x5a, 0xe4,
			0x4d, 0x2d, 0x26, 0x16, 0xfe, 0x17, 0x08, 0x29,
			0x2a, 0x3e, 0xf5, 0x0a, 0x07, 0x51, 0x70, 0xb3,
			0xa7, 0xeb, 0xab, 0x02, 0x91, 0x8a, 0xb0, 0x30,
			0x17, 0x94, 0xc1, 0x7f, 0xb3, 0x5e, 0x20, 0x38,
			0xf3, 0x69, 0xd9, 0x4d, 0xd4, 0x95, 0x69, 0xc0,
			0x66, 0xf7, 0xc3, 0x92, 0x88, 0x9d, 0xc4, 0xb8,
			0x78, 0xc5, 0x0c, 0x7e, 0x52, 0x58, 0x6b, 0x50,
			0x81, 0x11, 0x4d, 0x20, 0x23, 0x38, 0xd2, 0x33,
			0x04, 0xf1, 0x6f, 0x91, 0x2d, 0x51, 0x9a, 0x9a,
			0xd2, 0x1b, 0xaf, 0xf0, 0xe3, 0xd2, 0x17, 0x61,
			0xf3, 0x73, 0xd0, 0x84, 0x21, 0xe1, 0x01, 0x08,
			0xa9, 0x83, 0x04, 0x8f, 0xcb, 0x90, 0xeb, 0x2a,
			0xdc, 0x7c, 0x7f, 0x12, 0xff, 0xa1, 0x57, 0x1b,
			0x09, 0x1c, 0x78, 0x1b, 0x25, 0x5a, 0x77, 0xa8,
			0x80, 0xe9, 0x79, 0x75, 0xf1, 0x4f, 0x42, 0xba,
			0xf5, 0xaa, 0x28, 0x5e, 0xcc, 0x14, 0x21, 0x57,
			0xc3, 0xe1, 0xad, 0xdd, 0x6a, 0xa0, 0xc0, 0x92,
			0x53, 0xa1, 0x1c, 0x59, 0x14, 0x4a, 0xbd, 0x3b,
			0x1e, 0x21, 0x2d, 0x89, 0xe2, 0x7e, 0xd9, 0x6f,
			0xb7, 0x57, 0x56, 0xaf, 0xc2, 0x0e, 0xc6, 0x74,
			0x23, 0xb1, 0x51, 0x19, 0x4c, 0xb0, 0xb0, 0x64,
			0x8c, 0x65, 0x99, 0x87, 0xa5, 0x58, 0x3c, 0xb7,
			0x75, 0x77, 0x79, 0xd8, 0xa3, 0x9e, 0x20, 0x5e,
			0x71, 0x01, 0xa5, 0x35, 0x1c, 0xe1, 0xaf, 0x2c,
			0x9c, 0x6b, 0x08, 0x47, 0xcc, 0xa5, 0x7a, 0xf5,
			0x25, 0x93, 0x32, 0x39, 0x05, 0xe3, 0xd2, 0x29,
			0x7c, 0x0d, 0x54, 0x54, 0x1a, 0x01, 0x25, 0x62,
			0x16, 0x40, 0xfe, 0x1d, 0xee, 0xf1, 0x3e, 0x75,
			0x9f, 0x8f, 0x6c, 0x56, 0xa2, 0xec, 0x2a, 0x94,
			0x83, 0x1a, 0xc2, 0xc6, 0x14, 0xb9, 0x11, 0xe7,
			0x9e, 0xdd, 0x54, 0x2f, 0xef, 0x65, 0x1f, 0x5a,
			0x82, 0x7f, 0x48, 0x05, 0x75, 0xae, 0x22, 0x0c,
			0x49, 0x5f, 0x2a, 0x28, 0x42, 0xf9, 0x9e, 0xc4,
		},
		48, /* salt length */
		/* V1 : 48 byte */
		{
			0x61, 0xa7, 0x62, 0xf8, 0x96, 0x8d, 0x5f, 0x36,
			0x7e, 0x2d, 0xbc, 0xac, 0xb4, 0x02, 0x16, 0x53,
			0xdc, 0x75, 0x43, 0x7d, 0x90, 0x00, 0xe3, 0x16,
			0x9d, 0x94, 0x37, 0x29, 0x70, 0x38, 0x37, 0xa5,
			0xcb, 0xf4, 0xde, 0x62, 0xbd, 0xed, 0xc9, 0x5f,
			0xd0, 0xd1, 0x00, 0x4e, 0x84, 0x75, 0x14, 0x52,
		},
	},
	/* SHA384 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x04, 0x03, 0xef, 0x21, 0x99, 0x38, 0xb8, 0xcd,
			0xbf, 0x85, 0xd3, 0xb8, 0x8c, 0xbb, 0x9c, 0x60,
			0xd1, 0x74, 0x13, 0x4e, 0x43, 0xa7, 0x28, 0x4c,
			0xd8, 0x79, 0x36, 0xd3, 0x74, 0x56, 0xcd, 0xc3,
			0xc5, 0x41, 0xb4, 0x56, 0x6b, 0x68, 0x2e, 0x57,
			0x5d, 0xfc, 0x7d, 0x8f, 0x88, 0x3f, 0xa5, 0x81,
			0xb9, 0xdf, 0x72, 0x57, 0xbc, 0x82, 0xbc, 0x1b,
			0xc6, 0xa2, 0xea, 0x2a, 0x10, 0x9b, 0xb5, 0xe6,
			0xc5, 0x02, 0x2f, 0xac, 0x1e, 0x39, 0x03, 0x06,
			0xcb, 0x40, 0xfe, 0x21, 0x96, 0xce, 0xce, 0x81,
			0x43, 0xa1, 0x0a, 0xf3, 0xba, 0x12, 0x73, 0xc3,
			0x68, 0xec, 0x7a, 0x30, 0xe2, 0x7e, 0x02, 0x1d,
			0xcb, 0xef, 0x66, 0x09, 0xf9, 0xd2, 0xbe, 0x41,
			0xd3, 0xfa, 0x5e, 0x54, 0xfd, 0x90, 0xa0, 0xc8,
			0x38, 0x62, 0xe4, 0x0b, 0x83, 0x7e, 0xd4, 0xac,
			0x86, 0x00, 0xed, 0xcb, 0x31, 0x28, 0x3b, 0xcf,
		},
		/* S2 : 384 byte */
		{
			0x0a, 0x21, 0x75, 0x03, 0xfc, 0x48, 0x70, 0x48,
			0x12, 0x64, 0xd8, 0x30, 0x82, 0x92, 0xc6, 0x63,
			0x47, 0x6b, 0x25, 0xf8, 0xde, 0xc0, 0x8e, 0xa1,
			0xd1, 0x27, 0x6f, 0x09, 0x51, 0xec, 0x6d, 0xf2,
			0x7a, 0xae, 0x3b, 0xeb, 0x93, 0xd6, 0x30, 0xbf,
			0x8f, 0xac, 0x08, 0xb6, 0xcc, 0xe5, 0x0b, 0xd9,
			0x29, 0x94, 0x85, 0x1b, 0x4f, 0x31, 0x0f, 0xdd,
			0xdc, 0xe8, 0xe0, 0xd6, 0xa8, 0xb7, 0xa1, 0xe8,
			0x66, 0xa5, 0x67, 0xb2, 0x98, 0xc5, 0x57, 0x7d,
			0xc5, 0x0d, 0x8a, 0x90, 0x6a, 0xb1, 0xbe, 0x88,
			0x00, 0x84, 0xe6, 0x81, 0xb2, 0x64, 0x56, 0x27,
			0x91, 0x49, 0xb4, 0xb8, 0x52, 0x01, 0x62, 0x1c,
			0x44, 0x5d, 0xe1, 0x3d, 0x12, 0x7f, 0xb7, 0x7e,
			0x7f, 0x23, 0x6c, 0x39, 0xdf, 0x34, 0x05, 0x2b,
			0x46, 0x29, 0x57, 0x2a, 0xbe, 0x1c, 0x02, 0xc0,
			0x9e, 0xb1, 0x98, 0x18, 0x80, 0x03, 0xdd, 0x85,
			0x2f, 0x88, 0xf4, 0xf7, 0x67, 0xf1, 0x00, 0x04,
			0x58, 0x68, 0x02, 0x58, 0xfa, 0x4b, 0x63, 0xda,
			0xfc, 0x76, 0x18, 0x22, 0xca, 0x8b, 0x98, 0xc1,
			0xa1, 0x21, 0xb7, 0x2b, 0x14, 0x55, 0x39, 0x3b,
			0xee, 0x41, 0x6d, 0x24, 0x05, 0x12, 0x90, 0xf0,
			0x2a, 0x28, 0xa7, 0xb4, 0x9b, 0x18, 0xb3, 0x0c,
			0xcb, 0x29, 0xc2, 0x6f, 0xba, 0xc9, 0x91, 0x40,
			0x1a, 0x3a, 0x6f, 0xe0, 0x1f, 0xcd, 0x06, 0x08,
			0x92, 0x0f, 0xac, 0xae, 0x9d, 0x5b, 0xc5, 0x65,
			0x40, 0xc8, 0x0f, 0x47, 0x40, 0xaf, 0x02, 0xc9,
			0xb7, 0xa0, 0x78, 0x95, 0x8a, 0x8d, 0x8a, 0x7a,
			0x93, 0xa5, 0xe5, 0xb6, 0xd2, 0x57, 0x1f, 0x49,
			0xd7, 0x75, 0xef, 0x7c, 0x35, 0xa6, 0xd6, 0x74,
			0x29, 0x0b, 0x52, 0xcf, 0xbc, 0xd6, 0x72, 0x77,
			0xe2, 0xb2, 0xe8, 0x29, 0xec, 0x43, 0x7f, 0xb7,
			0x0e, 0x90, 0x53, 0x7e, 0xaa, 0x6f, 0xe4, 0x54,
			0x85, 0x51, 0x93, 0x9b, 0xfa, 0x98, 0xfc, 0x98,
			0xe2, 0x35, 0xb2, 0x64, 0xaa, 0x60, 0x64, 0xa5,
			0x05, 0xa8, 0xd6, 0x79, 0x46, 0xe2, 0xc3, 0x3e,
			0x5c, 0x6f, 0x0f, 0x34, 0xfa, 0x86, 0xba, 0x65,
			0x71, 0x5c, 0x25, 0x8f, 0x23, 0x8b, 0x69, 0xe4,
			0xf6, 0xe3, 0x6d, 0x86, 0xa8, 0x98, 0x22, 0xb4,
			0x80, 0x2d, 0x21, 0xba, 0x0b, 0xa7, 0x60, 0xb2,
			0xf3, 0xa5, 0xbd, 0x06, 0x1f, 0x50, 0xaa, 0xad,
			0xff, 0x12, 0xe0, 0xd8, 0x66, 0x27, 0x29, 0x4b,
			0xd0, 0xc4, 0xcd, 0x10, 0x85, 0xb5, 0xda, 0xb6,
			0xa6, 0xab, 0x30, 0x14, 0x6c, 0x9b, 0xbb, 0x37,
			0xde, 0x3a, 0xc5, 0xc4, 0xf8, 0xee, 0x29, 0x73,
			0x6d, 0x46, 0x04, 0x7e, 0x45, 0x0c, 0xfd, 0xcb,
			0x12, 0x79, 0xe4, 0xca, 0x83, 0xab, 0x69, 0xe8,
			0x58, 0x74, 0x1b, 0xfd, 0x01, 0xa7, 0x79, 0xd4,
			0x75, 0xdf, 0xc0, 0xf7, 0x1c, 0x62, 0x1d, 0x78,
		},
		48, /* salt length */
		/* V2 : 48 byte */
		{
			0x61, 0xa7, 0x62, 0xf8, 0x96, 0x8d, 0x5f, 0x36,
			0x7e, 0x2d, 0xbc, 0xac, 0xb4, 0x02, 0x16, 0x53,
			0xdc, 0x75, 0x43, 0x7d, 0x90, 0x00, 0xe3, 0x16,
			0x9d, 0x94, 0x37, 0x29, 0x70, 0x38, 0x37, 0xa5,
			0xcb, 0xf4, 0xde, 0x62, 0xbd, 0xed, 0xc9, 0x5f,
			0xd0, 0xd1, 0x00, 0x4e, 0x84, 0x75, 0x14, 0x52,
		},
	},
	/* SHA384 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0x45, 0x3e, 0x08, 0x35, 0xfe, 0xe7, 0xcd, 0xe8,
			0x1f, 0x18, 0xc2, 0xb3, 0x09, 0xb8, 0x04, 0xc6,
			0x7b, 0x9f, 0xd9, 0xe9, 0x6e, 0xf0, 0xa9, 0x6e,
			0x3d, 0xa9, 0x4b, 0x64, 0x09, 0x78, 0x83, 0x0e,
			0x5c, 0xd1, 0xc8, 0x94, 0x0c, 0x3d, 0x4a, 0xf7,
			0x63, 0xf5, 0x33, 0x4a, 0x7c, 0xaf, 0x2d, 0x20,
			0xf0, 0xb8, 0x25, 0x41, 0xb3, 0x43, 0x4f, 0xa1,
			0x38, 0x01, 0x6b, 0x92, 0xdc, 0xf1, 0x46, 0x38,
			0x81, 0x7a, 0x83, 0x3d, 0x79, 0xb7, 0x9b, 0xc7,
			0xa7, 0x12, 0x23, 0xa7, 0xe0, 0x14, 0x4e, 0xd4,
			0x97, 0x7b, 0xb2, 0x17, 0xba, 0x8d, 0x4f, 0x07,
			0xd7, 0xad, 0xcd, 0x38, 0x83, 0x2c, 0x05, 0xb0,
			0xfc, 0x61, 0xc3, 0x9a, 0x0d, 0xfc, 0xca, 0x3f,
			0x32, 0x97, 0x19, 0x31, 0xfd, 0x8e, 0x6d, 0xc9,
			0xb8, 0x11, 0x07, 0xd4, 0x4c, 0x77, 0xaf, 0x8a,
			0x62, 0xd5, 0xf9, 0xc0, 0xc7, 0xd0, 0xc7, 0x5e,
		},
		/* S3 : 384 byte */
		{
			0x6e, 0xc2, 0x2b, 0xd5, 0x8c, 0x32, 0xd4, 0x13,
			0x74, 0xc0, 0x17, 0xa7, 0x70, 0x27, 0xe7, 0x70,
			0xf6, 0x78, 0xfd, 0x81, 0x01, 0x7e, 0x20, 0xcd,
			0xaa, 0xab, 0x48, 0xa8, 0x32, 0x4b, 0x05, 0x07,
			0x49, 0xe5, 0xd8, 0x64, 0x08, 0x2f, 0x1f, 0x77,
			0xfe, 0xcf, 0x67, 0xa5, 0x9c, 0x28, 0x85, 0xe9,
			0x31, 0xc3, 0xc2, 0xf5, 0x81, 0x30, 0xfa, 0x68,
			0x06, 0xfe, 0x1c, 0xa8, 0x99, 0x04, 0x51, 0x14,
			0xb0, 0x9d, 0x09, 0xcf, 0x9c, 0x51, 0x3c, 0xe1,
			0x10, 0x9d, 0x22, 0x10, 0x51, 0x1a, 0x3b, 0x2e,
			0x93, 0xaf, 0x51, 0x1b, 0xad, 0xad, 0x27, 0x16,
			0xf4, 0x85, 0x55, 0x31, 0x0e, 0x6c, 0x5f, 0x54,
			0x7a, 0xfb, 0xdb, 0x0b, 0x9a, 0x68, 0x44, 0x91,
			0xff, 0x35, 0x88, 0xdf, 0x93, 0x3d, 0x6b, 0x04,
			0xda, 0xe8, 0x88, 0x3f, 0x5f, 0x8a, 0xad, 0x62,
			0xa4, 0x57, 0x06, 0x46, 0xf7, 0x2f, 0x36, 0x56,
			0xc4, 0xa7, 0x08, 0x56, 0x23, 0xf5, 0x15, 0x21,
			0x64, 0xa8, 0x1a, 0x06, 0xcc, 0xb5, 0x9c, 0xa4,
			0x78, 0xc5, 0xc2, 0x31, 0x54, 0x14, 0x55, 0x0b,
			0x0a, 0xd8, 0xee, 0xcd, 0x03, 0x28, 0xb2, 0xdb,
			0x01, 0xff, 0xf7, 0xdb, 0x0f, 0x26, 0x59, 0x6c,
			0x41, 0xf9, 0x70, 0xd0, 0x32, 0x92, 0x58, 0x87,
			0xf1, 0xc8, 0xa4, 0x46, 0xda, 0x88, 0x9b, 0xe6,
			0x4d, 0x48, 0x92, 0x5b, 0x9c, 0x6b, 0x79, 0xa3,
			0xd8, 0x97, 0x70, 0x0a, 0xb4, 0x0a, 0xf2, 0x0b,
			0x45, 0x1a, 0xaa, 0x6b, 0x42, 0x7e, 0xd1, 0x62,
			0x86, 0x4d, 0xb8, 0x9f, 0x78, 0x24, 0xb6, 0xae,
			0x9b, 0x47, 0x5b, 0x54, 0x33, 0xb8, 0x65, 0x33,
			0x5d, 0x6f, 0x91, 0x49, 0x1c, 0x1e, 0x32, 0xf6,
			0x35, 0xcb, 0x93, 0x0d, 0xec, 0x1a, 0xa3, 0xee,
			0x7d, 0xda, 0xa0, 0x8e, 0x8e, 0xbd, 0x67, 0xb6,
			0xb1, 0x1a, 0x46, 0xba, 0x04, 0x99, 0x22, 0x44,
			0x6f, 0xa6, 0x9f, 0x1a, 0x80, 0x4a, 0xcc, 0x29,
			0xf6, 0xce, 0xe4, 0x87, 0x72, 0x3f, 0x2e, 0x61,
			0xa4, 0x00, 0x07, 0x86, 0x5d, 0x80, 0xcd, 0xe0,
			0x11, 0x9f, 0x3f, 0xe6, 0xe1, 0x61, 0xa3, 0x39,
			0x48, 0x7f, 0x57, 0x89, 0xe1, 0xfd, 0x23, 0xac,
			0x0a, 0x63, 0xb4, 0x67, 0x39, 0x69, 0xfd, 0x87,
			0x22, 0xe3, 0xed, 0xc9, 0x43, 0x97, 0x78, 0x92,
			0x8f, 0x09, 0x61, 0x0c, 0xbe, 0xfb, 0xb4, 0x2f,
			0xe6, 0x24, 0x2c, 0x73, 0xb6, 0x8d, 0x46, 0x6c,
			0xef, 0x88, 0x9d, 0xa1, 0x56, 0xd9, 0xd4, 0xff,
			0x88, 0x83, 0x62, 0xdb, 0x4c, 0xf9, 0xa9, 0x41,
			0xe8, 0x0f, 0x57, 0x7c, 0x94, 0x4b, 0x79, 0xfb,
			0x27, 0xdb, 0xe0, 0xa6, 0x96, 0x7e, 0x88, 0xf1,
			0xf6, 0x7b, 0x91, 0xb0, 0xd3, 0x8e, 0x08, 0x3f,
			0xc0, 0xc0, 0x22, 0x8c, 0xd4, 0x9d, 0x27, 0x35,
			0x25, 0x21, 0x31, 0x21, 0x63, 0xf9, 0x0f, 0xba,
		},
		48, /* salt length */
		/* V3 : 48 byte */
		{
			0x61, 0xa7, 0x62, 0xf8, 0x96, 0x8d, 0x5f, 0x36,
			0x7e, 0x2d, 0xbc, 0xac, 0xb4, 0x02, 0x16, 0x53,
			0xdc, 0x75, 0x43, 0x7d, 0x90, 0x00, 0xe3, 0x16,
			0x9d, 0x94, 0x37, 0x29, 0x70, 0x38, 0x37, 0xa5,
			0xcb, 0xf4, 0xde, 0x62, 0xbd, 0xed, 0xc9, 0x5f,
			0xd0, 0xd1, 0x00, 0x4e, 0x84, 0x75, 0x14, 0x52,
		},
	},
	/* SHA384 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0x9a, 0xff, 0x46, 0xc1, 0x4f, 0xd8, 0x10, 0xa0,
			0x39, 0xc0, 0xa6, 0x2e, 0xda, 0x40, 0x3f, 0x5c,
			0xa9, 0x02, 0xac, 0x41, 0xb8, 0xe2, 0x25, 0xc6,
			0x94, 0x47, 0x48, 0xa3, 0x6c, 0xb4, 0x5f, 0x8a,
			0x76, 0x9a, 0xe2, 0xa1, 0x8f, 0x71, 0x3d, 0x36,
			0x22, 0x06, 0xd2, 0xaf, 0x4a, 0x17, 0x42, 0xbf,
			0x3b, 0x1d, 0xe8, 0xe0, 0xde, 0x69, 0xa7, 0xfd,
			0xbb, 0x72, 0xe6, 0x6e, 0x1c, 0x6e, 0xd8, 0x2a,
			0x6f, 0x1f, 0x01, 0x38, 0xed, 0xf0, 0xf6, 0x67,
			0x79, 0x40, 0x64, 0x3f, 0xcb, 0xfe, 0x53, 0x37,
			0xcd, 0x76, 0xac, 0x29, 0x45, 0x6a, 0xf9, 0x02,
			0xb5, 0x65, 0x6d, 0xbe, 0x7f, 0x4c, 0x24, 0x94,
			0x4d, 0x36, 0xab, 0x6d, 0xb0, 0x7d, 0xc3, 0x9b,
			0x08, 0x16, 0x62, 0xc8, 0xa3, 0x1d, 0xfb, 0x2c,
			0x29, 0xb4, 0xff, 0x04, 0x37, 0x0e, 0xa4, 0x3f,
			0x4a, 0xc7, 0xe5, 0x7a, 0xdf, 0x77, 0xca, 0x2e,
		},
		/* S4 : 384 byte */
		{
			0x62, 0xa5, 0x05, 0xb3, 0xf3, 0xad, 0xda, 0x45,
			0xc6, 0xba, 0xdb, 0x61, 0xb4, 0x64, 0xa2, 0x8b,
			0xc4, 0x5d, 0x4c, 0x66, 0x15, 0x9a, 0x34, 0xb6,
			0x3c, 0x1c, 0xce, 0x32, 0x60, 0x42, 0x42, 0xeb,
			0x8f, 0xcd, 0x9a, 0xc6, 0x92, 0x9e, 0xc6, 0xee,
			0x4a, 0xc1, 0x14, 0x49, 0x32, 0xd7, 0x25, 0xcb,
			0xf4, 0x63, 0x85, 0x11, 0x46, 0x4e, 0xc7, 0x0d,
			0xbb, 0x55, 0x43, 0xa4, 0x48, 0x7a, 0x24, 0x13,
			0x96, 0xad, 0xb8, 0x04, 0xc9, 0x79, 0x4b, 0x27,
			0x1f, 0x9d, 0x35, 0x31, 0x0e, 0xe5, 0x60, 0x36,
			0x8d, 0x94, 0x9a, 0x20, 0xa2, 0xb6, 0x4c, 0xb4,
			0x61, 0x7f, 0xcf, 0x63, 0xcf, 0x7b, 0x60, 0x97,
			0x8c, 0xad, 0x73, 0x46, 0x50, 0xda, 0xe8, 0x6c,
			0x7e, 0x51, 0xb7, 0x66, 0x52, 0x2e, 0xf0, 0xbe,
			0x48, 0xbc, 0xea, 0xfe, 0x20, 0x30, 0x56, 0x4d,
			0x5b, 0x7b, 0x17, 0xba, 0x12, 0x50, 0x97, 0xbd,
			0xaf, 0xee, 0x48, 0xe4, 0xdf, 0x60, 0xfb, 0xb7,
			0xac, 0x2d, 0x9f, 0x14, 0xaf, 0x9a, 0x27, 0x0c,
			0x2b, 0x7e, 0xf1, 0x8c, 0xad, 0xac, 0x45, 0xb9,
			0xb5, 0x4e, 0xf2, 0x30, 0x79, 0x43, 0x39, 0xd2,
			0x79, 0xd7, 0x2b, 0xa4, 0x87, 0x83, 0xbb, 0x09,
			0xb1, 0xd1, 0xd5, 0xc1, 0xc6, 0x53, 0x01, 0x27,
			0x65, 0x23, 0xfe, 0x90, 0xe6, 0x37, 0x89, 0xff,
			0xbc, 0xd4, 0x89, 0xe4, 0x5f, 0x8a, 0xa9, 0xcf,
			0x98, 0xf3, 0x3d, 0xe8, 0xf7, 0xd9, 0xc5, 0xcd,
			0xd2, 0x1a, 0x9a, 0xb2, 0x84, 0x78, 0x96, 0xde,
			0x6d, 0xce, 0x0b, 0x92, 0xf0, 0x7b, 0x1f, 0xfb,
			0x42, 0x30, 0xee, 0x71, 0xba, 0x1f, 0xe8, 0x04,
			0x8c, 0x22, 0xdd, 0x38, 0xaf, 0x80, 0xf8, 0x76,
			0x2e, 0x74, 0x7c, 0xde, 0xc6, 0xe9, 0x9f, 0x1c,
			0xe0, 0xd1, 0xc7, 0x43, 0xef, 0x98, 0xdd, 0xba,
			0xf7, 0xc7, 0x64, 0x41, 0x24, 0x46, 0xdc, 0xa5,
			0x8e, 0x6f, 0xf5, 0xac, 0x0d, 0xd1, 0x33, 0x22,
			0x64, 0x9a, 0xcb, 0xc9, 0x6f, 0x1c, 0x5e, 0x0b,
			0xc5, 0x8d, 0x1a, 0x82, 0x11, 0x85, 0x3a, 0x7d,
			0x2f, 0x51, 0x53, 0x8c, 0x5e, 0x5e, 0x80, 0x3d,
			0xe0, 0xb1, 0x30, 0x44, 0x60, 0x8d, 0x6e, 0x65,
			0x0b, 0xac, 0xe1, 0x29, 0x45, 0xa7, 0x00, 0x81,
			0x94, 0x58, 0x6e, 0x3b, 0x74, 0x80, 0x97, 0x14,
			0xb2, 0xa5, 0x2e, 0x9f, 0x38, 0x24, 0xbe, 0x41,
			0xde, 0x9f, 0xec, 0x3f, 0x36, 0x17, 0x5a, 0x28,
			0x9b, 0xaf, 0x9f, 0xd6, 0x8b, 0x7e, 0x92, 0xf3,
			0x75, 0x4e, 0x00, 0xb4, 0x17, 0x82, 0xd0, 0x55,
			0xfa, 0xa6, 0x54, 0x33, 0xc2, 0x52, 0x59, 0xaa,
			0x65, 0x3f, 0xda, 0x06, 0x93, 0x86, 0xb0, 0x83,
			0xfb, 0x31, 0xae, 0xec, 0x8e, 0x30, 0xc7, 0x69,
			0x55, 0x3f, 0x8f, 0x03, 0x89, 0xb6, 0xe6, 0xd4,
			0xb3, 0x92, 0xca, 0xdd, 0x24, 0xce, 0x3f, 0x74,
		},
		48, /* salt length */
		/* V4 : 48 byte */
		{
			0x61, 0xa7, 0x62, 0xf8, 0x96, 0x8d, 0x5f, 0x36,
			0x7e, 0x2d, 0xbc, 0xac, 0xb4, 0x02, 0x16, 0x53,
			0xdc, 0x75, 0x43, 0x7d, 0x90, 0x00, 0xe3, 0x16,
			0x9d, 0x94, 0x37, 0x29, 0x70, 0x38, 0x37, 0xa5,
			0xcb, 0xf4, 0xde, 0x62, 0xbd, 0xed, 0xc9, 0x5f,
			0xd0, 0xd1, 0x00, 0x4e, 0x84, 0x75, 0x14, 0x52,
		},
	},
	/* SHA384 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0xb5, 0x0b, 0xf2, 0x76, 0x72, 0x50, 0xf1, 0x4f,
			0xa7, 0xb6, 0xf5, 0xea, 0x21, 0xa5, 0x4d, 0xa8,
			0xd0, 0x1e, 0x91, 0x15, 0x1e, 0xb4, 0x91, 0x10,
			0x7f, 0xd8, 0x8b, 0x2d, 0x4a, 0x5a, 0xa1, 0x57,
			0xc7, 0x2d, 0x89, 0xba, 0x89, 0x6b, 0x87, 0xe0,
			0xfe, 0x98, 0x98, 0x19, 0x44, 0x2b, 0xf0, 0x21,
			0x3e, 0x4a, 0xa7, 0xfd, 0xe8, 0xd6, 0xb0, 0x26,
			0xe7, 0xa7, 0x0a, 0xe9, 0x65, 0x19, 0x3a, 0x0e,
			0x1b, 0xc7, 0xf8, 0xb8, 0xaf, 0x96, 0x29, 0x8c,
			0x41, 0xf6, 0x0d, 0x15, 0x41, 0x64, 0xba, 0x67,
			0x83, 0x33, 0xc9, 0x03, 0x95, 0x8d, 0x4f, 0xfb,
			0x50, 0xb5, 0x0f, 0x57, 0xad, 0x8e, 0xed, 0xb6,
			0xda, 0x61, 0xa6, 0x39, 0x8d, 0xdb, 0xbf, 0x9c,
			0x99, 0x55, 0xbb, 0xa6, 0xbf, 0x59, 0x91, 0xc4,
			0xc6, 0x61, 0x5d, 0xf1, 0xcd, 0xe1, 0x56, 0xd8,
			0xe1, 0x88, 0x00, 0x3d, 0xcb, 0xc3, 0xa3, 0x99,
		},
		/* S5 : 384 byte */
		{
			0x1f, 0x06, 0x8b, 0xd0, 0x83, 0xa2, 0x65, 0x34,
			0x04, 0x0f, 0x41, 0xc1, 0x38, 0x7e, 0x71, 0xa8,
			0xc0, 0x03, 0x70, 0xc5, 0xf1, 0xc9, 0x58, 0x12,
			0x7e, 0x0b, 0xc7, 0x21, 0x75, 0x1b, 0x59, 0x40,
			0x51, 0x30, 0x23, 0xfa, 0xd0, 0x2a, 0x61, 0x01,
			0xbb, 0xce, 0xfa, 0xaa, 0xae, 0xea, 0x28, 0x75,
			0x95, 0x2b, 0xf8, 0x59, 0xd4, 0x94, 0xbf, 0xb2,
			0x3f, 0xd8, 0x91, 0x49, 0xd9, 0x12, 0x90, 0x35,
			0x9e, 0xcb, 0x44, 0xec, 0xf2, 0xfc, 0xaa, 0x57,
			0x75, 0xe2, 0xe6, 0x1e, 0x5f, 0x8d, 0x51, 0x51,
			0x34, 0x35, 0x76, 0xfe, 0x9c, 0x71, 0x67, 0xe9,
			0x19, 0xa5, 0xd0, 0x81, 0xda, 0xc6, 0xbb, 0x81,
			0x17, 0x22, 0x9c, 0x42, 0x0f, 0xd2, 0xb0, 0xfc,
			0xb5, 0x21, 0xf4, 0xe7, 0x23, 0x66, 0xbf, 0xb4,
			0x43, 0xe6, 0x88, 0xa6, 0x5f, 0xa3, 0x92, 0xea,
			0xa5, 0x11, 0x5c, 0x29, 0x2a, 0xb0, 0x5b, 0xb4,
			0xdb, 0x65, 0x46, 0x8a, 0xab, 0x26, 0x71, 0x78,
			0x65, 0x3d, 0xfa, 0x0a, 0x5e, 0xfc, 0x96, 0x06,
			0x36, 0xfc, 0xce, 0x86, 0x43, 0x35, 0x28, 0xdb,
			0xce, 0x95, 0x5a, 0x0b, 0x1a, 0xa1, 0x88, 0xac,
			0x33, 0xea, 0x12, 0x82, 0x06, 0xec, 0xc0, 0xfe,
			0xea, 0xb8, 0xf7, 0xdf, 0x6f, 0x8c, 0x38, 0x1b,
			0x10, 0x48, 0x9c, 0x8c, 0xfb, 0x2d, 0x02, 0x45,
			0x9e, 0x4c, 0xff, 0xc1, 0x6f, 0x43, 0xa6, 0x6a,
			0xa4, 0xea, 0xa1, 0x9b, 0xc5, 0x18, 0xcc, 0xfc,
			0xf9, 0xfc, 0x1e, 0x48, 0x61, 0xcf, 0xa1, 0x3e,
			0x9b, 0x41, 0xfc, 0xef, 0xad, 0xe2, 0xcd, 0x2e,
			0xbc, 0x00, 0x1e, 0xc8, 0x43, 0x0a, 0x1c, 0xb9,
			0x49, 0xa0, 0xf2, 0xf8, 0x76, 0xba, 0xdc, 0x56,
			0x8c, 0x70, 0x3e, 0x42, 0x09, 0xe7, 0xca, 0x16,
			0xf6, 0x88, 0xba, 0x97, 0x05, 0xc1, 0x4f, 0xa1,
			0xc8, 0x82, 0xe6, 0xc4, 0x87, 0x1b, 0x9d, 0xef,
			0xf3, 0x15, 0x21, 0xd2, 0xd4, 0x18, 0xe0, 0x34,
			0x2e, 0x18, 0x9c, 0x40, 0xed, 0x19, 0xc1, 0xb6,
			0xf4, 0x32, 0x0d, 0x89, 0xa3, 0x6f, 0x78, 0xec,
			0xa1, 0x43, 0xd3, 0xc1, 0x6d, 0xd3, 0xeb, 0x33,
			0x8c, 0x07, 0x43, 0x64, 0x6f, 0xd3, 0x14, 0xc7,
			0x25, 0xc2, 0xd3, 0x6a, 0x13, 0x08, 0x0b, 0xfc,
			0xde, 0xea, 0x0e, 0x43, 0x1d, 0xe7, 0x1d, 0x61,
			0xf6, 0x52, 0x03, 0x3a, 0x75, 0x42, 0x4f, 0xe1,
			0xe1, 0x58, 0x66, 0x95, 0xc3, 0xdc, 0x46, 0x3a,
			0xd5, 0x53, 0xc1, 0xcf, 0x3a, 0xb2, 0x4a, 0x41,
			0xff, 0x4e, 0x03, 0x1f, 0x9e, 0x0c, 0x2c, 0xb0,
			0x02, 0x4c, 0xef, 0x68, 0x27, 0x3e, 0xa3, 0xb8,
			0xc1, 0xbe, 0x9d, 0x92, 0x3d, 0x3e, 0x9c, 0x96,
			0x86, 0xc4, 0x19, 0x77, 0xac, 0x7b, 0xe9, 0x4a,
			0x6d, 0x23, 0x18, 0x19, 0x36, 0x13, 0x1c, 0x17,
			0xa3, 0x9a, 0x89, 0x8c, 0x94, 0x3d, 0xcc, 0x8b,
		},
		48, /* salt length */
		/* V5 : 48 byte */
		{
			0x61, 0xa7, 0x62, 0xf8, 0x96, 0x8d, 0x5f, 0x36,
			0x7e, 0x2d, 0xbc, 0xac, 0xb4, 0x02, 0x16, 0x53,
			0xdc, 0x75, 0x43, 0x7d, 0x90, 0x00, 0xe3, 0x16,
			0x9d, 0x94, 0x37, 0x29, 0x70, 0x38, 0x37, 0xa5,
			0xcb, 0xf4, 0xde, 0x62, 0xbd, 0xed, 0xc9, 0x5f,
			0xd0, 0xd1, 0x00, 0x4e, 0x84, 0x75, 0x14, 0x52,
		},
	},
	/* SHA384 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0x5f, 0xf0, 0x00, 0xf8, 0x4a, 0x95, 0x1d, 0xbf,
			0xdd, 0x63, 0x5a, 0x4d, 0x9f, 0x18, 0x91, 0xe9,
			0x4f, 0xc2, 0xa6, 0xb1, 0x1c, 0x24, 0x5f, 0x26,
			0x19, 0x5b, 0x76, 0xeb, 0xeb, 0xc2, 0xed, 0xca,
			0xc4, 0x12, 0xa2, 0xf8, 0x96, 0xce, 0x23, 0x9a,
			0x80, 0xde, 0xc3, 0x87, 0x8d, 0x79, 0xee, 0x50,
			0x9d, 0x49, 0xb9, 0x7e, 0xa3, 0xca, 0xbd, 0x1a,
			0x11, 0xf4, 0x26, 0x73, 0x91, 0x19, 0x07, 0x1b,
			0xf6, 0x10, 0xf1, 0x33, 0x72, 0x93, 0xc3, 0xe8,
			0x09, 0xe6, 0xc3, 0x3e, 0x45, 0xb9, 0xee, 0x0d,
			0x2c, 0x50, 0x8d, 0x48, 0x6f, 0xe1, 0x09, 0x85,
			0xe4, 0x3e, 0x00, 0xba, 0x36, 0xb3, 0x98, 0x45,
			0xdc, 0x32, 0x14, 0x30, 0x47, 0xad, 0xa5, 0xb2,
			0x60, 0xc4, 0x82, 0xf9, 0x31, 0xa0, 0x3a, 0x26,
			0xe2, 0x1f, 0x49, 0x9a, 0xe8, 0x31, 0xea, 0x70,
			0x79, 0x82, 0x2d, 0x4a, 0x43, 0x59, 0x49, 0x51,
		},
		/* S6 : 384 byte */
		{
			0x18, 0xcb, 0x47, 0xbb, 0xf8, 0x0b, 0xad, 0x51,
			0x00, 0x64, 0x24, 0x83, 0x04, 0x12, 0xd2, 0x81,
			0xc6, 0x6a, 0xe4, 0x5c, 0x0b, 0x75, 0x6d, 0x03,
			0xe5, 0xd8, 0xd4, 0x9f, 0x73, 0x03, 0x79, 0x68,
			0xd1, 0x3d, 0xf4, 0x6e, 0xbe, 0xbd, 0x9b, 0x5b,
			0x4c, 0x58, 0xb1, 0x64, 0xd9, 0x1d, 0x06, 0x08,
			0xe8, 0xeb, 0xe3, 0x1d, 0x86, 0x44, 0xcb, 0x0b,
			0xeb, 0xfa, 0xa8, 0xe2, 0xcc, 0xaa, 0x1f, 0x57,
			0x46, 0xac, 0x8f, 0x3b, 0xc0, 0x2f, 0xf6, 0x93,
			0x0e, 0x21, 0x9f, 0x53, 0xfe, 0x13, 0xfc, 0x07,
			0x0f, 0x91, 0x0b, 0xa1, 0xcf, 0xf0, 0x61, 0x7a,
			0xea, 0x6e, 0xb3, 0x12, 0xc1, 0xef, 0x28, 0x58,
			0x69, 0x74, 0x66, 0x73, 0xac, 0x13, 0x48, 0xe8,
			0x9c, 0x36, 0x46, 0xf5, 0x83, 0xd7, 0x63, 0x3f,
			0x5a, 0x23, 0x41, 0x62, 0x6b, 0xc2, 0xe7, 0xe2,
			0x08, 0x7f, 0xf9, 0xd8, 0xf1, 0x3d, 0x57, 0x3d,
			0xc6, 0x45, 0x5d, 0xc0, 0x06, 0x8c, 0x7a, 0xc6,
			0xea, 0xf5, 0xb3, 0x09, 0x3b, 0x08, 0x16, 0x14,
			0xf7, 0xb2, 0x52, 0x17, 0x0c, 0x48, 0x93, 0x89,
			0x1e, 0x46, 0x91, 0x21, 0xfd, 0xa6, 0x55, 0xa2,
			0xa5, 0x5d, 0x67, 0xf5, 0xdf, 0x0f, 0xf6, 0xe2,
			0x9c, 0xe5, 0xf9, 0xb0, 0xc3, 0xa1, 0xa8, 0x83,
			0x42, 0x14, 0x0e, 0xad, 0x74, 0x8e, 0xde, 0xea,
			0x97, 0x06, 0xd6, 0x57, 0x0e, 0x90, 0x0f, 0x1c,
			0xf3, 0xa9, 0xad, 0xcd, 0x7a, 0xe6, 0x4f, 0x20,
			0x75, 0x85, 0x41, 0x79, 0x46, 0xb1, 0x04, 0xb3,
			0x99, 0x0d, 0x1a, 0x2d, 0x95, 0x0e, 0x0e, 0x6a,
			0x55, 0x33, 0xd3, 0xcf, 0xc8, 0xc4, 0x70, 0x25,
			0x0e, 0x4c, 0x79, 0x72, 0x73, 0x21, 0x0f, 0x24,
			0x8b, 0x89, 0x22, 0xab, 0x00, 0x42, 0x2f, 0x2e,
			0xcf, 0x85, 0xae, 0xf7, 0x35, 0x87, 0xe8, 0xc5,
			0xcd, 0x1c, 0x2e, 0xe6, 0xed, 0x95, 0x09, 0x50,
			0x84, 0x09, 0x67, 0x3f, 0xe0, 0x7e, 0xe2, 0xc4,
			0x62, 0xc5, 0x2d, 0x09, 0x1e, 0x7a, 0x79, 0x5d,
			0x8d, 0x3c, 0x55, 0xfd, 0xd5, 0xa7, 0x10, 0xd5,
			0x45, 0x06, 0x95, 0xa5, 0xa3, 0x1e, 0xd7, 0x6f,
			0x11, 0x5e, 0x71, 0xa7, 0x3c, 0x67, 0x57, 0xd2,
			0xde, 0xf7, 0xef, 0x47, 0x25, 0x71, 0xb0, 0xbd,
			0xc7, 0x55, 0x8c, 0x71, 0xea, 0xef, 0xed, 0xde,
			0xc9, 0x46, 0x86, 0x0b, 0x0c, 0x77, 0x93, 0x6d,
			0xb3, 0x1f, 0x20, 0x01, 0xd0, 0x49, 0x9a, 0x38,
			0x1e, 0x50, 0x18, 0x87, 0x0b, 0x41, 0xba, 0x04,
			0xc8, 0xd4, 0x2e, 0xc0, 0xdc, 0x55, 0xc9, 0xfa,
			0x2a, 0xf2, 0x37, 0xdc, 0x1c, 0x40, 0x5d, 0xd8,
			0xf5, 0x55, 0xb0, 0x7a, 0x23, 0x7c, 0xc5, 0x0c,
			0xbc, 0xe4, 0x6c, 0x30, 0x16, 0x11, 0x8c, 0xf4,
			0xea, 0x06, 0xc0, 0x47, 0x59, 0x92, 0x83, 0xad,
			0x47, 0x19, 0xd6, 0x47, 0xa2, 0x25, 0x20, 0x6e,
		},
		48, /* salt length */
		/* V6 : 48 byte */
		{
			0x61, 0xa7, 0x62, 0xf8, 0x96, 0x8d, 0x5f, 0x36,
			0x7e, 0x2d, 0xbc, 0xac, 0xb4, 0x02, 0x16, 0x53,
			0xdc, 0x75, 0x43, 0x7d, 0x90, 0x00, 0xe3, 0x16,
			0x9d, 0x94, 0x37, 0x29, 0x70, 0x38, 0x37, 0xa5,
			0xcb, 0xf4, 0xde, 0x62, 0xbd, 0xed, 0xc9, 0x5f,
			0xd0, 0xd1, 0x00, 0x4e, 0x84, 0x75, 0x14, 0x52,
		},
	},
	/* SHA384 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0x53, 0x1d, 0xc2, 0xb8, 0x56, 0x6e, 0x01, 0xa8,
			0xbf, 0xc5, 0x80, 0xda, 0x60, 0x7e, 0xc2, 0x12,
			0xfc, 0x1f, 0xbe, 0xbd, 0x5a, 0x25, 0x90, 0xd8,
			0x97, 0x04, 0x6f, 0x0e, 0xc0, 0x69, 0xdf, 0x20,
			0xa1, 0xc2, 0x27, 0x8a, 0xd7, 0x00, 0x06, 0x64,
			0x2d, 0x9b, 0xa2, 0x86, 0x25, 0xd7, 0xc1, 0xef,
			0xd4, 0x47, 0x3b, 0x68, 0xf3, 0x8f, 0xb0, 0x64,
			0x34, 0x6d, 0x76, 0x2b, 0xd2, 0xfb, 0xd5, 0x37,
			0x6c, 0x2e, 0x77, 0xde, 0x13, 0xa3, 0x1a, 0x32,
			0xa2, 0x9b, 0x88, 0x26, 0x4d, 0x44, 0xc9, 0xf2,
			0x7d, 0x3a, 0x97, 0xb8, 0xdc, 0x4d, 0x12, 0x67,
			0xab, 0x85, 0xb5, 0xe0, 0x5c, 0x63, 0x89, 0x57,
			0x5d, 0x6a, 0x98, 0xfc, 0x32, 0xde, 0xa5, 0xdb,
			0xc6, 0xcc, 0x1a, 0x01, 0x03, 0x4a, 0x42, 0xe1,
			0xa0, 0x00, 0xb8, 0xf6, 0x3a, 0xe7, 0x20, 0xa9,
			0xa7, 0x51, 0x14, 0x74, 0x87, 0x2a, 0x61, 0x48,
		},
		/* S7 : 384 byte */
		{
			0x80, 0xba, 0xa6, 0x63, 0x87, 0x76, 0x15, 0xc2,
			0xe7, 0xca, 0x9d, 0xd8, 0x99, 0x58, 0xa7, 0x4e,
			0x54, 0x01, 0x2e, 0xfa, 0xd5, 0x5a, 0xd0, 0x58,
			0x68, 0xdd, 0x74, 0xb0, 0xce, 0x78, 0xa6, 0x61,
			0xe2, 0xb8, 0x93, 0xc3, 0xac, 0x1f, 0xd8, 0x37,
			0xf2, 0x82, 0x32, 0x7e, 0xfe, 0x4b, 0x04, 0x12,
			0x20, 0x94, 0x26, 0x49, 0xb5, 0x47, 0x2c, 0x1a,
			0xc7, 0x02, 0x07, 0x07, 0x87, 0xae, 0x55, 0x49,
			0x39, 0x8a, 0x57, 0x65, 0x3d, 0x5f, 0xca, 0x69,
			0xcd, 0x54, 0x46, 0xd6, 0x3f, 0x6e, 0x9d, 0x06,
			0x84, 0x92, 0x5a, 0x23, 0x5a, 0xcc, 0x96, 0xb8,
			0xa1, 0x0b, 0xdf, 0x14, 0xfb, 0xe2, 0x09, 0xfc,
			0xd4, 0x93, 0x0b, 0x59, 0x45, 0x91, 0x0d, 0x84,
			0xb0, 0x88, 0x67, 0xb2, 0x05, 0x5f, 0xe8, 0xeb,
			0x1d, 0x77, 0x1b, 0x75, 0x37, 0x59, 0x59, 0x3b,
			0x90, 0xd6, 0xae, 0xc5, 0xef, 0x18, 0x2c, 0xb3,
			0x3b, 0xf2, 0xfe, 0x29, 0xe8, 0xc6, 0x7e, 0xa4,
			0xe8, 0x43, 0x3e, 0xcf, 0xa3, 0xf9, 0xba, 0x4c,
			0xe4, 0x61, 0xf0, 0xab, 0x19, 0x99, 0x7f, 0x29,
			0x9e, 0x95, 0x40, 0x9a, 0xf9, 0x7b, 0xf5, 0x7e,
			0x2d, 0xe4, 0x10, 0xef, 0x75, 0x38, 0xf6, 0x99,
			0xf3, 0x85, 0xc1, 0xab, 0xaf, 0xdf, 0x93, 0x37,
			0xf7, 0xf9, 0xd2, 0x68, 0xda, 0x87, 0xb2, 0xb3,
			0x89, 0x13, 0x1f, 0xe3, 0xdb, 0xef, 0xd8, 0xc6,
			0x7b, 0xd2, 0xa1, 0x58, 0xcc, 0x4e, 0x04, 0xf9,
			0xab, 0x7f, 0xee, 0x2a, 0x58, 0xd7, 0x4d, 0x06,
			0x3e, 0x6c, 0x16, 0x95, 0x8a, 0x90, 0x57, 0x4e,
			0x3e, 0x4c, 0xb8, 0x81, 0xd3, 0x2c, 0x31, 0x16,
			0x98, 0x7e, 0x46, 0xbf, 0x5b, 0xd4, 0x4f, 0x80,
			0xab, 0xe6, 0xb9, 0xeb, 0x71, 0x7a, 0x9f, 0xcd,
			0x4c, 0x0c, 0xfe, 0x80, 0xdd, 0x2c, 0xa6, 0x2c,
			0x33, 0xb5, 0xdd, 0x3a, 0x59, 0xc6, 0x48, 0x10,
			0x07, 0x3e, 0x04, 0x76, 0x08, 0x5e, 0xc7, 0xb7,
			0x66, 0x38, 0x98, 0x32, 0x91, 0xb6, 0x95, 0x59,
			0xc8, 0x15, 0xcd, 0x3b, 0xb8, 0x7d, 0x4b, 0x07,
			0xe2, 0x4c, 0x6b, 0x9e, 0xbb, 0x70, 0x28, 0xe8,
			0x00, 0xa0, 0x4f, 0x09, 0xb1, 0x10, 0xc1, 0x67,
			0xf6, 0xee, 0x3a, 0x3b, 0xbb, 0x73, 0x69, 0x5d,
			0x89, 0xbe, 0xe9, 0x24, 0x07, 0xd4, 0xad, 0xce,
			0xa3, 0xea, 0xa4, 0x78, 0x11, 0xe2, 0x3f, 0x8c,
			0x7f, 0x2f, 0xdf, 0xe8, 0x91, 0xf8, 0xcf, 0xc0,
			0x71, 0xcb, 0x98, 0x4a, 0x63, 0x84, 0x6b, 0x95,
			0xec, 0x04, 0xd6, 0x26, 0x1b, 0xb1, 0xc5, 0x98,
			0x00, 0x18, 0xfe, 0xee, 0x15, 0xc4, 0xe7, 0xbf,
			0x63, 0x2d, 0xc8, 0x30, 0x61, 0x28, 0xfa, 0x22,
			0xc4, 0x7d, 0xec, 0xfd, 0x9e, 0x8b, 0x09, 0x95,
			0x54, 0xf1, 0x72, 0x53, 0x63, 0x5e, 0x63, 0x16,
			0x71, 0x2e, 0x0b, 0x95, 0xef, 0xa3, 0xfb, 0x00,
		},
		48, /* salt length */
		/* V7 : 48 byte */
		{
			0x61, 0xa7, 0x62, 0xf8, 0x96, 0x8d, 0x5f, 0x36,
			0x7e, 0x2d, 0xbc, 0xac, 0xb4, 0x02, 0x16, 0x53,
			0xdc, 0x75, 0x43, 0x7d, 0x90, 0x00, 0xe3, 0x16,
			0x9d, 0x94, 0x37, 0x29, 0x70, 0x38, 0x37, 0xa5,
			0xcb, 0xf4, 0xde, 0x62, 0xbd, 0xed, 0xc9, 0x5f,
			0xd0, 0xd1, 0x00, 0x4e, 0x84, 0x75, 0x14, 0x52,
		},
	},
	/* SHA384 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0xa4, 0x54, 0x39, 0x1a, 0x7c, 0x36, 0x95, 0x48,
			0x6c, 0x33, 0x7a, 0x41, 0xc2, 0xad, 0xd4, 0x17,
			0xd8, 0xe9, 0xe9, 0xc6, 0x46, 0x6d, 0x2e, 0xbb,
			0x56, 0xad, 0x5f, 0x97, 0xb9, 0xe7, 0xce, 0x30,
			0x78, 0x4c, 0xfc, 0xd8, 0x2d, 0x60, 0x66, 0xe3,
			0x72, 0xa3, 0xa1, 0x63, 0x9a, 0x71, 0xa9, 0x36,
			0x9f, 0x27, 0x77, 0x43, 0x5c, 0x87, 0xd1, 0x00,
			0xfc, 0x5e, 0x66, 0x38, 0xb3, 0x63, 0x1a, 0x0b,
			0xac, 0x63, 0x9f, 0x36, 0x42, 0x9b, 0x45, 0x94,
			0x72, 0x66, 0x13, 0xe5, 0x90, 0x18, 0x16, 0xcf,
			0x3a, 0x29, 0xf9, 0x22, 0x8b, 0x96, 0xd6, 0x60,
			0x90, 0x84, 0x4c, 0x7d, 0x00, 0x26, 0xd2, 0xe3,
			0x27, 0xe2, 0x4a, 0xb9, 0x24, 0xaf, 0xda, 0x65,
			0x54, 0xc2, 0xf7, 0x4f, 0x0e, 0x69, 0xc2, 0xe8,
			0x91, 0x37, 0x98, 0xec, 0x3a, 0x61, 0xe4, 0xe4,
			0xfb, 0x68, 0x38, 0xee, 0x08, 0xf8, 0x9d, 0xc0,
		},
		/* S8 : 384 byte */
		{
			0x26, 0x11, 0x80, 0x71, 0x7e, 0xdd, 0x90, 0x5b,
			0x64, 0x7b, 0xc8, 0x69, 0xf5, 0x25, 0x92, 0x03,
			0x81, 0x16, 0x06, 0x22, 0x1f, 0x54, 0x5a, 0x3a,
			0xee, 0x5f, 0xc1, 0x23, 0xf2, 0x97, 0xcf, 0x7d,
			0x8a, 0x7e, 0xe6, 0xce, 0xe3, 0xdc, 0x8f, 0x97,
			0xd2, 0x42, 0x84, 0xcc, 0xde, 0xc2, 0xfd, 0x46,
			0x80, 0xf1, 0x42, 0x8e, 0xe7, 0x57, 0x97, 0xe0,
			0x37, 0x95, 0x12, 0xae, 0xcb, 0x9f, 0xc1, 0x66,
			0x75, 0x23, 0x41, 0x3e, 0x32, 0x3c, 0x4b, 0xd7,
			0xdd, 0xed, 0x5c, 0xaf, 0x9e, 0x5c, 0x60, 0x6e,
			0x5e, 0xe0, 0xc6, 0x94, 0xd4, 0xd1, 0xb5, 0xa1,
			0xf1, 0xcb, 0x61, 0x3b, 0x98, 0x01, 0x29, 0xf6,
			0x41, 0x46, 0xe4, 0x2e, 0x82, 0x61, 0xc1, 0xf7,
			0xef, 0x56, 0x03, 0x95, 0x4d, 0x34, 0xd5, 0x6a,
			0x50, 0xf7, 0x43, 0x1b, 0xee, 0xe5, 0xab, 0x29,
			0x1a, 0x47, 0x59, 0x16, 0x86, 0x55, 0xa5, 0x12,
			0x36, 0x40, 0xd5, 0x96, 0xb7, 0x44, 0xd9, 0x79,
			0x79, 0xd3, 0x9f, 0x87, 0x4e, 0xa7, 0xff, 0x13,
			0xa7, 0x46, 0x6a, 0x76, 0x55, 0xd0, 0x2e, 0xdb,
			0x49, 0x2b, 0x58, 0x04, 0x9f, 0x22, 0x08, 0x85,
			0x22, 0x97, 0xeb, 0x02, 0x3e, 0x65, 0x7f, 0x32,
			0x40, 0xc5, 0xda, 0x9a, 0x99, 0xfd, 0x37, 0x77,
			0x28, 0xbf, 0xf3, 0xcc, 0x07, 0x31, 0x09, 0xc3,
			0x17, 0x12, 0xd9, 0x4b, 0xc2, 0x4e, 0x08, 0xc4,
			0x33, 0x53, 0x3d, 0x4b, 0x86, 0xa7, 0x3b, 0x58,
			0xfb, 0xf2, 0xc5, 0x98, 0xcc, 0xad, 0x78, 0xd4,
			0x6c, 0xa0, 0xa0, 0x55, 0x60, 0x18, 0x50, 0x96,
			0x01, 0x95, 0xaa, 0xc1, 0x36, 0x4d, 0xfa, 0xdd,
			0xbd, 0x06, 0xf1, 0x4a, 0x78, 0xaa, 0xc2, 0xab,
			0x4d, 0x37, 0x45, 0x05, 0xcc, 0x61, 0xfc, 0x72,
			0xc1, 0x05, 0x06, 0x47, 0xd9, 0x5a, 0x73, 0x35,
			0x17, 0xb7, 0x09, 0xae, 0xd2, 0xd8, 0x96, 0x72,
			0x1e, 0x74, 0x84, 0x20, 0x85, 0x01, 0x48, 0x00,
			0x58, 0xfa, 0x4f, 0x60, 0x44, 0x30, 0x2d, 0xd7,
			0x05, 0xc2, 0x73, 0xfa, 0x7f, 0xb4, 0x2e, 0xae,
			0xb0, 0x2d, 0x02, 0x50, 0x92, 0xb2, 0x52, 0xe1,
			0x6d, 0x27, 0x0d, 0x88, 0xda, 0xb6, 0xf6, 0x8f,
			0xd7, 0xad, 0x57, 0x10, 0x11, 0xf8, 0x96, 0x27,
			0x68, 0x3e, 0x02, 0x9d, 0x1b, 0xf1, 0xed, 0xc1,
			0x49, 0xd4, 0x74, 0x52, 0xeb, 0xe8, 0x7e, 0xc6,
			0x86, 0x79, 0x57, 0x99, 0x40, 0xf5, 0xae, 0xc2,
			0x59, 0x99, 0xb0, 0xde, 0xdb, 0x82, 0x0a, 0x54,
			0x83, 0xec, 0x69, 0x01, 0xab, 0xfe, 0xe0, 0x41,
			0xc0, 0x3b, 0x1a, 0x7f, 0x74, 0x35, 0x48, 0xa2,
			0xca, 0xab, 0xca, 0x61, 0x3f, 0xf5, 0xd9, 0xf8,
			0xfd, 0x7c, 0x69, 0x4a, 0xf1, 0x2b, 0x29, 0xf2,
			0xc2, 0x46, 0x8e, 0xff, 0x55, 0xf9, 0xe0, 0x08,
			0x75, 0x74, 0x43, 0x96, 0x0f, 0xae, 0x45, 0x9e,
		},
		48, /* salt length */
		/* V8 : 48 byte */
		{
			0x61, 0xa7, 0x62, 0xf8, 0x96, 0x8d, 0x5f, 0x36,
			0x7e, 0x2d, 0xbc, 0xac, 0xb4, 0x02, 0x16, 0x53,
			0xdc, 0x75, 0x43, 0x7d, 0x90, 0x00, 0xe3, 0x16,
			0x9d, 0x94, 0x37, 0x29, 0x70, 0x38, 0x37, 0xa5,
			0xcb, 0xf4, 0xde, 0x62, 0xbd, 0xed, 0xc9, 0x5f,
			0xd0, 0xd1, 0x00, 0x4e, 0x84, 0x75, 0x14, 0x52,
		},
	},
	/* SHA384 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0xa0, 0x5e, 0x57, 0x82, 0xa9, 0x6e, 0xe6, 0xd6,
			0xf1, 0x0b, 0xe8, 0x83, 0x0d, 0x8c, 0x27, 0xc0,
			0xac, 0xf2, 0x72, 0xab, 0xbf, 0x77, 0xe6, 0x84,
			0xdd, 0x6a, 0x6c, 0x19, 0xe5, 0x39, 0x83, 0x81,
			0xe5, 0xd0, 0x40, 0x0d, 0x3a, 0x21, 0x92, 0x7c,
			0xf9, 0x04, 0xcb, 0x6e, 0x8e, 0x42, 0x5c, 0x1c,
			0xa3, 0xec, 0xe0, 0x45, 0x44, 0xf2, 0x5d, 0x6c,
			0x40, 0xf0, 0xc6, 0x40, 0xd2, 0x4b, 0xc4, 0x5c,
			0x80, 0x7d, 0xb5, 0x30, 0x44, 0xad, 0xf6, 0x3f,
			0xea, 0x83, 0x5d, 0x8c, 0xb9, 0x3a, 0x0a, 0x4e,
			0x55, 0xf7, 0x60, 0xeb, 0xe4, 0x59, 0x4e, 0x24,
			0x70, 0x51, 0xd3, 0x8d, 0x8c, 0x34, 0xc1, 0x41,
			0x3b, 0x0e, 0xc1, 0xd3, 0x0d, 0x3a, 0x97, 0x88,
			0x8b, 0x2f, 0xa7, 0xc3, 0xd5, 0x9d, 0xb8, 0xc0,
			0x8a, 0xb9, 0xf9, 0x85, 0xe8, 0xd4, 0x41, 0x16,
			0x35, 0x33, 0x9b, 0xe9, 0x5d, 0x1b, 0x02, 0x99,
		},
		/* S9 : 384 byte */
		{
			0x87, 0xd8, 0x02, 0x75, 0xdf, 0x7b, 0x19, 0x6b,
			0x7e, 0x1d, 0x0a, 0x41, 0x14, 0x77, 0x19, 0xd7,
			0x73, 0xed, 0xd8, 0x0b, 0x56, 0x27, 0x30, 0x1a,
			0x50, 0x0d, 0x91, 0x66, 0x5b, 0xa8, 0x60, 0x76,
			0xe6, 0xa3, 0x1c, 0x8f, 0x3a, 0xe8, 0x6a, 0xed,
			0xb6, 0x43, 0xfe, 0x2a, 0xf2, 0x23, 0x97, 0x6e,
			0xa4, 0xeb, 0x3d, 0x4d, 0xca, 0x2c, 0xbc, 0xf8,
			0x1f, 0xfd, 0x14, 0xb7, 0xef, 0x7d, 0xe3, 0xee,
			0x35, 0x5a, 0x8d, 0x0f, 0x41, 0x43, 0xe5, 0xb0,
			0xf0, 0xa0, 0x95, 0x0a, 0x42, 0x81, 0x11, 0x02,
			0xe6, 0x02, 0xcd, 0x21, 0x4e, 0x1c, 0x94, 0x5c,
			0x47, 0xe8, 0xb7, 0xb6, 0x6d, 0x50, 0x71, 0x03,
			0xc3, 0x45, 0x6f, 0x40, 0x4f, 0x9c, 0x48, 0xaa,
			0x7f, 0xe4, 0x8d, 0xee, 0x0a, 0xad, 0x05, 0xe5,
			0x99, 0xf2, 0x42, 0xad, 0xcf, 0x8c, 0xcb, 0x0c,
			0xc9, 0xdb, 0x3a, 0x6c, 0x24, 0x4a, 0x91, 0x35,
			0x51, 0xab, 0x59, 0x56, 0x00, 0xec, 0xfb, 0xb6,
			0x7c, 0x25, 0xa9, 0x5b, 0x54, 0xf4, 0x05, 0x43,
			0x97, 0xab, 0xe4, 0x76, 0x50, 0xe5, 0xc4, 0x99,
			0x1e, 0xda, 0xf1, 0x44, 0x1b, 0xa9, 0xc8, 0xe3,
			0xfb, 0xed, 0x90, 0x4f, 0xfb, 0xc9, 0x77, 0x14,
			0x2e, 0xbd, 0xc8, 0x47, 0x69, 0x86, 0x5a, 0x21,
			0x51, 0x58, 0xd5, 0xb0, 0x52, 0xe7, 0x5d, 0xe3,
			0x18, 0xd7, 0x50, 0x12, 0x17, 0x2e, 0x28, 0xc3,
			0x1d, 0xb2, 0xd8, 0xbd, 0x4e, 0xdc, 0xa7, 0x87,
			0x21, 0x6d, 0xde, 0x2a, 0x73, 0x87, 0xc5, 0x43,
			0xf1, 0x62, 0xfc, 0x91, 0x92, 0x49, 0x18, 0xfd,
			0x6c, 0x84, 0x5b, 0xf1, 0xeb, 0xc0, 0x22, 0x0a,
			0x10, 0x27, 0xfb, 0x42, 0x27, 0x34, 0x0c, 0xa4,
			0xcb, 0x0f, 0x18, 0x3e, 0x5b, 0x34, 0xb1, 0xe7,
			0xf9, 0x3e, 0x14, 0xfa, 0x57, 0xbb, 0x9d, 0x2d,
			0x2e, 0xa5, 0x3f, 0x86, 0xd8, 0x38, 0xbc, 0xbe,
			0x3f, 0x05, 0x5b, 0x47, 0x3b, 0x0b, 0x46, 0x9a,
			0xfd, 0x29, 0x60, 0xc0, 0xd7, 0x6c, 0xe2, 0xc3,
			0x0f, 0x3d, 0x49, 0xa3, 0xb2, 0x90, 0x65, 0xbb,
			0x92, 0x60, 0x24, 0x8e, 0x72, 0x8c, 0xbe, 0x32,
			0x8b, 0xdf, 0x50, 0x2b, 0x10, 0x9e, 0x1f, 0x20,
			0xb9, 0xd0, 0x37, 0x86, 0x0c, 0xf9, 0xe2, 0x61,
			0x61, 0x1b, 0x4c, 0xbf, 0x27, 0xff, 0x9b, 0x5b,
			0xf4, 0x25, 0xb2, 0x61, 0x2a, 0xfc, 0x7c, 0xfa,
			0x31, 0x38, 0xf7, 0x8a, 0xd2, 0x60, 0x77, 0xcb,
			0xfb, 0x94, 0x7f, 0xb2, 0xaa, 0xe6, 0xf4, 0xbe,
			0x85, 0xab, 0x2d, 0x1a, 0x15, 0x86, 0x08, 0x39,
			0xb8, 0x22, 0xdd, 0x03, 0xa1, 0xa9, 0x2a, 0x19,
			0xa5, 0xc7, 0x24, 0x4e, 0x98, 0xbd, 0xf5, 0x61,
			0x62, 0x5c, 0xa2, 0xa8, 0xdf, 0x41, 0x0f, 0xf8,
			0x55, 0x75, 0x2e, 0xbd, 0xf3, 0xd4, 0x9f, 0x5e,
			0xb9, 0x8f, 0x22, 0x8a, 0xcd, 0xd5, 0x27, 0x91,
		},
		48, /* salt length */
		/* V9 : 48 byte */
		{
			0x61, 0xa7, 0x62, 0xf8, 0x96, 0x8d, 0x5f, 0x36,
			0x7e, 0x2d, 0xbc, 0xac, 0xb4, 0x02, 0x16, 0x53,
			0xdc, 0x75, 0x43, 0x7d, 0x90, 0x00, 0xe3, 0x16,
			0x9d, 0x94, 0x37, 0x29, 0x70, 0x38, 0x37, 0xa5,
			0xcb, 0xf4, 0xde, 0x62, 0xbd, 0xed, 0xc9, 0x5f,
			0xd0, 0xd1, 0x00, 0x4e, 0x84, 0x75, 0x14, 0x52,
		},
	},
};

/* 10 elements */
static mod3072_t SHA512[] = {
	/* SHA512 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0x44, 0x24, 0x0c, 0xe5, 0x19, 0xf0, 0x02, 0x39,
			0xbd, 0x66, 0xba, 0x03, 0xc8, 0x4d, 0x31, 0x60,
			0xb1, 0xce, 0x39, 0xe3, 0x93, 0x28, 0x66, 0xe5,
			0x31, 0xa6, 0x2b, 0x1c, 0x37, 0xcf, 0x41, 0x70,
			0xc3, 0xdc, 0x48, 0x09, 0x23, 0x6f, 0xb1, 0xad,
			0xe1, 0x81, 0xdb, 0x49, 0xfc, 0x9c, 0x7c, 0xcd,
			0x79, 0x4b, 0x43, 0x3d, 0x1a, 0xd0, 0xbc, 0x05,
			0x6e, 0x14, 0x73, 0x8e, 0x0a, 0xe4, 0x5c, 0x0e,
			0x15, 0x59, 0x72, 0xa4, 0x0a, 0x98, 0x9f, 0xa4,
			0xb9, 0xbc, 0xdc, 0x30, 0x8f, 0x11, 0x99, 0x08,
			0x18, 0x83, 0x5f, 0xa2, 0xc2, 0x56, 0xb4, 0x7e,
			0xe4, 0x17, 0x3f, 0xb4, 0xfe, 0xd2, 0x2c, 0xcf,
			0x43, 0x85, 0xd2, 0xdd, 0x54, 0xd5, 0x93, 0xc7,
			0x4f, 0x00, 0x04, 0xdf, 0x08, 0x13, 0x4e, 0xb8,
			0x96, 0x5d, 0xd5, 0x3a, 0x12, 0x23, 0x17, 0xf5,
			0x9b, 0x95, 0xd6, 0xb6, 0x9d, 0x01, 0x79, 0x58,
		},
		/* S0 : 384 byte */
		{
			0x8f, 0x47, 0xab, 0xc2, 0x32, 0x6e, 0x22, 0xcf,
			0x62, 0x40, 0x45, 0x08, 0xb4, 0x42, 0xe8, 0x1a,
			0xd4, 0x5a, 0xff, 0xf7, 0x27, 0x40, 0x96, 0xb9,
			0xa1, 0x3e, 0x47, 0x8c, 0xdd, 0x0a, 0x72, 0xf9,
			0x9a, 0x76, 0xbf, 0x51, 0x7f, 0x1b, 0xb0, 0xf8,
			0x72, 0xa5, 0x23, 0xd8, 0xc5, 0x88, 0xd4, 0x40,
			0x25, 0x69, 0xe9, 0x48, 0xfd, 0x6a, 0x10, 0x8a,
			0xe1, 0xa4, 0x5c, 0x65, 0x83, 0x08, 0x28, 0xa1,
			0x0e, 0x94, 0xd4, 0x32, 0x76, 0x53, 0x14, 0xba,
			0x82, 0xea, 0xd3, 0x10, 0xfc, 0x87, 0xac, 0x99,
			0xa5, 0xb3, 0x9f, 0x30, 0xab, 0x88, 0x20, 0xbf,
			0x69, 0xe6, 0x93, 0x4a, 0x9c, 0x1c, 0x91, 0x5c,
			0x19, 0xf3, 0x6e, 0xa7, 0x71, 0x7e, 0xaf, 0xf7,
			0xaf, 0x67, 0xb4, 0x99, 0x13, 0x15, 0xb1, 0x87,
			0x3b, 0xa9, 0x29, 0xbe, 0xdf, 0x18, 0xa9, 0x75,
			0xbe, 0x80, 0x8e, 0x7a, 0xa1, 0x4a, 0x67, 0x26,
			0x12, 0x6c, 0x79, 0xcc, 0x93, 0xf6, 0x95, 0x41,
			0xc5, 0xce, 0xfd, 0xeb, 0x5b, 0x67, 0xec, 0x27,
			0x9d, 0x8f, 0x5a, 0x44, 0x65, 0x83, 0xe4, 0xb4,
			0xfa, 0xed, 0x16, 0x85, 0x14, 0x0e, 0xe4, 0xb3,
			0xb7, 0x57, 0xc8, 0xff, 0x4a, 0x1e, 0xf9, 0xcd,
			0x76, 0xa8, 0x8e, 0x05, 0x31, 0x9e, 0xe6, 0x20,
			0x03, 0xd2, 0xd7, 0x72, 0x90, 0xc9, 0x4c, 0x57,
			0x9b, 0x0c, 0xa2, 0xab, 0x0d, 0xeb, 0x31, 0x76,
			0xef, 0x10, 0xa3, 0xfd, 0xb8, 0x5c, 0x80, 0xff,
			0xbc, 0x9e, 0x2a, 0x66, 0x5a, 0x23, 0x74, 0x4f,
			0xc8, 0x36, 0xf9, 0xa9, 0xa1, 0x03, 0xcd, 0x9f,
			0xb7, 0x56, 0x95, 0x23, 0x56, 0xa2, 0xf1, 0xac,
			0xdd, 0x68, 0xa6, 0x45, 0xe2, 0x01, 0x79, 0x00,
			0x65, 0x58, 0xb5, 0xd4, 0xd0, 0xb9, 0xb0, 0xbd,
			0x3a, 0xdf, 0x5e, 0x29, 0x0f, 0x49, 0xda, 0xe6,
			0x0b, 0x9d, 0x19, 0x92, 0x09, 0x53, 0xea, 0x8b,
			0xb2, 0x37, 0xd5, 0xb3, 0xdc, 0xfe, 0x14, 0x9a,
			0x60, 0xf1, 0x2a, 0x4e, 0xe3, 0xa8, 0x89, 0xb3,
			0x3b, 0xcd, 0x3a, 0x3b, 0x75, 0x3d, 0x61, 0x07,
			0x57, 0xcb, 0xcd, 0x09, 0x3d, 0xd5, 0xa7, 0x34,
			0x25, 0x53, 0x33, 0x68, 0x96, 0x95, 0xab, 0x63,
			0x69, 0x63, 0xe3, 0xd2, 0x15, 0xa8, 0xe7, 0x7f,
			0xf3, 0x19, 0x73, 0x71, 0x8a, 0x49, 0x44, 0xa1,
			0xe9, 0xe4, 0x4f, 0x45, 0x75, 0x4d, 0x39, 0xf6,
			0xfa, 0x43, 0x1c, 0x53, 0xf9, 0xa2, 0xef, 0x36,
			0xe1, 0x6a, 0x5f, 0x70, 0x63, 0x6e, 0xb5, 0xfb,
			0xa5, 0x4e, 0x15, 0xc2, 0x0a, 0x71, 0x4f, 0x28,
			0x09, 0xa7, 0xcf, 0xf4, 0xb8, 0xdc, 0x11, 0x65,
			0xf8, 0x36, 0x60, 0x7e, 0xb5, 0xa5, 0xa3, 0xbb,
			0x0c, 0x45, 0x67, 0xee, 0xe2, 0x69, 0x41, 0xfe,
			0xf4, 0x6f, 0xb4, 0x1e, 0x73, 0xb5, 0x65, 0xc0,
			0xcf, 0x8c, 0x72, 0xe4, 0x04, 0x22, 0x12, 0x64,
		},
		62, /* salt length */
		/* V0 : 62 byte */
		{
			0x2d, 0x0c, 0x49, 0xb2, 0x07, 0x89, 0xf3, 0x95,
			0x02, 0xee, 0xfd, 0x09, 0x2a, 0x2b, 0x6a, 0x9b,
			0x27, 0x57, 0xc1, 0x45, 0x61, 0x47, 0x56, 0x9a,
			0x68, 0x5f, 0xca, 0x44, 0x92, 0xa8, 0xd5, 0xb0,
			0xe6, 0x23, 0x43, 0x08, 0x38, 0x5d, 0x3d, 0x62,
			0x96, 0x44, 0xca, 0x37, 0xe3, 0x39, 0x96, 0x16,
			0xc2, 0x66, 0xf1, 0x99, 0xb6, 0x52, 0x1a, 0x99,
			0x87, 0xb2, 0xbe, 0x9e, 0xe7, 0x83, 
		},
	},
	/* SHA512 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0x06, 0xd5, 0x53, 0x4b, 0x77, 0x69, 0x25, 0x6e,
			0x8c, 0xf6, 0x5c, 0x6c, 0xe5, 0x2a, 0x3e, 0x86,
			0x96, 0x5a, 0x1f, 0xd1, 0x2c, 0x75, 0x82, 0xd2,
			0xeb, 0x36, 0x82, 0x4a, 0x5a, 0x9d, 0x70, 0x53,
			0x02, 0x9f, 0xbe, 0xac, 0x72, 0x1d, 0x1b, 0x52,
			0x86, 0x13, 0xe0, 0x50, 0xe9, 0x12, 0xab, 0xd7,
			0xd9, 0xf0, 0x49, 0x91, 0x2a, 0xbe, 0xda, 0x33,
			0x8e, 0xfa, 0x2f, 0x52, 0x13, 0x06, 0x77, 0x77,
			0xed, 0xd9, 0x1b, 0x75, 0x76, 0xf5, 0xe6, 0xfa,
			0x73, 0x98, 0x69, 0x65, 0x99, 0x37, 0x9e, 0xd7,
			0x50, 0x28, 0xcb, 0x8d, 0xb6, 0x9f, 0xa9, 0x6d,
			0xe7, 0xdb, 0xc6, 0xde, 0x7c, 0xa1, 0x28, 0xdd,
			0x51, 0xea, 0x33, 0x4e, 0x8c, 0xd9, 0xcd, 0x8f,
			0xda, 0xef, 0xbf, 0x53, 0xfc, 0x82, 0x5e, 0xae,
			0x83, 0x6b, 0x6c, 0x6c, 0xd7, 0x00, 0x39, 0xa7,
			0x7e, 0x42, 0x0d, 0x99, 0x9b, 0x57, 0xca, 0xae,
		},
		/* S1 : 384 byte */
		{
			0x91, 0x3f, 0xc1, 0x18, 0xd5, 0xac, 0x1e, 0xdf,
			0xfb, 0x4b, 0x8f, 0xcf, 0xa4, 0xe8, 0x59, 0x86,
			0xb4, 0x62, 0x31, 0xce, 0xf3, 0xda, 0xd9, 0x11,
			0xd5, 0xe9, 0x53, 0x4c, 0xc8, 0x82, 0x61, 0xf6,
			0xb6, 0x96, 0x9b, 0x75, 0xa3, 0xf2, 0x5d, 0x83,
			0xec, 0xe7, 0xec, 0x20, 0x34, 0xb0, 0x1d, 0x3b,
			0x2b, 0xe6, 0xc5, 0xbd, 0x95, 0x8c, 0xc4, 0xaf,
			0xcd, 0x44, 0x83, 0x9e, 0x39, 0x53, 0xf0, 0x1e,
			0x4a, 0x15, 0xea, 0x5e, 0xf6, 0xe1, 0xb4, 0xb0,
			0xe8, 0xae, 0x90, 0xbd, 0xfd, 0x40, 0x41, 0x99,
			0xe8, 0xf8, 0x65, 0x47, 0xf6, 0x7f, 0xf6, 0xb8,
			0x4f, 0x21, 0x62, 0xc4, 0x31, 0x1c, 0xc9, 0xee,
			0xe0, 0x6b, 0xfb, 0x2f, 0xe4, 0x61, 0x98, 0xaf,
			0xb9, 0x74, 0x5d, 0x9c, 0x44, 0x38, 0x33, 0xbf,
			0x23, 0x87, 0xeb, 0x92, 0x40, 0x6a, 0x63, 0x39,
			0x52, 0x13, 0x96, 0xf2, 0xcb, 0xda, 0x55, 0xd9,
			0x8f, 0xe6, 0x40, 0x74, 0xd2, 0xf2, 0xe2, 0x7b,
			0x8b, 0xc6, 0xa7, 0x9b, 0xe3, 0xd1, 0xcc, 0x56,
			0x88, 0x69, 0xb0, 0xb5, 0x0f, 0xcb, 0xf7, 0x02,
			0xb0, 0x83, 0x16, 0x68, 0xfb, 0xfd, 0xed, 0xc2,
			0xd1, 0xb5, 0x49, 0x1e, 0x8e, 0xc6, 0x23, 0xed,
			0xeb, 0x60, 0xac, 0x87, 0x0e, 0x6e, 0x8d, 0x05,
			0x85, 0x93, 0xfb, 0xbc, 0x93, 0x8f, 0xbf, 0x74,
			0x17, 0x00, 0xef, 0xc2, 0xb2, 0x46, 0x7e, 0x7e,
			0xb2, 0x54, 0xae, 0x00, 0x85, 0x09, 0xe9, 0x16,
			0x07, 0xf8, 0xe5, 0x0a, 0xa1, 0x6a, 0x4e, 0x85,
			0x1a, 0xbc, 0xa7, 0xc8, 0xd2, 0x0c, 0x6f, 0xf6,
			0x1c, 0xfe, 0xe6, 0xc1, 0xfb, 0x67, 0x60, 0x98,
			0xe5, 0xcd, 0xf1, 0x27, 0xc9, 0xb7, 0x95, 0x38,
			0xfd, 0x1e, 0x6c, 0x01, 0x41, 0x61, 0x05, 0x4c,
			0xaf, 0x43, 0xb7, 0x34, 0xfa, 0x69, 0xfe, 0x06,
			0xa0, 0x0d, 0x76, 0xf7, 0x10, 0xac, 0xc1, 0x98,
			0xf3, 0xda, 0x90, 0x6a, 0x7d, 0x2e, 0x73, 0xa2,
			0xca, 0x88, 0x25, 0x26, 0xcc, 0x35, 0x4d, 0xd7,
			0x63, 0x0a, 0x30, 0x3d, 0x8f, 0x32, 0xc6, 0x55,
			0xb5, 0xb3, 0x3c, 0xf7, 0x88, 0x59, 0xbe, 0xea,
			0xba, 0x3f, 0x9a, 0xe0, 0x52, 0xc8, 0xd7, 0x47,
			0x1c, 0xd2, 0xbd, 0x9e, 0xdf, 0x42, 0xfd, 0x8f,
			0x70, 0xc3, 0xb0, 0xaa, 0x79, 0xc0, 0x76, 0x92,
			0x80, 0x68, 0xca, 0x97, 0x70, 0x95, 0x9a, 0xfa,
			0x63, 0x2c, 0xa6, 0xaa, 0xba, 0x66, 0x79, 0xe4,
			0x5d, 0x68, 0x88, 0xc5, 0x01, 0x25, 0xa7, 0x3b,
			0x9d, 0xeb, 0x00, 0xd4, 0x2a, 0x12, 0x5f, 0x25,
			0xdf, 0x54, 0x34, 0xbe, 0xff, 0x0d, 0x5b, 0x0e,
			0xe1, 0x3a, 0x16, 0xb1, 0x70, 0x45, 0xce, 0xce,
			0x0f, 0x2d, 0xa7, 0x57, 0x7d, 0x79, 0xd7, 0xcd,
			0x75, 0xa4, 0xb6, 0xc5, 0xbc, 0x34, 0x5f, 0x46,
			0x0a, 0x17, 0x34, 0x87, 0xb5, 0x1b, 0xc6, 0xa6,
		},
		62, /* salt length */
		/* V1 : 62 byte */
		{
			0x2d, 0x0c, 0x49, 0xb2, 0x07, 0x89, 0xf3, 0x95,
			0x02, 0xee, 0xfd, 0x09, 0x2a, 0x2b, 0x6a, 0x9b,
			0x27, 0x57, 0xc1, 0x45, 0x61, 0x47, 0x56, 0x9a,
			0x68, 0x5f, 0xca, 0x44, 0x92, 0xa8, 0xd5, 0xb0,
			0xe6, 0x23, 0x43, 0x08, 0x38, 0x5d, 0x3d, 0x62,
			0x96, 0x44, 0xca, 0x37, 0xe3, 0x39, 0x96, 0x16,
			0xc2, 0x66, 0xf1, 0x99, 0xb6, 0x52, 0x1a, 0x99,
			0x87, 0xb2, 0xbe, 0x9e, 0xe7, 0x83, 
		},
	},
	/* SHA512 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x75, 0x6c, 0x51, 0xba, 0xe6, 0x1d, 0x75, 0xe8,
			0xcf, 0x44, 0x93, 0x0e, 0x17, 0x81, 0xdd, 0x6b,
			0x8d, 0xb6, 0xbf, 0x8b, 0x1f, 0x68, 0xb4, 0xca,
			0x4c, 0x68, 0x5d, 0x14, 0xdc, 0xb2, 0xd4, 0xee,
			0xce, 0x95, 0x3e, 0xba, 0x92, 0x14, 0x9f, 0x36,
			0x78, 0x8d, 0xf3, 0x47, 0x69, 0x98, 0x7a, 0xf5,
			0xd5, 0x32, 0x53, 0xb6, 0xec, 0x1b, 0x4c, 0xef,
			0x11, 0x7c, 0xf9, 0xb8, 0x8b, 0xcd, 0x03, 0xe0,
			0x7e, 0xf6, 0xc3, 0x30, 0x1a, 0xb4, 0x0f, 0xf4,
			0x13, 0x3f, 0x54, 0xb8, 0x51, 0x2a, 0xe5, 0x50,
			0xe8, 0x8a, 0x93, 0x1b, 0x4a, 0x5a, 0x7e, 0x88,
			0xbc, 0x1e, 0x2b, 0xd8, 0x06, 0xc7, 0xd6, 0x26,
			0x6f, 0xd7, 0x09, 0xa5, 0xe8, 0xc5, 0x6d, 0x2a,
			0x88, 0xa3, 0xe1, 0xea, 0x38, 0xfe, 0xc9, 0x84,
			0xb0, 0x06, 0xa8, 0x42, 0xa2, 0xee, 0xf2, 0x9b,
			0x34, 0x96, 0x1b, 0xfd, 0xb4, 0x68, 0xf4, 0xca,
		},
		/* S2 : 384 byte */
		{
			0x73, 0x51, 0x86, 0xeb, 0xf0, 0x8d, 0x50, 0x51,
			0x61, 0xa8, 0xba, 0xb3, 0x67, 0x86, 0x13, 0x84,
			0x14, 0xbb, 0x5c, 0xa2, 0xf4, 0x02, 0x52, 0x89,
			0xaf, 0x23, 0x7a, 0x40, 0xf8, 0xd0, 0x96, 0x3d,
			0xf9, 0x11, 0x7b, 0x61, 0x9f, 0x83, 0xd9, 0xa9,
			0x8d, 0xfc, 0xf7, 0x4b, 0x8f, 0x00, 0x1a, 0x4a,
			0x74, 0x2c, 0x85, 0xae, 0x01, 0x8c, 0x3b, 0x51,
			0xf1, 0x6e, 0xb5, 0x01, 0x5b, 0xa7, 0x02, 0x7c,
			0xb9, 0xa0, 0xd0, 0xb9, 0xe6, 0xb6, 0x5c, 0x08,
			0xba, 0x58, 0xb6, 0x71, 0xa9, 0xb3, 0xdd, 0x62,
			0x10, 0x7b, 0xbd, 0x5a, 0xe9, 0x32, 0x78, 0x4d,
			0x32, 0x8c, 0xdb, 0x2e, 0x1a, 0x55, 0x1e, 0xb6,
			0x7e, 0x9d, 0x33, 0xff, 0x1c, 0xf9, 0xbf, 0xfd,
			0xb2, 0x23, 0xaf, 0xd7, 0x5d, 0x36, 0x50, 0x45,
			0x9f, 0xdb, 0x58, 0x14, 0x3c, 0xd4, 0x49, 0x09,
			0x81, 0xef, 0xb0, 0xb3, 0xfe, 0x36, 0xf6, 0x42,
			0xe1, 0x83, 0x7a, 0x5d, 0x95, 0xc3, 0xd4, 0x44,
			0xaf, 0x73, 0x72, 0x9d, 0xd1, 0xa5, 0xe9, 0x93,
			0x7b, 0x81, 0x14, 0xa2, 0x8e, 0x06, 0x5d, 0x10,
			0x81, 0xf0, 0x61, 0x04, 0x9e, 0x65, 0x0e, 0x45,
			0xff, 0x5c, 0xcf, 0x75, 0xc2, 0x46, 0xe2, 0xe9,
			0x43, 0x3b, 0x27, 0xe7, 0x9a, 0x1b, 0x06, 0xf7,
			0xb6, 0xb5, 0x7f, 0x9b, 0x00, 0x9e, 0x97, 0x16,
			0x8a, 0x61, 0x29, 0x7c, 0xfd, 0x0a, 0x81, 0x56,
			0xd0, 0x26, 0xa6, 0xbf, 0x8c, 0x37, 0x64, 0xd0,
			0xb7, 0x15, 0xc6, 0x19, 0xd8, 0x56, 0xb0, 0x61,
			0xdf, 0x35, 0x72, 0x54, 0x98, 0xd8, 0x6c, 0xec,
			0x25, 0xf7, 0xe1, 0xda, 0x65, 0xb9, 0x9d, 0x9e,
			0xcb, 0xb9, 0xa1, 0xa6, 0x36, 0x42, 0x52, 0xe4,
			0x79, 0x0d, 0x97, 0xea, 0x0f, 0xfd, 0x62, 0x34,
			0xb5, 0x15, 0x92, 0x9b, 0x5e, 0xf2, 0x26, 0x76,
			0xc2, 0x43, 0xd3, 0x86, 0xeb, 0xb9, 0x0a, 0x22,
			0xe6, 0x7a, 0x0e, 0x1d, 0x10, 0x94, 0xdd, 0xdf,
			0x77, 0x21, 0x09, 0x98, 0x68, 0xc3, 0x13, 0x26,
			0x81, 0x48, 0x87, 0xb6, 0x46, 0xca, 0x52, 0xa2,
			0xc4, 0xbc, 0xd4, 0x3f, 0x7c, 0x71, 0x39, 0x9e,
			0x7d, 0x13, 0xe1, 0x9d, 0xe6, 0x88, 0xae, 0x5c,
			0x20, 0x46, 0x3d, 0xf5, 0x96, 0x5d, 0x82, 0x55,
			0xa3, 0xe6, 0x92, 0x8d, 0x61, 0x4b, 0x60, 0x12,
			0x74, 0xb7, 0x57, 0xcf, 0xac, 0xdd, 0x40, 0x02,
			0xd9, 0xba, 0x8b, 0x24, 0x8a, 0xe7, 0x00, 0xd8,
			0x77, 0x64, 0x75, 0xd7, 0x9d, 0x0a, 0x55, 0xed,
			0x42, 0x41, 0xc9, 0x91, 0x9a, 0x3c, 0x44, 0xdf,
			0xb9, 0xa1, 0xf5, 0xd0, 0xfe, 0xc7, 0xca, 0x34,
			0x17, 0x74, 0xc5, 0x96, 0x14, 0x4c, 0x38, 0x17,
			0x4a, 0xf5, 0x9a, 0xf6, 0xde, 0xb8, 0x93, 0x7a,
			0x7d, 0x14, 0xc4, 0x59, 0xb5, 0xd7, 0x68, 0xa9,
			0x77, 0x44, 0x5d, 0xaf, 0xee, 0x1a, 0x4e, 0xeb,
		},
		62, /* salt length */
		/* V2 : 62 byte */
		{
			0x2d, 0x0c, 0x49, 0xb2, 0x07, 0x89, 0xf3, 0x95,
			0x02, 0xee, 0xfd, 0x09, 0x2a, 0x2b, 0x6a, 0x9b,
			0x27, 0x57, 0xc1, 0x45, 0x61, 0x47, 0x56, 0x9a,
			0x68, 0x5f, 0xca, 0x44, 0x92, 0xa8, 0xd5, 0xb0,
			0xe6, 0x23, 0x43, 0x08, 0x38, 0x5d, 0x3d, 0x62,
			0x96, 0x44, 0xca, 0x37, 0xe3, 0x39, 0x96, 0x16,
			0xc2, 0x66, 0xf1, 0x99, 0xb6, 0x52, 0x1a, 0x99,
			0x87, 0xb2, 0xbe, 0x9e, 0xe7, 0x83, 
		},
	},
	/* SHA512 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0xa9, 0x57, 0x9c, 0xce, 0x61, 0x9e, 0xba, 0xde,
			0x34, 0x5e, 0x10, 0x5a, 0x92, 0x14, 0xb9, 0x38,
			0xa2, 0x1f, 0x2b, 0x71, 0x91, 0xc4, 0x21, 0x1b,
			0x2b, 0x75, 0xd9, 0xd2, 0xa8, 0x53, 0x80, 0x5d,
			0xc8, 0xf1, 0xeb, 0x8f, 0x22, 0x5b, 0x87, 0x6a,
			0xb8, 0x57, 0x93, 0x8b, 0xd0, 0xea, 0x8c, 0xc2,
			0xff, 0x1e, 0xe9, 0x00, 0x87, 0x03, 0x09, 0x76,
			0xe3, 0xf4, 0x6a, 0xfb, 0x9f, 0x1b, 0x1b, 0xae,
			0x6d, 0x38, 0x74, 0xdd, 0x76, 0x9d, 0x04, 0x26,
			0xee, 0x7d, 0xcb, 0xdc, 0xeb, 0x67, 0xa9, 0xad,
			0x77, 0x0e, 0x17, 0x81, 0xe3, 0x4b, 0x15, 0xa4,
			0x5f, 0x65, 0x63, 0x28, 0xc8, 0x8f, 0xf4, 0x85,
			0xc1, 0xb2, 0xa0, 0x83, 0x05, 0x6d, 0x19, 0x5a,
			0xfc, 0x5b, 0x20, 0x17, 0x8c, 0x94, 0xf9, 0x41,
			0x31, 0x76, 0x1c, 0xbd, 0x50, 0xa5, 0x2d, 0xef,
			0xc8, 0x50, 0x2e, 0x22, 0xcb, 0xb6, 0xf4, 0x2a,
		},
		/* S3 : 384 byte */
		{
			0x60, 0x3f, 0xf6, 0x3f, 0xf6, 0x38, 0xf1, 0xad,
			0x41, 0x0e, 0x26, 0x6d, 0x82, 0xa0, 0x4c, 0x6d,
			0x47, 0x54, 0x16, 0xa0, 0x47, 0x0d, 0x97, 0xf4,
			0x83, 0xc0, 0xc9, 0x9e, 0x8f, 0xc7, 0x21, 0x2d,
			0x61, 0xe0, 0x2c, 0xc8, 0xb4, 0x49, 0x3c, 0x9a,
			0x9d, 0xac, 0x71, 0x1d, 0x2a, 0x8e, 0xdf, 0x19,
			0x6a, 0x26, 0x56, 0x38, 0x66, 0xd6, 0x8f, 0xb0,
			0x48, 0x49, 0xe8, 0x2d, 0xb0, 0xf9, 0x74, 0x1f,
			0x72, 0x1f, 0x2b, 0xa4, 0xe9, 0xdb, 0x62, 0xf6,
			0xec, 0xfe, 0x3b, 0x87, 0xeb, 0xe7, 0xfe, 0xed,
			0x0c, 0x9e, 0x2d, 0xd4, 0x6c, 0x3f, 0x92, 0x52,
			0xd4, 0xc1, 0x22, 0xc6, 0xbf, 0x1b, 0xf4, 0xce,
			0x21, 0x5b, 0xa8, 0x2f, 0xe7, 0xc5, 0xa9, 0x12,
			0x49, 0xda, 0x70, 0xdd, 0x30, 0xfc, 0x9c, 0x8a,
			0xc8, 0xb3, 0xbb, 0x28, 0x10, 0xb4, 0xff, 0x38,
			0xbf, 0xac, 0xc1, 0x3f, 0xd4, 0x1f, 0x6f, 0xa2,
			0x65, 0x07, 0xa0, 0x55, 0xe0, 0xf1, 0x24, 0x2f,
			0x18, 0xea, 0x8e, 0xd8, 0xa7, 0x02, 0xd2, 0x65,
			0xf8, 0x93, 0xcb, 0x4e, 0xb6, 0x1a, 0x3d, 0xc8,
			0xe1, 0x87, 0x77, 0x15, 0x75, 0x52, 0xa1, 0xc5,
			0x8d, 0xb1, 0x43, 0x49, 0xa0, 0xd0, 0xa2, 0xa9,
			0x00, 0xa0, 0xa1, 0xf4, 0xde, 0x86, 0x3f, 0xba,
			0xdb, 0x06, 0x3a, 0xd2, 0xa9, 0xe5, 0x26, 0xa0,
			0xa8, 0xc3, 0xbd, 0xcf, 0xca, 0x55, 0x24, 0xc1,
			0x81, 0x63, 0x7b, 0x1c, 0x4a, 0x57, 0x48, 0x09,
			0xfb, 0x45, 0xb2, 0xe4, 0xf0, 0x6f, 0x3f, 0x89,
			0xf4, 0xcc, 0xfb, 0x30, 0x21, 0x7b, 0x32, 0xfc,
			0x48, 0x4b, 0xb9, 0x08, 0x27, 0x6d, 0x65, 0x9a,
			0x0d, 0x9a, 0x3e, 0x7e, 0x3f, 0xbd, 0x46, 0x56,
			0x5a, 0x09, 0x24, 0xf9, 0x18, 0xb1, 0x6b, 0x2d,
			0x65, 0x27, 0xec, 0x4b, 0x5d, 0x1d, 0x6e, 0xf6,
			0xd6, 0x72, 0x0f, 0x3e, 0x00, 0x48, 0x5e, 0x87,
			0xde, 0x61, 0xed, 0x49, 0xed, 0x13, 0xe8, 0x5c,
			0xa6, 0xa1, 0x0d, 0x46, 0xd4, 0xca, 0x48, 0x39,
			0xf4, 0x86, 0x62, 0x1c, 0xca, 0x48, 0xa7, 0xf9,
			0x55, 0xa8, 0x78, 0xc4, 0x78, 0x5d, 0x55, 0xde,
			0x96, 0xfa, 0xcb, 0xb9, 0x1b, 0x6e, 0xa1, 0x2e,
			0x9e, 0x4f, 0xe4, 0xbe, 0xed, 0x00, 0x14, 0x1b,
			0x03, 0x72, 0xa3, 0x81, 0x24, 0x65, 0xe6, 0x50,
			0x30, 0xf4, 0xfb, 0x8d, 0xdd, 0x58, 0x70, 0x1a,
			0xa3, 0xda, 0x27, 0xd2, 0x6f, 0xeb, 0x86, 0x44,
			0xf7, 0xc8, 0x0b, 0x8e, 0xe2, 0xa3, 0xc3, 0xb2,
			0x0a, 0x51, 0x6c, 0x7f, 0x0b, 0x06, 0x8b, 0x50,
			0x3f, 0xbb, 0x65, 0xd3, 0xf3, 0xb8, 0x4b, 0x25,
			0x34, 0x66, 0xa8, 0x87, 0x31, 0x4a, 0xa8, 0xeb,
			0x9d, 0x85, 0xcd, 0x03, 0x5b, 0xf8, 0xdb, 0xb1,
			0x78, 0xeb, 0xd8, 0xd5, 0x49, 0x6f, 0xd1, 0xb6,
			0x84, 0x32, 0x45, 0x7c, 0x78, 0xc6, 0x9c, 0xad,
		},
		62, /* salt length */
		/* V3 : 62 byte */
		{
			0x2d, 0x0c, 0x49, 0xb2, 0x07, 0x89, 0xf3, 0x95,
			0x02, 0xee, 0xfd, 0x09, 0x2a, 0x2b, 0x6a, 0x9b,
			0x27, 0x57, 0xc1, 0x45, 0x61, 0x47, 0x56, 0x9a,
			0x68, 0x5f, 0xca, 0x44, 0x92, 0xa8, 0xd5, 0xb0,
			0xe6, 0x23, 0x43, 0x08, 0x38, 0x5d, 0x3d, 0x62,
			0x96, 0x44, 0xca, 0x37, 0xe3, 0x39, 0x96, 0x16,
			0xc2, 0x66, 0xf1, 0x99, 0xb6, 0x52, 0x1a, 0x99,
			0x87, 0xb2, 0xbe, 0x9e, 0xe7, 0x83, 
		},
	},
	/* SHA512 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0xc3, 0x28, 0x7c, 0x23, 0xb6, 0x13, 0xae, 0xfc,
			0x24, 0x25, 0xa8, 0xb8, 0x31, 0x7d, 0x64, 0x7a,
			0x44, 0x78, 0x16, 0xba, 0xc5, 0x6d, 0x0c, 0x99,
			0x25, 0x9b, 0xd9, 0x71, 0x1f, 0x5f, 0xb2, 0xb1,
			0x3e, 0xab, 0x18, 0xe8, 0xa0, 0xb3, 0xb8, 0x1f,
			0xf9, 0xe9, 0x8f, 0x6c, 0xda, 0x2c, 0x51, 0xc4,
			0x34, 0x3c, 0x0c, 0x11, 0x18, 0x72, 0x08, 0x84,
			0xc0, 0xae, 0xf3, 0x2d, 0xd3, 0x90, 0x3a, 0xc9,
			0xe5, 0xeb, 0xba, 0xdb, 0x3d, 0x76, 0x98, 0xfe,
			0xdc, 0xc5, 0x6d, 0x79, 0xbb, 0x78, 0xa7, 0x14,
			0x53, 0xb3, 0x2c, 0x2a, 0x62, 0xce, 0x40, 0x00,
			0xed, 0x4d, 0xa8, 0x55, 0x81, 0x12, 0x0f, 0x3a,
			0xbf, 0xd1, 0xaa, 0x24, 0x18, 0xc5, 0x18, 0x40,
			0xd4, 0xa1, 0x8c, 0x06, 0x59, 0xca, 0x2d, 0x11,
			0xaa, 0xc3, 0xbd, 0x2e, 0x2e, 0xe8, 0x79, 0xb3,
			0xb3, 0x60, 0x41, 0x12, 0xb2, 0x4d, 0xf9, 0xad,
		},
		/* S4 : 384 byte */
		{
			0x87, 0x8b, 0x9a, 0x44, 0x39, 0x21, 0xbc, 0x7d,
			0x72, 0x0e, 0x3e, 0x28, 0x8e, 0x8f, 0x39, 0xe5,
			0x50, 0x11, 0x3e, 0x01, 0xd0, 0x4f, 0xb1, 0x63,
			0x5a, 0x26, 0xf7, 0x96, 0xfb, 0x8b, 0x16, 0x1d,
			0x5b, 0x75, 0x8c, 0xff, 0x91, 0x4a, 0x24, 0x41,
			0xd8, 0x35, 0x0f, 0x8d, 0x39, 0x22, 0xaa, 0x56,
			0x15, 0xed, 0xfd, 0x86, 0x50, 0x1c, 0x9a, 0x05,
			0xc2, 0x10, 0xc9, 0x3a, 0x1a, 0xe0, 0x4f, 0xf7,
			0x61, 0x15, 0x1d, 0xc8, 0xd6, 0x52, 0xfb, 0x55,
			0x09, 0xed, 0x10, 0x09, 0x99, 0xd2, 0xbf, 0x6e,
			0x40, 0xb1, 0xbb, 0xb6, 0x4c, 0xf6, 0xc5, 0xd8,
			0xe0, 0x67, 0xb4, 0x45, 0xda, 0xf5, 0x67, 0x13,
			0x7c, 0xb8, 0xf0, 0x86, 0x39, 0x96, 0xde, 0x8d,
			0xe9, 0xa6, 0x47, 0xf9, 0x82, 0xc9, 0xe2, 0x1a,
			0x78, 0x7e, 0xe8, 0xd7, 0x26, 0x57, 0xa2, 0xdd,
			0x42, 0xec, 0x9f, 0xec, 0x49, 0xea, 0x1c, 0x33,
			0x45, 0xcf, 0x00, 0x4e, 0x94, 0x59, 0x4a, 0x06,
			0x4b, 0x6b, 0x6b, 0x22, 0x28, 0x45, 0xd6, 0x4c,
			0x93, 0x5b, 0x53, 0x9d, 0x3f, 0xd2, 0xd5, 0x35,
			0xfe, 0x0e, 0x47, 0xac, 0x67, 0x46, 0x02, 0x8e,
			0x74, 0x85, 0x56, 0xc2, 0xd8, 0x8e, 0x4d, 0x40,
			0x70, 0x7e, 0x74, 0xa1, 0xc0, 0xca, 0xd5, 0xcd,
			0x95, 0xda, 0xd2, 0x63, 0xef, 0xd3, 0xca, 0x63,
			0x7a, 0xc6, 0xb8, 0xf7, 0x8d, 0xdf, 0x7b, 0xa8,
			0x1e, 0x44, 0x3b, 0x83, 0x6d, 0x85, 0xa8, 0x3d,
			0xbe, 0x84, 0x3b, 0xd6, 0x27, 0x1e, 0x45, 0xd8,
			0x42, 0xe1, 0xbb, 0x24, 0x1c, 0x9c, 0x18, 0x80,
			0x5f, 0x37, 0xbc, 0x19, 0x83, 0x8b, 0xa2, 0xbc,
			0x6c, 0xd3, 0x84, 0x01, 0xdc, 0xe0, 0xcc, 0x97,
			0x80, 0x30, 0x6e, 0xa8, 0xa8, 0x7d, 0x43, 0x11,
			0x0b, 0x3e, 0x39, 0x5b, 0xbf, 0xb8, 0x1c, 0x3b,
			0xa4, 0x5c, 0xe1, 0xcd, 0x71, 0x59, 0x6e, 0xd2,
			0x7c, 0x03, 0xe2, 0x09, 0x0a, 0x7e, 0xe8, 0x1f,
			0x60, 0x11, 0x9e, 0x18, 0x7a, 0xdf, 0xf0, 0xd9,
			0x6a, 0xcf, 0xba, 0xac, 0x38, 0xf7, 0xcb, 0x50,
			0x30, 0x39, 0xea, 0xd9, 0xcf, 0x95, 0x50, 0xde,
			0xd5, 0x69, 0x3d, 0x3c, 0x25, 0x74, 0x06, 0xdd,
			0x0b, 0xc0, 0x61, 0xd4, 0x51, 0xbd, 0x81, 0xd6,
			0x4f, 0x96, 0x9b, 0x7c, 0x2b, 0x84, 0x61, 0x9f,
			0x0d, 0xd8, 0x24, 0x81, 0x78, 0x1e, 0xaf, 0x5b,
			0x8f, 0xc8, 0x2a, 0x3a, 0xc5, 0xb9, 0xfc, 0x20,
			0xb4, 0x2f, 0x86, 0xd4, 0x22, 0x5a, 0x43, 0x5b,
			0x90, 0x3d, 0x22, 0x58, 0xf5, 0xcf, 0x69, 0x3d,
			0x1b, 0x5c, 0x6a, 0x5d, 0x14, 0x4f, 0x7f, 0x4e,
			0xab, 0x9e, 0x70, 0xde, 0x2f, 0x38, 0x79, 0xf6,
			0x8e, 0x4c, 0x1c, 0x7a, 0x38, 0xdd, 0xa6, 0x3e,
			0x61, 0x86, 0x53, 0x4f, 0xcd, 0x78, 0xd5, 0x8d,
			0xb7, 0x09, 0xbf, 0x57, 0xa7, 0x8a, 0x84, 0x8c,
		},
		62, /* salt length */
		/* V4 : 62 byte */
		{
			0x2d, 0x0c, 0x49, 0xb2, 0x07, 0x89, 0xf3, 0x95,
			0x02, 0xee, 0xfd, 0x09, 0x2a, 0x2b, 0x6a, 0x9b,
			0x27, 0x57, 0xc1, 0x45, 0x61, 0x47, 0x56, 0x9a,
			0x68, 0x5f, 0xca, 0x44, 0x92, 0xa8, 0xd5, 0xb0,
			0xe6, 0x23, 0x43, 0x08, 0x38, 0x5d, 0x3d, 0x62,
			0x96, 0x44, 0xca, 0x37, 0xe3, 0x39, 0x96, 0x16,
			0xc2, 0x66, 0xf1, 0x99, 0xb6, 0x52, 0x1a, 0x99,
			0x87, 0xb2, 0xbe, 0x9e, 0xe7, 0x83, 
		},
	},
	/* SHA512 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0xd5, 0x4c, 0x51, 0xf9, 0x0b, 0x27, 0x8c, 0x1c,
			0x60, 0x2b, 0xb5, 0x4a, 0x23, 0x41, 0x9a, 0x62,
			0xc2, 0xe8, 0x52, 0x72, 0x29, 0x35, 0x2e, 0xd7,
			0x4a, 0x17, 0xed, 0xa6, 0xfd, 0xe0, 0x2f, 0x4b,
			0x0b, 0x01, 0x2d, 0x70, 0x85, 0x15, 0xa6, 0x21,
			0x5b, 0x22, 0x1d, 0x2d, 0x29, 0x1b, 0x41, 0xcf,
			0x54, 0xa9, 0xad, 0x8d, 0x56, 0x2a, 0xd1, 0x61,
			0x56, 0xfb, 0x30, 0x17, 0xfc, 0xf2, 0xcd, 0xf6,
			0x83, 0x2f, 0xdf, 0xa2, 0x10, 0x15, 0xcc, 0x41,
			0x42, 0x93, 0x55, 0xdd, 0x0a, 0xa8, 0x0e, 0x09,
			0xbd, 0x26, 0x12, 0xc8, 0x67, 0xb6, 0xf4, 0xaa,
			0x63, 0x1c, 0xf9, 0x38, 0x28, 0xbc, 0x84, 0x92,
			0x66, 0x5d, 0xd1, 0x57, 0x52, 0x2e, 0xe6, 0xc5,
			0x3d, 0x06, 0xc7, 0x22, 0x6c, 0xf0, 0xea, 0x5a,
			0x24, 0xe7, 0xea, 0xe9, 0x04, 0xde, 0x7f, 0xfb,
			0x98, 0x04, 0xae, 0xd2, 0x2a, 0x45, 0x3d, 0x69,
		},
		/* S5 : 384 byte */
		{
			0x26, 0x57, 0x49, 0xf7, 0xaf, 0xb1, 0xe1, 0xd1,
			0x64, 0x92, 0xee, 0xbc, 0xee, 0x9f, 0x50, 0x04,
			0x23, 0x4e, 0x1d, 0xcb, 0x95, 0xb8, 0x32, 0xd1,
			0x41, 0x65, 0x99, 0x2f, 0x4d, 0x1c, 0x49, 0xd5,
			0x18, 0xba, 0x15, 0xa6, 0xb3, 0xad, 0xed, 0xfd,
			0x80, 0x32, 0x87, 0xcf, 0x60, 0xce, 0x8c, 0x91,
			0x58, 0x82, 0xe2, 0xc7, 0x8d, 0x69, 0xff, 0xc4,
			0x6f, 0xde, 0xce, 0xf0, 0x08, 0xe5, 0xd7, 0xf1,
			0x46, 0xe3, 0x8f, 0x26, 0x8e, 0xfe, 0x49, 0x06,
			0x5d, 0xdb, 0x6f, 0xd7, 0x96, 0x9a, 0x84, 0x21,
			0x89, 0xb9, 0xd7, 0xb3, 0xcc, 0xb3, 0x2d, 0x62,
			0xaa, 0x05, 0xe8, 0x7e, 0x93, 0x29, 0x30, 0xf7,
			0xa1, 0x77, 0x5c, 0x33, 0x87, 0x36, 0xd9, 0xbc,
			0x8f, 0x36, 0x52, 0x16, 0x09, 0xd8, 0xbe, 0x0c,
			0x29, 0xfd, 0xd1, 0x72, 0x84, 0x30, 0xa5, 0x37,
			0xf0, 0xa2, 0xb9, 0xb9, 0xfe, 0xf2, 0xcd, 0x9f,
			0x09, 0x46, 0xc2, 0x21, 0xc0, 0x8a, 0xaa, 0x02,
			0x70, 0xe3, 0x18, 0x7e, 0xe5, 0xc5, 0x18, 0xcf,
			0xeb, 0x00, 0x16, 0x9e, 0x77, 0x18, 0xb0, 0x1a,
			0xc0, 0xfa, 0xef, 0x09, 0x7e, 0x9c, 0xb6, 0xa4,
			0xdf, 0x3e, 0x87, 0xa5, 0x54, 0x8a, 0x6c, 0x3d,
			0x9f, 0x1b, 0xa2, 0x30, 0xee, 0x1c, 0xaa, 0x01,
			0x29, 0x7e, 0x5f, 0x17, 0xd1, 0xbe, 0x1d, 0x77,
			0x65, 0x52, 0xf3, 0x66, 0x38, 0xcf, 0xf1, 0x3a,
			0xb7, 0x3a, 0x10, 0x58, 0xfe, 0x7c, 0x1e, 0xee,
			0x28, 0xc7, 0x6a, 0x14, 0x5e, 0x9f, 0xf9, 0xb1,
			0x70, 0x74, 0x96, 0x3c, 0x22, 0xc6, 0x43, 0x5b,
			0x6c, 0x5a, 0x61, 0x9a, 0x6f, 0x39, 0xdf, 0x94,
			0xce, 0x34, 0x8b, 0x24, 0x43, 0x20, 0xb2, 0x07,
			0xa9, 0x11, 0x7e, 0x98, 0xb9, 0xaa, 0x5a, 0x8c,
			0x58, 0x51, 0x6d, 0x39, 0xc7, 0x18, 0x78, 0xc4,
			0xec, 0xfd, 0x74, 0x1c, 0xe6, 0xe5, 0x12, 0x22,
			0xfc, 0xd9, 0x2a, 0xd3, 0x2d, 0x70, 0xc3, 0xb9,
			0x2c, 0xbb, 0xe3, 0x01, 0xda, 0xcd, 0xdf, 0x2e,
			0xc3, 0xae, 0xc2, 0x1f, 0xdd, 0x38, 0xa7, 0xe1,
			0x10, 0xf4, 0xf5, 0x44, 0x85, 0x77, 0xb9, 0x54,
			0x6f, 0x1a, 0x7c, 0xd7, 0x1a, 0x35, 0x67, 0x0c,
			0x1c, 0xa4, 0x7a, 0x91, 0x99, 0x43, 0x7c, 0xbb,
			0xc6, 0x59, 0x26, 0xcd, 0x17, 0xdd, 0xdd, 0x2c,
			0x0c, 0x3b, 0x1f, 0xfe, 0xbe, 0x68, 0x2b, 0xe6,
			0x16, 0xe6, 0x38, 0x83, 0x97, 0x44, 0xa1, 0x47,
			0xea, 0x89, 0x78, 0x85, 0xaf, 0xef, 0xbe, 0x6f,
			0x0e, 0x37, 0xd4, 0xe4, 0x82, 0xdd, 0x00, 0x5f,
			0x4f, 0xf1, 0x99, 0xd0, 0xd0, 0x33, 0xbb, 0x75,
			0x33, 0x80, 0x78, 0x0c, 0x90, 0x22, 0x8a, 0x87,
			0xd1, 0x4d, 0x8d, 0xbf, 0xb8, 0x29, 0xa1, 0x95,
			0xb5, 0xd8, 0xb2, 0xdb, 0xd6, 0x7c, 0x9e, 0xed,
			0xac, 0x48, 0xae, 0x63, 0x9c, 0x15, 0x8e, 0xb3,
		},
		62, /* salt length */
		/* V5 : 62 byte */
		{
			0x2d, 0x0c, 0x49, 0xb2, 0x07, 0x89, 0xf3, 0x95,
			0x02, 0xee, 0xfd, 0x09, 0x2a, 0x2b, 0x6a, 0x9b,
			0x27, 0x57, 0xc1, 0x45, 0x61, 0x47, 0x56, 0x9a,
			0x68, 0x5f, 0xca, 0x44, 0x92, 0xa8, 0xd5, 0xb0,
			0xe6, 0x23, 0x43, 0x08, 0x38, 0x5d, 0x3d, 0x62,
			0x96, 0x44, 0xca, 0x37, 0xe3, 0x39, 0x96, 0x16,
			0xc2, 0x66, 0xf1, 0x99, 0xb6, 0x52, 0x1a, 0x99,
			0x87, 0xb2, 0xbe, 0x9e, 0xe7, 0x83, 
		},
	},
	/* SHA512 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0x57, 0x72, 0x4b, 0x70, 0x62, 0x19, 0x3d, 0x22,
			0xf2, 0xb6, 0xbf, 0xd1, 0x84, 0x61, 0xd8, 0x7a,
			0xf1, 0x22, 0xc2, 0x7b, 0xf0, 0x60, 0x93, 0xa5,
			0xdd, 0x9c, 0x1d, 0x92, 0xb9, 0x5f, 0x12, 0x39,
			0x71, 0x70, 0x6c, 0xbf, 0x63, 0x4b, 0x0b, 0x91,
			0x1e, 0xcf, 0xa0, 0xaf, 0x69, 0x37, 0xcb, 0x4b,
			0x88, 0x4b, 0x80, 0x92, 0xba, 0xd7, 0xaf, 0xca,
			0x06, 0x5d, 0x24, 0x9d, 0x37, 0x07, 0xac, 0xb4,
			0x26, 0xdf, 0x79, 0x88, 0x37, 0x42, 0xc7, 0x75,
			0x26, 0x92, 0xc0, 0x11, 0x04, 0x2c, 0x9d, 0xbb,
			0x7c, 0x9a, 0x0f, 0x77, 0x5b, 0x09, 0xdd, 0xf9,
			0x50, 0xfd, 0xce, 0xff, 0xef, 0x43, 0xc9, 0xe4,
			0xfc, 0x28, 0x3b, 0x72, 0xe7, 0xe8, 0xb9, 0xf9,
			0x93, 0x69, 0xe7, 0x9d, 0x5b, 0x29, 0x98, 0xf4,
			0x57, 0x75, 0x36, 0xd1, 0xdb, 0xdd, 0x65, 0x5a,
			0x41, 0xe4, 0xe3, 0x61, 0xe9, 0xfc, 0xb2, 0xf1,
		},
		/* S6 : 384 byte */
		{
			0x84, 0xa2, 0x1a, 0x5c, 0xc0, 0x60, 0xd1, 0x41,
			0xba, 0x9c, 0xae, 0xca, 0x77, 0xfd, 0x04, 0xbe,
			0x8b, 0xa8, 0x27, 0x02, 0x35, 0xe9, 0x94, 0x8d,
			0x07, 0x06, 0xdc, 0xa7, 0x74, 0x13, 0xce, 0x7f,
			0x08, 0x11, 0xda, 0x8b, 0x2f, 0x53, 0x72, 0xf8,
			0xff, 0x5a, 0x2e, 0xb2, 0xbb, 0xea, 0xe4, 0x37,
			0x52, 0xc5, 0xd1, 0xc1, 0xe3, 0x87, 0x79, 0x92,
			0xa4, 0x95, 0x74, 0x89, 0x9a, 0x6e, 0xc9, 0xd2,
			0xa9, 0x48, 0x31, 0x56, 0x54, 0x03, 0x22, 0xfd,
			0xaa, 0x66, 0xee, 0xc4, 0xa2, 0x60, 0x1c, 0x28,
			0x1e, 0xa5, 0xae, 0x99, 0x61, 0x90, 0x85, 0x36,
			0x44, 0xb4, 0x82, 0x31, 0xbc, 0x22, 0x72, 0x9f,
			0x32, 0xc2, 0x18, 0x8e, 0x5f, 0x5f, 0x7b, 0x50,
			0x56, 0xfd, 0x3e, 0x99, 0xcc, 0xca, 0x3e, 0xff,
			0xcb, 0x97, 0x93, 0x34, 0x3f, 0x52, 0xa9, 0xee,
			0x60, 0x21, 0x7d, 0x1c, 0x49, 0x21, 0x02, 0x53,
			0x4a, 0x33, 0x4c, 0x1c, 0x60, 0xa9, 0xc4, 0xed,
			0x63, 0xae, 0x86, 0x1b, 0xec, 0x7d, 0xe9, 0x89,
			0x8c, 0x2d, 0xde, 0x02, 0x6d, 0x9a, 0x02, 0x9e,
			0x7d, 0x9f, 0xe4, 0x4d, 0x55, 0x2c, 0xd3, 0x76,
			0x3b, 0x8e, 0xc3, 0xf4, 0x37, 0x1f, 0x4e, 0x68,
			0x23, 0x15, 0x65, 0x7d, 0x72, 0xa8, 0x88, 0x91,
			0x3d, 0x15, 0xe1, 0xa8, 0x4a, 0x98, 0x1b, 0x3d,
			0x8d, 0x43, 0x75, 0x89, 0xa6, 0xde, 0xb3, 0x7d,
			0x14, 0xe8, 0x6a, 0xaa, 0x36, 0x51, 0x24, 0xbf,
			0x16, 0x50, 0x45, 0x04, 0x0b, 0x1f, 0x95, 0x9a,
			0xcc, 0xff, 0x35, 0x56, 0x52, 0x05, 0xd0, 0xee,
			0x72, 0xbc, 0x56, 0xd2, 0x73, 0xd1, 0x97, 0x34,
			0x10, 0x77, 0x4c, 0xea, 0x77, 0x35, 0xca, 0x79,
			0xc6, 0xbc, 0xb2, 0x56, 0xb5, 0x4f, 0xef, 0x01,
			0x72, 0xe0, 0x58, 0xba, 0x91, 0x61, 0x9c, 0x66,
			0xbc, 0x45, 0xe1, 0x1b, 0x6b, 0xcc, 0x0f, 0x68,
			0xb5, 0x29, 0xec, 0x3a, 0x41, 0x33, 0x59, 0x8b,
			0xcf, 0x09, 0xc9, 0xc4, 0xbb, 0x0f, 0x87, 0x4c,
			0x70, 0x95, 0xf3, 0xeb, 0xbf, 0x85, 0xa5, 0xf6,
			0x69, 0xbb, 0x37, 0x17, 0xee, 0xf9, 0x29, 0xfb,
			0x1c, 0x22, 0x94, 0x32, 0x68, 0xc3, 0x10, 0x28,
			0x2e, 0x88, 0x42, 0x84, 0x0a, 0xec, 0xfd, 0xc9,
			0x42, 0xa4, 0x68, 0x04, 0x5b, 0x02, 0x59, 0x5b,
			0xb1, 0x63, 0x36, 0x63, 0x4d, 0xa2, 0x0c, 0xa0,
			0xb8, 0xd7, 0x58, 0xcd, 0x30, 0xa2, 0xb7, 0xa0,
			0xbd, 0x0e, 0x3e, 0x2a, 0x6f, 0x30, 0xf3, 0x6a,
			0x14, 0x22, 0xad, 0xfe, 0xd8, 0x8e, 0x21, 0x14,
			0x85, 0x06, 0x6d, 0x6c, 0x0f, 0xa5, 0xc9, 0x86,
			0xf1, 0xdc, 0x5b, 0x4c, 0x1d, 0x96, 0x50, 0x21,
			0xdc, 0xc2, 0x4b, 0x3f, 0x72, 0x9f, 0x07, 0xc0,
			0x2b, 0x47, 0xaf, 0x75, 0xd0, 0x1f, 0x49, 0xda,
			0x3d, 0xea, 0x0f, 0x1b, 0xdd, 0x6b, 0x4c, 0x0f,
		},
		62, /* salt length */
		/* V6 : 62 byte */
		{
			0x2d, 0x0c, 0x49, 0xb2, 0x07, 0x89, 0xf3, 0x95,
			0x02, 0xee, 0xfd, 0x09, 0x2a, 0x2b, 0x6a, 0x9b,
			0x27, 0x57, 0xc1, 0x45, 0x61, 0x47, 0x56, 0x9a,
			0x68, 0x5f, 0xca, 0x44, 0x92, 0xa8, 0xd5, 0xb0,
			0xe6, 0x23, 0x43, 0x08, 0x38, 0x5d, 0x3d, 0x62,
			0x96, 0x44, 0xca, 0x37, 0xe3, 0x39, 0x96, 0x16,
			0xc2, 0x66, 0xf1, 0x99, 0xb6, 0x52, 0x1a, 0x99,
			0x87, 0xb2, 0xbe, 0x9e, 0xe7, 0x83, 
		},
	},
	/* SHA512 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0xbf, 0x5f, 0xf7, 0x76, 0x12, 0x28, 0x98, 0xe2,
			0x23, 0x33, 0xfb, 0x6d, 0xa9, 0x6d, 0x2a, 0x87,
			0xa3, 0xe6, 0xc4, 0xe6, 0x3f, 0x28, 0xfe, 0x7a,
			0xfb, 0xc8, 0xe8, 0xa4, 0x0a, 0x3a, 0xf2, 0xa3,
			0xf9, 0xe9, 0xae, 0x4f, 0x92, 0x87, 0xd7, 0x09,
			0x01, 0xa2, 0x93, 0xf2, 0x35, 0x79, 0xf5, 0x5b,
			0x89, 0x0d, 0xc6, 0x7d, 0xa4, 0x7b, 0x85, 0x6a,
			0x9d, 0x88, 0xac, 0x44, 0x63, 0x7e, 0x35, 0xad,
			0x5d, 0x37, 0x5d, 0x7e, 0x4d, 0x77, 0xa8, 0xbc,
			0x7a, 0x7f, 0x25, 0xc8, 0x0e, 0xde, 0xf3, 0xd5,
			0xbd, 0x8b, 0x04, 0x9f, 0xa7, 0x31, 0x21, 0x5b,
			0x80, 0xca, 0x2e, 0xe9, 0xee, 0x6f, 0xb0, 0x51,
			0x32, 0x6e, 0x8c, 0x6d, 0x0b, 0x9e, 0x11, 0xe3,
			0xd7, 0xef, 0x39, 0x57, 0xfc, 0x45, 0x2c, 0xde,
			0x86, 0x87, 0x06, 0xb5, 0x12, 0xf2, 0xda, 0x33,
			0xea, 0xb4, 0xf7, 0xfc, 0x71, 0xb6, 0x6a, 0x78,
		},
		/* S7 : 384 byte */
		{
			0x86, 0xec, 0xe9, 0x32, 0x1f, 0xaf, 0x13, 0x87,
			0xde, 0x6a, 0xfa, 0x7b, 0x1e, 0x16, 0xc2, 0x12,
			0x7e, 0x71, 0xe6, 0x47, 0x2e, 0x09, 0x37, 0x08,
			0xf0, 0xac, 0x4b, 0x40, 0xe6, 0xef, 0xb3, 0x0e,
			0xed, 0xc5, 0x46, 0x90, 0x71, 0x82, 0x79, 0x85,
			0x35, 0xad, 0x6b, 0x88, 0xae, 0x4a, 0x6f, 0x8c,
			0x4f, 0xae, 0x42, 0x9d, 0x22, 0x50, 0x58, 0x29,
			0x4e, 0xf7, 0x6d, 0x44, 0xca, 0x81, 0xde, 0xfd,
			0xad, 0xd1, 0x2c, 0xea, 0x16, 0xc5, 0x8c, 0x66,
			0x0a, 0x4d, 0x15, 0x8c, 0xb6, 0x72, 0x85, 0x45,
			0x30, 0x7f, 0x5a, 0x62, 0x34, 0xc3, 0xaa, 0x16,
			0xae, 0x6d, 0x98, 0x9b, 0x0b, 0x78, 0x8c, 0xc4,
			0xc1, 0x8b, 0x08, 0xc8, 0x9b, 0x57, 0xfe, 0x30,
			0x2c, 0xa6, 0x56, 0x0a, 0xff, 0xc5, 0x7b, 0xd5,
			0x33, 0xbd, 0xec, 0x6a, 0xe9, 0x0f, 0xc3, 0x71,
			0x67, 0xc4, 0x35, 0x5b, 0x07, 0xc6, 0xc7, 0xc7,
			0xaa, 0x2b, 0xda, 0xf9, 0x60, 0x02, 0x83, 0x2d,
			0x62, 0xc2, 0xdd, 0x09, 0x0c, 0x61, 0xcb, 0x86,
			0x58, 0xec, 0xc0, 0xe2, 0x24, 0x96, 0x4b, 0x50,
			0xb9, 0xab, 0xf1, 0xb4, 0x27, 0x18, 0x69, 0xa8,
			0x95, 0x1d, 0x81, 0xcd, 0x5b, 0x46, 0xaf, 0x4e,
			0xad, 0x70, 0xb0, 0x45, 0x4c, 0x01, 0xa7, 0x22,
			0x9e, 0xf2, 0xff, 0x27, 0x59, 0x9c, 0x73, 0x70,
			0xe7, 0x47, 0x98, 0x8b, 0x45, 0xb9, 0xa8, 0x14,
			0x85, 0x75, 0xd7, 0x30, 0x14, 0x16, 0x60, 0x82,
			0x94, 0x7c, 0x97, 0xe8, 0x73, 0x0d, 0x54, 0x58,
			0xff, 0x4a, 0x46, 0x06, 0xb1, 0x18, 0x5f, 0x1b,
			0xfd, 0x47, 0x6e, 0x8f, 0xea, 0x2d, 0x1d, 0x7f,
			0xb5, 0xd1, 0x4a, 0x06, 0x1f, 0x90, 0xe4, 0x38,
			0xce, 0x5e, 0x36, 0xb4, 0x89, 0xb5, 0x87, 0x3b,
			0x74, 0x00, 0xed, 0x77, 0x9e, 0xc8, 0x2a, 0xdf,
			0xdc, 0x2d, 0x93, 0x14, 0xd6, 0xe6, 0x54, 0x7d,
			0xec, 0x3b, 0xe9, 0x85, 0x33, 0x59, 0x82, 0x1e,
			0x6f, 0x6d, 0x85, 0x3c, 0x22, 0x92, 0xf1, 0x73,
			0x17, 0x89, 0x00, 0x20, 0x33, 0xec, 0xb4, 0x6c,
			0xfc, 0x3a, 0x7f, 0x19, 0x7a, 0x18, 0xa6, 0x77,
			0x57, 0x4f, 0xcf, 0x68, 0x70, 0xd7, 0xe4, 0x7d,
			0xb8, 0x74, 0xcf, 0xf2, 0x58, 0xf0, 0xf6, 0x58,
			0x93, 0x86, 0xfd, 0x96, 0x67, 0xaf, 0x29, 0x2c,
			0x31, 0x5f, 0xfd, 0x84, 0x9b, 0xf7, 0x17, 0x49,
			0xef, 0x1b, 0x4f, 0xc5, 0xa3, 0xfd, 0xf3, 0x9e,
			0x27, 0x82, 0xf9, 0x86, 0xbc, 0x8f, 0x52, 0x31,
			0x62, 0xc0, 0x01, 0x6c, 0x51, 0x70, 0x25, 0x13,
			0xed, 0x17, 0xc8, 0xf6, 0x86, 0x72, 0xcf, 0x42,
			0x5f, 0xd6, 0xef, 0x8b, 0x6c, 0x8e, 0x98, 0x3b,
			0xd2, 0x12, 0x8c, 0xe4, 0x61, 0x40, 0x85, 0xe7,
			0xfb, 0x21, 0x6a, 0xf7, 0xff, 0x01, 0x50, 0x19,
			0x41, 0xf2, 0x3f, 0xfb, 0xce, 0x55, 0x6f, 0x14,
		},
		62, /* salt length */
		/* V7 : 62 byte */
		{
			0x2d, 0x0c, 0x49, 0xb2, 0x07, 0x89, 0xf3, 0x95,
			0x02, 0xee, 0xfd, 0x09, 0x2a, 0x2b, 0x6a, 0x9b,
			0x27, 0x57, 0xc1, 0x45, 0x61, 0x47, 0x56, 0x9a,
			0x68, 0x5f, 0xca, 0x44, 0x92, 0xa8, 0xd5, 0xb0,
			0xe6, 0x23, 0x43, 0x08, 0x38, 0x5d, 0x3d, 0x62,
			0x96, 0x44, 0xca, 0x37, 0xe3, 0x39, 0x96, 0x16,
			0xc2, 0x66, 0xf1, 0x99, 0xb6, 0x52, 0x1a, 0x99,
			0x87, 0xb2, 0xbe, 0x9e, 0xe7, 0x83, 
		},
	},
	/* SHA512 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0x61, 0xb6, 0xdd, 0x24, 0x90, 0x36, 0x72, 0x62,
			0x18, 0x10, 0xcb, 0xe3, 0x34, 0x24, 0x97, 0xa6,
			0xb2, 0x98, 0xb5, 0x24, 0xf7, 0xcd, 0x50, 0xe3,
			0x42, 0x91, 0x4f, 0x48, 0x35, 0x96, 0xec, 0xad,
			0x91, 0x22, 0xa2, 0xb3, 0x41, 0x09, 0x4d, 0xd9,
			0x9a, 0xd9, 0x8d, 0x4e, 0xe1, 0x54, 0x6b, 0x04,
			0x0d, 0x23, 0x3f, 0x06, 0xcf, 0xc8, 0xd1, 0x0b,
			0xd0, 0xd5, 0xbe, 0x4b, 0x3a, 0x5b, 0x1d, 0x91,
			0x79, 0xa6, 0x63, 0x92, 0x43, 0x27, 0x84, 0x7d,
			0xd5, 0xb2, 0x5b, 0xd3, 0x80, 0xea, 0x4c, 0x79,
			0x65, 0xf9, 0x28, 0x0c, 0x7d, 0x84, 0x50, 0x74,
			0xdc, 0xdd, 0x1e, 0xbc, 0x36, 0x7b, 0x80, 0x20,
			0xa2, 0xa8, 0xe6, 0x68, 0x9e, 0x7a, 0x5f, 0x75,
			0x53, 0x04, 0xfe, 0x1a, 0x1b, 0xcd, 0x83, 0x2d,
			0x41, 0x82, 0x37, 0xdd, 0x08, 0xe7, 0x18, 0x45,
			0xee, 0x13, 0x36, 0x42, 0x31, 0xdd, 0x5d, 0x82,
		},
		/* S8 : 384 byte */
		{
			0x57, 0xd8, 0x27, 0x59, 0x3a, 0xd0, 0x9f, 0x00,
			0x00, 0x5f, 0xf1, 0xba, 0x45, 0x21, 0xa9, 0xab,
			0x27, 0x17, 0xfe, 0x34, 0xd7, 0xaf, 0x12, 0xd7,
			0xef, 0x5d, 0xc0, 0x78, 0x14, 0xcb, 0x93, 0x25,
			0x7a, 0x29, 0x03, 0xce, 0xdf, 0x0a, 0x80, 0x70,
			0x4b, 0x16, 0xfd, 0x8a, 0xa9, 0xdb, 0xd0, 0x6f,
			0xe3, 0xd9, 0x6f, 0xcc, 0x7b, 0xe3, 0x84, 0x3e,
			0xa1, 0x61, 0xe8, 0x0c, 0xa5, 0x6f, 0x3e, 0xf6,
			0xf7, 0x60, 0xdf, 0xc7, 0xf1, 0x70, 0x4e, 0xd4,
			0xa5, 0x01, 0x42, 0x26, 0x7b, 0x87, 0xd2, 0x44,
			0xc7, 0x1f, 0xc7, 0x21, 0x02, 0x11, 0x2f, 0xe4,
			0xea, 0x80, 0x1c, 0x82, 0xc6, 0x31, 0xed, 0xd9,
			0xd9, 0x17, 0x80, 0x8c, 0x0a, 0x1f, 0x1c, 0x81,
			0xa9, 0xde, 0x85, 0x9d, 0xd8, 0x75, 0x69, 0x89,
			0x8c, 0xba, 0x76, 0xb3, 0x57, 0x02, 0x23, 0x2a,
			0xa4, 0x92, 0x85, 0x07, 0x39, 0xec, 0x03, 0x71,
			0xb0, 0x34, 0x23, 0x18, 0xb9, 0x2e, 0xef, 0xc4,
			0x5e, 0x6a, 0xe8, 0x54, 0x7a, 0x60, 0x4d, 0x9a,
			0x15, 0xc2, 0x82, 0x9e, 0xa8, 0x55, 0x33, 0xd6,
			0xd2, 0x3f, 0xb6, 0x1e, 0xf5, 0x69, 0xbe, 0x63,
			0x77, 0x9d, 0x3d, 0x2c, 0x7c, 0xd3, 0xbf, 0xbc,
			0x26, 0xdf, 0x02, 0x61, 0x6b, 0x7b, 0xdb, 0xbc,
			0x0b, 0x4e, 0x2b, 0x5e, 0xbb, 0xa7, 0xec, 0x93,
			0x88, 0x6a, 0x36, 0x9d, 0x10, 0xb7, 0xbf, 0xc0,
			0xe7, 0xf5, 0x6e, 0x7b, 0x7c, 0xcc, 0x81, 0x48,
			0x80, 0xba, 0xa6, 0x34, 0xf4, 0xaf, 0xd8, 0x74,
			0xa8, 0x41, 0xd4, 0x0c, 0xdf, 0x9c, 0x8f, 0x11,
			0x75, 0x35, 0x65, 0x0b, 0x55, 0x12, 0x9b, 0x89,
			0x13, 0xd5, 0x34, 0x17, 0xbd, 0xaf, 0x16, 0x3d,
			0x68, 0xe7, 0x04, 0x4a, 0xc0, 0x11, 0xa5, 0x5a,
			0xc0, 0xe1, 0xaf, 0xd9, 0x27, 0x9d, 0x46, 0xd3,
			0x1e, 0xf8, 0x3a, 0x0b, 0xb4, 0xa7, 0xdb, 0xe7,
			0x0b, 0xde, 0x4b, 0x33, 0x39, 0x67, 0x50, 0xb6,
			0x76, 0x57, 0x64, 0x97, 0xe2, 0x02, 0xe4, 0x0c,
			0xd1, 0x40, 0x1f, 0xd6, 0xcb, 0x08, 0x87, 0x8a,
			0x6c, 0x22, 0xdb, 0x61, 0x40, 0x4b, 0x4c, 0x2a,
			0xa8, 0x80, 0x72, 0xf7, 0xa4, 0x85, 0x1d, 0x9f,
			0xaa, 0xf0, 0x16, 0xa6, 0x0a, 0x7a, 0x49, 0x14,
			0x7f, 0xc2, 0x34, 0xad, 0x67, 0xf8, 0x37, 0x5a,
			0x90, 0x06, 0x9c, 0x27, 0x4a, 0xad, 0xda, 0xea,
			0x43, 0xdf, 0x62, 0x92, 0xcc, 0xdf, 0x7d, 0xaa,
			0xb5, 0xf5, 0x11, 0x30, 0x70, 0xf8, 0xca, 0x5e,
			0x7f, 0x43, 0xc7, 0x91, 0xac, 0xc7, 0xe1, 0x73,
			0x7c, 0xbc, 0x31, 0x1b, 0xd5, 0x71, 0x4a, 0xbb,
			0x66, 0x56, 0x17, 0x03, 0xb9, 0xac, 0x36, 0x29,
			0xbb, 0x10, 0xbd, 0x1b, 0x77, 0x09, 0xf0, 0x81,
			0x84, 0x0e, 0xb3, 0xe9, 0x39, 0xc6, 0x96, 0x57,
			0xea, 0x8f, 0x7c, 0xfd, 0x59, 0x6b, 0x02, 0x65,
		},
		62, /* salt length */
		/* V8 : 62 byte */
		{
			0x2d, 0x0c, 0x49, 0xb2, 0x07, 0x89, 0xf3, 0x95,
			0x02, 0xee, 0xfd, 0x09, 0x2a, 0x2b, 0x6a, 0x9b,
			0x27, 0x57, 0xc1, 0x45, 0x61, 0x47, 0x56, 0x9a,
			0x68, 0x5f, 0xca, 0x44, 0x92, 0xa8, 0xd5, 0xb0,
			0xe6, 0x23, 0x43, 0x08, 0x38, 0x5d, 0x3d, 0x62,
			0x96, 0x44, 0xca, 0x37, 0xe3, 0x39, 0x96, 0x16,
			0xc2, 0x66, 0xf1, 0x99, 0xb6, 0x52, 0x1a, 0x99,
			0x87, 0xb2, 0xbe, 0x9e, 0xe7, 0x83, 
		},
	},
	/* SHA512 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0xdc, 0xc2, 0x71, 0xb1, 0xbb, 0x2e, 0x50, 0xeb,
			0xc2, 0x33, 0x30, 0xbe, 0x36, 0x53, 0x9d, 0x50,
			0x33, 0x8b, 0xaf, 0x2e, 0x9d, 0x7a, 0x96, 0x93,
			0x58, 0xc6, 0x77, 0xe8, 0xbc, 0xbc, 0x77, 0x87,
			0x43, 0x36, 0x15, 0xc4, 0x85, 0xc2, 0xbc, 0x2e,
			0x67, 0x00, 0x98, 0x12, 0x8f, 0x4c, 0xaa, 0x41,
			0x1b, 0x9d, 0x17, 0x13, 0x92, 0xad, 0xc6, 0xac,
			0x1a, 0x5b, 0x29, 0x7e, 0xec, 0x4d, 0x5b, 0x0f,
			0x06, 0xd9, 0x6c, 0xfd, 0x1f, 0x26, 0xf9, 0x3f,
			0xe0, 0x8e, 0xff, 0xad, 0x51, 0x47, 0xf0, 0xc3,
			0x92, 0x43, 0x07, 0xa2, 0xcb, 0x54, 0xd9, 0x57,
			0x65, 0x94, 0x2e, 0x60, 0x7b, 0x04, 0x0e, 0x6c,
			0x8b, 0x73, 0x1f, 0x63, 0x72, 0xa2, 0x2e, 0xa6,
			0x97, 0xa5, 0x0b, 0x98, 0x66, 0x8c, 0x9a, 0x5d,
			0x00, 0x43, 0x27, 0xe2, 0x30, 0xb7, 0xfa, 0x1d,
			0xa2, 0x3a, 0x2b, 0x96, 0x4f, 0x29, 0xb8, 0x26,
		},
		/* S9 : 384 byte */
		{
			0x0a, 0xc9, 0x38, 0xab, 0x04, 0xbf, 0x4e, 0xfa,
			0x58, 0x7e, 0x34, 0x14, 0x34, 0x36, 0xce, 0x60,
			0x8a, 0xd0, 0x89, 0x42, 0x09, 0x56, 0xa7, 0x2b,
			0x23, 0x10, 0x3f, 0xea, 0x76, 0x9c, 0x03, 0xf0,
			0x2c, 0x3a, 0x0d, 0xb7, 0x64, 0xcd, 0x5b, 0xf3,
			0xcc, 0x85, 0x18, 0x56, 0x5b, 0x7e, 0xff, 0xf7,
			0x0c, 0x74, 0xcc, 0x65, 0x36, 0x65, 0xdc, 0x06,
			0xe7, 0xf1, 0xd5, 0x84, 0xe9, 0x67, 0xba, 0x19,
			0x3a, 0x70, 0xf5, 0xe3, 0xf7, 0x41, 0x6e, 0xd0,
			0xd4, 0xd5, 0xdc, 0x0e, 0x76, 0x1b, 0x24, 0xac,
			0x8a, 0x8b, 0xe1, 0x72, 0xeb, 0x95, 0x69, 0x1f,
			0x02, 0x24, 0x43, 0x79, 0xc9, 0xae, 0xda, 0x8a,
			0x9f, 0x76, 0x0e, 0x06, 0x1f, 0xd4, 0x76, 0xb0,
			0x63, 0xb5, 0xed, 0xed, 0xa5, 0x6b, 0xed, 0x81,
			0x9f, 0xb7, 0x13, 0x6a, 0x46, 0x04, 0x87, 0x9a,
			0x92, 0xb2, 0xcd, 0x35, 0x50, 0x7f, 0xd4, 0x9b,
			0x7d, 0x47, 0x8f, 0xbd, 0x24, 0xc7, 0x64, 0xaa,
			0x5b, 0xc5, 0x35, 0xa6, 0xab, 0xd7, 0xbf, 0xf5,
			0xc7, 0x69, 0x20, 0x35, 0x62, 0x05, 0x97, 0xf6,
			0x32, 0x9a, 0x45, 0x4c, 0xe9, 0x18, 0x87, 0x31,
			0xc4, 0xe7, 0x4d, 0x56, 0xc5, 0xbd, 0xef, 0x11,
			0x37, 0x25, 0x40, 0xb9, 0x58, 0xcf, 0x2f, 0x8c,
			0x42, 0xcb, 0xdb, 0xf9, 0x15, 0xe0, 0xc0, 0x7c,
			0x77, 0xf0, 0x4b, 0x05, 0xd8, 0x76, 0xaf, 0xbc,
			0x3f, 0x2c, 0x20, 0x5a, 0x40, 0x48, 0x82, 0x63,
			0x19, 0x18, 0x4d, 0x65, 0x0a, 0x24, 0x3d, 0x19,
			0x2f, 0xbe, 0x35, 0xa1, 0x63, 0xab, 0x8e, 0xa8,
			0x4a, 0x00, 0x1d, 0xd7, 0xc1, 0x47, 0x29, 0x88,
			0xa7, 0x80, 0x42, 0xcf, 0x9f, 0xff, 0xd9, 0x6f,
			0x69, 0x48, 0xf0, 0xe6, 0x92, 0xfc, 0x3f, 0x3b,
			0x1c, 0x9c, 0x13, 0xde, 0x4b, 0x7a, 0x02, 0x1b,
			0xe2, 0x5c, 0x80, 0x60, 0x6e, 0x11, 0x05, 0xcd,
			0x56, 0x81, 0x5d, 0x27, 0xc4, 0x5f, 0xef, 0x99,
			0x5b, 0x1f, 0xea, 0x36, 0xe2, 0xe1, 0x2a, 0xaf,
			0xc4, 0xa6, 0x99, 0x24, 0x78, 0x5c, 0x48, 0x55,
			0xc5, 0x0c, 0x61, 0xb1, 0xf4, 0x3b, 0xe9, 0xa1,
			0xad, 0xfd, 0x8d, 0x7f, 0xf2, 0xef, 0x52, 0x40,
			0xdc, 0xfe, 0x5e, 0xa4, 0x61, 0x3d, 0xb4, 0xad,
			0x08, 0x5b, 0xb0, 0xa6, 0xfb, 0x86, 0x27, 0xb1,
			0xed, 0x94, 0xdd, 0x16, 0x4a, 0x4d, 0x9c, 0x4c,
			0x9f, 0x37, 0x59, 0x83, 0x73, 0x4f, 0x9d, 0x2c,
			0x35, 0xec, 0x69, 0xd6, 0xd7, 0x42, 0x11, 0x57,
			0xd8, 0x65, 0x8d, 0xce, 0xc1, 0xbf, 0x65, 0x99,
			0xea, 0x94, 0x28, 0x0a, 0x63, 0x42, 0x23, 0x76,
			0xbf, 0xab, 0xf1, 0xb9, 0xf7, 0x30, 0x29, 0x2c,
			0x49, 0x8c, 0x95, 0x36, 0x54, 0x40, 0x17, 0x43,
			0xc9, 0xe6, 0xbc, 0x49, 0x94, 0x46, 0x75, 0x94,
			0x84, 0xd9, 0x3e, 0x28, 0xd5, 0xf9, 0xf4, 0x86,
		},
		62, /* salt length */
		/* V9 : 62 byte */
		{
			0x2d, 0x0c, 0x49, 0xb2, 0x07, 0x89, 0xf3, 0x95,
			0x02, 0xee, 0xfd, 0x09, 0x2a, 0x2b, 0x6a, 0x9b,
			0x27, 0x57, 0xc1, 0x45, 0x61, 0x47, 0x56, 0x9a,
			0x68, 0x5f, 0xca, 0x44, 0x92, 0xa8, 0xd5, 0xb0,
			0xe6, 0x23, 0x43, 0x08, 0x38, 0x5d, 0x3d, 0x62,
			0x96, 0x44, 0xca, 0x37, 0xe3, 0x39, 0x96, 0x16,
			0xc2, 0x66, 0xf1, 0x99, 0xb6, 0x52, 0x1a, 0x99,
			0x87, 0xb2, 0xbe, 0x9e, 0xe7, 0x83, 
		},
	},
};

typedef mod3072_t tv_t;


#include "siggenPSStest.c"
