/* siggenPSS_2_3072_testfc.c */
/*
 * Copyright (c) 2017 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

/*
 * This test vector was created based on SigGen15_186-2.txt from
 * https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Algorithm-Validation-Program/documents/dss/186-2rsatestvectors.zip
 *
 * The link to the archive file exists in the following URL:
 * https://csrc.nist.gov/projects/cryptographic-algorithm-validation-program/digital-signatures/
 */
/*
 * https://csrc.nist.gov/projects/cryptographic-algorithm-validation-program/digital-signatures/
 * https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Algorithm-Validation-Program/documents/dss/186-2rsatestvectors.zip
 * SigGenPSS_186-2.txt
 */
#include "siggenPSStest.h"

static const unsigned short sn1[] = {
	0x9c43, 0xef52, 0x2cab, 0x1802, 0x2297, 0xd3d7, 0x0fa4, 0x91d0,
	0x3b97, 0x5b84, 0x4b76, 0xcedb, 0xa35d, 0x8d88, 0x5ddb, 0x2825,
	0xe31f, 0xd5c1, 0x01bd, 0x9e9a, 0x2155, 0x20bb, 0x8cdd, 0xdeb6,
	0xab2c, 0xf2dc, 0x8606, 0x5179, 0x477d, 0x80f7, 0x3301, 0x6929,
	0xd733, 0x4cdf, 0xdf81, 0x8c13, 0x78a4, 0xb942, 0x8fa1, 0xee2e,
	0x5253, 0x21f9, 0x05d0, 0xb949, 0xd3ab, 0xc9e9, 0x3d3f, 0x30b0,
	0x7779, 0x5338, 0xbd55, 0xc28a, 0x1ced, 0x134b, 0xb2d5, 0x75bf,
	0xa44b, 0x2fd8, 0xcf1d, 0x5c54, 0x168a, 0x12a1, 0xd6c5, 0x11f6,
	0x2ca9, 0x73cd, 0xb704, 0xc233, 0x487e, 0x1fd3, 0x9e5a, 0xdc88,
	0x70af, 0x352e, 0xc3c6, 0xa6a6, 0x4152, 0xfc82, 0xa1c1, 0x6ecc,
	0x43d1, 0xd581, 0x7f76, 0xa1b4, 0x6a5f, 0xab9d, 0xb892, 0x3311,
	0xedd3, 0xcc03, 0x2fed, 0x7eb6, 0x252e, 0x77db, 0x69d7, 0xbf9e,
	0xe35d, 0xc4dd, 0xd0fb, 0xdb9a, 0x76af, 0xe25a, 0x82f4, 0x495a,
	0xa4f0, 0x72ce, 0xf9b1, 0x247c, 0xb368, 0xbcc8, 0x6775, 0x65a4,
	0x7095, 0x2427, 0x02e6, 0x3412, 0x81f5, 0x0680, 0x5e20, 0xe8c8,
	0x9e9a, 0xf28a, 0xdff2, 0x1c80, 0x4c70, 0xcab1, 0x0ee2, 0xfe52,
	0x12ec, 0x0798, 0x7d13, 0xe7d4, 0xb605, 0x2961, 0x1e4d, 0x33a0,
	0x62d7, 0x24cd, 0xfb16, 0xcdc4, 0x8b96, 0x4ba0, 0x7dfa, 0x2620,
	0xd5a8, 0x805d, 0x0be9, 0x3380, 0x850c, 0x66f3, 0xfada, 0x0598,
	0x02a5, 0x946b, 0xfe3b, 0x0b10, 0xe19a, 0x8289, 0xec01, 0xf651,
	0x4abb, 0x883b, 0xb53c, 0x49da, 0xdbba, 0x42d4, 0x12ea, 0x264c,
	0x8a51, 0x22fd, 0xa1ea, 0x9b74, 0x2289, 0x642b, 0x0ea3, 0x4ceb,
	0x76ae, 0x8256, 0xa978, 0x45d3, 0x7594, 0xcfbf, 0xf8c7, 0xa443,
	0x0176, 0x223b, 0xacc3, 0xbef3, 0x95ce, 0xda13, 0xfd21, 0x1c71,
};

static const unsigned short se1[] = {
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0001, 0x0001,
};

static const unsigned short sd1[] = {
	0x0ca4, 0xcf8c, 0xb3da, 0x0d29, 0x8110, 0x8d5e, 0xb9a0, 0x8c7e,
	0xcf8f, 0x8de3, 0x5cb2, 0xca86, 0x8987, 0xa48f, 0xcbde, 0x5377,
	0x8b52, 0xd45a, 0x1750, 0xb730, 0x4bf1, 0xa622, 0x1ec2, 0x0606,
	0xfc3f, 0x3910, 0x6054, 0xcb0b, 0x57de, 0x3f0c, 0xe166, 0xe412,
	0x335c, 0x2e84, 0x6fc2, 0x7ce9, 0xc769, 0x76a6, 0xfcd5, 0xb29e,
	0xb361, 0x101e, 0xe4da, 0x505f, 0xb99d, 0x8d0b, 0x0efe, 0x59ab,
	0x71a3, 0x52b9, 0x44d2, 0x9758, 0xd301, 0xd8bd, 0x9784, 0xff8f,
	0xc7d4, 0xa094, 0xb79f, 0x6dcc, 0xccfd, 0x1388, 0x5e47, 0x184e,
	0xb5aa, 0xe958, 0x940e, 0x31c3, 0x7202, 0x6883, 0x582d, 0x0c6a,
	0x8ac9, 0x9671, 0x47fc, 0x7f17, 0xf531, 0xa886, 0x4e5d, 0xc720,
	0x067d, 0xc83b, 0x7feb, 0x82a6, 0x8533, 0xe400, 0x3142, 0xbd18,
	0x9e88, 0x7459, 0xaeab, 0x5605, 0x1969, 0x92df, 0xa147, 0x0875,
	0x4b9c, 0x429d, 0xf151, 0x48df, 0x842c, 0x8832, 0x4021, 0x5f49,
	0xaacc, 0xabb3, 0xd110, 0x6457, 0xa344, 0xce07, 0xbbaa, 0x07fa,
	0x68b9, 0xce0f, 0xe8ee, 0xf0fa, 0xd528, 0x309c, 0x81fc, 0xfa8a,
	0x8a78, 0x6c35, 0xc2d2, 0xcbff, 0xac59, 0x3936, 0x954e, 0xe104,
	0xa846, 0x65ce, 0xb49d, 0x3d0d, 0xd8da, 0xa450, 0xecb9, 0xdcf6,
	0xd16c, 0x3128, 0x5c0a, 0x3927, 0xb44a, 0xc09b, 0xec36, 0xd65f,
	0x885c, 0xf583, 0x6232, 0xca9e, 0x9498, 0x3dc3, 0x3f43, 0xe7a5,
	0x6e7f, 0xfdba, 0x6c60, 0xedb6, 0xf614, 0x1227, 0x22ee, 0xcefa,
	0xa0ab, 0x4b78, 0xd5a9, 0x218f, 0x5a98, 0x5dae, 0x8720, 0x76cf,
	0xdc85, 0x51a5, 0xfead, 0xc70a, 0x035b, 0xf24b, 0xcba2, 0x3c88,
	0x103e, 0x9580, 0x84c8, 0x5d80, 0x89b7, 0xe7ae, 0x3ea4, 0xd699,
	0x4fd7, 0x5384, 0x7844, 0xee58, 0xbe93, 0x7ad7, 0x51da, 0x71b5,
};

#define HAS_SHA1_TESTVECTOR

#define HAS_SHA224_TESTVECTOR

#define HAS_SHA256_TESTVECTOR

#define HAS_SHA384_TESTVECTOR

#define HAS_SHA512_TESTVECTOR

/* 10 elements */
static mod3072_t SHA1[] = {
	/* SHA1 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0x33, 0x11, 0x67, 0x89, 0xe2, 0x07, 0xba, 0x2d,
			0x5e, 0x9f, 0x0c, 0x26, 0xe6, 0xee, 0xae, 0xcf,
			0x7b, 0xbc, 0xd9, 0xcf, 0x39, 0x88, 0xa0, 0x3e,
			0xe8, 0xcc, 0x22, 0x09, 0xcf, 0x83, 0x04, 0x97,
			0xc8, 0xe7, 0x1f, 0xac, 0x9a, 0x7f, 0x75, 0x8e,
			0x82, 0x59, 0x6a, 0x67, 0x13, 0x95, 0x31, 0xeb,
			0x41, 0x23, 0x84, 0x3a, 0xb8, 0xcc, 0x98, 0x08,
			0x72, 0x77, 0x50, 0xfb, 0x88, 0xa5, 0xb1, 0x54,
			0x48, 0x3f, 0x96, 0xd2, 0x89, 0x91, 0xa4, 0xd3,
			0x3b, 0x96, 0x81, 0xbd, 0x42, 0x76, 0x41, 0x9c,
			0x85, 0x4d, 0x21, 0x0d, 0x41, 0x2e, 0x78, 0x5a,
			0xa4, 0x68, 0xea, 0xe6, 0x97, 0x07, 0xdf, 0xcc,
			0xdb, 0x9e, 0x33, 0xc3, 0x57, 0x44, 0x30, 0x45,
			0xb0, 0xdd, 0xb9, 0xf0, 0x1b, 0x9b, 0xc7, 0x2c,
			0x6f, 0xc4, 0x1a, 0x0e, 0x7f, 0xb5, 0x28, 0x7c,
			0xa6, 0x27, 0x5a, 0x92, 0xa3, 0xb1, 0x77, 0x45,
		},
		/* S0 : 384 byte */
		{
			0x53, 0x6d, 0x49, 0x43, 0x13, 0xfc, 0x8c, 0x9f,
			0x9c, 0x20, 0xb1, 0x06, 0x3a, 0x80, 0xe2, 0x5f,
			0x22, 0x1a, 0x60, 0x5f, 0x93, 0x2c, 0xf3, 0xe9,
			0x47, 0x51, 0x26, 0xcb, 0x9c, 0x54, 0x48, 0x0f,
			0xd6, 0x4e, 0xaf, 0x37, 0x31, 0x67, 0x7e, 0xb5,
			0xf4, 0x22, 0x4e, 0xf6, 0x4f, 0xac, 0x38, 0x71,
			0x8f, 0x45, 0x9b, 0x95, 0xe6, 0x35, 0xc7, 0x30,
			0xa7, 0xcf, 0x5c, 0x80, 0xba, 0x7c, 0xc6, 0x30,
			0xc5, 0x18, 0x3f, 0x3f, 0x04, 0x5b, 0x0d, 0xf8,
			0xc7, 0xab, 0x72, 0xfd, 0xd6, 0xfb, 0xab, 0xe0,
			0x3b, 0xc2, 0x8d, 0xa3, 0x9a, 0x35, 0x4d, 0x94,
			0x65, 0x6b, 0xab, 0x2f, 0xf8, 0xbf, 0x71, 0xd8,
			0x36, 0x97, 0x10, 0x18, 0x06, 0xe2, 0x93, 0x5f,
			0xed, 0x70, 0xfc, 0xd9, 0xe6, 0x75, 0xa7, 0x8f,
			0x19, 0x68, 0xb7, 0x99, 0x13, 0xa2, 0x86, 0x0c,
			0x2f, 0x91, 0xeb, 0xd5, 0x9d, 0x6f, 0xa6, 0x7f,
			0x8a, 0x38, 0x76, 0x2a, 0x78, 0xfc, 0xcc, 0xfc,
			0x67, 0x2a, 0x58, 0x04, 0x58, 0xf0, 0xaa, 0xa6,
			0x4f, 0x11, 0x14, 0x96, 0x16, 0xb1, 0xd3, 0xd9,
			0xfb, 0x9d, 0x68, 0xf1, 0x32, 0xdc, 0x9d, 0x32,
			0x89, 0x6b, 0x6c, 0x36, 0xda, 0xb6, 0x66, 0xcd,
			0xda, 0x36, 0xf0, 0x3e, 0xb9, 0xc0, 0x1f, 0xcb,
			0x74, 0xff, 0xf0, 0x25, 0xd4, 0xcd, 0xba, 0xa1,
			0xfd, 0xa1, 0xa0, 0xf2, 0x06, 0xe2, 0x64, 0x7b,
			0x8e, 0x72, 0xd6, 0x43, 0xed, 0x52, 0x26, 0xdb,
			0x59, 0x33, 0x30, 0x93, 0xd2, 0x5c, 0x03, 0x8e,
			0xbd, 0x94, 0xab, 0x97, 0x6b, 0x25, 0xd3, 0xbe,
			0x19, 0x1e, 0x9f, 0xcf, 0x66, 0x68, 0xb5, 0xd5,
			0x24, 0x8d, 0x48, 0xca, 0xe9, 0x90, 0xab, 0x9b,
			0xe4, 0x78, 0x30, 0x0c, 0x6a, 0x23, 0xc7, 0x4a,
			0x8e, 0x15, 0x32, 0x1d, 0x47, 0xaf, 0x19, 0x4f,
			0xcf, 0x32, 0xf6, 0x18, 0x0a, 0x21, 0xf2, 0x3f,
			0x16, 0x89, 0xa1, 0xd9, 0x4f, 0xb3, 0xc0, 0x77,
			0x81, 0x5c, 0xaf, 0xaa, 0x34, 0x17, 0x79, 0xc2,
			0xc9, 0x82, 0xd6, 0x4a, 0x25, 0xa6, 0x97, 0x8f,
			0x5f, 0x54, 0x2d, 0x10, 0x57, 0xef, 0x3a, 0xc6,
			0xbe, 0xcc, 0x70, 0x7c, 0xec, 0x95, 0x6d, 0xd9,
			0x24, 0x51, 0x89, 0x22, 0x70, 0xad, 0x38, 0xab,
			0xd9, 0xd4, 0xb3, 0x62, 0xa3, 0x59, 0xe1, 0x11,
			0x12, 0x51, 0x3c, 0x28, 0x86, 0x9b, 0x1c, 0xbf,
			0x3a, 0xb9, 0x72, 0xf5, 0x0b, 0x63, 0xb3, 0x24,
			0x2f, 0x83, 0xb3, 0x2a, 0x03, 0x40, 0x77, 0x42,
			0x9b, 0x26, 0x8b, 0xb4, 0x6b, 0x54, 0x1d, 0xa0,
			0xb6, 0x63, 0xf6, 0xce, 0x64, 0xaa, 0x14, 0x38,
			0x88, 0x8f, 0x19, 0x61, 0x21, 0xcb, 0x53, 0x11,
			0x9a, 0x96, 0x45, 0x58, 0xd5, 0x84, 0x44, 0x87,
			0x67, 0x33, 0x1b, 0xc5, 0xa4, 0x20, 0x60, 0xe8,
			0x9c, 0x86, 0xb8, 0xf5, 0x79, 0x5b, 0x99, 0xbd,
		},
		20, /* salt length */
		/* V0 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA1 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0xe6, 0x5d, 0xd6, 0xc4, 0x1d, 0x19, 0xbb, 0x00,
			0xe4, 0x8d, 0x41, 0x0a, 0x2f, 0xb2, 0x0a, 0x4e,
			0xca, 0xec, 0x75, 0xfc, 0x34, 0x92, 0x14, 0xad,
			0xb1, 0xaa, 0x61, 0x49, 0x9b, 0xf4, 0xac, 0x28,
			0xf9, 0x2b, 0x58, 0x77, 0x53, 0x87, 0x47, 0xde,
			0xc4, 0x86, 0x93, 0x5f, 0x50, 0xec, 0x7f, 0x82,
			0x78, 0x08, 0xad, 0xaf, 0x0c, 0x3a, 0x91, 0xdb,
			0x41, 0xb2, 0x25, 0x67, 0x03, 0x7a, 0xde, 0x44,
			0x8c, 0xb7, 0xdf, 0x5c, 0x36, 0x2c, 0x6e, 0x2d,
			0x8d, 0xff, 0xa4, 0xb7, 0xa9, 0xc0, 0x68, 0x99,
			0xbb, 0xa7, 0x9e, 0x77, 0xe5, 0x1d, 0x9b, 0x9e,
			0x4f, 0x7e, 0xdc, 0xeb, 0x77, 0x1e, 0xf8, 0xe9,
			0xc8, 0x8a, 0xd4, 0x39, 0x6f, 0x10, 0x16, 0x37,
			0x6a, 0x81, 0xdc, 0xda, 0xe1, 0x35, 0xad, 0xdc,
			0xba, 0x50, 0xb0, 0x1b, 0x68, 0x44, 0x1b, 0x40,
			0x03, 0x97, 0xef, 0x9e, 0x9e, 0xe7, 0xe3, 0x5d,
		},
		/* S1 : 384 byte */
		{
			0x8c, 0x61, 0xee, 0x0a, 0xfa, 0xa9, 0xb7, 0x3b,
			0xe9, 0xb8, 0x1c, 0x08, 0xf8, 0x80, 0xd2, 0x8c,
			0x8b, 0x86, 0x4d, 0xb1, 0xde, 0xa2, 0xfb, 0xf1,
			0x8d, 0xba, 0xa9, 0x1f, 0x21, 0x60, 0xb5, 0xac,
			0xf2, 0xad, 0x97, 0xdd, 0x95, 0x01, 0x26, 0xda,
			0x03, 0xd3, 0xc2, 0x31, 0x75, 0x89, 0x87, 0x7a,
			0x03, 0x95, 0xf7, 0x3f, 0xfe, 0x75, 0x2f, 0x60,
			0x7f, 0x32, 0xc4, 0x68, 0xa6, 0xdf, 0x9e, 0xcd,
			0xfe, 0x09, 0xb6, 0x98, 0x7e, 0xd8, 0xbb, 0x85,
			0x31, 0xa9, 0x4e, 0x45, 0x5e, 0x2b, 0xdc, 0x29,
			0x70, 0x8f, 0xb7, 0x82, 0x79, 0x7a, 0xa9, 0x72,
			0x0c, 0x19, 0x3a, 0x43, 0x05, 0x1a, 0xe4, 0x4f,
			0xb2, 0xb8, 0xcc, 0xa1, 0xaa, 0x98, 0x22, 0x4b,
			0xa6, 0x6d, 0x38, 0xa0, 0xd3, 0xc9, 0x70, 0x4e,
			0x31, 0x43, 0xb4, 0xfc, 0x05, 0x8d, 0x4f, 0xf6,
			0xd0, 0xfa, 0x5f, 0x03, 0x9f, 0xeb, 0x06, 0xda,
			0xed, 0xd2, 0x2d, 0xa8, 0x57, 0xa1, 0x7e, 0x1d,
			0xeb, 0xda, 0xe8, 0x0a, 0xbf, 0x56, 0xb5, 0x33,
			0x15, 0xc9, 0x16, 0x2c, 0xe5, 0x87, 0x9c, 0x67,
			0xa4, 0xdb, 0x12, 0x1f, 0x3f, 0xd5, 0x3f, 0x5e,
			0x27, 0x36, 0x6e, 0xa6, 0x10, 0x7d, 0x95, 0xd6,
			0x56, 0x89, 0x38, 0xae, 0xe9, 0x45, 0xb8, 0x8f,
			0x14, 0x31, 0x10, 0x66, 0x14, 0x0f, 0xab, 0xc3,
			0xa0, 0x3a, 0xb5, 0x7c, 0x5c, 0x66, 0xa7, 0xb4,
			0x6e, 0x26, 0x36, 0x93, 0x5d, 0x32, 0xe4, 0x15,
			0x14, 0x1f, 0x24, 0x43, 0x36, 0xfb, 0x56, 0x99,
			0x84, 0xac, 0xe2, 0xad, 0xad, 0x94, 0x84, 0xed,
			0x87, 0x9f, 0x71, 0x01, 0xc9, 0x65, 0xa4, 0xc2,
			0x3c, 0xba, 0x01, 0x9f, 0x47, 0x53, 0xba, 0xfe,
			0x79, 0x17, 0xf9, 0x56, 0x1c, 0x8c, 0xaf, 0x71,
			0xe1, 0x94, 0xa8, 0x8b, 0xa0, 0xd9, 0xd1, 0x38,
			0x27, 0xc1, 0x8b, 0x70, 0x5d, 0x4a, 0x8e, 0xdb,
			0xd3, 0x58, 0x05, 0x72, 0x44, 0x4d, 0xc8, 0x23,
			0xdb, 0x24, 0xda, 0x15, 0x10, 0x55, 0xf6, 0x9e,
			0xb0, 0xf8, 0xf0, 0x1b, 0x10, 0xac, 0x67, 0x63,
			0x62, 0xdc, 0x5f, 0xfb, 0x79, 0x9c, 0x2e, 0xb7,
			0xd1, 0x61, 0x76, 0x1a, 0x21, 0x18, 0x04, 0x0c,
			0x5f, 0x08, 0xbc, 0xbc, 0xf4, 0xe7, 0xb8, 0x14,
			0xfe, 0x8a, 0x20, 0x3b, 0xbc, 0x43, 0xa8, 0x4c,
			0x9c, 0x72, 0x3b, 0xa9, 0xb4, 0x92, 0x20, 0xea,
			0xb0, 0xf3, 0x3d, 0x79, 0x64, 0x53, 0x4f, 0x14,
			0xf8, 0xc3, 0xbf, 0xf6, 0xea, 0x71, 0x75, 0x53,
			0xd5, 0x69, 0xce, 0xcf, 0x1b, 0x5f, 0xd3, 0xc1,
			0xa3, 0x48, 0xcd, 0x00, 0xad, 0x2f, 0xbc, 0x6f,
			0xb2, 0x84, 0x5d, 0xdc, 0x80, 0xbb, 0x78, 0x0f,
			0x43, 0xf0, 0x60, 0x35, 0xf8, 0x3a, 0xa1, 0xc4,
			0xf7, 0x77, 0x0d, 0xc4, 0xe7, 0x05, 0x90, 0xee,
			0x23, 0xc3, 0xe2, 0x3b, 0xaa, 0xc4, 0xc6, 0x02,
		},
		20, /* salt length */
		/* V1 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA1 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x77, 0x07, 0x90, 0xf9, 0xa0, 0x39, 0x32, 0xaf,
			0xef, 0x37, 0xb8, 0xdb, 0x6d, 0xf4, 0x19, 0x2c,
			0x5d, 0xb9, 0x4d, 0x48, 0x1f, 0x75, 0x55, 0xbe,
			0x54, 0x54, 0x7d, 0x3b, 0x43, 0x92, 0xde, 0x00,
			0x7c, 0x92, 0x26, 0x93, 0xba, 0x2a, 0x9c, 0xb5,
			0x0e, 0x41, 0x5e, 0x95, 0x3a, 0x9d, 0xed, 0x43,
			0x5b, 0xf6, 0xec, 0x67, 0xcc, 0x4f, 0x25, 0x33,
			0x46, 0x86, 0xcd, 0x96, 0x4c, 0x29, 0x09, 0xb3,
			0xc6, 0x8b, 0xb1, 0x91, 0xec, 0x17, 0x53, 0x14,
			0x65, 0xe5, 0x7b, 0x2c, 0xb0, 0x8b, 0x31, 0x4c,
			0xc4, 0x36, 0xc4, 0x1e, 0xf3, 0xf9, 0xc6, 0x75,
			0x14, 0x64, 0x58, 0x83, 0xe1, 0x59, 0xa8, 0x4c,
			0x9e, 0x1b, 0xb7, 0x5a, 0xfc, 0xa8, 0x60, 0xb2,
			0x3c, 0x49, 0x96, 0x08, 0x97, 0x67, 0xed, 0x2f,
			0xde, 0xa0, 0x5b, 0xd3, 0x5e, 0x1a, 0x40, 0x6a,
			0x06, 0x17, 0xa5, 0x68, 0xcc, 0xca, 0xc1, 0xb2,
		},
		/* S2 : 384 byte */
		{
			0x0c, 0x5f, 0x83, 0x48, 0x06, 0x44, 0xd4, 0x46,
			0x10, 0x66, 0xe0, 0xd1, 0x25, 0x92, 0xa0, 0xaf,
			0x0a, 0xf2, 0xa0, 0x5d, 0x2d, 0x7f, 0x77, 0x1f,
			0x07, 0x63, 0x0d, 0x67, 0xe5, 0x44, 0x37, 0x9d,
			0x61, 0x23, 0x4c, 0x1c, 0x43, 0xf2, 0x93, 0x98,
			0x51, 0x7b, 0x59, 0x78, 0x14, 0x38, 0xa2, 0x79,
			0xf5, 0x9e, 0xdd, 0xba, 0x8f, 0x8c, 0xe5, 0x6c,
			0x90, 0x87, 0x38, 0x61, 0x85, 0x95, 0x30, 0xc5,
			0xef, 0x5f, 0x0c, 0x3d, 0xe4, 0xee, 0xb4, 0x53,
			0xf1, 0x9f, 0x2e, 0xd5, 0xae, 0x63, 0xb3, 0xa3,
			0xfe, 0xf6, 0xb8, 0x46, 0xac, 0xf4, 0xed, 0xae,
			0xe1, 0xde, 0x06, 0x6e, 0x50, 0x24, 0x95, 0xec,
			0x0e, 0x53, 0x44, 0x75, 0xdf, 0x6e, 0xa7, 0xdb,
			0xaf, 0x5e, 0x9b, 0x04, 0x76, 0xe9, 0x22, 0x6f,
			0x00, 0x40, 0x8a, 0xf7, 0x01, 0x6a, 0x48, 0x28,
			0x2a, 0xc3, 0x19, 0x1d, 0xf9, 0x00, 0x3b, 0x34,
			0x79, 0xc4, 0x24, 0xd0, 0x79, 0x7b, 0x02, 0xe9,
			0x9c, 0xe0, 0x82, 0x58, 0xfd, 0xb6, 0xfa, 0xcb,
			0x52, 0x04, 0x75, 0x21, 0xa8, 0xd1, 0x55, 0x03,
			0x53, 0x09, 0x27, 0xb1, 0x7a, 0x43, 0xb8, 0xee,
			0x89, 0x7e, 0x30, 0xad, 0xb3, 0x15, 0x66, 0x08,
			0xe9, 0xf2, 0x46, 0xb7, 0xf7, 0xd2, 0xe3, 0x04,
			0x27, 0xd5, 0x64, 0x8a, 0x3f, 0x65, 0x4d, 0x9d,
			0x7f, 0x9e, 0x44, 0xa8, 0x74, 0x7e, 0xe8, 0x34,
			0xdb, 0xcf, 0xc4, 0xe3, 0x9d, 0x8e, 0xeb, 0xae,
			0x41, 0x7b, 0x0a, 0xcc, 0x5e, 0xa0, 0x8d, 0x12,
			0x35, 0xa9, 0xc8, 0x38, 0xc8, 0x33, 0x4c, 0xc8,
			0xb7, 0x9b, 0x9b, 0xbe, 0xf2, 0x05, 0x48, 0xf6,
			0xa3, 0xfb, 0x3b, 0x4d, 0x73, 0xea, 0xe2, 0x01,
			0x70, 0x6a, 0xe9, 0xc7, 0x5d, 0x8a, 0xcf, 0x85,
			0xe8, 0xbd, 0xb9, 0x26, 0x3f, 0xf1, 0x02, 0x48,
			0x9b, 0xdf, 0x27, 0xe6, 0x2c, 0xb0, 0x57, 0xd3,
			0xf1, 0xe7, 0xe2, 0x93, 0x87, 0x0f, 0x73, 0xb3,
			0xea, 0x0b, 0x5c, 0x31, 0xf0, 0xf2, 0xe6, 0x45,
			0xce, 0x2e, 0xd4, 0x06, 0x8c, 0x81, 0xd5, 0xfe,
			0xf7, 0x45, 0x39, 0xb5, 0x15, 0x26, 0x5f, 0x70,
			0x73, 0xc2, 0x5e, 0xa3, 0xfc, 0x2e, 0x11, 0xd4,
			0x09, 0xb5, 0x86, 0x9a, 0x19, 0x4d, 0x61, 0xad,
			0xea, 0x3f, 0x3c, 0x77, 0x78, 0x99, 0x8d, 0x33,
			0x1f, 0x37, 0xa2, 0x4f, 0xbf, 0x3d, 0xd8, 0x81,
			0xc0, 0x45, 0xe4, 0x04, 0x82, 0x8f, 0x08, 0x0b,
			0x4b, 0xac, 0xf5, 0x9d, 0x76, 0xd2, 0x96, 0x56,
			0xd7, 0x76, 0xce, 0x84, 0xf1, 0xd2, 0xec, 0x7b,
			0x9e, 0x3b, 0x27, 0x1f, 0xa8, 0x98, 0xdd, 0x8c,
			0x1b, 0x57, 0x5e, 0x12, 0x7b, 0xe1, 0xed, 0x1b,
			0x13, 0xc2, 0x65, 0xb7, 0x22, 0xf8, 0x55, 0x9a,
			0xc1, 0x6a, 0x79, 0x66, 0xd4, 0xf1, 0x95, 0x12,
			0xd8, 0xe3, 0x90, 0x9c, 0x85, 0x2e, 0x3f, 0x6d,
		},
		20, /* salt length */
		/* V2 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA1 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0x29, 0x50, 0x4b, 0xf7, 0x46, 0xd2, 0x72, 0xb9,
			0xa0, 0x2a, 0x7d, 0x9f, 0x19, 0x7a, 0x94, 0x81,
			0x3c, 0xf9, 0xa6, 0xe4, 0x89, 0xf2, 0x5b, 0x1e,
			0x0c, 0x10, 0x4b, 0xbb, 0x35, 0xd8, 0x6d, 0x20,
			0xe3, 0x77, 0xd2, 0x6e, 0xc2, 0xbe, 0x93, 0xa9,
			0xd7, 0x6b, 0x08, 0x1a, 0x59, 0xaa, 0x3a, 0xf1,
			0xc3, 0xdb, 0x9b, 0x2f, 0x27, 0x96, 0xf5, 0xc3,
			0xc5, 0xec, 0xa1, 0xf3, 0xdc, 0xf0, 0xf9, 0xfe,
			0xc0, 0x77, 0x0f, 0x00, 0x2d, 0x1b, 0xa0, 0x15,
			0xdb, 0x86, 0x43, 0xc1, 0x52, 0xed, 0xe6, 0x90,
			0x00, 0xda, 0x7b, 0xf9, 0x89, 0x2f, 0xef, 0x88,
			0x5e, 0x68, 0x5c, 0x3d, 0x91, 0x20, 0x7f, 0x2f,
			0xe8, 0xd6, 0x6c, 0x74, 0x30, 0xf2, 0x80, 0x45,
			0xd3, 0x03, 0x98, 0x5e, 0xad, 0x4b, 0x80, 0xa4,
			0x1c, 0x7d, 0xaf, 0x0a, 0x56, 0xc5, 0x31, 0xb5,
			0x00, 0x09, 0xe4, 0x5f, 0xfc, 0x6f, 0xe7, 0xf9,
		},
		/* S3 : 384 byte */
		{
			0x0d, 0x75, 0x7b, 0x79, 0xd1, 0x76, 0x6a, 0xef,
			0x47, 0xa7, 0x49, 0x9c, 0xbf, 0x5b, 0xb8, 0xc0,
			0x82, 0xe4, 0xed, 0x3d, 0xb1, 0x74, 0xed, 0x5e,
			0x14, 0x89, 0x73, 0x6a, 0x3b, 0xbe, 0xe7, 0x22,
			0x47, 0x23, 0x18, 0x9a, 0xe6, 0x84, 0x79, 0x54,
			0x54, 0x7e, 0x57, 0xfb, 0x76, 0xbb, 0x65, 0xb2,
			0x62, 0xd7, 0x7c, 0x3a, 0x91, 0xf8, 0x08, 0xd4,
			0x07, 0x5f, 0xca, 0x17, 0x81, 0x96, 0x01, 0x19,
			0xc0, 0xbb, 0x48, 0x83, 0xdf, 0xba, 0xda, 0x4d,
			0x2c, 0x75, 0x1d, 0x32, 0xaf, 0xd1, 0xf8, 0x1b,
			0x14, 0x1c, 0xab, 0x1f, 0xf9, 0xcb, 0xb1, 0xc6,
			0x8b, 0x92, 0x73, 0xe1, 0x76, 0x8a, 0x4a, 0x7f,
			0x7b, 0xcd, 0x71, 0x67, 0x77, 0x00, 0x88, 0x1b,
			0xc0, 0xd2, 0x55, 0x0f, 0x97, 0x14, 0x4d, 0xc1,
			0xc2, 0xdd, 0xab, 0x6f, 0x20, 0xa0, 0x80, 0x79,
			0x14, 0xac, 0x94, 0x7f, 0x38, 0x84, 0x70, 0x26,
			0xd0, 0x08, 0x53, 0x9e, 0xb4, 0xe8, 0x85, 0x2b,
			0xf8, 0x72, 0x39, 0xd7, 0x63, 0x8e, 0x65, 0x65,
			0x06, 0x3f, 0x17, 0xb5, 0x31, 0xc6, 0xd2, 0xdb,
			0xcf, 0x27, 0x50, 0xdf, 0xcc, 0xf2, 0x43, 0xc9,
			0x5b, 0xac, 0x43, 0x8f, 0x48, 0x74, 0xe5, 0x64,
			0x32, 0x69, 0xd3, 0xd4, 0xc4, 0x88, 0x51, 0x57,
			0xdd, 0x42, 0x8e, 0xbb, 0x1a, 0x7b, 0xcf, 0x50,
			0x92, 0x3e, 0xba, 0xa7, 0x4b, 0x23, 0x17, 0x1f,
			0xf1, 0x98, 0x80, 0xfe, 0x49, 0xd8, 0x07, 0x40,
			0x31, 0x4d, 0xcb, 0xe9, 0x04, 0xd1, 0x2e, 0xbc,
			0x71, 0x69, 0x67, 0xf1, 0xc2, 0xdf, 0x41, 0x0f,
			0x7a, 0x62, 0xd1, 0x12, 0x08, 0x8d, 0xb0, 0x6f,
			0x98, 0x6c, 0xc8, 0x16, 0x32, 0x88, 0xaf, 0x25,
			0xfd, 0x5b, 0x4a, 0x43, 0xa2, 0x15, 0x73, 0xea,
			0xfe, 0x77, 0x73, 0x72, 0xec, 0x85, 0x74, 0xd7,
			0x13, 0x69, 0x1e, 0x6d, 0xf6, 0x05, 0xce, 0x9b,
			0x76, 0xf7, 0xc9, 0xa4, 0xd5, 0xb2, 0x63, 0x0d,
			0x34, 0x2c, 0xf0, 0xb8, 0x57, 0xaa, 0x01, 0x59,
			0xe6, 0x19, 0x48, 0x7d, 0xde, 0x42, 0x8b, 0x12,
			0x11, 0xbd, 0x26, 0x94, 0x0d, 0xd5, 0xd0, 0xbc,
			0xe0, 0xb7, 0xbf, 0xe6, 0x91, 0x30, 0x56, 0x6f,
			0x80, 0xa5, 0x0e, 0x5d, 0x5a, 0x9c, 0xe5, 0x5f,
			0x45, 0x39, 0x03, 0xce, 0xd4, 0xb3, 0xe3, 0x07,
			0x9d, 0xc3, 0x71, 0x07, 0xb0, 0x28, 0x7d, 0x74,
			0x38, 0x5a, 0x36, 0xa6, 0x74, 0x24, 0xf2, 0xaa,
			0xf6, 0xbf, 0xff, 0xe1, 0xef, 0x06, 0xa4, 0x53,
			0x33, 0x6b, 0x12, 0xe3, 0x03, 0xc4, 0x56, 0xe2,
			0x7e, 0x70, 0xf6, 0x0f, 0xa5, 0x44, 0x46, 0x88,
			0x81, 0xdd, 0x13, 0xc1, 0x65, 0x9f, 0xb2, 0x87,
			0x23, 0x86, 0x43, 0x53, 0xaa, 0x80, 0xb3, 0xbe,
			0x33, 0x98, 0xa7, 0x55, 0xe7, 0x07, 0xca, 0xb0,
			0x34, 0xef, 0xa6, 0x2c, 0x19, 0x4e, 0x8b, 0x2c,
		},
		20, /* salt length */
		/* V3 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA1 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0xec, 0xdc, 0xc3, 0x57, 0x6f, 0x99, 0xf4, 0xca,
			0x44, 0xd4, 0x5a, 0x99, 0x1b, 0xca, 0x49, 0x93,
			0x12, 0x70, 0xf7, 0xea, 0x2a, 0x12, 0x8e, 0xef,
			0xfc, 0x3a, 0x80, 0x1d, 0xa8, 0xd3, 0x1b, 0x59,
			0x0f, 0x8e, 0x5d, 0x78, 0x42, 0x85, 0x09, 0x02,
			0x39, 0x1f, 0xfa, 0x35, 0xc9, 0x33, 0xb8, 0x4d,
			0xd0, 0x70, 0x49, 0x03, 0xb3, 0xfc, 0x4a, 0x54,
			0x0a, 0x5b, 0x3e, 0x83, 0x99, 0x93, 0x25, 0x61,
			0x83, 0x8d, 0x51, 0xcc, 0xba, 0xbf, 0x75, 0x76,
			0xba, 0x08, 0xff, 0xe8, 0x77, 0xdb, 0xa7, 0xc6,
			0x93, 0xa0, 0x48, 0x54, 0xd5, 0x57, 0xef, 0x39,
			0xd7, 0x7f, 0x32, 0x83, 0xfe, 0x36, 0xc4, 0xec,
			0xaf, 0xa1, 0x46, 0x5b, 0xc0, 0x8e, 0x5f, 0xae,
			0x2d, 0x58, 0x0e, 0xb7, 0x0d, 0x0b, 0x3f, 0x84,
			0xc4, 0xc9, 0x9d, 0xe4, 0x79, 0x6d, 0xa8, 0x26,
			0xcb, 0x6d, 0x07, 0x24, 0xbf, 0x05, 0x1f, 0x81,
		},
		/* S4 : 384 byte */
		{
			0x04, 0x44, 0x77, 0xbb, 0x00, 0xbf, 0x31, 0xa9,
			0x41, 0x16, 0x26, 0xc4, 0xdf, 0xda, 0x90, 0x16,
			0x5e, 0x5d, 0x24, 0xa5, 0xa3, 0x84, 0x76, 0x6d,
			0xd7, 0x25, 0x9f, 0x5b, 0x60, 0xdd, 0x85, 0x38,
			0x8d, 0xa5, 0xf4, 0xc9, 0xaf, 0x4b, 0x37, 0x77,
			0xb0, 0x46, 0xe7, 0xf0, 0x3b, 0xd9, 0x7b, 0x31,
			0x26, 0xdf, 0xd6, 0xb1, 0x84, 0xcd, 0x8b, 0x69,
			0x78, 0xb6, 0x3d, 0x9a, 0xc6, 0x3d, 0x15, 0x05,
			0xbc, 0x8d, 0xe5, 0x0c, 0x54, 0xfe, 0x66, 0xf2,
			0x1a, 0x23, 0x6d, 0x97, 0xac, 0xde, 0xa7, 0x69,
			0x4b, 0x99, 0x2f, 0x53, 0x9b, 0x67, 0x57, 0x99,
			0xbc, 0xb0, 0x0f, 0x84, 0x5a, 0x00, 0x4f, 0xa7,
			0x3f, 0xe2, 0x1d, 0xd2, 0xd7, 0xad, 0x15, 0x87,
			0xbd, 0xd5, 0xc9, 0x4d, 0x78, 0x4b, 0xaa, 0xd1,
			0x50, 0x81, 0x8a, 0x9f, 0xff, 0xab, 0xb6, 0xa9,
			0x40, 0x4b, 0xe2, 0xa1, 0x8d, 0x20, 0x4e, 0x41,
			0xf4, 0xbb, 0x6d, 0x48, 0x29, 0x46, 0x40, 0xc8,
			0x5e, 0xf0, 0x1c, 0x3a, 0xfc, 0xce, 0x78, 0xad,
			0x20, 0x7a, 0xda, 0xc2, 0xcd, 0xf1, 0x61, 0x76,
			0xd1, 0x69, 0x20, 0xa6, 0x3a, 0xbc, 0xfc, 0xcf,
			0x31, 0x0d, 0x64, 0x79, 0x58, 0x1c, 0xa4, 0x99,
			0x0f, 0x33, 0xf4, 0x4c, 0x43, 0xba, 0x0d, 0x60,
			0xda, 0x1d, 0xb6, 0xc6, 0xa0, 0x41, 0x19, 0x98,
			0x2a, 0x94, 0x5c, 0xd1, 0xba, 0x06, 0xb9, 0x0c,
			0xfc, 0xfc, 0xff, 0x9f, 0x59, 0x19, 0x4b, 0x50,
			0x20, 0x49, 0x87, 0xd6, 0x65, 0xb3, 0x1b, 0x96,
			0x8d, 0x43, 0x3f, 0xa0, 0xfe, 0xef, 0x78, 0xba,
			0x50, 0xfd, 0x77, 0xe6, 0x84, 0xde, 0xee, 0x4e,
			0xd5, 0x1a, 0x39, 0xae, 0xcc, 0x05, 0x8e, 0x17,
			0x7d, 0x81, 0x77, 0x02, 0xbd, 0x9c, 0xd7, 0x3d,
			0x67, 0x13, 0x1f, 0xc8, 0x8d, 0x12, 0xa9, 0xae,
			0x1b, 0x1f, 0x83, 0x05, 0x53, 0x89, 0xdd, 0xbb,
			0x0b, 0x2e, 0x55, 0x07, 0x2a, 0xef, 0x49, 0x6b,
			0x60, 0xc4, 0xa4, 0xb9, 0xfb, 0x40, 0x5e, 0xb4,
			0xd9, 0x5a, 0x77, 0x8a, 0x7f, 0xe7, 0x81, 0xc9,
			0x8d, 0x06, 0x6e, 0xe8, 0x25, 0x9e, 0x67, 0x39,
			0xd8, 0x5a, 0xab, 0xdb, 0xd8, 0xe6, 0x86, 0xa1,
			0x15, 0x77, 0xd4, 0xbc, 0xe2, 0x70, 0xdc, 0x46,
			0x5d, 0xdb, 0x29, 0xb8, 0xbe, 0xfe, 0x59, 0xdc,
			0x9a, 0x9e, 0x0f, 0x0f, 0xd6, 0x9c, 0xe9, 0x5d,
			0x50, 0xbc, 0x31, 0x4f, 0xb5, 0x22, 0xa0, 0xba,
			0xc3, 0x1e, 0x56, 0xd1, 0x3a, 0x31, 0x90, 0xd3,
			0x54, 0xf6, 0x1a, 0x27, 0x71, 0x96, 0xd2, 0xf6,
			0xa7, 0x17, 0x24, 0xbd, 0xf8, 0x0d, 0xb0, 0x3b,
			0x2f, 0x24, 0x70, 0x75, 0x50, 0x48, 0xa5, 0xaf,
			0x5f, 0xfd, 0x34, 0x7e, 0xeb, 0x5d, 0x87, 0xfa,
			0xc6, 0x30, 0x2d, 0x3f, 0x9d, 0x92, 0x5b, 0xdd,
			0x5f, 0x39, 0x4b, 0xb8, 0x66, 0x98, 0x12, 0x37,
		},
		20, /* salt length */
		/* V4 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA1 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0x5e, 0x0e, 0xf5, 0x50, 0x3e, 0xe7, 0x43, 0x17,
			0x12, 0x2a, 0x78, 0xbb, 0x9f, 0x5d, 0x93, 0x8d,
			0xfc, 0x09, 0x89, 0xff, 0xe9, 0xbd, 0x3b, 0xc1,
			0x62, 0x55, 0xa5, 0x29, 0x74, 0x7c, 0x58, 0xd8,
			0x50, 0x70, 0x2a, 0x9b, 0x9e, 0x06, 0x2c, 0x8f,
			0x13, 0xed, 0xdc, 0x89, 0x46, 0x36, 0x9c, 0xb4,
			0xbf, 0x14, 0x63, 0x7c, 0x51, 0xe1, 0xb3, 0xab,
			0xab, 0x62, 0xcb, 0xd2, 0x29, 0x4c, 0x1d, 0xeb,
			0xf0, 0x07, 0x02, 0x3c, 0x39, 0xb1, 0xad, 0x7f,
			0xed, 0x66, 0x5d, 0x3f, 0x73, 0xbf, 0xd7, 0x85,
			0xc5, 0x98, 0x14, 0xb4, 0xca, 0x17, 0x37, 0xb2,
			0xe0, 0xea, 0xc2, 0x48, 0xb7, 0xcf, 0x08, 0xcc,
			0x66, 0x5a, 0xe6, 0xfd, 0xb8, 0xf3, 0xae, 0x2e,
			0xc7, 0x1e, 0x64, 0xa7, 0xcd, 0xff, 0x2d, 0x48,
			0x36, 0x1d, 0x04, 0x6f, 0xfa, 0x66, 0xaf, 0x1c,
			0x29, 0x73, 0xf1, 0x53, 0xc8, 0x0e, 0x02, 0xc2,
		},
		/* S5 : 384 byte */
		{
			0x49, 0xb8, 0xa5, 0xac, 0x98, 0x43, 0x13, 0x51,
			0xa9, 0xce, 0xed, 0x58, 0x6b, 0xf8, 0x6c, 0xd2,
			0x85, 0x6b, 0xb5, 0xdf, 0x81, 0xc1, 0xe1, 0x11,
			0x05, 0x84, 0x7e, 0xf2, 0x3e, 0xf6, 0xfa, 0x05,
			0x1c, 0xc6, 0xe2, 0x3e, 0xc7, 0xe7, 0xe9, 0x84,
			0x7a, 0x1c, 0xfe, 0x6d, 0xc2, 0xa7, 0xda, 0xa3,
			0x10, 0x2d, 0xdb, 0x9e, 0xf6, 0x0e, 0x5e, 0x68,
			0xa8, 0xdf, 0x3d, 0xeb, 0x30, 0xa7, 0x38, 0xee,
			0xc9, 0xd9, 0x5b, 0x83, 0x75, 0x8e, 0x9c, 0x83,
			0x87, 0x75, 0x3f, 0x87, 0xaf, 0x6b, 0xf1, 0xf5,
			0x05, 0xd9, 0x47, 0x6f, 0xd0, 0xa8, 0x78, 0x7e,
			0xa0, 0xee, 0x28, 0xd1, 0x90, 0x8b, 0x32, 0xc1,
			0xf0, 0x79, 0xc8, 0x0b, 0x00, 0x44, 0x34, 0xdd,
			0x3e, 0xe0, 0xc9, 0x97, 0xf2, 0x3a, 0xad, 0x89,
			0x38, 0xf2, 0xb3, 0x5e, 0xac, 0x15, 0x65, 0x71,
			0xc4, 0x4d, 0x23, 0x03, 0xc0, 0xd6, 0xc2, 0x5d,
			0x98, 0x1c, 0xf1, 0x4a, 0x05, 0x13, 0x6c, 0xfa,
			0x7c, 0x20, 0x1a, 0x2e, 0xb7, 0x2d, 0x5c, 0xe1,
			0x28, 0x96, 0x7d, 0x26, 0xff, 0x77, 0x6a, 0xb9,
			0x73, 0xcf, 0xc6, 0x30, 0xd7, 0x91, 0xe7, 0xcb,
			0x56, 0x2e, 0xb3, 0xca, 0x33, 0x8d, 0x16, 0xf8,
			0xff, 0xe1, 0x85, 0xea, 0x27, 0x76, 0xdf, 0xc1,
			0x21, 0xbe, 0x95, 0x8a, 0x7e, 0x10, 0x29, 0xf9,
			0xb8, 0xe2, 0xc5, 0xec, 0x59, 0x50, 0xe9, 0x7d,
			0x31, 0x80, 0xb2, 0x7b, 0xee, 0x12, 0x6c, 0xdf,
			0xba, 0xdd, 0x89, 0x4c, 0x63, 0x88, 0xe3, 0xcf,
			0x7f, 0x3f, 0x20, 0x58, 0x62, 0xdb, 0x51, 0x72,
			0x73, 0x6c, 0x2f, 0x1d, 0x05, 0xfb, 0xbd, 0x47,
			0x5e, 0x74, 0x37, 0xac, 0xe8, 0x71, 0x38, 0x6b,
			0x4b, 0x75, 0x15, 0x50, 0x49, 0x13, 0x0c, 0xb1,
			0xb1, 0x3f, 0x22, 0xe0, 0x0a, 0x6a, 0x98, 0x6a,
			0xee, 0xaf, 0xd1, 0x09, 0x23, 0xeb, 0x9d, 0x0d,
			0x6c, 0x88, 0x31, 0x77, 0xbb, 0x29, 0x44, 0x9c,
			0x24, 0xf4, 0xef, 0xdb, 0xf9, 0x12, 0xfd, 0x4c,
			0xdf, 0x4e, 0x46, 0x11, 0xe9, 0x05, 0x94, 0xb2,
			0xb4, 0x4d, 0xe5, 0x40, 0xd6, 0x21, 0x02, 0x36,
			0xb2, 0xdd, 0xca, 0xdb, 0xbd, 0xa3, 0x56, 0x98,
			0xb2, 0x6d, 0xe4, 0x58, 0x33, 0x02, 0x91, 0x4a,
			0xfc, 0xc1, 0x5f, 0xaf, 0x57, 0xd6, 0x62, 0xd4,
			0x25, 0xeb, 0x92, 0x7c, 0x9f, 0x50, 0x98, 0x73,
			0x35, 0xda, 0x04, 0xac, 0xed, 0x06, 0xd7, 0x3a,
			0xa6, 0xc6, 0x1b, 0x5f, 0x83, 0xd9, 0x81, 0x36,
			0x00, 0xcc, 0x62, 0x32, 0x94, 0x0c, 0x4e, 0xc3,
			0xb0, 0xe3, 0xb6, 0x17, 0x44, 0x59, 0x6a, 0xfc,
			0x0a, 0x8a, 0xa5, 0x21, 0xb4, 0xc5, 0xfe, 0x05,
			0xf9, 0x21, 0x19, 0x5a, 0x0e, 0x5b, 0x78, 0xc7,
			0x79, 0x40, 0x53, 0xcf, 0x80, 0xeb, 0x88, 0x69,
			0x77, 0xac, 0x14, 0x28, 0x46, 0x5d, 0x36, 0x0a,
		},
		20, /* salt length */
		/* V5 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA1 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0xb2, 0xb2, 0x42, 0x52, 0x94, 0x6c, 0x3c, 0x32,
			0xcf, 0x6b, 0x74, 0x30, 0x25, 0xb8, 0xdc, 0x44,
			0x3f, 0xab, 0x81, 0x1d, 0x9f, 0x0c, 0x1a, 0x64,
			0xd7, 0x78, 0xdc, 0x45, 0xf8, 0x06, 0x96, 0x2c,
			0x4e, 0x2b, 0xf9, 0xcc, 0x4a, 0xdb, 0x37, 0x66,
			0xc0, 0xf1, 0xd1, 0x48, 0x53, 0xd1, 0xa9, 0x94,
			0x2d, 0x4c, 0x5f, 0x42, 0xf3, 0xc4, 0xda, 0x68,
			0xf7, 0x90, 0x25, 0xb9, 0xb0, 0xa1, 0x7d, 0x4c,
			0x68, 0x66, 0xa8, 0xa0, 0x35, 0xee, 0x88, 0xdc,
			0x2a, 0x4d, 0xef, 0x5b, 0xce, 0x40, 0xb5, 0x67,
			0xcd, 0x14, 0x08, 0x45, 0xee, 0xc3, 0x04, 0x73,
			0x45, 0x61, 0x83, 0x2d, 0xaa, 0xb6, 0x37, 0x35,
			0x6a, 0x2d, 0xf6, 0xd4, 0x3c, 0xed, 0x54, 0x1c,
			0x76, 0x45, 0x77, 0x71, 0x80, 0x4d, 0x27, 0x48,
			0x89, 0xca, 0x25, 0x2d, 0x7a, 0x53, 0xbe, 0x0b,
			0x28, 0xb2, 0x9f, 0xa8, 0xce, 0x8d, 0xeb, 0x6f,
		},
		/* S6 : 384 byte */
		{
			0x7a, 0x61, 0x0d, 0x96, 0x62, 0xae, 0xd0, 0x62,
			0x1a, 0x8a, 0x84, 0xe8, 0xc1, 0xf5, 0x9c, 0x6c,
			0xcf, 0xb8, 0xbf, 0x8b, 0xf9, 0xea, 0x73, 0x2f,
			0x8f, 0xd3, 0x2f, 0x87, 0x91, 0xf4, 0x68, 0xda,
			0xf4, 0x5f, 0x94, 0x3a, 0x5f, 0xa1, 0xff, 0x5f,
			0xcd, 0x10, 0x5f, 0xde, 0x5b, 0xb2, 0xff, 0x77,
			0x02, 0xe2, 0x76, 0x1e, 0x99, 0x37, 0x00, 0x43,
			0xc7, 0xd6, 0x9e, 0xb9, 0xc6, 0x2c, 0x22, 0xe7,
			0xe7, 0x84, 0xef, 0xc0, 0x32, 0x49, 0x2d, 0x59,
			0xac, 0x6e, 0x1d, 0x22, 0x34, 0x4a, 0x15, 0x22,
			0x78, 0x2f, 0x38, 0x62, 0x90, 0xdd, 0xfd, 0x89,
			0x37, 0xa4, 0xe8, 0x60, 0xbb, 0x54, 0x77, 0x72,
			0xa7, 0x42, 0x33, 0x31, 0x92, 0xb1, 0x0c, 0xf0,
			0x49, 0xbc, 0xdc, 0x3d, 0xee, 0x6b, 0x8c, 0x9b,
			0xcf, 0x63, 0xae, 0xfb, 0xdc, 0x4f, 0x09, 0x4f,
			0x40, 0x00, 0x96, 0x4a, 0x6c, 0x6c, 0x41, 0x97,
			0x28, 0xfe, 0x2e, 0x15, 0xb5, 0xb5, 0xbc, 0x1d,
			0x25, 0xfb, 0x6b, 0x28, 0x00, 0x56, 0x6a, 0xe0,
			0x00, 0xcf, 0xeb, 0xa2, 0xcc, 0x98, 0x3f, 0xc3,
			0xd0, 0x43, 0x3c, 0x2c, 0x2d, 0xd6, 0xac, 0xc1,
			0x8f, 0xe2, 0xc9, 0xca, 0xda, 0xeb, 0x28, 0x24,
			0x94, 0x95, 0xb3, 0x3c, 0xaa, 0x45, 0x04, 0xae,
			0xeb, 0x88, 0x35, 0x4c, 0x14, 0x99, 0x03, 0x28,
			0xc3, 0xdd, 0x39, 0x0a, 0x0b, 0xff, 0x47, 0x78,
			0x26, 0x87, 0x88, 0x82, 0x1e, 0x7b, 0x72, 0x5c,
			0x5f, 0x87, 0x3d, 0x53, 0xed, 0x68, 0x55, 0x74,
			0x87, 0xcd, 0x1b, 0x8f, 0x6e, 0x40, 0xcd, 0xbb,
			0x82, 0x03, 0x89, 0xee, 0xaa, 0x33, 0x7e, 0x6d,
			0x19, 0x0e, 0x64, 0x1c, 0xd3, 0x5d, 0xcd, 0x1e,
			0x75, 0x29, 0xba, 0xbc, 0x37, 0xb2, 0xb8, 0x10,
			0xab, 0x73, 0x7e, 0x07, 0x42, 0xef, 0x31, 0x2a,
			0x90, 0x04, 0x45, 0xe7, 0xcf, 0x98, 0x32, 0xea,
			0x5d, 0x41, 0x7d, 0x6e, 0x8e, 0xf5, 0x0d, 0xbb,
			0xef, 0x18, 0x95, 0x63, 0x91, 0x75, 0x42, 0x84,
			0x39, 0xce, 0x88, 0x66, 0x3c, 0xd4, 0xb1, 0xe5,
			0xa6, 0x1a, 0x9e, 0x7e, 0x4e, 0x49, 0x16, 0xf8,
			0xe1, 0xad, 0x92, 0xf0, 0xfe, 0x8b, 0xa4, 0x7b,
			0xcb, 0x97, 0x69, 0xbe, 0x61, 0x37, 0x3a, 0xc8,
			0xf1, 0xfb, 0x10, 0xbf, 0x97, 0x48, 0x33, 0x73,
			0x54, 0x0b, 0xd9, 0x83, 0x97, 0x4e, 0xbe, 0xea,
			0xdf, 0x9e, 0xfb, 0x1e, 0x86, 0x8b, 0xd7, 0x7f,
			0x86, 0xa0, 0xaa, 0x5e, 0xa7, 0xf8, 0x50, 0x35,
			0x1b, 0x07, 0x9b, 0x6b, 0xc7, 0x0a, 0xe5, 0x84,
			0x8b, 0x1d, 0xe0, 0x6a, 0x6f, 0x29, 0x7d, 0xde,
			0x99, 0x11, 0xe9, 0x65, 0xf4, 0x89, 0x77, 0xab,
			0x85, 0x48, 0x80, 0xde, 0x7b, 0x0e, 0xce, 0x2f,
			0x7b, 0xad, 0xbc, 0xe0, 0x82, 0xdb, 0x7f, 0xda,
			0xca, 0xaa, 0xdb, 0xf6, 0xca, 0x6b, 0xee, 0x46,
		},
		20, /* salt length */
		/* V6 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA1 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0x74, 0x39, 0x1f, 0xfd, 0x23, 0x22, 0xc1, 0x3e,
			0x2b, 0xe5, 0x27, 0x3e, 0xd1, 0x43, 0x5d, 0xef,
			0x0d, 0xe5, 0xb7, 0x5c, 0x8d, 0x4e, 0x00, 0x3c,
			0xfd, 0x0d, 0xd4, 0xc7, 0x86, 0x37, 0x4b, 0xc8,
			0x94, 0x14, 0xd8, 0xf4, 0xdd, 0x19, 0x70, 0x7a,
			0x9f, 0x16, 0xee, 0xdb, 0x28, 0xb1, 0xb6, 0xfc,
			0x9c, 0x13, 0x98, 0x7a, 0xf3, 0xe1, 0xf8, 0x3d,
			0x41, 0x6e, 0x2c, 0x50, 0x3a, 0x2f, 0x26, 0xef,
			0xf5, 0x6f, 0x53, 0xe2, 0xf0, 0x43, 0xc2, 0x0a,
			0x55, 0xc1, 0x89, 0x3a, 0x7b, 0x6d, 0x65, 0x48,
			0xb0, 0xf4, 0xa0, 0x63, 0x36, 0x1d, 0x38, 0xa6,
			0x8b, 0xfc, 0x3d, 0xee, 0xed, 0x69, 0x7b, 0xe9,
			0x1e, 0xb0, 0x59, 0x70, 0x65, 0x8a, 0x01, 0x17,
			0xd5, 0x4c, 0xc1, 0x02, 0xd1, 0x5f, 0xb1, 0xfa,
			0xd0, 0xf0, 0x97, 0xbf, 0x5e, 0xe8, 0x07, 0xa0,
			0x64, 0x1f, 0xcf, 0x4a, 0xa8, 0xcc, 0x92, 0xde,
		},
		/* S7 : 384 byte */
		{
			0x3f, 0x2f, 0x89, 0xe9, 0x8f, 0xd0, 0x11, 0x8b,
			0x57, 0x77, 0xbd, 0x01, 0x94, 0x79, 0x79, 0xef,
			0x27, 0xf3, 0x6d, 0x71, 0x8c, 0xf6, 0xba, 0xa7,
			0x06, 0xb8, 0x15, 0xc9, 0xc2, 0x92, 0x41, 0x41,
			0x21, 0x8e, 0xab, 0x53, 0x89, 0x77, 0xe3, 0xea,
			0x4a, 0xd8, 0xef, 0x1a, 0x41, 0x68, 0xb3, 0x1c,
			0x5e, 0x33, 0xf4, 0x24, 0x60, 0xd8, 0x66, 0x04,
			0x84, 0x1a, 0x27, 0x0e, 0x1b, 0x49, 0xe9, 0x44,
			0xa8, 0xce, 0x86, 0xa1, 0x30, 0x56, 0xd4, 0x3a,
			0xb9, 0x83, 0xa4, 0xa0, 0xfd, 0x33, 0xce, 0x6a,
			0xbb, 0x8a, 0xe0, 0x5f, 0x03, 0x11, 0x2f, 0x4c,
			0x0e, 0xc5, 0x16, 0x82, 0xf3, 0xb2, 0x14, 0x84,
			0x2e, 0x6c, 0xca, 0x7e, 0xdb, 0x24, 0x4a, 0x12,
			0xd5, 0x59, 0x9e, 0xec, 0x49, 0xdd, 0x76, 0xf0,
			0x31, 0x06, 0x26, 0x03, 0xe9, 0x3d, 0x47, 0x47,
			0x3a, 0x06, 0x10, 0x6d, 0x66, 0x79, 0xa2, 0xe9,
			0xc7, 0xd8, 0x75, 0x89, 0x6b, 0xc7, 0xe0, 0xf3,
			0x46, 0x33, 0x7e, 0x0e, 0x50, 0x5b, 0x2b, 0x96,
			0xcf, 0x4f, 0x3c, 0x1e, 0x28, 0xa3, 0xce, 0x40,
			0x91, 0x9a, 0xab, 0xb1, 0x93, 0x9c, 0xbe, 0x90,
			0xc8, 0xda, 0x55, 0x6f, 0x22, 0xb4, 0xb5, 0xbc,
			0xca, 0x36, 0x91, 0xfe, 0xad, 0x48, 0xbf, 0x6a,
			0xb7, 0xf2, 0x55, 0x83, 0x6a, 0x43, 0x71, 0x9c,
			0x49, 0x19, 0x81, 0xe3, 0x79, 0x1a, 0x13, 0x4d,
			0x91, 0xb9, 0x49, 0x2d, 0x3e, 0x2e, 0x7f, 0xd9,
			0xf2, 0x1e, 0xfa, 0x7f, 0x46, 0x26, 0x3f, 0x66,
			0xe9, 0x80, 0x74, 0x0b, 0xed, 0x75, 0xce, 0x2c,
			0xce, 0x38, 0xbd, 0x87, 0xaa, 0xc5, 0x13, 0xae,
			0x7c, 0x51, 0x33, 0x46, 0x1c, 0xcc, 0x74, 0x71,
			0xee, 0xfb, 0x7b, 0x20, 0x19, 0x0a, 0x84, 0xe8,
			0xcf, 0x44, 0x67, 0x88, 0x98, 0xbc, 0xa9, 0x96,
			0xe6, 0x40, 0x3e, 0x2d, 0xf6, 0x2f, 0x5c, 0x70,
			0xc2, 0x8d, 0xc5, 0xed, 0x2e, 0xcf, 0xf3, 0xa9,
			0xe6, 0x2d, 0xce, 0x5f, 0x80, 0xc1, 0x08, 0x6f,
			0xae, 0x0d, 0xbd, 0x4f, 0xaa, 0x17, 0xe3, 0x54,
			0x9e, 0x8b, 0xd0, 0xc4, 0x73, 0x92, 0x59, 0xc2,
			0xdd, 0x24, 0x7e, 0x1b, 0x31, 0xc0, 0x69, 0xe5,
			0xe5, 0x7d, 0x0e, 0x65, 0x71, 0x5f, 0x7e, 0xa5,
			0x65, 0x41, 0x58, 0x25, 0xca, 0x71, 0x90, 0xbd,
			0xdf, 0x4e, 0x12, 0x2d, 0x1b, 0x9f, 0x6b, 0x0e,
			0x5b, 0x4a, 0x80, 0x3b, 0xb9, 0x83, 0x43, 0x10,
			0xb5, 0x9d, 0x3a, 0x8c, 0x82, 0x4e, 0x83, 0x88,
			0xd4, 0x98, 0x6c, 0x59, 0xde, 0x23, 0xf6, 0x55,
			0xcd, 0x4b, 0xf9, 0x2c, 0x42, 0x17, 0x2f, 0xdd,
			0x07, 0x6c, 0x2f, 0x63, 0x9e, 0x22, 0xfa, 0x81,
			0x35, 0xde, 0x4c, 0x4c, 0x7c, 0x86, 0x06, 0x3d,
			0xac, 0x80, 0x91, 0x00, 0xe3, 0x1a, 0xc5, 0xe5,
			0xf8, 0xed, 0x2d, 0x9f, 0x35, 0x44, 0x89, 0x9d,
		},
		20, /* salt length */
		/* V7 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA1 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0x22, 0xc9, 0xc3, 0xb4, 0x34, 0xef, 0x7d, 0x49,
			0x72, 0x7e, 0x30, 0x44, 0x7f, 0x9f, 0x3c, 0x1d,
			0x43, 0xa6, 0x6b, 0xfd, 0x90, 0x2f, 0xbf, 0x77,
			0x05, 0x6c, 0x6f, 0x73, 0x33, 0xbd, 0x2c, 0x6a,
			0xff, 0x81, 0xa8, 0xd9, 0x47, 0x2d, 0x59, 0xef,
			0xc9, 0x9b, 0x7b, 0x89, 0x40, 0xa4, 0x54, 0x61,
			0xcf, 0xf0, 0xbb, 0x82, 0xa3, 0x0d, 0xdb, 0x00,
			0xd5, 0xbf, 0x35, 0xe5, 0x8d, 0x97, 0xe4, 0x6e,
			0x6d, 0xc7, 0x25, 0x78, 0x2d, 0x35, 0x16, 0x35,
			0x82, 0xce, 0xbe, 0x23, 0x45, 0x1d, 0x4e, 0x3f,
			0xd1, 0x1c, 0xa6, 0x76, 0xad, 0x4a, 0x1b, 0x5c,
			0x6b, 0xfd, 0x78, 0x20, 0xcf, 0xfa, 0x42, 0xc0,
			0x34, 0x43, 0x41, 0x79, 0x2c, 0x30, 0x3b, 0x82,
			0x67, 0xc1, 0x03, 0xf9, 0xd6, 0x33, 0xb3, 0x1e,
			0x0f, 0xd1, 0x35, 0xbf, 0xf4, 0x0d, 0x08, 0xf3,
			0x8f, 0xcf, 0x40, 0x0b, 0x41, 0x0d, 0xcf, 0xc6,
		},
		/* S8 : 384 byte */
		{
			0x3d, 0xe2, 0x72, 0x8c, 0xab, 0xbf, 0x6e, 0xa6,
			0xb5, 0xde, 0x85, 0x54, 0xa2, 0x86, 0xeb, 0x29,
			0x62, 0x7f, 0x0f, 0x01, 0x21, 0xde, 0x44, 0x29,
			0xa1, 0xd1, 0xcb, 0xae, 0x8f, 0x21, 0x45, 0x8f,
			0xc5, 0x20, 0x11, 0x72, 0xb0, 0x2a, 0x5a, 0x61,
			0x62, 0x94, 0x8e, 0xcb, 0x93, 0x6a, 0x40, 0x0b,
			0xe6, 0x34, 0xbe, 0xa8, 0x52, 0x19, 0xc8, 0x93,
			0xcd, 0xec, 0xba, 0x89, 0x3c, 0x99, 0x88, 0x99,
			0x25, 0xd5, 0xf2, 0x4a, 0x66, 0x7e, 0x43, 0x32,
			0xf0, 0x3d, 0xeb, 0x06, 0x64, 0x17, 0x97, 0xbe,
			0x0a, 0xa9, 0xc9, 0xc8, 0xeb, 0x89, 0x07, 0xb7,
			0x58, 0x64, 0xfb, 0x99, 0x04, 0x29, 0xca, 0xb3,
			0x33, 0x41, 0x91, 0xec, 0x72, 0x8c, 0x09, 0x5a,
			0x2a, 0xef, 0x1c, 0x9f, 0x8f, 0xc5, 0x60, 0xaf,
			0x03, 0xb7, 0x53, 0xe9, 0x66, 0x81, 0x1a, 0xa7,
			0xbc, 0x6b, 0x8d, 0x3b, 0x94, 0x3a, 0xae, 0x8a,
			0x8d, 0x0b, 0x64, 0x74, 0x72, 0x0d, 0x28, 0x77,
			0x9b, 0x7a, 0x48, 0x80, 0x73, 0x37, 0x4a, 0x52,
			0x2a, 0x75, 0xcf, 0xc2, 0x9e, 0xc9, 0x49, 0x2b,
			0x75, 0x10, 0xf9, 0x9b, 0x7c, 0x24, 0x09, 0xa2,
			0x97, 0x8a, 0x79, 0xed, 0x52, 0x2a, 0x29, 0xe9,
			0x25, 0xd0, 0x3b, 0xfa, 0xf3, 0xda, 0x78, 0x99,
			0x6d, 0x03, 0x25, 0xb5, 0xf3, 0x51, 0x55, 0x0f,
			0x38, 0xe5, 0xc9, 0x89, 0xa1, 0x66, 0xd4, 0xbd,
			0x74, 0xe0, 0xf0, 0x92, 0xe0, 0x01, 0x87, 0xb8,
			0xd9, 0xe5, 0x95, 0x49, 0x77, 0xa6, 0x16, 0x08,
			0xa7, 0x87, 0xfe, 0x42, 0x5a, 0x8d, 0xf0, 0xfa,
			0x98, 0xd1, 0x7f, 0xae, 0xe5, 0x39, 0xce, 0x2e,
			0xdc, 0x04, 0xd9, 0x45, 0x5e, 0x57, 0x05, 0xe2,
			0xd5, 0x62, 0x94, 0x1e, 0x25, 0xaa, 0x18, 0xb8,
			0x6d, 0xbd, 0xb6, 0x48, 0x76, 0x9b, 0xbb, 0x15,
			0x69, 0xe8, 0x98, 0xa4, 0xcb, 0x26, 0x4a, 0x1c,
			0x88, 0xaa, 0xda, 0xe7, 0x5c, 0x39, 0xf5, 0x43,
			0xa7, 0x1f, 0x6b, 0x51, 0x78, 0x9d, 0x5d, 0x27,
			0x64, 0x1f, 0xce, 0x36, 0x82, 0x35, 0x33, 0x74,
			0xa0, 0xe6, 0xbb, 0x00, 0x85, 0xbb, 0x22, 0x3f,
			0x84, 0xed, 0x6b, 0x08, 0xb9, 0x41, 0xe2, 0x11,
			0x2d, 0x54, 0x32, 0xf3, 0x4b, 0xbb, 0x40, 0x67,
			0xe1, 0x82, 0xc6, 0xbd, 0xbf, 0xd4, 0x32, 0x13,
			0xd4, 0xeb, 0x0a, 0x31, 0x01, 0xd3, 0x40, 0x0f,
			0x4a, 0xed, 0x4e, 0x36, 0x65, 0x91, 0x71, 0x76,
			0x7b, 0xcf, 0x9a, 0x47, 0x51, 0xfb, 0x3d, 0xc9,
			0x76, 0xaf, 0xd6, 0xa9, 0x9f, 0x0e, 0x7c, 0x57,
			0x54, 0x76, 0x41, 0x54, 0x24, 0xe4, 0x8d, 0xa2,
			0x26, 0xaf, 0x75, 0x56, 0xa1, 0x25, 0x5a, 0x36,
			0x3e, 0xe8, 0xf6, 0x8a, 0x97, 0x1f, 0xf1, 0xc1,
			0x10, 0x61, 0xe1, 0xa7, 0x65, 0xdc, 0x78, 0xea,
			0xbc, 0xb5, 0x5f, 0x2e, 0xfe, 0xa4, 0xf3, 0x22,
		},
		20, /* salt length */
		/* V8 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA1 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0x40, 0x62, 0x28, 0xc2, 0xe2, 0xaa, 0x08, 0x3f,
			0x56, 0xce, 0x60, 0x2b, 0xf1, 0x1b, 0x58, 0xe2,
			0x31, 0x6e, 0x6e, 0xb8, 0xab, 0x51, 0x0e, 0x3a,
			0x44, 0x68, 0x62, 0xb6, 0xa4, 0x0c, 0x6b, 0xfd,
			0xe8, 0x35, 0x51, 0x67, 0x97, 0x9d, 0xf7, 0x18,
			0xd8, 0x22, 0xd3, 0x82, 0xe2, 0x6c, 0xfd, 0x2b,
			0xb3, 0xd1, 0x85, 0xb0, 0x05, 0x75, 0x0a, 0x0b,
			0x7e, 0xb6, 0x88, 0x11, 0x27, 0xc7, 0x72, 0x3e,
			0x1f, 0x9a, 0x01, 0xb5, 0x91, 0x04, 0xcb, 0x83,
			0xe3, 0x1c, 0x97, 0x24, 0x30, 0xe4, 0x70, 0x48,
			0xf9, 0x2b, 0xf8, 0xd7, 0x17, 0xb1, 0xff, 0x32,
			0xf4, 0x8d, 0x54, 0xbc, 0xfb, 0xcc, 0xdb, 0x99,
			0x4d, 0x5b, 0xe0, 0xb5, 0xb6, 0x27, 0xab, 0x08,
			0xdf, 0x82, 0x55, 0x19, 0xc5, 0xc9, 0xda, 0xc1,
			0x69, 0x43, 0x6e, 0x32, 0xca, 0x4d, 0x15, 0x37,
			0x10, 0xb4, 0x6e, 0xbe, 0x0a, 0x63, 0xcf, 0x93,
		},
		/* S9 : 384 byte */
		{
			0x5e, 0x67, 0x39, 0x79, 0x48, 0xa7, 0x2b, 0xb9,
			0xc3, 0x72, 0x50, 0xc2, 0xa7, 0xce, 0x9b, 0x72,
			0x15, 0xbc, 0x96, 0xaf, 0x18, 0x99, 0x17, 0x7d,
			0xfb, 0xd2, 0xb2, 0x7a, 0x33, 0xcd, 0x91, 0x4c,
			0x2d, 0x13, 0x79, 0xc4, 0x37, 0x83, 0xdf, 0xf0,
			0xd9, 0x31, 0x8e, 0x85, 0x13, 0xc0, 0x65, 0x17,
			0x8a, 0xb2, 0x86, 0x12, 0x8f, 0xef, 0xe4, 0x19,
			0x50, 0x0a, 0x9f, 0x8a, 0x81, 0xee, 0xdf, 0x11,
			0x4d, 0xca, 0xdf, 0xe6, 0x59, 0xb9, 0x83, 0x4e,
			0x74, 0xdc, 0x06, 0x85, 0x00, 0xdc, 0x2e, 0x19,
			0x30, 0xff, 0xdf, 0x4f, 0xeb, 0xc3, 0x9b, 0x92,
			0x44, 0x65, 0xe7, 0xe7, 0xb0, 0x17, 0x50, 0x4c,
			0x09, 0xdc, 0xd7, 0xd8, 0xd2, 0x5e, 0xea, 0x50,
			0xa7, 0xfe, 0x3b, 0xd3, 0x5f, 0x7e, 0xfc, 0xcf,
			0x90, 0x38, 0x4d, 0xfc, 0x88, 0x6f, 0xc3, 0x85,
			0xe5, 0xc8, 0xeb, 0x54, 0x1c, 0xc4, 0xcf, 0x13,
			0xc2, 0x9e, 0x53, 0xcd, 0xa9, 0xfe, 0xa7, 0xb5,
			0xf2, 0x7d, 0x6d, 0x94, 0x0f, 0xde, 0x0a, 0x7f,
			0x66, 0x63, 0x6d, 0xee, 0x87, 0x04, 0x49, 0x39,
			0x0e, 0xa8, 0xe8, 0x34, 0xd7, 0xb0, 0xff, 0x26,
			0x93, 0x71, 0x79, 0x9a, 0xa2, 0x09, 0x62, 0xe2,
			0x8e, 0xa1, 0xbb, 0x5f, 0xfa, 0xb6, 0x49, 0xbc,
			0x68, 0x9a, 0x67, 0x03, 0x45, 0x71, 0x5f, 0xbd,
			0x39, 0xde, 0x1c, 0x2e, 0x33, 0xb8, 0x45, 0x1a,
			0x3e, 0x4b, 0xda, 0x34, 0x6c, 0x35, 0x54, 0x3d,
			0xf2, 0x20, 0xd5, 0x4b, 0x5d, 0x9b, 0xcf, 0x73,
			0x7d, 0xad, 0x87, 0xdc, 0x8d, 0x7a, 0x45, 0x0f,
			0x56, 0xaa, 0x60, 0x54, 0x27, 0xdc, 0x70, 0xf6,
			0x12, 0x5a, 0xf1, 0x75, 0xb9, 0x00, 0x17, 0x50,
			0xde, 0xef, 0xbb, 0xdf, 0x9e, 0x5a, 0xb5, 0xd0,
			0x9b, 0xfa, 0xc4, 0xad, 0xc8, 0x3c, 0x1f, 0x2d,
			0x35, 0x1d, 0xca, 0x08, 0x78, 0xee, 0x96, 0x51,
			0x35, 0x89, 0xf5, 0xda, 0xc7, 0x76, 0x57, 0x23,
			0xdb, 0xa7, 0xa0, 0x81, 0x2a, 0x9d, 0x9d, 0xda,
			0x92, 0x93, 0xb0, 0x07, 0x48, 0xe7, 0x94, 0x81,
			0xcb, 0x0a, 0xf3, 0xeb, 0xa7, 0x16, 0xa9, 0x33,
			0xfb, 0x94, 0x52, 0xba, 0x25, 0xaf, 0x07, 0xfd,
			0xbf, 0xe6, 0x50, 0xb7, 0xc2, 0xc0, 0x03, 0x2b,
			0x10, 0x5d, 0x63, 0xf1, 0xe7, 0xb6, 0xfe, 0x84,
			0xef, 0x64, 0xbd, 0xcc, 0xd2, 0x13, 0xa7, 0x02,
			0x16, 0x5c, 0xd6, 0x0d, 0x29, 0xe3, 0x4a, 0x5b,
			0x14, 0x1d, 0x9e, 0xa1, 0x04, 0x99, 0x1c, 0x4b,
			0x8d, 0x09, 0x9f, 0xac, 0xf8, 0x11, 0x11, 0x73,
			0x48, 0xa9, 0xe7, 0x7b, 0x7d, 0xfa, 0x25, 0xee,
			0x3c, 0xd6, 0xbe, 0x45, 0xb7, 0x22, 0x52, 0x89,
			0xb1, 0x51, 0x79, 0x9e, 0x2b, 0xe2, 0x4e, 0xc9,
			0x7a, 0x4c, 0xb3, 0x9f, 0xb0, 0x9b, 0xa0, 0x07,
			0x2f, 0xa4, 0xd7, 0xc1, 0x10, 0x40, 0x13, 0x3b,
		},
		20, /* salt length */
		/* V9 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
};

/* 10 elements */
static mod3072_t SHA224[] = {
	/* SHA224 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0x1e, 0x4f, 0x71, 0xd6, 0x7b, 0x80, 0x41, 0x84,
			0x5a, 0x67, 0x41, 0xa2, 0xe8, 0x4b, 0x31, 0x3f,
			0x03, 0x5f, 0x04, 0xd6, 0x4e, 0x8c, 0x92, 0x2e,
			0x84, 0x71, 0x8d, 0x7f, 0x0c, 0xa9, 0xb6, 0xd6,
			0xce, 0x4c, 0x50, 0xba, 0x46, 0xb8, 0xd5, 0x10,
			0xd6, 0x91, 0xe9, 0x3c, 0x61, 0x06, 0x8c, 0x89,
			0x15, 0x56, 0x93, 0xcb, 0x88, 0x93, 0x59, 0x43,
			0x07, 0xa7, 0xb2, 0xc2, 0x2b, 0x94, 0x20, 0x11,
			0xac, 0x00, 0x4a, 0x91, 0x7a, 0xf0, 0xa9, 0x1f,
			0x0a, 0xd4, 0x85, 0x3a, 0xee, 0xc4, 0x20, 0x68,
			0xa9, 0x09, 0x31, 0xd5, 0xc1, 0xdf, 0x93, 0x3e,
			0x16, 0x79, 0x3f, 0x0d, 0x71, 0x46, 0x78, 0xc6,
			0x60, 0x73, 0x45, 0xa1, 0x42, 0xb1, 0x24, 0x79,
			0x9e, 0x38, 0xfd, 0xe4, 0xb9, 0x0b, 0x55, 0xa4,
			0x67, 0x7e, 0xc4, 0x3e, 0x21, 0xf6, 0xa9, 0xe8,
			0x58, 0xf1, 0x1c, 0xa8, 0x09, 0x46, 0x24, 0xbb,
		},
		/* S0 : 384 byte */
		{
			0x71, 0x71, 0xc7, 0x4d, 0xf2, 0x42, 0x72, 0xdf,
			0xe6, 0xb3, 0x4d, 0xb7, 0x8f, 0x24, 0x50, 0x7a,
			0x68, 0x06, 0x2b, 0xd7, 0x91, 0xf6, 0x87, 0x96,
			0xd5, 0x00, 0x1b, 0xe3, 0x54, 0xde, 0x6f, 0xdd,
			0xab, 0x81, 0xe9, 0x25, 0x2e, 0x15, 0x18, 0x84,
			0xf4, 0xcc, 0x1f, 0x3c, 0xd3, 0xe7, 0x76, 0x0e,
			0x26, 0x3c, 0x0c, 0x34, 0xe6, 0x3c, 0x55, 0x7e,
			0xb3, 0x2c, 0x83, 0x36, 0xe0, 0xce, 0xf4, 0x08,
			0x55, 0xc5, 0xe2, 0x79, 0xdb, 0xba, 0x31, 0x70,
			0xda, 0x5a, 0x14, 0xac, 0x60, 0xe4, 0xcc, 0x8d,
			0x40, 0x26, 0x33, 0xa3, 0x83, 0xb8, 0x87, 0x09,
			0xf3, 0x30, 0x6f, 0xb0, 0x27, 0x08, 0xe3, 0x9f,
			0x30, 0x39, 0xe7, 0xe6, 0x14, 0xed, 0xcb, 0x89,
			0x60, 0x9c, 0x8c, 0x71, 0x13, 0x7d, 0xe5, 0x21,
			0x16, 0x59, 0xa4, 0x1e, 0x9e, 0x56, 0x82, 0xcf,
			0xe0, 0x46, 0x3f, 0x3b, 0xc9, 0x75, 0x58, 0xd3,
			0xbf, 0x77, 0xbd, 0x79, 0x89, 0x76, 0xf0, 0x9d,
			0xb6, 0x91, 0x53, 0x12, 0x39, 0x23, 0x83, 0x5a,
			0xc9, 0xbb, 0xd7, 0x64, 0x8c, 0x27, 0x73, 0xe3,
			0x8b, 0x52, 0x28, 0x64, 0x0f, 0xde, 0x6d, 0xf0,
			0x05, 0xe9, 0xf4, 0x48, 0x19, 0xec, 0xa3, 0x1f,
			0x41, 0xcc, 0xdd, 0xbd, 0x45, 0xd6, 0x1a, 0xe7,
			0xe1, 0xed, 0x06, 0x40, 0xf0, 0x73, 0x6f, 0x52,
			0xbf, 0x5f, 0xc1, 0xc6, 0x2f, 0x54, 0x30, 0xde,
			0x6a, 0x96, 0xd5, 0xaa, 0xbc, 0xcf, 0xcf, 0xef,
			0x50, 0x8a, 0xc2, 0x99, 0xc7, 0xf3, 0xf0, 0xf7,
			0xd2, 0x22, 0xef, 0x1f, 0x19, 0xb2, 0x88, 0x27,
			0x36, 0x90, 0xb3, 0x27, 0x5b, 0x68, 0xf8, 0x74,
			0x30, 0x1a, 0xfa, 0x95, 0xd2, 0x43, 0x31, 0x62,
			0x84, 0xed, 0x11, 0x7b, 0xde, 0xd6, 0x9d, 0xa1,
			0x1f, 0x5c, 0xe1, 0x43, 0x5d, 0xd6, 0x77, 0x17,
			0xba, 0xe8, 0x2e, 0xd4, 0x68, 0xff, 0x1b, 0x6a,
			0xc7, 0xf2, 0x48, 0x33, 0x97, 0xd3, 0x10, 0xff,
			0xe9, 0x17, 0x75, 0x18, 0x9f, 0x67, 0x1a, 0x82,
			0xb4, 0x93, 0x03, 0x9d, 0x8c, 0x23, 0x38, 0x30,
			0xd2, 0x0e, 0x29, 0x0b, 0xc9, 0xbe, 0x88, 0x0a,
			0x47, 0xf0, 0xb3, 0x6b, 0xf2, 0xe1, 0xda, 0x2c,
			0x1f, 0x23, 0xda, 0xfe, 0xb9, 0xf4, 0x2d, 0x9f,
			0x08, 0x4f, 0xeb, 0x80, 0x8a, 0x98, 0xe8, 0x94,
			0xe8, 0x50, 0x19, 0x37, 0xba, 0x93, 0x25, 0x94,
			0xa6, 0xd2, 0x02, 0xe2, 0x0a, 0x0a, 0xfd, 0xdc,
			0xef, 0x8f, 0xa4, 0x8c, 0x16, 0x82, 0xd3, 0x17,
			0x9e, 0xde, 0xbf, 0x8e, 0xa4, 0x4e, 0xa1, 0x21,
			0x6a, 0x2f, 0x55, 0xc3, 0x05, 0xcd, 0xf4, 0x87,
			0x24, 0x90, 0x10, 0x90, 0x9f, 0xa8, 0xa2, 0x1d,
			0x9b, 0xa9, 0xe3, 0xdb, 0xbe, 0xec, 0x04, 0x6a,
			0x82, 0x39, 0x22, 0x39, 0x0b, 0x7d, 0x90, 0x2d,
			0x77, 0xec, 0x17, 0x6b, 0xb4, 0x47, 0xb0, 0x5d,
		},
		20, /* salt length */
		/* V0 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA224 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0xf2, 0x24, 0x8c, 0x0b, 0x53, 0x5f, 0x00, 0x79,
			0xeb, 0x32, 0xf9, 0x7e, 0x50, 0xeb, 0x59, 0x04,
			0x11, 0xed, 0xc3, 0xae, 0x99, 0x06, 0xa5, 0xc6,
			0x56, 0xfc, 0x76, 0x77, 0x1b, 0x69, 0x3c, 0x3c,
			0xc3, 0xc9, 0x52, 0x98, 0x18, 0xd2, 0x68, 0x08,
			0xf3, 0x04, 0xcb, 0x36, 0xa4, 0x80, 0xe7, 0x6f,
			0x93, 0x1d, 0x0a, 0x57, 0x28, 0xb9, 0x6a, 0xdb,
			0xe9, 0xcd, 0x6a, 0x68, 0xe2, 0x44, 0x19, 0x5a,
			0x4a, 0x8e, 0xfb, 0xb8, 0x63, 0xf0, 0x0d, 0x1e,
			0xa2, 0x1e, 0x03, 0xfa, 0x10, 0x0b, 0xd4, 0xfb,
			0xf0, 0x00, 0x78, 0xc7, 0xa2, 0xec, 0xf0, 0xed,
			0xc7, 0xfb, 0x88, 0x1a, 0xa6, 0x69, 0x59, 0x8f,
			0xcf, 0x99, 0x11, 0xd2, 0xfd, 0x61, 0xf7, 0xa4,
			0x41, 0xa8, 0x29, 0x34, 0x5d, 0x34, 0x28, 0x92,
			0x6e, 0xbd, 0x97, 0x66, 0xcf, 0x46, 0x40, 0xe2,
			0x39, 0xad, 0x58, 0xf5, 0x2f, 0x80, 0x04, 0xc2,
		},
		/* S1 : 384 byte */
		{
			0x60, 0x0d, 0xff, 0x1c, 0xd6, 0x31, 0xf8, 0x8f,
			0x57, 0x89, 0xcc, 0x0e, 0xcf, 0x0b, 0x8c, 0xf9,
			0x5b, 0xa8, 0xc8, 0x51, 0xe1, 0x56, 0x88, 0x80,
			0xc1, 0xde, 0x92, 0xeb, 0x44, 0x8d, 0x28, 0xf0,
			0x24, 0xca, 0xf2, 0x05, 0x2a, 0x1c, 0x3b, 0xcc,
			0x31, 0xaf, 0x85, 0xd7, 0x97, 0x64, 0x7b, 0xd4,
			0xd9, 0x9e, 0xab, 0x1a, 0x02, 0xa2, 0xfe, 0x0a,
			0x00, 0xfb, 0x1d, 0x07, 0x9b, 0x6b, 0x87, 0x0f,
			0xcf, 0x32, 0x47, 0x79, 0xf0, 0x88, 0x4e, 0xc2,
			0x73, 0xc4, 0x1b, 0x28, 0x0a, 0x42, 0x8e, 0x1c,
			0x5e, 0x7b, 0x17, 0x0d, 0xaa, 0xa2, 0x7e, 0xe0,
			0xfd, 0x46, 0xb1, 0x57, 0x69, 0x0f, 0xda, 0x21,
			0x46, 0xd3, 0x4e, 0x1f, 0x5d, 0x13, 0x99, 0x0a,
			0xc7, 0x3b, 0x41, 0x79, 0xff, 0xe5, 0x99, 0x0e,
			0x1f, 0x94, 0x93, 0x40, 0xc7, 0x71, 0xf5, 0xbb,
			0x21, 0xb7, 0x35, 0x0d, 0x9f, 0x9a, 0xb0, 0x91,
			0xc5, 0xeb, 0x4d, 0x90, 0xe4, 0xf3, 0x84, 0xee,
			0x71, 0x1e, 0xa4, 0xbb, 0x80, 0x17, 0xa6, 0x89,
			0x96, 0x9b, 0xfe, 0xc5, 0x45, 0x5b, 0x80, 0xf6,
			0x7a, 0x29, 0x98, 0x52, 0x8c, 0xd4, 0xdf, 0x85,
			0x5b, 0x75, 0x4c, 0x7c, 0xfc, 0x77, 0xc0, 0x2b,
			0xf7, 0xd7, 0x94, 0x0f, 0xc9, 0x86, 0xf0, 0x43,
			0xce, 0xf0, 0x13, 0x5b, 0xf4, 0x3e, 0xcc, 0x11,
			0x46, 0xe7, 0x01, 0x1c, 0xf4, 0xe2, 0xe1, 0xd4,
			0xed, 0x81, 0x14, 0xeb, 0xa8, 0x33, 0x4f, 0x07,
			0xf5, 0x36, 0x5b, 0x36, 0x34, 0xf6, 0x04, 0xb9,
			0x1a, 0xee, 0x02, 0x3e, 0x7c, 0x1e, 0xb5, 0x67,
			0x7e, 0x9d, 0x8e, 0xf9, 0xc1, 0xa9, 0x2e, 0xc7,
			0xbb, 0xac, 0x7e, 0x86, 0xc2, 0xa5, 0xd5, 0x9b,
			0x81, 0x65, 0x27, 0x7a, 0xd0, 0xd0, 0xcf, 0xf4,
			0xac, 0x23, 0x87, 0xf9, 0x3a, 0x56, 0x00, 0xd7,
			0x06, 0x67, 0x19, 0x61, 0xc5, 0x51, 0x87, 0xa4,
			0xe8, 0x56, 0x7a, 0x10, 0x6a, 0x9c, 0xfe, 0x6f,
			0x6d, 0x30, 0x08, 0xdc, 0x42, 0x45, 0xe9, 0xb0,
			0x53, 0xa9, 0xb2, 0xdf, 0xb0, 0x33, 0x6c, 0x4d,
			0xe2, 0x19, 0x02, 0x68, 0xe8, 0x64, 0xa3, 0x57,
			0xfd, 0x2c, 0x52, 0x1b, 0xd1, 0x29, 0x6b, 0x43,
			0x72, 0x3c, 0xf0, 0x9c, 0x3c, 0x9f, 0xa0, 0x24,
			0x12, 0x73, 0xff, 0xb5, 0x68, 0x96, 0x6e, 0x9a,
			0xfd, 0xc1, 0x18, 0x82, 0xa4, 0x92, 0xc5, 0x49,
			0xa4, 0xad, 0x5b, 0x04, 0x43, 0x39, 0xa6, 0xdc,
			0x75, 0x35, 0x7c, 0x60, 0xc4, 0x58, 0x56, 0x92,
			0x9f, 0x84, 0xf7, 0xce, 0xf1, 0x2c, 0xc4, 0x9c,
			0x9f, 0xf3, 0x2e, 0xe7, 0xac, 0x9c, 0xae, 0xcd,
			0x40, 0x71, 0x56, 0x10, 0x79, 0x3b, 0x58, 0xa0,
			0x99, 0xf1, 0x87, 0x54, 0x59, 0x6b, 0xd6, 0x08,
			0x75, 0x3b, 0x13, 0x4b, 0x05, 0x42, 0x55, 0x44,
			0x0e, 0xcd, 0x61, 0xd0, 0x6e, 0x0a, 0x1a, 0xf9,
		},
		20, /* salt length */
		/* V1 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA224 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x13, 0xb0, 0x2c, 0x35, 0x51, 0x46, 0xbb, 0xcb,
			0xfe, 0x2e, 0x75, 0x8e, 0x2c, 0x86, 0x0a, 0x34,
			0x44, 0x5e, 0x5c, 0xef, 0x40, 0x1d, 0x49, 0xf2,
			0x1b, 0x1f, 0xc4, 0x32, 0x86, 0xa3, 0x2b, 0x69,
			0xdd, 0xe5, 0xd4, 0x61, 0x59, 0x87, 0xea, 0x6f,
			0x87, 0x35, 0x77, 0x3d, 0xf9, 0x89, 0xc1, 0xcd,
			0x3b, 0x2c, 0xda, 0xbb, 0x08, 0x3f, 0x73, 0xc3,
			0xbe, 0xf9, 0xf1, 0x82, 0x10, 0x91, 0xc3, 0x81,
			0xfb, 0x5e, 0xf2, 0x6f, 0xfb, 0x3d, 0xf9, 0xae,
			0x7e, 0x33, 0xd8, 0x12, 0x99, 0x53, 0x64, 0xe5,
			0xfa, 0xe2, 0xe1, 0x2e, 0x1e, 0xba, 0xd2, 0x32,
			0xc4, 0xec, 0x10, 0x3c, 0xa1, 0xa9, 0x1a, 0x9a,
			0x55, 0x63, 0xac, 0xe8, 0x9e, 0x2f, 0x93, 0x91,
			0xcb, 0x6e, 0xbf, 0x92, 0x74, 0xac, 0x9a, 0x81,
			0x69, 0xc8, 0x98, 0xce, 0xe8, 0x55, 0x0f, 0x4a,
			0x13, 0x41, 0x4a, 0xe5, 0xa0, 0xb4, 0xd9, 0xbc,
		},
		/* S2 : 384 byte */
		{
			0x29, 0xf2, 0x56, 0x34, 0x4f, 0x37, 0xfa, 0x5e,
			0xd7, 0xed, 0x4b, 0xa7, 0x2b, 0xb7, 0xbf, 0x39,
			0xf6, 0xfa, 0x64, 0x84, 0x40, 0x14, 0xcb, 0x18,
			0xb5, 0xe2, 0xb3, 0x10, 0x7d, 0x87, 0x11, 0x41,
			0x1e, 0x0d, 0xc6, 0x89, 0x7e, 0x44, 0x76, 0xca,
			0x60, 0x75, 0xfd, 0x10, 0x82, 0xea, 0xde, 0x97,
			0x2f, 0xf4, 0xb6, 0x41, 0x60, 0x2c, 0xfe, 0x61,
			0xe2, 0x81, 0xf2, 0x3c, 0x24, 0x5f, 0xb6, 0xfc,
			0x30, 0xbe, 0x27, 0x88, 0x0b, 0x2e, 0x90, 0x24,
			0xbc, 0xd7, 0x0f, 0xf0, 0x8a, 0x1e, 0xd0, 0xc1,
			0xee, 0x0d, 0x12, 0xa6, 0xd4, 0x26, 0xce, 0x5a,
			0x30, 0x4d, 0x25, 0xf4, 0xec, 0x40, 0x7a, 0x5e,
			0x84, 0xda, 0xed, 0xf9, 0x65, 0x6a, 0xce, 0x90,
			0xe5, 0xa2, 0x1f, 0x12, 0x13, 0x12, 0xfc, 0xd7,
			0xa0, 0x50, 0x94, 0xc3, 0xc9, 0xe5, 0x7e, 0x21,
			0xee, 0x35, 0x06, 0xed, 0xae, 0x2d, 0x02, 0x4c,
			0x72, 0xd8, 0xb8, 0xa5, 0x07, 0xc3, 0x5a, 0x1a,
			0xa6, 0x86, 0x1d, 0x52, 0x0e, 0xec, 0xb7, 0x70,
			0x41, 0xfb, 0x93, 0xcb, 0xbe, 0xce, 0x0f, 0x70,
			0x5d, 0x1a, 0xe3, 0x4e, 0x79, 0x8e, 0x7a, 0xd1,
			0xd5, 0xd0, 0x66, 0x88, 0x30, 0xfa, 0x96, 0xd5,
			0x26, 0x96, 0xf2, 0x2c, 0xc4, 0x69, 0x5b, 0xbf,
			0x28, 0xda, 0x4f, 0xd4, 0x2b, 0x07, 0x64, 0x7b,
			0x3c, 0xe8, 0x2f, 0x9c, 0x4b, 0xfa, 0xba, 0xec,
			0x71, 0xe4, 0xbb, 0x36, 0x3e, 0x73, 0xa7, 0x73,
			0xb1, 0xf8, 0x05, 0xe6, 0x5e, 0x59, 0x3a, 0x3f,
			0x77, 0x79, 0xbb, 0x43, 0x5f, 0xc6, 0x5b, 0x09,
			0xa4, 0x88, 0xc0, 0x4f, 0x26, 0x81, 0xef, 0xf0,
			0xcf, 0x79, 0xe1, 0x3e, 0xbd, 0xf7, 0x29, 0xf5,
			0x20, 0xa5, 0xad, 0x23, 0x4a, 0x55, 0x36, 0x3a,
			0x3a, 0x4d, 0x79, 0xff, 0x4f, 0xd6, 0x86, 0xa2,
			0x1e, 0xfe, 0xd6, 0x0c, 0x1c, 0x0d, 0x27, 0x53,
			0x5b, 0x9e, 0x5b, 0x78, 0x3e, 0x55, 0x5c, 0x91,
			0x93, 0xd7, 0x13, 0x3c, 0xdf, 0xe7, 0xbb, 0x61,
			0xf1, 0xf0, 0x6b, 0xe2, 0xef, 0x54, 0xfc, 0xa7,
			0x42, 0x06, 0x78, 0x20, 0xf0, 0x5c, 0x83, 0x14,
			0x63, 0x00, 0x3a, 0xa6, 0x61, 0x2c, 0x6b, 0x66,
			0x3f, 0x61, 0x62, 0xa5, 0x34, 0x70, 0xcd, 0x22,
			0x32, 0x74, 0xb3, 0xa6, 0x14, 0x6d, 0x77, 0x38,
			0xd0, 0xa9, 0xde, 0xae, 0x49, 0xb4, 0x31, 0xab,
			0xcd, 0xa9, 0x73, 0x6a, 0x06, 0x4d, 0xa3, 0xc7,
			0x6e, 0xf8, 0xb1, 0xa1, 0xfe, 0x08, 0x73, 0xd1,
			0x18, 0xbc, 0xdb, 0x64, 0x3b, 0x60, 0x41, 0xa0,
			0xf6, 0x6e, 0xe4, 0x37, 0x6e, 0x6f, 0x75, 0x2a,
			0x3e, 0x21, 0xf4, 0xa2, 0x7d, 0x06, 0x83, 0x98,
			0x82, 0x77, 0x91, 0xae, 0xeb, 0x70, 0x97, 0x1b,
			0x25, 0xda, 0xfc, 0x4e, 0x3b, 0xc4, 0x00, 0x25,
			0x62, 0x6a, 0x22, 0x66, 0xfa, 0x28, 0xc1, 0xb5,
		},
		20, /* salt length */
		/* V2 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA224 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0xfd, 0xc5, 0xc5, 0xe6, 0x93, 0x0e, 0x04, 0x20,
			0x43, 0x20, 0xa7, 0xe8, 0x15, 0x72, 0x9c, 0x8a,
			0xf0, 0xde, 0xa6, 0xc8, 0x5f, 0x51, 0x48, 0x2d,
			0x94, 0xc7, 0xe4, 0x9e, 0xae, 0x77, 0x24, 0x32,
			0x37, 0x4a, 0x99, 0x44, 0xf1, 0x92, 0xa2, 0xc6,
			0xd2, 0xf9, 0x90, 0xd7, 0x4d, 0xb7, 0x07, 0x98,
			0xd3, 0x09, 0xa7, 0x3c, 0x9a, 0xde, 0xec, 0x5f,
			0x9d, 0xd3, 0x53, 0x99, 0xe4, 0x61, 0xbf, 0x52,
			0x05, 0x55, 0x1a, 0xd8, 0xeb, 0x81, 0x3f, 0xae,
			0xd6, 0xb2, 0x18, 0x28, 0xa3, 0xe2, 0x03, 0x34,
			0x4e, 0xae, 0x7b, 0x7b, 0xb5, 0x0a, 0xf4, 0xa9,
			0x9e, 0x33, 0x06, 0x08, 0xfc, 0xe5, 0xce, 0x54,
			0x6e, 0xcf, 0x92, 0xcb, 0x09, 0x44, 0xa3, 0x83,
			0x57, 0x33, 0x88, 0x00, 0x9f, 0x59, 0x58, 0x06,
			0x67, 0xb5, 0x68, 0xac, 0x38, 0x3c, 0x26, 0xb2,
			0xa1, 0xcf, 0x45, 0x14, 0x7e, 0x6a, 0x1b, 0xdf,
		},
		/* S3 : 384 byte */
		{
			0x51, 0x0e, 0x64, 0x52, 0xf9, 0x74, 0x6f, 0x3f,
			0x29, 0xa1, 0x4a, 0x28, 0xfb, 0xba, 0x3f, 0x76,
			0xe4, 0x6e, 0x75, 0xc9, 0x92, 0x52, 0xb1, 0xa8,
			0x79, 0xe2, 0x6a, 0x96, 0x6d, 0x31, 0xf2, 0xf4,
			0x73, 0x31, 0xe0, 0xa1, 0x3b, 0x66, 0xf1, 0x8b,
			0x47, 0x02, 0xaa, 0xd8, 0x02, 0x05, 0xd2, 0x23,
			0xe6, 0x45, 0xe9, 0x80, 0xca, 0x34, 0x9f, 0xb6,
			0x4c, 0x14, 0x96, 0xab, 0xe0, 0x43, 0xcf, 0xfe,
			0xf7, 0x36, 0xde, 0xfa, 0x04, 0x34, 0x0a, 0xa5,
			0xb7, 0x35, 0xe9, 0xd1, 0x90, 0xdd, 0x35, 0x09,
			0x84, 0xa6, 0x3e, 0xf5, 0x55, 0x4f, 0x34, 0x28,
			0x5e, 0x60, 0xb5, 0x28, 0x25, 0xae, 0xf5, 0x98,
			0x68, 0xdf, 0xe2, 0xa9, 0x66, 0xa4, 0x65, 0xc8,
			0x4c, 0xd2, 0xb9, 0xfa, 0x1f, 0x2c, 0x35, 0xc3,
			0xe3, 0x24, 0x5d, 0xfb, 0x33, 0x2d, 0xde, 0x9e,
			0x2d, 0xf4, 0x1c, 0xd0, 0x56, 0x30, 0x07, 0xd0,
			0xf6, 0xc4, 0xfd, 0xca, 0x68, 0x0b, 0x8a, 0xd9,
			0xe5, 0x8f, 0x49, 0x50, 0x58, 0xf5, 0x4c, 0xf6,
			0xa7, 0x81, 0x7b, 0xa0, 0x19, 0x69, 0xc3, 0xe2,
			0x5c, 0xe5, 0x74, 0x87, 0x42, 0x54, 0xf2, 0x4a,
			0x25, 0xbf, 0xdd, 0x8c, 0x13, 0x5c, 0x8c, 0x71,
			0xe5, 0xbb, 0xe7, 0x76, 0x41, 0x4a, 0xe2, 0x2b,
			0x52, 0x15, 0x70, 0x1a, 0x1a, 0x01, 0xb8, 0xcc,
			0x6b, 0x26, 0x45, 0xb0, 0x84, 0x45, 0x4f, 0x5e,
			0x3a, 0x00, 0xf5, 0x08, 0xa7, 0x54, 0xef, 0x2b,
			0x24, 0x19, 0x27, 0x0c, 0xae, 0xa6, 0xdb, 0x18,
			0x3c, 0x33, 0x0b, 0x81, 0xbe, 0x34, 0xae, 0xc0,
			0xd3, 0xea, 0x4a, 0x0e, 0x7b, 0x32, 0xb5, 0x27,
			0x7b, 0xe4, 0xec, 0x87, 0xdf, 0x07, 0x84, 0x19,
			0x2e, 0xa3, 0xf2, 0xe5, 0xa7, 0xee, 0x37, 0x4a,
			0xea, 0x2e, 0xe2, 0xaf, 0x80, 0x47, 0xa9, 0x1a,
			0x47, 0x31, 0x4e, 0x99, 0xbc, 0xbd, 0x6c, 0xc4,
			0x60, 0xad, 0x7c, 0x16, 0xc2, 0xfc, 0xba, 0xfd,
			0x4c, 0xa2, 0x2e, 0xe7, 0x8c, 0x19, 0x08, 0x91,
			0x94, 0x13, 0xff, 0x00, 0xea, 0x1e, 0x2d, 0xc8,
			0x29, 0xca, 0x28, 0x0a, 0xc1, 0x87, 0xce, 0x16,
			0xec, 0x14, 0xbd, 0x9b, 0x22, 0x2b, 0x8e, 0x8b,
			0x6a, 0x92, 0xcb, 0xe0, 0x90, 0x2e, 0xf6, 0x17,
			0x38, 0x75, 0x8e, 0x26, 0x75, 0x98, 0xfc, 0x06,
			0x5e, 0x35, 0x5e, 0xb2, 0x19, 0x45, 0x26, 0xd7,
			0x87, 0x96, 0x7e, 0xbf, 0x44, 0xa0, 0x5d, 0x26,
			0x89, 0xe7, 0xfd, 0xf7, 0xa6, 0xd2, 0xba, 0x78,
			0x02, 0x61, 0x37, 0xed, 0x59, 0x61, 0xa7, 0x6f,
			0xcf, 0x30, 0xc7, 0xac, 0xe8, 0x2b, 0x1d, 0xbb,
			0x8d, 0x06, 0x95, 0x91, 0x94, 0x9c, 0x9b, 0xb3,
			0x26, 0x13, 0x46, 0x61, 0x04, 0xb2, 0xac, 0x63,
			0x85, 0x69, 0x1b, 0xb9, 0xe1, 0x79, 0xf6, 0x95,
			0xb0, 0x9d, 0x78, 0x49, 0x64, 0x4f, 0x5a, 0xa1,
		},
		20, /* salt length */
		/* V3 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA224 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0xb5, 0xf6, 0xa9, 0x5d, 0x47, 0x26, 0xa5, 0x0a,
			0xa1, 0x33, 0x68, 0x3c, 0xd6, 0x75, 0x3e, 0x24,
			0x20, 0xce, 0xd1, 0xd2, 0x24, 0xde, 0x92, 0x03,
			0x77, 0x75, 0x5a, 0x3a, 0x60, 0x4e, 0x51, 0x12,
			0xe0, 0xfa, 0x10, 0xd2, 0xef, 0x1a, 0x02, 0x42,
			0x09, 0xd1, 0x10, 0xbd, 0xe9, 0x79, 0x7d, 0x88,
			0x95, 0x95, 0x0e, 0x62, 0x9a, 0x07, 0x6f, 0x5c,
			0x99, 0x84, 0xd2, 0xeb, 0x02, 0x44, 0xd9, 0x3c,
			0x20, 0xfe, 0xb6, 0xc7, 0x56, 0x13, 0x91, 0x27,
			0xa8, 0x0a, 0x29, 0x29, 0x7c, 0x3c, 0xbf, 0x23,
			0xa2, 0xd0, 0x30, 0x87, 0x93, 0xea, 0x61, 0xb7,
			0xf6, 0x20, 0xdf, 0x1b, 0x69, 0x2e, 0xc7, 0x72,
			0x7d, 0xea, 0xe4, 0x65, 0x02, 0x7a, 0x18, 0x62,
			0x86, 0xc4, 0xfd, 0xa7, 0x18, 0x26, 0xcb, 0x9d,
			0x52, 0x67, 0x7d, 0x68, 0x96, 0xee, 0x2f, 0xbc,
			0x97, 0x32, 0xcb, 0xf2, 0x1b, 0x72, 0xe4, 0x5b,
		},
		/* S4 : 384 byte */
		{
			0x18, 0x5a, 0x35, 0x8e, 0x80, 0x8b, 0xab, 0x45,
			0xfb, 0x9a, 0x42, 0x29, 0x68, 0x85, 0x67, 0x4e,
			0xad, 0x4e, 0xe8, 0x38, 0xfa, 0x18, 0x32, 0x2a,
			0xcd, 0xf9, 0xda, 0xea, 0x54, 0x56, 0x49, 0x9b,
			0xa4, 0x91, 0x9b, 0x39, 0x98, 0x44, 0x47, 0x20,
			0x2c, 0x38, 0x15, 0x2c, 0x81, 0x26, 0xa0, 0x95,
			0xeb, 0x5e, 0x8d, 0xeb, 0x86, 0x5d, 0x53, 0xd1,
			0xb6, 0xcc, 0x98, 0xdc, 0x93, 0x74, 0x1b, 0x67,
			0x33, 0xc6, 0x94, 0xd5, 0x6d, 0x48, 0xea, 0x36,
			0xbb, 0x08, 0x2c, 0x24, 0xc7, 0x16, 0x4f, 0x70,
			0x4c, 0xcc, 0x0c, 0x38, 0xb0, 0x11, 0x36, 0xa6,
			0x13, 0x37, 0x0d, 0x06, 0xbc, 0x4d, 0x79, 0x97,
			0xad, 0xec, 0x88, 0xfb, 0x06, 0x68, 0x82, 0xcb,
			0x4d, 0x81, 0x1d, 0xb9, 0x51, 0xa6, 0xc1, 0xd1,
			0x94, 0x82, 0x5d, 0x19, 0x29, 0x7e, 0x8b, 0x32,
			0x9c, 0xf3, 0xd1, 0xbd, 0xde, 0x0b, 0x4f, 0x4d,
			0x86, 0x3f, 0x73, 0x5b, 0x28, 0xfe, 0x00, 0x13,
			0x35, 0x0a, 0xbd, 0x09, 0x06, 0xdf, 0xf2, 0xd2,
			0x09, 0x90, 0xae, 0xe1, 0x66, 0xe1, 0xeb, 0xd6,
			0x8d, 0x76, 0x37, 0x85, 0x1d, 0x10, 0xbc, 0xe8,
			0x5c, 0xbb, 0x90, 0x43, 0x7d, 0x0e, 0x9d, 0x97,
			0x6b, 0xd8, 0xfd, 0x00, 0x77, 0x7e, 0xfc, 0xaf,
			0x32, 0x97, 0x16, 0x4e, 0x2d, 0x5d, 0x5b, 0xd3,
			0x71, 0xca, 0x2e, 0xcc, 0x6e, 0xc0, 0xe3, 0x90,
			0xb3, 0xc1, 0x16, 0xfe, 0xdb, 0xee, 0x75, 0x18,
			0x1c, 0x9a, 0xed, 0x20, 0x41, 0x0c, 0x98, 0xaf,
			0xa1, 0x95, 0x52, 0x7d, 0x26, 0xd0, 0xfc, 0x50,
			0x4f, 0xb0, 0xd0, 0x74, 0xbe, 0x43, 0x4a, 0x44,
			0x3b, 0xfb, 0x20, 0xb5, 0xf0, 0x94, 0xc5, 0xb2,
			0xe2, 0x10, 0xdb, 0xe7, 0x9a, 0x18, 0xbe, 0x66,
			0x4b, 0xa1, 0x26, 0xea, 0x39, 0x39, 0xed, 0xdb,
			0x73, 0x40, 0x27, 0xfa, 0x54, 0x62, 0xde, 0x4c,
			0x7b, 0xc3, 0xd2, 0xd1, 0x32, 0xad, 0xfb, 0x66,
			0xea, 0x56, 0xb5, 0x6b, 0x93, 0x3b, 0xe7, 0x6d,
			0xbb, 0x36, 0x21, 0x31, 0xe3, 0x4c, 0xb3, 0x76,
			0xc2, 0xd4, 0x07, 0x2f, 0x46, 0xed, 0x88, 0x92,
			0x73, 0x6e, 0xad, 0x9e, 0x31, 0x2d, 0x21, 0x13,
			0x8d, 0x81, 0x02, 0xd4, 0xed, 0x37, 0x2f, 0x3c,
			0x2e, 0xa0, 0x85, 0x0c, 0x51, 0xb7, 0x30, 0x9a,
			0x08, 0x0b, 0x8e, 0x93, 0x3c, 0xbb, 0x53, 0x68,
			0x8e, 0xb9, 0x0b, 0xac, 0xbd, 0x7b, 0xde, 0xc3,
			0x30, 0x77, 0xad, 0xd2, 0x6b, 0xbf, 0xf7, 0x6d,
			0xc9, 0x8b, 0xfb, 0x2e, 0xd1, 0xbc, 0xf4, 0x09,
			0xce, 0x7c, 0x24, 0x71, 0x9c, 0x05, 0x6f, 0xc5,
			0x0f, 0x9b, 0xe3, 0x49, 0x99, 0x0d, 0x42, 0xb3,
			0x31, 0xd0, 0xec, 0x7d, 0x9e, 0xbc, 0xba, 0x1d,
			0x3e, 0x0f, 0x4f, 0x5c, 0x06, 0xcf, 0x86, 0x5c,
			0xd5, 0x09, 0xdb, 0x84, 0x70, 0x67, 0xc6, 0x81,
		},
		20, /* salt length */
		/* V4 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA224 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0xed, 0x34, 0x8a, 0xf6, 0x93, 0x8b, 0xea, 0xec,
			0xb6, 0x54, 0x37, 0x49, 0x12, 0x2c, 0x2e, 0x29,
			0x5f, 0xb9, 0x69, 0x10, 0x26, 0x33, 0xd3, 0x8a,
			0xd2, 0x23, 0x5f, 0x0e, 0x0a, 0x6b, 0xfa, 0x28,
			0x3e, 0x87, 0xbe, 0xd8, 0x80, 0x12, 0x68, 0x35,
			0x96, 0x8c, 0x0c, 0x55, 0x4d, 0x5a, 0x07, 0x71,
			0x10, 0x0e, 0x36, 0x45, 0xb8, 0xe9, 0x65, 0xb6,
			0xa6, 0x9e, 0x0a, 0xb7, 0x0f, 0x59, 0x34, 0x6d,
			0x98, 0x00, 0xbe, 0x91, 0x23, 0xb9, 0xc8, 0x17,
			0xe8, 0xac, 0x67, 0x0d, 0xc7, 0x0a, 0x9c, 0xc5,
			0xdc, 0x4e, 0x83, 0x36, 0x55, 0xc3, 0xcd, 0xa0,
			0x80, 0x46, 0x70, 0x6f, 0x28, 0x4d, 0x9d, 0x62,
			0x22, 0x28, 0xf3, 0xec, 0x23, 0x87, 0xf2, 0xdb,
			0xd5, 0x3d, 0xb2, 0x37, 0x29, 0x43, 0xd5, 0x6d,
			0xee, 0x00, 0xba, 0xac, 0xa3, 0xc7, 0xf3, 0x90,
			0x89, 0xa2, 0xfa, 0xfe, 0xfd, 0x4e, 0x1f, 0x4f,
		},
		/* S5 : 384 byte */
		{
			0x3d, 0x24, 0xf4, 0xd6, 0x7f, 0xb8, 0x69, 0xb7,
			0x52, 0x5d, 0x16, 0xdc, 0xf6, 0x00, 0x4b, 0x5b,
			0x5c, 0x61, 0xe4, 0xba, 0xce, 0x1f, 0x4c, 0x58,
			0x52, 0xb7, 0x89, 0x8f, 0xc9, 0x0f, 0x8b, 0x48,
			0x33, 0xa7, 0x9a, 0x79, 0x93, 0xdb, 0x71, 0x81,
			0xb5, 0x2e, 0x98, 0xc2, 0xc1, 0xc7, 0xf5, 0x87,
			0x9a, 0x93, 0xb7, 0xf7, 0x3f, 0x26, 0x6d, 0x8b,
			0x91, 0xcf, 0xf1, 0xa7, 0x48, 0xb8, 0x0e, 0x40,
			0x1d, 0x2b, 0xd9, 0x45, 0xd2, 0x22, 0xb8, 0x7e,
			0x60, 0xc0, 0x59, 0x6f, 0x02, 0x5a, 0xa6, 0x4e,
			0x68, 0xf8, 0x31, 0x29, 0x34, 0xa0, 0xfb, 0x36,
			0xb4, 0x2e, 0x32, 0xb4, 0xee, 0xff, 0x07, 0x4b,
			0xdf, 0x01, 0x80, 0x55, 0xa7, 0xbd, 0xde, 0x48,
			0xac, 0xd4, 0x69, 0xbf, 0x40, 0xdc, 0xf7, 0x57,
			0x1b, 0x16, 0x0c, 0xfb, 0x59, 0x12, 0x55, 0x45,
			0xca, 0x52, 0xcd, 0xd6, 0xe2, 0x4a, 0xd6, 0x57,
			0x4d, 0xef, 0xf6, 0x49, 0x9c, 0xcb, 0xee, 0xd7,
			0x54, 0x3c, 0xb5, 0x74, 0xa4, 0xc0, 0xfc, 0x0b,
			0x03, 0xbd, 0x03, 0x53, 0xbc, 0xf4, 0x24, 0xb8,
			0x87, 0x2e, 0x00, 0x94, 0xdb, 0x19, 0xa6, 0x69,
			0xa1, 0x60, 0xff, 0xcb, 0x3d, 0x49, 0x0b, 0xcc,
			0x05, 0x60, 0x89, 0x20, 0x4d, 0x9e, 0x03, 0x81,
			0xcb, 0x3d, 0x6d, 0x87, 0x22, 0x63, 0x1c, 0x80,
			0xc2, 0x29, 0x6a, 0xe5, 0xa9, 0xd2, 0xee, 0x8d,
			0x6f, 0x90, 0x47, 0xf5, 0xde, 0x06, 0xd7, 0x43,
			0x39, 0x4a, 0x28, 0x89, 0xe6, 0xb2, 0x10, 0xef,
			0xbc, 0x21, 0x98, 0x2d, 0x8f, 0x15, 0xeb, 0x89,
			0x86, 0x54, 0x8d, 0xab, 0xb7, 0xdb, 0xda, 0xb9,
			0x46, 0xfc, 0xf7, 0x1c, 0x28, 0xd8, 0x1c, 0xf0,
			0x13, 0x60, 0xb3, 0xcb, 0xfd, 0xb5, 0x51, 0x8d,
			0x0f, 0x38, 0x38, 0x08, 0x40, 0x5b, 0xbe, 0x71,
			0xfe, 0x00, 0x97, 0xa6, 0xa9, 0x3b, 0xbd, 0xdf,
			0x97, 0x58, 0x4e, 0x5c, 0x44, 0x70, 0xaf, 0xfd,
			0xac, 0x95, 0x5b, 0xb8, 0x7b, 0x11, 0x08, 0xdb,
			0x3f, 0xe3, 0xdf, 0x26, 0x3a, 0x7c, 0x20, 0x93,
			0xe5, 0x8d, 0xd8, 0x8d, 0x8f, 0xf4, 0xa8, 0xc1,
			0x82, 0x47, 0x8d, 0x8f, 0x8e, 0x76, 0x65, 0xe4,
			0xbf, 0xdc, 0xe4, 0xfa, 0xf8, 0xbe, 0xfd, 0x7e,
			0x97, 0x84, 0xcb, 0xa3, 0xe7, 0x20, 0x89, 0x5e,
			0xc3, 0xb6, 0x7c, 0xba, 0x51, 0x25, 0x23, 0x13,
			0x84, 0xdd, 0x4c, 0xf9, 0x66, 0xba, 0xd2, 0x36,
			0x33, 0x37, 0xfe, 0xe0, 0x88, 0x13, 0x40, 0x53,
			0x70, 0x3b, 0x91, 0x0d, 0x5f, 0x4a, 0x1d, 0x65,
			0x1f, 0xad, 0x24, 0x19, 0x35, 0x24, 0xc1, 0x3c,
			0xf1, 0xc4, 0xa2, 0x52, 0xec, 0xaf, 0xe8, 0xf6,
			0xd1, 0xa4, 0x45, 0xf8, 0x6a, 0xa2, 0xf6, 0xe7,
			0xa4, 0x6c, 0x06, 0x3d, 0x5b, 0x07, 0x09, 0x78,
			0x7a, 0x9e, 0xa4, 0x57, 0x78, 0xa3, 0x56, 0x3a,
		},
		20, /* salt length */
		/* V5 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA224 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0x18, 0xf4, 0x4d, 0x68, 0xd5, 0xc3, 0x07, 0x04,
			0xdf, 0x47, 0xd8, 0xbe, 0x6f, 0xa2, 0xe6, 0x6f,
			0x06, 0x1f, 0xe5, 0x52, 0x03, 0x74, 0x42, 0x68,
			0x75, 0x44, 0xa1, 0x1a, 0x2e, 0xe8, 0x8a, 0x21,
			0x19, 0x6f, 0x5f, 0xa3, 0xcd, 0xe3, 0x92, 0x9e,
			0x61, 0xce, 0x14, 0x8a, 0x82, 0xfd, 0x11, 0x3a,
			0xc4, 0x84, 0xda, 0x04, 0x36, 0x03, 0x23, 0x21,
			0x39, 0x27, 0x6e, 0x56, 0x92, 0x73, 0xc7, 0xd4,
			0xbd, 0x40, 0xb2, 0x5e, 0x84, 0x02, 0x62, 0xa9,
			0xd1, 0xce, 0x2b, 0x08, 0xc1, 0x19, 0x3a, 0x49,
			0xf9, 0xc3, 0x84, 0xd9, 0x3d, 0x54, 0xfc, 0x2f,
			0x3f, 0x85, 0x09, 0x68, 0xb4, 0x02, 0x37, 0x2f,
			0xad, 0x6e, 0x2d, 0xe0, 0x25, 0xa7, 0xdf, 0xed,
			0xd6, 0x4c, 0x04, 0x80, 0x52, 0x7d, 0xcb, 0x62,
			0x4e, 0xe2, 0xc9, 0x1b, 0x41, 0xef, 0x36, 0x5b,
			0x2e, 0x66, 0x5a, 0x98, 0xbd, 0x1d, 0x42, 0xf9,
		},
		/* S6 : 384 byte */
		{
			0x2f, 0x2e, 0xff, 0x17, 0x8b, 0x54, 0x6f, 0xb5,
			0x7e, 0x57, 0x4f, 0x6c, 0x3f, 0xea, 0x53, 0xc3,
			0xae, 0x00, 0xf3, 0xde, 0xdf, 0x2e, 0x9d, 0xa5,
			0x31, 0x7c, 0x4d, 0x37, 0x12, 0x22, 0x9a, 0xe3,
			0xd3, 0x7e, 0xd0, 0x5a, 0xa8, 0x3c, 0x9d, 0xa8,
			0xce, 0x57, 0xd9, 0x12, 0x0d, 0xf5, 0xc3, 0x04,
			0x96, 0x65, 0x3b, 0x17, 0x98, 0xde, 0xae, 0x91,
			0x52, 0x6e, 0x03, 0xc5, 0x7c, 0xce, 0x57, 0xff,
			0x7d, 0x0f, 0xad, 0xc5, 0x5a, 0x61, 0xe3, 0x6f,
			0xeb, 0xb5, 0x16, 0x85, 0xf0, 0x5c, 0x6e, 0xa5,
			0xda, 0x2e, 0x5b, 0x71, 0xd4, 0x19, 0xd1, 0x1a,
			0x7d, 0x76, 0xf9, 0xcb, 0x29, 0xb1, 0x85, 0xf3,
			0xb0, 0x0d, 0xb5, 0x7a, 0xd1, 0xd9, 0x1c, 0x88,
			0x2f, 0x11, 0x47, 0x4f, 0x0c, 0xf7, 0x6e, 0x40,
			0x0e, 0xc3, 0xac, 0x93, 0xcc, 0xa3, 0xc3, 0x9c,
			0xcf, 0x46, 0x9e, 0xc4, 0xaf, 0xe7, 0x31, 0x4d,
			0xde, 0x04, 0xd3, 0xdf, 0xc1, 0xc9, 0x02, 0x2d,
			0x45, 0xe8, 0xb9, 0x9c, 0x65, 0x06, 0x00, 0x23,
			0x0c, 0x31, 0x9d, 0xaa, 0x0b, 0x21, 0x68, 0xb1,
			0x88, 0x54, 0xbd, 0x49, 0xce, 0x41, 0x14, 0x28,
			0x66, 0xc6, 0x8e, 0xcb, 0xb6, 0x53, 0x75, 0xb8,
			0x28, 0x42, 0x61, 0x2b, 0x16, 0x90, 0x68, 0xa2,
			0x6c, 0x60, 0x56, 0xca, 0x1a, 0xde, 0x35, 0x7a,
			0xf4, 0x2f, 0x56, 0xe0, 0xab, 0x5e, 0xe8, 0x77,
			0x90, 0x12, 0xce, 0x73, 0x71, 0xd6, 0x57, 0x54,
			0x67, 0x5e, 0x86, 0x13, 0x23, 0xd2, 0x84, 0xb5,
			0xb3, 0x7e, 0x12, 0x5a, 0x23, 0xe8, 0xc2, 0xa8,
			0xe0, 0x96, 0x36, 0xb8, 0x87, 0xbd, 0x63, 0x99,
			0x0d, 0xd4, 0xf8, 0x4f, 0xe1, 0xa2, 0x23, 0x44,
			0xd3, 0xde, 0xfd, 0x34, 0xbc, 0x8c, 0x20, 0x59,
			0x41, 0xfa, 0x96, 0xab, 0x10, 0xbb, 0x44, 0x29,
			0xee, 0x63, 0x63, 0x10, 0x8f, 0x80, 0x6d, 0xa4,
			0x19, 0x3e, 0x45, 0x5f, 0xf9, 0x70, 0x45, 0xc5,
			0x6e, 0x28, 0x90, 0x57, 0xe9, 0x6f, 0x02, 0xda,
			0x21, 0xbc, 0xcf, 0x3f, 0xd2, 0x89, 0x0c, 0x81,
			0x30, 0x38, 0x72, 0x6b, 0xf9, 0x25, 0xb5, 0xbb,
			0x3c, 0x1f, 0x97, 0x33, 0x0e, 0xd2, 0x62, 0xc0,
			0x82, 0x7d, 0xa2, 0xfe, 0x6b, 0xb1, 0xa3, 0xa3,
			0x77, 0x8b, 0x86, 0x65, 0x69, 0x3d, 0x92, 0x5f,
			0x2f, 0x47, 0xb0, 0x39, 0xac, 0x49, 0xd7, 0xa3,
			0x54, 0xdf, 0x92, 0x73, 0x4e, 0x95, 0xe4, 0xf2,
			0xe9, 0xc8, 0xc0, 0xc9, 0xb5, 0x49, 0xad, 0x05,
			0x8e, 0xa8, 0x6e, 0x68, 0xb8, 0xe5, 0x18, 0xad,
			0x18, 0xdd, 0x15, 0x7e, 0x4e, 0xda, 0x04, 0x6f,
			0x0a, 0x07, 0x77, 0x01, 0x27, 0x5d, 0x1b, 0xd1,
			0xb0, 0x64, 0x78, 0x45, 0xe3, 0x7e, 0x30, 0x43,
			0x83, 0x36, 0xac, 0x8e, 0x67, 0x0a, 0x87, 0x5a,
			0xdb, 0xea, 0x68, 0x11, 0x4e, 0xb6, 0xc1, 0x13,
		},
		20, /* salt length */
		/* V6 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA224 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0xf0, 0x25, 0xcb, 0xf8, 0x45, 0x52, 0xfb, 0x80,
			0xa4, 0x8c, 0x8c, 0x56, 0xca, 0x6a, 0x43, 0x7a,
			0xbf, 0xa2, 0x93, 0xe5, 0xae, 0x86, 0x11, 0x27,
			0xf4, 0xcf, 0x28, 0x4f, 0x77, 0x9a, 0x0e, 0x0b,
			0x39, 0xb0, 0xf6, 0x10, 0x2d, 0x0a, 0x88, 0xbc,
			0x49, 0x95, 0x8e, 0xfa, 0xdc, 0x32, 0xf0, 0x67,
			0x22, 0x01, 0xe2, 0xb3, 0x15, 0xdf, 0x5c, 0x49,
			0x87, 0x2d, 0xca, 0x32, 0x6e, 0xd4, 0x06, 0x36,
			0x7f, 0x85, 0x83, 0x45, 0xfe, 0x46, 0xcb, 0x16,
			0xd4, 0x2a, 0x2b, 0x0e, 0xe2, 0x5f, 0xac, 0x63,
			0x95, 0x6e, 0x46, 0x81, 0xba, 0xbf, 0x51, 0xad,
			0x94, 0x5c, 0x3c, 0xe8, 0x2c, 0xf6, 0x81, 0x16,
			0xe9, 0x2d, 0xd8, 0x5f, 0x60, 0x0c, 0xac, 0xd6,
			0x6d, 0xd3, 0xcb, 0x58, 0x83, 0xfa, 0x61, 0xb7,
			0x40, 0x72, 0x27, 0x18, 0x45, 0x2e, 0xd8, 0x9a,
			0x44, 0xe2, 0xe5, 0x39, 0x5a, 0x0b, 0x69, 0xf1,
		},
		/* S7 : 384 byte */
		{
			0x9a, 0x3d, 0x7a, 0x50, 0x27, 0x1e, 0xc6, 0xf2,
			0xc5, 0x21, 0xce, 0x2c, 0xf4, 0x53, 0x83, 0x69,
			0xbd, 0x7f, 0x9c, 0xc8, 0x05, 0xb0, 0xd0, 0x79,
			0x59, 0x0a, 0x1b, 0x87, 0xe9, 0x9d, 0xa9, 0x6b,
			0xb9, 0x5f, 0x46, 0xb3, 0xff, 0x4a, 0x3c, 0x75,
			0x54, 0x7a, 0xfa, 0x30, 0x69, 0x03, 0x51, 0x79,
			0x6a, 0xdf, 0x3c, 0xf8, 0xec, 0x65, 0xe0, 0x5a,
			0x5e, 0x1d, 0xd4, 0x2d, 0x0f, 0xf0, 0xed, 0xe9,
			0x37, 0x88, 0xca, 0xf0, 0xab, 0x45, 0x3d, 0x16,
			0xaa, 0x7a, 0xd2, 0xde, 0x67, 0xa4, 0x60, 0x50,
			0xab, 0xec, 0xcd, 0x3c, 0xbe, 0x25, 0xdd, 0x90,
			0x06, 0x14, 0x32, 0xca, 0x8d, 0xd7, 0xe7, 0x3e,
			0x44, 0x9c, 0x41, 0x8c, 0x87, 0x52, 0x57, 0xd2,
			0xc7, 0xe4, 0x91, 0x24, 0x0a, 0x60, 0xb6, 0x4f,
			0xf2, 0x8e, 0xe9, 0x02, 0x58, 0x7e, 0x75, 0xb3,
			0xdb, 0x8e, 0x66, 0x9e, 0x30, 0xbb, 0x2c, 0x2b,
			0x1f, 0x6b, 0x01, 0x60, 0xc1, 0x09, 0x08, 0x2d,
			0x32, 0x10, 0x53, 0x03, 0x60, 0x96, 0x23, 0x2a,
			0x03, 0xba, 0xdf, 0x1e, 0xf4, 0x2e, 0xac, 0xd4,
			0x13, 0x01, 0xef, 0xf2, 0x61, 0x11, 0x88, 0x19,
			0x19, 0xd9, 0x14, 0xa6, 0xf6, 0x6b, 0x91, 0xe2,
			0x26, 0x78, 0x52, 0x3b, 0x7f, 0x29, 0xd1, 0xf1,
			0x90, 0x50, 0xe0, 0x17, 0x1d, 0x02, 0xc6, 0xe4,
			0xc3, 0x50, 0x30, 0x9a, 0x75, 0xca, 0x8f, 0x4f,
			0x08, 0xfe, 0xb1, 0x42, 0x18, 0xcb, 0x54, 0xc4,
			0x0a, 0x57, 0x0e, 0xbd, 0xca, 0xa5, 0xc3, 0xab,
			0x6e, 0x47, 0x6b, 0xa7, 0x7e, 0xc9, 0x8b, 0xc9,
			0x51, 0x9c, 0x5a, 0x62, 0x89, 0x31, 0x43, 0xe4,
			0x99, 0x04, 0x0c, 0x7f, 0xcf, 0xe4, 0x94, 0x3f,
			0xe9, 0xc3, 0x33, 0x72, 0xde, 0xfe, 0xaf, 0x7e,
			0xf6, 0x18, 0xb8, 0xe1, 0xc9, 0x2a, 0xba, 0x1d,
			0xb7, 0xca, 0xfa, 0x0c, 0x84, 0x90, 0x9e, 0x40,
			0x90, 0x6f, 0xf7, 0x98, 0xeb, 0x9b, 0x93, 0x00,
			0x23, 0xc3, 0x92, 0x94, 0x91, 0x36, 0x46, 0x62,
			0xf0, 0x13, 0x76, 0xd1, 0xe9, 0xb0, 0x39, 0xd2,
			0xc0, 0xe1, 0xe0, 0x02, 0x5c, 0x6b, 0x64, 0x6b,
			0xf5, 0x9e, 0x85, 0xcf, 0xd4, 0x84, 0x83, 0x3a,
			0xfa, 0xb7, 0x3f, 0xb0, 0x7d, 0x3c, 0x67, 0xdf,
			0xc8, 0x61, 0x7a, 0x5e, 0x70, 0x33, 0x4c, 0x2f,
			0x89, 0x24, 0x22, 0xf4, 0xae, 0x61, 0x89, 0x12,
			0x19, 0xf0, 0x98, 0xb9, 0xc9, 0x10, 0x7d, 0x6b,
			0x01, 0xa4, 0x70, 0x34, 0x7e, 0x0a, 0x6f, 0x6f,
			0x4c, 0xed, 0x42, 0xb6, 0x67, 0xb5, 0x3e, 0x6b,
			0x9c, 0xb2, 0xb7, 0x39, 0xc2, 0x70, 0x22, 0xba,
			0xe0, 0x54, 0x91, 0x67, 0x80, 0x27, 0xe3, 0xd6,
			0x54, 0xa2, 0x2a, 0xe7, 0x16, 0x83, 0x63, 0x7b,
			0x1e, 0xf8, 0x0e, 0xb9, 0xb9, 0xf9, 0x7d, 0x25,
			0x58, 0xca, 0xc7, 0x12, 0xb5, 0x94, 0x6f, 0x1c,
		},
		20, /* salt length */
		/* V7 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA224 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0xbe, 0x5e, 0x04, 0x36, 0x22, 0x05, 0xd9, 0x94,
			0xad, 0xb3, 0x23, 0xf8, 0xfa, 0xa8, 0xf2, 0xc4,
			0x2c, 0x89, 0x26, 0xe4, 0xd6, 0xb2, 0x7a, 0xbc,
			0x18, 0xcd, 0xc1, 0x5e, 0x0b, 0xa3, 0x6f, 0x01,
			0x5f, 0x33, 0x5a, 0xaf, 0x43, 0x01, 0xc2, 0x3d,
			0x85, 0xb1, 0x8f, 0x80, 0xfb, 0xd4, 0xdb, 0xbe,
			0x21, 0x02, 0xb0, 0x63, 0x03, 0x29, 0x13, 0xf5,
			0x5e, 0x24, 0x41, 0x4a, 0x02, 0x90, 0x38, 0xf9,
			0x7a, 0xdb, 0x79, 0x0e, 0xef, 0x9d, 0x0d, 0x02,
			0x4a, 0x2c, 0xc6, 0xea, 0x99, 0x6b, 0xc7, 0xee,
			0xb4, 0xe2, 0x46, 0xfd, 0x20, 0x14, 0x91, 0xbf,
			0x33, 0x0c, 0x4f, 0xcb, 0x79, 0x3a, 0x09, 0x1b,
			0x57, 0x7d, 0xe9, 0xbc, 0xee, 0x81, 0xc1, 0xca,
			0x42, 0x4b, 0x4e, 0x9c, 0x9c, 0x12, 0xbf, 0x3c,
			0x2d, 0x51, 0x71, 0x1a, 0xf3, 0x1a, 0xfd, 0x01,
			0xe1, 0xad, 0x74, 0x49, 0x3a, 0x49, 0xe9, 0x4f,
		},
		/* S8 : 384 byte */
		{
			0x6c, 0xaa, 0x2f, 0xec, 0xda, 0xe1, 0x19, 0xc2,
			0x50, 0x73, 0x47, 0xe8, 0xf3, 0x52, 0x0a, 0x49,
			0x05, 0x27, 0xb2, 0x68, 0x66, 0x39, 0xdb, 0x4f,
			0x8d, 0x8a, 0xed, 0x44, 0x51, 0xf3, 0x8b, 0xfc,
			0x3f, 0x4a, 0xe7, 0xb4, 0x44, 0xa9, 0xa4, 0xde,
			0x23, 0x1a, 0xc6, 0xf6, 0x66, 0x64, 0x03, 0x39,
			0x51, 0xfb, 0x69, 0x9a, 0xad, 0xdd, 0x23, 0xae,
			0x9c, 0x06, 0x96, 0xad, 0xbd, 0x15, 0x6f, 0x01,
			0x7d, 0x5a, 0x92, 0x3d, 0xc0, 0xe6, 0xc1, 0x48,
			0x1c, 0x65, 0x6a, 0xd8, 0xc7, 0x1f, 0x55, 0xbe,
			0x1a, 0x28, 0x47, 0xb2, 0x34, 0x45, 0x6a, 0x75,
			0xb9, 0x76, 0x2f, 0x51, 0x5e, 0x8d, 0x19, 0x3a,
			0x8b, 0xae, 0xc8, 0x56, 0x3a, 0x55, 0xf8, 0x3c,
			0xbe, 0xde, 0x34, 0x7b, 0xd9, 0x8f, 0xb7, 0x1f,
			0xa2, 0xba, 0x64, 0x76, 0x76, 0x1e, 0x98, 0xdf,
			0xda, 0x7e, 0x56, 0xc8, 0x95, 0xbf, 0x25, 0x0d,
			0x5a, 0x17, 0xe2, 0x5b, 0x0a, 0xd2, 0x12, 0x9d,
			0xf4, 0x75, 0x33, 0xfb, 0x27, 0xee, 0x3e, 0x24,
			0x29, 0x66, 0xca, 0xfd, 0x48, 0x7d, 0x46, 0x10,
			0xa4, 0xbb, 0x34, 0x70, 0xef, 0x62, 0x2d, 0x63,
			0xab, 0xa3, 0xfc, 0xd3, 0x5c, 0x5b, 0x25, 0x22,
			0xef, 0x1f, 0x0d, 0x28, 0x5c, 0xe1, 0x68, 0x2f,
			0xbb, 0xd7, 0x8f, 0x99, 0x7b, 0xe0, 0xf8, 0xaf,
			0xcf, 0x56, 0x7c, 0x36, 0xdb, 0xd8, 0x1d, 0xf9,
			0x8c, 0x81, 0xbf, 0x6b, 0x3e, 0x4e, 0xef, 0x2f,
			0xce, 0x75, 0x45, 0x31, 0x4f, 0xc0, 0x76, 0xd8,
			0x66, 0xc1, 0x52, 0xfc, 0x0f, 0xd9, 0x64, 0x19,
			0x0f, 0xc0, 0x94, 0x59, 0xef, 0xda, 0x9e, 0x01,
			0xb9, 0x0e, 0x51, 0xad, 0xa8, 0xa7, 0xf8, 0xaa,
			0x3a, 0x39, 0x3e, 0xc2, 0xd0, 0xfb, 0x87, 0xfd,
			0xdb, 0x78, 0xe0, 0x6e, 0x23, 0x8f, 0xc6, 0xd2,
			0x2e, 0x40, 0xf7, 0xd2, 0xfc, 0x1a, 0x2e, 0x04,
			0xfe, 0x20, 0xa7, 0x94, 0x84, 0x1c, 0x59, 0xcc,
			0x71, 0x34, 0xd5, 0x2e, 0x0f, 0xc4, 0x6c, 0x18,
			0x05, 0x5d, 0x9c, 0x9c, 0x96, 0x4e, 0x9a, 0xd1,
			0xfd, 0x75, 0x7f, 0x50, 0x0b, 0xb5, 0x33, 0x07,
			0x31, 0xd1, 0xdd, 0xbd, 0x7e, 0xc3, 0x06, 0xd0,
			0xd9, 0xa8, 0x42, 0xf9, 0xcb, 0xfd, 0x23, 0x65,
			0x80, 0x64, 0xe1, 0xa3, 0xcd, 0xd6, 0x86, 0x36,
			0x64, 0xb9, 0xc2, 0x48, 0x63, 0x18, 0x9d, 0x7a,
			0x7c, 0xcc, 0xf8, 0x1e, 0x32, 0xa8, 0x89, 0x4e,
			0x79, 0xf9, 0xf3, 0xec, 0x8f, 0xc0, 0x5b, 0x2a,
			0xcf, 0x6d, 0x6c, 0x10, 0x06, 0xef, 0x0f, 0xc3,
			0xea, 0x5f, 0x6c, 0x64, 0x47, 0x91, 0x5c, 0x75,
			0xab, 0xd6, 0x2d, 0x9a, 0xfc, 0x11, 0x63, 0x10,
			0xa1, 0x0f, 0x47, 0x31, 0xec, 0x5a, 0xb2, 0x86,
			0x3f, 0x0d, 0x37, 0x86, 0xa2, 0x1e, 0x5a, 0x8f,
			0x62, 0xf3, 0x1b, 0x27, 0x56, 0x8a, 0xb2, 0x8b,
		},
		20, /* salt length */
		/* V8 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA224 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0x55, 0xc5, 0x7d, 0x2e, 0xbf, 0x8d, 0x53, 0x5a,
			0x6d, 0x31, 0x9e, 0x1d, 0x83, 0x17, 0xb5, 0xb7,
			0x4f, 0x22, 0x84, 0xbe, 0x22, 0x33, 0xf4, 0xc0,
			0x5e, 0xf0, 0xa8, 0x2b, 0xd1, 0xc4, 0xea, 0x6a,
			0x41, 0x26, 0xe5, 0x51, 0x48, 0x1f, 0x14, 0xbb,
			0x63, 0x06, 0xfa, 0x71, 0x45, 0x8f, 0x14, 0x4e,
			0x71, 0x04, 0x0c, 0xa1, 0x07, 0xd1, 0x2f, 0xd3,
			0xde, 0x46, 0x21, 0x4a, 0xd4, 0x99, 0xcf, 0x8c,
			0x62, 0x2a, 0xa6, 0xa8, 0x74, 0x0b, 0x03, 0xd2,
			0x77, 0xc0, 0xea, 0x4d, 0xb3, 0x43, 0xfa, 0x8e,
			0xdb, 0x4c, 0xa0, 0x9e, 0xe2, 0xce, 0x8c, 0xc0,
			0x11, 0xc5, 0x61, 0x54, 0x57, 0x2f, 0xb1, 0xfb,
			0xe0, 0x58, 0x28, 0xfc, 0xe8, 0x5b, 0x07, 0xe6,
			0xd2, 0xe6, 0xd4, 0x77, 0x74, 0x3d, 0x50, 0xbd,
			0xb6, 0x82, 0xaa, 0x7b, 0x5b, 0x33, 0xf2, 0xce,
			0x1e, 0xf4, 0xd0, 0x10, 0x01, 0x8f, 0x76, 0xfd,
		},
		/* S9 : 384 byte */
		{
			0x94, 0xb5, 0x39, 0x4f, 0x3c, 0xf0, 0x39, 0xa9,
			0xf3, 0xe3, 0xdd, 0xce, 0x50, 0x2a, 0x50, 0x5a,
			0x2b, 0xaf, 0xd8, 0x93, 0xa4, 0x86, 0x99, 0xd3,
			0x25, 0x5a, 0x29, 0x82, 0xdc, 0x56, 0xe0, 0xac,
			0x9c, 0x43, 0xa0, 0xb3, 0x5d, 0x59, 0x2e, 0xae,
			0x8a, 0xdb, 0xc3, 0xdc, 0x55, 0xa9, 0xa6, 0x2f,
			0xd5, 0xe0, 0x0e, 0x1f, 0x08, 0x12, 0x0b, 0xc9,
			0x0a, 0xdd, 0x00, 0xc2, 0x57, 0xf4, 0x76, 0xb2,
			0x43, 0x80, 0x82, 0x8d, 0x46, 0x86, 0xb3, 0x72,
			0x87, 0xa5, 0xba, 0xa7, 0x09, 0xa5, 0x80, 0x15,
			0xeb, 0x63, 0xbb, 0x81, 0x8f, 0xe7, 0x99, 0x64,
			0x3e, 0x06, 0xbd, 0xfb, 0xba, 0x9d, 0x0a, 0x55,
			0xa7, 0xfe, 0x32, 0xba, 0x24, 0xf7, 0x4f, 0x69,
			0x6f, 0x48, 0x76, 0xe4, 0x3c, 0x2b, 0x7e, 0xcd,
			0xcb, 0x83, 0x1e, 0xde, 0x7a, 0x10, 0x3d, 0x3d,
			0x33, 0x81, 0xd5, 0x91, 0xc7, 0xe7, 0x9c, 0xaf,
			0x76, 0xed, 0x7d, 0x0d, 0xd1, 0xd6, 0xf1, 0x1a,
			0x94, 0xcc, 0xf0, 0xa3, 0x83, 0xdf, 0xd3, 0x24,
			0x0d, 0x7c, 0x7c, 0x48, 0x5b, 0x9d, 0x6c, 0xe1,
			0x80, 0xf6, 0xd1, 0xaf, 0x30, 0xdc, 0xb4, 0xa3,
			0xb7, 0x79, 0xf5, 0x0b, 0x69, 0xf9, 0xae, 0x7b,
			0xee, 0x2d, 0xaa, 0x15, 0xea, 0xb7, 0x1e, 0xfc,
			0xbf, 0x21, 0x97, 0x16, 0x20, 0x4b, 0x8f, 0x88,
			0x66, 0x7d, 0x2e, 0xac, 0x69, 0x19, 0xdd, 0xec,
			0x4a, 0xcf, 0xba, 0x94, 0x9c, 0xe4, 0xf8, 0x10,
			0x54, 0x62, 0x0a, 0x03, 0xd5, 0x7d, 0x9e, 0xd2,
			0xf7, 0x02, 0x1c, 0x8f, 0xea, 0x26, 0xf0, 0xa4,
			0xb9, 0x63, 0xfc, 0xa0, 0x12, 0xa9, 0xd4, 0xa6,
			0xf9, 0x09, 0x10, 0x9d, 0x04, 0x53, 0x68, 0xd4,
			0xe0, 0x9a, 0xc9, 0x59, 0xec, 0x26, 0xa3, 0x53,
			0x88, 0xc3, 0x74, 0x3a, 0x45, 0x37, 0x7b, 0xc1,
			0x72, 0x63, 0x67, 0x03, 0xe9, 0x03, 0xf2, 0x9e,
			0xd3, 0x27, 0x61, 0x65, 0xe3, 0xb8, 0xd0, 0x1f,
			0x0d, 0x47, 0x5b, 0xc0, 0x03, 0x62, 0x27, 0xeb,
			0x4d, 0x00, 0x7b, 0xd6, 0xe3, 0xe6, 0x6a, 0x3c,
			0x8a, 0x96, 0x25, 0xd9, 0x3f, 0xa9, 0x38, 0xfd,
			0x17, 0x05, 0x74, 0xf6, 0x54, 0xa9, 0xec, 0xac,
			0x48, 0xb0, 0x90, 0x42, 0xf6, 0x1e, 0xd0, 0x72,
			0x01, 0x2e, 0xae, 0x18, 0xaa, 0x51, 0x96, 0xc2,
			0x25, 0x5b, 0x45, 0x6f, 0xbc, 0x44, 0xf1, 0xa2,
			0xa0, 0xe1, 0xc5, 0xe4, 0xb1, 0x4e, 0xbe, 0x64,
			0x57, 0x9e, 0x44, 0x0b, 0x1a, 0x49, 0xc8, 0xf8,
			0x25, 0x32, 0xea, 0x42, 0xac, 0xc6, 0xc3, 0x88,
			0x0d, 0x2e, 0x91, 0xd0, 0x67, 0x06, 0x55, 0x29,
			0xe1, 0xd4, 0xf5, 0xb6, 0xd1, 0x69, 0x9b, 0x0e,
			0xd7, 0xec, 0xcc, 0xac, 0x40, 0x6e, 0x0c, 0x3f,
			0xd9, 0xbd, 0x51, 0xa2, 0xe8, 0xda, 0xc5, 0x20,
			0x97, 0x7f, 0xa9, 0xa2, 0xe7, 0x65, 0x07, 0xef,
		},
		20, /* salt length */
		/* V9 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
};

/* 10 elements */
static mod3072_t SHA256[] = {
	/* SHA256 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0xe2, 0xf6, 0xdf, 0xa5, 0x01, 0x4f, 0xee, 0x6b,
			0x1b, 0x04, 0x10, 0x86, 0x82, 0xe8, 0x56, 0x19,
			0xde, 0xd7, 0xc4, 0x64, 0x7f, 0xaf, 0x4a, 0xe8,
			0xf1, 0x9c, 0xf6, 0xcb, 0xd1, 0x99, 0x67, 0x7f,
			0xe0, 0x33, 0x85, 0x9f, 0x56, 0x90, 0x6f, 0x19,
			0x79, 0xb1, 0xb5, 0x92, 0x6d, 0xf4, 0xc8, 0x06,
			0x4e, 0xdd, 0xae, 0xaf, 0x7c, 0x15, 0xfa, 0x29,
			0x36, 0xb3, 0xfc, 0xd3, 0x6b, 0xbb, 0x35, 0x78,
			0xcc, 0xe4, 0x0d, 0x2f, 0x26, 0x9f, 0xc9, 0x7f,
			0xef, 0x54, 0xb7, 0xc7, 0x1f, 0xef, 0xab, 0xdd,
			0x41, 0x9b, 0xaf, 0xf6, 0xc9, 0xcd, 0xf7, 0xc6,
			0xa8, 0x85, 0x13, 0xe8, 0x1e, 0xd1, 0x68, 0x7f,
			0xcf, 0x92, 0xe1, 0x1e, 0x1a, 0x82, 0xe2, 0xe5,
			0xa6, 0x76, 0x7e, 0xed, 0x3d, 0xe1, 0xe9, 0xe7,
			0xde, 0x9a, 0x30, 0xff, 0x0d, 0xdf, 0x27, 0x07,
			0x6e, 0x99, 0xa3, 0xd1, 0x92, 0xe1, 0xea, 0xdc,
		},
		/* S0 : 384 byte */
		{
			0x3a, 0x06, 0x22, 0xdd, 0xff, 0x5a, 0x0c, 0x1f,
			0x5b, 0x54, 0x5d, 0x68, 0x40, 0x54, 0xe4, 0x62,
			0x11, 0x78, 0x6a, 0x2e, 0x40, 0x62, 0x7e, 0x0c,
			0xb6, 0x79, 0x5e, 0xa0, 0xd1, 0x76, 0xf3, 0xc9,
			0x7e, 0x65, 0x36, 0xfb, 0x64, 0xc5, 0xec, 0xa7,
			0xb2, 0x8b, 0x7a, 0xc5, 0x2e, 0x48, 0xe3, 0xd5,
			0x0b, 0x91, 0x6d, 0x2f, 0xcc, 0xb8, 0x7d, 0x70,
			0xcd, 0x8e, 0xda, 0x7c, 0x15, 0xc2, 0x30, 0x87,
			0x34, 0x25, 0x47, 0x16, 0xe5, 0xb4, 0x00, 0x59,
			0x2c, 0xc2, 0xe5, 0xe0, 0x33, 0xba, 0x27, 0x86,
			0x6c, 0xb1, 0x4f, 0xef, 0xbd, 0xcb, 0xc3, 0x5d,
			0x5d, 0x85, 0xd4, 0xee, 0xe8, 0xba, 0x6b, 0xc2,
			0xda, 0x99, 0x5e, 0x8e, 0xbc, 0xc2, 0x7d, 0x50,
			0xc4, 0x8a, 0xa9, 0x88, 0xbf, 0x45, 0xfd, 0xe2,
			0x73, 0x11, 0xa9, 0xe2, 0xec, 0x02, 0x9d, 0x0f,
			0xa6, 0xfa, 0x6d, 0x3e, 0xfe, 0xa4, 0x60, 0xfc,
			0x1a, 0x90, 0xe4, 0x43, 0xd8, 0x07, 0xd2, 0x09,
			0xa4, 0xc0, 0x6b, 0xf3, 0x02, 0x2d, 0x52, 0x9a,
			0xb2, 0xe4, 0xa8, 0x77, 0x32, 0x5f, 0xcc, 0xcb,
			0x3f, 0x86, 0xac, 0x16, 0x20, 0x0a, 0xb9, 0x56,
			0x28, 0xbf, 0x0c, 0x1c, 0x8c, 0x70, 0xf6, 0xfe,
			0x1a, 0x9f, 0x28, 0x8b, 0xbc, 0x01, 0x62, 0xa3,
			0x92, 0xf4, 0x0a, 0xd1, 0x10, 0x9c, 0xdb, 0xba,
			0xf0, 0x3d, 0x9b, 0x2d, 0x51, 0x4a, 0x60, 0x98,
			0x38, 0x74, 0x35, 0x0b, 0xe9, 0xae, 0xf8, 0x86,
			0xc3, 0xc4, 0x81, 0xa6, 0x63, 0x25, 0xf1, 0x37,
			0xae, 0xcb, 0x4c, 0x82, 0xa8, 0xa7, 0x30, 0x46,
			0xdb, 0xc1, 0xdd, 0x85, 0x98, 0xff, 0xbd, 0xb8,
			0x28, 0xa3, 0xd6, 0x38, 0xf9, 0xdd, 0x81, 0x39,
			0xa7, 0x68, 0xdc, 0xd8, 0xd3, 0x0d, 0x79, 0x74,
			0x0e, 0xf3, 0x45, 0xc1, 0x64, 0x4d, 0x03, 0xe6,
			0xfb, 0x86, 0xa4, 0x63, 0x67, 0xf6, 0xd8, 0x2a,
			0x7a, 0x81, 0x90, 0x57, 0xae, 0x49, 0x0e, 0x1b,
			0x10, 0x0b, 0x58, 0x42, 0xed, 0x38, 0x58, 0x45,
			0xf3, 0x79, 0x10, 0x1e, 0x37, 0xce, 0x60, 0x45,
			0x31, 0xc6, 0x1d, 0xe4, 0x23, 0xdf, 0x66, 0x20,
			0x0d, 0x45, 0xb7, 0x22, 0x96, 0x62, 0xfd, 0x0e,
			0xc3, 0x57, 0x25, 0x93, 0xb0, 0x9a, 0x52, 0x13,
			0xec, 0x14, 0xc1, 0xd7, 0xb2, 0x33, 0x8c, 0xa9,
			0xc7, 0x63, 0xc0, 0xd1, 0x89, 0x46, 0xf0, 0x4e,
			0xaa, 0xf5, 0x7e, 0xa2, 0xeb, 0xc7, 0x9e, 0x09,
			0x3f, 0x2f, 0xd4, 0xc6, 0x4c, 0xb1, 0xc1, 0xa7,
			0xf0, 0xe8, 0x88, 0xdc, 0x2d, 0x87, 0xa1, 0x5e,
			0xb7, 0x69, 0xf5, 0x6d, 0xc1, 0x80, 0xcf, 0xe1,
			0x59, 0x7c, 0xc3, 0xe4, 0xe1, 0x81, 0x1d, 0x4e,
			0x27, 0x85, 0x2f, 0xa1, 0x88, 0xc8, 0xfe, 0xc4,
			0xfc, 0x91, 0x7d, 0x47, 0x24, 0xd3, 0x3c, 0xe5,
			0xf3, 0x21, 0x18, 0x95, 0xcf, 0x7e, 0x8b, 0x8c,
		},
		20, /* salt length */
		/* V0 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA256 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0xe6, 0x8a, 0x33, 0xad, 0x4c, 0xf6, 0x54, 0x59,
			0x5f, 0xc2, 0x26, 0x7e, 0x5c, 0xc3, 0x7d, 0xb2,
			0x56, 0x51, 0x9a, 0x42, 0x36, 0x43, 0x4e, 0xc6,
			0x01, 0x1b, 0x63, 0xf9, 0x59, 0x7d, 0x37, 0xc8,
			0x47, 0xe6, 0xaa, 0xa9, 0x37, 0x9a, 0x3a, 0x2e,
			0x44, 0xde, 0x6d, 0xda, 0xb6, 0x68, 0x70, 0x33,
			0xc9, 0x14, 0xfd, 0x18, 0xb1, 0xd8, 0x81, 0x55,
			0xda, 0xfa, 0x7f, 0x75, 0x74, 0x80, 0x2f, 0xd8,
			0x2f, 0xe9, 0x6c, 0x05, 0x43, 0x1a, 0xcd, 0x40,
			0xb4, 0x9b, 0x11, 0x60, 0xd4, 0x03, 0xc0, 0xdb,
			0x09, 0xb1, 0x0f, 0x23, 0xd0, 0xbd, 0x04, 0x35,
			0x02, 0x2e, 0xdc, 0x6e, 0x57, 0x6e, 0xaa, 0xa1,
			0x28, 0xfb, 0x40, 0x68, 0x79, 0x5e, 0x52, 0xb7,
			0x36, 0x61, 0xd9, 0xb7, 0xbd, 0x7d, 0xb6, 0xd1,
			0x9d, 0x92, 0x98, 0x08, 0x9d, 0xdb, 0x1f, 0x8d,
			0x01, 0x20, 0x42, 0x8d, 0x08, 0x2d, 0xee, 0x4e,
		},
		/* S1 : 384 byte */
		{
			0x11, 0xbc, 0x27, 0xd7, 0x8b, 0x27, 0xf7, 0xd0,
			0x5a, 0x5d, 0x77, 0x13, 0x62, 0x77, 0xb1, 0xb3,
			0x8b, 0x4d, 0xaf, 0x3a, 0x20, 0xbf, 0x4f, 0xba,
			0x3d, 0x95, 0xef, 0x36, 0x59, 0x91, 0x7e, 0xab,
			0x12, 0x9a, 0xe6, 0x03, 0x12, 0xaf, 0xc4, 0x15,
			0x53, 0xc2, 0x72, 0xde, 0xa4, 0xe5, 0x25, 0x22,
			0x08, 0x25, 0x6a, 0xf9, 0x94, 0x95, 0xf6, 0x7d,
			0x89, 0xff, 0xbc, 0x02, 0xc1, 0xc9, 0x61, 0x2c,
			0xb4, 0x3b, 0xb1, 0xf3, 0xf5, 0xc2, 0x42, 0x95,
			0xf2, 0xe0, 0x03, 0x48, 0xee, 0x90, 0x0a, 0x11,
			0x02, 0x9b, 0xcb, 0xf1, 0xb8, 0xd1, 0x4c, 0x92,
			0xda, 0xca, 0xb9, 0x54, 0xdb, 0x99, 0xbe, 0x61,
			0x61, 0x4f, 0xbe, 0x6c, 0x30, 0x4f, 0xe4, 0x51,
			0xfd, 0xbf, 0x35, 0x88, 0xdc, 0x57, 0xbd, 0xe7,
			0x7a, 0xfa, 0x1a, 0x95, 0x7e, 0x59, 0x52, 0xce,
			0xad, 0x41, 0x69, 0x30, 0xe9, 0x0f, 0x3f, 0x5c,
			0xb7, 0xaf, 0xdd, 0xed, 0xe2, 0x47, 0x99, 0x57,
			0xbc, 0x47, 0xcf, 0x25, 0x58, 0xe9, 0x61, 0xd0,
			0xcb, 0x11, 0x60, 0x8e, 0x8c, 0x84, 0xb6, 0xb6,
			0xfb, 0xd8, 0xda, 0x02, 0x27, 0x5a, 0x5e, 0xa5,
			0x56, 0xf6, 0x76, 0x0f, 0x30, 0xc9, 0xff, 0x2d,
			0x4b, 0x61, 0xab, 0xcf, 0x98, 0xfb, 0x27, 0xe8,
			0xf3, 0xd3, 0x08, 0x55, 0xa6, 0x2a, 0xeb, 0x0a,
			0x4d, 0x1f, 0xc9, 0xca, 0xe9, 0x4a, 0xb4, 0x06,
			0x1b, 0x83, 0x8c, 0x85, 0x6c, 0x15, 0xe7, 0x0a,
			0xc9, 0xf8, 0x42, 0xa6, 0xeb, 0x3f, 0x32, 0xff,
			0x2e, 0xe2, 0xc0, 0xda, 0x59, 0xc7, 0x89, 0x9c,
			0x4e, 0x05, 0x9d, 0xed, 0xa1, 0xef, 0x20, 0x4a,
			0x6b, 0x22, 0xd7, 0xab, 0xf5, 0x9f, 0xa8, 0x69,
			0x27, 0x62, 0x07, 0x82, 0xa1, 0x31, 0x28, 0x23,
			0x81, 0xbc, 0xda, 0x9b, 0xdd, 0x66, 0xec, 0x82,
			0x01, 0x92, 0x08, 0x1d, 0x0d, 0xe3, 0xd0, 0xbb,
			0x15, 0x72, 0xe7, 0xe6, 0x92, 0x47, 0x52, 0x45,
			0xed, 0xf1, 0xa4, 0x8b, 0x93, 0x4c, 0x67, 0x73,
			0x3d, 0xcc, 0xc6, 0x55, 0x84, 0xc6, 0xb6, 0xc2,
			0xba, 0x40, 0xc8, 0xba, 0x59, 0x4a, 0x58, 0x4d,
			0x15, 0xad, 0x07, 0x3e, 0xb8, 0x0a, 0xf1, 0xfc,
			0xc8, 0x21, 0x1e, 0x6f, 0x9b, 0xd8, 0x3d, 0x28,
			0x84, 0x60, 0x43, 0xfc, 0x23, 0xd9, 0xa5, 0x1c,
			0xa6, 0xe0, 0x99, 0x9d, 0x70, 0x2a, 0xc3, 0xeb,
			0x2f, 0xd7, 0xa6, 0x5a, 0x33, 0x19, 0x2c, 0x9d,
			0x8b, 0x14, 0x66, 0x89, 0x35, 0x0f, 0x30, 0x39,
			0xdc, 0xab, 0xee, 0xd8, 0x61, 0x27, 0x58, 0xb4,
			0x7c, 0x1f, 0xe4, 0x51, 0x07, 0xb4, 0xf6, 0x25,
			0xa6, 0x17, 0x72, 0x27, 0x50, 0x27, 0x0d, 0x8f,
			0xf6, 0x32, 0x5a, 0x85, 0xb7, 0xf9, 0x6a, 0x4c,
			0x8f, 0xe7, 0x17, 0xce, 0x72, 0x11, 0x69, 0x86,
			0x4c, 0x36, 0x57, 0x40, 0xa1, 0xee, 0xe9, 0xd7,
		},
		20, /* salt length */
		/* V1 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA256 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x2c, 0x5e, 0x7b, 0x76, 0x3e, 0xab, 0x4a, 0x2c,
			0xf7, 0x34, 0xd1, 0x73, 0xcf, 0xe2, 0x63, 0x20,
			0xdb, 0xc5, 0xe6, 0x55, 0x43, 0xb4, 0xe1, 0x49,
			0xf6, 0x34, 0xd7, 0x47, 0x9c, 0x9c, 0xd4, 0xfa,
			0xcd, 0x76, 0x45, 0xe7, 0x80, 0x27, 0x0c, 0x3b,
			0xc6, 0xd3, 0xd4, 0xc7, 0xb8, 0x7b, 0xa2, 0x49,
			0xad, 0x68, 0xad, 0xed, 0xb1, 0xcd, 0x70, 0x13,
			0x33, 0xa8, 0xfd, 0x53, 0xfe, 0xe9, 0x94, 0xbe,
			0x6b, 0xd6, 0x74, 0x67, 0xd4, 0xae, 0xf1, 0x23,
			0x4b, 0x8c, 0xc7, 0x8c, 0x8c, 0x8f, 0xae, 0x4c,
			0x31, 0x38, 0x2f, 0x98, 0x25, 0x13, 0xb5, 0xfa,
			0x5b, 0x9a, 0xe7, 0x54, 0xc0, 0x56, 0x35, 0x22,
			0x6e, 0x47, 0xb1, 0xff, 0x22, 0x86, 0x21, 0x6b,
			0xf0, 0x29, 0x52, 0xcb, 0x35, 0x67, 0xaf, 0xb1,
			0xce, 0xfd, 0x11, 0x5e, 0x87, 0xcf, 0xd9, 0x85,
			0xd6, 0xd3, 0x3d, 0x52, 0xca, 0x2b, 0xdf, 0xab,
		},
		/* S2 : 384 byte */
		{
			0x82, 0x4c, 0x11, 0x4d, 0x67, 0xb8, 0x01, 0x65,
			0x6a, 0xe5, 0x6e, 0xae, 0xac, 0x50, 0x2d, 0x94,
			0x2c, 0x8b, 0x0f, 0x20, 0xba, 0xeb, 0xbe, 0x8b,
			0x28, 0x51, 0x29, 0x03, 0x8b, 0x40, 0x0c, 0x75,
			0xdc, 0x6a, 0x53, 0x25, 0xc4, 0x06, 0x72, 0x4c,
			0x72, 0xa5, 0x25, 0x37, 0x14, 0x8e, 0xbd, 0x7a,
			0xd3, 0xab, 0x20, 0xaa, 0xf4, 0xcf, 0xec, 0x3e,
			0x3a, 0xef, 0xe0, 0x3b, 0xf0, 0x8f, 0xcf, 0xdc,
			0x1c, 0xd0, 0xd4, 0xc1, 0x3c, 0xb7, 0x1b, 0x2b,
			0xea, 0x65, 0x2a, 0x16, 0x29, 0x20, 0x3b, 0xaf,
			0x73, 0xfc, 0x44, 0x68, 0xd1, 0xf3, 0x41, 0x89,
			0x24, 0xe3, 0xd7, 0x6d, 0xd2, 0xb8, 0x2c, 0x8b,
			0xb1, 0xa4, 0xd1, 0xef, 0x49, 0x17, 0xe3, 0x3c,
			0xd1, 0x7b, 0x28, 0xb0, 0x32, 0x80, 0x66, 0x43,
			0x80, 0x6b, 0x3e, 0x24, 0x11, 0x71, 0x9d, 0xc4,
			0xf9, 0x15, 0x9e, 0xd2, 0xe8, 0x43, 0x0c, 0x83,
			0x2b, 0xd5, 0x4c, 0xbd, 0x84, 0xae, 0xe8, 0xf9,
			0xd4, 0x7a, 0x98, 0x3f, 0x69, 0xfc, 0x95, 0x5d,
			0x01, 0x5c, 0xdc, 0xf1, 0x48, 0xb3, 0xc8, 0xa7,
			0x0f, 0xfa, 0x24, 0x64, 0xe7, 0x94, 0x2d, 0x2c,
			0x44, 0xf4, 0x8f, 0xac, 0xdc, 0xc1, 0xd2, 0x25,
			0x61, 0x93, 0x4d, 0x74, 0xc8, 0x0e, 0x32, 0xc4,
			0xae, 0x5c, 0x6e, 0x7b, 0x99, 0x51, 0xc1, 0x01,
			0xc5, 0x63, 0xc8, 0x1b, 0x53, 0xce, 0xfb, 0xe1,
			0x0b, 0xdb, 0x32, 0xa1, 0xe1, 0x65, 0x04, 0xf4,
			0x99, 0xda, 0x9e, 0x35, 0xba, 0xe0, 0x34, 0xd5,
			0x8c, 0xf1, 0x96, 0xbc, 0x54, 0x0f, 0xfa, 0x84,
			0xf0, 0x34, 0x4c, 0x9c, 0x27, 0xa4, 0x2e, 0x40,
			0xb3, 0xfe, 0xc9, 0xec, 0xdd, 0x07, 0x6e, 0xd5,
			0x61, 0x0c, 0xee, 0x4c, 0x9f, 0xab, 0xd6, 0x73,
			0x4f, 0xd6, 0xc0, 0xeb, 0x78, 0x47, 0xc6, 0xc9,
			0xed, 0x34, 0x20, 0xa7, 0xa3, 0x0e, 0xa4, 0xae,
			0x88, 0xeb, 0x8b, 0x10, 0xaf, 0x33, 0x97, 0x59,
			0x95, 0x58, 0xf1, 0xc1, 0xfd, 0xed, 0x98, 0x16,
			0xfa, 0x13, 0x11, 0xe8, 0x09, 0xf2, 0x2d, 0x8a,
			0xcd, 0x45, 0x1e, 0xed, 0xf7, 0xe6, 0xae, 0xce,
			0x7f, 0xb7, 0x1f, 0xcd, 0xed, 0x79, 0xa3, 0xc0,
			0x3d, 0x8f, 0x7f, 0x06, 0xb8, 0xea, 0x95, 0xa4,
			0xf4, 0x7f, 0xf7, 0x62, 0xe1, 0xde, 0x7a, 0xca,
			0x88, 0x45, 0x3b, 0xeb, 0x37, 0x2b, 0x08, 0x68,
			0xe7, 0xf2, 0x23, 0x58, 0xc7, 0x3b, 0xfd, 0xef,
			0x4f, 0xd5, 0x9e, 0x3b, 0x88, 0x2f, 0x14, 0xde,
			0x6a, 0x8e, 0xe4, 0x8e, 0x4e, 0x5b, 0x64, 0x4a,
			0x39, 0xd0, 0x3f, 0x34, 0x4e, 0x67, 0xd1, 0xd4,
			0x87, 0xdd, 0x29, 0x7b, 0x25, 0xa7, 0x07, 0x65,
			0x11, 0x0b, 0xda, 0x7d, 0x43, 0x7c, 0xe7, 0xc6,
			0x08, 0x73, 0x71, 0x43, 0x7c, 0xe3, 0x4c, 0x1b,
			0x58, 0x3d, 0x4c, 0xde, 0xb3, 0xb8, 0x57, 0x2d,
		},
		20, /* salt length */
		/* V2 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA256 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0x74, 0x1a, 0x36, 0x65, 0x3f, 0xae, 0xdb, 0x98,
			0x80, 0x9e, 0x32, 0x5b, 0x21, 0x39, 0x22, 0x87,
			0xe9, 0x41, 0x36, 0x0c, 0x57, 0x3e, 0xa1, 0xe3,
			0xd1, 0x18, 0x7f, 0x44, 0xfe, 0xe5, 0x5d, 0x18,
			0x70, 0x00, 0xfc, 0xb1, 0x86, 0x5b, 0x52, 0xb9,
			0x90, 0x16, 0xd2, 0x68, 0x35, 0x50, 0xed, 0xfa,
			0x12, 0x3d, 0xd9, 0x61, 0xa4, 0x78, 0x91, 0x37,
			0x68, 0x9b, 0x1d, 0xf6, 0x40, 0x0e, 0x13, 0x6b,
			0x19, 0x1c, 0x5e, 0x68, 0xcb, 0x4c, 0x3e, 0xba,
			0x40, 0xee, 0x90, 0x5a, 0xd9, 0xf4, 0x50, 0xe8,
			0x0d, 0x03, 0x5c, 0x54, 0x54, 0xca, 0x7a, 0xdf,
			0x43, 0x93, 0x9d, 0x42, 0xf8, 0x17, 0x64, 0x2c,
			0xb8, 0x97, 0xe4, 0xf2, 0xd7, 0x28, 0xa7, 0x82,
			0xd8, 0x4f, 0xf4, 0x9a, 0xd6, 0xcd, 0x51, 0x37,
			0x23, 0xbc, 0x47, 0x4f, 0x2e, 0xdb, 0x65, 0xc2,
			0xaa, 0x27, 0x88, 0x8f, 0xec, 0xab, 0x59, 0x43,
		},
		/* S3 : 384 byte */
		{
			0x14, 0xf9, 0xaa, 0x92, 0x07, 0x00, 0x99, 0x90,
			0xcc, 0x26, 0x7a, 0x3e, 0xba, 0x82, 0xb7, 0x42,
			0x45, 0x20, 0x5d, 0x71, 0x4b, 0x00, 0x5a, 0xef,
			0x39, 0x9b, 0x9a, 0xf0, 0xd1, 0xf8, 0x53, 0x9c,
			0x8a, 0x43, 0x18, 0xb9, 0x1d, 0xb3, 0x9f, 0x0a,
			0x2f, 0x0f, 0x2d, 0xbe, 0x8d, 0x36, 0xd2, 0x68,
			0xff, 0x95, 0xd6, 0x71, 0x7b, 0x00, 0x40, 0xf7,
			0xbb, 0x0e, 0x4e, 0xe4, 0x22, 0x1f, 0x99, 0x20,
			0x88, 0xcc, 0x56, 0x6a, 0xb4, 0xe5, 0xbc, 0xfc,
			0x9a, 0x19, 0xe4, 0xb6, 0xe9, 0xb3, 0xf0, 0xe7,
			0x55, 0xd5, 0x61, 0x69, 0x22, 0x7c, 0xb0, 0x56,
			0xe0, 0x87, 0xbf, 0x3c, 0x0a, 0xa9, 0x9b, 0x73,
			0x09, 0xba, 0x20, 0xba, 0xa7, 0xd4, 0xdc, 0xc0,
			0x24, 0x21, 0xbf, 0xb5, 0x50, 0xa8, 0xf1, 0x69,
			0x08, 0x9e, 0x8c, 0xda, 0x05, 0xac, 0x86, 0xe3,
			0xfb, 0x25, 0x4d, 0x19, 0x58, 0xf8, 0x44, 0x6e,
			0xc2, 0x7f, 0x12, 0x19, 0xa8, 0x3a, 0xb7, 0xe2,
			0xac, 0x13, 0xd5, 0x9c, 0x8f, 0x25, 0x4f, 0x42,
			0x76, 0xf0, 0xa0, 0x26, 0xc3, 0x64, 0xbe, 0x50,
			0x2a, 0xc0, 0x3d, 0x2e, 0x62, 0x85, 0xa9, 0xc7,
			0xcc, 0x78, 0x36, 0xc6, 0x97, 0x1f, 0xdb, 0xe9,
			0xd2, 0x6e, 0x39, 0x6c, 0xf3, 0xc8, 0xa4, 0x2e,
			0xe2, 0x3b, 0xd3, 0x45, 0x59, 0xf4, 0x39, 0x15,
			0x8c, 0x93, 0x60, 0x9e, 0x14, 0xab, 0x2f, 0x33,
			0xbd, 0xb0, 0x7f, 0x47, 0x11, 0x03, 0xd1, 0xd0,
			0xec, 0x51, 0xc1, 0xc8, 0x3e, 0xbb, 0x01, 0x93,
			0x4e, 0x08, 0x92, 0x1f, 0x18, 0x28, 0x1b, 0x37,
			0xf5, 0x4b, 0xf7, 0x92, 0x34, 0x11, 0x2f, 0x38,
			0xdd, 0x4b, 0x21, 0x32, 0xfb, 0x73, 0x7f, 0x31,
			0x7b, 0x33, 0xa2, 0x78, 0x7e, 0xad, 0xd1, 0x3d,
			0x2d, 0xc6, 0x7c, 0xe7, 0x9a, 0x83, 0xfd, 0x0e,
			0x3e, 0x10, 0x7f, 0xcc, 0xdb, 0xaa, 0x4e, 0x1d,
			0xa7, 0x69, 0x33, 0xe9, 0xeb, 0x81, 0xaa, 0xdd,
			0x65, 0xed, 0x91, 0x13, 0xef, 0x5c, 0xcd, 0x4d,
			0x00, 0x7e, 0x54, 0x8f, 0xf3, 0x7a, 0x43, 0x2a,
			0xa9, 0x91, 0xd7, 0xbc, 0x46, 0xb4, 0x1c, 0x83,
			0x53, 0xdd, 0x87, 0x15, 0x5c, 0xf0, 0x3e, 0x44,
			0x49, 0xd7, 0xf9, 0x95, 0xce, 0x0a, 0xfd, 0xa8,
			0x51, 0xa1, 0x93, 0xff, 0x22, 0x58, 0x9b, 0xa6,
			0xe7, 0xa5, 0x7e, 0x3b, 0x2c, 0x1e, 0x82, 0xcf,
			0x2b, 0x5d, 0xc4, 0x4c, 0x4e, 0x44, 0x0f, 0xff,
			0x98, 0x87, 0x22, 0x00, 0x51, 0xb6, 0x4c, 0xb9,
			0x29, 0xe8, 0xa1, 0xc9, 0xfe, 0x61, 0x5c, 0xdf,
			0x27, 0x26, 0x80, 0x7b, 0x18, 0x2d, 0x84, 0xae,
			0x71, 0x83, 0xe5, 0x22, 0xfe, 0x74, 0xdb, 0x12,
			0xdf, 0x85, 0xea, 0xa3, 0xd6, 0x7d, 0x4d, 0xcc,
			0x38, 0xd1, 0xe7, 0x12, 0x4b, 0x90, 0x17, 0xe0,
			0x4a, 0x78, 0x93, 0xb5, 0x23, 0x98, 0xcd, 0x38,
		},
		20, /* salt length */
		/* V3 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA256 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0x43, 0x90, 0x3a, 0xf9, 0x35, 0x08, 0x25, 0x7f,
			0x13, 0x5e, 0x95, 0x1d, 0xae, 0x70, 0xfd, 0x4b,
			0x2f, 0x0e, 0xfe, 0xfa, 0x50, 0x5e, 0x35, 0x1d,
			0xcc, 0x7e, 0x58, 0x68, 0x13, 0x6c, 0x81, 0x57,
			0x43, 0x23, 0xc9, 0x73, 0x54, 0x07, 0x19, 0xd1,
			0x92, 0x83, 0x3d, 0xdb, 0x51, 0xf1, 0x3a, 0x52,
			0xdc, 0xba, 0xe2, 0x94, 0xae, 0xbe, 0xa5, 0x1b,
			0xe5, 0xf6, 0xaa, 0x47, 0xf3, 0x57, 0x1f, 0x5d,
			0x97, 0xfa, 0xcd, 0xcf, 0x0c, 0x7b, 0xef, 0xbe,
			0x80, 0x9f, 0x44, 0xbd, 0xc7, 0x39, 0x63, 0xd8,
			0x51, 0x4e, 0x4f, 0xd5, 0x59, 0x77, 0x4b, 0xb9,
			0x60, 0x87, 0xef, 0x8e, 0xda, 0x6e, 0x7c, 0x64,
			0x27, 0x5d, 0x6d, 0x96, 0xc4, 0x2b, 0x4e, 0x4e,
			0xc3, 0xf8, 0xd1, 0x4b, 0x78, 0x10, 0x1f, 0xe2,
			0x76, 0x11, 0x6c, 0x43, 0xba, 0x51, 0xce, 0xfe,
			0x25, 0x7a, 0x4f, 0xc6, 0x30, 0x77, 0x41, 0x74,
		},
		/* S4 : 384 byte */
		{
			0x0e, 0x5a, 0x58, 0x0e, 0xf7, 0xdc, 0x8e, 0x5f,
			0xe8, 0x3b, 0x1a, 0xaa, 0xc5, 0xec, 0xed, 0x0b,
			0xb6, 0x4f, 0x3a, 0x06, 0x5d, 0x24, 0x6d, 0x63,
			0x23, 0x61, 0xda, 0x17, 0x99, 0xb5, 0xbd, 0xa4,
			0x47, 0x27, 0x41, 0x9e, 0xb1, 0xf0, 0x4c, 0xbf,
			0x0d, 0x8e, 0x20, 0xb2, 0xea, 0x4c, 0xaf, 0xe6,
			0x44, 0xe7, 0x9e, 0x62, 0x27, 0x64, 0x3f, 0x75,
			0xc3, 0x6c, 0x30, 0x2b, 0x7c, 0x18, 0x42, 0x8f,
			0x12, 0x19, 0x1b, 0x60, 0xfe, 0x61, 0xdb, 0x4e,
			0xdc, 0x7f, 0xd6, 0x0f, 0x23, 0x72, 0x1f, 0x4e,
			0xea, 0x90, 0x35, 0x92, 0xc3, 0xc3, 0x0e, 0x12,
			0x10, 0x34, 0x1a, 0xa6, 0x9b, 0x95, 0x3e, 0x93,
			0xa9, 0x4b, 0x9b, 0xd6, 0x20, 0x1e, 0xc5, 0x9e,
			0xe3, 0x6f, 0x5f, 0x19, 0x85, 0x82, 0xda, 0x92,
			0xa0, 0xc2, 0x3e, 0xb8, 0xf8, 0xc9, 0x89, 0x49,
			0xca, 0x9e, 0x58, 0xf6, 0xb5, 0xa0, 0xd6, 0xdb,
			0xe4, 0x2c, 0xc9, 0x58, 0x70, 0x9b, 0x57, 0xf8,
			0x90, 0x18, 0x86, 0xff, 0xa3, 0x13, 0x4d, 0x0b,
			0x10, 0xee, 0xd0, 0x70, 0x7a, 0xc2, 0x2f, 0x62,
			0xc6, 0x5b, 0x6e, 0x1b, 0x7d, 0x3a, 0xf7, 0xe1,
			0x0f, 0xb9, 0x4d, 0xc9, 0xa3, 0xb9, 0x4f, 0xde,
			0x24, 0xb2, 0x4d, 0xc5, 0xa8, 0xc1, 0x47, 0x0e,
			0x6a, 0x3a, 0x00, 0x65, 0x59, 0xa8, 0xa3, 0x9c,
			0x0d, 0x78, 0x1b, 0x5f, 0x54, 0x0f, 0x22, 0x26,
			0xc1, 0x67, 0xdd, 0xa0, 0x22, 0xbc, 0x32, 0x7e,
			0x78, 0x25, 0xa9, 0xcc, 0x55, 0x38, 0x99, 0xe4,
			0xd5, 0x07, 0x40, 0x3c, 0x86, 0xc0, 0xf0, 0x24,
			0xd1, 0xfe, 0x40, 0xbf, 0x42, 0xef, 0x15, 0x40,
			0xb5, 0xbe, 0x20, 0xdb, 0xdc, 0x49, 0x12, 0x71,
			0x34, 0xac, 0xfb, 0x15, 0xa9, 0x6d, 0x54, 0x66,
			0x9f, 0x58, 0x63, 0x24, 0xbb, 0x9d, 0xf6, 0x93,
			0x01, 0xb1, 0xcd, 0xfa, 0x18, 0x1b, 0x0b, 0x35,
			0xa5, 0xce, 0x45, 0x41, 0x2b, 0x61, 0x01, 0x68,
			0x76, 0xa6, 0x74, 0x24, 0x1a, 0xa2, 0x7f, 0x0a,
			0x2b, 0xf3, 0x4d, 0xe1, 0x53, 0x45, 0xa1, 0xb3,
			0x98, 0x49, 0x06, 0x04, 0x05, 0xae, 0x35, 0x66,
			0x08, 0xa7, 0x0e, 0x3e, 0x62, 0x8c, 0x2a, 0xbf,
			0x25, 0xfd, 0x19, 0x2a, 0x62, 0x16, 0x20, 0x28,
			0xbe, 0xcf, 0xfc, 0x62, 0xfc, 0xe2, 0x29, 0x81,
			0x6a, 0x09, 0x46, 0xae, 0x38, 0x95, 0xee, 0xbd,
			0xc4, 0xd6, 0x66, 0xc3, 0x50, 0x78, 0x53, 0xb1,
			0xfe, 0x4d, 0x21, 0x30, 0xf3, 0xcd, 0xf0, 0x1b,
			0xa8, 0x0d, 0x06, 0x7f, 0x1d, 0xde, 0x40, 0x07,
			0xc3, 0xbe, 0x97, 0x5b, 0xc0, 0xda, 0xe0, 0x18,
			0x55, 0x54, 0x3b, 0x5a, 0x0a, 0xb3, 0xfe, 0x96,
			0xc7, 0x7f, 0xe9, 0x61, 0x26, 0xbf, 0xf0, 0x95,
			0xd6, 0x7d, 0x62, 0x60, 0xd1, 0x42, 0xf4, 0xbc,
			0x26, 0x08, 0xff, 0x69, 0x54, 0x48, 0x63, 0x46,
		},
		20, /* salt length */
		/* V4 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA256 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0x19, 0xe1, 0xe6, 0xed, 0x03, 0xee, 0x99, 0x81,
			0x81, 0x04, 0xe5, 0x35, 0xda, 0xde, 0xe3, 0xd2,
			0x7f, 0xba, 0x2f, 0x93, 0x91, 0xc5, 0x8e, 0x37,
			0xa3, 0xd4, 0xd3, 0x55, 0xc1, 0x54, 0x03, 0xf0,
			0xb1, 0x84, 0xb7, 0x4b, 0x78, 0xe2, 0xc2, 0xfe,
			0x21, 0xe2, 0x14, 0x4a, 0xb9, 0xe6, 0xde, 0xaa,
			0x9e, 0x6b, 0xf7, 0xaa, 0xfa, 0x80, 0x7f, 0xfc,
			0x8a, 0x7d, 0xc2, 0xc9, 0xa4, 0xf1, 0x5e, 0x45,
			0x78, 0xdc, 0xa9, 0x86, 0x96, 0x1f, 0x4d, 0x96,
			0xb0, 0x30, 0x35, 0xc7, 0xa1, 0x12, 0xad, 0x46,
			0xaf, 0x85, 0xc4, 0xf3, 0x0d, 0x7a, 0xf5, 0x70,
			0xa4, 0xc9, 0x07, 0x78, 0x14, 0x26, 0x32, 0x73,
			0xf5, 0x52, 0x83, 0x47, 0x62, 0x8e, 0x82, 0x70,
			0xb8, 0x55, 0x11, 0x53, 0x9b, 0x4a, 0x97, 0xcf,
			0x3a, 0x70, 0x5e, 0x16, 0xd4, 0x99, 0x3a, 0xbb,
			0x7c, 0x21, 0x6c, 0x0c, 0x19, 0xdb, 0xc5, 0xa1,
		},
		/* S5 : 384 byte */
		{
			0x57, 0x10, 0x10, 0xc6, 0x41, 0x38, 0x86, 0xf8,
			0x39, 0x45, 0xe5, 0xc9, 0xbb, 0x8d, 0x24, 0x32,
			0x2b, 0x19, 0x72, 0x7a, 0x15, 0x2b, 0xb5, 0x42,
			0x71, 0xaf, 0xd7, 0x46, 0x50, 0x28, 0xab, 0x43,
			0x90, 0xbd, 0xd9, 0x1a, 0xff, 0x52, 0x03, 0x77,
			0xd7, 0x51, 0x48, 0xa8, 0x00, 0xe4, 0x5b, 0x83,
			0xe5, 0xed, 0xeb, 0x73, 0xd9, 0x5a, 0x2d, 0x61,
			0x92, 0x74, 0x9b, 0x76, 0xe5, 0x78, 0x43, 0xf1,
			0x75, 0x0f, 0x56, 0x31, 0x67, 0x32, 0xd3, 0xed,
			0x49, 0xb3, 0x88, 0x66, 0x5f, 0x71, 0xb1, 0x73,
			0xac, 0xdd, 0x72, 0xa2, 0x41, 0xb1, 0x65, 0x5f,
			0x76, 0x69, 0x22, 0x8a, 0x00, 0x42, 0xcf, 0xb2,
			0x8b, 0x50, 0xc9, 0x04, 0x19, 0x57, 0x4f, 0x7e,
			0x4e, 0x85, 0x23, 0x5c, 0x6e, 0xfb, 0x3f, 0x96,
			0x47, 0x67, 0x64, 0x18, 0xb8, 0x9a, 0x7d, 0xe5,
			0x0e, 0xca, 0x24, 0x75, 0x90, 0xcc, 0x0f, 0x07,
			0xc6, 0x2b, 0xef, 0xd6, 0x51, 0x93, 0x10, 0x07,
			0xa2, 0x6d, 0xc3, 0x50, 0xfc, 0xc2, 0xdb, 0xd8,
			0xd1, 0xce, 0xbd, 0x9c, 0xa6, 0x65, 0xd2, 0xf4,
			0x03, 0xed, 0x98, 0x2e, 0xaa, 0xd6, 0xc4, 0x08,
			0x76, 0xac, 0x29, 0x5c, 0x80, 0xde, 0x88, 0x13,
			0x92, 0x6a, 0xb9, 0x39, 0x7d, 0x24, 0xdc, 0x26,
			0xc7, 0xc2, 0x59, 0x0a, 0x96, 0x5f, 0x7b, 0x33,
			0xa6, 0x5d, 0xf2, 0x81, 0xd8, 0xf5, 0x12, 0x8f,
			0xec, 0xb9, 0xaa, 0x1c, 0x72, 0x8e, 0x27, 0xc6,
			0xa6, 0x7d, 0xff, 0xa4, 0xec, 0xfc, 0x02, 0xb5,
			0xe5, 0xc5, 0xe4, 0x88, 0xd8, 0x76, 0xf0, 0x80,
			0x1c, 0xa4, 0x18, 0x55, 0xca, 0xe3, 0xc1, 0xdf,
			0x8a, 0x65, 0xa1, 0x0d, 0x4f, 0xe8, 0x93, 0x1f,
			0xcf, 0x7c, 0xb9, 0x80, 0x33, 0xd1, 0x39, 0xf1,
			0xd0, 0x73, 0xf6, 0x3f, 0xf5, 0xeb, 0x32, 0x1c,
			0x02, 0xbb, 0x24, 0xbe, 0x74, 0x20, 0xc1, 0x17,
			0x0e, 0xdb, 0x4a, 0x98, 0xb3, 0xfe, 0x96, 0xa2,
			0x1b, 0x40, 0x61, 0xb5, 0xee, 0x4a, 0x8e, 0xca,
			0xc4, 0x33, 0x11, 0xc2, 0xb5, 0xac, 0x81, 0xbd,
			0x39, 0x59, 0x93, 0x21, 0xe4, 0xa7, 0x04, 0x4c,
			0x91, 0xd7, 0x2d, 0xee, 0x60, 0xe3, 0x5a, 0x74,
			0x0a, 0x15, 0x8c, 0x6a, 0xe7, 0x9a, 0x10, 0xe1,
			0x4c, 0x44, 0x61, 0xfb, 0x44, 0x6a, 0x3e, 0xf4,
			0xc2, 0x73, 0x71, 0xbd, 0xbe, 0x5d, 0xc5, 0xf8,
			0x6d, 0x92, 0x29, 0xf4, 0x79, 0xa3, 0xe0, 0x8d,
			0x15, 0xc7, 0x72, 0xa7, 0x17, 0xba, 0xd1, 0x29,
			0x34, 0xbf, 0xb3, 0xf3, 0x94, 0x72, 0x44, 0x2e,
			0x1e, 0x05, 0x35, 0xf3, 0x21, 0x35, 0x7c, 0xd5,
			0x57, 0x0a, 0x58, 0xb4, 0xc5, 0x98, 0x50, 0xa0,
			0xf8, 0xf7, 0x88, 0xd0, 0xea, 0x80, 0x91, 0x4b,
			0xa7, 0xde, 0x1a, 0xa3, 0x07, 0xa3, 0xe3, 0x7e,
			0xa2, 0x1c, 0xdb, 0x93, 0xe0, 0x5b, 0x84, 0xdb,
		},
		20, /* salt length */
		/* V5 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA256 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0x77, 0x35, 0xe7, 0xfc, 0x6d, 0x79, 0xe9, 0xcf,
			0xaa, 0xe1, 0x93, 0x6a, 0x4c, 0xe6, 0x0d, 0xa9,
			0xcf, 0xde, 0xde, 0x41, 0x25, 0xc6, 0xa7, 0xa3,
			0x08, 0x40, 0x11, 0x23, 0x8f, 0xd8, 0x04, 0x59,
			0xff, 0x8b, 0x1a, 0xc7, 0xe4, 0x8a, 0x53, 0xbd,
			0x61, 0x3b, 0x0a, 0x7a, 0xc5, 0x81, 0x27, 0x94,
			0x5e, 0x3d, 0x97, 0x9a, 0x92, 0x15, 0xf9, 0xf7,
			0x2b, 0x74, 0x9a, 0x19, 0xf3, 0xd1, 0xa4, 0xd3,
			0x81, 0x35, 0x92, 0x00, 0x52, 0xf7, 0xe7, 0x6a,
			0x1c, 0xca, 0x20, 0xea, 0x5e, 0xfb, 0xe7, 0xd0,
			0x3c, 0xf3, 0x89, 0x84, 0x8a, 0x03, 0xaa, 0x6e,
			0x27, 0x61, 0x38, 0x14, 0x8c, 0x6d, 0x9d, 0x82,
			0xa4, 0x39, 0x39, 0xef, 0xdf, 0x49, 0x0f, 0x9d,
			0x82, 0x97, 0xbe, 0x01, 0x41, 0xda, 0xb4, 0x24,
			0x0f, 0x08, 0x9e, 0x49, 0x35, 0x1a, 0x22, 0xd0,
			0xa1, 0x0f, 0xd1, 0x5a, 0x81, 0x33, 0x57, 0x2e,
		},
		/* S6 : 384 byte */
		{
			0x3e, 0xaa, 0xb8, 0x6f, 0xd4, 0x6c, 0x48, 0x79,
			0x48, 0x41, 0xb4, 0x69, 0x5b, 0xf3, 0xd6, 0xfc,
			0x0f, 0xb3, 0x96, 0x94, 0x7e, 0x11, 0x8c, 0xe3,
			0xd1, 0x3c, 0x81, 0xb5, 0xdc, 0x4a, 0x3e, 0xd6,
			0xe2, 0x56, 0x71, 0x4e, 0x48, 0x09, 0x28, 0x9d,
			0xf2, 0xd6, 0x25, 0x5c, 0x89, 0x7c, 0xa8, 0xb2,
			0x84, 0xfb, 0xfc, 0xfd, 0xa7, 0x44, 0xc4, 0xbc,
			0x15, 0x38, 0xad, 0x39, 0x2b, 0x5a, 0xef, 0x9a,
			0xf3, 0x9b, 0xf1, 0x0c, 0xc3, 0x20, 0x19, 0x1f,
			0x3f, 0x45, 0x29, 0xdb, 0x9b, 0xfd, 0xc4, 0xd7,
			0xa6, 0x48, 0x54, 0x0a, 0xe8, 0xb5, 0x47, 0xe8,
			0x09, 0xf9, 0x3e, 0xcc, 0x81, 0xb9, 0x7d, 0xff,
			0x4f, 0xbd, 0x19, 0xa1, 0x85, 0x6c, 0xc1, 0x31,
			0x5a, 0xe0, 0x8c, 0x91, 0xde, 0xca, 0x2d, 0x70,
			0x50, 0xd4, 0x97, 0x00, 0x7f, 0x17, 0xdc, 0x5a,
			0xce, 0x82, 0xaf, 0xa2, 0x1e, 0xd1, 0x31, 0xac,
			0x3e, 0x42, 0x90, 0x31, 0xb6, 0xff, 0xb3, 0x12,
			0x4a, 0x95, 0xd5, 0x56, 0x0d, 0xa1, 0x4c, 0x64,
			0x1d, 0xcf, 0x74, 0x46, 0x7f, 0x2d, 0x7e, 0xff,
			0x31, 0x90, 0x33, 0x80, 0xb2, 0xbc, 0x92, 0xfc,
			0x66, 0x58, 0x98, 0x73, 0x04, 0x22, 0xd1, 0xe5,
			0x71, 0x05, 0x8e, 0x16, 0xc0, 0x83, 0xeb, 0xff,
			0xa6, 0x3f, 0x7f, 0xd8, 0xbe, 0x34, 0x75, 0x66,
			0xa1, 0x98, 0x4f, 0x0e, 0x2e, 0xc2, 0x28, 0x22,
			0x7f, 0xc0, 0xd5, 0xb0, 0x78, 0xf4, 0xc4, 0xbf,
			0xcd, 0x21, 0x33, 0x81, 0xaa, 0x64, 0x60, 0xb3,
			0xc6, 0x55, 0xd3, 0x7e, 0xd3, 0x77, 0xb4, 0xfb,
			0x38, 0xf7, 0x5c, 0x4d, 0xd4, 0x4f, 0x3c, 0x16,
			0x99, 0xbe, 0xcd, 0x54, 0xa8, 0x2e, 0x1d, 0x04,
			0x79, 0x8c, 0x42, 0x4a, 0xc5, 0x73, 0x0d, 0xce,
			0xc5, 0xbd, 0x0e, 0xbe, 0xa2, 0xc4, 0x61, 0xc8,
			0xdb, 0x4f, 0x63, 0x6c, 0x7f, 0x35, 0xa3, 0x6c,
			0x5c, 0xd3, 0x73, 0xc6, 0xc4, 0x61, 0xf6, 0x01,
			0x79, 0x7b, 0x7b, 0x18, 0x32, 0xa0, 0x9a, 0xb4,
			0xb1, 0xa0, 0x0b, 0x60, 0xf3, 0x5c, 0x4a, 0x67,
			0xe5, 0x3a, 0xaf, 0x81, 0x18, 0x28, 0x32, 0x85,
			0x7e, 0xcc, 0xf9, 0xfa, 0x7c, 0xfd, 0x0f, 0xac,
			0xc2, 0x0a, 0x7f, 0x08, 0x53, 0x3c, 0xd3, 0xb2,
			0x1b, 0x0d, 0x82, 0xc2, 0xb2, 0x6f, 0xe2, 0x5a,
			0x27, 0xd6, 0xc0, 0x53, 0x2d, 0x07, 0xad, 0xec,
			0x2a, 0xdd, 0x4f, 0x91, 0x94, 0x87, 0x96, 0x83,
			0x1a, 0xfd, 0x9b, 0x30, 0xac, 0x2c, 0x71, 0x84,
			0x5d, 0x8a, 0x8d, 0x91, 0xd3, 0x8b, 0x69, 0x19,
			0x04, 0xa7, 0x5c, 0xf0, 0x0b, 0x1e, 0x74, 0x2d,
			0x1a, 0x35, 0x47, 0x15, 0xe0, 0x69, 0x1a, 0x12,
			0x41, 0x95, 0x02, 0xd2, 0x78, 0x31, 0x9f, 0x47,
			0xaa, 0x63, 0xad, 0x68, 0x18, 0x1c, 0xdf, 0x92,
			0xb7, 0x5c, 0x4f, 0x69, 0x25, 0x41, 0xe8, 0xed,
		},
		20, /* salt length */
		/* V6 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA256 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0xda, 0xf9, 0xad, 0x0b, 0x70, 0xf5, 0xbe, 0x0c,
			0x15, 0x51, 0x05, 0x5d, 0xe8, 0xbf, 0x42, 0x56,
			0x37, 0x68, 0xc7, 0x86, 0xa0, 0x80, 0xc7, 0xed,
			0x76, 0x1d, 0xd1, 0x6b, 0x67, 0x49, 0x98, 0x79,
			0x5b, 0x55, 0xf5, 0x17, 0x9e, 0x71, 0x2f, 0x05,
			0x42, 0x7a, 0x79, 0x36, 0xbf, 0x13, 0xbc, 0x82,
			0xdd, 0xf7, 0x97, 0x9b, 0xd4, 0x8e, 0xa5, 0x42,
			0xb1, 0xbe, 0xfe, 0xbb, 0xfb, 0x03, 0x00, 0x7e,
			0x97, 0xc2, 0x4f, 0x2f, 0xe9, 0xda, 0x1a, 0xd0,
			0x3c, 0x7d, 0xa0, 0x36, 0x64, 0x3e, 0xf3, 0xb9,
			0xa1, 0x6a, 0xfc, 0x30, 0xc7, 0x9c, 0xc0, 0x1d,
			0x9c, 0x8c, 0xe1, 0x24, 0x82, 0x28, 0x68, 0xc2,
			0x56, 0xe6, 0xc1, 0x3b, 0x97, 0x58, 0x07, 0xd4,
			0xcb, 0x01, 0x83, 0x43, 0x1e, 0x67, 0x6f, 0x64,
			0x4d, 0x69, 0xff, 0x2d, 0xc2, 0xd3, 0xa2, 0xe4,
			0x01, 0x30, 0x85, 0x8f, 0x42, 0xdf, 0x58, 0xad,
		},
		/* S7 : 384 byte */
		{
			0x5a, 0x25, 0x2b, 0x81, 0xa8, 0x07, 0x03, 0xd8,
			0x01, 0x9f, 0xb7, 0x2f, 0xe0, 0xd6, 0x71, 0x9b,
			0x1d, 0x5f, 0xd0, 0x9f, 0x04, 0x19, 0x04, 0x30,
			0x1e, 0x64, 0x49, 0x23, 0xa2, 0x1f, 0x56, 0x26,
			0x0e, 0x62, 0xfe, 0x65, 0x59, 0x71, 0x78, 0xdf,
			0x62, 0x57, 0xcd, 0xe7, 0xb2, 0xe3, 0xd1, 0x35,
			0x1a, 0x57, 0x1f, 0x0c, 0xf2, 0x6f, 0x0b, 0xcf,
			0x45, 0x56, 0x57, 0x28, 0xcc, 0xc7, 0x56, 0xf3,
			0x09, 0x7d, 0x3f, 0xab, 0x6a, 0x4b, 0xa5, 0xc7,
			0x67, 0x07, 0xba, 0x13, 0xc4, 0x6e, 0xd0, 0x51,
			0x3d, 0x92, 0x3f, 0x5b, 0x11, 0x9d, 0x84, 0x95,
			0x39, 0x65, 0xe4, 0x67, 0x01, 0xd7, 0x3f, 0xf2,
			0x58, 0xfa, 0xec, 0xd9, 0xf3, 0xaa, 0x68, 0x40,
			0x78, 0x12, 0x9e, 0xe8, 0x81, 0xd0, 0xc6, 0xed,
			0x2d, 0x86, 0x64, 0xd5, 0xb2, 0xbd, 0x17, 0xc7,
			0x52, 0xd3, 0xd0, 0x02, 0xf8, 0x7b, 0xbe, 0xb9,
			0x6a, 0x33, 0x35, 0x6a, 0x10, 0x3f, 0x29, 0x73,
			0x8c, 0x47, 0x22, 0x3f, 0x98, 0xa5, 0x6f, 0x65,
			0xb6, 0x8f, 0xd4, 0x14, 0x3d, 0x8f, 0x2d, 0xdd,
			0xcd, 0x84, 0xf5, 0xd5, 0x83, 0xf5, 0xcc, 0xad,
			0x1b, 0xb4, 0xa8, 0xea, 0xb1, 0x89, 0x14, 0x98,
			0x3f, 0xad, 0xc5, 0x3a, 0x1f, 0x3d, 0x61, 0x05,
			0x12, 0xf9, 0x96, 0x4e, 0x0e, 0x61, 0xeb, 0x8e,
			0x47, 0x80, 0xee, 0x29, 0x51, 0x40, 0x01, 0xc4,
			0x8f, 0x6f, 0x26, 0x83, 0x38, 0x46, 0x30, 0x99,
			0xc5, 0x01, 0x61, 0xba, 0xfa, 0x69, 0x62, 0x30,
			0xd7, 0x05, 0x70, 0x85, 0xff, 0x57, 0xa7, 0x98,
			0xa1, 0xd3, 0xf3, 0xfd, 0xbe, 0xbe, 0x51, 0x59,
			0xb9, 0x54, 0xc1, 0xdd, 0xc6, 0x52, 0xd2, 0x30,
			0x9a, 0x52, 0xc7, 0x9e, 0x1d, 0xb5, 0xbb, 0x33,
			0xec, 0x2f, 0x4a, 0xcd, 0xa4, 0x73, 0x3b, 0xf9,
			0x8b, 0x34, 0xe1, 0x76, 0x3b, 0x8d, 0x0d, 0x92,
			0x21, 0xb8, 0x75, 0x4d, 0x1f, 0x31, 0x10, 0xe2,
			0xea, 0x8d, 0x32, 0x64, 0xab, 0x44, 0x8a, 0x06,
			0x90, 0x82, 0x74, 0xd2, 0xe7, 0x18, 0xa0, 0x0a,
			0x57, 0x78, 0x12, 0xf9, 0x5c, 0xb9, 0x14, 0x86,
			0x7d, 0xa2, 0x11, 0x1f, 0xae, 0xb8, 0x26, 0x43,
			0x1c, 0x6f, 0x76, 0x3d, 0x4a, 0x46, 0x34, 0x08,
			0xba, 0x33, 0x20, 0x8f, 0x72, 0xaf, 0xb7, 0x6c,
			0xa8, 0xdf, 0x01, 0xb5, 0x67, 0x09, 0x09, 0xe8,
			0x7a, 0x80, 0x76, 0x88, 0x95, 0xec, 0x22, 0x1f,
			0x02, 0xbf, 0x2f, 0x62, 0x23, 0xa3, 0xd1, 0x74,
			0x4e, 0xd6, 0xa8, 0xed, 0xb3, 0xca, 0x49, 0x09,
			0x2c, 0x26, 0x6e, 0x04, 0x9d, 0x96, 0x85, 0x6e,
			0x78, 0x08, 0xb9, 0xc4, 0xae, 0xb5, 0x88, 0x0b,
			0x18, 0x25, 0xeb, 0x32, 0x4d, 0x7d, 0x3d, 0xa6,
			0xce, 0xc0, 0xe0, 0xee, 0xad, 0xf9, 0x81, 0x90,
			0xf2, 0xdb, 0xa8, 0x9d, 0x2d, 0x4a, 0x87, 0x8d,
		},
		20, /* salt length */
		/* V7 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA256 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0xb4, 0x6a, 0xa5, 0x52, 0xe0, 0x65, 0x7e, 0xaa,
			0x6f, 0x34, 0xd6, 0x8f, 0x3a, 0xa1, 0x62, 0xaf,
			0xcc, 0x5d, 0x1e, 0xe9, 0x13, 0xe8, 0x9d, 0x54,
			0x5b, 0xb0, 0x9d, 0x83, 0x7e, 0x7d, 0xdf, 0x07,
			0x54, 0x0c, 0x36, 0x36, 0x71, 0xfa, 0xdd, 0x2b,
			0x10, 0xe5, 0x0d, 0x0e, 0x4a, 0xcd, 0xcf, 0x96,
			0x8c, 0xc0, 0x10, 0xcb, 0x47, 0xb6, 0xf3, 0x11,
			0xe9, 0xba, 0x6b, 0x62, 0x0d, 0x4a, 0x08, 0xdb,
			0xee, 0x01, 0x11, 0x7a, 0x22, 0x73, 0xd4, 0x26,
			0x85, 0x59, 0x3a, 0xef, 0x73, 0xee, 0x9e, 0x98,
			0x3a, 0x98, 0xdf, 0x52, 0xce, 0xcb, 0xb4, 0xcb,
			0xbf, 0x2a, 0x40, 0x67, 0x69, 0xf0, 0xe7, 0xcd,
			0x6b, 0x8f, 0x5f, 0xa6, 0x5a, 0x97, 0x0a, 0xa1,
			0xb6, 0xd8, 0x83, 0xbb, 0x1c, 0xcb, 0x77, 0xbd,
			0xbe, 0x2e, 0xb6, 0x06, 0x11, 0xef, 0x89, 0x8a,
			0xc5, 0x4b, 0x47, 0x1b, 0xfa, 0x37, 0x23, 0xe8,
		},
		/* S8 : 384 byte */
		{
			0x1a, 0x7a, 0xa6, 0xae, 0x05, 0xb5, 0x39, 0xdb,
			0x68, 0x8a, 0xb6, 0x29, 0x81, 0x44, 0x6e, 0xe7,
			0xf3, 0x23, 0xb5, 0x92, 0xe7, 0x71, 0x4e, 0x9f,
			0x5b, 0xd6, 0x67, 0x90, 0x6f, 0xa3, 0x44, 0x32,
			0xeb, 0xaf, 0x26, 0x67, 0x61, 0x41, 0xd5, 0x8a,
			0x36, 0xd6, 0x9b, 0x0a, 0x2b, 0x46, 0x7a, 0x7a,
			0x62, 0x64, 0x28, 0xfa, 0xdc, 0xf9, 0x89, 0x25,
			0xba, 0x5a, 0x18, 0xbf, 0x83, 0x1a, 0xda, 0xaf,
			0x25, 0x05, 0x04, 0x4d, 0xe1, 0xad, 0xb9, 0xc1,
			0x2b, 0x7e, 0xaa, 0x68, 0xd7, 0x45, 0xe1, 0xdc,
			0x82, 0x39, 0xad, 0x21, 0x00, 0x22, 0x84, 0xc3,
			0x42, 0xca, 0x74, 0xbb, 0x54, 0x67, 0x75, 0x20,
			0xe8, 0x30, 0x05, 0x30, 0xac, 0x0e, 0xe4, 0xf0,
			0x9b, 0x22, 0xb9, 0x44, 0x2f, 0xd1, 0xe3, 0x27,
			0x98, 0x93, 0xdf, 0xf9, 0xd8, 0x3d, 0x0a, 0x38,
			0x9a, 0x15, 0x6d, 0x5e, 0x7b, 0xa7, 0xcf, 0x2d,
			0x25, 0xb5, 0xe2, 0x60, 0x4a, 0xe9, 0x00, 0x97,
			0x21, 0xbe, 0xfe, 0x6a, 0x55, 0x29, 0xf8, 0xeb,
			0x25, 0xc4, 0xa6, 0x7d, 0x51, 0x36, 0x03, 0x52,
			0x6a, 0x2b, 0x59, 0x19, 0x30, 0x6e, 0x08, 0xa3,
			0x2e, 0x3e, 0x6e, 0xff, 0x81, 0xfb, 0x21, 0xf7,
			0x08, 0x8c, 0x97, 0xf8, 0xae, 0x41, 0xef, 0xb8,
			0xb8, 0xb1, 0x2b, 0x1b, 0xb9, 0x1d, 0xd0, 0xe8,
			0x40, 0xa4, 0x49, 0x60, 0x31, 0xdd, 0x01, 0x80,
			0x5d, 0x2e, 0xff, 0x2c, 0x89, 0x3f, 0x4d, 0xc4,
			0x8a, 0x72, 0x08, 0x5a, 0xf5, 0xf5, 0x3f, 0xf6,
			0xbe, 0x18, 0xea, 0x26, 0xce, 0xab, 0x0b, 0x3c,
			0xa1, 0x79, 0x81, 0xae, 0x8e, 0xe9, 0xfd, 0x1e,
			0x33, 0x93, 0xc1, 0x35, 0xc6, 0x82, 0x9b, 0xca,
			0x33, 0x45, 0x2b, 0x3a, 0xb1, 0xc8, 0xa0, 0x23,
			0x67, 0xbf, 0xf2, 0x24, 0xea, 0xf4, 0xbb, 0xfa,
			0xa1, 0x0b, 0x7e, 0x06, 0xf5, 0x4c, 0x1e, 0x3d,
			0x61, 0x75, 0xd3, 0x68, 0x21, 0xf1, 0x67, 0x0e,
			0x39, 0xc1, 0x58, 0xe9, 0xfe, 0x68, 0x9b, 0xb0,
			0x83, 0x3b, 0xba, 0xc7, 0x50, 0x04, 0x5b, 0xb1,
			0x50, 0xbc, 0x03, 0xc3, 0x6a, 0x3d, 0x70, 0xbc,
			0x25, 0x60, 0x82, 0x94, 0xe4, 0xc5, 0x01, 0x22,
			0x49, 0x4c, 0x7b, 0xce, 0x7d, 0xf8, 0xa7, 0x0c,
			0x0d, 0x6f, 0x8a, 0x44, 0xfc, 0xd4, 0xa1, 0x36,
			0x8a, 0x18, 0xc2, 0x9b, 0x8e, 0x9a, 0xe7, 0x51,
			0xfe, 0x72, 0x45, 0x1a, 0x38, 0x7c, 0xdf, 0x65,
			0x22, 0xe1, 0xac, 0x0c, 0xbe, 0x2f, 0xe7, 0xd6,
			0x10, 0x8c, 0x7f, 0x02, 0x4f, 0xee, 0x4a, 0x85,
			0xc0, 0x6c, 0xfc, 0x01, 0xf8, 0x55, 0xd1, 0x76,
			0xe7, 0x52, 0x69, 0x5d, 0xa6, 0x4e, 0x69, 0x9b,
			0x11, 0xff, 0x97, 0xa8, 0x4c, 0xc6, 0xd6, 0xe4,
			0x70, 0x47, 0xad, 0xd8, 0x04, 0xad, 0x79, 0x77,
			0xa2, 0x5b, 0x92, 0x20, 0xb1, 0x56, 0x70, 0x16,
		},
		20, /* salt length */
		/* V8 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA256 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0xdd, 0xd9, 0x80, 0xa7, 0x12, 0x9e, 0x3e, 0x85,
			0x6c, 0xf0, 0x69, 0xdd, 0x17, 0x18, 0xb8, 0xda,
			0xa0, 0x3d, 0x45, 0x80, 0x0a, 0x2b, 0x3e, 0x92,
			0x1e, 0x53, 0xdd, 0xb3, 0x02, 0x12, 0x9a, 0xcb,
			0x0d, 0x36, 0x9f, 0xac, 0xda, 0xbe, 0x7c, 0xf6,
			0x9b, 0xe1, 0x1c, 0x37, 0xb0, 0x52, 0xf7, 0x57,
			0x79, 0xc3, 0x8a, 0x34, 0x68, 0xa8, 0x04, 0x89,
			0x5a, 0x3d, 0x43, 0x14, 0x27, 0xca, 0xfe, 0xf8,
			0x77, 0x25, 0x44, 0x60, 0xd8, 0x79, 0x25, 0x65,
			0x0e, 0x47, 0xaf, 0x32, 0xaa, 0xaa, 0x1c, 0x65,
			0xdf, 0xdf, 0x49, 0xb9, 0x90, 0x04, 0x6a, 0xe2,
			0xad, 0x20, 0xfd, 0xb9, 0xc0, 0x64, 0xfc, 0x99,
			0xc3, 0xb1, 0x57, 0x09, 0x35, 0x5a, 0xa1, 0x9b,
			0x6d, 0x2a, 0x09, 0x14, 0x2d, 0xa9, 0x8d, 0xcc,
			0x7a, 0x9b, 0x6a, 0x58, 0xab, 0xca, 0xd7, 0x67,
			0xc1, 0x05, 0xf2, 0xea, 0xf6, 0x69, 0xfb, 0x78,
		},
		/* S9 : 384 byte */
		{
			0x14, 0x72, 0xa8, 0xd6, 0x4f, 0x3b, 0x80, 0x18,
			0xce, 0x15, 0x8d, 0x27, 0xd8, 0x50, 0x7e, 0x71,
			0x11, 0x77, 0xb1, 0x8d, 0x07, 0x15, 0xaf, 0x9b,
			0xe5, 0xdf, 0xd7, 0x5d, 0x93, 0xac, 0xa0, 0x7e,
			0x77, 0x0d, 0x62, 0x1a, 0xf4, 0x6f, 0xab, 0x2e,
			0x23, 0x37, 0x15, 0x38, 0xe0, 0x0c, 0x62, 0x41,
			0xbc, 0x64, 0xa2, 0x2c, 0x01, 0xcb, 0x77, 0x56,
			0x44, 0x6a, 0x6e, 0xbd, 0x45, 0x1f, 0x22, 0x39,
			0x7f, 0x8d, 0xbd, 0x6d, 0xcb, 0xa4, 0x6e, 0xb1,
			0xa1, 0xae, 0xe7, 0x0c, 0x5e, 0xde, 0xcc, 0xa9,
			0x23, 0x1c, 0x50, 0xc0, 0xd5, 0x85, 0xab, 0x7b,
			0x1e, 0xd7, 0xb1, 0x57, 0xe0, 0xfb, 0xbf, 0xe9,
			0xb1, 0x03, 0x8e, 0x70, 0x96, 0xfb, 0xb7, 0x4b,
			0x6e, 0x7a, 0xf8, 0x16, 0xda, 0xd0, 0xba, 0xe3,
			0x6e, 0x09, 0x5a, 0x93, 0xa9, 0x1e, 0xa4, 0x05,
			0x0b, 0x96, 0x79, 0x9a, 0x60, 0x9d, 0x86, 0x11,
			0x7b, 0x9b, 0x92, 0xe0, 0x3e, 0x3e, 0xe4, 0xc9,
			0xfe, 0x9c, 0xc4, 0x60, 0x34, 0x0f, 0x6d, 0xaf,
			0x99, 0x85, 0x62, 0x78, 0xfc, 0xf4, 0x11, 0xc7,
			0x02, 0x08, 0xc1, 0xf5, 0x01, 0x87, 0xb3, 0x08,
			0x8e, 0x8b, 0x70, 0x8c, 0x0f, 0x85, 0xf9, 0x91,
			0x40, 0xfa, 0xaf, 0xf2, 0x38, 0xfe, 0x0f, 0x8b,
			0x6e, 0x87, 0x0e, 0xb3, 0x6b, 0x97, 0xa0, 0x1a,
			0x32, 0x14, 0x52, 0x97, 0xb5, 0x12, 0x92, 0x6d,
			0xc1, 0xf0, 0x8d, 0xa9, 0x5d, 0x79, 0x6c, 0x8b,
			0xda, 0x96, 0x60, 0xb4, 0x2f, 0xe6, 0xef, 0x09,
			0xc4, 0xe6, 0x99, 0x49, 0x47, 0xce, 0xc7, 0x21,
			0x9a, 0x17, 0xc2, 0x28, 0xcf, 0x2f, 0xda, 0x45,
			0xbf, 0x00, 0x36, 0x03, 0x82, 0xa7, 0xcb, 0xfb,
			0x03, 0xba, 0xca, 0x75, 0xfd, 0x4d, 0x19, 0x57,
			0x74, 0x0d, 0xab, 0x20, 0xb9, 0x01, 0xe4, 0x71,
			0xb8, 0xad, 0xea, 0xbf, 0xbc, 0xa5, 0x1d, 0x2c,
			0xd3, 0x84, 0xa6, 0xbb, 0x7c, 0x6d, 0xe7, 0x49,
			0x1e, 0x00, 0x8f, 0x50, 0xb9, 0xdc, 0x80, 0x49,
			0x0e, 0xc0, 0x12, 0xb6, 0x14, 0x75, 0x6f, 0x2f,
			0x44, 0x89, 0x68, 0xb8, 0x83, 0xf7, 0xa7, 0xef,
			0x4d, 0x40, 0x20, 0xa3, 0x55, 0xea, 0x26, 0xa5,
			0x69, 0xe2, 0x82, 0x57, 0x48, 0x64, 0x8d, 0x70,
			0x60, 0x89, 0x17, 0x84, 0x2e, 0x42, 0xb6, 0x59,
			0x0e, 0xb2, 0x84, 0xad, 0x14, 0xd0, 0xae, 0xfd,
			0x57, 0xa7, 0xe5, 0x32, 0x58, 0x73, 0x91, 0xb5,
			0xb5, 0x27, 0x49, 0x10, 0x89, 0xc2, 0x7b, 0x29,
			0x3a, 0x2f, 0xf0, 0xb9, 0xe3, 0xf4, 0x33, 0x90,
			0x0b, 0x12, 0x14, 0xfe, 0xe2, 0xa3, 0xe0, 0xd3,
			0x91, 0x05, 0xd6, 0x04, 0x9b, 0x34, 0x09, 0x08,
			0x0c, 0x6f, 0x40, 0xf8, 0x57, 0xe4, 0xb4, 0x75,
			0xca, 0x50, 0x5d, 0x31, 0x9a, 0x43, 0xdd, 0x91,
			0x5f, 0xf4, 0x8d, 0x84, 0x0a, 0xd8, 0x9f, 0x09,
		},
		20, /* salt length */
		/* V9 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
};

/* 10 elements */
static mod3072_t SHA384[] = {
	/* SHA384 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0x69, 0x2a, 0xca, 0xaf, 0x5e, 0x27, 0x7c, 0xdd,
			0x4b, 0x3f, 0xdc, 0x0a, 0x1f, 0xf1, 0x78, 0x5b,
			0xfd, 0x28, 0xa3, 0xa8, 0xec, 0x1b, 0xc9, 0x7f,
			0xd0, 0x72, 0xff, 0x6c, 0x99, 0xaa, 0xde, 0x77,
			0xba, 0xba, 0x92, 0xef, 0xdc, 0xf7, 0x2e, 0x66,
			0xd4, 0x35, 0x42, 0xfd, 0xd3, 0x2f, 0xb0, 0xe2,
			0xdd, 0x29, 0xbb, 0x16, 0x7d, 0xd3, 0x61, 0x74,
			0xb6, 0x71, 0xeb, 0xef, 0x3c, 0x39, 0xc2, 0x1b,
			0xe5, 0xfc, 0x84, 0xef, 0x5a, 0x09, 0x57, 0xc9,
			0x12, 0x4f, 0x7e, 0xb2, 0x81, 0xc1, 0x2a, 0xe3,
			0x8c, 0xff, 0x92, 0x89, 0x41, 0x32, 0x45, 0xc6,
			0xc5, 0x37, 0xbf, 0xf8, 0x8d, 0x01, 0x3b, 0x3d,
			0xd1, 0x38, 0xc9, 0x37, 0x3e, 0x26, 0xa0, 0x0c,
			0xec, 0xd4, 0xb5, 0xb1, 0x8f, 0x70, 0x8d, 0x69,
			0xf1, 0xf2, 0x4f, 0x88, 0xa0, 0x00, 0x1d, 0x7d,
			0xe3, 0x0e, 0xa4, 0x0f, 0xf3, 0xc9, 0xf2, 0xe7,
		},
		/* S0 : 384 byte */
		{
			0x3f, 0x90, 0xae, 0xab, 0xfa, 0x9a, 0x5f, 0x00,
			0xe2, 0x41, 0xf3, 0xf6, 0x5d, 0xfe, 0x61, 0xba,
			0xf6, 0x7c, 0x13, 0x56, 0x35, 0x30, 0x42, 0xc3,
			0x56, 0x6e, 0xda, 0xcb, 0x11, 0xc7, 0x64, 0x97,
			0x37, 0xe5, 0xad, 0xf9, 0x4c, 0xfb, 0x05, 0xf2,
			0x61, 0x9a, 0xec, 0xc8, 0x89, 0x5d, 0xb4, 0x51,
			0x90, 0xfb, 0xdf, 0x35, 0xda, 0xb0, 0x11, 0x44,
			0xe2, 0x07, 0xb6, 0xf0, 0x92, 0x39, 0x27, 0xa6,
			0x14, 0x8d, 0x3f, 0x16, 0xea, 0xad, 0x05, 0xe7,
			0x3b, 0xcc, 0xb5, 0x62, 0xdc, 0x08, 0x7e, 0x2d,
			0x82, 0xdb, 0x3d, 0xce, 0x13, 0x0a, 0x83, 0xe8,
			0x30, 0x3b, 0xd7, 0xc3, 0x44, 0x7b, 0x3a, 0xe4,
			0xd3, 0x70, 0x0d, 0x47, 0x63, 0xba, 0x69, 0x81,
			0xd8, 0x26, 0x18, 0xac, 0x82, 0xa6, 0xe6, 0x64,
			0x23, 0xf2, 0x94, 0x78, 0x1a, 0x59, 0xb2, 0x0c,
			0xc9, 0x78, 0xc7, 0x9e, 0x2d, 0x5c, 0x10, 0x3b,
			0xfb, 0x9d, 0x47, 0x11, 0x92, 0x94, 0xc3, 0xc8,
			0x5b, 0x1d, 0x3c, 0x45, 0xa3, 0x68, 0x97, 0xd4,
			0x2e, 0x18, 0x35, 0x14, 0xcc, 0x8e, 0xdb, 0xbf,
			0xa1, 0xbe, 0x9e, 0xf1, 0x7b, 0x78, 0x28, 0x0b,
			0x5b, 0x62, 0x14, 0xda, 0xd7, 0x9d, 0x60, 0xdb,
			0x05, 0x7f, 0x22, 0x50, 0x65, 0x15, 0xb6, 0x84,
			0x3c, 0xe7, 0xd4, 0xdd, 0x6b, 0xd8, 0x61, 0xa8,
			0x89, 0xb3, 0x61, 0x64, 0xc3, 0x25, 0x14, 0x7b,
			0xae, 0xed, 0x71, 0x4d, 0x7a, 0x3f, 0x55, 0xae,
			0x51, 0xef, 0x6e, 0x6d, 0x4a, 0xe9, 0xe8, 0x62,
			0xd6, 0x77, 0xca, 0xba, 0x1a, 0x2d, 0xf3, 0x69,
			0xc2, 0x3d, 0x3f, 0xfe, 0x33, 0xdd, 0x42, 0xfe,
			0x70, 0x7e, 0x1f, 0xd8, 0xba, 0x62, 0x83, 0xaa,
			0xa0, 0xb5, 0x70, 0x35, 0x3b, 0x48, 0xa8, 0xe3,
			0x9f, 0xf7, 0x2a, 0x09, 0xf7, 0x00, 0xe0, 0x24,
			0x15, 0x0c, 0xe8, 0x7c, 0x04, 0x4a, 0x3e, 0xc7,
			0x45, 0xb2, 0x12, 0xae, 0x81, 0xaa, 0x57, 0x43,
			0xb9, 0x81, 0xa8, 0xbb, 0x95, 0xde, 0xb6, 0xb3,
			0xe1, 0x5c, 0x24, 0x87, 0xf7, 0x90, 0x01, 0x78,
			0xd5, 0x84, 0x0f, 0x8e, 0x79, 0x46, 0x62, 0x70,
			0x6d, 0xcd, 0xb1, 0x9b, 0xc0, 0xbd, 0xd5, 0x6c,
			0xb7, 0xfd, 0xf0, 0xe2, 0x1d, 0x10, 0xb0, 0x3a,
			0xda, 0xc4, 0x1b, 0x74, 0x9f, 0x31, 0xbd, 0x3e,
			0x7c, 0x4d, 0x07, 0xd5, 0xd4, 0xec, 0x8e, 0x79,
			0xd4, 0x24, 0x81, 0x2b, 0x6e, 0x83, 0xf1, 0xc7,
			0xb5, 0x97, 0x79, 0xe5, 0x80, 0x29, 0xf9, 0xb0,
			0x7d, 0xa3, 0xe7, 0x77, 0x95, 0xfc, 0xff, 0x6a,
			0xe8, 0xbb, 0x09, 0x8b, 0x1c, 0x00, 0xd1, 0xd2,
			0xa5, 0xbc, 0x0c, 0xb0, 0x05, 0xef, 0x3d, 0x8a,
			0xab, 0x63, 0xdd, 0xd8, 0x83, 0xd3, 0x8b, 0xac,
			0xdc, 0x64, 0x30, 0x7e, 0x91, 0x1c, 0x6e, 0x51,
			0x94, 0x67, 0x44, 0xf3, 0x61, 0xfe, 0x97, 0x8d,
		},
		20, /* salt length */
		/* V0 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA384 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0xd6, 0x9a, 0x51, 0x97, 0x3a, 0xe1, 0x6b, 0xbd,
			0x9f, 0x61, 0x13, 0x86, 0x44, 0xb7, 0x8c, 0xf7,
			0xa7, 0xb3, 0x6c, 0x18, 0x3d, 0x46, 0xf5, 0xfa,
			0x65, 0xa1, 0xa1, 0xf7, 0x9f, 0x9b, 0xff, 0x3c,
			0xf9, 0x53, 0xbe, 0x85, 0xcb, 0x17, 0x07, 0x28,
			0xa4, 0x85, 0x11, 0x55, 0x84, 0xf0, 0xef, 0x9c,
			0x4c, 0x06, 0xac, 0xaf, 0x72, 0xaf, 0x99, 0x6a,
			0xf9, 0x38, 0xe6, 0x10, 0xa1, 0xb0, 0x70, 0xd8,
			0x5b, 0x92, 0xda, 0xa8, 0xff, 0xa5, 0xdf, 0x20,
			0x41, 0x21, 0xe6, 0xda, 0xe2, 0x13, 0xd7, 0xf0,
			0x23, 0x02, 0x2e, 0xbf, 0x7f, 0xd3, 0x4e, 0xeb,
			0x99, 0xea, 0x17, 0xa4, 0x74, 0x92, 0xba, 0x27,
			0xe2, 0x9f, 0xcc, 0x87, 0x3f, 0x52, 0x9a, 0xad,
			0x5e, 0x7c, 0xbd, 0x9d, 0xc4, 0xe6, 0xee, 0xfc,
			0x14, 0xf1, 0x1b, 0xcf, 0xcb, 0x7a, 0xba, 0x86,
			0x2d, 0x00, 0x5d, 0x39, 0x80, 0x09, 0x86, 0x31,
		},
		/* S1 : 384 byte */
		{
			0x09, 0xac, 0x71, 0xe4, 0x1c, 0xe1, 0x3b, 0x20,
			0x46, 0x90, 0x56, 0xdb, 0xe4, 0x2b, 0x26, 0x37,
			0xfa, 0xb0, 0x0b, 0x87, 0xc4, 0x79, 0xef, 0x33,
			0xb6, 0xa3, 0xfd, 0x25, 0x6c, 0x5f, 0x94, 0x95,
			0xa5, 0xf3, 0xb0, 0xc8, 0x7e, 0xc0, 0x2a, 0xa3,
			0x72, 0x6a, 0xb2, 0xc5, 0x2f, 0xb8, 0x20, 0x3d,
			0x6e, 0xb1, 0x86, 0x6f, 0x1e, 0x44, 0x9a, 0x81,
			0x00, 0x72, 0x6d, 0xe2, 0x2b, 0x0a, 0xb2, 0x9e,
			0x7c, 0x79, 0x64, 0x0c, 0x35, 0x6b, 0x1c, 0xe0,
			0xf8, 0x1f, 0x55, 0xf7, 0x56, 0x38, 0x34, 0x90,
			0xa2, 0xe5, 0xd9, 0x5e, 0x8a, 0x28, 0x92, 0x9a,
			0xf9, 0x2c, 0xdf, 0x4e, 0x3c, 0xf6, 0xdf, 0xc0,
			0xe2, 0x92, 0x5a, 0x61, 0xb2, 0x44, 0x84, 0x00,
			0x29, 0x8e, 0x39, 0x5c, 0xdb, 0xec, 0x5c, 0x69,
			0x8c, 0xdb, 0x0c, 0xb4, 0x77, 0x83, 0xd2, 0xd5,
			0x98, 0xed, 0xe5, 0x3f, 0x83, 0x08, 0x8c, 0x70,
			0xa9, 0xbb, 0x52, 0xa0, 0x9b, 0x4d, 0x19, 0x61,
			0xf0, 0x5c, 0x8b, 0x7e, 0xee, 0x7d, 0x5a, 0x7a,
			0x6f, 0xe1, 0x1e, 0xe4, 0xc4, 0x90, 0xba, 0xd6,
			0x97, 0x20, 0x4b, 0xcd, 0x76, 0xeb, 0xa9, 0x60,
			0xd0, 0x20, 0xba, 0xa4, 0x6e, 0xf0, 0xd9, 0x7d,
			0x04, 0x48, 0x8b, 0x82, 0xa2, 0x27, 0x6d, 0x2a,
			0x4a, 0x40, 0xad, 0x53, 0x8a, 0xf6, 0xcc, 0xe9,
			0x41, 0x67, 0xed, 0x01, 0x4c, 0x88, 0xf6, 0x11,
			0x52, 0xa9, 0x0a, 0xa7, 0xf1, 0xf2, 0x4c, 0x7d,
			0xd3, 0x09, 0x3a, 0x3e, 0x24, 0xdf, 0x1b, 0xb7,
			0xbc, 0x90, 0xfe, 0x98, 0xa4, 0x5a, 0xfe, 0xf0,
			0x10, 0xdc, 0xa2, 0x70, 0x8e, 0x21, 0x31, 0x0b,
			0xb4, 0x9d, 0xcb, 0xd2, 0xb6, 0xd0, 0x18, 0x88,
			0x44, 0xcd, 0x50, 0x8c, 0xd2, 0x36, 0x5b, 0x6f,
			0xb0, 0xc2, 0x6b, 0xdb, 0x03, 0xf5, 0xf6, 0xa8,
			0xfc, 0xe2, 0xa2, 0xf1, 0xe5, 0x62, 0x7c, 0x0e,
			0x6e, 0xac, 0x64, 0x03, 0x0e, 0xb5, 0x7d, 0x28,
			0xe5, 0x64, 0xd8, 0x26, 0xa5, 0x28, 0xa4, 0x58,
			0xa6, 0x94, 0xc9, 0xaf, 0x96, 0x98, 0x08, 0x9a,
			0x8b, 0x27, 0xb3, 0xef, 0xbc, 0xb9, 0x86, 0x72,
			0x50, 0x13, 0x22, 0x57, 0x3a, 0x5c, 0x73, 0x87,
			0x34, 0xee, 0x8d, 0xd0, 0x06, 0x1c, 0xb9, 0x37,
			0x49, 0xe1, 0x64, 0xd1, 0xce, 0xfd, 0x28, 0x12,
			0x06, 0x92, 0x22, 0x8b, 0x9e, 0xa6, 0x4f, 0x71,
			0xd7, 0x69, 0x7a, 0x9a, 0x3a, 0xac, 0x4d, 0x7d,
			0xf6, 0x7a, 0x58, 0x0f, 0xd7, 0x96, 0x59, 0x4d,
			0xa3, 0x97, 0x15, 0x26, 0xc0, 0x5d, 0xa5, 0xdd,
			0x7f, 0xce, 0x6b, 0x8e, 0xa1, 0x66, 0xc0, 0x92,
			0xd2, 0xa8, 0x83, 0x64, 0x2a, 0xd9, 0x98, 0xe2,
			0xbe, 0x9e, 0x31, 0x9b, 0xc8, 0xd6, 0x67, 0x70,
			0x3c, 0x64, 0xe2, 0x26, 0xfb, 0x25, 0x28, 0x79,
			0xe5, 0x56, 0x33, 0xae, 0x7d, 0x56, 0x75, 0x7b,
		},
		20, /* salt length */
		/* V1 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA384 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x76, 0x4e, 0x2f, 0x04, 0x09, 0x9b, 0x72, 0xf6,
			0x42, 0x03, 0x40, 0xbb, 0xa4, 0x0f, 0xca, 0x53,
			0x7c, 0xe3, 0x76, 0x19, 0x0c, 0xcd, 0x27, 0xf5,
			0xe9, 0x1e, 0xbe, 0xd4, 0xa2, 0x39, 0xe1, 0x55,
			0xa2, 0xe9, 0xfb, 0xc5, 0x58, 0xf8, 0x83, 0x4e,
			0x95, 0xfd, 0x82, 0x6b, 0x35, 0x75, 0x4f, 0x9f,
			0x01, 0x15, 0x9d, 0x67, 0xd4, 0x5f, 0xb8, 0x6d,
			0xfd, 0x74, 0x0c, 0xfe, 0xe5, 0x29, 0xe7, 0xbe,
			0xf2, 0x5c, 0x80, 0x9b, 0xaa, 0x89, 0x37, 0xff,
			0x1c, 0x98, 0x1e, 0x4a, 0x7b, 0xf9, 0xb9, 0x7f,
			0x0d, 0xf3, 0x02, 0x3d, 0x43, 0x39, 0xb0, 0xf2,
			0x2a, 0xbf, 0xb9, 0x4b, 0x02, 0xcb, 0x16, 0xef,
			0x2b, 0x4f, 0xc6, 0x6c, 0x49, 0x76, 0x15, 0x71,
			0xa0, 0x7d, 0x1e, 0x3f, 0xc0, 0xc4, 0x8f, 0x5a,
			0x67, 0x25, 0xac, 0x84, 0xa3, 0x84, 0x97, 0xeb,
			0x37, 0xa7, 0xce, 0xa2, 0x40, 0x48, 0xf5, 0x4e,
		},
		/* S2 : 384 byte */
		{
			0x53, 0xfb, 0x56, 0x25, 0x09, 0x1c, 0xa2, 0xd5,
			0x23, 0xfc, 0x7a, 0x25, 0x82, 0x6f, 0x9b, 0xe4,
			0x1e, 0xdc, 0x09, 0x12, 0x17, 0xcd, 0x2f, 0xb0,
			0x47, 0x16, 0xa5, 0x3e, 0xff, 0xf7, 0xff, 0xb9,
			0x0a, 0xbb, 0x41, 0x52, 0x0b, 0x14, 0x05, 0xf7,
			0xd3, 0xc2, 0x22, 0xc7, 0x3e, 0xc9, 0xa5, 0x5d,
			0x13, 0xbf, 0xd7, 0x35, 0x70, 0xaf, 0xcd, 0xcc,
			0x5e, 0x5c, 0xa8, 0xfc, 0xda, 0xaa, 0xe5, 0xd9,
			0x36, 0xdb, 0x03, 0x2a, 0xfd, 0xae, 0xa1, 0xd1,
			0xcb, 0x10, 0x34, 0x1f, 0x98, 0xf8, 0x88, 0x44,
			0x32, 0xd0, 0xba, 0x69, 0x52, 0xc5, 0x8d, 0x0c,
			0x3a, 0xd6, 0x57, 0x0d, 0xc7, 0x2d, 0x34, 0x5c,
			0xb5, 0x1d, 0x68, 0xb6, 0xf8, 0xe0, 0xb9, 0x40,
			0x4d, 0x94, 0x5c, 0xd3, 0xd7, 0x4c, 0x25, 0x37,
			0x51, 0xbd, 0x58, 0x31, 0x59, 0xef, 0xda, 0x3e,
			0x38, 0xce, 0x7b, 0x4d, 0xe4, 0x2c, 0x86, 0x0b,
			0xcc, 0xb1, 0x4c, 0x32, 0x9c, 0x76, 0xfa, 0x8a,
			0x85, 0xe4, 0x41, 0xb0, 0x01, 0xe2, 0xf3, 0x01,
			0x8b, 0xa9, 0x22, 0x5c, 0xaa, 0x7b, 0x70, 0xc9,
			0x13, 0x03, 0xdd, 0x55, 0x96, 0xab, 0x07, 0x9e,
			0xd6, 0x72, 0xf8, 0xd6, 0x53, 0x2f, 0x04, 0xdf,
			0x58, 0x92, 0x49, 0xf4, 0x73, 0xec, 0xb1, 0x07,
			0xdc, 0x29, 0x6a, 0xb2, 0xce, 0x3c, 0x9d, 0x5e,
			0xf7, 0x40, 0x2c, 0x82, 0x1a, 0x65, 0x5b, 0xb1,
			0x5e, 0xd5, 0x68, 0xd5, 0x0a, 0x42, 0xde, 0x56,
			0xbb, 0x3e, 0x52, 0x97, 0xf8, 0x35, 0x08, 0x10,
			0x29, 0x7c, 0x9a, 0x76, 0x77, 0xb3, 0x75, 0x5f,
			0xab, 0x43, 0x7b, 0xa6, 0x1d, 0xba, 0x0d, 0xac,
			0xdd, 0x45, 0x4e, 0x5e, 0x4f, 0xc4, 0xca, 0xf4,
			0xcd, 0x42, 0x56, 0x14, 0xa0, 0x27, 0x2d, 0x07,
			0x0b, 0x55, 0x66, 0x1a, 0xa5, 0xa0, 0xc0, 0x70,
			0xbd, 0xc4, 0xe7, 0x44, 0x4d, 0xfa, 0xc5, 0x4c,
			0xe2, 0xa2, 0x2b, 0xec, 0x28, 0x0d, 0x2a, 0x5a,
			0x42, 0x95, 0xab, 0xea, 0xd4, 0x92, 0x95, 0x5c,
			0x56, 0xed, 0x16, 0xec, 0xd7, 0x05, 0x04, 0x1b,
			0x5c, 0x4a, 0x5d, 0x6c, 0x1a, 0x0c, 0xb6, 0x15,
			0xd2, 0xd1, 0x90, 0x00, 0xc6, 0x0f, 0x8b, 0xbb,
			0xa8, 0x7f, 0x48, 0x40, 0x8a, 0x98, 0x41, 0x56,
			0x9f, 0xa6, 0xb0, 0x83, 0x6e, 0x4f, 0xc7, 0x57,
			0xc6, 0x43, 0xf0, 0x45, 0x5f, 0x2d, 0x8b, 0x9d,
			0x46, 0xd9, 0xc8, 0x6c, 0x5d, 0x0f, 0x94, 0x20,
			0x1a, 0xdc, 0x3a, 0xb2, 0xb7, 0xdd, 0xf7, 0xd8,
			0x28, 0xda, 0x6c, 0x5e, 0x0c, 0xb5, 0xf0, 0xee,
			0xb7, 0x6a, 0x1d, 0xf3, 0xcf, 0xc8, 0xa0, 0x21,
			0x31, 0x45, 0x90, 0x59, 0xbe, 0x9b, 0xef, 0x9a,
			0xf6, 0x5c, 0x61, 0xa8, 0x02, 0x9b, 0xd9, 0xdd,
			0x94, 0x8a, 0xa8, 0x11, 0x8f, 0xd6, 0x3f, 0xac,
			0xe9, 0xa9, 0xe3, 0xd8, 0xd6, 0xda, 0x92, 0xe2,
		},
		20, /* salt length */
		/* V2 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA384 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0xdb, 0x81, 0xbf, 0xe8, 0x76, 0x9b, 0xe9, 0x3c,
			0x3f, 0x30, 0xa0, 0xf2, 0xc8, 0xd5, 0x37, 0xe2,
			0x17, 0xc6, 0x05, 0xd9, 0x5a, 0xcf, 0xb7, 0x98,
			0xd6, 0x6c, 0xea, 0x1f, 0x21, 0x2f, 0xfa, 0x2b,
			0x91, 0x62, 0x14, 0xfd, 0xf4, 0xee, 0x93, 0x5d,
			0x81, 0x7b, 0x89, 0xdf, 0x0f, 0xd9, 0x9b, 0x65,
			0x85, 0x9e, 0x13, 0xe5, 0x92, 0x11, 0x30, 0x6c,
			0x9e, 0x47, 0xc0, 0x63, 0x00, 0x8c, 0xf1, 0x12,
			0x71, 0x05, 0x65, 0x62, 0xc0, 0x93, 0x84, 0xe7,
			0xc8, 0xf9, 0x16, 0x1d, 0xa0, 0x42, 0x11, 0xf2,
			0x11, 0x5d, 0xaf, 0x85, 0x4d, 0x0b, 0xc4, 0x2e,
			0xdd, 0xfb, 0x4f, 0xbe, 0xdc, 0x32, 0x4f, 0x08,
			0x63, 0xb1, 0xd8, 0xa0, 0x44, 0x52, 0x65, 0xe2,
			0xfb, 0xf4, 0x70, 0x36, 0xe1, 0xd7, 0x40, 0x93,
			0xa3, 0x49, 0x03, 0x42, 0xf3, 0x83, 0x1c, 0xe5,
			0x7d, 0xce, 0xbe, 0xb7, 0x1d, 0xea, 0xba, 0x15,
		},
		/* S3 : 384 byte */
		{
			0x42, 0xdc, 0x91, 0x72, 0x2a, 0x4f, 0x20, 0x47,
			0x05, 0x5a, 0xad, 0xc2, 0xbf, 0x7f, 0x7e, 0x66,
			0xde, 0x4c, 0x95, 0xf2, 0xe6, 0xb5, 0x27, 0xf7,
			0xe2, 0x4a, 0x16, 0xfa, 0x2e, 0x95, 0xc2, 0xc6,
			0x1e, 0xda, 0x58, 0x1e, 0xea, 0xac, 0x72, 0xae,
			0x5d, 0xd2, 0xe0, 0x96, 0xb8, 0x9a, 0xad, 0xc0,
			0x30, 0x9d, 0x9d, 0x10, 0x2c, 0xf1, 0x13, 0xff,
			0xc7, 0x53, 0xb8, 0x00, 0x83, 0x7c, 0xc8, 0xb2,
			0x6c, 0xd1, 0xb1, 0x66, 0x8b, 0xd0, 0x47, 0xda,
			0x8e, 0x2e, 0xc0, 0x56, 0x54, 0xd7, 0xe1, 0x68,
			0x66, 0xda, 0xf6, 0x6b, 0x88, 0x64, 0x54, 0xc0,
			0x6d, 0xe8, 0x61, 0x78, 0x15, 0xe3, 0x65, 0x2c,
			0x51, 0x85, 0xe7, 0xf5, 0x9d, 0xd9, 0x8f, 0xd9,
			0xfe, 0xd0, 0x66, 0xa2, 0x31, 0x74, 0x14, 0x3f,
			0x4a, 0xb2, 0x38, 0x74, 0x4b, 0x67, 0x7d, 0x52,
			0xb6, 0xdf, 0x4f, 0x5a, 0xbf, 0xcb, 0x12, 0x26,
			0x11, 0x46, 0x42, 0x4c, 0x2b, 0xfe, 0x03, 0x3c,
			0xa4, 0xda, 0x22, 0xab, 0x28, 0xf9, 0xfc, 0x0d,
			0x0a, 0xbd, 0xd8, 0x32, 0x0f, 0x9e, 0x2d, 0x13,
			0x73, 0x9b, 0xf5, 0x85, 0x4b, 0x73, 0x41, 0x7b,
			0x16, 0xc5, 0x8c, 0xee, 0x45, 0x64, 0x02, 0xe0,
			0xec, 0xaa, 0x05, 0x5c, 0x4c, 0x2f, 0x76, 0xb2,
			0x80, 0x32, 0x3a, 0xfc, 0x1a, 0xc9, 0xb6, 0x65,
			0xd8, 0x59, 0xa3, 0x59, 0x43, 0xb8, 0x7e, 0x3e,
			0x5b, 0xba, 0xd6, 0x20, 0xbc, 0x46, 0x1f, 0x49,
			0xc1, 0xcc, 0x60, 0x3f, 0x23, 0xd4, 0x42, 0xcc,
			0x87, 0xff, 0xbb, 0x03, 0xaa, 0x0d, 0x16, 0x80,
			0xe2, 0x55, 0x6a, 0x1e, 0xaf, 0x49, 0x06, 0x36,
			0x20, 0xfd, 0xc9, 0xb1, 0xc8, 0x1c, 0xea, 0x48,
			0xf7, 0xd8, 0x79, 0xda, 0xc2, 0x40, 0xe2, 0x9a,
			0xcf, 0xdd, 0xc4, 0x74, 0x49, 0xe0, 0xd2, 0xfc,
			0x0a, 0xc3, 0x2b, 0xa7, 0x7d, 0x9b, 0x08, 0x9d,
			0x7b, 0x1d, 0x68, 0xf3, 0x5d, 0x52, 0x4a, 0xc1,
			0xc6, 0x4a, 0xa5, 0x80, 0xd4, 0x3b, 0xca, 0x3a,
			0xb5, 0x3d, 0x4b, 0x98, 0x97, 0x05, 0x44, 0x9b,
			0x18, 0xc5, 0x56, 0x51, 0x67, 0x59, 0x5a, 0x8b,
			0xba, 0x0e, 0xe7, 0xef, 0x7e, 0x9c, 0x56, 0x44,
			0xbf, 0xf5, 0xa1, 0xca, 0x8a, 0xb7, 0x10, 0x5a,
			0x2d, 0xb7, 0xb5, 0x61, 0x32, 0x37, 0x60, 0xfe,
			0x7a, 0x05, 0xf3, 0xd0, 0x7d, 0x4f, 0x76, 0xf3,
			0xfd, 0xff, 0xc1, 0x42, 0x40, 0x92, 0xd9, 0xa5,
			0x4e, 0x91, 0x71, 0xed, 0x47, 0x8c, 0x61, 0x31,
			0x0f, 0x3c, 0xec, 0xe7, 0xb6, 0x68, 0xbb, 0x2b,
			0x85, 0x9a, 0xc2, 0xa7, 0x6e, 0x8c, 0xe6, 0xc8,
			0x0a, 0x7d, 0x62, 0xfe, 0xb8, 0x54, 0xd7, 0x24,
			0xfc, 0x7d, 0xa4, 0xa3, 0x7f, 0x3a, 0x7b, 0x45,
			0x40, 0xbe, 0xa4, 0x5c, 0x86, 0xe4, 0x3f, 0x7d,
			0x81, 0x9c, 0x3f, 0x4b, 0xfa, 0x91, 0x09, 0x83,
		},
		20, /* salt length */
		/* V3 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA384 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0xc5, 0xe4, 0x92, 0x5a, 0xbc, 0xd7, 0x6f, 0xde,
			0x1f, 0x1f, 0x85, 0x18, 0x3d, 0xa1, 0xaf, 0xc7,
			0xea, 0xac, 0x92, 0x49, 0x1a, 0x3a, 0xd4, 0x04,
			0xae, 0x73, 0x89, 0x04, 0x8d, 0x73, 0xee, 0xe5,
			0xf7, 0x4d, 0x5a, 0xf4, 0x16, 0x50, 0x71, 0x91,
			0xd6, 0x89, 0x9b, 0xd8, 0xf7, 0x88, 0xb5, 0x80,
			0x61, 0x67, 0x06, 0x02, 0xb7, 0x09, 0x74, 0xfe,
			0x2e, 0xa0, 0xf7, 0xdb, 0xa0, 0x72, 0x4f, 0xaf,
			0xdc, 0xdc, 0xf2, 0x5e, 0x41, 0x97, 0x49, 0x05,
			0x8b, 0x3a, 0x1b, 0x99, 0xec, 0x00, 0x4a, 0xcb,
			0x5f, 0xcc, 0xbd, 0x2f, 0x39, 0x6a, 0x9d, 0x9e,
			0x00, 0x18, 0xc4, 0xd9, 0x81, 0x45, 0x72, 0x6e,
			0x1d, 0x98, 0x42, 0xe0, 0x64, 0x34, 0x5d, 0x02,
			0xe5, 0x3b, 0xee, 0xa4, 0x43, 0x90, 0xd6, 0x71,
			0x8c, 0xe2, 0xa0, 0x18, 0xc7, 0xe6, 0xb7, 0xaa,
			0xcc, 0xe3, 0xd6, 0x43, 0x59, 0x74, 0xc1, 0xed,
		},
		/* S4 : 384 byte */
		{
			0x54, 0x68, 0xa3, 0xe3, 0xc9, 0xe8, 0x08, 0x03,
			0x27, 0x42, 0x76, 0x7a, 0xcb, 0x41, 0xa6, 0x75,
			0x11, 0x58, 0x0c, 0x4a, 0x82, 0x44, 0xe9, 0x7d,
			0x2a, 0xc9, 0x46, 0xde, 0xb6, 0xe2, 0xca, 0xf9,
			0x89, 0x47, 0x64, 0x61, 0x96, 0x9b, 0x0f, 0x43,
			0xfd, 0x17, 0x25, 0xf6, 0x57, 0x24, 0x82, 0x35,
			0xcc, 0xa1, 0x7b, 0x63, 0x3e, 0xa6, 0x4a, 0x92,
			0xfd, 0x76, 0xe1, 0xb6, 0x27, 0x5c, 0xf2, 0x72,
			0x9d, 0x48, 0x8c, 0x4a, 0x42, 0x57, 0x45, 0x2b,
			0xe3, 0x2d, 0xdf, 0x26, 0x5e, 0x1e, 0xdf, 0x72,
			0x2f, 0x50, 0xad, 0xa0, 0x27, 0x36, 0x5c, 0xb6,
			0x91, 0xca, 0x51, 0xd8, 0x64, 0x16, 0x88, 0x62,
			0x42, 0x8d, 0xe7, 0x17, 0xeb, 0x29, 0x65, 0xd3,
			0x56, 0x2c, 0x64, 0xaf, 0xee, 0xaa, 0x80, 0x60,
			0xf8, 0x93, 0xeb, 0x40, 0x0e, 0x89, 0x45, 0xc8,
			0xc7, 0x27, 0xe2, 0xb1, 0x65, 0x28, 0x94, 0x22,
			0x13, 0x46, 0xdc, 0x31, 0x02, 0xdf, 0xc8, 0x09,
			0xc8, 0xe4, 0x77, 0xa8, 0x57, 0x08, 0x06, 0xe2,
			0x05, 0x4d, 0xb3, 0xda, 0x79, 0x82, 0x87, 0x9b,
			0x49, 0x7f, 0x22, 0xa1, 0x8b, 0xec, 0x25, 0x08,
			0x74, 0xca, 0x59, 0x25, 0xde, 0xcf, 0x26, 0xb6,
			0x32, 0x27, 0xc9, 0x48, 0x40, 0x1b, 0xa7, 0xd1,
			0x5e, 0x3e, 0x79, 0x10, 0x6d, 0x2a, 0x73, 0x38,
			0x3c, 0x38, 0x17, 0x10, 0xd1, 0xb7, 0x50, 0xfe,
			0x82, 0x88, 0xd2, 0xd8, 0x0e, 0x00, 0x33, 0x85,
			0xf2, 0xe2, 0x82, 0x9c, 0xf4, 0xa6, 0x82, 0x3e,
			0xb0, 0xd9, 0x49, 0x68, 0x1c, 0x08, 0x43, 0xc4,
			0xb2, 0xc5, 0x82, 0xfc, 0x8b, 0x62, 0x86, 0x64,
			0x05, 0xcf, 0x8e, 0x51, 0xc8, 0xa3, 0x06, 0x27,
			0xf5, 0xf4, 0xb9, 0xef, 0x8a, 0x4f, 0x52, 0x21,
			0xfa, 0x89, 0xb0, 0x52, 0x15, 0x20, 0x44, 0xbe,
			0xdd, 0x99, 0x61, 0xfb, 0x45, 0xd0, 0xad, 0xde,
			0x68, 0x92, 0x8f, 0x69, 0xcb, 0xdc, 0x6b, 0x68,
			0xc3, 0x7e, 0x3c, 0xde, 0xeb, 0xae, 0xfe, 0xa0,
			0x20, 0x32, 0x02, 0x11, 0x64, 0x2f, 0xa1, 0xdf,
			0x95, 0xd2, 0x4a, 0x22, 0x9e, 0xb6, 0x4c, 0xbd,
			0x5b, 0x2c, 0xb2, 0x1e, 0xeb, 0xf6, 0x31, 0x3b,
			0x69, 0x8c, 0xda, 0x79, 0xc8, 0x97, 0x79, 0xa3,
			0x45, 0x53, 0x46, 0x63, 0xfa, 0x2b, 0xb6, 0x82,
			0x54, 0x12, 0x45, 0x71, 0x52, 0x48, 0xab, 0xb1,
			0x17, 0x5f, 0x4f, 0x5a, 0xfe, 0xbe, 0xcb, 0xcc,
			0x67, 0xe9, 0x91, 0x51, 0x38, 0x46, 0xab, 0x49,
			0xbf, 0x3f, 0x61, 0xb5, 0xd9, 0x12, 0x68, 0x85,
			0xec, 0x45, 0x46, 0x17, 0x70, 0x02, 0x69, 0x5a,
			0x0a, 0x7f, 0xc7, 0x5c, 0x0b, 0xcc, 0x5b, 0x97,
			0x24, 0x1e, 0xc9, 0x27, 0xff, 0x43, 0x90, 0x01,
			0x3f, 0x65, 0xf6, 0xd6, 0x9d, 0xee, 0xe8, 0xae,
			0xe8, 0x3d, 0xa1, 0x9b, 0x72, 0xfd, 0x2b, 0x76,
		},
		20, /* salt length */
		/* V4 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA384 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0x22, 0x59, 0xae, 0x2c, 0xed, 0xec, 0x5e, 0x26,
			0x23, 0x1c, 0x92, 0x7d, 0x87, 0xf6, 0x35, 0x3e,
			0x83, 0x87, 0x31, 0x7f, 0x62, 0x42, 0x3d, 0xe8,
			0xbf, 0xa2, 0xf6, 0x73, 0x3d, 0x10, 0x32, 0x6f,
			0x79, 0x06, 0x91, 0xcb, 0x3d, 0x75, 0x55, 0x91,
			0xf7, 0x64, 0x73, 0xb1, 0x69, 0xda, 0xe9, 0x00,
			0xa1, 0xff, 0x53, 0x80, 0x2c, 0x7e, 0xb9, 0xe2,
			0x87, 0x2d, 0xac, 0x49, 0xd6, 0xf0, 0x91, 0x51,
			0xd8, 0xdb, 0x3c, 0x50, 0x1b, 0x96, 0xc1, 0xdc,
			0x6c, 0x06, 0x86, 0x8d, 0x92, 0x29, 0x9d, 0x03,
			0x3c, 0x42, 0x51, 0x2b, 0x37, 0x37, 0xff, 0xbf,
			0xe4, 0x37, 0x28, 0x0c, 0xea, 0x9f, 0xc3, 0xf8,
			0x28, 0x1f, 0xd8, 0x43, 0xee, 0x19, 0x4a, 0x0d,
			0x6c, 0x49, 0xf5, 0x99, 0x6a, 0xab, 0xf8, 0x50,
			0x3a, 0x99, 0x54, 0x09, 0xeb, 0x34, 0xb7, 0x87,
			0xc0, 0x06, 0x92, 0x43, 0xe0, 0xe6, 0x71, 0x6c,
		},
		/* S5 : 384 byte */
		{
			0x7b, 0x80, 0x3d, 0xa9, 0x35, 0xde, 0x10, 0xf8,
			0x63, 0x96, 0xc2, 0x25, 0xb2, 0x6b, 0x7a, 0xa5,
			0x9d, 0xf2, 0xb3, 0x80, 0xa2, 0x8c, 0x49, 0x24,
			0x12, 0xff, 0x03, 0x46, 0x43, 0xf6, 0x8b, 0x8a,
			0x87, 0x19, 0x8b, 0x4a, 0x6a, 0xc1, 0xbf, 0xae,
			0x5f, 0xb6, 0x74, 0x8a, 0x5c, 0xd0, 0x96, 0x62,
			0xa1, 0xbe, 0xce, 0xb1, 0xc2, 0x11, 0xcd, 0xb3,
			0xa6, 0x02, 0x36, 0x2d, 0x95, 0x34, 0xff, 0xc9,
			0x24, 0x5c, 0x2e, 0x3f, 0x29, 0xa1, 0x34, 0xae,
			0x7a, 0x88, 0xfb, 0xf8, 0xeb, 0xd1, 0x1b, 0xdd,
			0xaa, 0x09, 0x17, 0x7a, 0x0a, 0xbe, 0x55, 0x5e,
			0x6c, 0xee, 0xe2, 0x70, 0x34, 0xbb, 0xec, 0x25,
			0x47, 0x76, 0x2f, 0x78, 0xa3, 0xad, 0x33, 0xd1,
			0x7c, 0x29, 0x77, 0xb2, 0x76, 0x7f, 0xba, 0x71,
			0xee, 0xdc, 0x17, 0xe6, 0x00, 0x5f, 0x00, 0xf6,
			0xa8, 0x7b, 0x20, 0x85, 0x8e, 0x75, 0xaf, 0x03,
			0x4e, 0xbe, 0x03, 0x8e, 0xc8, 0x27, 0xea, 0xff,
			0x4b, 0xe0, 0x0c, 0x1d, 0xba, 0xde, 0xff, 0xee,
			0x25, 0x3a, 0x05, 0xb1, 0xa2, 0x64, 0x0c, 0xc0,
			0x7f, 0x8e, 0x49, 0xef, 0xd2, 0x75, 0xe5, 0x58,
			0x54, 0xd4, 0x54, 0x67, 0xfa, 0x8a, 0x7e, 0x7f,
			0x76, 0xcd, 0x9f, 0x2b, 0x8e, 0x3b, 0xb1, 0x11,
			0xeb, 0x8e, 0xb8, 0x1d, 0x62, 0x12, 0x9e, 0x8f,
			0x9c, 0xb0, 0xa4, 0x43, 0xd9, 0x97, 0x9a, 0x3d,
			0x90, 0x40, 0x5f, 0xd1, 0xf9, 0x8f, 0x09, 0x14,
			0x44, 0xbc, 0xc7, 0xb5, 0xab, 0x03, 0xd2, 0x6b,
			0x0e, 0xb1, 0x2a, 0x96, 0x81, 0x30, 0xa4, 0xfa,
			0xca, 0x13, 0x8e, 0x9c, 0xe2, 0x01, 0x52, 0x00,
			0x6c, 0x11, 0x11, 0xed, 0xd1, 0x44, 0xc3, 0x98,
			0x60, 0xcd, 0xe1, 0x82, 0xab, 0xa9, 0xd0, 0xcd,
			0x9a, 0x1e, 0x52, 0x6d, 0xd2, 0x16, 0x15, 0x80,
			0xcc, 0x4d, 0xee, 0xe8, 0x04, 0x03, 0x30, 0x90,
			0xb6, 0x80, 0x24, 0xe6, 0x82, 0x28, 0xda, 0xda,
			0xae, 0x67, 0x33, 0x5e, 0x54, 0x90, 0xb9, 0xf8,
			0x62, 0x23, 0xf1, 0x65, 0x95, 0xeb, 0x5f, 0xeb,
			0xfc, 0x8a, 0x39, 0x7b, 0x25, 0xe9, 0xaa, 0x90,
			0x9d, 0x6f, 0xd7, 0xde, 0xea, 0xb1, 0x9f, 0xfd,
			0x0b, 0x96, 0xa1, 0x4f, 0xc6, 0xdb, 0x33, 0xfb,
			0x80, 0x6c, 0x01, 0x1e, 0x6f, 0xf3, 0xd2, 0xca,
			0x72, 0xe8, 0xd0, 0x2f, 0x28, 0x11, 0x79, 0x0a,
			0x67, 0xb1, 0xff, 0xb4, 0xd8, 0xd4, 0x39, 0x08,
			0x36, 0x08, 0x42, 0xff, 0x52, 0x48, 0xdd, 0x55,
			0x13, 0x60, 0x2b, 0xa4, 0xda, 0xfa, 0x39, 0xeb,
			0xae, 0x5b, 0xbb, 0x80, 0x6e, 0xb8, 0xd0, 0xfd,
			0x1a, 0xf3, 0x83, 0x69, 0xba, 0xbf, 0x31, 0x37,
			0x79, 0x66, 0x2d, 0x82, 0x94, 0x1e, 0x5d, 0x8f,
			0xea, 0x37, 0x49, 0xe3, 0xe9, 0xbd, 0xd8, 0x89,
			0xff, 0xc1, 0x4b, 0x9c, 0x2f, 0xdf, 0xdb, 0x96,
		},
		20, /* salt length */
		/* V5 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA384 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0x12, 0x1b, 0x99, 0x41, 0x14, 0xad, 0x5b, 0xd7,
			0x01, 0x15, 0xcc, 0xbe, 0xd8, 0x2c, 0xf9, 0x25,
			0x1d, 0x2f, 0xb6, 0xda, 0x74, 0x07, 0xd8, 0xb2,
			0x6e, 0xce, 0xa5, 0x2c, 0xa3, 0x8e, 0x89, 0x51,
			0x01, 0x4c, 0x0b, 0xde, 0x63, 0x45, 0xfe, 0xa5,
			0x65, 0x77, 0xfc, 0x31, 0x77, 0x63, 0x10, 0xa9,
			0x31, 0x12, 0xb3, 0xf6, 0xe7, 0x56, 0xfa, 0x28,
			0x7c, 0x52, 0x0d, 0x16, 0x7b, 0x3c, 0xdb, 0x45,
			0x63, 0x62, 0x0c, 0x0c, 0x43, 0x6b, 0x37, 0xf5,
			0x87, 0xe0, 0x56, 0x65, 0x12, 0xeb, 0x77, 0x80,
			0x8d, 0x5e, 0xb4, 0x47, 0xfe, 0xf6, 0x64, 0x03,
			0x9c, 0xe2, 0x93, 0xe7, 0xe2, 0x7f, 0xb0, 0xf1,
			0xe2, 0x66, 0x86, 0x11, 0xdc, 0xa8, 0x6e, 0x8d,
			0x0f, 0x58, 0xc2, 0xa4, 0xcf, 0x4a, 0x94, 0x72,
			0xd8, 0x1b, 0xa0, 0x13, 0xe2, 0x71, 0x80, 0x0b,
			0x75, 0x84, 0x1f, 0xe5, 0xff, 0xde, 0x70, 0x1b,
		},
		/* S6 : 384 byte */
		{
			0x5e, 0x3e, 0x7f, 0xd4, 0x7d, 0x04, 0x95, 0xb0,
			0xd4, 0xb5, 0x4b, 0xda, 0x88, 0x2e, 0xb4, 0x6e,
			0x87, 0xa5, 0xdf, 0xcc, 0xce, 0x73, 0xa8, 0xca,
			0x21, 0xa9, 0x4c, 0x38, 0xd5, 0xfb, 0x78, 0x4d,
			0x5a, 0x37, 0x5b, 0xe8, 0xd9, 0xec, 0xfc, 0xa3,
			0x73, 0xe2, 0x69, 0x34, 0xbe, 0xcf, 0xe1, 0xaa,
			0x71, 0x3e, 0x2a, 0x49, 0x8e, 0x86, 0x64, 0x01,
			0x95, 0x34, 0xfe, 0x58, 0xa8, 0xad, 0x8b, 0x64,
			0xdc, 0x69, 0x1a, 0x2c, 0x22, 0x5b, 0x11, 0x11,
			0x25, 0x1f, 0x86, 0x0c, 0xc0, 0x35, 0x8e, 0x3c,
			0x4e, 0x67, 0x6c, 0x47, 0x82, 0x0c, 0xb4, 0xa5,
			0x15, 0xf1, 0x48, 0xf6, 0x9d, 0xcf, 0xa9, 0xc4,
			0x2a, 0xd5, 0x6b, 0xdb, 0x97, 0x5d, 0x78, 0x65,
			0x97, 0xa3, 0x75, 0x9c, 0x0f, 0xd5, 0x73, 0x03,
			0xdb, 0xfe, 0xae, 0x96, 0xca, 0x9b, 0x57, 0xa9,
			0x26, 0x73, 0x4d, 0xc3, 0x22, 0x9d, 0xf5, 0xd5,
			0x7b, 0xe1, 0xca, 0x6f, 0x9b, 0x4d, 0x8e, 0xad,
			0x91, 0x43, 0xc0, 0x72, 0x0a, 0x08, 0xc8, 0xdc,
			0xbe, 0x47, 0x7a, 0x43, 0x26, 0x72, 0xce, 0xa2,
			0x6d, 0xc0, 0xbd, 0x29, 0xde, 0x1f, 0xd8, 0xcb,
			0xb9, 0xf4, 0x23, 0xa7, 0x59, 0xa3, 0x26, 0xb1,
			0x69, 0x77, 0x77, 0x93, 0x33, 0x14, 0x67, 0x9b,
			0xe6, 0xb9, 0xd2, 0x30, 0x31, 0x1e, 0x0e, 0x0b,
			0x8b, 0xa8, 0x66, 0x1e, 0x70, 0x03, 0xc7, 0xcb,
			0xfb, 0x9e, 0x88, 0x72, 0xf9, 0xf4, 0x0a, 0x44,
			0xd3, 0x0d, 0x12, 0x83, 0x48, 0xce, 0x94, 0x80,
			0x06, 0x07, 0x5b, 0x1a, 0xe0, 0xab, 0xc3, 0xc6,
			0x2e, 0xdb, 0x32, 0x85, 0xa1, 0x2c, 0xc2, 0x1b,
			0x03, 0x0c, 0xbc, 0x99, 0xba, 0x38, 0x20, 0x19,
			0x85, 0x24, 0x1c, 0xc5, 0x05, 0x16, 0x64, 0x67,
			0x56, 0x48, 0xe5, 0x7c, 0xc5, 0x4a, 0xc5, 0x11,
			0x27, 0x15, 0x9d, 0xaa, 0xee, 0x5a, 0x1a, 0x8e,
			0x65, 0x9d, 0x87, 0xdd, 0x59, 0x80, 0xaf, 0x8a,
			0x1a, 0xa9, 0x7a, 0xdb, 0xfa, 0x50, 0xbd, 0x9b,
			0x04, 0x5f, 0x6f, 0x60, 0x5d, 0x3a, 0x1e, 0x56,
			0x94, 0x5c, 0xfc, 0x33, 0x1f, 0xff, 0x81, 0xe5,
			0x36, 0x93, 0xed, 0xd6, 0xd2, 0x0f, 0xaa, 0xcc,
			0x93, 0x03, 0x13, 0xfa, 0xb4, 0x8a, 0x27, 0x03,
			0x1f, 0xa9, 0x25, 0xce, 0x10, 0x1a, 0x96, 0x28,
			0xd7, 0xb1, 0x48, 0x44, 0xf1, 0x61, 0x84, 0xae,
			0xe0, 0x50, 0x4d, 0x07, 0x9a, 0x3f, 0x82, 0x40,
			0x2f, 0x46, 0x85, 0x9c, 0x5f, 0x24, 0xbf, 0x15,
			0x58, 0x0e, 0x62, 0xe6, 0x7b, 0xa4, 0x34, 0x9f,
			0x11, 0xf3, 0xbc, 0x26, 0x01, 0x84, 0x66, 0xbc,
			0x2e, 0x4f, 0xb8, 0x9f, 0x64, 0x0d, 0xc8, 0x82,
			0x27, 0x5f, 0xba, 0x95, 0xf5, 0x49, 0x6b, 0x1a,
			0xff, 0x5b, 0x56, 0xe0, 0x77, 0xce, 0xc4, 0x7e,
			0x31, 0x59, 0x7b, 0xb3, 0x13, 0xce, 0x0c, 0xd4,
		},
		20, /* salt length */
		/* V6 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA384 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0x7e, 0x11, 0x88, 0x7d, 0xbd, 0x16, 0x73, 0x12,
			0x40, 0x07, 0x6d, 0xfb, 0x44, 0x01, 0xb3, 0xcf,
			0x14, 0xed, 0x16, 0x6d, 0x22, 0x4a, 0xad, 0xe3,
			0x56, 0x2d, 0xc7, 0x8e, 0xa6, 0x85, 0x23, 0x83,
			0xe7, 0xa4, 0x64, 0x2e, 0xe2, 0xfb, 0xbf, 0x39,
			0xab, 0x25, 0xf7, 0xaf, 0x2e, 0x28, 0x8b, 0x62,
			0x0b, 0xf7, 0x2e, 0x8a, 0xa4, 0x3b, 0x77, 0xae,
			0x73, 0x52, 0xda, 0xc8, 0xa2, 0x98, 0xd7, 0xa4,
			0xd5, 0xe6, 0x71, 0x89, 0x4b, 0x59, 0xe2, 0x1b,
			0x25, 0x55, 0x8d, 0x80, 0x05, 0xc9, 0x34, 0xc2,
			0xdb, 0xb4, 0x6b, 0x74, 0xfc, 0xe8, 0x4c, 0x99,
			0x75, 0x0e, 0x6d, 0xc1, 0x9e, 0xea, 0x0f, 0x75,
			0xf2, 0xe0, 0x98, 0xd5, 0x1d, 0xb7, 0x44, 0x80,
			0x57, 0x9e, 0x17, 0xb4, 0xf1, 0x6c, 0x16, 0xb7,
			0x2d, 0x2c, 0xb6, 0x75, 0x53, 0xd9, 0x95, 0x6a,
			0x01, 0x66, 0x68, 0xc2, 0xc3, 0x01, 0x34, 0xbf,
		},
		/* S7 : 384 byte */
		{
			0x7f, 0xc0, 0x82, 0x93, 0x87, 0x06, 0x53, 0x9b,
			0x7e, 0x99, 0x88, 0x74, 0xf9, 0xb1, 0xda, 0xeb,
			0x6a, 0xbd, 0x62, 0xc2, 0xdd, 0xc9, 0x48, 0xd2,
			0xff, 0x7b, 0xa2, 0x19, 0xb2, 0xb1, 0x23, 0x35,
			0x15, 0xb6, 0x51, 0x3f, 0x49, 0x03, 0x2c, 0x4c,
			0xad, 0xed, 0xb1, 0x95, 0x83, 0xd6, 0xa0, 0x69,
			0x36, 0x8b, 0x0c, 0xd6, 0x34, 0x93, 0x65, 0x67,
			0x1a, 0xeb, 0x41, 0xc1, 0x27, 0x30, 0x4b, 0x41,
			0x7a, 0x84, 0xa3, 0x8f, 0x33, 0x8c, 0x59, 0x94,
			0xb4, 0x5a, 0x4d, 0x03, 0x1c, 0xad, 0x05, 0x65,
			0xa9, 0xeb, 0x75, 0xa3, 0xed, 0x29, 0x79, 0x71,
			0x11, 0x1a, 0x5b, 0x84, 0x81, 0x39, 0x76, 0x67,
			0x94, 0xcf, 0x22, 0x9b, 0xf3, 0xda, 0x15, 0x2c,
			0x3d, 0xa0, 0x96, 0x6c, 0xa4, 0xf0, 0x67, 0x0c,
			0x6a, 0x42, 0xfe, 0xc9, 0x86, 0x85, 0x2e, 0x2c,
			0x15, 0xa8, 0xa2, 0x0e, 0x55, 0x16, 0x69, 0xb0,
			0x89, 0x82, 0x19, 0x06, 0x79, 0x8b, 0x6a, 0x93,
			0xaf, 0x26, 0x9a, 0xd4, 0x93, 0xfd, 0xea, 0x73,
			0x36, 0x11, 0xbb, 0xca, 0xfc, 0xa4, 0xa5, 0x20,
			0x78, 0xf9, 0x83, 0x7d, 0xa0, 0x80, 0x6a, 0x63,
			0xf0, 0x74, 0x56, 0xd1, 0x65, 0x03, 0x3b, 0xe3,
			0x7e, 0xfd, 0xc5, 0x21, 0x5e, 0xe9, 0x78, 0x14,
			0xf9, 0xca, 0xf2, 0x17, 0x73, 0x3d, 0x10, 0x7d,
			0x67, 0xd5, 0xe4, 0x36, 0x49, 0xcc, 0x74, 0x92,
			0x5d, 0x02, 0x2b, 0xdf, 0x3e, 0x03, 0x86, 0x82,
			0x13, 0x4c, 0xcd, 0xe5, 0x4d, 0xdc, 0xc5, 0x76,
			0x1b, 0xb9, 0xb8, 0x22, 0x5c, 0x02, 0xea, 0x1c,
			0xd8, 0xa2, 0x5d, 0xde, 0x40, 0xb0, 0x33, 0x1d,
			0xcc, 0xb5, 0xf3, 0x9c, 0x12, 0xd3, 0x19, 0xaf,
			0xdd, 0x24, 0x7a, 0xef, 0xbf, 0xf9, 0x70, 0xdc,
			0x2d, 0x2f, 0xe5, 0x07, 0x7b, 0x74, 0xc1, 0xd8,
			0xa5, 0x2e, 0x89, 0xba, 0x54, 0xbe, 0x60, 0xd4,
			0xeb, 0x83, 0xef, 0x11, 0xfa, 0xe0, 0x31, 0x93,
			0xa7, 0xab, 0x73, 0x0f, 0x8b, 0xc8, 0x85, 0xfa,
			0x77, 0x9f, 0x1e, 0x0b, 0xaa, 0x12, 0x0a, 0xeb,
			0x42, 0x1c, 0x9a, 0x65, 0x00, 0x10, 0x7d, 0xb1,
			0x9d, 0xb7, 0x4f, 0x40, 0x06, 0x4f, 0xf2, 0x6b,
			0xec, 0x7e, 0x66, 0x78, 0xc3, 0x3a, 0x36, 0x60,
			0xc5, 0x64, 0x61, 0x34, 0x0d, 0x32, 0x76, 0xac,
			0xd7, 0x6e, 0x28, 0x04, 0x78, 0x20, 0xdb, 0x6d,
			0xd0, 0xf4, 0x00, 0xed, 0xe2, 0x15, 0x75, 0xb3,
			0x56, 0x9c, 0xcd, 0xfb, 0x8d, 0x20, 0xdd, 0x94,
			0xd3, 0x58, 0xb4, 0x7d, 0xa7, 0xe1, 0xa4, 0xb7,
			0x35, 0x0f, 0xfc, 0x5c, 0xaf, 0xb1, 0xe3, 0x9f,
			0x22, 0x79, 0xe6, 0x44, 0xc2, 0x85, 0xc7, 0x59,
			0x22, 0x3b, 0xff, 0xb1, 0x4c, 0x33, 0x9a, 0x9c,
			0xe0, 0x46, 0x50, 0xe4, 0x26, 0xc6, 0xd7, 0xa5,
			0x8a, 0x22, 0x7f, 0x15, 0x12, 0xb2, 0x21, 0xd7,
		},
		20, /* salt length */
		/* V7 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA384 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0x4a, 0x75, 0x70, 0x3e, 0x08, 0xbf, 0x48, 0xd0,
			0x3e, 0x93, 0xa3, 0x77, 0x60, 0x2c, 0x22, 0x5f,
			0xb5, 0x3d, 0xf7, 0x03, 0xa3, 0x2b, 0x4a, 0xfb,
			0x2d, 0x79, 0xdb, 0x14, 0xba, 0x84, 0xe8, 0xa6,
			0x18, 0x70, 0x51, 0x22, 0xc0, 0x62, 0xfa, 0xeb,
			0x5c, 0xef, 0x71, 0x67, 0x86, 0xd5, 0xd1, 0x55,
			0xd3, 0xe9, 0xe6, 0x61, 0x10, 0x51, 0x91, 0xf5,
			0xcb, 0xf3, 0x6e, 0x53, 0x16, 0x2d, 0xb5, 0xe1,
			0x87, 0xc7, 0x5e, 0xc3, 0x83, 0x26, 0x89, 0xaf,
			0xbc, 0xc2, 0x1c, 0xfd, 0xf9, 0xda, 0xae, 0xfe,
			0x16, 0x66, 0xa4, 0x8f, 0x4b, 0x4d, 0x99, 0xee,
			0xb1, 0xd8, 0x04, 0xcc, 0x01, 0x66, 0x14, 0xa3,
			0x9f, 0x62, 0xdf, 0x49, 0xd8, 0x73, 0xbc, 0xc6,
			0x68, 0xf2, 0xed, 0x63, 0x40, 0xa1, 0x82, 0x03,
			0x86, 0x9a, 0x79, 0xb6, 0xdc, 0x85, 0x2a, 0x8c,
			0xe5, 0x0e, 0xe3, 0xa7, 0x05, 0x97, 0xcf, 0x94,
		},
		/* S8 : 384 byte */
		{
			0x05, 0xb6, 0x89, 0x6c, 0x8c, 0x0d, 0x0a, 0x86,
			0x66, 0x71, 0x52, 0xf4, 0x61, 0x35, 0x3d, 0x20,
			0xce, 0x3b, 0xae, 0xce, 0xa9, 0x03, 0x7e, 0x37,
			0xa9, 0x1d, 0x2b, 0x49, 0xee, 0x26, 0xda, 0x9a,
			0x47, 0xff, 0x63, 0x47, 0x3f, 0x19, 0x71, 0xc2,
			0xa0, 0x92, 0xc0, 0xe3, 0x6f, 0xb4, 0x78, 0x10,
			0xcf, 0x89, 0xae, 0xab, 0xaa, 0xda, 0xe6, 0x5f,
			0xa4, 0xa6, 0x30, 0xaa, 0x7b, 0xe2, 0xf6, 0xd6,
			0xad, 0x9d, 0x81, 0xae, 0x1f, 0x5d, 0x67, 0x6b,
			0x25, 0xb4, 0xdb, 0xe7, 0xaf, 0x6e, 0xb2, 0xf1,
			0x72, 0x18, 0x31, 0x75, 0xd6, 0x99, 0x9e, 0x04,
			0xff, 0xad, 0x0f, 0x7a, 0xfa, 0x41, 0x08, 0x42,
			0x2c, 0xbb, 0xa5, 0x57, 0x37, 0x44, 0x3c, 0x99,
			0x60, 0x5b, 0x06, 0x2a, 0x26, 0x5d, 0x97, 0xd1,
			0xa1, 0xdf, 0x7e, 0xf4, 0x56, 0xc0, 0xd6, 0x43,
			0xc3, 0x43, 0xbf, 0x49, 0x3c, 0x2c, 0x81, 0x3d,
			0x9c, 0x2b, 0xa4, 0x58, 0xf8, 0x15, 0xd9, 0xc5,
			0x8f, 0xba, 0xa9, 0x5e, 0x2e, 0x18, 0x3a, 0x16,
			0x14, 0x99, 0xb9, 0xec, 0x59, 0x6f, 0x68, 0x71,
			0xc5, 0x1e, 0xcd, 0x62, 0xe6, 0x1c, 0x6f, 0xa3,
			0xbf, 0x76, 0x12, 0x9a, 0xe9, 0x34, 0x10, 0x99,
			0xe5, 0x92, 0xa1, 0xbe, 0x47, 0x5e, 0x7c, 0xf5,
			0x5c, 0xcc, 0x00, 0xa3, 0x6e, 0xa1, 0x13, 0x88,
			0xd0, 0xf8, 0xc4, 0xec, 0xe5, 0x4f, 0xff, 0x6a,
			0xd5, 0xe2, 0xbc, 0x00, 0x4e, 0x55, 0x35, 0xd5,
			0xa6, 0x2e, 0xae, 0x96, 0x9e, 0x84, 0x67, 0x97,
			0x2d, 0xb5, 0x19, 0xae, 0xe5, 0x00, 0xea, 0x1a,
			0xdf, 0x33, 0x48, 0xf8, 0xa4, 0x3a, 0x05, 0xed,
			0x63, 0xc0, 0xa2, 0x02, 0x31, 0xab, 0xc6, 0xfe,
			0xb0, 0x08, 0x8d, 0x6d, 0x14, 0x8c, 0x6b, 0x2e,
			0x86, 0xa2, 0x0f, 0x94, 0xb9, 0x53, 0x4f, 0x8b,
			0x0f, 0x35, 0x29, 0x5a, 0xac, 0xe5, 0x0c, 0xf7,
			0xe8, 0x7d, 0x6f, 0x72, 0x32, 0xf6, 0x41, 0x6e,
			0xc1, 0xd2, 0x80, 0x48, 0xb8, 0x90, 0x10, 0x68,
			0xdd, 0xd0, 0x1d, 0xdd, 0xa8, 0x39, 0xb7, 0x63,
			0xb7, 0x5a, 0x5d, 0x83, 0x93, 0xb6, 0x48, 0xe3,
			0xab, 0x73, 0x66, 0xb0, 0x6e, 0x65, 0x00, 0x39,
			0x6b, 0x11, 0x66, 0x38, 0xde, 0x92, 0x34, 0x3b,
			0x8b, 0x27, 0x33, 0x17, 0xaf, 0x88, 0xa8, 0xee,
			0xa2, 0x6e, 0x6a, 0xd9, 0x5b, 0x10, 0xbf, 0x7d,
			0x30, 0xab, 0x04, 0x74, 0xe9, 0x18, 0xc1, 0x29,
			0xec, 0x12, 0x42, 0x84, 0xdc, 0x23, 0x74, 0xe2,
			0x09, 0x30, 0x01, 0x64, 0x2d, 0xdf, 0xb7, 0x40,
			0x25, 0x86, 0xac, 0x98, 0xc9, 0xa3, 0x25, 0xc4,
			0x50, 0xd1, 0x47, 0xe7, 0x61, 0x64, 0xf9, 0x46,
			0xeb, 0xf3, 0xd3, 0x87, 0xa4, 0xfd, 0x3c, 0xb1,
			0x66, 0x5c, 0xdc, 0xcd, 0xf8, 0xf0, 0x02, 0x91,
			0x05, 0x9b, 0xe3, 0xd0, 0xfc, 0xb4, 0x03, 0x9b,
		},
		20, /* salt length */
		/* V8 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA384 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0x90, 0x6f, 0x3b, 0x4e, 0x9d, 0x4c, 0xe0, 0xef,
			0x50, 0xb0, 0x77, 0x1c, 0x31, 0x18, 0x5c, 0x54,
			0x10, 0x25, 0x7d, 0x5e, 0xe9, 0xee, 0x5f, 0x3c,
			0x0b, 0x9b, 0x62, 0x5b, 0x79, 0x30, 0x45, 0x0c,
			0x91, 0xa6, 0xec, 0x38, 0xeb, 0x40, 0x61, 0x4e,
			0x84, 0x1c, 0x1a, 0x81, 0x1c, 0x09, 0x7a, 0x31,
			0xdc, 0x98, 0x99, 0xa6, 0x3c, 0x76, 0x48, 0x04,
			0x24, 0xf8, 0x50, 0x7b, 0x51, 0xf8, 0xfb, 0xbf,
			0xfb, 0xe0, 0xd6, 0xb8, 0xb5, 0x04, 0xb4, 0x7a,
			0x94, 0x35, 0xf6, 0x77, 0x93, 0x92, 0x0a, 0xf3,
			0x36, 0xa2, 0x32, 0xb9, 0x6e, 0xa0, 0x87, 0x0d,
			0xbd, 0x16, 0x3d, 0xe1, 0x9a, 0xae, 0x28, 0x4c,
			0x17, 0x44, 0x7f, 0x37, 0xc1, 0x3f, 0xe1, 0x5a,
			0xc8, 0x22, 0x95, 0x67, 0x60, 0x5e, 0x16, 0x84,
			0x66, 0x6b, 0xcf, 0xff, 0x47, 0x16, 0x23, 0x3d,
			0x1e, 0x1d, 0xb0, 0xf4, 0x1c, 0xf9, 0xd5, 0x9a,
		},
		/* S9 : 384 byte */
		{
			0x5e, 0x15, 0x88, 0x7c, 0x68, 0x10, 0xa9, 0x71,
			0x1b, 0x1d, 0x4b, 0xb5, 0x66, 0xd4, 0xa5, 0x0e,
			0x05, 0x7c, 0xbd, 0xb8, 0x2f, 0x33, 0x71, 0xcb,
			0xc3, 0x2c, 0xfc, 0x98, 0xfa, 0x00, 0x44, 0xbe,
			0xe9, 0x56, 0x5b, 0xa7, 0x3c, 0xbc, 0x36, 0xcf,
			0xd9, 0x1f, 0x12, 0xed, 0x72, 0x87, 0x14, 0x68,
			0x04, 0xd0, 0x68, 0x8e, 0xb5, 0x30, 0x0a, 0x68,
			0xe3, 0xfd, 0x66, 0x1c, 0x68, 0xec, 0xa1, 0x98,
			0x6b, 0xab, 0xd7, 0x82, 0x0a, 0x4c, 0xe5, 0x61,
			0x72, 0x78, 0x1d, 0x67, 0x74, 0xf1, 0x63, 0xf6,
			0xc6, 0x2f, 0x23, 0xc8, 0x9e, 0xf0, 0x93, 0xc1,
			0xff, 0x67, 0xf7, 0x18, 0x1c, 0xe5, 0x9e, 0x00,
			0x63, 0x19, 0xd0, 0xd1, 0x5b, 0x57, 0x04, 0xe2,
			0x6e, 0xe1, 0x16, 0x52, 0x22, 0x88, 0x0c, 0x26,
			0x52, 0x1d, 0xcd, 0xe3, 0x29, 0x1a, 0x6d, 0x05,
			0x8f, 0x28, 0x3c, 0xd1, 0x14, 0x43, 0x74, 0x74,
			0x41, 0xce, 0x8f, 0x39, 0xdc, 0x8e, 0xf6, 0x62,
			0xf6, 0xb4, 0x0d, 0x64, 0x19, 0x52, 0xc2, 0x55,
			0xad, 0xdc, 0x8e, 0x67, 0x56, 0x1c, 0x03, 0xb1,
			0x7b, 0xc9, 0xa7, 0x4a, 0x2b, 0xcd, 0xbc, 0x30,
			0x85, 0x01, 0x0a, 0xe8, 0x7a, 0x77, 0x7e, 0x01,
			0x44, 0x67, 0x99, 0xa0, 0x85, 0x6f, 0xe5, 0xb5,
			0x8e, 0x63, 0xd2, 0x7f, 0x6f, 0xfe, 0x65, 0x7c,
			0xd9, 0xb0, 0x91, 0xe0, 0x56, 0x1a, 0x95, 0xa5,
			0x2a, 0x2b, 0x5e, 0x99, 0x81, 0xaa, 0x8b, 0xa3,
			0x67, 0x0c, 0x2c, 0x44, 0x69, 0x96, 0x44, 0xae,
			0xb7, 0x7b, 0x81, 0x93, 0x60, 0xb7, 0x67, 0x64,
			0x7d, 0x46, 0xbd, 0x39, 0xfe, 0x72, 0xd3, 0x90,
			0xd0, 0xee, 0xca, 0xb3, 0x89, 0x61, 0xad, 0x67,
			0x8a, 0xcb, 0xe2, 0x16, 0x09, 0x06, 0xfc, 0x9f,
			0x07, 0x2a, 0xd9, 0x69, 0x5c, 0xb0, 0xaf, 0x85,
			0x4d, 0x03, 0xe1, 0x7f, 0xc1, 0x99, 0x9b, 0x38,
			0xd5, 0xe7, 0x08, 0xe3, 0x04, 0x20, 0x6e, 0xa4,
			0xe7, 0xa5, 0x3b, 0xc7, 0xf1, 0xaa, 0x23, 0x8e,
			0xd2, 0xa9, 0x89, 0x8d, 0x74, 0xd3, 0x5a, 0x64,
			0x4e, 0x92, 0xf4, 0x86, 0xcb, 0x0f, 0xef, 0x8a,
			0xe7, 0x46, 0x82, 0xa8, 0x06, 0xbb, 0x86, 0xa3,
			0x84, 0x3d, 0x04, 0x51, 0xbb, 0x56, 0x5d, 0x29,
			0x9e, 0x05, 0x3b, 0xd4, 0x58, 0xb8, 0xf6, 0x76,
			0x44, 0xea, 0x63, 0x00, 0xe8, 0xcf, 0x0e, 0x4e,
			0xb1, 0xa3, 0xde, 0xd0, 0xbe, 0x00, 0xfe, 0x34,
			0x0d, 0x40, 0xfd, 0x11, 0x57, 0x30, 0xa4, 0xb5,
			0x4f, 0xaf, 0x56, 0x55, 0x96, 0x9d, 0x74, 0x4d,
			0x5b, 0x4f, 0xfb, 0xf5, 0xaf, 0xcd, 0x44, 0xa5,
			0x4f, 0xa8, 0x72, 0x2e, 0xd0, 0xc1, 0xb9, 0x0b,
			0xcb, 0x48, 0xc4, 0xa7, 0xc4, 0xfe, 0xa2, 0xa5,
			0x4d, 0xf9, 0x96, 0xf2, 0x8c, 0x44, 0x49, 0x64,
			0xa2, 0xcc, 0xd3, 0x23, 0xf2, 0x8f, 0x21, 0x8b,
		},
		20, /* salt length */
		/* V9 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
};

/* 10 elements */
static mod3072_t SHA512[] = {
	/* SHA512 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0xe9, 0x90, 0xc8, 0x83, 0x5f, 0x18, 0xb1, 0x85,
			0x62, 0x32, 0x3b, 0xa5, 0x09, 0x6a, 0x4e, 0x7b,
			0x99, 0xbd, 0x84, 0x89, 0x9e, 0x5c, 0xdd, 0x1f,
			0x3b, 0xad, 0xb4, 0x7c, 0xbf, 0x93, 0xf1, 0x36,
			0x78, 0xef, 0x81, 0xdc, 0xcc, 0x67, 0x03, 0xd9,
			0x85, 0x66, 0xc4, 0x9b, 0x6d, 0x63, 0xee, 0xf5,
			0x1b, 0x67, 0xfc, 0xc2, 0x0c, 0xc9, 0x71, 0xcc,
			0xf6, 0x3c, 0xca, 0xec, 0x58, 0x0d, 0xb1, 0x72,
			0x56, 0xa5, 0x73, 0xc6, 0xc4, 0x55, 0xb4, 0x50,
			0x81, 0x53, 0x62, 0x96, 0x06, 0xff, 0xe7, 0xa4,
			0x3e, 0x6b, 0xa3, 0xb1, 0x99, 0x1b, 0x99, 0xff,
			0x5c, 0x09, 0x68, 0x03, 0x3b, 0xec, 0x7e, 0xc6,
			0x29, 0xba, 0x88, 0x8b, 0x6f, 0x6c, 0x2c, 0xb2,
			0xfb, 0x01, 0xfb, 0xdc, 0xfb, 0xc5, 0xa1, 0x50,
			0xab, 0xd3, 0x5f, 0x9e, 0x6b, 0xd9, 0xbc, 0x82,
			0x15, 0x1b, 0x77, 0x0a, 0x8d, 0xbb, 0xbf, 0xfb,
		},
		/* S0 : 384 byte */
		{
			0x60, 0x7b, 0x77, 0x31, 0xec, 0xb2, 0x32, 0xf9,
			0xb8, 0xe9, 0xea, 0x03, 0xbe, 0x28, 0xcc, 0x1e,
			0x94, 0x8a, 0xcc, 0x3e, 0xc1, 0x2a, 0x12, 0x22,
			0xba, 0x0f, 0x63, 0x93, 0x54, 0x40, 0xc3, 0xef,
			0xfe, 0xaf, 0x46, 0x0d, 0x70, 0x66, 0xd2, 0x60,
			0xd1, 0x74, 0xd0, 0xed, 0x18, 0xa9, 0x19, 0x35,
			0x50, 0x00, 0x0c, 0x2f, 0xa0, 0x11, 0x97, 0x12,
			0xfb, 0x1a, 0xb1, 0xe2, 0x7b, 0x4e, 0x6f, 0x5f,
			0x84, 0xbe, 0x9b, 0x63, 0xa1, 0xed, 0xe1, 0x7a,
			0x01, 0x17, 0x40, 0x60, 0xe2, 0xd9, 0xe4, 0x61,
			0x21, 0xcc, 0x5d, 0x10, 0x51, 0x5a, 0x34, 0x2a,
			0x26, 0x64, 0x95, 0x39, 0x34, 0x1e, 0xb1, 0xb4,
			0x4b, 0x82, 0xe3, 0x46, 0xa0, 0x10, 0x2e, 0x7c,
			0xa4, 0x5b, 0xe3, 0x14, 0x9b, 0x5f, 0x14, 0x44,
			0xbd, 0x7f, 0xdf, 0x43, 0xda, 0x44, 0x1c, 0x59,
			0xde, 0xb3, 0x7d, 0xa9, 0xa2, 0x23, 0xbc, 0xd7,
			0xa8, 0x24, 0x42, 0x37, 0xbb, 0x54, 0x04, 0xea,
			0x53, 0x2e, 0xb4, 0x70, 0xe8, 0x08, 0x91, 0xc0,
			0xfe, 0x94, 0x03, 0xd1, 0x27, 0x34, 0x10, 0x02,
			0x84, 0xe9, 0x9c, 0xfd, 0x96, 0xde, 0x2a, 0xb4,
			0x05, 0x85, 0x29, 0xd9, 0x1b, 0xf3, 0x48, 0xc6,
			0xcb, 0xdb, 0x7f, 0xcf, 0xee, 0xa3, 0xf9, 0x92,
			0x5e, 0x93, 0xef, 0xd6, 0xad, 0xb3, 0xef, 0x69,
			0x46, 0x00, 0x87, 0x38, 0xf4, 0x57, 0x7a, 0x49,
			0xc4, 0x2a, 0xc0, 0x20, 0x3a, 0x2d, 0x98, 0x2f,
			0xd7, 0x7c, 0xb4, 0x21, 0xae, 0x03, 0x0b, 0x81,
			0xb9, 0x7d, 0xd0, 0x44, 0x90, 0x60, 0x51, 0x79,
			0x62, 0x69, 0x03, 0x47, 0x1c, 0xf6, 0x88, 0x35,
			0xdd, 0x5e, 0x4a, 0xc4, 0x1a, 0xcf, 0xe5, 0x4e,
			0x04, 0x88, 0x78, 0xdf, 0x89, 0xdb, 0x9c, 0x2d,
			0xe5, 0xf1, 0xe8, 0x22, 0x26, 0x6c, 0x32, 0x5e,
			0x0b, 0xe0, 0x99, 0x1c, 0x7f, 0x18, 0xcd, 0x3d,
			0xe4, 0xb2, 0x11, 0x0e, 0x14, 0xf5, 0x61, 0x00,
			0xe4, 0x5f, 0x8b, 0xa1, 0x9e, 0xdf, 0x91, 0x71,
			0x50, 0xc2, 0x07, 0x4f, 0x37, 0x92, 0x93, 0xf7,
			0x3c, 0xb5, 0x87, 0xff, 0x77, 0xad, 0x63, 0xe4,
			0xcb, 0xec, 0x9e, 0xea, 0xed, 0x77, 0xca, 0x90,
			0x26, 0x1b, 0x28, 0x13, 0xae, 0x8e, 0x65, 0x33,
			0xb0, 0x9b, 0x22, 0x3a, 0x68, 0xab, 0xe2, 0xbe,
			0xee, 0xc8, 0x88, 0x08, 0x8f, 0xf9, 0x1f, 0xea,
			0x5c, 0x63, 0xde, 0x3b, 0x55, 0x23, 0x8a, 0xef,
			0x01, 0x8c, 0x36, 0x8f, 0x98, 0x65, 0x15, 0x72,
			0xbc, 0x7b, 0x8c, 0xf3, 0xd1, 0x4c, 0x15, 0xb2,
			0x4b, 0xb5, 0x53, 0x4a, 0xe0, 0x7a, 0x6c, 0x4c,
			0x9d, 0x5e, 0xcd, 0x0b, 0x86, 0x96, 0x1b, 0x55,
			0x08, 0x59, 0x03, 0x6b, 0xa6, 0xfa, 0x8e, 0x50,
			0xd0, 0x62, 0x28, 0xd8, 0x9b, 0xcc, 0x94, 0x35,
			0x81, 0xb2, 0x6e, 0x30, 0x27, 0x95, 0xd1, 0xe3,
		},
		20, /* salt length */
		/* V0 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA512 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0x31, 0x91, 0x2f, 0x49, 0x8e, 0xb7, 0x46, 0xa9,
			0xbc, 0x7e, 0x38, 0x36, 0x2f, 0x1b, 0xbd, 0xd7,
			0xfc, 0xe2, 0x25, 0xac, 0x98, 0xfe, 0x3e, 0x59,
			0xe2, 0x64, 0x78, 0xa2, 0x64, 0x0a, 0x47, 0x39,
			0xd6, 0x53, 0xc9, 0xd3, 0xe4, 0xce, 0xa9, 0x81,
			0x32, 0x21, 0x43, 0x3e, 0x9c, 0x40, 0x08, 0xc4,
			0xc2, 0xe3, 0x24, 0xa5, 0xc4, 0x36, 0x5f, 0x96,
			0x4a, 0xd2, 0x41, 0x0e, 0xc8, 0xec, 0x7a, 0x10,
			0x00, 0xcc, 0x81, 0x44, 0xc8, 0x81, 0x7a, 0x4d,
			0x0d, 0xd4, 0x5d, 0x57, 0x18, 0x7a, 0x55, 0xef,
			0x10, 0x8e, 0x66, 0x18, 0xc0, 0x3e, 0x53, 0x9b,
			0x9b, 0xc4, 0xbf, 0x9e, 0xfd, 0x98, 0x92, 0x78,
			0xb4, 0xe3, 0x9e, 0xc4, 0xbf, 0x37, 0x83, 0xb4,
			0x55, 0x7e, 0xcf, 0xa9, 0x2a, 0x31, 0x6a, 0x00,
			0xed, 0xcb, 0x32, 0x2e, 0x15, 0x7c, 0xe2, 0x0f,
			0xdd, 0x21, 0x38, 0x7b, 0x8d, 0x74, 0x57, 0x1b,
		},
		/* S1 : 384 byte */
		{
			0x61, 0xe8, 0xb4, 0x13, 0xf7, 0xf7, 0x31, 0xfd,
			0x44, 0xc1, 0x45, 0xd6, 0x1f, 0xad, 0xf4, 0x9d,
			0xe1, 0xcb, 0x14, 0x67, 0xca, 0x6d, 0x81, 0xfb,
			0xc9, 0xeb, 0xd5, 0xb7, 0x3f, 0xae, 0xbe, 0x42,
			0x32, 0xa8, 0x9c, 0x59, 0x76, 0x96, 0x53, 0x7d,
			0xe1, 0x64, 0xa3, 0x8d, 0x59, 0x73, 0x62, 0xd6,
			0x8e, 0x19, 0xd4, 0x39, 0xa5, 0x71, 0x6e, 0xab,
			0xc1, 0xc2, 0xfd, 0x6b, 0x2d, 0x22, 0x4f, 0xfb,
			0x63, 0xfd, 0xac, 0x8c, 0x68, 0x92, 0x75, 0xed,
			0x60, 0x4a, 0x12, 0xe9, 0x61, 0x26, 0xd8, 0x5e,
			0x71, 0x72, 0xff, 0xc4, 0x09, 0x0e, 0x45, 0x11,
			0x8c, 0x5a, 0xf9, 0xa8, 0x37, 0xad, 0x66, 0xe0,
			0x76, 0x2d, 0x42, 0xcd, 0x7c, 0x53, 0xed, 0x90,
			0x94, 0xa0, 0x9e, 0xb1, 0x96, 0x3b, 0xaf, 0x64,
			0x64, 0xc1, 0xac, 0x9a, 0xcc, 0xab, 0xf1, 0xae,
			0x58, 0x18, 0x17, 0xf6, 0x59, 0x16, 0x9a, 0xc5,
			0x39, 0xfd, 0x0b, 0x51, 0x09, 0x80, 0x7a, 0x48,
			0xe8, 0xad, 0xc3, 0xc8, 0xe3, 0xf8, 0x91, 0x4d,
			0x19, 0x90, 0x50, 0x1c, 0xd5, 0x20, 0x0c, 0x4c,
			0xac, 0x4b, 0xbc, 0x02, 0xe2, 0x3f, 0x9b, 0x84,
			0xdb, 0xc7, 0x03, 0x0a, 0x11, 0xdb, 0x51, 0x09,
			0x6a, 0xeb, 0xbd, 0x8f, 0x3e, 0xf8, 0x52, 0xca,
			0x77, 0x9a, 0x7f, 0x91, 0x22, 0x18, 0x31, 0x27,
			0x2c, 0xbb, 0x78, 0xd9, 0x95, 0x5d, 0xe5, 0xc6,
			0x4d, 0x66, 0x58, 0x3b, 0xde, 0x19, 0xae, 0x2b,
			0x04, 0xaa, 0x50, 0xb2, 0xa7, 0xeb, 0xec, 0xb5,
			0x1a, 0xba, 0x1f, 0x6e, 0x74, 0x0d, 0x94, 0xef,
			0x0d, 0x7f, 0x01, 0x08, 0xaa, 0xaf, 0xec, 0x40,
			0xdd, 0x31, 0xc6, 0xec, 0x84, 0xc0, 0xc5, 0x5d,
			0xfd, 0x37, 0x2d, 0xea, 0x5c, 0x57, 0xfd, 0x50,
			0x22, 0x92, 0x07, 0xfa, 0xf6, 0x11, 0x70, 0xda,
			0x63, 0x70, 0x09, 0xf2, 0xfd, 0xd8, 0x91, 0x83,
			0xaf, 0x61, 0xd9, 0x0e, 0x13, 0x99, 0xd6, 0xae,
			0x70, 0x43, 0xd2, 0x80, 0x78, 0x68, 0xee, 0x56,
			0xa4, 0xd9, 0x17, 0x92, 0x5e, 0x20, 0x98, 0x42,
			0x0c, 0x2f, 0x4b, 0xda, 0xd3, 0x6b, 0xb9, 0x25,
			0xc3, 0xc7, 0xaa, 0x21, 0xff, 0xb8, 0x7d, 0xcd,
			0x2c, 0xc3, 0xd5, 0x8f, 0xc0, 0x5b, 0xf7, 0x4b,
			0x75, 0x69, 0xea, 0xb0, 0x3a, 0x17, 0x9b, 0x19,
			0xb5, 0xfc, 0xc7, 0x99, 0x18, 0x81, 0xec, 0x5e,
			0xcd, 0xba, 0x31, 0x70, 0xb9, 0x45, 0xa6, 0x92,
			0x7a, 0xfd, 0x7c, 0x4d, 0x2e, 0x52, 0xdb, 0x7a,
			0x1d, 0x71, 0x1b, 0x5c, 0xdf, 0xab, 0x39, 0xed,
			0xe5, 0xda, 0x14, 0x39, 0x8c, 0x25, 0x78, 0x97,
			0xd5, 0xf6, 0xde, 0x44, 0x93, 0x90, 0xea, 0xbc,
			0x9d, 0x97, 0x4c, 0x44, 0xf9, 0xc7, 0xf1, 0x19,
			0x34, 0x9e, 0x2c, 0x61, 0x54, 0x6c, 0x75, 0x27,
			0x4a, 0x2f, 0xc8, 0xdc, 0x55, 0xed, 0x98, 0xe4,
		},
		20, /* salt length */
		/* V1 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA512 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x11, 0x28, 0x1a, 0x53, 0x70, 0xf2, 0x4f, 0x3b,
			0x5a, 0x84, 0xe5, 0xd9, 0xa6, 0x6e, 0xf9, 0x5e,
			0x0e, 0x92, 0xbc, 0xce, 0xd6, 0x33, 0x2f, 0xb4,
			0xaf, 0x4f, 0x2a, 0x13, 0xda, 0x2f, 0xa3, 0x9f,
			0x90, 0x3c, 0x2d, 0x2d, 0xa2, 0x14, 0xbf, 0x69,
			0x5d, 0xdd, 0xb7, 0xc7, 0xb0, 0xda, 0x1e, 0xbc,
			0x91, 0x3b, 0x2e, 0x2d, 0x9c, 0x3a, 0x11, 0x28,
			0xfd, 0x63, 0xec, 0xd2, 0x91, 0x33, 0x1c, 0xa0,
			0x48, 0xe5, 0xc0, 0xca, 0xce, 0x89, 0x77, 0xfd,
			0x6a, 0x56, 0x6b, 0x51, 0x26, 0x41, 0x8e, 0x7a,
			0x30, 0xcf, 0x23, 0x42, 0x80, 0x27, 0x81, 0x37,
			0xbb, 0x6b, 0x13, 0xa0, 0x57, 0x48, 0xa4, 0xb6,
			0x01, 0xcc, 0xd9, 0x11, 0x3b, 0x78, 0xff, 0x61,
			0x4a, 0x97, 0x07, 0x5e, 0x4f, 0xce, 0xcf, 0x02,
			0xb4, 0xf3, 0xa4, 0xf5, 0xc9, 0x22, 0x04, 0x49,
			0xaf, 0x0e, 0xa8, 0x67, 0x76, 0x96, 0xc2, 0x49,
		},
		/* S2 : 384 byte */
		{
			0x16, 0xa6, 0xf2, 0xfe, 0xed, 0x28, 0x42, 0xc1,
			0xee, 0xa4, 0x34, 0x70, 0xa5, 0xd7, 0x98, 0x92,
			0x60, 0x01, 0x03, 0x35, 0x6d, 0x04, 0x9c, 0xa0,
			0x93, 0x75, 0xa5, 0xe8, 0xf2, 0xc4, 0x75, 0x32,
			0x06, 0xaf, 0x5c, 0x94, 0xe4, 0x36, 0x7f, 0x5f,
			0x33, 0x1f, 0xb7, 0xef, 0x6c, 0x65, 0x43, 0x40,
			0x1f, 0x54, 0x86, 0x3e, 0x3e, 0xe1, 0x84, 0xd6,
			0xa5, 0x32, 0x08, 0xc1, 0xca, 0x53, 0x22, 0xc4,
			0xd0, 0xf3, 0xfe, 0x00, 0x1e, 0xfe, 0xa4, 0x89,
			0xb4, 0xb3, 0x56, 0x74, 0x43, 0x2b, 0x3b, 0xf9,
			0xab, 0xfe, 0x31, 0x89, 0xf9, 0xef, 0xe3, 0xb2,
			0xb8, 0x11, 0x2b, 0x55, 0x8b, 0xc1, 0xae, 0x96,
			0x30, 0x0a, 0xe6, 0xa9, 0x43, 0x4d, 0x6f, 0x24,
			0x02, 0x26, 0x66, 0x1f, 0xe6, 0x0c, 0xa6, 0x02,
			0x4d, 0xf7, 0x3b, 0x98, 0x9f, 0xd4, 0xa1, 0x15,
			0x88, 0x21, 0x89, 0x93, 0x9c, 0xab, 0x1f, 0x53,
			0x2e, 0xe9, 0xb6, 0x82, 0x3a, 0xa7, 0xba, 0x21,
			0xeb, 0x61, 0x60, 0x33, 0x1d, 0xdf, 0x19, 0x6a,
			0xad, 0xeb, 0xd2, 0xc7, 0x6b, 0x34, 0xdf, 0x96,
			0xcd, 0x52, 0x53, 0x46, 0x81, 0xdf, 0x1b, 0x86,
			0x73, 0xf7, 0xaa, 0xfd, 0x66, 0xa3, 0x0b, 0x76,
			0x44, 0xed, 0xdf, 0x4f, 0xd8, 0x0a, 0x2f, 0x36,
			0xce, 0x18, 0x70, 0xd0, 0xcb, 0x67, 0x8e, 0x9e,
			0xf7, 0x3e, 0x01, 0x2e, 0x84, 0x50, 0x9e, 0x1a,
			0x7f, 0xc9, 0x49, 0x10, 0xcd, 0xa1, 0x28, 0xe6,
			0x26, 0x7e, 0xc9, 0x74, 0x01, 0x1e, 0x31, 0x18,
			0xde, 0x55, 0xb9, 0x36, 0x37, 0xde, 0xc7, 0x3d,
			0x11, 0xaf, 0x02, 0x2b, 0xf5, 0x24, 0x41, 0x4d,
			0x7f, 0xd7, 0xd8, 0xd1, 0x5c, 0x6b, 0xff, 0xc0,
			0x26, 0xd9, 0x38, 0x66, 0x9b, 0x94, 0xcc, 0x3b,
			0x4c, 0x9b, 0x19, 0xc3, 0xda, 0x86, 0x13, 0xf7,
			0x1d, 0x01, 0xc7, 0xb0, 0xa2, 0xad, 0xf3, 0x1e,
			0x20, 0xf2, 0x72, 0xf2, 0x3e, 0x42, 0xb2, 0xaf,
			0xcc, 0x7a, 0x61, 0x42, 0xee, 0x7c, 0x22, 0xe2,
			0xc0, 0x85, 0xc8, 0x09, 0x82, 0xde, 0x38, 0x76,
			0xfb, 0x73, 0xb4, 0x46, 0xfd, 0xd8, 0xc9, 0x11,
			0x0d, 0x26, 0x29, 0xcd, 0x6c, 0xb6, 0x7b, 0xe9,
			0xaf, 0xdf, 0x35, 0xbf, 0x42, 0xe5, 0x61, 0xfe,
			0x7c, 0x8e, 0xe4, 0xb5, 0xcc, 0x6d, 0x9c, 0x0c,
			0xf4, 0x54, 0x86, 0x5b, 0xfb, 0x6c, 0xa9, 0xeb,
			0x62, 0x6d, 0xab, 0x13, 0x5c, 0x73, 0xfc, 0x62,
			0x29, 0xbf, 0x23, 0x97, 0xc1, 0x58, 0x58, 0x53,
			0x96, 0x3f, 0x4d, 0x05, 0x15, 0x64, 0xd9, 0x2d,
			0xaf, 0xb1, 0xa0, 0xfe, 0x5f, 0x28, 0xd6, 0x3a,
			0x2e, 0x2d, 0x3c, 0x83, 0x0a, 0xaa, 0xb0, 0x86,
			0xe4, 0xee, 0x2a, 0x17, 0x00, 0x55, 0x86, 0x33,
			0x3a, 0xe0, 0x36, 0x5c, 0xe1, 0xf7, 0x50, 0xaa,
			0xda, 0x16, 0xb6, 0x27, 0x56, 0x45, 0xb8, 0x48,
		},
		20, /* salt length */
		/* V2 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA512 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0x6b, 0x03, 0x02, 0x9b, 0x41, 0x6d, 0x40, 0xb1,
			0xd4, 0x79, 0x0c, 0xbd, 0x9f, 0x88, 0xd6, 0xc0,
			0xb4, 0x2f, 0x69, 0xa9, 0x53, 0x83, 0x1d, 0xbe,
			0xbb, 0x94, 0x4e, 0x3e, 0x40, 0x37, 0x0e, 0x45,
			0x9e, 0x09, 0x17, 0xef, 0x8d, 0xb0, 0x03, 0xc0,
			0x95, 0x64, 0xc0, 0x44, 0x24, 0xaf, 0x3e, 0xf2,
			0xde, 0x70, 0x6a, 0x56, 0x34, 0x26, 0x60, 0x70,
			0x38, 0x6b, 0xdc, 0x57, 0xaf, 0xbf, 0xf2, 0x21,
			0x68, 0x87, 0x00, 0x07, 0xcd, 0x55, 0xe0, 0xc5,
			0xb8, 0x6a, 0x62, 0x40, 0x88, 0x7a, 0xf9, 0x6b,
			0x73, 0xaf, 0xb4, 0x6b, 0x1e, 0xf0, 0x70, 0xfa,
			0x6a, 0x62, 0x4d, 0x09, 0x91, 0x2f, 0x5e, 0xa9,
			0x73, 0x87, 0xa5, 0x2b, 0x2a, 0xe5, 0x3e, 0x86,
			0xe4, 0x94, 0xda, 0xfa, 0xf2, 0x71, 0x6f, 0xf3,
			0x1c, 0xf2, 0x2e, 0x30, 0x38, 0x67, 0xb6, 0x24,
			0xfb, 0x80, 0x87, 0x9d, 0x0e, 0x10, 0xb9, 0xa3,
		},
		/* S3 : 384 byte */
		{
			0x99, 0x50, 0xab, 0x6b, 0x10, 0xc1, 0x24, 0x45,
			0xa5, 0xf9, 0xd6, 0xfb, 0x6b, 0xd9, 0x5a, 0x64,
			0xe6, 0xd5, 0x33, 0xe6, 0x15, 0xb6, 0xbb, 0x75,
			0xb0, 0x8a, 0x4e, 0x88, 0x1a, 0x3f, 0x8f, 0x35,
			0xe4, 0x77, 0xb4, 0xf5, 0x85, 0x2b, 0xf4, 0x12,
			0xc6, 0x78, 0x2e, 0xb9, 0x8e, 0xa3, 0x07, 0xa2,
			0x05, 0x45, 0xe9, 0x00, 0x88, 0x94, 0xa7, 0x00,
			0x67, 0xd5, 0xd2, 0xb4, 0xb1, 0x28, 0x1a, 0x73,
			0x11, 0x85, 0x49, 0xd9, 0xdf, 0x23, 0x73, 0x95,
			0x69, 0xc5, 0x1f, 0x0c, 0xa4, 0x5b, 0x8e, 0x8c,
			0xf9, 0xb7, 0xa9, 0xe2, 0xfd, 0x44, 0x94, 0xce,
			0x39, 0xcf, 0x95, 0x35, 0xca, 0x7e, 0xc3, 0x4e,
			0xe5, 0xd9, 0xc1, 0x82, 0x69, 0x24, 0x8d, 0x15,
			0x68, 0xf8, 0x4a, 0x82, 0x5b, 0xb0, 0x0a, 0x10,
			0xce, 0x88, 0x02, 0xd8, 0x3c, 0x8e, 0x65, 0x90,
			0xe3, 0xc5, 0x8c, 0x9d, 0xac, 0xb2, 0x8b, 0xc6,
			0x25, 0xd5, 0x90, 0x21, 0x1c, 0x34, 0xf0, 0xa1,
			0x3c, 0x1e, 0xc1, 0x53, 0x8b, 0xba, 0xf2, 0xa2,
			0xb0, 0x3e, 0xd7, 0x6b, 0x4f, 0x2a, 0x5f, 0x18,
			0xc1, 0x6c, 0x76, 0x3e, 0xce, 0x69, 0x68, 0x26,
			0x7b, 0x98, 0x2d, 0x0b, 0xac, 0xf8, 0xfc, 0xac,
			0x6d, 0xd4, 0x0e, 0x83, 0x1a, 0xfd, 0xdd, 0x49,
			0x25, 0x49, 0x4b, 0x15, 0xca, 0x5e, 0x36, 0x41,
			0x21, 0x7b, 0xc6, 0xee, 0x1c, 0x43, 0x37, 0x20,
			0xda, 0x77, 0xa1, 0x37, 0x0a, 0x92, 0x78, 0xd4,
			0x1c, 0xe8, 0x3e, 0xd3, 0x7f, 0xa1, 0x9e, 0xac,
			0x55, 0x1b, 0x5c, 0x77, 0xbe, 0x81, 0x90, 0x74,
			0x37, 0xab, 0x52, 0x66, 0xf1, 0x7a, 0xb9, 0x20,
			0x37, 0x58, 0x50, 0x1a, 0xfe, 0xc5, 0xbd, 0x2d,
			0x85, 0x18, 0x0e, 0xb7, 0x9f, 0xac, 0x88, 0xa5,
			0xbc, 0x58, 0x7e, 0x59, 0xc0, 0x60, 0x00, 0x66,
			0x9d, 0xe9, 0x0d, 0x67, 0x8d, 0x70, 0x21, 0xeb,
			0x59, 0x33, 0x89, 0x04, 0x91, 0xcc, 0x8e, 0x8d,
			0x23, 0xf3, 0x00, 0xb0, 0xd8, 0x54, 0x85, 0x5d,
			0x6e, 0xd7, 0xb8, 0x85, 0xe3, 0x17, 0xe6, 0xe6,
			0x26, 0x83, 0xec, 0x55, 0xad, 0x1c, 0x4c, 0x92,
			0x27, 0x43, 0x4a, 0x4d, 0x29, 0xb7, 0x5a, 0xf1,
			0x33, 0x61, 0x7c, 0x1b, 0x0e, 0xba, 0x21, 0xa7,
			0x84, 0x2a, 0x2d, 0x80, 0x83, 0x44, 0x52, 0xf1,
			0x9e, 0x9e, 0x92, 0xe0, 0x43, 0x46, 0x37, 0x86,
			0x12, 0x51, 0x38, 0x3c, 0xde, 0x9a, 0x0c, 0x0f,
			0xc2, 0x65, 0x79, 0x91, 0x3e, 0x75, 0x4b, 0x0f,
			0x87, 0xee, 0x51, 0x64, 0x5a, 0x7d, 0x4a, 0x1f,
			0xab, 0x35, 0x67, 0xe8, 0x4a, 0xbe, 0xf0, 0xb7,
			0x57, 0x4b, 0xfc, 0x68, 0x02, 0x6b, 0xf9, 0xf0,
			0x5f, 0x2d, 0xf4, 0x89, 0xf3, 0xfd, 0xc2, 0xa4,
			0xcd, 0x20, 0xe2, 0xf1, 0x30, 0x8b, 0x81, 0xba,
			0x2d, 0xfa, 0xcf, 0xe8, 0xde, 0x56, 0x1d, 0xb5,
		},
		20, /* salt length */
		/* V3 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA512 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0xd3, 0xaf, 0xf0, 0x1d, 0x77, 0xb9, 0x34, 0x6e,
			0x72, 0x3e, 0x84, 0x22, 0xaa, 0xdc, 0x15, 0x6b,
			0x18, 0xfb, 0xc1, 0x8e, 0x79, 0xb0, 0x57, 0xeb,
			0x46, 0xe2, 0x36, 0x38, 0xe7, 0xab, 0x1a, 0x0d,
			0x24, 0x00, 0x35, 0xa5, 0xdc, 0x95, 0x82, 0xe3,
			0xf9, 0xfb, 0x56, 0x93, 0xa3, 0x0b, 0x38, 0xf3,
			0xb4, 0x88, 0xf2, 0x71, 0x6c, 0x00, 0xe8, 0xb7,
			0x67, 0xf3, 0xa9, 0xe5, 0xcb, 0x38, 0xc4, 0x1d,
			0xc6, 0x9b, 0xdd, 0x45, 0x38, 0xcb, 0xff, 0x08,
			0xac, 0x74, 0x34, 0x1e, 0x8d, 0xa8, 0x53, 0xce,
			0x97, 0x83, 0x1a, 0xb1, 0x8b, 0x0f, 0x7c, 0xb8,
			0x27, 0x65, 0x3c, 0x70, 0x55, 0x15, 0xbb, 0x85,
			0xa3, 0xc8, 0x0f, 0x86, 0xf3, 0x24, 0x54, 0xe7,
			0x73, 0xf0, 0x47, 0x4c, 0xd0, 0x79, 0x11, 0x05,
			0xa7, 0x34, 0x60, 0xd4, 0x3d, 0xa4, 0xbc, 0xf7,
			0x6f, 0x7e, 0x19, 0x62, 0xec, 0x0c, 0xa8, 0x4d,
		},
		/* S4 : 384 byte */
		{
			0x1a, 0x5f, 0xd5, 0x06, 0x3e, 0x84, 0x68, 0x7c,
			0xdf, 0x77, 0xbd, 0xac, 0x36, 0x78, 0xd5, 0x46,
			0x02, 0xe2, 0x41, 0x58, 0xbe, 0xf0, 0x7a, 0x5c,
			0x2d, 0x14, 0x24, 0x9c, 0x1c, 0x94, 0x29, 0x7e,
			0x3a, 0xaa, 0x27, 0xc2, 0x39, 0x49, 0x3b, 0xd9,
			0x5e, 0xeb, 0x4c, 0x29, 0xbb, 0xea, 0xad, 0xde,
			0xf6, 0x57, 0x78, 0xef, 0x02, 0x9a, 0x6b, 0x94,
			0x6d, 0x25, 0x42, 0xea, 0x5f, 0x56, 0xb7, 0xd9,
			0x7c, 0x68, 0xc0, 0xa2, 0xc2, 0x55, 0x14, 0xca,
			0xa8, 0x2b, 0x9a, 0x1e, 0x85, 0xb3, 0xf8, 0x62,
			0x3c, 0xe5, 0x52, 0xb9, 0xf9, 0x09, 0x40, 0x24,
			0x7f, 0xc7, 0x7f, 0xbf, 0x80, 0xab, 0x8b, 0xc3,
			0x8a, 0x9b, 0x77, 0xc3, 0xf7, 0x05, 0x6b, 0x5a,
			0x25, 0x6d, 0x7b, 0x75, 0x56, 0x12, 0x09, 0x66,
			0xc3, 0xeb, 0x7e, 0xd3, 0xd6, 0xd7, 0x76, 0xfb,
			0xde, 0x61, 0x7b, 0xae, 0x0b, 0x94, 0xa2, 0xb8,
			0x22, 0x6c, 0xfa, 0x76, 0x5e, 0xd4, 0x91, 0x24,
			0x75, 0x5c, 0x5a, 0x72, 0x17, 0x7a, 0x4d, 0xc5,
			0xf1, 0x40, 0xff, 0xb1, 0x91, 0x79, 0x62, 0x49,
			0xb8, 0x68, 0x3a, 0xf4, 0x67, 0x9f, 0x3e, 0x69,
			0x7b, 0x18, 0xc5, 0x78, 0x1b, 0xc6, 0xd3, 0x9d,
			0x46, 0x0f, 0xb6, 0xff, 0x5a, 0xcc, 0x0d, 0x3e,
			0xcc, 0xe0, 0x65, 0xdd, 0x1f, 0xb9, 0x89, 0xf8,
			0x38, 0xdb, 0xb0, 0x91, 0xed, 0x3e, 0xca, 0xcc,
			0x83, 0x3e, 0xb5, 0x30, 0xea, 0xde, 0x2e, 0x44,
			0xa7, 0x66, 0x86, 0x60, 0xae, 0xa0, 0xae, 0xf6,
			0x5d, 0x66, 0x7b, 0x52, 0x86, 0x96, 0x66, 0xec,
			0xfd, 0xda, 0xde, 0x32, 0xef, 0x2a, 0xd6, 0x81,
			0xb7, 0x76, 0x63, 0x95, 0x08, 0x18, 0x4f, 0x3f,
			0x2f, 0xd9, 0x14, 0x7f, 0x55, 0x49, 0xed, 0x8d,
			0x29, 0x4a, 0xf3, 0xe0, 0xf8, 0xeb, 0x6c, 0xb6,
			0x86, 0xad, 0xef, 0xaf, 0xb1, 0xd3, 0x6b, 0x17,
			0x87, 0x27, 0xf9, 0xc2, 0xef, 0xcf, 0xce, 0x26,
			0x6c, 0xea, 0x74, 0x0c, 0x28, 0x0f, 0xd2, 0xdf,
			0x52, 0xd2, 0x92, 0xe4, 0x49, 0x16, 0xae, 0x9b,
			0x2b, 0xad, 0x98, 0xc0, 0x62, 0xc9, 0x4e, 0xfd,
			0xbf, 0x52, 0x21, 0xbc, 0xc1, 0x5b, 0x8c, 0xb2,
			0x7e, 0x7c, 0xa1, 0xbc, 0xbd, 0x4f, 0x86, 0xea,
			0x75, 0x38, 0x4d, 0x8a, 0x31, 0xca, 0x4e, 0x8c,
			0xe5, 0xc7, 0x28, 0xf2, 0xf6, 0x95, 0xe6, 0x7f,
			0x8e, 0xad, 0x13, 0x13, 0x2b, 0x2d, 0xfe, 0x52,
			0xf3, 0x1a, 0x23, 0xc5, 0xc1, 0x2a, 0x60, 0xb5,
			0x78, 0xc0, 0x4f, 0xbe, 0x62, 0x1f, 0x96, 0x9f,
			0xb0, 0xf1, 0x29, 0x7d, 0x4e, 0xcb, 0xc8, 0x6f,
			0x41, 0x84, 0x6a, 0x67, 0xe6, 0x96, 0xab, 0x71,
			0x5f, 0xf2, 0x2c, 0x6b, 0x5c, 0x60, 0x6d, 0x8c,
			0xa2, 0x9d, 0xfe, 0xd1, 0x6d, 0xd4, 0x7b, 0x3f,
			0x65, 0xec, 0xaa, 0x7a, 0x5f, 0xe6, 0xbf, 0x6a,
		},
		20, /* salt length */
		/* V4 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA512 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0xd6, 0x6d, 0xf3, 0xad, 0x0a, 0x04, 0xd2, 0x65,
			0x25, 0x2c, 0x2e, 0x13, 0x17, 0xa6, 0x18, 0x93,
			0x1b, 0x4e, 0x47, 0xd2, 0x3b, 0x94, 0xbb, 0xe0,
			0x16, 0x30, 0x9f, 0x47, 0xe1, 0x12, 0x74, 0x9c,
			0xac, 0x98, 0x6c, 0x53, 0xe6, 0x04, 0x14, 0xd7,
			0x24, 0x2d, 0x36, 0xdb, 0x1e, 0x75, 0xfe, 0xea,
			0xfa, 0xfa, 0x5b, 0x46, 0xce, 0x11, 0xa8, 0xef,
			0x60, 0xf0, 0x8c, 0x02, 0x30, 0xaa, 0x1f, 0x44,
			0xb8, 0x61, 0x4e, 0xff, 0xf2, 0xb6, 0x80, 0x0d,
			0x1b, 0x85, 0x7a, 0x32, 0xb7, 0xd0, 0x7f, 0xaf,
			0xd5, 0xf8, 0xbc, 0x13, 0x94, 0x33, 0xe4, 0x55,
			0xe6, 0x3a, 0x18, 0x20, 0x94, 0x42, 0x09, 0x6e,
			0x82, 0x2e, 0x60, 0x58, 0x35, 0x03, 0x5e, 0x2c,
			0x92, 0x9c, 0xbf, 0xc1, 0xe8, 0x9f, 0xe5, 0x07,
			0x31, 0xaf, 0x32, 0xe2, 0x95, 0xe3, 0xb6, 0x39,
			0x30, 0x77, 0x0a, 0x4a, 0x15, 0xc2, 0x7b, 0x40,
		},
		/* S5 : 384 byte */
		{
			0x03, 0x11, 0xec, 0xa8, 0x0f, 0xc0, 0xc3, 0x65,
			0x32, 0x2c, 0x53, 0x57, 0xf6, 0x2f, 0x4a, 0x8c,
			0xef, 0xf8, 0x7e, 0x57, 0xdb, 0x4e, 0x5d, 0xcc,
			0xcb, 0x9a, 0x4c, 0x2b, 0x6c, 0x8f, 0x4b, 0x69,
			0x81, 0x60, 0x75, 0xc9, 0x46, 0xad, 0xea, 0xc2,
			0xdc, 0xd1, 0x46, 0x87, 0x27, 0x2d, 0xdb, 0x70,
			0x12, 0x57, 0x9a, 0xf9, 0x88, 0x6a, 0x9f, 0xf1,
			0x67, 0x6e, 0x28, 0x0c, 0xb8, 0xf6, 0x7a, 0x74,
			0xa6, 0x65, 0x53, 0x2c, 0x78, 0xb4, 0x2e, 0x15,
			0xb0, 0x79, 0x79, 0xf8, 0xb3, 0xf2, 0xa5, 0xbc,
			0x81, 0x6e, 0x46, 0x90, 0x49, 0xf9, 0x23, 0xae,
			0xf3, 0xcb, 0xa3, 0x69, 0xbf, 0x9f, 0xef, 0x89,
			0xc5, 0xba, 0xcf, 0xa1, 0x15, 0xe7, 0xa0, 0x18,
			0x79, 0x5d, 0x7a, 0x92, 0xa7, 0xfb, 0x80, 0x00,
			0x7d, 0xf1, 0xf2, 0x35, 0x54, 0xb5, 0x6b, 0x64,
			0x23, 0xef, 0x64, 0x07, 0x5a, 0x28, 0x87, 0x98,
			0x94, 0x83, 0x8c, 0xc4, 0x85, 0x63, 0x14, 0x8b,
			0xa3, 0x31, 0xb3, 0xec, 0x70, 0xf7, 0xf0, 0x05,
			0x21, 0x19, 0x15, 0xd9, 0x6b, 0xb6, 0x93, 0xad,
			0x3c, 0xfe, 0x33, 0x6e, 0x75, 0x29, 0xb8, 0xf8,
			0xf3, 0x9f, 0xe8, 0x48, 0xba, 0xaf, 0x11, 0x04,
			0x51, 0xd6, 0xac, 0x64, 0x66, 0x76, 0x6c, 0x13,
			0x5f, 0x5a, 0xfe, 0x64, 0x09, 0x42, 0xfa, 0xdf,
			0xac, 0x3e, 0x83, 0xce, 0xc3, 0x12, 0x19, 0x3d,
			0x26, 0x24, 0x9e, 0xb1, 0xf9, 0x96, 0x4b, 0x38,
			0x75, 0x33, 0xd6, 0x80, 0x37, 0x44, 0x80, 0x57,
			0xc4, 0x37, 0xa6, 0x94, 0x31, 0xd9, 0x6a, 0x9e,
			0x3e, 0x51, 0x97, 0x0a, 0xa8, 0x0c, 0x78, 0xe8,
			0xc3, 0x71, 0x8b, 0x87, 0x78, 0x7e, 0xfe, 0xda,
			0x29, 0xf1, 0xdd, 0x9d, 0x53, 0x2e, 0x7a, 0xab,
			0x76, 0x2d, 0x5f, 0x2b, 0x07, 0xeb, 0xa4, 0x0e,
			0x7b, 0x5e, 0xe9, 0xe0, 0x97, 0x72, 0x39, 0xe0,
			0x60, 0x42, 0xed, 0x1f, 0xae, 0xf8, 0x69, 0x58,
			0x6e, 0x26, 0xb1, 0xce, 0xa1, 0xd2, 0x17, 0xd1,
			0xaf, 0x91, 0x2c, 0xed, 0x22, 0x01, 0x64, 0xab,
			0x23, 0x38, 0x17, 0xc3, 0x87, 0x79, 0x5c, 0xad,
			0xe4, 0xf8, 0x8b, 0xfc, 0xa0, 0xfb, 0xc5, 0x3d,
			0x6f, 0x3b, 0xa5, 0x01, 0x77, 0x00, 0x73, 0x08,
			0x9c, 0x96, 0x5f, 0xd5, 0x84, 0x4b, 0x14, 0x49,
			0xab, 0xb7, 0x4e, 0x38, 0x26, 0x89, 0xb0, 0xd6,
			0xcc, 0x06, 0x7d, 0xd8, 0xd7, 0xae, 0xea, 0x1c,
			0x93, 0x17, 0x1f, 0xb8, 0x11, 0x8e, 0x22, 0x87,
			0x6a, 0xe0, 0x9a, 0xfa, 0xae, 0xd0, 0xce, 0x2c,
			0x26, 0xca, 0x70, 0x21, 0xe0, 0x5d, 0x8c, 0xe9,
			0x20, 0x04, 0x24, 0x04, 0x60, 0xb5, 0x2d, 0x67,
			0x9c, 0x7b, 0x4c, 0xd2, 0x5c, 0x8f, 0xf9, 0xd7,
			0xc1, 0x34, 0xd5, 0x60, 0x54, 0xb3, 0x17, 0xc2,
			0xac, 0x3c, 0x70, 0x3e, 0xed, 0x8c, 0x84, 0x77,
		},
		20, /* salt length */
		/* V5 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA512 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0x17, 0xfd, 0x20, 0x24, 0xff, 0x2b, 0x1d, 0xd8,
			0x22, 0x45, 0x8a, 0x9d, 0xef, 0xc7, 0x86, 0x65,
			0xa2, 0xd2, 0x3c, 0x02, 0xd0, 0x73, 0x12, 0x8e,
			0x40, 0x28, 0x23, 0x0a, 0xfd, 0xa0, 0xf0, 0x45,
			0xa9, 0xc3, 0x52, 0xff, 0x14, 0xf2, 0xb4, 0x4f,
			0x0e, 0x90, 0x7e, 0x42, 0x79, 0xc1, 0xd4, 0xc5,
			0xa2, 0x71, 0x81, 0xc5, 0xd1, 0x99, 0x2e, 0x94,
			0x1e, 0xbd, 0xd8, 0x8f, 0x87, 0xba, 0x81, 0xfe,
			0x40, 0xbe, 0x25, 0x3b, 0x4a, 0x98, 0xef, 0x17,
			0x47, 0x32, 0xad, 0x79, 0x8d, 0x5b, 0x82, 0x48,
			0x79, 0xca, 0xda, 0x8b, 0x05, 0x60, 0xa3, 0xcf,
			0x9d, 0xae, 0xbb, 0xc9, 0x2f, 0xb7, 0xa4, 0x3d,
			0x82, 0xf7, 0x7f, 0x1f, 0xc8, 0xa4, 0x38, 0xf7,
			0x76, 0x33, 0xfe, 0x8a, 0x53, 0x1e, 0xf6, 0xb6,
			0xcf, 0xe5, 0x30, 0x9e, 0x97, 0x53, 0xda, 0x07,
			0x66, 0x02, 0xb3, 0x02, 0x1e, 0x21, 0xc3, 0xcc,
		},
		/* S6 : 384 byte */
		{
			0x1f, 0xc7, 0xe4, 0x6d, 0xe0, 0xb6, 0xd9, 0x30,
			0x08, 0xd8, 0x13, 0xa6, 0x42, 0x13, 0x7f, 0x42,
			0xb2, 0x5a, 0x77, 0xa0, 0x4e, 0x17, 0xd4, 0x78,
			0x8c, 0x71, 0x62, 0x9b, 0x17, 0xba, 0xec, 0x3e,
			0x1b, 0x35, 0xa8, 0xc8, 0x59, 0x58, 0x77, 0xa1,
			0x7a, 0xaa, 0xbe, 0xa3, 0x51, 0xef, 0x69, 0x47,
			0x7c, 0xce, 0xdf, 0xa3, 0xbf, 0x83, 0x41, 0xe3,
			0x9d, 0x3e, 0x10, 0x62, 0x2f, 0x39, 0xbf, 0x09,
			0xce, 0xdd, 0xe7, 0xa5, 0xd3, 0x41, 0x17, 0x8e,
			0xfb, 0x09, 0xdf, 0x92, 0xe8, 0x7a, 0x5f, 0xca,
			0x95, 0x11, 0x5f, 0xde, 0x58, 0xb8, 0x99, 0x2c,
			0x6b, 0xcc, 0xa0, 0x0b, 0x34, 0x05, 0x57, 0xa4,
			0x47, 0x7b, 0x43, 0xd8, 0xd4, 0x35, 0x2f, 0x72,
			0xf5, 0x42, 0x54, 0x60, 0x85, 0x5b, 0x3a, 0xfb,
			0x29, 0x1d, 0x92, 0x42, 0xb7, 0x35, 0x24, 0x68,
			0x9e, 0x22, 0x79, 0x93, 0x00, 0xfc, 0x55, 0x63,
			0x39, 0xd1, 0x03, 0x25, 0x2d, 0x31, 0xee, 0x1e,
			0x22, 0xf4, 0xbc, 0x97, 0x40, 0x66, 0x55, 0x3e,
			0x17, 0x2c, 0xb9, 0x0a, 0x66, 0x57, 0x82, 0x75,
			0x00, 0x36, 0xeb, 0x8c, 0x05, 0x36, 0xfd, 0xe1,
			0x96, 0x67, 0x60, 0x52, 0x0b, 0x13, 0xda, 0x0b,
			0x27, 0x84, 0x5a, 0xdc, 0xbb, 0x72, 0x60, 0x34,
			0x72, 0xaf, 0x8c, 0x9e, 0x6b, 0xf2, 0xfd, 0x92,
			0x03, 0x37, 0xf6, 0xfe, 0xd4, 0x9c, 0xda, 0x2c,
			0x2c, 0xf5, 0xa0, 0xea, 0xf1, 0x4b, 0x94, 0x64,
			0xc6, 0x7a, 0xb5, 0x3b, 0x2a, 0xf9, 0x15, 0x5e,
			0x11, 0x62, 0x26, 0xd9, 0xa2, 0xfa, 0xd4, 0x7c,
			0x5e, 0x2b, 0x43, 0xdc, 0x47, 0x7b, 0xac, 0x1f,
			0x59, 0x79, 0x92, 0xe5, 0x1a, 0x9c, 0x1b, 0x04,
			0x5b, 0x76, 0x92, 0x3c, 0x68, 0x5f, 0xd4, 0xf2,
			0x5e, 0x3c, 0xd4, 0x1b, 0x75, 0xaa, 0x45, 0xa3,
			0xf4, 0xa3, 0xe2, 0xda, 0x1b, 0xa4, 0x9a, 0xa7,
			0x92, 0xfc, 0x61, 0xff, 0x8a, 0xf1, 0x14, 0xad,
			0xba, 0x97, 0xab, 0xd2, 0xf6, 0x3f, 0xbc, 0x80,
			0xbe, 0x7d, 0x00, 0x5f, 0x81, 0x14, 0x0e, 0xc5,
			0x30, 0x2f, 0x07, 0x53, 0xec, 0xc4, 0xd3, 0xbc,
			0xbf, 0x6b, 0xcb, 0x29, 0xbe, 0x5d, 0x94, 0xa2,
			0x77, 0x11, 0xd1, 0x8f, 0x77, 0x67, 0x47, 0x9e,
			0xf7, 0x87, 0x48, 0x4d, 0x96, 0x51, 0x16, 0xa1,
			0x82, 0x81, 0x5a, 0xfc, 0xa4, 0xd3, 0x2a, 0xe4,
			0xea, 0xb0, 0x5c, 0x7f, 0x1a, 0x4d, 0x70, 0xbe,
			0xf1, 0x7f, 0x0b, 0xbc, 0xcc, 0x0a, 0x59, 0xfd,
			0x0d, 0x80, 0xf1, 0xb4, 0x11, 0x9f, 0x4f, 0xce,
			0x97, 0x69, 0x8b, 0x62, 0x3d, 0x5d, 0xf5, 0x52,
			0xd5, 0x13, 0x1c, 0x9b, 0x83, 0x0b, 0xd9, 0xb4,
			0xf0, 0x7e, 0x10, 0x5f, 0xa8, 0x70, 0xf1, 0xe4,
			0x06, 0x61, 0x74, 0x61, 0xd0, 0xe3, 0x85, 0x6c,
			0x06, 0xd1, 0xe2, 0x26, 0x1c, 0x9e, 0x86, 0x06,
		},
		20, /* salt length */
		/* V6 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA512 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0x54, 0xc6, 0x7c, 0x29, 0xe4, 0xe0, 0x12, 0xfb,
			0xf3, 0x66, 0xa8, 0xfd, 0xdc, 0x15, 0x9a, 0x6e,
			0xf9, 0xb4, 0x21, 0xef, 0x71, 0x99, 0xca, 0x2d,
			0x51, 0x6e, 0x27, 0xbf, 0xdf, 0x35, 0xc4, 0x52,
			0x2b, 0x8a, 0x5c, 0x03, 0xe6, 0x28, 0x4d, 0x0c,
			0x77, 0xaf, 0x4a, 0xa3, 0x4c, 0xa5, 0x50, 0x1f,
			0x6e, 0xba, 0xdb, 0x5d, 0x59, 0x50, 0x59, 0x62,
			0x0a, 0x3a, 0xdf, 0x41, 0x59, 0xc4, 0x1b, 0x3f,
			0x89, 0xf4, 0x8a, 0x36, 0x1e, 0x10, 0xea, 0x37,
			0xee, 0x60, 0xf3, 0x73, 0x7b, 0x35, 0x42, 0x5c,
			0x80, 0x2b, 0x96, 0x07, 0xca, 0xaa, 0x3e, 0xd5,
			0x49, 0xb3, 0xd2, 0x52, 0x67, 0xd9, 0x22, 0x5d,
			0x98, 0x8d, 0x6c, 0x8a, 0x33, 0x9f, 0xd2, 0xc4,
			0x80, 0x02, 0x0a, 0x37, 0x12, 0xd0, 0x58, 0x6d,
			0x57, 0x8d, 0xb9, 0xb6, 0x6d, 0xb0, 0x61, 0xce,
			0x37, 0x5f, 0x68, 0xba, 0xb1, 0x7b, 0xc1, 0xf3,
		},
		/* S7 : 384 byte */
		{
			0x2c, 0xe6, 0x51, 0x8f, 0x19, 0x34, 0x51, 0xf4,
			0x57, 0xbb, 0x64, 0xc4, 0xa6, 0x4a, 0xeb, 0x76,
			0x43, 0xa6, 0x22, 0x89, 0x0e, 0xcb, 0x42, 0x25,
			0xe6, 0xc0, 0xdf, 0xc6, 0x96, 0x05, 0xcd, 0x6c,
			0x85, 0x87, 0x29, 0x23, 0x8b, 0x1f, 0x9b, 0xa2,
			0xe7, 0x70, 0x13, 0xd7, 0x9b, 0x02, 0x41, 0x22,
			0xac, 0x76, 0x03, 0x0e, 0x09, 0xe2, 0xb7, 0x68,
			0xcf, 0xf1, 0x5b, 0xea, 0x6d, 0xc1, 0xfe, 0x25,
			0x77, 0xf2, 0xe4, 0x1f, 0xb0, 0x85, 0xcb, 0x61,
			0xc3, 0x08, 0xa1, 0xdf, 0x50, 0x2f, 0x4f, 0xea,
			0x7c, 0x0a, 0xeb, 0xf6, 0x5c, 0x02, 0x09, 0x8b,
			0x4b, 0x83, 0xcf, 0x30, 0x8c, 0x98, 0x81, 0x7b,
			0xd9, 0x79, 0xf6, 0xab, 0x9b, 0x36, 0x4d, 0x7e,
			0xab, 0x31, 0x36, 0x52, 0x01, 0x5f, 0x8d, 0x8e,
			0xe0, 0x87, 0xed, 0x65, 0x02, 0x0e, 0xc4, 0x9d,
			0x50, 0x7d, 0x06, 0x94, 0x73, 0xda, 0xba, 0x2b,
			0x7c, 0xd9, 0xe8, 0x70, 0x82, 0x65, 0x99, 0x84,
			0x95, 0x10, 0x59, 0xdb, 0xc2, 0xe9, 0xfd, 0x92,
			0xd2, 0xa1, 0x2b, 0x8a, 0x84, 0x86, 0x42, 0xa6,
			0x4f, 0x13, 0xc8, 0xa8, 0x52, 0x30, 0x11, 0x87,
			0x69, 0x2e, 0xee, 0x5d, 0xf9, 0x93, 0x63, 0x33,
			0x39, 0x2f, 0x84, 0x3b, 0x19, 0x78, 0x0a, 0x5a,
			0xbf, 0x63, 0x0c, 0x18, 0xde, 0x8b, 0x2a, 0xa4,
			0xb2, 0xd4, 0x0f, 0x22, 0xe6, 0x13, 0x52, 0x9b,
			0x2b, 0xa3, 0x78, 0xc8, 0x04, 0x33, 0x55, 0x37,
			0x33, 0x7a, 0xb4, 0x28, 0xd4, 0x6e, 0x61, 0xfe,
			0x00, 0x3d, 0x73, 0xfd, 0xda, 0x78, 0x0d, 0x2a,
			0x49, 0x90, 0x97, 0x7e, 0x8a, 0x69, 0xd0, 0xa3,
			0xeb, 0x87, 0x78, 0xc3, 0x7d, 0x68, 0xf3, 0x15,
			0x70, 0x4c, 0xcc, 0x57, 0x06, 0x54, 0x36, 0x0c,
			0x2a, 0x27, 0xc1, 0x1b, 0xdd, 0x5e, 0x41, 0xb1,
			0x7f, 0x92, 0x49, 0x8d, 0xc8, 0xaf, 0xe8, 0xb4,
			0xae, 0x22, 0xb5, 0xdf, 0x30, 0xa5, 0xea, 0x84,
			0x9a, 0x78, 0xd6, 0x82, 0x48, 0x11, 0x3c, 0xf8,
			0xf4, 0x84, 0xe2, 0x73, 0xe2, 0x00, 0x4c, 0x11,
			0x05, 0xe9, 0xef, 0xf7, 0x26, 0xcd, 0x09, 0x2c,
			0xd7, 0xfd, 0x39, 0x8f, 0x59, 0x3a, 0xd3, 0x25,
			0xe6, 0x3b, 0x1e, 0x17, 0x6e, 0xcb, 0xf4, 0x56,
			0x41, 0x08, 0x70, 0x2f, 0x78, 0x6f, 0x22, 0x0d,
			0x1e, 0x07, 0x5c, 0xe9, 0x1c, 0x85, 0x93, 0xcb,
			0x9f, 0x3e, 0x30, 0x45, 0xb8, 0xf1, 0x16, 0xd0,
			0x4c, 0x6b, 0x43, 0x38, 0x1c, 0x2b, 0xe4, 0xdf,
			0x4d, 0x25, 0xb7, 0x52, 0xaf, 0xe7, 0xa2, 0xf0,
			0x89, 0x9f, 0xe4, 0x9e, 0x5c, 0x86, 0x53, 0x22,
			0x80, 0x78, 0xec, 0xf3, 0x70, 0x99, 0x11, 0x53,
			0x02, 0x99, 0xc7, 0xeb, 0xec, 0xbc, 0x4f, 0x56,
			0xc1, 0x31, 0x94, 0xce, 0xe2, 0x17, 0x96, 0xac,
			0xd2, 0xe1, 0xab, 0x75, 0x3c, 0x3d, 0x6c, 0x03,
		},
		20, /* salt length */
		/* V7 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA512 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0xc0, 0xf7, 0x83, 0x79, 0x38, 0xca, 0x95, 0xe0,
			0x6d, 0x2b, 0xaf, 0x1a, 0x62, 0xa3, 0x86, 0x1d,
			0x03, 0xf9, 0x10, 0x11, 0xae, 0xa9, 0x31, 0x5f,
			0xf2, 0x92, 0x97, 0x0b, 0x97, 0xef, 0xe3, 0x43,
			0xde, 0x88, 0x09, 0x38, 0x09, 0x2a, 0xf9, 0x2c,
			0x1e, 0x54, 0x7f, 0xcc, 0xe9, 0x33, 0x67, 0xc1,
			0x40, 0xd0, 0xe6, 0x82, 0x8b, 0x89, 0x35, 0x6c,
			0x83, 0xbe, 0x7f, 0x14, 0x39, 0xcc, 0xfd, 0xdc,
			0x5a, 0x3f, 0x60, 0xc4, 0xb2, 0x43, 0x6f, 0x83,
			0xf0, 0x5c, 0xf1, 0xd7, 0xa9, 0x57, 0xcd, 0x19,
			0x9f, 0x81, 0x70, 0x14, 0x42, 0x16, 0x6f, 0x17,
			0x77, 0xfc, 0x6d, 0x49, 0x99, 0xb1, 0x40, 0x3d,
			0xbd, 0x83, 0x4f, 0xc6, 0x3b, 0xff, 0x3e, 0x0d,
			0x6a, 0xab, 0xce, 0xe0, 0xaa, 0xf8, 0xfe, 0x4d,
			0xa8, 0x72, 0x77, 0x70, 0xdd, 0x3a, 0x26, 0x89,
			0x58, 0xb6, 0x2b, 0x51, 0xbe, 0x87, 0xf2, 0x8f,
		},
		/* S8 : 384 byte */
		{
			0x20, 0xf7, 0xc4, 0xa9, 0x5f, 0x19, 0x58, 0x11,
			0x65, 0x30, 0x9a, 0xcf, 0x45, 0xda, 0xb8, 0x93,
			0xc7, 0x15, 0xe0, 0x9f, 0x16, 0x21, 0x0e, 0x46,
			0x61, 0x3f, 0xc5, 0x71, 0x85, 0x6e, 0x3e, 0x31,
			0x46, 0xe7, 0x99, 0x46, 0x58, 0x68, 0xfb, 0xb3,
			0x64, 0xfc, 0xd1, 0xfe, 0x39, 0x73, 0xe4, 0x53,
			0x5a, 0x79, 0xbf, 0xee, 0xcc, 0xb1, 0x06, 0x95,
			0xee, 0x6d, 0x8f, 0x5a, 0x96, 0xc7, 0x8b, 0x41,
			0x63, 0x1d, 0x2d, 0x62, 0x91, 0x44, 0x1e, 0x76,
			0x11, 0xd4, 0x5f, 0x48, 0xc1, 0xe5, 0xab, 0x8d,
			0xbf, 0x54, 0x02, 0x11, 0xf5, 0x0d, 0x2d, 0xf0,
			0x12, 0xb6, 0x58, 0xfe, 0x68, 0x27, 0x71, 0x34,
			0xfb, 0x47, 0x52, 0x9a, 0xea, 0x33, 0x9e, 0x75,
			0x0c, 0x15, 0x26, 0x76, 0x86, 0xa2, 0x39, 0x82,
			0x50, 0x57, 0x58, 0xb5, 0x0e, 0x4e, 0x34, 0xae,
			0x84, 0x5e, 0xa3, 0x14, 0x30, 0x38, 0xc1, 0xb2,
			0x51, 0xcd, 0x17, 0x7a, 0xa2, 0x5a, 0x99, 0x6c,
			0xf2, 0x79, 0x1c, 0x74, 0x9f, 0x2e, 0x4a, 0x28,
			0x95, 0x16, 0x05, 0x2d, 0x0c, 0xfb, 0x40, 0xcf,
			0x21, 0x5a, 0xe0, 0x0b, 0x50, 0x32, 0x19, 0xa1,
			0x80, 0x97, 0xff, 0x26, 0x8c, 0xb8, 0xcc, 0x83,
			0x37, 0x00, 0x91, 0x2d, 0x23, 0x05, 0x45, 0x34,
			0xa6, 0x31, 0x76, 0x22, 0xf8, 0xb6, 0xc5, 0xd1,
			0x76, 0x4e, 0xc7, 0x24, 0x8f, 0x9c, 0xfc, 0xd4,
			0xa2, 0x01, 0xa9, 0xa8, 0x2b, 0x8b, 0x43, 0x83,
			0x05, 0x61, 0x93, 0x77, 0xe6, 0x21, 0xb4, 0x1c,
			0x92, 0xce, 0x61, 0xed, 0x66, 0x62, 0xaf, 0x84,
			0xd5, 0x19, 0xf9, 0xba, 0x51, 0x9d, 0xec, 0xe7,
			0xd6, 0x07, 0xa8, 0x77, 0x87, 0xba, 0x2d, 0x21,
			0x98, 0xb5, 0x40, 0x47, 0x35, 0xbb, 0x8c, 0xe1,
			0x34, 0x94, 0x93, 0x68, 0x8d, 0xee, 0x1c, 0x8a,
			0xd6, 0xc6, 0xc7, 0xd0, 0xff, 0x11, 0x6f, 0x45,
			0xb9, 0xe0, 0x32, 0x78, 0x59, 0x01, 0xf5, 0xfe,
			0xfb, 0x17, 0x11, 0x74, 0x0f, 0xce, 0x61, 0x5b,
			0x87, 0xf4, 0xe9, 0x3c, 0x8a, 0xbc, 0x09, 0x32,
			0xc1, 0x34, 0x6d, 0xa0, 0x05, 0xcc, 0x81, 0x5a,
			0x9a, 0x6d, 0x01, 0x7a, 0x2b, 0x16, 0x14, 0xd1,
			0xd9, 0x96, 0xdf, 0x8c, 0xe1, 0xb1, 0x23, 0xb1,
			0xce, 0x5d, 0xae, 0xa0, 0x4d, 0x98, 0x3f, 0xee,
			0x05, 0xcd, 0xd6, 0xbf, 0xa0, 0x5f, 0xe2, 0xfc,
			0xf0, 0xe7, 0x3b, 0x16, 0x63, 0xf3, 0x6a, 0x41,
			0x03, 0x6a, 0x2e, 0xe6, 0x7d, 0xb9, 0x24, 0x68,
			0x6e, 0xee, 0xd3, 0x80, 0x34, 0x18, 0x97, 0xc5,
			0xf2, 0xb1, 0xc5, 0x1a, 0x7f, 0x6f, 0x01, 0x05,
			0x54, 0xe5, 0xfa, 0x40, 0x63, 0x83, 0x49, 0x7f,
			0x42, 0x26, 0x0a, 0x31, 0xdc, 0x06, 0x6c, 0xd0,
			0xc1, 0xa2, 0xcb, 0xe4, 0xc0, 0x61, 0xcd, 0xad,
			0x0b, 0xbc, 0x22, 0x66, 0x87, 0x24, 0x5e, 0xfc,
		},
		20, /* salt length */
		/* V8 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA512 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0x32, 0x07, 0xdf, 0x80, 0x95, 0x1d, 0x21, 0x72,
			0xd9, 0x45, 0x0a, 0xc9, 0xd6, 0x86, 0xdc, 0xd5,
			0x6d, 0xcf, 0x17, 0x22, 0xc9, 0x2e, 0xe2, 0x4d,
			0xe5, 0x42, 0xac, 0x17, 0x4e, 0xf3, 0x13, 0xb8,
			0x2b, 0x60, 0x05, 0x76, 0x7a, 0x19, 0x25, 0x75,
			0x9c, 0x8f, 0x70, 0xd1, 0x14, 0xc9, 0x20, 0xea,
			0x8c, 0x58, 0xd0, 0xff, 0x18, 0x8c, 0xe2, 0xff,
			0xbf, 0xe8, 0x1e, 0x9b, 0x38, 0xd4, 0x9b, 0xa0,
			0xf1, 0x92, 0xe3, 0x84, 0x56, 0x73, 0x52, 0x3e,
			0x4f, 0x4b, 0xbf, 0x5b, 0x0a, 0x22, 0x5f, 0x4f,
			0xfc, 0x2c, 0xe5, 0x09, 0x22, 0xfc, 0xee, 0xca,
			0x8d, 0xf4, 0x9d, 0x36, 0x1d, 0x00, 0x82, 0xac,
			0x8d, 0x84, 0xc1, 0xd3, 0xaf, 0x92, 0x70, 0x75,
			0xf9, 0x61, 0x40, 0x91, 0x41, 0xfa, 0x60, 0xab,
			0xc5, 0x35, 0x9e, 0x67, 0x6e, 0xe4, 0xd0, 0x56,
			0x56, 0x4e, 0x72, 0x10, 0x87, 0xdf, 0x90, 0x83,
		},
		/* S9 : 384 byte */
		{
			0x45, 0xd1, 0xee, 0x2b, 0x69, 0xf8, 0x52, 0xb7,
			0xf7, 0x8a, 0x43, 0x7c, 0x28, 0x49, 0x9c, 0xeb,
			0x5c, 0x68, 0x84, 0x2e, 0x4c, 0xf7, 0xcd, 0x2c,
			0x87, 0x9d, 0xd2, 0xb0, 0xae, 0x62, 0x43, 0xc8,
			0xa3, 0x8f, 0x62, 0x17, 0x9f, 0x50, 0x88, 0x01,
			0x26, 0x6a, 0x2d, 0x5d, 0x0f, 0x32, 0xd2, 0x2f,
			0x66, 0x9a, 0xbd, 0xdd, 0x45, 0xf4, 0xdf, 0xe3,
			0x09, 0xae, 0x64, 0x07, 0xff, 0xdc, 0x96, 0x5f,
			0x1f, 0x68, 0xce, 0xf5, 0x74, 0x19, 0x48, 0xba,
			0x29, 0x7f, 0x3d, 0x8d, 0xf7, 0xda, 0xa8, 0x3b,
			0xda, 0x95, 0xa8, 0x28, 0x62, 0xff, 0xa7, 0x0b,
			0xb5, 0xd4, 0x97, 0xaf, 0xf4, 0x2d, 0x51, 0x6c,
			0x3a, 0x6f, 0xb4, 0xd2, 0x6c, 0x19, 0xf7, 0xb3,
			0x3c, 0x16, 0x72, 0xa2, 0x1a, 0x45, 0x74, 0xc4,
			0x65, 0x86, 0xa2, 0x0b, 0xfd, 0x5a, 0xd8, 0xde,
			0x49, 0x51, 0x26, 0xe9, 0x93, 0x94, 0x2d, 0x25,
			0xd8, 0x7a, 0x23, 0x74, 0x60, 0x26, 0xb3, 0xef,
			0x17, 0x60, 0x55, 0x18, 0xbd, 0x35, 0xd5, 0x1d,
			0x09, 0xb1, 0x41, 0x7c, 0x3c, 0x30, 0xac, 0xe1,
			0xcd, 0x35, 0x62, 0x87, 0xa6, 0x6e, 0x86, 0x41,
			0x1b, 0xb3, 0xde, 0x93, 0x4e, 0x21, 0x87, 0xe8,
			0xb1, 0x55, 0x7f, 0xd5, 0x41, 0xc9, 0x2c, 0xc5,
			0x2b, 0x19, 0x3b, 0x97, 0x8e, 0x6b, 0xfd, 0xa5,
			0xe9, 0x3e, 0xe0, 0x65, 0x5e, 0x11, 0x1f, 0xd8,
			0x1d, 0x29, 0xce, 0xe7, 0xbb, 0x9e, 0xec, 0x66,
			0x68, 0x71, 0x5a, 0x24, 0x14, 0x00, 0xdf, 0xc9,
			0x82, 0x27, 0x11, 0xc1, 0x4f, 0x94, 0x78, 0xe4,
			0xa0, 0x08, 0xa8, 0x8b, 0xd8, 0x1b, 0x61, 0xb2,
			0x7e, 0x1d, 0xfc, 0x5b, 0x1b, 0x64, 0xfc, 0x65,
			0x9b, 0xcc, 0x20, 0xdd, 0x44, 0xd4, 0x1e, 0x7f,
			0x11, 0xba, 0xe6, 0xea, 0x85, 0x3e, 0xe7, 0x00,
			0xa0, 0x6c, 0x54, 0xec, 0x15, 0x5e, 0xd2, 0x11,
			0x50, 0x98, 0x17, 0x8e, 0x3f, 0xe9, 0x82, 0x7d,
			0x49, 0x39, 0x4b, 0xfa, 0x4c, 0xb8, 0xb0, 0xac,
			0xe6, 0x54, 0x71, 0xac, 0x96, 0xb6, 0xc8, 0x0c,
			0x88, 0x37, 0x79, 0x9d, 0xac, 0x1c, 0x75, 0x34,
			0x59, 0x24, 0xd2, 0x11, 0x77, 0xeb, 0xf4, 0xa7,
			0xe9, 0xb6, 0xa3, 0xea, 0x78, 0x40, 0x03, 0x10,
			0x7a, 0x19, 0x9c, 0x29, 0x1e, 0x5f, 0x71, 0xf0,
			0x75, 0x7a, 0x7c, 0x70, 0xc6, 0xf6, 0x1a, 0xde,
			0xda, 0x58, 0x84, 0x90, 0x1b, 0x13, 0x1c, 0xbe,
			0x43, 0x1e, 0xad, 0x0a, 0x7f, 0xb7, 0x8e, 0x61,
			0xd9, 0xc1, 0x93, 0x0d, 0x6b, 0xcc, 0x81, 0x75,
			0xcf, 0x4b, 0xaa, 0x04, 0x7f, 0x49, 0x29, 0x0d,
			0x85, 0x59, 0xb5, 0x7a, 0xe8, 0x59, 0xfb, 0xdd,
			0x90, 0xf0, 0x88, 0xb1, 0xa4, 0x13, 0xb0, 0xf3,
			0xcb, 0xd4, 0x19, 0x9f, 0xb1, 0xd0, 0xcd, 0x84,
			0x5e, 0x3c, 0xcd, 0xf7, 0x03, 0x34, 0x70, 0xbf,
		},
		20, /* salt length */
		/* V9 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
};

typedef mod3072_t tv_t;


#include "siggenPSStest.c"
