/* siggenPSS_2_1024_testfc.c */
/*
 * Copyright (c) 2017 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

/*
 * This test vector was created based on SigGenPSS_186-2.txt from
 * https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Algorithm-Validation-Program/documents/dss/186-2rsatestvectors.zip.
 *
 * The link to the archive file exists in the following URL:
 * https://csrc.nist.gov/projects/cryptographic-algorithm-validation-program/digital-signatures/
 */
#include "siggenPSStest.h"

static const unsigned short sn1[] = {
	0xbcb4, 0x7b2e, 0x0daf, 0xcba8, 0x1ff2, 0xa2b5, 0xcb11, 0x5ca7,
	0xe757, 0x184c, 0x9d72, 0xbcdc, 0xda70, 0x7a14, 0x6b3b, 0x4e29,
	0x989d, 0xdc66, 0x0bd6, 0x9486, 0x5b93, 0x2b71, 0xca24, 0xa335,
	0xcf4d, 0x339c, 0x7191, 0x83e6, 0x222e, 0x4c9e, 0xa687, 0x5acd,
	0x528a, 0x49ba, 0x2186, 0x3fe0, 0x8147, 0xc3a4, 0x7e41, 0x990b,
	0x51a0, 0x3f77, 0xd221, 0x37f8, 0xd74c, 0x43a5, 0xa45f, 0x4e9e,
	0x18a2, 0xd15d, 0xb051, 0xdc89, 0x385d, 0xb9cf, 0x8374, 0xb63a,
	0x8cc8, 0x8113, 0x710e, 0x6d81, 0x7907, 0x5b7d, 0xc79e, 0xe76b,
};

static const unsigned short se1[] = {
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0001, 0x0001,
};

static const unsigned short sd1[] = {
	0x383a, 0x6f19, 0xe1ea, 0x27fd, 0x08c7, 0xfbc3, 0xbfa6, 0x84bd,
	0x6329, 0x888c, 0x0bbe, 0x4c98, 0x625e, 0x7181, 0xf411, 0xcfd0,
	0x8531, 0x44a3, 0x0394, 0x04dd, 0xa41b, 0xce2e, 0x31d5, 0x88ec,
	0x57c0, 0xe148, 0x146f, 0x0fa6, 0x5b39, 0x008b, 0xa583, 0x5f82,
	0x9ba3, 0x5ae2, 0xf155, 0xd61b, 0x8a12, 0x581b, 0x99c9, 0x27fd,
	0x2f22, 0x252c, 0x5e73, 0xcba4, 0xa610, 0xdb39, 0x73e0, 0x19ee,
	0x0f95, 0x130d, 0x4319, 0xed41, 0x3432, 0xf2e5, 0xe20d, 0x5215,
	0xcdd2, 0x7c21, 0x6420, 0x6b3f, 0x80ed, 0xee51, 0x938a, 0x25c1,
};

#define HAS_SHA1_TESTVECTOR

#define HAS_SHA224_TESTVECTOR

#define HAS_SHA256_TESTVECTOR

#define HAS_SHA384_TESTVECTOR

#define HAS_SHA512_TESTVECTOR

/* 10 elements */
static mod1024_t SHA1[] = {
	/* SHA1 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0x12, 0x48, 0xf6, 0x2a, 0x43, 0x89, 0xf4, 0x2f,
			0x7b, 0x4b, 0xb1, 0x31, 0x05, 0x3d, 0x6c, 0x88,
			0xa9, 0x94, 0xdb, 0x20, 0x75, 0xb9, 0x12, 0xcc,
			0xbe, 0x3e, 0xa7, 0xdc, 0x61, 0x17, 0x14, 0xf1,
			0x4e, 0x07, 0x5c, 0x10, 0x48, 0x58, 0xf2, 0xf6,
			0xe6, 0xcf, 0xd6, 0xab, 0xde, 0xdf, 0x01, 0x5a,
			0x82, 0x1d, 0x03, 0x60, 0x8b, 0xf4, 0xeb, 0xa3,
			0x16, 0x9a, 0x67, 0x25, 0xec, 0x42, 0x2c, 0xd9,
			0x06, 0x94, 0x98, 0xb5, 0x51, 0x5a, 0x96, 0x08,
			0xae, 0x7c, 0xc3, 0x0e, 0x3d, 0x2e, 0xcf, 0xc1,
			0xdb, 0x68, 0x25, 0xf3, 0xe9, 0x96, 0xce, 0x9a,
			0x50, 0x92, 0x92, 0x6b, 0xc1, 0xcf, 0x61, 0xaa,
			0x42, 0xd7, 0xf2, 0x40, 0xe6, 0xf7, 0xaa, 0x0e,
			0xdb, 0x38, 0xbf, 0x81, 0xaa, 0x92, 0x9d, 0x66,
			0xbb, 0x5d, 0x89, 0x00, 0x18, 0x08, 0x84, 0x58,
			0x72, 0x0d, 0x72, 0xd5, 0x69, 0x24, 0x7b, 0x0c,
		},
		/* S0 : 128 byte */
		{
			0x68, 0x2c, 0xf5, 0x3c, 0x11, 0x45, 0xd2, 0x2a,
			0x50, 0xca, 0xa9, 0xeb, 0x1a, 0x9b, 0xa7, 0x06,
			0x70, 0xc5, 0x91, 0x5e, 0x0f, 0xdf, 0xde, 0x64,
			0x57, 0xa7, 0x65, 0xde, 0x2a, 0x8f, 0xe1, 0x2d,
			0xe9, 0x79, 0x41, 0x72, 0xa7, 0x8d, 0x14, 0xe6,
			0x68, 0xd4, 0x98, 0xac, 0xed, 0xad, 0x61, 0x65,
			0x04, 0xbb, 0x17, 0x64, 0xd0, 0x94, 0x60, 0x70,
			0x70, 0x08, 0x05, 0x92, 0xc3, 0xa6, 0x9c, 0x34,
			0x3d, 0x98, 0x2b, 0xd7, 0x78, 0x65, 0x87, 0x3d,
			0x35, 0xe2, 0x48, 0x22, 0xca, 0xf4, 0x34, 0x43,
			0xcc, 0x10, 0x24, 0x9a, 0xf6, 0xa1, 0xe2, 0x6e,
			0xf3, 0x44, 0xf2, 0x8b, 0x9e, 0xf6, 0xf1, 0x4e,
			0x09, 0xad, 0x83, 0x97, 0x48, 0xe5, 0x14, 0x8b,
			0xcc, 0xeb, 0x0f, 0xd2, 0xaa, 0x63, 0x70, 0x9c,
			0xb4, 0x89, 0x75, 0xcb, 0xf9, 0xc7, 0xb4, 0x9a,
			0xbc, 0x66, 0xa1, 0xdc, 0x6c, 0xb5, 0xb3, 0x1a,
		},
		20, /* salt length */
		/* V0 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA1 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0x99, 0x68, 0x80, 0x9a, 0x55, 0x7b, 0xb4, 0xf8,
			0x92, 0x03, 0x9f, 0xf2, 0xb6, 0xa0, 0xef, 0xcd,
			0x06, 0x52, 0x36, 0x24, 0xbc, 0x3b, 0x9a, 0xd3,
			0x59, 0xa7, 0xcf, 0x14, 0x3c, 0x49, 0x42, 0xe8,
			0x74, 0xc7, 0x97, 0xb9, 0xd3, 0x7a, 0x56, 0x3d,
			0x43, 0x6f, 0xe1, 0x9d, 0x5d, 0xb1, 0xaa, 0xd7,
			0x38, 0xca, 0xa2, 0x61, 0x7f, 0x87, 0xf5, 0x0f,
			0xc7, 0xfc, 0xf4, 0x36, 0x1f, 0xc8, 0x52, 0x12,
			0xe8, 0x9a, 0x94, 0x65, 0xe7, 0xf4, 0xc3, 0x61,
			0x98, 0x2f, 0x64, 0xc8, 0xc5, 0xc0, 0xaa, 0x52,
			0x58, 0xb9, 0xe9, 0x4f, 0x6e, 0x93, 0x4e, 0x8d,
			0xac, 0x2a, 0xce, 0x7c, 0xd6, 0x09, 0x5c, 0x90,
			0x9d, 0xe8, 0x5f, 0xe7, 0xb9, 0x73, 0x63, 0x2c,
			0x38, 0x4d, 0x0e, 0xbb, 0x16, 0x55, 0x56, 0x05,
			0x0d, 0x28, 0xf2, 0x36, 0xae, 0xe7, 0x0e, 0x16,
			0xb1, 0x3a, 0x43, 0x2d, 0x8a, 0x94, 0xc6, 0x2b,
		},
		/* S1 : 128 byte */
		{
			0x8f, 0x5e, 0xa7, 0x03, 0x73, 0x67, 0xe0, 0xdb,
			0x75, 0x67, 0x05, 0x04, 0x08, 0x57, 0x90, 0xac,
			0xd6, 0xd9, 0x7d, 0x96, 0xf5, 0x1e, 0x76, 0xdf,
			0x91, 0x6a, 0x0c, 0x2e, 0x4c, 0xd6, 0x6e, 0x1a,
			0xb5, 0x1c, 0x4c, 0xd8, 0xe2, 0xc3, 0xe4, 0xef,
			0x78, 0x1f, 0x63, 0x8a, 0xd6, 0x5d, 0xc4, 0x9c,
			0x8d, 0x6d, 0x7f, 0x69, 0x30, 0xf8, 0x0b, 0x6a,
			0xe1, 0x99, 0xea, 0x28, 0x3a, 0x89, 0x24, 0x92,
			0x5a, 0x50, 0xed, 0xab, 0x79, 0xbb, 0x3f, 0x34,
			0x86, 0x1f, 0xfa, 0x8b, 0x2f, 0x96, 0xfd, 0xf9,
			0xf8, 0xca, 0xd3, 0xd3, 0xf8, 0xf0, 0x25, 0x47,
			0x8c, 0x81, 0xf3, 0x16, 0xda, 0x61, 0xb0, 0xd6,
			0xa7, 0xf7, 0x1b, 0x90, 0x68, 0xef, 0xdf, 0xb3,
			0x3c, 0x21, 0x98, 0x3a, 0x92, 0x2f, 0x46, 0x69,
			0x28, 0x0d, 0x8e, 0x84, 0xf9, 0x63, 0xff, 0x88,
			0x5e, 0xf5, 0x6d, 0xd3, 0xf5, 0x03, 0x81, 0xdb,
		},
		20, /* salt length */
		/* V1 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA1 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0xf9, 0xbe, 0x76, 0xe9, 0xb0, 0x29, 0xc4, 0x93,
			0x32, 0x60, 0x24, 0x9d, 0xd7, 0x6c, 0x50, 0x67,
			0xcc, 0xd4, 0x83, 0xd4, 0xb9, 0xb3, 0x38, 0xfd,
			0xa0, 0x0d, 0x34, 0x27, 0x09, 0x63, 0xc6, 0xf3,
			0x5c, 0x85, 0x4e, 0xd5, 0x8e, 0xa8, 0xfc, 0xc7,
			0xff, 0xb8, 0xda, 0x3f, 0xa3, 0xf0, 0x0d, 0x5e,
			0x61, 0xa7, 0x58, 0x6a, 0xb8, 0x6d, 0xe1, 0x7e,
			0xa8, 0x56, 0x38, 0x80, 0xd0, 0x96, 0x95, 0x54,
			0xd4, 0x4e, 0x61, 0x4f, 0x01, 0xa6, 0xf8, 0xef,
			0x34, 0x1c, 0xae, 0xc9, 0xf7, 0x1c, 0x10, 0xc2,
			0xee, 0xd0, 0x6c, 0x82, 0x72, 0x39, 0x93, 0x26,
			0x7b, 0x7f, 0xdd, 0x35, 0xc3, 0x85, 0x6e, 0xd6,
			0x28, 0xc1, 0xb8, 0x40, 0x52, 0x4b, 0x41, 0x71,
			0x97, 0x33, 0xa6, 0x23, 0x1b, 0x18, 0xe1, 0xfc,
			0x3c, 0xf6, 0xc7, 0x05, 0x2d, 0x40, 0xd4, 0x5d,
			0xe0, 0x2f, 0x2b, 0x2f, 0x2a, 0x59, 0xd9, 0xe1,
		},
		/* S2 : 128 byte */
		{
			0x4f, 0x7f, 0xf1, 0x12, 0xa7, 0x93, 0x77, 0x07,
			0xd5, 0x29, 0xb5, 0x60, 0x2e, 0x1d, 0x01, 0x7b,
			0x79, 0x73, 0x9b, 0x13, 0x23, 0x4a, 0x72, 0x5d,
			0xc3, 0x64, 0x87, 0xe0, 0x5e, 0xe4, 0xf5, 0xe0,
			0x72, 0x27, 0x0b, 0x67, 0x12, 0xb5, 0xf7, 0x1e,
			0x47, 0x74, 0x7d, 0xd4, 0xb9, 0xb2, 0x89, 0xdf,
			0x31, 0xf9, 0x1f, 0xc9, 0x7d, 0xb3, 0x9b, 0xa1,
			0x46, 0x94, 0xff, 0x89, 0x4b, 0x42, 0x93, 0x25,
			0x55, 0xd0, 0x1d, 0xfd, 0x41, 0x46, 0xa1, 0x67,
			0x23, 0x38, 0xcc, 0x60, 0x52, 0x1e, 0x4b, 0x59,
			0x88, 0xfd, 0x22, 0x89, 0x6b, 0x51, 0x2f, 0xaa,
			0xf5, 0x88, 0x8d, 0xed, 0xcd, 0x5f, 0x96, 0x62,
			0x31, 0x9e, 0x9f, 0x75, 0x2c, 0x99, 0xc3, 0x41,
			0xbe, 0x08, 0xac, 0x2b, 0xd0, 0x34, 0x68, 0x45,
			0xcc, 0xb7, 0x4d, 0x0a, 0x7d, 0x4d, 0x81, 0x65,
			0xaa, 0x76, 0x13, 0xb6, 0x68, 0x47, 0x70, 0x2f,
		},
		20, /* salt length */
		/* V2 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA1 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0x5e, 0xec, 0x1f, 0x64, 0xa7, 0x99, 0xc1, 0xbb,
			0x29, 0xb9, 0xaa, 0x66, 0x35, 0xb0, 0xbb, 0x93,
			0xd1, 0xb4, 0x61, 0xc9, 0xf6, 0x31, 0x6a, 0x71,
			0xfc, 0x6c, 0x17, 0x94, 0x52, 0x66, 0x72, 0xe6,
			0x72, 0x03, 0xe1, 0x04, 0x14, 0x6b, 0xfc, 0xfa,
			0xa2, 0xb1, 0x9c, 0x08, 0xed, 0xb3, 0x34, 0xa6,
			0x1e, 0x41, 0x65, 0xc5, 0xef, 0x81, 0xcc, 0x58,
			0x1b, 0x91, 0x1f, 0x13, 0x12, 0x6f, 0x78, 0x72,
			0xf2, 0x16, 0xf0, 0xfe, 0xb3, 0xea, 0x70, 0xcc,
			0xa1, 0x56, 0xc4, 0x46, 0x10, 0xb4, 0xf1, 0x2d,
			0x4b, 0x6d, 0xc9, 0xdf, 0xcd, 0x61, 0xbc, 0xd6,
			0xb4, 0x89, 0xf1, 0x72, 0x72, 0xda, 0x96, 0xf8,
			0xc6, 0x6d, 0x74, 0xda, 0xeb, 0xde, 0xc6, 0x39,
			0x0f, 0xf2, 0x4b, 0xa8, 0x06, 0x7c, 0x9a, 0x33,
			0x40, 0x81, 0xb6, 0xa0, 0xf8, 0x9a, 0xe2, 0xf4,
			0xaf, 0x9a, 0xb6, 0x39, 0xd9, 0x71, 0x73, 0x7f,
		},
		/* S3 : 128 byte */
		{
			0x34, 0x4c, 0x60, 0x2f, 0x68, 0xbd, 0x95, 0x29,
			0xaf, 0x10, 0xa8, 0x0a, 0x1a, 0x3c, 0x25, 0xe2,
			0x4c, 0xce, 0x3d, 0x27, 0xe6, 0xba, 0xac, 0xc1,
			0xb9, 0x98, 0xe8, 0xc0, 0xb6, 0x34, 0xe1, 0x68,
			0x9a, 0x97, 0xd2, 0xeb, 0x60, 0x04, 0xad, 0x08,
			0xb6, 0x51, 0xa2, 0xde, 0xc5, 0x3c, 0x8a, 0x6b,
			0xae, 0x51, 0xde, 0x69, 0x4f, 0x07, 0x2a, 0xdd,
			0x1f, 0x90, 0x75, 0x8a, 0xbb, 0x6f, 0x37, 0x47,
			0x1c, 0x73, 0x79, 0x49, 0x8f, 0x6a, 0x63, 0xb4,
			0x9f, 0xf1, 0x12, 0xa9, 0x9d, 0x4d, 0xe5, 0x85,
			0x5e, 0x9b, 0x1d, 0x17, 0x83, 0xf7, 0x5c, 0x91,
			0x97, 0x30, 0xe9, 0xfc, 0x29, 0x40, 0x17, 0xbc,
			0x96, 0xba, 0x82, 0xcc, 0x00, 0x97, 0x97, 0x35,
			0x38, 0x81, 0xea, 0x25, 0xc4, 0xfa, 0xc9, 0x23,
			0x3a, 0x01, 0x18, 0x9a, 0xc4, 0x82, 0x5a, 0xa5,
			0x2b, 0xf0, 0x1f, 0xa3, 0x00, 0xe6, 0x30, 0xd5,
		},
		20, /* salt length */
		/* V3 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA1 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0xbd, 0x75, 0x51, 0x1e, 0x45, 0x16, 0x02, 0x95,
			0x89, 0x95, 0xe5, 0x0e, 0xfa, 0xef, 0x13, 0x0d,
			0x9f, 0x5e, 0xef, 0x3b, 0x09, 0x7e, 0xb8, 0xaf,
			0x88, 0xae, 0x5a, 0x6d, 0x9f, 0xb9, 0x8a, 0x2c,
			0xb6, 0xb6, 0x11, 0x5d, 0x0f, 0x2c, 0x68, 0x68,
			0xfb, 0x7a, 0x86, 0x39, 0x4e, 0x76, 0xf6, 0xfb,
			0x10, 0x55, 0x2c, 0x23, 0x75, 0xcb, 0x19, 0xb1,
			0x0f, 0x72, 0xfd, 0xb2, 0x43, 0x70, 0x7d, 0x0a,
			0xfa, 0x55, 0xf5, 0x70, 0x98, 0x42, 0x0f, 0xba,
			0x32, 0x50, 0x95, 0x5b, 0xfc, 0x2f, 0x83, 0x2a,
			0x00, 0x0e, 0x61, 0x65, 0x16, 0xbb, 0xcf, 0xf5,
			0x10, 0x0c, 0x62, 0xf3, 0x35, 0x7d, 0xae, 0x6c,
			0xf1, 0x1a, 0xa4, 0x61, 0xca, 0xa0, 0x1f, 0x12,
			0x96, 0xdf, 0xb9, 0x7d, 0xeb, 0x62, 0xe3, 0x21,
			0xd8, 0x52, 0x35, 0x07, 0xd2, 0x98, 0x02, 0x03,
			0xf1, 0x42, 0x30, 0x38, 0x5d, 0x28, 0x46, 0xda,
		},
		/* S4 : 128 byte */
		{
			0x0d, 0xbb, 0x79, 0x54, 0xb4, 0x11, 0xac, 0x1c,
			0x42, 0x5d, 0x77, 0x88, 0x75, 0xde, 0xdd, 0xa9,
			0xad, 0xd8, 0x36, 0xf7, 0x28, 0x30, 0x40, 0x79,
			0x7b, 0x58, 0x99, 0xb9, 0x93, 0xe2, 0x8f, 0x8e,
			0x84, 0x39, 0x09, 0xc2, 0x8b, 0xaf, 0x41, 0x69,
			0xa0, 0x34, 0x4f, 0x5e, 0xed, 0x76, 0x9b, 0x22,
			0xc5, 0x67, 0xba, 0xb9, 0x7a, 0x18, 0x16, 0xa6,
			0x34, 0xd0, 0xfe, 0x0e, 0x78, 0x60, 0x67, 0x4c,
			0xc6, 0x27, 0x3e, 0xc2, 0x23, 0x2d, 0x01, 0xce,
			0x93, 0xea, 0x2c, 0xd0, 0x3f, 0x42, 0xce, 0xb4,
			0xc4, 0xbd, 0xa7, 0xa4, 0x0d, 0xc1, 0xa5, 0xc8,
			0x9a, 0x84, 0xae, 0x7a, 0xfd, 0xdb, 0x72, 0x99,
			0x56, 0xa8, 0x3f, 0xc6, 0x26, 0xa9, 0x9f, 0x4b,
			0xe5, 0xd5, 0xda, 0xf2, 0xef, 0x47, 0xf4, 0xec,
			0xca, 0xa1, 0x03, 0xac, 0xf9, 0x2f, 0xb5, 0xee,
			0x5e, 0x1b, 0x4f, 0x8c, 0xd8, 0xd4, 0x64, 0x8a,
		},
		20, /* salt length */
		/* V4 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA1 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0xca, 0x77, 0xe6, 0x62, 0x07, 0x7e, 0xd5, 0xcb,
			0x63, 0xdd, 0x9b, 0x10, 0x17, 0x7b, 0xd6, 0x60,
			0x62, 0x27, 0x68, 0x28, 0xd1, 0xcf, 0xd8, 0xe3,
			0xbd, 0xf2, 0xf6, 0x32, 0x15, 0x57, 0xbe, 0x2f,
			0x5e, 0x00, 0x7f, 0xfe, 0xf7, 0x5a, 0x3d, 0xa6,
			0xa2, 0xb6, 0x53, 0x1f, 0x40, 0xfc, 0x5c, 0xad,
			0xdc, 0x9e, 0x54, 0x6c, 0xeb, 0xcc, 0xfb, 0x2b,
			0xd2, 0xe7, 0x7a, 0x7d, 0xa9, 0x37, 0x49, 0x0e,
			0x3f, 0xe0, 0x50, 0x34, 0x5e, 0x12, 0x4d, 0xc6,
			0x50, 0x81, 0x75, 0x31, 0xa1, 0x15, 0x5e, 0xc8,
			0x27, 0xe2, 0x1c, 0x5c, 0x40, 0x5a, 0xab, 0x0d,
			0x5b, 0xc1, 0x8d, 0x5f, 0xba, 0xe2, 0xb3, 0x91,
			0x76, 0x01, 0xa1, 0x2a, 0xc3, 0x10, 0xce, 0xd4,
			0x75, 0x26, 0x49, 0x6d, 0xc3, 0x2c, 0xa2, 0x5c,
			0x4b, 0xda, 0x08, 0x21, 0x57, 0xe2, 0xaf, 0x30,
			0x5f, 0x6a, 0x75, 0x00, 0xcf, 0xc0, 0xc4, 0x5f,
		},
		/* S5 : 128 byte */
		{
			0x46, 0x2d, 0x92, 0xb7, 0xfb, 0xac, 0x49, 0xda,
			0xb2, 0xfe, 0xf1, 0x3c, 0xae, 0xaf, 0x9b, 0xd1,
			0x1b, 0xb5, 0x05, 0x3a, 0x8a, 0xef, 0x55, 0x4a,
			0x05, 0xe1, 0xab, 0x6c, 0xe0, 0x95, 0x67, 0x82,
			0x12, 0xb8, 0xe1, 0xfb, 0xea, 0x03, 0x96, 0xb1,
			0x8a, 0xae, 0x3d, 0x12, 0x75, 0xda, 0x85, 0x20,
			0xab, 0xc1, 0x23, 0xf8, 0x8c, 0x3b, 0x5c, 0x05,
			0x62, 0x71, 0xd4, 0x41, 0x56, 0x42, 0xca, 0x4a,
			0xbc, 0x20, 0x30, 0xde, 0x60, 0x0a, 0x48, 0x28,
			0x01, 0x15, 0x15, 0x1c, 0x84, 0xa7, 0xd9, 0xc7,
			0xf1, 0x77, 0x69, 0xe0, 0x92, 0x64, 0x50, 0xef,
			0xe3, 0xfc, 0xe6, 0x2f, 0xb6, 0x3d, 0xb5, 0x30,
			0x82, 0xa3, 0xed, 0x21, 0xf7, 0x29, 0x94, 0x33,
			0xae, 0xd0, 0x78, 0xcf, 0xb7, 0xbe, 0x3b, 0x48,
			0xbc, 0xd6, 0xc1, 0x3f, 0x11, 0xe3, 0xea, 0x0f,
			0x49, 0x46, 0xa1, 0x39, 0xa9, 0x41, 0x30, 0x3d,
		},
		20, /* salt length */
		/* V5 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA1 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0x63, 0x82, 0x60, 0xc0, 0xa5, 0x19, 0xc4, 0x5f,
			0xab, 0xa2, 0x99, 0xd0, 0xca, 0x2e, 0x80, 0xbf,
			0x50, 0xdf, 0xde, 0x8d, 0x6a, 0x42, 0xe0, 0x46,
			0x45, 0xdf, 0xbc, 0xd4, 0x74, 0x0f, 0x3a, 0x72,
			0x92, 0x0e, 0x74, 0x63, 0x28, 0x51, 0xd9, 0xe3,
			0xd0, 0x1a, 0x78, 0x5e, 0x9b, 0x49, 0x7c, 0xe0,
			0xb1, 0x75, 0xf2, 0xcd, 0x37, 0x3b, 0xd3, 0xd2,
			0x76, 0xd6, 0x3e, 0x1b, 0x39, 0xf0, 0x05, 0xc6,
			0x76, 0xb8, 0x6b, 0x98, 0x31, 0x35, 0x2c, 0xef,
			0x9e, 0xda, 0xbe, 0xf8, 0x86, 0x5a, 0xd7, 0x22,
			0xeb, 0xbe, 0x2f, 0xd3, 0xef, 0xb4, 0x87, 0x59,
			0xf2, 0x2a, 0xea, 0x23, 0xfb, 0x1b, 0x33, 0x31,
			0x59, 0xa9, 0xcf, 0xc9, 0x8a, 0x6d, 0xc4, 0x6c,
			0x5b, 0x0b, 0xf2, 0x08, 0x51, 0x09, 0x99, 0x6f,
			0x48, 0xe1, 0x1c, 0x08, 0x5a, 0x54, 0x05, 0xc4,
			0x02, 0x02, 0x48, 0x54, 0x81, 0x38, 0xcc, 0x9a,
		},
		/* S6 : 128 byte */
		{
			0x01, 0x5c, 0x92, 0x29, 0x38, 0xf6, 0x7d, 0xd5,
			0x70, 0xf9, 0x60, 0xab, 0x30, 0xa4, 0xfc, 0x72,
			0xb9, 0x3d, 0xf7, 0x6a, 0xa7, 0x69, 0x73, 0xc5,
			0x13, 0x05, 0x57, 0x93, 0x80, 0x7a, 0xec, 0xc3,
			0x14, 0xcb, 0x0e, 0xf1, 0xf7, 0xde, 0x40, 0x1e,
			0x60, 0x99, 0x04, 0xed, 0x20, 0xca, 0xb6, 0x42,
			0xba, 0x46, 0x64, 0xed, 0x8b, 0x9e, 0x58, 0x23,
			0x70, 0x5b, 0xb2, 0x7a, 0x20, 0x1c, 0xdd, 0xa9,
			0x45, 0x5f, 0xa9, 0x75, 0x2c, 0x69, 0x06, 0x7c,
			0x74, 0x73, 0x06, 0x4e, 0x0e, 0x39, 0x9e, 0x87,
			0xca, 0x5f, 0x9d, 0x28, 0x79, 0x8e, 0xe7, 0xbf,
			0xda, 0x18, 0x15, 0xc3, 0xc3, 0x0e, 0x83, 0xda,
			0x41, 0x1c, 0x33, 0x93, 0x80, 0x01, 0xe6, 0xce,
			0xbc, 0x38, 0x8b, 0xd1, 0x3e, 0xe2, 0x06, 0xea,
			0x93, 0xf9, 0xfd, 0x45, 0x8b, 0x1a, 0x83, 0x00,
			0xab, 0x8a, 0xaa, 0x79, 0xbd, 0x22, 0xd1, 0x82,
		},
		20, /* salt length */
		/* V6 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA1 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0xc6, 0xac, 0x68, 0x9a, 0xc5, 0x2f, 0xa1, 0xea,
			0x22, 0x76, 0xf7, 0xaa, 0xde, 0x8b, 0x1b, 0x54,
			0xdd, 0xd5, 0xaa, 0xa0, 0x47, 0xe9, 0xb2, 0xf9,
			0x2f, 0xa3, 0x73, 0x2d, 0x72, 0x76, 0x62, 0x68,
			0xe2, 0x00, 0x74, 0x63, 0xa4, 0xe1, 0x98, 0xd9,
			0xd7, 0xce, 0x85, 0x3b, 0x76, 0xe1, 0xb7, 0xa4,
			0x80, 0x33, 0xb4, 0x0f, 0x56, 0xad, 0x94, 0xc8,
			0x64, 0x1b, 0xc0, 0x20, 0x03, 0x82, 0xfa, 0x65,
			0x9e, 0xb4, 0x9e, 0x11, 0x14, 0x21, 0x27, 0x42,
			0x5e, 0xf0, 0xf4, 0x6a, 0x72, 0x4f, 0x4d, 0x48,
			0x67, 0x07, 0xa1, 0x17, 0xd9, 0x53, 0x11, 0x05,
			0x8f, 0xf2, 0x33, 0xe6, 0x5b, 0x7b, 0x0f, 0x2a,
			0x42, 0xf1, 0x6e, 0x0c, 0xdd, 0x97, 0x55, 0x0d,
			0xff, 0x05, 0xcc, 0xa1, 0x99, 0x7c, 0xe3, 0x28,
			0xda, 0x77, 0xf7, 0x1f, 0x9a, 0x82, 0x32, 0x98,
			0xf8, 0x4f, 0x51, 0xe7, 0x42, 0x07, 0xaa, 0x9f,
		},
		/* S7 : 128 byte */
		{
			0x8f, 0x7e, 0x7b, 0xfe, 0x45, 0x44, 0x8c, 0x20,
			0x3e, 0xd2, 0xb7, 0xb8, 0xb6, 0x8e, 0x3a, 0xab,
			0xef, 0x2c, 0xa3, 0xf7, 0xc2, 0x86, 0x10, 0x42,
			0xd5, 0x91, 0xfc, 0x96, 0x46, 0x49, 0xec, 0xda,
			0x06, 0xda, 0x1b, 0x0f, 0x77, 0xd2, 0x30, 0xc4,
			0x6b, 0x0d, 0x3f, 0xed, 0x02, 0xce, 0xaf, 0xf9,
			0x13, 0x36, 0xab, 0xd5, 0x7b, 0xe2, 0x95, 0xf4,
			0x16, 0x55, 0x27, 0xce, 0x85, 0x65, 0x56, 0x34,
			0xfc, 0x77, 0xe9, 0xd1, 0xb3, 0x4e, 0x0a, 0x82,
			0x6a, 0xa1, 0xc6, 0x87, 0x81, 0x8f, 0xa2, 0xe1,
			0xef, 0x31, 0x9e, 0xfb, 0x6d, 0x1d, 0xe1, 0x08,
			0xf1, 0x40, 0xa3, 0x0f, 0xe9, 0x42, 0xaa, 0xd2,
			0x5b, 0xca, 0xf2, 0xbc, 0x2a, 0x8c, 0xa4, 0xac,
			0x08, 0x3a, 0x10, 0xc1, 0x18, 0x8c, 0xad, 0x40,
			0xab, 0x72, 0x3b, 0x5f, 0xcf, 0x47, 0xcc, 0x70,
			0x9f, 0xef, 0x4b, 0xbc, 0x71, 0xfe, 0xd7, 0xb4,
		},
		20, /* salt length */
		/* V7 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA1 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0x75, 0x23, 0xcb, 0x2f, 0x9d, 0x75, 0xb6, 0x06,
			0x8f, 0x78, 0x5c, 0x44, 0x82, 0xdc, 0xbc, 0xbf,
			0x80, 0xd0, 0x5a, 0xb7, 0x5e, 0x22, 0xda, 0x14,
			0xf1, 0xa8, 0xcd, 0xf0, 0x7f, 0xde, 0xcb, 0x55,
			0x74, 0xd0, 0xc2, 0x48, 0x56, 0x3b, 0x8d, 0x06,
			0xfb, 0xef, 0xc9, 0x8d, 0xfa, 0x77, 0xe4, 0x5e,
			0x5f, 0xfc, 0x76, 0x82, 0xa3, 0x67, 0xd5, 0xcb,
			0xdb, 0xc6, 0xe4, 0x5c, 0x2f, 0xf1, 0x2b, 0x6b,
			0x8e, 0x6d, 0x2d, 0x46, 0xa0, 0x90, 0xf5, 0x94,
			0xfc, 0xe6, 0xf3, 0xde, 0x9c, 0xd4, 0x06, 0x4e,
			0x8d, 0x7b, 0xda, 0xb6, 0xeb, 0xdf, 0x74, 0xd3,
			0x0d, 0x49, 0x0b, 0xd4, 0xfd, 0xea, 0x1a, 0x16,
			0x2b, 0xc0, 0xb3, 0x32, 0x62, 0xbe, 0x16, 0x39,
			0xfa, 0x27, 0x82, 0x3e, 0x50, 0x3b, 0x4b, 0x15,
			0x73, 0x15, 0x2d, 0x5b, 0x20, 0xd4, 0xe0, 0xb6,
			0xf1, 0xf9, 0xec, 0x5e, 0xd1, 0x10, 0xbf, 0xdd,
		},
		/* S8 : 128 byte */
		{
			0x31, 0x52, 0x7f, 0x8b, 0x5b, 0x28, 0x9a, 0x67,
			0xac, 0xc9, 0xc5, 0x25, 0xee, 0xa4, 0x85, 0xb4,
			0x0b, 0xee, 0xed, 0xbb, 0x77, 0x7d, 0x68, 0x20,
			0x1c, 0xaa, 0x54, 0xeb, 0xc4, 0xa0, 0x8c, 0x15,
			0xb4, 0xbc, 0x5f, 0xe8, 0x5d, 0xe9, 0xf6, 0x95,
			0xf2, 0x1c, 0x95, 0xb3, 0x61, 0x7f, 0xc4, 0xc3,
			0xe3, 0xf8, 0xaa, 0x87, 0x2f, 0x5e, 0xbf, 0xbf,
			0x2a, 0x4f, 0x61, 0x7f, 0xa3, 0x9b, 0xb3, 0x95,
			0x90, 0x76, 0xf3, 0xe6, 0x1d, 0x41, 0x00, 0x5e,
			0xe8, 0x30, 0x6c, 0x8d, 0x8d, 0xf6, 0xf4, 0xcb,
			0xa0, 0xd2, 0x74, 0x30, 0xd9, 0x81, 0xe8, 0x41,
			0xe2, 0xdd, 0x9a, 0x5d, 0xb0, 0x1c, 0xd6, 0xf4,
			0x81, 0xe5, 0xce, 0xdc, 0x32, 0x55, 0xd9, 0x4e,
			0xab, 0x10, 0x39, 0x4e, 0xfc, 0xfd, 0xb1, 0x6e,
			0xe9, 0x0d, 0x25, 0xd0, 0x2c, 0x37, 0xad, 0x41,
			0xb0, 0x70, 0x8e, 0x91, 0x6f, 0x5a, 0xe0, 0xb5,
		},
		20, /* salt length */
		/* V8 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA1 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0x03, 0x20, 0xf0, 0x3a, 0x9c, 0x68, 0x1f, 0xd9,
			0x7d, 0xcc, 0x1a, 0xb0, 0xdc, 0x3b, 0x66, 0x42,
			0x07, 0x5d, 0x59, 0x9a, 0xa5, 0xc9, 0x7f, 0x28,
			0x3f, 0x79, 0x3d, 0xcb, 0xff, 0x7a, 0xb4, 0x2f,
			0xb7, 0xe8, 0xe8, 0x59, 0x24, 0x1a, 0x22, 0x80,
			0x75, 0x1a, 0x93, 0x1f, 0x07, 0x58, 0x39, 0x48,
			0x73, 0x9f, 0x07, 0x3b, 0xca, 0x47, 0x40, 0x37,
			0x1f, 0xad, 0x24, 0x90, 0x46, 0xe0, 0xce, 0x02,
			0xdc, 0x9d, 0x5d, 0x45, 0x87, 0x7e, 0x52, 0xe1,
			0x7f, 0x6e, 0x39, 0x44, 0x4e, 0x60, 0x8c, 0x12,
			0xd4, 0x41, 0xd4, 0xbc, 0x8d, 0x2a, 0x11, 0x54,
			0x17, 0xdb, 0x5c, 0xa0, 0xee, 0x65, 0xcb, 0x6d,
			0x7b, 0x26, 0x95, 0x25, 0x4d, 0xfa, 0x76, 0xa4,
			0xa6, 0xb0, 0x55, 0x48, 0x39, 0xbb, 0x25, 0x47,
			0x32, 0x2a, 0x09, 0x46, 0x77, 0xdb, 0xbb, 0x27,
			0x2d, 0xeb, 0x2e, 0x22, 0x7f, 0x70, 0x72, 0x17,
		},
		/* S9 : 128 byte */
		{
			0x82, 0x31, 0x56, 0x70, 0x2a, 0xc0, 0xca, 0xa2,
			0x7f, 0xc3, 0x46, 0x1b, 0x7a, 0x2c, 0x0d, 0x71,
			0x41, 0x1e, 0x5c, 0x45, 0x70, 0x81, 0x84, 0xb2,
			0x69, 0x75, 0xfa, 0xee, 0x55, 0x51, 0xc3, 0x85,
			0x4a, 0x9a, 0xd6, 0x8f, 0x9c, 0x13, 0x27, 0xc4,
			0x3c, 0xcf, 0x46, 0xd2, 0x29, 0xa4, 0x43, 0x45,
			0xb0, 0xa0, 0x8d, 0x11, 0xff, 0xdb, 0x6f, 0x61,
			0x80, 0xa2, 0x40, 0xac, 0x68, 0xc8, 0x95, 0xda,
			0xae, 0x10, 0x86, 0xd6, 0x0e, 0x95, 0xe1, 0xf3,
			0xdb, 0x39, 0x31, 0xee, 0x36, 0xa4, 0xf7, 0x51,
			0xde, 0x56, 0x99, 0xfa, 0x53, 0x74, 0xd1, 0x95,
			0x9b, 0x78, 0xde, 0x0f, 0x47, 0x9a, 0xb0, 0x70,
			0xc2, 0x20, 0x25, 0x68, 0x03, 0x86, 0x4a, 0xa1,
			0x23, 0x02, 0xde, 0xa9, 0x6a, 0xa0, 0x73, 0x2a,
			0xc4, 0xc4, 0x0d, 0x79, 0x9b, 0xd3, 0x06, 0x08,
			0x7b, 0xd4, 0x3e, 0xf6, 0xb3, 0x68, 0x68, 0xf0,
		},
		20, /* salt length */
		/* V9 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
};

/* 10 elements */
static mod1024_t SHA224[] = {
	/* SHA224 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0x12, 0x48, 0xf6, 0x2a, 0x43, 0x89, 0xf4, 0x2f,
			0x7b, 0x4b, 0xb1, 0x31, 0x05, 0x3d, 0x6c, 0x88,
			0xa9, 0x94, 0xdb, 0x20, 0x75, 0xb9, 0x12, 0xcc,
			0xbe, 0x3e, 0xa7, 0xdc, 0x61, 0x17, 0x14, 0xf1,
			0x4e, 0x07, 0x5c, 0x10, 0x48, 0x58, 0xf2, 0xf6,
			0xe6, 0xcf, 0xd6, 0xab, 0xde, 0xdf, 0x01, 0x5a,
			0x82, 0x1d, 0x03, 0x60, 0x8b, 0xf4, 0xeb, 0xa3,
			0x16, 0x9a, 0x67, 0x25, 0xec, 0x42, 0x2c, 0xd9,
			0x06, 0x94, 0x98, 0xb5, 0x51, 0x5a, 0x96, 0x08,
			0xae, 0x7c, 0xc3, 0x0e, 0x3d, 0x2e, 0xcf, 0xc1,
			0xdb, 0x68, 0x25, 0xf3, 0xe9, 0x96, 0xce, 0x9a,
			0x50, 0x92, 0x92, 0x6b, 0xc1, 0xcf, 0x61, 0xaa,
			0x42, 0xd7, 0xf2, 0x40, 0xe6, 0xf7, 0xaa, 0x0e,
			0xdb, 0x38, 0xbf, 0x81, 0xaa, 0x92, 0x9d, 0x66,
			0xbb, 0x5d, 0x89, 0x00, 0x18, 0x08, 0x84, 0x58,
			0x72, 0x0d, 0x72, 0xd5, 0x69, 0x24, 0x7b, 0x0c,
		},
		/* S0 : 128 byte */
		{
			0x53, 0xd8, 0x59, 0xc9, 0xf1, 0x0a, 0xbf, 0x1c,
			0x00, 0x28, 0x4a, 0x4b, 0x55, 0xbf, 0x2b, 0xd8,
			0x4d, 0x8e, 0x31, 0x3b, 0x4f, 0x3c, 0x35, 0xb8,
			0xde, 0xc7, 0xbc, 0x3a, 0xfe, 0x39, 0xb9, 0xb8,
			0xa1, 0x55, 0x41, 0x8e, 0xad, 0x19, 0x31, 0x89,
			0x57, 0x69, 0xce, 0x23, 0x40, 0xbe, 0x20, 0x91,
			0xf2, 0x38, 0x5b, 0xbc, 0xf1, 0x0d, 0x9e, 0x92,
			0xbc, 0xf5, 0xd0, 0xe2, 0x96, 0x0d, 0x10, 0xe7,
			0x92, 0xe7, 0xd8, 0x65, 0xc6, 0x4e, 0x50, 0xd1,
			0x9f, 0xfa, 0x13, 0xe5, 0x28, 0x17, 0xd7, 0xd8,
			0xd8, 0xdb, 0x34, 0x39, 0x2c, 0x23, 0x74, 0xa2,
			0xe9, 0xb6, 0x91, 0x84, 0xf9, 0x2a, 0x4a, 0xd9,
			0xb1, 0xb8, 0xba, 0xe9, 0x9c, 0xa6, 0x14, 0xd2,
			0x04, 0xb6, 0x5a, 0x43, 0x8e, 0x38, 0xdb, 0xbf,
			0xc8, 0xc7, 0xcc, 0x44, 0xed, 0x56, 0x77, 0xaf,
			0x70, 0xce, 0x6c, 0x4f, 0x95, 0x1f, 0x02, 0x44,
		},
		20, /* salt length */
		/* V0 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA224 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 0x2b, 0x90, 0x75, 0x7b,
			0x77, 0xc7, 0xc2, 0x01, 0xe0, 0xdf, 0xbe, 0x7b,
			0x68, 0xa3, 0x9c, 0x9f, 0x01, 0x0c, 0x04, 0xc9,
			0xa4, 0x04, 0xc0, 0xe5, 0x9d, 0x6f, 0xb8, 0x8f,
			0x29, 0x6c, 0x20, 0x79, 0x42, 0xd8, 0xf0, 0x0d,
			0xba, 0x15, 0x94, 0x54, 0x87, 0x09, 0xe5, 0x17,
			0x5e, 0x69, 0x59, 0x44, 0x83, 0x37, 0x76, 0x92,
			0xdf, 0x89, 0x8d, 0x68, 0x53, 0x8d, 0xa4, 0xfc,
			0x4c, 0xca, 0xb4, 0xb8, 0x97, 0xa8, 0x15, 0xe1,
			0x77, 0x33, 0x36, 0x7a, 0xf5, 0x1c, 0x91, 0x66,
			0x78, 0x00, 0xde, 0xce, 0x95, 0xed, 0x87, 0xc2,
			0x28, 0x23, 0x5d, 0x23, 0xa4, 0x17, 0x89, 0xc0,
			0xa7, 0xbe, 0xc5, 0xbf, 0xd2, 0x07, 0xcc, 0x3f,
			0xd7, 0xaa, 0x0e, 0x3b, 0xd3, 0x20, 0xa9, 0xb1,
		},
		/* S1 : 128 byte */
		{
			0x15, 0xf9, 0x61, 0x6e, 0xed, 0x46, 0x47, 0x77,
			0x65, 0x5b, 0x7b, 0x6a, 0x00, 0xfc, 0x8e, 0x3d,
			0x18, 0x6d, 0x60, 0xa1, 0x43, 0xa8, 0x41, 0x1d,
			0x89, 0x15, 0x39, 0xb8, 0x86, 0x23, 0x2b, 0xc4,
			0xcc, 0x9e, 0x0d, 0x71, 0xff, 0x8d, 0xf4, 0x82,
			0x8b, 0x32, 0xea, 0x9a, 0xfe, 0xcc, 0xa3, 0xad,
			0xc5, 0x7c, 0x2a, 0xc6, 0xd3, 0x55, 0x44, 0x42,
			0x01, 0xd2, 0xad, 0x5a, 0x13, 0x85, 0xae, 0x40,
			0x3e, 0x2f, 0xae, 0xba, 0xc1, 0x11, 0xb0, 0xc3,
			0xf6, 0x3a, 0x0b, 0xf7, 0xbb, 0xa9, 0x27, 0xf5,
			0x37, 0xfa, 0x30, 0x1c, 0xed, 0x57, 0xe2, 0x74,
			0x83, 0x4d, 0x62, 0xc6, 0xe4, 0xd0, 0x54, 0x03,
			0x5f, 0x77, 0x7e, 0xb7, 0xdb, 0x6a, 0x97, 0xbe,
			0x38, 0x96, 0x95, 0xd1, 0x78, 0x5f, 0x9a, 0x9a,
			0xa0, 0x2d, 0x8d, 0x36, 0x80, 0xdd, 0x91, 0x1b,
			0xc1, 0xbf, 0x83, 0xd3, 0x10, 0xcc, 0x8a, 0x8d,
		},
		20, /* salt length */
		/* V1 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA224 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x46, 0xd0, 0xb4, 0xb0, 0x35, 0xe9, 0xf1, 0xa1,
			0xa9, 0xfc, 0xb1, 0xb7, 0xa5, 0x8a, 0xc1, 0x98,
			0x0a, 0x81, 0xa6, 0xc9, 0x18, 0xb3, 0x3c, 0xc4,
			0x32, 0x87, 0x9d, 0x4b, 0xe1, 0xee, 0x44, 0xfb,
			0x1c, 0x99, 0x73, 0x24, 0xc7, 0x6a, 0xd1, 0xd8,
			0x79, 0x28, 0x58, 0x42, 0x82, 0x76, 0x6e, 0x39,
			0x2d, 0x06, 0x52, 0x15, 0x42, 0x0a, 0xba, 0xc2,
			0xcf, 0xeb, 0xbb, 0x13, 0x28, 0x3e, 0xe8, 0x58,
			0x2d, 0xf4, 0xbe, 0x32, 0xcb, 0xd0, 0x40, 0xa6,
			0xc6, 0x1f, 0xde, 0x71, 0xb5, 0xa3, 0x9b, 0x9b,
			0x4e, 0xd0, 0x6f, 0x4f, 0xe3, 0x3b, 0xea, 0xea,
			0x2d, 0x4f, 0x1b, 0x52, 0x49, 0x41, 0xb0, 0x89,
			0x00, 0x47, 0x5e, 0x82, 0x4b, 0x05, 0x84, 0x33,
			0x16, 0x48, 0x0b, 0xeb, 0x45, 0xf7, 0x90, 0xe6,
			0xf5, 0x47, 0xc5, 0x1f, 0x04, 0x2d, 0x14, 0x65,
			0xd3, 0x19, 0x86, 0xb1, 0x4a, 0xe1, 0xe4, 0xb7,
		},
		/* S2 : 128 byte */
		{
			0x73, 0x8c, 0xb4, 0x61, 0x4a, 0xb1, 0xc9, 0x3f,
			0xe6, 0x9e, 0x9b, 0x2f, 0xa1, 0x28, 0x12, 0x0e,
			0xbf, 0x13, 0xb5, 0xa1, 0x63, 0x84, 0xc7, 0xa9,
			0x73, 0x25, 0xb0, 0x1c, 0xb6, 0xa5, 0x79, 0xa1,
			0x3a, 0xa8, 0xb6, 0xa5, 0xe6, 0x5d, 0x4e, 0x52,
			0x21, 0xc6, 0x70, 0xf5, 0x9d, 0xca, 0xd1, 0x15,
			0xdc, 0xb9, 0x82, 0xfe, 0x79, 0x6c, 0xee, 0x8a,
			0x8a, 0x15, 0x0a, 0xbf, 0x3b, 0x62, 0xdd, 0x28,
			0xb5, 0xcd, 0xd7, 0xee, 0x85, 0x62, 0xd9, 0x2c,
			0x1d, 0xc0, 0x44, 0xde, 0xa8, 0xd6, 0x9d, 0x64,
			0x11, 0x6e, 0x70, 0xb9, 0x92, 0xe1, 0x90, 0x0d,
			0x99, 0x11, 0x0a, 0x98, 0x73, 0x83, 0x3f, 0xc6,
			0x89, 0xf3, 0x90, 0x2f, 0x8f, 0x93, 0x9a, 0x51,
			0x18, 0x68, 0x94, 0x52, 0xd5, 0xdf, 0x06, 0xa9,
			0x72, 0x92, 0x73, 0x5f, 0xac, 0x20, 0x3b, 0xa0,
			0x29, 0x6d, 0x1d, 0xb2, 0xb6, 0x3b, 0x10, 0x8c,
		},
		20, /* salt length */
		/* V2 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA224 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0x5e, 0x61, 0x14, 0x73, 0xdd, 0x3c, 0xb9, 0x22,
			0x38, 0x30, 0x0e, 0xd5, 0x4a, 0xbd, 0x60, 0x36,
			0x62, 0x04, 0x1b, 0x92, 0xe9, 0xdc, 0x8d, 0x8d,
			0x45, 0x23, 0xec, 0x15, 0xfc, 0x52, 0x9b, 0x94,
			0x1d, 0x54, 0xa9, 0x6a, 0xdc, 0x99, 0x9b, 0x9a,
			0x7c, 0x66, 0x6b, 0xc3, 0x72, 0x6f, 0xd0, 0x53,
			0xc7, 0xa7, 0xc0, 0xf7, 0xbe, 0x57, 0x33, 0x56,
			0xb7, 0x6d, 0x2b, 0xfe, 0x53, 0x17, 0xf1, 0x9e,
			0xd9, 0x91, 0xa1, 0x77, 0xd8, 0x3c, 0xe8, 0x0d,
			0x0e, 0xef, 0x0d, 0x22, 0x89, 0x91, 0x2e, 0xe4,
			0x0c, 0xe8, 0xcd, 0x66, 0xb4, 0xac, 0x8b, 0x4d,
			0xdb, 0xe0, 0x32, 0xcc, 0x3d, 0x62, 0xf9, 0xd2,
			0x59, 0xc0, 0x04, 0x81, 0x1b, 0x4d, 0x2b, 0xe3,
			0xb7, 0x74, 0xfc, 0xd8, 0xd8, 0x4d, 0x03, 0x53,
			0xfa, 0x6e, 0x49, 0xc6, 0x14, 0x04, 0x1a, 0xdb,
			0x7b, 0x22, 0x0b, 0x65, 0x03, 0x58, 0x3c, 0x96,
		},
		/* S3 : 128 byte */
		{
			0x49, 0x1c, 0x76, 0x78, 0x11, 0xe8, 0x7e, 0x17,
			0x8d, 0x39, 0x9a, 0x0a, 0x21, 0xbe, 0xb7, 0x64,
			0xd6, 0xad, 0xd2, 0x5a, 0x79, 0x8c, 0x5b, 0xb0,
			0xfa, 0x58, 0x59, 0x7f, 0x11, 0x64, 0x8c, 0x92,
			0xbd, 0x60, 0x20, 0xa2, 0x7d, 0xc3, 0xae, 0x77,
			0xd3, 0xb3, 0x33, 0xbe, 0x41, 0xd7, 0x27, 0x9f,
			0xf8, 0xc2, 0x50, 0x53, 0xea, 0x86, 0x2b, 0xcc,
			0xec, 0x00, 0x42, 0x95, 0xe9, 0xd9, 0x45, 0x57,
			0xd0, 0x84, 0xf3, 0x33, 0x3b, 0x0a, 0x19, 0x9c,
			0x5e, 0x02, 0x3f, 0xe0, 0xf2, 0x55, 0x6d, 0xa9,
			0x6d, 0xd7, 0xc8, 0x56, 0x76, 0x06, 0x1e, 0x6b,
			0x96, 0x63, 0x47, 0x3e, 0xbe, 0x04, 0x56, 0xc6,
			0xd3, 0xd6, 0x32, 0xf0, 0x89, 0xc9, 0x72, 0xa1,
			0x0c, 0x60, 0xcc, 0x29, 0xca, 0xbe, 0x01, 0x1f,
			0x81, 0x43, 0xce, 0x1d, 0x0c, 0x15, 0xbb, 0x91,
			0x87, 0xb2, 0xbe, 0xe5, 0x5d, 0xc3, 0xbe, 0xd0,
		},
		20, /* salt length */
		/* V3 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA224 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0xf1, 0x4b, 0x4e, 0x3f, 0xe8, 0x2b, 0x9c, 0xcd,
			0x1d, 0x37, 0xb5, 0x05, 0x28, 0x37, 0x97, 0x01,
			0x95, 0x15, 0xcd, 0x2d, 0xe7, 0xd9, 0x93, 0xe6,
			0xdc, 0xd7, 0x3a, 0x47, 0xf3, 0x27, 0xce, 0x3f,
			0x2c, 0x56, 0xf7, 0xe7, 0x0f, 0x5b, 0x1d, 0x13,
			0xef, 0xcd, 0xcf, 0x28, 0x70, 0x53, 0x66, 0x2d,
			0x47, 0x5b, 0xe5, 0x24, 0x41, 0x0f, 0x24, 0x9a,
			0x87, 0x84, 0x2e, 0x7d, 0x20, 0x35, 0xe7, 0x2e,
			0xb7, 0xb1, 0xf2, 0xd7, 0x9f, 0x90, 0xd1, 0xff,
			0x14, 0xab, 0x4e, 0x5b, 0xc4, 0x8d, 0x1a, 0xe7,
			0x8e, 0x24, 0xb5, 0x36, 0x89, 0xbc, 0x8d, 0x06,
			0x48, 0x2f, 0x6a, 0x18, 0x5c, 0x55, 0x09, 0x3f,
			0x1b, 0xc2, 0x07, 0xb6, 0xa0, 0xb1, 0x02, 0xb6,
			0x13, 0x3b, 0xfa, 0x48, 0x2b, 0xcd, 0xfd, 0x59,
			0xf0, 0xd8, 0x03, 0x0c, 0xc6, 0xca, 0x17, 0x54,
			0xa7, 0x3e, 0xb7, 0xc1, 0xb0, 0x6f, 0x7e, 0x9a,
		},
		/* S4 : 128 byte */
		{
			0x41, 0xeb, 0x90, 0x61, 0x20, 0x12, 0xfb, 0xa3,
			0xf0, 0x61, 0x22, 0xfb, 0x0c, 0x20, 0xe1, 0x6b,
			0x0f, 0x4d, 0x4a, 0xca, 0xe6, 0x97, 0x24, 0x57,
			0xff, 0x25, 0xbd, 0xcb, 0x70, 0xf4, 0x16, 0x3b,
			0x6a, 0xa6, 0x8f, 0x8d, 0x39, 0x39, 0x57, 0x40,
			0x55, 0x40, 0xda, 0xee, 0xc3, 0x68, 0x06, 0x1d,
			0x40, 0x09, 0x58, 0xb6, 0x33, 0x5d, 0xaf, 0x9f,
			0x4b, 0xe5, 0xa1, 0x1b, 0x1c, 0xa5, 0xa9, 0x50,
			0xba, 0x7d, 0x9d, 0xe6, 0xfa, 0xf4, 0xb7, 0x19,
			0x57, 0xcc, 0x55, 0x09, 0x2f, 0xb1, 0xda, 0x8c,
			0x51, 0x35, 0x28, 0x14, 0xb3, 0xed, 0xed, 0x56,
			0xb4, 0x8e, 0xd2, 0xe8, 0x13, 0x2b, 0x2d, 0x44,
			0x2a, 0x57, 0xa1, 0xe8, 0x33, 0xc0, 0x9e, 0xa2,
			0x9f, 0x0d, 0x2f, 0x29, 0x18, 0x57, 0x63, 0xb5,
			0x29, 0xfa, 0x7b, 0xbb, 0x45, 0xfb, 0x83, 0x17,
			0xd3, 0x09, 0xfc, 0x97, 0x0c, 0x7f, 0x4d, 0xd5,
		},
		20, /* salt length */
		/* V4 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA224 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0xe5, 0x11, 0x90, 0x3c, 0x2f, 0x1b, 0xfb, 0xa2,
			0x45, 0x46, 0x72, 0x95, 0xac, 0x95, 0x41, 0x3a,
			0xc4, 0x74, 0x6c, 0x98, 0x4c, 0x37, 0x50, 0xa7,
			0x28, 0xc3, 0x88, 0xaa, 0x62, 0x8b, 0x0e, 0xbf,
			0x70, 0xbf, 0xa2, 0xb5, 0xb5, 0xbf, 0x70, 0x1d,
			0x4a, 0xd6, 0xed, 0xa4, 0xfd, 0x25, 0x02, 0xa0,
			0x9a, 0xdd, 0x6c, 0xe8, 0x6a, 0xb2, 0x63, 0xe9,
			0xfa, 0x8c, 0x79, 0xf6, 0x1d, 0x7f, 0xc6, 0xe0,
			0x34, 0xfd, 0x42, 0xc7, 0xad, 0x4a, 0x72, 0x2f,
			0xca, 0x31, 0x47, 0x56, 0xa4, 0xf6, 0x43, 0xc6,
			0x6f, 0x8b, 0xdd, 0x26, 0xff, 0x06, 0x27, 0x55,
			0x8b, 0x53, 0xae, 0xba, 0xb2, 0x26, 0xa1, 0xd5,
			0xbb, 0x34, 0x36, 0x1a, 0x21, 0xa3, 0x4a, 0x00,
			0x4f, 0xbf, 0x48, 0x55, 0xa8, 0xee, 0x4b, 0xd4,
			0xca, 0xe6, 0x87, 0xf9, 0x14, 0x1d, 0xe5, 0x14,
			0x66, 0x81, 0xed, 0x9d, 0x27, 0x6a, 0xe9, 0xc7,
		},
		/* S5 : 128 byte */
		{
			0x48, 0x59, 0x82, 0x69, 0x51, 0x99, 0x18, 0x95,
			0xd8, 0x03, 0xbf, 0xda, 0xee, 0xe2, 0x26, 0xf3,
			0xff, 0x3d, 0xae, 0xf1, 0x26, 0xbf, 0x30, 0xf7,
			0xde, 0xc1, 0x2e, 0x6a, 0xd3, 0x16, 0xe7, 0xa1,
			0x93, 0x86, 0xa9, 0xc3, 0x92, 0x09, 0x81, 0xbc,
			0x0c, 0xdf, 0x19, 0x21, 0x2e, 0x04, 0x20, 0x81,
			0x89, 0x87, 0x99, 0xc5, 0x01, 0xa3, 0xca, 0x7f,
			0x2e, 0x93, 0x88, 0xe6, 0x42, 0xcc, 0xcf, 0x5f,
			0x54, 0x5e, 0x5f, 0x33, 0xca, 0x10, 0xe7, 0x1f,
			0xcd, 0x3b, 0x8f, 0xa1, 0x0f, 0xe1, 0x3e, 0x2b,
			0x45, 0x84, 0xaf, 0x52, 0xa1, 0x2c, 0x72, 0x96,
			0x1c, 0x87, 0x42, 0xfe, 0xef, 0x96, 0xde, 0xec,
			0x26, 0xf3, 0x8d, 0x8a, 0x5c, 0x35, 0xa5, 0x6b,
			0x51, 0x16, 0xd3, 0x7f, 0x91, 0x55, 0xb0, 0x3c,
			0x64, 0x25, 0x4a, 0x8e, 0xa3, 0xee, 0x27, 0x74,
			0x8c, 0x82, 0x38, 0xc5, 0x2b, 0xa3, 0x7c, 0xa1,
		},
		20, /* salt length */
		/* V5 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA224 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0x10, 0x34, 0xe0, 0x43, 0xd5, 0x21, 0xa2, 0x9e,
			0x1f, 0x6d, 0x84, 0xef, 0x8a, 0x54, 0x9b, 0x86,
			0x75, 0xc5, 0xeb, 0xe4, 0xab, 0x74, 0x2e, 0x72,
			0x43, 0x2b, 0xd1, 0x72, 0xc4, 0x60, 0x40, 0x35,
			0x8d, 0x0d, 0x63, 0x8e, 0x74, 0xbc, 0x68, 0x88,
			0xae, 0xe1, 0xc4, 0x54, 0xef, 0x6d, 0x74, 0x85,
			0xd1, 0xa0, 0x89, 0x80, 0x7a, 0x1f, 0xe3, 0x2d,
			0x79, 0x46, 0x3d, 0xf3, 0xf2, 0xbf, 0x6a, 0xe4,
			0xfb, 0x54, 0xdd, 0x34, 0x45, 0x01, 0x6f, 0xf0,
			0xff, 0x4e, 0x5d, 0xec, 0xf7, 0xdc, 0x90, 0xfe,
			0x03, 0x3d, 0x22, 0x69, 0x9f, 0x08, 0x1d, 0x9c,
			0xd7, 0x2d, 0x86, 0x1d, 0x68, 0x87, 0x9a, 0xc0,
			0x22, 0xb0, 0x57, 0x1c, 0x90, 0x19, 0x3e, 0x3f,
			0xdb, 0x5a, 0x58, 0xa1, 0xec, 0xc6, 0x77, 0x57,
			0xd1, 0x43, 0xbe, 0x89, 0x62, 0x59, 0x63, 0x26,
			0x24, 0x88, 0xb4, 0xd6, 0x6d, 0xdf, 0x58, 0x31,
		},
		/* S6 : 128 byte */
		{
			0x0e, 0x6c, 0x36, 0x3d, 0xa4, 0x90, 0x38, 0x1a,
			0x14, 0x0c, 0x8b, 0x25, 0xe7, 0xa7, 0xe4, 0xe7,
			0x10, 0xcc, 0x61, 0x98, 0x4e, 0xa1, 0x09, 0x23,
			0x10, 0x1b, 0xe5, 0xe7, 0x59, 0x8b, 0x3f, 0xef,
			0x8a, 0xd9, 0x5a, 0x24, 0x85, 0x7f, 0xbd, 0x31,
			0x5d, 0x5e, 0x42, 0x61, 0x15, 0x99, 0xd2, 0xdd,
			0x8d, 0xf6, 0xdf, 0xa1, 0xcf, 0xcf, 0x88, 0x1b,
			0x9c, 0x8e, 0x0a, 0x59, 0x3a, 0x00, 0xcf, 0xc5,
			0xb9, 0x91, 0x30, 0xcf, 0xf4, 0x54, 0x59, 0x57,
			0x9a, 0xa5, 0x2f, 0xfb, 0xf5, 0x7b, 0x14, 0x74,
			0x4e, 0xce, 0x6a, 0xce, 0x55, 0x94, 0x70, 0xcc,
			0x59, 0x14, 0x0a, 0x09, 0x86, 0x69, 0x5d, 0x27,
			0x49, 0x83, 0x40, 0xb6, 0x3a, 0x24, 0xa0, 0xa5,
			0x3a, 0x33, 0xb0, 0x6e, 0x6c, 0xfb, 0x68, 0x18,
			0x7d, 0xf6, 0xa1, 0x27, 0x4b, 0x8d, 0x1a, 0xa0,
			0xa8, 0x52, 0xef, 0xfc, 0x28, 0x10, 0xee, 0x33,
		},
		20, /* salt length */
		/* V6 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA224 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0x12, 0x67, 0xd9, 0xe2, 0x9f, 0x0f, 0x39, 0xfc,
			0xe1, 0xd1, 0x92, 0x4f, 0x0c, 0x2a, 0x43, 0x51,
			0x09, 0x68, 0x2b, 0xf0, 0x60, 0x03, 0xb4, 0x84,
			0x5b, 0x30, 0x10, 0x02, 0x69, 0xb4, 0xc3, 0x72,
			0xc0, 0xf6, 0x62, 0xa3, 0x46, 0x59, 0x9a, 0x1d,
			0xb2, 0xc8, 0x73, 0x57, 0xee, 0x93, 0xd6, 0x2a,
			0xc0, 0x79, 0x4e, 0x86, 0xaa, 0x06, 0x0b, 0x64,
			0x0f, 0x1f, 0x2c, 0xfb, 0x96, 0xdb, 0x3e, 0x37,
			0xd4, 0x94, 0xff, 0x67, 0x25, 0x3e, 0x68, 0x36,
			0xda, 0xf8, 0xeb, 0xdc, 0x9a, 0xe1, 0xfa, 0x95,
			0x02, 0x2d, 0xc8, 0x52, 0x8f, 0x78, 0x53, 0xaf,
			0xbe, 0x59, 0xa2, 0x26, 0x71, 0x3b, 0x4b, 0x82,
			0x95, 0x68, 0x39, 0x96, 0x02, 0x67, 0xac, 0x2e,
			0xa2, 0x88, 0x81, 0x46, 0xd6, 0xbd, 0xb4, 0x7a,
			0x15, 0xab, 0x23, 0xbe, 0xd7, 0x01, 0x95, 0x4f,
			0xb1, 0x09, 0xfa, 0xea, 0xc1, 0x7d, 0xf5, 0x3b,
		},
		/* S7 : 128 byte */
		{
			0x83, 0x97, 0x8d, 0xb0, 0x27, 0x93, 0x2e, 0x2d,
			0x67, 0x62, 0x06, 0xc1, 0xb9, 0xfe, 0x63, 0x7f,
			0x0a, 0x13, 0x7d, 0x11, 0x50, 0x55, 0x23, 0x0f,
			0xd3, 0xb2, 0x17, 0x7e, 0x6f, 0xd1, 0xd7, 0x5f,
			0x7e, 0x54, 0x2f, 0xb5, 0x2c, 0x6f, 0x3e, 0x81,
			0x29, 0x86, 0x40, 0x53, 0x4f, 0x36, 0x4c, 0x8b,
			0x63, 0x6a, 0xf0, 0xc3, 0x61, 0xb2, 0xd2, 0x3b,
			0xba, 0xd9, 0x0a, 0x00, 0x1e, 0xfc, 0x8c, 0x0f,
			0x08, 0x51, 0xc8, 0x0e, 0x76, 0x5e, 0xb0, 0xeb,
			0x17, 0x91, 0x2c, 0xb3, 0x25, 0xf2, 0x36, 0x54,
			0x08, 0x73, 0xc9, 0xf2, 0x95, 0xed, 0x37, 0x25,
			0x35, 0x1a, 0x4b, 0x4b, 0x13, 0x50, 0xf4, 0x9c,
			0xc1, 0x8b, 0x30, 0x8c, 0x58, 0x54, 0xc8, 0x27,
			0x2c, 0x19, 0xb0, 0xf2, 0xdc, 0x11, 0x65, 0x47,
			0x5e, 0xaf, 0xaf, 0x60, 0x89, 0x53, 0x9b, 0xaa,
			0xad, 0x87, 0x9f, 0x4c, 0x46, 0xe3, 0xcd, 0x94,
		},
		20, /* salt length */
		/* V7 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA224 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0xb0, 0x66, 0x48, 0x83, 0x5d, 0x99, 0x3a, 0x4c,
			0x66, 0x94, 0x4a, 0x5c, 0xb9, 0x5a, 0xfd, 0x74,
			0x1c, 0x66, 0x70, 0x8b, 0xc4, 0xf0, 0x92, 0x28,
			0x23, 0x6d, 0x91, 0x10, 0x01, 0x55, 0x92, 0x21,
			0xce, 0x28, 0xa5, 0x7e, 0x37, 0xb3, 0xf8, 0xec,
			0xf7, 0x90, 0x4b, 0x7c, 0xf0, 0xec, 0xa3, 0x6b,
			0x7a, 0x8c, 0x43, 0x46, 0xfb, 0x12, 0xd8, 0x51,
			0xb8, 0x7d, 0xf4, 0x2a, 0xea, 0xf0, 0xbd, 0xaa,
			0x15, 0xb4, 0xe7, 0x0c, 0x95, 0x7f, 0xdf, 0x51,
			0x79, 0xf7, 0x49, 0xe2, 0x95, 0x72, 0xad, 0x78,
			0xd3, 0x00, 0x6d, 0x39, 0xca, 0xab, 0xfb, 0x24,
			0x92, 0xfd, 0x48, 0xae, 0xd6, 0xc3, 0x80, 0xad,
			0x2a, 0x11, 0xf4, 0x76, 0x9f, 0x86, 0x58, 0x42,
			0x95, 0xd1, 0x2e, 0xd7, 0xd1, 0x74, 0x82, 0x64,
			0xd0, 0xc8, 0xd8, 0xac, 0x5a, 0x42, 0x63, 0x66,
			0x5a, 0xf4, 0x77, 0xe4, 0xec, 0x57, 0x41, 0xf4,
		},
		/* S8 : 128 byte */
		{
			0x50, 0x62, 0xdd, 0xc2, 0x96, 0x21, 0x09, 0x72,
			0xb9, 0x0e, 0x88, 0xcf, 0xf2, 0x15, 0x59, 0x34,
			0x1b, 0x1a, 0x82, 0x61, 0xc0, 0x2a, 0x9e, 0x72,
			0x70, 0x10, 0xa6, 0x4a, 0x1a, 0x91, 0xd4, 0xb7,
			0xaa, 0xa3, 0xcc, 0x06, 0xcd, 0xbb, 0xcc, 0xb2,
			0xb7, 0x22, 0x7b, 0xb6, 0x11, 0x5b, 0x61, 0x6e,
			0xa6, 0xab, 0x65, 0x1d, 0x3c, 0x9c, 0x66, 0xa6,
			0xad, 0xe8, 0x22, 0x2c, 0x66, 0x23, 0xaa, 0x26,
			0xa4, 0x86, 0xd0, 0x7a, 0x16, 0xe9, 0x0a, 0x14,
			0xfe, 0x73, 0x6b, 0xa5, 0x88, 0x9d, 0x97, 0xfc,
			0x19, 0x9c, 0x4a, 0x4d, 0x55, 0x1a, 0x41, 0x05,
			0xb9, 0xea, 0xaf, 0xa8, 0xb3, 0x5f, 0x9e, 0xa1,
			0xfb, 0xc9, 0x55, 0xf5, 0xc4, 0x17, 0x5b, 0x80,
			0x66, 0xb8, 0x59, 0x95, 0xf6, 0xb0, 0x0e, 0x4c,
			0x3e, 0x0e, 0x3f, 0x1d, 0x94, 0x21, 0xd5, 0x4a,
			0x5c, 0x3f, 0x23, 0x5a, 0xe8, 0x38, 0x59, 0x72,
		},
		20, /* salt length */
		/* V8 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA224 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0x5d, 0x95, 0x51, 0xdf, 0x21, 0x0c, 0xea, 0x61,
			0xf9, 0xae, 0xc3, 0xd2, 0xef, 0x2f, 0x57, 0x16,
			0x69, 0x2c, 0x5c, 0x55, 0xff, 0xb1, 0x3d, 0xa2,
			0x8a, 0xfe, 0xca, 0xda, 0x7d, 0x8b, 0x2f, 0xcf,
			0x79, 0x49, 0x22, 0xa6, 0x1b, 0x2b, 0x61, 0x7e,
			0x19, 0xe5, 0x5c, 0x53, 0x8d, 0xf4, 0x2e, 0x21,
			0x90, 0xde, 0xd2, 0x4c, 0x0b, 0xbb, 0x94, 0x3f,
			0xca, 0xf4, 0xc5, 0x53, 0xd2, 0x4b, 0xb3, 0xf3,
			0xf5, 0x1b, 0xd9, 0xff, 0xa3, 0xe1, 0x52, 0x6d,
			0x03, 0xfc, 0x94, 0x36, 0x42, 0xb1, 0x5a, 0x70,
			0x2c, 0x72, 0xe2, 0xb7, 0xf9, 0x60, 0x59, 0xd2,
			0x69, 0x0e, 0x94, 0x94, 0x13, 0x88, 0x00, 0xff,
			0xf9, 0x94, 0xdc, 0xb0, 0x61, 0x37, 0xa5, 0x77,
			0xe0, 0x7b, 0xd2, 0x45, 0xb8, 0x70, 0xc2, 0x4b,
			0x64, 0x71, 0xf3, 0x60, 0x70, 0xa8, 0x74, 0xa3,
			0x8e, 0xd5, 0x9d, 0x62, 0xe7, 0x4a, 0xfd, 0x3b,
		},
		/* S9 : 128 byte */
		{
			0x69, 0x6a, 0x03, 0x91, 0xfa, 0xd6, 0x96, 0xf0,
			0x98, 0xb3, 0xe1, 0xbd, 0xe9, 0x09, 0xe5, 0x1c,
			0x08, 0x93, 0x58, 0x20, 0x50, 0xaf, 0xe5, 0xee,
			0x63, 0x4e, 0xe3, 0x1d, 0x9c, 0x84, 0xb7, 0x72,
			0x0f, 0x06, 0xb5, 0x90, 0xa1, 0x89, 0x28, 0x7d,
			0x1b, 0x0c, 0xdf, 0x75, 0x81, 0x73, 0xed, 0x6d,
			0xe8, 0xac, 0x16, 0x4d, 0xa9, 0xbe, 0x8a, 0xa0,
			0x0a, 0xca, 0x38, 0x70, 0x97, 0x47, 0x87, 0xd1,
			0x5f, 0x84, 0xc5, 0x66, 0xa5, 0x19, 0x5a, 0xec,
			0x03, 0x63, 0x68, 0x3e, 0xc0, 0xb6, 0x91, 0x84,
			0xbb, 0x7d, 0xde, 0x3d, 0xf3, 0x33, 0x27, 0xf2,
			0xd5, 0x39, 0xe1, 0xe4, 0x2d, 0x9e, 0xe5, 0x40,
			0x47, 0xdf, 0x7f, 0x82, 0x95, 0xf1, 0x49, 0x6c,
			0xa6, 0x44, 0xd4, 0x11, 0xc8, 0x54, 0x74, 0xa5,
			0x2d, 0x74, 0xa2, 0x6c, 0x65, 0x56, 0xd5, 0xea,
			0x52, 0x45, 0x8a, 0xc1, 0x30, 0x70, 0x10, 0x21,
		},
		20, /* salt length */
		/* V9 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
};

/* 10 elements */
static mod1024_t SHA256[] = {
	/* SHA256 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0x12, 0x48, 0xf6, 0x2a, 0x43, 0x89, 0xf4, 0x2f,
			0x7b, 0x4b, 0xb1, 0x31, 0x05, 0x3d, 0x6c, 0x88,
			0xa9, 0x94, 0xdb, 0x20, 0x75, 0xb9, 0x12, 0xcc,
			0xbe, 0x3e, 0xa7, 0xdc, 0x61, 0x17, 0x14, 0xf1,
			0x4e, 0x07, 0x5c, 0x10, 0x48, 0x58, 0xf2, 0xf6,
			0xe6, 0xcf, 0xd6, 0xab, 0xde, 0xdf, 0x01, 0x5a,
			0x82, 0x1d, 0x03, 0x60, 0x8b, 0xf4, 0xeb, 0xa3,
			0x16, 0x9a, 0x67, 0x25, 0xec, 0x42, 0x2c, 0xd9,
			0x06, 0x94, 0x98, 0xb5, 0x51, 0x5a, 0x96, 0x08,
			0xae, 0x7c, 0xc3, 0x0e, 0x3d, 0x2e, 0xcf, 0xc1,
			0xdb, 0x68, 0x25, 0xf3, 0xe9, 0x96, 0xce, 0x9a,
			0x50, 0x92, 0x92, 0x6b, 0xc1, 0xcf, 0x61, 0xaa,
			0x42, 0xd7, 0xf2, 0x40, 0xe6, 0xf7, 0xaa, 0x0e,
			0xdb, 0x38, 0xbf, 0x81, 0xaa, 0x92, 0x9d, 0x66,
			0xbb, 0x5d, 0x89, 0x00, 0x18, 0x08, 0x84, 0x58,
			0x72, 0x0d, 0x72, 0xd5, 0x69, 0x24, 0x7b, 0x0c,
		},
		/* S0 : 128 byte */
		{
			0x7b, 0x1d, 0x37, 0x27, 0x8e, 0x54, 0x98, 0x98,
			0xd4, 0x08, 0x4e, 0x22, 0x10, 0xc4, 0xa9, 0x96,
			0x1e, 0xdf, 0xe7, 0xb5, 0x96, 0x35, 0x50, 0xcc,
			0xa1, 0x90, 0x42, 0x48, 0xc8, 0x68, 0x15, 0x13,
			0x53, 0x90, 0x17, 0x82, 0x0f, 0x0e, 0x9b, 0xd0,
			0x74, 0xb9, 0xf8, 0xa0, 0x67, 0xb9, 0xfe, 0xff,
			0xf7, 0xf1, 0xfa, 0x20, 0xbf, 0x2d, 0x0c, 0x75,
			0x01, 0x5f, 0xf0, 0x20, 0xb2, 0x21, 0x0c, 0xc7,
			0xf7, 0x90, 0x34, 0xfe, 0xdf, 0x68, 0xe8, 0xd4,
			0x4a, 0x00, 0x7a, 0xbf, 0x4d, 0xd8, 0x2c, 0x26,
			0xe8, 0xb0, 0x03, 0x93, 0x72, 0x3a, 0xea, 0x15,
			0xab, 0xfb, 0xc2, 0x29, 0x41, 0xc8, 0xcf, 0x79,
			0x48, 0x17, 0x18, 0xc0, 0x08, 0xda, 0x71, 0x3f,
			0xb8, 0xf5, 0x4c, 0xb3, 0xfc, 0xa8, 0x90, 0xbd,
			0xe1, 0x13, 0x73, 0x14, 0x33, 0x4b, 0x9b, 0x0a,
			0x18, 0x51, 0x5b, 0xfa, 0x48, 0xe5, 0xcc, 0xd0,
		},
		20, /* salt length */
		/* V0 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA256 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 0x2b, 0x90, 0x75, 0x7b,
			0x77, 0xc7, 0xc2, 0x01, 0xe0, 0xdf, 0xbe, 0x7b,
			0x68, 0xa3, 0x9c, 0x9f, 0x01, 0x0c, 0x04, 0xc9,
			0xa4, 0x04, 0xc0, 0xe5, 0x9d, 0x6f, 0xb8, 0x8f,
			0x29, 0x6c, 0x20, 0x79, 0x42, 0xd8, 0xf0, 0x0d,
			0xba, 0x15, 0x94, 0x54, 0x87, 0x09, 0xe5, 0x17,
			0x5e, 0x69, 0x59, 0x44, 0x83, 0x37, 0x76, 0x92,
			0xdf, 0x89, 0x8d, 0x68, 0x53, 0x8d, 0xa4, 0xfc,
			0x4c, 0xca, 0xb4, 0xb8, 0x97, 0xa8, 0x15, 0xe1,
			0x77, 0x33, 0x36, 0x7a, 0xf5, 0x1c, 0x91, 0x66,
			0x78, 0x00, 0xde, 0xce, 0x95, 0xed, 0x87, 0xc2,
			0x28, 0x23, 0x5d, 0x23, 0xa4, 0x17, 0x89, 0xc0,
			0xa7, 0xbe, 0xc5, 0xbf, 0xd2, 0x07, 0xcc, 0x3f,
			0xd7, 0xaa, 0x0e, 0x3b, 0xd3, 0x20, 0xa9, 0xb1,
		},
		/* S1 : 128 byte */
		{
			0x5d, 0x6e, 0x26, 0xa5, 0xe8, 0x59, 0x9a, 0xe4,
			0xfb, 0xc4, 0x9c, 0x68, 0x7f, 0x18, 0x12, 0xa9,
			0x0f, 0xba, 0x94, 0x6f, 0x2d, 0x21, 0x7c, 0xf0,
			0xae, 0xa5, 0x3f, 0x54, 0x31, 0x91, 0xe1, 0x1a,
			0xca, 0x9b, 0xd5, 0xad, 0x66, 0xc3, 0x91, 0x41,
			0x1e, 0x7c, 0xdb, 0x1f, 0xd1, 0xa4, 0x65, 0x55,
			0x9c, 0xf3, 0x71, 0xc5, 0x87, 0x03, 0xb3, 0x5c,
			0xc8, 0xea, 0xfc, 0x50, 0xb7, 0x8f, 0x1d, 0xdd,
			0xbc, 0x91, 0x35, 0x85, 0xd1, 0x84, 0x64, 0x65,
			0x1c, 0x69, 0xff, 0xa5, 0xf7, 0x85, 0xba, 0x92,
			0x13, 0x85, 0x56, 0x68, 0xcc, 0x7b, 0x3c, 0x55,
			0x3c, 0xb2, 0xc1, 0x12, 0xc9, 0x89, 0x12, 0x79,
			0x85, 0x01, 0xbd, 0x61, 0xd6, 0x5d, 0xe9, 0xee,
			0xdd, 0x65, 0xca, 0x26, 0x05, 0x1b, 0x0d, 0xb4,
			0xb5, 0xc1, 0xd6, 0x0e, 0x91, 0x92, 0xaf, 0x9c,
			0x04, 0xd1, 0xa6, 0xc9, 0x62, 0xc8, 0x49, 0x16,
		},
		20, /* salt length */
		/* V1 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA256 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x46, 0xd0, 0xb4, 0xb0, 0x35, 0xe9, 0xf1, 0xa1,
			0xa9, 0xfc, 0xb1, 0xb7, 0xa5, 0x8a, 0xc1, 0x98,
			0x0a, 0x81, 0xa6, 0xc9, 0x18, 0xb3, 0x3c, 0xc4,
			0x32, 0x87, 0x9d, 0x4b, 0xe1, 0xee, 0x44, 0xfb,
			0x1c, 0x99, 0x73, 0x24, 0xc7, 0x6a, 0xd1, 0xd8,
			0x79, 0x28, 0x58, 0x42, 0x82, 0x76, 0x6e, 0x39,
			0x2d, 0x06, 0x52, 0x15, 0x42, 0x0a, 0xba, 0xc2,
			0xcf, 0xeb, 0xbb, 0x13, 0x28, 0x3e, 0xe8, 0x58,
			0x2d, 0xf4, 0xbe, 0x32, 0xcb, 0xd0, 0x40, 0xa6,
			0xc6, 0x1f, 0xde, 0x71, 0xb5, 0xa3, 0x9b, 0x9b,
			0x4e, 0xd0, 0x6f, 0x4f, 0xe3, 0x3b, 0xea, 0xea,
			0x2d, 0x4f, 0x1b, 0x52, 0x49, 0x41, 0xb0, 0x89,
			0x00, 0x47, 0x5e, 0x82, 0x4b, 0x05, 0x84, 0x33,
			0x16, 0x48, 0x0b, 0xeb, 0x45, 0xf7, 0x90, 0xe6,
			0xf5, 0x47, 0xc5, 0x1f, 0x04, 0x2d, 0x14, 0x65,
			0xd3, 0x19, 0x86, 0xb1, 0x4a, 0xe1, 0xe4, 0xb7,
		},
		/* S2 : 128 byte */
		{
			0x55, 0x76, 0xd0, 0x90, 0x15, 0x47, 0x61, 0x93,
			0x4a, 0x35, 0x76, 0x92, 0x83, 0xc1, 0xe6, 0xb6,
			0x6e, 0x72, 0x18, 0x50, 0xcb, 0x95, 0x89, 0xbf,
			0x4a, 0xa8, 0xb6, 0xd7, 0xb6, 0x95, 0x01, 0x6d,
			0x2d, 0xdc, 0xca, 0x77, 0xc1, 0x89, 0x66, 0x05,
			0x19, 0x49, 0xf6, 0x58, 0xeb, 0x48, 0x13, 0x9b,
			0x5b, 0x5b, 0xaf, 0x6b, 0xbc, 0x1e, 0x16, 0xdf,
			0x82, 0x5a, 0x55, 0x2e, 0x4a, 0x38, 0xd9, 0x7b,
			0x83, 0x37, 0xa2, 0x47, 0xef, 0x0f, 0x59, 0x1c,
			0xc1, 0xec, 0x9a, 0xe0, 0x8f, 0x5a, 0x91, 0xf0,
			0xc7, 0x3d, 0x07, 0x8b, 0x60, 0x6b, 0x8a, 0x52,
			0xd5, 0x67, 0x96, 0x58, 0xf6, 0x6b, 0x17, 0xdb,
			0x2d, 0xe1, 0x9b, 0xc4, 0x8e, 0x0c, 0xdb, 0x2f,
			0x14, 0xdf, 0x86, 0xc1, 0x26, 0x39, 0x12, 0x52,
			0xba, 0x71, 0xbb, 0x60, 0x0f, 0xde, 0xf6, 0xd7,
			0x12, 0x41, 0xc2, 0xbb, 0x9c, 0xbd, 0xc3, 0xb5,
		},
		20, /* salt length */
		/* V2 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA256 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0x5e, 0x61, 0x14, 0x73, 0xdd, 0x3c, 0xb9, 0x22,
			0x38, 0x30, 0x0e, 0xd5, 0x4a, 0xbd, 0x60, 0x36,
			0x62, 0x04, 0x1b, 0x92, 0xe9, 0xdc, 0x8d, 0x8d,
			0x45, 0x23, 0xec, 0x15, 0xfc, 0x52, 0x9b, 0x94,
			0x1d, 0x54, 0xa9, 0x6a, 0xdc, 0x99, 0x9b, 0x9a,
			0x7c, 0x66, 0x6b, 0xc3, 0x72, 0x6f, 0xd0, 0x53,
			0xc7, 0xa7, 0xc0, 0xf7, 0xbe, 0x57, 0x33, 0x56,
			0xb7, 0x6d, 0x2b, 0xfe, 0x53, 0x17, 0xf1, 0x9e,
			0xd9, 0x91, 0xa1, 0x77, 0xd8, 0x3c, 0xe8, 0x0d,
			0x0e, 0xef, 0x0d, 0x22, 0x89, 0x91, 0x2e, 0xe4,
			0x0c, 0xe8, 0xcd, 0x66, 0xb4, 0xac, 0x8b, 0x4d,
			0xdb, 0xe0, 0x32, 0xcc, 0x3d, 0x62, 0xf9, 0xd2,
			0x59, 0xc0, 0x04, 0x81, 0x1b, 0x4d, 0x2b, 0xe3,
			0xb7, 0x74, 0xfc, 0xd8, 0xd8, 0x4d, 0x03, 0x53,
			0xfa, 0x6e, 0x49, 0xc6, 0x14, 0x04, 0x1a, 0xdb,
			0x7b, 0x22, 0x0b, 0x65, 0x03, 0x58, 0x3c, 0x96,
		},
		/* S3 : 128 byte */
		{
			0xb3, 0x2c, 0xb5, 0xcb, 0xa0, 0x65, 0x25, 0x96,
			0x81, 0xb9, 0xb9, 0x1e, 0xb4, 0x43, 0x45, 0xfb,
			0x89, 0xd2, 0x81, 0xd1, 0xd7, 0x7e, 0x76, 0xea,
			0xcf, 0xae, 0x05, 0xb1, 0xa2, 0xbf, 0x98, 0x8b,
			0x97, 0xa1, 0xb7, 0x51, 0xc1, 0x69, 0x16, 0x8c,
			0x58, 0xd7, 0x36, 0x10, 0x59, 0x9a, 0x6f, 0x85,
			0x6f, 0x65, 0x6e, 0xc4, 0xc7, 0x09, 0x5f, 0x9d,
			0x7c, 0x5c, 0x87, 0xed, 0xdf, 0x04, 0x6b, 0xd5,
			0xea, 0xe0, 0x29, 0x8c, 0x73, 0xe1, 0xb3, 0x60,
			0xe1, 0x33, 0xdc, 0x0d, 0x40, 0x06, 0x25, 0x9d,
			0x6c, 0xef, 0x6a, 0xff, 0x26, 0x83, 0x2c, 0x08,
			0x6d, 0xed, 0x86, 0xca, 0x0a, 0x7b, 0x5f, 0x7f,
			0x36, 0xb6, 0x07, 0xa4, 0xeb, 0x8e, 0xbb, 0xe2,
			0x61, 0x9a, 0x74, 0x64, 0x7f, 0xaa, 0x9e, 0xe3,
			0x83, 0x52, 0x00, 0x6c, 0x41, 0xe7, 0xe6, 0x41,
			0x4a, 0x52, 0x40, 0xa7, 0x28, 0x94, 0x9e, 0x63,
		},
		20, /* salt length */
		/* V3 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA256 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0xf1, 0x4b, 0x4e, 0x3f, 0xe8, 0x2b, 0x9c, 0xcd,
			0x1d, 0x37, 0xb5, 0x05, 0x28, 0x37, 0x97, 0x01,
			0x95, 0x15, 0xcd, 0x2d, 0xe7, 0xd9, 0x93, 0xe6,
			0xdc, 0xd7, 0x3a, 0x47, 0xf3, 0x27, 0xce, 0x3f,
			0x2c, 0x56, 0xf7, 0xe7, 0x0f, 0x5b, 0x1d, 0x13,
			0xef, 0xcd, 0xcf, 0x28, 0x70, 0x53, 0x66, 0x2d,
			0x47, 0x5b, 0xe5, 0x24, 0x41, 0x0f, 0x24, 0x9a,
			0x87, 0x84, 0x2e, 0x7d, 0x20, 0x35, 0xe7, 0x2e,
			0xb7, 0xb1, 0xf2, 0xd7, 0x9f, 0x90, 0xd1, 0xff,
			0x14, 0xab, 0x4e, 0x5b, 0xc4, 0x8d, 0x1a, 0xe7,
			0x8e, 0x24, 0xb5, 0x36, 0x89, 0xbc, 0x8d, 0x06,
			0x48, 0x2f, 0x6a, 0x18, 0x5c, 0x55, 0x09, 0x3f,
			0x1b, 0xc2, 0x07, 0xb6, 0xa0, 0xb1, 0x02, 0xb6,
			0x13, 0x3b, 0xfa, 0x48, 0x2b, 0xcd, 0xfd, 0x59,
			0xf0, 0xd8, 0x03, 0x0c, 0xc6, 0xca, 0x17, 0x54,
			0xa7, 0x3e, 0xb7, 0xc1, 0xb0, 0x6f, 0x7e, 0x9a,
		},
		/* S4 : 128 byte */
		{
			0x65, 0x7d, 0xaa, 0xc0, 0xad, 0xe9, 0x99, 0x46,
			0xd5, 0x57, 0xa5, 0x41, 0x3b, 0xa6, 0x8f, 0x6b,
			0x3e, 0x3d, 0x69, 0xfe, 0x1c, 0xb1, 0xc2, 0xd8,
			0x73, 0x76, 0x26, 0xae, 0xd7, 0xf6, 0x0c, 0xb3,
			0xa1, 0xac, 0xa9, 0xe2, 0xca, 0xac, 0x92, 0x67,
			0x92, 0x71, 0x12, 0x13, 0x84, 0xd5, 0xa7, 0xa3,
			0x00, 0x45, 0x12, 0x4e, 0x1f, 0x54, 0xac, 0x78,
			0x9e, 0x45, 0x63, 0x26, 0x46, 0x48, 0xda, 0x18,
			0x5e, 0x51, 0x74, 0xc2, 0xf8, 0x5a, 0xcb, 0xb8,
			0x3f, 0xe9, 0xd2, 0xfd, 0xed, 0x35, 0xc2, 0x38,
			0x82, 0xe7, 0x15, 0x94, 0xa5, 0xdd, 0x7e, 0xd4,
			0xb5, 0x0a, 0x08, 0xd5, 0x33, 0x70, 0x2b, 0x5b,
			0x35, 0x8b, 0xf1, 0x91, 0xa5, 0x9f, 0x56, 0x8d,
			0xb3, 0x26, 0xa7, 0xf2, 0x38, 0x33, 0x21, 0x9f,
			0x26, 0xf5, 0xe7, 0x77, 0x24, 0xd9, 0x13, 0x87,
			0x26, 0x08, 0x74, 0x9a, 0x58, 0x62, 0x56, 0xb2,
		},
		20, /* salt length */
		/* V4 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA256 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0xe5, 0x11, 0x90, 0x3c, 0x2f, 0x1b, 0xfb, 0xa2,
			0x45, 0x46, 0x72, 0x95, 0xac, 0x95, 0x41, 0x3a,
			0xc4, 0x74, 0x6c, 0x98, 0x4c, 0x37, 0x50, 0xa7,
			0x28, 0xc3, 0x88, 0xaa, 0x62, 0x8b, 0x0e, 0xbf,
			0x70, 0xbf, 0xa2, 0xb5, 0xb5, 0xbf, 0x70, 0x1d,
			0x4a, 0xd6, 0xed, 0xa4, 0xfd, 0x25, 0x02, 0xa0,
			0x9a, 0xdd, 0x6c, 0xe8, 0x6a, 0xb2, 0x63, 0xe9,
			0xfa, 0x8c, 0x79, 0xf6, 0x1d, 0x7f, 0xc6, 0xe0,
			0x34, 0xfd, 0x42, 0xc7, 0xad, 0x4a, 0x72, 0x2f,
			0xca, 0x31, 0x47, 0x56, 0xa4, 0xf6, 0x43, 0xc6,
			0x6f, 0x8b, 0xdd, 0x26, 0xff, 0x06, 0x27, 0x55,
			0x8b, 0x53, 0xae, 0xba, 0xb2, 0x26, 0xa1, 0xd5,
			0xbb, 0x34, 0x36, 0x1a, 0x21, 0xa3, 0x4a, 0x00,
			0x4f, 0xbf, 0x48, 0x55, 0xa8, 0xee, 0x4b, 0xd4,
			0xca, 0xe6, 0x87, 0xf9, 0x14, 0x1d, 0xe5, 0x14,
			0x66, 0x81, 0xed, 0x9d, 0x27, 0x6a, 0xe9, 0xc7,
		},
		/* S5 : 128 byte */
		{
			0x56, 0xc6, 0x29, 0xaf, 0x3b, 0xe9, 0xc9, 0xa7,
			0x1e, 0x06, 0xc8, 0x46, 0x81, 0xe6, 0x8a, 0x82,
			0x5a, 0x63, 0x1a, 0x1c, 0xa8, 0x2d, 0x4e, 0xeb,
			0x3b, 0xe5, 0x02, 0x99, 0x49, 0xd1, 0x85, 0xda,
			0x53, 0x79, 0xe4, 0x27, 0xdd, 0x00, 0x8e, 0x32,
			0x53, 0x8c, 0xe9, 0x4f, 0xd5, 0x4d, 0xf8, 0xb7,
			0x4d, 0x8a, 0xcf, 0x04, 0xf6, 0xcd, 0xa6, 0xb5,
			0xc0, 0xc6, 0x91, 0x11, 0x46, 0x31, 0xa4, 0x36,
			0x8b, 0xf7, 0xc4, 0xb6, 0x0e, 0xc1, 0x24, 0x70,
			0x8c, 0x43, 0x58, 0x45, 0xb7, 0xae, 0x1a, 0xd4,
			0x98, 0xd1, 0xf9, 0x03, 0xe0, 0xe9, 0xef, 0x36,
			0xda, 0xef, 0xf1, 0x7b, 0x11, 0xba, 0xa7, 0x86,
			0x2e, 0xcc, 0x16, 0xea, 0x9c, 0x69, 0xd6, 0x1b,
			0x2e, 0x3a, 0xaf, 0xf4, 0x40, 0x48, 0xa3, 0x99,
			0xfb, 0x38, 0xbd, 0x7c, 0x4d, 0x45, 0xb3, 0x18,
			0xac, 0x58, 0x15, 0xcd, 0x60, 0x50, 0x61, 0x92,
		},
		20, /* salt length */
		/* V5 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA256 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0x10, 0x34, 0xe0, 0x43, 0xd5, 0x21, 0xa2, 0x9e,
			0x1f, 0x6d, 0x84, 0xef, 0x8a, 0x54, 0x9b, 0x86,
			0x75, 0xc5, 0xeb, 0xe4, 0xab, 0x74, 0x2e, 0x72,
			0x43, 0x2b, 0xd1, 0x72, 0xc4, 0x60, 0x40, 0x35,
			0x8d, 0x0d, 0x63, 0x8e, 0x74, 0xbc, 0x68, 0x88,
			0xae, 0xe1, 0xc4, 0x54, 0xef, 0x6d, 0x74, 0x85,
			0xd1, 0xa0, 0x89, 0x80, 0x7a, 0x1f, 0xe3, 0x2d,
			0x79, 0x46, 0x3d, 0xf3, 0xf2, 0xbf, 0x6a, 0xe4,
			0xfb, 0x54, 0xdd, 0x34, 0x45, 0x01, 0x6f, 0xf0,
			0xff, 0x4e, 0x5d, 0xec, 0xf7, 0xdc, 0x90, 0xfe,
			0x03, 0x3d, 0x22, 0x69, 0x9f, 0x08, 0x1d, 0x9c,
			0xd7, 0x2d, 0x86, 0x1d, 0x68, 0x87, 0x9a, 0xc0,
			0x22, 0xb0, 0x57, 0x1c, 0x90, 0x19, 0x3e, 0x3f,
			0xdb, 0x5a, 0x58, 0xa1, 0xec, 0xc6, 0x77, 0x57,
			0xd1, 0x43, 0xbe, 0x89, 0x62, 0x59, 0x63, 0x26,
			0x24, 0x88, 0xb4, 0xd6, 0x6d, 0xdf, 0x58, 0x31,
		},
		/* S6 : 128 byte */
		{
			0x97, 0xd0, 0x1e, 0xb9, 0x37, 0x79, 0xf6, 0x7b,
			0x3c, 0x21, 0x36, 0x4e, 0x83, 0x8d, 0xf0, 0x79,
			0xb2, 0x3b, 0xb4, 0x1e, 0x17, 0x37, 0x51, 0x2a,
			0x79, 0xcd, 0xac, 0xa0, 0xbf, 0x72, 0xe0, 0x64,
			0x93, 0xce, 0x00, 0x5f, 0xe5, 0xe9, 0x79, 0x15,
			0x32, 0x36, 0x1b, 0x07, 0x2f, 0x61, 0xac, 0x8b,
			0x01, 0x40, 0x45, 0x8b, 0x1f, 0x2e, 0x3f, 0x85,
			0x2c, 0x0f, 0xf8, 0xc4, 0xf0, 0xb5, 0x9c, 0x48,
			0x08, 0x45, 0x86, 0x80, 0xb4, 0xe2, 0x00, 0xbe,
			0x87, 0xa8, 0x37, 0x73, 0xb5, 0xdd, 0x11, 0x45,
			0x95, 0xfc, 0xa4, 0xe2, 0x03, 0x64, 0x07, 0x8f,
			0x04, 0x9c, 0xb3, 0x88, 0xac, 0x8e, 0x2e, 0x35,
			0x4d, 0xb1, 0x13, 0xd6, 0x90, 0x17, 0xf3, 0xe7,
			0x10, 0x01, 0x0c, 0x68, 0x55, 0x06, 0xcf, 0xbd,
			0x2c, 0x0b, 0x46, 0xc0, 0x84, 0x00, 0x02, 0x5c,
			0xd2, 0x6b, 0x17, 0xc7, 0x36, 0x5d, 0x68, 0xab,
		},
		20, /* salt length */
		/* V6 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA256 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0x12, 0x67, 0xd9, 0xe2, 0x9f, 0x0f, 0x39, 0xfc,
			0xe1, 0xd1, 0x92, 0x4f, 0x0c, 0x2a, 0x43, 0x51,
			0x09, 0x68, 0x2b, 0xf0, 0x60, 0x03, 0xb4, 0x84,
			0x5b, 0x30, 0x10, 0x02, 0x69, 0xb4, 0xc3, 0x72,
			0xc0, 0xf6, 0x62, 0xa3, 0x46, 0x59, 0x9a, 0x1d,
			0xb2, 0xc8, 0x73, 0x57, 0xee, 0x93, 0xd6, 0x2a,
			0xc0, 0x79, 0x4e, 0x86, 0xaa, 0x06, 0x0b, 0x64,
			0x0f, 0x1f, 0x2c, 0xfb, 0x96, 0xdb, 0x3e, 0x37,
			0xd4, 0x94, 0xff, 0x67, 0x25, 0x3e, 0x68, 0x36,
			0xda, 0xf8, 0xeb, 0xdc, 0x9a, 0xe1, 0xfa, 0x95,
			0x02, 0x2d, 0xc8, 0x52, 0x8f, 0x78, 0x53, 0xaf,
			0xbe, 0x59, 0xa2, 0x26, 0x71, 0x3b, 0x4b, 0x82,
			0x95, 0x68, 0x39, 0x96, 0x02, 0x67, 0xac, 0x2e,
			0xa2, 0x88, 0x81, 0x46, 0xd6, 0xbd, 0xb4, 0x7a,
			0x15, 0xab, 0x23, 0xbe, 0xd7, 0x01, 0x95, 0x4f,
			0xb1, 0x09, 0xfa, 0xea, 0xc1, 0x7d, 0xf5, 0x3b,
		},
		/* S7 : 128 byte */
		{
			0xb8, 0x48, 0x98, 0xf0, 0x03, 0x25, 0x9f, 0x6d,
			0x6c, 0xe4, 0xe4, 0xbd, 0xa1, 0xff, 0x9e, 0x77,
			0x2c, 0x51, 0x6f, 0x04, 0xd8, 0xa2, 0xf9, 0x55,
			0x9b, 0x40, 0x65, 0x0f, 0x87, 0xe3, 0x74, 0x84,
			0xb4, 0xec, 0x8f, 0xac, 0xe4, 0x03, 0x57, 0x0f,
			0x19, 0xdb, 0x16, 0x37, 0x7a, 0x1e, 0x28, 0x4e,
			0xb8, 0x65, 0xdf, 0xe8, 0x18, 0x25, 0x39, 0xe7,
			0xb9, 0x3b, 0x6b, 0xf9, 0x36, 0x07, 0x54, 0x83,
			0x09, 0x0f, 0xc2, 0xd3, 0x33, 0x55, 0xf3, 0x2a,
			0x78, 0x40, 0x7b, 0x80, 0x98, 0x34, 0xd6, 0x56,
			0x73, 0xbc, 0xad, 0xa8, 0xab, 0x3b, 0xf8, 0x1d,
			0x19, 0x85, 0x1b, 0x43, 0xd9, 0x9b, 0x81, 0xbc,
			0x8f, 0x21, 0xb2, 0xb0, 0x85, 0x04, 0xbe, 0xef,
			0x0f, 0x67, 0x47, 0x75, 0x9c, 0xe3, 0x31, 0x8b,
			0x67, 0xc3, 0xa4, 0x4e, 0xe5, 0x4d, 0xfd, 0x04,
			0x04, 0x24, 0xc9, 0xb2, 0xeb, 0xd8, 0xbc, 0x9d,
		},
		20, /* salt length */
		/* V7 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA256 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0xb0, 0x66, 0x48, 0x83, 0x5d, 0x99, 0x3a, 0x4c,
			0x66, 0x94, 0x4a, 0x5c, 0xb9, 0x5a, 0xfd, 0x74,
			0x1c, 0x66, 0x70, 0x8b, 0xc4, 0xf0, 0x92, 0x28,
			0x23, 0x6d, 0x91, 0x10, 0x01, 0x55, 0x92, 0x21,
			0xce, 0x28, 0xa5, 0x7e, 0x37, 0xb3, 0xf8, 0xec,
			0xf7, 0x90, 0x4b, 0x7c, 0xf0, 0xec, 0xa3, 0x6b,
			0x7a, 0x8c, 0x43, 0x46, 0xfb, 0x12, 0xd8, 0x51,
			0xb8, 0x7d, 0xf4, 0x2a, 0xea, 0xf0, 0xbd, 0xaa,
			0x15, 0xb4, 0xe7, 0x0c, 0x95, 0x7f, 0xdf, 0x51,
			0x79, 0xf7, 0x49, 0xe2, 0x95, 0x72, 0xad, 0x78,
			0xd3, 0x00, 0x6d, 0x39, 0xca, 0xab, 0xfb, 0x24,
			0x92, 0xfd, 0x48, 0xae, 0xd6, 0xc3, 0x80, 0xad,
			0x2a, 0x11, 0xf4, 0x76, 0x9f, 0x86, 0x58, 0x42,
			0x95, 0xd1, 0x2e, 0xd7, 0xd1, 0x74, 0x82, 0x64,
			0xd0, 0xc8, 0xd8, 0xac, 0x5a, 0x42, 0x63, 0x66,
			0x5a, 0xf4, 0x77, 0xe4, 0xec, 0x57, 0x41, 0xf4,
		},
		/* S8 : 128 byte */
		{
			0x52, 0x34, 0x42, 0xef, 0x2a, 0x7b, 0xda, 0xf0,
			0x05, 0x70, 0xf1, 0xb0, 0x6e, 0x6c, 0xfa, 0xae,
			0xd9, 0x18, 0x37, 0x1a, 0x07, 0xc9, 0xff, 0xa6,
			0xdf, 0x16, 0x80, 0x66, 0x10, 0xcf, 0xb9, 0x34,
			0x20, 0x1a, 0x06, 0x8e, 0xf6, 0xa3, 0xd2, 0x7c,
			0x74, 0xef, 0x2d, 0xaa, 0x81, 0xec, 0x56, 0x91,
			0xb7, 0xd0, 0x3b, 0x48, 0x22, 0x31, 0x29, 0xa1,
			0xf1, 0x42, 0xcc, 0x81, 0xb3, 0xb7, 0x15, 0x1c,
			0xce, 0xe7, 0x53, 0x0e, 0xf5, 0x05, 0x05, 0x42,
			0x94, 0x06, 0xb9, 0x6d, 0xdd, 0xf2, 0x14, 0x72,
			0x63, 0xa6, 0x26, 0xf7, 0x51, 0x2c, 0x1c, 0xd3,
			0xd1, 0x1a, 0x10, 0x0b, 0x8f, 0x84, 0xe0, 0x74,
			0x01, 0xb2, 0xa4, 0xa3, 0xf4, 0xfc, 0x31, 0x30,
			0x9f, 0x08, 0xc9, 0x69, 0x3e, 0xae, 0x5d, 0x0e,
			0xea, 0x2a, 0xef, 0x0f, 0x40, 0x41, 0xcd, 0xf9,
			0x40, 0xcb, 0xc3, 0x9e, 0x0c, 0xfa, 0xc9, 0xc3,
		},
		20, /* salt length */
		/* V8 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA256 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0x5d, 0x95, 0x51, 0xdf, 0x21, 0x0c, 0xea, 0x61,
			0xf9, 0xae, 0xc3, 0xd2, 0xef, 0x2f, 0x57, 0x16,
			0x69, 0x2c, 0x5c, 0x55, 0xff, 0xb1, 0x3d, 0xa2,
			0x8a, 0xfe, 0xca, 0xda, 0x7d, 0x8b, 0x2f, 0xcf,
			0x79, 0x49, 0x22, 0xa6, 0x1b, 0x2b, 0x61, 0x7e,
			0x19, 0xe5, 0x5c, 0x53, 0x8d, 0xf4, 0x2e, 0x21,
			0x90, 0xde, 0xd2, 0x4c, 0x0b, 0xbb, 0x94, 0x3f,
			0xca, 0xf4, 0xc5, 0x53, 0xd2, 0x4b, 0xb3, 0xf3,
			0xf5, 0x1b, 0xd9, 0xff, 0xa3, 0xe1, 0x52, 0x6d,
			0x03, 0xfc, 0x94, 0x36, 0x42, 0xb1, 0x5a, 0x70,
			0x2c, 0x72, 0xe2, 0xb7, 0xf9, 0x60, 0x59, 0xd2,
			0x69, 0x0e, 0x94, 0x94, 0x13, 0x88, 0x00, 0xff,
			0xf9, 0x94, 0xdc, 0xb0, 0x61, 0x37, 0xa5, 0x77,
			0xe0, 0x7b, 0xd2, 0x45, 0xb8, 0x70, 0xc2, 0x4b,
			0x64, 0x71, 0xf3, 0x60, 0x70, 0xa8, 0x74, 0xa3,
			0x8e, 0xd5, 0x9d, 0x62, 0xe7, 0x4a, 0xfd, 0x3b,
		},
		/* S9 : 128 byte */
		{
			0x07, 0xa4, 0xce, 0xc5, 0xc0, 0xe2, 0xa5, 0x8b,
			0x6c, 0x75, 0x77, 0xd0, 0x09, 0x75, 0xb0, 0x90,
			0x77, 0xe6, 0x33, 0x72, 0xff, 0x67, 0x2e, 0x62,
			0x3e, 0x11, 0x78, 0xc1, 0x0e, 0xab, 0xf3, 0x9b,
			0x29, 0x20, 0x80, 0x68, 0xb0, 0xad, 0x7f, 0xcc,
			0x67, 0x37, 0x5c, 0xf4, 0x4f, 0x71, 0x5c, 0xf3,
			0x7a, 0xc7, 0xd2, 0x79, 0x30, 0x2a, 0x98, 0xdb,
			0x7a, 0x15, 0xba, 0x6b, 0xf8, 0xe3, 0x9e, 0xbd,
			0x54, 0x2b, 0x01, 0x77, 0x2d, 0x7d, 0x1b, 0x4b,
			0xca, 0x9d, 0xc5, 0x07, 0xf9, 0x09, 0x88, 0xb2,
			0x80, 0xa3, 0x45, 0x02, 0x63, 0xc4, 0xed, 0xc1,
			0x4d, 0x3e, 0x9a, 0x22, 0x01, 0xe8, 0x49, 0x28,
			0x18, 0x37, 0x6a, 0xe7, 0x37, 0x74, 0x5c, 0x42,
			0xa3, 0x90, 0xac, 0x04, 0x2b, 0x9a, 0x90, 0x6d,
			0x81, 0x6f, 0x7f, 0x4f, 0xef, 0x4f, 0x43, 0x1b,
			0x0d, 0x56, 0xbb, 0xd0, 0x7e, 0xa2, 0x55, 0x21,
		},
		20, /* salt length */
		/* V9 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
};

/* 10 elements */
static mod1024_t SHA384[] = {
	/* SHA384 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0x12, 0x48, 0xf6, 0x2a, 0x43, 0x89, 0xf4, 0x2f,
			0x7b, 0x4b, 0xb1, 0x31, 0x05, 0x3d, 0x6c, 0x88,
			0xa9, 0x94, 0xdb, 0x20, 0x75, 0xb9, 0x12, 0xcc,
			0xbe, 0x3e, 0xa7, 0xdc, 0x61, 0x17, 0x14, 0xf1,
			0x4e, 0x07, 0x5c, 0x10, 0x48, 0x58, 0xf2, 0xf6,
			0xe6, 0xcf, 0xd6, 0xab, 0xde, 0xdf, 0x01, 0x5a,
			0x82, 0x1d, 0x03, 0x60, 0x8b, 0xf4, 0xeb, 0xa3,
			0x16, 0x9a, 0x67, 0x25, 0xec, 0x42, 0x2c, 0xd9,
			0x06, 0x94, 0x98, 0xb5, 0x51, 0x5a, 0x96, 0x08,
			0xae, 0x7c, 0xc3, 0x0e, 0x3d, 0x2e, 0xcf, 0xc1,
			0xdb, 0x68, 0x25, 0xf3, 0xe9, 0x96, 0xce, 0x9a,
			0x50, 0x92, 0x92, 0x6b, 0xc1, 0xcf, 0x61, 0xaa,
			0x42, 0xd7, 0xf2, 0x40, 0xe6, 0xf7, 0xaa, 0x0e,
			0xdb, 0x38, 0xbf, 0x81, 0xaa, 0x92, 0x9d, 0x66,
			0xbb, 0x5d, 0x89, 0x00, 0x18, 0x08, 0x84, 0x58,
			0x72, 0x0d, 0x72, 0xd5, 0x69, 0x24, 0x7b, 0x0c,
		},
		/* S0 : 128 byte */
		{
			0x8f, 0x16, 0xc8, 0x07, 0xbe, 0xf3, 0xed, 0x6f,
			0x74, 0xee, 0x7f, 0xf5, 0xc3, 0x60, 0xa5, 0x42,
			0x8c, 0x6c, 0x2f, 0x10, 0x51, 0x78, 0xb5, 0x8f,
			0xf7, 0xd0, 0x73, 0xe5, 0x66, 0xda, 0xd6, 0xe7,
			0x71, 0x8d, 0x31, 0x29, 0xc7, 0x68, 0xcd, 0x5a,
			0x96, 0x66, 0xde, 0x2b, 0x6c, 0x94, 0x71, 0x77,
			0xb4, 0x57, 0x09, 0xdc, 0x7c, 0xd0, 0xf4, 0x3b,
			0x0b, 0xa6, 0xfc, 0x75, 0x57, 0x8e, 0x11, 0x96,
			0xac, 0xc1, 0x5c, 0xa3, 0xaf, 0xe4, 0xa7, 0x8c,
			0x14, 0x4c, 0xb6, 0x88, 0x5c, 0x1c, 0xc8, 0x15,
			0xf7, 0xf9, 0x89, 0x25, 0xbc, 0x04, 0xad, 0x2f,
			0xf2, 0x0f, 0xc1, 0x06, 0x8b, 0x04, 0x5d, 0x94,
			0x50, 0xe2, 0xa1, 0xdc, 0xf5, 0xa1, 0x61, 0xce,
			0xab, 0xba, 0x2b, 0x0b, 0x66, 0xc7, 0x35, 0x4f,
			0xdb, 0x80, 0xfa, 0x1d, 0x72, 0x9e, 0x5f, 0x97,
			0x63, 0x87, 0xf2, 0x4a, 0x69, 0x7a, 0x7e, 0x56,
		},
		20, /* salt length */
		/* V0 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA384 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 0x2b, 0x90, 0x75, 0x7b,
			0x77, 0xc7, 0xc2, 0x01, 0xe0, 0xdf, 0xbe, 0x7b,
			0x68, 0xa3, 0x9c, 0x9f, 0x01, 0x0c, 0x04, 0xc9,
			0xa4, 0x04, 0xc0, 0xe5, 0x9d, 0x6f, 0xb8, 0x8f,
			0x29, 0x6c, 0x20, 0x79, 0x42, 0xd8, 0xf0, 0x0d,
			0xba, 0x15, 0x94, 0x54, 0x87, 0x09, 0xe5, 0x17,
			0x5e, 0x69, 0x59, 0x44, 0x83, 0x37, 0x76, 0x92,
			0xdf, 0x89, 0x8d, 0x68, 0x53, 0x8d, 0xa4, 0xfc,
			0x4c, 0xca, 0xb4, 0xb8, 0x97, 0xa8, 0x15, 0xe1,
			0x77, 0x33, 0x36, 0x7a, 0xf5, 0x1c, 0x91, 0x66,
			0x78, 0x00, 0xde, 0xce, 0x95, 0xed, 0x87, 0xc2,
			0x28, 0x23, 0x5d, 0x23, 0xa4, 0x17, 0x89, 0xc0,
			0xa7, 0xbe, 0xc5, 0xbf, 0xd2, 0x07, 0xcc, 0x3f,
			0xd7, 0xaa, 0x0e, 0x3b, 0xd3, 0x20, 0xa9, 0xb1,
		},
		/* S1 : 128 byte */
		{
			0xb0, 0x94, 0x57, 0xcb, 0x82, 0x09, 0x25, 0xbd,
			0xc5, 0x4e, 0xd3, 0x98, 0x1a, 0x13, 0xe6, 0x40,
			0xcd, 0x9d, 0x8c, 0x96, 0x90, 0x81, 0x56, 0x4b,
			0xc8, 0x26, 0x53, 0x81, 0x35, 0x4c, 0x4a, 0xfe,
			0xa0, 0x34, 0x08, 0x9d, 0x81, 0x98, 0x71, 0x64,
			0xac, 0xf5, 0xa3, 0x1a, 0xea, 0x1f, 0x69, 0x98,
			0x35, 0x66, 0x18, 0xfa, 0xfc, 0xab, 0xbc, 0xd3,
			0x20, 0x37, 0x8c, 0x82, 0x8d, 0x1d, 0xe6, 0x48,
			0x38, 0x58, 0x3d, 0xc0, 0xf6, 0x44, 0x5c, 0x6b,
			0x86, 0x7c, 0x94, 0x36, 0xfb, 0x04, 0x05, 0x70,
			0x18, 0xd7, 0x9b, 0x88, 0x7f, 0xd7, 0x46, 0xc7,
			0xab, 0x07, 0x89, 0xb5, 0x4e, 0x67, 0x17, 0xcb,
			0x31, 0xda, 0x9e, 0x74, 0x68, 0x8b, 0xe7, 0xe7,
			0xf0, 0xa1, 0xb6, 0x5a, 0x44, 0xf8, 0xfc, 0xe2,
			0x93, 0x30, 0xc4, 0x5e, 0x79, 0x56, 0x4e, 0x45,
			0x2e, 0x83, 0x3b, 0xd9, 0x73, 0x62, 0xbd, 0xa6,
		},
		20, /* salt length */
		/* V1 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA384 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x46, 0xd0, 0xb4, 0xb0, 0x35, 0xe9, 0xf1, 0xa1,
			0xa9, 0xfc, 0xb1, 0xb7, 0xa5, 0x8a, 0xc1, 0x98,
			0x0a, 0x81, 0xa6, 0xc9, 0x18, 0xb3, 0x3c, 0xc4,
			0x32, 0x87, 0x9d, 0x4b, 0xe1, 0xee, 0x44, 0xfb,
			0x1c, 0x99, 0x73, 0x24, 0xc7, 0x6a, 0xd1, 0xd8,
			0x79, 0x28, 0x58, 0x42, 0x82, 0x76, 0x6e, 0x39,
			0x2d, 0x06, 0x52, 0x15, 0x42, 0x0a, 0xba, 0xc2,
			0xcf, 0xeb, 0xbb, 0x13, 0x28, 0x3e, 0xe8, 0x58,
			0x2d, 0xf4, 0xbe, 0x32, 0xcb, 0xd0, 0x40, 0xa6,
			0xc6, 0x1f, 0xde, 0x71, 0xb5, 0xa3, 0x9b, 0x9b,
			0x4e, 0xd0, 0x6f, 0x4f, 0xe3, 0x3b, 0xea, 0xea,
			0x2d, 0x4f, 0x1b, 0x52, 0x49, 0x41, 0xb0, 0x89,
			0x00, 0x47, 0x5e, 0x82, 0x4b, 0x05, 0x84, 0x33,
			0x16, 0x48, 0x0b, 0xeb, 0x45, 0xf7, 0x90, 0xe6,
			0xf5, 0x47, 0xc5, 0x1f, 0x04, 0x2d, 0x14, 0x65,
			0xd3, 0x19, 0x86, 0xb1, 0x4a, 0xe1, 0xe4, 0xb7,
		},
		/* S2 : 128 byte */
		{
			0x56, 0x1c, 0xbb, 0x9f, 0x03, 0x89, 0x64, 0x17,
			0x36, 0x06, 0xac, 0x41, 0xbe, 0x9d, 0xfb, 0xc9,
			0x22, 0xa9, 0x74, 0x0a, 0xbc, 0xaf, 0x22, 0xe1,
			0x3f, 0x47, 0x5c, 0x03, 0x65, 0xe3, 0xe8, 0xc0,
			0x8b, 0xec, 0x94, 0x0d, 0xd9, 0x58, 0x78, 0xa1,
			0xfd, 0xb4, 0x9c, 0xd1, 0x24, 0x2b, 0xc5, 0x03,
			0x67, 0x7a, 0x83, 0xcf, 0xea, 0xde, 0x94, 0xbe,
			0x63, 0x93, 0x71, 0x5b, 0x52, 0xda, 0x7e, 0xb2,
			0xce, 0xdc, 0x89, 0x3d, 0xcd, 0x95, 0xa7, 0xd4,
			0x57, 0x4b, 0xa1, 0x34, 0x41, 0xb5, 0x23, 0x3e,
			0xb3, 0x56, 0xe9, 0x36, 0xf2, 0xc8, 0xf8, 0x91,
			0xd2, 0x5f, 0xf7, 0x23, 0x20, 0x19, 0xb6, 0xf3,
			0x44, 0xf9, 0x02, 0x95, 0x1e, 0xcb, 0x15, 0x25,
			0x9c, 0xa4, 0xe4, 0x79, 0x06, 0x78, 0x86, 0xe0,
			0xf5, 0xd3, 0x41, 0xfc, 0xf9, 0x5d, 0x25, 0xf7,
			0x44, 0x4d, 0xc3, 0x3c, 0x23, 0x00, 0x24, 0x99,
		},
		20, /* salt length */
		/* V2 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA384 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0x5e, 0x61, 0x14, 0x73, 0xdd, 0x3c, 0xb9, 0x22,
			0x38, 0x30, 0x0e, 0xd5, 0x4a, 0xbd, 0x60, 0x36,
			0x62, 0x04, 0x1b, 0x92, 0xe9, 0xdc, 0x8d, 0x8d,
			0x45, 0x23, 0xec, 0x15, 0xfc, 0x52, 0x9b, 0x94,
			0x1d, 0x54, 0xa9, 0x6a, 0xdc, 0x99, 0x9b, 0x9a,
			0x7c, 0x66, 0x6b, 0xc3, 0x72, 0x6f, 0xd0, 0x53,
			0xc7, 0xa7, 0xc0, 0xf7, 0xbe, 0x57, 0x33, 0x56,
			0xb7, 0x6d, 0x2b, 0xfe, 0x53, 0x17, 0xf1, 0x9e,
			0xd9, 0x91, 0xa1, 0x77, 0xd8, 0x3c, 0xe8, 0x0d,
			0x0e, 0xef, 0x0d, 0x22, 0x89, 0x91, 0x2e, 0xe4,
			0x0c, 0xe8, 0xcd, 0x66, 0xb4, 0xac, 0x8b, 0x4d,
			0xdb, 0xe0, 0x32, 0xcc, 0x3d, 0x62, 0xf9, 0xd2,
			0x59, 0xc0, 0x04, 0x81, 0x1b, 0x4d, 0x2b, 0xe3,
			0xb7, 0x74, 0xfc, 0xd8, 0xd8, 0x4d, 0x03, 0x53,
			0xfa, 0x6e, 0x49, 0xc6, 0x14, 0x04, 0x1a, 0xdb,
			0x7b, 0x22, 0x0b, 0x65, 0x03, 0x58, 0x3c, 0x96,
		},
		/* S3 : 128 byte */
		{
			0x3d, 0x01, 0xf5, 0xa9, 0x51, 0x9e, 0xe8, 0x1f,
			0xb9, 0x3f, 0x73, 0x1b, 0xa6, 0x8e, 0x2a, 0xcf,
			0x16, 0x77, 0xb1, 0x4c, 0x50, 0x15, 0x70, 0x31,
			0x3f, 0x3b, 0x12, 0xfa, 0x12, 0x40, 0xdc, 0x95,
			0x50, 0x7d, 0x6d, 0xa6, 0xb7, 0xaf, 0x3e, 0x55,
			0xf4, 0xcd, 0x3d, 0x4f, 0x01, 0x1f, 0x98, 0x81,
			0x5a, 0x2c, 0xa8, 0xbb, 0xd8, 0x20, 0xa7, 0xb1,
			0xe1, 0xd4, 0x8b, 0x38, 0x83, 0x2b, 0x09, 0x25,
			0x71, 0x91, 0x44, 0xf9, 0xa2, 0x0b, 0x72, 0xf0,
			0x79, 0xa8, 0xcf, 0x65, 0x84, 0xf9, 0x44, 0x39,
			0xb6, 0x2e, 0x00, 0xb7, 0x0f, 0x4d, 0xa8, 0x28,
			0x63, 0xcd, 0x82, 0xb8, 0x66, 0x79, 0x14, 0xd7,
			0x6f, 0x91, 0x27, 0x98, 0x66, 0xe3, 0xd7, 0x59,
			0x3f, 0x30, 0x23, 0x96, 0x2c, 0x0b, 0x27, 0x34,
			0x04, 0x90, 0x0d, 0xaf, 0x49, 0x27, 0xe7, 0x35,
			0xa6, 0xa9, 0xfa, 0x71, 0x6f, 0xa4, 0x1f, 0xcf,
		},
		20, /* salt length */
		/* V3 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA384 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0xf1, 0x4b, 0x4e, 0x3f, 0xe8, 0x2b, 0x9c, 0xcd,
			0x1d, 0x37, 0xb5, 0x05, 0x28, 0x37, 0x97, 0x01,
			0x95, 0x15, 0xcd, 0x2d, 0xe7, 0xd9, 0x93, 0xe6,
			0xdc, 0xd7, 0x3a, 0x47, 0xf3, 0x27, 0xce, 0x3f,
			0x2c, 0x56, 0xf7, 0xe7, 0x0f, 0x5b, 0x1d, 0x13,
			0xef, 0xcd, 0xcf, 0x28, 0x70, 0x53, 0x66, 0x2d,
			0x47, 0x5b, 0xe5, 0x24, 0x41, 0x0f, 0x24, 0x9a,
			0x87, 0x84, 0x2e, 0x7d, 0x20, 0x35, 0xe7, 0x2e,
			0xb7, 0xb1, 0xf2, 0xd7, 0x9f, 0x90, 0xd1, 0xff,
			0x14, 0xab, 0x4e, 0x5b, 0xc4, 0x8d, 0x1a, 0xe7,
			0x8e, 0x24, 0xb5, 0x36, 0x89, 0xbc, 0x8d, 0x06,
			0x48, 0x2f, 0x6a, 0x18, 0x5c, 0x55, 0x09, 0x3f,
			0x1b, 0xc2, 0x07, 0xb6, 0xa0, 0xb1, 0x02, 0xb6,
			0x13, 0x3b, 0xfa, 0x48, 0x2b, 0xcd, 0xfd, 0x59,
			0xf0, 0xd8, 0x03, 0x0c, 0xc6, 0xca, 0x17, 0x54,
			0xa7, 0x3e, 0xb7, 0xc1, 0xb0, 0x6f, 0x7e, 0x9a,
		},
		/* S4 : 128 byte */
		{
			0x66, 0xff, 0x3f, 0xb0, 0xc5, 0x83, 0x9e, 0x83,
			0x19, 0x1f, 0xfe, 0x6d, 0xbd, 0x57, 0xd9, 0x12,
			0x99, 0xbc, 0xd6, 0x53, 0xf6, 0xaa, 0x1c, 0x8a,
			0x53, 0x92, 0x16, 0x47, 0xa6, 0x2d, 0xb1, 0x66,
			0x55, 0x74, 0xff, 0xc4, 0xd0, 0xd2, 0x1a, 0xcf,
			0x55, 0x54, 0xfb, 0x0f, 0xbd, 0xfd, 0xab, 0x32,
			0xa4, 0xee, 0x4b, 0x2e, 0x93, 0x4b, 0x23, 0x72,
			0xb4, 0xf5, 0x03, 0xa8, 0xee, 0x59, 0xe6, 0xf1,
			0x08, 0x42, 0x14, 0x03, 0x6a, 0xa5, 0x37, 0x9c,
			0xf3, 0x04, 0x87, 0xbb, 0x0f, 0x3c, 0x44, 0x8c,
			0x24, 0x7d, 0x19, 0xb5, 0xd1, 0xd9, 0x01, 0xaf,
			0x9d, 0x2f, 0xdf, 0xe6, 0x86, 0x29, 0x08, 0x27,
			0xf5, 0x76, 0xf1, 0xda, 0xd2, 0x67, 0xf1, 0xde,
			0x0b, 0x37, 0xe8, 0x47, 0x00, 0xbc, 0xc1, 0x25,
			0xaa, 0x47, 0xb1, 0x66, 0x1d, 0xdd, 0x5f, 0x32,
			0xe1, 0x41, 0x8c, 0x23, 0x83, 0x5e, 0x7e, 0x33,
		},
		20, /* salt length */
		/* V4 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA384 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0xe5, 0x11, 0x90, 0x3c, 0x2f, 0x1b, 0xfb, 0xa2,
			0x45, 0x46, 0x72, 0x95, 0xac, 0x95, 0x41, 0x3a,
			0xc4, 0x74, 0x6c, 0x98, 0x4c, 0x37, 0x50, 0xa7,
			0x28, 0xc3, 0x88, 0xaa, 0x62, 0x8b, 0x0e, 0xbf,
			0x70, 0xbf, 0xa2, 0xb5, 0xb5, 0xbf, 0x70, 0x1d,
			0x4a, 0xd6, 0xed, 0xa4, 0xfd, 0x25, 0x02, 0xa0,
			0x9a, 0xdd, 0x6c, 0xe8, 0x6a, 0xb2, 0x63, 0xe9,
			0xfa, 0x8c, 0x79, 0xf6, 0x1d, 0x7f, 0xc6, 0xe0,
			0x34, 0xfd, 0x42, 0xc7, 0xad, 0x4a, 0x72, 0x2f,
			0xca, 0x31, 0x47, 0x56, 0xa4, 0xf6, 0x43, 0xc6,
			0x6f, 0x8b, 0xdd, 0x26, 0xff, 0x06, 0x27, 0x55,
			0x8b, 0x53, 0xae, 0xba, 0xb2, 0x26, 0xa1, 0xd5,
			0xbb, 0x34, 0x36, 0x1a, 0x21, 0xa3, 0x4a, 0x00,
			0x4f, 0xbf, 0x48, 0x55, 0xa8, 0xee, 0x4b, 0xd4,
			0xca, 0xe6, 0x87, 0xf9, 0x14, 0x1d, 0xe5, 0x14,
			0x66, 0x81, 0xed, 0x9d, 0x27, 0x6a, 0xe9, 0xc7,
		},
		/* S5 : 128 byte */
		{
			0x9c, 0x74, 0x87, 0x02, 0xbb, 0xcc, 0x1f, 0x94,
			0x68, 0x86, 0x4c, 0xd3, 0x60, 0xc8, 0xc3, 0x9d,
			0x00, 0x7b, 0x2d, 0x8a, 0xae, 0xe8, 0x33, 0x60,
			0x6c, 0x70, 0xf7, 0x59, 0x3c, 0xf0, 0xd1, 0x51,
			0x92, 0x4e, 0xac, 0x33, 0xd3, 0x6b, 0x6b, 0x81,
			0x50, 0x19, 0xfa, 0x05, 0x75, 0xa5, 0x18, 0xde,
			0x4e, 0xa8, 0xce, 0x8a, 0x3a, 0x8e, 0x31, 0xc4,
			0x24, 0x2d, 0x34, 0x71, 0xb3, 0x0b, 0xc1, 0x98,
			0xda, 0xb3, 0x41, 0xbe, 0xa9, 0x77, 0xec, 0xcc,
			0x4f, 0x69, 0xb8, 0xfb, 0x4b, 0xa2, 0x1b, 0x0b,
			0x90, 0xbf, 0xc0, 0x47, 0x8f, 0x2e, 0x34, 0xb3,
			0x20, 0x06, 0xeb, 0x7b, 0xf9, 0x15, 0xf7, 0x2d,
			0xa2, 0x47, 0xe1, 0x3c, 0xdc, 0x6d, 0x00, 0xff,
			0xe3, 0x8c, 0x28, 0x53, 0x03, 0x0c, 0x83, 0x2e,
			0x4c, 0x06, 0x5f, 0x8a, 0xc3, 0x35, 0x0e, 0xf4,
			0x03, 0xa8, 0x95, 0x3f, 0x95, 0x1e, 0x08, 0x32,
		},
		20, /* salt length */
		/* V5 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA384 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0x10, 0x34, 0xe0, 0x43, 0xd5, 0x21, 0xa2, 0x9e,
			0x1f, 0x6d, 0x84, 0xef, 0x8a, 0x54, 0x9b, 0x86,
			0x75, 0xc5, 0xeb, 0xe4, 0xab, 0x74, 0x2e, 0x72,
			0x43, 0x2b, 0xd1, 0x72, 0xc4, 0x60, 0x40, 0x35,
			0x8d, 0x0d, 0x63, 0x8e, 0x74, 0xbc, 0x68, 0x88,
			0xae, 0xe1, 0xc4, 0x54, 0xef, 0x6d, 0x74, 0x85,
			0xd1, 0xa0, 0x89, 0x80, 0x7a, 0x1f, 0xe3, 0x2d,
			0x79, 0x46, 0x3d, 0xf3, 0xf2, 0xbf, 0x6a, 0xe4,
			0xfb, 0x54, 0xdd, 0x34, 0x45, 0x01, 0x6f, 0xf0,
			0xff, 0x4e, 0x5d, 0xec, 0xf7, 0xdc, 0x90, 0xfe,
			0x03, 0x3d, 0x22, 0x69, 0x9f, 0x08, 0x1d, 0x9c,
			0xd7, 0x2d, 0x86, 0x1d, 0x68, 0x87, 0x9a, 0xc0,
			0x22, 0xb0, 0x57, 0x1c, 0x90, 0x19, 0x3e, 0x3f,
			0xdb, 0x5a, 0x58, 0xa1, 0xec, 0xc6, 0x77, 0x57,
			0xd1, 0x43, 0xbe, 0x89, 0x62, 0x59, 0x63, 0x26,
			0x24, 0x88, 0xb4, 0xd6, 0x6d, 0xdf, 0x58, 0x31,
		},
		/* S6 : 128 byte */
		{
			0x0f, 0xdc, 0x48, 0xfa, 0x2f, 0xe5, 0x6e, 0xd5,
			0x35, 0xf7, 0xe1, 0xb0, 0x32, 0x05, 0x4c, 0xcc,
			0x64, 0xa7, 0x36, 0x3e, 0xb1, 0x45, 0xeb, 0x01,
			0xb2, 0xe8, 0x4e, 0xfa, 0x35, 0x88, 0xde, 0xf1,
			0xf3, 0x31, 0x25, 0xcf, 0x71, 0xa8, 0xef, 0x5b,
			0x6f, 0x2f, 0x7b, 0x6a, 0xe1, 0xe6, 0x16, 0xd9,
			0xaf, 0xcc, 0xd7, 0x37, 0x2c, 0x84, 0x3a, 0x97,
			0x0f, 0x16, 0x5c, 0x10, 0x3a, 0x69, 0x41, 0x63,
			0x18, 0xf5, 0x57, 0x4f, 0xf3, 0xb2, 0x86, 0x99,
			0x2b, 0x78, 0xcb, 0x8e, 0x77, 0x0f, 0x85, 0xa1,
			0xbd, 0x0f, 0x0f, 0x0b, 0x3d, 0xdf, 0xb7, 0x77,
			0x12, 0xb4, 0x27, 0x7c, 0x19, 0x10, 0xc6, 0x8a,
			0xa8, 0xc6, 0x00, 0x6c, 0xe8, 0x46, 0x3a, 0x45,
			0xc9, 0x32, 0xa8, 0x4c, 0x23, 0x34, 0xf2, 0xf1,
			0xd4, 0xf6, 0x7e, 0x7e, 0x55, 0x01, 0x07, 0x56,
			0x86, 0xcf, 0x80, 0x96, 0xb1, 0x75, 0x14, 0xc6,
		},
		20, /* salt length */
		/* V6 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA384 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0x12, 0x67, 0xd9, 0xe2, 0x9f, 0x0f, 0x39, 0xfc,
			0xe1, 0xd1, 0x92, 0x4f, 0x0c, 0x2a, 0x43, 0x51,
			0x09, 0x68, 0x2b, 0xf0, 0x60, 0x03, 0xb4, 0x84,
			0x5b, 0x30, 0x10, 0x02, 0x69, 0xb4, 0xc3, 0x72,
			0xc0, 0xf6, 0x62, 0xa3, 0x46, 0x59, 0x9a, 0x1d,
			0xb2, 0xc8, 0x73, 0x57, 0xee, 0x93, 0xd6, 0x2a,
			0xc0, 0x79, 0x4e, 0x86, 0xaa, 0x06, 0x0b, 0x64,
			0x0f, 0x1f, 0x2c, 0xfb, 0x96, 0xdb, 0x3e, 0x37,
			0xd4, 0x94, 0xff, 0x67, 0x25, 0x3e, 0x68, 0x36,
			0xda, 0xf8, 0xeb, 0xdc, 0x9a, 0xe1, 0xfa, 0x95,
			0x02, 0x2d, 0xc8, 0x52, 0x8f, 0x78, 0x53, 0xaf,
			0xbe, 0x59, 0xa2, 0x26, 0x71, 0x3b, 0x4b, 0x82,
			0x95, 0x68, 0x39, 0x96, 0x02, 0x67, 0xac, 0x2e,
			0xa2, 0x88, 0x81, 0x46, 0xd6, 0xbd, 0xb4, 0x7a,
			0x15, 0xab, 0x23, 0xbe, 0xd7, 0x01, 0x95, 0x4f,
			0xb1, 0x09, 0xfa, 0xea, 0xc1, 0x7d, 0xf5, 0x3b,
		},
		/* S7 : 128 byte */
		{
			0xb8, 0x2a, 0xff, 0xc8, 0x32, 0x4e, 0x78, 0x92,
			0x7d, 0x46, 0x6a, 0x57, 0x72, 0x6d, 0xeb, 0x19,
			0x20, 0x24, 0xa3, 0xd0, 0xca, 0x15, 0x24, 0xb1,
			0xc5, 0xad, 0x5c, 0x08, 0x86, 0x44, 0x6c, 0x9c,
			0x33, 0x1f, 0x7c, 0x20, 0xfe, 0x19, 0x8b, 0x92,
			0xd1, 0x7f, 0x7f, 0xe5, 0x5b, 0x15, 0x47, 0x4a,
			0x4d, 0x0a, 0xe0, 0xf5, 0xe0, 0x99, 0x86, 0x78,
			0x93, 0x23, 0xfb, 0xa7, 0xd8, 0x3d, 0xca, 0x1f,
			0x35, 0xb2, 0xcf, 0xd8, 0x06, 0xf3, 0x50, 0xf3,
			0x54, 0x5c, 0x1f, 0x7a, 0x5a, 0x2c, 0xec, 0xca,
			0x45, 0xfe, 0x13, 0x17, 0xe6, 0x01, 0x7c, 0xa5,
			0xca, 0x10, 0xaf, 0x41, 0x4a, 0x0f, 0x00, 0xcc,
			0x94, 0xb9, 0xd2, 0xec, 0xd8, 0xc4, 0xd0, 0xd5,
			0xe7, 0xd2, 0xcc, 0xe6, 0x3a, 0x16, 0xe2, 0x48,
			0xb9, 0x93, 0x2f, 0xa8, 0xfa, 0xd8, 0xa5, 0x62,
			0xb8, 0xbf, 0x82, 0x4b, 0xda, 0x7d, 0x92, 0xd4,
		},
		20, /* salt length */
		/* V7 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA384 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0xb0, 0x66, 0x48, 0x83, 0x5d, 0x99, 0x3a, 0x4c,
			0x66, 0x94, 0x4a, 0x5c, 0xb9, 0x5a, 0xfd, 0x74,
			0x1c, 0x66, 0x70, 0x8b, 0xc4, 0xf0, 0x92, 0x28,
			0x23, 0x6d, 0x91, 0x10, 0x01, 0x55, 0x92, 0x21,
			0xce, 0x28, 0xa5, 0x7e, 0x37, 0xb3, 0xf8, 0xec,
			0xf7, 0x90, 0x4b, 0x7c, 0xf0, 0xec, 0xa3, 0x6b,
			0x7a, 0x8c, 0x43, 0x46, 0xfb, 0x12, 0xd8, 0x51,
			0xb8, 0x7d, 0xf4, 0x2a, 0xea, 0xf0, 0xbd, 0xaa,
			0x15, 0xb4, 0xe7, 0x0c, 0x95, 0x7f, 0xdf, 0x51,
			0x79, 0xf7, 0x49, 0xe2, 0x95, 0x72, 0xad, 0x78,
			0xd3, 0x00, 0x6d, 0x39, 0xca, 0xab, 0xfb, 0x24,
			0x92, 0xfd, 0x48, 0xae, 0xd6, 0xc3, 0x80, 0xad,
			0x2a, 0x11, 0xf4, 0x76, 0x9f, 0x86, 0x58, 0x42,
			0x95, 0xd1, 0x2e, 0xd7, 0xd1, 0x74, 0x82, 0x64,
			0xd0, 0xc8, 0xd8, 0xac, 0x5a, 0x42, 0x63, 0x66,
			0x5a, 0xf4, 0x77, 0xe4, 0xec, 0x57, 0x41, 0xf4,
		},
		/* S8 : 128 byte */
		{
			0x84, 0x90, 0xa7, 0x8e, 0x4c, 0x23, 0x69, 0x70,
			0xc0, 0xc3, 0xd7, 0xcd, 0xaa, 0xca, 0xc9, 0x0f,
			0x00, 0xb2, 0xa3, 0xb3, 0x2a, 0x6d, 0x81, 0xd7,
			0xec, 0xbe, 0x8a, 0x79, 0xf2, 0xf9, 0x78, 0x28,
			0x61, 0xcb, 0xc7, 0x2c, 0xab, 0xa0, 0xc2, 0xe2,
			0x22, 0x0a, 0x47, 0x46, 0x03, 0x87, 0xf7, 0x29,
			0x18, 0xf2, 0xfc, 0x40, 0x1c, 0xfc, 0x92, 0xe7,
			0xf0, 0x0d, 0xaa, 0xa5, 0x14, 0xcb, 0x1c, 0xef,
			0x34, 0x31, 0x00, 0xb5, 0x6c, 0x38, 0x86, 0x34,
			0x17, 0xfc, 0xe3, 0x65, 0x0e, 0x8b, 0x96, 0xfc,
			0xca, 0x05, 0xfe, 0xbf, 0xb8, 0x7b, 0x1f, 0x2e,
			0xc7, 0xe3, 0x59, 0x9d, 0xdc, 0xe7, 0x1c, 0x18,
			0x55, 0x29, 0xfa, 0x22, 0xe8, 0x08, 0x04, 0xfb,
			0xbc, 0xec, 0xe5, 0x74, 0x51, 0x65, 0x36, 0xd6,
			0x6a, 0x8c, 0x30, 0x3b, 0x6d, 0x81, 0x9d, 0x50,
			0xa2, 0x06, 0x63, 0x06, 0x7b, 0x6b, 0x54, 0x60,
		},
		20, /* salt length */
		/* V8 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA384 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0x5d, 0x95, 0x51, 0xdf, 0x21, 0x0c, 0xea, 0x61,
			0xf9, 0xae, 0xc3, 0xd2, 0xef, 0x2f, 0x57, 0x16,
			0x69, 0x2c, 0x5c, 0x55, 0xff, 0xb1, 0x3d, 0xa2,
			0x8a, 0xfe, 0xca, 0xda, 0x7d, 0x8b, 0x2f, 0xcf,
			0x79, 0x49, 0x22, 0xa6, 0x1b, 0x2b, 0x61, 0x7e,
			0x19, 0xe5, 0x5c, 0x53, 0x8d, 0xf4, 0x2e, 0x21,
			0x90, 0xde, 0xd2, 0x4c, 0x0b, 0xbb, 0x94, 0x3f,
			0xca, 0xf4, 0xc5, 0x53, 0xd2, 0x4b, 0xb3, 0xf3,
			0xf5, 0x1b, 0xd9, 0xff, 0xa3, 0xe1, 0x52, 0x6d,
			0x03, 0xfc, 0x94, 0x36, 0x42, 0xb1, 0x5a, 0x70,
			0x2c, 0x72, 0xe2, 0xb7, 0xf9, 0x60, 0x59, 0xd2,
			0x69, 0x0e, 0x94, 0x94, 0x13, 0x88, 0x00, 0xff,
			0xf9, 0x94, 0xdc, 0xb0, 0x61, 0x37, 0xa5, 0x77,
			0xe0, 0x7b, 0xd2, 0x45, 0xb8, 0x70, 0xc2, 0x4b,
			0x64, 0x71, 0xf3, 0x60, 0x70, 0xa8, 0x74, 0xa3,
			0x8e, 0xd5, 0x9d, 0x62, 0xe7, 0x4a, 0xfd, 0x3b,
		},
		/* S9 : 128 byte */
		{
			0x59, 0xce, 0x3e, 0xfb, 0xd1, 0x92, 0xd6, 0xd8,
			0xb5, 0x3f, 0x59, 0x1c, 0x04, 0x79, 0xd5, 0x71,
			0x04, 0x84, 0x9b, 0x6e, 0x3d, 0x1f, 0x91, 0xbe,
			0xc2, 0x7c, 0x54, 0xf1, 0x72, 0x75, 0xd2, 0x3e,
			0xd2, 0x54, 0xa4, 0xde, 0x98, 0x3a, 0xaf, 0x43,
			0x97, 0x1e, 0xcc, 0xd5, 0x4b, 0x4c, 0xd6, 0xd9,
			0xc2, 0xdc, 0xc0, 0xa6, 0xf6, 0xc8, 0x19, 0xdf,
			0x0e, 0x85, 0xab, 0x59, 0xe2, 0x11, 0x35, 0xe6,
			0x53, 0x63, 0x18, 0x13, 0xf5, 0x54, 0x0c, 0xf8,
			0xaa, 0x65, 0xa7, 0x1b, 0xb1, 0x46, 0x54, 0xb4,
			0xae, 0x03, 0x79, 0xd5, 0xf0, 0x70, 0x88, 0x9c,
			0x57, 0xc2, 0x3a, 0x74, 0xcd, 0x6e, 0x17, 0x2e,
			0xa7, 0xb6, 0xfa, 0xa7, 0x74, 0xdf, 0x21, 0x43,
			0x77, 0x7b, 0x56, 0x7a, 0x63, 0x8b, 0xe8, 0x6e,
			0xba, 0x11, 0xdd, 0x04, 0x0d, 0xe4, 0x76, 0x32,
			0x92, 0x3d, 0xdd, 0x71, 0xaa, 0x0f, 0x94, 0xa6,
		},
		20, /* salt length */
		/* V9 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
};

/* 10 elements */
static mod1024_t SHA512[] = {
	/* SHA512 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0x12, 0x48, 0xf6, 0x2a, 0x43, 0x89, 0xf4, 0x2f,
			0x7b, 0x4b, 0xb1, 0x31, 0x05, 0x3d, 0x6c, 0x88,
			0xa9, 0x94, 0xdb, 0x20, 0x75, 0xb9, 0x12, 0xcc,
			0xbe, 0x3e, 0xa7, 0xdc, 0x61, 0x17, 0x14, 0xf1,
			0x4e, 0x07, 0x5c, 0x10, 0x48, 0x58, 0xf2, 0xf6,
			0xe6, 0xcf, 0xd6, 0xab, 0xde, 0xdf, 0x01, 0x5a,
			0x82, 0x1d, 0x03, 0x60, 0x8b, 0xf4, 0xeb, 0xa3,
			0x16, 0x9a, 0x67, 0x25, 0xec, 0x42, 0x2c, 0xd9,
			0x06, 0x94, 0x98, 0xb5, 0x51, 0x5a, 0x96, 0x08,
			0xae, 0x7c, 0xc3, 0x0e, 0x3d, 0x2e, 0xcf, 0xc1,
			0xdb, 0x68, 0x25, 0xf3, 0xe9, 0x96, 0xce, 0x9a,
			0x50, 0x92, 0x92, 0x6b, 0xc1, 0xcf, 0x61, 0xaa,
			0x42, 0xd7, 0xf2, 0x40, 0xe6, 0xf7, 0xaa, 0x0e,
			0xdb, 0x38, 0xbf, 0x81, 0xaa, 0x92, 0x9d, 0x66,
			0xbb, 0x5d, 0x89, 0x00, 0x18, 0x08, 0x84, 0x58,
			0x72, 0x0d, 0x72, 0xd5, 0x69, 0x24, 0x7b, 0x0c,
		},
		/* S0 : 128 byte */
		{
			0xa8, 0x33, 0xba, 0x31, 0x63, 0x4f, 0x87, 0x73,
			0xe4, 0xfe, 0x6e, 0xa0, 0xc6, 0x9e, 0x1a, 0x23,
			0x76, 0x6a, 0x93, 0x9d, 0x34, 0xb3, 0x2f, 0xc7,
			0x8b, 0x77, 0x4b, 0x22, 0xe4, 0x6a, 0x64, 0x6c,
			0x25, 0xe6, 0xe1, 0x06, 0x2d, 0x23, 0x4e, 0xd4,
			0x8b, 0x1a, 0xba, 0x0f, 0x83, 0x05, 0x29, 0xff,
			0x6a, 0xfc, 0x29, 0x6c, 0xc8, 0xdc, 0x20, 0x7b,
			0xbc, 0x15, 0x39, 0x16, 0x23, 0xbe, 0xac, 0x5f,
			0x6c, 0x3d, 0xb5, 0x57, 0xca, 0x49, 0xd0, 0xe4,
			0x2c, 0x96, 0x2d, 0xe9, 0x5b, 0x5f, 0xf5, 0x48,
			0xcf, 0xf9, 0x70, 0xf5, 0xc7, 0x3f, 0x43, 0x9c,
			0xfe, 0x82, 0xd3, 0x90, 0x7b, 0xe6, 0x02, 0x40,
			0xf5, 0x6b, 0x6a, 0x42, 0x59, 0xcc, 0x96, 0xdf,
			0xd8, 0xfe, 0x02, 0xa0, 0xbf, 0xa2, 0x6e, 0x02,
			0x23, 0xf6, 0x82, 0x14, 0x42, 0x8f, 0xff, 0x0a,
			0xe4, 0x01, 0x62, 0x19, 0x8c, 0xc5, 0xcb, 0xd1,
		},
		20, /* salt length */
		/* V0 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA512 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 0x2b, 0x90, 0x75, 0x7b,
			0x77, 0xc7, 0xc2, 0x01, 0xe0, 0xdf, 0xbe, 0x7b,
			0x68, 0xa3, 0x9c, 0x9f, 0x01, 0x0c, 0x04, 0xc9,
			0xa4, 0x04, 0xc0, 0xe5, 0x9d, 0x6f, 0xb8, 0x8f,
			0x29, 0x6c, 0x20, 0x79, 0x42, 0xd8, 0xf0, 0x0d,
			0xba, 0x15, 0x94, 0x54, 0x87, 0x09, 0xe5, 0x17,
			0x5e, 0x69, 0x59, 0x44, 0x83, 0x37, 0x76, 0x92,
			0xdf, 0x89, 0x8d, 0x68, 0x53, 0x8d, 0xa4, 0xfc,
			0x4c, 0xca, 0xb4, 0xb8, 0x97, 0xa8, 0x15, 0xe1,
			0x77, 0x33, 0x36, 0x7a, 0xf5, 0x1c, 0x91, 0x66,
			0x78, 0x00, 0xde, 0xce, 0x95, 0xed, 0x87, 0xc2,
			0x28, 0x23, 0x5d, 0x23, 0xa4, 0x17, 0x89, 0xc0,
			0xa7, 0xbe, 0xc5, 0xbf, 0xd2, 0x07, 0xcc, 0x3f,
			0xd7, 0xaa, 0x0e, 0x3b, 0xd3, 0x20, 0xa9, 0xb1,
		},
		/* S1 : 128 byte */
		{
			0xaf, 0x2d, 0x6d, 0xc7, 0xfc, 0x64, 0xb1, 0xbb,
			0xb5, 0x7a, 0xf3, 0x8d, 0xef, 0xe3, 0xfa, 0xa2,
			0xdb, 0x9d, 0x64, 0xb7, 0x30, 0x10, 0x04, 0xd7,
			0xca, 0x9f, 0x8e, 0x4a, 0x46, 0x27, 0x7e, 0x35,
			0x94, 0xfd, 0x0b, 0xae, 0xbf, 0x13, 0xa8, 0xa4,
			0x4d, 0x78, 0xda, 0x16, 0x55, 0x36, 0xb3, 0xb9,
			0x50, 0x43, 0x95, 0xba, 0x78, 0x70, 0xa7, 0x5c,
			0x6d, 0xc9, 0xf2, 0x22, 0x9e, 0xd0, 0xa3, 0x49,
			0x32, 0x6c, 0xa1, 0x15, 0xc5, 0x49, 0xd4, 0x12,
			0xfc, 0xa3, 0x2e, 0x22, 0xf1, 0x1e, 0x13, 0x29,
			0xc4, 0x33, 0xf8, 0x8a, 0x24, 0x10, 0x6e, 0xa1,
			0x54, 0x9e, 0xbc, 0x45, 0x03, 0xba, 0xd9, 0x00,
			0x19, 0x63, 0x54, 0xa6, 0x19, 0xcb, 0xa6, 0xff,
			0xd0, 0xb4, 0x9e, 0xc5, 0x0e, 0x00, 0xd3, 0x95,
			0xe5, 0x4a, 0x48, 0x83, 0x55, 0x0b, 0xf3, 0xa0,
			0xa0, 0xfd, 0x67, 0xd3, 0xdb, 0xff, 0x7d, 0xbc,
		},
		20, /* salt length */
		/* V1 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA512 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x46, 0xd0, 0xb4, 0xb0, 0x35, 0xe9, 0xf1, 0xa1,
			0xa9, 0xfc, 0xb1, 0xb7, 0xa5, 0x8a, 0xc1, 0x98,
			0x0a, 0x81, 0xa6, 0xc9, 0x18, 0xb3, 0x3c, 0xc4,
			0x32, 0x87, 0x9d, 0x4b, 0xe1, 0xee, 0x44, 0xfb,
			0x1c, 0x99, 0x73, 0x24, 0xc7, 0x6a, 0xd1, 0xd8,
			0x79, 0x28, 0x58, 0x42, 0x82, 0x76, 0x6e, 0x39,
			0x2d, 0x06, 0x52, 0x15, 0x42, 0x0a, 0xba, 0xc2,
			0xcf, 0xeb, 0xbb, 0x13, 0x28, 0x3e, 0xe8, 0x58,
			0x2d, 0xf4, 0xbe, 0x32, 0xcb, 0xd0, 0x40, 0xa6,
			0xc6, 0x1f, 0xde, 0x71, 0xb5, 0xa3, 0x9b, 0x9b,
			0x4e, 0xd0, 0x6f, 0x4f, 0xe3, 0x3b, 0xea, 0xea,
			0x2d, 0x4f, 0x1b, 0x52, 0x49, 0x41, 0xb0, 0x89,
			0x00, 0x47, 0x5e, 0x82, 0x4b, 0x05, 0x84, 0x33,
			0x16, 0x48, 0x0b, 0xeb, 0x45, 0xf7, 0x90, 0xe6,
			0xf5, 0x47, 0xc5, 0x1f, 0x04, 0x2d, 0x14, 0x65,
			0xd3, 0x19, 0x86, 0xb1, 0x4a, 0xe1, 0xe4, 0xb7,
		},
		/* S2 : 128 byte */
		{
			0xa2, 0xe5, 0x50, 0xf0, 0x26, 0x18, 0xbd, 0x7a,
			0xb3, 0x29, 0x43, 0x54, 0x36, 0x97, 0xfe, 0x28,
			0x7d, 0x26, 0xc7, 0xa0, 0x04, 0xe1, 0x24, 0x63,
			0x27, 0x5b, 0x0d, 0xb6, 0x2b, 0xb7, 0x4e, 0x48,
			0x1e, 0xc5, 0xef, 0x66, 0xb6, 0x30, 0x18, 0xa0,
			0x54, 0xe6, 0x42, 0x32, 0xbc, 0x41, 0xe4, 0x2d,
			0xf0, 0xcd, 0x22, 0x32, 0x12, 0x41, 0x17, 0x5b,
			0x08, 0xdf, 0x8d, 0xff, 0x7b, 0x7c, 0x4c, 0x5b,
			0xc4, 0x2d, 0x0f, 0x68, 0xf1, 0xb1, 0x62, 0xf5,
			0x0f, 0xe3, 0x38, 0x0b, 0x54, 0x7d, 0xb3, 0x00,
			0xa0, 0x6a, 0x32, 0x49, 0x7a, 0xd6, 0x22, 0x3e,
			0xac, 0xc0, 0x96, 0x33, 0xf3, 0x5f, 0x98, 0x9b,
			0xf8, 0x68, 0xd7, 0x7b, 0x24, 0x08, 0xd2, 0x04,
			0x06, 0x03, 0x72, 0xf0, 0x2d, 0x5c, 0x7d, 0xfb,
			0x90, 0xc5, 0x81, 0xd8, 0x5f, 0xd0, 0x1f, 0x9d,
			0x5f, 0x94, 0xd7, 0x7d, 0x77, 0x25, 0x32, 0xea,
		},
		20, /* salt length */
		/* V2 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA512 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0x5e, 0x61, 0x14, 0x73, 0xdd, 0x3c, 0xb9, 0x22,
			0x38, 0x30, 0x0e, 0xd5, 0x4a, 0xbd, 0x60, 0x36,
			0x62, 0x04, 0x1b, 0x92, 0xe9, 0xdc, 0x8d, 0x8d,
			0x45, 0x23, 0xec, 0x15, 0xfc, 0x52, 0x9b, 0x94,
			0x1d, 0x54, 0xa9, 0x6a, 0xdc, 0x99, 0x9b, 0x9a,
			0x7c, 0x66, 0x6b, 0xc3, 0x72, 0x6f, 0xd0, 0x53,
			0xc7, 0xa7, 0xc0, 0xf7, 0xbe, 0x57, 0x33, 0x56,
			0xb7, 0x6d, 0x2b, 0xfe, 0x53, 0x17, 0xf1, 0x9e,
			0xd9, 0x91, 0xa1, 0x77, 0xd8, 0x3c, 0xe8, 0x0d,
			0x0e, 0xef, 0x0d, 0x22, 0x89, 0x91, 0x2e, 0xe4,
			0x0c, 0xe8, 0xcd, 0x66, 0xb4, 0xac, 0x8b, 0x4d,
			0xdb, 0xe0, 0x32, 0xcc, 0x3d, 0x62, 0xf9, 0xd2,
			0x59, 0xc0, 0x04, 0x81, 0x1b, 0x4d, 0x2b, 0xe3,
			0xb7, 0x74, 0xfc, 0xd8, 0xd8, 0x4d, 0x03, 0x53,
			0xfa, 0x6e, 0x49, 0xc6, 0x14, 0x04, 0x1a, 0xdb,
			0x7b, 0x22, 0x0b, 0x65, 0x03, 0x58, 0x3c, 0x96,
		},
		/* S3 : 128 byte */
		{
			0x8f, 0xf2, 0x14, 0xf4, 0x22, 0xba, 0x0c, 0xf7,
			0x10, 0xf4, 0xb6, 0xeb, 0x1f, 0x85, 0xd2, 0x7b,
			0x92, 0x36, 0x6d, 0x50, 0x45, 0xc7, 0x9d, 0x8a,
			0xdb, 0x98, 0x3e, 0x46, 0xc5, 0xbe, 0x65, 0x44,
			0xd1, 0x31, 0x83, 0x11, 0x9a, 0x95, 0x2b, 0x6a,
			0x3e, 0x51, 0xe2, 0x8a, 0x36, 0x7d, 0x74, 0x41,
			0xa9, 0x47, 0x88, 0xd5, 0x24, 0xfe, 0xf2, 0xfc,
			0x55, 0xc5, 0xbb, 0x82, 0x60, 0xdf, 0xc7, 0x48,
			0xa7, 0x0c, 0xf7, 0xa7, 0xfb, 0x00, 0x21, 0xa9,
			0x79, 0x12, 0xee, 0x89, 0x32, 0xb5, 0xfa, 0xc9,
			0x4c, 0xdb, 0xa1, 0x89, 0xa9, 0xed, 0x84, 0x02,
			0x5a, 0xba, 0x66, 0x2b, 0xf8, 0x94, 0xb9, 0x36,
			0x5c, 0x5a, 0x67, 0x84, 0xaf, 0xb7, 0x31, 0x5c,
			0xf9, 0x38, 0xef, 0xb3, 0x07, 0xa1, 0x0a, 0x14,
			0x45, 0xaa, 0xf3, 0x4f, 0x76, 0x94, 0xc6, 0xd9,
			0x7d, 0x15, 0x6b, 0xad, 0xcd, 0x23, 0xcc, 0x13,
		},
		20, /* salt length */
		/* V3 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA512 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0xf1, 0x4b, 0x4e, 0x3f, 0xe8, 0x2b, 0x9c, 0xcd,
			0x1d, 0x37, 0xb5, 0x05, 0x28, 0x37, 0x97, 0x01,
			0x95, 0x15, 0xcd, 0x2d, 0xe7, 0xd9, 0x93, 0xe6,
			0xdc, 0xd7, 0x3a, 0x47, 0xf3, 0x27, 0xce, 0x3f,
			0x2c, 0x56, 0xf7, 0xe7, 0x0f, 0x5b, 0x1d, 0x13,
			0xef, 0xcd, 0xcf, 0x28, 0x70, 0x53, 0x66, 0x2d,
			0x47, 0x5b, 0xe5, 0x24, 0x41, 0x0f, 0x24, 0x9a,
			0x87, 0x84, 0x2e, 0x7d, 0x20, 0x35, 0xe7, 0x2e,
			0xb7, 0xb1, 0xf2, 0xd7, 0x9f, 0x90, 0xd1, 0xff,
			0x14, 0xab, 0x4e, 0x5b, 0xc4, 0x8d, 0x1a, 0xe7,
			0x8e, 0x24, 0xb5, 0x36, 0x89, 0xbc, 0x8d, 0x06,
			0x48, 0x2f, 0x6a, 0x18, 0x5c, 0x55, 0x09, 0x3f,
			0x1b, 0xc2, 0x07, 0xb6, 0xa0, 0xb1, 0x02, 0xb6,
			0x13, 0x3b, 0xfa, 0x48, 0x2b, 0xcd, 0xfd, 0x59,
			0xf0, 0xd8, 0x03, 0x0c, 0xc6, 0xca, 0x17, 0x54,
			0xa7, 0x3e, 0xb7, 0xc1, 0xb0, 0x6f, 0x7e, 0x9a,
		},
		/* S4 : 128 byte */
		{
			0x28, 0xc8, 0x2d, 0x56, 0x24, 0xa0, 0xf2, 0xa0,
			0xef, 0x61, 0xa6, 0x54, 0xaa, 0xe4, 0x3f, 0x28,
			0x66, 0x93, 0xd3, 0x53, 0x29, 0xb6, 0xcc, 0x7a,
			0x1b, 0x08, 0x49, 0x59, 0x0f, 0x53, 0x69, 0x93,
			0x6a, 0x6a, 0x13, 0x9b, 0x25, 0xdd, 0xef, 0x92,
			0x4c, 0x53, 0x2e, 0x93, 0x9c, 0xa9, 0x54, 0x82,
			0xc3, 0x7c, 0x68, 0x96, 0xfd, 0x14, 0x19, 0xc7,
			0x68, 0x64, 0x78, 0xc0, 0xf5, 0x00, 0x70, 0x03,
			0xb4, 0x2f, 0xc0, 0xfe, 0xfe, 0xa9, 0x2b, 0xbe,
			0xcd, 0x0d, 0xb9, 0xc6, 0x0e, 0x6f, 0xd6, 0x08,
			0xbf, 0x41, 0x69, 0xd0, 0xd3, 0xd5, 0x5c, 0x97,
			0xa5, 0x9d, 0xd9, 0x42, 0xd5, 0xcb, 0x35, 0xd9,
			0x5c, 0x34, 0xd1, 0x11, 0x1c, 0xd1, 0x83, 0x83,
			0x32, 0x94, 0x6b, 0x02, 0xf1, 0xfa, 0xe5, 0x42,
			0xf4, 0x4d, 0xb7, 0xa2, 0xa5, 0x20, 0xb4, 0x7f,
			0x8a, 0x53, 0x68, 0x93, 0x63, 0x09, 0x98, 0xea,
		},
		20, /* salt length */
		/* V4 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA512 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0xe5, 0x11, 0x90, 0x3c, 0x2f, 0x1b, 0xfb, 0xa2,
			0x45, 0x46, 0x72, 0x95, 0xac, 0x95, 0x41, 0x3a,
			0xc4, 0x74, 0x6c, 0x98, 0x4c, 0x37, 0x50, 0xa7,
			0x28, 0xc3, 0x88, 0xaa, 0x62, 0x8b, 0x0e, 0xbf,
			0x70, 0xbf, 0xa2, 0xb5, 0xb5, 0xbf, 0x70, 0x1d,
			0x4a, 0xd6, 0xed, 0xa4, 0xfd, 0x25, 0x02, 0xa0,
			0x9a, 0xdd, 0x6c, 0xe8, 0x6a, 0xb2, 0x63, 0xe9,
			0xfa, 0x8c, 0x79, 0xf6, 0x1d, 0x7f, 0xc6, 0xe0,
			0x34, 0xfd, 0x42, 0xc7, 0xad, 0x4a, 0x72, 0x2f,
			0xca, 0x31, 0x47, 0x56, 0xa4, 0xf6, 0x43, 0xc6,
			0x6f, 0x8b, 0xdd, 0x26, 0xff, 0x06, 0x27, 0x55,
			0x8b, 0x53, 0xae, 0xba, 0xb2, 0x26, 0xa1, 0xd5,
			0xbb, 0x34, 0x36, 0x1a, 0x21, 0xa3, 0x4a, 0x00,
			0x4f, 0xbf, 0x48, 0x55, 0xa8, 0xee, 0x4b, 0xd4,
			0xca, 0xe6, 0x87, 0xf9, 0x14, 0x1d, 0xe5, 0x14,
			0x66, 0x81, 0xed, 0x9d, 0x27, 0x6a, 0xe9, 0xc7,
		},
		/* S5 : 128 byte */
		{
			0xb2, 0x87, 0x09, 0x91, 0x2d, 0xbe, 0x6e, 0x63,
			0x44, 0x24, 0xda, 0x3c, 0x2c, 0x5c, 0xab, 0x50,
			0x32, 0x53, 0xf1, 0xce, 0xf2, 0x6a, 0x0b, 0x63,
			0xa7, 0x19, 0x0f, 0x63, 0xdb, 0xfd, 0xe9, 0x77,
			0xa7, 0x52, 0x7b, 0xf3, 0x88, 0x3d, 0x11, 0x8d,
			0x94, 0x2c, 0x39, 0xf9, 0xf7, 0x2f, 0x20, 0xa4,
			0xf0, 0xc4, 0x94, 0x8b, 0x48, 0x6e, 0xa8, 0x0f,
			0x09, 0xbe, 0xe7, 0x40, 0xc3, 0xaa, 0x42, 0x65,
			0x82, 0xee, 0xa1, 0x05, 0x28, 0x2e, 0x6e, 0x08,
			0x26, 0xaa, 0xb4, 0x39, 0x07, 0x20, 0xb1, 0x27,
			0xa1, 0x7e, 0x86, 0xe4, 0xc6, 0x2f, 0x15, 0x3d,
			0x79, 0x49, 0x46, 0x2c, 0x87, 0x1c, 0x5d, 0xf5,
			0xa4, 0xc9, 0x1c, 0xe7, 0xf7, 0xbc, 0x7a, 0x1f,
			0x17, 0xef, 0xa1, 0x71, 0x80, 0x65, 0x36, 0xab,
			0x98, 0x30, 0x24, 0xa6, 0xda, 0x69, 0x04, 0xd3,
			0xf9, 0x74, 0x3f, 0x95, 0x2e, 0xcf, 0xe9, 0xbd,
		},
		20, /* salt length */
		/* V5 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA512 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0x10, 0x34, 0xe0, 0x43, 0xd5, 0x21, 0xa2, 0x9e,
			0x1f, 0x6d, 0x84, 0xef, 0x8a, 0x54, 0x9b, 0x86,
			0x75, 0xc5, 0xeb, 0xe4, 0xab, 0x74, 0x2e, 0x72,
			0x43, 0x2b, 0xd1, 0x72, 0xc4, 0x60, 0x40, 0x35,
			0x8d, 0x0d, 0x63, 0x8e, 0x74, 0xbc, 0x68, 0x88,
			0xae, 0xe1, 0xc4, 0x54, 0xef, 0x6d, 0x74, 0x85,
			0xd1, 0xa0, 0x89, 0x80, 0x7a, 0x1f, 0xe3, 0x2d,
			0x79, 0x46, 0x3d, 0xf3, 0xf2, 0xbf, 0x6a, 0xe4,
			0xfb, 0x54, 0xdd, 0x34, 0x45, 0x01, 0x6f, 0xf0,
			0xff, 0x4e, 0x5d, 0xec, 0xf7, 0xdc, 0x90, 0xfe,
			0x03, 0x3d, 0x22, 0x69, 0x9f, 0x08, 0x1d, 0x9c,
			0xd7, 0x2d, 0x86, 0x1d, 0x68, 0x87, 0x9a, 0xc0,
			0x22, 0xb0, 0x57, 0x1c, 0x90, 0x19, 0x3e, 0x3f,
			0xdb, 0x5a, 0x58, 0xa1, 0xec, 0xc6, 0x77, 0x57,
			0xd1, 0x43, 0xbe, 0x89, 0x62, 0x59, 0x63, 0x26,
			0x24, 0x88, 0xb4, 0xd6, 0x6d, 0xdf, 0x58, 0x31,
		},
		/* S6 : 128 byte */
		{
			0x26, 0x25, 0xbc, 0x34, 0xfc, 0xab, 0x3a, 0xf0,
			0x7f, 0x63, 0x34, 0xe0, 0x4a, 0x65, 0x67, 0x74,
			0xe9, 0x84, 0x86, 0x9c, 0xd2, 0x8d, 0x98, 0xab,
			0x5a, 0xbc, 0xf7, 0x5a, 0xdf, 0xed, 0x9d, 0x83,
			0xdf, 0x99, 0x1f, 0x4f, 0x41, 0xae, 0x77, 0xd0,
			0x40, 0xf9, 0xb2, 0xc0, 0x0a, 0x12, 0x49, 0x86,
			0xce, 0xc4, 0x37, 0x16, 0x1e, 0x9a, 0x92, 0xaa,
			0xec, 0x57, 0x1a, 0x13, 0xe4, 0x89, 0xf4, 0xbb,
			0xef, 0x58, 0xec, 0x07, 0xb5, 0xf4, 0x78, 0x37,
			0x32, 0x9f, 0x2d, 0x36, 0x11, 0x1c, 0xbc, 0x7a,
			0x81, 0xf3, 0x20, 0x48, 0x30, 0x1f, 0x12, 0x11,
			0x05, 0x05, 0x5f, 0xa1, 0x25, 0x0f, 0xb2, 0xf2,
			0xb6, 0x94, 0x9e, 0xa6, 0x03, 0x55, 0x57, 0x1f,
			0x83, 0xf4, 0xc1, 0xe7, 0x14, 0x1a, 0x10, 0xf2,
			0x4a, 0xc4, 0x8a, 0x6d, 0x3f, 0xbb, 0x2d, 0xc3,
			0xf6, 0x98, 0xef, 0x0d, 0xba, 0x62, 0x24, 0xf9,
		},
		20, /* salt length */
		/* V6 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA512 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0x12, 0x67, 0xd9, 0xe2, 0x9f, 0x0f, 0x39, 0xfc,
			0xe1, 0xd1, 0x92, 0x4f, 0x0c, 0x2a, 0x43, 0x51,
			0x09, 0x68, 0x2b, 0xf0, 0x60, 0x03, 0xb4, 0x84,
			0x5b, 0x30, 0x10, 0x02, 0x69, 0xb4, 0xc3, 0x72,
			0xc0, 0xf6, 0x62, 0xa3, 0x46, 0x59, 0x9a, 0x1d,
			0xb2, 0xc8, 0x73, 0x57, 0xee, 0x93, 0xd6, 0x2a,
			0xc0, 0x79, 0x4e, 0x86, 0xaa, 0x06, 0x0b, 0x64,
			0x0f, 0x1f, 0x2c, 0xfb, 0x96, 0xdb, 0x3e, 0x37,
			0xd4, 0x94, 0xff, 0x67, 0x25, 0x3e, 0x68, 0x36,
			0xda, 0xf8, 0xeb, 0xdc, 0x9a, 0xe1, 0xfa, 0x95,
			0x02, 0x2d, 0xc8, 0x52, 0x8f, 0x78, 0x53, 0xaf,
			0xbe, 0x59, 0xa2, 0x26, 0x71, 0x3b, 0x4b, 0x82,
			0x95, 0x68, 0x39, 0x96, 0x02, 0x67, 0xac, 0x2e,
			0xa2, 0x88, 0x81, 0x46, 0xd6, 0xbd, 0xb4, 0x7a,
			0x15, 0xab, 0x23, 0xbe, 0xd7, 0x01, 0x95, 0x4f,
			0xb1, 0x09, 0xfa, 0xea, 0xc1, 0x7d, 0xf5, 0x3b,
		},
		/* S7 : 128 byte */
		{
			0x03, 0xf8, 0x97, 0x0b, 0x2b, 0xe5, 0xa7, 0x60,
			0xd0, 0x3f, 0x09, 0x51, 0x85, 0x1d, 0x18, 0xc2,
			0x2c, 0x3b, 0xe1, 0x40, 0xb9, 0x3d, 0x1e, 0xc1,
			0x3d, 0xb2, 0xdc, 0xad, 0x98, 0x5a, 0xcb, 0x89,
			0x97, 0x06, 0xf9, 0xb5, 0x0b, 0x08, 0x53, 0x0f,
			0xa5, 0x82, 0x89, 0xbd, 0xd2, 0xba, 0x5b, 0xd5,
			0xeb, 0xc8, 0x18, 0x6f, 0xa7, 0xad, 0x9f, 0x1f,
			0x90, 0xae, 0xd8, 0x4e, 0x31, 0x36, 0xf6, 0xed,
			0xc3, 0x1d, 0xc8, 0x4d, 0x4e, 0xb8, 0xc8, 0x3a,
			0xe5, 0x2c, 0x63, 0x14, 0x91, 0xbb, 0xfa, 0x6d,
			0x11, 0x6d, 0xc2, 0x4a, 0x6a, 0x58, 0xed, 0x52,
			0xe3, 0x18, 0xa8, 0xe8, 0x5b, 0xf8, 0x7c, 0xf7,
			0xca, 0x6b, 0x34, 0x78, 0xa1, 0x57, 0xc5, 0x81,
			0x50, 0xe7, 0xe8, 0xcf, 0x48, 0xe4, 0xac, 0x67,
			0x31, 0x19, 0x58, 0xd8, 0x8a, 0x30, 0xef, 0xcb,
			0x57, 0x8e, 0x56, 0x1b, 0x05, 0x0d, 0xf1, 0x86,
		},
		20, /* salt length */
		/* V7 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA512 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0xb0, 0x66, 0x48, 0x83, 0x5d, 0x99, 0x3a, 0x4c,
			0x66, 0x94, 0x4a, 0x5c, 0xb9, 0x5a, 0xfd, 0x74,
			0x1c, 0x66, 0x70, 0x8b, 0xc4, 0xf0, 0x92, 0x28,
			0x23, 0x6d, 0x91, 0x10, 0x01, 0x55, 0x92, 0x21,
			0xce, 0x28, 0xa5, 0x7e, 0x37, 0xb3, 0xf8, 0xec,
			0xf7, 0x90, 0x4b, 0x7c, 0xf0, 0xec, 0xa3, 0x6b,
			0x7a, 0x8c, 0x43, 0x46, 0xfb, 0x12, 0xd8, 0x51,
			0xb8, 0x7d, 0xf4, 0x2a, 0xea, 0xf0, 0xbd, 0xaa,
			0x15, 0xb4, 0xe7, 0x0c, 0x95, 0x7f, 0xdf, 0x51,
			0x79, 0xf7, 0x49, 0xe2, 0x95, 0x72, 0xad, 0x78,
			0xd3, 0x00, 0x6d, 0x39, 0xca, 0xab, 0xfb, 0x24,
			0x92, 0xfd, 0x48, 0xae, 0xd6, 0xc3, 0x80, 0xad,
			0x2a, 0x11, 0xf4, 0x76, 0x9f, 0x86, 0x58, 0x42,
			0x95, 0xd1, 0x2e, 0xd7, 0xd1, 0x74, 0x82, 0x64,
			0xd0, 0xc8, 0xd8, 0xac, 0x5a, 0x42, 0x63, 0x66,
			0x5a, 0xf4, 0x77, 0xe4, 0xec, 0x57, 0x41, 0xf4,
		},
		/* S8 : 128 byte */
		{
			0x7a, 0xed, 0x46, 0x11, 0xe0, 0x4d, 0x87, 0x57,
			0x71, 0x52, 0x44, 0xeb, 0xf4, 0x94, 0x2f, 0x9b,
			0xe0, 0x13, 0xdd, 0x00, 0x09, 0xb9, 0xe7, 0x7e,
			0x1f, 0xab, 0x05, 0x67, 0x9c, 0x22, 0x4c, 0xdf,
			0x9a, 0xb3, 0x98, 0x74, 0x96, 0xe7, 0x52, 0xd4,
			0xc3, 0x9e, 0x27, 0xfc, 0x31, 0xce, 0x76, 0xf0,
			0x1d, 0x7b, 0x2c, 0x44, 0x57, 0x7d, 0xa9, 0x82,
			0x72, 0x60, 0x2a, 0x21, 0x15, 0x7d, 0xfa, 0x30,
			0x09, 0x25, 0x3b, 0x86, 0x89, 0x5b, 0x53, 0x7f,
			0x7f, 0xc0, 0x89, 0xed, 0x68, 0x9f, 0x5c, 0xf4,
			0x89, 0xe8, 0x2f, 0x2a, 0xb1, 0x1e, 0x5e, 0x4f,
			0x05, 0xea, 0xf0, 0x72, 0x84, 0xa5, 0xb2, 0xed,
			0x39, 0xb8, 0xc3, 0x59, 0xa2, 0x31, 0x17, 0x21,
			0x38, 0xcc, 0x54, 0x0c, 0xc3, 0xc8, 0x30, 0xfa,
			0xf8, 0x04, 0x1c, 0xcf, 0x55, 0xd7, 0x8c, 0x63,
			0x31, 0x67, 0xa3, 0x86, 0x63, 0x43, 0x45, 0xb8,
		},
		20, /* salt length */
		/* V8 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
	/* SHA512 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0x5d, 0x95, 0x51, 0xdf, 0x21, 0x0c, 0xea, 0x61,
			0xf9, 0xae, 0xc3, 0xd2, 0xef, 0x2f, 0x57, 0x16,
			0x69, 0x2c, 0x5c, 0x55, 0xff, 0xb1, 0x3d, 0xa2,
			0x8a, 0xfe, 0xca, 0xda, 0x7d, 0x8b, 0x2f, 0xcf,
			0x79, 0x49, 0x22, 0xa6, 0x1b, 0x2b, 0x61, 0x7e,
			0x19, 0xe5, 0x5c, 0x53, 0x8d, 0xf4, 0x2e, 0x21,
			0x90, 0xde, 0xd2, 0x4c, 0x0b, 0xbb, 0x94, 0x3f,
			0xca, 0xf4, 0xc5, 0x53, 0xd2, 0x4b, 0xb3, 0xf3,
			0xf5, 0x1b, 0xd9, 0xff, 0xa3, 0xe1, 0x52, 0x6d,
			0x03, 0xfc, 0x94, 0x36, 0x42, 0xb1, 0x5a, 0x70,
			0x2c, 0x72, 0xe2, 0xb7, 0xf9, 0x60, 0x59, 0xd2,
			0x69, 0x0e, 0x94, 0x94, 0x13, 0x88, 0x00, 0xff,
			0xf9, 0x94, 0xdc, 0xb0, 0x61, 0x37, 0xa5, 0x77,
			0xe0, 0x7b, 0xd2, 0x45, 0xb8, 0x70, 0xc2, 0x4b,
			0x64, 0x71, 0xf3, 0x60, 0x70, 0xa8, 0x74, 0xa3,
			0x8e, 0xd5, 0x9d, 0x62, 0xe7, 0x4a, 0xfd, 0x3b,
		},
		/* S9 : 128 byte */
		{
			0x38, 0x88, 0x9a, 0x9f, 0x36, 0x46, 0xf6, 0x75,
			0x67, 0x53, 0x8d, 0xe1, 0xdd, 0xdb, 0xee, 0xed,
			0x36, 0x40, 0x74, 0xba, 0x58, 0x2f, 0xcb, 0x1a,
			0x13, 0x02, 0x9c, 0x00, 0xf1, 0x3c, 0x88, 0xfc,
			0x76, 0x9f, 0xf1, 0xcb, 0x6c, 0x4f, 0xd0, 0x51,
			0x8d, 0x41, 0xd0, 0x88, 0xb5, 0xfd, 0x37, 0xaf,
			0x92, 0xd1, 0x63, 0xe2, 0x24, 0xe4, 0x18, 0x30,
			0x18, 0xed, 0xc9, 0xa7, 0xd1, 0xe4, 0x64, 0x8b,
			0x6d, 0x95, 0xe2, 0x1a, 0xda, 0xf7, 0x7b, 0x39,
			0x01, 0x50, 0xb2, 0x21, 0xf4, 0x56, 0x07, 0x5e,
			0x5d, 0x56, 0x6f, 0x20, 0x49, 0xac, 0xf7, 0xb6,
			0x13, 0x5c, 0x7a, 0xfa, 0x2f, 0x0c, 0xd8, 0x2e,
			0x7d, 0xb1, 0xad, 0x24, 0x2e, 0x5b, 0xe0, 0xe0,
			0xc8, 0xd2, 0x9a, 0x36, 0xc8, 0x29, 0x2e, 0x42,
			0x9a, 0xd6, 0xaa, 0x52, 0xd3, 0x32, 0xb2, 0x9e,
			0xb9, 0x50, 0x7a, 0xe9, 0xc1, 0xf3, 0x8d, 0xd1,
		},
		20, /* salt length */
		/* V9 : 20 byte */
		{
			0x6f, 0x28, 0x41, 0x16, 0x6a, 0x64, 0x47, 0x1d,
			0x4f, 0x0b, 0x8e, 0xd0, 0xdb, 0xb7, 0xdb, 0x32,
			0x16, 0x1d, 0xa1, 0x3b, 
		},
	},
};

typedef mod1024_t tv_t;


#include "siggenPSStest.c"
