/* siggen15test.c */
/*
 * Copyright (c) 2017 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

static inline void dump(const char *label, const uint8_t *buf,
			const size_t bytes)
{
	int i;

	printf("%s:\n", label);
	for (i = 0; i < bytes; i++) {
		if (i % 8 == 0) {
			printf("    ");
		}
		printf("%02x ", buf[i]);
		if ((i+1) % 8 == 0) {
			printf("\n");
		}
	}
	printf("\n");
}

static int check_sign(Prvkey_RSA *prv, tv_t *vector, int sig_algo)
{
	unsigned char *sig = NULL;
	int slen;
	int rc;
	int i, j;
	char sb[80];

	switch_str(sig_algo, sb);

	for (i = 0; i < 10; i++) {
		if (OK_do_signature((Key *) prv,
				    vector[i].msg, sizeof(vector[i].msg),
				    &sig, &slen, sig_algo)) {
			printf("test OK_do_signature %s %d -- error!\n", sb, i);
			rc = -1;
			goto error;
		}
		printf("slen = %d\n", slen);
		for (j = 0; j < slen; j++) {
			printf("%02x ", sig[j]);
		}
		printf("\n");
		if (memcmp(sig, vector[i].s, slen) != 0) {
			printf("test memcmp %s %d -- error!\n", sb, i);
			rc = -1;
			goto error;
		} else {
			printf("test memcmp %s %d -- ok!\n", sb, i);
		}
	}
	rc = 0;

error:
	if (sig != NULL) {
		free(sig);
	}

	return rc;
}

static int check_verify(Pubkey_RSA *pub, tv_t *vector, int sig_algo)
{
	int rc;
	int i;
	char sb[80];
	unsigned char digest[AC_MAX_DIGESTSIZE];
	int dlen;

	switch_str(sig_algo, sb);

	for (i = 0; i < 10; i++) {
		if (OK_do_digest(sig_algo, vector[i].msg, sizeof(vector[i].msg),
				 digest, &dlen) == NULL){
			printf("test OK_do_digest %s %d -- error!\n", sb, i);
			rc = -1;
			goto error;
		}
		if (OK_do_verify((Key *) pub, digest, vector[i].s, sig_algo)) {
			printf("test OK_do_verify %s %d -- error!\n", sb, i);
			rc = -1;
			goto error;
		} else {
			printf("test OK_do_verify %s %d -- ok!\n", sb, i);
		}
	}
	rc = 0;

error:
	return rc;
}

int test_rsa_sig()
{
	LNm *e,*d,*n;
	Pubkey_RSA	   *key;
	Prvkey_RSA	   *prv;
	int rc = -1;

	n = LN_alloc_s(sizeof(sn1)/sizeof(unsigned short), (unsigned short *) sn1);
	d = LN_alloc_s(sizeof(sd1)/sizeof(unsigned short), (unsigned short *) sd1);
	e = LN_alloc_s(sizeof(se1)/sizeof(unsigned short), (unsigned short *) se1);

	key = (Pubkey_RSA *) malloc(sizeof(Pubkey_RSA));
	prv = (Prvkey_RSA *) malloc(sizeof(Prvkey_RSA));
	key->size = prv->size = sizeof(sn1);
	key->key_type = KEY_RSA_PUB;
	prv->key_type = KEY_RSA_PRV;

	printf("n:");
	LN_print(n);
	printf("e:");
	LN_print(e);
	printf("d:");
	LN_print(d);

	OK_RSA_set_pubkey(key, n, e);
	OK_RSA_set_prvkey(prv, n, d);

#ifdef HAS_SHA1_TESTVECTOR
	if (check_sign(prv, SHA1, OBJ_SIG_SHA1RSA_PKCS1) != 0) {
		goto error;
	}
	if (check_verify(key, SHA1, OBJ_SIG_SHA1RSA_PKCS1) != 0) {
		goto error;
	}
#endif /* HAS_SHA1_TESTVECTOR */

#ifdef HAS_SHA224_TESTVECTOR
	if (check_sign(prv, SHA224, OBJ_SIG_SHA224RSA_PKCS1) != 0) {
		goto error;
	}
	if (check_verify(key, SHA224, OBJ_SIG_SHA224RSA_PKCS1) != 0) {
		goto error;
	}
#endif /* HAS_SHA224_TESTVECTOR */

#ifdef HAS_SHA256_TESTVECTOR
	if (check_sign(prv, SHA256, OBJ_SIG_SHA256RSA_PKCS1) != 0) {
		goto error;
	}
	if (check_verify(key, SHA256, OBJ_SIG_SHA256RSA_PKCS1) != 0) {
		goto error;
	}
#endif /* HAS_SHA256_TESTVECTOR */

#ifdef HAS_SHA384_TESTVECTOR
	if (check_sign(prv, SHA384, OBJ_SIG_SHA384RSA_PKCS1) != 0) {
		goto error;
	}
	if (check_verify(key, SHA384, OBJ_SIG_SHA384RSA_PKCS1) != 0) {
		goto error;
	}
#endif /* HAS_SHA384_TESTVECTOR */

#ifdef HAS_SHA512_TESTVECTOR
	if (check_sign(prv, SHA512, OBJ_SIG_SHA512RSA_PKCS1) != 0) {
		goto error;
	}
	if (check_verify(key, SHA512, OBJ_SIG_SHA512RSA_PKCS1) != 0) {
		goto error;
	}
#endif /* HAS_SHA512_TESTVECTOR */

#ifdef HAS_SHA512224_TESTVECTOR
	if (check_sign(prv, SHA512224, OBJ_SIG_SHA512224RSA_PKCS1) != 0) {
		goto error;
	}
	if (check_verify(key, SHA512224, OBJ_SIG_SHA512224RSA_PKCS1) != 0) {
		goto error;
	}
#endif /* HAS_SHA512224_TESTVECTOR */

#ifdef HAS_SHA512256_TESTVECTOR
	if (check_sign(prv, SHA512256, OBJ_SIG_SHA512256RSA_PKCS1) != 0) {
		goto error;
	}
	if (check_verify(key, SHA512256, OBJ_SIG_SHA512256RSA_PKCS1) != 0) {
		goto error;
	}
#endif /* HAS_SHA512256_TESTVECTOR */

	rc = 0;
error:
	free(key);
	free(prv);
	LN_free(d);
	LN_free(e);
	LN_free(n);

	return rc;
}

int main(int argc,char **argv)
{

	if(test_rsa_sig())
		exit(EXIT_FAILURE);

	return EXIT_SUCCESS;
}
