/* siggen15_3_TruncatedSHAs_3072_testfc.c */
/*
 * Copyright (c) 2017 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

/*
 * This test vector was created based on SigGen15_186-3.txt from
 * https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Algorithm-Validation-Program/documents/dss/186-3rsatestvectors.zip
 *
 * The link to the archive file exists in the following URL:
 * https://csrc.nist.gov/projects/cryptographic-algorithm-validation-program/digital-signatures/
 */
/*
 * https://csrc.nist.gov/projects/cryptographic-algorithm-validation-program/digital-signatures/
 * https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Algorithm-Validation-Program/documents/dss/186-3rsatestvectors.zip
 * SigGen15_186-3_TruncatedSHAs.txt
 */ 
#include "siggen15test.h"

static const unsigned short sn1[] = {
	0xa5e6, 0x8e37, 0x9f3c, 0x4c3c, 0x908a, 0x04ce, 0x63c4, 0x6042,
	0x3a7a, 0x10a1, 0x6df4, 0xda61, 0x66fc, 0xd8b1, 0x4100, 0x0b14,
	0xc113, 0xb216, 0xfe4f, 0x5f17, 0x4a59, 0x35c9, 0x37e0, 0xc6d8,
	0x3286, 0xba13, 0xb137, 0xc4d0, 0xbe7d, 0x7b41, 0x5ced, 0x1d6d,
	0xc095, 0x1098, 0xa477, 0xdac4, 0x1df6, 0x382b, 0x9a50, 0x108f,
	0x5745, 0x5d85, 0x3d4d, 0x8663, 0xfefd, 0xd501, 0x7a53, 0x3cb7,
	0x1a87, 0x07ae, 0x4803, 0xdc2c, 0xb93e, 0x5453, 0x5736, 0x5248,
	0x5eaf, 0x9e17, 0x5e09, 0x5c7f, 0xe72d, 0xf7db, 0xae96, 0x902c,
	0x15ed, 0x0ef9, 0x6883, 0x5e72, 0xcfbd, 0x5290, 0xd2e3, 0xea35,
	0xace7, 0xa2d3, 0x0e58, 0x687d, 0x388a, 0x3a65, 0x40cb, 0x4706,
	0xa6e5, 0x96a8, 0xa7cb, 0x9e18, 0x741c, 0x6eab, 0x632a, 0x2300,
	0x453f, 0xb47d, 0x3b54, 0x3381, 0x519b, 0x7c72, 0x7913, 0xdbd6,
	0xe037, 0xaf5b, 0xc8d0, 0x13c3, 0xfd16, 0xc04a, 0xd2ca, 0x4640,
	0x4189, 0x8c54, 0xb006, 0x7db7, 0x57b8, 0x8dda, 0x0a29, 0x86b4,
	0x520c, 0x514c, 0xaa07, 0x39b3, 0x0d1d, 0xa85c, 0x0c84, 0x1885,
	0x07ed, 0x3979, 0xbafb, 0x9f8c, 0xca0b, 0x2d6d, 0xde20, 0xc6ac,
	0x06ed, 0xa11e, 0x6a69, 0xbd9e, 0x9bfc, 0x3c35, 0xa1d4, 0xb52b,
	0x8556, 0x8105, 0xed69, 0x4c33, 0xddc8, 0x10b1, 0xd4a5, 0x0653,
	0x8cf9, 0xf4b3, 0xde2f, 0xca2b, 0xf899, 0xc9b8, 0xb891, 0x1466,
	0xfc08, 0x30cc, 0x1f3a, 0x9c70, 0x356f, 0xb9b2, 0xcb9a, 0x6202,
	0x1674, 0xe80e, 0x024b, 0x89bc, 0xfff0, 0x0e63, 0x3c54, 0x8852,
	0xd4ee, 0xa21a, 0x5e5f, 0xfe7f, 0x7ff8, 0x6ace, 0x0792, 0x6f4c,
	0xd9eb, 0xdcfe, 0xb90f, 0xb2cb, 0xaa91, 0x52f7, 0x9223, 0xaac5,
	0xfbcc, 0xe9e6, 0x6b6c, 0xa9d7, 0x21f0, 0x714e, 0x6def, 0x0699,
};

static const unsigned short se1[] = {
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x004c, 0x9c53,
};

static const unsigned short sd1[] = {
	0x0690, 0x3d86, 0xfc4b, 0x1351, 0x727e, 0xc45d, 0x3bae, 0x17c7,
	0x0b18, 0x0f88, 0x24a0, 0x87e8, 0xafd4, 0x5285, 0x0ee2, 0x803a,
	0xfaaf, 0x599b, 0x71c0, 0x7982, 0xc23c, 0xd8b0, 0x9713, 0xcd1e,
	0x9352, 0x6ba8, 0x0389, 0x8a26, 0x74fd, 0xb2d8, 0xc271, 0x6b8d,
	0xa670, 0x116c, 0xa4aa, 0x4fc5, 0xf6c0, 0x2079, 0x6ad5, 0xc27a,
	0x6d1a, 0x27af, 0xe8c2, 0x8067, 0x9462, 0x3116, 0x494e, 0x6374,
	0xe773, 0xc9b2, 0x686f, 0x2453, 0xda69, 0x747a, 0xd2e0, 0xb68a,
	0x490e, 0x7e51, 0x3289, 0xace6, 0xf6d1, 0xa0dd, 0x990b, 0x222c,
	0xfadb, 0x6a80, 0x0f9d, 0xce7b, 0xd399, 0x5cda, 0x61e1, 0xf590,
	0xe412, 0x5555, 0x15af, 0x9b91, 0x9bb3, 0x823d, 0xa1b5, 0xd6be,
	0xb47f, 0xff2a, 0x28ac, 0xf8d7, 0xf40d, 0xbae7, 0x2630, 0x3868,
	0x0a4b, 0x642d, 0x8a28, 0x0485, 0x09e2, 0xdb9d, 0x41c6, 0x5c23,
	0xeb60, 0x5a4d, 0x1cc6, 0x4f14, 0x0ff5, 0xec7d, 0x46af, 0xaef5,
	0x532a, 0xe140, 0xf9a5, 0x0f9d, 0x2763, 0xf2d2, 0x5080, 0xb9af,
	0xc7d3, 0x5bec, 0x13a1, 0x1d34, 0xc4ab, 0x95ce, 0xebfe, 0xdf0c,
	0xeac7, 0x6e7a, 0xb1d6, 0x6b9d, 0xab61, 0x8aa0, 0x0667, 0x715e,
	0x6054, 0xa06a, 0xdd76, 0xa73a, 0xc5b9, 0xa0fb, 0xb7f1, 0x4015,
	0x757b, 0x9d07, 0xc701, 0x14f4, 0xd375, 0x9fac, 0x34c2, 0x2ea8,
	0xea1c, 0xc9af, 0x9761, 0xb688, 0xfdfd, 0xb0f4, 0x8161, 0x5675,
	0xc5b7, 0x0759, 0xa202, 0xed80, 0xeac1, 0xf602, 0x5965, 0x3537,
	0xd132, 0xbf2e, 0xdf77, 0x70fb, 0x376f, 0x5130, 0x5659, 0x4a55,
	0x8ea9, 0xff66, 0xdf85, 0x9ca4, 0xcfad, 0x1447, 0xcde2, 0x299f,
	0xd265, 0x8f8d, 0xb01b, 0xdd15, 0xabfc, 0xef39, 0x1a1c, 0xe4e0,
	0xd612, 0x7dd9, 0x4640, 0x48b2, 0xf92a, 0x185d, 0xa88a, 0xbc6d,
};

#define HAS_SHA512224_TESTVECTOR

#define HAS_SHA512256_TESTVECTOR

/* 10 elements */
static mod3072_t SHA512224[] = {
	/* SHA512224 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0x14, 0x80, 0x47, 0x3f, 0x19, 0x36, 0x0f, 0x66,
			0x6e, 0x20, 0xdb, 0xa9, 0xf0, 0xf2, 0x9b, 0x07,
			0x39, 0x32, 0xce, 0xd8, 0xcf, 0x9b, 0x50, 0x52,
			0x9a, 0xc4, 0x73, 0x52, 0x9c, 0xfd, 0x52, 0x5a,
			0xdc, 0x79, 0x62, 0xd5, 0xa3, 0xb3, 0x4a, 0xa3,
			0xeb, 0x0a, 0xf5, 0xd1, 0x15, 0xae, 0xe5, 0xa8,
			0xda, 0xc0, 0xca, 0xea, 0x84, 0xb5, 0x53, 0x58,
			0x5e, 0xfc, 0x44, 0x77, 0x69, 0xe0, 0xc9, 0x94,
			0x50, 0xaf, 0x09, 0x06, 0x9e, 0x1c, 0xff, 0xb3,
			0x2f, 0xe2, 0x85, 0xd6, 0xe4, 0xaa, 0xb2, 0xff,
			0x9b, 0x2f, 0x66, 0x5b, 0x5f, 0xa0, 0x23, 0x35,
			0xdb, 0x46, 0x1f, 0xbd, 0x15, 0x66, 0x97, 0x08,
			0x59, 0x69, 0x65, 0x59, 0x72, 0x3a, 0xd5, 0x9f,
			0x21, 0xf0, 0xb5, 0x3e, 0xc7, 0xad, 0xab, 0x12,
			0x65, 0xc5, 0xcc, 0xb1, 0xc6, 0x23, 0x69, 0xf1,
			0xfb, 0xe9, 0x2c, 0xab, 0x08, 0x57, 0x64, 0xbe,
		},
		/* S0 : 384 byte */
		{
			0x1f, 0x95, 0x4d, 0xa5, 0x3b, 0x48, 0xfd, 0xcf,
			0x3d, 0xc3, 0xe8, 0x2c, 0x62, 0x38, 0x81, 0x75,
			0x6a, 0x17, 0xbd, 0x93, 0x8f, 0xe0, 0xc3, 0x61,
			0x0d, 0xef, 0xb7, 0xb6, 0x48, 0xb7, 0xca, 0x0c,
			0x27, 0x4d, 0xf3, 0x69, 0xa5, 0x54, 0x8e, 0xdc,
			0xde, 0x37, 0xac, 0x77, 0xe8, 0xff, 0x07, 0x03,
			0xe2, 0x38, 0xa1, 0x18, 0x81, 0xe7, 0xa9, 0xda,
			0x5b, 0x62, 0xbd, 0xc6, 0x14, 0x7c, 0x29, 0x52,
			0x61, 0x26, 0xde, 0x03, 0xfe, 0x75, 0xd5, 0xe8,
			0x60, 0x2c, 0xf3, 0x29, 0x36, 0xa2, 0x05, 0xc2,
			0x69, 0x28, 0xd6, 0xb5, 0x90, 0xcb, 0x10, 0xd4,
			0xd0, 0x98, 0x2f, 0x2b, 0x72, 0x01, 0xbb, 0xdb,
			0xee, 0xce, 0xcb, 0x9e, 0x8b, 0xf5, 0x5a, 0xd6,
			0x70, 0x12, 0xb7, 0x22, 0x24, 0x3c, 0x4a, 0xa7,
			0x21, 0xce, 0x83, 0x70, 0x96, 0x83, 0x16, 0xfe,
			0x93, 0x26, 0xfc, 0xe6, 0xc8, 0x9c, 0x59, 0x51,
			0xeb, 0xfd, 0xd7, 0xc3, 0xce, 0x84, 0x75, 0x32,
			0xa4, 0x38, 0x8a, 0x98, 0x04, 0x95, 0xaf, 0x4a,
			0x67, 0x33, 0xd8, 0x23, 0xf9, 0xdf, 0xcd, 0x0c,
			0x19, 0xc1, 0x53, 0x32, 0x94, 0x62, 0x82, 0xf9,
			0x98, 0xd3, 0x8e, 0x19, 0x92, 0xba, 0x4b, 0x72,
			0x1c, 0x45, 0xfc, 0xa9, 0xe5, 0x12, 0x89, 0xa9,
			0x69, 0x34, 0x21, 0x20, 0x6a, 0x96, 0x8a, 0x49,
			0xb1, 0x49, 0x11, 0xd8, 0x9a, 0x2f, 0xb0, 0x30,
			0xe6, 0xcb, 0x4b, 0x94, 0x9b, 0x0a, 0x3d, 0x6b,
			0x24, 0x86, 0x80, 0x7c, 0x8e, 0x2f, 0x6b, 0xe8,
			0x1a, 0xc4, 0x36, 0x9c, 0xce, 0x00, 0xd6, 0x00,
			0x00, 0x98, 0x80, 0x50, 0x3d, 0xd4, 0xf5, 0x63,
			0x5c, 0xde, 0xb0, 0x56, 0xb2, 0x0e, 0xd3, 0x20,
			0xe9, 0xe9, 0xeb, 0x44, 0x32, 0xe9, 0xc1, 0x3c,
			0xc2, 0xd0, 0x91, 0xf2, 0xbe, 0x45, 0x9b, 0x39,
			0x00, 0xcf, 0xf1, 0x2b, 0x2c, 0x27, 0x0d, 0x56,
			0x77, 0x5c, 0xa6, 0x1b, 0x2d, 0x80, 0xb8, 0x35,
			0x28, 0xc0, 0x2d, 0x9c, 0xe3, 0x5e, 0x42, 0xbc,
			0x36, 0xb1, 0xa5, 0x6b, 0x3d, 0xd6, 0xce, 0x4e,
			0x33, 0x4b, 0x33, 0x10, 0x34, 0xe6, 0xfc, 0x0b,
			0xa0, 0x78, 0x31, 0x90, 0x5e, 0x83, 0xd0, 0x23,
			0x1e, 0x34, 0x68, 0xdb, 0x4c, 0x04, 0x91, 0x8b,
			0x5a, 0x90, 0xd5, 0x08, 0xf5, 0x97, 0x57, 0x3c,
			0xaa, 0x1f, 0x26, 0x5c, 0x5e, 0x9a, 0x25, 0xe2,
			0x18, 0x01, 0xaa, 0x6c, 0x39, 0xad, 0x95, 0xb5,
			0x6d, 0xd0, 0x24, 0xb3, 0x81, 0x8f, 0x82, 0xa7,
			0x28, 0x92, 0xf9, 0x82, 0x7f, 0x65, 0xe9, 0xf5,
			0xf1, 0x34, 0x6e, 0xb7, 0x1b, 0xac, 0xa0, 0xc2,
			0xf4, 0xf2, 0xf1, 0xae, 0x5f, 0x7f, 0x24, 0x92,
			0x86, 0xa4, 0xf3, 0x01, 0xe5, 0xe3, 0x2c, 0x57,
			0x46, 0x22, 0x0d, 0x41, 0x33, 0x50, 0x78, 0xcf,
			0xbc, 0x51, 0x1c, 0x19, 0xb2, 0xad, 0x5e, 0xc0,
		},
	},
	/* SHA512224 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0x8a, 0x09, 0xfb, 0x05, 0xb7, 0xe6, 0x77, 0x96,
			0x36, 0x12, 0x3a, 0xc0, 0xa3, 0x47, 0x60, 0x3a,
			0xa7, 0x1f, 0x6d, 0x8f, 0x77, 0x87, 0xed, 0x2a,
			0x9d, 0x71, 0x57, 0xaf, 0xfb, 0xca, 0x37, 0x92,
			0x0d, 0xac, 0x8c, 0xe2, 0x6e, 0xd4, 0x6f, 0x65,
			0x9e, 0x56, 0xf6, 0xea, 0x02, 0x8f, 0x58, 0xdb,
			0x2c, 0x5c, 0x52, 0x12, 0x5d, 0x89, 0xc9, 0xed,
			0x48, 0x8d, 0xb0, 0xa5, 0x19, 0x74, 0x2c, 0xdb,
			0xb6, 0x1c, 0xf6, 0x74, 0x45, 0xa4, 0x02, 0xa7,
			0xee, 0x22, 0x5e, 0x54, 0xe2, 0x94, 0x5c, 0x98,
			0x9c, 0x1a, 0xf3, 0x9e, 0x67, 0x62, 0x63, 0xb8,
			0x20, 0x63, 0x18, 0xae, 0x3d, 0x4e, 0xd1, 0x56,
			0x0e, 0xc1, 0x01, 0x64, 0x43, 0x3e, 0x75, 0x84,
			0xaf, 0xdd, 0x38, 0xa6, 0x4b, 0x3d, 0xb4, 0x99,
			0x7e, 0xc0, 0x19, 0xdc, 0x9b, 0xf8, 0x03, 0xb1,
			0xac, 0x5c, 0x57, 0x72, 0x6e, 0x40, 0x6d, 0x28,
		},
		/* S1 : 384 byte */
		{
			0xa3, 0xba, 0xa7, 0xce, 0x19, 0x3c, 0xa9, 0x8c,
			0x05, 0xae, 0xb3, 0xde, 0xfd, 0xe2, 0x82, 0x0b,
			0xea, 0x07, 0x2d, 0x4d, 0xc0, 0x27, 0x37, 0xfe,
			0x8e, 0x34, 0x92, 0xae, 0x4b, 0x32, 0x9a, 0x7b,
			0x16, 0x31, 0xc9, 0xab, 0x4f, 0xfb, 0x61, 0x80,
			0xd3, 0x3a, 0xa6, 0xca, 0x27, 0xb6, 0x39, 0xd0,
			0x4f, 0xca, 0x9d, 0xd4, 0x6e, 0xa8, 0x2f, 0x0c,
			0xe0, 0x60, 0xae, 0x56, 0xac, 0x15, 0xb5, 0x2e,
			0x51, 0x16, 0x43, 0x21, 0x84, 0xab, 0x00, 0x9e,
			0x57, 0xdf, 0x08, 0x0c, 0x2a, 0xe1, 0x7e, 0xc7,
			0xa4, 0xf3, 0x36, 0x13, 0xb8, 0x46, 0xb1, 0x60,
			0x97, 0x66, 0xb0, 0x84, 0xde, 0x34, 0xc0, 0x08,
			0x0d, 0x36, 0x95, 0xea, 0xc5, 0xdd, 0x3e, 0xee,
			0x83, 0xee, 0x57, 0xbf, 0x70, 0xb3, 0x03, 0xd5,
			0xea, 0x7b, 0x1a, 0x96, 0xde, 0x0b, 0x32, 0x57,
			0x80, 0x08, 0xd6, 0x14, 0x90, 0x69, 0x76, 0x71,
			0x25, 0xfa, 0x1a, 0xfd, 0x05, 0xa5, 0x4b, 0x73,
			0x4c, 0x70, 0xdd, 0x8c, 0x58, 0xb0, 0x7d, 0x46,
			0xe7, 0xe0, 0xd0, 0x65, 0x71, 0x7a, 0x2b, 0x44,
			0x5d, 0xce, 0x21, 0xb1, 0x15, 0x52, 0x2d, 0x5c,
			0x05, 0x98, 0x50, 0x4e, 0x1f, 0xa6, 0x9a, 0x63,
			0x32, 0x28, 0xb7, 0x13, 0xe4, 0xad, 0x28, 0x0d,
			0x4d, 0x85, 0x5f, 0xd1, 0xca, 0xef, 0xbd, 0xc0,
			0xff, 0x87, 0x7a, 0x00, 0x35, 0x84, 0xac, 0xc5,
			0x7f, 0xa6, 0x2e, 0x79, 0x1d, 0x16, 0x5f, 0x5d,
			0x63, 0xd3, 0x3d, 0x80, 0x87, 0xc3, 0x02, 0x31,
			0x1d, 0x78, 0x89, 0xc5, 0x94, 0x43, 0x40, 0x67,
			0x08, 0x56, 0x6b, 0xd2, 0x44, 0x6b, 0xeb, 0xbd,
			0xf9, 0x75, 0xde, 0xc7, 0xd4, 0x2a, 0xf6, 0x61,
			0xf7, 0x5e, 0xa9, 0xb3, 0x7a, 0x0e, 0x85, 0xde,
			0x3e, 0x7d, 0x59, 0xfa, 0x53, 0xa6, 0x2f, 0xfd,
			0xca, 0xe0, 0x4b, 0x1e, 0xd6, 0xd9, 0xcb, 0x69,
			0xac, 0x58, 0x6b, 0x94, 0x8d, 0xcb, 0x64, 0x5b,
			0xc6, 0xcd, 0xe6, 0x78, 0x9d, 0x46, 0x5a, 0x1b,
			0x72, 0x3d, 0x85, 0x04, 0xcb, 0xbf, 0xf1, 0xc2,
			0x7a, 0xa7, 0xc9, 0xfc, 0x0e, 0x8e, 0x8c, 0x69,
			0x7e, 0x25, 0x3a, 0xd1, 0xd8, 0x1c, 0xa6, 0xf5,
			0x6e, 0x6c, 0x73, 0x05, 0xb7, 0xb5, 0x1e, 0xa9,
			0x30, 0x81, 0xa7, 0x5b, 0x5d, 0x85, 0x88, 0x35,
			0x7f, 0x6a, 0xfe, 0x89, 0x19, 0x12, 0x2f, 0x79,
			0xe6, 0x3e, 0xf7, 0xec, 0xee, 0x57, 0x2b, 0xc1,
			0xd0, 0x16, 0xe6, 0x8b, 0xa1, 0xad, 0x93, 0xc5,
			0xd4, 0xaa, 0x26, 0xfa, 0xf6, 0x38, 0xab, 0xe0,
			0x9a, 0x34, 0xb1, 0xaf, 0xcb, 0xaf, 0xaf, 0x4c,
			0x32, 0xd1, 0x59, 0x98, 0x11, 0x93, 0x50, 0xa3,
			0x9d, 0xf6, 0xb4, 0x8a, 0xff, 0xbd, 0xa5, 0x9e,
			0xa6, 0x42, 0xbc, 0x4a, 0xeb, 0x86, 0x9a, 0x8d,
			0xbc, 0xbd, 0x2a, 0x5a, 0x31, 0xe6, 0x8f, 0x8b,
		},
	},
	/* SHA512224 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0xd3, 0x47, 0x27, 0x40, 0x77, 0x3e, 0xd7, 0xb3,
			0xd6, 0x37, 0xed, 0xab, 0x7e, 0xd3, 0x97, 0x71,
			0x1e, 0x59, 0x11, 0x21, 0xf7, 0x58, 0xbc, 0x68,
			0x8d, 0x8e, 0x6f, 0x8d, 0xa2, 0x09, 0x11, 0xae,
			0x64, 0x8c, 0x68, 0x4d, 0xe6, 0x34, 0x2c, 0x0b,
			0xcd, 0x5b, 0x60, 0xb2, 0x4c, 0xbd, 0xe2, 0x79,
			0x52, 0xd9, 0x40, 0x84, 0x01, 0x9b, 0xcd, 0xab,
			0x63, 0x37, 0xf4, 0x7c, 0x58, 0x58, 0xcd, 0x83,
			0xd3, 0x7a, 0xee, 0xc9, 0x46, 0x94, 0x84, 0x9a,
			0x5b, 0xfb, 0xa2, 0x8b, 0xe5, 0x83, 0xd8, 0xbc,
			0x16, 0xe7, 0x08, 0x5b, 0xf2, 0x6b, 0x75, 0x69,
			0x03, 0xbf, 0x1d, 0xc1, 0x4f, 0x27, 0x49, 0x56,
			0x87, 0xd9, 0x62, 0xbc, 0x81, 0xa6, 0x06, 0xea,
			0xe8, 0xdd, 0x5b, 0x3e, 0x33, 0x6d, 0xa3, 0xc1,
			0xd3, 0x49, 0x11, 0xad, 0xb1, 0x11, 0xdb, 0x2c,
			0xd7, 0xee, 0x91, 0x63, 0x6d, 0xbe, 0xad, 0xaf,
		},
		/* S2 : 384 byte */
		{
			0x92, 0x89, 0xad, 0xe3, 0x52, 0x70, 0xd2, 0xb5,
			0x08, 0x10, 0x36, 0xfc, 0xe1, 0x86, 0xae, 0x1d,
			0xfb, 0xfc, 0xcd, 0x4b, 0xbd, 0xdf, 0xe1, 0xea,
			0x5e, 0x68, 0x5f, 0xbc, 0x47, 0xd6, 0x70, 0x5f,
			0x02, 0x12, 0x2b, 0xe8, 0x63, 0x33, 0x70, 0x73,
			0x5e, 0xc3, 0x08, 0x67, 0x6c, 0xcb, 0x8b, 0x96,
			0x93, 0x53, 0xfa, 0xf5, 0xfc, 0x01, 0xb8, 0xa3,
			0x0b, 0x35, 0xfd, 0x6d, 0xb0, 0x88, 0xd2, 0x16,
			0x38, 0x0b, 0xc0, 0x07, 0x6f, 0x54, 0xe7, 0x7e,
			0xed, 0x25, 0x8b, 0x38, 0x2b, 0xc0, 0x51, 0xd8,
			0xea, 0x37, 0xb9, 0x86, 0x07, 0x4a, 0xd2, 0xbd,
			0xfc, 0x51, 0xfa, 0xf4, 0xfa, 0xcf, 0xb3, 0x3f,
			0x9d, 0x5b, 0x00, 0x78, 0x36, 0xb9, 0xed, 0xbc,
			0x85, 0x7e, 0x4f, 0xfd, 0xd1, 0x3f, 0xd5, 0x4a,
			0x16, 0x7d, 0x54, 0x49, 0x2c, 0xc4, 0xa5, 0xb8,
			0x08, 0xa5, 0xcf, 0x23, 0xd3, 0x16, 0x39, 0xae,
			0x19, 0x53, 0x90, 0xeb, 0xfc, 0x8b, 0x72, 0x20,
			0x14, 0xf1, 0xc1, 0xa1, 0x80, 0x6c, 0xe6, 0xd7,
			0xad, 0x94, 0x32, 0xf7, 0x06, 0xa5, 0x5c, 0xf6,
			0x18, 0x47, 0x37, 0xe9, 0x9b, 0xeb, 0x58, 0x80,
			0xe7, 0x00, 0x78, 0xb8, 0x21, 0xff, 0x41, 0x4c,
			0x0c, 0x65, 0x50, 0x6d, 0xa1, 0x70, 0xb2, 0x61,
			0x34, 0x18, 0xdc, 0xf2, 0xd1, 0xdf, 0xfc, 0x51,
			0xfb, 0x9e, 0x4a, 0x10, 0x93, 0x82, 0xc4, 0xc5,
			0x66, 0x12, 0x01, 0xcd, 0x3a, 0x69, 0x6b, 0x64,
			0x18, 0x6f, 0x18, 0x4b, 0x5a, 0xc1, 0xac, 0x3f,
			0xa5, 0x0f, 0xcf, 0xef, 0x1f, 0x87, 0x75, 0x7d,
			0x01, 0x68, 0x5d, 0x39, 0x29, 0xe9, 0x97, 0xa3,
			0xa2, 0x3e, 0xf3, 0x5f, 0x2b, 0xfa, 0xb0, 0x9c,
			0x60, 0x5c, 0xdc, 0x82, 0xdc, 0x32, 0x2c, 0x29,
			0x07, 0x41, 0xf7, 0x6b, 0x45, 0x61, 0xf1, 0xdd,
			0x38, 0x8a, 0x1b, 0x91, 0x60, 0x30, 0xf0, 0x0a,
			0x4c, 0xed, 0x80, 0x5c, 0x68, 0xb8, 0x33, 0xfb,
			0xa9, 0x56, 0x41, 0x3f, 0x77, 0xaa, 0xf4, 0xba,
			0x08, 0xcf, 0x2c, 0xd0, 0x4f, 0x2d, 0x91, 0xf5,
			0xcb, 0xdd, 0x4b, 0x65, 0x55, 0xb8, 0xbd, 0x49,
			0x60, 0xb4, 0x3d, 0xbd, 0x24, 0x44, 0x64, 0xd6,
			0xef, 0xb5, 0xe3, 0x5e, 0xcd, 0x19, 0x10, 0x2b,
			0x4b, 0xc2, 0x7b, 0x43, 0xe4, 0x0e, 0x62, 0x45,
			0x05, 0x76, 0x89, 0x00, 0xc1, 0x65, 0xa2, 0x3f,
			0x40, 0x74, 0xd5, 0x42, 0x06, 0xc2, 0xe1, 0x16,
			0xf1, 0x57, 0x65, 0x29, 0xb2, 0x9b, 0xd8, 0x38,
			0x5c, 0x1c, 0xd9, 0x1b, 0x83, 0xae, 0xb5, 0x20,
			0xcc, 0x4b, 0xa9, 0xd0, 0x8e, 0x13, 0x3f, 0x22,
			0x4b, 0xbb, 0x0d, 0xff, 0xa9, 0x5d, 0x9f, 0xf5,
			0xb6, 0x05, 0x04, 0xba, 0x9b, 0x3c, 0x00, 0x8a,
			0x90, 0xb0, 0xe4, 0x95, 0x21, 0x57, 0x0d, 0x58,
			0x6c, 0x4a, 0xe9, 0x6f, 0x89, 0x89, 0x20, 0x95,
		},
	},
	/* SHA512224 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0xc4, 0xdb, 0x70, 0x98, 0x1a, 0x92, 0x5e, 0xcc,
			0xc1, 0x16, 0xa6, 0xbb, 0xd9, 0x59, 0x7e, 0x97,
			0x63, 0xa8, 0x0e, 0xb1, 0xf6, 0x8c, 0xc9, 0x84,
			0xb3, 0xf3, 0xfc, 0xf5, 0x7b, 0x8b, 0xa7, 0xd9,
			0x5f, 0x6f, 0x8d, 0xb8, 0x57, 0x05, 0x87, 0xd0,
			0x37, 0x26, 0x1d, 0x5d, 0xe0, 0x80, 0x57, 0x18,
			0xb3, 0x39, 0xd1, 0xfe, 0x4c, 0xd5, 0x0e, 0x52,
			0x6c, 0x5a, 0x0e, 0xe5, 0x36, 0xb2, 0x23, 0x95,
			0x9f, 0x50, 0x00, 0x14, 0x20, 0x17, 0x13, 0xeb,
			0xad, 0x77, 0x15, 0xbe, 0xeb, 0xd9, 0xdf, 0xd1,
			0x9e, 0x3d, 0x7f, 0xcb, 0x60, 0x21, 0x8b, 0xba,
			0x99, 0xa5, 0xb5, 0x58, 0xab, 0xee, 0xa0, 0x8d,
			0x6d, 0xc8, 0xf2, 0x34, 0xdb, 0x8f, 0xab, 0x22,
			0x0c, 0x4f, 0xb5, 0x64, 0x64, 0x2a, 0xbb, 0xc9,
			0x0a, 0xfa, 0x3f, 0xa1, 0x9c, 0x38, 0xe7, 0xc3,
			0x22, 0x1d, 0x18, 0xd5, 0x41, 0x06, 0xc6, 0xc7,
		},
		/* S3 : 384 byte */
		{
			0xa5, 0x8c, 0x3f, 0x48, 0x0e, 0x18, 0x71, 0xea,
			0x49, 0x81, 0x35, 0x2e, 0x85, 0x01, 0xf6, 0x9c,
			0x91, 0xfe, 0x6c, 0x86, 0x74, 0xbe, 0xd6, 0xbe,
			0x01, 0x29, 0xed, 0xa5, 0xd8, 0x6f, 0x14, 0x25,
			0x58, 0x4b, 0xb4, 0xc1, 0xa0, 0x39, 0x9e, 0x5e,
			0x1a, 0x8f, 0xef, 0xf2, 0x19, 0x0e, 0x8c, 0xfc,
			0xc9, 0x94, 0x52, 0x36, 0xf5, 0x27, 0xc3, 0x4a,
			0x35, 0xfd, 0xc3, 0x44, 0x85, 0x35, 0xc8, 0x28,
			0x7f, 0xf7, 0xac, 0xe9, 0x99, 0x45, 0x19, 0xf7,
			0x8c, 0xa5, 0x30, 0x49, 0x84, 0xc4, 0x49, 0x16,
			0x5f, 0x37, 0x17, 0x2b, 0x99, 0xbf, 0x20, 0x95,
			0xb1, 0x2c, 0x8e, 0x58, 0xf5, 0x7f, 0xaa, 0xdb,
			0x77, 0xff, 0xcf, 0x76, 0xa3, 0xd5, 0x32, 0x9c,
			0xed, 0xcf, 0xcf, 0x82, 0x5e, 0xdd, 0x25, 0x4d,
			0x37, 0x9f, 0xa8, 0xc4, 0xab, 0xbb, 0x5c, 0xed,
			0xf7, 0x5a, 0xad, 0xf0, 0x4e, 0x84, 0x1f, 0x8d,
			0x0f, 0x4b, 0xd4, 0xa1, 0x70, 0xc3, 0xf6, 0xca,
			0x6a, 0xae, 0xa4, 0xec, 0x6a, 0x2e, 0xef, 0x57,
			0xf4, 0x02, 0xf0, 0x2b, 0x99, 0xaa, 0x45, 0x4a,
			0xf1, 0x91, 0x6a, 0x8b, 0x11, 0x7a, 0x48, 0xa8,
			0x74, 0x3c, 0x2b, 0xe2, 0x64, 0x2d, 0x8b, 0xc9,
			0xb5, 0x1b, 0xc0, 0x44, 0xea, 0x66, 0x1d, 0xc7,
			0xa0, 0x98, 0x57, 0x72, 0x34, 0x6c, 0x1f, 0x05,
			0x35, 0x5e, 0x03, 0xfd, 0x16, 0xf1, 0x2a, 0xf2,
			0x4e, 0x4b, 0xae, 0xbc, 0x28, 0x53, 0x85, 0x37,
			0xb8, 0xb7, 0x9f, 0x0c, 0x81, 0xe4, 0x77, 0x7a,
			0xa7, 0xaf, 0xc7, 0xb8, 0x2a, 0xe7, 0x6a, 0x0b,
			0x8e, 0xe9, 0x61, 0x85, 0x01, 0x5d, 0x65, 0x86,
			0xe2, 0xd7, 0xdb, 0x07, 0xcf, 0x50, 0x3f, 0xf4,
			0x1f, 0xf4, 0x3e, 0xaa, 0x97, 0xcc, 0xb7, 0xa1,
			0xc4, 0x8e, 0x45, 0xd2, 0xdd, 0x45, 0xcd, 0x0c,
			0x53, 0x1d, 0xdf, 0x1a, 0x34, 0x24, 0x4a, 0x1f,
			0xb1, 0x55, 0xd9, 0xbc, 0x49, 0x90, 0x09, 0x9c,
			0x4b, 0x2f, 0x69, 0x2f, 0x95, 0x1d, 0x59, 0x92,
			0xaf, 0xe8, 0x8d, 0xcc, 0x98, 0xfd, 0x18, 0xf3,
			0x2a, 0xf1, 0x3c, 0x47, 0x9d, 0xbd, 0x6a, 0x72,
			0x5a, 0x01, 0xfb, 0x2f, 0x4a, 0x82, 0xa7, 0x95,
			0x7e, 0xe2, 0x89, 0x6f, 0x38, 0x4e, 0xcb, 0x3d,
			0x19, 0xc5, 0xc1, 0xc8, 0x02, 0xe9, 0xe1, 0xdd,
			0x97, 0xff, 0x2f, 0x8b, 0xc6, 0x70, 0x31, 0x22,
			0x5a, 0xec, 0x77, 0x3d, 0x5d, 0x52, 0xcc, 0x9b,
			0x90, 0x79, 0xf3, 0x54, 0x3a, 0xa0, 0xd0, 0x53,
			0xe9, 0x12, 0x2f, 0x6b, 0x68, 0xa2, 0x22, 0x4b,
			0x87, 0x27, 0x18, 0x90, 0x38, 0xfd, 0xaf, 0x43,
			0x9a, 0x1d, 0x03, 0x67, 0x18, 0x91, 0xc3, 0xc7,
			0xc1, 0x3b, 0x30, 0x64, 0x36, 0x72, 0xce, 0xb7,
			0xcf, 0xee, 0xdd, 0xf7, 0x9b, 0xb5, 0x06, 0x30,
			0x04, 0x22, 0x1e, 0xcb, 0x28, 0x4f, 0xb6, 0xf1,
		},
	},
	/* SHA512224 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0x4c, 0x4d, 0x20, 0x12, 0x2e, 0x76, 0x76, 0xd8,
			0xbe, 0x5f, 0x34, 0x7b, 0x85, 0x3a, 0x4d, 0x6b,
			0x41, 0xa0, 0x8e, 0x96, 0x38, 0x8e, 0x8b, 0x8c,
			0xcd, 0xad, 0xd9, 0xe1, 0x51, 0xd6, 0x7d, 0xd3,
			0x1b, 0xd5, 0xa5, 0x2d, 0xdb, 0x6d, 0x96, 0x6c,
			0x2b, 0x9d, 0xe6, 0x1f, 0x67, 0x22, 0xeb, 0x94,
			0xbf, 0x3d, 0x73, 0xb0, 0x8a, 0xb6, 0xb5, 0x1f,
			0xdd, 0x21, 0x88, 0xae, 0x5d, 0x60, 0xd3, 0xf4,
			0x51, 0x69, 0x45, 0x38, 0xfc, 0x4a, 0x46, 0x8f,
			0x29, 0x6a, 0x2d, 0x45, 0x07, 0x11, 0xb3, 0x7e,
			0x37, 0x2a, 0xa7, 0x1d, 0x24, 0x48, 0xe4, 0xe5,
			0x93, 0xe8, 0x81, 0xde, 0x7b, 0xf6, 0x47, 0xfb,
			0x16, 0x92, 0x68, 0xf9, 0x38, 0x13, 0x6d, 0x8a,
			0xe0, 0x4c, 0x71, 0xb3, 0x0c, 0x0f, 0x8d, 0x73,
			0xd2, 0xf0, 0x93, 0xa5, 0xad, 0x4b, 0xff, 0x27,
			0x15, 0x89, 0x2a, 0x3c, 0x52, 0x9e, 0xc1, 0x2f,
		},
		/* S4 : 384 byte */
		{
			0xa2, 0x79, 0x3a, 0x49, 0x88, 0x18, 0x5c, 0x97,
			0x75, 0x54, 0x37, 0xfb, 0x1f, 0x0b, 0xc5, 0x36,
			0xf5, 0xf2, 0xd9, 0x13, 0xcc, 0xff, 0xbb, 0xab,
			0x02, 0xc4, 0xc6, 0x1c, 0x96, 0x79, 0x5d, 0x5d,
			0x6d, 0x32, 0x29, 0x60, 0x75, 0x76, 0xff, 0x90,
			0x2f, 0x23, 0x14, 0xa3, 0x34, 0x7e, 0x89, 0xb3,
			0x1c, 0x9a, 0xb7, 0x4c, 0x10, 0x2e, 0x7a, 0x3e,
			0x6f, 0x6b, 0xb0, 0x9a, 0x40, 0x5f, 0xc7, 0xd2,
			0x10, 0xd3, 0xdd, 0x70, 0x7a, 0x83, 0x30, 0x12,
			0x7e, 0xe4, 0x37, 0xd4, 0x98, 0x94, 0x66, 0x80,
			0xe0, 0xe3, 0x7e, 0xec, 0x83, 0x18, 0xa0, 0x6a,
			0xeb, 0xb7, 0x2e, 0xb7, 0x70, 0x06, 0xde, 0x51,
			0xbb, 0x43, 0x66, 0x11, 0x90, 0xc0, 0xa0, 0x64,
			0x54, 0xe7, 0x2f, 0xa8, 0x11, 0xb6, 0x20, 0x8b,
			0x17, 0x22, 0x21, 0x69, 0xb0, 0x83, 0x4b, 0x2a,
			0xda, 0xff, 0xe3, 0x5b, 0x5f, 0x9f, 0xf1, 0xf2,
			0xa5, 0x5a, 0xe9, 0x4e, 0x96, 0x80, 0xae, 0x15,
			0x96, 0xe2, 0x8a, 0x9d, 0x9a, 0x70, 0x67, 0xa7,
			0xa1, 0x85, 0x8b, 0x5b, 0x8f, 0xfd, 0x1e, 0x94,
			0xb3, 0x27, 0x45, 0x20, 0xe1, 0x4b, 0xee, 0xd4,
			0xf8, 0xac, 0x3e, 0xdd, 0xb1, 0x61, 0xb3, 0xfe,
			0x8a, 0xfc, 0x2e, 0xee, 0x5d, 0xc2, 0x03, 0x81,
			0x4b, 0x66, 0x5d, 0x62, 0x52, 0x90, 0x17, 0x17,
			0xd6, 0xde, 0x61, 0xdb, 0xb8, 0x6c, 0x74, 0xb8,
			0x21, 0x34, 0xad, 0xec, 0x31, 0x7e, 0x19, 0xf7,
			0x45, 0x6d, 0x22, 0xed, 0x40, 0xac, 0xd4, 0x16,
			0x74, 0xbd, 0xde, 0x76, 0xbe, 0x7d, 0xa4, 0x52,
			0x34, 0x9c, 0xe9, 0xcb, 0x52, 0x11, 0x48, 0x9b,
			0xa9, 0xb4, 0x2e, 0x3a, 0xf4, 0xa0, 0xe6, 0xb6,
			0xae, 0x6f, 0xf3, 0x45, 0xa4, 0xc4, 0xc8, 0x9f,
			0x7e, 0xbf, 0x5d, 0x88, 0xc6, 0x10, 0xcb, 0x87,
			0xd6, 0xca, 0xc6, 0x09, 0xa1, 0xe3, 0x2f, 0xa9,
			0xd6, 0x13, 0xf4, 0xf2, 0xcd, 0xe3, 0x29, 0x98,
			0x47, 0x66, 0xcb, 0xc0, 0x66, 0xa5, 0xf6, 0x1a,
			0x01, 0x5e, 0x7c, 0xd7, 0xe0, 0x8c, 0x61, 0xff,
			0x2b, 0x9c, 0xeb, 0x5d, 0xdc, 0xab, 0xac, 0x7c,
			0x5c, 0x41, 0xa5, 0x7f, 0x56, 0x82, 0x9c, 0x5a,
			0x3b, 0x86, 0xe8, 0x5e, 0xef, 0xac, 0x30, 0x44,
			0x06, 0x04, 0x7e, 0xf6, 0xec, 0x32, 0x0e, 0x1d,
			0x27, 0xc2, 0x37, 0xf5, 0x9b, 0x45, 0x6f, 0x5d,
			0xc2, 0xfe, 0x4a, 0x35, 0x1e, 0x56, 0x28, 0x3d,
			0xa7, 0xda, 0x6b, 0x41, 0xe5, 0xbe, 0x7e, 0x95,
			0x52, 0xa2, 0xd4, 0x1f, 0xd1, 0xa5, 0x64, 0xbe,
			0x8d, 0xdc, 0x3e, 0x9c, 0xea, 0x9f, 0x1c, 0xff,
			0x9f, 0x76, 0xfd, 0x80, 0x8f, 0x9e, 0x9d, 0x73,
			0x35, 0x42, 0x17, 0xcc, 0xd9, 0x32, 0x44, 0x68,
			0x2e, 0xe0, 0x06, 0xaa, 0xa4, 0x65, 0x75, 0x4b,
			0xc6, 0xd1, 0xea, 0x93, 0x6a, 0xa6, 0x8d, 0x47,
		},
	},
	/* SHA512224 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0x5d, 0xc6, 0x0e, 0x1f, 0xe5, 0x96, 0xa9, 0x6b,
			0xe8, 0x14, 0xd8, 0x23, 0x2d, 0x3c, 0xb5, 0xcc,
			0xb6, 0x5d, 0x64, 0xc5, 0xf0, 0x7b, 0xec, 0xd0,
			0x2c, 0xf1, 0x45, 0xce, 0x52, 0x5a, 0xb3, 0xca,
			0xd4, 0xb2, 0x52, 0x39, 0xe2, 0xb9, 0xf1, 0x80,
			0xdd, 0x98, 0x23, 0xb7, 0x0d, 0xc7, 0x94, 0xad,
			0x96, 0x2e, 0x9a, 0x2c, 0xa8, 0x8b, 0x0c, 0x7e,
			0xf7, 0x13, 0x57, 0x2f, 0x6b, 0x5c, 0xce, 0x15,
			0x18, 0x7a, 0x42, 0x8e, 0x70, 0x69, 0xd3, 0x09,
			0xb7, 0xab, 0x05, 0xc8, 0xb8, 0x31, 0x17, 0xe4,
			0xb7, 0x7f, 0x90, 0x91, 0xa6, 0x0c, 0x1a, 0xa3,
			0x9c, 0xe7, 0x93, 0x51, 0x82, 0xa0, 0x62, 0x39,
			0x10, 0x66, 0x07, 0xa5, 0xf8, 0x6d, 0xf6, 0x0d,
			0x60, 0x93, 0xa6, 0xdd, 0x96, 0x42, 0xe5, 0x77,
			0xff, 0x98, 0x6e, 0x7c, 0x51, 0xc5, 0xbd, 0x48,
			0x02, 0xb5, 0x22, 0xbb, 0x01, 0xf0, 0x15, 0x3f,
		},
		/* S5 : 384 byte */
		{
			0x55, 0x60, 0x80, 0xed, 0x70, 0xdf, 0xd8, 0x93,
			0xef, 0x30, 0x4b, 0xce, 0x1e, 0x8e, 0xb4, 0x0d,
			0x1c, 0x15, 0x42, 0xf7, 0x6f, 0xd9, 0x5b, 0x99,
			0x23, 0x21, 0x1b, 0xdb, 0xa6, 0x27, 0xee, 0xb1,
			0x1f, 0x68, 0x6e, 0xf9, 0x15, 0x18, 0xcb, 0x9c,
			0x84, 0x16, 0xd1, 0xbd, 0xb2, 0xb4, 0xa5, 0xc5,
			0x39, 0xa8, 0xfc, 0x72, 0x9e, 0xf4, 0xa7, 0x51,
			0x2d, 0x75, 0xec, 0x14, 0xfb, 0x70, 0xc8, 0x77,
			0x34, 0x1c, 0x9f, 0x85, 0xd3, 0x4e, 0x43, 0x9a,
			0xf1, 0xf6, 0x66, 0x9a, 0xf1, 0x6f, 0x47, 0xa6,
			0x2c, 0x5a, 0x45, 0xa3, 0xc0, 0x05, 0x93, 0x32,
			0x21, 0xaf, 0x60, 0xbd, 0x36, 0xcd, 0x28, 0x74,
			0xa0, 0x0f, 0x09, 0xd0, 0xfa, 0x85, 0xe6, 0x15,
			0xb3, 0x32, 0x2e, 0xe6, 0xb5, 0x38, 0x29, 0x80,
			0x36, 0x34, 0xe6, 0x5b, 0xd1, 0x22, 0xe4, 0x1e,
			0xe9, 0x7d, 0xf1, 0xec, 0x69, 0x95, 0x9d, 0xc2,
			0x97, 0x02, 0xf3, 0x23, 0xa7, 0xb4, 0x45, 0xaa,
			0x4a, 0xcb, 0xc7, 0x78, 0x30, 0x4c, 0xa5, 0x65,
			0x86, 0x61, 0xc3, 0x8e, 0x07, 0x80, 0x27, 0xec,
			0x5b, 0x28, 0x31, 0x57, 0x62, 0xa9, 0x63, 0xb8,
			0x2e, 0xfd, 0x4d, 0xcb, 0xb6, 0x87, 0x93, 0xe4,
			0x1e, 0x27, 0x1d, 0x4f, 0x37, 0x40, 0x04, 0xdb,
			0xa5, 0xed, 0xca, 0xe3, 0x03, 0x9c, 0x92, 0xe0,
			0x2f, 0x97, 0xe5, 0x0f, 0x0e, 0x7b, 0xb1, 0x15,
			0x07, 0x26, 0x89, 0x5e, 0xae, 0xc3, 0x82, 0x14,
			0xe6, 0x54, 0xdb, 0xab, 0x14, 0x26, 0xba, 0x28,
			0x0c, 0xe4, 0xcf, 0xdd, 0x54, 0x9f, 0x47, 0xcd,
			0xe6, 0x18, 0x02, 0x5e, 0x74, 0x1d, 0xbe, 0x46,
			0x9c, 0xca, 0xbe, 0x3e, 0xd3, 0x41, 0xde, 0xae,
			0x95, 0x00, 0x3d, 0xa5, 0x50, 0x3e, 0x23, 0x2d,
			0xae, 0x9a, 0x27, 0xe3, 0xa8, 0xa1, 0x98, 0xf2,
			0xc7, 0x65, 0x3b, 0xa8, 0x67, 0xcb, 0x37, 0xc9,
			0x83, 0xb2, 0x9e, 0x66, 0xec, 0xce, 0xfa, 0x74,
			0x68, 0x21, 0x67, 0x0b, 0xb5, 0x65, 0x7d, 0x6e,
			0xd8, 0xbc, 0xca, 0xf4, 0xf6, 0x61, 0x24, 0x02,
			0xda, 0x43, 0x62, 0xf7, 0xe6, 0xdc, 0x9c, 0x16,
			0x43, 0xff, 0x79, 0x65, 0x72, 0xd0, 0x0f, 0x17,
			0xfb, 0x37, 0x83, 0xe6, 0xc5, 0x32, 0xeb, 0xcc,
			0x42, 0xad, 0x36, 0x00, 0x86, 0x56, 0x58, 0x48,
			0x44, 0xb6, 0x9a, 0x25, 0xdb, 0x5f, 0x77, 0xca,
			0x9b, 0x23, 0x4e, 0x45, 0x58, 0x7e, 0xac, 0xd6,
			0x7c, 0x49, 0x63, 0x5d, 0x6a, 0xc7, 0xd2, 0x47,
			0x21, 0xb3, 0xfa, 0x61, 0xa6, 0x7e, 0x40, 0xfa,
			0x65, 0x94, 0x1e, 0x6a, 0xe4, 0x5b, 0x2b, 0x4c,
			0x6b, 0x5a, 0xf8, 0x6c, 0x17, 0x29, 0xc5, 0xbe,
			0x6e, 0xd9, 0x79, 0xc2, 0xbe, 0x03, 0x5a, 0xe2,
			0xba, 0x28, 0x56, 0x92, 0xb5, 0xd5, 0xf5, 0xe5,
			0x58, 0xce, 0x98, 0x66, 0xa7, 0xfd, 0xc3, 0x29,
		},
	},
	/* SHA512224 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0xc3, 0x7e, 0xc2, 0x8b, 0xa5, 0x64, 0xae, 0x2c,
			0xd1, 0xf7, 0xfa, 0x7e, 0x76, 0x74, 0x7b, 0x55,
			0x2a, 0xee, 0x89, 0xf2, 0x3a, 0x4e, 0xfb, 0x26,
			0x3e, 0x99, 0x53, 0xbb, 0xf1, 0x71, 0x35, 0x0f,
			0x40, 0xdf, 0x1e, 0x02, 0xa1, 0x4f, 0x52, 0x12,
			0xa6, 0xdc, 0x16, 0xc5, 0xd2, 0xce, 0x20, 0x48,
			0xab, 0x61, 0x4c, 0x55, 0x10, 0xc9, 0x90, 0x59,
			0xcd, 0x72, 0x32, 0xd7, 0x6d, 0x8c, 0x38, 0x07,
			0x50, 0xc7, 0x21, 0xc4, 0xfd, 0x5b, 0xd2, 0xa5,
			0xb8, 0x4c, 0xd3, 0x69, 0x58, 0xec, 0xbb, 0x98,
			0x53, 0xa1, 0xe9, 0x67, 0x1b, 0xe8, 0x75, 0xdc,
			0xad, 0x9a, 0x66, 0x34, 0x66, 0x6f, 0x26, 0x82,
			0x19, 0xc0, 0x33, 0x98, 0x60, 0x91, 0x16, 0x2a,
			0x2f, 0xce, 0x99, 0x31, 0xaf, 0xd5, 0x35, 0x8e,
			0x49, 0x35, 0xca, 0xf0, 0x00, 0xb5, 0x93, 0xf7,
			0x05, 0x99, 0x58, 0x9a, 0x85, 0x20, 0xe5, 0xc6,
		},
		/* S6 : 384 byte */
		{
			0xa4, 0xd5, 0x89, 0x1d, 0xba, 0x2e, 0x4f, 0xdb,
			0x56, 0x26, 0xd2, 0xc9, 0x2a, 0x38, 0x74, 0xc8,
			0xd3, 0x78, 0xc8, 0x72, 0x5b, 0x51, 0x00, 0x41,
			0xb3, 0x23, 0x4a, 0xf3, 0xcc, 0xd1, 0xd3, 0x74,
			0x44, 0x69, 0xab, 0x4f, 0x97, 0x05, 0xff, 0x9e,
			0x16, 0xfe, 0xa9, 0x19, 0x22, 0xf0, 0x2c, 0xb0,
			0xe8, 0x16, 0xb6, 0x5c, 0x19, 0xf4, 0xac, 0x5a,
			0xbf, 0xf1, 0xd1, 0x88, 0x71, 0xd8, 0x08, 0x6e,
			0x5d, 0x8c, 0x84, 0x6f, 0x43, 0xe6, 0xcb, 0x6e,
			0x88, 0x17, 0xc9, 0x9e, 0xcd, 0xb9, 0x4f, 0x5f,
			0x2f, 0xc9, 0xc4, 0x30, 0x91, 0x19, 0x3b, 0xca,
			0x25, 0x12, 0x76, 0x70, 0xeb, 0x2c, 0x00, 0x6b,
			0x20, 0xe5, 0xf2, 0x7f, 0x32, 0xd4, 0xa1, 0xcc,
			0x8f, 0xfa, 0x14, 0xdf, 0xe9, 0x10, 0xa3, 0xb3,
			0x81, 0x3a, 0x8b, 0xff, 0xd2, 0xeb, 0x65, 0x67,
			0x53, 0xca, 0x66, 0x7c, 0xfa, 0xab, 0x81, 0xf9,
			0xad, 0x7e, 0x6a, 0x48, 0x18, 0x4e, 0xa0, 0x5d,
			0xdb, 0x98, 0x0a, 0xca, 0x57, 0x31, 0x17, 0x1b,
			0xb7, 0xcb, 0x58, 0xdc, 0x30, 0xc3, 0x79, 0xdd,
			0x40, 0x18, 0x6e, 0x68, 0x9f, 0xbd, 0xe3, 0x78,
			0xaa, 0x9d, 0x76, 0xea, 0x5a, 0x41, 0x28, 0xcf,
			0x58, 0xe4, 0x44, 0x0c, 0x1a, 0x69, 0x29, 0xac,
			0x1a, 0xde, 0x86, 0xfe, 0xd7, 0x5a, 0x4a, 0x10,
			0xf6, 0x1a, 0xe7, 0x89, 0x2f, 0x69, 0x61, 0xdb,
			0x17, 0xb0, 0xe2, 0xca, 0x99, 0xd2, 0x4e, 0xf3,
			0xfe, 0x72, 0xd6, 0x47, 0xb9, 0x8e, 0xd8, 0x3c,
			0x74, 0x3c, 0xe0, 0x04, 0xb7, 0x08, 0x63, 0xa3,
			0x5c, 0xd3, 0x56, 0x16, 0x35, 0x8e, 0xa2, 0x34,
			0x22, 0x88, 0xe7, 0x4b, 0x5e, 0x4b, 0x22, 0x9f,
			0xf5, 0x69, 0xed, 0x47, 0xf9, 0x7e, 0x57, 0xaf,
			0x8b, 0x55, 0xe8, 0x96, 0x8c, 0x8d, 0x03, 0x15,
			0x60, 0x0b, 0x0b, 0xc0, 0x49, 0xcc, 0xdf, 0xee,
			0xd4, 0x90, 0x4c, 0xd5, 0xab, 0xa0, 0x55, 0x38,
			0x36, 0x1f, 0xeb, 0xad, 0x22, 0x92, 0x64, 0x69,
			0xe4, 0x4a, 0xcd, 0xa4, 0x2b, 0x46, 0x63, 0xa2,
			0xc2, 0x75, 0x5f, 0xe2, 0x60, 0x41, 0xa5, 0xb8,
			0x49, 0x1f, 0x82, 0x4a, 0x40, 0x5e, 0xa3, 0x77,
			0xb5, 0x5d, 0x4d, 0x86, 0x95, 0x17, 0x36, 0xb6,
			0x99, 0xa1, 0x53, 0x7d, 0x24, 0xf9, 0x9f, 0xb9,
			0xb7, 0x2f, 0x14, 0x2c, 0x25, 0xa4, 0x9b, 0x62,
			0x16, 0x56, 0x91, 0xd6, 0x20, 0x2c, 0x9c, 0xcf,
			0x2b, 0x14, 0x90, 0x05, 0x36, 0x4b, 0x7d, 0x45,
			0x5f, 0x6b, 0x23, 0x5a, 0x95, 0x30, 0x97, 0xfd,
			0x6a, 0xa2, 0xf5, 0xb9, 0xc5, 0xf4, 0x29, 0x7a,
			0x47, 0x3f, 0x1f, 0xb8, 0x83, 0xbd, 0x74, 0xb9,
			0xa4, 0x8f, 0x71, 0xfa, 0xd4, 0x37, 0x67, 0x37,
			0xac, 0x39, 0x06, 0xd6, 0xe1, 0xb1, 0x4c, 0x53,
			0xa3, 0xc0, 0x35, 0xb5, 0x85, 0x79, 0x87, 0x11,
		},
	},
	/* SHA512224 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0x8a, 0xa6, 0x67, 0x95, 0xa6, 0xd6, 0x8e, 0x37,
			0xba, 0xe9, 0xa9, 0x34, 0x20, 0xe2, 0x76, 0xde,
			0x2e, 0xd2, 0x9b, 0xd4, 0x5c, 0x61, 0xeb, 0x30,
			0xa4, 0xaa, 0xe5, 0x02, 0x2f, 0x4c, 0x7d, 0xfe,
			0x2d, 0xbd, 0x20, 0x71, 0x05, 0xe2, 0xf2, 0x7a,
			0xae, 0xdd, 0x5a, 0x76, 0x5c, 0x27, 0xc0, 0xbc,
			0x60, 0xde, 0x95, 0x8b, 0x49, 0x60, 0x94, 0x40,
			0x50, 0x18, 0x48, 0xcc, 0xf3, 0x98, 0xcf, 0x66,
			0xdf, 0xe8, 0xdd, 0x7d, 0x13, 0x1e, 0x04, 0xf1,
			0x43, 0x2f, 0x32, 0x82, 0x7a, 0x05, 0x7b, 0x89,
			0x04, 0xd2, 0x18, 0xe6, 0x8b, 0xa3, 0xb0, 0x39,
			0x80, 0x38, 0xd7, 0x55, 0xbd, 0x13, 0xd5, 0xf1,
			0x68, 0xcf, 0xa8, 0xa1, 0x1a, 0xb3, 0x4c, 0x05,
			0x40, 0x87, 0x39, 0x40, 0xc2, 0xa6, 0x2e, 0xac,
			0xe3, 0x55, 0x2d, 0xcd, 0x69, 0x53, 0xc6, 0x83,
			0xfd, 0xb2, 0x99, 0x83, 0xd4, 0xe4, 0x17, 0x07,
		},
		/* S7 : 384 byte */
		{
			0x39, 0x76, 0xc3, 0x7b, 0x76, 0x25, 0xb0, 0x2d,
			0x79, 0x95, 0x94, 0xd2, 0xcb, 0x85, 0x34, 0x19,
			0xee, 0x71, 0x4d, 0x72, 0xe8, 0x95, 0xa1, 0x07,
			0xd4, 0x9e, 0x26, 0xa3, 0x56, 0x40, 0x9f, 0x9d,
			0x84, 0x70, 0xd4, 0x37, 0xa2, 0x48, 0x29, 0x6e,
			0xa4, 0xc1, 0x41, 0xd4, 0xad, 0x3a, 0x3f, 0xdc,
			0xaa, 0xb0, 0xcc, 0x1c, 0x84, 0x71, 0x45, 0x50,
			0x56, 0x7c, 0x1a, 0x9a, 0x30, 0x09, 0x23, 0x4b,
			0x87, 0x86, 0xf5, 0x91, 0x2e, 0x9c, 0x4f, 0x6c,
			0x02, 0x47, 0x0e, 0x66, 0x5f, 0x71, 0x63, 0x8d,
			0x55, 0x99, 0xed, 0x8c, 0x1d, 0x66, 0x64, 0x5a,
			0xb5, 0x6e, 0xa1, 0x12, 0x11, 0xdf, 0xae, 0xdf,
			0x53, 0xdd, 0x4b, 0xa6, 0x4e, 0x26, 0x49, 0x2b,
			0x4e, 0x83, 0xcf, 0xa7, 0xea, 0x7b, 0x9a, 0xf3,
			0xda, 0x71, 0x3e, 0x8e, 0xcb, 0xf9, 0x0b, 0x6d,
			0xd8, 0xb4, 0xca, 0x35, 0xa2, 0x9c, 0xcd, 0xa8,
			0x0e, 0x33, 0x7a, 0x40, 0xe6, 0x71, 0x11, 0xe2,
			0x66, 0x58, 0x65, 0x93, 0xa6, 0x7a, 0xde, 0x79,
			0x10, 0x4d, 0x51, 0xfa, 0xdc, 0x33, 0x50, 0x61,
			0x77, 0xfd, 0xda, 0xda, 0x9e, 0xc9, 0xbb, 0x69,
			0x08, 0xf5, 0xd2, 0x5f, 0xf8, 0x54, 0x23, 0x69,
			0x4a, 0xd6, 0x6f, 0xdc, 0x2c, 0x4a, 0x34, 0xba,
			0x13, 0xbf, 0x0f, 0x25, 0xf0, 0x96, 0xac, 0x28,
			0x47, 0x26, 0x3a, 0xad, 0x34, 0xe9, 0x17, 0xcc,
			0x0f, 0xc3, 0x14, 0x2c, 0x23, 0x7f, 0x07, 0x39,
			0x22, 0xa7, 0x63, 0x32, 0x59, 0x8e, 0xe4, 0x9a,
			0xaa, 0x18, 0x16, 0xeb, 0xb8, 0x8d, 0x05, 0x4f,
			0x35, 0xe0, 0x3a, 0x25, 0x29, 0x5d, 0x6f, 0x5f,
			0x32, 0x0a, 0x94, 0x9c, 0x73, 0x42, 0x8e, 0xe9,
			0xeb, 0xde, 0x93, 0x0e, 0xce, 0x4d, 0xe6, 0x33,
			0xae, 0xd1, 0x0f, 0x1d, 0x50, 0x20, 0x86, 0xcb,
			0x46, 0x66, 0xd0, 0x63, 0x84, 0xd8, 0x4f, 0x4f,
			0xb0, 0xf6, 0x05, 0x58, 0x27, 0x87, 0x13, 0xc4,
			0x9c, 0xf0, 0x07, 0x26, 0x8b, 0x98, 0x70, 0x73,
			0x4a, 0xda, 0x1c, 0xe8, 0x20, 0xfc, 0xe3, 0xc5,
			0xac, 0x3b, 0x95, 0xd3, 0x95, 0x6c, 0x2a, 0x38,
			0xb0, 0x3c, 0x07, 0xb1, 0x60, 0x09, 0xff, 0x67,
			0x3e, 0x2f, 0x3f, 0x29, 0xf8, 0xcc, 0x3d, 0x92,
			0xeb, 0x53, 0xf1, 0x38, 0xf1, 0xd4, 0x65, 0x3e,
			0x42, 0x03, 0xf7, 0xfd, 0x05, 0xe0, 0xe1, 0xae,
			0xd0, 0xec, 0xd7, 0xd4, 0xbc, 0xab, 0x95, 0xe5,
			0x29, 0x6a, 0xc6, 0x3b, 0x58, 0xae, 0xe6, 0x16,
			0x4f, 0xcb, 0x74, 0xd7, 0xa2, 0x22, 0xea, 0xe8,
			0x2c, 0xc5, 0xb5, 0x05, 0x06, 0x20, 0xfa, 0x26,
			0x17, 0x1e, 0xe0, 0xcc, 0xfe, 0xb5, 0x37, 0x7e,
			0xe5, 0x27, 0x58, 0x90, 0x5a, 0x4c, 0xd7, 0x59,
			0xa8, 0xb2, 0xda, 0x02, 0xe1, 0x54, 0x01, 0xaf,
			0xab, 0x3b, 0x56, 0xf3, 0x45, 0xfb, 0x59, 0xce,
		},
	},
	/* SHA512224 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0x10, 0xf5, 0x8d, 0xdb, 0xd9, 0x98, 0xe2, 0x60,
			0x04, 0x30, 0x0e, 0x40, 0x79, 0xd6, 0x56, 0xfc,
			0xc6, 0xa9, 0xeb, 0x55, 0xb9, 0x16, 0x19, 0xb2,
			0x2d, 0x56, 0xc0, 0xc3, 0x24, 0x37, 0x08, 0x64,
			0x16, 0x20, 0x51, 0xc4, 0x89, 0x4f, 0x69, 0xc6,
			0xe2, 0x66, 0x02, 0xa7, 0x7b, 0x26, 0x63, 0xe2,
			0xf5, 0x7a, 0x5d, 0x50, 0xd0, 0xb7, 0x81, 0x94,
			0x38, 0x22, 0x90, 0xb6, 0x22, 0xb3, 0x33, 0x0e,
			0x98, 0x18, 0xec, 0x64, 0x51, 0xfd, 0x4d, 0x55,
			0x85, 0x8b, 0x68, 0xf8, 0xff, 0x2e, 0x86, 0xc2,
			0x74, 0x9c, 0x1c, 0xa7, 0x13, 0x13, 0xfa, 0x82,
			0x5d, 0xe1, 0xc6, 0xb7, 0x37, 0x2a, 0xa9, 0x17,
			0x3e, 0xe6, 0x48, 0x04, 0x5a, 0x29, 0xf6, 0xd5,
			0x96, 0xa7, 0x27, 0x7c, 0x2f, 0x77, 0x28, 0x65,
			0xea, 0x17, 0x0e, 0xa2, 0xab, 0xb1, 0xed, 0x46,
			0x44, 0x9c, 0x4a, 0x0e, 0x8b, 0x26, 0xd2, 0x47,
		},
		/* S8 : 384 byte */
		{
			0x8e, 0x3a, 0x05, 0xba, 0xe9, 0x44, 0x08, 0x53,
			0x0c, 0xc0, 0x0a, 0x60, 0xfd, 0xfb, 0x33, 0x40,
			0x5b, 0x2a, 0x08, 0x7f, 0xb2, 0x61, 0x3b, 0x30,
			0xf4, 0x46, 0xea, 0x55, 0x20, 0xc2, 0x9c, 0x66,
			0x15, 0xf5, 0x12, 0x2f, 0xb5, 0xc1, 0xd2, 0x9a,
			0x2e, 0x45, 0xb4, 0x88, 0xca, 0x11, 0x8b, 0x97,
			0x08, 0x85, 0x60, 0xfa, 0x1a, 0x7d, 0x86, 0xe1,
			0x1c, 0xcc, 0xfa, 0xd2, 0x37, 0x26, 0x4d, 0x4f,
			0xbb, 0x95, 0x1a, 0x3a, 0x4f, 0xec, 0xc4, 0xa1,
			0x17, 0x1a, 0x50, 0x8e, 0x1b, 0x7c, 0xa4, 0x61,
			0x90, 0x47, 0xd6, 0xd2, 0x16, 0xe9, 0x7c, 0x19,
			0x55, 0xab, 0x73, 0xac, 0x77, 0x94, 0x2e, 0x00,
			0x5c, 0x07, 0xb2, 0xee, 0x95, 0xdc, 0xa8, 0x24,
			0xe1, 0x35, 0x45, 0xf6, 0x7d, 0x79, 0x84, 0x7a,
			0x53, 0xbc, 0xf0, 0xd4, 0x22, 0x2e, 0x70, 0xe2,
			0xe2, 0x6b, 0x2b, 0x28, 0xc1, 0xbe, 0x91, 0x51,
			0x41, 0xc6, 0x38, 0xe7, 0x7b, 0xc9, 0x3e, 0x8f,
			0x1d, 0x47, 0x93, 0xfd, 0xce, 0x77, 0x50, 0x7a,
			0xb7, 0x10, 0x00, 0x7e, 0x07, 0xc1, 0xb5, 0x53,
			0xf5, 0x7e, 0x59, 0x9d, 0x7f, 0xd9, 0xd4, 0xff,
			0xe2, 0x08, 0x31, 0x2d, 0x84, 0x47, 0x28, 0x1d,
			0x1e, 0xba, 0xda, 0x5c, 0x51, 0x7a, 0x77, 0x35,
			0xa4, 0x67, 0x68, 0x21, 0x3e, 0x85, 0x6e, 0xa1,
			0x12, 0x9a, 0xd3, 0x26, 0xf7, 0xc1, 0x51, 0x24,
			0x22, 0x3c, 0x33, 0xce, 0x13, 0x6c, 0xef, 0xf3,
			0xe6, 0x71, 0x5a, 0x47, 0xb4, 0xae, 0x7c, 0xf8,
			0xe9, 0x5b, 0xe9, 0x43, 0x17, 0x14, 0x2d, 0x12,
			0xea, 0xd5, 0xed, 0x59, 0xfa, 0x91, 0xce, 0x15,
			0x7d, 0xb2, 0x93, 0xf5, 0xf2, 0xbd, 0x30, 0x42,
			0x50, 0x0e, 0x7f, 0x0a, 0x7a, 0x28, 0x1c, 0xf8,
			0x25, 0x17, 0x1d, 0x66, 0x20, 0xd6, 0x1b, 0x51,
			0x88, 0xe0, 0x95, 0x1c, 0x53, 0xc1, 0x7b, 0xd2,
			0x2b, 0x72, 0xe7, 0xd7, 0x52, 0x0b, 0x49, 0x07,
			0x81, 0xa8, 0x46, 0xe5, 0x0e, 0xe2, 0x1a, 0xed,
			0x2f, 0x32, 0x62, 0xb6, 0x36, 0x9f, 0x59, 0xde,
			0x19, 0xd0, 0x88, 0x84, 0x34, 0x18, 0x0d, 0x36,
			0x7b, 0x55, 0x3c, 0x0e, 0xb4, 0x7b, 0xcf, 0xd0,
			0x85, 0xba, 0x1b, 0x8c, 0xc2, 0xb5, 0xe4, 0x8a,
			0x11, 0x75, 0x8f, 0xcb, 0xfc, 0x8f, 0x5a, 0xdb,
			0xa9, 0x5b, 0x06, 0x58, 0x10, 0x79, 0x97, 0xd5,
			0x32, 0x07, 0xd2, 0x20, 0x5f, 0xd3, 0x3f, 0x0e,
			0x78, 0x91, 0xc5, 0x4e, 0xb5, 0xa8, 0xca, 0xc0,
			0x31, 0xc7, 0xf1, 0x38, 0x82, 0x4b, 0x15, 0xe5,
			0xa1, 0xd2, 0x76, 0xad, 0x9c, 0x64, 0x20, 0x05,
			0x44, 0x2c, 0x6a, 0xda, 0x0b, 0x21, 0x61, 0xad,
			0x5b, 0xa6, 0xdd, 0x33, 0x9a, 0x38, 0x94, 0x64,
			0x75, 0xe2, 0xef, 0xd0, 0xa5, 0xc0, 0x1a, 0x6b,
			0x6a, 0x93, 0x36, 0x0f, 0x26, 0x98, 0xe3, 0x4d,
		},
	},
	/* SHA512224 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0x01, 0x3d, 0x90, 0xb2, 0x69, 0x9c, 0xeb, 0x79,
			0xcb, 0x94, 0xed, 0xfa, 0xf8, 0x41, 0x8f, 0x56,
			0x5b, 0xee, 0x41, 0x50, 0x80, 0x89, 0x8c, 0x5b,
			0xd8, 0x55, 0xf6, 0xf7, 0xb6, 0xdf, 0x26, 0x3e,
			0x9e, 0x33, 0x48, 0xba, 0xbf, 0x4f, 0xb3, 0x7a,
			0xbe, 0xd8, 0xc9, 0x65, 0xb5, 0x0c, 0x1f, 0xce,
			0x7c, 0xf8, 0xdf, 0x48, 0x27, 0xeb, 0x6a, 0xbb,
			0xcf, 0x2a, 0x41, 0xb7, 0x96, 0xe5, 0x25, 0x09,
			0xe5, 0x6a, 0x7d, 0x8c, 0xf8, 0x99, 0xf9, 0x42,
			0x9d, 0x98, 0x75, 0xa3, 0x38, 0xc8, 0x20, 0x35,
			0x0b, 0xf7, 0xdd, 0x5d, 0xb4, 0xd7, 0xe9, 0x75,
			0xf7, 0xae, 0xbe, 0x1e, 0xbe, 0x50, 0x38, 0xd8,
			0x5e, 0x4b, 0xf7, 0xd0, 0x1c, 0x61, 0x04, 0xf8,
			0xf0, 0x3b, 0xb7, 0x5d, 0x89, 0x5b, 0xd8, 0xb6,
			0x90, 0x53, 0x04, 0x38, 0x31, 0x34, 0x52, 0xb2,
			0xd8, 0x4a, 0x37, 0xd5, 0x38, 0x06, 0xa7, 0xd3,
		},
		/* S9 : 384 byte */
		{
			0x21, 0xe6, 0x5d, 0x04, 0xc4, 0x1b, 0xfb, 0x83,
			0x7c, 0xb9, 0xb9, 0xf2, 0x6b, 0xd1, 0xd9, 0x20,
			0xf6, 0xc1, 0x6f, 0x11, 0x78, 0x98, 0x0a, 0xdf,
			0x3d, 0xb2, 0x97, 0xc1, 0x6a, 0x76, 0x75, 0xf9,
			0xe1, 0xad, 0x15, 0xe0, 0x0e, 0x1c, 0x86, 0xb3,
			0x97, 0x25, 0x0e, 0x1c, 0x5f, 0xab, 0xd7, 0xbf,
			0xe6, 0x91, 0x55, 0xb8, 0xdb, 0x4b, 0x5e, 0x18,
			0xc5, 0xc8, 0xb1, 0x69, 0x48, 0x34, 0x8f, 0x8b,
			0x73, 0xdb, 0xa2, 0x74, 0xc9, 0xa4, 0x7d, 0x7a,
			0x35, 0x32, 0x9e, 0x4d, 0x6a, 0xc3, 0x60, 0x1a,
			0x08, 0xaf, 0x34, 0xfb, 0x62, 0x72, 0x26, 0x4e,
			0x4a, 0xe5, 0x26, 0x8c, 0x1e, 0x59, 0xa7, 0x3f,
			0xe4, 0xab, 0xde, 0x5b, 0xe0, 0x2c, 0xc1, 0x4a,
			0xb3, 0x3e, 0x3d, 0xe7, 0x7a, 0xb0, 0x64, 0x46,
			0xfc, 0x4f, 0x32, 0x0f, 0x08, 0x05, 0x27, 0x9d,
			0x74, 0x73, 0x36, 0x45, 0xbc, 0x2d, 0xe6, 0xb6,
			0xf5, 0x18, 0xb8, 0x41, 0x35, 0x00, 0x48, 0xe1,
			0x07, 0x8c, 0x07, 0x52, 0x1a, 0x02, 0xd7, 0x56,
			0x9f, 0x5e, 0x70, 0x0e, 0x80, 0x09, 0x9b, 0x69,
			0xb8, 0xc4, 0xdc, 0xf6, 0xd4, 0xaa, 0x89, 0x6a,
			0x64, 0x47, 0x54, 0xa3, 0xa9, 0x74, 0x92, 0xf8,
			0x2a, 0x96, 0x64, 0xb9, 0xfe, 0xf0, 0x41, 0xf6,
			0xfd, 0xa2, 0x96, 0x7d, 0xdf, 0x88, 0xda, 0x84,
			0x82, 0x15, 0x8a, 0x17, 0x97, 0x85, 0x6b, 0xf6,
			0xb8, 0xd0, 0x71, 0xa1, 0xee, 0xf5, 0x0d, 0xf8,
			0x3d, 0x6f, 0x64, 0x84, 0xe2, 0xa3, 0xdb, 0x02,
			0xc6, 0x52, 0x53, 0x1d, 0x8c, 0x04, 0x46, 0x6a,
			0x55, 0xe7, 0x9f, 0x46, 0x7f, 0xba, 0xce, 0xd5,
			0x67, 0x0d, 0x65, 0x32, 0x5b, 0x87, 0x88, 0x5a,
			0xbd, 0x6d, 0x16, 0xc7, 0x8c, 0xb4, 0x24, 0xc1,
			0x9a, 0x09, 0x0b, 0x4d, 0xbd, 0xac, 0xb1, 0xfa,
			0x8e, 0xeb, 0xfd, 0xe9, 0x7b, 0xa4, 0xfa, 0xf5,
			0x1e, 0x57, 0x65, 0x21, 0x75, 0x6b, 0xe3, 0xc7,
			0x61, 0xcd, 0xfa, 0x85, 0xda, 0xb3, 0xd4, 0xba,
			0x89, 0xf2, 0x76, 0xff, 0xf9, 0x43, 0x6f, 0x17,
			0x6b, 0x83, 0x9f, 0xa5, 0x93, 0x4c, 0x88, 0x61,
			0x2d, 0x7a, 0x77, 0x5c, 0xcd, 0xba, 0xc8, 0xea,
			0x59, 0xe3, 0xb6, 0xf7, 0xe2, 0x47, 0x90, 0x22,
			0x17, 0x2f, 0x17, 0xf1, 0xad, 0x74, 0x9a, 0xc6,
			0xfc, 0xa2, 0x8d, 0x69, 0x8a, 0xa5, 0x0b, 0xc4,
			0xab, 0x92, 0xd8, 0xea, 0x36, 0xe4, 0xb3, 0x40,
			0x9d, 0xd4, 0xb6, 0x71, 0xcb, 0x00, 0xcd, 0x2f,
			0x7d, 0x3e, 0x54, 0x4b, 0x96, 0xd7, 0x49, 0x53,
			0xd0, 0x50, 0x9a, 0x15, 0xbc, 0xb0, 0xb5, 0x2a,
			0xa6, 0x67, 0x1a, 0x5e, 0x13, 0x28, 0x14, 0xd1,
			0x89, 0x08, 0xb7, 0xda, 0x88, 0xfc, 0x9e, 0x3a,
			0x7e, 0xe8, 0x43, 0x95, 0x7f, 0x6b, 0xcd, 0x0a,
			0x79, 0x3f, 0x41, 0xef, 0x90, 0x54, 0x82, 0x3d,
		},
	},
};

/* 10 elements */
static mod3072_t SHA512256[] = {
	/* SHA512256 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0xf2, 0x36, 0x60, 0xb3, 0x33, 0x75, 0xda, 0xde,
			0x2f, 0x51, 0x19, 0xec, 0xdf, 0xee, 0x7c, 0x53,
			0x39, 0x7a, 0x4a, 0xce, 0x54, 0xc0, 0xf5, 0x9b,
			0xcc, 0xfa, 0x9b, 0xe7, 0x68, 0xe7, 0xd2, 0x87,
			0xfa, 0x1e, 0xa5, 0xe8, 0x87, 0xa2, 0xda, 0x49,
			0xa8, 0xb6, 0x77, 0x27, 0x30, 0xf3, 0x84, 0xf9,
			0x0f, 0x97, 0x11, 0x1e, 0x77, 0x20, 0x7b, 0x14,
			0x3d, 0x1a, 0xec, 0xea, 0x81, 0x35, 0x21, 0xf8,
			0x91, 0x9a, 0x70, 0xcd, 0x0b, 0x7f, 0x09, 0x69,
			0x42, 0xf9, 0x7c, 0xad, 0xe3, 0x12, 0x7b, 0x0f,
			0xd9, 0x1c, 0xe1, 0x92, 0x6d, 0x45, 0xf4, 0xf6,
			0xb2, 0x6b, 0xe7, 0x2b, 0x20, 0x31, 0xa4, 0x0c,
			0x7f, 0x52, 0xc8, 0x4a, 0x07, 0x35, 0xee, 0xa6,
			0xc5, 0xc2, 0x30, 0x64, 0x40, 0x75, 0xeb, 0xfc,
			0x5d, 0xb0, 0xc3, 0x12, 0x80, 0x56, 0xe7, 0xa8,
			0xf4, 0x95, 0x96, 0x93, 0x8d, 0x03, 0xe3, 0x2b,
		},
		/* S0 : 384 byte */
		{
			0x31, 0xf3, 0x55, 0x1b, 0x29, 0x9c, 0x1e, 0xbc,
			0xf1, 0xa6, 0x71, 0x11, 0xe6, 0x4e, 0x56, 0xba,
			0x63, 0x61, 0xa0, 0xfc, 0xf6, 0xad, 0x81, 0x3f,
			0xcc, 0x3f, 0x7b, 0xbc, 0x93, 0x7b, 0xfc, 0xb9,
			0x91, 0xda, 0x38, 0x4b, 0x7c, 0x6b, 0x5e, 0x6a,
			0x11, 0x0d, 0xbd, 0x6c, 0x4c, 0x9d, 0x94, 0xde,
			0x21, 0x45, 0x43, 0x78, 0x55, 0x44, 0x2d, 0x1e,
			0xca, 0x3a, 0x4b, 0xcb, 0xf4, 0x21, 0xde, 0x8d,
			0x7b, 0x5a, 0x8d, 0x54, 0x52, 0x5a, 0xbb, 0x1d,
			0xf6, 0x09, 0x5d, 0x52, 0xff, 0xf3, 0x3d, 0x4d,
			0x74, 0xed, 0x2f, 0xb4, 0xc3, 0xaf, 0x43, 0x0f,
			0xc0, 0xcc, 0x3b, 0xde, 0x9f, 0xd0, 0x5f, 0x10,
			0x21, 0xca, 0x26, 0x7e, 0x95, 0xbc, 0x82, 0xf0,
			0x9c, 0xd5, 0x0b, 0xd5, 0xcc, 0x7e, 0xc7, 0xed,
			0xe6, 0x3f, 0x3e, 0x5b, 0x8e, 0xef, 0xb3, 0xd0,
			0x24, 0xff, 0xf0, 0xec, 0x74, 0x79, 0xde, 0x98,
			0x3b, 0xea, 0x9e, 0xd3, 0x6e, 0xd8, 0x71, 0xef,
			0x5c, 0x65, 0x38, 0x9d, 0x83, 0x32, 0x64, 0x38,
			0x2b, 0xbb, 0x6c, 0x48, 0xa6, 0xdc, 0x5b, 0xde,
			0x54, 0x0a, 0x4b, 0xa8, 0xd9, 0x41, 0x63, 0xe6,
			0x6d, 0x93, 0xc4, 0x15, 0xc3, 0x4b, 0x5c, 0x06,
			0xb9, 0x65, 0x62, 0x6e, 0x85, 0x0d, 0x1e, 0xc7,
			0x94, 0x54, 0x0d, 0xb2, 0x7c, 0x55, 0x18, 0x01,
			0x17, 0x7a, 0x78, 0x1a, 0xaa, 0x32, 0x1f, 0x77,
			0x12, 0x4e, 0x63, 0xc6, 0x3b, 0x42, 0x8a, 0xe8,
			0x81, 0x6c, 0x5f, 0x14, 0x50, 0x5b, 0x33, 0xe4,
			0xf1, 0x0a, 0x93, 0x06, 0x0c, 0xb4, 0x10, 0x5b,
			0x03, 0x4b, 0x54, 0x57, 0x5c, 0x62, 0x52, 0xd1,
			0x56, 0x1b, 0xbe, 0x24, 0x2f, 0xb9, 0xbe, 0x13,
			0x71, 0xe4, 0x72, 0x61, 0x47, 0x2f, 0x91, 0x47,
			0x2f, 0x81, 0xf6, 0x02, 0x94, 0x23, 0x57, 0xcd,
			0xe6, 0x41, 0xcf, 0x73, 0x01, 0x5e, 0x16, 0x1d,
			0x03, 0xfc, 0xce, 0x07, 0x55, 0xab, 0x6a, 0x7c,
			0x92, 0x78, 0xa5, 0x35, 0xb2, 0x7a, 0x9d, 0xa4,
			0xc0, 0x02, 0xf6, 0xf6, 0x12, 0x31, 0x22, 0x15,
			0x43, 0xbe, 0x57, 0x9e, 0x9b, 0xc9, 0xd0, 0xd5,
			0x23, 0x9e, 0x2f, 0xce, 0x7f, 0xcd, 0x1d, 0xd6,
			0xea, 0x0a, 0xc1, 0x95, 0x4a, 0xaf, 0x27, 0xd7,
			0xd8, 0x04, 0xf5, 0x49, 0x25, 0x6e, 0x95, 0x7b,
			0x7e, 0x82, 0x66, 0xee, 0xf0, 0x2b, 0x6b, 0x90,
			0x5f, 0x5f, 0xe6, 0xec, 0xb4, 0x72, 0x88, 0xa5,
			0xf3, 0xd9, 0x74, 0x49, 0x84, 0x6c, 0xcb, 0x3a,
			0xc5, 0x64, 0x04, 0x66, 0x97, 0xda, 0xcd, 0x05,
			0xf1, 0xb6, 0x35, 0x76, 0x37, 0xe0, 0x00, 0x05,
			0x2e, 0x44, 0x31, 0xd4, 0x2a, 0x48, 0x4a, 0xfe,
			0x7d, 0xb8, 0x5d, 0x82, 0x1b, 0x1c, 0xcd, 0x37,
			0xea, 0xa0, 0x13, 0x1f, 0x8d, 0x16, 0x67, 0xaf,
			0x94, 0x0f, 0x32, 0xf2, 0xc2, 0xfe, 0x96, 0x9b,
		},
	},
	/* SHA512256 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0x43, 0xe2, 0xcc, 0x2a, 0x6a, 0x8e, 0xa6, 0x45,
			0x65, 0xff, 0x6c, 0xe2, 0xfd, 0x2c, 0x4f, 0x43,
			0xfc, 0x02, 0x92, 0x6e, 0xe4, 0x4e, 0xe0, 0x2f,
			0xe1, 0xdc, 0xe2, 0x5c, 0xfd, 0xe0, 0x11, 0x5c,
			0x93, 0x96, 0xc9, 0xea, 0x06, 0x26, 0x9f, 0x17,
			0xb2, 0xca, 0xf5, 0x8e, 0x23, 0x32, 0xcc, 0x1c,
			0x85, 0x28, 0xd9, 0x70, 0x5c, 0x70, 0xda, 0x1f,
			0x76, 0xf2, 0x2a, 0xeb, 0x1d, 0x1b, 0x93, 0x44,
			0x91, 0x80, 0x64, 0x0f, 0xb5, 0xc4, 0xc4, 0xa7,
			0x08, 0xbc, 0x46, 0x21, 0xd7, 0xd2, 0xbe, 0xd5,
			0xb1, 0xa7, 0x52, 0x19, 0x1c, 0xfd, 0xd4, 0x50,
			0x86, 0xd3, 0x4f, 0x24, 0x7e, 0xd1, 0xdf, 0x0f,
			0x24, 0xe7, 0xc6, 0x20, 0xde, 0x32, 0xbd, 0xfc,
			0x4d, 0x1f, 0x88, 0x23, 0x80, 0xd2, 0xcd, 0x74,
			0x67, 0xc9, 0x26, 0xf4, 0x8a, 0xbc, 0x75, 0xcb,
			0xfa, 0xc8, 0x78, 0x8f, 0x88, 0xcd, 0x9d, 0xc5,
		},
		/* S1 : 384 byte */
		{
			0x85, 0xf3, 0xe1, 0xc6, 0x0a, 0x2d, 0x06, 0x29,
			0x0b, 0x0c, 0x8a, 0xd0, 0xb2, 0xd9, 0x21, 0x94,
			0x04, 0x8e, 0x3a, 0xd1, 0x7b, 0xd0, 0xa1, 0xf2,
			0x27, 0x1f, 0x5c, 0xb1, 0x45, 0x75, 0x13, 0xdb,
			0x2d, 0x06, 0x01, 0x18, 0x65, 0x08, 0xcd, 0x15,
			0x1a, 0x7a, 0x86, 0xc2, 0x38, 0xcd, 0xbb, 0xa8,
			0xd4, 0x51, 0xee, 0xbe, 0xc4, 0x2f, 0x3c, 0x7b,
			0xae, 0x03, 0x17, 0xe0, 0x2a, 0xe9, 0xfb, 0xe2,
			0x9a, 0x04, 0x51, 0xee, 0x7d, 0xc2, 0x3b, 0xe3,
			0x18, 0x5f, 0x76, 0xc2, 0xb3, 0xc8, 0xf3, 0x02,
			0xb2, 0x05, 0x95, 0x8d, 0x32, 0x11, 0xd2, 0x44,
			0xa3, 0xb0, 0x21, 0xe6, 0xa7, 0x03, 0xf0, 0xbb,
			0x34, 0x65, 0x12, 0xc7, 0x3d, 0x0a, 0x2b, 0x65,
			0x67, 0x98, 0x47, 0x1d, 0xa4, 0x93, 0xe1, 0x81,
			0x3e, 0x13, 0x17, 0xb1, 0x98, 0xd4, 0x85, 0x2b,
			0xf2, 0xe8, 0x65, 0xfb, 0x53, 0xc1, 0xed, 0xdc,
			0x4c, 0xe8, 0xf4, 0x46, 0xa7, 0xb3, 0x2b, 0xec,
			0x75, 0x5c, 0x0c, 0x57, 0x0d, 0xc1, 0xde, 0xcc,
			0x62, 0xfc, 0x05, 0x1d, 0x27, 0x29, 0x22, 0x6d,
			0xb9, 0x3f, 0x39, 0xa0, 0x14, 0x99, 0xf9, 0x45,
			0xe8, 0x5c, 0x07, 0x23, 0xe9, 0x19, 0xc1, 0xd0,
			0xa0, 0x04, 0x01, 0xdf, 0xa0, 0xd1, 0xc2, 0x1f,
			0x4a, 0x19, 0xdb, 0x81, 0x74, 0xac, 0xa1, 0x0b,
			0x07, 0xdc, 0x70, 0xbd, 0xf0, 0x91, 0x4e, 0x15,
			0x23, 0x5f, 0x14, 0x83, 0xc2, 0x51, 0x39, 0x93,
			0x2a, 0x8e, 0xb3, 0x33, 0x64, 0xe7, 0xee, 0x79,
			0x56, 0xcb, 0x2f, 0xdc, 0x5f, 0x67, 0x47, 0x89,
			0x8b, 0x80, 0x8f, 0x0d, 0x72, 0x27, 0x67, 0x93,
			0xe7, 0xab, 0x79, 0x82, 0x28, 0x3d, 0x07, 0x21,
			0xd7, 0xd5, 0x60, 0xfe, 0x36, 0xdb, 0x6f, 0xc8,
			0xe6, 0x76, 0x12, 0x45, 0xf9, 0x64, 0xa0, 0x4c,
			0xb4, 0x9a, 0xbf, 0xbe, 0xf8, 0xd2, 0xab, 0x33,
			0x0a, 0x0d, 0x50, 0x7c, 0x7a, 0x14, 0x7f, 0x0a,
			0xb1, 0x7a, 0xc1, 0xac, 0x6f, 0x8d, 0xdf, 0x97,
			0xa7, 0x4b, 0x7f, 0x63, 0xdf, 0x61, 0xf6, 0xc6,
			0x90, 0xd5, 0xc5, 0x20, 0x9c, 0x78, 0x82, 0x01,
			0xa0, 0xf2, 0x2c, 0x42, 0x9f, 0x8f, 0x05, 0x24,
			0x86, 0x1f, 0x3b, 0x78, 0x0b, 0xa0, 0x3f, 0xac,
			0xf9, 0x94, 0x9b, 0x57, 0x0f, 0xe6, 0x20, 0xdb,
			0x18, 0xc5, 0x56, 0xb2, 0xca, 0x83, 0x79, 0x29,
			0x52, 0x48, 0x99, 0x4d, 0x6a, 0x4a, 0xe8, 0xe5,
			0x93, 0x75, 0x1d, 0xc0, 0xbc, 0x90, 0x8b, 0xef,
			0x26, 0x16, 0x9f, 0x48, 0xb2, 0x5d, 0x16, 0xad,
			0x63, 0x4a, 0xd1, 0x3b, 0xd3, 0xa0, 0xd9, 0xf8,
			0x7e, 0x2d, 0x01, 0x13, 0x68, 0x09, 0x96, 0xc8,
			0x47, 0x22, 0xa6, 0xf4, 0xc8, 0x7e, 0x46, 0x93,
			0x14, 0x69, 0x01, 0xe9, 0x4e, 0x07, 0x93, 0xe1,
			0xe1, 0xec, 0xcf, 0xf3, 0x36, 0xad, 0xf6, 0x97,
		},
	},
	/* SHA512256 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x6e, 0x32, 0x80, 0xf4, 0xed, 0xb2, 0x45, 0x13,
			0x84, 0x50, 0x83, 0x56, 0x0a, 0x17, 0x65, 0x49,
			0xa8, 0x1b, 0x04, 0xb1, 0xdf, 0x66, 0x8b, 0x1f,
			0xcc, 0x35, 0x99, 0xc5, 0xab, 0x65, 0xe6, 0x89,
			0x9b, 0x28, 0x2a, 0x58, 0xa0, 0xfc, 0x3a, 0xbd,
			0xee, 0xde, 0x74, 0xb2, 0x65, 0xba, 0x5e, 0xb6,
			0x58, 0x27, 0x8a, 0x1f, 0x92, 0x51, 0xbd, 0xb2,
			0x9b, 0x36, 0x4f, 0x71, 0x37, 0x16, 0xd5, 0xb4,
			0x30, 0x24, 0xfe, 0x7b, 0x55, 0x82, 0xbb, 0x03,
			0xc3, 0x6c, 0xa3, 0x97, 0x63, 0xb4, 0x95, 0xa9,
			0xb4, 0x6e, 0x9f, 0x21, 0xcb, 0xec, 0x1e, 0xf5,
			0x98, 0xed, 0x27, 0xfa, 0x6a, 0x11, 0x26, 0xfa,
			0xb5, 0x90, 0xc5, 0x06, 0x14, 0x2c, 0x10, 0x0d,
			0x8a, 0x64, 0xd6, 0xae, 0x0d, 0xeb, 0x52, 0x4b,
			0x45, 0x58, 0x0a, 0x5f, 0x91, 0x1f, 0xf8, 0x11,
			0x4b, 0xc0, 0xe8, 0x09, 0x4d, 0x3e, 0x21, 0x82,
		},
		/* S2 : 384 byte */
		{
			0x2b, 0x14, 0x6c, 0x0a, 0x99, 0x63, 0x50, 0x07,
			0x95, 0xdc, 0xc7, 0xfc, 0xcd, 0x53, 0xce, 0x8a,
			0xd2, 0x23, 0xba, 0x89, 0x9a, 0x98, 0x07, 0x26,
			0xc1, 0xd0, 0x80, 0x40, 0x68, 0x8e, 0xcd, 0xd3,
			0xcb, 0xd7, 0xf6, 0x48, 0xfc, 0x7e, 0x88, 0x65,
			0xda, 0x6b, 0x12, 0xe7, 0x43, 0xc4, 0xf3, 0x4a,
			0x54, 0x4a, 0x74, 0x6b, 0x1c, 0x90, 0x2c, 0xf5,
			0xf2, 0xf1, 0x30, 0x24, 0x5c, 0xca, 0x3c, 0x50,
			0x06, 0x75, 0xd1, 0x78, 0x6e, 0xfc, 0x2f, 0x66,
			0x85, 0xbd, 0x6c, 0x82, 0xf9, 0xc0, 0x6d, 0xfe,
			0xa9, 0x84, 0x0a, 0x17, 0x2d, 0xe2, 0x7b, 0xe6,
			0x78, 0x8c, 0xaa, 0xe3, 0xff, 0xc5, 0x36, 0x5e,
			0xac, 0xc7, 0x50, 0x6f, 0x88, 0xf6, 0x0b, 0xdf,
			0x9b, 0x9e, 0x95, 0xd7, 0x8d, 0xf1, 0x02, 0x0f,
			0xf8, 0x97, 0xce, 0xad, 0xb3, 0x5d, 0x00, 0x6d,
			0xcf, 0x08, 0x0a, 0x25, 0xf4, 0x5e, 0x65, 0xd7,
			0x48, 0x80, 0xd5, 0xa0, 0x05, 0xa2, 0x34, 0x14,
			0x62, 0x77, 0x33, 0x22, 0x83, 0xf5, 0x75, 0xbb,
			0xab, 0x12, 0x27, 0x89, 0xdc, 0xb6, 0x2b, 0x68,
			0x52, 0x90, 0x64, 0x8d, 0x1e, 0x24, 0x0f, 0x47,
			0x1e, 0x8e, 0xa0, 0x36, 0xc4, 0xb2, 0x06, 0x4e,
			0xfa, 0x53, 0x3d, 0x58, 0x87, 0x67, 0x0b, 0x8e,
			0xb6, 0x06, 0x7e, 0x1b, 0x99, 0xfc, 0x2d, 0x44,
			0xd2, 0xaf, 0xd1, 0x34, 0xf2, 0x6b, 0x71, 0x68,
			0x11, 0x76, 0x7e, 0x8d, 0xe0, 0x69, 0x63, 0xf8,
			0x49, 0x88, 0xa6, 0x24, 0x7a, 0xeb, 0x58, 0xd9,
			0xc8, 0x6a, 0x86, 0x0c, 0xbe, 0x48, 0xe9, 0x97,
			0x84, 0x8c, 0xb5, 0xaa, 0x7d, 0x42, 0xd8, 0x33,
			0xd4, 0xe7, 0x28, 0x26, 0xec, 0x67, 0xa8, 0x22,
			0xae, 0x61, 0x87, 0x96, 0xf5, 0x13, 0x97, 0xf4,
			0x2f, 0xfe, 0x55, 0x9d, 0xb9, 0xcd, 0x33, 0x14,
			0x0e, 0x48, 0x25, 0x86, 0xc8, 0x14, 0x2b, 0x43,
			0x41, 0x39, 0x28, 0x39, 0xd4, 0x1e, 0xc1, 0xb5,
			0xe9, 0xe5, 0xc8, 0x3e, 0xb3, 0x5b, 0x5f, 0x53,
			0xcc, 0xc7, 0x90, 0x86, 0xfa, 0x4e, 0x71, 0xde,
			0xda, 0x67, 0x8d, 0xb7, 0x11, 0x30, 0xa7, 0x3f,
			0x7f, 0x68, 0xd4, 0x5c, 0x13, 0x54, 0xf1, 0x53,
			0x6d, 0xa7, 0x81, 0xaa, 0x5c, 0x17, 0x57, 0x5f,
			0x78, 0xcb, 0x58, 0xa4, 0x56, 0xfe, 0xce, 0x8d,
			0x9d, 0x97, 0x55, 0xe6, 0xb8, 0xd5, 0xb8, 0xc8,
			0x2c, 0x42, 0xb0, 0x37, 0x99, 0x6a, 0x30, 0xfc,
			0x82, 0x3c, 0xcf, 0xb4, 0xc1, 0xd0, 0x6f, 0x6a,
			0x58, 0x97, 0xbd, 0x64, 0xd2, 0xa4, 0x91, 0x2d,
			0xda, 0xdb, 0x7d, 0x24, 0x7e, 0xb1, 0xb5, 0x61,
			0x40, 0xcf, 0x8b, 0x43, 0x57, 0x88, 0xd6, 0xca,
			0xc7, 0xc9, 0x56, 0x3f, 0x01, 0x9d, 0xa5, 0xaf,
			0xe3, 0x50, 0xef, 0xdc, 0xcf, 0x55, 0x45, 0xe8,
			0x82, 0xec, 0x59, 0xfc, 0x76, 0x60, 0xbe, 0x83,
		},
	},
	/* SHA512256 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0x59, 0xdc, 0xad, 0xa0, 0x47, 0xc1, 0x30, 0xbd,
			0x40, 0x26, 0x53, 0x9f, 0xed, 0x60, 0x23, 0xf7,
			0xd0, 0x37, 0x57, 0x6e, 0x7b, 0x4b, 0x24, 0x2d,
			0x00, 0xb0, 0x59, 0xa3, 0xa8, 0xdc, 0xa7, 0x82,
			0xe8, 0xc3, 0x40, 0x0b, 0x90, 0x40, 0xa5, 0x8c,
			0x46, 0xb7, 0xb8, 0x27, 0x33, 0xf9, 0x19, 0xd4,
			0xfc, 0x67, 0xaf, 0xa9, 0xf4, 0x5b, 0x85, 0x5a,
			0x90, 0xc3, 0xd8, 0xfc, 0x1b, 0x0f, 0x6e, 0x0e,
			0xaa, 0x4e, 0x29, 0xb9, 0x53, 0x93, 0xdb, 0x57,
			0x9a, 0x9e, 0x5c, 0x33, 0x2c, 0xb8, 0xda, 0x8f,
			0xd0, 0xdf, 0x71, 0xec, 0x9a, 0x23, 0xfb, 0x85,
			0x61, 0x50, 0x2a, 0x1d, 0x74, 0xcd, 0xd4, 0xfa,
			0x89, 0xc4, 0x8d, 0x32, 0xf6, 0x83, 0x7b, 0x25,
			0x24, 0x23, 0x66, 0x4b, 0x40, 0x67, 0x0f, 0x30,
			0x33, 0xe6, 0xc3, 0xbc, 0xd5, 0x6d, 0x2f, 0xb8,
			0x5d, 0x9f, 0x76, 0x8e, 0x1d, 0xde, 0x81, 0x50,
		},
		/* S3 : 384 byte */
		{
			0x84, 0xcc, 0xaa, 0x67, 0x2f, 0x52, 0xaf, 0x4f,
			0x28, 0x5e, 0x81, 0x15, 0x3d, 0xf4, 0x26, 0x88,
			0x3f, 0x53, 0x50, 0x21, 0xe8, 0x97, 0x16, 0xc3,
			0x07, 0x70, 0x6c, 0x35, 0xd9, 0x3f, 0xd6, 0x1b,
			0xe1, 0x41, 0x31, 0x75, 0x3c, 0x8b, 0xb5, 0x93,
			0x2f, 0xe1, 0x23, 0x6e, 0x9a, 0xda, 0x13, 0x76,
			0x5f, 0xa7, 0x4b, 0xe3, 0x39, 0x79, 0x6a, 0x75,
			0x01, 0xe3, 0xfd, 0xf1, 0x53, 0xc4, 0x14, 0x2f,
			0xbe, 0x42, 0x48, 0xfb, 0xd9, 0x10, 0x48, 0xc4,
			0xac, 0x76, 0x64, 0x57, 0x15, 0xb9, 0x91, 0x49,
			0x0c, 0x0f, 0x46, 0x3f, 0xeb, 0xf4, 0x89, 0x3c,
			0x02, 0xb5, 0x42, 0xa6, 0x55, 0xf0, 0x73, 0xc6,
			0x4e, 0xfe, 0x20, 0xab, 0x2b, 0x5c, 0xb7, 0x7c,
			0x8e, 0xcd, 0x41, 0xc3, 0x9a, 0x73, 0x88, 0xa5,
			0xb5, 0x30, 0xf6, 0xeb, 0xf8, 0x22, 0xe2, 0x63,
			0xb8, 0x8c, 0xdd, 0x31, 0x29, 0x64, 0xe8, 0x5c,
			0x75, 0x88, 0xc8, 0x72, 0x0b, 0x7f, 0xcd, 0xcf,
			0x2b, 0x52, 0x89, 0xb0, 0x44, 0x94, 0x96, 0x40,
			0x03, 0x62, 0x49, 0xfe, 0xd3, 0x0e, 0x34, 0x7c,
			0x1a, 0x40, 0xd9, 0x57, 0x46, 0x01, 0xad, 0x86,
			0x37, 0xfa, 0xa4, 0xac, 0xf9, 0x29, 0x59, 0x72,
			0x96, 0x64, 0xc4, 0x11, 0x03, 0xda, 0x6b, 0xb6,
			0x6e, 0xdf, 0x95, 0xd1, 0xa2, 0x0d, 0xab, 0x6d,
			0xc6, 0xa4, 0x01, 0x0d, 0xe7, 0x6f, 0x0c, 0xcd,
			0x8e, 0xe6, 0x12, 0x96, 0x96, 0xb2, 0xe1, 0x40,
			0x53, 0xda, 0x51, 0x49, 0x43, 0x68, 0x5d, 0x6f,
			0x52, 0x70, 0x03, 0x89, 0xd9, 0x31, 0xbe, 0xad,
			0x03, 0xaf, 0x43, 0xd6, 0xfc, 0x31, 0xfa, 0x28,
			0x1b, 0x1e, 0x41, 0xda, 0x24, 0x4c, 0x1b, 0x69,
			0xf3, 0x77, 0xe4, 0xd7, 0xb1, 0xd7, 0x5d, 0x1f,
			0xa5, 0xf1, 0xe6, 0x3a, 0xf8, 0x3a, 0x39, 0xb8,
			0xc5, 0x9b, 0x2a, 0x19, 0x28, 0x1e, 0x6f, 0x65,
			0xaf, 0x83, 0x56, 0x8d, 0x04, 0xe2, 0x83, 0xa5,
			0xe7, 0x44, 0xce, 0xcf, 0xd4, 0xa2, 0x61, 0xdb,
			0x48, 0xd2, 0x15, 0x9b, 0x0b, 0x7f, 0x0c, 0x8a,
			0x34, 0x16, 0xd8, 0x73, 0xeb, 0x72, 0xc7, 0x68,
			0x98, 0xa1, 0x1b, 0xf9, 0xad, 0xa0, 0x70, 0xf4,
			0xb9, 0xd3, 0x76, 0x2f, 0x94, 0x2f, 0xfa, 0x74,
			0xa5, 0xb5, 0x7d, 0x26, 0xdd, 0x52, 0x7e, 0x95,
			0xd7, 0x62, 0xe1, 0xd6, 0x2f, 0x38, 0xe1, 0x69,
			0x3c, 0x16, 0x88, 0xe3, 0x0b, 0x6e, 0x56, 0x81,
			0x2f, 0x0e, 0x01, 0xe4, 0x25, 0xbf, 0x4c, 0xc2,
			0xa2, 0x97, 0xde, 0xb1, 0x43, 0x19, 0xe5, 0xa3,
			0xf0, 0xec, 0x24, 0x4c, 0xd0, 0x50, 0x43, 0x7d,
			0x93, 0x01, 0x26, 0x8f, 0xd5, 0x5a, 0x9f, 0x75,
			0x81, 0x23, 0x98, 0x5d, 0x4a, 0x0a, 0xd5, 0x96,
			0xfd, 0x45, 0x6b, 0x4b, 0x24, 0xc0, 0x95, 0x72,
			0xc0, 0x78, 0x1a, 0xe2, 0x13, 0x64, 0xe0, 0xc3,
		},
	},
	/* SHA512256 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0x07, 0x3b, 0x07, 0x19, 0x75, 0xd8, 0xb6, 0xb8,
			0xcf, 0x60, 0x9f, 0xc0, 0xb3, 0x07, 0xf2, 0x2e,
			0x74, 0xe3, 0x3f, 0xe3, 0xbf, 0xfe, 0x38, 0x58,
			0x7f, 0xaa, 0xb9, 0x8d, 0xfd, 0x14, 0x05, 0x42,
			0x4c, 0x1c, 0x53, 0x6a, 0xe4, 0x64, 0x89, 0xdd,
			0x8d, 0xa7, 0x47, 0x4d, 0x7f, 0x4d, 0x27, 0x7b,
			0xc7, 0x3c, 0x43, 0xaa, 0xec, 0xf0, 0xf8, 0x54,
			0xd3, 0xce, 0xea, 0x1d, 0xe0, 0xb5, 0x48, 0x05,
			0x5e, 0xd7, 0x52, 0xe1, 0x21, 0xc7, 0x18, 0x0c,
			0x69, 0xd2, 0x84, 0x5a, 0x0a, 0x93, 0x9a, 0x4d,
			0xca, 0xc2, 0xff, 0x8c, 0x0b, 0x4d, 0xb8, 0x90,
			0xaa, 0xaa, 0x35, 0xa1, 0x26, 0x6b, 0x8e, 0x01,
			0x05, 0x13, 0x0c, 0x67, 0x73, 0x28, 0xeb, 0xae,
			0x2d, 0x89, 0x5e, 0xcf, 0x9d, 0x02, 0xd4, 0x0d,
			0x46, 0x1e, 0x7a, 0x6f, 0x63, 0x3c, 0x01, 0x73,
			0xcb, 0xe5, 0xa1, 0x01, 0x18, 0x5d, 0x5e, 0x9e,
		},
		/* S4 : 384 byte */
		{
			0x09, 0x9d, 0x03, 0x20, 0x90, 0x73, 0xb4, 0x00,
			0x94, 0x82, 0x91, 0x1b, 0x6d, 0xc9, 0xa2, 0xa9,
			0xd9, 0x38, 0x39, 0x7c, 0xee, 0x56, 0x94, 0xb6,
			0x1d, 0xe2, 0x81, 0x78, 0xe4, 0x0f, 0xd0, 0x74,
			0x35, 0x12, 0x13, 0x23, 0xb3, 0x6d, 0x10, 0x74,
			0xb9, 0xde, 0xbd, 0x50, 0x38, 0x6c, 0x0b, 0xdd,
			0x63, 0xe1, 0xf2, 0x6b, 0xd1, 0xf2, 0x9c, 0x7b,
			0xee, 0xc2, 0xbe, 0x3e, 0x4e, 0x02, 0x33, 0xbb,
			0x88, 0x55, 0x23, 0x19, 0xda, 0xca, 0x3d, 0x35,
			0xf4, 0x60, 0x2a, 0x6e, 0xb6, 0x53, 0xe2, 0x29,
			0x26, 0x31, 0x5a, 0x95, 0x23, 0x1d, 0x9c, 0xfd,
			0x7a, 0xef, 0x40, 0xe2, 0x80, 0xac, 0x1a, 0xc5,
			0x72, 0xd4, 0x5f, 0x11, 0xd1, 0x15, 0x39, 0xb8,
			0x58, 0xbd, 0xf2, 0xbf, 0x78, 0x5c, 0xb3, 0x1c,
			0xb3, 0x8d, 0xbc, 0xcb, 0xd7, 0x07, 0xb6, 0x3e,
			0xaf, 0xcb, 0x0c, 0xa1, 0x62, 0x64, 0x48, 0xba,
			0x81, 0x0c, 0xaa, 0xf1, 0xed, 0xd6, 0xfa, 0xf3,
			0xd5, 0x3a, 0x02, 0x29, 0xc1, 0xef, 0xfa, 0x62,
			0x56, 0xe4, 0x1d, 0x3e, 0xb7, 0x96, 0x2f, 0xb9,
			0x3b, 0xf4, 0xad, 0xc0, 0xdc, 0x83, 0x5b, 0x6c,
			0x46, 0x0e, 0x4d, 0xb2, 0x8a, 0x3d, 0xd7, 0xea,
			0x6e, 0x81, 0x8c, 0xab, 0x8d, 0xc0, 0xfe, 0x3c,
			0x35, 0x25, 0xae, 0xec, 0x5f, 0xa6, 0xf6, 0x4d,
			0xed, 0x10, 0x5e, 0xf9, 0x0c, 0x46, 0xb5, 0x59,
			0x8f, 0x39, 0xe8, 0x26, 0x3e, 0x20, 0x26, 0x8d,
			0x07, 0x2e, 0xad, 0x3c, 0xfe, 0x07, 0xbc, 0x73,
			0xdf, 0xee, 0x40, 0xcb, 0x25, 0x7b, 0x55, 0x93,
			0x8c, 0xe1, 0xb7, 0xdd, 0x40, 0x23, 0x67, 0x71,
			0x0a, 0x18, 0xb5, 0x05, 0xd5, 0x83, 0x26, 0xd2,
			0xde, 0xf2, 0x4e, 0x1b, 0x76, 0xb3, 0x42, 0xc5,
			0xb9, 0x96, 0xa5, 0x99, 0x1d, 0x9d, 0xad, 0x6f,
			0x36, 0x9b, 0x91, 0x58, 0xeb, 0x60, 0x3d, 0x7a,
			0x9d, 0x78, 0x3e, 0xef, 0x6b, 0xc8, 0xf1, 0x6c,
			0x1c, 0x45, 0x8c, 0x4e, 0xe1, 0x11, 0x27, 0x09,
			0x6f, 0xb3, 0xe7, 0x82, 0x61, 0xe3, 0x96, 0x5e,
			0x0b, 0xe0, 0x37, 0xaa, 0x02, 0xfe, 0xe0, 0x83,
			0xa2, 0xc6, 0x61, 0x73, 0x9f, 0xaf, 0x44, 0xc1,
			0x01, 0x7a, 0x22, 0x5f, 0x6d, 0x7e, 0x1b, 0x9f,
			0xb6, 0x86, 0x08, 0x00, 0x7f, 0xa9, 0x3f, 0xd5,
			0x60, 0x4e, 0x4c, 0x0c, 0x0a, 0xde, 0x97, 0x83,
			0x4f, 0x10, 0x2b, 0xfd, 0xad, 0x72, 0xfb, 0xe3,
			0x2d, 0xd7, 0xea, 0x07, 0x0e, 0xc4, 0xd0, 0xa7,
			0x34, 0x5d, 0x2e, 0xbb, 0x2e, 0xb4, 0x05, 0x28,
			0x5a, 0xb4, 0xa4, 0xc9, 0xc6, 0xe6, 0xa1, 0x15,
			0x3f, 0x1d, 0xa8, 0x42, 0x02, 0x75, 0x2e, 0x4e,
			0xca, 0xc3, 0x8c, 0x80, 0x9a, 0xe2, 0xb7, 0x58,
			0x99, 0x79, 0xcc, 0x62, 0x4d, 0x68, 0x3d, 0x05,
			0x4c, 0x17, 0xac, 0x33, 0x6a, 0xc1, 0x8f, 0x83,
		},
	},
	/* SHA512256 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0xd3, 0xc6, 0xaa, 0x33, 0xa8, 0x0c, 0xc8, 0x50,
			0x35, 0xfc, 0x72, 0x1d, 0xa5, 0x63, 0x57, 0x65,
			0x2c, 0xf8, 0x5b, 0x48, 0xf5, 0x88, 0x30, 0x83,
			0x5b, 0x50, 0x26, 0xf1, 0x5b, 0xab, 0x44, 0x52,
			0x49, 0xed, 0x97, 0x4c, 0xc3, 0x8e, 0xca, 0xaf,
			0xe4, 0x8b, 0xf7, 0xfd, 0xb9, 0x09, 0xbd, 0x2a,
			0xf9, 0x3b, 0x93, 0xa0, 0xf0, 0x87, 0x5b, 0x86,
			0xbd, 0xe4, 0x3b, 0x42, 0xbd, 0x87, 0xa8, 0x3c,
			0x47, 0xbe, 0xc3, 0xe5, 0x9a, 0x1c, 0xe9, 0xf9,
			0x13, 0xd6, 0x85, 0x01, 0xa4, 0x6d, 0x29, 0x1b,
			0x81, 0x8d, 0xde, 0xfd, 0x1c, 0x36, 0xb8, 0x38,
			0x54, 0x19, 0xad, 0xba, 0xe9, 0x45, 0xa6, 0x97,
			0x34, 0x02, 0xd7, 0x09, 0x15, 0xfd, 0x4c, 0xb5,
			0x2c, 0xa6, 0xc5, 0x2f, 0x4c, 0xd7, 0xc4, 0xc1,
			0x2c, 0xb6, 0xe4, 0x68, 0x60, 0xdc, 0x6c, 0x20,
			0x88, 0xb7, 0x24, 0x61, 0xc8, 0x2c, 0x68, 0xea,
		},
		/* S5 : 384 byte */
		{
			0x9b, 0xef, 0xb0, 0x9c, 0xfd, 0xd9, 0x6b, 0x3e,
			0x42, 0x32, 0xe6, 0x6e, 0xad, 0xb8, 0x85, 0x80,
			0x65, 0xf1, 0x5e, 0x5e, 0x6e, 0xf0, 0x0b, 0x6c,
			0x1f, 0xe6, 0x6d, 0xef, 0xe6, 0x16, 0x96, 0xbf,
			0x05, 0x76, 0x6a, 0x81, 0x83, 0x0e, 0xc8, 0x8d,
			0x87, 0x5f, 0x33, 0x06, 0x28, 0x45, 0x90, 0x55,
			0x91, 0xee, 0xa0, 0xda, 0xcb, 0x00, 0x4c, 0xc0,
			0x22, 0x7a, 0x49, 0x7a, 0xd6, 0x2d, 0x89, 0xa1,
			0x2c, 0xc2, 0xcd, 0x3f, 0xd1, 0xf8, 0xd0, 0x3a,
			0x74, 0x4d, 0xae, 0x91, 0x29, 0xe8, 0x38, 0x2d,
			0x9c, 0xb3, 0x5e, 0xe0, 0x42, 0x13, 0xbd, 0x02,
			0xde, 0x5f, 0x4c, 0x7f, 0xc7, 0xb2, 0xac, 0x0b,
			0xea, 0x68, 0xbc, 0x03, 0xeb, 0xcb, 0x9e, 0x77,
			0xc2, 0xad, 0x3c, 0x62, 0x4b, 0x5b, 0xb3, 0xa9,
			0x69, 0x96, 0x0b, 0x1e, 0xba, 0xfc, 0x70, 0xf2,
			0x26, 0xcd, 0xd6, 0xda, 0xa2, 0xde, 0xa5, 0xf8,
			0x47, 0x5b, 0x77, 0x70, 0x6e, 0x3f, 0x34, 0x1f,
			0xb4, 0x28, 0x83, 0x3c, 0x44, 0x85, 0x12, 0x7a,
			0x07, 0x87, 0x85, 0x50, 0x04, 0x08, 0xda, 0xe6,
			0xbe, 0xf7, 0x91, 0xc8, 0x39, 0xd8, 0xa0, 0xe1,
			0xa9, 0x35, 0xf5, 0x43, 0x47, 0x77, 0xd5, 0x2a,
			0x47, 0xb8, 0xb8, 0xbd, 0x6b, 0xd1, 0x42, 0x44,
			0xd6, 0x12, 0x9d, 0xd6, 0x3f, 0x0e, 0xd5, 0x34,
			0x26, 0xd6, 0xb0, 0x34, 0x47, 0x14, 0xed, 0x68,
			0x77, 0x3b, 0x59, 0xeb, 0x3a, 0x4e, 0x2c, 0x39,
			0xfa, 0x09, 0x07, 0x96, 0x9c, 0x40, 0xbe, 0x66,
			0x18, 0xc8, 0x35, 0x5a, 0xeb, 0x52, 0x69, 0xf5,
			0xab, 0x32, 0x8a, 0xeb, 0x35, 0x52, 0x11, 0xc8,
			0xee, 0xe7, 0x2f, 0xe8, 0x14, 0x1c, 0xfa, 0xcc,
			0x8b, 0x22, 0xae, 0x1e, 0x1d, 0x6b, 0x45, 0x44,
			0x3f, 0xaa, 0x7c, 0x6f, 0x89, 0x09, 0x38, 0x5f,
			0x78, 0x5d, 0xb0, 0x74, 0xb3, 0x17, 0xb3, 0x7a,
			0x7e, 0x94, 0x60, 0x14, 0x8f, 0x7f, 0x44, 0x0f,
			0x14, 0x2c, 0xe0, 0x3b, 0xbc, 0xd7, 0xdc, 0xbc,
			0x25, 0x39, 0x03, 0x1a, 0x4e, 0x33, 0x4d, 0x5d,
			0xd9, 0xfb, 0x6c, 0xa6, 0xec, 0xb3, 0x67, 0xe0,
			0xa7, 0x8a, 0x21, 0x53, 0xdd, 0x9b, 0xb9, 0xae,
			0xad, 0x00, 0x95, 0x78, 0xfe, 0xf5, 0x58, 0xdd,
			0x61, 0x8c, 0x02, 0x25, 0x48, 0xff, 0x5b, 0xfd,
			0x63, 0xbc, 0x2a, 0x40, 0x12, 0x37, 0xd4, 0x8b,
			0x56, 0x5b, 0xb0, 0x2b, 0xc5, 0x5c, 0x92, 0x46,
			0x6e, 0xf2, 0x9a, 0x75, 0x45, 0x73, 0xe1, 0x46,
			0xd8, 0xb0, 0x2e, 0xff, 0xd1, 0x16, 0xa5, 0x17,
			0x50, 0x63, 0x39, 0x5b, 0xcb, 0xde, 0xfd, 0xc9,
			0xdf, 0xb5, 0x80, 0x27, 0xe6, 0xa3, 0xf6, 0x1a,
			0x32, 0x3c, 0x73, 0x67, 0x00, 0x75, 0x00, 0x4f,
			0x3e, 0x53, 0x97, 0x90, 0x74, 0xf9, 0x9f, 0xcb,
			0xf2, 0xdb, 0xa3, 0xfe, 0x0e, 0x47, 0xf0, 0x30,
		},
	},
	/* SHA512256 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0x84, 0x9e, 0x5d, 0x6b, 0x17, 0x80, 0x2a, 0xea,
			0x21, 0x0d, 0xc2, 0xa5, 0xab, 0x7b, 0x94, 0x18,
			0xd4, 0xbc, 0xdc, 0x63, 0xc6, 0xa2, 0x2f, 0x96,
			0xea, 0x47, 0x54, 0x2c, 0x58, 0xac, 0x56, 0xd5,
			0x92, 0xa0, 0xd7, 0x79, 0x71, 0xe4, 0xa7, 0x79,
			0x5f, 0x8c, 0x96, 0x38, 0xc7, 0xa8, 0x86, 0xda,
			0xca, 0x24, 0xc4, 0x4b, 0x34, 0xed, 0x98, 0x83,
			0xae, 0x55, 0x1a, 0x09, 0x35, 0xb9, 0x48, 0xb6,
			0xca, 0x61, 0x56, 0xbc, 0x6b, 0xa7, 0x45, 0xe7,
			0x43, 0x5c, 0xb7, 0x22, 0x1e, 0x67, 0x02, 0x3c,
			0x1f, 0xaf, 0x83, 0xec, 0xb5, 0x3c, 0x35, 0x13,
			0xcb, 0x99, 0x81, 0x42, 0x3f, 0x7c, 0x58, 0x7c,
			0x96, 0xa8, 0x80, 0x3d, 0xee, 0x13, 0x2b, 0xba,
			0x33, 0x44, 0xce, 0x6b, 0x96, 0x00, 0x30, 0xc7,
			0x3b, 0x25, 0xc2, 0x4f, 0x32, 0xd6, 0x2e, 0xcd,
			0xa9, 0xd8, 0x32, 0xde, 0x60, 0x3c, 0xaf, 0xae,
		},
		/* S6 : 384 byte */
		{
			0x7a, 0x2f, 0x7a, 0x8a, 0x22, 0xda, 0xe7, 0x02,
			0x11, 0x67, 0xb4, 0x0a, 0xf3, 0x7f, 0xc2, 0x79,
			0x36, 0x02, 0xb3, 0x4f, 0x1b, 0xef, 0x03, 0x9b,
			0xff, 0x7e, 0xbf, 0xdb, 0xb8, 0xd8, 0x9c, 0x32,
			0x6b, 0xf9, 0x29, 0x57, 0xd8, 0xe4, 0x21, 0x8c,
			0xb1, 0x58, 0x39, 0x12, 0x8e, 0x6f, 0x6c, 0x71,
			0xf9, 0xcf, 0xcc, 0x52, 0x13, 0x2a, 0x32, 0x00,
			0x57, 0xb0, 0x41, 0x5f, 0x83, 0x17, 0x62, 0xaa,
			0xae, 0xf1, 0xb1, 0x91, 0x75, 0xd5, 0x18, 0x63,
			0x66, 0xdf, 0x74, 0x96, 0x3d, 0xbb, 0xc4, 0x02,
			0x02, 0x3a, 0xa4, 0x16, 0xd1, 0x7e, 0x2a, 0xa5,
			0x15, 0xe0, 0x4c, 0x8e, 0xb0, 0xb2, 0x6e, 0x8d,
			0x8c, 0x00, 0xf1, 0xa7, 0x46, 0x10, 0xfb, 0xea,
			0x45, 0x74, 0x7f, 0xef, 0xc6, 0xfa, 0x9a, 0x90,
			0x02, 0xb8, 0x20, 0x28, 0xe4, 0x44, 0xeb, 0x45,
			0x30, 0x2e, 0x2a, 0xe9, 0x3a, 0x4b, 0xd4, 0xa1,
			0x26, 0xe4, 0x13, 0xc7, 0x01, 0x22, 0x6a, 0x23,
			0x43, 0x82, 0xc9, 0x2d, 0x38, 0xd8, 0x46, 0xac,
			0xd6, 0x63, 0xb5, 0xac, 0x99, 0xf2, 0xd3, 0x37,
			0x4c, 0x1a, 0x34, 0xca, 0xfa, 0x8c, 0xf4, 0xdb,
			0x8f, 0xad, 0xf8, 0xd6, 0xac, 0x6f, 0x84, 0xc8,
			0xc1, 0xe8, 0x2a, 0x34, 0xcb, 0x94, 0x8e, 0x3c,
			0x21, 0xba, 0xc8, 0x05, 0x87, 0xff, 0xe8, 0x86,
			0x4c, 0x7b, 0xd1, 0x06, 0xa4, 0x57, 0x2f, 0x1f,
			0xeb, 0x1d, 0xbe, 0x8b, 0xf4, 0x7b, 0xea, 0x91,
			0xc5, 0xb4, 0xf8, 0xec, 0xbd, 0x5c, 0x3a, 0x03,
			0x91, 0xc3, 0x5a, 0x24, 0x29, 0xef, 0x66, 0x18,
			0x4e, 0xf7, 0x64, 0x91, 0x92, 0xb9, 0x61, 0x00,
			0x5c, 0xfe, 0x96, 0x57, 0x2a, 0xb1, 0xf1, 0x47,
			0x9f, 0xed, 0x56, 0x9b, 0x9d, 0xad, 0xa7, 0xcd,
			0xe7, 0x4a, 0x03, 0x29, 0xeb, 0x63, 0xf6, 0xe9,
			0x1e, 0x16, 0x47, 0xc4, 0x24, 0xac, 0xa2, 0xa4,
			0xbe, 0x2c, 0x10, 0x6e, 0xb9, 0xe6, 0x7b, 0x81,
			0x92, 0x02, 0x1f, 0xa2, 0x3f, 0x3e, 0xb1, 0x75,
			0x95, 0xd5, 0xb0, 0xe6, 0xbd, 0x3c, 0xe5, 0xd9,
			0xc9, 0x93, 0x91, 0x92, 0xb7, 0x86, 0xac, 0xab,
			0x77, 0x8a, 0x45, 0x6b, 0x4a, 0xce, 0x7b, 0xc8,
			0xaa, 0x96, 0xf8, 0x0c, 0x1f, 0x40, 0x76, 0x07,
			0xd3, 0xfe, 0x01, 0x30, 0xd8, 0x21, 0x93, 0xa4,
			0xf6, 0x30, 0x80, 0x80, 0x0d, 0xe1, 0x0d, 0xdf,
			0xf6, 0x92, 0xe3, 0x85, 0x48, 0x9e, 0x67, 0xdb,
			0xa5, 0xfb, 0x33, 0xa8, 0x77, 0x86, 0x0a, 0xa1,
			0xf3, 0x00, 0x37, 0x62, 0x20, 0xc5, 0x71, 0x17,
			0xa3, 0xf4, 0x11, 0xdd, 0x18, 0xec, 0x60, 0x3c,
			0xd2, 0xb6, 0x6a, 0x56, 0xe5, 0x7f, 0xf4, 0xb9,
			0xc4, 0x43, 0x12, 0x67, 0x64, 0x3f, 0x3f, 0x99,
			0xae, 0x6b, 0x76, 0x9d, 0x86, 0x09, 0x99, 0xba,
			0xe7, 0xb7, 0x43, 0x03, 0x58, 0x3e, 0x8b, 0x1c,
		},
	},
	/* SHA512256 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0xd7, 0xf9, 0x5e, 0x00, 0x1d, 0x6a, 0x82, 0x6e,
			0x7f, 0x7c, 0xe4, 0xc0, 0x5f, 0x99, 0x95, 0xd3,
			0xf6, 0xa7, 0x37, 0xd0, 0x99, 0x37, 0x62, 0x00,
			0x3a, 0xff, 0x46, 0xe1, 0x31, 0x8a, 0x91, 0x79,
			0x3d, 0x2e, 0x93, 0xea, 0xc5, 0x3f, 0x9f, 0x47,
			0x64, 0x82, 0xb5, 0xa6, 0xa5, 0xe4, 0x5f, 0x76,
			0x0b, 0x6c, 0xd9, 0x13, 0x55, 0x6f, 0x74, 0x98,
			0xff, 0x03, 0x3c, 0xf5, 0x0c, 0xb5, 0xd9, 0x41,
			0x03, 0x7f, 0xb3, 0x51, 0x38, 0xf4, 0x5a, 0x89,
			0x4c, 0xbe, 0x24, 0xf2, 0xf7, 0x4a, 0x18, 0x8d,
			0x05, 0xc2, 0x0a, 0xe7, 0x9f, 0x3c, 0x08, 0xee,
			0xcc, 0x31, 0xf5, 0x03, 0x38, 0x30, 0x74, 0x50,
			0x33, 0xd3, 0x08, 0x5c, 0xfc, 0xf3, 0x79, 0xdc,
			0x40, 0x18, 0x79, 0xcd, 0xde, 0x33, 0x87, 0xbf,
			0xed, 0x90, 0x14, 0x74, 0x00, 0x06, 0xe4, 0xa9,
			0xa8, 0x71, 0xb8, 0x34, 0x3b, 0x62, 0x2a, 0x4f,
		},
		/* S7 : 384 byte */
		{
			0x72, 0x36, 0x4e, 0x03, 0x1c, 0xd5, 0x7c, 0xeb,
			0x29, 0xa4, 0x7b, 0xb0, 0x68, 0x5b, 0x12, 0x13,
			0xd7, 0xe0, 0xe9, 0x71, 0x72, 0x86, 0x73, 0xfd,
			0x19, 0xc0, 0x26, 0x58, 0x79, 0xc5, 0x89, 0x69,
			0x58, 0xe3, 0xd7, 0x8a, 0xc9, 0x0a, 0x39, 0x8f,
			0x63, 0x6c, 0x09, 0x3f, 0x97, 0x9b, 0xee, 0x99,
			0xf5, 0xff, 0x70, 0x53, 0x98, 0x30, 0xe4, 0xad,
			0x44, 0xe9, 0x88, 0xb6, 0x3b, 0xce, 0xd4, 0x27,
			0x72, 0x8a, 0xd4, 0x3f, 0x8d, 0xf0, 0x64, 0xc8,
			0x2e, 0x9f, 0xb6, 0x82, 0x18, 0x5f, 0x2a, 0xbc,
			0xde, 0x5c, 0x52, 0xb2, 0x9a, 0x59, 0x18, 0x17,
			0x3c, 0x13, 0xe9, 0x89, 0x0e, 0xff, 0x41, 0xf7,
			0xa1, 0x5b, 0xd9, 0xef, 0xd4, 0xb6, 0x5b, 0x2f,
			0xf5, 0xa2, 0xf7, 0x06, 0x82, 0x1e, 0x38, 0xf6,
			0x27, 0x80, 0x51, 0x42, 0xe6, 0xf5, 0xce, 0x52,
			0x76, 0x41, 0xae, 0x7b, 0x46, 0xc1, 0xe9, 0x40,
			0x5e, 0x6f, 0xaa, 0x78, 0x48, 0xac, 0x7d, 0xa9,
			0x8d, 0x00, 0xab, 0x21, 0x96, 0x07, 0x4b, 0x1c,
			0x09, 0x20, 0x18, 0x33, 0x7a, 0x0a, 0x25, 0x47,
			0x31, 0x39, 0xff, 0x0c, 0xb7, 0x1e, 0x68, 0x80,
			0xe7, 0xaa, 0x6e, 0xa2, 0xea, 0x0d, 0x6b, 0x25,
			0x96, 0x6d, 0xe4, 0x85, 0x89, 0xdf, 0xde, 0x90,
			0xe4, 0x9a, 0x74, 0x11, 0x34, 0xe0, 0x48, 0x94,
			0x06, 0x83, 0x9a, 0xbf, 0x97, 0xa2, 0xe3, 0x78,
			0xde, 0xf5, 0x9d, 0xf5, 0x38, 0x62, 0x87, 0x83,
			0x05, 0x69, 0x90, 0x63, 0x05, 0x20, 0x01, 0xa4,
			0x29, 0x7e, 0xe1, 0xdc, 0x99, 0x38, 0xc4, 0xdb,
			0x68, 0x6c, 0x9b, 0xed, 0xab, 0x9a, 0x8d, 0x95,
			0xee, 0x84, 0xe0, 0x16, 0x36, 0x5c, 0x81, 0x2c,
			0x40, 0x7d, 0x91, 0x5d, 0xd1, 0xea, 0xf3, 0x8f,
			0xae, 0x32, 0xfa, 0x40, 0xce, 0x0b, 0xfe, 0xf6,
			0xc0, 0x92, 0xf3, 0x0c, 0x4b, 0x9e, 0x9e, 0xdd,
			0x77, 0x41, 0xe3, 0xa4, 0xa1, 0xf2, 0xc7, 0x9b,
			0x52, 0xd3, 0xe7, 0x5c, 0x23, 0xbc, 0xb6, 0x41,
			0xff, 0x1e, 0x81, 0xd8, 0x63, 0xa9, 0x41, 0x90,
			0x2a, 0xeb, 0x87, 0xb7, 0xd9, 0xe8, 0xd1, 0x75,
			0x76, 0x62, 0x0a, 0x35, 0xd7, 0x5e, 0x23, 0xe5,
			0x6e, 0x3c, 0x2e, 0x23, 0x14, 0x63, 0x24, 0xac,
			0x31, 0x74, 0x0c, 0x1b, 0xbd, 0x30, 0xaa, 0xf4,
			0x27, 0xef, 0xb1, 0xa3, 0xb8, 0x50, 0x58, 0x9c,
			0xae, 0x30, 0x5a, 0x9e, 0x51, 0x6e, 0xe2, 0x1f,
			0x00, 0x45, 0xc2, 0x13, 0x66, 0x13, 0x94, 0x67,
			0x4d, 0x58, 0x9b, 0x2e, 0x66, 0x53, 0x49, 0x92,
			0xd9, 0xbd, 0x09, 0x38, 0xce, 0xc8, 0x28, 0x91,
			0x67, 0xc8, 0x24, 0xa4, 0x9c, 0xb0, 0x95, 0xd7,
			0xde, 0x57, 0xee, 0x8e, 0x5e, 0xc7, 0xbb, 0xf1,
			0x07, 0x88, 0x87, 0x59, 0xd4, 0xd1, 0x3d, 0x81,
			0x24, 0x30, 0x18, 0xc6, 0x2d, 0x60, 0xef, 0xa8,
		},
	},
	/* SHA512256 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0x00, 0x74, 0xba, 0x72, 0x61, 0x44, 0xb9, 0x6d,
			0xc9, 0x3c, 0x7b, 0x68, 0x57, 0x57, 0x23, 0xc9,
			0x68, 0xc6, 0x29, 0xf8, 0x53, 0xdb, 0xae, 0x66,
			0x57, 0x25, 0x0c, 0xcf, 0xd1, 0xdd, 0xb2, 0xbc,
			0xb2, 0x80, 0x2f, 0x4b, 0x31, 0xbe, 0x8c, 0x32,
			0xbd, 0x93, 0xdf, 0x02, 0x36, 0xb4, 0x1d, 0xa9,
			0x5c, 0x2c, 0x30, 0x3b, 0x88, 0x47, 0x49, 0xf2,
			0x9a, 0xaf, 0x78, 0x73, 0x32, 0xd5, 0x7a, 0x52,
			0x23, 0x92, 0xd2, 0xd9, 0x24, 0x0f, 0x1b, 0x05,
			0xc9, 0xe2, 0x9c, 0xd2, 0xb2, 0x79, 0x1d, 0xb8,
			0x03, 0xbc, 0xf9, 0x76, 0x13, 0xee, 0x75, 0x0e,
			0x69, 0xd4, 0x50, 0x11, 0xe4, 0x1a, 0x9b, 0x1c,
			0x39, 0xf2, 0xcb, 0xa3, 0xa3, 0xfe, 0x0d, 0xed,
			0xd5, 0x70, 0xd9, 0x60, 0x34, 0x70, 0xca, 0xff,
			0x43, 0xbe, 0xad, 0x31, 0x5e, 0x98, 0xd8, 0xc4,
			0xac, 0xde, 0xbd, 0x31, 0x4f, 0x74, 0xbe, 0x22,
		},
		/* S8 : 384 byte */
		{
			0x99, 0xeb, 0x49, 0x19, 0x65, 0xdc, 0x7d, 0xac,
			0x37, 0x93, 0xcf, 0xf5, 0x73, 0xbf, 0x11, 0x78,
			0xa0, 0x31, 0x34, 0xad, 0x99, 0x77, 0xdb, 0x88,
			0x3c, 0x4b, 0x41, 0x88, 0xfa, 0x57, 0x4e, 0x7e,
			0xa8, 0x92, 0x96, 0x9c, 0x6a, 0x0e, 0x08, 0x4b,
			0xf0, 0x3c, 0xac, 0xb4, 0xd2, 0xa7, 0x56, 0xe5,
			0xad, 0xc8, 0x2a, 0x8b, 0x8d, 0xb7, 0xaf, 0x9f,
			0x3c, 0xee, 0x6f, 0x77, 0xf1, 0xab, 0x23, 0xf0,
			0xa5, 0xdc, 0x40, 0xc1, 0x45, 0x0d, 0x2d, 0x23,
			0x53, 0x9c, 0xa2, 0x93, 0x36, 0x55, 0x8a, 0x52,
			0x01, 0xb6, 0x74, 0x32, 0xa2, 0x2e, 0x88, 0x1f,
			0xa3, 0xc3, 0x04, 0xee, 0x5e, 0x79, 0x6d, 0xbd,
			0xec, 0x9d, 0xd5, 0xe9, 0xac, 0xa6, 0x3e, 0x99,
			0xc4, 0xb2, 0xe5, 0x2b, 0xaf, 0x29, 0x2b, 0xd2,
			0x78, 0xb4, 0xc1, 0x31, 0xef, 0xbd, 0x54, 0xee,
			0x95, 0x3b, 0x7e, 0x84, 0x9e, 0xb8, 0x35, 0xcd,
			0x17, 0xb9, 0xf0, 0xdf, 0x4a, 0x1b, 0x86, 0x9c,
			0x03, 0x13, 0xdf, 0x23, 0x55, 0x1a, 0xa6, 0x0f,
			0x57, 0x05, 0x15, 0xc6, 0xc1, 0x86, 0xae, 0xda,
			0x72, 0x13, 0x5a, 0xa8, 0xc1, 0xff, 0x5a, 0x42,
			0xe8, 0xbf, 0x31, 0x2f, 0x14, 0xb0, 0xd9, 0xf4,
			0xf0, 0x42, 0xd5, 0xe6, 0x9c, 0x44, 0x67, 0xe9,
			0xba, 0x87, 0x15, 0xf0, 0xd2, 0x06, 0xea, 0xb2,
			0x91, 0xa2, 0x88, 0x98, 0x11, 0x09, 0xbc, 0x3c,
			0xb7, 0x51, 0x2f, 0x19, 0x98, 0xf5, 0xd2, 0xdd,
			0xfc, 0x99, 0x04, 0x0b, 0x10, 0x8f, 0x15, 0xc7,
			0xd4, 0xf7, 0x06, 0x20, 0xfc, 0x48, 0x82, 0xf8,
			0xb1, 0x76, 0x54, 0xdc, 0x70, 0xcf, 0xf9, 0x3c,
			0x5d, 0x4f, 0xf8, 0x34, 0x80, 0xe3, 0x05, 0x68,
			0x3c, 0x73, 0xb0, 0x38, 0xe1, 0x0b, 0xf5, 0xf0,
			0xc4, 0xea, 0x56, 0xf0, 0x3e, 0xdf, 0x38, 0xd5,
			0xe8, 0x4e, 0x0a, 0xe6, 0xc9, 0x36, 0x48, 0x46,
			0xac, 0xba, 0xea, 0xa2, 0x72, 0x93, 0x14, 0x06,
			0x46, 0x66, 0xf1, 0xd9, 0x3f, 0x63, 0x51, 0xe6,
			0x10, 0xed, 0xbe, 0x30, 0xe2, 0x95, 0x27, 0x96,
			0xdc, 0x01, 0xed, 0x8e, 0x77, 0x27, 0xda, 0xd3,
			0x2e, 0xcf, 0xb4, 0x0d, 0x78, 0xca, 0x23, 0xe5,
			0x26, 0xbe, 0x97, 0xff, 0x07, 0x0c, 0x91, 0x9f,
			0x61, 0xa2, 0xb9, 0xdc, 0xfe, 0x10, 0xd8, 0xe3,
			0x54, 0x42, 0xee, 0x76, 0xcc, 0x1b, 0x3c, 0x22,
			0xd6, 0x0a, 0xf7, 0xb7, 0xc2, 0x3d, 0xc2, 0xfe,
			0xa3, 0xb0, 0xa8, 0x9d, 0x1e, 0xde, 0xd7, 0xc7,
			0xfc, 0xf8, 0x98, 0x26, 0x01, 0x1b, 0xca, 0xb7,
			0x43, 0x0c, 0x31, 0xac, 0x86, 0xf7, 0xb0, 0xb9,
			0xc8, 0x57, 0x06, 0x6a, 0x6b, 0x7d, 0x1f, 0x4c,
			0xbf, 0xcc, 0x64, 0xb4, 0xa1, 0x4f, 0x71, 0x56,
			0xd3, 0xd6, 0x9d, 0xcd, 0x3b, 0x9f, 0xa6, 0xe2,
			0xa4, 0x68, 0xfe, 0x13, 0x86, 0x16, 0x24, 0xa3,
		},
	},
	/* SHA512256 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0xb6, 0xb2, 0x6a, 0xae, 0x4d, 0x1f, 0xcb, 0xd4,
			0x3d, 0x62, 0x05, 0x59, 0x57, 0x58, 0x93, 0x99,
			0x17, 0xdf, 0xef, 0xfa, 0xd6, 0x37, 0x60, 0x74,
			0x41, 0xaf, 0x13, 0x50, 0x0a, 0x7d, 0x6e, 0x2f,
			0xde, 0xed, 0xa1, 0x4f, 0x96, 0xbe, 0x7e, 0x0a,
			0xb1, 0xc8, 0x8a, 0x9f, 0x02, 0xe3, 0x54, 0x35,
			0xa6, 0x05, 0x05, 0x1d, 0x39, 0xaa, 0x39, 0x58,
			0x49, 0x79, 0x0d, 0x59, 0x0b, 0x24, 0x79, 0x0b,
			0x4d, 0x90, 0xe1, 0x16, 0xac, 0x36, 0xe6, 0x5f,
			0x07, 0xc9, 0x82, 0xcd, 0x34, 0x18, 0x54, 0x53,
			0xf1, 0x37, 0x38, 0x2b, 0x1d, 0xd9, 0x1b, 0x5f,
			0x28, 0x01, 0x0e, 0x8e, 0xc8, 0xda, 0x98, 0xed,
			0xed, 0xe2, 0x51, 0x8c, 0xf8, 0x4e, 0xac, 0xdb,
			0x27, 0xab, 0x15, 0x6c, 0x54, 0xc1, 0x3c, 0xfa,
			0xe0, 0xbc, 0xb0, 0xab, 0xea, 0x8e, 0xaf, 0x92,
			0xc4, 0x8f, 0x3d, 0x78, 0xa7, 0x66, 0x73, 0xdc,
		},
		/* S9 : 384 byte */
		{
			0x46, 0x08, 0x76, 0xef, 0xb3, 0xad, 0xe4, 0xc1,
			0x19, 0xb9, 0xa1, 0xac, 0x22, 0xc7, 0xf3, 0xfa,
			0x72, 0x78, 0xaf, 0x8f, 0xb0, 0x65, 0x62, 0x92,
			0xd0, 0x34, 0x53, 0xc6, 0x83, 0x52, 0xc5, 0xa4,
			0x62, 0xbc, 0x72, 0x0a, 0xf4, 0x60, 0xad, 0xd4,
			0x0f, 0xe2, 0x80, 0x78, 0xd9, 0xdd, 0xa4, 0x69,
			0x22, 0xdd, 0xb9, 0xb7, 0x8c, 0x85, 0x8f, 0xef,
			0xf5, 0xc2, 0x43, 0x36, 0x88, 0x90, 0x2b, 0xdd,
			0x34, 0x47, 0x85, 0x0d, 0xc8, 0x8f, 0x9d, 0x8c,
			0x1b, 0x00, 0x38, 0x91, 0xbd, 0xa0, 0x81, 0x11,
			0x68, 0x00, 0xe2, 0xbb, 0x32, 0xf9, 0x79, 0x54,
			0x27, 0xdb, 0x84, 0xed, 0x10, 0x84, 0x34, 0x22,
			0x83, 0xf5, 0x90, 0x3e, 0xa9, 0xb8, 0x98, 0xbc,
			0x8e, 0xe0, 0x83, 0xd0, 0x69, 0xda, 0x45, 0x73,
			0x33, 0xc2, 0x6f, 0x4c, 0x4d, 0xc7, 0x56, 0x82,
			0xdd, 0x0b, 0xe7, 0x3c, 0x68, 0x5d, 0x9a, 0xcd,
			0xab, 0xe5, 0xa5, 0x9d, 0xe8, 0xd1, 0x2c, 0x4d,
			0x8e, 0x7d, 0x62, 0xcf, 0xd4, 0x29, 0x92, 0x38,
			0xf8, 0x49, 0x32, 0x5e, 0x1a, 0xbe, 0xaa, 0x77,
			0x63, 0xe3, 0x92, 0xa5, 0x01, 0x90, 0x59, 0xe3,
			0x3f, 0x31, 0xb0, 0xa6, 0xa4, 0xca, 0xd3, 0x6b,
			0x3b, 0x58, 0x01, 0xc2, 0x33, 0x58, 0xe7, 0xb9,
			0xd0, 0x55, 0xcc, 0x4d, 0xd0, 0xc9, 0xb1, 0x99,
			0x11, 0x69, 0xed, 0xf6, 0x14, 0x80, 0xe0, 0x5f,
			0x74, 0xf4, 0x38, 0x81, 0x45, 0xaf, 0xc4, 0x47,
			0xf1, 0x70, 0x41, 0x14, 0x81, 0x82, 0x05, 0x09,
			0x6c, 0x11, 0x7d, 0x2a, 0xe8, 0xee, 0x74, 0x7e,
			0xfe, 0x03, 0xb6, 0x21, 0x3a, 0x6f, 0x8c, 0xfb,
			0x6a, 0xab, 0xed, 0x9f, 0x85, 0x90, 0x2f, 0xb9,
			0x68, 0x2c, 0x17, 0xbc, 0x92, 0xdf, 0x25, 0x0b,
			0xbf, 0xcf, 0x57, 0xa9, 0x4f, 0xd9, 0xce, 0xa7,
			0x4c, 0x90, 0xf1, 0xbb, 0x92, 0xa7, 0x7d, 0xba,
			0x5f, 0x31, 0x8c, 0xdb, 0x09, 0x50, 0xfb, 0xce,
			0x14, 0xf4, 0xd8, 0x2c, 0xa0, 0x97, 0x04, 0x22,
			0x2d, 0x01, 0xc8, 0xd6, 0xf0, 0x8c, 0x31, 0xb1,
			0x1e, 0x11, 0xd5, 0x5c, 0xb9, 0x98, 0x56, 0xcb,
			0xe6, 0xf1, 0x99, 0xda, 0x4a, 0x41, 0xa3, 0x4c,
			0xe8, 0x78, 0xa3, 0xb3, 0x64, 0x1a, 0x1f, 0x4b,
			0x69, 0x21, 0x97, 0x70, 0x25, 0xf4, 0xd2, 0x29,
			0x78, 0x23, 0xf7, 0x73, 0xda, 0xbb, 0xb8, 0xb5,
			0x87, 0xa2, 0x80, 0xbf, 0x87, 0xf3, 0x04, 0x5a,
			0x66, 0x09, 0x09, 0x0b, 0x92, 0x5c, 0xea, 0x76,
			0x76, 0xa4, 0xf1, 0xc6, 0x3d, 0x4f, 0xd8, 0xef,
			0xaa, 0x6e, 0x8c, 0x3b, 0xe7, 0xdd, 0xe0, 0x35,
			0xd8, 0xe9, 0x43, 0x34, 0x99, 0xcd, 0x19, 0x35,
			0x42, 0x3d, 0xb1, 0x8e, 0x92, 0xad, 0xd2, 0xd7,
			0xa5, 0x95, 0xee, 0x6f, 0x80, 0xc1, 0xa2, 0x33,
			0xd1, 0x0d, 0xe1, 0xa3, 0xfe, 0xba, 0xeb, 0xac,
		},
	},
};

typedef mod3072_t tv_t;


#include "siggen15test.c"
