/* siggen15_3_3072_testfc.c */
/*
 * Copyright (c) 2017 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

/*
 * This test vector was created based on SigGen15_186-3.txt from
 * https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Algorithm-Validation-Program/documents/dss/186-3rsatestvectors.zip
 *
 * The link to the archive file exists in the following URL:
 * https://csrc.nist.gov/projects/cryptographic-algorithm-validation-program/digital-signatures/
 */
/*
 * https://csrc.nist.gov/projects/cryptographic-algorithm-validation-program/digital-signatures/
 * https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Algorithm-Validation-Program/documents/dss/186-3rsatestvectors.zip
 * SigGen15_186-3.txt
 */ 
#include "siggen15test.h"

static const unsigned short sn1[] = {
	0xdca9, 0x8304, 0xb729, 0xe819, 0xb340, 0xe26c, 0xecb7, 0x30ae,
	0xcbd8, 0x930e, 0x334c, 0x7314, 0x93b1, 0x80de, 0x970e, 0x6d3b,
	0xc579, 0xf86c, 0x8d5d, 0x032f, 0x8cd3, 0x3c43, 0x97ee, 0x7ffd,
	0x019d, 0x51b0, 0xa7db, 0xe4f5, 0x2505, 0xa1a3, 0x4ae3, 0x5d23,
	0xcfaa, 0xf594, 0x419d, 0x509f, 0x469b, 0x1369, 0x589f, 0x9c86,
	0x16a7, 0xd698, 0x513b, 0xc1d4, 0x23d7, 0x0070, 0xd3d7, 0x2b99,
	0x6c23, 0xabe6, 0x8b22, 0xccc3, 0x9aab, 0xd165, 0x0712, 0x4042,
	0xc88d, 0x4da6, 0xa745, 0x1288, 0xec87, 0xc924, 0x4be2, 0x26aa,
	0xc02d, 0x1817, 0x682f, 0x80cc, 0x34c6, 0xeaf3, 0x7ec8, 0x4d24,
	0x7aae, 0xdebb, 0x56c3, 0xbbca, 0xffb5, 0xcf42, 0xf61f, 0xe1b7,
	0xf3fc, 0x8974, 0x8e21, 0x3973, 0xbf5f, 0x679d, 0x8b8b, 0x42a4,
	0x7ac4, 0xafd9, 0xe51e, 0x1d12, 0x14df, 0xe1a7, 0xe116, 0x9080,
	0xbd9a, 0xd917, 0x58f6, 0xc0f9, 0xb22a, 0xe40a, 0xf6b4, 0x1403,
	0xd8f2, 0xd96d, 0xb5a0, 0x88da, 0xa5ef, 0x8683, 0xf86f, 0x501f,
	0x7ad3, 0xf358, 0xb633, 0x7da5, 0x5c6c, 0xfc00, 0x3197, 0x420c,
	0x1c75, 0xabdb, 0x7be1, 0x403e, 0xa4f3, 0xe642, 0x59f5, 0xc6da,
	0x3325, 0xbb87, 0xd605, 0xb6e1, 0x4b53, 0x50e6, 0xe145, 0x5c9d,
	0x497d, 0x8104, 0x6608, 0xe387, 0x95dc, 0x85ab, 0xa406, 0xc9de,
	0x1f4f, 0x9990, 0xd515, 0x3b98, 0xbbab, 0xbdcb, 0xd6bb, 0x1885,
	0x4312, 0xb2da, 0x48b4, 0x11e8, 0x38f2, 0x6ae3, 0x109f, 0x104d,
	0xfd16, 0x19f9, 0x9182, 0x4ec8, 0x1986, 0x1e51, 0x99f2, 0x6bb9,
	0xb3b2, 0x99bf, 0xa9ec, 0x2fd6, 0x9127, 0x1b58, 0xa8ad, 0xecbf,
	0x0ff6, 0x27b5, 0x4336, 0xf3df, 0x7003, 0xd70e, 0x37d1, 0x1ddb,
	0xd930, 0xd9ab, 0xa7e8, 0x8ed4, 0x01ac, 0xb440, 0x92fd, 0x53d5,
};

static const unsigned short se1[] = {
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x00ea, 0xf05d,
};

static const unsigned short sd1[] = {
	0x2d6d, 0xb91e, 0xb32e, 0x36e5, 0xd512, 0x7deb, 0x034d, 0x1407,
	0x2fe6, 0x0c1c, 0xd13c, 0x8c3d, 0xd9ad, 0xbc87, 0x140b, 0x5e71,
	0x36f4, 0xf89e, 0x61bb, 0xee78, 0x26f4, 0x5ac1, 0xd991, 0x94fb,
	0xaa8c, 0x5a0b, 0xb94d, 0xb31d, 0x9372, 0x3b51, 0x419d, 0x9c6f,
	0x6eeb, 0x5f36, 0x10b6, 0x7f4b, 0x4e2a, 0xde05, 0xcc6b, 0x8990,
	0xe883, 0x2cf4, 0xcd40, 0xf2df, 0x0388, 0xc9a5, 0x2072, 0xe27e,
	0xfeba, 0xe20b, 0x4ad5, 0x951f, 0x4d20, 0xdd18, 0x943e, 0x58b7,
	0x86d8, 0x7976, 0x52b2, 0xbb75, 0x9c31, 0x9d2b, 0x0046, 0xdbf6,
	0x9c53, 0xc075, 0xd00c, 0x287b, 0x8760, 0x42fa, 0xfa23, 0xfe4d,
	0xd705, 0xe4e4, 0x2327, 0x7c90, 0x0031, 0x1e94, 0xea3f, 0x7456,
	0xe32f, 0xd12a, 0xfe4a, 0x2bde, 0x358a, 0x6582, 0x4f10, 0x5506,
	0x4823, 0xc893, 0xfc93, 0xbe3b, 0x8c65, 0x8bb4, 0x41d7, 0xf0b0,
	0x0ac2, 0x46bf, 0x043a, 0x9c00, 0x53d3, 0x19f0, 0x03ef, 0x5a55,
	0x33f7, 0x4d63, 0x0d8c, 0xe93b, 0xab41, 0x6a82, 0x951e, 0x05b8,
	0x2c60, 0x3659, 0x3eca, 0x89f0, 0xebac, 0xd7d5, 0x1ed9, 0x610a,
	0xf435, 0x37fc, 0xd266, 0xe5e4, 0x7c0d, 0x25fe, 0xdad6, 0xd047,
	0xa1a1, 0xee3e, 0xb444, 0x367e, 0x3eff, 0x7c75, 0x20ca, 0x4f77,
	0x9f20, 0x27fe, 0x4503, 0x6204, 0x1684, 0x54df, 0x4918, 0xb547,
	0xa4d1, 0x9e93, 0x8f3c, 0x6db6, 0xca27, 0x02ad, 0x9bbd, 0xa126,
	0x1c64, 0xd00b, 0x5782, 0x85bd, 0xcfc9, 0x851f, 0x96a4, 0xf2cd,
	0x14d6, 0x6b9c, 0x1f65, 0x742a, 0x1344, 0x948c, 0x9f1d, 0xa8d3,
	0x38ed, 0x4e3d, 0xeb1e, 0xbadf, 0x11f8, 0xc281, 0x944e, 0x8849,
	0x8234, 0x96f8, 0x6111, 0xf378, 0xbdd0, 0x84c9, 0x9f65, 0xfb9b,
	0x4ee6, 0x271b, 0x1d1b, 0xe424, 0xc294, 0xd185, 0xd9fd, 0x9cdf,
};

#define HAS_SHA224_TESTVECTOR

#define HAS_SHA256_TESTVECTOR

#define HAS_SHA384_TESTVECTOR

#define HAS_SHA512_TESTVECTOR

/* 10 elements */
static mod3072_t SHA224[] = {
	/* SHA224 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0x25, 0x4c, 0xe3, 0x6e, 0x8e, 0xd6, 0x2e, 0x08,
			0x87, 0xd4, 0xbe, 0x00, 0xee, 0xfa, 0x82, 0x51,
			0x5a, 0xce, 0xf9, 0x56, 0x54, 0x0c, 0xff, 0x45,
			0xc4, 0x48, 0xe7, 0xf9, 0xa9, 0xd5, 0xc9, 0xf4,
			0x0d, 0xe6, 0x1d, 0xa4, 0x39, 0xf3, 0x89, 0xe5,
			0x25, 0x5e, 0xf8, 0xc8, 0x32, 0x57, 0xec, 0x92,
			0x1b, 0xfd, 0x15, 0x08, 0x29, 0xc5, 0x22, 0xea,
			0xa7, 0x20, 0xd7, 0xbe, 0x96, 0x58, 0x60, 0xce,
			0xa2, 0xbb, 0xe5, 0x74, 0x54, 0xfc, 0x5e, 0x95,
			0x88, 0xd6, 0xa9, 0x6c, 0x22, 0xf2, 0xd9, 0x89,
			0xfd, 0x0b, 0xd2, 0x19, 0x24, 0x50, 0x13, 0x67,
			0x45, 0x0a, 0xd2, 0xa3, 0x62, 0x7e, 0x4e, 0xe3,
			0xca, 0x15, 0x61, 0x67, 0x48, 0xba, 0x54, 0x21,
			0x9a, 0x84, 0xf8, 0x74, 0x24, 0x95, 0xf2, 0x3d,
			0xe6, 0x42, 0x57, 0x10, 0xac, 0x74, 0x79, 0xc4,
			0x84, 0x4d, 0x00, 0x31, 0x75, 0x0f, 0x3c, 0x38,
		},
		/* S0 : 384 byte */
		{
			0x9d, 0xfd, 0x3f, 0x32, 0x09, 0x1b, 0x91, 0x6a,
			0x56, 0xf4, 0xf3, 0x57, 0xa9, 0x61, 0xa5, 0x25,
			0xa5, 0x27, 0xfd, 0x29, 0xb1, 0x14, 0xb3, 0xf0,
			0x38, 0x29, 0xdf, 0x1b, 0x25, 0xc0, 0xc5, 0x97,
			0x3b, 0x1c, 0x51, 0xaf, 0x36, 0x63, 0x31, 0x16,
			0xf4, 0xc7, 0x7a, 0xa2, 0xf6, 0x77, 0xa3, 0xb0,
			0xf8, 0x23, 0x68, 0xa5, 0x38, 0xbd, 0xb3, 0x3e,
			0x49, 0xdb, 0x9f, 0xa7, 0x04, 0xbd, 0x51, 0x23,
			0xed, 0xef, 0xbd, 0x3a, 0x86, 0xdc, 0xc3, 0x92,
			0x83, 0xc2, 0xa0, 0x3c, 0x96, 0xc6, 0x90, 0x30,
			0xf0, 0x4c, 0x64, 0x84, 0x17, 0xf5, 0x44, 0xf0,
			0x8a, 0x9b, 0x4e, 0x01, 0xae, 0x4d, 0x42, 0x9e,
			0xf2, 0x14, 0x22, 0xdd, 0xfe, 0x76, 0x83, 0x4f,
			0x92, 0x5c, 0x56, 0x53, 0xb1, 0x22, 0x78, 0x35,
			0xa9, 0xa4, 0x41, 0x3d, 0xa7, 0x94, 0x2b, 0x0a,
			0x01, 0x51, 0x96, 0xfa, 0xec, 0x31, 0x50, 0x41,
			0x11, 0xc9, 0xf0, 0x84, 0xd6, 0xdd, 0x6d, 0x5a,
			0x69, 0x56, 0xc5, 0x54, 0x12, 0xc1, 0xbd, 0x14,
			0xaa, 0xf9, 0x5d, 0x82, 0x8e, 0x84, 0x49, 0x61,
			0xfd, 0xd6, 0x0c, 0xc0, 0x78, 0xf1, 0x69, 0xf6,
			0xe1, 0x18, 0x6c, 0xb0, 0xcb, 0x6b, 0xa3, 0xd2,
			0x11, 0x68, 0xc5, 0xbf, 0xd0, 0x67, 0xdc, 0x6e,
			0x46, 0x07, 0x84, 0xe7, 0xc6, 0xa7, 0x6e, 0xe3,
			0xd8, 0xb3, 0x32, 0xac, 0xfa, 0x97, 0xe5, 0xd4,
			0xb6, 0x56, 0xec, 0x8c, 0x61, 0x1e, 0xbd, 0x89,
			0x6f, 0xe9, 0x0e, 0x61, 0x9b, 0x58, 0x8d, 0x0c,
			0x61, 0x54, 0x92, 0x46, 0x4a, 0x1b, 0x3d, 0x05,
			0xd3, 0xa9, 0x63, 0xf4, 0x51, 0x05, 0x1c, 0x65,
			0xd8, 0xf8, 0x1f, 0xee, 0xa9, 0x25, 0xbc, 0xbe,
			0xe9, 0xce, 0x7a, 0x39, 0xba, 0x3c, 0x91, 0x5a,
			0x18, 0xa2, 0x4a, 0x45, 0x1e, 0x47, 0x0e, 0x76,
			0x1d, 0x09, 0x85, 0x5a, 0x96, 0x5e, 0x83, 0xed,
			0xae, 0x3f, 0xca, 0x41, 0x67, 0x8c, 0xc9, 0xd0,
			0x98, 0xba, 0x99, 0x28, 0xb5, 0x25, 0xb5, 0x0e,
			0x48, 0xcb, 0x03, 0x0c, 0x51, 0x0c, 0x4c, 0xe7,
			0x27, 0xc6, 0xb9, 0x3b, 0xd0, 0x91, 0xb7, 0xd2,
			0x0b, 0x4b, 0x96, 0x11, 0x65, 0xae, 0x0e, 0x28,
			0x48, 0xaa, 0x99, 0x5b, 0xb7, 0x3a, 0xbe, 0x9a,
			0x26, 0x34, 0x37, 0x8d, 0x22, 0x41, 0x28, 0x54,
			0x1a, 0xb0, 0x56, 0xa3, 0x1b, 0x78, 0x48, 0x85,
			0xae, 0xf8, 0x03, 0x4d, 0xed, 0xac, 0x13, 0x16,
			0x74, 0x02, 0xf9, 0xf6, 0x2b, 0x55, 0x74, 0x12,
			0x20, 0xdf, 0x8a, 0xff, 0x5d, 0xef, 0xb6, 0x9c,
			0x03, 0x5d, 0x9a, 0x31, 0xe2, 0xa5, 0xb8, 0x81,
			0x70, 0x57, 0x24, 0x1b, 0xcf, 0x85, 0x49, 0x32,
			0xf5, 0xed, 0xee, 0x7e, 0xe6, 0x6e, 0x89, 0x17,
			0xaa, 0x4a, 0x71, 0x8b, 0x6c, 0x44, 0x6b, 0xdd,
			0xf0, 0x84, 0xf5, 0xcd, 0x76, 0x9c, 0xae, 0xff,
		},
	},
	/* SHA224 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0x35, 0xad, 0xcd, 0x3f, 0x24, 0xb6, 0x72, 0x55,
			0x18, 0x81, 0x5c, 0xf4, 0x60, 0x6f, 0x7b, 0x1d,
			0x94, 0x0c, 0x39, 0x63, 0x84, 0x37, 0x0a, 0x37,
			0x6e, 0x84, 0x45, 0x69, 0x74, 0xde, 0x32, 0xec,
			0x4c, 0x71, 0x64, 0xda, 0x3a, 0xc7, 0x49, 0xb7,
			0x3b, 0x30, 0xff, 0xfa, 0x83, 0x68, 0x69, 0xc9,
			0x2a, 0x74, 0x83, 0x05, 0x23, 0xcd, 0xf2, 0x86,
			0x6d, 0xc0, 0xe0, 0xa8, 0x8d, 0x15, 0x06, 0x06,
			0x3b, 0xef, 0x0a, 0x85, 0x5c, 0xf3, 0x0c, 0x95,
			0x30, 0xac, 0x7c, 0xb3, 0xcd, 0x2e, 0x2e, 0x32,
			0xcc, 0xfa, 0xb0, 0x3c, 0x42, 0x22, 0xdb, 0x14,
			0xf2, 0xae, 0xa8, 0x9d, 0xc0, 0x3d, 0x45, 0x20,
			0x69, 0xf0, 0x68, 0x4a, 0x72, 0x83, 0xe4, 0x74,
			0x5e, 0xbd, 0x7a, 0x28, 0x24, 0x0b, 0xf1, 0xe0,
			0xe0, 0x68, 0x68, 0x10, 0xc9, 0x7f, 0xec, 0x67,
			0x63, 0x14, 0x46, 0x52, 0xf6, 0xa0, 0x16, 0xc3,
		},
		/* S1 : 384 byte */
		{
			0xb5, 0x12, 0x0b, 0xe9, 0x8b, 0xcd, 0xfd, 0xc1,
			0xe1, 0xe3, 0x31, 0x2d, 0xd7, 0xb5, 0x91, 0x0f,
			0x07, 0x31, 0x32, 0xa4, 0x27, 0x76, 0xc4, 0xda,
			0x75, 0x69, 0x0c, 0x64, 0x1f, 0x32, 0xd2, 0x89,
			0x91, 0x87, 0xd9, 0xb3, 0x9b, 0x55, 0xf9, 0x9e,
			0xbe, 0x6c, 0xa0, 0xa0, 0x80, 0x36, 0x37, 0x27,
			0x49, 0x81, 0x67, 0x06, 0x66, 0x4f, 0x39, 0xb2,
			0x73, 0x12, 0x13, 0x5c, 0x50, 0x33, 0x9f, 0x2b,
			0xb1, 0x7b, 0x2c, 0xee, 0xe2, 0x57, 0x68, 0xc2,
			0xbc, 0x0a, 0xc3, 0x7d, 0x6c, 0xa6, 0xee, 0x90,
			0x3c, 0x84, 0xe8, 0x2e, 0x2f, 0x4d, 0x00, 0x5d,
			0x73, 0xbd, 0xc3, 0x35, 0xf1, 0x35, 0x39, 0x9c,
			0x49, 0x12, 0x36, 0x62, 0xe8, 0x90, 0x81, 0x19,
			0x91, 0x84, 0x37, 0xed, 0xb6, 0x15, 0xb1, 0x4e,
			0x90, 0x6c, 0x9f, 0x8b, 0xa1, 0xb8, 0x5d, 0x5b,
			0x45, 0x90, 0x9f, 0x43, 0x9c, 0xc8, 0x99, 0x29,
			0x51, 0xbe, 0x16, 0x84, 0xa9, 0x9e, 0xba, 0x04,
			0xec, 0xb0, 0xf6, 0xdf, 0x92, 0x33, 0x53, 0x51,
			0x69, 0x77, 0x77, 0x4f, 0x69, 0xe8, 0x26, 0x65,
			0x11, 0x90, 0xaf, 0xfa, 0x86, 0xa4, 0x0b, 0xe7,
			0x5b, 0x06, 0xa4, 0x12, 0x8e, 0x55, 0x09, 0xc5,
			0x15, 0x57, 0xae, 0x4f, 0xb4, 0x10, 0xc7, 0xe5,
			0x84, 0x1a, 0xc9, 0xfd, 0xc4, 0xbc, 0x1f, 0x97,
			0xe2, 0x86, 0x24, 0x29, 0xf3, 0x71, 0xaa, 0xaf,
			0x99, 0x82, 0x4d, 0xac, 0xfe, 0xe0, 0xbc, 0x39,
			0x61, 0xfb, 0x98, 0xb3, 0xff, 0xc0, 0x91, 0xf7,
			0x79, 0x56, 0x22, 0x3e, 0xbf, 0x5b, 0xb5, 0x46,
			0x55, 0x23, 0x58, 0x20, 0x8a, 0x32, 0xef, 0x9c,
			0x37, 0x82, 0x5e, 0x81, 0x66, 0x8f, 0xd2, 0xc2,
			0x30, 0xf7, 0x88, 0xca, 0x16, 0xff, 0xbc, 0xc0,
			0xf1, 0xd8, 0x84, 0xb3, 0x0f, 0xe8, 0xef, 0xe6,
			0x49, 0x82, 0x95, 0x00, 0x4c, 0xa7, 0xc7, 0xf2,
			0xb1, 0x73, 0xe5, 0x66, 0x6b, 0x8b, 0x0f, 0xdf,
			0x9d, 0x32, 0x75, 0x65, 0x59, 0xf9, 0x9d, 0x10,
			0x5c, 0x1e, 0x80, 0x42, 0xa7, 0xae, 0xd7, 0x26,
			0x2c, 0xa9, 0xa1, 0x70, 0x25, 0xaa, 0x09, 0x60,
			0x75, 0xfe, 0x44, 0x33, 0xf3, 0x4d, 0xb6, 0xb0,
			0xf1, 0x97, 0x77, 0x6c, 0x21, 0xfb, 0xe0, 0x0e,
			0x83, 0x2e, 0xba, 0x02, 0x8e, 0x66, 0x52, 0x65,
			0x30, 0x18, 0x07, 0x9f, 0xee, 0x04, 0xeb, 0x3e,
			0x3c, 0x12, 0x80, 0x3c, 0x39, 0x83, 0x0d, 0x07,
			0x2a, 0xb4, 0x97, 0x1b, 0xca, 0xb4, 0xb7, 0x97,
			0x58, 0x69, 0x4b, 0x5d, 0x3d, 0x8a, 0xb2, 0x1c,
			0xe8, 0x74, 0xb7, 0xc4, 0x2b, 0xed, 0xd5, 0x26,
			0x52, 0x21, 0x9f, 0xf5, 0x16, 0xfd, 0x69, 0x4c,
			0x3d, 0x7c, 0xb0, 0xbe, 0xf0, 0x18, 0x1b, 0xb8,
			0x5e, 0xb4, 0xb1, 0x31, 0x84, 0xea, 0x3a, 0xef,
			0xe3, 0xcc, 0xee, 0xa5, 0xc5, 0x75, 0x96, 0xf7,
		},
	},
	/* SHA224 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x0b, 0xa5, 0x73, 0xa9, 0xdb, 0xb7, 0xf6, 0x2e,
			0x5a, 0x4d, 0x3d, 0x84, 0x1b, 0xfd, 0x92, 0x98,
			0xe8, 0xbb, 0x29, 0x9e, 0xb4, 0xfd, 0xb2, 0x56,
			0xd1, 0x1d, 0x2f, 0x8d, 0x64, 0xfe, 0x03, 0xe6,
			0x15, 0xf2, 0x4c, 0xda, 0x0b, 0xdb, 0x73, 0xfe,
			0x17, 0x91, 0x02, 0x84, 0x2f, 0x84, 0xb5, 0x05,
			0x1f, 0xa3, 0xd3, 0x7e, 0x7b, 0x7c, 0xbe, 0x98,
			0xd8, 0xa4, 0xc9, 0x2c, 0x3b, 0x59, 0x4b, 0x06,
			0xd2, 0x66, 0xf2, 0xe9, 0xe2, 0x47, 0x59, 0xd4,
			0x01, 0x8e, 0xdc, 0x84, 0x85, 0x85, 0xab, 0x3a,
			0x3c, 0x15, 0x1d, 0xbe, 0x5e, 0xe6, 0x47, 0xa4,
			0xbf, 0xc8, 0xce, 0xce, 0x49, 0x52, 0xf9, 0x32,
			0xaa, 0xc8, 0x0a, 0xdd, 0x4a, 0x42, 0xcf, 0x38,
			0x80, 0x0b, 0x74, 0x8b, 0x05, 0x48, 0x9b, 0xbf,
			0xa9, 0xda, 0xae, 0x68, 0x44, 0x85, 0x74, 0x03,
			0xf0, 0x51, 0xe3, 0x7b, 0x75, 0x30, 0x36, 0xf3,
		},
		/* S2 : 384 byte */
		{
			0x36, 0xfd, 0x68, 0x13, 0xab, 0x41, 0x1c, 0x4d,
			0xcb, 0x2d, 0x7b, 0xe1, 0xed, 0x61, 0x6c, 0x1e,
			0x40, 0xde, 0x29, 0x1a, 0x00, 0xac, 0xd8, 0x7d,
			0x2b, 0x4d, 0x9d, 0x4b, 0x73, 0xc8, 0x86, 0x4a,
			0x44, 0x41, 0x3c, 0x51, 0xf0, 0x9b, 0x37, 0x84,
			0x4a, 0x98, 0x04, 0xf8, 0x23, 0xb2, 0x7a, 0x90,
			0x94, 0x62, 0x7a, 0xaa, 0xf0, 0x0a, 0x6b, 0xe9,
			0x42, 0xd7, 0x55, 0x8b, 0xe1, 0x1b, 0x84, 0xa7,
			0x3d, 0x98, 0x02, 0x9c, 0x2e, 0x26, 0xeb, 0x8f,
			0x65, 0x05, 0x80, 0xec, 0xb1, 0x1b, 0x4e, 0xc2,
			0x36, 0x35, 0x97, 0x33, 0x34, 0x44, 0x56, 0x96,
			0x34, 0x35, 0x16, 0x00, 0x21, 0x29, 0x62, 0xfe,
			0xf5, 0x35, 0x2b, 0xdb, 0xa3, 0x67, 0x83, 0x28,
			0x99, 0xd3, 0x18, 0x8a, 0x74, 0x72, 0x36, 0xf0,
			0x85, 0x28, 0xf0, 0x86, 0xd9, 0x3c, 0xa3, 0x3a,
			0x06, 0xb1, 0x03, 0x92, 0xbb, 0xbd, 0x62, 0x5c,
			0x86, 0x7d, 0xdb, 0xa7, 0x4b, 0xb1, 0x51, 0xdc,
			0xc6, 0xaf, 0xdd, 0x4c, 0xe4, 0x10, 0x16, 0xdc,
			0x2e, 0xf0, 0xce, 0xea, 0x2c, 0xa2, 0x09, 0x17,
			0xfb, 0xdb, 0x07, 0x77, 0xe2, 0x35, 0x03, 0x46,
			0x4d, 0x0b, 0xb5, 0x9c, 0xd4, 0xe1, 0x2c, 0x10,
			0x94, 0x52, 0x50, 0x88, 0x9b, 0xae, 0x2e, 0xd8,
			0x39, 0xb7, 0x09, 0x64, 0xb2, 0xc9, 0xd9, 0x57,
			0xea, 0xc6, 0x22, 0x2a, 0x49, 0xb3, 0x37, 0x73,
			0x04, 0x11, 0x98, 0x44, 0x48, 0xe5, 0x8c, 0x02,
			0x73, 0x71, 0xbc, 0xf9, 0xc2, 0xc7, 0xd6, 0x86,
			0xde, 0x3b, 0xda, 0xe1, 0x67, 0x38, 0xdb, 0x52,
			0x76, 0xe0, 0xf5, 0x38, 0xd1, 0x5b, 0x35, 0x41,
			0xc0, 0xed, 0x86, 0xd3, 0x18, 0xb4, 0x23, 0xd8,
			0xc7, 0xf1, 0x85, 0x96, 0x02, 0x10, 0x8a, 0x4b,
			0x11, 0xc2, 0x77, 0x29, 0x41, 0x39, 0x6a, 0x14,
			0xa2, 0xa8, 0x8e, 0xc7, 0x97, 0x12, 0x97, 0xc1,
			0x86, 0x33, 0x02, 0x09, 0x98, 0xee, 0x02, 0xb3,
			0x11, 0x4d, 0x19, 0x01, 0x2a, 0x09, 0xa1, 0x81,
			0xd0, 0x1f, 0x11, 0xcb, 0x8f, 0x8c, 0xb5, 0xf4,
			0x38, 0xe8, 0x2f, 0xb4, 0x5e, 0x76, 0x78, 0xbc,
			0x8d, 0xf9, 0xa2, 0x6f, 0x1a, 0x34, 0x95, 0x43,
			0x9a, 0x7a, 0xc1, 0xf1, 0xbd, 0xa6, 0xfb, 0x86,
			0xc9, 0xb3, 0xed, 0x6c, 0xb5, 0xf7, 0x88, 0x63,
			0x49, 0x46, 0x34, 0x8b, 0x7e, 0x24, 0xb0, 0x89,
			0x4c, 0x39, 0xc5, 0x06, 0xce, 0xd2, 0xda, 0x65,
			0x7a, 0x33, 0x5e, 0x54, 0xe8, 0xf9, 0x97, 0x38,
			0x4e, 0x40, 0xc5, 0x6a, 0x17, 0xa2, 0x8a, 0x9b,
			0xb6, 0x48, 0x75, 0xa1, 0x59, 0xca, 0xda, 0x5a,
			0x64, 0x4a, 0xb3, 0xbd, 0x6e, 0xa7, 0xbc, 0x4c,
			0xca, 0xed, 0x43, 0xdd, 0x09, 0x55, 0xf6, 0xbe,
			0x6e, 0x45, 0x9e, 0x2e, 0x6a, 0x7b, 0xa6, 0x52,
			0xf1, 0xe9, 0xa3, 0xf8, 0xa8, 0x3e, 0x47, 0x95,
		},
	},
	/* SHA224 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0x89, 0x53, 0x0f, 0x81, 0x6a, 0x5e, 0x2a, 0xbd,
			0x4b, 0x42, 0x2f, 0xdf, 0x96, 0x8f, 0xfd, 0x96,
			0x4e, 0x0c, 0xcf, 0x82, 0xa4, 0xfc, 0x6d, 0x9a,
			0xc5, 0xa1, 0xa4, 0xcb, 0xf7, 0xff, 0xf3, 0xe1,
			0xe4, 0xe2, 0x87, 0xab, 0x35, 0x22, 0x6a, 0x5a,
			0x63, 0x26, 0xf7, 0x2b, 0xca, 0xa7, 0x91, 0x46,
			0x00, 0xb6, 0x94, 0xe5, 0x64, 0x01, 0x8c, 0xb8,
			0xfa, 0x52, 0xa5, 0x89, 0x76, 0x58, 0x63, 0x1c,
			0x96, 0xaa, 0x93, 0x59, 0xb5, 0x09, 0x82, 0xac,
			0x9e, 0xe5, 0x6c, 0xad, 0x9e, 0x23, 0x37, 0xfc,
			0xdd, 0x1e, 0x61, 0x6f, 0xed, 0xec, 0x38, 0x70,
			0xa4, 0xe2, 0x49, 0xa0, 0x27, 0x5a, 0x1a, 0xc1,
			0x48, 0xb3, 0x1c, 0xd2, 0x12, 0x9a, 0xdb, 0x7b,
			0xa1, 0x88, 0x78, 0xac, 0x38, 0x8c, 0x59, 0x82,
			0x8d, 0x4b, 0x1f, 0x6a, 0x67, 0x45, 0xd8, 0x88,
			0x6b, 0x5a, 0x76, 0x5a, 0x33, 0x8c, 0x81, 0x98,
		},
		/* S3 : 384 byte */
		{
			0x27, 0xc7, 0x96, 0xca, 0xee, 0xe6, 0xb4, 0xbc,
			0xd7, 0x50, 0xd8, 0xdf, 0x13, 0xcb, 0xe5, 0x16,
			0x4f, 0xd7, 0x26, 0xf9, 0x1b, 0xaa, 0x57, 0x5f,
			0x70, 0x2f, 0xe2, 0x96, 0x67, 0x44, 0xcf, 0x2b,
			0xef, 0x38, 0xc9, 0x3e, 0xfa, 0x11, 0x11, 0xc9,
			0x27, 0x7d, 0x77, 0xf3, 0xec, 0xf6, 0x97, 0xd0,
			0x20, 0x30, 0xf0, 0x1e, 0x3d, 0x96, 0x4c, 0x31,
			0x25, 0x53, 0x3d, 0x40, 0x88, 0x34, 0xb7, 0xce,
			0x65, 0x28, 0x24, 0x30, 0x3e, 0xb2, 0x78, 0xdc,
			0xa6, 0x10, 0x23, 0xa2, 0xf9, 0x28, 0x03, 0x52,
			0xf8, 0x9b, 0x5d, 0x03, 0xd0, 0x08, 0xc1, 0x03,
			0x03, 0x2b, 0x2b, 0x5c, 0x6b, 0x8c, 0xf7, 0xbe,
			0xfc, 0x1f, 0xff, 0xfa, 0x9b, 0x55, 0x9a, 0x99,
			0x57, 0x59, 0xa8, 0xd3, 0x3c, 0x6f, 0x49, 0xae,
			0x57, 0x4a, 0x2d, 0x31, 0x80, 0x5a, 0xb0, 0x55,
			0xe6, 0x46, 0xab, 0xed, 0x71, 0xb3, 0x0e, 0xcf,
			0x73, 0x67, 0x03, 0x0b, 0xf2, 0x6b, 0x96, 0x2d,
			0x41, 0xa2, 0xc7, 0xd7, 0x73, 0x5d, 0xdc, 0x0e,
			0x5f, 0x1e, 0xda, 0x30, 0xb1, 0xae, 0x6e, 0xfe,
			0xaa, 0xe9, 0xa4, 0xcf, 0x50, 0xb6, 0x85, 0x06,
			0xc2, 0x1b, 0x12, 0xe3, 0xdf, 0x2b, 0x99, 0x3f,
			0xea, 0xee, 0x44, 0x8a, 0x64, 0x43, 0xe6, 0x13,
			0xcf, 0x53, 0x6e, 0x2a, 0x71, 0x1a, 0xa5, 0x26,
			0x48, 0x71, 0x87, 0xb4, 0xfc, 0xd1, 0xfa, 0x68,
			0x4e, 0x99, 0x47, 0x8c, 0x28, 0xb8, 0x4d, 0x9a,
			0xf0, 0xeb, 0x6a, 0x49, 0x56, 0xc0, 0x37, 0x7d,
			0x08, 0xee, 0x26, 0xeb, 0xd2, 0xd8, 0xd2, 0xf4,
			0xce, 0x7e, 0x66, 0x04, 0x8d, 0xa3, 0xc0, 0x9c,
			0x05, 0x38, 0xff, 0x8e, 0xfa, 0x17, 0x86, 0x90,
			0xd4, 0x2f, 0x03, 0x41, 0xb2, 0x8a, 0x8f, 0xcb,
			0x64, 0x9b, 0x53, 0x1a, 0x07, 0xaf, 0x1f, 0x21,
			0xc4, 0x24, 0x32, 0x42, 0xe0, 0x45, 0xb1, 0x94,
			0xa0, 0x4a, 0xd0, 0xf9, 0x2e, 0xdc, 0xe4, 0x82,
			0xf3, 0x55, 0xf6, 0x69, 0x69, 0xcd, 0x90, 0x25,
			0x4a, 0xb1, 0x59, 0xff, 0x9d, 0x9c, 0x0c, 0x66,
			0x80, 0xf7, 0x8c, 0x99, 0x6d, 0x70, 0x48, 0xe2,
			0xc5, 0xf0, 0x07, 0xad, 0x36, 0x21, 0x9d, 0x67,
			0x2a, 0x0e, 0x76, 0xf1, 0xbf, 0x8b, 0xc8, 0x90,
			0xfa, 0xa5, 0x6e, 0x49, 0x3f, 0x0c, 0x52, 0xd0,
			0x9f, 0xa1, 0x26, 0x5c, 0xe5, 0x38, 0xe1, 0x66,
			0x70, 0x9a, 0x00, 0xa2, 0xcd, 0x64, 0xe4, 0x5b,
			0x9e, 0x5a, 0xca, 0xe2, 0xb9, 0x5d, 0xcb, 0x22,
			0xbc, 0xfe, 0x96, 0x30, 0xe3, 0x2f, 0x37, 0xd0,
			0xbb, 0x52, 0x9e, 0xfc, 0x8d, 0x29, 0x8c, 0x0b,
			0xa7, 0xb8, 0xd6, 0x5e, 0x16, 0xde, 0xe9, 0x9a,
			0xd7, 0x44, 0x6a, 0x39, 0x39, 0x46, 0x25, 0x87,
			0x24, 0xd0, 0x8d, 0x84, 0x76, 0xe7, 0xf1, 0x6c,
			0xcb, 0xc0, 0xe4, 0x26, 0x38, 0x38, 0x1a, 0x58,
		},
	},
	/* SHA224 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0xe3, 0x76, 0x56, 0xde, 0xfd, 0xee, 0xdf, 0xb4,
			0x6b, 0x14, 0x62, 0x8d, 0xff, 0x3f, 0x69, 0x17,
			0xb8, 0x42, 0x0e, 0x5a, 0x97, 0xef, 0x6c, 0x54,
			0xaf, 0xda, 0x55, 0xe0, 0x7c, 0x60, 0x43, 0xdd,
			0x75, 0xe7, 0x90, 0x8b, 0xe4, 0x66, 0xe9, 0x38,
			0xf6, 0x29, 0x00, 0x1d, 0x0e, 0xce, 0x81, 0x83,
			0x5f, 0x94, 0x48, 0x2a, 0xba, 0xd5, 0xd1, 0xea,
			0xa4, 0xd0, 0xef, 0x9b, 0xac, 0xac, 0xc1, 0x33,
			0xfc, 0xba, 0xe2, 0x2e, 0x2d, 0xfb, 0xe1, 0x33,
			0x60, 0xe2, 0xf1, 0xf4, 0x8a, 0x5a, 0xe1, 0x56,
			0x0f, 0x0b, 0x4e, 0xd2, 0x93, 0xd9, 0x17, 0x1a,
			0x0c, 0xae, 0x11, 0x00, 0x1c, 0x7a, 0xfc, 0x94,
			0x9f, 0x78, 0xb6, 0x8d, 0x80, 0xb2, 0xaf, 0xeb,
			0xd0, 0xc7, 0x9d, 0xda, 0x19, 0xec, 0x71, 0xd8,
			0xef, 0x31, 0x89, 0x1a, 0xc9, 0x06, 0x27, 0x2c,
			0x0f, 0xfd, 0x22, 0xd9, 0x74, 0xd1, 0xdb, 0x4a,
		},
		/* S4 : 384 byte */
		{
			0xa9, 0x27, 0xec, 0x4c, 0xeb, 0x2e, 0xc1, 0x47,
			0xcc, 0x45, 0x7e, 0x66, 0xc1, 0x2a, 0x64, 0x6f,
			0xdc, 0x41, 0x2d, 0x9e, 0xeb, 0x1d, 0x51, 0xf3,
			0xb5, 0xa3, 0xe5, 0xa8, 0xf4, 0xb0, 0xd3, 0x6d,
			0xeb, 0xa3, 0xa7, 0x19, 0x14, 0xcc, 0x6f, 0x23,
			0x21, 0xc3, 0x9d, 0x83, 0x4a, 0xdd, 0xb4, 0x85,
			0x7c, 0x82, 0xab, 0xe9, 0x28, 0x0c, 0x7c, 0x82,
			0x31, 0x89, 0x39, 0x04, 0xbd, 0x27, 0x47, 0x4c,
			0xb2, 0xcc, 0xe1, 0x01, 0x2b, 0x92, 0x1f, 0x0a,
			0x4d, 0x63, 0x80, 0xaa, 0xed, 0x61, 0x43, 0x56,
			0xd6, 0x53, 0x65, 0x33, 0x88, 0xce, 0x86, 0xac,
			0x71, 0xa2, 0x7c, 0x97, 0x67, 0x47, 0xc9, 0x21,
			0x3c, 0xf2, 0x97, 0xe7, 0x59, 0xfc, 0x3e, 0x2d,
			0x7b, 0x1a, 0xd5, 0xba, 0x8c, 0xb3, 0x10, 0x6c,
			0x0a, 0x67, 0x62, 0x44, 0x79, 0xce, 0x55, 0xd0,
			0xcd, 0x67, 0xc2, 0x4b, 0x5a, 0x45, 0xc1, 0x80,
			0xef, 0xb5, 0x83, 0x0f, 0xc2, 0x0d, 0x87, 0xad,
			0x3b, 0x15, 0x15, 0xe9, 0x0b, 0x77, 0xaf, 0x87,
			0xf0, 0x6c, 0x6b, 0x0e, 0x71, 0x29, 0x71, 0x8a,
			0x2f, 0x93, 0xae, 0xfb, 0xd1, 0x02, 0x8b, 0x1a,
			0xc6, 0x3f, 0x6b, 0xd7, 0xec, 0xa0, 0xa0, 0x02,
			0x69, 0xc0, 0x47, 0x3e, 0xaa, 0xc5, 0x57, 0x97,
			0x51, 0x19, 0x50, 0xb1, 0x15, 0x25, 0xc2, 0x41,
			0x41, 0xcb, 0x5a, 0xc4, 0xcf, 0xe2, 0xd9, 0xfd,
			0xbf, 0xfc, 0xbd, 0xdf, 0x84, 0x12, 0xa7, 0x0e,
			0xb1, 0xb8, 0xf4, 0x56, 0x48, 0x55, 0x3b, 0x70,
			0x67, 0x58, 0x1b, 0xc8, 0xee, 0x2d, 0x6a, 0xa0,
			0x89, 0xb9, 0x7e, 0x40, 0xdf, 0xe6, 0x1c, 0x33,
			0xfa, 0xf9, 0xfc, 0xd5, 0x65, 0x0f, 0x61, 0x07,
			0x85, 0x71, 0xf0, 0x3c, 0x6d, 0xf9, 0x4e, 0x01,
			0xdd, 0x7f, 0x90, 0xf1, 0xdb, 0xea, 0xf0, 0x42,
			0xd9, 0xbb, 0xc8, 0xb3, 0x63, 0x5c, 0x4c, 0x89,
			0x93, 0x28, 0x52, 0xb3, 0x11, 0xf6, 0x3f, 0xf6,
			0x19, 0x55, 0x0a, 0xab, 0xa0, 0x0f, 0x06, 0x14,
			0x18, 0x88, 0x62, 0x24, 0xf8, 0x47, 0x87, 0x08,
			0xf9, 0xec, 0xdb, 0xd9, 0x6f, 0x0f, 0x25, 0x15,
			0x35, 0x31, 0x92, 0xad, 0x93, 0xd4, 0x6c, 0xfa,
			0x8a, 0x4b, 0x3a, 0xc3, 0xea, 0xf7, 0xab, 0x9d,
			0x1a, 0x3c, 0x4d, 0xfc, 0x62, 0x74, 0x6c, 0xeb,
			0x08, 0x9e, 0xd3, 0xab, 0x40, 0x51, 0xae, 0x09,
			0x27, 0x4f, 0x54, 0xf2, 0xa9, 0xc3, 0x79, 0xff,
			0xe8, 0xc8, 0xc0, 0x10, 0x94, 0x87, 0xb6, 0x88,
			0x3a, 0x48, 0x49, 0x41, 0x5c, 0x6a, 0x0c, 0xcc,
			0xc6, 0x8b, 0x30, 0x96, 0x93, 0x8d, 0x6e, 0x54,
			0x66, 0x9e, 0xda, 0xf7, 0xb8, 0x2e, 0xc9, 0x01,
			0xc0, 0x53, 0x33, 0xe6, 0xc3, 0x10, 0x55, 0x41,
			0xf0, 0x31, 0xab, 0x59, 0x04, 0x61, 0xe7, 0xf1,
			0xf7, 0x76, 0xa2, 0x93, 0xe5, 0x93, 0xd0, 0x0d,
		},
	},
	/* SHA224 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0x99, 0xea, 0x30, 0xdf, 0xbb, 0x1e, 0xff, 0x6f,
			0x56, 0xad, 0x6e, 0x0b, 0x05, 0x59, 0x89, 0xa2,
			0xcb, 0xa1, 0x1f, 0xd3, 0x9e, 0x38, 0x6b, 0x00,
			0x26, 0xb5, 0xf3, 0xa4, 0xc2, 0x8c, 0xb1, 0xe6,
			0xcc, 0x3d, 0x71, 0x6e, 0x1e, 0xcb, 0x7a, 0x77,
			0xd4, 0x70, 0x70, 0x25, 0x54, 0x8f, 0x79, 0x19,
			0x8c, 0xea, 0x9f, 0x44, 0x7b, 0x15, 0x76, 0xf8,
			0xf3, 0x2b, 0xfe, 0x45, 0x9d, 0xbf, 0xca, 0x82,
			0x3d, 0x15, 0x62, 0x2a, 0x37, 0x92, 0xe7, 0xea,
			0x53, 0x72, 0xf5, 0xf7, 0xbd, 0xb9, 0xcd, 0xa5,
			0x50, 0x6c, 0xb4, 0x36, 0x13, 0x00, 0x96, 0xef,
			0x04, 0x13, 0xef, 0x72, 0x15, 0x5a, 0xec, 0x47,
			0x75, 0xdb, 0xcd, 0xbc, 0x10, 0x5c, 0x8d, 0xef,
			0x59, 0x1b, 0xc5, 0x29, 0x47, 0xbf, 0xce, 0x6d,
			0x9d, 0x8f, 0x25, 0x51, 0x6f, 0xe2, 0x14, 0x0d,
			0xe2, 0xd6, 0x8f, 0xd2, 0x33, 0x45, 0x5d, 0x5d,
		},
		/* S5 : 384 byte */
		{
			0x69, 0x21, 0x0e, 0xe2, 0x7a, 0x00, 0xdf, 0xbf,
			0xcd, 0x50, 0xaa, 0xf2, 0xeb, 0x50, 0x2c, 0x57,
			0x06, 0xdd, 0xff, 0x6d, 0x9d, 0x23, 0xfb, 0x38,
			0xd1, 0x11, 0x2f, 0x25, 0xc0, 0x47, 0xea, 0xac,
			0x57, 0xdc, 0x90, 0xa6, 0xda, 0x67, 0x38, 0x76,
			0x31, 0x9d, 0x5c, 0x04, 0x49, 0x4e, 0xce, 0x80,
			0x37, 0xc2, 0xfb, 0x60, 0x20, 0x3c, 0x9f, 0x23,
			0x32, 0x2e, 0x2c, 0x20, 0x63, 0xfa, 0x7d, 0x19,
			0x16, 0x5e, 0xdd, 0xd8, 0x9e, 0x1b, 0x91, 0x93,
			0x5a, 0x2b, 0x50, 0x02, 0x1e, 0x62, 0x68, 0x25,
			0xbf, 0x19, 0xcc, 0x46, 0xaa, 0xeb, 0xfa, 0xb0,
			0x9b, 0x49, 0x04, 0xde, 0xde, 0xf8, 0xc4, 0x63,
			0x2a, 0xae, 0xdb, 0x42, 0x9f, 0xeb, 0x68, 0x7b,
			0xba, 0xc2, 0xb4, 0x06, 0xf9, 0x23, 0xff, 0x1e,
			0x84, 0x49, 0x41, 0xb0, 0xc0, 0x2b, 0x08, 0xdc,
			0x2d, 0x8b, 0x42, 0x65, 0xfc, 0xeb, 0x61, 0xa8,
			0x2f, 0xce, 0xf0, 0x62, 0x4f, 0x28, 0xee, 0xf3,
			0xa9, 0x19, 0x3b, 0x86, 0xf1, 0x5f, 0x7a, 0xc4,
			0x70, 0xdf, 0x59, 0x0a, 0xe8, 0x55, 0xa7, 0xaa,
			0x75, 0x40, 0x49, 0x9d, 0xd4, 0x6a, 0x67, 0x85,
			0x5a, 0x5b, 0xae, 0x6e, 0xc5, 0xdc, 0xa8, 0xb0,
			0xc1, 0x6b, 0xcc, 0x69, 0xc0, 0xa1, 0xf9, 0x21,
			0x8e, 0xc7, 0xcc, 0xae, 0x21, 0x7a, 0xc9, 0xb4,
			0x7e, 0x8f, 0x7c, 0xae, 0xfc, 0x1e, 0x10, 0x2e,
			0x3b, 0xdb, 0x42, 0xa6, 0x77, 0xfa, 0xbe, 0x18,
			0x27, 0x4a, 0x5e, 0x69, 0x44, 0x7b, 0x33, 0x41,
			0x4d, 0xf5, 0xbb, 0x29, 0xcc, 0xeb, 0x2a, 0xbd,
			0x35, 0xc9, 0x4d, 0x36, 0x9e, 0xed, 0x25, 0x63,
			0x02, 0xd7, 0x58, 0xdf, 0x99, 0x48, 0xbe, 0xe4,
			0xef, 0xbd, 0xcc, 0x4a, 0xe3, 0x56, 0xe7, 0x8b,
			0xe7, 0x35, 0xf7, 0x42, 0x5b, 0x64, 0x43, 0xcb,
			0xff, 0x7e, 0x85, 0xc6, 0x53, 0xa6, 0x66, 0xde,
			0xd2, 0xe7, 0x4e, 0xc7, 0xf6, 0x11, 0x03, 0xd6,
			0xe8, 0xba, 0xc1, 0x10, 0xb1, 0x57, 0xae, 0xbf,
			0x61, 0xce, 0x32, 0xf8, 0xb6, 0xf5, 0x67, 0xac,
			0xbe, 0x92, 0xf6, 0xe3, 0xe2, 0x6e, 0xfd, 0xd3,
			0x94, 0x2a, 0xf6, 0xc2, 0x79, 0xc2, 0xc7, 0xb4,
			0xf1, 0x83, 0x98, 0xcc, 0x0a, 0xb4, 0xe2, 0x76,
			0x88, 0x1b, 0x60, 0x46, 0xcc, 0x55, 0x25, 0x94,
			0xcd, 0x96, 0x56, 0xf2, 0x2c, 0x3e, 0xe4, 0x98,
			0x07, 0xcc, 0xe0, 0xf0, 0x9f, 0x2b, 0xfa, 0x7a,
			0xbb, 0x87, 0x97, 0x27, 0xb7, 0x34, 0xdc, 0x19,
			0xc4, 0x68, 0xf4, 0xaf, 0x4d, 0x72, 0x0d, 0xa8,
			0xff, 0xd6, 0x50, 0xcd, 0xd6, 0x93, 0x82, 0x49,
			0xb6, 0xa4, 0xc8, 0x47, 0xa5, 0x13, 0x83, 0x88,
			0x8d, 0x12, 0x92, 0xa6, 0x16, 0x32, 0x22, 0x12,
			0x6d, 0x5a, 0x42, 0xdc, 0xa6, 0xfb, 0x22, 0x83,
			0xe7, 0xbb, 0xb6, 0xc2, 0x0d, 0x7b, 0x60, 0xb1,
		},
	},
	/* SHA224 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0x1e, 0xe4, 0x3d, 0xe2, 0xd8, 0x79, 0x7e, 0x65,
			0xbf, 0xaf, 0x52, 0xc2, 0x5a, 0x0b, 0x00, 0xb6,
			0xc0, 0x4e, 0x0e, 0x40, 0x46, 0x92, 0x05, 0x56,
			0x5a, 0x96, 0x74, 0xd6, 0xaf, 0x57, 0x37, 0xbf,
			0x9b, 0xe0, 0xf9, 0xa1, 0xbd, 0x62, 0xf8, 0x52,
			0xe2, 0x87, 0x08, 0xe3, 0x29, 0x04, 0xdb, 0xd2,
			0x66, 0x6e, 0x7f, 0x81, 0xc5, 0x4e, 0x28, 0xe7,
			0xb0, 0x77, 0x30, 0x86, 0xf2, 0x97, 0x5c, 0x9d,
			0x1c, 0x0d, 0x61, 0x9e, 0x27, 0xfa, 0xa7, 0xe2,
			0x5c, 0x9b, 0x4c, 0x9c, 0x71, 0xd6, 0x6c, 0x0c,
			0xf5, 0x12, 0xa0, 0xf5, 0xee, 0x4c, 0xc4, 0x50,
			0xc0, 0x67, 0xfe, 0x42, 0x50, 0xc4, 0xfb, 0x4c,
			0x6a, 0x13, 0x7c, 0xc2, 0x60, 0x69, 0x12, 0x7e,
			0xf4, 0x92, 0x25, 0xd5, 0x78, 0xa8, 0x3b, 0xca,
			0x34, 0xe4, 0x77, 0x82, 0x08, 0xb5, 0x60, 0xf8,
			0x53, 0x0f, 0xe5, 0xf2, 0x13, 0x06, 0x9d, 0x34,
		},
		/* S6 : 384 byte */
		{
			0x3d, 0xd7, 0x22, 0xd2, 0xf0, 0x54, 0x3e, 0x66,
			0x74, 0x3f, 0x8c, 0xdb, 0x60, 0x34, 0x1d, 0x61,
			0xfd, 0x7b, 0x6e, 0xf8, 0xcb, 0x23, 0xa9, 0xe9,
			0xf3, 0x40, 0x57, 0xd7, 0xa0, 0xaf, 0x49, 0xe3,
			0x08, 0x26, 0xaa, 0x0a, 0xaf, 0x1f, 0xd3, 0x4e,
			0xfe, 0xbd, 0xbf, 0xc9, 0x3a, 0xe5, 0x21, 0x27,
			0x11, 0xa1, 0x60, 0xf2, 0xb8, 0x78, 0x6f, 0x4f,
			0x5b, 0xec, 0xc4, 0x92, 0x09, 0xbd, 0x05, 0xdd,
			0xf8, 0xde, 0x9f, 0xec, 0xd0, 0x0a, 0xf5, 0x30,
			0x4d, 0x66, 0x15, 0x27, 0x2f, 0x2e, 0x49, 0x40,
			0xbc, 0x8c, 0x39, 0xc2, 0xfb, 0xc6, 0x36, 0xf8,
			0xc1, 0x05, 0x56, 0x5e, 0xc0, 0xf1, 0x57, 0x00,
			0xcd, 0xb0, 0x66, 0xc5, 0xca, 0x1f, 0xd0, 0xe3,
			0xe3, 0xf4, 0x94, 0x52, 0xe4, 0xf6, 0x71, 0x5a,
			0x58, 0x22, 0x27, 0xd5, 0x9e, 0xc1, 0x04, 0x57,
			0x5c, 0x17, 0x4f, 0x8c, 0xd1, 0x3e, 0xca, 0xbc,
			0x4d, 0x58, 0x99, 0xe0, 0x2e, 0xbd, 0x3e, 0x81,
			0xbd, 0x2c, 0x00, 0x32, 0x42, 0x73, 0x8b, 0x3b,
			0x95, 0xb0, 0xe0, 0xcf, 0x0e, 0xf0, 0x2f, 0x8e,
			0xe0, 0x28, 0x96, 0xdf, 0x64, 0x60, 0x68, 0xae,
			0x23, 0x3f, 0xfc, 0x44, 0x36, 0xf1, 0xe9, 0x7d,
			0x37, 0xd4, 0x5d, 0x49, 0x7e, 0x1a, 0x54, 0xa0,
			0xd6, 0xfc, 0x5a, 0xaf, 0x27, 0x5e, 0xc5, 0x0c,
			0xbf, 0x0b, 0x40, 0x20, 0x52, 0x20, 0x0f, 0x6b,
			0xc3, 0x53, 0x73, 0x82, 0x8b, 0xcd, 0xb4, 0x8a,
			0x17, 0x8c, 0x96, 0x88, 0x65, 0x8a, 0x23, 0x63,
			0xa8, 0x68, 0x3a, 0xb9, 0xea, 0xfa, 0x97, 0x90,
			0xee, 0xf2, 0xc7, 0x9d, 0xa1, 0x48, 0xa9, 0xd9,
			0x95, 0x39, 0x5d, 0x9f, 0x6a, 0x7b, 0x31, 0x0f,
			0x6f, 0x71, 0x41, 0xd3, 0xcb, 0x0f, 0x20, 0x6e,
			0x8b, 0xaa, 0x82, 0xa3, 0x38, 0xd5, 0x19, 0xee,
			0x88, 0x1c, 0xf6, 0x1d, 0x5e, 0x1f, 0x90, 0x6d,
			0x42, 0xc2, 0xe8, 0x5f, 0x25, 0xcd, 0x19, 0xd9,
			0x86, 0x4a, 0xb5, 0x4a, 0x32, 0x96, 0x9c, 0x8e,
			0xdf, 0x29, 0xe5, 0xac, 0x52, 0xf6, 0x20, 0x06,
			0xd9, 0x21, 0x9c, 0x21, 0x14, 0x00, 0x07, 0xb0,
			0x5c, 0x63, 0xe3, 0xba, 0x4c, 0x04, 0xec, 0xe5,
			0xd8, 0x80, 0x50, 0x26, 0xdb, 0xe8, 0xff, 0x66,
			0x52, 0x52, 0xd5, 0x37, 0xd0, 0x13, 0xf7, 0x09,
			0xd8, 0x49, 0x99, 0xf8, 0x4b, 0x43, 0x82, 0xa8,
			0x94, 0xc1, 0xba, 0x03, 0x18, 0x49, 0x37, 0x83,
			0xa5, 0x98, 0xf6, 0x37, 0xbc, 0x2d, 0x8d, 0x56,
			0x78, 0xcf, 0x65, 0xd0, 0x38, 0x33, 0x80, 0xad,
			0xa0, 0xdb, 0x5a, 0x51, 0x07, 0x37, 0xa8, 0xb7,
			0x0c, 0x3b, 0xae, 0xee, 0xe4, 0x70, 0x85, 0x08,
			0x8e, 0x96, 0xd9, 0x94, 0x38, 0xba, 0x5e, 0x98,
			0x87, 0x88, 0xf2, 0x88, 0x6a, 0xa7, 0xe2, 0x95,
			0xd8, 0x57, 0x8e, 0xb2, 0x7f, 0x1d, 0x68, 0x38,
		},
	},
	/* SHA224 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0x74, 0x03, 0x22, 0x95, 0x3b, 0xfc, 0x8e, 0x84,
			0x0c, 0xec, 0xd9, 0x96, 0x3f, 0x58, 0xbe, 0xa7,
			0x0d, 0x2b, 0xd2, 0x0a, 0x50, 0x63, 0x66, 0xd7,
			0x8a, 0x0b, 0xad, 0x86, 0x29, 0x69, 0x22, 0xbd,
			0x09, 0x78, 0x24, 0x67, 0x3b, 0x99, 0xe3, 0x06,
			0x05, 0x85, 0x80, 0x4a, 0x29, 0x86, 0x70, 0xe2,
			0x6a, 0xe7, 0x22, 0x92, 0x4d, 0xa8, 0xe1, 0x86,
			0x1d, 0x77, 0xfb, 0xe6, 0x31, 0xdc, 0x23, 0xaa,
			0x72, 0xb4, 0x14, 0xb0, 0x17, 0xe0, 0x77, 0x0b,
			0xb3, 0x3e, 0x07, 0x9f, 0x72, 0xd8, 0xf3, 0xeb,
			0x9f, 0x5c, 0x83, 0x25, 0x6a, 0xcd, 0xff, 0x8c,
			0xe9, 0x77, 0xcd, 0xfa, 0x5c, 0x28, 0xd9, 0xb8,
			0xb5, 0x9d, 0x3a, 0x97, 0x58, 0x3b, 0x12, 0x3c,
			0x1f, 0x00, 0xb5, 0xbc, 0xa1, 0xb8, 0x0e, 0x69,
			0xb4, 0x74, 0x3f, 0xeb, 0x30, 0x38, 0x88, 0x92,
			0xf6, 0xf4, 0x6a, 0xea, 0x31, 0xb5, 0x0c, 0x90,
		},
		/* S7 : 384 byte */
		{
			0x7c, 0x41, 0x48, 0x40, 0x91, 0x0c, 0xa0, 0x8f,
			0xec, 0xd2, 0x3f, 0xf1, 0x2c, 0xee, 0xbc, 0xd4,
			0x8b, 0x7a, 0xfa, 0x4e, 0x6a, 0x87, 0xa4, 0x06,
			0x54, 0xba, 0xae, 0xc6, 0xc9, 0x05, 0x00, 0x87,
			0xb1, 0xf0, 0xb6, 0xfa, 0x04, 0xe3, 0x6c, 0xd5,
			0x95, 0xad, 0x29, 0x3d, 0x08, 0x27, 0xe9, 0xe1,
			0xc9, 0x4f, 0xe0, 0x33, 0xec, 0x42, 0xbb, 0xd0,
			0x21, 0xf7, 0xce, 0x2e, 0x75, 0xda, 0x6d, 0xd2,
			0x06, 0xb9, 0x91, 0x51, 0x76, 0x8d, 0x6d, 0x5a,
			0xe7, 0xb1, 0xf0, 0x44, 0x16, 0x80, 0x4c, 0x2a,
			0xd7, 0xc6, 0x74, 0x4c, 0x73, 0x43, 0xc8, 0xf0,
			0x1b, 0xe2, 0x59, 0x36, 0x1c, 0x11, 0x68, 0x10,
			0xf0, 0xad, 0xa1, 0xc6, 0x43, 0x48, 0x05, 0x5b,
			0x25, 0x94, 0xa0, 0x6b, 0xdc, 0x08, 0xdb, 0x39,
			0x0d, 0x75, 0x0e, 0x4a, 0xee, 0xa5, 0x43, 0x59,
			0x32, 0xa0, 0x4d, 0x0e, 0x69, 0xd5, 0x90, 0x63,
			0x04, 0xc8, 0x4e, 0x19, 0xd5, 0xfb, 0x86, 0x88,
			0xca, 0x25, 0x98, 0xb6, 0xfa, 0xe6, 0xd1, 0x69,
			0x59, 0x3f, 0xac, 0x29, 0x09, 0x23, 0x8c, 0x55,
			0x3c, 0x66, 0x4d, 0xe9, 0x2c, 0xba, 0x6d, 0x89,
			0x15, 0xe0, 0x1a, 0x6e, 0x99, 0xd8, 0xd9, 0x2f,
			0xec, 0xbc, 0x6e, 0xae, 0xfd, 0x93, 0x15, 0x1c,
			0x61, 0xfb, 0xbd, 0xe2, 0xea, 0xcf, 0x26, 0x34,
			0xe7, 0xb6, 0x11, 0x6a, 0xd2, 0xfe, 0x88, 0x59,
			0xb6, 0x5a, 0x70, 0x66, 0xd7, 0xb5, 0xb7, 0x76,
			0x38, 0x65, 0x0b, 0x60, 0xa4, 0x3d, 0x82, 0x77,
			0xda, 0xb0, 0xac, 0xa1, 0x45, 0x06, 0x5b, 0x3c,
			0xf0, 0x0d, 0x96, 0x3b, 0x7f, 0x81, 0x8d, 0xda,
			0xdd, 0x7c, 0x54, 0xbe, 0x5b, 0x4b, 0xa7, 0x69,
			0xae, 0x01, 0x34, 0x46, 0xa5, 0x74, 0xdb, 0xbb,
			0x8f, 0x7c, 0x22, 0xb2, 0xd1, 0x54, 0x3e, 0x7b,
			0x5e, 0xc0, 0x8d, 0xfd, 0xe3, 0x8e, 0xf9, 0xad,
			0x84, 0x3c, 0x1b, 0xb6, 0xd9, 0x55, 0x8a, 0xef,
			0xcd, 0x45, 0xd3, 0xb1, 0x2c, 0x82, 0x06, 0xb7,
			0x92, 0xca, 0x72, 0xbf, 0x49, 0x50, 0xbe, 0xfb,
			0xee, 0xc0, 0x4f, 0xc1, 0xa2, 0x8c, 0x37, 0x20,
			0x58, 0x85, 0x13, 0xa2, 0x9a, 0xf9, 0x69, 0x1d,
			0x2f, 0x31, 0xdd, 0x7d, 0x39, 0xa5, 0x6b, 0xcb,
			0x5f, 0x49, 0x9f, 0xb1, 0x4c, 0xa4, 0x7f, 0xa5,
			0x41, 0xe2, 0xea, 0x67, 0x84, 0x33, 0x99, 0xe0,
			0xc8, 0xab, 0x89, 0xc8, 0x1e, 0x58, 0x93, 0x41,
			0x59, 0x42, 0xbf, 0xe4, 0xe4, 0x70, 0xa6, 0x78,
			0xc0, 0xe5, 0x61, 0xed, 0x64, 0x55, 0x47, 0x11,
			0xb1, 0x6b, 0xe3, 0x35, 0x0c, 0x98, 0x5b, 0x61,
			0xf2, 0x92, 0x19, 0xc5, 0x27, 0x4d, 0x87, 0x93,
			0x08, 0xdd, 0x25, 0xfc, 0x03, 0x3f, 0x81, 0x9c,
			0x38, 0x59, 0x04, 0x65, 0x43, 0x99, 0xe5, 0x43,
			0x8f, 0xd9, 0xc8, 0xcf, 0x1e, 0xc7, 0x6e, 0xcc,
		},
	},
	/* SHA224 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0xf7, 0xe3, 0x78, 0x20, 0xa1, 0x9d, 0x5f, 0x6a,
			0x05, 0xeb, 0x47, 0x79, 0xc2, 0x40, 0xe7, 0xfb,
			0x58, 0x6a, 0xe8, 0xc3, 0xdf, 0x71, 0x3b, 0xcd,
			0xf9, 0xc2, 0xaf, 0x7c, 0x05, 0x8c, 0xc3, 0x27,
			0x95, 0x6b, 0xb8, 0xd4, 0x22, 0x44, 0xeb, 0x43,
			0xff, 0x70, 0x62, 0x2f, 0x8c, 0x1c, 0xa5, 0xd0,
			0xac, 0xef, 0xcf, 0xa4, 0x79, 0xee, 0xe4, 0x6f,
			0x36, 0x9d, 0x65, 0x81, 0x84, 0x67, 0x22, 0x37,
			0xd9, 0x40, 0x50, 0xc4, 0x2f, 0x89, 0xdb, 0x31,
			0xf9, 0x34, 0xfe, 0xa3, 0x5b, 0x28, 0x10, 0xdd,
			0x9a, 0xe7, 0xa1, 0x05, 0xd2, 0x6e, 0xc5, 0xab,
			0xe7, 0x5d, 0xb0, 0x07, 0xbd, 0x57, 0x83, 0x82,
			0xac, 0xac, 0x66, 0x79, 0x2e, 0x35, 0xd7, 0x3d,
			0xdb, 0x80, 0x41, 0x5e, 0x98, 0x2d, 0xd1, 0x29,
			0x0b, 0x98, 0x85, 0x6f, 0x52, 0xb9, 0x86, 0x88,
			0xf4, 0x48, 0xb7, 0x98, 0x17, 0x24, 0x8e, 0x11,
		},
		/* S8 : 384 byte */
		{
			0x56, 0x3e, 0x22, 0x61, 0x7d, 0xd8, 0x89, 0xe7,
			0xbe, 0x8d, 0xd2, 0x6a, 0x17, 0x6e, 0xe9, 0xf6,
			0x7b, 0x9b, 0x3e, 0xb0, 0x40, 0xad, 0x7a, 0x7f,
			0xab, 0xc0, 0x89, 0xb2, 0x7e, 0xd4, 0xe7, 0xa7,
			0x82, 0xf1, 0x52, 0x2b, 0x44, 0x6f, 0x42, 0xa5,
			0x67, 0x49, 0x21, 0x37, 0x77, 0x0c, 0x61, 0x2d,
			0xc5, 0xe4, 0x28, 0xec, 0x28, 0xa3, 0xc5, 0x02,
			0xaa, 0x25, 0x08, 0xfb, 0x46, 0xb7, 0x03, 0xd7,
			0x9d, 0x1f, 0xde, 0x8e, 0x1a, 0x50, 0x7d, 0x70,
			0x62, 0xe2, 0x64, 0x40, 0xb3, 0xa3, 0xff, 0x16,
			0xbc, 0x82, 0xfc, 0xc9, 0xb3, 0x01, 0xf2, 0xb5,
			0x8f, 0xa8, 0x18, 0x52, 0xb5, 0x7f, 0x95, 0x1d,
			0x92, 0x51, 0x64, 0xbe, 0x0c, 0x70, 0xbd, 0x28,
			0x1d, 0x72, 0x6c, 0x9a, 0x71, 0xb9, 0x84, 0x28,
			0x03, 0x52, 0x28, 0x9f, 0x8c, 0x1b, 0x39, 0x4a,
			0x85, 0xdf, 0x9e, 0x17, 0x32, 0xa4, 0x53, 0x9a,
			0x30, 0xa7, 0x59, 0xe8, 0xf1, 0x26, 0x09, 0x6b,
			0xf7, 0x3f, 0x7b, 0x25, 0xa5, 0xed, 0x34, 0xc3,
			0x2a, 0xf3, 0x45, 0xbc, 0x32, 0xe4, 0x12, 0xe0,
			0x8b, 0x6c, 0xa9, 0xb6, 0x56, 0xa6, 0x92, 0x85,
			0x19, 0x65, 0x5e, 0xc9, 0x76, 0x9c, 0xf1, 0xda,
			0xe7, 0xc9, 0x85, 0x50, 0x5a, 0x81, 0x2e, 0xe4,
			0x4b, 0xb3, 0xb4, 0x2e, 0xcb, 0xec, 0x91, 0x1b,
			0xec, 0xed, 0x8f, 0xe8, 0x73, 0x65, 0xf1, 0x13,
			0xaa, 0xc0, 0x0a, 0x65, 0x9c, 0x0e, 0xb3, 0x7b,
			0xfe, 0x75, 0x36, 0xf9, 0x17, 0x6a, 0xfe, 0x9c,
			0x45, 0x9a, 0x08, 0xae, 0x23, 0x60, 0x0d, 0x4c,
			0x85, 0x43, 0xef, 0x3c, 0x3a, 0xf4, 0xcd, 0x10,
			0x11, 0xe0, 0x8f, 0xdc, 0xf1, 0x99, 0xba, 0x49,
			0x02, 0x4f, 0x08, 0x80, 0x8c, 0x47, 0x59, 0x86,
			0x87, 0x05, 0x61, 0xd6, 0xa0, 0x88, 0xb7, 0x9c,
			0x38, 0xae, 0x8c, 0xe0, 0xe6, 0xec, 0x40, 0x26,
			0x8b, 0xc9, 0xfb, 0x7a, 0x3b, 0x61, 0x85, 0x87,
			0xf5, 0x5f, 0xbc, 0xd3, 0x1c, 0xea, 0x93, 0x70,
			0x24, 0x38, 0x65, 0x49, 0x2e, 0x5f, 0x13, 0xc9,
			0xfd, 0xad, 0x61, 0xf4, 0x0b, 0x32, 0xd3, 0xa9,
			0x15, 0x19, 0x42, 0x44, 0x94, 0x9a, 0xdd, 0x15,
			0x02, 0x6c, 0x0a, 0xe1, 0x9f, 0x52, 0xad, 0x5b,
			0x70, 0x36, 0x5e, 0x77, 0xf2, 0xcf, 0x53, 0x29,
			0x8c, 0x9e, 0x2b, 0xad, 0x06, 0x17, 0x1b, 0x09,
			0x08, 0xdf, 0x26, 0xb2, 0x2e, 0xf1, 0xc7, 0x37,
			0xc3, 0xb3, 0x21, 0x39, 0x5f, 0xfc, 0xdb, 0x71,
			0xc8, 0x22, 0x8f, 0xe9, 0xde, 0x02, 0x7f, 0x0d,
			0x31, 0x06, 0x86, 0xb1, 0x68, 0x3a, 0x67, 0x41,
			0x9e, 0xa0, 0x89, 0x71, 0xcf, 0x0b, 0xf1, 0xa3,
			0xe5, 0xa1, 0x07, 0x27, 0x24, 0x83, 0x46, 0x01,
			0xd5, 0xf9, 0x44, 0xfa, 0x23, 0xf7, 0x7d, 0x8e,
			0x77, 0xe8, 0x87, 0xf8, 0x8d, 0xdb, 0xee, 0xb1,
		},
	},
	/* SHA224 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0x87, 0x10, 0xa8, 0x77, 0xb7, 0xa4, 0xc2, 0xe5,
			0x78, 0x79, 0x3b, 0xd3, 0xe4, 0xd1, 0x9c, 0xb5,
			0x6d, 0xe9, 0x7f, 0xcd, 0x1f, 0x2a, 0xf5, 0xfb,
			0x25, 0xa3, 0x26, 0xd6, 0x8f, 0xb7, 0x37, 0xfb,
			0x52, 0x13, 0x71, 0xa6, 0x90, 0xe4, 0x9f, 0x7f,
			0x1a, 0x46, 0xb7, 0xb6, 0x34, 0xff, 0xbd, 0x51,
			0x98, 0x6d, 0xe5, 0xc5, 0xbd, 0xbd, 0xf8, 0xc4,
			0x58, 0x5e, 0xf8, 0x57, 0x24, 0xb5, 0x07, 0x2c,
			0xde, 0x13, 0x85, 0x31, 0x94, 0xe4, 0x79, 0x62,
			0x20, 0x29, 0x32, 0xde, 0xf0, 0x28, 0x2e, 0x41,
			0x08, 0x61, 0x3a, 0x2e, 0x49, 0xc5, 0xdb, 0x2b,
			0xf3, 0x23, 0xed, 0xb2, 0x69, 0xe3, 0x8a, 0x84,
			0x34, 0xf6, 0x2d, 0x41, 0x4b, 0x0d, 0x17, 0x36,
			0x91, 0x09, 0xf2, 0x76, 0xa0, 0xb3, 0xb5, 0x2c,
			0xc5, 0xae, 0xc7, 0x2f, 0x4b, 0xaa, 0x67, 0xd7,
			0xfd, 0xd9, 0x4b, 0x10, 0xe6, 0xa7, 0x87, 0xac,
		},
		/* S9 : 384 byte */
		{
			0xa7, 0x83, 0x58, 0xef, 0x28, 0x30, 0x3d, 0xeb,
			0xa1, 0xbf, 0x1b, 0xc3, 0xca, 0xe5, 0x9a, 0xb0,
			0xff, 0x66, 0x14, 0xc5, 0x20, 0xee, 0xb7, 0xd8,
			0xc8, 0xfd, 0x5c, 0xed, 0x34, 0xda, 0x74, 0x54,
			0xad, 0x14, 0x0b, 0x53, 0x9e, 0xf7, 0x5e, 0x2d,
			0x65, 0xdd, 0x89, 0x1e, 0xbf, 0x89, 0x9a, 0x88,
			0xad, 0xa2, 0x5b, 0xcc, 0x35, 0x72, 0x60, 0x53,
			0xda, 0x68, 0xe2, 0xe0, 0x2b, 0x6a, 0xcd, 0x2e,
			0x7e, 0x21, 0xcb, 0x8b, 0x37, 0x35, 0x5d, 0x19,
			0xbd, 0x4c, 0x3e, 0x36, 0xa8, 0xc1, 0x64, 0x7e,
			0x1a, 0x38, 0x4c, 0x8a, 0xd2, 0xab, 0x39, 0xbd,
			0x22, 0xf3, 0xd3, 0x0f, 0x0f, 0x9d, 0xd6, 0x85,
			0xfe, 0x4d, 0xd7, 0xf8, 0x36, 0xec, 0x46, 0xbb,
			0xce, 0xf0, 0x80, 0x5d, 0x08, 0xa7, 0x84, 0xa6,
			0x96, 0x4c, 0xd5, 0x0f, 0x58, 0x07, 0x1e, 0xd7,
			0x9f, 0x88, 0x24, 0x91, 0xa3, 0x31, 0xb4, 0x45,
			0x39, 0x0b, 0x43, 0xf2, 0xa2, 0x95, 0xa1, 0x3a,
			0x28, 0xce, 0x0f, 0x44, 0xbb, 0x6d, 0x63, 0xf3,
			0x19, 0xd8, 0xde, 0x90, 0xe3, 0x90, 0x17, 0xf4,
			0xcb, 0xc1, 0x45, 0x33, 0xda, 0x33, 0x38, 0x0f,
			0x55, 0x3f, 0x09, 0x7e, 0x79, 0x6a, 0x67, 0x1b,
			0xa2, 0x9c, 0x94, 0x58, 0x2c, 0xd5, 0x19, 0xf1,
			0xf6, 0x4d, 0xb3, 0xbe, 0x89, 0x4b, 0x66, 0x15,
			0xf6, 0x84, 0x4f, 0xf2, 0xfc, 0x62, 0x10, 0x13,
			0x82, 0xb0, 0x44, 0xf5, 0x85, 0x6b, 0x9b, 0xb9,
			0x78, 0x71, 0xcf, 0x13, 0x7c, 0x4e, 0x9e, 0x48,
			0x4e, 0x84, 0xa3, 0xcd, 0x2d, 0xae, 0xa8, 0xe1,
			0xc6, 0x35, 0x8d, 0x66, 0xcd, 0x83, 0x26, 0xc1,
			0x92, 0x5c, 0xe1, 0xf7, 0xd2, 0xd2, 0xe9, 0x04,
			0x57, 0xad, 0xaa, 0x65, 0xec, 0x3a, 0x67, 0xf4,
			0x86, 0x5b, 0xf6, 0x12, 0x0e, 0xff, 0xa0, 0x6a,
			0x79, 0xde, 0xb6, 0xb6, 0xca, 0x9f, 0x85, 0xc9,
			0xdd, 0x96, 0x7f, 0x2f, 0x31, 0xa2, 0x2d, 0x5d,
			0xb2, 0x5b, 0x15, 0x53, 0x0a, 0x9e, 0x85, 0x0a,
			0xca, 0x48, 0x6b, 0xc0, 0xca, 0xc2, 0xbe, 0x6b,
			0x0a, 0xf6, 0x6e, 0xcb, 0x56, 0x8c, 0x09, 0x55,
			0xa3, 0x04, 0x95, 0xbd, 0xd5, 0xd0, 0x5a, 0x22,
			0x0c, 0xd0, 0x6c, 0xb0, 0x6f, 0x04, 0xf2, 0x16,
			0x07, 0x6a, 0xaa, 0xd4, 0x38, 0x2a, 0x94, 0x04,
			0x0d, 0xcc, 0xda, 0x68, 0xa1, 0x9d, 0x55, 0xb4,
			0x93, 0x38, 0xc9, 0x31, 0x5a, 0xa8, 0x02, 0x91,
			0x06, 0x55, 0xfe, 0x93, 0x94, 0xaa, 0x73, 0x59,
			0x0a, 0x6b, 0x2a, 0x04, 0x39, 0xbb, 0xef, 0x5e,
			0xc7, 0xcc, 0xb5, 0x20, 0xf2, 0xc5, 0xcb, 0x71,
			0xd3, 0x93, 0xa6, 0xcc, 0xe2, 0x5b, 0xf7, 0x7d,
			0x80, 0x33, 0x44, 0x4f, 0xb3, 0xda, 0x8a, 0xc8,
			0x61, 0xc6, 0x3d, 0xc2, 0x56, 0x1f, 0xfd, 0xcc,
			0xe8, 0xc2, 0x06, 0x5b, 0x35, 0xb5, 0xc8, 0x3b,
		},
	},
};

/* 10 elements */
static mod3072_t SHA256[] = {
	/* SHA256 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0xbc, 0xf6, 0x07, 0x43, 0x33, 0xa7, 0xed, 0xe5,
			0x92, 0xff, 0xc9, 0xec, 0xf1, 0xc5, 0x11, 0x81,
			0x28, 0x7e, 0x0a, 0x69, 0x36, 0x3f, 0x46, 0x7d,
			0xe4, 0xbf, 0x6b, 0x5a, 0xa5, 0xb0, 0x37, 0x59,
			0xc1, 0x50, 0xc1, 0xc2, 0xb2, 0x3b, 0x02, 0x3c,
			0xce, 0x83, 0x93, 0x88, 0x27, 0x02, 0xb8, 0x6f,
			0xb0, 0xef, 0x9e, 0xf9, 0xa1, 0xb0, 0xe1, 0xe0,
			0x1c, 0xef, 0x51, 0x44, 0x10, 0xf0, 0xf6, 0xa0,
			0x5e, 0x22, 0x52, 0xfd, 0x3a, 0xf4, 0xe5, 0x66,
			0xd4, 0xe9, 0xf7, 0x9b, 0x38, 0xef, 0x91, 0x0a,
			0x73, 0xed, 0xcd, 0xfa, 0xf8, 0x9b, 0x4f, 0x0a,
			0x42, 0x96, 0x14, 0xda, 0xba, 0xb4, 0x6b, 0x08,
			0xda, 0x94, 0x40, 0x5e, 0x93, 0x7a, 0xa0, 0x49,
			0xec, 0x5a, 0x7a, 0x8d, 0xed, 0x33, 0xa3, 0x38,
			0xbb, 0x9f, 0x1d, 0xd4, 0x04, 0xa7, 0x99, 0xe1,
			0x9d, 0xdb, 0x3a, 0x83, 0x6a, 0xa3, 0x9c, 0x77,
		},
		/* S0 : 384 byte */
		{
			0xd1, 0xd2, 0x1b, 0x8d, 0xfa, 0x55, 0xf0, 0x68,
			0x1e, 0x8f, 0xa8, 0x61, 0x35, 0xcf, 0x29, 0x2d,
			0x71, 0xb7, 0x66, 0x97, 0x13, 0xc2, 0x91, 0xd8,
			0xf8, 0xdc, 0x24, 0x64, 0x64, 0xde, 0x3b, 0xbb,
			0x96, 0x1b, 0x59, 0x6d, 0xfc, 0x8f, 0xda, 0x6c,
			0x82, 0x3c, 0x38, 0x40, 0x08, 0xd0, 0x5b, 0xcb,
			0x3d, 0xcc, 0xc3, 0x6a, 0xcc, 0xf1, 0xb2, 0xbe,
			0xde, 0x1a, 0x95, 0xe5, 0x22, 0x58, 0xd7, 0xd1,
			0xbd, 0xf1, 0xfc, 0x44, 0xe1, 0x80, 0x72, 0xab,
			0xd4, 0x5c, 0x13, 0x92, 0x01, 0x5e, 0xe7, 0x16,
			0x92, 0x69, 0x0e, 0xf8, 0xcd, 0xaa, 0xed, 0x33,
			0x7d, 0xd8, 0x54, 0x67, 0x83, 0xf9, 0x61, 0xbb,
			0x96, 0x20, 0xeb, 0x5c, 0x7b, 0x8b, 0x67, 0x16,
			0xe8, 0xc6, 0x00, 0x35, 0x1f, 0xab, 0x77, 0x65,
			0xee, 0x38, 0xa1, 0x5d, 0x32, 0xd8, 0xa2, 0xc0,
			0x94, 0x98, 0x25, 0xc4, 0x9a, 0x7f, 0x25, 0xee,
			0xdd, 0x9b, 0xe7, 0xb8, 0x07, 0xbb, 0xfd, 0x51,
			0x79, 0x13, 0x78, 0x66, 0x20, 0xd2, 0x49, 0x82,
			0x3d, 0xae, 0x6f, 0xe2, 0xfd, 0x39, 0xac, 0x63,
			0x9d, 0xd7, 0x48, 0x21, 0xb0, 0xc1, 0x20, 0xb4,
			0x2f, 0x31, 0xc2, 0xc6, 0x39, 0xd2, 0xc6, 0x1b,
			0x39, 0x5f, 0x09, 0xf8, 0x68, 0x51, 0xbc, 0x80,
			0x9b, 0x34, 0xc4, 0x98, 0x1a, 0xc6, 0x5c, 0xf2,
			0x5b, 0x2e, 0x8a, 0xdc, 0xbc, 0xe1, 0x90, 0xef,
			0x2e, 0xf6, 0x7a, 0x01, 0x89, 0x03, 0x9c, 0x91,
			0x10, 0xf2, 0x67, 0x01, 0xc3, 0xee, 0xd7, 0x31,
			0xc8, 0xd9, 0xea, 0xd1, 0x78, 0x22, 0x0f, 0xfc,
			0xac, 0x7f, 0x0f, 0x67, 0x8a, 0xa2, 0x22, 0x68,
			0xe1, 0xd0, 0x19, 0x42, 0xec, 0x51, 0xe8, 0x0e,
			0xef, 0x06, 0xe2, 0x11, 0x28, 0x30, 0x85, 0x5e,
			0x87, 0xba, 0xfe, 0x8c, 0xc9, 0xc2, 0x2f, 0xd7,
			0x37, 0xc7, 0xab, 0xbc, 0xa5, 0xeb, 0x7a, 0x22,
			0x1d, 0x38, 0x35, 0xa8, 0x66, 0x10, 0xd2, 0x4b,
			0x50, 0x7b, 0x5d, 0xcb, 0x46, 0x18, 0xaa, 0x42,
			0x1f, 0x63, 0xa5, 0x60, 0x9e, 0xf5, 0xd6, 0x8f,
			0x57, 0x60, 0xfd, 0xdf, 0x97, 0x01, 0x35, 0x60,
			0x2e, 0xfa, 0xd0, 0x85, 0x1b, 0xbf, 0xf9, 0x8f,
			0xe8, 0x7f, 0xa5, 0x8b, 0xc3, 0x65, 0xf3, 0x8e,
			0xe7, 0xec, 0x8e, 0xf5, 0xaa, 0xb1, 0x7f, 0xd1,
			0x1d, 0x89, 0xd9, 0x1e, 0xf4, 0xc6, 0x04, 0xe0,
			0xd1, 0xf0, 0x01, 0xd0, 0xe0, 0x88, 0x69, 0xdf,
			0x92, 0x25, 0xe3, 0xb4, 0xce, 0xf5, 0x2f, 0xf8,
			0x68, 0x15, 0xe1, 0x3b, 0x3e, 0xfd, 0xf4, 0x57,
			0x76, 0xf9, 0x35, 0x37, 0x69, 0xa8, 0xa5, 0x1f,
			0xe7, 0xd8, 0x91, 0xa7, 0xef, 0x70, 0x35, 0xee,
			0xcf, 0xa2, 0x59, 0x84, 0x87, 0x38, 0x37, 0x68,
			0x86, 0xed, 0xc9, 0x1c, 0xc7, 0x8f, 0x6d, 0xa3,
			0x1c, 0x2f, 0x07, 0xee, 0x36, 0x2c, 0x3d, 0x82,
		},
	},
	/* SHA256 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0x2b, 0xca, 0xd6, 0xe7, 0x44, 0xf2, 0x49, 0x0b,
			0xa6, 0xa6, 0xe0, 0x72, 0x28, 0x32, 0x41, 0x7e,
			0xbd, 0x91, 0x0f, 0x91, 0x46, 0xeb, 0x62, 0xba,
			0xaa, 0x5c, 0x74, 0x95, 0x29, 0xf7, 0x9d, 0x6c,
			0xed, 0x0b, 0x81, 0xa2, 0xe2, 0xa4, 0x88, 0x52,
			0xc8, 0x55, 0x8e, 0x33, 0x87, 0x35, 0xdc, 0xbf,
			0xc2, 0x28, 0x57, 0x94, 0xae, 0x60, 0xf8, 0x1a,
			0x25, 0x23, 0x7c, 0x66, 0xf6, 0xce, 0x5d, 0x5e,
			0x80, 0x1a, 0x00, 0x1e, 0x7f, 0x9e, 0x30, 0x9b,
			0x25, 0x95, 0xcb, 0x86, 0x6d, 0xe2, 0xbb, 0x74,
			0xac, 0x51, 0x28, 0x3b, 0x68, 0x20, 0xec, 0x9f,
			0x6e, 0xbe, 0x48, 0x2e, 0x1f, 0xd2, 0xd5, 0x68,
			0x0b, 0x7f, 0xbd, 0x23, 0xc1, 0xe6, 0x2a, 0x2e,
			0xe4, 0xed, 0xff, 0x35, 0x82, 0x3f, 0xc7, 0xe4,
			0xa2, 0x95, 0xea, 0x4f, 0x1c, 0x33, 0x27, 0x92,
			0xae, 0xb5, 0x3e, 0xb4, 0x4b, 0x0b, 0xed, 0xd2,
		},
		/* S1 : 384 byte */
		{
			0x37, 0xd9, 0x60, 0xfe, 0x39, 0x12, 0x98, 0xbb,
			0xdc, 0x22, 0x3f, 0xa1, 0xeb, 0x1d, 0x3c, 0xd9,
			0xa4, 0x6b, 0xa8, 0xc6, 0x2e, 0x1d, 0xa8, 0xc5,
			0x63, 0xc8, 0x9a, 0x8f, 0x0e, 0x67, 0xb8, 0x64,
			0xfc, 0x89, 0x83, 0x7f, 0xfc, 0x08, 0xaa, 0xb7,
			0x12, 0x2b, 0x84, 0xc4, 0x35, 0xc7, 0xf9, 0x40,
			0x6e, 0x16, 0x5a, 0x10, 0x29, 0x85, 0x7c, 0x1e,
			0x4d, 0xea, 0x65, 0x35, 0x69, 0x27, 0x72, 0x73,
			0xb1, 0xd9, 0xb0, 0xa9, 0xf5, 0xb0, 0xdc, 0x24,
			0xaf, 0xdd, 0x21, 0x44, 0x76, 0xd4, 0x72, 0x08,
			0xad, 0x52, 0x21, 0xa7, 0xd7, 0x93, 0xca, 0xb8,
			0x06, 0x71, 0xfb, 0x49, 0x87, 0xc8, 0x6b, 0xd6,
			0x14, 0x48, 0x80, 0xc5, 0x9d, 0x24, 0x87, 0x14,
			0x00, 0xf6, 0x4b, 0xdc, 0x6d, 0x49, 0x6d, 0xbd,
			0x49, 0x7f, 0x3d, 0xbf, 0x64, 0x28, 0x64, 0xfe,
			0x49, 0xaf, 0x3e, 0x21, 0x51, 0x5e, 0x62, 0xd6,
			0x0f, 0x00, 0x71, 0xdb, 0x48, 0x84, 0xf4, 0x96,
			0x70, 0xea, 0xa9, 0xe4, 0xe4, 0x98, 0x2f, 0x26,
			0x9a, 0xbe, 0x72, 0x42, 0x44, 0x28, 0x88, 0x59,
			0xc2, 0xad, 0xf6, 0x0a, 0x09, 0xfa, 0xaa, 0xbb,
			0x07, 0x99, 0x0e, 0x09, 0xe5, 0x6d, 0xe2, 0x54,
			0xba, 0xbb, 0xee, 0x14, 0xbe, 0x7e, 0xb6, 0xed,
			0xa0, 0xcd, 0xb2, 0x2f, 0x3d, 0x0d, 0xe8, 0x72,
			0x48, 0x04, 0x67, 0x3f, 0xb9, 0x9f, 0x86, 0xef,
			0xb4, 0x26, 0x3d, 0xcc, 0x50, 0x17, 0xab, 0xc9,
			0x1b, 0xd9, 0xcd, 0x83, 0x36, 0x79, 0x47, 0x5b,
			0xfa, 0xc5, 0x0a, 0x2b, 0xe8, 0xdb, 0x86, 0x29,
			0x6b, 0xbf, 0x80, 0x17, 0x88, 0x93, 0x57, 0x37,
			0x13, 0x14, 0x60, 0x4e, 0x83, 0xd6, 0x8b, 0x6e,
			0xfe, 0xcd, 0x4b, 0x79, 0xf0, 0xa8, 0xaf, 0xa0,
			0xdf, 0xfa, 0x44, 0x8f, 0xb7, 0xfc, 0xe6, 0xd3,
			0x44, 0x70, 0x9a, 0x67, 0x0e, 0x0c, 0xff, 0x43,
			0x2c, 0x3e, 0x18, 0x7b, 0xcf, 0xf7, 0xfd, 0xc4,
			0xf4, 0xe9, 0xab, 0xe1, 0x09, 0x5c, 0x46, 0xb0,
			0x1d, 0x88, 0xb6, 0x04, 0x4b, 0xb9, 0x50, 0xe9,
			0x28, 0x59, 0x01, 0x0d, 0x9a, 0x0e, 0x3b, 0x2d,
			0x1f, 0x27, 0xa0, 0x96, 0xea, 0xca, 0xa2, 0x42,
			0x63, 0xa2, 0xa0, 0x52, 0x3d, 0x6e, 0x0d, 0xa1,
			0xfb, 0xa8, 0xaf, 0x76, 0x81, 0x96, 0xf7, 0xa5,
			0x1f, 0x92, 0xfd, 0xf1, 0x52, 0xbe, 0xf0, 0x62,
			0xdd, 0x1f, 0x83, 0x27, 0xce, 0xe1, 0xd3, 0x44,
			0xc2, 0x00, 0xc2, 0x11, 0x5a, 0xc6, 0xec, 0x1d,
			0xd8, 0x51, 0x4c, 0xef, 0x9e, 0x36, 0xd0, 0xce,
			0x8c, 0x32, 0xe5, 0x87, 0x83, 0xc4, 0xfc, 0xba,
			0x90, 0x1a, 0xa7, 0x0c, 0x2b, 0x42, 0x96, 0x64,
			0x88, 0x00, 0x2f, 0xf1, 0x71, 0xd3, 0x64, 0x14,
			0xa1, 0x44, 0xbf, 0x46, 0x77, 0x51, 0x83, 0xa8,
			0x81, 0x5d, 0xe9, 0xee, 0x3e, 0x81, 0xf3, 0x1b,
		},
	},
	/* SHA256 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0xc3, 0x97, 0x8b, 0xd0, 0x50, 0xd4, 0x6d, 0xa4,
			0xa7, 0x92, 0x27, 0xd8, 0x27, 0x0a, 0x22, 0x02,
			0x95, 0x34, 0x82, 0x87, 0x59, 0x30, 0xfb, 0x1a,
			0xea, 0xe4, 0xe6, 0x7f, 0x87, 0xe7, 0x94, 0x95,
			0x28, 0x9d, 0xe2, 0x93, 0xb4, 0xa4, 0x0d, 0x92,
			0x74, 0x6f, 0xc8, 0x4c, 0xc8, 0x31, 0x8c, 0x23,
			0x18, 0xfd, 0x30, 0x65, 0x0e, 0x2b, 0xb9, 0xce,
			0x02, 0xfd, 0x73, 0x4e, 0xb6, 0x83, 0x41, 0x0d,
			0x44, 0xbb, 0x31, 0xad, 0x54, 0xfd, 0x53, 0xcf,
			0x92, 0x96, 0xcc, 0xd8, 0x60, 0xb4, 0x26, 0xf5,
			0xc7, 0x82, 0xea, 0x5c, 0xb4, 0x93, 0x71, 0xd5,
			0x61, 0x84, 0xf7, 0x79, 0x11, 0xdd, 0xf1, 0xba,
			0x00, 0x39, 0xa0, 0xa4, 0x9a, 0xa7, 0xe7, 0x63,
			0xeb, 0x4f, 0x5a, 0x04, 0x57, 0x59, 0x97, 0x80,
			0x8b, 0x0a, 0xd9, 0xf6, 0xb3, 0x30, 0xca, 0x38,
			0xed, 0xc1, 0x99, 0x89, 0xfe, 0xbf, 0x4d, 0xa5,
		},
		/* S2 : 384 byte */
		{
			0x9a, 0xed, 0x20, 0xa8, 0xbd, 0xaf, 0x26, 0xf1,
			0xf1, 0x19, 0x02, 0x0d, 0x8f, 0x3e, 0xa6, 0xce,
			0x91, 0x51, 0x38, 0xd4, 0xc8, 0x7d, 0xce, 0x02,
			0x5e, 0x7f, 0x4e, 0x49, 0x53, 0x6c, 0x8e, 0xc0,
			0x79, 0xed, 0xc6, 0xca, 0xf0, 0xd6, 0x03, 0xbf,
			0x42, 0xbd, 0x6a, 0x45, 0x4a, 0x6d, 0x52, 0xd0,
			0xd9, 0x9f, 0xd0, 0xf5, 0x9f, 0xfb, 0x3b, 0x22,
			0xe9, 0xe6, 0x7b, 0x3d, 0x0b, 0xb2, 0xd2, 0x75,
			0xd9, 0xae, 0xdc, 0x6d, 0xa9, 0x6a, 0x72, 0xcb,
			0xff, 0x35, 0xc4, 0x3e, 0x7f, 0x39, 0xa9, 0x96,
			0xfa, 0x8a, 0x6d, 0x33, 0x8a, 0x07, 0x25, 0xf7,
			0x85, 0x25, 0x4f, 0xe9, 0x1a, 0x20, 0x83, 0x4b,
			0xa5, 0x57, 0xfe, 0xdf, 0xe7, 0x15, 0x2b, 0x99,
			0x56, 0xfe, 0xdd, 0xfd, 0x94, 0x17, 0x41, 0xef,
			0xf9, 0x17, 0x7c, 0x2f, 0xbb, 0x55, 0xe2, 0x00,
			0xbb, 0xe4, 0x21, 0x62, 0xb3, 0x2a, 0x94, 0x0c,
			0xc3, 0x00, 0xab, 0x37, 0x55, 0x57, 0xdf, 0xfd,
			0x48, 0xdf, 0xa5, 0x39, 0xf5, 0x0e, 0xdd, 0x52,
			0xdf, 0x15, 0x8d, 0x90, 0x72, 0xd1, 0x49, 0x82,
			0xe9, 0x63, 0x03, 0xbc, 0x61, 0x2c, 0x2c, 0x25,
			0x06, 0xdb, 0xca, 0x3a, 0x93, 0x9d, 0x62, 0x6d,
			0x2e, 0x7f, 0xb4, 0x44, 0xc6, 0xad, 0x7d, 0x8d,
			0x9f, 0x3b, 0xba, 0x82, 0x10, 0xb2, 0xac, 0x2f,
			0x69, 0x67, 0x83, 0xc3, 0x49, 0xfc, 0x52, 0x80,
			0xc1, 0x05, 0x40, 0x2a, 0x4b, 0x3d, 0x86, 0xbe,
			0xf5, 0x02, 0x6c, 0x3d, 0xd9, 0x99, 0xe3, 0xb2,
			0x23, 0x80, 0xf9, 0xdc, 0xce, 0x40, 0xe3, 0xa9,
			0xcc, 0x9f, 0x1d, 0x7b, 0xc3, 0x8e, 0xf3, 0xdd,
			0x7e, 0x94, 0x13, 0xbb, 0x57, 0x98, 0x00, 0xc0,
			0xe6, 0xc3, 0xe9, 0xab, 0x91, 0x2d, 0xa8, 0xfe,
			0xc1, 0xa4, 0xab, 0x21, 0x39, 0x8e, 0x96, 0x80,
			0xba, 0x0d, 0x04, 0xf3, 0xb4, 0xc8, 0xd5, 0x3c,
			0x02, 0xf0, 0x5c, 0x7a, 0xe4, 0x9b, 0x70, 0xa5,
			0x61, 0x1c, 0xf8, 0x2e, 0x38, 0xde, 0x84, 0xaa,
			0x8c, 0x24, 0x26, 0xf0, 0xb6, 0x3e, 0xa0, 0x1b,
			0x28, 0x9f, 0x20, 0x1d, 0x3a, 0xf4, 0x0d, 0xad,
			0x5d, 0x6e, 0x5b, 0xcc, 0xc7, 0x5b, 0x99, 0x59,
			0xe5, 0xc9, 0x75, 0x8e, 0x79, 0x10, 0x5a, 0xf7,
			0xa9, 0xaf, 0xb1, 0x2a, 0xee, 0x57, 0x7c, 0xb3,
			0x99, 0x18, 0x79, 0xdb, 0x0f, 0xd8, 0x66, 0x2c,
			0x5b, 0xc4, 0x90, 0x22, 0x75, 0x24, 0x98, 0xa3,
			0x01, 0xd9, 0x5f, 0x4b, 0x1d, 0x08, 0xc0, 0x1e,
			0xbc, 0x31, 0x3f, 0x89, 0xc0, 0x0b, 0x1e, 0xc2,
			0x73, 0x5a, 0x07, 0x98, 0x3f, 0xd5, 0x28, 0xe6,
			0x38, 0x82, 0x45, 0x03, 0x6f, 0x0e, 0xd4, 0xa2,
			0xdb, 0xb6, 0x5d, 0xd3, 0x3a, 0xb7, 0xf1, 0x24,
			0xc0, 0x14, 0xec, 0x16, 0x79, 0xf1, 0xc2, 0xf1,
			0x1e, 0xdf, 0xfb, 0x93, 0xfa, 0x2d, 0x1d, 0x73,
		},
	},
	/* SHA256 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0x0c, 0x11, 0x95, 0x02, 0xc2, 0xa0, 0x19, 0x20,
			0xa0, 0x90, 0xe4, 0x33, 0x57, 0xe7, 0xb2, 0x8e,
			0x33, 0xc7, 0xee, 0x85, 0x8b, 0x43, 0x30, 0xe0,
			0x5c, 0x71, 0x04, 0x89, 0x31, 0xc0, 0xed, 0x88,
			0x46, 0x8c, 0xa9, 0x31, 0xec, 0xf0, 0xb7, 0x9c,
			0x2f, 0xdc, 0x17, 0x56, 0xb7, 0x67, 0x51, 0x56,
			0xec, 0x66, 0xb8, 0x33, 0x5e, 0x3d, 0xf0, 0x94,
			0x63, 0xf5, 0xae, 0xe7, 0x02, 0x8f, 0xbf, 0x56,
			0x0f, 0x98, 0x4c, 0xf6, 0x98, 0xfe, 0x5c, 0x42,
			0x80, 0x22, 0x9a, 0xc9, 0x6a, 0x2e, 0x59, 0x23,
			0xd8, 0xa9, 0xd5, 0x29, 0x94, 0x49, 0xbb, 0x66,
			0x50, 0x08, 0xec, 0xc8, 0x89, 0x79, 0x7e, 0x9b,
			0xb1, 0x5d, 0x04, 0xb8, 0x8c, 0x72, 0x10, 0xfa,
			0xdb, 0x8b, 0xf6, 0xf2, 0x38, 0xe5, 0xd2, 0xdc,
			0x41, 0xb9, 0xcc, 0xd1, 0xf8, 0x0e, 0x9a, 0x3e,
			0x6a, 0xd1, 0x47, 0x94, 0x8f, 0x27, 0x33, 0x41,
		},
		/* S3 : 384 byte */
		{
			0x8a, 0xbf, 0x2a, 0x30, 0x77, 0x4e, 0x6e, 0x73,
			0x38, 0xec, 0xa0, 0x9c, 0xcc, 0xac, 0xa3, 0x68,
			0x43, 0x99, 0x94, 0x04, 0x92, 0xfb, 0x94, 0xb2,
			0x3b, 0x5a, 0xd6, 0x2c, 0xe3, 0xe1, 0x1d, 0x2d,
			0xbe, 0xf8, 0x96, 0x6b, 0xa5, 0x26, 0x99, 0x79,
			0xeb, 0x96, 0x53, 0xba, 0xad, 0x71, 0x95, 0x16,
			0xd3, 0xe8, 0x39, 0x90, 0x79, 0xa2, 0xf6, 0x70,
			0x27, 0x5a, 0x2e, 0xd4, 0x2c, 0x82, 0x0a, 0x9a,
			0x31, 0xfc, 0xd7, 0x03, 0xa7, 0x66, 0x37, 0xe0,
			0xd7, 0x13, 0xf3, 0x2d, 0x79, 0x2b, 0x9a, 0xe3,
			0x6d, 0x72, 0x88, 0xf6, 0x0c, 0x2d, 0x1a, 0xe5,
			0x26, 0x83, 0xbb, 0x15, 0x94, 0x1b, 0x1c, 0xd8,
			0x90, 0xd2, 0xcd, 0x64, 0x99, 0x8b, 0x77, 0x25,
			0x85, 0xe7, 0x60, 0x32, 0xa1, 0x70, 0x2e, 0x06,
			0x52, 0xcb, 0xf2, 0x59, 0xa1, 0xce, 0xae, 0x69,
			0x5d, 0x40, 0xcf, 0x2f, 0x4f, 0x6d, 0x81, 0x34,
			0x1c, 0x8b, 0xc9, 0x08, 0x2c, 0xb9, 0x6c, 0x75,
			0x2c, 0x35, 0x5d, 0xfb, 0xe2, 0x96, 0xdd, 0x21,
			0xd6, 0x98, 0x46, 0xfa, 0x37, 0x61, 0x3e, 0x73,
			0x81, 0x7b, 0x2a, 0x07, 0x04, 0x66, 0x58, 0xc9,
			0xe3, 0xfc, 0x6d, 0x09, 0x1e, 0x17, 0x59, 0x1b,
			0xb1, 0xa4, 0xfb, 0x6e, 0x2a, 0xc0, 0x0a, 0x31,
			0x94, 0xc1, 0x48, 0x8e, 0x16, 0xa9, 0xd2, 0x90,
			0x37, 0x86, 0xdb, 0x86, 0xae, 0x90, 0xe9, 0x6a,
			0xcb, 0x4d, 0xe9, 0x90, 0x1a, 0xaf, 0x1b, 0x06,
			0x51, 0xfb, 0x76, 0xa5, 0x8d, 0xcb, 0x3d, 0xb4,
			0x73, 0xef, 0xbf, 0xb8, 0x31, 0xef, 0x8e, 0x30,
			0xf8, 0x99, 0x67, 0xdd, 0xd3, 0xa6, 0xc2, 0xf1,
			0x89, 0x79, 0xa0, 0x45, 0x06, 0x57, 0xcd, 0xae,
			0xef, 0x6e, 0x59, 0x37, 0x7c, 0x6d, 0xb1, 0xec,
			0x46, 0x06, 0x5f, 0x61, 0x40, 0x24, 0xa6, 0x9c,
			0x51, 0x8a, 0x55, 0x99, 0x42, 0x59, 0x4a, 0x46,
			0x26, 0x6e, 0x0d, 0x3c, 0xa1, 0x33, 0x42, 0x96,
			0xb9, 0x68, 0xa2, 0x3a, 0x4b, 0x11, 0xc6, 0x3a,
			0x97, 0xe2, 0x9e, 0xb1, 0x6b, 0x24, 0xc0, 0x2d,
			0x54, 0x5d, 0x5b, 0x42, 0x7e, 0x6a, 0xa5, 0x85,
			0x33, 0x33, 0x18, 0xe6, 0x3a, 0x20, 0x45, 0x24,
			0xe0, 0xe4, 0x2a, 0xc1, 0xed, 0xb7, 0x0d, 0x34,
			0x56, 0x78, 0x0d, 0xbe, 0xad, 0x31, 0xf7, 0x85,
			0xf0, 0xb2, 0xa7, 0x7f, 0xfe, 0xb0, 0xd3, 0x73,
			0x84, 0xcb, 0x5f, 0x65, 0xb4, 0xe3, 0x6c, 0xa2,
			0x41, 0xf3, 0xb2, 0xb0, 0x59, 0x10, 0x5f, 0xaa,
			0xa3, 0x22, 0x2d, 0x6c, 0x13, 0x5e, 0xa5, 0xa3,
			0x66, 0x51, 0xae, 0xa3, 0x96, 0xd2, 0x2f, 0xc4,
			0xea, 0x1b, 0x40, 0x4d, 0x7e, 0x83, 0x4b, 0x6d,
			0xf1, 0xfb, 0x83, 0x8b, 0xb5, 0xba, 0x0d, 0x78,
			0x4a, 0x96, 0xe2, 0xae, 0x28, 0x43, 0xdb, 0x3e,
			0xee, 0xa4, 0x96, 0xc7, 0xad, 0x2b, 0x42, 0x41,
		},
	},
	/* SHA256 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0xdd, 0xbd, 0x84, 0x68, 0xbd, 0xb0, 0x36, 0xf4,
			0x79, 0x9f, 0x42, 0x8b, 0xc8, 0xb4, 0x37, 0x4e,
			0xd9, 0xb7, 0xcd, 0xe5, 0x41, 0x33, 0x7a, 0xc4,
			0x39, 0xd4, 0x41, 0xac, 0x06, 0x14, 0xcb, 0x75,
			0xb8, 0x16, 0xb8, 0x0c, 0x17, 0xd2, 0x37, 0xb8,
			0xdb, 0x73, 0xd4, 0xa1, 0x1b, 0xfd, 0x92, 0x92,
			0x08, 0x33, 0x3a, 0xfe, 0xdb, 0xb8, 0xf2, 0x41,
			0x0c, 0x74, 0x11, 0x29, 0xc5, 0x39, 0x32, 0xb5,
			0x96, 0xa7, 0x88, 0x1c, 0x6a, 0x4d, 0x71, 0x11,
			0xba, 0x10, 0x4d, 0x46, 0x00, 0xd1, 0x90, 0x2f,
			0x6f, 0x4a, 0x16, 0x08, 0xe1, 0x39, 0xb7, 0x19,
			0x11, 0xc1, 0x1c, 0x39, 0x0a, 0x0d, 0xd0, 0x91,
			0xdf, 0x36, 0x9a, 0xa2, 0x9d, 0x67, 0x0b, 0x8a,
			0x7e, 0x3f, 0x53, 0x82, 0x5f, 0x76, 0x59, 0xac,
			0x74, 0xc4, 0x0a, 0x0c, 0x3b, 0xfe, 0xf0, 0xd3,
			0xae, 0x83, 0x07, 0xe4, 0xbd, 0xd6, 0xcd, 0x91,
		},
		/* S4 : 384 byte */
		{
			0x4e, 0x37, 0x7e, 0x24, 0x59, 0x81, 0x5d, 0x5b,
			0x33, 0x91, 0x5f, 0xa6, 0x3c, 0xd4, 0x77, 0xb5,
			0xbe, 0x7c, 0x6b, 0x7f, 0x78, 0x14, 0xd1, 0x35,
			0x00, 0x34, 0xce, 0x71, 0x0b, 0xe6, 0x7e, 0xd6,
			0x91, 0x39, 0xdb, 0x62, 0x2e, 0xf6, 0x0e, 0xc6,
			0xb7, 0x63, 0x8e, 0x94, 0xb2, 0x02, 0x36, 0x8b,
			0xac, 0x63, 0x1e, 0x05, 0x77, 0x02, 0xb0, 0xe6,
			0x48, 0x7b, 0x32, 0x4a, 0x6b, 0x98, 0xed, 0x7e,
			0x03, 0xd1, 0xf3, 0xf2, 0x0a, 0x98, 0x14, 0xb0,
			0x0e, 0x21, 0x7a, 0x46, 0x48, 0xe4, 0xbb, 0xc4,
			0x49, 0xa2, 0xaf, 0x40, 0x5c, 0xa4, 0xb5, 0x9f,
			0x84, 0x38, 0xdd, 0xfd, 0x75, 0xd3, 0x4d, 0x10,
			0x64, 0xe5, 0x8b, 0xfb, 0x32, 0x5c, 0x55, 0xbd,
			0x54, 0xea, 0x6c, 0xdf, 0x77, 0x12, 0xba, 0x80,
			0x7c, 0x3e, 0x4c, 0x66, 0x5d, 0x62, 0x0c, 0xd5,
			0x95, 0x13, 0xd7, 0xbc, 0x08, 0x55, 0x24, 0x7e,
			0xb6, 0x70, 0xec, 0xc2, 0x92, 0x50, 0x96, 0x61,
			0x81, 0x27, 0x02, 0x70, 0x32, 0x75, 0xd9, 0xb2,
			0xf8, 0x7e, 0xf2, 0x79, 0xd7, 0x70, 0x0e, 0x69,
			0xd9, 0x95, 0xdb, 0x98, 0x14, 0x4a, 0x14, 0xc8,
			0x17, 0x74, 0xa4, 0xcd, 0x89, 0x0e, 0xc0, 0x3d,
			0x13, 0xf8, 0x58, 0xf3, 0x76, 0x9e, 0x50, 0x48,
			0xed, 0x55, 0xca, 0xa8, 0x12, 0x01, 0xe8, 0x78,
			0x5d, 0x37, 0x71, 0xce, 0x6d, 0xa5, 0x11, 0x75,
			0xd0, 0x17, 0xd2, 0x11, 0xfa, 0x70, 0x37, 0x94,
			0x41, 0x6f, 0x46, 0x9b, 0x11, 0x29, 0xd7, 0x31,
			0xab, 0xde, 0x74, 0x4d, 0xa5, 0xb2, 0xfa, 0xcd,
			0x7a, 0x9b, 0x09, 0x3d, 0x6c, 0x97, 0x43, 0x50,
			0x9b, 0x01, 0x03, 0xba, 0xb9, 0xc8, 0x1c, 0x6e,
			0x5f, 0x38, 0xbc, 0x97, 0x18, 0xe3, 0xe4, 0xfa,
			0xa8, 0x64, 0x75, 0xd1, 0x37, 0x25, 0xa8, 0x29,
			0xac, 0x61, 0xdf, 0x8d, 0x15, 0xf0, 0xb2, 0x7c,
			0xb4, 0x0d, 0x0e, 0xba, 0x0b, 0x24, 0x6b, 0x9c,
			0x36, 0x0b, 0x56, 0x9b, 0x81, 0xb3, 0xab, 0xf3,
			0x80, 0xee, 0xc2, 0x74, 0x92, 0x31, 0x6b, 0xc2,
			0x92, 0xe5, 0x15, 0x0e, 0xe0, 0x60, 0x72, 0x19,
			0xa2, 0xbd, 0x80, 0xba, 0x98, 0x4c, 0x7e, 0x3f,
			0x19, 0x89, 0xbc, 0x51, 0xe4, 0xc5, 0xda, 0x3a,
			0xe5, 0x07, 0x06, 0x76, 0xe0, 0xc1, 0x50, 0xd0,
			0x37, 0xa8, 0x6a, 0x0f, 0x91, 0xbf, 0xc0, 0x7c,
			0xde, 0x64, 0xc1, 0x9f, 0x9c, 0x7a, 0x7a, 0xf4,
			0x4d, 0x69, 0x29, 0x97, 0x00, 0x41, 0x44, 0x8d,
			0x3b, 0x17, 0xc2, 0x49, 0xd5, 0xe0, 0xb5, 0x86,
			0x2e, 0x9a, 0x25, 0x20, 0x9e, 0x8f, 0x97, 0xd7,
			0xa0, 0xf0, 0x30, 0x18, 0x15, 0x04, 0xfe, 0xad,
			0x22, 0x66, 0xc8, 0x73, 0xfd, 0x23, 0x59, 0x83,
			0xdf, 0x3d, 0x06, 0x57, 0xb9, 0x20, 0x96, 0xe2,
			0xb4, 0x90, 0xdf, 0x33, 0xca, 0x11, 0x57, 0x33,
		},
	},
	/* SHA256 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0xf9, 0x96, 0xf3, 0xad, 0xc2, 0xab, 0xa5, 0x05,
			0xad, 0x4a, 0xe5, 0x2b, 0xc5, 0xa4, 0x33, 0x71,
			0xa3, 0x3d, 0x0f, 0x28, 0xe1, 0x95, 0x0b, 0x66,
			0xd2, 0x08, 0x24, 0x06, 0x70, 0xf3, 0x52, 0xef,
			0x96, 0x18, 0x5e, 0x9a, 0x70, 0x44, 0xf4, 0xce,
			0x2f, 0x2f, 0xf9, 0xae, 0x01, 0xa3, 0x1e, 0xf6,
			0x40, 0xe0, 0xb6, 0x82, 0xe9, 0x40, 0xc5, 0x10,
			0x51, 0x17, 0x59, 0x46, 0x13, 0xdd, 0x1d, 0xf7,
			0x4d, 0x8f, 0x2b, 0xa2, 0x0c, 0x52, 0x22, 0x3b,
			0x04, 0x5a, 0x78, 0x2e, 0x85, 0x0a, 0x12, 0xa2,
			0xaa, 0x5c, 0x12, 0xfa, 0xd4, 0x84, 0xf1, 0xa2,
			0x56, 0xd0, 0xcd, 0x08, 0x72, 0xd3, 0x04, 0xe8,
			0x85, 0xc2, 0x01, 0xcd, 0x7e, 0x1e, 0x56, 0xd5,
			0x94, 0x93, 0x0b, 0xb4, 0x39, 0x21, 0x36, 0xfb,
			0x49, 0x79, 0xcc, 0x9b, 0x88, 0xaa, 0xb7, 0xa4,
			0x4b, 0xfc, 0x29, 0x53, 0x75, 0x1c, 0x2f, 0x4c,
		},
		/* S5 : 384 byte */
		{
			0x30, 0xb3, 0x48, 0x62, 0x4f, 0xaa, 0x99, 0x85,
			0xfc, 0xd9, 0x5f, 0x9c, 0x7e, 0xad, 0x3a, 0xfe,
			0x64, 0x56, 0xba, 0xdf, 0x8c, 0x0f, 0xed, 0xbd,
			0xad, 0xb3, 0xa9, 0x00, 0x3a, 0x67, 0x02, 0x97,
			0x3a, 0xcd, 0xb4, 0xe8, 0x66, 0x52, 0x36, 0x7d,
			0xb2, 0x3e, 0x0a, 0x81, 0x41, 0x88, 0x0d, 0x66,
			0x31, 0x83, 0x4f, 0x9f, 0x17, 0x1c, 0x94, 0xa8,
			0xfe, 0x9c, 0x31, 0x5b, 0xcb, 0x86, 0x80, 0xec,
			0xfb, 0x5a, 0x4f, 0x59, 0xb4, 0x5d, 0x4e, 0x4c,
			0x3c, 0x05, 0x82, 0x8b, 0x7f, 0xaa, 0xa8, 0xe4,
			0x23, 0x4a, 0xad, 0xa4, 0xe7, 0x66, 0x64, 0x6c,
			0xc5, 0x10, 0xd0, 0x7b, 0x42, 0xbd, 0x38, 0x83,
			0xa8, 0x3b, 0x5b, 0xcb, 0x92, 0xd9, 0xe7, 0xcc,
			0x1d, 0xdf, 0x59, 0x0a, 0x69, 0x01, 0x11, 0xbf,
			0xc6, 0x2a, 0x51, 0xaf, 0x7e, 0x55, 0x54, 0x3e,
			0xa5, 0x18, 0x8c, 0x92, 0x45, 0x3d, 0x41, 0xd3,
			0xe8, 0xfd, 0xab, 0xee, 0x3e, 0x1d, 0xef, 0xa9,
			0xd0, 0xaf, 0xdb, 0x85, 0xc8, 0x15, 0x3a, 0x50,
			0x19, 0xae, 0x45, 0x56, 0x3e, 0xa3, 0x08, 0x0a,
			0x30, 0x22, 0x66, 0x81, 0x68, 0xf0, 0xc2, 0x73,
			0xa6, 0xdb, 0x1a, 0xfa, 0xdc, 0xd5, 0xed, 0xbc,
			0xa5, 0x02, 0x1c, 0x2e, 0x53, 0xf4, 0xd9, 0x51,
			0xc6, 0x04, 0x20, 0x6a, 0xe1, 0x0f, 0x28, 0x7f,
			0x45, 0x18, 0x67, 0x27, 0x1d, 0x37, 0x04, 0x82,
			0x79, 0x1c, 0xdf, 0xdc, 0xb6, 0xa4, 0x01, 0x0f,
			0x6b, 0x3d, 0x9b, 0x92, 0x85, 0x63, 0xd1, 0x68,
			0xda, 0x19, 0xf1, 0xc1, 0xe5, 0x70, 0xf8, 0xc1,
			0x58, 0xf3, 0xd4, 0x90, 0xb2, 0x9a, 0xa2, 0x3a,
			0xbd, 0x1f, 0xfd, 0xf2, 0x08, 0x66, 0xc3, 0x4c,
			0x6e, 0x63, 0xb9, 0xe8, 0xa9, 0xa0, 0x2d, 0x7a,
			0x1b, 0x19, 0x6d, 0x05, 0x5f, 0x4c, 0x53, 0xce,
			0x82, 0xb4, 0x00, 0xe4, 0xab, 0x9e, 0x1b, 0x9d,
			0x70, 0xd0, 0x04, 0x9d, 0x6d, 0x57, 0xcf, 0x0a,
			0x49, 0x49, 0xcf, 0xc6, 0x8d, 0x63, 0x38, 0x82,
			0x88, 0x2d, 0xcf, 0xdf, 0xc5, 0x0c, 0xf4, 0x49,
			0xdf, 0x10, 0xac, 0xf2, 0x03, 0x05, 0xc2, 0xaa,
			0x43, 0xbd, 0xa1, 0x0f, 0xd8, 0xa1, 0x0b, 0x4e,
			0xca, 0xa2, 0x31, 0x00, 0xaa, 0x47, 0xe9, 0x29,
			0x36, 0xdc, 0xe1, 0xbf, 0xb8, 0xd6, 0x59, 0x52,
			0x35, 0xbb, 0xfe, 0x2c, 0x85, 0x85, 0xcb, 0x16,
			0x47, 0xb2, 0xbe, 0xac, 0xb1, 0xe1, 0xd4, 0xb6,
			0xce, 0xf7, 0x58, 0x81, 0x1a, 0x68, 0x33, 0x0f,
			0xa9, 0xc3, 0xa8, 0x25, 0x73, 0xc0, 0x8f, 0xa2,
			0xcd, 0xa5, 0xa0, 0x3f, 0x34, 0x25, 0x55, 0x4e,
			0x45, 0xd9, 0x8c, 0x16, 0x45, 0xc5, 0xbd, 0x27,
			0xd1, 0x2e, 0x6c, 0x20, 0xb2, 0xc4, 0x62, 0xa7,
			0x46, 0xe8, 0x82, 0xa3, 0x42, 0x1a, 0x7b, 0x1b,
			0x1e, 0x25, 0xb4, 0xc3, 0x6c, 0x8b, 0x16, 0xa1,
		},
	},
	/* SHA256 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0x6a, 0xce, 0x05, 0x2d, 0x7e, 0x99, 0xcd, 0x97,
			0x3b, 0xb5, 0xc9, 0xf6, 0x67, 0x9b, 0x1c, 0x30,
			0x5e, 0x07, 0x20, 0x89, 0x65, 0xfe, 0x58, 0xc6,
			0x3b, 0x10, 0xa6, 0x92, 0xf1, 0xdb, 0xbe, 0x22,
			0xfc, 0xd0, 0xdb, 0x15, 0x89, 0x3a, 0xb1, 0x9e,
			0x10, 0x7b, 0xa2, 0xe4, 0x2c, 0x99, 0x34, 0xa9,
			0xaa, 0xfa, 0xc3, 0x2a, 0xdf, 0x6c, 0x73, 0x47,
			0x3f, 0x69, 0x69, 0xe4, 0x2c, 0x98, 0x3b, 0x8f,
			0x0c, 0x96, 0xa4, 0x63, 0x9e, 0xf7, 0x7d, 0x2c,
			0x8e, 0x88, 0xe8, 0xcc, 0x47, 0xd7, 0xcf, 0xdd,
			0x08, 0xf6, 0x8d, 0x97, 0x3a, 0x7b, 0xea, 0xf4,
			0x01, 0xcb, 0x4d, 0x13, 0x11, 0x99, 0x2d, 0xda,
			0xc3, 0xa9, 0xc9, 0xe0, 0x67, 0xda, 0x19, 0x8a,
			0xdc, 0x63, 0x04, 0x74, 0x5f, 0x5d, 0xd3, 0x12,
			0xa1, 0x82, 0xe6, 0x97, 0x1c, 0x34, 0xa5, 0x15,
			0xa6, 0xc1, 0xba, 0xe6, 0x47, 0xe5, 0x7e, 0x4c,
		},
		/* S6 : 384 byte */
		{
			0x5f, 0x0e, 0x74, 0xf4, 0x54, 0x75, 0x4a, 0x30,
			0x74, 0xfa, 0xaf, 0xc6, 0x05, 0xf3, 0xc9, 0xaf,
			0x47, 0x60, 0x4a, 0x89, 0x83, 0x65, 0x0a, 0x9b,
			0x62, 0x11, 0xfb, 0x19, 0x1d, 0x9a, 0xfa, 0x53,
			0x15, 0xdf, 0x4d, 0xb4, 0x50, 0x1f, 0xd4, 0xf0,
			0x4c, 0x74, 0x1d, 0x76, 0x46, 0x56, 0xd4, 0xa5,
			0xd0, 0x06, 0x38, 0x8a, 0xd8, 0xfd, 0xb2, 0x19,
			0xec, 0x6b, 0x75, 0x69, 0x08, 0xe2, 0x3b, 0x30,
			0xcb, 0x63, 0x9f, 0xfa, 0x7b, 0xbf, 0x28, 0x74,
			0x71, 0x3b, 0xfd, 0x5a, 0x10, 0x62, 0xc1, 0x9d,
			0x04, 0xe0, 0xe4, 0xa7, 0x4b, 0x14, 0x44, 0x6a,
			0x7f, 0xdf, 0x5c, 0xb8, 0x12, 0xe9, 0xac, 0x7b,
			0x60, 0x12, 0xd9, 0xae, 0x99, 0x1c, 0x47, 0x65,
			0x6d, 0x2a, 0xde, 0xd2, 0x40, 0x74, 0xbb, 0x8a,
			0x38, 0xb1, 0xa8, 0x8b, 0x1c, 0x2b, 0x13, 0x1e,
			0x5b, 0x09, 0xc9, 0x37, 0x57, 0xfd, 0xb2, 0xd6,
			0xb6, 0x9a, 0xa8, 0x26, 0x5a, 0x43, 0x5f, 0xba,
			0x00, 0xae, 0xb3, 0x6a, 0x1f, 0x62, 0x9b, 0xc3,
			0x4b, 0x87, 0x60, 0x89, 0xd2, 0x8a, 0x94, 0x8d,
			0xd6, 0xab, 0x4c, 0x89, 0x94, 0x30, 0xda, 0x60,
			0xa2, 0x6f, 0x6c, 0x13, 0x60, 0x3f, 0xc8, 0x89,
			0xc7, 0xb2, 0x93, 0x6c, 0xa3, 0xc5, 0x15, 0x6b,
			0xd7, 0xfa, 0x6e, 0x34, 0xea, 0xc9, 0xe0, 0x48,
			0x00, 0x83, 0x3e, 0xf0, 0xcb, 0x9b, 0x6e, 0xef,
			0x78, 0x8c, 0x0e, 0xf0, 0x02, 0x1a, 0x45, 0x36,
			0xfb, 0x83, 0x71, 0xfa, 0x3e, 0x2c, 0x8b, 0xb8,
			0xbe, 0xfa, 0xc1, 0x6e, 0x80, 0x92, 0xd6, 0x9c,
			0x57, 0x1c, 0x1e, 0x15, 0xfd, 0x25, 0x5e, 0xc0,
			0xa0, 0x7a, 0xcf, 0x9a, 0xe9, 0x95, 0x38, 0x31,
			0xef, 0xd3, 0xdc, 0xbe, 0xf4, 0x4e, 0x0f, 0xcc,
			0xeb, 0xb1, 0xaf, 0x95, 0x9d, 0x71, 0xf5, 0x01,
			0x30, 0xe8, 0xac, 0xb4, 0xfa, 0x23, 0x19, 0x26,
			0x1f, 0xba, 0x12, 0xf2, 0x71, 0x5d, 0xef, 0x82,
			0xbf, 0xaf, 0xbf, 0x40, 0xe3, 0x45, 0xec, 0x5d,
			0xcd, 0xab, 0x5c, 0x1b, 0xf5, 0xf6, 0x6b, 0x1d,
			0x0e, 0x9f, 0x7a, 0x9c, 0x62, 0xc9, 0x37, 0x57,
			0x46, 0xe1, 0xae, 0x0c, 0x8f, 0x14, 0xa4, 0x89,
			0x18, 0x43, 0x83, 0xe8, 0x1d, 0xce, 0x20, 0x70,
			0xad, 0x4b, 0x52, 0x5d, 0xf7, 0x6b, 0x44, 0x6b,
			0x1f, 0x22, 0x92, 0x1d, 0x42, 0x4d, 0x9b, 0xa3,
			0xce, 0x21, 0x57, 0x75, 0x01, 0xdf, 0x62, 0x80,
			0xfd, 0xc6, 0x9f, 0x02, 0x39, 0xae, 0x11, 0x27,
			0xb6, 0x99, 0x50, 0x75, 0x9d, 0x5f, 0x0b, 0x69,
			0x3f, 0x54, 0xe8, 0x7e, 0x07, 0x63, 0x62, 0x3b,
			0xf5, 0xd3, 0xff, 0x69, 0x43, 0x00, 0x81, 0xb9,
			0xc9, 0xe2, 0x44, 0x5a, 0x05, 0xe1, 0x15, 0x67,
			0x5e, 0x09, 0x0b, 0xca, 0xb2, 0xaa, 0x1d, 0x75,
			0xce, 0xee, 0x2a, 0xd6, 0x19, 0xec, 0x8b, 0x80,
		},
	},
	/* SHA256 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0x0e, 0x49, 0x74, 0x0f, 0xdc, 0xca, 0x6b, 0xfc,
			0xe2, 0x94, 0xc1, 0x1f, 0x45, 0x40, 0x78, 0x05,
			0xb3, 0xda, 0x41, 0x2b, 0x01, 0xef, 0x3f, 0xb5,
			0x13, 0xe7, 0x0e, 0x62, 0xfd, 0x95, 0x04, 0xc0,
			0x67, 0x0d, 0xb6, 0x9c, 0x36, 0xb6, 0xbe, 0xbd,
			0x69, 0xa0, 0xbc, 0xd2, 0x40, 0x17, 0x9b, 0xa8,
			0xa4, 0x78, 0x16, 0xa0, 0xc3, 0x43, 0x7a, 0x61,
			0xfb, 0x72, 0xad, 0xca, 0xf9, 0x09, 0x6f, 0x2a,
			0x22, 0xef, 0xe0, 0xb4, 0x31, 0xfc, 0x42, 0x2d,
			0x22, 0x53, 0x01, 0xe8, 0x50, 0xf2, 0xf0, 0xf4,
			0xda, 0x87, 0xd6, 0x94, 0x4a, 0x85, 0x29, 0xef,
			0x79, 0x78, 0x19, 0x09, 0xad, 0x96, 0xd1, 0xf2,
			0x05, 0x96, 0xf9, 0x3e, 0x17, 0xc5, 0x7f, 0xb4,
			0xd7, 0x56, 0x97, 0x4b, 0xbb, 0xf9, 0x00, 0x52,
			0x1c, 0xb0, 0x89, 0xee, 0xe0, 0xde, 0xd5, 0xc9,
			0x56, 0xa1, 0x5b, 0x09, 0x61, 0x62, 0xb0, 0x7f,
		},
		/* S7 : 384 byte */
		{
			0x7b, 0xbb, 0x3d, 0xdd, 0x17, 0xa4, 0x2b, 0xe7,
			0xcc, 0x4e, 0x7e, 0xaf, 0x45, 0x65, 0x09, 0xa4,
			0xba, 0x58, 0xd4, 0x0c, 0x49, 0xa3, 0xd9, 0x95,
			0x73, 0xb7, 0x33, 0xe1, 0x94, 0x2f, 0x9f, 0xca,
			0x20, 0xba, 0x8b, 0x91, 0x07, 0x08, 0xd6, 0xe7,
			0x50, 0x36, 0x7e, 0x84, 0x73, 0x02, 0xfc, 0x60,
			0x3b, 0x80, 0x63, 0xc1, 0x9a, 0xf8, 0x83, 0xe7,
			0x50, 0x7f, 0xb0, 0xd9, 0xcc, 0x2b, 0xe3, 0x74,
			0x79, 0xa3, 0x7c, 0xca, 0x25, 0xb8, 0xc7, 0xc4,
			0x6f, 0x6b, 0xf6, 0x61, 0xdc, 0x6a, 0x32, 0x32,
			0xf8, 0x8b, 0x48, 0x3f, 0x1b, 0x8f, 0x41, 0xb4,
			0x6d, 0x49, 0xba, 0x3f, 0x17, 0x95, 0xd6, 0x8e,
			0xaa, 0xd4, 0xa2, 0x55, 0x6f, 0xb5, 0xd7, 0x87,
			0x3b, 0xbb, 0x65, 0x01, 0xec, 0xf0, 0x6a, 0xc5,
			0x58, 0x23, 0x5e, 0xd1, 0x39, 0x90, 0xb0, 0xe1,
			0x6f, 0x67, 0x96, 0x5b, 0x09, 0x36, 0x6b, 0xcb,
			0x36, 0x2c, 0xfc, 0x6f, 0xb9, 0x78, 0xf4, 0xf6,
			0x8d, 0x81, 0x46, 0xdc, 0x8b, 0x81, 0x98, 0x04,
			0xdf, 0x42, 0x4e, 0x8c, 0xa5, 0xb6, 0x3c, 0xf1,
			0xfc, 0xf9, 0x7b, 0xbf, 0x30, 0x0d, 0x0b, 0x99,
			0x88, 0x60, 0x79, 0x8a, 0x63, 0x42, 0x43, 0x83,
			0xfc, 0xd8, 0x1d, 0x37, 0x77, 0x3d, 0x59, 0xbb,
			0x13, 0xb4, 0xfa, 0x5d, 0x46, 0x8c, 0xd1, 0x28,
			0xbb, 0xab, 0x18, 0xa8, 0xce, 0x51, 0x73, 0xbe,
			0x5d, 0x9d, 0x54, 0xd3, 0x17, 0x7f, 0x02, 0x45,
			0x78, 0x84, 0x09, 0x97, 0x3d, 0xf4, 0xa9, 0x01,
			0x6b, 0x94, 0x4b, 0xae, 0xfb, 0xf3, 0xbf, 0x11,
			0x46, 0xa9, 0x39, 0x3d, 0x22, 0xe3, 0x5e, 0xc2,
			0xbe, 0x0a, 0xe6, 0xf4, 0xc3, 0x1d, 0xc4, 0x98,
			0x1f, 0x40, 0xfc, 0x1b, 0xaf, 0x38, 0x26, 0x00,
			0x69, 0x9e, 0xaf, 0xce, 0xa9, 0x2c, 0xbe, 0x24,
			0xe2, 0x6e, 0xe8, 0x46, 0xfa, 0x23, 0xbc, 0x19,
			0x3b, 0x6e, 0x72, 0x14, 0x01, 0xb7, 0xac, 0x3f,
			0x5f, 0x4e, 0xbe, 0xb6, 0x33, 0x97, 0x9f, 0x8e,
			0xf3, 0x5f, 0x4a, 0xb1, 0x11, 0x7a, 0x86, 0x9d,
			0x5b, 0x9d, 0xbb, 0x74, 0x82, 0xf0, 0xd5, 0xa5,
			0x9e, 0x41, 0x63, 0x54, 0x8d, 0x25, 0x12, 0xae,
			0x06, 0x72, 0x05, 0xb5, 0x7d, 0x03, 0x0c, 0x48,
			0x3f, 0x72, 0x0d, 0x2c, 0x44, 0x35, 0x04, 0x28,
			0xf5, 0x26, 0x89, 0x43, 0xfc, 0x5f, 0x6e, 0xa1,
			0xc8, 0x8e, 0x2e, 0xc1, 0x3a, 0xb3, 0xdc, 0x14,
			0x56, 0xe9, 0x6a, 0x3b, 0x8e, 0x7c, 0x12, 0x1a,
			0xf4, 0xd6, 0xa5, 0xfe, 0x4e, 0xe5, 0x5e, 0x99,
			0xfb, 0xc3, 0x59, 0x2a, 0x48, 0x7c, 0x19, 0x4b,
			0xc2, 0xf2, 0xbf, 0x6e, 0x79, 0xfb, 0x79, 0xc2,
			0x87, 0x6c, 0xf3, 0x36, 0x5e, 0x07, 0x5b, 0xee,
			0xac, 0xc7, 0xdb, 0x4d, 0xb7, 0xee, 0x69, 0xe7,
			0xf1, 0xfe, 0x12, 0xa3, 0x27, 0xe6, 0xcb, 0x0f,
		},
	},
	/* SHA256 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0x0e, 0x67, 0x5d, 0xac, 0x9a, 0xec, 0x91, 0x01,
			0x06, 0xa6, 0xab, 0x21, 0x9b, 0x4c, 0xce, 0xb5,
			0x2d, 0xed, 0x25, 0x49, 0xe8, 0x99, 0xc9, 0xa2,
			0x4d, 0x5e, 0xe5, 0x51, 0x77, 0x76, 0x18, 0x88,
			0xa3, 0xbe, 0x1a, 0x2d, 0xef, 0x6a, 0xa3, 0x2d,
			0x62, 0xf7, 0x88, 0x13, 0x2d, 0x62, 0x27, 0xd9,
			0x30, 0x98, 0x06, 0xfd, 0xc0, 0x2d, 0xb7, 0xd8,
			0xa8, 0x50, 0xff, 0x2c, 0x6d, 0xff, 0x37, 0xfc,
			0xd7, 0x77, 0xf1, 0xa0, 0xac, 0xef, 0xdf, 0x18,
			0xbf, 0x85, 0xf1, 0xa1, 0x29, 0x79, 0xbe, 0x86,
			0xd7, 0x99, 0x25, 0x39, 0x45, 0xfc, 0x34, 0xa2,
			0x88, 0xf3, 0x48, 0xb7, 0x92, 0x3d, 0x76, 0x4d,
			0xb2, 0x7a, 0x2a, 0x2d, 0x5a, 0xe2, 0x0e, 0x6b,
			0x25, 0x37, 0x2e, 0xf3, 0x18, 0xf8, 0x59, 0x65,
			0x29, 0xd8, 0xca, 0x23, 0xfd, 0x6f, 0x08, 0xa8,
			0xf6, 0x2e, 0x0a, 0x1b, 0x6d, 0x98, 0x9f, 0x23,
		},
		/* S8 : 384 byte */
		{
			0x80, 0x52, 0xd9, 0x5f, 0x12, 0xce, 0x0e, 0x6e,
			0x53, 0xa5, 0xa3, 0x56, 0xa0, 0xeb, 0x35, 0x3b,
			0xdc, 0xc1, 0xa6, 0x65, 0x14, 0xd6, 0xcf, 0xb3,
			0xa3, 0xd9, 0x61, 0x55, 0x31, 0x0b, 0xdd, 0xa0,
			0xa0, 0xd1, 0x79, 0x5f, 0x97, 0x64, 0x3f, 0x3a,
			0x44, 0x96, 0x63, 0x4f, 0x2d, 0xd9, 0xb9, 0x5a,
			0x21, 0x38, 0xee, 0x39, 0x0e, 0x1e, 0x74, 0xbe,
			0x31, 0x34, 0xf3, 0xf4, 0x7a, 0x91, 0x9e, 0xe7,
			0xb5, 0x9f, 0x8e, 0xcd, 0x27, 0x2a, 0xb8, 0x8c,
			0x82, 0xcb, 0xce, 0x7c, 0x21, 0x7e, 0x5f, 0x92,
			0xd0, 0x57, 0xa5, 0xb0, 0x0f, 0xbf, 0x05, 0x75,
			0xcd, 0xae, 0xcd, 0x7d, 0xc2, 0x85, 0xa4, 0x21,
			0x8c, 0x8a, 0x95, 0x52, 0x16, 0x59, 0x8f, 0x07,
			0x42, 0x67, 0x1e, 0x01, 0x8e, 0x8e, 0x4e, 0x76,
			0x83, 0x9a, 0x57, 0x5f, 0x50, 0xb2, 0x10, 0x2a,
			0x8b, 0x77, 0xd1, 0xb8, 0x4f, 0x6d, 0xce, 0x98,
			0xd7, 0x8e, 0x57, 0x58, 0xe0, 0xa6, 0xf9, 0x2b,
			0xf3, 0x5d, 0x6a, 0x2f, 0x18, 0xad, 0x40, 0x09,
			0x25, 0xd7, 0x88, 0x0f, 0x9e, 0xfc, 0x77, 0x4a,
			0x8c, 0x7e, 0xbf, 0x64, 0x88, 0x5c, 0xd2, 0xf6,
			0xf6, 0x29, 0xb5, 0x4a, 0x7c, 0x12, 0xec, 0x91,
			0xd3, 0x9b, 0x3c, 0x25, 0x18, 0x24, 0x1f, 0xdc,
			0x32, 0x2d, 0x9b, 0x23, 0x5a, 0x8e, 0xa4, 0x4f,
			0x77, 0xe8, 0x2f, 0x3d, 0xc4, 0xf7, 0x28, 0xf6,
			0x20, 0xc0, 0x7d, 0x1e, 0x7f, 0xf4, 0x09, 0x4f,
			0x29, 0xc6, 0x74, 0xab, 0x0f, 0x08, 0x02, 0xef,
			0xa1, 0xc9, 0xe6, 0x48, 0x1e, 0xbb, 0x84, 0xe0,
			0xbf, 0x13, 0xef, 0x46, 0x8d, 0x8c, 0xca, 0x11,
			0x45, 0x70, 0xb9, 0xed, 0xcd, 0xdf, 0x98, 0xac,
			0x4a, 0x83, 0x4f, 0xe7, 0xa0, 0xd5, 0xc6, 0xfa,
			0xe8, 0xa6, 0x0a, 0x48, 0x39, 0x9f, 0x3c, 0x8a,
			0xf4, 0x2f, 0xf4, 0x02, 0x6e, 0x42, 0xa8, 0x1a,
			0xac, 0x36, 0x11, 0x4f, 0xfc, 0x05, 0x3f, 0x3f,
			0x72, 0x9b, 0x7c, 0xf9, 0xa9, 0x7a, 0x56, 0x84,
			0x8e, 0xbe, 0xa0, 0x11, 0x5a, 0xa8, 0x29, 0x83,
			0x41, 0xaa, 0x22, 0x69, 0x63, 0xeb, 0xdf, 0x57,
			0xab, 0x2d, 0x8e, 0x4b, 0x90, 0x00, 0xdd, 0x05,
			0x1a, 0x6c, 0x5d, 0x69, 0xf6, 0x0e, 0x1d, 0xc1,
			0xb3, 0x3f, 0x20, 0x94, 0xfd, 0xbf, 0x8e, 0x5b,
			0x62, 0x7b, 0xc0, 0x76, 0x4d, 0xb9, 0x52, 0x2c,
			0xbb, 0xc0, 0x81, 0xdb, 0xf3, 0x8c, 0x21, 0xb1,
			0x3f, 0x98, 0x08, 0x13, 0xbd, 0x2b, 0x00, 0xc7,
			0x57, 0xeb, 0xb8, 0xc0, 0xb2, 0x12, 0x13, 0x15,
			0x2e, 0x69, 0x40, 0x39, 0xf3, 0x06, 0xf7, 0x34,
			0x28, 0x57, 0x65, 0x1f, 0x72, 0x2b, 0xdd, 0xa0,
			0x12, 0x12, 0xa8, 0x55, 0x27, 0x99, 0xbd, 0xa6,
			0xef, 0x07, 0xc5, 0x20, 0x7d, 0xc7, 0x44, 0xef,
			0x79, 0x69, 0xaf, 0xd5, 0xaf, 0x2e, 0x6f, 0x12,
		},
	},
	/* SHA256 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0xf6, 0xa7, 0xa6, 0xe5, 0x26, 0x59, 0x12, 0x5f,
			0xbb, 0xc8, 0x72, 0x74, 0x17, 0x28, 0x3b, 0x9a,
			0x64, 0x44, 0x1f, 0x87, 0x12, 0x1e, 0x27, 0xf3,
			0x86, 0xd5, 0x01, 0x9f, 0x10, 0xcc, 0x9b, 0x96,
			0x1e, 0x09, 0xf1, 0xb3, 0xb0, 0xdb, 0x23, 0x63,
			0x0c, 0xc0, 0xca, 0xac, 0xb3, 0x85, 0x8c, 0x6f,
			0x93, 0xaf, 0xee, 0xea, 0x7e, 0x1a, 0x6a, 0x80,
			0xdb, 0xe0, 0xc2, 0xbd, 0x9c, 0x7c, 0x93, 0x95,
			0x70, 0x30, 0x2d, 0xec, 0x39, 0xa4, 0xa2, 0x5c,
			0xc0, 0xcf, 0x1d, 0x32, 0xa7, 0x1a, 0x75, 0xb9,
			0xa0, 0xc3, 0x02, 0xbc, 0xdd, 0x80, 0xb0, 0x46,
			0xc8, 0x66, 0x51, 0xac, 0xf3, 0x08, 0x38, 0xcd,
			0x52, 0xe3, 0x03, 0x99, 0xa8, 0xfa, 0xb8, 0xd0,
			0x3f, 0xbd, 0x14, 0x0c, 0xdc, 0x2f, 0x1f, 0x02,
			0xf2, 0x48, 0x04, 0x05, 0x16, 0x98, 0x20, 0xcc,
			0xb3, 0x2e, 0x59, 0x74, 0xff, 0xb8, 0xb1, 0xc8,
		},
		/* S9 : 384 byte */
		{
			0x84, 0x60, 0x3a, 0xcb, 0xfe, 0x1f, 0x2f, 0x76,
			0x9f, 0x1a, 0x62, 0xb0, 0xf2, 0x87, 0xf3, 0x06,
			0x94, 0x0b, 0x22, 0x54, 0x76, 0x71, 0x4a, 0x4b,
			0x68, 0x27, 0xc0, 0x2d, 0x7b, 0xd0, 0x52, 0xf3,
			0x03, 0xf3, 0x0a, 0x5f, 0xa6, 0xda, 0x83, 0xe6,
			0x06, 0x15, 0x30, 0x56, 0x69, 0xca, 0x9e, 0xc1,
			0x77, 0xc5, 0xb3, 0x2b, 0x14, 0x15, 0xee, 0xbe,
			0xf7, 0x86, 0x20, 0x29, 0x6e, 0xba, 0xd6, 0xdb,
			0xbd, 0x52, 0x08, 0x39, 0xd3, 0xaa, 0xcc, 0x97,
			0x81, 0xac, 0x86, 0x02, 0xdd, 0xce, 0x07, 0x36,
			0xdc, 0xfa, 0x72, 0x90, 0xb4, 0x5f, 0x15, 0x5b,
			0x8e, 0x92, 0x4d, 0x0a, 0xfd, 0xf7, 0xdf, 0xc8,
			0xd1, 0x99, 0xbf, 0x09, 0x50, 0x9d, 0x01, 0x76,
			0xa6, 0x8b, 0x14, 0x57, 0x56, 0xee, 0xf5, 0x3d,
			0xe4, 0x56, 0xe1, 0x70, 0x78, 0x85, 0x98, 0x49,
			0xa3, 0x52, 0xa5, 0xbb, 0x65, 0x42, 0x39, 0xd8,
			0xeb, 0xaf, 0x88, 0x00, 0xca, 0x82, 0x63, 0xd3,
			0x4a, 0x86, 0x8d, 0x52, 0xbf, 0x8f, 0x22, 0x64,
			0x4d, 0xd9, 0xf3, 0xc0, 0x5b, 0xd8, 0x91, 0xcd,
			0x92, 0xf2, 0x63, 0x53, 0x0c, 0x58, 0x96, 0x02,
			0x3c, 0x6b, 0x21, 0x3d, 0xdb, 0x64, 0xed, 0xe1,
			0x77, 0x0f, 0xf1, 0x68, 0x6c, 0x34, 0x03, 0x6e,
			0x28, 0x1e, 0x91, 0x1d, 0x9d, 0xc9, 0x60, 0x35,
			0x4f, 0xd8, 0x44, 0xcb, 0x7b, 0x22, 0xdc, 0x0c,
			0xd8, 0x1a, 0x96, 0x20, 0x3b, 0xa8, 0x18, 0x40,
			0x1c, 0xcc, 0x22, 0x5f, 0x85, 0x7e, 0x59, 0xa5,
			0xcb, 0x7b, 0xa6, 0xdf, 0xc7, 0xf5, 0x13, 0x5e,
			0xa3, 0x27, 0x81, 0xe6, 0x3d, 0xaa, 0x14, 0xfb,
			0xda, 0x1b, 0xac, 0xc1, 0x8e, 0xbc, 0x50, 0x82,
			0x4d, 0x40, 0x28, 0xb8, 0xfd, 0xec, 0xda, 0x49,
			0xe8, 0x10, 0xba, 0xe5, 0xac, 0xc8, 0xad, 0xc0,
			0xdc, 0xa2, 0xe2, 0x36, 0xfc, 0x83, 0x2a, 0x97,
			0x33, 0x0a, 0x12, 0x14, 0xfa, 0x0a, 0xed, 0x15,
			0xcd, 0x10, 0xc0, 0x49, 0xef, 0xb6, 0x5c, 0xe8,
			0x55, 0xc0, 0x60, 0xf0, 0x5b, 0xef, 0xb3, 0x17,
			0xb8, 0x06, 0x58, 0x43, 0xc4, 0xeb, 0x5a, 0x03,
			0x71, 0xfc, 0x6f, 0x20, 0x9f, 0x6f, 0xfb, 0x94,
			0x8c, 0x88, 0x1f, 0x2f, 0x20, 0x91, 0xca, 0xf0,
			0xf5, 0x9f, 0x60, 0xb7, 0x2c, 0x5f, 0x67, 0x27,
			0x1b, 0xae, 0x96, 0xb9, 0x13, 0xfd, 0x21, 0xfa,
			0x1d, 0xfa, 0x97, 0x5d, 0x5e, 0xcd, 0x62, 0xb0,
			0xd5, 0x08, 0x73, 0xb6, 0x86, 0xd2, 0x9c, 0x88,
			0x0d, 0x36, 0xed, 0xca, 0xd3, 0x3e, 0xc3, 0xe2,
			0x21, 0x6c, 0x9c, 0xfc, 0xfb, 0x4f, 0x98, 0x4c,
			0x23, 0xfd, 0xe8, 0x15, 0xe2, 0x80, 0xa8, 0x02,
			0x42, 0x86, 0x08, 0xbe, 0xd3, 0x73, 0x9a, 0xf9,
			0x20, 0x0d, 0xe1, 0xf8, 0x5e, 0xde, 0xe2, 0x83,
			0x4c, 0x04, 0x94, 0x2c, 0x06, 0x8a, 0xac, 0xd2,
		},
	},
};

/* 10 elements */
static mod3072_t SHA384[] = {
	/* SHA384 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0xbb, 0x29, 0x4b, 0x95, 0xd9, 0x13, 0x00, 0x5b,
			0x11, 0x09, 0x87, 0xcd, 0xe4, 0x58, 0x87, 0x48,
			0x4a, 0xe6, 0xdf, 0x79, 0x48, 0x73, 0xdf, 0xc5,
			0xc4, 0x1f, 0xb7, 0xe8, 0x99, 0x2c, 0x2f, 0xdc,
			0xe7, 0x06, 0x99, 0xfc, 0xac, 0x80, 0x04, 0x69,
			0x99, 0x61, 0xb3, 0xad, 0x1e, 0x1f, 0xce, 0x9e,
			0xc8, 0xea, 0x56, 0x85, 0xcc, 0xec, 0x5e, 0x80,
			0xe4, 0xd0, 0x79, 0x25, 0x59, 0x81, 0x6f, 0x68,
			0x61, 0x34, 0x34, 0xbf, 0xac, 0xa8, 0x1a, 0x84,
			0x3a, 0xac, 0x45, 0x9a, 0x6f, 0xe3, 0x5f, 0x53,
			0x69, 0xc4, 0x8e, 0x91, 0x91, 0xe4, 0xa3, 0x2c,
			0x70, 0x78, 0x95, 0x94, 0xc5, 0x15, 0x2d, 0xb8,
			0xd4, 0xbb, 0x02, 0x26, 0x00, 0x12, 0xa8, 0x73,
			0x9c, 0xf3, 0x25, 0xdd, 0xff, 0x2a, 0xa4, 0x2f,
			0xd6, 0x7b, 0x6e, 0xe5, 0xbf, 0xe3, 0x15, 0x91,
			0x13, 0x1f, 0xf2, 0x7d, 0x02, 0x73, 0xd2, 0x92,
		},
		/* S0 : 384 byte */
		{
			0x32, 0x63, 0x7c, 0x60, 0x79, 0x8b, 0x45, 0x0b,
			0xff, 0x10, 0x0b, 0xff, 0x12, 0x83, 0x83, 0x57,
			0xde, 0xff, 0x28, 0x1d, 0x5b, 0x31, 0xe4, 0xf4,
			0xc2, 0xcf, 0xc9, 0x6e, 0xb7, 0x79, 0xce, 0x6d,
			0x31, 0xb1, 0xce, 0x8b, 0xd7, 0xaa, 0x7f, 0xa8,
			0x8d, 0xdc, 0x42, 0x79, 0xc8, 0xc3, 0x28, 0x06,
			0x04, 0xb0, 0x18, 0xcc, 0xf4, 0x52, 0x00, 0x4a,
			0x14, 0x88, 0xed, 0x47, 0x50, 0x18, 0x1c, 0x50,
			0x25, 0x63, 0x65, 0x11, 0xac, 0x67, 0x24, 0xfe,
			0x51, 0x76, 0x1c, 0x27, 0xd7, 0xcf, 0x9a, 0x0c,
			0x87, 0x82, 0xea, 0x22, 0x31, 0x26, 0x88, 0x53,
			0xc4, 0xb1, 0xf7, 0xac, 0xb0, 0x00, 0x5e, 0x56,
			0x87, 0xc8, 0xf3, 0xdf, 0x16, 0xc9, 0x62, 0xf0,
			0x2c, 0xe5, 0x6b, 0x23, 0xd3, 0x87, 0xa2, 0xba,
			0xad, 0xc8, 0xbe, 0xc9, 0x42, 0x29, 0xc3, 0x55,
			0x75, 0x26, 0xe6, 0x17, 0x07, 0xa8, 0xb5, 0x92,
			0x93, 0xa9, 0x76, 0xe3, 0x2c, 0x7f, 0xa1, 0x33,
			0x28, 0x50, 0x88, 0xf3, 0xce, 0x3e, 0x67, 0x77,
			0x88, 0xaa, 0xa9, 0x47, 0xe7, 0x62, 0x2c, 0x75,
			0x7e, 0x84, 0x4b, 0x11, 0x75, 0x92, 0xbe, 0x99,
			0xfe, 0x45, 0x37, 0x6f, 0x8b, 0x30, 0x13, 0xe8,
			0x77, 0x2e, 0xc9, 0x2c, 0x5b, 0xb0, 0xb9, 0xfa,
			0x30, 0x1b, 0x95, 0x54, 0x45, 0x99, 0x69, 0x0a,
			0xd9, 0x36, 0x68, 0xd8, 0x3b, 0x2d, 0xaa, 0x7d,
			0xf0, 0x5c, 0x66, 0x21, 0x4e, 0x27, 0x50, 0x14,
			0x78, 0x0a, 0x91, 0x2d, 0x8b, 0x19, 0x32, 0xd7,
			0xa6, 0x55, 0x05, 0x8e, 0x74, 0x3f, 0x50, 0xb0,
			0x74, 0xb1, 0xd9, 0x69, 0x1c, 0xa2, 0x3a, 0x2f,
			0x95, 0xf6, 0xaf, 0xfb, 0xd5, 0x16, 0xd6, 0x4c,
			0xcb, 0x2a, 0xa4, 0x3c, 0x23, 0x6e, 0xb9, 0x5d,
			0x36, 0xd2, 0x72, 0x54, 0x5e, 0x3b, 0xeb, 0x8f,
			0xf5, 0xaa, 0xcd, 0x95, 0xb3, 0x0f, 0x7f, 0x1d,
			0x64, 0x18, 0xaf, 0x04, 0x2c, 0xd9, 0xa0, 0xcf,
			0x01, 0x89, 0x84, 0x62, 0x62, 0x32, 0x2a, 0x18,
			0x87, 0x5a, 0xe4, 0xc3, 0xe6, 0x8e, 0x4e, 0x8f,
			0xfa, 0xa0, 0x27, 0x6c, 0xdd, 0x99, 0xa0, 0x04,
			0x7c, 0x86, 0xc0, 0xf7, 0x1d, 0x2d, 0xee, 0xfd,
			0x50, 0x64, 0x2d, 0x29, 0xc1, 0x95, 0xe6, 0xd1,
			0x4f, 0xb4, 0x6f, 0xba, 0xc3, 0x3a, 0x50, 0x8c,
			0x1f, 0x03, 0xa2, 0x32, 0xde, 0x08, 0xaa, 0xe0,
			0x9f, 0xaf, 0x1d, 0xa8, 0xed, 0x2b, 0xa2, 0xae,
			0x84, 0xbc, 0xca, 0x88, 0xb7, 0x8d, 0xcc, 0xbd,
			0xe9, 0xaf, 0xde, 0x08, 0xa3, 0xbe, 0xb3, 0x22,
			0xdc, 0x79, 0x35, 0x6b, 0x29, 0xc8, 0x48, 0x41,
			0x69, 0x89, 0x14, 0xb0, 0x50, 0xbe, 0xb7, 0x5a,
			0x7b, 0x2f, 0x67, 0x01, 0xaa, 0x81, 0x01, 0xa5,
			0xa4, 0x95, 0x5e, 0xe2, 0x7b, 0xaf, 0xe8, 0x1b,
			0x21, 0xd0, 0x3b, 0x43, 0xe3, 0xc7, 0x73, 0x98,
		},
	},
	/* SHA384 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0xf9, 0x46, 0xc6, 0xbd, 0x5e, 0x1d, 0x6b, 0x89,
			0x09, 0x2f, 0x3c, 0x48, 0x7c, 0x05, 0x68, 0xfa,
			0x07, 0xc3, 0x56, 0xfa, 0xe9, 0xb8, 0xe8, 0x31,
			0xb8, 0x32, 0x02, 0x89, 0x03, 0x97, 0x46, 0xa4,
			0x35, 0xb1, 0x22, 0xcf, 0xbc, 0x4a, 0x0d, 0x31,
			0x6b, 0xf9, 0x0d, 0x48, 0x1d, 0x3b, 0x7d, 0x97,
			0x9c, 0xc5, 0x0d, 0x98, 0xc1, 0x19, 0x0a, 0xf8,
			0xdc, 0x58, 0xe0, 0x03, 0x55, 0x57, 0xdd, 0x5e,
			0x94, 0xf4, 0x37, 0xf4, 0x1f, 0xab, 0x51, 0x32,
			0x02, 0x64, 0x3a, 0x77, 0x74, 0x8f, 0x76, 0xc6,
			0xb7, 0x73, 0x02, 0xbf, 0x40, 0xc3, 0x92, 0xcd,
			0x18, 0x73, 0x1d, 0xa0, 0x82, 0xc9, 0x9b, 0xde,
			0xde, 0xb7, 0x0e, 0x15, 0xcd, 0x68, 0xbf, 0xf5,
			0x96, 0x19, 0xca, 0xbc, 0xc9, 0x2a, 0xdc, 0xf1,
			0x22, 0x75, 0x3c, 0x55, 0xaf, 0xde, 0x08, 0x17,
			0x35, 0x2b, 0xc2, 0x47, 0xd1, 0x17, 0x0b, 0x8d,
		},
		/* S1 : 384 byte */
		{
			0x50, 0x70, 0x6b, 0xa4, 0x9d, 0x9a, 0x31, 0x66,
			0x88, 0xa3, 0xee, 0x80, 0xa0, 0xbd, 0x98, 0x67,
			0x57, 0xd4, 0x3e, 0xc8, 0x32, 0x85, 0xaf, 0x9e,
			0x78, 0x19, 0x6b, 0xd5, 0x2c, 0x90, 0x0d, 0x40,
			0xb2, 0x80, 0xfa, 0x0d, 0xe5, 0x4e, 0x35, 0xac,
			0xe7, 0xd6, 0x66, 0x00, 0x12, 0xf1, 0xa6, 0x62,
			0x04, 0x09, 0x2f, 0x0e, 0x63, 0x4b, 0x97, 0xe0,
			0xe5, 0x16, 0x65, 0xb4, 0x07, 0x5e, 0x36, 0xf1,
			0x42, 0x22, 0x66, 0xc7, 0xca, 0xd7, 0xb2, 0xd9,
			0x98, 0x1b, 0x91, 0x3d, 0xf3, 0xfa, 0x3e, 0x6a,
			0x5a, 0x1c, 0xad, 0xfc, 0x63, 0x78, 0xa8, 0x54,
			0x0e, 0x0f, 0xaa, 0x26, 0xf1, 0xcc, 0x6f, 0xb2,
			0xfb, 0x49, 0x2a, 0x80, 0xd0, 0xa6, 0x94, 0x5b,
			0xce, 0x5b, 0xbc, 0x23, 0xdd, 0xb3, 0xb1, 0x07,
			0x01, 0xf0, 0x24, 0x9b, 0x27, 0x40, 0x7a, 0x67,
			0x00, 0x80, 0x2e, 0x88, 0x42, 0xef, 0x3c, 0xc7,
			0x61, 0xc4, 0x82, 0x3a, 0xcb, 0x5d, 0x14, 0x53,
			0x50, 0x8d, 0xcd, 0xbb, 0x97, 0x9e, 0x7b, 0xd8,
			0xd0, 0x01, 0x28, 0xe6, 0x0a, 0x9b, 0x37, 0x89,
			0x16, 0x7c, 0x91, 0x41, 0x7d, 0x93, 0xf0, 0xe9,
			0xfb, 0xb0, 0x0c, 0x9a, 0xf1, 0x49, 0x8e, 0x09,
			0xeb, 0x64, 0x85, 0xeb, 0x94, 0xce, 0xa4, 0x88,
			0x3f, 0x6a, 0x25, 0x6e, 0xab, 0x2c, 0xaa, 0x82,
			0x6d, 0xe4, 0xfd, 0xac, 0x01, 0xba, 0xca, 0x3a,
			0x21, 0x6e, 0x3d, 0x20, 0x4a, 0x3d, 0x83, 0x7f,
			0xfd, 0x4d, 0x0b, 0xe2, 0xb2, 0xce, 0xf7, 0x11,
			0x90, 0x90, 0x54, 0xc4, 0xda, 0x1d, 0x5b, 0x93,
			0xa8, 0xf9, 0x84, 0x51, 0xc7, 0x00, 0x2a, 0xe8,
			0x4a, 0x5e, 0x70, 0x80, 0xd9, 0x86, 0x71, 0xc5,
			0x0e, 0x3c, 0x91, 0xc4, 0x08, 0x7d, 0x04, 0x77,
			0xb1, 0x04, 0xf9, 0x16, 0x01, 0x0e, 0x74, 0x2f,
			0x2d, 0x20, 0x7f, 0xb4, 0x0d, 0x12, 0x2d, 0x8f,
			0x21, 0x1a, 0xf6, 0xd7, 0xc5, 0xec, 0xa4, 0x95,
			0x42, 0xd9, 0xac, 0xb0, 0xf1, 0x66, 0xe3, 0x6a,
			0xbc, 0x37, 0x15, 0x50, 0x70, 0xc1, 0x2e, 0x9f,
			0x28, 0xb9, 0x07, 0xd6, 0x7a, 0x2c, 0xa7, 0x0b,
			0xfc, 0xe5, 0x54, 0xe1, 0xc4, 0x4c, 0x91, 0x52,
			0x0e, 0x98, 0xfc, 0x9a, 0xd0, 0xc0, 0xee, 0x47,
			0x7f, 0x75, 0x05, 0x16, 0x47, 0x6a, 0x94, 0x16,
			0x80, 0x66, 0xce, 0x47, 0x00, 0x00, 0x30, 0xa9,
			0x9c, 0x23, 0xe2, 0xc3, 0x87, 0x55, 0xde, 0x94,
			0x6d, 0x5e, 0xdf, 0x0d, 0x6a, 0xa9, 0x42, 0x12,
			0xf9, 0x92, 0x31, 0x5b, 0x24, 0x8c, 0x1f, 0x82,
			0x72, 0x3b, 0x29, 0xc4, 0x22, 0x16, 0xc7, 0x8c,
			0xdc, 0xb6, 0x68, 0xf1, 0x12, 0x78, 0x26, 0x1c,
			0xee, 0x92, 0x52, 0xc8, 0xfd, 0x0e, 0xd3, 0x7d,
			0x0a, 0x85, 0x80, 0xca, 0x9b, 0x9f, 0xde, 0x75,
			0x05, 0x61, 0x59, 0x43, 0x71, 0x2d, 0xa1, 0x9a,
		},
	},
	/* SHA384 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x9a, 0x33, 0x7d, 0x4c, 0x0b, 0xb9, 0xa0, 0x05,
			0xb4, 0x7f, 0x47, 0x65, 0xd6, 0x96, 0xd1, 0x9d,
			0xec, 0x58, 0xbc, 0x84, 0x82, 0xf2, 0x17, 0x3a,
			0x4a, 0x20, 0x3a, 0x0b, 0x6d, 0x38, 0xb4, 0x96,
			0x1f, 0x6a, 0x85, 0x2e, 0x76, 0x46, 0x8e, 0x80,
			0x7c, 0x7e, 0x45, 0x76, 0x83, 0xee, 0xad, 0x5c,
			0xb8, 0xd9, 0x86, 0x42, 0xfb, 0x76, 0xc0, 0xa1,
			0xee, 0xab, 0x36, 0x41, 0x4c, 0x18, 0x99, 0x59,
			0x7d, 0x57, 0xaa, 0xf9, 0x67, 0x82, 0xad, 0xa5,
			0x86, 0xf6, 0x1a, 0x42, 0x3f, 0x57, 0x95, 0x37,
			0x71, 0xd5, 0x20, 0xcc, 0x4e, 0xad, 0x90, 0xd5,
			0x69, 0xf2, 0x3d, 0x95, 0x0f, 0x8d, 0xfe, 0xdd,
			0xdb, 0x83, 0x55, 0x74, 0x85, 0x76, 0xe6, 0xbb,
			0xfb, 0x6f, 0x2e, 0x91, 0xb3, 0xda, 0x71, 0x75,
			0x3f, 0xd2, 0xf4, 0xea, 0x22, 0x9f, 0x6d, 0x20,
			0xe2, 0x7d, 0xb8, 0xd0, 0x5e, 0x9f, 0xcb, 0x68,
		},
		/* S2 : 384 byte */
		{
			0xcf, 0xf7, 0xaa, 0x7f, 0x87, 0x56, 0x42, 0xfb,
			0x93, 0x43, 0xe0, 0x7e, 0xf5, 0xe7, 0x30, 0x3b,
			0xbf, 0x5f, 0x06, 0x9b, 0x44, 0xc1, 0x9f, 0xbf,
			0x83, 0xe5, 0x9d, 0x42, 0x2e, 0x25, 0x26, 0x7e,
			0xf9, 0x30, 0x74, 0x14, 0xb6, 0xb1, 0xef, 0x61,
			0x71, 0x1e, 0xd0, 0x01, 0x32, 0x76, 0xd1, 0xa2,
			0xad, 0x98, 0x39, 0x04, 0x74, 0x02, 0x7a, 0x0a,
			0x70, 0x3b, 0xfe, 0x8a, 0x6e, 0x87, 0x70, 0x60,
			0x59, 0xd8, 0x9c, 0x06, 0x09, 0x80, 0xc9, 0xc9,
			0xe6, 0x0d, 0xc7, 0xe1, 0xfb, 0x9f, 0x77, 0x7a,
			0x41, 0x78, 0x5a, 0xb4, 0xd2, 0xb6, 0x63, 0xba,
			0x0e, 0x3c, 0x19, 0x21, 0x54, 0x5c, 0x47, 0x9c,
			0x2a, 0x38, 0x3a, 0x50, 0xda, 0x8e, 0x48, 0x9c,
			0xb2, 0x2b, 0x71, 0x10, 0x1d, 0x0e, 0xc1, 0x48,
			0xac, 0x70, 0x92, 0x87, 0x32, 0xa7, 0x72, 0x19,
			0x5a, 0x14, 0x0d, 0x08, 0x01, 0x52, 0x76, 0x2a,
			0x9c, 0x40, 0x80, 0x3a, 0x39, 0xfa, 0x2a, 0x69,
			0x78, 0xc2, 0xa7, 0x5a, 0xc4, 0xd8, 0xbd, 0x1b,
			0xcc, 0xaa, 0x1f, 0x42, 0x04, 0xba, 0x65, 0xed,
			0xdd, 0xf3, 0x2f, 0xed, 0xf2, 0xd9, 0xd0, 0xa3,
			0xae, 0xd9, 0xb0, 0x6c, 0x47, 0xe7, 0x17, 0x73,
			0x3c, 0x57, 0x78, 0x12, 0xd7, 0x23, 0xdb, 0xa7,
			0x4a, 0x85, 0x2b, 0x29, 0x05, 0x23, 0x5c, 0x81,
			0x2d, 0xc5, 0xf1, 0xd0, 0xdf, 0x0f, 0x0d, 0xe7,
			0x3d, 0xfb, 0x86, 0x22, 0x1c, 0x6f, 0xfd, 0xd1,
			0xed, 0xa1, 0x19, 0xbb, 0xe9, 0x8d, 0x14, 0x8a,
			0xdd, 0x36, 0xa4, 0xfe, 0x50, 0x48, 0x9b, 0x06,
			0xaa, 0xee, 0xfc, 0xb5, 0xc2, 0x06, 0x6d, 0x90,
			0xfa, 0x79, 0x73, 0x87, 0x06, 0xcd, 0x18, 0xe4,
			0x74, 0xd6, 0x96, 0x09, 0xff, 0x12, 0x10, 0xc7,
			0x7d, 0xe7, 0xcd, 0x23, 0xba, 0x2a, 0x77, 0x5a,
			0x43, 0x29, 0xcb, 0x27, 0x1a, 0x82, 0x6d, 0x60,
			0x2c, 0x40, 0x1a, 0x71, 0x43, 0x90, 0x19, 0xce,
			0xc1, 0x0c, 0xd9, 0xf1, 0x84, 0xc4, 0xd0, 0x45,
			0x84, 0x21, 0x18, 0x27, 0xb1, 0x9e, 0xad, 0xac,
			0x32, 0x58, 0xd8, 0xa0, 0xf2, 0x63, 0x16, 0x13,
			0xf0, 0x51, 0xaa, 0xe0, 0xc6, 0x13, 0x05, 0x0c,
			0xb2, 0x44, 0x42, 0xf1, 0x5e, 0xd4, 0xfe, 0x0d,
			0xbd, 0x29, 0x0e, 0x42, 0x62, 0x91, 0x41, 0xbd,
			0x2c, 0xd5, 0x6d, 0x20, 0x58, 0x4a, 0x1d, 0x10,
			0xe1, 0xf2, 0xc2, 0xa9, 0xec, 0x73, 0x14, 0x33,
			0xd5, 0xbc, 0xd1, 0xd3, 0x18, 0xbe, 0xd5, 0x24,
			0x3b, 0x4b, 0x7d, 0x0f, 0x9a, 0x79, 0x82, 0x06,
			0x1c, 0x55, 0xdf, 0xaa, 0x86, 0xb2, 0xc0, 0x18,
			0x45, 0xc0, 0x21, 0xfd, 0xd2, 0xa9, 0x78, 0xd4,
			0x20, 0x34, 0x21, 0x2f, 0x43, 0xb3, 0x35, 0x1b,
			0x6a, 0xde, 0xb0, 0x3b, 0xdd, 0x6c, 0xaf, 0x7d,
			0xe0, 0x59, 0x50, 0x2f, 0x16, 0xd7, 0x73, 0x48,
		},
	},
	/* SHA384 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0x32, 0xfd, 0x45, 0xe7, 0x3f, 0x6f, 0x69, 0x49,
			0xf2, 0x0c, 0xab, 0x78, 0xc0, 0xcc, 0x31, 0xd8,
			0x14, 0xba, 0xea, 0x63, 0x89, 0x54, 0x6a, 0x36,
			0x5d, 0x35, 0xf5, 0x4f, 0x23, 0xf1, 0xd9, 0x95,
			0xb7, 0x41, 0x01, 0x18, 0x77, 0x60, 0xc8, 0x9b,
			0xb0, 0xb4, 0x0b, 0x50, 0x57, 0xb1, 0x82, 0xe2,
			0xfa, 0xfb, 0x50, 0xb8, 0xf5, 0xca, 0xd8, 0x79,
			0xe9, 0x93, 0xd3, 0xcb, 0x6a, 0xe5, 0x9f, 0x61,
			0xf8, 0x91, 0xda, 0x34, 0x31, 0x0d, 0x30, 0x10,
			0x44, 0x1a, 0x71, 0x53, 0xa9, 0xa5, 0xe7, 0xf2,
			0x10, 0xeb, 0xe6, 0xbc, 0x97, 0xe1, 0xa4, 0xe3,
			0x3f, 0xd3, 0x4b, 0xb8, 0xa1, 0x4b, 0x4d, 0xb6,
			0xdd, 0x34, 0xf8, 0xc2, 0xd4, 0x3f, 0x4a, 0xb1,
			0x97, 0x86, 0x06, 0x0b, 0x1e, 0x70, 0x07, 0x0e,
			0x3e, 0xd4, 0xd5, 0xf6, 0xd5, 0x61, 0x76, 0x7c,
			0x48, 0x3d, 0x87, 0x9d, 0x2f, 0xec, 0x8b, 0x9c,
		},
		/* S3 : 384 byte */
		{
			0xc3, 0x89, 0x61, 0x37, 0x17, 0xec, 0x74, 0x76,
			0xec, 0xda, 0x21, 0x44, 0xd0, 0xe8, 0xc8, 0xf9,
			0xd6, 0x6f, 0xb4, 0x69, 0xc1, 0x67, 0xc4, 0x20,
			0x9e, 0xc0, 0xbd, 0xee, 0xbf, 0xb4, 0x71, 0x66,
			0x5d, 0x33, 0xda, 0xd4, 0x7b, 0x8f, 0x3c, 0x31,
			0x9a, 0x76, 0xfe, 0x8a, 0x8a, 0x9f, 0x66, 0x2b,
			0x6c, 0x69, 0x0b, 0x74, 0x90, 0x3d, 0x17, 0xf6,
			0x1e, 0x23, 0x14, 0xe5, 0xea, 0x8d, 0x26, 0x67,
			0x0e, 0xe4, 0xdb, 0x4d, 0xad, 0x29, 0x5b, 0x27,
			0x7c, 0xa0, 0x8a, 0xde, 0x88, 0x0d, 0xe2, 0xe4,
			0x2d, 0x12, 0xb9, 0x29, 0x52, 0x76, 0x4c, 0x1d,
			0xc8, 0x08, 0xc2, 0x66, 0xdb, 0xbe, 0xdb, 0x67,
			0x01, 0x58, 0xee, 0xf3, 0x6e, 0x89, 0x6f, 0x55,
			0xa2, 0x03, 0xfb, 0x99, 0x55, 0x6d, 0xed, 0x05,
			0x97, 0x41, 0x0b, 0xa3, 0x74, 0x86, 0xb1, 0xd8,
			0x41, 0xf3, 0xd6, 0xd5, 0xc0, 0xb3, 0x9f, 0x2f,
			0x49, 0xf0, 0xc5, 0x79, 0x48, 0x24, 0xfb, 0xa9,
			0x4a, 0x8e, 0xc7, 0xc2, 0xb2, 0xc9, 0x1e, 0xad,
			0xd5, 0xc8, 0xcb, 0xe4, 0x48, 0x95, 0xfe, 0x3b,
			0xe3, 0xbc, 0x17, 0x27, 0xd6, 0xfc, 0x0e, 0x53,
			0x64, 0xf5, 0x35, 0x78, 0x63, 0x9d, 0x3b, 0x3a,
			0xf6, 0x96, 0xb7, 0x50, 0xa0, 0x78, 0x53, 0x69,
			0x4f, 0xfe, 0x14, 0x5a, 0x28, 0xc0, 0x36, 0x20,
			0xc7, 0x8d, 0xd7, 0x37, 0x7d, 0x09, 0x4d, 0x92,
			0xc3, 0xe0, 0x95, 0x46, 0x88, 0x3d, 0x47, 0x03,
			0xe6, 0x2a, 0x98, 0xdd, 0xf8, 0x1f, 0xd0, 0x1f,
			0xcd, 0xf3, 0xc4, 0xb2, 0x15, 0x22, 0x4f, 0xe2,
			0xb1, 0xb4, 0x99, 0x2a, 0xbf, 0x31, 0xf2, 0x0d,
			0x12, 0xaf, 0xa8, 0x68, 0x20, 0x23, 0x90, 0xde,
			0x33, 0x4a, 0x84, 0x6b, 0x2d, 0x58, 0xb2, 0x53,
			0xea, 0x8a, 0xb3, 0xc5, 0x26, 0x5d, 0x84, 0x77,
			0x3a, 0x65, 0x9e, 0x8b, 0xac, 0x7a, 0xf4, 0x41,
			0x23, 0xd9, 0xea, 0x15, 0x06, 0x2e, 0x65, 0xd4,
			0xd4, 0x19, 0xcf, 0x2d, 0x97, 0x07, 0x7d, 0x06,
			0x24, 0xf8, 0xe5, 0xc3, 0x6f, 0x2c, 0x7b, 0x35,
			0xcc, 0xf9, 0x54, 0x35, 0xd5, 0xc3, 0x68, 0x86,
			0xff, 0x91, 0x05, 0xa6, 0xc1, 0xea, 0x22, 0x5e,
			0x15, 0xea, 0x8c, 0xbc, 0x7b, 0x6b, 0xf6, 0x85,
			0x61, 0x51, 0xcd, 0x76, 0xfb, 0xb7, 0x5b, 0x5b,
			0x98, 0xf0, 0xe3, 0xdb, 0x51, 0x6a, 0x8e, 0x21,
			0x81, 0x89, 0xfc, 0xb1, 0xcd, 0x5d, 0xe3, 0xca,
			0xfe, 0xaa, 0x33, 0xef, 0x13, 0x5c, 0x5d, 0x8b,
			0x8a, 0xa5, 0xf8, 0x81, 0xaf, 0xaa, 0xca, 0xf4,
			0xc0, 0x8b, 0xd7, 0x28, 0x12, 0x55, 0xbc, 0x2a,
			0x33, 0xb7, 0x6d, 0x4a, 0x36, 0xe0, 0xb1, 0x70,
			0xc4, 0x55, 0x88, 0x23, 0x9e, 0x5b, 0x38, 0xc6,
			0x79, 0xb0, 0x8c, 0xf8, 0x02, 0xaf, 0x73, 0xb6,
			0xd7, 0x9b, 0x39, 0x35, 0x94, 0x94, 0x61, 0xe7,
		},
	},
	/* SHA384 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0xab, 0x66, 0xcc, 0x48, 0x7e, 0xc9, 0x51, 0xf2,
			0x11, 0x9d, 0x6e, 0x0f, 0xa1, 0x7a, 0x6d, 0x8f,
			0xeb, 0x7d, 0x07, 0x14, 0x9b, 0xec, 0x7d, 0xb2,
			0x07, 0x18, 0xe4, 0xf3, 0x1d, 0x88, 0xc0, 0x1f,
			0x9a, 0x53, 0xd5, 0xba, 0x7e, 0xce, 0x3a, 0x4d,
			0xbc, 0x67, 0xaf, 0x6a, 0x35, 0xd1, 0x30, 0xea,
			0xe7, 0x62, 0xcb, 0x79, 0x62, 0xb9, 0xae, 0x55,
			0x7c, 0xa3, 0x84, 0x52, 0x46, 0x40, 0x02, 0x22,
			0x3f, 0x61, 0xbc, 0xd3, 0xc7, 0x35, 0x3e, 0x99,
			0xd6, 0x25, 0x58, 0xce, 0xed, 0xfc, 0xb9, 0x37,
			0x4d, 0x4b, 0xbf, 0x89, 0x68, 0x0c, 0x8e, 0x2b,
			0x95, 0x85, 0x60, 0x3e, 0x07, 0x6f, 0x1c, 0xdb,
			0x00, 0x58, 0x29, 0x9b, 0x42, 0x46, 0x84, 0x5d,
			0xc7, 0x9d, 0x10, 0x43, 0xb1, 0x42, 0x2e, 0xfe,
			0x84, 0x01, 0x8e, 0x4c, 0x93, 0x2c, 0x45, 0xbe,
			0xb8, 0x85, 0x1f, 0xbf, 0x48, 0x5e, 0x36, 0xd2,
		},
		/* S4 : 384 byte */
		{
			0xb5, 0x13, 0x31, 0x55, 0x2b, 0x08, 0xbe, 0x35,
			0xa1, 0x69, 0x8a, 0xa6, 0x20, 0x3d, 0x84, 0xdb,
			0xff, 0xf9, 0x00, 0x1e, 0xd5, 0xdd, 0x77, 0x6f,
			0x2b, 0xe4, 0xdd, 0xfc, 0x07, 0xdd, 0x46, 0x20,
			0xe9, 0x65, 0x4e, 0x82, 0xa3, 0x34, 0x65, 0xbd,
			0x20, 0xf1, 0x18, 0x63, 0xc0, 0xed, 0x02, 0xa0,
			0xae, 0xa2, 0x7a, 0x44, 0xd4, 0x14, 0xc3, 0x28,
			0xa9, 0x38, 0xbf, 0x87, 0x7e, 0x15, 0x83, 0x8a,
			0xb9, 0x9d, 0x67, 0x0d, 0x01, 0x41, 0x42, 0x62,
			0xe8, 0x86, 0x5d, 0xc1, 0xd9, 0xfc, 0x30, 0xfd,
			0x08, 0x12, 0x69, 0x9f, 0xa6, 0x90, 0xc3, 0x4f,
			0x30, 0x2f, 0x63, 0x7e, 0xc8, 0x02, 0xcd, 0x40,
			0xac, 0x85, 0x91, 0xe9, 0x76, 0xc0, 0xb8, 0xbc,
			0xcb, 0x1b, 0x01, 0x37, 0xaf, 0x64, 0xa2, 0x87,
			0x02, 0x10, 0xe8, 0xfa, 0x3d, 0xc4, 0x31, 0xfe,
			0x09, 0x56, 0xb8, 0xad, 0xdf, 0xf1, 0xe4, 0xb1,
			0x8c, 0xf0, 0x7e, 0x07, 0x8a, 0xa9, 0x3a, 0xf8,
			0x1b, 0xb3, 0x02, 0x3c, 0x9e, 0x59, 0x4e, 0x66,
			0x59, 0x5f, 0xd9, 0x2b, 0x10, 0x22, 0x6e, 0xa1,
			0x26, 0x00, 0x5f, 0x47, 0x24, 0x42, 0x73, 0x52,
			0xc3, 0x8e, 0x9e, 0x85, 0xfc, 0x2e, 0x07, 0x23,
			0xf8, 0x0a, 0xf1, 0xf6, 0x15, 0x99, 0x55, 0x0b,
			0x5e, 0xf5, 0x4c, 0x5b, 0x38, 0xca, 0x40, 0x57,
			0x38, 0x01, 0x7b, 0x89, 0xcb, 0x94, 0x68, 0xd9,
			0x74, 0x1c, 0xd6, 0xbd, 0xf7, 0x11, 0x21, 0x62,
			0x25, 0x1b, 0xa1, 0xd0, 0x83, 0xcc, 0x37, 0x0a,
			0x4a, 0x82, 0x61, 0xc3, 0x9b, 0x6b, 0x94, 0xbf,
			0x21, 0xa5, 0x3b, 0x75, 0x64, 0x53, 0x1a, 0xe9,
			0xeb, 0xc4, 0xcc, 0xea, 0x7e, 0xbb, 0x8b, 0xd3,
			0x14, 0xb2, 0xe1, 0x3b, 0x58, 0xed, 0x10, 0x18,
			0xae, 0x5b, 0x41, 0x5e, 0x0f, 0x9e, 0x3e, 0x19,
			0xa5, 0xea, 0xd3, 0xa4, 0x46, 0x03, 0xf9, 0x06,
			0x74, 0xa1, 0x90, 0xfe, 0xbd, 0xe2, 0x5f, 0x8a,
			0xd8, 0x77, 0x8a, 0xee, 0xad, 0x4d, 0x0f, 0x64,
			0xfb, 0xae, 0x37, 0x16, 0x6a, 0x54, 0xe3, 0xa7,
			0x63, 0xe3, 0x55, 0x59, 0xbf, 0x8c, 0x3f, 0x17,
			0x3f, 0x19, 0xff, 0x7b, 0xab, 0x98, 0xf3, 0xef,
			0x80, 0x3d, 0xd5, 0x6c, 0x07, 0x62, 0x83, 0x99,
			0xaf, 0xf8, 0x74, 0x85, 0xee, 0x73, 0xdb, 0xc3,
			0xdb, 0x34, 0xec, 0xc7, 0xbf, 0xf3, 0xa5, 0x32,
			0x26, 0xcf, 0x87, 0xbc, 0x81, 0xd2, 0x56, 0xe8,
			0x0c, 0x09, 0x52, 0x0c, 0x8f, 0x38, 0xe9, 0xbc,
			0xda, 0x09, 0x5e, 0x36, 0x35, 0x12, 0x8e, 0x1b,
			0xed, 0xd9, 0x97, 0x06, 0x00, 0x54, 0x6a, 0x75,
			0x1e, 0xb1, 0x1d, 0xab, 0x42, 0xe2, 0x89, 0xd6,
			0xfd, 0xfe, 0xa0, 0x4b, 0xd5, 0x8d, 0x45, 0x71,
			0xa7, 0x9d, 0x24, 0xbc, 0xe4, 0x50, 0x8c, 0x54,
			0xe1, 0xec, 0x4c, 0xf7, 0x5b, 0x98, 0x5f, 0xd3,
		},
	},
	/* SHA384 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0xfe, 0xf7, 0xfe, 0x89, 0xb9, 0xa5, 0x99, 0x02,
			0xa7, 0x0a, 0x1d, 0x9c, 0xaa, 0xd0, 0x9c, 0xed,
			0x8b, 0xee, 0x41, 0x45, 0xed, 0xcb, 0xe3, 0xef,
			0x7f, 0xa6, 0xda, 0xb3, 0x76, 0x35, 0x12, 0x9f,
			0x3b, 0x8c, 0x5e, 0x08, 0x60, 0x41, 0x0e, 0xcb,
			0xd9, 0xce, 0xc3, 0xd8, 0x69, 0x36, 0x82, 0xf2,
			0x5a, 0xec, 0x08, 0xb0, 0x71, 0xf0, 0x5d, 0xc8,
			0x21, 0x3b, 0xac, 0x8c, 0xff, 0x5d, 0x52, 0xb5,
			0x76, 0x65, 0x35, 0x60, 0xbc, 0x01, 0x57, 0x56,
			0x04, 0xe6, 0xab, 0x90, 0xf6, 0x72, 0x27, 0xfb,
			0x5c, 0x90, 0x1a, 0x78, 0x1e, 0xdd, 0xc0, 0x27,
			0x70, 0x09, 0x13, 0xe5, 0x4a, 0x7f, 0xe5, 0x13,
			0x18, 0x48, 0x2c, 0x9a, 0xb4, 0x2c, 0x9d, 0x2b,
			0x91, 0x1b, 0x7c, 0xcc, 0x39, 0xcc, 0xb2, 0x90,
			0xf9, 0xa4, 0x20, 0xa5, 0xda, 0xd9, 0x33, 0x94,
			0xd4, 0xd7, 0xb8, 0xc5, 0x3f, 0xe3, 0xf2, 0x42,
		},
		/* S5 : 384 byte */
		{
			0x45, 0x06, 0x8c, 0xa6, 0xd8, 0x2f, 0x2c, 0x12,
			0x39, 0x25, 0xcd, 0xe1, 0x19, 0x71, 0x21, 0x5d,
			0x8f, 0xa4, 0xa4, 0xdf, 0x68, 0x48, 0xbb, 0x76,
			0x54, 0x86, 0x87, 0x00, 0x97, 0x87, 0x64, 0x85,
			0x46, 0x38, 0x92, 0x1b, 0xea, 0x58, 0x69, 0x28,
			0x0d, 0xc6, 0xad, 0x95, 0x81, 0xab, 0x43, 0xff,
			0x70, 0x12, 0x96, 0x99, 0x48, 0xa5, 0x67, 0x7f,
			0xa0, 0xa6, 0x61, 0x36, 0xa3, 0x16, 0xa4, 0xbf,
			0xec, 0xb8, 0x9a, 0xdf, 0x41, 0x31, 0xb5, 0xbe,
			0xdf, 0x3d, 0x46, 0x93, 0xb7, 0x80, 0xd1, 0x33,
			0xaf, 0x9b, 0xf9, 0xc1, 0x33, 0x30, 0x5b, 0xe7,
			0x83, 0x74, 0xaf, 0xda, 0x3b, 0xa3, 0x85, 0x42,
			0x03, 0x32, 0x44, 0x81, 0xa9, 0xd1, 0x0b, 0x9c,
			0xa9, 0xb9, 0x2d, 0xc7, 0xd7, 0x4d, 0xf5, 0x31,
			0x87, 0x2d, 0xdf, 0xc7, 0x6c, 0xaa, 0x82, 0xde,
			0x02, 0x0e, 0x2c, 0x41, 0x56, 0x43, 0xcb, 0xcc,
			0x42, 0x80, 0xe6, 0xd2, 0xf4, 0x37, 0x1f, 0xda,
			0x7d, 0x92, 0x49, 0x31, 0x4a, 0x8f, 0x43, 0x76,
			0x48, 0x99, 0x1a, 0x9b, 0x03, 0xd7, 0x1b, 0x58,
			0x39, 0xad, 0x38, 0xa1, 0x55, 0x5a, 0xd3, 0x45,
			0x26, 0x99, 0x4b, 0xa5, 0x68, 0x70, 0xb6, 0xea,
			0x18, 0x01, 0x12, 0x95, 0xf2, 0xca, 0x2b, 0x07,
			0x13, 0xb2, 0xe9, 0x2a, 0xd7, 0x76, 0x80, 0xc0,
			0xdc, 0x5b, 0xed, 0x8d, 0x3b, 0x9b, 0x31, 0xac,
			0x14, 0xdf, 0x76, 0x99, 0x49, 0xc4, 0xa4, 0x3e,
			0xa6, 0x7f, 0x6d, 0xee, 0xb3, 0xdc, 0x9e, 0xd5,
			0x89, 0xea, 0x4e, 0x8a, 0x2c, 0xf6, 0x69, 0x5d,
			0xf4, 0x6f, 0x94, 0x6f, 0x14, 0x67, 0xb2, 0x8e,
			0x87, 0x54, 0x77, 0xae, 0x4e, 0x64, 0x50, 0x80,
			0xfa, 0xfd, 0xa6, 0xdd, 0x55, 0x1d, 0x2c, 0x02,
			0xfd, 0x6b, 0x2b, 0x19, 0x4f, 0xc0, 0xbd, 0xb0,
			0x50, 0xe0, 0x6d, 0x4c, 0x78, 0x41, 0x05, 0xf5,
			0xa3, 0x3b, 0x53, 0xe7, 0x30, 0x98, 0x05, 0x59,
			0x63, 0x07, 0x1e, 0xfc, 0x1b, 0xf3, 0x97, 0xfd,
			0x32, 0x5f, 0x3a, 0x6f, 0x4e, 0x10, 0xd7, 0x6f,
			0x04, 0x11, 0xa0, 0x01, 0xe6, 0x2e, 0xc7, 0x37,
			0x29, 0x01, 0x83, 0x16, 0xf5, 0x63, 0x10, 0xf8,
			0x93, 0xa5, 0x93, 0x63, 0xd1, 0xf6, 0xfe, 0x5c,
			0x17, 0x44, 0x4b, 0x6c, 0x72, 0x8a, 0x49, 0x33,
			0xb7, 0x52, 0x12, 0xfd, 0xfa, 0x25, 0x8e, 0x40,
			0x18, 0xb7, 0x76, 0x39, 0x51, 0xab, 0x4e, 0x50,
			0x96, 0x41, 0x1d, 0xf9, 0xe5, 0xbc, 0x16, 0xdf,
			0x38, 0x96, 0xe4, 0x6c, 0x97, 0x3d, 0x32, 0xac,
			0x92, 0x76, 0xa4, 0xe2, 0xb5, 0xb8, 0x0e, 0x3d,
			0x8d, 0x79, 0x8d, 0xc0, 0x47, 0x0b, 0x45, 0x09,
			0x6b, 0x4d, 0x73, 0x86, 0x69, 0xce, 0x05, 0x2e,
			0xd8, 0x18, 0xe5, 0x60, 0xaf, 0x1e, 0x92, 0xc9,
			0x15, 0x18, 0x7d, 0x66, 0xcc, 0x30, 0x8b, 0x70,
		},
	},
	/* SHA384 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0x82, 0xb3, 0x84, 0x0e, 0xeb, 0x95, 0xc9, 0xc5,
			0x77, 0x24, 0xc7, 0x0f, 0x11, 0x2b, 0x6c, 0x2d,
			0xc6, 0x17, 0xc3, 0x17, 0x85, 0xac, 0xd0, 0xc8,
			0x23, 0xf8, 0xbc, 0xdd, 0xa2, 0x85, 0x32, 0x5e,
			0xb3, 0xd3, 0x08, 0xdc, 0x79, 0x05, 0x22, 0xbc,
			0x90, 0xdb, 0x93, 0xd2, 0x4e, 0xe0, 0x06, 0x32,
			0x49, 0xe5, 0x5d, 0x42, 0x19, 0xad, 0x97, 0x14,
			0x5f, 0xea, 0xf7, 0xf3, 0x06, 0x68, 0x62, 0x3c,
			0xc8, 0x89, 0x0a, 0x70, 0xf4, 0xf1, 0x49, 0x86,
			0x6f, 0x82, 0xcf, 0x86, 0xf9, 0x8b, 0x00, 0x53,
			0xb2, 0x3c, 0x98, 0xc8, 0xdd, 0x5e, 0x91, 0x07,
			0xe3, 0x41, 0x46, 0x0e, 0x9b, 0xf5, 0xd8, 0x8c,
			0xc8, 0xbc, 0xd1, 0xf2, 0xe4, 0xc0, 0x07, 0xcc,
			0x1c, 0x02, 0xc4, 0x52, 0x9b, 0x93, 0x23, 0x3a,
			0x0b, 0x06, 0xbd, 0xd1, 0x59, 0x25, 0x85, 0x4a,
			0xb9, 0xe3, 0xf1, 0x56, 0xeb, 0x92, 0x5b, 0xf5,
		},
		/* S6 : 384 byte */
		{
			0x05, 0x93, 0xb9, 0xfd, 0x44, 0x21, 0x45, 0x23,
			0x76, 0xd2, 0x7b, 0xc7, 0xa2, 0x80, 0x10, 0x1c,
			0xfd, 0x6e, 0x88, 0xa6, 0x72, 0x7d, 0x7d, 0x77,
			0xcf, 0x65, 0xce, 0xb7, 0x23, 0xec, 0xd2, 0x57,
			0xf3, 0x2f, 0xe1, 0x02, 0x77, 0xe8, 0x57, 0x98,
			0xe0, 0xda, 0x75, 0x91, 0x77, 0x36, 0xda, 0x1a,
			0x3b, 0xfc, 0x22, 0xad, 0xc7, 0x65, 0x8f, 0xbb,
			0x84, 0xda, 0x6e, 0xbe, 0xa0, 0xb0, 0x7d, 0x1c,
			0xc4, 0x05, 0x73, 0x2f, 0xb0, 0x40, 0xb5, 0x85,
			0xc1, 0xb6, 0x3c, 0x80, 0x34, 0x06, 0x9b, 0xff,
			0xb8, 0x22, 0x06, 0x56, 0xf1, 0xac, 0x54, 0xce,
			0x69, 0x37, 0x20, 0xd6, 0xfb, 0x1b, 0x5a, 0xec,
			0x67, 0xb0, 0x3c, 0x88, 0x7c, 0x80, 0x77, 0xda,
			0x14, 0x8d, 0x10, 0xf4, 0x8a, 0xf7, 0xc0, 0x28,
			0xf9, 0x92, 0xb1, 0x8f, 0x13, 0xc0, 0xe5, 0x75,
			0x30, 0xc0, 0x86, 0xd7, 0x75, 0x48, 0x3d, 0xa5,
			0xf6, 0x6f, 0x3a, 0x6a, 0x19, 0x18, 0x78, 0x68,
			0x34, 0x0a, 0xc6, 0x3c, 0x62, 0x12, 0xbc, 0xbd,
			0x6c, 0xbb, 0x7b, 0xed, 0xa8, 0x62, 0x0a, 0xfd,
			0x9b, 0x66, 0xde, 0x47, 0x47, 0x3e, 0xf2, 0x4d,
			0x1b, 0x6a, 0x36, 0xf4, 0xec, 0xe9, 0xad, 0xd4,
			0x95, 0x14, 0xfd, 0xf1, 0xd8, 0x4c, 0x7a, 0x78,
			0x5b, 0x7f, 0x0e, 0x00, 0xf3, 0x82, 0x23, 0x58,
			0x99, 0x79, 0x0f, 0x47, 0x2d, 0x13, 0xf4, 0x85,
			0x58, 0xa4, 0x31, 0x47, 0x42, 0xf3, 0x76, 0x80,
			0x8d, 0xec, 0x96, 0xed, 0xd2, 0xe2, 0x29, 0xe9,
			0x43, 0xf7, 0xb9, 0x83, 0xbe, 0xa5, 0xec, 0x6e,
			0xdf, 0xa5, 0xe9, 0xbb, 0x37, 0xf5, 0x88, 0xe5,
			0x5e, 0xf6, 0x2e, 0xbc, 0x92, 0x14, 0xbe, 0xaf,
			0x9d, 0xa5, 0x02, 0x43, 0x4e, 0x10, 0x88, 0xdf,
			0x27, 0x2c, 0x6c, 0x77, 0xc1, 0xe1, 0xd8, 0x97,
			0xc4, 0x7b, 0xea, 0xb7, 0x7e, 0x3b, 0xbe, 0x31,
			0x7f, 0x8d, 0x43, 0xd2, 0x1f, 0xd7, 0xe9, 0x43,
			0x37, 0xc7, 0xe2, 0x63, 0xe2, 0x86, 0x7b, 0xf5,
			0x80, 0xa2, 0xa8, 0xec, 0xb9, 0xe3, 0x6a, 0xb7,
			0xd3, 0xe1, 0xd5, 0xcf, 0x9a, 0x23, 0x23, 0x09,
			0x53, 0xd5, 0x9d, 0xf0, 0xd7, 0xe2, 0x35, 0x58,
			0xfb, 0x61, 0x2b, 0x79, 0x18, 0xab, 0xba, 0x31,
			0xb1, 0x64, 0xce, 0x17, 0x88, 0x18, 0xa1, 0xa9,
			0xe6, 0xb6, 0x68, 0x7f, 0x4d, 0xe6, 0x85, 0xd7,
			0x0e, 0x16, 0xbe, 0xf6, 0xe1, 0x92, 0xfa, 0xed,
			0xfe, 0x0b, 0x2b, 0x95, 0x47, 0x7d, 0x37, 0xb0,
			0xa3, 0xa2, 0xd0, 0x02, 0xf3, 0x3e, 0xf4, 0x32,
			0x1c, 0xb9, 0x05, 0x04, 0x0c, 0xe0, 0x6f, 0xda,
			0x1c, 0x98, 0xa0, 0x08, 0x76, 0x7f, 0xbc, 0x78,
			0x1a, 0x1e, 0xaf, 0x33, 0x75, 0xda, 0xb8, 0x66,
			0x4b, 0x59, 0x03, 0x36, 0xb9, 0x9e, 0x15, 0x7b,
			0x86, 0x87, 0xa6, 0x60, 0x2f, 0xef, 0x6a, 0x3b,
		},
	},
	/* SHA384 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0xe1, 0x53, 0xcc, 0xa4, 0x43, 0x1e, 0xd9, 0x71,
			0x3f, 0x47, 0x44, 0xba, 0x05, 0x4f, 0x5f, 0x19,
			0x1c, 0xb3, 0x7b, 0x28, 0x01, 0x08, 0xae, 0x3a,
			0x11, 0x4a, 0xd3, 0x49, 0xa8, 0x72, 0xd1, 0x30,
			0x8b, 0x46, 0x21, 0x1a, 0x83, 0x75, 0x8a, 0x3b,
			0x4b, 0xe3, 0x2f, 0xbe, 0xac, 0x42, 0xcc, 0xfe,
			0xe7, 0xe2, 0x3d, 0xf8, 0x53, 0xca, 0x40, 0x01,
			0x47, 0x07, 0x7b, 0xb4, 0x3a, 0x44, 0xc1, 0x2f,
			0x29, 0x9b, 0x91, 0x7f, 0x3a, 0xab, 0xdf, 0x58,
			0x9e, 0xeb, 0x17, 0x09, 0xbb, 0x3d, 0x60, 0xb0,
			0x8b, 0xc7, 0x1e, 0xaa, 0x3f, 0xfe, 0xba, 0x4e,
			0x29, 0x03, 0xa5, 0xdb, 0xd8, 0x33, 0x9a, 0xae,
			0x85, 0xfa, 0x24, 0xb9, 0xae, 0xe7, 0x61, 0x30,
			0x00, 0x06, 0x05, 0x85, 0x7a, 0x6a, 0xa1, 0x97,
			0xd0, 0x09, 0x26, 0x27, 0x0d, 0xcd, 0xa5, 0x8b,
			0x7d, 0xe7, 0x58, 0xa6, 0xca, 0x67, 0xe6, 0x17,
		},
		/* S7 : 384 byte */
		{
			0xa8, 0x35, 0xcd, 0x41, 0x46, 0xbe, 0xf4, 0x65,
			0x64, 0x2d, 0x49, 0x49, 0x36, 0x26, 0x8a, 0x31,
			0x1a, 0x54, 0x90, 0xd2, 0xc9, 0xf9, 0x16, 0x6c,
			0x6c, 0xe9, 0x82, 0x16, 0xa9, 0xa2, 0x3a, 0x64,
			0x35, 0x97, 0x30, 0x0a, 0x00, 0x50, 0xe6, 0x44,
			0x5a, 0xbd, 0x5a, 0x9b, 0xfc, 0x7a, 0x2d, 0x9b,
			0x70, 0x72, 0x6c, 0x82, 0x4c, 0x38, 0x3b, 0xf5,
			0xac, 0xad, 0xdd, 0xdc, 0x34, 0xd4, 0x34, 0xa3,
			0x1e, 0x53, 0x14, 0xd2, 0x5f, 0xb5, 0x8e, 0x25,
			0x8f, 0x51, 0x88, 0x66, 0xc1, 0x36, 0xe5, 0x28,
			0x55, 0xc1, 0x6f, 0xe6, 0x4f, 0xf8, 0xf1, 0xc4,
			0xd6, 0x6c, 0x4e, 0x9e, 0x39, 0xb8, 0xcb, 0x11,
			0x96, 0xd8, 0x09, 0x44, 0xd0, 0x74, 0x6c, 0x0a,
			0x3e, 0x17, 0x69, 0xcd, 0x41, 0x67, 0xdf, 0x72,
			0xab, 0x5e, 0x4c, 0x9d, 0xba, 0xe9, 0xcb, 0x35,
			0xf4, 0x82, 0x8e, 0x12, 0x09, 0x9f, 0x9b, 0x36,
			0xa5, 0xa7, 0x0c, 0x48, 0xd4, 0xae, 0xc9, 0x87,
			0x2d, 0x7b, 0x19, 0xe1, 0x29, 0x1b, 0x33, 0xcb,
			0xdf, 0x08, 0xa2, 0x26, 0x3d, 0x50, 0x0c, 0x0a,
			0x83, 0xb5, 0x23, 0x7e, 0xf6, 0xce, 0x92, 0xde,
			0x34, 0x4b, 0x3b, 0x41, 0xd0, 0xd0, 0x74, 0x04,
			0xfc, 0xd5, 0x46, 0x7b, 0x04, 0x6b, 0x52, 0xb8,
			0xf8, 0x5f, 0xc6, 0xb5, 0xd7, 0xaf, 0xc4, 0x37,
			0xf1, 0xee, 0x9e, 0x78, 0x39, 0x0c, 0xa9, 0xbb,
			0x6c, 0xec, 0x61, 0x88, 0x85, 0xec, 0xe2, 0x97,
			0x58, 0xf2, 0xfd, 0x6f, 0x4e, 0x5f, 0x4f, 0x89,
			0x69, 0x35, 0xde, 0x5f, 0x67, 0xcc, 0x04, 0x05,
			0x5a, 0x4c, 0x4c, 0x0f, 0xba, 0x5d, 0xef, 0x8d,
			0x2c, 0xaa, 0x17, 0x93, 0x31, 0xa8, 0x55, 0x01,
			0xed, 0x25, 0x82, 0x2a, 0xe7, 0x9d, 0xa9, 0xbc,
			0x81, 0x5c, 0xc3, 0x9c, 0x6a, 0x97, 0x92, 0x11,
			0x08, 0x3e, 0x86, 0x83, 0x13, 0x6c, 0x94, 0x2e,
			0x1e, 0x17, 0xe9, 0xeb, 0x8f, 0x84, 0xaa, 0xcf,
			0x09, 0x1a, 0xa1, 0xe5, 0x16, 0x65, 0xfa, 0xe4,
			0x46, 0xbc, 0x48, 0xc3, 0x04, 0xaf, 0x65, 0x39,
			0x1f, 0x27, 0x9a, 0xfb, 0x98, 0xb9, 0x2e, 0x04,
			0xc2, 0xb7, 0x3d, 0x9d, 0x94, 0xe9, 0x91, 0x19,
			0x8f, 0xe7, 0x78, 0x1f, 0x0f, 0x96, 0x96, 0xfc,
			0xba, 0x2c, 0x03, 0x48, 0x5f, 0x76, 0xe6, 0xde,
			0x30, 0xb9, 0x53, 0x5c, 0xf3, 0x90, 0x3d, 0xb2,
			0xf3, 0xaf, 0xa8, 0x51, 0xa4, 0x7b, 0xcd, 0xe7,
			0x2d, 0x4e, 0xd2, 0xe8, 0xfa, 0xbf, 0x9b, 0xb7,
			0xd4, 0x69, 0x6c, 0xb4, 0xab, 0x8c, 0x28, 0x9b,
			0x0c, 0x21, 0xe1, 0xf9, 0x79, 0xeb, 0xc5, 0x32,
			0xe2, 0x80, 0xcd, 0x90, 0x10, 0xdf, 0x4e, 0xe7,
			0x2f, 0x84, 0xbb, 0x9e, 0x82, 0x75, 0x28, 0x28,
			0xf1, 0x67, 0x03, 0x0c, 0x0f, 0xe3, 0x48, 0xeb,
			0xc3, 0x1e, 0xc1, 0x7b, 0x8f, 0x07, 0xd9, 0x4b,
		},
	},
	/* SHA384 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0x9c, 0x63, 0x89, 0x9d, 0xfc, 0x7b, 0xdc, 0x0d,
			0xb3, 0x84, 0x72, 0x72, 0x44, 0xca, 0xf7, 0x1e,
			0xcf, 0xb9, 0xb8, 0x79, 0x2b, 0x9f, 0x57, 0xe9,
			0x36, 0xb3, 0xc2, 0xf5, 0x69, 0x55, 0x65, 0xa9,
			0xb0, 0x97, 0x9f, 0x3c, 0x78, 0xfd, 0x73, 0xf0,
			0x09, 0x81, 0x81, 0x3a, 0x16, 0xda, 0x34, 0x23,
			0x92, 0xfe, 0x3c, 0xee, 0xc6, 0xe6, 0x3f, 0xfb,
			0xa1, 0x91, 0xcb, 0xeb, 0x4f, 0x4b, 0x90, 0x05,
			0x0d, 0x2f, 0xcc, 0xd8, 0x3b, 0xeb, 0x06, 0x22,
			0xb2, 0xc3, 0xff, 0xf1, 0x59, 0xd9, 0xe6, 0x08,
			0xf3, 0xab, 0xcb, 0x84, 0x3b, 0xdd, 0x56, 0xc0,
			0x33, 0x39, 0xb9, 0x75, 0xb9, 0xf4, 0xe3, 0x26,
			0x5b, 0x32, 0xf6, 0xbb, 0x6c, 0xcd, 0xfc, 0x6c,
			0x57, 0x52, 0xd6, 0xe0, 0x34, 0x4d, 0x74, 0x96,
			0x99, 0xc7, 0x4c, 0x85, 0xb3, 0x0c, 0x04, 0xff,
			0x95, 0xb2, 0x72, 0xdb, 0xcf, 0xd6, 0xc7, 0xd3,
		},
		/* S8 : 384 byte */
		{
			0x4d, 0x38, 0xa2, 0x97, 0x30, 0x2a, 0xd0, 0x77,
			0x0d, 0x97, 0x29, 0xce, 0x5b, 0x72, 0x12, 0xee,
			0xf2, 0x87, 0xce, 0x02, 0x50, 0xf4, 0x03, 0xe3,
			0x2b, 0x4a, 0xcc, 0x36, 0x17, 0xdc, 0x0d, 0x2e,
			0xdc, 0xcc, 0xc2, 0xd5, 0x80, 0xdd, 0xbd, 0xbc,
			0xa5, 0x72, 0x2b, 0x70, 0x70, 0x40, 0x58, 0xa3,
			0xb8, 0x07, 0xf5, 0x92, 0xe4, 0x00, 0xbd, 0x56,
			0x3f, 0xca, 0xa8, 0xb0, 0x66, 0xa6, 0x14, 0xb4,
			0x90, 0x6f, 0x14, 0x33, 0x96, 0x8e, 0xd2, 0xf5,
			0x20, 0xa2, 0xf6, 0xb0, 0x34, 0xd4, 0xb2, 0xd6,
			0x89, 0x0a, 0x24, 0x1a, 0xfd, 0x1a, 0xdb, 0x86,
			0x39, 0xa6, 0xca, 0xd9, 0xdb, 0xfd, 0x2e, 0x27,
			0x8d, 0xfe, 0xbf, 0x79, 0x74, 0x0d, 0x75, 0xf2,
			0x95, 0x75, 0x9d, 0x29, 0x13, 0x0b, 0x19, 0xab,
			0x19, 0x98, 0x3d, 0xd6, 0x8f, 0x77, 0x9d, 0xe4,
			0x1f, 0xfe, 0xfd, 0x4e, 0x82, 0xb5, 0xe6, 0x2f,
			0x72, 0xf9, 0x0e, 0xfb, 0x73, 0x43, 0x7f, 0x08,
			0xa2, 0x50, 0x3d, 0xd9, 0x81, 0x9d, 0xae, 0x20,
			0xba, 0x97, 0x06, 0xc1, 0x99, 0xde, 0x9c, 0xf8,
			0x84, 0x43, 0x3e, 0xeb, 0x75, 0x62, 0x86, 0xa8,
			0x5e, 0xae, 0x14, 0xbf, 0x9f, 0x6d, 0xbe, 0xb7,
			0x05, 0x46, 0x1d, 0x91, 0x82, 0x22, 0x82, 0xf1,
			0x8e, 0xfb, 0xb1, 0x05, 0x89, 0xa5, 0x78, 0xf2,
			0xc9, 0xc3, 0x45, 0xb0, 0x79, 0xa7, 0xe9, 0xdd,
			0x07, 0xfd, 0x4b, 0x34, 0x05, 0x1b, 0x27, 0x11,
			0x97, 0x29, 0x90, 0x6c, 0x77, 0xdf, 0xb7, 0xd2,
			0xf8, 0xfa, 0x6b, 0xdd, 0x5f, 0xaa, 0x1e, 0x13,
			0x2b, 0xfb, 0xa9, 0xd3, 0x91, 0xe6, 0x63, 0x95,
			0xe6, 0x7f, 0x01, 0x35, 0x3f, 0xa2, 0x75, 0xea,
			0xce, 0x8b, 0x53, 0xaa, 0x91, 0xcb, 0x6f, 0xb6,
			0x93, 0xe1, 0x91, 0x91, 0xd4, 0x2a, 0x4c, 0x1a,
			0x85, 0xa0, 0xc5, 0x04, 0xb1, 0xc8, 0x5f, 0x49,
			0xa4, 0xd6, 0x09, 0x36, 0xde, 0xe4, 0x64, 0x6a,
			0xca, 0x62, 0xa9, 0x4a, 0xa4, 0xbc, 0x78, 0x28,
			0xc1, 0xff, 0xaf, 0xde, 0x8b, 0xe6, 0x56, 0x31,
			0x7d, 0x50, 0x6a, 0xbe, 0xc1, 0x79, 0xcc, 0x90,
			0x19, 0x1d, 0x12, 0x35, 0x6f, 0xf5, 0x06, 0x44,
			0xd3, 0xe0, 0x1a, 0xa5, 0xbc, 0xfd, 0xd7, 0x1d,
			0x3c, 0x82, 0x8d, 0xc3, 0x53, 0x9d, 0xc0, 0xcf,
			0x3f, 0xe8, 0xb9, 0xb9, 0x1e, 0x0c, 0x25, 0x24,
			0xf6, 0xa3, 0x71, 0x03, 0x79, 0xc9, 0x0a, 0xff,
			0xd0, 0xd0, 0xa5, 0x0d, 0x74, 0x38, 0x7f, 0x9c,
			0xa8, 0x8b, 0x46, 0x46, 0x3e, 0xf1, 0xbd, 0xba,
			0x58, 0xcc, 0x9a, 0x36, 0xe5, 0xc2, 0xc4, 0x35,
			0xa2, 0x0d, 0x96, 0x83, 0x50, 0xd1, 0x5d, 0x94,
			0x1c, 0x32, 0x12, 0xcd, 0xce, 0x81, 0x55, 0x92,
			0xb3, 0x10, 0xd2, 0x59, 0x86, 0x0d, 0xe1, 0xdc,
			0x1a, 0x3d, 0x70, 0xac, 0x22, 0x30, 0x2a, 0x51,
		},
	},
	/* SHA384 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0x04, 0x84, 0x6c, 0x2e, 0x67, 0x6a, 0xc7, 0x31,
			0x60, 0xbf, 0x4e, 0x45, 0x65, 0x2b, 0xdc, 0x6c,
			0xc4, 0xd4, 0xc9, 0x28, 0x45, 0x77, 0xb4, 0x32,
			0x0a, 0xb7, 0x7f, 0x6e, 0xbb, 0xb5, 0x9a, 0x1f,
			0xe0, 0xe0, 0x85, 0x58, 0x8e, 0x0f, 0x90, 0xb3,
			0x46, 0xcd, 0xe6, 0x44, 0x1a, 0xf3, 0xc9, 0xd0,
			0x11, 0x7d, 0x1f, 0x3b, 0xcd, 0x96, 0x2e, 0x40,
			0x6b, 0xf5, 0xa4, 0x65, 0xab, 0x6c, 0xda, 0x2d,
			0x51, 0xbe, 0x59, 0x8f, 0xcb, 0xb2, 0x9e, 0xa7,
			0x13, 0x65, 0x1a, 0xac, 0xd7, 0xe4, 0x7d, 0x22,
			0xd8, 0xfa, 0x34, 0x50, 0x90, 0x47, 0x30, 0xf5,
			0x17, 0x92, 0xea, 0x37, 0x47, 0x61, 0xa4, 0xdc,
			0x1f, 0xc6, 0xf1, 0xbc, 0x65, 0x7b, 0x77, 0x76,
			0x8f, 0x31, 0xf4, 0x63, 0xe4, 0x26, 0x7f, 0xc8,
			0xdf, 0xf6, 0x11, 0x50, 0xd4, 0xb3, 0x43, 0xb9,
			0xd5, 0x37, 0x59, 0xcd, 0xd7, 0xb9, 0x80, 0x94,
		},
		/* S9 : 384 byte */
		{
			0x10, 0x3b, 0xee, 0x57, 0xe2, 0x5b, 0xe8, 0xc3,
			0xa2, 0xf7, 0x74, 0xe7, 0x39, 0xb4, 0x7f, 0x93,
			0x43, 0x5e, 0x41, 0x49, 0x32, 0xc0, 0x49, 0x4b,
			0x6b, 0x6a, 0xa2, 0x47, 0x5b, 0xf7, 0xc9, 0x30,
			0x5c, 0x73, 0x74, 0x7e, 0x0a, 0xdf, 0x82, 0xc2,
			0x03, 0x20, 0x07, 0xb3, 0xf7, 0x5a, 0x69, 0xc9,
			0x31, 0x12, 0x61, 0x7a, 0x62, 0x56, 0x6c, 0x5a,
			0x2d, 0xea, 0xa2, 0x5f, 0xb9, 0x52, 0x09, 0xda,
			0x49, 0xfe, 0x9c, 0x16, 0x1c, 0xb2, 0xff, 0xa4,
			0x0f, 0xd9, 0xd7, 0x7f, 0x1f, 0xf6, 0x60, 0xc8,
			0xb6, 0xcd, 0x3b, 0x54, 0xe3, 0xe7, 0x9a, 0x75,
			0x9c, 0x57, 0xc5, 0x71, 0x98, 0x02, 0xc9, 0x31,
			0x1d, 0xb7, 0x04, 0xba, 0x3c, 0x67, 0xb4, 0xa3,
			0x11, 0x37, 0x54, 0xa4, 0x1b, 0x8d, 0xa5, 0x9c,
			0x64, 0x5b, 0xe3, 0x90, 0x9e, 0x7d, 0xb7, 0xe7,
			0xcf, 0x72, 0x94, 0xda, 0xb4, 0x4f, 0x74, 0x24,
			0x0f, 0x81, 0xa2, 0x81, 0xee, 0xcd, 0x6e, 0xf3,
			0x1c, 0x7c, 0xf1, 0x8b, 0x1a, 0x19, 0xc7, 0xd0,
			0x2a, 0x31, 0x2b, 0x91, 0xd6, 0xed, 0xfa, 0xa9,
			0x54, 0x46, 0x2d, 0x34, 0x74, 0x0a, 0xf5, 0xab,
			0x70, 0x8d, 0xb5, 0xa1, 0x0b, 0x00, 0xc5, 0x42,
			0xbe, 0x82, 0xfa, 0x2b, 0x20, 0x26, 0xb0, 0x9e,
			0xf3, 0x8a, 0x40, 0x01, 0x45, 0x7e, 0x27, 0xa6,
			0x02, 0x37, 0x70, 0xe4, 0xb4, 0xd5, 0x00, 0x32,
			0x67, 0xc8, 0x5c, 0x9e, 0xea, 0x1d, 0x5f, 0x8d,
			0x77, 0x0b, 0xd4, 0x0b, 0x55, 0x4d, 0x5b, 0x4d,
			0xaf, 0x14, 0x6d, 0xcc, 0xab, 0xac, 0x3e, 0xa8,
			0xa1, 0x3a, 0x05, 0xc3, 0xbd, 0xdf, 0xc9, 0x71,
			0xc5, 0x15, 0x8f, 0xac, 0x02, 0x7c, 0xa1, 0x9b,
			0x72, 0x32, 0x62, 0x1e, 0x9d, 0x2e, 0x37, 0xb6,
			0xa6, 0x55, 0xaf, 0x54, 0x5e, 0x44, 0xa2, 0x98,
			0xbe, 0x78, 0xcd, 0x47, 0x5c, 0x22, 0xa4, 0x8b,
			0xff, 0x7c, 0x34, 0x94, 0xa5, 0xf8, 0xa6, 0xab,
			0xdf, 0x1a, 0x46, 0xf9, 0xde, 0x08, 0x2e, 0x37,
			0x4f, 0xd5, 0x98, 0x86, 0x7d, 0x61, 0xe4, 0xd5,
			0x1d, 0xae, 0xd8, 0x41, 0x52, 0xe4, 0x3c, 0xc6,
			0xa2, 0xaf, 0xfa, 0xe2, 0x05, 0xed, 0xc5, 0x26,
			0x13, 0x48, 0x0d, 0x41, 0x1a, 0xba, 0x84, 0xfc,
			0xc9, 0xb6, 0x9d, 0x1c, 0x28, 0xf1, 0x6f, 0x76,
			0x83, 0x69, 0x01, 0xa7, 0xc5, 0xb3, 0xeb, 0x2f,
			0x2c, 0x94, 0x0d, 0x0a, 0x3f, 0xad, 0x38, 0xa8,
			0xef, 0xab, 0x96, 0x8a, 0x0c, 0x85, 0xeb, 0x22,
			0xe1, 0x1d, 0x3d, 0x08, 0x61, 0x13, 0x6c, 0xed,
			0x5f, 0x06, 0x73, 0x4f, 0xdf, 0x8d, 0x4f, 0x15,
			0x1d, 0x23, 0x86, 0x1b, 0x1c, 0xba, 0x9b, 0x9c,
			0x58, 0x0d, 0x33, 0x50, 0xc7, 0x6d, 0x4d, 0xc8,
			0x08, 0x46, 0x1d, 0x5f, 0x87, 0x2e, 0xc5, 0x48,
			0xb2, 0xb4, 0x27, 0xdf, 0xf7, 0x4b, 0x1d, 0x1a,
		},
	},
};

/* 10 elements */
static mod3072_t SHA512[] = {
	/* SHA512 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0xdb, 0x6c, 0x9d, 0x4b, 0xad, 0xb1, 0xd9, 0xb7,
			0x4d, 0x68, 0x34, 0x64, 0x48, 0xb4, 0xd5, 0x34,
			0x06, 0x31, 0x78, 0x3b, 0x5a, 0x35, 0xac, 0x24,
			0x58, 0x56, 0x3e, 0xd0, 0x67, 0x2c, 0xf5, 0x41,
			0x97, 0x58, 0x7f, 0xb7, 0x34, 0xc4, 0xac, 0x18,
			0x9b, 0x2d, 0xda, 0x95, 0x4c, 0xdf, 0xb1, 0x8b,
			0x41, 0xc0, 0x10, 0xa7, 0x7e, 0x90, 0x46, 0x4e,
			0xea, 0x6f, 0x86, 0x3c, 0x5d, 0xa0, 0x95, 0x6b,
			0xfa, 0x8c, 0xc6, 0x36, 0xbf, 0x0a, 0x28, 0xbe,
			0x5a, 0xdd, 0xfe, 0x8d, 0x3e, 0x7e, 0x6f, 0x79,
			0xf7, 0x1d, 0x7f, 0xcb, 0xba, 0xe2, 0x3e, 0xa1,
			0x41, 0x78, 0x3f, 0x91, 0xd6, 0xcc, 0x4c, 0x8f,
			0xad, 0x12, 0x58, 0x11, 0x76, 0x0a, 0xb5, 0x71,
			0x33, 0x81, 0x88, 0x92, 0x47, 0x1a, 0x79, 0xc6,
			0xd0, 0x4e, 0xaf, 0xef, 0x37, 0xb2, 0xfb, 0xe5,
			0x06, 0x78, 0x53, 0x18, 0xf9, 0x39, 0x83, 0x77,
		},
		/* S0 : 384 byte */
		{
			0xd4, 0x80, 0xd5, 0xa9, 0x79, 0xad, 0x1a, 0x0c,
			0x4c, 0xa3, 0x29, 0xeb, 0xd8, 0x8a, 0x4a, 0xa6,
			0x94, 0x8a, 0x8c, 0xf6, 0x6a, 0x3c, 0x0b, 0xfe,
			0xe2, 0x25, 0x44, 0x09, 0xc5, 0x30, 0x54, 0xd6,
			0xff, 0xf5, 0x9f, 0x72, 0xa4, 0x6f, 0x02, 0xc6,
			0x68, 0x14, 0x6a, 0x14, 0x4f, 0x8f, 0x2b, 0xa7,
			0xc4, 0xe6, 0xb4, 0xde, 0x31, 0x40, 0x0e, 0xba,
			0x00, 0xae, 0x3e, 0xe8, 0x75, 0x89, 0xdc, 0xb6,
			0xea, 0x13, 0x9e, 0x70, 0xf7, 0x70, 0x4f, 0x69,
			0x1b, 0xc3, 0x7d, 0x72, 0x2f, 0x62, 0xbb, 0x3b,
			0x2c, 0xd3, 0x03, 0xa3, 0x4d, 0x92, 0xfd, 0xe4,
			0xde, 0xb5, 0x4a, 0x64, 0xdd, 0x39, 0x18, 0x43,
			0x82, 0xd5, 0x9c, 0xca, 0xf0, 0xc0, 0x7a, 0x7e,
			0xa4, 0x10, 0x7d, 0x08, 0x08, 0x26, 0x0e, 0xd8,
			0xd4, 0x21, 0xcb, 0x8b, 0x14, 0x07, 0xcd, 0xf9,
			0xe9, 0x15, 0x15, 0x92, 0x82, 0xb9, 0xf7, 0xbf,
			0xfd, 0xbf, 0x40, 0xd8, 0x77, 0x88, 0x5d, 0xa7,
			0x39, 0x9e, 0xde, 0xbd, 0x30, 0x0a, 0x7e, 0x77,
			0xa9, 0x08, 0xf7, 0x56, 0x65, 0x9a, 0x18, 0x24,
			0xf9, 0x5c, 0x8a, 0x81, 0x2a, 0xa5, 0x40, 0xeb,
			0xaa, 0x64, 0xab, 0x54, 0xa2, 0x33, 0x72, 0x3d,
			0xb5, 0x5c, 0xaa, 0x8b, 0x44, 0x66, 0xea, 0x9a,
			0xe6, 0x61, 0x4a, 0xd1, 0xbb, 0x86, 0x9e, 0x9d,
			0x8e, 0x0d, 0x03, 0x2f, 0x39, 0x01, 0x67, 0x1e,
			0x94, 0xc0, 0xb6, 0x73, 0xbe, 0x65, 0x37, 0xcd,
			0x54, 0x27, 0x8e, 0xd3, 0xda, 0x2e, 0x1e, 0xdb,
			0xc0, 0x4e, 0xe3, 0xa9, 0xe8, 0x07, 0x0d, 0x73,
			0xba, 0x0f, 0xfb, 0x93, 0xe6, 0x0f, 0x30, 0xb8,
			0x7f, 0xf3, 0x86, 0x2e, 0x9c, 0x53, 0x90, 0x8f,
			0x2c, 0x8e, 0x99, 0x91, 0x56, 0x68, 0xc1, 0xf4,
			0x66, 0x35, 0xe0, 0x5b, 0xf7, 0x16, 0x30, 0x51,
			0xff, 0x9d, 0x92, 0xbc, 0x71, 0xa6, 0x26, 0x55,
			0x3c, 0x69, 0xdf, 0xdd, 0x06, 0xa4, 0x9f, 0x7f,
			0xf1, 0xed, 0x51, 0xe9, 0x18, 0xf3, 0xed, 0x80,
			0x1d, 0xae, 0x62, 0xca, 0x27, 0x6d, 0x70, 0x63,
			0xd7, 0x2a, 0x6e, 0xbc, 0x13, 0x6b, 0xa0, 0x6c,
			0xfe, 0xdf, 0x5a, 0xa2, 0x32, 0x77, 0xe8, 0x10,
			0x08, 0xc6, 0x3b, 0x2e, 0x00, 0x83, 0xd0, 0xfd,
			0x68, 0x14, 0xf6, 0xd4, 0xb4, 0xb4, 0x0a, 0x42,
			0xe8, 0xc0, 0x20, 0x6f, 0x3c, 0x35, 0x6a, 0x5e,
			0xc7, 0x09, 0xb7, 0xc8, 0xa4, 0xb7, 0x4b, 0x7b,
			0x48, 0xd5, 0x3c, 0x9d, 0x86, 0x94, 0xd2, 0x73,
			0x59, 0xc2, 0xc7, 0x70, 0x19, 0x38, 0xd2, 0xf0,
			0x16, 0x17, 0x21, 0xa5, 0x73, 0x13, 0xbb, 0x1a,
			0x2e, 0x11, 0xda, 0x21, 0x58, 0x72, 0x49, 0x81,
			0x82, 0x49, 0x3d, 0x85, 0x17, 0x04, 0x3b, 0x4c,
			0x03, 0xf9, 0x34, 0x46, 0xaa, 0xc9, 0x38, 0x30,
			0x27, 0x65, 0x42, 0x02, 0x6c, 0xe8, 0x30, 0x55,
		},
	},
	/* SHA512 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0xd5, 0xdd, 0x3b, 0x6c, 0xe9, 0x77, 0x2d, 0x9a,
			0x97, 0xfe, 0x21, 0x64, 0x84, 0x97, 0x78, 0x3b,
			0xac, 0x5b, 0xb5, 0x25, 0x4a, 0xad, 0x82, 0xb6,
			0xf7, 0xcb, 0xf4, 0x3b, 0x15, 0xa4, 0x0f, 0x38,
			0x6e, 0xea, 0x8d, 0x15, 0x19, 0x67, 0xdb, 0x14,
			0x9e, 0x94, 0x65, 0x86, 0x59, 0x68, 0x13, 0x3f,
			0x24, 0x6e, 0x13, 0x47, 0x30, 0x1a, 0xda, 0xd2,
			0x34, 0x5d, 0x65, 0x72, 0xca, 0x77, 0xc5, 0x8c,
			0x15, 0x0d, 0xda, 0x09, 0xa8, 0x7b, 0x5f, 0x4d,
			0xa3, 0x6b, 0x26, 0x6d, 0x1f, 0xa7, 0xa5, 0x9c,
			0xcd, 0x2b, 0xb2, 0xe7, 0xd9, 0x7f, 0x8b, 0x23,
			0x15, 0x43, 0x19, 0x23, 0x53, 0x0b, 0x76, 0x2e,
			0x12, 0x6e, 0xac, 0xaf, 0x5e, 0x5a, 0xc0, 0x2f,
			0xf1, 0xaa, 0xef, 0x81, 0x9e, 0xfb, 0x37, 0x3c,
			0xf0, 0xbb, 0x19, 0x6f, 0x0e, 0x82, 0x9e, 0x8f,
			0xe1, 0xa6, 0x98, 0xb4, 0x79, 0x0a, 0x2a, 0x05,
		},
		/* S1 : 384 byte */
		{
			0xbf, 0x9e, 0x8b, 0x4f, 0x2a, 0xe5, 0x13, 0xf7,
			0x3d, 0x78, 0x89, 0x58, 0x00, 0x37, 0x33, 0xdb,
			0xe2, 0x09, 0x57, 0xb1, 0x47, 0xb1, 0x7c, 0x3f,
			0x4f, 0xd6, 0xd0, 0x24, 0xe8, 0xe8, 0x3f, 0x07,
			0xb6, 0x5d, 0x9f, 0x3d, 0xbc, 0x3b, 0x1f, 0xe8,
			0x4d, 0xa0, 0x21, 0xce, 0xab, 0xfc, 0xcd, 0x8c,
			0x57, 0xa0, 0x14, 0xfb, 0xe5, 0xa2, 0xbc, 0xe3,
			0xe4, 0x05, 0x1b, 0x7d, 0x03, 0xe0, 0x9f, 0xc0,
			0x35, 0x0b, 0x6a, 0x21, 0xfa, 0xd2, 0x14, 0xae,
			0x7a, 0x07, 0x32, 0x77, 0xc7, 0x7a, 0x40, 0xdc,
			0x44, 0xa5, 0xae, 0xea, 0x51, 0x94, 0xa7, 0x56,
			0xb6, 0x9c, 0x93, 0x97, 0x7b, 0x69, 0xee, 0x92,
			0x94, 0x36, 0x0e, 0xaa, 0x73, 0xa5, 0x74, 0x54,
			0x8f, 0xa6, 0xa9, 0x74, 0xa7, 0xcd, 0x5a, 0x6a,
			0xdc, 0xf0, 0x9e, 0x80, 0x63, 0x11, 0x56, 0xaf,
			0x85, 0xa8, 0xe5, 0xc5, 0x31, 0x7e, 0x18, 0x9e,
			0xea, 0xd4, 0x7e, 0x2e, 0xad, 0x65, 0xc3, 0x81,
			0x39, 0x6b, 0x5c, 0xac, 0xde, 0x26, 0x0e, 0x93,
			0x72, 0x84, 0xa8, 0xe9, 0x0e, 0xff, 0x2c, 0xbc,
			0xb9, 0xde, 0xe2, 0x29, 0x25, 0xf2, 0xf7, 0x25,
			0x6f, 0x74, 0xc6, 0x7c, 0xf3, 0xff, 0xc7, 0xb8,
			0xce, 0x65, 0x7e, 0x8d, 0x13, 0x5f, 0x0f, 0x37,
			0x6d, 0x9d, 0x93, 0x6a, 0x79, 0x79, 0x2c, 0x98,
			0x16, 0x14, 0xd9, 0x8e, 0x3f, 0x7d, 0x66, 0x2a,
			0x4f, 0xd4, 0x6d, 0xcd, 0xa9, 0x69, 0x16, 0xb3,
			0x2f, 0x36, 0x6e, 0xd2, 0x7d, 0xab, 0x18, 0x8f,
			0x18, 0x4b, 0x98, 0x4d, 0xf0, 0xb5, 0x59, 0x71,
			0x0d, 0x8f, 0xf2, 0x04, 0x0b, 0xe4, 0x62, 0xf9,
			0x19, 0x43, 0x50, 0x1b, 0xda, 0x48, 0x40, 0xfd,
			0xd5, 0xc8, 0xec, 0x15, 0xd1, 0x89, 0x06, 0x4d,
			0xef, 0x75, 0x6e, 0x54, 0x5d, 0xb3, 0x19, 0xe0,
			0x07, 0xc4, 0x33, 0xf0, 0x46, 0x8a, 0x67, 0x23,
			0x35, 0x7b, 0xa4, 0x7d, 0x15, 0x6a, 0xb7, 0x65,
			0x2b, 0x06, 0xae, 0x2b, 0x18, 0x87, 0x4f, 0x07,
			0x71, 0xc6, 0x26, 0x46, 0x6d, 0xbd, 0x64, 0x23,
			0xe6, 0xcb, 0xc5, 0x18, 0xb5, 0xe4, 0xae, 0x7b,
			0x8f, 0x15, 0xe0, 0xf2, 0xd0, 0x47, 0x1a, 0x95,
			0x16, 0xdf, 0xa9, 0x59, 0x16, 0x97, 0xf7, 0x42,
			0x86, 0x23, 0x24, 0xd8, 0xd1, 0x03, 0xfb, 0x63,
			0x1d, 0x6c, 0x20, 0x73, 0xd4, 0x06, 0xb6, 0x5c,
			0xde, 0xe7, 0xbd, 0xa5, 0x43, 0xe2, 0xe9, 0xeb,
			0xff, 0x99, 0x06, 0x98, 0x5d, 0x1c, 0xb3, 0x65,
			0x17, 0x2e, 0xa6, 0x23, 0xed, 0x7a, 0xa4, 0xc7,
			0xa3, 0x22, 0xf0, 0x98, 0x46, 0x80, 0xe3, 0x4e,
			0x99, 0xbc, 0x62, 0x31, 0xb0, 0x2e, 0x3d, 0x14,
			0x58, 0x16, 0x08, 0xbc, 0x55, 0xbc, 0xa7, 0xfb,
			0xe2, 0x2d, 0x7f, 0x03, 0xe9, 0x04, 0xda, 0x45,
			0x52, 0xe0, 0x09, 0xe5, 0x60, 0x7f, 0x04, 0x18,
		},
	},
	/* SHA512 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x59, 0x16, 0x52, 0xb6, 0xeb, 0x1b, 0x52, 0xc9,
			0xbe, 0xbd, 0x58, 0x32, 0x56, 0xc2, 0x22, 0x86,
			0x80, 0x11, 0x0b, 0x87, 0x89, 0x17, 0xde, 0xa5,
			0xad, 0x69, 0xe8, 0xc5, 0xd2, 0xab, 0x51, 0x42,
			0x77, 0xb0, 0xac, 0x31, 0xe7, 0xe2, 0xcc, 0xea,
			0xb2, 0xe5, 0xd9, 0xc4, 0x5d, 0x77, 0xa4, 0x1f,
			0x59, 0x9b, 0x38, 0xa8, 0x32, 0xf6, 0xb2, 0xd8,
			0x09, 0x79, 0x52, 0xbe, 0x44, 0x40, 0xd1, 0xff,
			0x84, 0xba, 0xf5, 0x1b, 0xd7, 0x0b, 0x64, 0xf1,
			0x30, 0xae, 0xb6, 0x86, 0x14, 0x5f, 0xcd, 0x02,
			0x95, 0x38, 0x69, 0xfb, 0x84, 0x1a, 0xf7, 0xf6,
			0xe3, 0x4e, 0xaa, 0x2b, 0x99, 0x6c, 0xcd, 0x89,
			0x69, 0x7c, 0x58, 0xfa, 0x25, 0x5c, 0xc1, 0xe8,
			0x1f, 0x62, 0x14, 0x00, 0xe1, 0x41, 0x46, 0x36,
			0x1e, 0x31, 0xc7, 0x09, 0xe8, 0x4a, 0x56, 0x08,
			0x22, 0x31, 0x19, 0x95, 0x39, 0xf7, 0xed, 0xe9,
		},
		/* S2 : 384 byte */
		{
			0x1d, 0xe7, 0x9d, 0x72, 0x16, 0xdd, 0xe1, 0x25,
			0xde, 0xb7, 0x7c, 0x34, 0xd9, 0x0a, 0xb3, 0x21,
			0xa4, 0xde, 0x5f, 0xb1, 0x1c, 0x29, 0x66, 0x56,
			0xad, 0x9b, 0xf9, 0xa2, 0x46, 0x53, 0x59, 0x11,
			0x17, 0xac, 0xe4, 0x15, 0xe1, 0x8e, 0xad, 0xce,
			0x92, 0x82, 0x3f, 0x31, 0xaf, 0xe5, 0x6f, 0xc8,
			0xe2, 0x94, 0x94, 0xe3, 0x7c, 0xf2, 0xba, 0x85,
			0xab, 0xc3, 0xba, 0xc6, 0x6e, 0x01, 0x95, 0x84,
			0x79, 0x9a, 0xee, 0x23, 0x4a, 0xd5, 0x55, 0x9e,
			0x21, 0xc7, 0xfd, 0x4f, 0xfd, 0x24, 0xd8, 0x26,
			0x49, 0xf6, 0x79, 0xb4, 0xc0, 0x5d, 0x8c, 0x15,
			0xd3, 0xd4, 0x57, 0x4a, 0x2e, 0x76, 0xb1, 0xf3,
			0xee, 0x9f, 0x8d, 0xec, 0x0a, 0xf6, 0x0b, 0x0c,
			0xed, 0x1b, 0xe8, 0xa1, 0x9c, 0x2f, 0xa7, 0x1b,
			0xcb, 0xc1, 0xfb, 0x19, 0x08, 0x99, 0xec, 0x85,
			0x56, 0x95, 0x8e, 0x07, 0x82, 0xac, 0xe7, 0x19,
			0x6b, 0x36, 0x65, 0x86, 0x56, 0xcf, 0x36, 0x4d,
			0x37, 0x73, 0xde, 0x86, 0x26, 0x0f, 0xd8, 0x98,
			0x76, 0x04, 0xef, 0x35, 0xea, 0xe8, 0xf3, 0x8e,
			0xc2, 0xcb, 0x0d, 0xa8, 0x64, 0xcc, 0xa7, 0x19,
			0x21, 0x9c, 0x2a, 0xd7, 0x1c, 0x08, 0x50, 0x6c,
			0x41, 0x2e, 0xc7, 0x79, 0x95, 0xf3, 0x74, 0x39,
			0xc8, 0x56, 0x97, 0x7b, 0x71, 0xdf, 0xb9, 0x64,
			0x79, 0x90, 0xef, 0x70, 0xfa, 0xf4, 0x32, 0x73,
			0xae, 0x60, 0x83, 0x9c, 0xd0, 0x67, 0x9e, 0xc9,
			0xaa, 0x42, 0xbf, 0x91, 0x4e, 0x42, 0x1b, 0x79,
			0x7c, 0xba, 0x21, 0x8a, 0x40, 0x0f, 0xf9, 0xdb,
			0xaa, 0x20, 0x6c, 0xb9, 0xc2, 0xb0, 0x59, 0x6c,
			0x70, 0x9a, 0x32, 0x2b, 0x73, 0xcb, 0x82, 0x72,
			0x1d, 0x79, 0xf9, 0xdb, 0x24, 0x21, 0x1b, 0xf0,
			0x75, 0xa1, 0xce, 0xf7, 0x4e, 0x8f, 0x6d, 0x2b,
			0xa0, 0x7f, 0xe0, 0xdc, 0x8a, 0x60, 0xf4, 0x8a,
			0xf5, 0x11, 0xad, 0x46, 0x9d, 0xcd, 0x06, 0xe0,
			0x7a, 0x4c, 0xe6, 0x80, 0x72, 0x13, 0x9c, 0x46,
			0xd8, 0xbe, 0x5e, 0x72, 0x12, 0x53, 0xc3, 0xb1,
			0x8b, 0x3c, 0x94, 0x48, 0x5c, 0xe5, 0x5c, 0x0e,
			0x7c, 0x1c, 0xbc, 0x39, 0xb7, 0x7b, 0xc6, 0xbb,
			0x7e, 0x5e, 0x9f, 0x42, 0xb1, 0x53, 0x9e, 0x44,
			0x2d, 0xa8, 0x57, 0x65, 0x8c, 0x9e, 0x77, 0x1c,
			0xcb, 0x86, 0xbe, 0x73, 0x97, 0x64, 0x7e, 0xfb,
			0xc0, 0xcc, 0xb2, 0xc3, 0xad, 0x31, 0xac, 0x4e,
			0x32, 0xbf, 0x24, 0x8c, 0xc0, 0xce, 0xd3, 0xa4,
			0xf0, 0x94, 0x52, 0x6b, 0x25, 0x63, 0x1c, 0xb5,
			0x02, 0x47, 0x09, 0x61, 0x29, 0xb0, 0x8a, 0x9c,
			0x2c, 0xdf, 0xb7, 0x75, 0x97, 0x8b, 0x0f, 0xee,
			0xe2, 0x65, 0xa6, 0xc4, 0x19, 0x91, 0xc1, 0xdc,
			0x44, 0x52, 0x61, 0x5b, 0x78, 0xc9, 0x06, 0xc7,
			0xed, 0x1b, 0xd2, 0x07, 0x96, 0x9d, 0x98, 0xd0,
		},
	},
	/* SHA512 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0x8d, 0xff, 0xaa, 0x91, 0x51, 0x27, 0x1a, 0xd2,
			0x26, 0x22, 0xf2, 0x28, 0xc8, 0x92, 0xe1, 0xd9,
			0x74, 0x8b, 0x3c, 0x39, 0x43, 0x97, 0xf2, 0xcb,
			0xb6, 0xfe, 0xbe, 0xaa, 0x92, 0x44, 0xa0, 0x27,
			0xee, 0xf2, 0x8d, 0xb4, 0x8a, 0x9a, 0x66, 0x01,
			0x62, 0x15, 0x27, 0x64, 0x83, 0x0f, 0x61, 0x7e,
			0x1e, 0xc6, 0xea, 0x1c, 0xdb, 0x0e, 0xd2, 0x5b,
			0x6f, 0x99, 0x9a, 0x10, 0x71, 0x75, 0xa1, 0x66,
			0x69, 0xd6, 0xdf, 0xc9, 0x2b, 0x16, 0xd5, 0x03,
			0x63, 0xfa, 0xc4, 0xa5, 0x70, 0x37, 0x1e, 0xa9,
			0x76, 0x34, 0x3a, 0x55, 0xae, 0x12, 0x4b, 0x63,
			0x01, 0xea, 0x93, 0x5e, 0xd6, 0x55, 0xd4, 0x4f,
			0x28, 0x32, 0x08, 0x99, 0xdb, 0xa3, 0x51, 0x22,
			0x50, 0x59, 0x33, 0xb3, 0x37, 0x12, 0x01, 0xa2,
			0xa4, 0x5f, 0x95, 0xae, 0x65, 0xab, 0x44, 0x2a,
			0x94, 0x79, 0x12, 0x5e, 0x68, 0xed, 0x21, 0x2a,
		},
		/* S3 : 384 byte */
		{
			0xb3, 0x29, 0xae, 0xf8, 0x3a, 0x56, 0xdd, 0xc5,
			0x7c, 0xd9, 0xa0, 0xe1, 0x5e, 0xb0, 0xb0, 0xb7,
			0xae, 0xa7, 0xd7, 0x8d, 0x5e, 0x8c, 0xa3, 0x98,
			0x2b, 0xd3, 0x1c, 0xc8, 0x25, 0xa0, 0xcd, 0x1c,
			0x44, 0x4d, 0x9f, 0x7b, 0xea, 0x9e, 0x7a, 0x27,
			0xf3, 0xbb, 0xb3, 0x76, 0x10, 0x60, 0xff, 0x95,
			0xfe, 0xe1, 0xa3, 0xe8, 0x64, 0xd2, 0x10, 0x8f,
			0xc4, 0x0b, 0x64, 0x78, 0x6a, 0x96, 0xa6, 0xd6,
			0x2d, 0x20, 0x12, 0x17, 0xe0, 0x3a, 0x8b, 0xa2,
			0xc0, 0x7e, 0xe9, 0x4c, 0x26, 0x71, 0x49, 0xd1,
			0xe7, 0x2c, 0xc5, 0x77, 0x9b, 0x73, 0x7e, 0x85,
			0x47, 0xac, 0xd6, 0xaa, 0x4b, 0xba, 0x3f, 0xf3,
			0x8b, 0xf9, 0x68, 0x7e, 0x9e, 0x82, 0xf5, 0x11,
			0xb5, 0x97, 0xad, 0x7e, 0xc1, 0xd7, 0x95, 0xc3,
			0x6a, 0x98, 0xbf, 0x83, 0xa9, 0x0f, 0xc8, 0x6b,
			0x0c, 0xad, 0x41, 0x95, 0x33, 0x60, 0x73, 0x89,
			0x21, 0x93, 0x6a, 0x45, 0x86, 0x74, 0xb2, 0xe9,
			0xa7, 0x01, 0x2a, 0xc3, 0x02, 0x9f, 0xdb, 0x0a,
			0x9d, 0x12, 0x31, 0x82, 0x02, 0xd2, 0x54, 0x4a,
			0x0d, 0x97, 0x6e, 0xe5, 0x36, 0xe0, 0x3b, 0x7e,
			0x8d, 0x89, 0x4b, 0x3b, 0x9c, 0x76, 0x2d, 0xab,
			0x01, 0x10, 0x84, 0x9c, 0xc1, 0xea, 0xad, 0x74,
			0x7e, 0x3d, 0x88, 0xd7, 0xdc, 0xf4, 0x9f, 0x82,
			0x4d, 0xf0, 0x27, 0xe6, 0x45, 0xc0, 0xb9, 0x29,
			0x4e, 0x65, 0x5d, 0x9f, 0xc9, 0xe1, 0xef, 0x95,
			0xeb, 0x53, 0xaa, 0xff, 0x57, 0x75, 0xc3, 0x49,
			0x48, 0x6d, 0x4b, 0x5d, 0x67, 0xdb, 0xa2, 0x9b,
			0x62, 0x17, 0xf8, 0xb9, 0x97, 0x66, 0x12, 0xb5,
			0x7e, 0x16, 0xfc, 0x1f, 0x99, 0x98, 0x3f, 0x2a,
			0xf0, 0x45, 0x79, 0x93, 0x86, 0x06, 0x87, 0x9b,
			0x7c, 0x72, 0x53, 0xe8, 0x70, 0x71, 0x4b, 0x4f,
			0x0f, 0x24, 0xe2, 0x6d, 0xc8, 0xc7, 0xa6, 0xfc,
			0xef, 0xfb, 0x5f, 0x98, 0xe3, 0xb2, 0xfb, 0x5d,
			0xb9, 0x49, 0xd2, 0xf9, 0x8c, 0xd1, 0xae, 0x1a,
			0xa5, 0x52, 0x69, 0x6b, 0x48, 0xc3, 0x9f, 0x67,
			0x8e, 0x15, 0x43, 0x51, 0xcc, 0x75, 0x6d, 0x3e,
			0x9a, 0x97, 0xf7, 0x92, 0x79, 0x85, 0x3e, 0xbd,
			0x0d, 0xb9, 0xae, 0x68, 0x59, 0xfb, 0x2d, 0x57,
			0x21, 0x38, 0x5d, 0x06, 0xf5, 0x56, 0x5a, 0x3a,
			0x8f, 0xf0, 0x99, 0x2d, 0x51, 0x7a, 0xcd, 0xa1,
			0xaf, 0x69, 0xa9, 0x28, 0x54, 0xa1, 0xb3, 0x2a,
			0x79, 0xcb, 0x9e, 0x44, 0x2a, 0x90, 0xb0, 0x55,
			0xbb, 0x2e, 0xc3, 0xaf, 0x8d, 0x99, 0x26, 0xa0,
			0xd8, 0x57, 0xe3, 0xcb, 0x1e, 0x7e, 0x4a, 0x73,
			0x00, 0xd1, 0xac, 0xcb, 0x94, 0x92, 0xec, 0x78,
			0x32, 0xaf, 0x45, 0x35, 0x29, 0xff, 0x0f, 0x4a,
			0x6a, 0xd3, 0x25, 0x97, 0x57, 0xf7, 0x07, 0xf7,
			0x13, 0xaa, 0xa5, 0xdf, 0x23, 0x1f, 0x74, 0x87,
		},
	},
	/* SHA512 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0x71, 0xd4, 0x16, 0x3e, 0x70, 0x8c, 0x12, 0x1e,
			0x93, 0x1b, 0xb9, 0x69, 0x2b, 0x21, 0x7d, 0xdd,
			0xd3, 0x5c, 0x73, 0x46, 0xf6, 0x1c, 0xfc, 0x95,
			0x91, 0xf7, 0xa4, 0x31, 0x3a, 0xbd, 0x4a, 0x92,
			0x62, 0xaf, 0x82, 0x0b, 0xd7, 0xeb, 0x37, 0xe7,
			0x8c, 0x2b, 0x95, 0xb8, 0x9d, 0xaf, 0x25, 0xec,
			0x8e, 0x78, 0x3a, 0xa1, 0xd4, 0xb7, 0x8d, 0xbb,
			0x96, 0x85, 0x24, 0x33, 0xb4, 0xd4, 0x78, 0xb1,
			0x09, 0xa6, 0xd6, 0x5e, 0xed, 0x7d, 0x06, 0xf3,
			0xfe, 0x12, 0x2b, 0x17, 0x21, 0x49, 0xea, 0xe7,
			0xc3, 0x65, 0xce, 0xd6, 0x65, 0x78, 0xeb, 0xb7,
			0x57, 0x1e, 0xc2, 0x18, 0xc3, 0x6b, 0x65, 0xd2,
			0xee, 0x22, 0xdc, 0xde, 0xbb, 0x28, 0xc6, 0x6a,
			0x71, 0x38, 0x43, 0x2c, 0xbd, 0xd7, 0x12, 0xf7,
			0xfb, 0x8b, 0xf7, 0x8c, 0xb1, 0x48, 0x60, 0xb2,
			0x5c, 0x2b, 0x47, 0x89, 0x70, 0x6b, 0x5a, 0x1b,
		},
		/* S4 : 384 byte */
		{
			0x25, 0x22, 0xee, 0x3b, 0xda, 0x30, 0xc0, 0x43,
			0x4e, 0x54, 0xb1, 0x99, 0xda, 0x8c, 0x97, 0x33,
			0x96, 0x4f, 0xd4, 0x02, 0xb7, 0x07, 0xf5, 0xb3,
			0x30, 0xf4, 0xf7, 0x54, 0xa0, 0x50, 0x2c, 0x7a,
			0x71, 0x3c, 0x78, 0x14, 0xf0, 0xe8, 0x51, 0xa4,
			0xa4, 0xdb, 0x72, 0x69, 0x0d, 0xb9, 0x6e, 0xa8,
			0xb8, 0x81, 0x3b, 0xd8, 0x62, 0x9a, 0x94, 0x8b,
			0xb3, 0x0c, 0x1b, 0x82, 0x72, 0xa8, 0x16, 0xb3,
			0x0a, 0x75, 0x5f, 0xc6, 0xfb, 0x17, 0x54, 0x16,
			0x7c, 0x3e, 0xb1, 0xf1, 0x94, 0x39, 0x59, 0x07,
			0xa5, 0x6c, 0xf5, 0xa7, 0x3b, 0x41, 0x54, 0x38,
			0x3a, 0x05, 0xb7, 0x8b, 0x73, 0x1f, 0xed, 0xd9,
			0x07, 0x7f, 0x3c, 0x22, 0x67, 0xa5, 0xcf, 0x92,
			0x66, 0x97, 0x87, 0x1f, 0xe0, 0xa4, 0xbe, 0xd9,
			0xc2, 0x19, 0x55, 0x2d, 0xd1, 0xc8, 0x7a, 0xff,
			0x50, 0x61, 0x30, 0x94, 0xbc, 0xaa, 0x2d, 0xec,
			0x42, 0xa3, 0x53, 0x80, 0xa6, 0xba, 0xc6, 0x73,
			0xda, 0x25, 0x94, 0xf8, 0x24, 0xa8, 0xf3, 0x2f,
			0x21, 0xd7, 0x59, 0x3a, 0x3e, 0x49, 0xc7, 0x8e,
			0xe2, 0x80, 0x19, 0x3a, 0x47, 0x86, 0x21, 0xd3,
			0xb0, 0x95, 0xc1, 0x6d, 0xce, 0x72, 0x93, 0x53,
			0x14, 0xd4, 0xa2, 0x32, 0x3e, 0xeb, 0xe7, 0x85,
			0x5c, 0xa4, 0x73, 0x8a, 0x19, 0xb5, 0xa3, 0x1a,
			0x5f, 0x95, 0xab, 0x91, 0xfb, 0xe1, 0x28, 0x9c,
			0x02, 0xfe, 0xa7, 0xa6, 0x5b, 0x91, 0x32, 0x7b,
			0x7b, 0x97, 0x90, 0x55, 0x62, 0x89, 0xe1, 0xb9,
			0x88, 0xe4, 0x5d, 0x50, 0xeb, 0x8c, 0xea, 0x15,
			0x81, 0xde, 0x5d, 0x5d, 0xfd, 0x21, 0x00, 0x1c,
			0x73, 0xb4, 0x39, 0x21, 0xd8, 0xb2, 0x1b, 0x96,
			0x44, 0xb0, 0xf2, 0xb9, 0x6e, 0xe6, 0xb0, 0x9d,
			0x73, 0x70, 0x9c, 0x33, 0x33, 0x81, 0x43, 0xd6,
			0xa2, 0xfe, 0xc5, 0x59, 0xa4, 0x36, 0xc5, 0xec,
			0x86, 0x5d, 0x3a, 0xcc, 0xa5, 0xfe, 0xe6, 0x54,
			0xf1, 0x32, 0x5a, 0xe5, 0x72, 0x55, 0xdf, 0xd4,
			0x21, 0x88, 0xc8, 0x4d, 0xcb, 0x1f, 0x7c, 0x1e,
			0x86, 0x02, 0x8a, 0x74, 0xe3, 0x1d, 0x73, 0x60,
			0x78, 0x74, 0x1e, 0xe9, 0x7c, 0x39, 0xa5, 0x6e,
			0x4d, 0xe0, 0x0f, 0xc1, 0x2b, 0x80, 0x51, 0x83,
			0x5b, 0xbd, 0x0d, 0x8f, 0xca, 0xe7, 0x37, 0x32,
			0x20, 0x99, 0xad, 0xc1, 0x01, 0x71, 0x07, 0x02,
			0x2d, 0xd1, 0x5c, 0x11, 0x4d, 0xa5, 0x7e, 0x78,
			0xb9, 0x56, 0x81, 0xba, 0x99, 0x45, 0x61, 0x5b,
			0x59, 0xda, 0x90, 0xf5, 0xa2, 0xa9, 0x9a, 0x25,
			0x2e, 0xb4, 0x2b, 0x20, 0x06, 0xee, 0xdd, 0x6e,
			0x78, 0x47, 0x6c, 0x29, 0x05, 0x47, 0x3e, 0xe6,
			0xb4, 0xf2, 0x3c, 0x1c, 0x5c, 0xf0, 0xb8, 0x04,
			0x51, 0xc5, 0x42, 0x6e, 0xa0, 0x09, 0x14, 0x1c,
			0xb3, 0xfc, 0xb0, 0xdf, 0x2d, 0xed, 0x92, 0xbe,
		},
	},
	/* SHA512 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0xd0, 0x0e, 0x15, 0x29, 0x22, 0x8c, 0x79, 0xa2,
			0x0a, 0x1c, 0x36, 0x68, 0xff, 0xa4, 0xa5, 0x41,
			0x40, 0xbb, 0x17, 0x0b, 0xc5, 0xc6, 0x69, 0xfd,
			0x75, 0x60, 0xd9, 0x30, 0x99, 0x00, 0x17, 0x5e,
			0x91, 0xd5, 0xa0, 0xe9, 0xc5, 0xf5, 0x47, 0x1f,
			0xdf, 0xb7, 0x14, 0xbc, 0x38, 0x5d, 0x52, 0xb0,
			0x8f, 0xf7, 0xe4, 0x23, 0x01, 0x84, 0xd8, 0xb7,
			0x35, 0x59, 0x3f, 0x0d, 0xd8, 0xc7, 0x3b, 0x8a,
			0x49, 0xf8, 0x59, 0x5b, 0x95, 0x1a, 0x21, 0xb6,
			0xa5, 0xbf, 0xec, 0x63, 0xb6, 0x84, 0xf6, 0x7c,
			0x0a, 0xf1, 0xb4, 0x71, 0xdd, 0xa1, 0x68, 0x4e,
			0x9b, 0xa3, 0xf2, 0x41, 0x50, 0x1f, 0xe9, 0x57,
			0x60, 0x3d, 0xea, 0x86, 0x78, 0x42, 0x30, 0xf0,
			0xc4, 0xfd, 0x65, 0x66, 0x63, 0x61, 0xb8, 0x2b,
			0x18, 0x73, 0x30, 0xfb, 0x42, 0x67, 0x40, 0x4c,
			0x0e, 0x05, 0x9b, 0xd4, 0xeb, 0x52, 0x49, 0x4b,
		},
		/* S5 : 384 byte */
		{
			0x18, 0x35, 0xdd, 0x97, 0xe5, 0x09, 0x3a, 0x33,
			0xce, 0x1e, 0x62, 0xd6, 0x83, 0x86, 0x3f, 0x6b,
			0x35, 0x07, 0xf3, 0x58, 0xa6, 0x2f, 0xc8, 0x79,
			0xb5, 0x24, 0x35, 0x0f, 0xbc, 0x73, 0x30, 0x68,
			0x1c, 0xb0, 0xc6, 0x82, 0xee, 0xf4, 0x33, 0x04,
			0x19, 0xca, 0xf8, 0x54, 0x3b, 0xd9, 0x26, 0x9b,
			0x6d, 0x91, 0xd8, 0xe1, 0x07, 0xec, 0x38, 0xb6,
			0xe9, 0xc6, 0xea, 0xab, 0xf9, 0x06, 0x45, 0x72,
			0x05, 0xd5, 0x2a, 0x90, 0x0e, 0x05, 0x57, 0x9a,
			0xa1, 0x1f, 0xc5, 0x81, 0x37, 0x52, 0x64, 0xe6,
			0x9a, 0x92, 0x57, 0x98, 0xe5, 0xa3, 0x48, 0xe5,
			0xa1, 0x6f, 0x15, 0x67, 0xd5, 0xd0, 0xe4, 0x08,
			0x53, 0x38, 0x0b, 0x34, 0xde, 0xac, 0x93, 0xad,
			0x73, 0x77, 0xaa, 0xe8, 0xa2, 0x7b, 0x09, 0x0d,
			0x0d, 0x3a, 0x92, 0xbf, 0x7a, 0x82, 0x4d, 0x92,
			0x6e, 0x2e, 0x35, 0xa0, 0xc3, 0xbd, 0x0e, 0x99,
			0x0b, 0x59, 0x11, 0x20, 0xd7, 0x4d, 0xd9, 0xb0,
			0x52, 0xa7, 0x35, 0x68, 0xe3, 0xc3, 0xf2, 0x9c,
			0x5a, 0x77, 0xfb, 0x1c, 0x92, 0x1b, 0xce, 0x9c,
			0x1e, 0x7f, 0x76, 0x4a, 0xa6, 0x7b, 0xac, 0x11,
			0x9f, 0x58, 0x39, 0xa5, 0x30, 0x38, 0x60, 0xed,
			0xeb, 0x63, 0x48, 0x14, 0xc2, 0x38, 0x6c, 0x83,
			0x1f, 0xee, 0x62, 0x00, 0xcf, 0x55, 0xb6, 0xbf,
			0xea, 0x05, 0x8b, 0x79, 0x5a, 0x0f, 0xcf, 0x26,
			0xeb, 0x72, 0x16, 0xae, 0x1b, 0x75, 0x87, 0xc8,
			0x2e, 0x56, 0x85, 0xe5, 0x84, 0x17, 0x0c, 0xbd,
			0xdc, 0x89, 0xa7, 0x7e, 0x09, 0x89, 0xd4, 0xce,
			0x5c, 0x3c, 0x7f, 0xdb, 0x66, 0x4a, 0xae, 0xaa,
			0xdb, 0xce, 0x1f, 0x23, 0x1e, 0x64, 0x79, 0x8f,
			0x6f, 0x9a, 0x85, 0x45, 0x6b, 0x5a, 0x93, 0xa5,
			0x02, 0x12, 0x6a, 0x80, 0xe2, 0xd2, 0x1f, 0x46,
			0x92, 0x1c, 0xc3, 0x60, 0x1f, 0x5e, 0xcd, 0xbd,
			0x56, 0x99, 0x8a, 0x63, 0xb8, 0x65, 0xfc, 0xe7,
			0xeb, 0x29, 0x9f, 0x76, 0xaf, 0x40, 0xe9, 0x12,
			0x81, 0xbf, 0xc0, 0x19, 0xf4, 0x0e, 0x0d, 0x46,
			0x81, 0x1e, 0x38, 0x36, 0x91, 0xe4, 0x02, 0x4c,
			0x94, 0x56, 0x6f, 0x18, 0x02, 0x4f, 0xf2, 0xb2,
			0x2a, 0xa7, 0xe1, 0x27, 0x02, 0x33, 0xff, 0x16,
			0xe9, 0x2f, 0x89, 0xc6, 0x85, 0x09, 0xea, 0x0b,
			0xe2, 0xd3, 0x45, 0x11, 0x58, 0x1d, 0x47, 0x22,
			0x07, 0xd1, 0xb6, 0x5f, 0x7e, 0xde, 0x45, 0x13,
			0x3d, 0xe8, 0x7a, 0x5f, 0xfb, 0x92, 0x62, 0xc1,
			0xff, 0x84, 0x08, 0x8f, 0xf0, 0x4c, 0x01, 0x83,
			0xf4, 0x84, 0x67, 0x99, 0x6a, 0x94, 0xd8, 0x2b,
			0xa7, 0x51, 0x0c, 0xb0, 0xb3, 0x6c, 0xf2, 0x54,
			0x82, 0x09, 0xa5, 0x06, 0x03, 0x37, 0x5c, 0xb8,
			0x2e, 0x67, 0x8f, 0x51, 0x49, 0x33, 0x45, 0xca,
			0x33, 0xf9, 0x34, 0x5f, 0xfd, 0xf5, 0x4b, 0xe9,
		},
	},
	/* SHA512 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0xa3, 0x59, 0x26, 0x68, 0x55, 0x61, 0xf0, 0x9f,
			0x30, 0x92, 0x5e, 0x94, 0xd7, 0x4e, 0x56, 0x61,
			0x89, 0x2a, 0x2d, 0xdd, 0x52, 0x4f, 0x75, 0x1f,
			0x83, 0x21, 0x16, 0x3d, 0x61, 0x1e, 0xa1, 0x59,
			0x1a, 0x08, 0xe0, 0xdf, 0xfd, 0x46, 0xb2, 0x08,
			0xe9, 0x88, 0x15, 0xa3, 0x06, 0xaa, 0x85, 0x14,
			0xb4, 0xdb, 0x85, 0x9d, 0xc1, 0xfe, 0x7b, 0xdc,
			0xdf, 0x50, 0xc0, 0x95, 0x55, 0x4b, 0xf8, 0xb2,
			0xf4, 0xcb, 0x9f, 0x88, 0x4d, 0x70, 0xe5, 0x5c,
			0x21, 0x43, 0xbc, 0x26, 0x19, 0x9c, 0x2f, 0x94,
			0xb7, 0x43, 0xf5, 0x52, 0x8d, 0xd5, 0x46, 0x89,
			0xad, 0x69, 0xed, 0xa6, 0x60, 0x74, 0x9f, 0x5c,
			0x1b, 0xea, 0x8b, 0xec, 0xae, 0xa6, 0x32, 0xa4,
			0xbf, 0x0c, 0x79, 0xa5, 0x77, 0xed, 0xfc, 0xea,
			0x7b, 0xaa, 0xa6, 0x86, 0x1e, 0x9d, 0x7f, 0x2d,
			0xd5, 0xb4, 0xc4, 0xf6, 0xeb, 0x5f, 0x3d, 0x5f,
		},
		/* S6 : 384 byte */
		{
			0xb1, 0xa9, 0xc4, 0x5a, 0x26, 0x4d, 0x2c, 0x9a,
			0xf4, 0x41, 0xa7, 0xb2, 0xd3, 0x30, 0xdd, 0x78,
			0x80, 0x89, 0xcc, 0xef, 0x20, 0x5d, 0x5d, 0x66,
			0x6b, 0xfe, 0x86, 0x43, 0x67, 0xbe, 0x97, 0x38,
			0x12, 0x4e, 0x9d, 0x74, 0x64, 0x8a, 0xd9, 0x91,
			0x60, 0xbd, 0x3a, 0xf8, 0x1a, 0x81, 0x85, 0x8b,
			0xab, 0xe6, 0x67, 0xa5, 0xd9, 0x5c, 0x98, 0x0f,
			0xe2, 0xf6, 0xac, 0x34, 0x86, 0x1e, 0xb2, 0xec,
			0x9b, 0x4b, 0x4e, 0x8b, 0x64, 0x2e, 0xf3, 0x82,
			0x0f, 0x56, 0xca, 0x38, 0x8a, 0x55, 0x65, 0x30,
			0xd4, 0x27, 0x54, 0xc4, 0x72, 0x12, 0xe9, 0xb2,
			0xf2, 0x52, 0x38, 0xa1, 0xef, 0x5a, 0xfe, 0x29,
			0xbe, 0x63, 0x40, 0x8c, 0xf3, 0x8c, 0xaa, 0x2d,
			0x23, 0xa7, 0x88, 0x24, 0xae, 0x0b, 0x92, 0x59,
			0x75, 0xd3, 0xe9, 0x83, 0x55, 0x8d, 0xf6, 0xd2,
			0xe9, 0xb1, 0xd3, 0x4a, 0x18, 0xb1, 0xd9, 0x73,
			0xff, 0xac, 0xcc, 0x74, 0x5e, 0x52, 0x7c, 0xe7,
			0x6c, 0x66, 0x3e, 0x90, 0x37, 0x19, 0x35, 0x5e,
			0x45, 0xcd, 0x6d, 0x11, 0x8e, 0xd0, 0xb8, 0x5b,
			0x70, 0xcb, 0xb8, 0xe4, 0x96, 0x41, 0x13, 0x53,
			0xf8, 0x4f, 0x88, 0x66, 0xa0, 0x1f, 0xad, 0xc8,
			0x19, 0xca, 0x0f, 0xf9, 0x5b, 0xbe, 0x2c, 0xc6,
			0x8c, 0x8c, 0xf7, 0x8d, 0xa5, 0x58, 0x1b, 0xec,
			0xc9, 0x62, 0x47, 0xb9, 0x11, 0xd1, 0x85, 0xed,
			0x1f, 0xae, 0x36, 0xc4, 0xca, 0xd2, 0x62, 0x08,
			0xeb, 0x80, 0x88, 0x3f, 0x42, 0xa0, 0x81, 0x23,
			0xda, 0xc6, 0x8d, 0x88, 0xf2, 0xf9, 0x89, 0x3c,
			0xde, 0x02, 0xef, 0x5a, 0x57, 0x66, 0x1d, 0xb2,
			0xb3, 0xe1, 0xe9, 0x26, 0x9c, 0xbb, 0x0e, 0x15,
			0xc4, 0x07, 0xbc, 0xf5, 0x5d, 0x92, 0xe6, 0x79,
			0x38, 0x3c, 0x90, 0x80, 0x2c, 0xd0, 0xbf, 0xfd,
			0x46, 0x96, 0x46, 0xdc, 0xb6, 0x0c, 0xa0, 0x1a,
			0x1d, 0xea, 0xd4, 0x32, 0x28, 0x93, 0x40, 0x18,
			0x39, 0x1d, 0xd8, 0x1f, 0x8b, 0x7e, 0x79, 0x7e,
			0x52, 0x7f, 0xbe, 0x18, 0x15, 0xb9, 0x1b, 0xf3,
			0xcd, 0x6a, 0x1f, 0x2f, 0xfb, 0xf5, 0xdd, 0x16,
			0x6a, 0xcd, 0x55, 0x26, 0x76, 0x1c, 0xa8, 0xba,
			0xb5, 0xd4, 0x63, 0xfb, 0x9f, 0xb8, 0x20, 0x65,
			0x9f, 0x5c, 0xd5, 0x0f, 0x81, 0x50, 0xf1, 0x2f,
			0x7e, 0x8d, 0x52, 0xe7, 0x77, 0x73, 0xc1, 0xe6,
			0x48, 0x0c, 0x2c, 0xc1, 0x84, 0xd4, 0x11, 0xd6,
			0x41, 0xf7, 0x1a, 0x9d, 0xed, 0xc2, 0xc5, 0xfc,
			0x2e, 0xc3, 0x7a, 0x27, 0x70, 0xa9, 0x38, 0x3b,
			0xfb, 0xf6, 0xa4, 0x89, 0xcf, 0x32, 0xb5, 0x6a,
			0x12, 0xcf, 0x99, 0x37, 0x8e, 0x39, 0xb5, 0x0b,
			0xda, 0xdb, 0x9f, 0x05, 0x91, 0xb2, 0x06, 0x5f,
			0x9d, 0x44, 0xe5, 0x11, 0xc9, 0xdf, 0xb6, 0x15,
			0x8f, 0xdd, 0xdd, 0xd1, 0xbc, 0x2c, 0xec, 0xe6,
		},
	},
	/* SHA512 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0x12, 0x71, 0xa0, 0xdd, 0xb9, 0x9a, 0x0e, 0x1e,
			0x9a, 0x50, 0x1c, 0xa3, 0x3c, 0x13, 0x1b, 0x0a,
			0x1c, 0x78, 0x20, 0xa3, 0x97, 0x79, 0x08, 0x69,
			0x09, 0x0f, 0xba, 0x37, 0x37, 0x03, 0xac, 0x38,
			0xea, 0x00, 0xa9, 0xa0, 0xdd, 0xee, 0xd1, 0x99,
			0xd9, 0x7b, 0xe1, 0x80, 0x1f, 0xfa, 0xb4, 0x52,
			0x06, 0x71, 0x0a, 0x61, 0xe5, 0xed, 0x89, 0x4c,
			0x33, 0x19, 0x01, 0x2d, 0xed, 0x0f, 0xf4, 0x14,
			0x38, 0x6e, 0x56, 0xb5, 0x48, 0xad, 0x91, 0x5d,
			0x80, 0xaf, 0xcc, 0x2b, 0xdb, 0x97, 0x6d, 0x7c,
			0x8a, 0xdd, 0xdc, 0xa7, 0xdf, 0xa2, 0x8a, 0xeb,
			0x69, 0x40, 0x33, 0xa5, 0x61, 0x26, 0x60, 0xc6,
			0x44, 0xe3, 0x2f, 0x85, 0xc2, 0x80, 0x56, 0x51,
			0xd7, 0x13, 0x66, 0x0a, 0x38, 0x91, 0x4d, 0x70,
			0xf0, 0xe4, 0x1f, 0xdc, 0x4b, 0x3d, 0x16, 0x2e,
			0xf3, 0xac, 0xd7, 0x06, 0x59, 0xee, 0xf6, 0x37,
		},
		/* S7 : 384 byte */
		{
			0xbf, 0xfd, 0x01, 0x0b, 0x2e, 0xc4, 0xe4, 0xa3,
			0x27, 0x77, 0xb7, 0x76, 0x19, 0xb8, 0x76, 0x22,
			0xf8, 0x92, 0x1d, 0xab, 0x56, 0xe1, 0x02, 0xc8,
			0xd8, 0x24, 0xfe, 0x52, 0xb5, 0xdf, 0x7a, 0x20,
			0x3f, 0xe7, 0x17, 0x99, 0xee, 0xaf, 0xdc, 0xc0,
			0xc8, 0x87, 0x2d, 0xba, 0x6a, 0x37, 0x44, 0x07,
			0xb5, 0x63, 0x9a, 0xeb, 0x5a, 0x30, 0xa9, 0x04,
			0x71, 0x2f, 0x15, 0x09, 0x7d, 0xba, 0x0f, 0x2d,
			0x62, 0xe8, 0x45, 0x41, 0x23, 0x95, 0xcf, 0x09,
			0x54, 0x0a, 0xbd, 0x6e, 0x10, 0xc1, 0xa2, 0xe2,
			0x3d, 0xbf, 0x2f, 0xe1, 0xdf, 0xd2, 0xb0, 0x2a,
			0xf4, 0xee, 0xa4, 0x75, 0x15, 0x95, 0x7f, 0xa3,
			0x73, 0x8b, 0x06, 0x41, 0x1a, 0x55, 0x1f, 0x8f,
			0x8d, 0xc4, 0xb8, 0x5e, 0xa7, 0xf5, 0xa3, 0xa1,
			0xe2, 0x6c, 0xcc, 0x44, 0x98, 0xbd, 0x64, 0xaf,
			0x80, 0x38, 0xc1, 0xda, 0x5c, 0xbd, 0x8e, 0x80,
			0xb3, 0xcb, 0xac, 0xde, 0xf1, 0xa4, 0x1e, 0xc5,
			0xaf, 0x20, 0x55, 0x66, 0xc8, 0xdd, 0x80, 0xb2,
			0xea, 0xda, 0xf9, 0x7d, 0xd0, 0xaa, 0x98, 0x33,
			0xba, 0x3f, 0xd0, 0xe4, 0xb6, 0x73, 0xe2, 0xf8,
			0x96, 0x0b, 0x04, 0xed, 0xa7, 0x61, 0x61, 0x64,
			0x39, 0x14, 0x24, 0x2b, 0x96, 0x1e, 0x74, 0xde,
			0xae, 0x49, 0x7c, 0xaf, 0x00, 0x5b, 0x00, 0x51,
			0x5d, 0x78, 0x49, 0x2e, 0xc2, 0xc2, 0xde, 0xb6,
			0x0a, 0x57, 0xb9, 0xdc, 0xe3, 0x6e, 0x68, 0xdd,
			0x82, 0x00, 0x7d, 0x94, 0x2a, 0xe7, 0xc0, 0x23,
			0xe1, 0x21, 0x0f, 0x0b, 0xe8, 0xa3, 0xeb, 0x3f,
			0x00, 0x48, 0x24, 0x07, 0x4b, 0x8f, 0x72, 0x5e,
			0xaf, 0x8a, 0xc7, 0x73, 0xe6, 0x0f, 0xbb, 0xb7,
			0xcb, 0xa9, 0x63, 0x0e, 0x88, 0xb6, 0x9c, 0x8b,
			0xcb, 0x2d, 0x74, 0xdb, 0xdb, 0x29, 0xbf, 0xff,
			0x8b, 0x22, 0x54, 0x5b, 0x80, 0xbb, 0x63, 0x4e,
			0x4c, 0x05, 0xf7, 0x3e, 0x00, 0x2a, 0x92, 0x8e,
			0xfd, 0x5a, 0x6a, 0xa4, 0x56, 0x21, 0xce, 0x1b,
			0x03, 0x2a, 0x22, 0x44, 0xde, 0x48, 0xf4, 0xdf,
			0x43, 0x58, 0x15, 0x66, 0x78, 0xcb, 0xe0, 0x39,
			0xc9, 0xeb, 0xe4, 0xce, 0xe9, 0x45, 0xa2, 0x5b,
			0x90, 0x38, 0x46, 0x9f, 0xe0, 0x0c, 0x30, 0x92,
			0x93, 0x6a, 0x8c, 0xff, 0x93, 0x69, 0x04, 0x5f,
			0x90, 0x67, 0x33, 0xa9, 0xd2, 0xab, 0x36, 0x60,
			0x18, 0x20, 0x69, 0xb1, 0x57, 0xca, 0x8f, 0x9b,
			0x99, 0xa7, 0x1f, 0xc1, 0x53, 0xc6, 0x83, 0x01,
			0xe9, 0x7a, 0x38, 0xfc, 0x3a, 0x87, 0xae, 0x2b,
			0x6f, 0x03, 0x75, 0x4e, 0x6d, 0xa8, 0x2d, 0x0b,
			0x07, 0x26, 0xe0, 0x70, 0x39, 0x79, 0xc9, 0x32,
			0x02, 0x89, 0xfe, 0xef, 0xbc, 0xdd, 0xcd, 0x9d,
			0x70, 0x6b, 0x71, 0xb5, 0x1e, 0x9a, 0x1b, 0x9d,
			0xc1, 0x41, 0x2e, 0x6e, 0xd4, 0xb5, 0x66, 0x76,
		},
	},
	/* SHA512 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0xf3, 0x0c, 0x78, 0x3b, 0x4e, 0xae, 0xb4, 0x65,
			0x76, 0x7f, 0xa1, 0xb9, 0x6d, 0x0a, 0xf5, 0x24,
			0x35, 0xd8, 0x5f, 0xab, 0x91, 0x2b, 0x6a, 0xba,
			0x10, 0xef, 0xa5, 0xb9, 0x46, 0xed, 0x01, 0xe1,
			0x5d, 0x42, 0x7a, 0x4e, 0xcd, 0x0f, 0xf9, 0x55,
			0x67, 0x73, 0x79, 0x17, 0x98, 0xb6, 0x69, 0x56,
			0xec, 0xc7, 0x52, 0x88, 0xd1, 0xe9, 0xba, 0x2a,
			0x9e, 0xa9, 0x48, 0x57, 0xd3, 0x13, 0x29, 0x99,
			0xa2, 0x25, 0xb1, 0xff, 0xaf, 0x84, 0x46, 0x70,
			0x15, 0x6e, 0x7a, 0x3e, 0xa9, 0xf0, 0x77, 0xfe,
			0x82, 0x59, 0xa0, 0x98, 0xb9, 0xee, 0x75, 0x9a,
			0x6d, 0xdf, 0xb7, 0xd2, 0x0a, 0x7a, 0xcd, 0x1b,
			0xcb, 0x9f, 0x67, 0x77, 0x7e, 0x74, 0x61, 0x5e,
			0x88, 0x59, 0xea, 0x56, 0x28, 0x1f, 0xe5, 0xc4,
			0x00, 0x74, 0x8f, 0x02, 0xd1, 0xa2, 0x63, 0xb1,
			0x86, 0x7a, 0x3b, 0x51, 0x74, 0x8a, 0xb7, 0x0f,
		},
		/* S8 : 384 byte */
		{
			0x34, 0x5e, 0x2f, 0x60, 0xf7, 0xc8, 0x2c, 0x89,
			0xef, 0x7d, 0xfd, 0x7d, 0xff, 0x2b, 0xc2, 0x34,
			0x8b, 0xab, 0x02, 0x04, 0x79, 0x33, 0x08, 0x99,
			0xd4, 0x41, 0x02, 0x13, 0xb3, 0x5e, 0x98, 0xd9,
			0xba, 0xc9, 0x2f, 0xd8, 0xae, 0x80, 0x6b, 0x5b,
			0xce, 0x8a, 0x6c, 0x4b, 0xd8, 0x27, 0x5b, 0x0f,
			0xac, 0xb4, 0xdd, 0x13, 0xf9, 0xd6, 0x8b, 0xa6,
			0x71, 0x41, 0xfa, 0x50, 0x85, 0x26, 0x4d, 0xa6,
			0xdd, 0x68, 0x5a, 0x6d, 0x21, 0x21, 0x70, 0xa2,
			0xc9, 0xcb, 0xf2, 0xcf, 0x59, 0x30, 0x18, 0x0e,
			0xff, 0xc2, 0x50, 0x86, 0x8c, 0x98, 0x4b, 0xf5,
			0x0f, 0xf6, 0x9d, 0x60, 0x69, 0xea, 0x28, 0xf5,
			0xbc, 0x1b, 0x63, 0x70, 0x5d, 0x07, 0x32, 0x41,
			0x6f, 0xd8, 0x29, 0xa5, 0xf5, 0xd6, 0x21, 0x74,
			0x62, 0xc2, 0x2a, 0x33, 0xfd, 0x46, 0x52, 0xf7,
			0xc1, 0xd1, 0x98, 0x79, 0x46, 0x46, 0xc0, 0x84,
			0x06, 0x02, 0x4e, 0x81, 0x63, 0xa7, 0xeb, 0xe3,
			0x9c, 0xfb, 0x51, 0x4c, 0x54, 0x43, 0x89, 0x7b,
			0x58, 0x94, 0xdd, 0x19, 0xa2, 0x13, 0xe0, 0x37,
			0xf2, 0x7e, 0x0f, 0xfb, 0xd6, 0xc5, 0x44, 0x7a,
			0x80, 0x5a, 0x54, 0xdf, 0xdf, 0x4f, 0x65, 0x81,
			0x9d, 0x4e, 0x0f, 0xbe, 0xe2, 0x5e, 0x3d, 0xac,
			0x47, 0xfb, 0x6b, 0x63, 0x6e, 0x8d, 0xe6, 0x19,
			0x0a, 0xdc, 0xcb, 0xce, 0xe9, 0x37, 0xd0, 0x97,
			0x7b, 0x35, 0xb9, 0x73, 0x60, 0x6b, 0x0c, 0xa3,
			0x48, 0x75, 0x8b, 0x50, 0xcd, 0xbb, 0xa0, 0x28,
			0xb7, 0x3d, 0x0e, 0xf0, 0x1c, 0x56, 0x01, 0x4c,
			0x03, 0x1c, 0x59, 0x8f, 0xe8, 0xdb, 0x87, 0xd2,
			0xca, 0x46, 0x44, 0x77, 0x0a, 0xaa, 0x04, 0x51,
			0xc3, 0x76, 0xde, 0xd8, 0x2f, 0xf5, 0xc6, 0xb8,
			0xe7, 0xd2, 0xed, 0x9d, 0x1c, 0x8a, 0x17, 0xc3,
			0x12, 0x2c, 0x12, 0x82, 0x73, 0xc6, 0x0f, 0xd1,
			0xb0, 0x08, 0x8d, 0xfb, 0xc9, 0xc9, 0x27, 0xf1,
			0x62, 0xe4, 0x38, 0x79, 0x40, 0x59, 0x64, 0xcb,
			0x11, 0xef, 0x78, 0x99, 0x12, 0x3f, 0xeb, 0x8f,
			0x88, 0xdd, 0x27, 0x34, 0xdf, 0x98, 0xaa, 0x69,
			0x6d, 0x93, 0x6a, 0x8d, 0xf0, 0x70, 0x00, 0xe8,
			0x4a, 0xf9, 0x01, 0x01, 0xf7, 0x00, 0x6a, 0x9b,
			0xd2, 0x54, 0x9f, 0xdd, 0x0a, 0xd3, 0xf9, 0xde,
			0x09, 0x30, 0x12, 0xd3, 0x2d, 0x2a, 0xfa, 0xa8,
			0x28, 0x01, 0x7e, 0xe9, 0xc6, 0x07, 0xcb, 0xf5,
			0xb5, 0x4f, 0x22, 0x36, 0x66, 0xd4, 0xb5, 0xf3,
			0xe2, 0x6e, 0x0d, 0xfe, 0xc0, 0x03, 0x96, 0x1b,
			0x83, 0xd8, 0x3d, 0xe3, 0x9f, 0xf6, 0xa0, 0xe8,
			0x1e, 0x18, 0x83, 0xc1, 0xdb, 0x4a, 0xaa, 0xf0,
			0x82, 0xfe, 0xc5, 0xaa, 0x30, 0xa7, 0xe5, 0x78,
			0x55, 0x3d, 0x89, 0x77, 0x4c, 0x67, 0x90, 0x77,
			0x90, 0xc9, 0x6d, 0xc4, 0xf5, 0xbe, 0x4c, 0x8c,
		},
	},
	/* SHA512 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0x13, 0x2c, 0xf5, 0x0c, 0x66, 0xac, 0x4c, 0xc5,
			0x43, 0x39, 0x75, 0x1a, 0x0e, 0xbb, 0x86, 0x5e,
			0x1d, 0x3d, 0x32, 0x05, 0x62, 0xfc, 0x90, 0x5c,
			0x4a, 0xbd, 0x1e, 0x78, 0xe4, 0x64, 0x06, 0x6c,
			0x46, 0xc3, 0xa0, 0xc0, 0x2d, 0xb0, 0x37, 0x1e,
			0xe3, 0x5a, 0x10, 0x4d, 0x66, 0xdd, 0xa8, 0x64,
			0xc6, 0x13, 0x3e, 0x37, 0xcf, 0xad, 0x91, 0x16,
			0xe8, 0x83, 0xeb, 0xb7, 0x3b, 0x29, 0x5e, 0x70,
			0x16, 0xc3, 0x4e, 0xa9, 0x91, 0x1a, 0x30, 0x92,
			0x72, 0xef, 0x90, 0x11, 0x4d, 0x8f, 0x59, 0xff,
			0xf0, 0xa7, 0x51, 0x93, 0xfe, 0x5a, 0xe3, 0x1e,
			0xd9, 0x91, 0x21, 0xf9, 0xc5, 0x92, 0x09, 0xbc,
			0x4b, 0xd5, 0x07, 0xb1, 0xdc, 0x12, 0xbc, 0x89,
			0xb7, 0x9f, 0xfe, 0x4d, 0x0d, 0xf9, 0x20, 0x97,
			0x62, 0xa1, 0x73, 0x01, 0x36, 0x29, 0x0c, 0xde,
			0xe5, 0x8e, 0xc8, 0x28, 0xcc, 0xc8, 0x8e, 0xba,
		},
		/* S9 : 384 byte */
		{
			0xb1, 0x25, 0x03, 0xb7, 0xb2, 0xf7, 0x83, 0x61,
			0x88, 0x84, 0x17, 0x4b, 0xcb, 0x9b, 0xe1, 0x08,
			0x77, 0x96, 0x04, 0x31, 0xed, 0x63, 0x63, 0xc8,
			0x07, 0xe1, 0x2d, 0xb7, 0x1b, 0x8b, 0x6b, 0xd9,
			0xd6, 0x40, 0x1d, 0x06, 0x4e, 0x25, 0x37, 0x40,
			0x15, 0x8e, 0x8b, 0x90, 0x01, 0x52, 0xd3, 0x7f,
			0xaf, 0x20, 0x33, 0x3a, 0x7d, 0x80, 0xb3, 0xd4,
			0x7c, 0x7c, 0x7a, 0x3f, 0xa1, 0x20, 0x91, 0xce,
			0x31, 0xcd, 0x8a, 0xae, 0x27, 0x2a, 0x4d, 0xa1,
			0x5f, 0xe2, 0xcb, 0x5c, 0xfd, 0xea, 0x54, 0x11,
			0x95, 0xa4, 0x69, 0xc9, 0x6b, 0xcf, 0x69, 0x5e,
			0x0b, 0x52, 0x6d, 0xfa, 0x48, 0xa5, 0x90, 0x03,
			0xc6, 0x76, 0x3a, 0xf8, 0x13, 0x63, 0x92, 0xc4,
			0xb8, 0xd2, 0x4d, 0xb3, 0x14, 0x74, 0x6f, 0x42,
			0xac, 0xa5, 0x50, 0xac, 0xc6, 0x5e, 0x07, 0x49,
			0x13, 0xab, 0x82, 0x23, 0x2e, 0xb8, 0x59, 0x35,
			0x09, 0x15, 0x8a, 0x8b, 0xa3, 0x4b, 0xc0, 0xf0,
			0xe3, 0x12, 0x5a, 0x83, 0x4a, 0x3e, 0xd2, 0xd6,
			0xa8, 0xcb, 0x1d, 0x08, 0x5f, 0x23, 0x4a, 0xe8,
			0x68, 0xb8, 0x6a, 0xea, 0x8d, 0x6f, 0x82, 0xe1,
			0x3a, 0x08, 0x84, 0x24, 0x85, 0x06, 0x6e, 0x48,
			0xaa, 0xe4, 0x83, 0x78, 0x73, 0x15, 0x0f, 0x44,
			0x47, 0x5e, 0x12, 0x60, 0x2b, 0x55, 0x2d, 0xcb,
			0x34, 0xd1, 0xf9, 0xfd, 0xaa, 0xdb, 0xc6, 0xbf,
			0xf5, 0x13, 0x4c, 0x6f, 0xc7, 0x62, 0x63, 0x88,
			0x8b, 0xe6, 0x7e, 0xfe, 0x63, 0xee, 0x18, 0x40,
			0xfa, 0x08, 0xc4, 0x99, 0x38, 0x85, 0x8a, 0x9d,
			0x48, 0xb1, 0x05, 0x8d, 0x18, 0x97, 0x6b, 0xf2,
			0xe3, 0xbf, 0xc6, 0x25, 0x55, 0x2f, 0x75, 0xb3,
			0xea, 0x44, 0xeb, 0x91, 0xdd, 0x36, 0x68, 0x65,
			0xf2, 0x40, 0xa0, 0xc3, 0x36, 0xa0, 0x11, 0x0e,
			0x0f, 0xa0, 0x9d, 0x09, 0xcd, 0x94, 0xc7, 0x0c,
			0xbc, 0x88, 0x95, 0xae, 0x3d, 0x44, 0xae, 0x3d,
			0xff, 0x54, 0x5f, 0x0e, 0x8c, 0x8c, 0xc6, 0x62,
			0xec, 0xd4, 0x0f, 0x90, 0x99, 0xa9, 0x52, 0x49,
			0x43, 0x96, 0xc6, 0xb4, 0x23, 0xeb, 0xb4, 0x63,
			0x40, 0x99, 0x69, 0x28, 0x1c, 0xdd, 0x54, 0xad,
			0x87, 0xa3, 0x08, 0xe4, 0x87, 0xce, 0x19, 0x74,
			0x5b, 0x30, 0xd5, 0xda, 0x76, 0xb9, 0x8d, 0x2a,
			0xa9, 0xa0, 0x07, 0xa5, 0x57, 0x83, 0xb3, 0x03,
			0x7e, 0x5b, 0x86, 0x62, 0x32, 0x28, 0x10, 0xbd,
			0xd1, 0x1d, 0x86, 0xdc, 0x3f, 0x61, 0x45, 0x11,
			0x49, 0x39, 0x1f, 0xb2, 0xf1, 0x4e, 0xd9, 0xc1,
			0x7c, 0x75, 0x16, 0x23, 0xa4, 0x04, 0x2c, 0xe7,
			0xed, 0xb8, 0x75, 0xee, 0x27, 0xbc, 0xd1, 0xf1,
			0x9d, 0x6d, 0xc9, 0x28, 0x3a, 0xd0, 0x6d, 0x15,
			0xe0, 0x97, 0xe2, 0xb0, 0xb1, 0x5a, 0x7e, 0xb7,
			0x12, 0x8a, 0xdb, 0xca, 0x0a, 0xa6, 0xad, 0xcc,
		},
	},
};

typedef mod3072_t tv_t;


#include "siggen15test.c"
