/* siggen15_3_2048_testfc.c */
/*
 * Copyright (c) 2017 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

/*
 * This test vector was created based on SigGen15_186-3.txt from
 * https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Algorithm-Validation-Program/documents/dss/186-3rsatestvectors.zip
 *
 * The link to the archive file exists in the following URL:
 * https://csrc.nist.gov/projects/cryptographic-algorithm-validation-program/digital-signatures/
 */
#include "siggen15test.h"

static const unsigned short sn1[] = {
	0xcea8, 0x0475, 0x324c, 0x1dc8, 0x3478, 0x2781, 0x8da5, 0x8bac,
	0x069d, 0x3419, 0xc614, 0xa6ea, 0x1ac6, 0xa3b5, 0x10dc, 0xd72c,
	0xc516, 0x9549, 0x05e9, 0xfef9, 0x08d4, 0x5e13, 0x006a, 0xdf27,
	0xd467, 0xa7d8, 0x3c11, 0x1d1a, 0x5df1, 0x5ef2, 0x9377, 0x1aef,
	0xb920, 0x032a, 0x5bb9, 0x89f8, 0xe4f5, 0xe1b0, 0x5093, 0xd3f1,
	0x30f9, 0x84c0, 0x7a77, 0x2a36, 0x83f4, 0xdc6f, 0xb28a, 0x9681,
	0x5b32, 0x123c, 0xcdd1, 0x3954, 0xf19d, 0x5b8b, 0x24a1, 0x03e7,
	0x71a3, 0x4c32, 0x8755, 0xc65e, 0xd64e, 0x1924, 0xffd0, 0x4d30,
	0xb214, 0x2cc2, 0x62f6, 0xe004, 0x8fef, 0x6dbc, 0x652f, 0x2147,
	0x9ea1, 0xc4b1, 0xd66d, 0x28f4, 0xd46e, 0xf718, 0x5e39, 0x0cbf,
	0xa2e0, 0x2380, 0x582f, 0x3188, 0xbb94, 0xebbf, 0x05d3, 0x1487,
	0xa09a, 0xff01, 0xfcbb, 0x4cd4, 0xbfd1, 0xf0a8, 0x33b3, 0x8c11,
	0x813c, 0x8436, 0x0bb5, 0x3c7d, 0x4481, 0x031c, 0x40ba, 0xd871,
	0x3bb6, 0xb835, 0xcb08, 0x098e, 0xd15b, 0xa31e, 0xe4ba, 0x728a,
	0x8c8e, 0x10f7, 0x294e, 0x1b41, 0x63b7, 0xaee5, 0x7277, 0xbfd8,
	0x81a6, 0xf9d4, 0x3e02, 0xc692, 0x5aa3, 0xa043, 0xfb7f, 0xb78d,
};

static const unsigned short se1[] = {
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0026, 0x0445,
};

static const unsigned short sd1[] = {
	0x0997, 0x634c, 0x477c, 0x1a03, 0x9d44, 0xc810, 0xb2aa, 0xa3c7,
	0x862b, 0x0b88, 0xd370, 0x8272, 0xe1e1, 0x5f66, 0xfc93, 0x8970,
	0x9f8a, 0x11f3, 0xea6a, 0x5af7, 0xeffa, 0x2d01, 0xc189, 0xc50f,
	0x0d5b, 0xcbe3, 0xfa27, 0x2e56, 0xcfc4, 0xa4e1, 0xd388, 0xa9dc,
	0xd65d, 0xf862, 0x8902, 0x556c, 0x8b6b, 0xb6a6, 0x4170, 0x9b5a,
	0x35dd, 0x2622, 0xc73d, 0x4640, 0xbfa1, 0x359d, 0x0e76, 0xe1f2,
	0x19f8, 0xe33e, 0xb9bd, 0x0b59, 0xec19, 0x8eb2, 0xfcca, 0xae03,
	0x46bd, 0x8b40, 0x1e12, 0xe3c6, 0x7cb6, 0x2956, 0x9c18, 0x5a2e,
	0x0f35, 0xa2f7, 0x4164, 0x4c1c, 0xca5e, 0xbb13, 0x9d77, 0xa89a,
	0x2953, 0xfc5e, 0x3004, 0x8c0e, 0x619f, 0x07c8, 0xd21d, 0x1e56,
	0xb8af, 0x0719, 0x3d0f, 0xdf3f, 0x49cd, 0x49f2, 0xef31, 0x38b5,
	0x1388, 0x62f1, 0x470b, 0xd2d1, 0x6e34, 0xa2b9, 0xe777, 0x7a6c,
	0x8c8d, 0x4cb9, 0x4b4e, 0x8b5d, 0x616c, 0xd539, 0x3753, 0xe7b0,
	0xf31c, 0xc7da, 0x559b, 0xa8e9, 0x8d88, 0x8914, 0xe334, 0x773b,
	0xaf49, 0x8ad8, 0x8d96, 0x31eb, 0x5fe3, 0x2e53, 0xa414, 0x5bf0,
	0xba54, 0x8bf2, 0xb0a5, 0x0c63, 0xf67b, 0x14e3, 0x98a3, 0x4b0d,
};

#define HAS_SHA224_TESTVECTOR

#define HAS_SHA256_TESTVECTOR

#define HAS_SHA384_TESTVECTOR

#define HAS_SHA512_TESTVECTOR

/* 10 elements */
static mod2048_t SHA224[] = {
	/* SHA224 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0x74, 0x23, 0x04, 0x47, 0xbc, 0xd4, 0x92, 0xf2,
			0xf8, 0xa8, 0xc5, 0x94, 0xa0, 0x43, 0x79, 0x27,
			0x16, 0x90, 0xbf, 0x0c, 0x8a, 0x13, 0xdd, 0xfc,
			0x1b, 0x7b, 0x96, 0x41, 0x3e, 0x77, 0xab, 0x26,
			0x64, 0xcb, 0xa1, 0xac, 0xd7, 0xa3, 0xc5, 0x7e,
			0xe5, 0x27, 0x6e, 0x27, 0x41, 0x4f, 0x82, 0x83,
			0xa6, 0xf9, 0x3b, 0x73, 0xbd, 0x39, 0x2b, 0xd5,
			0x41, 0xf0, 0x7e, 0xb4, 0x61, 0xa0, 0x80, 0xbb,
			0x66, 0x7e, 0x5f, 0xf0, 0x95, 0xc9, 0x31, 0x9f,
			0x57, 0x5b, 0x38, 0x93, 0x97, 0x7e, 0x65, 0x8c,
			0x6c, 0x00, 0x1c, 0xee, 0xf8, 0x8a, 0x37, 0xb7,
			0x90, 0x2d, 0x4d, 0xb3, 0x1c, 0x3e, 0x34, 0xf3,
			0xc1, 0x64, 0xc4, 0x7b, 0xbe, 0xef, 0xde, 0x3b,
			0x94, 0x6b, 0xad, 0x41, 0x6a, 0x75, 0x2c, 0x2c,
			0xaf, 0xce, 0xe9, 0xe4, 0x01, 0xae, 0x08, 0x88,
			0x4e, 0x5b, 0x8a, 0xa8, 0x39, 0xf9, 0xd0, 0xb5,
		},
		/* S0 : 256 byte */
		{
			0x27, 0xda, 0x41, 0x04, 0xea, 0xce, 0x19, 0x91,
			0xe0, 0x8b, 0xd8, 0xe7, 0xcf, 0xcc, 0xd9, 0x7e,
			0xc4, 0x8b, 0x89, 0x6a, 0x0e, 0x15, 0x6c, 0xe7,
			0xbd, 0xc2, 0x3f, 0xd5, 0x70, 0xaa, 0xa9, 0xa0,
			0x0e, 0xd0, 0x15, 0x10, 0x1f, 0x0c, 0x62, 0x61,
			0xc7, 0x37, 0x1c, 0xec, 0xa3, 0x27, 0xa7, 0x3c,
			0x3c, 0xec, 0xfc, 0xf6, 0xb2, 0xd9, 0xed, 0x92,
			0x0c, 0x96, 0x98, 0x04, 0x6e, 0x25, 0xc8, 0x9a,
			0xdb, 0x23, 0x60, 0x88, 0x7d, 0x99, 0x98, 0x3b,
			0xf6, 0x32, 0xf9, 0xe6, 0xeb, 0x0e, 0x5d, 0xf6,
			0x07, 0x15, 0x90, 0x2b, 0x9a, 0xea, 0xa7, 0x4b,
			0xf5, 0x02, 0x7a, 0xa2, 0x46, 0x51, 0x08, 0x91,
			0xc7, 0x4a, 0xe3, 0x66, 0xa1, 0x6f, 0x39, 0x7e,
			0x2c, 0x8c, 0xcd, 0xc8, 0xbd, 0x56, 0xaa, 0x10,
			0xe0, 0xd0, 0x15, 0x85, 0xe6, 0x9f, 0x8c, 0x48,
			0x56, 0xe7, 0x6b, 0x53, 0xac, 0xfd, 0x3d, 0x78,
			0x2b, 0x81, 0x71, 0x52, 0x90, 0x08, 0xfa, 0x5e,
			0xff, 0x03, 0x0f, 0x46, 0x95, 0x67, 0x04, 0xa3,
			0xf5, 0xd9, 0x16, 0x73, 0x48, 0xf3, 0x70, 0x21,
			0xfc, 0x27, 0x7c, 0x6c, 0x0a, 0x8f, 0x93, 0xb8,
			0xa2, 0x3c, 0xfb, 0xf9, 0x18, 0x99, 0x0f, 0x98,
			0x2a, 0x56, 0xd0, 0xed, 0x2a, 0xa0, 0x81, 0x61,
			0x56, 0x07, 0x55, 0xad, 0xc0, 0xce, 0x2c, 0x3e,
			0x2a, 0xb2, 0x92, 0x9f, 0x79, 0xbf, 0xc0, 0xb2,
			0x4f, 0xf3, 0xe0, 0xff, 0x35, 0x2e, 0x64, 0x45,
			0xd8, 0xa6, 0x17, 0xf1, 0x78, 0x5d, 0x66, 0xc3,
			0x22, 0x95, 0xbb, 0x36, 0x5d, 0x61, 0xcf, 0xb1,
			0x07, 0xe9, 0x99, 0x3b, 0xbd, 0x93, 0x42, 0x1f,
			0x2d, 0x34, 0x4a, 0x86, 0xe4, 0x12, 0x78, 0x27,
			0xfa, 0x0d, 0x0b, 0x25, 0x35, 0xf9, 0xb1, 0xd5,
			0x47, 0xde, 0x12, 0xba, 0x28, 0x68, 0xac, 0xde,
			0xcf, 0x2c, 0xb5, 0xf9, 0x2a, 0x6a, 0x15, 0x9a,
		},
	},
	/* SHA224 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0x9a, 0xf2, 0xc5, 0xa9, 0x19, 0xe5, 0xda, 0xdc,
			0x66, 0x87, 0x99, 0xf3, 0x65, 0xfc, 0x23, 0xda,
			0x62, 0x31, 0x43, 0x7e, 0xa5, 0x1c, 0xa5, 0x31,
			0x46, 0x45, 0x42, 0x50, 0x43, 0x85, 0x1f, 0x23,
			0xd0, 0x0d, 0x37, 0x04, 0xee, 0xab, 0xb5, 0xc4,
			0x3f, 0x49, 0x67, 0x4a, 0x19, 0xb7, 0x70, 0x7d,
			0xd9, 0xaa, 0x3d, 0x65, 0x7a, 0x04, 0xba, 0x8c,
			0x66, 0x55, 0xc5, 0xab, 0x8b, 0xa2, 0xe3, 0x82,
			0xb2, 0x66, 0x31, 0x08, 0x0c, 0xd7, 0x9e, 0xc4,
			0x0e, 0x6a, 0x58, 0x7b, 0x7f, 0x99, 0x84, 0x0b,
			0xd0, 0xe4, 0x32, 0x97, 0xab, 0x16, 0x90, 0xe4,
			0xce, 0xc9, 0x5d, 0x03, 0x1a, 0x2c, 0xa1, 0x31,
			0xe7, 0x04, 0x9c, 0xfb, 0x9b, 0xf1, 0xfc, 0xa6,
			0x7b, 0xf3, 0x53, 0xcd, 0xc1, 0x2c, 0xc7, 0x4c,
			0xee, 0xe8, 0x0c, 0x5d, 0x61, 0xda, 0x8f, 0x01,
			0x29, 0xa8, 0xf4, 0xa2, 0x18, 0xab, 0xc3, 0xf6,
		},
		/* S1 : 256 byte */
		{
			0xc5, 0xdf, 0xbe, 0xfd, 0x35, 0xce, 0xc8, 0x46,
			0xe2, 0xc7, 0xb2, 0x43, 0x4d, 0xc9, 0xc4, 0x6a,
			0x5a, 0x9b, 0x1b, 0x6c, 0xe6, 0x5b, 0x2b, 0x18,
			0x66, 0x5a, 0xed, 0xb1, 0x40, 0x4d, 0xe1, 0xf4,
			0x66, 0xe0, 0x24, 0xf8, 0x49, 0xee, 0xc3, 0x08,
			0xc2, 0xd2, 0xf2, 0xf0, 0x19, 0x3d, 0xf1, 0x89,
			0x8a, 0x58, 0x1c, 0x9e, 0xa3, 0x25, 0x81, 0x18,
			0x55, 0x53, 0xb1, 0x71, 0xb6, 0x50, 0x70, 0x82,
			0x61, 0x7c, 0x5c, 0x01, 0x8a, 0xfe, 0x0c, 0x3a,
			0xf6, 0x4d, 0x2e, 0xc5, 0xa5, 0x63, 0x79, 0x5a,
			0xa5, 0x85, 0xe7, 0x77, 0x53, 0xcd, 0x18, 0x83,
			0x6f, 0x6f, 0x0c, 0x29, 0x53, 0x5f, 0x62, 0x00,
			0xca, 0x89, 0x99, 0x28, 0xfe, 0x78, 0xe9, 0x49,
			0xb0, 0xa2, 0x16, 0xec, 0x47, 0xa6, 0xad, 0xf2,
			0x22, 0x3e, 0x17, 0x23, 0x6c, 0xfc, 0x16, 0x7c,
			0xf0, 0x0e, 0xd6, 0x13, 0x6f, 0x03, 0xcf, 0x6f,
			0xfd, 0x4f, 0x3f, 0x77, 0x87, 0xae, 0xb0, 0x05,
			0x84, 0x09, 0x78, 0xd8, 0xd6, 0xba, 0x59, 0x3d,
			0x4f, 0x4c, 0xfe, 0x69, 0x20, 0xbe, 0x10, 0x2b,
			0x98, 0x47, 0xd1, 0x01, 0x40, 0xdf, 0xf8, 0x6b,
			0x0d, 0xb1, 0x4f, 0xfc, 0xcc, 0x9a, 0x96, 0xe6,
			0x73, 0xc6, 0x72, 0xc1, 0x12, 0x8a, 0xe4, 0x54,
			0x89, 0xd2, 0xcb, 0xfe, 0x6e, 0x19, 0x5c, 0xa5,
			0x20, 0x6e, 0xda, 0x51, 0x9c, 0xad, 0x3d, 0x6e,
			0x0a, 0xbf, 0x46, 0x53, 0xe3, 0x6b, 0x5a, 0x26,
			0x4e, 0x87, 0x49, 0x4a, 0x4d, 0x63, 0xee, 0x91,
			0xff, 0x7c, 0x35, 0xa6, 0xab, 0x12, 0xad, 0xfa,
			0x3b, 0xb5, 0x37, 0xf6, 0x19, 0x8b, 0x06, 0xf5,
			0xde, 0x07, 0x17, 0x07, 0x6b, 0x0e, 0xc8, 0x3a,
			0xe0, 0xda, 0x9e, 0xa4, 0x19, 0xcc, 0x0c, 0x96,
			0x66, 0x9d, 0x1d, 0x7c, 0x9e, 0x52, 0x92, 0x71,
			0x42, 0x84, 0x01, 0xe0, 0x9e, 0x04, 0x88, 0x8a,
		},
	},
	/* SHA224 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x59, 0xb5, 0xb8, 0x5b, 0x9d, 0xc2, 0x46, 0xd3,
			0x0a, 0x3f, 0xc8, 0xa2, 0xde, 0x3c, 0x9d, 0xfa,
			0x97, 0x16, 0x43, 0xb0, 0xc1, 0xf7, 0xc9, 0xe4,
			0x0c, 0x9c, 0x87, 0xe4, 0xa1, 0x5b, 0x0c, 0x4e,
			0xb6, 0x64, 0x58, 0x75, 0x60, 0x47, 0x4c, 0x06,
			0xa9, 0xb6, 0x5e, 0xec, 0xe3, 0x8c, 0x91, 0x70,
			0x3c, 0x0f, 0xa5, 0xa5, 0x92, 0x72, 0x8a, 0x03,
			0x88, 0x9f, 0x1b, 0x52, 0xd9, 0x33, 0x09, 0xca,
			0xec, 0xc9, 0x15, 0x78, 0xa9, 0x7b, 0x83, 0xe3,
			0x8c, 0xa6, 0xcb, 0xf0, 0xf7, 0xee, 0x91, 0x03,
			0xcd, 0x82, 0xd7, 0x67, 0x3c, 0xa1, 0x72, 0xf0,
			0xda, 0x5e, 0xba, 0xde, 0xf4, 0xa0, 0x86, 0x05,
			0x22, 0x6c, 0x58, 0x2b, 0x1f, 0x67, 0xd4, 0xb2,
			0xd8, 0x96, 0x77, 0x77, 0xc3, 0x69, 0x85, 0xf9,
			0x72, 0xf8, 0x43, 0xbe, 0x68, 0x8c, 0x67, 0xf2,
			0x2b, 0x61, 0xcd, 0x52, 0x9b, 0xaa, 0x6b, 0x48,
		},
		/* S2 : 256 byte */
		{
			0x29, 0xb5, 0xac, 0x41, 0x72, 0x26, 0x44, 0x4b,
			0xc8, 0x57, 0x0a, 0x27, 0x9e, 0x0e, 0x56, 0x1a,
			0x4c, 0x39, 0x70, 0x7b, 0xdb, 0xea, 0x93, 0x60,
			0x64, 0xed, 0x60, 0x3b, 0xa9, 0x68, 0x89, 0xeb,
			0x3d, 0x78, 0x6b, 0x19, 0x99, 0xb5, 0x18, 0x0c,
			0xd5, 0xd0, 0x61, 0x17, 0x88, 0x83, 0x7a, 0x9d,
			0xf1, 0x49, 0x6b, 0xac, 0xea, 0x31, 0xcb, 0xf8,
			0xf2, 0x4a, 0x1a, 0x22, 0x32, 0xd4, 0x15, 0x89,
			0x13, 0xc9, 0x63, 0xf5, 0x06, 0x6a, 0xad, 0x4b,
			0x65, 0xe6, 0x17, 0xd0, 0x90, 0x33, 0x59, 0x69,
			0x6d, 0x75, 0x9d, 0x84, 0xc1, 0x39, 0x2e, 0x22,
			0xc2, 0x46, 0xd5, 0xf5, 0xbe, 0xd4, 0xb8, 0x06,
			0xf4, 0x09, 0x1d, 0x5e, 0x8f, 0x71, 0xa5, 0x13,
			0xf1, 0x31, 0x9b, 0xb4, 0xe5, 0x69, 0x71, 0xcd,
			0x3e, 0x16, 0x8c, 0x9a, 0x7e, 0x27, 0x89, 0x83,
			0x22, 0x93, 0x99, 0x1a, 0x73, 0xd3, 0x02, 0x70,
			0x72, 0xec, 0xee, 0x68, 0x63, 0x51, 0x45, 0x49,
			0x02, 0x9f, 0xb3, 0x55, 0x34, 0x78, 0xc8, 0xf4,
			0x10, 0x3b, 0xf6, 0x2d, 0x7d, 0xe1, 0xfb, 0x53,
			0xfe, 0x76, 0xce, 0x97, 0x78, 0xad, 0xa3, 0xbb,
			0x9e, 0xfa, 0x62, 0xda, 0x44, 0xcd, 0x00, 0xd0,
			0x2b, 0xb0, 0xeb, 0x74, 0x88, 0xac, 0x24, 0xda,
			0x38, 0x14, 0xc6, 0x53, 0xcb, 0xa6, 0x12, 0x30,
			0x13, 0x73, 0x83, 0x7a, 0x0c, 0x3f, 0x11, 0x88,
			0x54, 0x93, 0xcb, 0xf3, 0x02, 0x4c, 0x35, 0x72,
			0xea, 0xed, 0x39, 0x6d, 0x0e, 0xbb, 0x80, 0x39,
			0xdd, 0xf8, 0x43, 0xc2, 0x18, 0xd8, 0xbc, 0x77,
			0x83, 0x54, 0x90, 0x46, 0xc3, 0x35, 0x86, 0xfb,
			0x34, 0x28, 0x56, 0x2c, 0xb8, 0x04, 0x60, 0x90,
			0x04, 0x0c, 0x0e, 0x4e, 0xea, 0x50, 0xa1, 0x9a,
			0x42, 0x8b, 0xde, 0x34, 0x62, 0x62, 0x77, 0xff,
			0x48, 0xa8, 0x4f, 0xaa, 0x18, 0x9b, 0x54, 0x40,
		},
	},
	/* SHA224 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0x49, 0xa5, 0xf3, 0x93, 0x0a, 0xd4, 0x5a, 0xca,
			0x5e, 0x22, 0xca, 0xac, 0x66, 0x46, 0xf0, 0xbe,
			0xde, 0x12, 0x28, 0x83, 0x8d, 0x49, 0xf8, 0xf2,
			0xe0, 0xb2, 0xdd, 0x27, 0xd2, 0x6a, 0x4b, 0x59,
			0x0e, 0x7e, 0xef, 0x0c, 0x58, 0xb9, 0x37, 0x88,
			0x29, 0xbb, 0x14, 0x89, 0x99, 0x4b, 0xff, 0x38,
			0x82, 0xef, 0x3a, 0x5a, 0xe3, 0xb9, 0x58, 0xc8,
			0x82, 0x63, 0xff, 0x1f, 0xd6, 0x9f, 0xed, 0xb8,
			0x23, 0xa8, 0x39, 0xdb, 0xe7, 0x1d, 0xdb, 0x2f,
			0x75, 0x0f, 0x6f, 0x75, 0xe0, 0x59, 0x36, 0x76,
			0x1a, 0x2f, 0x5e, 0x3a, 0x5d, 0xfa, 0x83, 0x7b,
			0xca, 0x63, 0x75, 0x59, 0x51, 0xae, 0x3c, 0x50,
			0xd0, 0x4a, 0x59, 0x66, 0x7f, 0xa6, 0x4f, 0xa9,
			0x8b, 0x46, 0x62, 0xd8, 0x01, 0x15, 0x9f, 0x61,
			0xee, 0xfd, 0x1c, 0x8b, 0xc5, 0xb5, 0x81, 0xf5,
			0x00, 0xda, 0xc7, 0x3f, 0x0a, 0x42, 0x40, 0x07,
		},
		/* S3 : 256 byte */
		{
			0x60, 0x4e, 0xb6, 0x37, 0xca, 0x54, 0xbe, 0xa5,
			0xad, 0x1f, 0xd3, 0x16, 0x59, 0x11, 0xf3, 0xba,
			0xa2, 0xe0, 0x6c, 0x85, 0x9d, 0xc7, 0x39, 0x45,
			0xa3, 0x8b, 0xca, 0x7f, 0xf9, 0xbf, 0xa9, 0xed,
			0x39, 0x43, 0x53, 0x48, 0x62, 0x3d, 0x3e, 0x60,
			0xf1, 0xce, 0x48, 0x74, 0x43, 0x84, 0x0c, 0x6b,
			0x2c, 0x00, 0x0f, 0x15, 0x82, 0xe8, 0x52, 0x60,
			0x67, 0xa5, 0xe8, 0x92, 0x3f, 0x1a, 0x1b, 0xda,
			0xab, 0xb1, 0xa4, 0x0c, 0x0f, 0x49, 0xee, 0x69,
			0x06, 0xa4, 0xc8, 0xfc, 0x9b, 0x8c, 0xfa, 0x6d,
			0x07, 0xc2, 0xcc, 0x5b, 0xdf, 0x2a, 0xda, 0x65,
			0xc5, 0x3d, 0x79, 0x54, 0x80, 0x89, 0xc5, 0x24,
			0xfa, 0x36, 0x43, 0x19, 0xa9, 0x0d, 0x46, 0x21,
			0x3f, 0xeb, 0xdc, 0xe6, 0xdb, 0x79, 0x59, 0x14,
			0xcb, 0xda, 0x04, 0xd7, 0xbb, 0xbf, 0x26, 0xbb,
			0xb2, 0x99, 0xfc, 0x7d, 0x14, 0x49, 0xdc, 0xc8,
			0x1d, 0x13, 0x9e, 0x3c, 0x33, 0xd4, 0xc1, 0xde,
			0x96, 0x47, 0x39, 0x94, 0x73, 0x0a, 0x4b, 0x63,
			0x96, 0x33, 0xd6, 0x77, 0xdb, 0x25, 0x69, 0x5f,
			0xfd, 0x15, 0x7e, 0x59, 0x1b, 0xdd, 0xea, 0xd0,
			0x3d, 0xd2, 0xf1, 0xc1, 0xb8, 0xf5, 0xc8, 0xa2,
			0x13, 0xb7, 0x85, 0x87, 0x9b, 0xf7, 0xc9, 0xa9,
			0x92, 0xbb, 0x11, 0xdd, 0x5e, 0x91, 0xdf, 0x3a,
			0xff, 0x09, 0x31, 0xca, 0x76, 0xc4, 0x06, 0x23,
			0x0a, 0x19, 0xe3, 0x07, 0xf3, 0x34, 0x19, 0xc9,
			0xd9, 0xd3, 0xf6, 0xf6, 0x4b, 0xf8, 0x88, 0x1c,
			0x0d, 0xdf, 0x74, 0xa5, 0x71, 0x6c, 0xbc, 0x43,
			0x33, 0x29, 0x36, 0x8d, 0x6e, 0x55, 0xf1, 0xf7,
			0x51, 0xd7, 0xb9, 0xf9, 0xb0, 0xa2, 0x6e, 0xb5,
			0x81, 0x17, 0x72, 0xf5, 0xf6, 0x98, 0x53, 0x0e,
			0xfc, 0x1e, 0xac, 0xee, 0xe6, 0xe1, 0xdc, 0x68,
			0x39, 0xb2, 0x13, 0x3c, 0x2f, 0xcc, 0xfa, 0x8c,
		},
	},
	/* SHA224 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0x9b, 0xfc, 0x4d, 0xac, 0x8c, 0x22, 0x32, 0x38,
			0x72, 0x16, 0xa5, 0x32, 0xce, 0x62, 0xd9, 0x8c,
			0x1a, 0xaf, 0xa3, 0x5c, 0x65, 0xdc, 0x38, 0x8e,
			0x3d, 0x4d, 0x37, 0xd6, 0xd1, 0x86, 0xea, 0xe9,
			0x57, 0xf8, 0xc9, 0xed, 0xac, 0x1a, 0x3f, 0x2e,
			0x3a, 0xbc, 0xb1, 0x12, 0x1f, 0x99, 0xbd, 0x4f,
			0x8c, 0x2b, 0xbf, 0x5b, 0x6a, 0xc3, 0x9a, 0x25,
			0x44, 0xd8, 0xb5, 0x02, 0x61, 0x9f, 0x43, 0xea,
			0x30, 0xdd, 0xc8, 0xe4, 0xea, 0xfa, 0xd8, 0xbf,
			0x72, 0x56, 0x22, 0x03, 0x80, 0xe0, 0xae, 0x27,
			0xfe, 0xe4, 0x63, 0x04, 0xb2, 0x24, 0xcc, 0x8a,
			0x1e, 0x2b, 0x1c, 0xb2, 0xa4, 0xde, 0x6f, 0xb3,
			0xee, 0x54, 0x52, 0x79, 0x8d, 0xe7, 0x86, 0x53,
			0xe0, 0x8b, 0x01, 0xec, 0x38, 0x5f, 0x36, 0x7c,
			0x39, 0x82, 0x96, 0x3f, 0x84, 0x28, 0x57, 0x27,
			0x93, 0xed, 0x74, 0xce, 0xe3, 0x69, 0xf5, 0xae,
		},
		/* S4 : 256 byte */
		{
			0x44, 0x4f, 0x7e, 0xfb, 0xfe, 0xf5, 0x86, 0xfa,
			0xd4, 0x31, 0xe1, 0x7f, 0xea, 0x1a, 0x2d, 0x59,
			0xf1, 0x9b, 0x3d, 0x61, 0x9b, 0xb6, 0xfa, 0x36,
			0x64, 0x30, 0x18, 0x33, 0xa4, 0xdb, 0x12, 0x43,
			0x45, 0x9e, 0x31, 0xaa, 0x6a, 0x70, 0x3b, 0x22,
			0x57, 0x2f, 0x09, 0x12, 0x75, 0x4e, 0x56, 0xf7,
			0x23, 0x1a, 0x55, 0xac, 0x7a, 0xbc, 0xa5, 0x14,
			0xc7, 0x9d, 0x9f, 0xb3, 0x56, 0x42, 0x14, 0xb4,
			0xaf, 0x83, 0x5d, 0x7d, 0x1e, 0xaf, 0x2b, 0x58,
			0xce, 0xb6, 0xa3, 0x44, 0xf1, 0xc3, 0x68, 0x90,
			0xf5, 0xe8, 0x3b, 0x50, 0x18, 0x8c, 0x01, 0x47,
			0xd6, 0xd1, 0x15, 0x6d, 0xa2, 0x89, 0xcc, 0xf4,
			0xbd, 0xb0, 0xb9, 0xa6, 0x6f, 0x1e, 0x4a, 0x1f,
			0x26, 0x43, 0x59, 0x1d, 0x5f, 0xfb, 0x53, 0x70,
			0x2c, 0xf7, 0x0d, 0xdf, 0x35, 0x15, 0x92, 0x57,
			0x54, 0x88, 0xf1, 0x92, 0x90, 0x10, 0xac, 0xa3,
			0x77, 0x14, 0xb2, 0x34, 0xee, 0xb5, 0xb9, 0x52,
			0xb9, 0x32, 0x3a, 0xe2, 0x65, 0x33, 0xe9, 0xec,
			0xd5, 0x16, 0xdf, 0x26, 0x39, 0x2d, 0x12, 0x54,
			0x22, 0x8b, 0xd9, 0xca, 0x21, 0xa3, 0x69, 0xbb,
			0x6a, 0xb0, 0xa3, 0x3d, 0x5e, 0xb4, 0x4c, 0xee,
			0x92, 0xb0, 0xea, 0x74, 0x71, 0xff, 0xe5, 0xfa,
			0x43, 0xc2, 0x1d, 0xe2, 0xa8, 0x97, 0x5d, 0x4c,
			0x5c, 0x8e, 0x18, 0x5f, 0xcb, 0x7a, 0xab, 0x33,
			0xd8, 0x8a, 0x83, 0x65, 0xdd, 0xf0, 0x11, 0x9c,
			0x10, 0x88, 0x03, 0xc5, 0x62, 0x88, 0x64, 0x3a,
			0x05, 0x6e, 0x78, 0x1a, 0xbd, 0x4a, 0x02, 0x42,
			0xa9, 0x2e, 0x25, 0x29, 0xd4, 0x05, 0xef, 0xcf,
			0xd4, 0x24, 0x86, 0x62, 0xcf, 0xbb, 0x33, 0x2d,
			0x6e, 0x6f, 0xad, 0x6a, 0xce, 0xb9, 0x0b, 0x5b,
			0x58, 0xa5, 0x54, 0x1a, 0xbe, 0x07, 0xbe, 0xf2,
			0x5d, 0x9d, 0x89, 0x21, 0x5e, 0x39, 0x84, 0x26,
		},
	},
	/* SHA224 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0xbf, 0x5f, 0xf1, 0x96, 0x8a, 0x39, 0xf8, 0x09,
			0xde, 0x73, 0xe6, 0xa8, 0x01, 0x4f, 0xc6, 0xe8,
			0xdf, 0x15, 0x93, 0x67, 0xf4, 0x63, 0x40, 0xda,
			0x6c, 0xc5, 0xfb, 0x46, 0x89, 0x85, 0xb3, 0x74,
			0x46, 0xc5, 0xd8, 0x9f, 0x3a, 0xca, 0x62, 0x6f,
			0xbe, 0x9b, 0x14, 0x2b, 0x52, 0xcb, 0x02, 0x2a,
			0x3d, 0x93, 0x51, 0x8a, 0x74, 0x24, 0x3e, 0x25,
			0xbd, 0x3a, 0x61, 0xc1, 0x14, 0xf5, 0x33, 0x87,
			0x4e, 0xe5, 0xcf, 0xb7, 0xfc, 0x63, 0xf5, 0x99,
			0x92, 0x28, 0x54, 0xb7, 0xc9, 0x18, 0x09, 0x49,
			0x41, 0x5f, 0x63, 0xf1, 0x6b, 0xbf, 0xe9, 0xa8,
			0xa6, 0x28, 0x9e, 0xf8, 0xa8, 0x8a, 0x83, 0x6d,
			0x20, 0xe7, 0x5e, 0x46, 0x99, 0xac, 0xba, 0x6f,
			0xa2, 0x41, 0x2f, 0xb4, 0x2c, 0xdf, 0xe3, 0x2f,
			0x33, 0xa2, 0x51, 0x02, 0xa1, 0xdf, 0x49, 0x4c,
			0x6f, 0xb7, 0x38, 0x55, 0x0d, 0xec, 0xaa, 0x0c,
		},
		/* S5 : 256 byte */
		{
			0x01, 0x7e, 0x05, 0x3d, 0x1e, 0xf8, 0x5c, 0x43,
			0x19, 0x3a, 0x00, 0x09, 0xa9, 0x03, 0x95, 0x2a,
			0xaf, 0x40, 0x0f, 0xbc, 0xfe, 0xe9, 0xc0, 0x28,
			0x97, 0x57, 0x77, 0xab, 0x54, 0x0d, 0x2d, 0x22,
			0xab, 0x5c, 0x25, 0xf4, 0xcf, 0x1d, 0x37, 0x94,
			0xaf, 0xac, 0x66, 0x97, 0xe1, 0xf2, 0x43, 0x82,
			0x90, 0x52, 0xa8, 0x4e, 0x28, 0x43, 0xcc, 0x0e,
			0x25, 0x4d, 0xba, 0xc1, 0x02, 0x15, 0x72, 0x99,
			0x9f, 0x2d, 0xca, 0xfa, 0xb5, 0x8b, 0x9d, 0xfe,
			0xf2, 0xfc, 0xaf, 0x70, 0x1e, 0x43, 0x1b, 0xdc,
			0xd1, 0x6d, 0xbe, 0xf1, 0x10, 0x09, 0x5b, 0xcf,
			0xba, 0x50, 0x10, 0x59, 0xd7, 0x99, 0x4d, 0xad,
			0x5b, 0x0b, 0x54, 0xd0, 0x81, 0x2a, 0x43, 0x80,
			0xa1, 0xf0, 0xba, 0x8e, 0xc2, 0xbc, 0xba, 0x76,
			0x8b, 0xf5, 0xb5, 0x44, 0x69, 0x56, 0x26, 0xa5,
			0xf3, 0x95, 0xe7, 0x84, 0xd4, 0xb2, 0x96, 0x2f,
			0xb7, 0x53, 0x38, 0x18, 0xde, 0x1d, 0x6e, 0xc6,
			0x86, 0xed, 0xc9, 0xf6, 0x68, 0x68, 0xad, 0x03,
			0xee, 0x64, 0x36, 0x1a, 0x6c, 0xb9, 0x1f, 0xd8,
			0xef, 0x53, 0x6c, 0xa6, 0x45, 0x4d, 0x16, 0xc5,
			0x37, 0xc0, 0x7a, 0xa4, 0x29, 0x23, 0xe6, 0x20,
			0x57, 0xdf, 0x9d, 0xd9, 0xe7, 0xfa, 0x4a, 0xd0,
			0x38, 0x4f, 0x35, 0x72, 0x1f, 0x6e, 0xb3, 0xb8,
			0x16, 0xd3, 0x52, 0xa0, 0x95, 0xc6, 0x05, 0xd5,
			0xc1, 0x0e, 0x0a, 0x7a, 0x2e, 0x86, 0x40, 0xe2,
			0x73, 0x07, 0xcd, 0x44, 0xb9, 0xd7, 0x1a, 0xc5,
			0x0c, 0x00, 0x43, 0xca, 0xca, 0x28, 0xae, 0x8d,
			0x6f, 0x8f, 0xa5, 0xbb, 0x48, 0x31, 0x58, 0xa4,
			0xe4, 0x15, 0xef, 0x6c, 0xfa, 0xd4, 0x7f, 0x34,
			0xc0, 0x04, 0x2a, 0x2d, 0x58, 0x8a, 0xce, 0x0f,
			0x13, 0x71, 0xd9, 0x38, 0x65, 0x39, 0x7b, 0xd2,
			0x15, 0x16, 0xda, 0x2c, 0xc1, 0x5e, 0x90, 0x9c,
		},
	},
	/* SHA224 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0x2f, 0xf4, 0xfc, 0xd0, 0xbe, 0x26, 0x0b, 0xf4,
			0xa0, 0xd7, 0x31, 0x12, 0xd0, 0xe5, 0x64, 0x9c,
			0x0b, 0xef, 0x5b, 0xbc, 0xdf, 0x15, 0x42, 0x3a,
			0x05, 0xff, 0xb2, 0xa1, 0xf0, 0x21, 0xe0, 0x9d,
			0xa6, 0x3d, 0x15, 0xa8, 0xcf, 0x29, 0x5e, 0xe5,
			0x0b, 0xd2, 0x84, 0x4c, 0x89, 0x81, 0x3e, 0x08,
			0xd6, 0x5d, 0xa6, 0x1d, 0xf2, 0x32, 0xea, 0x4e,
			0xa9, 0x70, 0x44, 0x3e, 0x20, 0x77, 0x2c, 0xd5,
			0xaf, 0x11, 0xcc, 0xe5, 0xee, 0x40, 0xb4, 0x0e,
			0x13, 0x3b, 0xcf, 0xdf, 0x7b, 0xb3, 0x95, 0x3d,
			0x86, 0x5a, 0x83, 0x09, 0xa8, 0xa6, 0xc8, 0xfd,
			0xbd, 0xd2, 0x42, 0xd7, 0x9d, 0x27, 0xa8, 0xba,
			0xf1, 0x79, 0x09, 0xd1, 0x45, 0xf4, 0x75, 0x35,
			0x5e, 0x19, 0xfa, 0x11, 0xcd, 0x03, 0xd2, 0x04,
			0xc4, 0xef, 0xda, 0xc6, 0x29, 0xfb, 0x46, 0x0f,
			0xe9, 0x2e, 0x93, 0xb4, 0x8f, 0xb9, 0xbe, 0x13,
		},
		/* S6 : 256 byte */
		{
			0xab, 0xee, 0x5c, 0x86, 0x8f, 0x85, 0x0c, 0x17,
			0x79, 0x4f, 0x02, 0x1e, 0xe9, 0x70, 0x9c, 0xc2,
			0x30, 0x13, 0x20, 0xdd, 0x24, 0x6f, 0xb3, 0xea,
			0xdb, 0x78, 0x02, 0xa3, 0x00, 0xa9, 0x8a, 0x67,
			0x08, 0x3a, 0x2e, 0x4e, 0x25, 0x0d, 0xf1, 0x33,
			0x14, 0xc2, 0x54, 0x53, 0xb8, 0x98, 0x11, 0x08,
			0x01, 0xf7, 0xe7, 0xac, 0xb9, 0xb6, 0x94, 0x64,
			0x4e, 0x5c, 0x4a, 0x26, 0x23, 0xdf, 0xf1, 0x88,
			0x49, 0x13, 0xc0, 0x5e, 0x63, 0x6f, 0xe7, 0x7e,
			0xd5, 0x15, 0x5d, 0x95, 0x4e, 0xe3, 0x8f, 0x12,
			0x62, 0xc6, 0xc2, 0xe3, 0x8d, 0x11, 0x14, 0xcf,
			0x6c, 0xc5, 0x14, 0x3c, 0x72, 0x77, 0xc8, 0x64,
			0x9f, 0x5a, 0x42, 0x3f, 0x83, 0xdf, 0xd5, 0xf8,
			0x29, 0xd9, 0xdc, 0x74, 0xaa, 0x4b, 0x2f, 0xcd,
			0xc8, 0x96, 0x0c, 0xde, 0x5c, 0xe1, 0x46, 0xb2,
			0x89, 0x13, 0x60, 0x64, 0xb1, 0x3b, 0xd0, 0xd3,
			0x6a, 0x1e, 0x64, 0xa2, 0x61, 0xd6, 0x80, 0xfb,
			0x7e, 0x23, 0xd2, 0xae, 0x92, 0xef, 0xb7, 0x43,
			0xc3, 0xdb, 0x54, 0x60, 0x9e, 0xca, 0x7a, 0x1b,
			0xe0, 0xe4, 0x7e, 0x6f, 0x72, 0x4d, 0xc5, 0xcf,
			0x61, 0xcb, 0x2a, 0x36, 0x9c, 0x2b, 0xb1, 0x73,
			0xf2, 0xc6, 0xcf, 0xec, 0xb9, 0xa8, 0x87, 0xd5,
			0x83, 0xd2, 0x77, 0xb8, 0xe3, 0x0b, 0x24, 0xec,
			0x85, 0x49, 0xc4, 0xd5, 0x3b, 0xa3, 0x98, 0x86,
			0x42, 0xa6, 0x1f, 0x1f, 0x93, 0x9f, 0x0f, 0x38,
			0x98, 0x00, 0x5c, 0x5d, 0x13, 0xaa, 0xaa, 0x54,
			0xbc, 0xb8, 0xae, 0x83, 0xb7, 0x2b, 0x3c, 0xb6,
			0x44, 0xb9, 0x43, 0x9d, 0x1d, 0x2a, 0xcc, 0xc8,
			0x00, 0x27, 0x1d, 0x23, 0xe5, 0x2f, 0x98, 0x48,
			0x0d, 0x27, 0x0f, 0xad, 0x6a, 0xce, 0xd5, 0x12,
			0x25, 0x2e, 0xe9, 0x83, 0x32, 0xaf, 0x90, 0x35,
			0x63, 0xd9, 0x82, 0xd8, 0xcb, 0xde, 0xfb, 0x7d,
		},
	},
	/* SHA224 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0xb5, 0xdc, 0xa1, 0x53, 0x2d, 0xff, 0xda, 0x08,
			0x31, 0xcb, 0x2d, 0x21, 0xeb, 0xd1, 0xbd, 0xca,
			0x23, 0xc9, 0x31, 0x9c, 0x64, 0x27, 0xfd, 0xcc,
			0x5a, 0xef, 0xe3, 0xa2, 0x7f, 0xc9, 0xb9, 0x2d,
			0xf7, 0x58, 0x6c, 0x36, 0xb7, 0xc8, 0x45, 0x72,
			0xed, 0xa6, 0x6b, 0xfb, 0x9c, 0xf5, 0xaa, 0x01,
			0x87, 0x7e, 0x72, 0xbd, 0x51, 0x67, 0x23, 0xa7,
			0xe2, 0x07, 0x87, 0xe9, 0x0d, 0xf9, 0xa0, 0x13,
			0x6f, 0x6f, 0xa5, 0x10, 0x9a, 0xc9, 0x47, 0x59,
			0x73, 0x67, 0x38, 0x68, 0xd8, 0xbb, 0xee, 0x70,
			0x86, 0xa2, 0xa5, 0x4b, 0x3a, 0xf4, 0xa3, 0xb4,
			0x17, 0x59, 0xbf, 0xb6, 0x48, 0x5f, 0x24, 0x64,
			0xe6, 0xca, 0x53, 0xcb, 0x1c, 0x2c, 0x67, 0x25,
			0x89, 0xb5, 0x9d, 0x50, 0xe5, 0x4b, 0x13, 0x7e,
			0xe8, 0xdd, 0xd0, 0x2d, 0x67, 0xf5, 0x05, 0x5a,
			0xc1, 0x8d, 0x92, 0xf1, 0x79, 0x24, 0xcc, 0x89,
		},
		/* S7 : 256 byte */
		{
			0x9a, 0xe5, 0xb9, 0x63, 0x3f, 0x9a, 0xdc, 0x7f,
			0xf9, 0x23, 0xd8, 0x87, 0x57, 0x48, 0xbc, 0x62,
			0x20, 0xdd, 0x8f, 0x67, 0x81, 0xb3, 0xd4, 0x6d,
			0x60, 0x08, 0xae, 0x69, 0xfd, 0xa0, 0x72, 0xd2,
			0x05, 0xf8, 0x7a, 0x12, 0xd5, 0x4c, 0x3c, 0x7e,
			0xcc, 0x85, 0xb8, 0x8b, 0x6e, 0xf4, 0x77, 0x0e,
			0xeb, 0x4b, 0x71, 0xde, 0xbe, 0xff, 0x84, 0x01,
			0xe3, 0x29, 0xf6, 0xb3, 0xe8, 0xdc, 0x8a, 0x9a,
			0xf1, 0x3a, 0x53, 0x3b, 0x60, 0xb9, 0x62, 0x93,
			0x0b, 0xc0, 0xce, 0x3d, 0x65, 0xd0, 0xb5, 0xa2,
			0x76, 0xe8, 0x5a, 0x0c, 0x74, 0xf4, 0x59, 0xfb,
			0x07, 0x29, 0x92, 0x99, 0x1b, 0xa9, 0x68, 0x49,
			0x02, 0x34, 0x78, 0xab, 0x28, 0xd3, 0x81, 0xaa,
			0x67, 0xd2, 0x2c, 0x9c, 0x3b, 0x09, 0x2a, 0x02,
			0x3f, 0x06, 0xc9, 0x6e, 0x11, 0xfd, 0x2f, 0x1b,
			0x4d, 0x9d, 0xaf, 0x0f, 0x34, 0x49, 0xde, 0x17,
			0x97, 0x61, 0x2a, 0x81, 0x13, 0xd6, 0xe6, 0x26,
			0xcc, 0x3f, 0x99, 0x5e, 0x1c, 0x11, 0x0e, 0x65,
			0xd1, 0x7c, 0x63, 0x6c, 0x92, 0x92, 0x9f, 0x91,
			0x36, 0x39, 0xa9, 0x7c, 0xd0, 0x49, 0x15, 0x58,
			0x30, 0xdc, 0x0f, 0x76, 0x04, 0x91, 0x23, 0xbe,
			0x3d, 0x3d, 0x79, 0x15, 0x9f, 0xc2, 0xb4, 0x25,
			0x8e, 0x94, 0xb8, 0xbf, 0x80, 0x8d, 0x7c, 0x46,
			0xbe, 0xef, 0xe6, 0xdf, 0x0a, 0x83, 0x03, 0x7d,
			0x15, 0xa7, 0x2a, 0x58, 0x1d, 0x8a, 0xde, 0xdd,
			0x8f, 0x01, 0x3b, 0x38, 0xf5, 0x50, 0x2d, 0x73,
			0x6d, 0x1d, 0x2f, 0x04, 0xb0, 0xe5, 0xdc, 0x22,
			0xeb, 0x1a, 0x41, 0x4e, 0x52, 0xb1, 0xa9, 0xe8,
			0x73, 0x5e, 0x05, 0x92, 0x28, 0x8c, 0x9e, 0x5a,
			0x0a, 0x78, 0x53, 0x1e, 0x95, 0x97, 0x4a, 0x5d,
			0x48, 0x86, 0x0f, 0x8e, 0x5b, 0x04, 0xeb, 0xd3,
			0xeb, 0x56, 0xad, 0x12, 0xad, 0xc4, 0x6e, 0xc7,
		},
	},
	/* SHA224 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0x1e, 0x56, 0x3f, 0xc3, 0xad, 0x02, 0x7a, 0x9c,
			0xc6, 0x06, 0xbe, 0x19, 0xb2, 0x58, 0xbf, 0x70,
			0xdd, 0x8b, 0x52, 0x73, 0xe2, 0x96, 0x23, 0x6e,
			0xe8, 0xd7, 0xa6, 0x53, 0x31, 0x58, 0x50, 0x14,
			0xf0, 0x50, 0x06, 0x51, 0x5b, 0xed, 0xd6, 0x33,
			0x02, 0x50, 0xe5, 0x98, 0x5f, 0xda, 0xa8, 0x70,
			0xae, 0xa6, 0x57, 0x66, 0xff, 0x56, 0x9f, 0xc4,
			0x89, 0x13, 0x98, 0x90, 0x41, 0xcf, 0xf6, 0xfb,
			0xab, 0xcd, 0x83, 0xfd, 0xf0, 0x64, 0xcd, 0x39,
			0x32, 0x00, 0x1b, 0x26, 0x1c, 0x69, 0xa6, 0x70,
			0xbd, 0x48, 0x06, 0x9c, 0x96, 0xe7, 0xeb, 0xec,
			0xf1, 0x38, 0x0d, 0x82, 0x75, 0x19, 0x66, 0xc7,
			0xf8, 0xd6, 0x9e, 0x0e, 0x94, 0xef, 0xc7, 0x75,
			0xfd, 0x1c, 0x4a, 0x0c, 0x11, 0x8f, 0x21, 0x3a,
			0xb1, 0x79, 0x47, 0x5c, 0xd0, 0xcf, 0x6d, 0xae,
			0xc9, 0x4e, 0xef, 0x6f, 0xf6, 0xbd, 0x06, 0x40,
		},
		/* S8 : 256 byte */
		{
			0x80, 0xd3, 0xff, 0x1f, 0x74, 0xa8, 0x10, 0x95,
			0xd0, 0xba, 0xa2, 0xe9, 0xde, 0x24, 0x8c, 0x03,
			0x12, 0xca, 0x5a, 0x81, 0x7b, 0xc9, 0xf5, 0x15,
			0x6a, 0x29, 0x3d, 0x80, 0x89, 0x6a, 0xde, 0xc5,
			0x50, 0x7e, 0xe8, 0xf2, 0xdf, 0x41, 0x7a, 0xfe,
			0x87, 0x79, 0x66, 0x8e, 0x25, 0xb4, 0x6f, 0x49,
			0xe4, 0x35, 0x7a, 0x71, 0x70, 0x53, 0x1e, 0xd3,
			0x07, 0x61, 0x10, 0x3d, 0xbb, 0x99, 0x41, 0x35,
			0xb5, 0x10, 0xd9, 0x1d, 0xb9, 0xfe, 0x1f, 0x12,
			0x68, 0xf4, 0x37, 0xe0, 0xf3, 0xa7, 0xa4, 0xba,
			0x6a, 0x4d, 0x0b, 0x9e, 0xb7, 0x0d, 0xfc, 0x09,
			0xfe, 0xd4, 0xb4, 0x4b, 0x35, 0x60, 0x85, 0x01,
			0xc2, 0xdf, 0xd7, 0xa2, 0x30, 0xa2, 0x8d, 0xad,
			0x14, 0x92, 0x6d, 0xa4, 0x60, 0x0b, 0xa7, 0x85,
			0xe4, 0x96, 0x21, 0x2e, 0x57, 0x73, 0x8d, 0xd5,
			0x75, 0xb4, 0x0c, 0x23, 0x34, 0x7b, 0x16, 0x35,
			0xec, 0xdf, 0x2b, 0x91, 0x94, 0xd9, 0x6b, 0x14,
			0x50, 0xa6, 0x87, 0x6a, 0xa7, 0x6d, 0x04, 0xaa,
			0x59, 0x47, 0xcc, 0xe7, 0x1d, 0x85, 0x12, 0x1e,
			0x0b, 0xf5, 0x78, 0xe8, 0x1c, 0xf7, 0x8c, 0x6a,
			0x04, 0x7e, 0x30, 0xfc, 0x1d, 0x87, 0xcf, 0xd3,
			0x01, 0x9d, 0xe4, 0xbb, 0x48, 0x29, 0x4c, 0x25,
			0x86, 0x0b, 0x45, 0x03, 0x55, 0xbc, 0x26, 0x62,
			0xaa, 0x36, 0xd6, 0xe3, 0x3f, 0x00, 0xad, 0x79,
			0x25, 0x7d, 0x2d, 0x8b, 0x91, 0xf7, 0x3f, 0x27,
			0xc3, 0x2a, 0x9a, 0xfc, 0xb1, 0xe1, 0xf0, 0x15,
			0xf7, 0x7c, 0xb6, 0xb0, 0xdf, 0x51, 0xfb, 0x39,
			0xee, 0x1b, 0xd7, 0x6a, 0xc4, 0x2c, 0x20, 0x79,
			0x1d, 0x79, 0xcf, 0x3f, 0x36, 0x3f, 0xb3, 0x24,
			0xdb, 0x30, 0xee, 0x82, 0xbc, 0xc1, 0xdf, 0x1a,
			0x95, 0x64, 0x33, 0x0c, 0x12, 0xa5, 0x49, 0x65,
			0x9b, 0xd3, 0x01, 0x00, 0x01, 0x57, 0x31, 0x33,
		},
	},
	/* SHA224 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0x90, 0x0a, 0xe7, 0xe2, 0xe7, 0xe5, 0xf6, 0x15,
			0x75, 0x0c, 0x4e, 0xe4, 0xc1, 0x3c, 0xca, 0x8f,
			0x9f, 0x45, 0x07, 0x14, 0xa6, 0xb2, 0x73, 0xf2,
			0xe4, 0xac, 0xa6, 0x32, 0xd1, 0x1c, 0xf6, 0xa8,
			0x82, 0x10, 0x45, 0x77, 0x1f, 0x60, 0x1e, 0xd3,
			0x97, 0x91, 0x01, 0x0b, 0x92, 0xf9, 0xfa, 0xc6,
			0xa8, 0x24, 0x78, 0x8c, 0xd0, 0x77, 0x5d, 0x89,
			0x1b, 0x13, 0x52, 0x8e, 0xa2, 0xfd, 0x5d, 0x59,
			0xbc, 0x7b, 0xb5, 0x16, 0x75, 0xc1, 0xd5, 0x26,
			0x3c, 0xcc, 0xcf, 0x1e, 0xdc, 0x8f, 0xe3, 0x13,
			0xae, 0x4d, 0x50, 0x15, 0x0c, 0x46, 0x6a, 0xf9,
			0x08, 0x95, 0xed, 0x5c, 0x5e, 0x59, 0x91, 0xe4,
			0xa8, 0x13, 0xde, 0xc9, 0xd1, 0x4f, 0x42, 0x94,
			0xcc, 0x87, 0x61, 0x27, 0x86, 0x44, 0xac, 0xfe,
			0x19, 0x86, 0x35, 0xb4, 0x42, 0x66, 0xc1, 0xc9,
			0x15, 0xfa, 0x1f, 0xa2, 0xef, 0x79, 0xb9, 0xd1,
		},
		/* S9 : 256 byte */
		{
			0x39, 0xc6, 0x48, 0x91, 0xd9, 0xac, 0x47, 0x41,
			0xa5, 0x7d, 0xd8, 0xae, 0xc7, 0xf7, 0x24, 0x36,
			0x13, 0xd1, 0x55, 0xdf, 0x44, 0x92, 0x81, 0x4b,
			0x40, 0xce, 0xab, 0xee, 0x79, 0xea, 0xdb, 0x8d,
			0x8b, 0xc5, 0xfa, 0x61, 0x1b, 0xde, 0xbe, 0x0e,
			0x0d, 0x97, 0x14, 0xc4, 0x3d, 0x6d, 0x29, 0xef,
			0x30, 0x9f, 0x78, 0x2b, 0xc8, 0xe6, 0x8a, 0x4d,
			0x31, 0x7c, 0xe1, 0xec, 0xe4, 0x68, 0x55, 0x23,
			0x05, 0xa7, 0x3d, 0xb9, 0xd0, 0xd2, 0x89, 0x1e,
			0x28, 0x04, 0xf4, 0x20, 0x1b, 0x1b, 0xf8, 0xa3,
			0x24, 0x6f, 0xa0, 0x82, 0xad, 0xde, 0x1f, 0xc9,
			0xb3, 0xd2, 0x99, 0xf8, 0x8c, 0xb9, 0x3b, 0x7b,
			0x47, 0xfe, 0x9f, 0x73, 0x13, 0x70, 0x96, 0xc2,
			0xb8, 0xc5, 0x9e, 0xc0, 0x61, 0x2a, 0x08, 0x53,
			0x63, 0xc0, 0x4c, 0xc3, 0x74, 0x76, 0x9a, 0x96,
			0x4f, 0xea, 0xf1, 0xf8, 0xe4, 0x91, 0x38, 0x1e,
			0x16, 0xd7, 0xae, 0x2a, 0x0c, 0x67, 0x2e, 0x69,
			0xa3, 0x66, 0x73, 0x10, 0xfe, 0xed, 0x01, 0x21,
			0x56, 0xdc, 0xa6, 0x30, 0xa6, 0x8d, 0x33, 0x9e,
			0xc8, 0x04, 0x96, 0xc6, 0xb5, 0x94, 0xfe, 0xd1,
			0x70, 0x91, 0xd3, 0xa1, 0xc6, 0xac, 0x3e, 0x4d,
			0xa1, 0x41, 0x9b, 0x05, 0xd5, 0x89, 0xcb, 0x32,
			0x46, 0x82, 0x88, 0xf7, 0xdf, 0x4d, 0xaa, 0xce,
			0xff, 0x5a, 0x39, 0xbc, 0xf2, 0x97, 0xdc, 0x50,
			0x8c, 0xe9, 0x54, 0x9f, 0x60, 0x2e, 0x97, 0x3e,
			0xdb, 0xc2, 0xaa, 0x44, 0x33, 0x2e, 0xc3, 0x66,
			0x1b, 0x19, 0xc8, 0xc5, 0x8c, 0x56, 0x16, 0x92,
			0x4b, 0xeb, 0x89, 0x2f, 0x77, 0xb5, 0xe2, 0x00,
			0xd6, 0xfb, 0x3f, 0xc7, 0x59, 0x26, 0x3a, 0x74,
			0x9d, 0x15, 0x7e, 0xff, 0x9f, 0x73, 0x67, 0x98,
			0xd2, 0x81, 0xb2, 0x5b, 0x71, 0xfb, 0x47, 0x0b,
			0xdb, 0x70, 0x0f, 0x21, 0x1f, 0x84, 0x1d, 0xb7,
		},
	},
};

/* 10 elements */
static mod2048_t SHA256[] = {
	/* SHA256 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0x5a, 0xf2, 0x83, 0xb1, 0xb7, 0x6a, 0xb2, 0xa6,
			0x95, 0xd7, 0x94, 0xc2, 0x3b, 0x35, 0xca, 0x73,
			0x71, 0xfc, 0x77, 0x9e, 0x92, 0xeb, 0xf5, 0x89,
			0xe3, 0x04, 0xc7, 0xf9, 0x23, 0xd8, 0xcf, 0x97,
			0x63, 0x04, 0xc1, 0x98, 0x18, 0xfc, 0xd8, 0x9d,
			0x6f, 0x07, 0xc8, 0xd8, 0xe0, 0x8b, 0xf3, 0x71,
			0x06, 0x8b, 0xdf, 0x28, 0xae, 0x6e, 0xe8, 0x3b,
			0x2e, 0x02, 0x32, 0x8a, 0xf8, 0xc0, 0xe2, 0xf9,
			0x6e, 0x52, 0x8e, 0x16, 0xf8, 0x52, 0xf1, 0xfc,
			0x54, 0x55, 0xe4, 0x77, 0x2e, 0x28, 0x8a, 0x68,
			0xf1, 0x59, 0xca, 0x6b, 0xdc, 0xf9, 0x02, 0xb8,
			0x58, 0xa1, 0xf9, 0x47, 0x89, 0xb3, 0x16, 0x38,
			0x23, 0xe2, 0xd0, 0x71, 0x7f, 0xf5, 0x66, 0x89,
			0xee, 0xc7, 0xd0, 0xe5, 0x4d, 0x93, 0xf5, 0x20,
			0xd9, 0x6e, 0x1e, 0xb0, 0x45, 0x15, 0xab, 0xc7,
			0x0a, 0xe9, 0x05, 0x78, 0xff, 0x38, 0xd3, 0x1b,
		},
		/* S0 : 256 byte */
		{
			0x6b, 0x8b, 0xe9, 0x7d, 0x9e, 0x51, 0x8a, 0x2e,
			0xde, 0x74, 0x6f, 0xf4, 0xa7, 0xd9, 0x1a, 0x84,
			0xa1, 0xfc, 0x66, 0x5b, 0x52, 0xf1, 0x54, 0xa9,
			0x27, 0x65, 0x0d, 0xb6, 0xe7, 0x34, 0x8c, 0x69,
			0xf8, 0xc8, 0x88, 0x1f, 0x7b, 0xcf, 0x9b, 0x1a,
			0x6d, 0x33, 0x66, 0xee, 0xd3, 0x0c, 0x3a, 0xed,
			0x4e, 0x93, 0xc2, 0x03, 0xc4, 0x3f, 0x55, 0x28,
			0xa4, 0x5d, 0xe7, 0x91, 0x89, 0x57, 0x47, 0xad,
			0xe9, 0xc5, 0xfa, 0x5e, 0xee, 0x81, 0x42, 0x7e,
			0xde, 0xe0, 0x20, 0x82, 0x14, 0x7a, 0xa3, 0x11,
			0x71, 0x2a, 0x6a, 0xd5, 0xfb, 0x17, 0x32, 0xe9,
			0x3b, 0x3d, 0x6c, 0xd2, 0x3f, 0xfd, 0x46, 0xa0,
			0xb3, 0xca, 0xf6, 0x2a, 0x8b, 0x69, 0x95, 0x7c,
			0xc6, 0x8a, 0xe3, 0x9f, 0x99, 0x93, 0xc1, 0xa7,
			0x79, 0x59, 0x9c, 0xdd, 0xa9, 0x49, 0xbd, 0xaa,
			0xba, 0xbb, 0x77, 0xf2, 0x48, 0xfc, 0xfe, 0xaa,
			0x44, 0x05, 0x9b, 0xe5, 0x45, 0x9f, 0xb9, 0xb8,
			0x99, 0x27, 0x8e, 0x92, 0x95, 0x28, 0xee, 0x13,
			0x0f, 0xac, 0xd5, 0x33, 0x72, 0xec, 0xbc, 0x42,
			0xf3, 0xe8, 0xde, 0x29, 0x98, 0x42, 0x58, 0x60,
			0x40, 0x64, 0x40, 0xf2, 0x48, 0xd8, 0x17, 0x43,
			0x2d, 0xe6, 0x87, 0x11, 0x2e, 0x50, 0x4d, 0x73,
			0x40, 0x28, 0xe6, 0xc5, 0x62, 0x0f, 0xa2, 0x82,
			0xca, 0x07, 0x64, 0x70, 0x06, 0xcf, 0x0a, 0x2f,
			0xf8, 0x3e, 0x19, 0xa9, 0x16, 0x55, 0x4c, 0xc6,
			0x18, 0x10, 0xc2, 0xe8, 0x55, 0x30, 0x5d, 0xb4,
			0xe5, 0xcf, 0x89, 0x3a, 0x6a, 0x96, 0x76, 0x73,
			0x65, 0x79, 0x45, 0x56, 0xff, 0x03, 0x33, 0x59,
			0x08, 0x4d, 0x7e, 0x38, 0xa8, 0x45, 0x6e, 0x68,
			0xe2, 0x11, 0x55, 0xb7, 0x61, 0x51, 0x31, 0x4a,
			0x29, 0x87, 0x5f, 0xee, 0xe0, 0x95, 0x57, 0x16,
			0x1c, 0xbc, 0x65, 0x45, 0x41, 0xe8, 0x9e, 0x42,
		},
	},
	/* SHA256 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0xc4, 0x30, 0x11, 0xf3, 0xee, 0x88, 0xc9, 0xc9,
			0xad, 0xca, 0xc8, 0xbf, 0x37, 0x22, 0x1a, 0xfa,
			0x31, 0x76, 0x9d, 0x34, 0x7d, 0xec, 0x70, 0x5e,
			0x53, 0xac, 0xa9, 0x89, 0x93, 0xe7, 0x46, 0x06,
			0x59, 0x18, 0x67, 0xcc, 0xd2, 0x89, 0xba, 0x1b,
			0x4f, 0x19, 0x36, 0x5f, 0x98, 0x3e, 0x0c, 0x57,
			0x83, 0x46, 0xda, 0x76, 0xc5, 0xe2, 0x22, 0x8a,
			0x07, 0xe4, 0xfc, 0x9b, 0x3d, 0x48, 0x07, 0x16,
			0x33, 0x71, 0xa5, 0x2b, 0x68, 0xb6, 0x68, 0x73,
			0x20, 0x1d, 0xc7, 0xd6, 0xb5, 0x66, 0x16, 0xac,
			0x2e, 0x4c, 0xb5, 0x22, 0x12, 0x07, 0x87, 0xdf,
			0x7f, 0x15, 0xa5, 0xe8, 0x76, 0x3a, 0x54, 0xc1,
			0x79, 0xc6, 0x35, 0xd6, 0x58, 0x16, 0xbc, 0x19,
			0x48, 0x5d, 0xe3, 0xeb, 0x35, 0xa5, 0x20, 0x40,
			0x59, 0x10, 0x94, 0xfe, 0x0e, 0x64, 0x85, 0xa7,
			0xe0, 0xc6, 0x0e, 0x38, 0xe7, 0xc6, 0x15, 0x51,
		},
		/* S1 : 256 byte */
		{
			0xaa, 0x3a, 0x4e, 0x12, 0xeb, 0x87, 0x59, 0x6c,
			0x71, 0x1c, 0x9a, 0x22, 0xbc, 0xab, 0xcb, 0x9d,
			0xad, 0xff, 0xca, 0xbc, 0xec, 0xbd, 0x16, 0x22,
			0x88, 0x89, 0xe9, 0xbb, 0x45, 0x7d, 0x5d, 0x22,
			0x57, 0x1a, 0x72, 0xf0, 0x34, 0xbe, 0x47, 0x83,
			0x38, 0x4f, 0x43, 0xce, 0x6f, 0xff, 0xc6, 0x05,
			0x34, 0xb8, 0x33, 0x1c, 0xdd, 0x5d, 0x7c, 0x77,
			0xf4, 0x91, 0x80, 0xbf, 0xd1, 0x94, 0xb5, 0xfd,
			0x43, 0xa5, 0x08, 0xc6, 0x6d, 0x78, 0x6c, 0x55,
			0x88, 0x76, 0x73, 0x58, 0x94, 0xe6, 0xa9, 0x30,
			0x09, 0x52, 0xde, 0x79, 0x2f, 0x74, 0x70, 0x45,
			0xe7, 0x4d, 0x87, 0xfd, 0x50, 0x98, 0x02, 0x30,
			0x70, 0x7a, 0x34, 0xa4, 0xdf, 0x01, 0x3c, 0xe0,
			0x50, 0xbb, 0xff, 0x0d, 0x6f, 0x57, 0x08, 0x85,
			0xc9, 0xc7, 0xbf, 0x8d, 0xc4, 0x99, 0x13, 0x2c,
			0xae, 0xe0, 0x71, 0xb4, 0x1d, 0x81, 0xff, 0x91,
			0xb8, 0xce, 0x21, 0xaa, 0x2f, 0x28, 0x2c, 0xbf,
			0x52, 0x38, 0x9f, 0x23, 0x9a, 0xfe, 0x14, 0x90,
			0x89, 0x0b, 0xe2, 0x1f, 0x9d, 0x80, 0x8b, 0x3d,
			0x70, 0xb9, 0x7e, 0xfd, 0x59, 0xc0, 0xb6, 0x0e,
			0x46, 0x60, 0x88, 0xbb, 0x42, 0x71, 0x4f, 0x21,
			0x2b, 0xc9, 0x0d, 0xb7, 0xe9, 0x42, 0xeb, 0xce,
			0xe6, 0x0e, 0x7b, 0x10, 0x7f, 0xff, 0x44, 0xfb,
			0x35, 0x64, 0xff, 0x07, 0xd6, 0xd0, 0x28, 0x50,
			0x21, 0x5f, 0xd3, 0x57, 0xd8, 0x97, 0xc4, 0xd3,
			0x2b, 0xef, 0x86, 0x61, 0x68, 0x9f, 0x2d, 0x84,
			0xff, 0x89, 0x76, 0x37, 0xfb, 0x6d, 0x55, 0x68,
			0xa7, 0x27, 0x0e, 0x78, 0x34, 0x26, 0xb7, 0x4b,
			0x70, 0x37, 0x49, 0x3e, 0x51, 0x55, 0xfd, 0x7c,
			0xb3, 0xdd, 0xdd, 0xfd, 0x36, 0xbd, 0x8a, 0x9c,
			0x87, 0x7d, 0x71, 0xd2, 0xa9, 0x66, 0x05, 0x7c,
			0x08, 0x26, 0x3d, 0x29, 0x39, 0xc8, 0x49, 0x87,
		},
	},
	/* SHA256 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x61, 0xd7, 0xb3, 0x15, 0x01, 0x31, 0x35, 0x1e,
			0x7b, 0x4c, 0x8e, 0x56, 0x45, 0xd3, 0x8b, 0xe9,
			0x33, 0x5b, 0x40, 0x28, 0x9a, 0xf3, 0x4c, 0xc6,
			0xb6, 0xfc, 0x5e, 0x48, 0x49, 0x3b, 0xf8, 0xb7,
			0x85, 0x2c, 0x73, 0x98, 0x2c, 0x99, 0x44, 0x1e,
			0xf6, 0x6c, 0x7d, 0x9d, 0x33, 0xc2, 0x97, 0x42,
			0xb1, 0x40, 0x6e, 0x02, 0xe0, 0xaa, 0x8d, 0xd0,
			0x34, 0xb1, 0xac, 0x13, 0xcb, 0x0d, 0x77, 0x57,
			0x50, 0xcc, 0x91, 0x42, 0x1f, 0xea, 0xd9, 0xca,
			0xa9, 0x21, 0xec, 0xa6, 0x1a, 0x02, 0xeb, 0x02,
			0x3a, 0x45, 0x7e, 0x77, 0x91, 0x5e, 0x18, 0x3a,
			0xcf, 0x51, 0x7d, 0x94, 0x6b, 0xc6, 0x82, 0x92,
			0x89, 0x60, 0x14, 0xfd, 0x21, 0x4b, 0x7c, 0x8c,
			0x5e, 0x14, 0xe1, 0x59, 0x44, 0xbe, 0x0f, 0x92,
			0x96, 0x12, 0x77, 0x71, 0xf7, 0x36, 0x76, 0x6e,
			0x4f, 0x81, 0xda, 0xb3, 0x70, 0x8e, 0xa2, 0xd0,
		},
		/* S2 : 256 byte */
		{
			0x84, 0xe9, 0x2a, 0x14, 0x5a, 0xe6, 0xbe, 0x1f,
			0xf9, 0x24, 0x2d, 0x9e, 0xd2, 0xd6, 0x8d, 0xe6,
			0x68, 0xe8, 0x02, 0x52, 0x4e, 0x8a, 0xc0, 0xa7,
			0x9d, 0xe6, 0x2f, 0xe7, 0x40, 0x48, 0xc3, 0x54,
			0x91, 0xfd, 0x2f, 0xfd, 0xb1, 0x85, 0x05, 0x7e,
			0x66, 0x6d, 0xbf, 0xaa, 0xc8, 0x4c, 0x34, 0xfd,
			0xe7, 0x89, 0x12, 0x63, 0xf8, 0xb2, 0xbc, 0x74,
			0x74, 0x62, 0x30, 0x32, 0x0f, 0x67, 0xa7, 0xbd,
			0x73, 0x19, 0xc9, 0xb9, 0xde, 0x41, 0x90, 0x54,
			0x70, 0x14, 0xe2, 0xd7, 0xa2, 0xa5, 0x06, 0x0d,
			0x62, 0x00, 0xaa, 0xdc, 0x3a, 0x44, 0xba, 0xc0,
			0x29, 0xff, 0x39, 0x92, 0xed, 0xd3, 0x0e, 0xc5,
			0x3a, 0xb0, 0xd9, 0x12, 0x3e, 0xaa, 0x6b, 0x14,
			0x73, 0x52, 0xa0, 0x73, 0xa9, 0x81, 0x61, 0xe6,
			0x4f, 0x39, 0x4b, 0xb9, 0x94, 0x92, 0xc6, 0x97,
			0x7e, 0x24, 0xf4, 0x45, 0xc7, 0x12, 0x5b, 0xfb,
			0x90, 0xf8, 0x7f, 0xaf, 0x26, 0x22, 0x72, 0x13,
			0x4a, 0xcb, 0x18, 0x82, 0x3a, 0x99, 0xa5, 0x22,
			0x8d, 0x14, 0x95, 0x46, 0x32, 0x97, 0xfd, 0x77,
			0x48, 0x77, 0xfb, 0x63, 0xd4, 0x91, 0x81, 0x06,
			0x34, 0x7e, 0x6f, 0x29, 0x31, 0x5e, 0x48, 0x36,
			0x3f, 0x39, 0xb3, 0x32, 0x99, 0xea, 0xa3, 0x2d,
			0x8d, 0xa7, 0x1b, 0x22, 0x9d, 0x8f, 0xfe, 0xe5,
			0xf6, 0x6f, 0x72, 0x2a, 0xd3, 0xaa, 0x41, 0x75,
			0xd3, 0xf8, 0x4e, 0xce, 0x9c, 0xc8, 0xec, 0xa8,
			0xd6, 0xf2, 0xf3, 0x56, 0xa8, 0x5c, 0x15, 0x24,
			0x89, 0x6c, 0x18, 0xf7, 0xb5, 0xc8, 0xf9, 0xbc,
			0xde, 0xf4, 0x5c, 0x49, 0x6d, 0x53, 0x91, 0x79,
			0x89, 0x1d, 0xdc, 0x76, 0xe5, 0x20, 0x8a, 0xd8,
			0x35, 0x3d, 0x48, 0xc6, 0x24, 0x05, 0x4f, 0x34,
			0x40, 0xee, 0xba, 0x44, 0x32, 0xa1, 0x06, 0x54,
			0xa1, 0x1e, 0xf5, 0x37, 0x83, 0xbd, 0x11, 0x6f,
		},
	},
	/* SHA256 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0xb6, 0x77, 0x1a, 0xb0, 0xe1, 0x28, 0xb4, 0x1b,
			0x32, 0xb8, 0xb0, 0x5e, 0x05, 0xad, 0xd2, 0x3c,
			0xe0, 0xfb, 0x87, 0x7b, 0x40, 0xbf, 0xcc, 0x3b,
			0x99, 0x2f, 0x4c, 0x86, 0x98, 0xd1, 0xc8, 0x28,
			0xab, 0xec, 0xbc, 0xc1, 0xc3, 0x3d, 0x40, 0x18,
			0x59, 0xea, 0x2c, 0xb2, 0xaf, 0xbc, 0x7f, 0xa4,
			0x58, 0x88, 0x02, 0xa5, 0xfa, 0xee, 0x28, 0x67,
			0x53, 0x46, 0x39, 0x28, 0x7a, 0xd8, 0xaf, 0x84,
			0x67, 0x4b, 0xe1, 0x8d, 0xb6, 0x61, 0xde, 0x1d,
			0xa8, 0xe1, 0x9c, 0x6b, 0x6b, 0xd4, 0x52, 0xdd,
			0x9b, 0xf3, 0x22, 0x1d, 0x08, 0x61, 0xfb, 0x6f,
			0xba, 0x96, 0xbe, 0x42, 0x32, 0x9b, 0x9f, 0x04,
			0xf3, 0x7d, 0xcf, 0x3b, 0x41, 0xfc, 0x58, 0xd2,
			0x29, 0x83, 0x48, 0xb0, 0xc1, 0x5d, 0x11, 0x90,
			0xb1, 0x25, 0x30, 0x0c, 0xf2, 0x7e, 0x0d, 0xfa,
			0xd6, 0x05, 0x22, 0xfc, 0x49, 0x84, 0x60, 0x53,
		},
		/* S3 : 256 byte */
		{
			0x62, 0x76, 0x92, 0x55, 0x68, 0x62, 0x6f, 0x0c,
			0xbe, 0x6f, 0x51, 0x50, 0xb0, 0x50, 0xe1, 0x70,
			0x25, 0x82, 0xf8, 0xda, 0xf9, 0x9a, 0x6f, 0x88,
			0x0e, 0xf7, 0x5c, 0xd9, 0x6c, 0x2d, 0x42, 0x08,
			0xfb, 0x6e, 0x91, 0xb0, 0x1b, 0xa6, 0xab, 0xa2,
			0xa8, 0x16, 0xb2, 0xd3, 0xcb, 0x97, 0x5d, 0xf8,
			0x50, 0xb1, 0xd2, 0x68, 0xc4, 0x66, 0x2d, 0xd1,
			0xea, 0x3a, 0x30, 0x0c, 0x1d, 0x71, 0x71, 0xc6,
			0x33, 0xdd, 0x2e, 0xfb, 0xac, 0x30, 0x00, 0xc5,
			0x6a, 0xb8, 0x0f, 0x98, 0x9d, 0xbc, 0x18, 0x24,
			0x3e, 0x63, 0x6b, 0xa5, 0xd4, 0xd2, 0x6a, 0x7d,
			0x3f, 0x19, 0x65, 0xad, 0x3c, 0xb0, 0xf1, 0xa8,
			0x51, 0x3f, 0x99, 0x80, 0x03, 0xf7, 0xb6, 0x7e,
			0x2a, 0xc5, 0xc7, 0x18, 0xcb, 0x68, 0x8b, 0x32,
			0x01, 0xd5, 0x6e, 0x68, 0xf0, 0xb9, 0xf8, 0x62,
			0x57, 0xb8, 0x47, 0x94, 0xcd, 0xff, 0xbc, 0x1f,
			0xe3, 0xea, 0x24, 0xb7, 0xbb, 0x6e, 0x9e, 0xf0,
			0x53, 0x9b, 0xd4, 0xfb, 0xc1, 0xaf, 0xb5, 0x5b,
			0xc1, 0xdc, 0xa3, 0x99, 0x96, 0xea, 0x8a, 0x63,
			0x76, 0x9f, 0x6e, 0x22, 0x57, 0x07, 0xf6, 0x90,
			0x47, 0x55, 0x5e, 0x1a, 0x4e, 0xf3, 0xc6, 0x39,
			0xc5, 0xf2, 0xa4, 0x97, 0xb8, 0x89, 0x42, 0x4a,
			0x90, 0x14, 0x86, 0x39, 0xbb, 0x64, 0xdf, 0x0a,
			0x06, 0xe0, 0xb7, 0xf0, 0xe8, 0xed, 0x46, 0x6a,
			0x97, 0x7b, 0xac, 0xa3, 0x2f, 0x48, 0x23, 0x37,
			0xb2, 0xab, 0xe3, 0x98, 0x3e, 0xae, 0xc3, 0xfe,
			0x10, 0x75, 0x01, 0x6e, 0x58, 0x67, 0x52, 0x17,
			0x60, 0xfd, 0x06, 0x07, 0xd7, 0x99, 0xf1, 0x76,
			0x6b, 0x3f, 0xf6, 0xe2, 0xae, 0x15, 0x5d, 0x69,
			0x25, 0x0f, 0x8b, 0xf0, 0x8c, 0x8e, 0xdc, 0xa0,
			0xb4, 0xf3, 0x1d, 0x0f, 0x83, 0x8c, 0xfd, 0x29,
			0x8c, 0xb7, 0x31, 0x2d, 0xf9, 0x3f, 0x09, 0x97,
		},
	},
	/* SHA256 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0x6a, 0x81, 0xcb, 0x6c, 0x7b, 0x26, 0x8f, 0x4b,
			0x9f, 0xb9, 0x17, 0x2a, 0xdb, 0xbb, 0x36, 0xa2,
			0x37, 0xa0, 0xdc, 0xf1, 0xc3, 0xc8, 0x3a, 0x95,
			0xdc, 0xb0, 0x27, 0x1a, 0xac, 0x6a, 0xc3, 0x30,
			0xf0, 0x4a, 0x5a, 0x00, 0xfe, 0xe3, 0x8b, 0xc0,
			0x06, 0x31, 0xa9, 0x85, 0x98, 0x18, 0x61, 0x59,
			0x66, 0x0d, 0x9d, 0x8e, 0x4c, 0x14, 0xa9, 0x52,
			0x8d, 0xea, 0x94, 0x83, 0x60, 0x83, 0xda, 0xc4,
			0xab, 0xb7, 0x3f, 0xd0, 0x0e, 0x38, 0xfe, 0x0e,
			0x23, 0xc7, 0x23, 0x66, 0x04, 0xa7, 0x36, 0x54,
			0x0e, 0x52, 0x19, 0x3a, 0xe5, 0x6c, 0x33, 0xfb,
			0xb8, 0xf5, 0xcf, 0xc5, 0xc7, 0xc2, 0xbe, 0x2e,
			0x22, 0x2e, 0x44, 0x83, 0xb3, 0x0d, 0x32, 0x5c,
			0x7e, 0xe1, 0x4f, 0x74, 0x28, 0x51, 0xfc, 0xb8,
			0xb6, 0xd6, 0x18, 0x9e, 0x98, 0xb8, 0x22, 0xb8,
			0xe6, 0x39, 0x9d, 0x89, 0xe9, 0x0f, 0xb9, 0x97,
		},
		/* S4 : 256 byte */
		{
			0xb6, 0x79, 0x91, 0x05, 0x0c, 0x08, 0x3e, 0x64,
			0x50, 0x97, 0xdb, 0x03, 0xff, 0xf3, 0x47, 0x58,
			0x86, 0x8b, 0xeb, 0x19, 0xe9, 0xc0, 0xc4, 0x84,
			0x75, 0xf0, 0xf9, 0x13, 0x36, 0x1e, 0x71, 0xd3,
			0xd6, 0xf2, 0x7a, 0x8c, 0x4f, 0x0b, 0x26, 0x9b,
			0x49, 0xe8, 0x53, 0x40, 0x39, 0xe5, 0x3a, 0xd3,
			0xba, 0xb9, 0xa3, 0xe6, 0x2a, 0xbe, 0x07, 0x8e,
			0xe7, 0x5e, 0x7f, 0xb5, 0x95, 0x90, 0x06, 0xfb,
			0xfb, 0x01, 0x4c, 0xa7, 0xb8, 0x1b, 0x3d, 0x5a,
			0xfe, 0x0e, 0xe5, 0xf6, 0xfc, 0x2d, 0xfb, 0xc4,
			0x50, 0xf2, 0x83, 0x95, 0x43, 0x00, 0x2f, 0x33,
			0xf4, 0xf3, 0x54, 0xf8, 0x27, 0x27, 0x8c, 0x76,
			0xc0, 0x41, 0x68, 0x6e, 0xea, 0x78, 0x86, 0xeb,
			0xb2, 0xa7, 0xaf, 0xa5, 0x99, 0x5c, 0x6c, 0xdd,
			0xb1, 0xc0, 0xb5, 0x80, 0x66, 0xdd, 0xb8, 0xdc,
			0x54, 0xa6, 0x92, 0x7c, 0x14, 0x6c, 0x3b, 0x2a,
			0x0f, 0xa7, 0xce, 0xf2, 0x89, 0x03, 0xc6, 0xc6,
			0x72, 0xbc, 0x20, 0xef, 0x68, 0xff, 0xbf, 0xab,
			0x24, 0x7e, 0xb6, 0x88, 0xab, 0x4b, 0xde, 0x71,
			0x06, 0xd9, 0xc5, 0x9d, 0x21, 0x53, 0x09, 0x6d,
			0xc9, 0xe5, 0x20, 0x72, 0x67, 0x03, 0x8d, 0x88,
			0xe2, 0x17, 0x4e, 0x76, 0xad, 0xc1, 0x50, 0x8a,
			0xe2, 0x4e, 0xb6, 0x02, 0x33, 0x2e, 0x53, 0xc0,
			0xc2, 0xe3, 0x31, 0x54, 0xa6, 0x6a, 0x97, 0xa0,
			0xf1, 0x2f, 0x66, 0xc6, 0x12, 0x58, 0xc7, 0xbf,
			0x6b, 0xbf, 0x3f, 0x1d, 0xcb, 0xe9, 0xca, 0xf2,
			0xfd, 0x30, 0xec, 0x68, 0xc0, 0xa9, 0xd0, 0x9f,
			0x4f, 0xd7, 0x76, 0x30, 0x4b, 0x54, 0x0e, 0x62,
			0xfc, 0x85, 0x12, 0xbe, 0xaa, 0xbc, 0x4b, 0xe2,
			0x10, 0x7a, 0x1e, 0xc1, 0x8e, 0x87, 0xf6, 0x1f,
			0x9d, 0xb2, 0x5e, 0x87, 0x1d, 0xc0, 0x69, 0x3c,
			0xef, 0x17, 0xc2, 0xa6, 0x87, 0xfc, 0x85, 0x4f,
		},
	},
	/* SHA256 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0x05, 0x6c, 0x1e, 0x46, 0x44, 0x59, 0x9e, 0x31,
			0x83, 0xdd, 0x8d, 0x2f, 0x64, 0xe4, 0xbb, 0x23,
			0x52, 0xff, 0x00, 0xd0, 0x12, 0xab, 0x76, 0x3f,
			0x9a, 0xd6, 0xe5, 0x60, 0x27, 0x9f, 0x7f, 0xf3,
			0x8a, 0x5e, 0xce, 0xa9, 0xc2, 0xe4, 0xea, 0x87,
			0xd0, 0x04, 0xef, 0x8c, 0xc7, 0x52, 0xae, 0x93,
			0x23, 0x2a, 0xa3, 0x7b, 0x5b, 0xf4, 0x28, 0x84,
			0xba, 0xa7, 0xe7, 0xfc, 0x6a, 0x8c, 0x95, 0x1c,
			0xd2, 0x45, 0xde, 0x2d, 0x22, 0x0d, 0x9b, 0xee,
			0x2b, 0x41, 0x4b, 0x3a, 0x75, 0x20, 0xc1, 0xe6,
			0x8b, 0xcf, 0x1a, 0xe9, 0x9a, 0x9f, 0xf2, 0xbf,
			0x3a, 0x93, 0xd8, 0x0f, 0x8c, 0x1d, 0xfe, 0x8b,
			0x85, 0x29, 0x35, 0x17, 0x89, 0x5c, 0x19, 0x2e,
			0x3c, 0x9e, 0x89, 0x82, 0x95, 0xd6, 0x5b, 0xe3,
			0x34, 0xf4, 0x4d, 0x62, 0xf5, 0x35, 0x3e, 0xb6,
			0xc5, 0xa2, 0x9e, 0xdf, 0xb4, 0xdb, 0x23, 0x09,
		},
		/* S5 : 256 byte */
		{
			0xae, 0x05, 0x20, 0x4e, 0x40, 0x9d, 0x72, 0x7e,
			0xb9, 0xe4, 0xdc, 0x24, 0xbe, 0x8f, 0x86, 0x33,
			0x28, 0xc2, 0x81, 0x3d, 0xa4, 0xfc, 0xef, 0x28,
			0x86, 0x6e, 0x21, 0xa5, 0xda, 0xb2, 0x1a, 0x48,
			0x53, 0x21, 0xb7, 0x35, 0x27, 0x4a, 0xf0, 0x6b,
			0xf1, 0x7e, 0x27, 0x15, 0x18, 0xe1, 0x11, 0x64,
			0xd7, 0x22, 0xab, 0x07, 0x35, 0x48, 0xf0, 0x2e,
			0x1b, 0x44, 0x19, 0x23, 0xdb, 0x6f, 0x1c, 0xee,
			0x65, 0xa0, 0x17, 0xed, 0xfb, 0xaf, 0x33, 0x61,
			0xc6, 0x7f, 0xbc, 0x2b, 0x39, 0xfe, 0x03, 0x8c,
			0xb5, 0xcb, 0x65, 0xa6, 0x40, 0xf9, 0x58, 0x87,
			0x38, 0x9c, 0xe8, 0xa5, 0xad, 0x2e, 0xc6, 0xe6,
			0x9d, 0x3d, 0x60, 0x35, 0x05, 0xb0, 0x25, 0xf6,
			0xd6, 0x33, 0x0c, 0x8b, 0x64, 0x88, 0x02, 0xca,
			0xf7, 0xe6, 0xfa, 0x3f, 0xe7, 0xb3, 0x81, 0x41,
			0x65, 0x99, 0x86, 0xcb, 0x89, 0xe6, 0x23, 0x2f,
			0x10, 0x62, 0x22, 0x56, 0x4d, 0x5e, 0x51, 0x95,
			0xed, 0xa6, 0xa2, 0x5f, 0x99, 0x06, 0x85, 0x72,
			0xc2, 0xfa, 0xfe, 0x97, 0xf1, 0x47, 0xf7, 0xf2,
			0xf4, 0x11, 0x9f, 0x21, 0x38, 0x5a, 0xf1, 0xfc,
			0xed, 0x97, 0xf7, 0x86, 0x32, 0xd8, 0xbf, 0x4f,
			0xd9, 0xa9, 0x05, 0x4d, 0x8b, 0x9a, 0xa2, 0xa9,
			0xf4, 0xde, 0xd5, 0x87, 0x84, 0x7a, 0x91, 0xd4,
			0x2c, 0x63, 0x91, 0x12, 0x5f, 0x10, 0x3a, 0xe2,
			0x88, 0x54, 0x7e, 0x84, 0x89, 0x69, 0x3a, 0xe8,
			0x68, 0x6b, 0x84, 0x89, 0x1b, 0x77, 0x2b, 0x10,
			0xc4, 0x79, 0x68, 0x83, 0xf6, 0x6c, 0xd4, 0x59,
			0xa8, 0xc1, 0xa6, 0xa4, 0x18, 0x7b, 0xd6, 0xb3,
			0x87, 0xd3, 0x49, 0xe9, 0x2d, 0x7b, 0x60, 0x49,
			0x53, 0x72, 0x7c, 0x9e, 0x9f, 0xdc, 0x44, 0x9e,
			0x73, 0x45, 0xe7, 0xca, 0x6b, 0x33, 0x9e, 0x26,
			0xb0, 0x86, 0xf5, 0x54, 0x88, 0x98, 0xcb, 0xe9,
		},
	},
	/* SHA256 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0xce, 0xc5, 0xc9, 0xb6, 0xf8, 0x44, 0x97, 0xac,
			0x32, 0x7f, 0x68, 0xef, 0x88, 0x66, 0x41, 0xfe,
			0xc9, 0x95, 0x17, 0x8b, 0x30, 0x71, 0x92, 0x30,
			0x43, 0x74, 0x11, 0x5e, 0xfc, 0xc5, 0xee, 0x96,
			0x27, 0x0c, 0x03, 0xdb, 0x0b, 0x84, 0x6d, 0x67,
			0x4c, 0x52, 0x8f, 0x9d, 0x10, 0x15, 0x5a, 0x3f,
			0x61, 0xbe, 0xcc, 0xe1, 0xd3, 0xa2, 0xb7, 0x9d,
			0x66, 0xcd, 0xc4, 0x09, 0xad, 0x99, 0xb7, 0x66,
			0x30, 0x80, 0xf5, 0x1a, 0x10, 0x2f, 0x43, 0x61,
			0xe9, 0xdb, 0xd0, 0x3f, 0xfc, 0xd8, 0x76, 0xb9,
			0x8e, 0x68, 0x3d, 0x44, 0x8b, 0xd1, 0x21, 0x7e,
			0x6f, 0xb2, 0x15, 0x1c, 0x66, 0x96, 0x47, 0x23,
			0xb2, 0xca, 0xa6, 0x5c, 0x4e, 0x6c, 0xa2, 0x01,
			0xd1, 0xc5, 0x32, 0xbd, 0x94, 0xd9, 0x1c, 0xd4,
			0x17, 0x3b, 0x71, 0x9d, 0xa1, 0x26, 0x56, 0x39,
			0x27, 0xca, 0x0a, 0x7f, 0x6f, 0xe4, 0x25, 0x36,
		},
		/* S6 : 256 byte */
		{
			0xc4, 0x8a, 0x8e, 0x01, 0xd4, 0xbb, 0xfe, 0x0f,
			0x2f, 0x05, 0x65, 0x93, 0x37, 0xea, 0x71, 0xd2,
			0x1f, 0x38, 0xd7, 0xf7, 0xa1, 0x0b, 0x00, 0xb0,
			0x6e, 0x1f, 0x89, 0x9e, 0xaf, 0x40, 0xa8, 0xe9,
			0x7e, 0xad, 0x64, 0xbc, 0xa3, 0x7f, 0x13, 0xa5,
			0x5e, 0xf1, 0xcf, 0x3f, 0xb5, 0x2c, 0xee, 0x27,
			0x9c, 0xdc, 0xb0, 0x96, 0x08, 0x5a, 0x46, 0x7a,
			0xfa, 0x97, 0xb0, 0x3d, 0x78, 0xd6, 0x07, 0x6e,
			0x47, 0x2b, 0x12, 0xd6, 0xbe, 0x96, 0x47, 0xce,
			0xc3, 0x2d, 0x8d, 0x91, 0xa2, 0x62, 0x47, 0x69,
			0x37, 0x71, 0x68, 0x74, 0x60, 0xba, 0x52, 0x69,
			0xde, 0x18, 0xe1, 0xed, 0xef, 0x60, 0x22, 0x53,
			0x3a, 0x95, 0x79, 0xf9, 0x1d, 0x58, 0x4f, 0x9e,
			0x0c, 0xee, 0x11, 0x00, 0xc4, 0x47, 0xb7, 0x75,
			0x76, 0xb1, 0xb4, 0xee, 0x16, 0x3e, 0xd4, 0x70,
			0x01, 0x47, 0xa9, 0xaa, 0x61, 0xbd, 0xc4, 0xe2,
			0x31, 0x6d, 0x2d, 0x81, 0x8c, 0x10, 0x28, 0xed,
			0x1c, 0x3e, 0x37, 0x2c, 0x9f, 0x6a, 0x17, 0x45,
			0x57, 0x24, 0x44, 0x63, 0x72, 0x48, 0x09, 0x1b,
			0x83, 0xf7, 0xb5, 0x39, 0xf9, 0xbd, 0x58, 0xb7,
			0x67, 0x56, 0x76, 0x03, 0x4c, 0x20, 0xe4, 0xca,
			0x11, 0x9b, 0x91, 0xc4, 0xca, 0x5d, 0xc7, 0x6a,
			0xcb, 0xff, 0x3d, 0x04, 0x62, 0x89, 0x83, 0x52,
			0xc5, 0x91, 0xc2, 0xca, 0x6f, 0x2d, 0x8b, 0x09,
			0xe2, 0xe6, 0x33, 0x8a, 0x84, 0x33, 0x6e, 0x06,
			0xf0, 0xcc, 0x02, 0x0e, 0x9e, 0xb8, 0xda, 0x78,
			0x58, 0x89, 0xb4, 0x97, 0xf3, 0xb9, 0x8e, 0x82,
			0x7e, 0xe7, 0xa7, 0xd3, 0xf1, 0xb0, 0xb7, 0x3c,
			0x19, 0x58, 0xe1, 0x6a, 0xa9, 0x78, 0x61, 0xe6,
			0x67, 0x59, 0x70, 0xce, 0x31, 0xd9, 0xd1, 0x19,
			0xbb, 0x34, 0x0b, 0xe8, 0x0f, 0xd0, 0xf4, 0x3c,
			0x3d, 0xbe, 0x64, 0xf2, 0xa5, 0x9d, 0x62, 0x9d,
		},
	},
	/* SHA256 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0x91, 0x93, 0xf8, 0xb9, 0x14, 0xdf, 0xe0, 0xe6,
			0x25, 0x21, 0xf3, 0x5a, 0xfa, 0x4f, 0xa5, 0xd4,
			0x28, 0x35, 0xe1, 0x98, 0xaf, 0x67, 0x38, 0x09,
			0x37, 0x7a, 0x3e, 0x7a, 0x99, 0x73, 0x31, 0x42,
			0xa1, 0x80, 0xdc, 0x0e, 0x13, 0xe6, 0xbb, 0x7c,
			0xeb, 0x3b, 0x60, 0xe5, 0xe9, 0xd5, 0x15, 0x79,
			0x4d, 0x82, 0xc3, 0x92, 0xe0, 0x79, 0x13, 0x42,
			0x33, 0x91, 0xd2, 0x2e, 0x2b, 0xb1, 0x9a, 0xa0,
			0xbd, 0x88, 0xaf, 0xd7, 0xf7, 0x7e, 0x27, 0xa2,
			0x40, 0xea, 0x4e, 0x2d, 0xe0, 0x85, 0x48, 0x1a,
			0xc3, 0x1f, 0xf8, 0xd3, 0x79, 0x90, 0x21, 0x1f,
			0x82, 0xf2, 0xcb, 0xf4, 0xc9, 0x0d, 0xe9, 0x8d,
			0x6e, 0x13, 0x38, 0xbb, 0xc8, 0x8e, 0x6a, 0x80,
			0xab, 0x96, 0x84, 0xda, 0xe6, 0x47, 0x85, 0xdd,
			0x10, 0x72, 0x48, 0x04, 0x85, 0x93, 0xab, 0xc9,
			0xab, 0x03, 0xf1, 0x73, 0x7a, 0x6f, 0x65, 0x30,
		},
		/* S7 : 256 byte */
		{
			0x5c, 0x2f, 0xe4, 0x53, 0xa8, 0xb0, 0x8c, 0x90,
			0xb0, 0x2e, 0xb2, 0xc9, 0x99, 0x42, 0x42, 0xd5,
			0x18, 0xf3, 0xf2, 0x1b, 0x36, 0x88, 0x95, 0xcf,
			0xfd, 0x62, 0x40, 0x50, 0xe4, 0x8a, 0xa7, 0x14,
			0x00, 0x5a, 0xe6, 0x75, 0xfe, 0x79, 0xaa, 0x3c,
			0xad, 0xd4, 0xdf, 0x55, 0xbd, 0xf1, 0x2b, 0xec,
			0x5b, 0xe8, 0xa4, 0x1d, 0x87, 0x53, 0x8f, 0x7e,
			0x03, 0x1b, 0x78, 0x2e, 0x34, 0xd3, 0x92, 0x46,
			0x8e, 0x5f, 0x14, 0xbc, 0x61, 0x3b, 0x8f, 0x4d,
			0x28, 0xc8, 0xfb, 0x79, 0xa2, 0x53, 0x7e, 0x1e,
			0x60, 0x10, 0x31, 0xda, 0x72, 0x0a, 0xcd, 0x7b,
			0x2c, 0x8d, 0xcb, 0xe9, 0x85, 0x86, 0x24, 0xa7,
			0xa9, 0xa9, 0x2a, 0x06, 0xf9, 0x18, 0x45, 0xf7,
			0x32, 0x37, 0x0d, 0x67, 0x36, 0x5c, 0x64, 0x64,
			0xf7, 0xb6, 0x8f, 0x22, 0xeb, 0x3e, 0xdf, 0xee,
			0xc9, 0x7e, 0x32, 0x85, 0x02, 0x4d, 0x7f, 0x69,
			0x43, 0xb6, 0xd5, 0x0a, 0x16, 0xcc, 0x96, 0xd6,
			0x0f, 0x68, 0x03, 0x51, 0xde, 0xaa, 0x25, 0xf0,
			0xbc, 0x86, 0x89, 0x48, 0x60, 0x7a, 0x6b, 0xa7,
			0xf1, 0x94, 0x9b, 0x85, 0x94, 0x3c, 0x6a, 0x92,
			0xbd, 0x61, 0x72, 0xe8, 0x1b, 0xcc, 0x05, 0x50,
			0x14, 0xb7, 0x8a, 0x73, 0x39, 0x72, 0xe3, 0xf3,
			0x9d, 0x14, 0x09, 0x9d, 0x16, 0x07, 0xa2, 0x0f,
			0xf8, 0x68, 0x1c, 0x29, 0xae, 0x1e, 0xf9, 0x9e,
			0xf1, 0x15, 0xed, 0x6a, 0x10, 0x84, 0xb5, 0x14,
			0xb8, 0x1a, 0x69, 0xd4, 0xa1, 0x5c, 0xe1, 0xe2,
			0x57, 0x6f, 0xdc, 0xf2, 0xb2, 0xaf, 0x61, 0x5b,
			0x52, 0xfe, 0xc7, 0x01, 0x32, 0x11, 0x2d, 0xcc,
			0x5b, 0xc1, 0x9e, 0xc1, 0x7f, 0x32, 0x28, 0x14,
			0x60, 0x62, 0x34, 0x20, 0x31, 0x73, 0x53, 0xe8,
			0xa2, 0x55, 0xfd, 0xa5, 0x02, 0xbd, 0x1f, 0xb1,
			0x1a, 0x58, 0x83, 0x2a, 0xe2, 0xc0, 0x4f, 0x9a,
		},
	},
	/* SHA256 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0x0e, 0x57, 0xef, 0x40, 0xb0, 0x21, 0xbf, 0x87,
			0xf6, 0x42, 0xc5, 0x75, 0x6b, 0x65, 0x15, 0xa0,
			0xe0, 0x6c, 0x15, 0xa0, 0x18, 0x56, 0xd7, 0x16,
			0xc5, 0x66, 0xa6, 0xed, 0xb3, 0x81, 0xdf, 0xdf,
			0x44, 0xd9, 0x03, 0x3b, 0x1c, 0xc8, 0x09, 0xe6,
			0x1d, 0xfe, 0xf9, 0xa0, 0x96, 0xdf, 0xb6, 0x89,
			0xb7, 0x27, 0x1b, 0xe4, 0x49, 0xd0, 0x4a, 0x1a,
			0x9c, 0x35, 0x41, 0x02, 0xc0, 0x77, 0xaf, 0x5f,
			0xf7, 0x20, 0x05, 0xab, 0x6b, 0x06, 0xcf, 0x13,
			0x1d, 0x73, 0x45, 0xc2, 0x1e, 0x82, 0x1d, 0x62,
			0x01, 0xcc, 0xa4, 0xe0, 0x90, 0x44, 0x0d, 0x70,
			0xbe, 0x60, 0x09, 0xd2, 0xdd, 0x7a, 0x98, 0xd3,
			0x11, 0x75, 0x1e, 0x16, 0x05, 0xa3, 0xb9, 0x14,
			0xdc, 0xe6, 0xd2, 0x62, 0x6b, 0x16, 0xf2, 0x33,
			0xa5, 0xa3, 0xd7, 0x1d, 0x56, 0x7c, 0xc8, 0x20,
			0x15, 0x2f, 0x25, 0xe4, 0x73, 0x51, 0x42, 0x42,
		},
		/* S8 : 256 byte */
		{
			0x76, 0x43, 0xaa, 0x3f, 0xe6, 0x3e, 0x66, 0xf7,
			0x9d, 0x6b, 0x40, 0x9d, 0x14, 0x5e, 0xa8, 0x20,
			0xc9, 0xf7, 0x35, 0x6f, 0x71, 0xb4, 0xac, 0xdc,
			0xbd, 0x43, 0xfe, 0x1e, 0x99, 0xf8, 0x80, 0x2c,
			0xd1, 0x66, 0x2b, 0x16, 0x24, 0x0f, 0x5c, 0xfd,
			0x94, 0xa7, 0x69, 0xb0, 0xb3, 0xf2, 0xcb, 0x0b,
			0x11, 0x88, 0x7e, 0x88, 0x6e, 0x5b, 0xa4, 0x37,
			0x33, 0x36, 0x74, 0x90, 0xb3, 0xfc, 0x18, 0x8f,
			0x2f, 0xb3, 0xa0, 0xc0, 0xc8, 0xa6, 0x8b, 0x5d,
			0x27, 0x26, 0xc8, 0xf7, 0xa3, 0x19, 0x02, 0xb6,
			0xb8, 0x6c, 0xd4, 0x02, 0x28, 0x7d, 0x38, 0x5c,
			0x3e, 0x3c, 0x06, 0x50, 0x3c, 0xe1, 0x7f, 0xd6,
			0xe5, 0x4e, 0x58, 0x2f, 0x4a, 0x90, 0x7a, 0x91,
			0xf9, 0x52, 0xd2, 0xa3, 0x60, 0xe2, 0xfb, 0xa0,
			0x00, 0x28, 0xe4, 0xd3, 0xb0, 0x2a, 0xab, 0xf7,
			0xd2, 0x20, 0xb3, 0x1d, 0x1f, 0x8e, 0xe7, 0xfa,
			0xa0, 0x70, 0x14, 0x76, 0x82, 0xcc, 0xc8, 0xbc,
			0xc7, 0x56, 0xca, 0x6a, 0x68, 0xfc, 0x20, 0x95,
			0x45, 0x50, 0xc3, 0x17, 0xe8, 0x79, 0x18, 0x78,
			0x1a, 0x3d, 0x1f, 0x19, 0x23, 0x50, 0x30, 0x91,
			0x09, 0x0c, 0x3c, 0x60, 0xca, 0x1c, 0x0b, 0x1c,
			0x69, 0x99, 0x06, 0xfb, 0xf8, 0x5a, 0xa7, 0x0a,
			0xd9, 0xae, 0x48, 0x70, 0x9f, 0xf7, 0x43, 0xb8,
			0x2d, 0xcc, 0x31, 0x07, 0x4c, 0xfc, 0xea, 0x62,
			0x3e, 0xa4, 0x5e, 0x48, 0x64, 0x4b, 0x19, 0xa2,
			0x17, 0x72, 0xca, 0x10, 0x7e, 0xd6, 0x42, 0x39,
			0xc5, 0x65, 0x74, 0xa0, 0x87, 0xf1, 0xa6, 0xaa,
			0xdf, 0x0f, 0x4b, 0x00, 0xff, 0xe5, 0x81, 0xc1,
			0x41, 0x02, 0x74, 0xc8, 0x75, 0xe4, 0x59, 0x90,
			0x63, 0xe4, 0x6e, 0x51, 0x68, 0x80, 0x3f, 0x0d,
			0x28, 0xd2, 0x1f, 0xcd, 0x35, 0x09, 0xb4, 0xc6,
			0x22, 0x29, 0x95, 0xad, 0xd7, 0x75, 0x3b, 0xf3,
		},
	},
	/* SHA256 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0x0c, 0x84, 0x91, 0xfc, 0x34, 0x8d, 0x34, 0x1f,
			0xe8, 0x5c, 0x46, 0xa5, 0x61, 0x15, 0xf2, 0x60,
			0x35, 0xc5, 0x9e, 0x6a, 0x2b, 0xe7, 0x65, 0xc4,
			0x4e, 0x2e, 0xc8, 0x3d, 0x40, 0x7e, 0xa0, 0x96,
			0xd1, 0x3b, 0x57, 0xe3, 0xd0, 0xc7, 0x58, 0x34,
			0x22, 0x46, 0xc4, 0x75, 0x10, 0xa5, 0x67, 0x93,
			0xe5, 0xda, 0xea, 0xe1, 0xb9, 0x6d, 0x4a, 0xb9,
			0x88, 0x37, 0x89, 0x66, 0x87, 0x6a, 0xa3, 0x41,
			0xb7, 0xd1, 0xc3, 0x1b, 0xba, 0x59, 0xb7, 0xdb,
			0xe6, 0xd1, 0xa1, 0x68, 0x98, 0xee, 0xf0, 0xca,
			0xca, 0x92, 0x8f, 0x8c, 0xe8, 0x4d, 0x5c, 0x64,
			0xe0, 0x25, 0xdc, 0x16, 0x79, 0x92, 0x2d, 0x95,
			0xe5, 0xcd, 0x3c, 0x6b, 0x99, 0x4a, 0x38, 0x5c,
			0x5c, 0x83, 0x46, 0x46, 0x9e, 0xf8, 0x76, 0x4c,
			0x0c, 0x74, 0xf5, 0x33, 0x61, 0x91, 0x85, 0x0c,
			0x7f, 0x7e, 0x2b, 0x14, 0xbe, 0x00, 0x27, 0xd8,
		},
		/* S9 : 256 byte */
		{
			0xca, 0xcc, 0x8d, 0x9f, 0x5e, 0xcd, 0x34, 0xc1,
			0x43, 0x48, 0x84, 0x61, 0x13, 0x5c, 0x49, 0x51,
			0x67, 0x61, 0x45, 0xc6, 0xe4, 0x72, 0xb9, 0x2f,
			0x12, 0xf7, 0x58, 0x04, 0x6f, 0x17, 0x21, 0x42,
			0xfa, 0x38, 0x8f, 0x28, 0x5f, 0x3f, 0xff, 0x06,
			0x82, 0x42, 0x02, 0x88, 0x29, 0x04, 0x7e, 0x24,
			0x80, 0x59, 0xed, 0x4f, 0xd3, 0x9d, 0x2c, 0x5a,
			0xde, 0x46, 0x9d, 0xc7, 0xc3, 0x93, 0x45, 0xe5,
			0x11, 0x49, 0x50, 0xd2, 0x03, 0x1c, 0xc7, 0x46,
			0x5f, 0xe7, 0x12, 0xc4, 0x04, 0x1d, 0x05, 0xc7,
			0x56, 0xd3, 0xf2, 0xd8, 0x8a, 0x46, 0xce, 0xb9,
			0x9f, 0x2e, 0x24, 0xa5, 0x2e, 0x95, 0x8a, 0x03,
			0xcd, 0x25, 0x19, 0xa9, 0xb1, 0x37, 0xe6, 0x2d,
			0x5c, 0xa2, 0xb3, 0x53, 0xf7, 0xb0, 0x47, 0xb6,
			0x25, 0xc3, 0x60, 0x23, 0x13, 0xfd, 0xb5, 0x3c,
			0x8d, 0xb2, 0x3d, 0x83, 0x95, 0x1a, 0x59, 0x9d,
			0xb3, 0x28, 0xfe, 0xdc, 0x4a, 0xe0, 0x6d, 0xa8,
			0x9c, 0xe7, 0xf5, 0x62, 0x59, 0xb5, 0xc8, 0x22,
			0x2f, 0x7b, 0xd3, 0xd9, 0x74, 0x04, 0x78, 0xfd,
			0x28, 0xe5, 0x81, 0x0d, 0xb7, 0x8a, 0xee, 0x86,
			0x23, 0xfd, 0xd3, 0x9f, 0x60, 0x3f, 0x8d, 0xdf,
			0x98, 0x08, 0x1d, 0x78, 0x73, 0x98, 0x0c, 0x4e,
			0xb0, 0xe2, 0x2a, 0x9c, 0xd4, 0x08, 0xf7, 0xc4,
			0x13, 0x4c, 0x12, 0xd2, 0x04, 0x9a, 0x2d, 0x12,
			0x0f, 0x4b, 0x62, 0xe6, 0xb3, 0x82, 0xb9, 0x97,
			0xfc, 0x37, 0x5e, 0xf7, 0xac, 0x95, 0x5f, 0xcf,
			0x80, 0xb0, 0x45, 0xc3, 0xd6, 0x38, 0x5f, 0xf4,
			0x22, 0xda, 0xd3, 0x50, 0xc6, 0x88, 0x70, 0x53,
			0x90, 0x68, 0xa1, 0x62, 0xa2, 0xed, 0xbb, 0x93,
			0xce, 0xef, 0xed, 0x96, 0x77, 0x93, 0x9b, 0x90,
			0xbd, 0x3d, 0xfa, 0x0d, 0xc0, 0x53, 0x46, 0x0b,
			0x4e, 0x23, 0x32, 0xef, 0xa6, 0x92, 0x17, 0x9a,
		},
	},
};

/* 10 elements */
static mod2048_t SHA384[] = {
	/* SHA384 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0x6c, 0xd5, 0x9f, 0xdd, 0x3e, 0xfd, 0x89, 0x3d,
			0x09, 0x1a, 0xfd, 0xc3, 0x15, 0x5d, 0x35, 0x4f,
			0x10, 0xd6, 0xd8, 0x81, 0x67, 0x42, 0x7a, 0x2c,
			0xf7, 0x24, 0x62, 0x07, 0xe5, 0x17, 0x91, 0xa6,
			0xca, 0x62, 0x00, 0xa9, 0x14, 0xcd, 0x28, 0x34,
			0xa9, 0xb3, 0xc7, 0x9f, 0xcd, 0x59, 0xe2, 0x6e,
			0x45, 0x7e, 0x06, 0x83, 0xbc, 0x33, 0xd4, 0x92,
			0x67, 0xed, 0xbd, 0xd6, 0xe5, 0xd9, 0x09, 0x02,
			0x69, 0x6f, 0x1e, 0x7b, 0x1a, 0x4a, 0xff, 0xc4,
			0xba, 0x37, 0x13, 0x39, 0x86, 0x8c, 0x28, 0x01,
			0x5e, 0xbb, 0xb7, 0x3e, 0x26, 0x26, 0x69, 0x86,
			0x6c, 0x35, 0xdb, 0x97, 0x4b, 0xa6, 0x9e, 0x46,
			0x8f, 0x25, 0x83, 0xb9, 0x19, 0x1d, 0x15, 0xd6,
			0x86, 0xcd, 0x66, 0xfb, 0x0b, 0x9e, 0x0f, 0xf0,
			0xa3, 0xb4, 0x72, 0x1a, 0x6d, 0xc3, 0x42, 0xf1,
			0x4f, 0x24, 0x46, 0xb4, 0xe0, 0x28, 0x59, 0x5b,
		},
		/* S0 : 256 byte */
		{
			0x39, 0x74, 0x90, 0x0b, 0xec, 0x3f, 0xcb, 0x08,
			0x1f, 0x0e, 0x5a, 0x29, 0x9a, 0xdf, 0x30, 0xd0,
			0x87, 0xaa, 0xba, 0xa6, 0x33, 0x91, 0x14, 0x10,
			0xe8, 0x7a, 0x49, 0x79, 0xbb, 0xe3, 0xfa, 0x80,
			0xc3, 0xab, 0xcf, 0x22, 0x16, 0x86, 0x39, 0x9a,
			0x49, 0xbc, 0x2f, 0x1e, 0x5a, 0xc4, 0x0c, 0x35,
			0xdf, 0x17, 0x00, 0xe4, 0xb9, 0xcb, 0x7c, 0x80,
			0x5a, 0x89, 0x66, 0x46, 0x57, 0x3f, 0x4a, 0x57,
			0x0a, 0x97, 0x04, 0xd2, 0xa2, 0xe6, 0xba, 0xee,
			0x4b, 0x43, 0xd9, 0x16, 0x90, 0x68, 0x84, 0xad,
			0x3c, 0xf2, 0x83, 0x52, 0x9e, 0xa2, 0x65, 0xe8,
			0xfc, 0xb5, 0xcc, 0x1b, 0xdf, 0x7b, 0x7d, 0xee,
			0x85, 0x94, 0x1e, 0x4b, 0x4f, 0xb2, 0x5c, 0x1f,
			0xc7, 0xb9, 0x51, 0xfb, 0x12, 0x9a, 0xb3, 0x93,
			0xcb, 0x06, 0x9b, 0xe2, 0x71, 0xc1, 0xd9, 0x54,
			0xda, 0x3c, 0x43, 0x67, 0x43, 0x09, 0xf1, 0xd2,
			0x12, 0x82, 0x6f, 0xab, 0xb8, 0xe8, 0x12, 0xde,
			0x2d, 0x53, 0xd1, 0x25, 0x97, 0xde, 0x04, 0x0d,
			0x32, 0xcb, 0x28, 0xc9, 0xf8, 0x13, 0x15, 0x9c,
			0xb1, 0x8c, 0x1b, 0x51, 0xf7, 0xa8, 0x74, 0xcb,
			0xf2, 0x29, 0xcc, 0x22, 0x2c, 0xae, 0xb9, 0x8e,
			0x35, 0xec, 0x5e, 0x4b, 0xf5, 0xc5, 0xe2, 0x2c,
			0xc8, 0x52, 0x86, 0x31, 0xf1, 0x51, 0x17, 0xe8,
			0xc2, 0xbe, 0x6e, 0xac, 0x91, 0xf4, 0x07, 0x0e,
			0xec, 0xdd, 0x07, 0xec, 0xc6, 0xdb, 0x6c, 0x46,
			0xea, 0xa6, 0x5f, 0x47, 0x2f, 0x20, 0x06, 0x98,
			0x8e, 0xfe, 0xf0, 0xb5, 0x1c, 0x53, 0x8c, 0x6e,
			0x04, 0xd7, 0x51, 0x9c, 0x8e, 0x3d, 0xa4, 0xb1,
			0x72, 0xb1, 0xe2, 0x76, 0x10, 0x89, 0xed, 0x3a,
			0xd1, 0x19, 0x79, 0x92, 0xef, 0x37, 0xc1, 0x68,
			0xdc, 0x88, 0x1c, 0x8b, 0x5f, 0x8b, 0xbf, 0xee,
			0x91, 0x9f, 0x7c, 0x7a, 0xfd, 0x25, 0xb8, 0xfc,
		},
	},
	/* SHA384 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0xac, 0xb3, 0x0b, 0xe9, 0x09, 0x2b, 0x2f, 0x18,
			0xf2, 0x59, 0x34, 0xa0, 0xd6, 0x78, 0xb6, 0xbc,
			0xd6, 0xb6, 0x7c, 0x2b, 0x88, 0xe7, 0x58, 0x84,
			0xf4, 0x7b, 0x4f, 0xca, 0xe3, 0xad, 0xfa, 0x40,
			0x5a, 0xfe, 0x2c, 0x7e, 0x61, 0xe2, 0xd6, 0xc5,
			0x08, 0xb9, 0x27, 0x90, 0xac, 0x00, 0xf7, 0x6b,
			0x77, 0xc9, 0x65, 0x08, 0x26, 0x68, 0xbf, 0x90,
			0x0f, 0x70, 0xa3, 0x37, 0x62, 0xde, 0x64, 0x13,
			0xaf, 0x93, 0xaf, 0x2e, 0xa8, 0x08, 0x6f, 0xda,
			0x29, 0x3d, 0xed, 0x44, 0x75, 0xf2, 0x3c, 0x4c,
			0xc3, 0x1a, 0xd4, 0x94, 0xf9, 0x8d, 0x7d, 0xd7,
			0xb7, 0xfd, 0x6f, 0x7d, 0x97, 0x2b, 0xb7, 0x6c,
			0xb3, 0x5a, 0xdc, 0x20, 0x68, 0x04, 0xc3, 0xfe,
			0x5a, 0xcd, 0xd0, 0xe5, 0xb8, 0xb5, 0x4e, 0x07,
			0xc2, 0x91, 0x11, 0xf7, 0x88, 0xbc, 0x59, 0x02,
			0xf4, 0x0a, 0xfa, 0xc3, 0x0a, 0xfd, 0xba, 0xf2,
		},
		/* S1 : 256 byte */
		{
			0xb5, 0xc6, 0x0d, 0x8d, 0xa9, 0xb3, 0x94, 0x38,
			0x78, 0xcb, 0x23, 0x59, 0xcf, 0x65, 0xe4, 0x81,
			0x7c, 0x07, 0x94, 0xf9, 0x50, 0x45, 0x3c, 0xa7,
			0x7c, 0x81, 0xa5, 0xa1, 0xc1, 0x58, 0x55, 0x91,
			0xaa, 0x50, 0xa6, 0x74, 0x68, 0xe3, 0xb3, 0x99,
			0xe4, 0xfa, 0xf1, 0xd6, 0x06, 0xbe, 0xa0, 0xd9,
			0xe6, 0xcc, 0x1d, 0x2d, 0x70, 0xdb, 0x80, 0x63,
			0x73, 0x9e, 0x0c, 0x27, 0xd3, 0xdc, 0x9f, 0x9a,
			0xfe, 0x88, 0xde, 0xa5, 0x2e, 0x73, 0x29, 0x8a,
			0x07, 0xd0, 0x5c, 0x7d, 0x97, 0x07, 0x00, 0x2e,
			0xfa, 0x53, 0x7c, 0x38, 0x9e, 0x38, 0xbd, 0x37,
			0xbc, 0xa7, 0x4e, 0xb0, 0xaf, 0x62, 0x61, 0xa5,
			0xda, 0x06, 0x13, 0x62, 0x02, 0xc8, 0xad, 0x48,
			0x7e, 0xeb, 0xd5, 0x0b, 0xef, 0x74, 0x76, 0x70,
			0x89, 0xc7, 0x08, 0x70, 0xbe, 0x1d, 0x8f, 0xab,
			0x91, 0x56, 0xf9, 0xfd, 0xbc, 0x2f, 0x2e, 0x9c,
			0xc3, 0x30, 0xa9, 0x50, 0x18, 0xce, 0x79, 0x43,
			0x98, 0x4b, 0xec, 0xc2, 0x56, 0x21, 0xbf, 0xa6,
			0x60, 0x18, 0xef, 0x83, 0x20, 0xb6, 0x00, 0x59,
			0xf9, 0x41, 0x15, 0x6e, 0x9c, 0xdd, 0x87, 0xff,
			0x0d, 0x82, 0xcf, 0x7b, 0xe7, 0x74, 0x65, 0xe0,
			0x20, 0x3e, 0x71, 0x20, 0xaa, 0xec, 0xed, 0x84,
			0xab, 0xd8, 0x18, 0x69, 0x47, 0xd4, 0xac, 0x3d,
			0xaf, 0x3f, 0x99, 0x39, 0x02, 0xae, 0xc4, 0x7c,
			0x30, 0x90, 0x47, 0x5c, 0x85, 0x7b, 0x5d, 0x35,
			0x9f, 0x0a, 0x55, 0x72, 0xd4, 0x68, 0x8e, 0x5a,
			0x76, 0xa4, 0x65, 0x38, 0x68, 0xff, 0x54, 0xce,
			0x9f, 0x99, 0x9e, 0x6b, 0xb5, 0x59, 0xd1, 0xc1,
			0x1c, 0x67, 0xc1, 0x5b, 0xe9, 0xd7, 0xfe, 0x5f,
			0x8c, 0x17, 0x04, 0x30, 0x1d, 0x05, 0x5f, 0x3d,
			0x29, 0x07, 0x72, 0x27, 0x79, 0xd6, 0x01, 0x20,
			0x36, 0x08, 0x4e, 0x95, 0x0d, 0xe3, 0x6f, 0x4f,
		},
	},
	/* SHA384 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x60, 0x1a, 0x6a, 0xad, 0x3f, 0xaa, 0x79, 0x88,
			0xd5, 0xae, 0x52, 0x8a, 0x69, 0x69, 0x03, 0x1b,
			0x10, 0xa6, 0xf3, 0x92, 0x16, 0x94, 0x6a, 0xa8,
			0x9f, 0xd4, 0x53, 0x2c, 0x8e, 0xd1, 0x41, 0xf9,
			0xa6, 0x50, 0xb1, 0x26, 0xef, 0x48, 0x8f, 0x7c,
			0x5c, 0xf3, 0xfb, 0x2d, 0xaa, 0x25, 0x4c, 0xc2,
			0x8b, 0xdd, 0x55, 0x56, 0x04, 0x19, 0xe8, 0x02,
			0x14, 0xef, 0x99, 0x98, 0x96, 0xda, 0xc4, 0x94,
			0x68, 0x52, 0xd2, 0x4f, 0xcd, 0x9f, 0xb7, 0x76,
			0x10, 0xee, 0xbf, 0xbb, 0x6b, 0xa5, 0x8b, 0xca,
			0x26, 0xf4, 0x56, 0x7f, 0x03, 0xac, 0x7e, 0x56,
			0xda, 0x55, 0x3f, 0x23, 0x81, 0x7b, 0xc1, 0x03,
			0xee, 0x48, 0x55, 0x92, 0xa0, 0x58, 0xfb, 0x5e,
			0x3b, 0xc8, 0x29, 0x9c, 0x72, 0x90, 0xc7, 0x1a,
			0x29, 0x13, 0x7e, 0x75, 0xdb, 0xf5, 0x32, 0x8c,
			0x3a, 0x2d, 0xcd, 0x34, 0x16, 0x5b, 0x3f, 0x2e,
		},
		/* S2 : 256 byte */
		{
			0x30, 0x1d, 0x60, 0xd5, 0x65, 0x76, 0xf3, 0x66,
			0x3a, 0x7f, 0xbe, 0x80, 0x36, 0xbb, 0xe4, 0xfb,
			0xc0, 0xfb, 0xd8, 0x2c, 0xd6, 0xa4, 0x2e, 0x36,
			0xd7, 0xbb, 0xc8, 0xb2, 0x06, 0x54, 0x3d, 0xc2,
			0xd5, 0x6d, 0x31, 0x98, 0xe7, 0x91, 0x1a, 0xd1,
			0x38, 0xca, 0xd2, 0x22, 0xdd, 0x99, 0x05, 0x0d,
			0xd1, 0xf8, 0x5f, 0xe1, 0x9c, 0x8a, 0x88, 0xbf,
			0x67, 0x13, 0x5e, 0x7f, 0x8f, 0x11, 0xb5, 0xf5,
			0xe4, 0x85, 0xc9, 0x1f, 0xc7, 0xd4, 0x78, 0x06,
			0x9b, 0x72, 0xf4, 0x6e, 0xbc, 0xdc, 0xf2, 0xd2,
			0xae, 0x7d, 0xe6, 0xac, 0x8f, 0xe5, 0x3b, 0xb6,
			0xc0, 0x49, 0x11, 0xd1, 0x22, 0xcc, 0x23, 0x1d,
			0xc2, 0x10, 0xb2, 0x14, 0x7e, 0xbe, 0x8b, 0x05,
			0x2e, 0x8b, 0x2c, 0xcc, 0x09, 0xf3, 0x38, 0xb3,
			0x49, 0xde, 0x20, 0x25, 0xcc, 0x87, 0xb2, 0x61,
			0x9a, 0x7b, 0x16, 0x33, 0x47, 0xca, 0x66, 0xa3,
			0x47, 0x91, 0xa2, 0xe4, 0x6b, 0x4e, 0x2a, 0xc5,
			0x7e, 0xb9, 0xf6, 0x02, 0x9c, 0xdb, 0xe0, 0x24,
			0xe8, 0x96, 0xd5, 0x7f, 0x7d, 0x04, 0x91, 0xf7,
			0x78, 0x33, 0x12, 0xf8, 0xf0, 0x6c, 0x79, 0x07,
			0x70, 0x15, 0x0c, 0xd1, 0x39, 0xf6, 0x1f, 0xd2,
			0xb3, 0xe7, 0x04, 0x1b, 0x37, 0x26, 0x1c, 0x6e,
			0x7e, 0xa8, 0x6d, 0x4e, 0x06, 0xd9, 0x30, 0x0b,
			0x1a, 0x56, 0x67, 0xcb, 0x02, 0x88, 0xc5, 0x50,
			0xb2, 0xaf, 0xb3, 0x55, 0x94, 0x48, 0x34, 0xb4,
			0x61, 0xce, 0xad, 0x13, 0x79, 0x42, 0x76, 0xbb,
			0x46, 0xe5, 0xe2, 0x0a, 0xec, 0x7b, 0x63, 0xaa,
			0xca, 0x4d, 0x49, 0x1a, 0x50, 0x0f, 0xac, 0xd5,
			0x9a, 0x37, 0xc5, 0x27, 0x79, 0xcf, 0x46, 0x7d,
			0x74, 0xaf, 0x1e, 0x62, 0xb1, 0xeb, 0xe0, 0xfd,
			0x0b, 0xe1, 0xca, 0xcb, 0x7c, 0xe6, 0xd0, 0x50,
			0xd8, 0x6e, 0x4e, 0xb7, 0x6c, 0xde, 0x06, 0x93,
		},
	},
	/* SHA384 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0x44, 0xd3, 0xe0, 0xfc, 0x90, 0x10, 0x0a, 0x1c,
			0x93, 0x16, 0x06, 0x3f, 0x26, 0xb1, 0x80, 0x32,
			0x6c, 0xc2, 0xe3, 0x83, 0x4c, 0xe5, 0x6e, 0x43,
			0x24, 0x52, 0x8a, 0x0b, 0xbb, 0x01, 0x5b, 0x3d,
			0x78, 0x12, 0x95, 0x8c, 0xd2, 0x6b, 0x91, 0xbf,
			0x08, 0xa3, 0xa0, 0xb1, 0x12, 0x1f, 0x9f, 0x9d,
			0xd7, 0x7a, 0xcb, 0x98, 0xa0, 0x2a, 0xd7, 0x5f,
			0xcd, 0x61, 0x3c, 0x53, 0xc7, 0x32, 0xd1, 0xc2,
			0x35, 0xf5, 0x9b, 0x68, 0x73, 0xec, 0xe6, 0x36,
			0x3f, 0x27, 0x94, 0x52, 0xb6, 0xa4, 0xb6, 0x5e,
			0x80, 0xbb, 0x59, 0xfd, 0x47, 0xb9, 0xa2, 0x93,
			0x6d, 0xcc, 0x1e, 0x4d, 0xfe, 0x1f, 0x53, 0x62,
			0xe3, 0x45, 0x9b, 0x98, 0x59, 0xdb, 0x32, 0x09,
			0xa2, 0x69, 0x8d, 0x27, 0xfa, 0x8a, 0xed, 0xfe,
			0xcd, 0x4d, 0x35, 0xb9, 0x27, 0xda, 0xf8, 0x68,
			0x6c, 0x59, 0xd7, 0x00, 0x49, 0x0f, 0x0a, 0xa3,
		},
		/* S3 : 256 byte */
		{
			0xaf, 0x22, 0x29, 0xe9, 0x4a, 0x85, 0x7b, 0x89,
			0xe0, 0xe8, 0x90, 0xda, 0xca, 0x3a, 0x8f, 0xe1,
			0x2e, 0xbd, 0xba, 0x04, 0x94, 0x8d, 0x18, 0x83,
			0xa7, 0xd7, 0x81, 0x6a, 0x3b, 0x68, 0x2f, 0x7d,
			0xa3, 0x03, 0x25, 0x40, 0xa8, 0x76, 0x9f, 0x9c,
			0xca, 0xc9, 0x58, 0x6c, 0xf2, 0x4e, 0x8c, 0x20,
			0x4b, 0x45, 0xb8, 0x5d, 0x1b, 0xdc, 0xc5, 0xa5,
			0x45, 0x0a, 0x21, 0x5b, 0x40, 0x48, 0xea, 0x42,
			0x98, 0x3b, 0x34, 0x56, 0xfa, 0x8c, 0x76, 0xc6,
			0x78, 0x6e, 0x02, 0x4f, 0x70, 0x5e, 0x08, 0x8d,
			0x69, 0x45, 0x59, 0xd6, 0x68, 0xca, 0xa8, 0x68,
			0x4c, 0xad, 0x0f, 0xc5, 0x78, 0x50, 0xfc, 0xaf,
			0x34, 0xe4, 0x58, 0xae, 0xe8, 0xfa, 0xd4, 0xe0,
			0x9e, 0x6f, 0x19, 0x65, 0x57, 0xd4, 0xe8, 0x86,
			0x02, 0x84, 0xd9, 0x82, 0xc0, 0x10, 0x5d, 0x98,
			0xce, 0x49, 0x12, 0xe9, 0x6c, 0x35, 0x50, 0xe2,
			0xa0, 0xc7, 0xe8, 0xba, 0xd5, 0xab, 0xc2, 0x9a,
			0x9a, 0x54, 0x2f, 0x57, 0xa8, 0xc6, 0x05, 0x79,
			0x03, 0x80, 0x67, 0xb3, 0xd5, 0x39, 0x1a, 0xbc,
			0x21, 0xb4, 0xf9, 0xde, 0xb0, 0x24, 0xca, 0x58,
			0xf9, 0xb0, 0xc3, 0x8c, 0x0d, 0x1f, 0x82, 0x37,
			0x3f, 0x52, 0x8e, 0x93, 0x9b, 0xd7, 0x3a, 0x24,
			0xd5, 0x01, 0xc5, 0x91, 0x16, 0x88, 0x14, 0xc8,
			0x72, 0xc5, 0x25, 0xdb, 0x0e, 0x56, 0xca, 0xe4,
			0x7d, 0xf0, 0x0f, 0xa3, 0x72, 0x8d, 0xc3, 0xa0,
			0x97, 0x69, 0x65, 0x32, 0x3c, 0xe8, 0xd2, 0xde,
			0xe2, 0xb1, 0x38, 0xb5, 0x0a, 0xb7, 0xaf, 0xd4,
			0x84, 0x95, 0x11, 0x46, 0x73, 0xe9, 0x1b, 0xb3,
			0xed, 0x22, 0x05, 0xe2, 0x6a, 0x84, 0x55, 0x47,
			0x4c, 0x3d, 0x4e, 0xc8, 0x73, 0x9b, 0xbf, 0xf6,
			0xdf, 0x39, 0xb2, 0xb7, 0x2e, 0xe0, 0x50, 0x41,
			0x09, 0x30, 0x42, 0x3b, 0x14, 0x72, 0xb6, 0xed,
		},
	},
	/* SHA384 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0x5a, 0xf0, 0x90, 0x77, 0xa1, 0xf5, 0x34, 0xb8,
			0x98, 0x22, 0xb2, 0x6c, 0x32, 0x72, 0xad, 0xf8,
			0x50, 0x0d, 0x3c, 0x6b, 0xd9, 0x0f, 0x9b, 0x5e,
			0x0d, 0x8b, 0x21, 0x1f, 0x16, 0xd0, 0x72, 0x0e,
			0xe0, 0xea, 0xf6, 0x46, 0x2b, 0x6c, 0x8a, 0x80,
			0xdf, 0x6d, 0x75, 0x35, 0x9f, 0xd1, 0x9d, 0x03,
			0xa0, 0xca, 0xfb, 0x52, 0xbc, 0x9d, 0x4c, 0x37,
			0xc2, 0xaa, 0x09, 0x99, 0x11, 0xa7, 0x9a, 0x92,
			0x65, 0x2c, 0xc7, 0x17, 0xf0, 0x74, 0x6f, 0xdc,
			0xad, 0x62, 0x7c, 0x72, 0xf1, 0xc2, 0x16, 0xb2,
			0x43, 0xd2, 0x17, 0x5f, 0x6d, 0x00, 0xbf, 0x07,
			0xd3, 0xf6, 0xaa, 0x2a, 0x04, 0xd4, 0xfe, 0x9f,
			0x8f, 0xbc, 0xe9, 0x32, 0x18, 0x94, 0x4b, 0x92,
			0xaa, 0x07, 0xaf, 0x6b, 0x4f, 0xcd, 0x80, 0xcf,
			0xde, 0x2d, 0x7a, 0xda, 0x15, 0xc0, 0x5e, 0x96,
			0xe7, 0x77, 0xea, 0x1c, 0x17, 0xdf, 0x08, 0xfc,
		},
		/* S4 : 256 byte */
		{
			0xa5, 0x68, 0x23, 0xfa, 0x57, 0x7e, 0x89, 0x46,
			0xf1, 0xd2, 0xf6, 0xe3, 0x51, 0xb7, 0x38, 0xb5,
			0x35, 0x92, 0x54, 0x43, 0x58, 0x52, 0x8a, 0xf8,
			0x88, 0x07, 0xea, 0x4f, 0x19, 0x01, 0x7d, 0xfe,
			0x81, 0xa3, 0xd6, 0x9f, 0x62, 0xfb, 0xff, 0x64,
			0x95, 0x50, 0xd9, 0xb3, 0x10, 0xfa, 0xf2, 0x7a,
			0x04, 0x1f, 0xe6, 0x24, 0xf0, 0xa0, 0x2b, 0xdc,
			0xdd, 0xb7, 0x9b, 0xfb, 0x0a, 0x46, 0x57, 0x39,
			0xec, 0x8b, 0x64, 0xb7, 0x48, 0xcc, 0x29, 0xe5,
			0xa0, 0x2c, 0x77, 0x7e, 0x18, 0x26, 0xd3, 0xe2,
			0xf1, 0xee, 0xe6, 0xfe, 0x2e, 0xde, 0xe4, 0xa8,
			0xbc, 0xac, 0x51, 0x9c, 0x7c, 0x7c, 0xa5, 0xc0,
			0x39, 0xe7, 0x6d, 0x63, 0x06, 0x68, 0x94, 0x5a,
			0x1e, 0x5e, 0x86, 0x18, 0xe2, 0x35, 0x86, 0x45,
			0x61, 0xa4, 0x40, 0xe7, 0x3e, 0x39, 0xf6, 0xd6,
			0x84, 0x2a, 0xd7, 0xda, 0x64, 0xef, 0x5b, 0x0c,
			0xe1, 0xc4, 0xab, 0x88, 0xdb, 0x15, 0x7b, 0x68,
			0x10, 0x71, 0x74, 0xad, 0x7d, 0x5c, 0x9a, 0x60,
			0x65, 0x06, 0x87, 0x68, 0xc1, 0x1c, 0x4c, 0x96,
			0xff, 0x67, 0x05, 0x0b, 0x5d, 0x07, 0xb8, 0xcd,
			0x02, 0x7f, 0xcd, 0x0d, 0x34, 0x7e, 0xc7, 0x9a,
			0x19, 0x7c, 0xf4, 0x34, 0x35, 0x98, 0x5b, 0xc1,
			0xae, 0xb4, 0x79, 0xdb, 0x00, 0x22, 0x28, 0x9e,
			0x8d, 0xd3, 0xb3, 0x1b, 0xb7, 0xc6, 0x2d, 0x88,
			0x31, 0xcf, 0xe6, 0x95, 0x2f, 0x41, 0xd2, 0x4f,
			0x89, 0xd7, 0x53, 0x78, 0x95, 0x35, 0xf9, 0x18,
			0xff, 0x68, 0xb3, 0x69, 0x50, 0xaf, 0x6f, 0xd3,
			0x1d, 0xee, 0x1a, 0xc4, 0x76, 0xa0, 0xcf, 0x93,
			0xaf, 0xe9, 0xf4, 0xa7, 0x66, 0xf3, 0xc4, 0xd2,
			0xc0, 0xc3, 0xf9, 0x28, 0x25, 0xd5, 0x57, 0x2e,
			0xb2, 0xeb, 0x8a, 0x2b, 0x64, 0x4e, 0x32, 0x9e,
			0xea, 0x16, 0x83, 0xf9, 0x08, 0x10, 0xed, 0x77,
		},
	},
	/* SHA384 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0xf6, 0x0a, 0x3a, 0x54, 0x37, 0x68, 0xfa, 0xbe,
			0x37, 0xf0, 0x03, 0x00, 0x9a, 0x8c, 0x26, 0xf7,
			0xdc, 0x91, 0xf1, 0x42, 0x2d, 0x44, 0x29, 0xed,
			0x7f, 0x9d, 0x74, 0x4c, 0xdd, 0x4b, 0x55, 0x2a,
			0xfe, 0xf7, 0x5d, 0x24, 0x1a, 0xcd, 0xa0, 0x4f,
			0xfc, 0x39, 0x67, 0x21, 0x59, 0xee, 0x24, 0x8e,
			0x60, 0x2d, 0xab, 0x71, 0x92, 0x44, 0x9e, 0x2e,
			0xd4, 0x55, 0x29, 0x95, 0xc2, 0x58, 0xf0, 0x0a,
			0x47, 0x63, 0x46, 0xe3, 0x6a, 0x29, 0xa0, 0x12,
			0x6b, 0xc2, 0x49, 0x04, 0x0f, 0xaa, 0x57, 0xc9,
			0x38, 0x0b, 0xdd, 0x74, 0xb8, 0x3f, 0x62, 0xc5,
			0x67, 0x90, 0x92, 0x05, 0x74, 0x43, 0x34, 0x32,
			0xf8, 0xd6, 0x5c, 0x5c, 0xd1, 0x85, 0xe2, 0x4f,
			0xad, 0x13, 0x12, 0x72, 0x65, 0xc6, 0xa5, 0xef,
			0x8d, 0xb4, 0xf1, 0x14, 0x49, 0x3d, 0x5c, 0xfa,
			0x61, 0xd9, 0x16, 0x64, 0x98, 0x14, 0x08, 0xe9,
		},
		/* S5 : 256 byte */
		{
			0x08, 0xd3, 0x96, 0x48, 0x1d, 0xee, 0xf1, 0x8c,
			0xb0, 0xbe, 0xf7, 0xc3, 0xe8, 0x26, 0xfe, 0x6e,
			0x5c, 0x9e, 0xcc, 0x85, 0xe5, 0x23, 0x0d, 0x35,
			0xd6, 0x67, 0x72, 0xb8, 0xd2, 0xd0, 0x15, 0xd4,
			0xe5, 0xf5, 0x79, 0x4f, 0xbe, 0x05, 0x50, 0xdf,
			0x2f, 0x74, 0x57, 0x30, 0xd6, 0xf8, 0xd1, 0xd3,
			0xb8, 0x50, 0xd1, 0x64, 0xfc, 0xe4, 0x63, 0x08,
			0x05, 0xe7, 0x11, 0xb5, 0x93, 0x08, 0xf8, 0x60,
			0x85, 0x06, 0xb7, 0xe0, 0x1e, 0x8e, 0x92, 0x94,
			0xed, 0x8b, 0x7e, 0x75, 0x82, 0x16, 0x56, 0x77,
			0xf1, 0x80, 0xe9, 0x65, 0x16, 0x9d, 0xca, 0x81,
			0xb3, 0xda, 0xf2, 0x4d, 0x7b, 0x92, 0xfe, 0x32,
			0xd6, 0xa9, 0xac, 0x63, 0x82, 0x1d, 0x48, 0xb1,
			0xa0, 0xa1, 0x44, 0xfc, 0x7a, 0x04, 0xb0, 0xbf,
			0xc6, 0x3a, 0x3b, 0xc1, 0x6a, 0x0f, 0xd8, 0x37,
			0xb0, 0x20, 0x37, 0xed, 0x76, 0xe5, 0x0d, 0x46,
			0xcb, 0xfa, 0x38, 0x57, 0xe6, 0x58, 0xe3, 0x70,
			0xc5, 0x86, 0xab, 0x1e, 0xed, 0x82, 0x50, 0x76,
			0x32, 0x1a, 0xc8, 0xe8, 0x2b, 0xe3, 0x74, 0xba,
			0xcb, 0x29, 0x5e, 0x4d, 0x34, 0x08, 0xf0, 0xcc,
			0x1f, 0xc4, 0xc3, 0x00, 0xb8, 0x42, 0x75, 0xa5,
			0x1c, 0x35, 0x73, 0xe9, 0xca, 0xbf, 0xdb, 0xe3,
			0xdc, 0x51, 0xe4, 0xa6, 0xf5, 0x81, 0x1d, 0x86,
			0x0d, 0x72, 0x5a, 0xaf, 0x8f, 0xd0, 0xaf, 0x19,
			0xa2, 0x43, 0x7b, 0x0f, 0x1c, 0x80, 0xf5, 0xac,
			0x22, 0x2f, 0x6b, 0x25, 0xf1, 0xfa, 0x09, 0xe9,
			0x33, 0x99, 0xa6, 0x97, 0x6b, 0x1b, 0x3c, 0xa7,
			0x6a, 0xfe, 0x60, 0x86, 0xe9, 0xb2, 0x32, 0xaa,
			0xe6, 0xc7, 0xb8, 0x18, 0x25, 0x5b, 0xf9, 0x63,
			0xf3, 0x1c, 0x04, 0xae, 0x3f, 0xa2, 0x13, 0x6c,
			0x0a, 0x44, 0x29, 0x97, 0xd4, 0xcf, 0x12, 0xf3,
			0x95, 0xfb, 0x80, 0x4a, 0x47, 0x55, 0xb5, 0x6b,
		},
	},
	/* SHA384 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0x2c, 0x07, 0xa8, 0x1d, 0xe5, 0x89, 0x55, 0xb6,
			0x76, 0xfe, 0xc0, 0x57, 0x2d, 0x48, 0xd1, 0x95,
			0x5b, 0x48, 0x75, 0xff, 0x62, 0xa4, 0x4b, 0x00,
			0x10, 0xc7, 0xa1, 0x07, 0x2b, 0x29, 0x9e, 0xe4,
			0x4d, 0xd0, 0xc0, 0x76, 0xf2, 0x17, 0x8a, 0x83,
			0xd0, 0xae, 0x76, 0xe7, 0x67, 0xe2, 0x31, 0xf1,
			0xd8, 0x1e, 0x07, 0x0a, 0xfa, 0xb2, 0x9c, 0x97,
			0xab, 0xd4, 0xde, 0x21, 0x64, 0xe4, 0x37, 0xb3,
			0x11, 0xf5, 0x07, 0x84, 0x1f, 0x88, 0x51, 0xd6,
			0xd6, 0x9a, 0xb5, 0x1e, 0xe9, 0xe2, 0x9e, 0x65,
			0x4b, 0x54, 0xbc, 0xee, 0x45, 0xe9, 0xb5, 0x19,
			0xc6, 0xa2, 0x17, 0x87, 0xfa, 0xcb, 0x92, 0x7f,
			0x1d, 0x7d, 0x64, 0x91, 0x92, 0x66, 0x14, 0x79,
			0x2f, 0xcc, 0x63, 0x46, 0xdc, 0xd0, 0x80, 0xbb,
			0x5c, 0xf0, 0x7b, 0xf5, 0x6a, 0xd0, 0xfc, 0x4e,
			0x08, 0x3a, 0x35, 0x82, 0x14, 0x63, 0x15, 0x10,
		},
		/* S6 : 256 byte */
		{
			0x9a, 0xa3, 0x91, 0xe7, 0xc2, 0xf0, 0xe9, 0x20,
			0xaa, 0xc2, 0x7e, 0xd9, 0xfc, 0x20, 0x81, 0xd3,
			0xc9, 0xca, 0xa3, 0x73, 0x58, 0x83, 0xd0, 0x1a,
			0xd7, 0xa7, 0xe3, 0xb1, 0x18, 0x67, 0xd0, 0xad,
			0x62, 0x41, 0x56, 0x47, 0x7b, 0xbb, 0xdd, 0xe6,
			0x59, 0xf4, 0x74, 0x68, 0x2d, 0x0d, 0x77, 0x44,
			0x89, 0xe2, 0xb5, 0xb0, 0x39, 0xd1, 0xeb, 0x35,
			0x45, 0x4c, 0x9e, 0x3e, 0xed, 0x78, 0xcf, 0xf9,
			0xc4, 0x26, 0x2e, 0x3a, 0xec, 0xfc, 0xa1, 0xd8,
			0x17, 0x54, 0x2b, 0x48, 0x60, 0x96, 0x59, 0x8e,
			0x11, 0x14, 0xbf, 0xc0, 0x3f, 0x20, 0xa4, 0x5d,
			0xe3, 0x6f, 0x6d, 0xf7, 0x0d, 0x14, 0x4d, 0x01,
			0xdc, 0x48, 0x66, 0xa0, 0xf8, 0x33, 0x19, 0xe7,
			0xc2, 0xb8, 0x53, 0x0f, 0x8c, 0x27, 0xa4, 0x1b,
			0x7a, 0xdd, 0x9f, 0x69, 0x2d, 0x8a, 0x8e, 0x64,
			0x64, 0x55, 0xb6, 0x7c, 0x9e, 0xc4, 0x7a, 0x4d,
			0x2c, 0xe3, 0xdf, 0xe3, 0x5d, 0x6a, 0x2e, 0x89,
			0xd9, 0xbe, 0x50, 0xc5, 0xb6, 0xda, 0x39, 0xbb,
			0x02, 0x54, 0xbd, 0x23, 0xa8, 0x09, 0xab, 0x97,
			0xb2, 0xb4, 0x8a, 0x06, 0x8a, 0x87, 0xab, 0xde,
			0x6b, 0x6a, 0x6e, 0x35, 0x95, 0x5f, 0xc9, 0x2a,
			0x96, 0x26, 0xf9, 0x60, 0x7d, 0x5b, 0x3f, 0x40,
			0x15, 0x17, 0x27, 0x15, 0x94, 0xbe, 0xf7, 0x38,
			0x59, 0x81, 0x2b, 0x6a, 0x62, 0x1e, 0xd6, 0xbd,
			0xaf, 0x3c, 0x5f, 0x2a, 0x90, 0xb1, 0xe1, 0x68,
			0x0f, 0x68, 0xdc, 0xfc, 0xca, 0xcb, 0x65, 0xe0,
			0x08, 0x1f, 0x1c, 0xcb, 0x6a, 0x20, 0x73, 0x70,
			0x9d, 0x1b, 0xa0, 0x67, 0x06, 0x50, 0x16, 0xed,
			0x73, 0xeb, 0xd7, 0xeb, 0xe9, 0xe7, 0xa7, 0xb6,
			0x0c, 0x8c, 0x9d, 0xd0, 0x4a, 0x56, 0xfa, 0xb3,
			0x07, 0x02, 0xc8, 0xa6, 0xdf, 0x6a, 0x35, 0x3a,
			0x30, 0x10, 0x47, 0xdf, 0x4c, 0x7a, 0xff, 0x62,
		},
	},
	/* SHA384 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0x35, 0xec, 0x92, 0xaf, 0xdb, 0xc2, 0xfc, 0xef,
			0xe4, 0x8f, 0x1e, 0x2f, 0x6e, 0x48, 0x29, 0xae,
			0x53, 0xb3, 0xda, 0x04, 0x59, 0xcc, 0x4e, 0xa8,
			0xa9, 0x68, 0x18, 0xb5, 0x83, 0x18, 0x91, 0xee,
			0x2f, 0x50, 0x6f, 0xff, 0x37, 0xc8, 0x99, 0x06,
			0xd3, 0x23, 0x3a, 0x51, 0xa5, 0xcf, 0x14, 0x69,
			0xa6, 0x2c, 0x18, 0x50, 0x61, 0xf0, 0x33, 0x08,
			0x5f, 0xca, 0x6a, 0x54, 0xe2, 0x45, 0x29, 0xc3,
			0xd6, 0xf0, 0xd8, 0xe9, 0x04, 0xbc, 0xb0, 0xf0,
			0x89, 0xa5, 0xcd, 0x50, 0x86, 0x94, 0x84, 0xda,
			0x1a, 0x84, 0xf6, 0xfb, 0x8d, 0xe4, 0xe5, 0x3f,
			0xce, 0x3d, 0xc7, 0x14, 0x20, 0x15, 0x19, 0xd1,
			0x10, 0x13, 0xf6, 0xf6, 0xaa, 0x64, 0xe8, 0xb5,
			0xec, 0x5c, 0xfe, 0xb2, 0x7b, 0x61, 0x1f, 0x08,
			0x95, 0x05, 0x9d, 0x8c, 0x47, 0x72, 0x0d, 0x55,
			0xe0, 0x0b, 0x57, 0x7c, 0xa5, 0x50, 0x09, 0x20,
		},
		/* S7 : 256 byte */
		{
			0x6b, 0x0f, 0x5b, 0x50, 0xe6, 0x78, 0xda, 0x08,
			0x3e, 0xd0, 0xf1, 0xb6, 0x4e, 0x94, 0x3e, 0x8c,
			0x62, 0x79, 0xc7, 0x24, 0x6a, 0xf5, 0xad, 0x07,
			0x9c, 0xdb, 0xf2, 0x23, 0xe4, 0x2a, 0x0d, 0x47,
			0x1e, 0x56, 0x31, 0x4b, 0xc0, 0xd5, 0x8f, 0x20,
			0x2a, 0xa6, 0xc5, 0xe1, 0xe5, 0x25, 0x59, 0x85,
			0xb0, 0x79, 0x5d, 0x48, 0xeb, 0x3d, 0x4b, 0x8e,
			0x3f, 0xc9, 0x22, 0x40, 0xae, 0x02, 0xb4, 0x08,
			0x8c, 0x6c, 0xe8, 0xab, 0x0e, 0x8c, 0x79, 0xc6,
			0x8d, 0xfd, 0xc4, 0x86, 0x57, 0xd6, 0xa2, 0x82,
			0x95, 0x39, 0x1b, 0x9a, 0x5a, 0x5f, 0x35, 0x25,
			0x51, 0x26, 0xbf, 0x8c, 0xa5, 0x3c, 0xbc, 0xc0,
			0x08, 0x2e, 0xab, 0x52, 0xec, 0x10, 0x9d, 0x22,
			0xa1, 0x18, 0x5f, 0x6d, 0xc7, 0x92, 0xfc, 0x29,
			0x0a, 0xa8, 0xdb, 0xae, 0xbb, 0x2f, 0xbe, 0x40,
			0x4f, 0x1d, 0x03, 0x9a, 0xa6, 0x34, 0x3c, 0xd7,
			0xaf, 0x9f, 0xcb, 0x2d, 0x1e, 0x05, 0xde, 0xf4,
			0x80, 0x96, 0xc2, 0x37, 0xe1, 0x0d, 0xaa, 0x7c,
			0xfa, 0xc5, 0xae, 0x9b, 0x3b, 0x30, 0x22, 0x00,
			0x5d, 0x0d, 0x2d, 0x5c, 0x9c, 0x5c, 0x50, 0x2b,
			0x2f, 0x23, 0x59, 0x4e, 0x80, 0xd1, 0x60, 0x4b,
			0xbb, 0x8f, 0x5d, 0xec, 0x07, 0xcd, 0x3a, 0xfe,
			0x1f, 0x77, 0x77, 0x43, 0xb0, 0xb5, 0x8a, 0x4e,
			0x0e, 0x4e, 0x5c, 0xaa, 0x14, 0x88, 0x30, 0xee,
			0xe0, 0x47, 0x96, 0x8e, 0x7f, 0x40, 0x66, 0x1f,
			0x9f, 0x1a, 0x02, 0xe1, 0xa7, 0xfd, 0x2b, 0x6c,
			0xaf, 0x19, 0x32, 0x6a, 0x75, 0xe9, 0x56, 0x5e,
			0xfd, 0xc0, 0x11, 0x4b, 0xce, 0xcb, 0x14, 0xdd,
			0xa0, 0x6c, 0x32, 0x9c, 0xf3, 0x22, 0xa5, 0xbd,
			0x3e, 0x6a, 0xb4, 0x8d, 0x95, 0xf2, 0xd2, 0xa9,
			0xc1, 0xc1, 0x23, 0x3a, 0x0a, 0xa0, 0x15, 0xa7,
			0x38, 0xf9, 0x01, 0xf1, 0x31, 0x48, 0xb4, 0x54,
		},
	},
	/* SHA384 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0x80, 0xc9, 0xde, 0xbd, 0xf9, 0x31, 0x74, 0xd7,
			0x57, 0x50, 0xa6, 0xcf, 0x09, 0xaf, 0x71, 0xfc,
			0x18, 0xfd, 0x51, 0x3b, 0xff, 0x9c, 0xb4, 0x91,
			0xbe, 0x60, 0xaf, 0x11, 0x2a, 0x93, 0xf0, 0x00,
			0x87, 0x3c, 0xf4, 0x38, 0x58, 0xa0, 0x7a, 0xca,
			0x76, 0x0a, 0x37, 0xe7, 0x60, 0xc8, 0xcb, 0x01,
			0xd2, 0x76, 0xf4, 0x2d, 0x99, 0x7f, 0x01, 0xcc,
			0xa5, 0xe0, 0x8a, 0x6a, 0x60, 0x2f, 0x5f, 0xe6,
			0x3e, 0xdc, 0xbe, 0xd3, 0x95, 0xb8, 0xc9, 0x1f,
			0xb0, 0xb3, 0x36, 0xf2, 0x1f, 0xea, 0x49, 0xd9,
			0x50, 0xe1, 0xff, 0x24, 0x64, 0x0c, 0x8d, 0x8d,
			0x3b, 0x95, 0x08, 0x1a, 0xd1, 0x59, 0x66, 0x44,
			0xce, 0x34, 0xa5, 0x58, 0x58, 0x7e, 0x4a, 0x1e,
			0x2c, 0xd5, 0x0d, 0xb9, 0xed, 0x1d, 0xd3, 0xce,
			0xbb, 0xc6, 0xdc, 0xe8, 0x08, 0x4d, 0x3e, 0x1b,
			0xa7, 0x06, 0x92, 0xe8, 0x26, 0x18, 0xed, 0x61,
		},
		/* S8 : 256 byte */
		{
			0x4a, 0x15, 0xa7, 0x83, 0xad, 0xbf, 0x27, 0x46,
			0x22, 0xd5, 0xa6, 0x10, 0xbb, 0x6f, 0xc7, 0x33,
			0x37, 0x99, 0x9e, 0x44, 0x5d, 0xc2, 0x13, 0x3a,
			0xcc, 0xb7, 0x88, 0xd6, 0x20, 0x3d, 0x70, 0xf3,
			0xcd, 0xc6, 0x3e, 0x67, 0xda, 0xa4, 0x17, 0x1a,
			0x79, 0x52, 0xa4, 0x98, 0x64, 0x56, 0xfa, 0xb3,
			0xc0, 0x77, 0xa8, 0x94, 0x1f, 0xb2, 0x59, 0xe3,
			0x7a, 0x5c, 0x0c, 0xbb, 0x20, 0xc4, 0x08, 0xfa,
			0x24, 0xad, 0x0e, 0xc8, 0x50, 0xe9, 0xbf, 0x02,
			0x8c, 0x36, 0x04, 0x60, 0x99, 0x41, 0xf5, 0xae,
			0x2f, 0x18, 0xbf, 0x1a, 0xc3, 0x7a, 0x24, 0xf7,
			0x55, 0xab, 0xb9, 0xc8, 0x5d, 0xdc, 0xd0, 0xbf,
			0x4a, 0x12, 0xfa, 0xbd, 0x9d, 0x25, 0x30, 0x29,
			0xe0, 0x81, 0xf6, 0x28, 0xe2, 0xbb, 0xe9, 0xf9,
			0xaf, 0xe9, 0x22, 0x49, 0x54, 0xd8, 0x31, 0x5d,
			0xb8, 0x6c, 0x21, 0x25, 0x51, 0x2b, 0xb9, 0x8c,
			0xe9, 0xb3, 0x69, 0x30, 0x99, 0x4b, 0x09, 0x1a,
			0x8a, 0x1d, 0x7d, 0x4e, 0x2f, 0x4a, 0x0e, 0x58,
			0xd0, 0xa3, 0x58, 0x76, 0xad, 0xad, 0x14, 0x30,
			0x05, 0x30, 0xb3, 0x9c, 0x8d, 0xc1, 0x1d, 0xed,
			0x3e, 0xf2, 0xfa, 0x95, 0xd5, 0xf2, 0x2e, 0x67,
			0xca, 0xe3, 0x4c, 0xc2, 0x1a, 0xd5, 0xe2, 0x3f,
			0x91, 0x22, 0xb5, 0x3d, 0xfb, 0x79, 0xf1, 0xa2,
			0xac, 0x63, 0xc1, 0x84, 0x4e, 0x9e, 0xf0, 0x69,
			0xa2, 0xe4, 0x1f, 0x17, 0x8d, 0x6d, 0xce, 0xdc,
			0x51, 0x8a, 0xaf, 0xcf, 0x81, 0xe0, 0xeb, 0xd8,
			0x82, 0x55, 0x6e, 0x73, 0x1c, 0xb0, 0xab, 0x41,
			0xd9, 0x57, 0x27, 0x4a, 0x3f, 0xbb, 0xb7, 0xce,
			0xf2, 0x60, 0x87, 0x91, 0x00, 0x0c, 0x6b, 0x86,
			0x08, 0x68, 0xcb, 0x73, 0x93, 0xe7, 0xd0, 0x3d,
			0x94, 0x56, 0x89, 0xff, 0xb7, 0x75, 0x55, 0xef,
			0xe0, 0x8f, 0x46, 0x14, 0x51, 0xd3, 0x3c, 0x11,
		},
	},
	/* SHA384 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0x31, 0x39, 0x5c, 0xef, 0x34, 0x95, 0x51, 0x34,
			0x3a, 0x49, 0x27, 0x1a, 0x8d, 0x81, 0x2b, 0x4c,
			0x7b, 0x65, 0xb4, 0x55, 0xb7, 0xed, 0xa8, 0x11,
			0xfc, 0xf7, 0x41, 0x61, 0xf3, 0x97, 0x11, 0x23,
			0x57, 0xae, 0x44, 0x62, 0x57, 0xbe, 0x26, 0xc9,
			0x3c, 0xfc, 0xe5, 0x5e, 0x4b, 0xa7, 0x97, 0x6d,
			0xed, 0x99, 0x7e, 0xc1, 0x0d, 0x1c, 0x8b, 0x1a,
			0xc2, 0xfe, 0x22, 0xdc, 0x2e, 0xe8, 0x1d, 0x05,
			0xa6, 0xeb, 0x13, 0x61, 0x12, 0x5c, 0xda, 0x01,
			0x97, 0xe2, 0x4a, 0xe9, 0x74, 0xcd, 0x44, 0x09,
			0x2a, 0xa9, 0xf3, 0x6f, 0xe0, 0x13, 0x52, 0xba,
			0x05, 0xcc, 0xef, 0xd2, 0x37, 0x0c, 0xee, 0xd6,
			0x64, 0x19, 0x50, 0x56, 0x2f, 0x17, 0x76, 0xc3,
			0x95, 0x22, 0xe0, 0x23, 0xd0, 0x9a, 0x3b, 0x09,
			0x7b, 0xbe, 0x9b, 0xc5, 0xf8, 0x7d, 0x05, 0xd8,
			0x0f, 0x88, 0x30, 0xab, 0xd7, 0xac, 0x8c, 0x80,
		},
		/* S9 : 256 byte */
		{
			0x16, 0x2f, 0x38, 0x76, 0x95, 0xcf, 0x9d, 0x82,
			0xdd, 0xa8, 0x9c, 0x74, 0x93, 0x18, 0xe4, 0x6c,
			0x9b, 0xe8, 0x95, 0xec, 0x36, 0x4e, 0xa4, 0xae,
			0xce, 0x97, 0xcc, 0xfa, 0x63, 0x92, 0x5a, 0xf3,
			0x71, 0x08, 0x94, 0xda, 0x2b, 0x7b, 0x59, 0x67,
			0xe4, 0x6f, 0x4e, 0xfa, 0x80, 0xca, 0x25, 0xd2,
			0xa9, 0x65, 0xa7, 0xe1, 0x5f, 0x75, 0xe0, 0xaa,
			0x1b, 0xd4, 0x25, 0x0f, 0x8f, 0x41, 0x09, 0x9e,
			0x6e, 0x97, 0x14, 0xc3, 0xfc, 0x43, 0x11, 0x07,
			0x7a, 0xe9, 0xbd, 0xdf, 0xe3, 0x5b, 0xa4, 0x72,
			0x75, 0x31, 0x52, 0x9c, 0x23, 0x9d, 0x54, 0x6a,
			0xb1, 0xc2, 0x98, 0x18, 0x7f, 0x16, 0x5f, 0x70,
			0x8c, 0xcc, 0x0a, 0xe3, 0x97, 0x9a, 0x8d, 0xa1,
			0x93, 0xe3, 0x48, 0x59, 0xa5, 0x9c, 0x2c, 0x3b,
			0xc4, 0x22, 0x53, 0xc8, 0x34, 0x66, 0x88, 0xe6,
			0xbb, 0xa6, 0xfb, 0x1b, 0x01, 0xb1, 0x0c, 0x1e,
			0xc2, 0xc6, 0x49, 0x3d, 0xed, 0xcc, 0x26, 0x96,
			0x26, 0x9d, 0x85, 0x1b, 0xde, 0x63, 0xe2, 0x7e,
			0x37, 0xbe, 0xd3, 0x57, 0x45, 0x5c, 0x8f, 0xee,
			0x56, 0x29, 0xf9, 0x4a, 0xfa, 0x7a, 0x98, 0x66,
			0x95, 0xcf, 0xd5, 0xb9, 0x92, 0x12, 0x65, 0x7a,
			0x6c, 0x88, 0x46, 0x44, 0x59, 0x60, 0x86, 0xb8,
			0x9e, 0x0c, 0x7c, 0x05, 0xe8, 0x19, 0xfa, 0xeb,
			0xeb, 0xef, 0x74, 0x5f, 0xd2, 0x95, 0xaf, 0x88,
			0x66, 0xe0, 0x75, 0x0f, 0x54, 0x79, 0xba, 0xed,
			0x50, 0xcb, 0xb3, 0xd0, 0x59, 0xf8, 0xa5, 0xeb,
			0x7e, 0x0e, 0x61, 0xe2, 0x73, 0x3a, 0xe5, 0x0f,
			0x0c, 0x1e, 0xc4, 0x2b, 0xe7, 0x1f, 0x5d, 0xff,
			0x32, 0x41, 0x95, 0xcb, 0x4f, 0x0e, 0x94, 0x1a,
			0x21, 0x56, 0x15, 0x13, 0xc3, 0x03, 0x7d, 0xb9,
			0x2f, 0xec, 0x95, 0x56, 0xb7, 0x72, 0xcc, 0xab,
			0x23, 0x9e, 0x34, 0xb1, 0x87, 0x6c, 0x56, 0xb1,
		},
	},
};

/* 10 elements */
static mod2048_t SHA512[] = {
	/* SHA512 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0xa7, 0xc3, 0x09, 0xd4, 0x4a, 0x57, 0x18, 0x8b,
			0xbd, 0x7b, 0x72, 0x6b, 0x98, 0xb9, 0x8c, 0xe1,
			0x25, 0x82, 0x22, 0x8e, 0x14, 0x15, 0x86, 0x48,
			0x70, 0xa2, 0x39, 0x61, 0xd2, 0xaf, 0xb8, 0x2c,
			0xd5, 0xbc, 0x98, 0xbe, 0xc9, 0x22, 0xd5, 0xf2,
			0xac, 0x41, 0x68, 0xb0, 0x56, 0xda, 0x17, 0x6e,
			0xf3, 0xba, 0x91, 0xf6, 0xb6, 0x99, 0xba, 0x6a,
			0xcc, 0x41, 0x44, 0x86, 0x8f, 0xf3, 0x7f, 0x26,
			0xfd, 0x06, 0x72, 0x08, 0x68, 0xd1, 0x2a, 0xd2,
			0x6e, 0xcb, 0x52, 0x57, 0x2c, 0xf1, 0x04, 0x16,
			0xaf, 0x68, 0xdf, 0x03, 0xab, 0x64, 0x5a, 0x8b,
			0x70, 0x48, 0x57, 0xd2, 0x19, 0x0f, 0xfc, 0x3f,
			0x07, 0xea, 0xbe, 0x3a, 0x8e, 0x2a, 0xbe, 0x34,
			0xed, 0x61, 0x59, 0xe8, 0x84, 0xc4, 0xfa, 0xe1,
			0x41, 0xd4, 0x33, 0x3d, 0x5c, 0x3e, 0x0d, 0xb0,
			0x44, 0xff, 0x9c, 0xcc, 0xd9, 0xcb, 0xd6, 0x7f,
		},
		/* S0 : 256 byte */
		{
			0x14, 0x8a, 0xf6, 0x1e, 0xd5, 0xea, 0x8a, 0x87,
			0xa0, 0x8b, 0x3f, 0x40, 0x39, 0x29, 0xbf, 0x80,
			0x31, 0xdb, 0x4f, 0xd3, 0x99, 0x9b, 0x64, 0x40,
			0x9b, 0xa4, 0x89, 0xf9, 0x7a, 0x3e, 0xe5, 0x20,
			0x8e, 0xa4, 0x20, 0x2d, 0x2e, 0xc1, 0x87, 0x34,
			0xf6, 0x15, 0x00, 0x3a, 0x51, 0xf7, 0x74, 0x41,
			0x08, 0x5b, 0xe6, 0xac, 0x0f, 0x11, 0x81, 0x0f,
			0xfa, 0x2d, 0xad, 0x58, 0xf0, 0xe1, 0x86, 0xd5,
			0x52, 0x0a, 0xc2, 0xb8, 0xa5, 0xd3, 0x96, 0x6e,
			0x8d, 0x2a, 0xbb, 0x80, 0x74, 0xe1, 0x3b, 0x50,
			0xa4, 0xe7, 0xde, 0x83, 0xbe, 0x10, 0xa6, 0x6f,
			0xdc, 0x7c, 0xa1, 0x81, 0x18, 0xc5, 0x77, 0x4f,
			0x78, 0x12, 0x12, 0xde, 0x9e, 0xfe, 0xbc, 0x63,
			0x76, 0xfc, 0xdd, 0xdc, 0x65, 0xa3, 0xb1, 0xb8,
			0xf1, 0xab, 0x31, 0x49, 0x2f, 0xe4, 0x78, 0x25,
			0x9c, 0xe7, 0x19, 0xb3, 0xdb, 0x58, 0x74, 0x98,
			0xd8, 0x79, 0xa0, 0x1d, 0xec, 0x96, 0xe8, 0xea,
			0xbe, 0xb0, 0x7f, 0xf7, 0x07, 0x3f, 0x3f, 0x3e,
			0xb4, 0x46, 0x08, 0x49, 0x55, 0xca, 0x26, 0x32,
			0x9a, 0x79, 0x13, 0x15, 0xa2, 0xc2, 0x59, 0xd2,
			0x25, 0xe2, 0x6b, 0x21, 0x54, 0xb2, 0x04, 0x7b,
			0x21, 0xfa, 0xba, 0x68, 0x11, 0x5b, 0xfd, 0x96,
			0x2e, 0x5e, 0x24, 0xec, 0x52, 0xd7, 0xc5, 0xd2,
			0x31, 0xe3, 0x04, 0x4c, 0xbc, 0xd8, 0xc8, 0x80,
			0x48, 0x55, 0x70, 0x3c, 0xba, 0xa6, 0x22, 0xb1,
			0x5b, 0x6e, 0xf7, 0x8c, 0x74, 0x21, 0xa3, 0x67,
			0x16, 0x6f, 0x1b, 0x02, 0x57, 0x6c, 0x87, 0x36,
			0x05, 0x93, 0xda, 0x75, 0xb7, 0x18, 0x9e, 0xfa,
			0xfd, 0x10, 0x82, 0xbd, 0x59, 0xf6, 0x85, 0x7f,
			0x17, 0x01, 0xf6, 0x46, 0xc2, 0x4d, 0x70, 0xc9,
			0x52, 0x73, 0xc4, 0x9d, 0x5b, 0x11, 0xe6, 0xaf,
			0xe2, 0x58, 0x82, 0x1b, 0x55, 0xc1, 0x68, 0x0c,
		},
	},
	/* SHA512 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0xca, 0x50, 0x5d, 0x45, 0x91, 0x12, 0x16, 0x64,
			0x99, 0x07, 0x47, 0xd9, 0x5d, 0x95, 0x55, 0xcc,
			0x75, 0xbf, 0xc3, 0xfd, 0xae, 0xec, 0xee, 0xaa,
			0x60, 0xea, 0xfa, 0xb3, 0xfc, 0x32, 0x0c, 0xfc,
			0xe5, 0x6e, 0xb9, 0x13, 0x81, 0x38, 0xbf, 0x13,
			0x8f, 0x25, 0xf3, 0xc8, 0xbb, 0x02, 0x7b, 0x13,
			0x6f, 0x5d, 0x3d, 0x90, 0xed, 0x48, 0x97, 0x77,
			0x9b, 0x59, 0x51, 0xc0, 0x9d, 0xf5, 0xd0, 0x8b,
			0xa9, 0xce, 0x8c, 0xbe, 0x17, 0xab, 0xc4, 0xf0,
			0x38, 0x68, 0x70, 0x86, 0xe9, 0x3d, 0x77, 0x1b,
			0x68, 0x43, 0x22, 0x26, 0x66, 0x33, 0xd0, 0xd6,
			0x5d, 0x71, 0xec, 0x41, 0x23, 0x4a, 0x1d, 0xbe,
			0xc0, 0x7a, 0xbc, 0x8f, 0x7d, 0xf2, 0x8b, 0xc4,
			0x3d, 0xd8, 0xa4, 0x5b, 0x10, 0xce, 0xaf, 0xac,
			0x06, 0x77, 0x58, 0x05, 0x41, 0x37, 0x01, 0x91,
			0x4e, 0x3b, 0xb3, 0x7e, 0xb6, 0xba, 0x5b, 0x5e,
		},
		/* S1 : 256 byte */
		{
			0x58, 0x9c, 0xcd, 0x4e, 0xbf, 0x97, 0x64, 0xf8,
			0x7e, 0x6a, 0xfa, 0x7f, 0x13, 0xc4, 0x06, 0x25,
			0x79, 0xb0, 0x22, 0x28, 0x11, 0x7b, 0x15, 0xa8,
			0x73, 0x8a, 0xb3, 0x9c, 0xd6, 0x44, 0x77, 0x06,
			0x9c, 0xb4, 0xf5, 0x2c, 0xd8, 0xd5, 0xf4, 0x57,
			0x4c, 0x65, 0x7b, 0x45, 0x38, 0x35, 0xca, 0x3c,
			0xed, 0xb8, 0x24, 0xf0, 0x3b, 0x92, 0xa5, 0x73,
			0xd6, 0xd3, 0xd9, 0x13, 0x61, 0x31, 0x3f, 0x11,
			0xbd, 0xcb, 0x34, 0xd2, 0x05, 0x9f, 0xe2, 0xe6,
			0xce, 0x2b, 0x85, 0x44, 0x61, 0xaf, 0x58, 0xa9,
			0x29, 0x4c, 0x88, 0xcb, 0xfb, 0x2a, 0x63, 0x99,
			0x76, 0xb5, 0x6e, 0x47, 0x48, 0x02, 0x6f, 0x30,
			0x40, 0xe2, 0xfd, 0x71, 0x12, 0xd6, 0xad, 0x44,
			0x50, 0x06, 0x89, 0xac, 0x77, 0x7c, 0x07, 0x1d,
			0x17, 0x39, 0x19, 0x69, 0x76, 0x2e, 0x18, 0x64,
			0x17, 0xc4, 0x40, 0x0a, 0xbd, 0xda, 0x5c, 0x16,
			0xdc, 0xe0, 0x07, 0x76, 0x42, 0xf1, 0xfc, 0x13,
			0x54, 0xe0, 0xe8, 0xc1, 0x4e, 0x55, 0x8c, 0x92,
			0x3c, 0x1b, 0xfb, 0x85, 0x48, 0x8b, 0x83, 0x50,
			0xf4, 0x15, 0x86, 0x6a, 0x60, 0x87, 0x1e, 0xd7,
			0x15, 0x1f, 0x5f, 0xbc, 0x5b, 0x88, 0x05, 0x00,
			0x01, 0x19, 0x77, 0xc7, 0x78, 0xe1, 0x7f, 0xe8,
			0x91, 0x8c, 0x5d, 0x34, 0x3f, 0x70, 0xb0, 0x0d,
			0x58, 0xf7, 0x18, 0x95, 0x61, 0x25, 0xfe, 0x28,
			0xb3, 0xa5, 0xe2, 0xd0, 0x76, 0x04, 0xa2, 0xb8,
			0xa8, 0x77, 0x20, 0x44, 0x34, 0xce, 0x90, 0x3b,
			0x35, 0xa0, 0x30, 0x93, 0x6b, 0xc7, 0x19, 0x51,
			0xca, 0x59, 0x3d, 0xf9, 0x7d, 0x24, 0xe8, 0xe8,
			0xad, 0x8f, 0x2d, 0xc9, 0xb7, 0x8f, 0x76, 0xef,
			0x13, 0xa1, 0xd3, 0x86, 0xca, 0x85, 0x7c, 0xed,
			0x48, 0xf1, 0x9f, 0x3e, 0xbe, 0x39, 0x10, 0x8f,
			0x9b, 0x33, 0xff, 0x59, 0xeb, 0x05, 0x56, 0xb1,
		},
	},
	/* SHA512 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x23, 0x7a, 0x7e, 0x44, 0xb0, 0xa6, 0xc2, 0x68,
			0xbb, 0x63, 0x36, 0x4b, 0x95, 0x8a, 0xe0, 0x2b,
			0x95, 0xe7, 0xee, 0xd3, 0x6b, 0x3e, 0xa5, 0xbf,
			0xb1, 0x8b, 0x9b, 0x81, 0xc3, 0x8e, 0x26, 0x63,
			0xd1, 0x87, 0x14, 0x4e, 0x32, 0x3f, 0x9c, 0xea,
			0xfb, 0x47, 0x95, 0x07, 0xd1, 0x84, 0xe6, 0x3c,
			0xfb, 0xec, 0x3e, 0xcd, 0xbb, 0x8a, 0x05, 0xd2,
			0xdf, 0xc8, 0x92, 0x96, 0x93, 0xed, 0x9e, 0x3e,
			0x79, 0xe5, 0xf8, 0xab, 0xfc, 0x41, 0x7b, 0xa1,
			0xe1, 0x7e, 0x3e, 0x28, 0x1e, 0x8a, 0x0a, 0x32,
			0xf0, 0x84, 0x11, 0x7f, 0x28, 0xc3, 0xdc, 0xbe,
			0xc5, 0x1b, 0x86, 0xf5, 0xc8, 0x5b, 0x28, 0x22,
			0x44, 0x1a, 0x94, 0x23, 0xb5, 0xb4, 0x46, 0xd3,
			0x92, 0x8f, 0x97, 0x76, 0x26, 0xa3, 0x34, 0x57,
			0x9b, 0x39, 0xcf, 0xaf, 0x58, 0xf2, 0x14, 0xc9,
			0x8d, 0x0c, 0xdf, 0x64, 0x0b, 0xe1, 0xac, 0x59,
		},
		/* S2 : 256 byte */
		{
			0xaf, 0x07, 0x6b, 0xc2, 0x13, 0xca, 0xf7, 0x56,
			0x19, 0xf4, 0xbd, 0x1d, 0x78, 0x7c, 0xc1, 0x98,
			0xf7, 0xdf, 0x33, 0x24, 0xa0, 0xdd, 0x87, 0xa8,
			0x84, 0x16, 0xe0, 0xa4, 0xb8, 0x1c, 0x2f, 0xb9,
			0xa9, 0xdb, 0x5f, 0x98, 0xae, 0xd4, 0x3b, 0xc1,
			0x5f, 0xe2, 0x35, 0x71, 0x43, 0xa6, 0xe4, 0xff,
			0x70, 0x1d, 0x9c, 0x48, 0xf5, 0x1d, 0xe9, 0xeb,
			0x80, 0x36, 0x70, 0xbb, 0xc4, 0xb0, 0xae, 0xa7,
			0x22, 0x0b, 0xe2, 0xf8, 0x4b, 0x83, 0x00, 0x31,
			0x8c, 0x77, 0xa9, 0xf6, 0x15, 0x98, 0x6c, 0x49,
			0x80, 0xab, 0xda, 0x85, 0xe3, 0xad, 0x00, 0x89,
			0x56, 0x4d, 0xba, 0xf7, 0xf4, 0x4d, 0x81, 0xb6,
			0x66, 0x4e, 0xec, 0x03, 0x11, 0xad, 0xb1, 0x94,
			0xd4, 0x6d, 0xe9, 0x6b, 0xb1, 0x7d, 0x5a, 0x5d,
			0x47, 0x42, 0x68, 0x45, 0x80, 0x2c, 0xa0, 0xf4,
			0x9a, 0x16, 0x9e, 0xb8, 0x2b, 0x75, 0xaf, 0xa1,
			0x91, 0x02, 0x7a, 0x0c, 0xc8, 0xfc, 0xe9, 0xdd,
			0x16, 0x05, 0x53, 0x50, 0xdf, 0x97, 0x45, 0xfc,
			0x72, 0x00, 0xff, 0x9f, 0x4e, 0xa3, 0xcf, 0xbf,
			0xc6, 0x6c, 0x42, 0x84, 0x81, 0x13, 0xe3, 0xbe,
			0x32, 0x93, 0xd5, 0x10, 0x38, 0x2d, 0x09, 0x99,
			0xf0, 0x32, 0x51, 0x55, 0x27, 0xbd, 0x99, 0xf6,
			0x6e, 0xfa, 0x2a, 0x75, 0x5e, 0x01, 0x12, 0x47,
			0xb2, 0x23, 0xa6, 0x8e, 0x51, 0x25, 0x8b, 0x6b,
			0xc3, 0x19, 0xa7, 0xcd, 0xef, 0x4a, 0xec, 0x53,
			0x3e, 0x9d, 0xcd, 0x8a, 0xe2, 0x6e, 0x34, 0x9e,
			0x5b, 0x33, 0xc7, 0x91, 0x21, 0x90, 0x7d, 0xe5,
			0x09, 0xa1, 0xcb, 0x83, 0xc2, 0xe5, 0x9a, 0x47,
			0xc1, 0xa8, 0x84, 0xbf, 0x68, 0xe7, 0x22, 0x93,
			0x16, 0xa6, 0x2e, 0x3c, 0x49, 0xd1, 0xf5, 0x42,
			0xeb, 0xe7, 0x10, 0x5c, 0xfc, 0x27, 0x09, 0x92,
			0x68, 0x12, 0x0a, 0x77, 0x43, 0x90, 0x84, 0x71,
		},
	},
	/* SHA512 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0xab, 0x18, 0x93, 0x92, 0x30, 0xb0, 0x96, 0x64,
			0x6a, 0x37, 0xa7, 0x81, 0x62, 0x9f, 0xbd, 0x92,
			0x70, 0xf3, 0x89, 0x1a, 0x5c, 0xea, 0xb4, 0xa8,
			0xc3, 0xbc, 0x68, 0x51, 0xbc, 0x34, 0x11, 0x5d,
			0xbc, 0x06, 0x65, 0x41, 0xb7, 0x64, 0xa2, 0xce,
			0x88, 0xcc, 0x16, 0xa7, 0x93, 0x24, 0xe5, 0xf8,
			0xa9, 0x08, 0x07, 0x65, 0x2c, 0x63, 0x90, 0x41,
			0x73, 0x3c, 0x34, 0x01, 0x6f, 0xd3, 0x0a, 0xf0,
			0x8f, 0xed, 0x90, 0x24, 0xe2, 0x6c, 0xf0, 0xb0,
			0x7c, 0x22, 0x81, 0x1b, 0x1a, 0xe7, 0x91, 0x11,
			0x09, 0xe9, 0x62, 0x59, 0x43, 0x44, 0x72, 0x07,
			0xdc, 0xd3, 0xff, 0xf3, 0x9c, 0x45, 0xcb, 0x69,
			0xee, 0x73, 0x1d, 0x22, 0xf8, 0xf0, 0x08, 0x73,
			0x0c, 0xe2, 0xef, 0xc5, 0x3f, 0x11, 0x49, 0x45,
			0x57, 0x3e, 0xa2, 0xdd, 0xeb, 0xb6, 0xe2, 0x62,
			0xc5, 0x27, 0xd2, 0x0f, 0x8b, 0xb1, 0xdc, 0x32,
		},
		/* S3 : 256 byte */
		{
			0x95, 0xbd, 0x0b, 0xf2, 0x36, 0x2f, 0x34, 0xb2,
			0xe0, 0x40, 0x75, 0xb2, 0x93, 0x4f, 0x40, 0x47,
			0x98, 0x70, 0x3e, 0xa4, 0x72, 0xb8, 0x1a, 0xc3,
			0xcc, 0x22, 0x3a, 0xec, 0x48, 0x6e, 0x4c, 0x3d,
			0x9c, 0x5d, 0x1c, 0x2f, 0x9e, 0xe2, 0x24, 0x17,
			0x13, 0x29, 0x64, 0xed, 0x58, 0xe4, 0x99, 0x37,
			0xf5, 0xb2, 0x57, 0xd3, 0x16, 0xca, 0x7f, 0xff,
			0xe2, 0x90, 0xb1, 0x9f, 0x5b, 0x58, 0x10, 0x38,
			0x36, 0x81, 0x2b, 0xef, 0x30, 0xca, 0x03, 0x27,
			0x03, 0x9d, 0x8b, 0x9e, 0xa9, 0x12, 0x95, 0x39,
			0x2f, 0xc3, 0x94, 0xb8, 0x81, 0xe2, 0xd2, 0xac,
			0x9e, 0x30, 0xc5, 0xa4, 0x42, 0x56, 0x70, 0x0f,
			0xc9, 0xde, 0x0d, 0xba, 0x29, 0x82, 0x73, 0xae,
			0xc3, 0x0c, 0x4f, 0x77, 0x8d, 0x2e, 0x71, 0x27,
			0xe8, 0xb8, 0xa8, 0x8b, 0x02, 0x74, 0xfc, 0xe0,
			0x40, 0x81, 0xcc, 0x13, 0xad, 0xbe, 0xfe, 0x55,
			0x50, 0x14, 0xe1, 0xb5, 0xd5, 0xdc, 0xf6, 0x22,
			0x4c, 0x5a, 0xe2, 0x77, 0x54, 0x23, 0xa6, 0x6c,
			0x81, 0x81, 0x8e, 0xec, 0x01, 0x4a, 0x3f, 0xaf,
			0x9e, 0xe7, 0x5a, 0x3f, 0x6c, 0x3e, 0x51, 0xc5,
			0x56, 0xb0, 0xa2, 0x88, 0xe8, 0xc2, 0x62, 0x94,
			0x66, 0x84, 0xeb, 0x62, 0x8b, 0x88, 0xe3, 0xf8,
			0x75, 0xe6, 0x2e, 0xf6, 0xe8, 0x01, 0xca, 0xe7,
			0x5f, 0x61, 0xce, 0xe4, 0x04, 0x97, 0x1c, 0x39,
			0xd2, 0x4a, 0x97, 0x12, 0xeb, 0x34, 0x2d, 0xdc,
			0x66, 0x35, 0x15, 0xde, 0xc1, 0x03, 0xb1, 0x8d,
			0x97, 0xd7, 0x8e, 0xd6, 0x82, 0x12, 0xf2, 0x79,
			0x00, 0xe7, 0x7c, 0x04, 0x9b, 0x60, 0xc8, 0x53,
			0x00, 0x2b, 0x08, 0x02, 0x2d, 0xf5, 0x6f, 0x70,
			0x7e, 0xfa, 0x71, 0x02, 0x75, 0x89, 0xe1, 0xa3,
			0xca, 0x6e, 0x41, 0x5b, 0xa5, 0xf4, 0x43, 0x7e,
			0x97, 0x8b, 0x07, 0xaf, 0x3b, 0x73, 0xba, 0x0d,
		},
	},
	/* SHA512 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0xa2, 0x80, 0xe8, 0x9c, 0xeb, 0x2c, 0x8c, 0xf2,
			0x62, 0x97, 0x19, 0x1b, 0xaf, 0x9a, 0x95, 0x5d,
			0x0d, 0x52, 0x37, 0x5d, 0xa0, 0x23, 0x63, 0x3e,
			0x0a, 0xfc, 0xdb, 0x0d, 0x39, 0xdc, 0x33, 0x5d,
			0x82, 0x95, 0x85, 0x2e, 0xf4, 0xd0, 0x67, 0x14,
			0xe6, 0x51, 0x1a, 0x95, 0xd3, 0x7c, 0x04, 0xd2,
			0x68, 0x18, 0x60, 0x6a, 0xda, 0x54, 0x35, 0x9b,
			0x7d, 0x07, 0x84, 0xaa, 0x93, 0x3c, 0xc6, 0x85,
			0x61, 0xee, 0x96, 0xa8, 0x89, 0x10, 0xaa, 0x3d,
			0x93, 0xd1, 0x07, 0x87, 0xcd, 0x1d, 0x75, 0x80,
			0x55, 0x67, 0x31, 0xc1, 0x74, 0xa6, 0xe3, 0xa3,
			0x2d, 0x9d, 0xcf, 0xa4, 0x16, 0x60, 0x4f, 0x0c,
			0x67, 0x14, 0x81, 0xd0, 0x51, 0xf6, 0x3d, 0xb6,
			0x91, 0x9f, 0x4a, 0xba, 0x44, 0x86, 0xd1, 0xb0,
			0xfd, 0xc6, 0x11, 0x2c, 0x15, 0x21, 0x55, 0x9f,
			0x42, 0x45, 0x23, 0xc2, 0x6b, 0x4f, 0xb7, 0x38,
		},
		/* S4 : 256 byte */
		{
			0xcd, 0x60, 0xde, 0x3b, 0x4a, 0x12, 0x89, 0xa8,
			0x4c, 0xa7, 0x61, 0xf9, 0x0f, 0xa6, 0x3f, 0x4d,
			0x56, 0x88, 0xbd, 0x88, 0x5f, 0x4b, 0x53, 0x1c,
			0x85, 0x15, 0xad, 0xd2, 0xde, 0x12, 0x51, 0xf9,
			0x93, 0xff, 0x7f, 0x98, 0x6b, 0xef, 0x3f, 0xba,
			0x69, 0x2e, 0xcd, 0xeb, 0xc8, 0x19, 0x42, 0xd7,
			0x42, 0x9c, 0x7a, 0x59, 0xc5, 0xd3, 0xf1, 0xfb,
			0x87, 0x2f, 0xc1, 0xda, 0x19, 0x15, 0xe9, 0x45,
			0x86, 0xa5, 0xc3, 0xd9, 0x63, 0x60, 0x36, 0x19,
			0x00, 0x8f, 0x7e, 0xfe, 0xde, 0xd1, 0xd7, 0x0b,
			0x0a, 0x11, 0xce, 0x2c, 0xd8, 0x1b, 0x5b, 0x0d,
			0x86, 0xb3, 0x76, 0x0c, 0x94, 0x83, 0x67, 0x4f,
			0x55, 0xe9, 0xfa, 0x47, 0xf2, 0xf3, 0x10, 0xd5,
			0x88, 0xfb, 0x21, 0x60, 0xe8, 0xb5, 0xc3, 0x2b,
			0xe4, 0xe7, 0xa9, 0x68, 0xd5, 0xa8, 0xd4, 0xac,
			0x65, 0x76, 0xb7, 0x1a, 0x2b, 0x91, 0xcd, 0x6a,
			0xf0, 0x01, 0x6c, 0xbc, 0x81, 0x6d, 0x4a, 0xae,
			0x8c, 0x70, 0x64, 0x9e, 0x08, 0xdc, 0xe9, 0x0b,
			0x3c, 0xe5, 0x2a, 0xb4, 0x9c, 0xe2, 0xcb, 0x5b,
			0x0e, 0xd8, 0xa4, 0x5e, 0x33, 0xd9, 0x4c, 0xf2,
			0xd4, 0xcf, 0xde, 0xe1, 0x15, 0x12, 0x70, 0xb2,
			0x07, 0x3a, 0xef, 0xfe, 0xaf, 0x71, 0x7d, 0x39,
			0xe0, 0x41, 0x92, 0xb8, 0xb6, 0x93, 0xc5, 0x3f,
			0x21, 0xa6, 0x12, 0x38, 0x13, 0x28, 0x08, 0x06,
			0x92, 0x0b, 0x7d, 0xc5, 0x82, 0x20, 0x1c, 0x9d,
			0x11, 0x70, 0x50, 0x32, 0x06, 0x71, 0xe8, 0x61,
			0x39, 0xa0, 0x27, 0x97, 0x6b, 0x7e, 0xcf, 0x41,
			0x33, 0x69, 0xa9, 0xfc, 0x28, 0xe0, 0xbd, 0x71,
			0x9c, 0xeb, 0x5e, 0x10, 0x7d, 0xe7, 0x99, 0xf1,
			0xbc, 0x2e, 0x25, 0x5a, 0x9f, 0x29, 0x47, 0x6d,
			0x45, 0x74, 0xd1, 0x33, 0x2f, 0x66, 0x46, 0x8a,
			0xfb, 0x90, 0x04, 0xff, 0x7b, 0x53, 0x53, 0x02,
		},
	},
	/* SHA512 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0x85, 0xed, 0x1e, 0x3d, 0xfc, 0xd5, 0xbc, 0xa2,
			0x4c, 0xad, 0x1d, 0x01, 0xeb, 0xe1, 0x92, 0xb7,
			0xd0, 0x59, 0xec, 0x9b, 0x88, 0x44, 0x36, 0xe1,
			0x87, 0x14, 0xa4, 0x3f, 0xbc, 0xc9, 0xc6, 0x4f,
			0x68, 0x73, 0x01, 0x35, 0x2f, 0xf2, 0x40, 0x81,
			0x70, 0x01, 0xe7, 0x57, 0xd2, 0x73, 0x09, 0xcd,
			0x1f, 0xbb, 0xda, 0x94, 0x56, 0xb2, 0x67, 0xdb,
			0xfb, 0x95, 0x84, 0x70, 0xb2, 0x4d, 0x06, 0x28,
			0x0c, 0xf4, 0x33, 0x82, 0xa1, 0x94, 0x77, 0x87,
			0x5f, 0x32, 0x59, 0xf4, 0x21, 0x0b, 0xac, 0x9b,
			0x83, 0x1d, 0x0a, 0x07, 0xf5, 0xe9, 0x7e, 0x5f,
			0x0f, 0x78, 0x81, 0x8c, 0x25, 0x9c, 0x28, 0x9e,
			0x1a, 0x78, 0x9b, 0x6c, 0x79, 0x42, 0xc9, 0x7b,
			0xc1, 0x48, 0x5a, 0x22, 0x01, 0x31, 0xe5, 0xeb,
			0xa5, 0x86, 0x64, 0x3b, 0x90, 0x71, 0xe5, 0x36,
			0x6b, 0xc4, 0x82, 0xdd, 0x3c, 0x3c, 0x92, 0x79,
		},
		/* S5 : 256 byte */
		{
			0x13, 0x81, 0x34, 0xbb, 0xec, 0xef, 0xaf, 0xc7,
			0xca, 0x8b, 0x10, 0x2c, 0xbe, 0x87, 0xb0, 0x12,
			0xf8, 0xaa, 0xda, 0x88, 0x78, 0x99, 0x50, 0x02,
			0xcf, 0x18, 0x87, 0x69, 0x4b, 0x5b, 0xe3, 0xb8,
			0xf0, 0xbb, 0x61, 0x6b, 0xc6, 0xe0, 0x79, 0x62,
			0xd5, 0x48, 0x2d, 0x3a, 0x52, 0xc5, 0x2a, 0xb9,
			0x1b, 0x3e, 0xe0, 0x06, 0x4d, 0x24, 0x55, 0x8e,
			0x13, 0xc7, 0x5c, 0x80, 0xf6, 0xa9, 0x5b, 0x7d,
			0xc4, 0x98, 0x44, 0x28, 0x79, 0xd5, 0xba, 0xf8,
			0xff, 0xa7, 0xe2, 0xf6, 0x38, 0x80, 0x8b, 0x97,
			0xff, 0x70, 0x13, 0x6b, 0xb6, 0x45, 0xe3, 0x09,
			0x44, 0xdd, 0x97, 0xa9, 0x97, 0xa0, 0x20, 0x51,
			0x69, 0x55, 0x3a, 0x5b, 0x9e, 0x87, 0x4c, 0x5a,
			0x94, 0x41, 0xe1, 0x8c, 0x15, 0xeb, 0xed, 0x76,
			0x04, 0x3b, 0x63, 0x9d, 0xfd, 0x64, 0xdb, 0x79,
			0xe1, 0x74, 0x84, 0x7a, 0x10, 0x27, 0x24, 0xa2,
			0xa0, 0x5c, 0x64, 0x94, 0x73, 0xcc, 0x7d, 0xac,
			0xd3, 0x9e, 0x2e, 0x1d, 0x56, 0x66, 0xbb, 0xb5,
			0xf0, 0x12, 0x46, 0x74, 0x70, 0x48, 0xff, 0xfc,
			0xdf, 0xcd, 0xdf, 0x78, 0x2d, 0xa2, 0x4a, 0x6d,
			0xcc, 0x02, 0x2b, 0x26, 0x95, 0xf7, 0x07, 0x81,
			0xbd, 0x9f, 0x8f, 0xf7, 0xd0, 0x3b, 0xe2, 0x2e,
			0xb8, 0xfc, 0x79, 0x3f, 0x5c, 0x07, 0x1a, 0x66,
			0xd9, 0xa6, 0xea, 0x46, 0xc6, 0xa2, 0xcf, 0x05,
			0x56, 0x52, 0x6b, 0xa8, 0xb0, 0x85, 0x07, 0x35,
			0x46, 0x44, 0x80, 0x81, 0x73, 0x2a, 0xc1, 0x5f,
			0x12, 0x83, 0x3c, 0x1d, 0xb1, 0x70, 0x1f, 0xf7,
			0xf6, 0x83, 0x44, 0xca, 0x65, 0xdf, 0xf8, 0x62,
			0x11, 0xa0, 0x03, 0xad, 0xbf, 0x51, 0x89, 0xcf,
			0xae, 0x79, 0xea, 0xa8, 0xc8, 0xb7, 0x14, 0x1e,
			0xa3, 0x78, 0xe4, 0x4c, 0xc9, 0xc5, 0xbf, 0x02,
			0x4d, 0x2c, 0x71, 0x0f, 0xf5, 0xcd, 0x68, 0xaf,
		},
	},
	/* SHA512 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0x0b, 0xdb, 0xa3, 0x4e, 0x35, 0xfc, 0xa6, 0x5a,
			0x17, 0x81, 0xd4, 0xd7, 0xc9, 0x33, 0xa5, 0xf2,
			0x10, 0xd3, 0xa5, 0x94, 0x83, 0xae, 0xbc, 0x95,
			0xec, 0x71, 0xb3, 0x2d, 0xf1, 0x3f, 0xf4, 0xab,
			0xf4, 0x01, 0x91, 0x69, 0x37, 0xfd, 0x88, 0xff,
			0x44, 0xab, 0x46, 0xb7, 0x8c, 0xc3, 0x69, 0x41,
			0x4e, 0x9b, 0xca, 0xa8, 0xba, 0xb0, 0xbb, 0x85,
			0x57, 0x82, 0x8d, 0x73, 0xa2, 0xa6, 0x56, 0xc2,
			0xf8, 0x16, 0xf0, 0x70, 0xb5, 0xcb, 0x45, 0x54,
			0x9e, 0x8e, 0xca, 0x9d, 0x7c, 0x0b, 0x4a, 0x7b,
			0x0a, 0x27, 0xe5, 0x1c, 0x11, 0x93, 0x58, 0xda,
			0xd2, 0xa1, 0x7f, 0xb3, 0xa4, 0x57, 0x18, 0xf9,
			0xde, 0xc3, 0xc9, 0x4a, 0xf7, 0x8d, 0x65, 0xc3,
			0xec, 0xd3, 0x6b, 0x71, 0xe2, 0x30, 0xcf, 0x08,
			0x0d, 0x1e, 0xfd, 0xd8, 0xd0, 0x7f, 0x1c, 0xfc,
			0x26, 0x76, 0x8f, 0xd5, 0x40, 0x7b, 0xc2, 0xb7,
		},
		/* S6 : 256 byte */
		{
			0x9f, 0x48, 0xde, 0xb9, 0x6b, 0xec, 0x0b, 0x72,
			0xfb, 0xc4, 0xf1, 0x2f, 0x08, 0xaf, 0xb4, 0x6b,
			0xcc, 0xf1, 0x9d, 0x9e, 0x0c, 0xd0, 0x36, 0x8e,
			0xbe, 0xb3, 0x12, 0xd8, 0x38, 0x72, 0x62, 0x63,
			0x80, 0xac, 0x92, 0x8b, 0x61, 0x2c, 0x5c, 0xd7,
			0x74, 0x38, 0xd4, 0x7a, 0xa9, 0xce, 0xea, 0x90,
			0x5a, 0x9d, 0xe7, 0x18, 0x2c, 0x8e, 0xf7, 0x6e,
			0x8a, 0x7a, 0x03, 0xd6, 0xef, 0xec, 0x84, 0x00,
			0xb6, 0x49, 0x63, 0x62, 0xbf, 0x6a, 0x30, 0xce,
			0xb1, 0xce, 0xd2, 0x18, 0x5f, 0xc7, 0xc2, 0x11,
			0x7b, 0x6a, 0x6d, 0x88, 0x8a, 0xc2, 0x0c, 0x16,
			0x87, 0xb0, 0xf2, 0xaa, 0x9b, 0x76, 0x70, 0x5f,
			0xd3, 0x15, 0x48, 0x89, 0xb6, 0xac, 0xaf, 0x4e,
			0x63, 0xbe, 0x25, 0x88, 0x0c, 0x71, 0xe6, 0xc2,
			0x39, 0xec, 0xfb, 0x96, 0x50, 0x04, 0xcd, 0x63,
			0x21, 0x25, 0x7f, 0x84, 0x6a, 0xfd, 0x2a, 0x65,
			0x90, 0xc7, 0x2a, 0xd8, 0x31, 0x46, 0xee, 0xfc,
			0x7b, 0x0d, 0xc4, 0x79, 0x63, 0x39, 0xa7, 0xf6,
			0x4d, 0xa0, 0xfb, 0xe3, 0x59, 0xf9, 0x4a, 0xce,
			0x1f, 0xd1, 0x51, 0xc5, 0xac, 0x7b, 0xb5, 0x70,
			0x7b, 0x32, 0xea, 0xcf, 0x56, 0x4f, 0xe1, 0x62,
			0x2e, 0x66, 0xe1, 0x84, 0x4e, 0x63, 0x96, 0x02,
			0xca, 0x36, 0x27, 0x4a, 0xe0, 0x1f, 0x93, 0xe6,
			0xb2, 0xbd, 0x1e, 0xff, 0xd3, 0x4a, 0xb6, 0x3d,
			0x85, 0x2c, 0xc9, 0xca, 0xf3, 0xce, 0x84, 0x46,
			0xc2, 0x9c, 0x8a, 0xe3, 0xc6, 0x11, 0x0f, 0xb7,
			0x53, 0x8c, 0xc8, 0x37, 0x1c, 0x2a, 0x39, 0x81,
			0x24, 0x9c, 0xdc, 0x1b, 0xe2, 0xb2, 0x4b, 0x6a,
			0x0c, 0x95, 0x17, 0x64, 0xd0, 0xb7, 0xef, 0xa9,
			0x2a, 0x22, 0xcd, 0x8e, 0xd1, 0x65, 0xe1, 0x82,
			0x86, 0x35, 0x79, 0x37, 0x79, 0x97, 0xa9, 0xee,
			0x50, 0xc8, 0xac, 0x3a, 0xa4, 0xdf, 0x1a, 0xca,
		},
	},
	/* SHA512 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0x9a, 0xee, 0xd8, 0x5b, 0x40, 0xba, 0x7f, 0x86,
			0xa2, 0x28, 0xb5, 0xa1, 0x51, 0x5b, 0xa1, 0x90,
			0xb2, 0xef, 0xff, 0x66, 0x99, 0x3a, 0x5e, 0xce,
			0x19, 0xd1, 0x8b, 0xaa, 0x9b, 0x4e, 0x4d, 0xf9,
			0x2e, 0x51, 0x52, 0xfe, 0x1e, 0xc5, 0x6a, 0x9f,
			0xc8, 0x65, 0xf3, 0x0b, 0xac, 0x7e, 0x94, 0x9f,
			0xc4, 0xf6, 0x2f, 0x0b, 0x15, 0x8d, 0x10, 0xb0,
			0x83, 0x63, 0x6b, 0x4d, 0xe9, 0xbb, 0x05, 0xdb,
			0x69, 0xfe, 0x31, 0xb5, 0x01, 0x03, 0xfe, 0xfc,
			0x5f, 0x8d, 0xaf, 0x3a, 0xf7, 0x15, 0x6b, 0x45,
			0x52, 0xca, 0x36, 0x67, 0xa9, 0xd7, 0x20, 0xbb,
			0xb2, 0xe4, 0xbc, 0xda, 0xba, 0xdf, 0xd4, 0xb7,
			0xf4, 0xfc, 0x5b, 0xc8, 0x11, 0xfa, 0xa3, 0x67,
			0x10, 0xa9, 0xd1, 0x77, 0x58, 0xa9, 0x8d, 0x4a,
			0x04, 0x74, 0xfe, 0xc2, 0x7e, 0x9e, 0xf5, 0xb7,
			0x4f, 0x5c, 0x68, 0x99, 0x35, 0x44, 0x23, 0x57,
		},
		/* S7 : 256 byte */
		{
			0x9e, 0xec, 0xdb, 0xd7, 0xfb, 0xf6, 0x18, 0xdd,
			0xdd, 0xfb, 0x6e, 0x75, 0xd6, 0x44, 0x40, 0xf6,
			0x04, 0x45, 0xb8, 0x53, 0xc5, 0x42, 0xfe, 0x0f,
			0xba, 0xaa, 0x6a, 0x43, 0x12, 0x94, 0xe6, 0xcb,
			0x66, 0x83, 0xae, 0x1a, 0x71, 0xea, 0x05, 0x5e,
			0xb4, 0x9c, 0xd2, 0xa3, 0xcb, 0x51, 0x54, 0xdc,
			0x93, 0xd9, 0xaa, 0x16, 0x63, 0x99, 0xf4, 0xe6,
			0x29, 0x4f, 0x0e, 0xb0, 0x65, 0x28, 0x00, 0xd7,
			0x1e, 0x04, 0x1c, 0x1c, 0xe1, 0xad, 0x84, 0x9c,
			0x03, 0xc9, 0x63, 0xbc, 0x09, 0x29, 0xdc, 0xdd,
			0x11, 0xbe, 0x5d, 0x67, 0xa0, 0x50, 0xd0, 0x2b,
			0x64, 0xb2, 0x9e, 0xab, 0xa6, 0x55, 0x64, 0x2b,
			0x64, 0x36, 0xfb, 0xfb, 0x16, 0x36, 0x90, 0xbf,
			0x43, 0x2f, 0xdc, 0xee, 0xdd, 0x10, 0x6c, 0x2f,
			0x49, 0x72, 0xec, 0xbf, 0x30, 0x77, 0xed, 0x8b,
			0x75, 0x3b, 0xb6, 0x05, 0xec, 0x1e, 0xa0, 0x30,
			0x20, 0x83, 0x9a, 0x31, 0x8a, 0x24, 0xf8, 0xd4,
			0xc1, 0xd7, 0xd8, 0xdf, 0x99, 0xa7, 0xf0, 0x01,
			0x0a, 0xe4, 0x1a, 0x8b, 0x06, 0x8e, 0x28, 0x88,
			0x53, 0x10, 0x56, 0xa7, 0xda, 0xbb, 0xe9, 0x21,
			0x87, 0x8d, 0xcd, 0x3c, 0x7d, 0x69, 0x41, 0x68,
			0x67, 0xf4, 0x01, 0x2a, 0x60, 0x6a, 0xe8, 0x68,
			0x55, 0xf1, 0x5a, 0xed, 0x0d, 0xa1, 0x25, 0x0e,
			0x59, 0x68, 0x77, 0x06, 0xe8, 0x9c, 0x94, 0x94,
			0xba, 0xf3, 0x7f, 0x61, 0xfb, 0x17, 0x03, 0xb7,
			0x99, 0x28, 0x79, 0x5f, 0x90, 0xcc, 0xbe, 0x29,
			0x3a, 0x1e, 0x94, 0x72, 0xf6, 0xe0, 0xf4, 0xb8,
			0x90, 0xfd, 0xda, 0x3e, 0xa2, 0x52, 0x2e, 0x3d,
			0x11, 0xd5, 0xab, 0xdf, 0x00, 0x69, 0x51, 0x94,
			0x24, 0xd1, 0x47, 0xb5, 0x64, 0x6a, 0x5a, 0x60,
			0x1f, 0x19, 0xec, 0x89, 0x72, 0x9a, 0x8b, 0x48,
			0x46, 0x1e, 0x71, 0xc0, 0x8b, 0xbe, 0x9c, 0xda,
		},
	},
	/* SHA512 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0x65, 0x4e, 0x18, 0x9f, 0x06, 0xc7, 0xd4, 0x2d,
			0x55, 0x39, 0xa5, 0x87, 0x21, 0x84, 0xf8, 0x33,
			0x6c, 0xf1, 0x00, 0x69, 0x1f, 0x19, 0x08, 0x18,
			0xfd, 0x02, 0x08, 0x2a, 0xd6, 0x8a, 0x76, 0x09,
			0xfd, 0x09, 0x5e, 0x62, 0xfc, 0x32, 0xb5, 0x29,
			0x85, 0x3a, 0xeb, 0xdd, 0xac, 0x3d, 0xbf, 0x0d,
			0x54, 0xdd, 0x57, 0x1b, 0xe7, 0x2c, 0x90, 0x40,
			0x4b, 0xcc, 0x93, 0xd0, 0x11, 0x54, 0xa9, 0xbf,
			0xef, 0xf6, 0x50, 0x65, 0x70, 0x5f, 0x8e, 0x7e,
			0xea, 0xdf, 0x85, 0x75, 0xb1, 0xca, 0x48, 0xe2,
			0x8a, 0x1e, 0xed, 0x51, 0x62, 0x65, 0xe3, 0x45,
			0x40, 0xdd, 0x86, 0x7c, 0x79, 0xd7, 0xf1, 0x75,
			0x23, 0x5d, 0x13, 0x30, 0xcb, 0x17, 0x06, 0x35,
			0x6b, 0x70, 0x9b, 0xd7, 0x96, 0xf4, 0x3a, 0xba,
			0xf6, 0xfc, 0xe9, 0x93, 0xf8, 0x8e, 0xaa, 0x2f,
			0xc6, 0x7f, 0x0a, 0xb7, 0x76, 0xda, 0xf7, 0x32,
		},
		/* S8 : 256 byte */
		{
			0xaf, 0x90, 0x29, 0x8b, 0xce, 0xf6, 0x15, 0x30,
			0x9f, 0x23, 0x5d, 0x5c, 0x33, 0x60, 0xf0, 0xdf,
			0x11, 0xf5, 0xfb, 0x98, 0x87, 0x89, 0xf2, 0x13,
			0xd4, 0xc4, 0x61, 0x34, 0xfe, 0xe5, 0xeb, 0x10,
			0x4a, 0xa1, 0xfa, 0xbb, 0x13, 0x07, 0xc9, 0xa9,
			0x04, 0x70, 0x9d, 0xe8, 0x86, 0x73, 0xed, 0x99,
			0x51, 0xcb, 0xa9, 0x31, 0x67, 0xc6, 0x7c, 0x09,
			0xd8, 0x27, 0x02, 0x1b, 0x08, 0xa2, 0x2c, 0x05,
			0x05, 0x82, 0x8a, 0xb4, 0xbe, 0xb4, 0x2e, 0x59,
			0xa3, 0x88, 0x32, 0xcb, 0x4d, 0xa2, 0x4e, 0xcf,
			0x91, 0xf4, 0x70, 0xa3, 0xb4, 0x12, 0xc0, 0x71,
			0x2a, 0x8a, 0x59, 0xf6, 0xf2, 0x73, 0x9d, 0x4e,
			0x9e, 0xb4, 0xcc, 0x58, 0xd2, 0xc5, 0x25, 0x92,
			0xf1, 0x45, 0x2d, 0xc6, 0x57, 0x59, 0xab, 0xe4,
			0x3e, 0x8d, 0x2b, 0xc8, 0x04, 0xe2, 0xef, 0xb3,
			0xef, 0xc9, 0xb2, 0x3c, 0xc1, 0x73, 0x4f, 0xf7,
			0xca, 0xef, 0xa4, 0x6b, 0x03, 0xba, 0x4b, 0x39,
			0x7d, 0x07, 0x14, 0xcd, 0xb8, 0x50, 0x1a, 0x81,
			0x2c, 0x1b, 0x9f, 0x47, 0x41, 0x1c, 0x91, 0xcb,
			0xa5, 0x3a, 0x3d, 0x3b, 0x13, 0x9e, 0xdb, 0xd7,
			0xcb, 0xb5, 0x43, 0xf5, 0xbf, 0x38, 0x29, 0xba,
			0x7f, 0x5f, 0xaf, 0xd8, 0xa7, 0x12, 0xc0, 0xb1,
			0x11, 0x94, 0x3f, 0x53, 0x20, 0x93, 0x53, 0xaf,
			0xab, 0xa1, 0x76, 0xb3, 0xf5, 0xdc, 0x06, 0x03,
			0x39, 0xd0, 0x9b, 0x1f, 0xb3, 0xc2, 0x13, 0xda,
			0xe5, 0xd0, 0xf0, 0x04, 0xd3, 0x02, 0x82, 0x85,
			0x60, 0xfb, 0x5d, 0xeb, 0xf9, 0xfe, 0x49, 0x1e,
			0xaa, 0x66, 0xf5, 0x97, 0xaa, 0x4d, 0xe2, 0x3e,
			0xee, 0xf9, 0x17, 0x63, 0x58, 0x75, 0x5c, 0x95,
			0x2e, 0xf9, 0x6e, 0x36, 0x72, 0x58, 0x3b, 0x6e,
			0xcd, 0x95, 0xa0, 0x2e, 0x8c, 0xa7, 0xb2, 0x1d,
			0x7c, 0x20, 0xcb, 0xb7, 0xa7, 0x57, 0xaf, 0x71,
		},
	},
	/* SHA512 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0x12, 0x1f, 0x80, 0xb4, 0x3f, 0x97, 0x57, 0xb3,
			0xfa, 0x80, 0x90, 0x6a, 0xea, 0xb2, 0x32, 0x19,
			0x5f, 0x0e, 0x2c, 0x41, 0xe5, 0xbf, 0x8c, 0x09,
			0x1a, 0xc0, 0xf1, 0xe0, 0xbc, 0x9e, 0x43, 0x64,
			0x06, 0x80, 0xa1, 0x82, 0x3d, 0x64, 0x9b, 0xdf,
			0x86, 0xab, 0xa2, 0x77, 0xfa, 0xd8, 0xbc, 0x85,
			0xfc, 0x95, 0x7d, 0xa2, 0xca, 0xf7, 0x32, 0x30,
			0x53, 0x02, 0x5f, 0xf9, 0x49, 0x70, 0x6c, 0x14,
			0x76, 0xae, 0x9b, 0x09, 0x53, 0x28, 0x3d, 0x34,
			0xd7, 0xc6, 0x26, 0x6f, 0x8d, 0xb6, 0x5e, 0xeb,
			0xe9, 0x6d, 0x19, 0x5f, 0xdc, 0xe8, 0xe9, 0x65,
			0xa6, 0x38, 0x33, 0x20, 0xec, 0x3d, 0xe0, 0x23,
			0x0a, 0xb2, 0x54, 0x8e, 0xaa, 0x69, 0xa4, 0x7a,
			0x96, 0xd8, 0x03, 0x98, 0xca, 0xd5, 0x7e, 0x14,
			0xce, 0x9e, 0xea, 0xc0, 0x42, 0x1c, 0x1a, 0x6e,
			0xba, 0x69, 0x55, 0x9d, 0xcd, 0x8f, 0x06, 0x59,
		},
		/* S9 : 256 byte */
		{
			0x06, 0xa2, 0xd7, 0x45, 0x85, 0xf1, 0x2e, 0xa7,
			0xa8, 0x05, 0x27, 0xb8, 0xc6, 0x35, 0xa2, 0x1c,
			0xc1, 0x1b, 0x45, 0xdb, 0xb0, 0x88, 0x5a, 0x12,
			0x72, 0x21, 0x26, 0x81, 0x1d, 0xd2, 0x5d, 0x65,
			0x7b, 0xfa, 0x9f, 0xda, 0x77, 0x43, 0x01, 0xca,
			0x34, 0x98, 0xd0, 0x5d, 0xfd, 0xfb, 0x78, 0xa6,
			0xaa, 0x16, 0xa9, 0xf8, 0xa9, 0x5f, 0x40, 0xf1,
			0xf0, 0x4b, 0xd3, 0x54, 0xa5, 0x22, 0xf6, 0xa2,
			0xd6, 0x2b, 0x32, 0x4e, 0xfa, 0x3c, 0x00, 0x6c,
			0x22, 0xc2, 0x31, 0x4b, 0x01, 0xfa, 0x0e, 0x91,
			0xa3, 0xdb, 0xa4, 0x9a, 0xa3, 0x5b, 0x46, 0xb1,
			0x98, 0x04, 0xb0, 0x7a, 0xd9, 0x8f, 0xe4, 0xbc,
			0x99, 0x03, 0x93, 0xa4, 0xa2, 0x73, 0xce, 0x8f,
			0x1c, 0x85, 0xfc, 0x19, 0xcd, 0x5e, 0xae, 0x9a,
			0xf0, 0xb7, 0xd1, 0x95, 0x7b, 0xb2, 0x34, 0x09,
			0x77, 0x8a, 0x01, 0x0b, 0x00, 0xc6, 0x95, 0x9e,
			0x1b, 0x67, 0x06, 0x6f, 0xdb, 0x9f, 0x84, 0x95,
			0xb4, 0xde, 0x4d, 0xcb, 0xb9, 0x87, 0x35, 0x81,
			0x45, 0xb1, 0xff, 0x6a, 0x39, 0xef, 0x6f, 0xc5,
			0x88, 0xcd, 0xa1, 0x74, 0x4e, 0x0a, 0xb9, 0xe7,
			0xeb, 0x00, 0x2c, 0x29, 0xa7, 0x85, 0x31, 0xd2,
			0x51, 0x57, 0xc5, 0xc2, 0xcd, 0x64, 0x70, 0x55,
			0x15, 0x60, 0xa0, 0x28, 0x45, 0xdb, 0x6d, 0xbe,
			0xe2, 0x42, 0xf9, 0x65, 0xa2, 0x55, 0x40, 0x6f,
			0x6e, 0xf4, 0x7b, 0x32, 0x21, 0xa5, 0x11, 0x0e,
			0xdb, 0x44, 0xd3, 0x8b, 0x94, 0x19, 0x1a, 0xea,
			0xf4, 0x33, 0xc0, 0xec, 0xe3, 0x48, 0x0b, 0x9d,
			0x1b, 0x06, 0xd8, 0xb8, 0xb6, 0xc0, 0xa2, 0x32,
			0xa0, 0x4c, 0x56, 0x78, 0x88, 0xe6, 0x37, 0x2f,
			0x2e, 0x94, 0xbc, 0x2b, 0xe6, 0xb8, 0x27, 0xf8,
			0x71, 0x2a, 0xf4, 0x8c, 0x6f, 0x1e, 0x4f, 0x22,
			0x3f, 0x55, 0x28, 0xfc, 0xf3, 0x48, 0x79, 0x9d,
		},
	},
};

typedef mod2048_t tv_t;


#include "siggen15test.c"
