/* siggen15_2_4096_testfc.c */
/*
 * Copyright (c) 2017 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

/*
 * This test vector was created based on SigGen15_186-2.txt from
 * https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Algorithm-Validation-Program/documents/dss/186-2rsatestvectors.zip
 *
 * The link to the archive file exists in the following URL:
 * https://csrc.nist.gov/projects/cryptographic-algorithm-validation-program/digital-signatures/
 */
/*
 * https://csrc.nist.gov/projects/cryptographic-algorithm-validation-program/digital-signatures/
 * https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Algorithm-Validation-Program/documents/dss/186-2rsatestvectors.zip
 * SigGen15_186-2.txt
 */
#include "siggen15test.h"

static const unsigned short sn1[] = {
	0xd25c, 0x8f8c, 0x97b7, 0xe5a6, 0xa98d, 0xe78f, 0x19ef, 0x1807,
	0x7cf9, 0x52e6, 0x4622, 0xe414, 0x2e6b, 0xa5c8, 0xee26, 0x5ea9,
	0xb34b, 0x9e5c, 0xb2c1, 0xac03, 0x0754, 0x7faa, 0x7012, 0x6229,
	0xb320, 0x8598, 0x8c09, 0x74a0, 0x3198, 0xc89f, 0x8305, 0xbe4b,
	0xdc44, 0x5ee5, 0x77c6, 0x0b7a, 0x7fdf, 0xd5cf, 0x7283, 0xf0a1,
	0xd0f2, 0xd65b, 0x1ba2, 0xfbfb, 0x76c5, 0x039e, 0xe06a, 0x944d,
	0x40ae, 0x3843, 0xc9a4, 0xd605, 0x71db, 0x4cd1, 0x6e28, 0xce5b,
	0x9fb9, 0xfa83, 0xfa3c, 0xf8cf, 0xc1e8, 0x7103, 0x6125, 0x68ec,
	0xdb1a, 0x5c50, 0xfbd0, 0x911b, 0x6459, 0x83b6, 0x29ba, 0x72bc,
	0x2d22, 0x8207, 0xb49e, 0x00bd, 0xae1b, 0x906a, 0xa15e, 0xa2d2,
	0x3f8c, 0x0e12, 0x4c9e, 0x7de1, 0xb3cd, 0x3b9b, 0xc6a0, 0x17c4,
	0xde3f, 0x1d65, 0x48da, 0xcad6, 0xd4ab, 0xd5f4, 0xe179, 0xa2a5,
	0xe782, 0xfb8f, 0x818c, 0xf8f3, 0x07f8, 0x4f7f, 0x3bcd, 0x8105,
	0x9e73, 0x1920, 0x36ac, 0xa02f, 0x8050, 0x70ed, 0xed8b, 0x5de0,
	0x1037, 0xf2b3, 0x4b41, 0xe86b, 0xb8c4, 0x4760, 0x492c, 0xde4b,
	0xc3b0, 0x80d5, 0xb062, 0x4bfa, 0x1002, 0xbd6a, 0x75d0, 0x513b,
	0x07c3, 0x54e8, 0xedf8, 0xa6f5, 0x7dc1, 0x4290, 0xeb88, 0x5065,
	0x9788, 0x098b, 0x9971, 0xfc04, 0x92c3, 0xced7, 0xc347, 0x8689,
	0x444e, 0x771c, 0x9434, 0xfb62, 0x4eb4, 0x243f, 0x740c, 0x3ef9,
	0xc3f3, 0xf084, 0xbfb5, 0x9f44, 0xa46a, 0x1d3b, 0x1438, 0xe661,
	0xba3b, 0xa08d, 0x5252, 0x5522, 0x4816, 0x348b, 0x4d69, 0x6475,
	0xccac, 0x3871, 0xfc78, 0x3638, 0x5f70, 0x7841, 0x2cc5, 0x8e36,
	0x747d, 0x248b, 0xb1ab, 0x123f, 0xa332, 0x56c1, 0x0d11, 0x02f7,
	0x7a60, 0x643f, 0x4247, 0xac1c, 0xdfdf, 0x4d8d, 0x51de, 0xea80,
	0x7959, 0x64f8, 0x01f9, 0x7c4f, 0x955e, 0xb181, 0x136b, 0xb39a,
	0xbecd, 0x3549, 0x541c, 0xcc9d, 0x29e3, 0x4a63, 0x241a, 0x4548,
	0xe836, 0xc4d2, 0x1599, 0x6859, 0x6130, 0xa222, 0x608a, 0xe0b0,
	0x9d9e, 0xd30c, 0xbf04, 0xd830, 0x8345, 0x7f60, 0xe8a6, 0x86a4,
	0x52ba, 0xa2b0, 0x8c17, 0x7ade, 0x1a3c, 0xbe7b, 0xc0d6, 0xed84,
	0x6c24, 0xc4ad, 0x5bca, 0xff38, 0x17fd, 0xf338, 0x1e64, 0xc39c,
	0x18dd, 0x4f54, 0xeb82, 0xfc23, 0xd313, 0x9843, 0x2284, 0x8939,
	0x9c15, 0x6565, 0xdf75, 0xf380, 0x8cbd, 0xfaa5, 0x3594, 0xbac5,
};

static const unsigned short se1[] = {
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0001, 0x0001,
};

static const unsigned short sd1[] = {
	0x28a6, 0xfc92, 0xc5eb, 0x22f5, 0x4f44, 0xfe1f, 0xd87a, 0x2d21,
	0x9d54, 0x2d62, 0x14e1, 0xbf43, 0xeb28, 0xa074, 0x3119, 0xb727,
	0x825f, 0xb5ad, 0xe259, 0xad1d, 0x34b8, 0x6ea7, 0x885a, 0xf51f,
	0x6bdf, 0x7811, 0xdfd6, 0x1ff4, 0xa941, 0x065c, 0x551c, 0xb206,
	0xc4fd, 0x6d7d, 0x58e5, 0x2be8, 0xb377, 0x9541, 0x2eb7, 0x32b4,
	0x82c5, 0xbb09, 0x0651, 0x3bf2, 0xffa3, 0xd92c, 0x45e0, 0x32e2,
	0xaf02, 0x3c1b, 0x5587, 0x8658, 0xad05, 0x1d56, 0x47bc, 0x990c,
	0x79d4, 0x9e06, 0x0d40, 0xe530, 0x791f, 0xab5f, 0x64d1, 0xaef2,
	0x3ef1, 0x5c2f, 0xc64a, 0x2b10, 0xa916, 0xe76e, 0x37a8, 0x316e,
	0x0e26, 0xe20c, 0x0d2a, 0x8fa8, 0x1852, 0x7959, 0x31f0, 0x30f5,
	0x0294, 0xd54f, 0x5201, 0x3b00, 0x7817, 0xcc52, 0x3153, 0x54e4,
	0x95b2, 0x8ac2, 0xbd0e, 0x932e, 0x6b10, 0xfe1b, 0x01cb, 0xd8d3,
	0xc8e6, 0xb628, 0xa877, 0x8ed0, 0xa094, 0xab00, 0x4295, 0x573e,
	0xb7bd, 0x3478, 0xd638, 0x0c08, 0xc1fa, 0x49a8, 0xa144, 0x5292,
	0xef6f, 0x7076, 0x97af, 0x97f3, 0xe154, 0x051b, 0x2569, 0x0758,
	0x8aeb, 0xbc2a, 0x0f99, 0x1f67, 0xa6da, 0x70b2, 0x772c, 0x17d6,
	0x3fd8, 0xd12b, 0x94ab, 0x4e0f, 0xaf93, 0x30a4, 0xb4bf, 0xdcc6,
	0xd913, 0x7262, 0xcff6, 0x0ec0, 0x03e6, 0x1182, 0xf201, 0xbd07,
	0xc7d3, 0xacad, 0x67cf, 0x2e78, 0x0e29, 0xa112, 0x3878, 0x5d48,
	0x02ff, 0xa813, 0x6bfb, 0x4d35, 0x58b4, 0x8e8c, 0xb77b, 0xd63f,
	0xfd9b, 0xbeac, 0x0f51, 0xd297, 0xf009, 0xa7b6, 0x03a3, 0x8860,
	0x3c59, 0xd7d6, 0x8a02, 0xe2e0, 0x27bc, 0x7249, 0x177b, 0x185e,
	0x567b, 0x7a09, 0x2986, 0xe800, 0x8101, 0xd4f5, 0x8e91, 0x7c5e,
	0x9ece, 0xf56d, 0x5acc, 0x0076, 0x7e0d, 0x74b4, 0x9c2d, 0x3bd2,
	0x9ca2, 0x90e8, 0xde04, 0xf6e1, 0x61d5, 0xe48a, 0xf325, 0x3a17,
	0x10d1, 0xf5b7, 0xa50f, 0x888a, 0x1417, 0xb12c, 0x221f, 0xf25d,
	0x9b10, 0x6128, 0x8298, 0x94cb, 0x16ee, 0xd843, 0x33a9, 0x6abb,
	0x2ac3, 0x10b4, 0xa72b, 0xd207, 0xead3, 0xdc25, 0x6093, 0x9307,
	0xa74f, 0x3ba3, 0xb8ae, 0xa576, 0xcd84, 0x6ca9, 0x63cf, 0x337f,
	0xb41b, 0x506b, 0xd232, 0x0036, 0xe3b4, 0x69af, 0x5a1d, 0xb5a2,
	0xd741, 0x9dfc, 0x7624, 0x41da, 0x1127, 0x5c88, 0x25d9, 0x6d4c,
	0xe270, 0x3a4e, 0x183f, 0xd20d, 0x4a9f, 0xb85d, 0xa6a0, 0x7b8d,
};

#define HAS_SHA1_TESTVECTOR

#define HAS_SHA224_TESTVECTOR

#define HAS_SHA256_TESTVECTOR

#define HAS_SHA384_TESTVECTOR

#define HAS_SHA512_TESTVECTOR

/* 10 elements */
static mod4096_t SHA1[] = {
	/* SHA1 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0x80, 0xf1, 0x61, 0xf9, 0xe7, 0xae, 0x94, 0x65,
			0xdc, 0x66, 0xe5, 0x40, 0x1a, 0x77, 0x8e, 0xf1,
			0x41, 0x94, 0xc5, 0xe6, 0xd7, 0xba, 0xe1, 0x28,
			0x8c, 0x65, 0xbf, 0x33, 0xc2, 0x03, 0x64, 0x04,
			0x4c, 0xa9, 0xe3, 0x73, 0x76, 0x05, 0x81, 0xc3,
			0x46, 0x94, 0x18, 0x78, 0x02, 0x14, 0xb9, 0x68,
			0xf2, 0x3d, 0x95, 0x15, 0x67, 0xac, 0x1e, 0x3a,
			0xfd, 0x20, 0x69, 0xa4, 0x1c, 0xe8, 0xd7, 0x43,
			0xc5, 0x9c, 0xf4, 0x81, 0x8b, 0x0c, 0x9f, 0xd2,
			0xdf, 0x76, 0x6c, 0x8d, 0x91, 0xfc, 0x46, 0xfa,
			0x94, 0x54, 0x59, 0xac, 0x02, 0xc2, 0x2f, 0x11,
			0x5f, 0x43, 0x18, 0x47, 0x22, 0x0b, 0xd0, 0x31,
			0x71, 0x32, 0x5c, 0xcb, 0x2d, 0xab, 0x35, 0xba,
			0x2e, 0x75, 0xa8, 0x28, 0xd0, 0x15, 0x7d, 0xce,
			0xae, 0x43, 0xd7, 0x52, 0xad, 0xe4, 0x5b, 0xd4,
			0x3d, 0x38, 0x94, 0xc4, 0xdb, 0x55, 0x96, 0xf6,
		},
		/* S0 : 512 byte */
		{
			0x25, 0xc2, 0x47, 0xf9, 0x9a, 0x2f, 0xf5, 0xd6,
			0x98, 0xed, 0x3d, 0xbe, 0x4d, 0xde, 0x8e, 0xb4,
			0x91, 0x32, 0x30, 0xe1, 0x30, 0x39, 0xb2, 0x31,
			0x3d, 0xcd, 0x0c, 0xbf, 0xa8, 0x17, 0x4f, 0x82,
			0x51, 0xe1, 0xfb, 0x17, 0xae, 0xea, 0xe3, 0x60,
			0xda, 0x0c, 0xbf, 0xcd, 0xc7, 0x08, 0x75, 0x1e,
			0x84, 0x24, 0xd4, 0x7e, 0x69, 0x2e, 0x48, 0xea,
			0x06, 0xd4, 0x61, 0x5f, 0x7b, 0xd8, 0xb7, 0x1c,
			0xd2, 0xac, 0x1d, 0xc9, 0x43, 0xd5, 0xec, 0x2c,
			0x35, 0xe1, 0x58, 0x5c, 0x8b, 0x97, 0x57, 0x02,
			0x6a, 0x2e, 0x10, 0x3e, 0x22, 0xff, 0x6c, 0x2b,
			0xbb, 0x70, 0xb0, 0x51, 0xe8, 0x3a, 0xc9, 0x3d,
			0x81, 0x08, 0x92, 0xa3, 0x80, 0x55, 0xf0, 0x3f,
			0xf9, 0x0d, 0x49, 0x72, 0x78, 0x73, 0x87, 0x13,
			0xae, 0xb6, 0x71, 0xfb, 0xb6, 0xb2, 0x1c, 0xc2,
			0x30, 0x97, 0xc5, 0xd5, 0xd0, 0xd4, 0x9c, 0x49,
			0x7d, 0xdc, 0xec, 0x61, 0x17, 0x78, 0xd9, 0x05,
			0xea, 0x63, 0x5e, 0xcf, 0x9c, 0x50, 0x0b, 0x35,
			0x1d, 0x0c, 0x09, 0xae, 0xd6, 0xa5, 0x9e, 0xb3,
			0x3b, 0xc4, 0x61, 0x08, 0x18, 0x41, 0xba, 0x28,
			0xe0, 0xea, 0xc4, 0x33, 0x1e, 0x02, 0x22, 0xa8,
			0xe8, 0xe7, 0x1b, 0x0e, 0x5c, 0x06, 0xad, 0xda,
			0xf4, 0x69, 0x69, 0xfb, 0x83, 0x6b, 0x87, 0x2a,
			0xed, 0xba, 0x9a, 0xff, 0x01, 0xf2, 0xc5, 0x31,
			0xb8, 0x18, 0x6e, 0xde, 0xa7, 0x73, 0x67, 0xdf,
			0x9c, 0x01, 0x5f, 0xdf, 0xf6, 0x9e, 0xdc, 0xa8,
			0x43, 0xef, 0x24, 0xa8, 0x16, 0xa0, 0xbd, 0xd4,
			0x64, 0xeb, 0xc7, 0xf7, 0xb2, 0x0d, 0xde, 0xc2,
			0xb7, 0x4f, 0x9e, 0x85, 0x64, 0x76, 0xe2, 0xd9,
			0xdc, 0xf6, 0xef, 0xa4, 0xc3, 0x52, 0x43, 0x94,
			0x70, 0xb8, 0x5c, 0xb1, 0xd4, 0xaa, 0x21, 0x71,
			0x9a, 0x03, 0x5d, 0xde, 0x4e, 0xee, 0x22, 0xea,
			0xf6, 0x74, 0x38, 0x74, 0xeb, 0x7e, 0xcd, 0x93,
			0x33, 0x4e, 0xdc, 0x82, 0x95, 0x7c, 0xdb, 0x8e,
			0x1d, 0x6f, 0x56, 0xf7, 0x31, 0xb6, 0xb0, 0xd5,
			0x01, 0x9f, 0xcf, 0xf9, 0x2c, 0xc1, 0xdb, 0x18,
			0xf6, 0xba, 0xff, 0xf1, 0x46, 0xa9, 0xe9, 0xbf,
			0xb5, 0x24, 0x2f, 0x63, 0x41, 0x8c, 0x71, 0x6b,
			0x50, 0x81, 0xeb, 0xf0, 0xc8, 0x19, 0xbe, 0xe1,
			0x51, 0x68, 0xe4, 0x7e, 0xed, 0x40, 0x28, 0xa2,
			0x0f, 0xfa, 0xf9, 0xa7, 0x62, 0x14, 0x83, 0x84,
			0xfa, 0x76, 0xe6, 0xc2, 0x2f, 0x1c, 0xed, 0xc9,
			0xe1, 0x82, 0x58, 0x8e, 0xe8, 0x87, 0xdf, 0xe6,
			0x68, 0xc3, 0x0a, 0xc4, 0xf6, 0xf2, 0x7c, 0x4c,
			0x3f, 0xf3, 0xe2, 0xdd, 0x73, 0xbf, 0x7e, 0xbe,
			0x13, 0x99, 0xb5, 0xad, 0xdb, 0xc6, 0xb5, 0xa1,
			0xeb, 0xe2, 0xd9, 0x71, 0x34, 0xfb, 0x26, 0x82,
			0x56, 0x49, 0x12, 0x88, 0x79, 0x00, 0x6f, 0xc3,
			0xdd, 0x20, 0x67, 0xb6, 0xc6, 0x81, 0xab, 0xb6,
			0x78, 0xd1, 0x86, 0xfd, 0x97, 0xf2, 0x32, 0x31,
			0xa7, 0x98, 0xaf, 0xb6, 0x40, 0x72, 0x1a, 0xd9,
			0xb8, 0x4f, 0xe5, 0xf5, 0x1e, 0x71, 0x2a, 0x51,
			0x55, 0xe7, 0xa4, 0x33, 0x34, 0x58, 0xeb, 0x14,
			0x5b, 0x67, 0xab, 0x48, 0xa8, 0x5b, 0x5e, 0xee,
			0xe2, 0xd1, 0xca, 0xeb, 0x76, 0x3a, 0x37, 0x01,
			0x8c, 0xd7, 0xfd, 0x5e, 0x7c, 0xd4, 0x1f, 0x89,
			0x65, 0xb2, 0x5b, 0xb3, 0x09, 0x90, 0x8f, 0xd7,
			0x53, 0x2b, 0x5e, 0x63, 0x6a, 0x85, 0xc7, 0x60,
			0x37, 0x59, 0xc0, 0x70, 0x7a, 0xb5, 0x8d, 0xe6,
			0xeb, 0x2e, 0xc4, 0x6e, 0x9c, 0xe1, 0xeb, 0x5c,
			0xff, 0x08, 0x61, 0xc1, 0xd3, 0x12, 0x36, 0x73,
			0x5b, 0x01, 0xe9, 0xd6, 0xb8, 0x3e, 0x0a, 0x2a,
			0xd4, 0x2a, 0x47, 0xaa, 0xc9, 0x1a, 0x0d, 0x01,
			0x46, 0xdb, 0x13, 0xcd, 0x44, 0x79, 0xc3, 0xb0,
		},
	},
	/* SHA1 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0x9f, 0x99, 0x3b, 0x53, 0x07, 0x24, 0x92, 0xae,
			0x16, 0xd9, 0x10, 0x99, 0x65, 0xf9, 0xc8, 0x6b,
			0x41, 0xae, 0x08, 0xc8, 0x0c, 0xe4, 0xdf, 0x18,
			0xe1, 0x5b, 0xdd, 0x37, 0x77, 0xfa, 0x1a, 0x6e,
			0x48, 0x67, 0x79, 0xa3, 0xf2, 0xd4, 0x2e, 0xcc,
			0xdd, 0x1c, 0x7e, 0xaa, 0x1e, 0x0e, 0xc6, 0xb3,
			0x9b, 0x08, 0xbf, 0x50, 0x70, 0xc9, 0x00, 0x48,
			0x92, 0x80, 0xc5, 0x7c, 0xb2, 0x28, 0xee, 0xd6,
			0x65, 0x18, 0x4b, 0x7c, 0x81, 0xd6, 0x16, 0x49,
			0xca, 0x2d, 0xc1, 0x7d, 0xce, 0x7f, 0xf3, 0xb3,
			0x0d, 0x5c, 0xcd, 0x14, 0x5d, 0x52, 0x71, 0xcb,
			0x8d, 0x07, 0xc5, 0xb8, 0x49, 0x86, 0x74, 0x69,
			0x3d, 0xd7, 0x36, 0x46, 0x7e, 0xd2, 0x52, 0x0b,
			0x25, 0x34, 0x61, 0x7c, 0x3d, 0xf3, 0x54, 0x54,
			0xdf, 0xcf, 0xb8, 0x7e, 0x9b, 0x2a, 0x39, 0x87,
			0x1a, 0x19, 0x65, 0x54, 0x02, 0xbb, 0xb3, 0x10,
		},
		/* S1 : 512 byte */
		{
			0x2a, 0xe6, 0xd9, 0x37, 0x58, 0x9e, 0xfc, 0x4a,
			0x87, 0x13, 0x78, 0xfb, 0xf6, 0x46, 0x33, 0x78,
			0x96, 0x3b, 0x32, 0x90, 0x00, 0x2f, 0xd9, 0x61,
			0xc8, 0xfd, 0x52, 0xfa, 0xf1, 0x90, 0x9f, 0x86,
			0x46, 0x9b, 0x73, 0x8a, 0x8f, 0x64, 0x4f, 0xb3,
			0xe9, 0xcf, 0x41, 0x44, 0x86, 0x77, 0x1b, 0x69,
			0x47, 0x22, 0x61, 0x23, 0x51, 0x70, 0x46, 0xd8,
			0xff, 0xf8, 0xd9, 0x26, 0x2a, 0x99, 0xd3, 0xae,
			0x94, 0x01, 0x2b, 0x14, 0x34, 0x99, 0x95, 0x90,
			0x46, 0x77, 0xd3, 0x87, 0xe4, 0xa1, 0x2b, 0x33,
			0x4b, 0xea, 0x6b, 0xaf, 0xb0, 0xa2, 0x55, 0x58,
			0x8c, 0xb7, 0x64, 0x86, 0x7c, 0xdf, 0x9f, 0xd6,
			0x26, 0xb0, 0xc4, 0x2f, 0x76, 0xe4, 0xdd, 0x30,
			0xc5, 0x2a, 0x69, 0x12, 0x09, 0xa8, 0x30, 0xfc,
			0xe1, 0x43, 0x44, 0x55, 0x21, 0x20, 0x53, 0xbf,
			0x6a, 0xd9, 0x1d, 0x2c, 0x7b, 0xf8, 0x0d, 0xc7,
			0x9c, 0x1c, 0x4b, 0x01, 0x57, 0xe8, 0x32, 0x05,
			0x86, 0x7a, 0x00, 0x04, 0x87, 0x58, 0xd7, 0xd1,
			0x36, 0xd9, 0x83, 0x91, 0xef, 0x20, 0x74, 0x8f,
			0xed, 0x0a, 0xa9, 0x8f, 0x6b, 0x7b, 0x88, 0xe0,
			0xdb, 0x49, 0x81, 0xf0, 0xe9, 0xbb, 0xca, 0x30,
			0x0a, 0x50, 0x31, 0x4e, 0x1c, 0xaf, 0xb1, 0xf2,
			0x65, 0x80, 0x49, 0x5a, 0x2b, 0xb3, 0xf0, 0xf0,
			0x58, 0x60, 0xc6, 0xdc, 0x6e, 0x70, 0x89, 0xc0,
			0x3b, 0xb5, 0x8e, 0x29, 0xac, 0x61, 0xa9, 0x08,
			0x48, 0x27, 0x57, 0x0b, 0x95, 0xd9, 0x0f, 0x56,
			0x86, 0x70, 0xd8, 0x9e, 0x91, 0x73, 0xa3, 0x2d,
			0x4e, 0x54, 0xdd, 0x80, 0x47, 0xb8, 0xbc, 0x04,
			0xd2, 0x1a, 0x6b, 0xfb, 0xb9, 0x48, 0xa9, 0x6c,
			0x64, 0xf7, 0x4d, 0xd6, 0x49, 0xbf, 0x3c, 0x91,
			0x1f, 0xab, 0x9b, 0x02, 0x69, 0x7b, 0x3f, 0x74,
			0x75, 0xf4, 0x5c, 0xe0, 0x17, 0xc6, 0x96, 0xda,
			0x06, 0xa2, 0xe9, 0x55, 0xe2, 0x81, 0x52, 0xfe,
			0xa9, 0x50, 0x2c, 0x59, 0x09, 0x37, 0x04, 0xda,
			0x6d, 0x7d, 0xf9, 0x63, 0xcb, 0xd9, 0x4c, 0x42,
			0xdf, 0x9f, 0xf7, 0xfb, 0xa8, 0x9e, 0x7b, 0x11,
			0x3f, 0x1e, 0x32, 0x29, 0x88, 0x7c, 0x0b, 0x26,
			0xf7, 0x69, 0x4d, 0x1c, 0xf6, 0x94, 0x99, 0x3f,
			0xa7, 0x90, 0x2c, 0xaa, 0x92, 0xf0, 0xec, 0x30,
			0x47, 0x96, 0x48, 0x8c, 0xc9, 0x3f, 0x78, 0x9f,
			0xf2, 0x8f, 0x6d, 0x12, 0x2c, 0x64, 0x2a, 0x07,
			0x98, 0x4a, 0x6f, 0x71, 0x54, 0x3c, 0x8e, 0xdd,
			0xe6, 0xc6, 0x12, 0x2b, 0xd1, 0x62, 0x1d, 0xe3,
			0x49, 0xea, 0x2b, 0x67, 0xbe, 0x2e, 0x87, 0xec,
			0x01, 0x42, 0xf1, 0x3e, 0x47, 0x26, 0x6e, 0xb5,
			0xf0, 0x4d, 0xd1, 0x22, 0xc7, 0x37, 0x7b, 0xcb,
			0x6c, 0xef, 0xf9, 0x22, 0xdd, 0x74, 0x87, 0x2f,
			0x09, 0x18, 0x18, 0x07, 0x0f, 0xff, 0xef, 0x24,
			0x61, 0x8b, 0x27, 0xb1, 0x28, 0x27, 0xce, 0xbe,
			0xd7, 0x41, 0x03, 0xde, 0x9b, 0x9e, 0x00, 0x91,
			0xff, 0x22, 0x76, 0xad, 0x2b, 0x2d, 0xf0, 0xd1,
			0x90, 0x6b, 0x72, 0x28, 0xdd, 0x4c, 0x8b, 0x1f,
			0xb2, 0x2e, 0x75, 0xde, 0xa8, 0x7d, 0x6c, 0xb5,
			0x9a, 0xe6, 0xea, 0xcb, 0xad, 0x6d, 0xde, 0xe0,
			0xde, 0x44, 0x96, 0x8e, 0x8b, 0x24, 0x50, 0xdf,
			0x32, 0x7c, 0x88, 0x57, 0x67, 0xe2, 0x19, 0x60,
			0x1c, 0x91, 0x70, 0x2c, 0xae, 0xe6, 0xba, 0xd1,
			0xc2, 0x8d, 0xdc, 0x47, 0xab, 0xba, 0x77, 0x23,
			0xfe, 0xad, 0x69, 0x92, 0x43, 0x26, 0x85, 0x66,
			0x5d, 0xcf, 0x8a, 0x69, 0xc0, 0xf5, 0xdd, 0x22,
			0x75, 0xff, 0x55, 0x1f, 0x13, 0xc9, 0x22, 0xf9,
			0x08, 0xc9, 0xca, 0x64, 0x7f, 0xa9, 0x85, 0xf0,
			0x5c, 0x4d, 0x89, 0x0e, 0x9b, 0xec, 0x98, 0x21,
			0xc0, 0x40, 0xd9, 0x99, 0xe9, 0xe9, 0x4d, 0x2b,
		},
	},
	/* SHA1 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0xfa, 0x47, 0x87, 0xc0, 0xdc, 0x22, 0xae, 0x64,
			0xe2, 0x75, 0xca, 0x13, 0x10, 0x80, 0x2b, 0xab,
			0x56, 0xc4, 0x5d, 0x58, 0xc2, 0x67, 0xfd, 0x15,
			0xf3, 0xdb, 0xcc, 0x6f, 0xd1, 0xbb, 0x02, 0x8c,
			0x7b, 0xf5, 0xe5, 0x6d, 0x81, 0x2e, 0x99, 0x70,
			0x54, 0x62, 0x3a, 0x19, 0xfa, 0xcb, 0x28, 0x11,
			0x06, 0x05, 0xa2, 0x6f, 0xc8, 0x6f, 0x97, 0x37,
			0xc1, 0xd4, 0xb8, 0x20, 0x59, 0x4b, 0xd2, 0x9b,
			0xd5, 0x5b, 0x5b, 0x10, 0x83, 0x63, 0x4b, 0x6b,
			0x37, 0x39, 0x2d, 0xd7, 0xfb, 0x12, 0x75, 0xc9,
			0x03, 0xa2, 0x14, 0x41, 0xdf, 0x84, 0x4a, 0x4d,
			0xf4, 0xdb, 0xbb, 0xce, 0x2d, 0x3a, 0xc6, 0x7e,
			0xee, 0xc1, 0x12, 0x32, 0x4a, 0x89, 0x69, 0x5d,
			0x75, 0x43, 0xca, 0xd6, 0x7c, 0x1b, 0xb9, 0xdb,
			0x33, 0xe3, 0xda, 0x55, 0x6f, 0x6d, 0xbe, 0x5d,
			0x76, 0x38, 0xfd, 0x00, 0xb4, 0x4f, 0x85, 0x3f,
		},
		/* S2 : 512 byte */
		{
			0x87, 0xd3, 0x24, 0x41, 0xd5, 0x14, 0x09, 0x82,
			0x01, 0xab, 0x4c, 0x43, 0x36, 0x59, 0x92, 0x92,
			0xde, 0xb5, 0x9d, 0xcc, 0x53, 0x6e, 0x35, 0xaf,
			0x55, 0x03, 0x2b, 0xee, 0xdf, 0xb6, 0xdf, 0xe9,
			0xf5, 0x5f, 0xbd, 0x1c, 0x9e, 0xb0, 0x88, 0xce,
			0xe1, 0xdc, 0xf2, 0x48, 0x7d, 0x35, 0x5d, 0xbf,
			0x0c, 0x39, 0x2a, 0x98, 0x54, 0x1c, 0xa3, 0xb4,
			0x16, 0xce, 0x2a, 0xb1, 0x76, 0xb3, 0xf7, 0xb0,
			0x31, 0xa2, 0x79, 0x69, 0xf7, 0xa4, 0x59, 0x28,
			0xdb, 0x90, 0x2e, 0x1b, 0x70, 0x67, 0x7a, 0x17,
			0x06, 0xe2, 0x33, 0x85, 0xcf, 0xec, 0xa1, 0x8f,
			0x10, 0x28, 0xab, 0xa3, 0xf0, 0x84, 0x10, 0x4e,
			0x45, 0x48, 0xd1, 0xf1, 0xc4, 0xc4, 0x2b, 0xa2,
			0x58, 0x7f, 0x59, 0x27, 0x89, 0x62, 0x36, 0x4c,
			0x67, 0xda, 0x15, 0xec, 0x5e, 0xd0, 0x86, 0x3c,
			0x91, 0xbf, 0xf4, 0xef, 0x5b, 0x65, 0x62, 0xbc,
			0x74, 0x19, 0x7b, 0xef, 0xb7, 0xa1, 0x87, 0x0e,
			0x9e, 0xad, 0xb9, 0x00, 0x02, 0xee, 0x27, 0x44,
			0x90, 0x26, 0x6f, 0x7d, 0xd7, 0xee, 0x83, 0x2b,
			0x81, 0x82, 0x04, 0xc2, 0xd9, 0x02, 0xce, 0xbb,
			0x03, 0x50, 0x2b, 0x3a, 0x4b, 0x43, 0x64, 0x4b,
			0xca, 0x90, 0x55, 0x19, 0xab, 0x78, 0x9c, 0x3b,
			0x01, 0xda, 0x27, 0x06, 0x50, 0x60, 0xaf, 0xe3,
			0x94, 0x5d, 0x37, 0x28, 0x78, 0xd4, 0xe3, 0xdd,
			0x95, 0x4a, 0x7f, 0xcf, 0x75, 0xbc, 0x07, 0x72,
			0x13, 0x3b, 0x6c, 0x22, 0x31, 0x3b, 0x2f, 0x38,
			0x62, 0x3c, 0xaf, 0x36, 0x34, 0x9e, 0x9a, 0xb2,
			0x90, 0xe9, 0x5d, 0xb7, 0x81, 0x02, 0x54, 0xa0,
			0xe6, 0xaa, 0x57, 0xdb, 0x54, 0x12, 0xb9, 0xb1,
			0x0d, 0x08, 0x85, 0x0a, 0x51, 0xcf, 0xcf, 0xfc,
			0xc3, 0xf0, 0x01, 0x3a, 0x52, 0x4d, 0x12, 0xc5,
			0xf6, 0xd4, 0x31, 0xf5, 0xc6, 0x79, 0xfd, 0x2f,
			0x98, 0x40, 0x4d, 0x7d, 0xb8, 0xd3, 0xf6, 0x7b,
			0x65, 0xa0, 0x6d, 0x5f, 0x57, 0xe4, 0xac, 0xeb,
			0x8b, 0xbb, 0x06, 0x0f, 0xd0, 0x54, 0xa7, 0x9b,
			0xdb, 0xf1, 0x14, 0x2b, 0x33, 0xa4, 0x58, 0xea,
			0xbe, 0x16, 0x5e, 0x57, 0x89, 0xf0, 0x75, 0xb6,
			0x98, 0xc2, 0x3a, 0xbd, 0x04, 0x4e, 0xb3, 0x1f,
			0x4e, 0xea, 0xf2, 0x9c, 0x5f, 0x82, 0x55, 0xa1,
			0x57, 0xe8, 0x1f, 0xfc, 0xf5, 0x7b, 0xb8, 0x56,
			0x17, 0x3f, 0xf9, 0x3b, 0xc5, 0x1c, 0x88, 0x9d,
			0xa0, 0x65, 0xf1, 0x4f, 0x83, 0x08, 0xe3, 0x58,
			0x7b, 0xf0, 0x1e, 0xd3, 0xa7, 0x28, 0x90, 0x59,
			0xdb, 0x19, 0xb7, 0xea, 0x43, 0xd0, 0xc6, 0x91,
			0x94, 0xbb, 0x8b, 0x40, 0xbd, 0xfa, 0xb9, 0x4d,
			0xcc, 0xf8, 0x0b, 0x6f, 0xc0, 0x90, 0x3d, 0x3b,
			0xa7, 0x3d, 0x4d, 0x6b, 0x2b, 0xa0, 0xd4, 0x8a,
			0xcb, 0x5a, 0x73, 0x42, 0xcc, 0x2e, 0x14, 0x71,
			0x49, 0x91, 0x63, 0x2c, 0x8f, 0xaa, 0xf2, 0xc2,
			0x57, 0xf9, 0x31, 0x0f, 0xbe, 0xd2, 0x29, 0x59,
			0x70, 0x36, 0x5e, 0x02, 0xdd, 0x6b, 0x28, 0x49,
			0xee, 0x13, 0x4f, 0x07, 0x02, 0xb8, 0x6d, 0x35,
			0x0a, 0x32, 0x02, 0x97, 0x22, 0xcc, 0xc2, 0xc7,
			0x7c, 0x76, 0xc4, 0x37, 0x93, 0x25, 0x3a, 0x2f,
			0xfa, 0x0a, 0xc8, 0x3c, 0x07, 0x53, 0x2b, 0xa9,
			0x56, 0x13, 0x2c, 0x1c, 0xf0, 0x3a, 0x6d, 0xca,
			0xf1, 0x28, 0xe4, 0x98, 0xeb, 0x17, 0x5b, 0xe1,
			0xe5, 0xa7, 0x0d, 0x15, 0x73, 0xb1, 0xd4, 0xcc,
			0x93, 0xbb, 0x92, 0x4c, 0xc7, 0x63, 0xa6, 0x2d,
			0x5f, 0xcb, 0x06, 0xc3, 0x52, 0x26, 0x42, 0x43,
			0x3b, 0x36, 0xe8, 0x66, 0xfb, 0x6c, 0x62, 0x80,
			0x87, 0x10, 0xec, 0x2d, 0xfc, 0xdd, 0xe7, 0xd2,
			0x12, 0xb3, 0x1b, 0xb5, 0x3c, 0xbc, 0x79, 0xde,
			0x49, 0x13, 0x2c, 0x60, 0xa9, 0x0d, 0x25, 0x85,
		},
	},
	/* SHA1 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0xc8, 0x21, 0x18, 0xdd, 0x81, 0xf0, 0xfe, 0x71,
			0x32, 0xd2, 0xb4, 0x7c, 0xd1, 0xfa, 0x71, 0xe3,
			0xfa, 0x9f, 0x63, 0xb3, 0xda, 0x4e, 0xab, 0x14,
			0x4b, 0x84, 0xaa, 0x78, 0x40, 0x7c, 0x7b, 0x18,
			0x64, 0x46, 0xc0, 0xa1, 0xfe, 0x19, 0xa2, 0x2a,
			0x18, 0xea, 0x13, 0x6a, 0xe0, 0x3d, 0x56, 0x28,
			0x4e, 0x3c, 0xc2, 0xf2, 0x07, 0x06, 0xc3, 0x12,
			0x82, 0x49, 0x21, 0xbe, 0x69, 0x13, 0x85, 0x8f,
			0x42, 0xe5, 0x39, 0x35, 0x4e, 0x0a, 0x2f, 0x68,
			0xb1, 0x26, 0x45, 0x21, 0xd4, 0x12, 0xc7, 0x0b,
			0x08, 0xe7, 0x35, 0x34, 0x6c, 0x59, 0x47, 0x09,
			0x10, 0x47, 0x2f, 0x81, 0x5a, 0x8f, 0x1e, 0x98,
			0xa9, 0x27, 0x09, 0xff, 0x39, 0x6a, 0xad, 0x12,
			0x47, 0xaf, 0xd1, 0x0a, 0x75, 0x20, 0xcb, 0x74,
			0x6f, 0xca, 0x45, 0xe2, 0xcf, 0xf0, 0x41, 0xe0,
			0x3e, 0xa5, 0x5b, 0x29, 0xdc, 0x9a, 0x4e, 0x1b,
		},
		/* S3 : 512 byte */
		{
			0x6a, 0xa3, 0xb2, 0xee, 0xfb, 0x2c, 0xe0, 0xeb,
			0xa6, 0x04, 0x22, 0x2f, 0xcd, 0x93, 0x06, 0x90,
			0x0d, 0xb0, 0x19, 0x87, 0x7b, 0xf5, 0xb0, 0x32,
			0x43, 0x30, 0xfc, 0x78, 0x4f, 0xcb, 0xbb, 0xfe,
			0x93, 0xeb, 0x01, 0xf5, 0xbe, 0x96, 0x4d, 0xfd,
			0x23, 0x18, 0x23, 0x88, 0x50, 0xe5, 0x19, 0x7d,
			0x43, 0xa9, 0x73, 0x82, 0x3f, 0xc4, 0xb9, 0xd5,
			0x96, 0x86, 0x0d, 0x4f, 0xf4, 0xbc, 0x4a, 0x1f,
			0x40, 0x3f, 0xac, 0xf9, 0x92, 0x75, 0x06, 0x2e,
			0x41, 0xc7, 0xca, 0x86, 0x0b, 0xae, 0xf4, 0x52,
			0x79, 0x44, 0xd8, 0x69, 0xa5, 0x13, 0xa0, 0x6d,
			0x96, 0x0f, 0x61, 0xe2, 0x72, 0xfa, 0xe7, 0xa8,
			0x99, 0xf5, 0xa1, 0x33, 0xa0, 0x2c, 0xd2, 0x0f,
			0x45, 0xaa, 0x9a, 0xbb, 0x7f, 0x37, 0x2c, 0xd6,
			0x91, 0x21, 0xcb, 0x9f, 0x3c, 0x0c, 0x51, 0x24,
			0x94, 0xd8, 0xf9, 0x7a, 0xd3, 0xc9, 0xcc, 0x1d,
			0x3a, 0x1a, 0xa3, 0xcb, 0x29, 0xc7, 0xd0, 0x75,
			0x21, 0x8a, 0x56, 0xbe, 0x1f, 0x01, 0x51, 0x64,
			0x50, 0xe4, 0x10, 0x43, 0xdf, 0x5b, 0x18, 0x2a,
			0x59, 0x9d, 0xdd, 0xad, 0xca, 0x1e, 0xae, 0x1c,
			0xb2, 0xd7, 0xae, 0xc9, 0x09, 0x84, 0x08, 0x94,
			0x40, 0x2e, 0x8b, 0x6a, 0x0b, 0x0b, 0x56, 0x67,
			0x71, 0x11, 0x68, 0x81, 0x64, 0xe8, 0xdf, 0x90,
			0x4c, 0x0e, 0xb0, 0x6c, 0xc4, 0xa3, 0x79, 0xfc,
			0x0f, 0x25, 0xfa, 0xc3, 0x24, 0xdc, 0x3d, 0x3d,
			0x4e, 0xf6, 0xbb, 0x2a, 0x59, 0xb5, 0x0a, 0xeb,
			0xa8, 0x15, 0xe4, 0x9d, 0xd7, 0x4e, 0xe1, 0x9e,
			0x79, 0xaf, 0xb1, 0x5b, 0x07, 0xcb, 0x09, 0x15,
			0x1a, 0xea, 0xe0, 0xde, 0xc2, 0x1c, 0x05, 0x2d,
			0xe0, 0xee, 0x9d, 0x85, 0x87, 0x0b, 0xb8, 0xe7,
			0x26, 0x94, 0x78, 0xed, 0xc8, 0x32, 0x04, 0xc3,
			0x00, 0x88, 0x71, 0x82, 0x16, 0x22, 0xae, 0xe8,
			0x76, 0x36, 0x01, 0x8a, 0x62, 0x8b, 0xad, 0x4e,
			0x6c, 0xd9, 0xda, 0x74, 0xca, 0x2a, 0x1b, 0xc9,
			0x00, 0xc1, 0xad, 0x34, 0x62, 0x2e, 0x1d, 0xdf,
			0x4d, 0x13, 0x65, 0x6b, 0x84, 0x93, 0x66, 0x92,
			0xa9, 0x69, 0x75, 0x9e, 0xa9, 0x5c, 0x26, 0xda,
			0xae, 0x78, 0x6c, 0x72, 0x9e, 0x90, 0xd3, 0xef,
			0x2a, 0x76, 0x72, 0xfe, 0xdc, 0xb3, 0x57, 0x2a,
			0x93, 0xdf, 0xea, 0x27, 0x35, 0xee, 0x38, 0xd0,
			0x33, 0xfa, 0xdd, 0x37, 0x52, 0x45, 0xc0, 0xa2,
			0xcd, 0x0b, 0x60, 0x68, 0x7a, 0x96, 0x20, 0xaa,
			0x64, 0x10, 0x5f, 0x41, 0x31, 0xbb, 0x06, 0x76,
			0x06, 0x74, 0x2b, 0x5c, 0xf0, 0x16, 0x96, 0x6c,
			0xe8, 0x07, 0x9b, 0x31, 0x72, 0x25, 0x23, 0x8e,
			0xdb, 0xaf, 0xbc, 0xfd, 0xee, 0x2a, 0x82, 0xce,
			0x19, 0x27, 0x4b, 0xc3, 0x3d, 0x0a, 0x0b, 0xd8,
			0xb3, 0xbf, 0xd6, 0x99, 0x87, 0xf9, 0x5a, 0x4b,
			0x4a, 0x21, 0xed, 0xb3, 0xd7, 0x0d, 0xbc, 0xdd,
			0x68, 0x60, 0x35, 0x5f, 0xd6, 0x2e, 0x98, 0x76,
			0x60, 0x24, 0x03, 0xee, 0xe8, 0xac, 0xc7, 0x9c,
			0xb4, 0x3a, 0x7a, 0x58, 0x36, 0x2d, 0x64, 0x8d,
			0xaf, 0xf0, 0x56, 0x11, 0xf9, 0x7e, 0x85, 0xf3,
			0xd3, 0x51, 0x36, 0x5e, 0x28, 0xfc, 0x4d, 0xfb,
			0x32, 0x8c, 0x4e, 0xcf, 0x6c, 0x0c, 0x88, 0x61,
			0x3d, 0x5b, 0xcc, 0x21, 0xeb, 0xdc, 0xca, 0xad,
			0x2c, 0x0c, 0x62, 0xd4, 0xea, 0xbf, 0x96, 0x8b,
			0x34, 0x03, 0x66, 0x17, 0xe1, 0x92, 0x5f, 0xd9,
			0x08, 0xb0, 0xbf, 0x64, 0x9b, 0xf2, 0xf2, 0x02,
			0xf7, 0xe5, 0xfb, 0xed, 0x6c, 0x8f, 0x4e, 0x96,
			0x05, 0xaf, 0xaa, 0xec, 0xe1, 0x9c, 0x1e, 0xcb,
			0xde, 0x5a, 0x53, 0x04, 0x12, 0x1f, 0xb5, 0x37,
			0xde, 0xbb, 0xab, 0x97, 0x21, 0xb0, 0xb9, 0xb9,
			0x96, 0x4d, 0x8a, 0xb2, 0xb8, 0xcf, 0x98, 0x9f,
		},
	},
	/* SHA1 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0x43, 0x0d, 0xca, 0xb2, 0x51, 0x61, 0x98, 0x2d,
			0x76, 0x6e, 0x38, 0x39, 0xcc, 0x3d, 0xa0, 0xc4,
			0x1b, 0xbe, 0x0d, 0xed, 0x95, 0xb1, 0x50, 0x0c,
			0xc4, 0xc7, 0x68, 0xf2, 0xc0, 0x05, 0x56, 0x72,
			0xa5, 0x6a, 0x2d, 0x20, 0x44, 0xd1, 0x69, 0x58,
			0x19, 0x19, 0x81, 0x82, 0x7e, 0x82, 0xc0, 0x95,
			0x1d, 0x5a, 0x50, 0xda, 0xab, 0xda, 0x98, 0xf3,
			0xd4, 0x4b, 0xc8, 0x1b, 0x71, 0x6c, 0xb1, 0xaf,
			0x01, 0x18, 0xdb, 0xee, 0x57, 0x24, 0xd1, 0xfd,
			0x90, 0x87, 0xc0, 0x2d, 0x48, 0xbd, 0x40, 0x81,
			0x0e, 0x6d, 0x72, 0x6b, 0x14, 0x49, 0x4b, 0xd7,
			0x2b, 0x35, 0x2f, 0x6a, 0xef, 0xac, 0xc2, 0x0c,
			0x3f, 0x60, 0xfb, 0xa4, 0xf1, 0x21, 0x7a, 0x22,
			0xc3, 0x00, 0x1b, 0xc3, 0x94, 0xb4, 0xcb, 0x90,
			0xd4, 0x3d, 0x9a, 0x29, 0x39, 0xc7, 0x12, 0xc0,
			0xb3, 0xd0, 0xc7, 0x68, 0xa3, 0x8d, 0x30, 0x90,
		},
		/* S4 : 512 byte */
		{
			0x42, 0x68, 0x1a, 0xc1, 0x82, 0x8f, 0xc8, 0x64,
			0x03, 0xb3, 0x8e, 0x24, 0xe2, 0x02, 0x24, 0x91,
			0xb8, 0xae, 0xc6, 0xc7, 0x5c, 0x80, 0xfb, 0x3f,
			0x6a, 0x96, 0xd8, 0xdc, 0x7d, 0xf5, 0xe2, 0xa5,
			0xf8, 0xb9, 0xbb, 0xb3, 0xad, 0x68, 0x79, 0xcd,
			0x81, 0x3a, 0x71, 0xeb, 0x2b, 0x84, 0x3b, 0x23,
			0x4d, 0x5f, 0xe7, 0x6e, 0x20, 0x85, 0xf8, 0xd8,
			0x2e, 0x08, 0x5e, 0x4c, 0xbd, 0x1a, 0x05, 0xa1,
			0x52, 0xc8, 0x06, 0x62, 0x92, 0x26, 0xf5, 0xde,
			0x85, 0x34, 0x77, 0x3c, 0x15, 0xb5, 0x66, 0xe5,
			0x38, 0xc2, 0x72, 0xc8, 0xea, 0xd0, 0xb6, 0xad,
			0xb9, 0x86, 0x23, 0xf8, 0x68, 0xf5, 0x4b, 0x76,
			0x5c, 0x7b, 0x09, 0x81, 0xef, 0x91, 0x36, 0xec,
			0x14, 0xb4, 0x2d, 0x79, 0x8e, 0x1a, 0x04, 0x1b,
			0x2b, 0x8e, 0x7e, 0xa7, 0x4b, 0x1d, 0x4b, 0xa5,
			0x6b, 0xa6, 0x4a, 0x92, 0x64, 0x19, 0xc5, 0xc5,
			0x04, 0xd6, 0x36, 0x5d, 0x7f, 0x64, 0xf3, 0xce,
			0x6a, 0x3c, 0x68, 0x9f, 0x67, 0xc4, 0xfe, 0x17,
			0x04, 0xfe, 0x23, 0x73, 0x90, 0x3c, 0xe6, 0x3c,
			0x07, 0xab, 0xa6, 0x75, 0x62, 0xbc, 0x2b, 0xe8,
			0x7a, 0x48, 0x83, 0xe2, 0x0b, 0xe5, 0xb0, 0xa3,
			0xcc, 0xae, 0xb9, 0x0a, 0x97, 0xdb, 0x92, 0x1a,
			0xd5, 0xe6, 0x8c, 0xc7, 0xdd, 0x4c, 0x9c, 0xfb,
			0x5e, 0xa0, 0x4d, 0x89, 0x5e, 0x32, 0xa0, 0x40,
			0xa8, 0x41, 0xb3, 0x4b, 0x78, 0x84, 0x4f, 0x2f,
			0x68, 0x53, 0xb3, 0x85, 0x5b, 0xf4, 0x89, 0xc1,
			0x09, 0x19, 0x64, 0x6a, 0x9d, 0x79, 0xc2, 0xc4,
			0x69, 0x0f, 0xf7, 0xe3, 0x21, 0x85, 0xc1, 0x3d,
			0xcc, 0xa6, 0x6e, 0x21, 0x47, 0x87, 0xdd, 0x58,
			0x06, 0xb4, 0x92, 0xaa, 0xb7, 0x46, 0xf3, 0x5e,
			0xa9, 0x45, 0x8d, 0x7f, 0xaa, 0xbb, 0x35, 0xad,
			0x45, 0xec, 0x76, 0x6d, 0x43, 0x00, 0x96, 0xa2,
			0x0e, 0x51, 0x68, 0xed, 0xe4, 0xc6, 0x36, 0xe8,
			0xdf, 0x4a, 0x2b, 0x99, 0xbc, 0xbe, 0xd3, 0x97,
			0xc4, 0x76, 0xc8, 0xdb, 0x9a, 0xbb, 0x11, 0x35,
			0xf1, 0xb9, 0x5f, 0xb0, 0x22, 0xfb, 0xa0, 0x3a,
			0x6b, 0xb3, 0x57, 0xad, 0x7c, 0x02, 0xee, 0x2e,
			0x6f, 0xc3, 0xbd, 0xec, 0x12, 0x54, 0x25, 0x18,
			0x5c, 0xe4, 0xaf, 0x50, 0xc5, 0xad, 0xb7, 0x17,
			0x5e, 0xac, 0x5f, 0x76, 0x93, 0x78, 0xbd, 0xc9,
			0x5f, 0x1f, 0x1a, 0x06, 0xd2, 0x41, 0x99, 0xdd,
			0x7f, 0x84, 0x65, 0x88, 0x17, 0x61, 0x72, 0xe4,
			0x2e, 0x33, 0x1a, 0x41, 0x86, 0x76, 0x5b, 0x2e,
			0xa9, 0xf9, 0xc8, 0xaf, 0xd7, 0x2a, 0x95, 0xff,
			0x35, 0xdd, 0xfe, 0xd7, 0xf3, 0xc8, 0x0e, 0x2e,
			0x7a, 0x2b, 0x5b, 0x6d, 0x5e, 0x63, 0xad, 0x09,
			0xfd, 0xa5, 0x75, 0x95, 0x48, 0xf7, 0x8e, 0xb1,
			0x25, 0x2f, 0x31, 0x85, 0xf0, 0x23, 0xca, 0x3b,
			0xdd, 0xff, 0x2c, 0x12, 0xcd, 0x52, 0x38, 0x6c,
			0x4c, 0xb0, 0x44, 0x4e, 0x2f, 0x0b, 0xb2, 0x5c,
			0x62, 0x10, 0x2e, 0x4e, 0x4d, 0xef, 0xaf, 0x21,
			0x66, 0x70, 0xf9, 0x53, 0x60, 0x24, 0x35, 0xd1,
			0x38, 0xd1, 0xc2, 0xf3, 0xe6, 0xce, 0xa5, 0xde,
			0x73, 0x5d, 0x9d, 0x75, 0x3b, 0x1e, 0x07, 0x36,
			0x59, 0xae, 0xd8, 0xdd, 0x29, 0x2c, 0xc1, 0x5e,
			0xd5, 0x6d, 0x61, 0x69, 0xd9, 0x16, 0xd1, 0x8f,
			0xe4, 0x7d, 0x31, 0x07, 0x3f, 0x38, 0x67, 0x25,
			0x75, 0xee, 0xa4, 0x80, 0xe4, 0xc6, 0x43, 0x0d,
			0xc4, 0x60, 0x44, 0x7d, 0xbd, 0xde, 0xa2, 0xc0,
			0x8d, 0x92, 0xa0, 0x6f, 0xa8, 0x49, 0x58, 0x95,
			0x63, 0xed, 0xe7, 0xb3, 0x7e, 0x16, 0xba, 0xd4,
			0x33, 0xeb, 0xf9, 0xe2, 0xe5, 0xc3, 0xc3, 0x61,
			0x65, 0x73, 0x23, 0x05, 0xa2, 0x7a, 0x3a, 0x9a,
			0xcd, 0x59, 0x99, 0xc1, 0xf0, 0xb9, 0x25, 0x07,
		},
	},
	/* SHA1 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0xed, 0xd8, 0xd6, 0x09, 0x97, 0x6e, 0x99, 0x73,
			0x81, 0x75, 0x77, 0x46, 0x06, 0x29, 0xc7, 0x49,
			0x61, 0x2e, 0xb3, 0xd2, 0x1e, 0x30, 0x48, 0x8b,
			0xcf, 0xc7, 0xc9, 0x76, 0x00, 0x9d, 0x49, 0x2f,
			0x6d, 0xd2, 0x78, 0x72, 0x72, 0x9f, 0xbe, 0xdc,
			0xc2, 0x3a, 0x86, 0xf6, 0x5c, 0xfa, 0xdd, 0x6f,
			0xbb, 0xbc, 0x3b, 0xa9, 0x4d, 0x85, 0xa0, 0x78,
			0x27, 0x69, 0xe3, 0x26, 0xf7, 0xab, 0x47, 0xba,
			0x3a, 0x25, 0x50, 0x79, 0xaf, 0xea, 0xd5, 0xae,
			0x0b, 0x2b, 0x55, 0xa7, 0xee, 0x57, 0x8f, 0x00,
			0x16, 0x86, 0x4c, 0x24, 0xd3, 0x18, 0x80, 0x10,
			0xba, 0x99, 0x91, 0x5a, 0x9e, 0xe8, 0xf7, 0x72,
			0xbc, 0x99, 0x05, 0x29, 0x36, 0x97, 0x06, 0x6f,
			0xc3, 0x0e, 0x8c, 0x61, 0xa4, 0x88, 0x05, 0x83,
			0xda, 0x56, 0x8f, 0x49, 0x96, 0x31, 0x0e, 0xdd,
			0xf1, 0x21, 0x7b, 0x1c, 0xdd, 0x9e, 0x7f, 0xe1,
		},
		/* S5 : 512 byte */
		{
			0x69, 0xc8, 0x16, 0xe5, 0x5d, 0x6b, 0x3b, 0x3b,
			0xb4, 0x42, 0xad, 0xc9, 0x08, 0xa9, 0x8a, 0x0e,
			0x07, 0x50, 0x0a, 0xf7, 0x4f, 0x9b, 0x63, 0xf8,
			0x8f, 0x4e, 0x31, 0x27, 0x44, 0x0a, 0x43, 0x26,
			0x8b, 0x66, 0xb6, 0xf5, 0x28, 0x46, 0x95, 0xb9,
			0xba, 0x22, 0xed, 0x1a, 0x47, 0x6f, 0x20, 0x6c,
			0x29, 0xf1, 0x55, 0x39, 0xf8, 0x79, 0xbd, 0x45,
			0x1f, 0xfd, 0x40, 0xfa, 0x46, 0x5c, 0xb8, 0xf3,
			0x06, 0xe1, 0x79, 0xc6, 0x3a, 0x5e, 0x9d, 0xdc,
			0x5c, 0x6b, 0x71, 0x5d, 0x3c, 0xa8, 0xcb, 0x5c,
			0xf9, 0x7f, 0xdb, 0x0d, 0xe4, 0xa3, 0x76, 0xe0,
			0xe8, 0x55, 0x3d, 0x4e, 0x0f, 0x86, 0x5e, 0x90,
			0x76, 0x06, 0xcb, 0x1d, 0xdd, 0x93, 0xef, 0x3c,
			0x39, 0xfe, 0xe3, 0x1d, 0xe3, 0xe6, 0x69, 0x87,
			0x44, 0x94, 0xaa, 0x6f, 0x9c, 0x56, 0x53, 0x2d,
			0x3a, 0x7a, 0x89, 0x9c, 0x6b, 0x6c, 0xba, 0x95,
			0x8e, 0x86, 0x02, 0xff, 0x27, 0xa6, 0x9b, 0xd1,
			0xd0, 0x6a, 0x81, 0x9c, 0x6e, 0x64, 0x70, 0xf0,
			0xfa, 0xfe, 0xaa, 0x26, 0x36, 0x1e, 0x0e, 0xa0,
			0xf6, 0x0f, 0x70, 0xb9, 0x4a, 0x15, 0xd7, 0x1f,
			0x89, 0x94, 0xa9, 0x3c, 0x65, 0x06, 0x01, 0x78,
			0xb0, 0x78, 0xc1, 0x44, 0xb5, 0x36, 0x3c, 0x8f,
			0x72, 0xc2, 0x2e, 0x50, 0x73, 0x83, 0x2b, 0x35,
			0xe0, 0x85, 0x7d, 0x6a, 0x44, 0x0b, 0x3a, 0xf2,
			0x0d, 0x27, 0x69, 0x45, 0x9d, 0x05, 0xe4, 0x3f,
			0x82, 0x04, 0x94, 0xc8, 0xad, 0xd8, 0x1b, 0x30,
			0xdf, 0xd9, 0x36, 0xa3, 0xdb, 0x8e, 0xef, 0xdc,
			0xc3, 0xfb, 0xf0, 0x88, 0x76, 0x08, 0xfd, 0xde,
			0x38, 0xe9, 0xad, 0xb3, 0x00, 0xa5, 0x7b, 0x27,
			0xf9, 0xc4, 0x47, 0x64, 0xce, 0xf4, 0x9b, 0x49,
			0xcd, 0x6d, 0xa0, 0xe9, 0xac, 0xdf, 0xd5, 0xd7,
			0x6f, 0xea, 0xa8, 0xac, 0xd1, 0x2d, 0xb5, 0x71,
			0xd5, 0x37, 0xcc, 0xbe, 0x5d, 0xca, 0x7a, 0xaf,
			0x73, 0xcf, 0xdc, 0xba, 0x93, 0xc6, 0x39, 0x13,
			0xc1, 0x56, 0xd2, 0xf3, 0xf5, 0xa2, 0x53, 0xf7,
			0xd8, 0xba, 0xf1, 0xc4, 0x0d, 0x99, 0x04, 0xb4,
			0xb4, 0x4b, 0x36, 0x75, 0x54, 0x42, 0xe4, 0x2c,
			0x82, 0xba, 0x54, 0x83, 0x8d, 0x77, 0x03, 0xbd,
			0x62, 0x56, 0x10, 0x0d, 0xa4, 0x63, 0x94, 0x8a,
			0xf5, 0xa9, 0x90, 0xfc, 0x06, 0x80, 0xf2, 0xdf,
			0xc7, 0xb8, 0x74, 0x1d, 0x85, 0xaf, 0xda, 0x74,
			0x44, 0x5c, 0xd1, 0xe0, 0xe4, 0xe3, 0x19, 0xce,
			0x7a, 0xe6, 0x17, 0x3a, 0xcb, 0x29, 0x86, 0x5e,
			0x8d, 0x94, 0xf2, 0xe0, 0x51, 0x9e, 0xd3, 0x82,
			0x6f, 0x58, 0x71, 0x42, 0x37, 0xdb, 0xcf, 0xac,
			0x73, 0x16, 0x52, 0xa1, 0x20, 0xb9, 0x00, 0xb8,
			0x10, 0x17, 0x66, 0x3d, 0x3d, 0x54, 0xf6, 0x10,
			0xd4, 0x93, 0x83, 0x3f, 0x48, 0x2b, 0x6b, 0xda,
			0xd2, 0xd4, 0x6b, 0x5d, 0x11, 0xf5, 0x6c, 0x48,
			0xc2, 0x05, 0xb1, 0xc1, 0x58, 0x51, 0xf7, 0x22,
			0x8f, 0x1f, 0xca, 0x4f, 0xb3, 0x20, 0xc0, 0xbc,
			0x4b, 0x0a, 0x28, 0x51, 0x03, 0x07, 0x9a, 0x33,
			0x37, 0x08, 0x4a, 0xc4, 0xe1, 0x4b, 0x6a, 0xb2,
			0xfd, 0xc3, 0x9d, 0x4b, 0x44, 0x7f, 0x52, 0x22,
			0xa8, 0x0c, 0xaa, 0x33, 0xf0, 0x15, 0xfb, 0x69,
			0xf6, 0x88, 0xce, 0x3a, 0x6d, 0x05, 0x2b, 0x57,
			0xfe, 0xdc, 0xa1, 0x2d, 0xcb, 0x17, 0x6d, 0x2a,
			0x65, 0x6e, 0x6c, 0x9b, 0x31, 0x9e, 0x39, 0x9a,
			0xa5, 0x5d, 0xca, 0x95, 0x12, 0xd0, 0x9d, 0x2b,
			0x5c, 0x0d, 0x5f, 0xd7, 0x65, 0x08, 0x13, 0x34,
			0xed, 0x5f, 0xe2, 0x7d, 0xad, 0x1e, 0x04, 0x32,
			0x3d, 0xd6, 0x27, 0x96, 0xbc, 0xc6, 0xe8, 0xab,
			0x9e, 0xee, 0xc1, 0x14, 0x9c, 0x20, 0xae, 0xa0,
			0x4e, 0xa3, 0x98, 0x94, 0xe1, 0x49, 0x95, 0x88,
		},
	},
	/* SHA1 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0x82, 0x76, 0x86, 0x81, 0x3a, 0x7c, 0x22, 0xdf,
			0xc8, 0x1c, 0xa7, 0x64, 0x3f, 0xea, 0xe6, 0xeb,
			0x1b, 0x42, 0x5b, 0x53, 0xd3, 0x8c, 0x60, 0xdc,
			0xb2, 0xbf, 0x9f, 0xd4, 0xe2, 0x24, 0xcd, 0x74,
			0xca, 0x5d, 0x96, 0x05, 0x9a, 0xa5, 0xd3, 0x46,
			0x23, 0x59, 0x03, 0x12, 0xed, 0xb8, 0xf9, 0x3e,
			0x83, 0xe7, 0xd2, 0xc1, 0x12, 0x37, 0x2a, 0x17,
			0x32, 0xab, 0x90, 0xe7, 0x04, 0xb5, 0x36, 0xe3,
			0x7d, 0x41, 0xed, 0xef, 0x01, 0xf4, 0x51, 0x86,
			0x34, 0xb6, 0x41, 0x5a, 0x0d, 0x6e, 0x93, 0x3b,
			0x2c, 0x0c, 0x05, 0x3a, 0x6d, 0xcd, 0x76, 0x0e,
			0xbd, 0xba, 0x56, 0xb7, 0x2c, 0x73, 0x5b, 0x61,
			0x44, 0x2a, 0x75, 0x8a, 0x99, 0xf2, 0x05, 0x67,
			0xa2, 0x3a, 0x4a, 0x84, 0xc7, 0x94, 0x1b, 0xac,
			0xbb, 0xbc, 0xdc, 0x08, 0x0c, 0xd5, 0xab, 0x8a,
			0xf9, 0xf5, 0xda, 0x8d, 0x83, 0xe4, 0xa2, 0xb5,
		},
		/* S6 : 512 byte */
		{
			0x9b, 0x02, 0x8d, 0x8d, 0x6c, 0xbd, 0x7e, 0xbf,
			0x0e, 0x70, 0xd0, 0x6f, 0x14, 0x79, 0x43, 0xe0,
			0xb3, 0x6d, 0xfa, 0x5c, 0xa7, 0xc5, 0x14, 0x3f,
			0x98, 0x1b, 0x1e, 0x76, 0xce, 0xac, 0x88, 0x98,
			0x83, 0x9f, 0x10, 0x7b, 0x5b, 0xfc, 0xd8, 0x5c,
			0xd9, 0xaf, 0x5c, 0x27, 0xa2, 0x7c, 0xd1, 0xd4,
			0x5e, 0xbc, 0x28, 0xdf, 0xac, 0x53, 0x4d, 0xe6,
			0x58, 0xb3, 0x87, 0xd8, 0xc4, 0xcc, 0xc5, 0xa0,
			0xd4, 0x6d, 0xe3, 0x6d, 0xf9, 0x55, 0xbb, 0xfb,
			0x36, 0x85, 0xb2, 0x64, 0x15, 0x6b, 0xe3, 0x33,
			0xff, 0xf4, 0xd8, 0xef, 0x8a, 0x34, 0xbd, 0x20,
			0x6f, 0x80, 0xcc, 0xcc, 0xf6, 0xfa, 0xa5, 0x55,
			0x48, 0x31, 0x69, 0x38, 0xf6, 0x8a, 0xd7, 0x5f,
			0xdd, 0xb5, 0x52, 0xc4, 0xb7, 0x2c, 0xf0, 0xfb,
			0x17, 0x28, 0x09, 0x0a, 0x1e, 0xbf, 0x2f, 0xcc,
			0xc1, 0xcb, 0x94, 0xc3, 0xad, 0xa6, 0x5d, 0x78,
			0xdc, 0x07, 0x6b, 0x5c, 0x78, 0xb6, 0xbb, 0x99,
			0x7b, 0xef, 0x8c, 0xb5, 0xce, 0xe6, 0xb1, 0x38,
			0x31, 0x88, 0x81, 0xe5, 0x62, 0x86, 0xf2, 0xb9,
			0xd5, 0x1f, 0xec, 0xf1, 0xda, 0x44, 0x8e, 0xd9,
			0xe4, 0xb0, 0x32, 0x28, 0xe9, 0xfb, 0xb6, 0x58,
			0xd6, 0xbd, 0xdb, 0x04, 0x3e, 0xb9, 0x41, 0x43,
			0x1f, 0xe0, 0x90, 0x8a, 0xfb, 0x80, 0xa9, 0x0a,
			0xda, 0x63, 0xcf, 0x92, 0xe1, 0x6c, 0xc1, 0xaf,
			0x65, 0xba, 0x8c, 0x4e, 0xeb, 0x91, 0x0c, 0x32,
			0x38, 0x3d, 0xbf, 0xa1, 0x40, 0x33, 0x06, 0x05,
			0x03, 0x5e, 0xd1, 0x79, 0x78, 0x80, 0xc8, 0x34,
			0x8c, 0xfb, 0x61, 0x35, 0x25, 0xe0, 0xc5, 0x4f,
			0x82, 0xe6, 0xae, 0x83, 0x85, 0x4e, 0x0e, 0x76,
			0x4d, 0xbd, 0x1b, 0x50, 0x2f, 0x19, 0xaa, 0x37,
			0x2f, 0x0d, 0xd4, 0x81, 0xb1, 0x42, 0x38, 0x69,
			0xf1, 0xfc, 0x21, 0xcf, 0xf3, 0x25, 0xd0, 0xc6,
			0x43, 0xde, 0xb7, 0x12, 0x66, 0xc9, 0x91, 0xc1,
			0x75, 0xa1, 0x5a, 0x2d, 0x3a, 0x0c, 0x6b, 0x04,
			0x39, 0x4f, 0x9d, 0xd9, 0x35, 0x4e, 0x92, 0x15,
			0x46, 0x09, 0x19, 0x95, 0x98, 0x1f, 0xb3, 0xfb,
			0xfc, 0x16, 0x31, 0x06, 0x7c, 0x51, 0x56, 0x0f,
			0x10, 0xfe, 0xc0, 0x97, 0xd2, 0x7a, 0x58, 0x74,
			0x32, 0x38, 0x0d, 0xb3, 0xbc, 0xb6, 0x77, 0xc0,
			0xc2, 0x1a, 0x17, 0x62, 0x26, 0x0f, 0x7f, 0xae,
			0x5f, 0x6d, 0xcb, 0xe5, 0x5e, 0x1d, 0xcc, 0x5d,
			0x60, 0xf8, 0x3a, 0xb3, 0x18, 0x11, 0x00, 0xa2,
			0x83, 0x48, 0x61, 0xe2, 0xc3, 0xe5, 0xdb, 0x2e,
			0x4f, 0xae, 0xe1, 0xa3, 0xda, 0x48, 0xfe, 0x94,
			0xbb, 0xa5, 0x2f, 0x13, 0x83, 0x5a, 0xef, 0x2e,
			0xcf, 0xd9, 0xbe, 0x1c, 0x3e, 0x8f, 0x0e, 0x86,
			0x68, 0xbc, 0x05, 0x46, 0x2d, 0x72, 0x20, 0x46,
			0x57, 0x83, 0xe2, 0x5b, 0xb1, 0xed, 0x20, 0xea,
			0xac, 0xeb, 0xc4, 0x27, 0xc2, 0x09, 0xd2, 0x12,
			0x6d, 0x36, 0xb6, 0x50, 0x93, 0x50, 0xf2, 0x31,
			0x3f, 0x0c, 0x2f, 0x24, 0x78, 0xc2, 0x43, 0x55,
			0x5a, 0x31, 0x97, 0xd7, 0xfa, 0x9f, 0xde, 0x28,
			0xb0, 0xdf, 0xb4, 0x34, 0x18, 0x59, 0x39, 0x2c,
			0xa3, 0xcf, 0x8f, 0xd6, 0xba, 0x6f, 0x81, 0xb5,
			0x79, 0xd6, 0x77, 0x13, 0xcf, 0x9e, 0x9f, 0xef,
			0xbf, 0x22, 0xc6, 0x0d, 0x0b, 0x99, 0xf7, 0x12,
			0x28, 0x57, 0x0b, 0x61, 0x71, 0x72, 0xc8, 0xfd,
			0x2d, 0x02, 0xe8, 0xb1, 0x40, 0x37, 0xa1, 0xb6,
			0x13, 0x0e, 0x7c, 0x35, 0x12, 0x6f, 0xf5, 0xdf,
			0x19, 0x50, 0x33, 0xcb, 0x22, 0x05, 0x88, 0x48,
			0xbc, 0x78, 0x93, 0xb0, 0x6d, 0x21, 0x6b, 0x30,
			0xb2, 0x71, 0xa1, 0x74, 0xb8, 0xe7, 0x33, 0xd7,
			0x56, 0x3c, 0xf4, 0xb2, 0xd1, 0xb6, 0x1f, 0x00,
			0xbf, 0xa7, 0x96, 0xd6, 0x50, 0xf2, 0x83, 0x49,
		},
	},
	/* SHA1 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0xf0, 0x8f, 0xfa, 0x17, 0x16, 0xc6, 0xa0, 0x85,
			0xee, 0x19, 0x51, 0xae, 0xfc, 0x91, 0x47, 0xf5,
			0xe9, 0xbe, 0x1d, 0x99, 0xcc, 0xf1, 0x35, 0x86,
			0xa9, 0xab, 0x57, 0x71, 0x37, 0x54, 0x9a, 0x23,
			0xe7, 0x30, 0x88, 0x15, 0x21, 0x90, 0x7d, 0x3e,
			0xd1, 0x45, 0x50, 0x29, 0x22, 0x5a, 0x35, 0x4a,
			0x89, 0x03, 0xe2, 0x92, 0x06, 0xd0, 0xf1, 0x41,
			0xa7, 0x26, 0x86, 0xfc, 0x4d, 0xf0, 0x31, 0x1f,
			0xb2, 0x92, 0x16, 0x54, 0xab, 0x1e, 0x4b, 0x65,
			0xab, 0xcb, 0x81, 0x4e, 0x89, 0xa3, 0xe7, 0x97,
			0x83, 0x7b, 0x4b, 0xe0, 0x82, 0x28, 0x82, 0xbe,
			0xa0, 0xf0, 0x0b, 0xc5, 0xe7, 0x3f, 0xef, 0xe6,
			0x5b, 0x9b, 0xeb, 0xfc, 0x3b, 0xda, 0xd1, 0xa1,
			0x85, 0x93, 0x2c, 0x46, 0xb8, 0xd2, 0x23, 0x83,
			0xdd, 0x10, 0xa0, 0xab, 0xc8, 0x62, 0xb1, 0xa4,
			0x9c, 0xf0, 0x2e, 0xf5, 0x8d, 0xa9, 0x9c, 0x21,
		},
		/* S7 : 512 byte */
		{
			0xca, 0xe0, 0x90, 0xd0, 0x65, 0xe0, 0x37, 0x83,
			0x74, 0xd9, 0x43, 0x3f, 0x4c, 0xeb, 0xaf, 0x7b,
			0xce, 0xbc, 0xa8, 0x2a, 0x0b, 0x79, 0xec, 0x21,
			0x31, 0x3c, 0x49, 0x06, 0xbf, 0x0f, 0xe1, 0x37,
			0x7a, 0xa3, 0x2f, 0x49, 0xbb, 0x67, 0x1d, 0xb0,
			0x96, 0x07, 0xea, 0xdf, 0xa3, 0xc4, 0xd3, 0x02,
			0xcd, 0x4d, 0xff, 0xb1, 0xc1, 0x4e, 0xa2, 0xf2,
			0x1f, 0xa2, 0xcd, 0x6b, 0x9e, 0x8e, 0x96, 0xa5,
			0x86, 0x7d, 0xd9, 0xdc, 0x0a, 0x5c, 0xf4, 0xa0,
			0x1c, 0xfd, 0xa8, 0x14, 0xb1, 0x36, 0x50, 0xba,
			0xb3, 0x3c, 0xd0, 0xbf, 0x34, 0xa3, 0xa4, 0xb8,
			0x7c, 0x89, 0x83, 0x34, 0x53, 0xf2, 0x47, 0x47,
			0x5a, 0x7b, 0xad, 0x06, 0x52, 0x56, 0x26, 0x7d,
			0x72, 0x6f, 0xa4, 0x13, 0xfa, 0xf4, 0x6d, 0x59,
			0x52, 0x77, 0xe6, 0x8c, 0x70, 0xde, 0x02, 0xb7,
			0x3e, 0x9e, 0x4a, 0xbe, 0x58, 0x94, 0xde, 0x75,
			0x13, 0x2e, 0x23, 0x98, 0xde, 0x94, 0x1f, 0x6e,
			0x05, 0x1d, 0xa9, 0xea, 0xfc, 0x66, 0xca, 0xbc,
			0xf5, 0x96, 0xe2, 0x87, 0x77, 0x5e, 0xa9, 0xb5,
			0x00, 0x42, 0xdc, 0x36, 0x53, 0x04, 0xd1, 0x10,
			0xad, 0x33, 0x11, 0xb0, 0x6e, 0x01, 0x46, 0x09,
			0x6d, 0xb4, 0x9b, 0x93, 0x55, 0x31, 0xc1, 0xf1,
			0x32, 0xd6, 0x4a, 0xb4, 0x2a, 0x84, 0x12, 0x8d,
			0x33, 0x89, 0xb0, 0x31, 0xcf, 0xca, 0xb1, 0x54,
			0xd8, 0x8e, 0x28, 0x1f, 0x27, 0x79, 0x1f, 0x17,
			0xce, 0x18, 0x1d, 0x24, 0xcb, 0x64, 0x54, 0x9f,
			0x0a, 0x93, 0xee, 0x76, 0x0f, 0x19, 0x90, 0x5c,
			0x4d, 0xa8, 0x09, 0x5a, 0x12, 0xcb, 0xb0, 0x1b,
			0x81, 0x62, 0xba, 0xd9, 0x97, 0x6e, 0x07, 0x65,
			0x14, 0x26, 0x54, 0xe7, 0x58, 0x16, 0x27, 0x17,
			0x3b, 0xea, 0x35, 0xd9, 0xaf, 0xd3, 0x68, 0x2f,
			0x0c, 0x25, 0x81, 0xa3, 0x7c, 0x47, 0x20, 0xd1,
			0x10, 0xe3, 0x68, 0xc5, 0x58, 0xc2, 0x20, 0xea,
			0x4d, 0xdd, 0xe5, 0x58, 0xe8, 0x59, 0xbd, 0xb5,
			0x3b, 0x09, 0x1b, 0xaa, 0xd2, 0xde, 0x07, 0xe1,
			0x01, 0xf5, 0x6a, 0xdb, 0x0a, 0x1a, 0x24, 0x0f,
			0x69, 0x73, 0x58, 0xe8, 0xfa, 0x7b, 0x74, 0x5d,
			0xe4, 0x3e, 0x0f, 0x76, 0xe5, 0xff, 0xd5, 0xda,
			0x1d, 0xd3, 0xcf, 0x63, 0xc8, 0x13, 0x6b, 0x00,
			0x1d, 0x10, 0x19, 0x03, 0x4e, 0x68, 0x8a, 0x96,
			0x02, 0x06, 0xe3, 0x0e, 0x04, 0xbb, 0xeb, 0x37,
			0xe7, 0x07, 0x05, 0x50, 0x4a, 0x8e, 0x68, 0x02,
			0xcb, 0x92, 0x40, 0x7d, 0x55, 0xba, 0x80, 0x0f,
			0x7e, 0x93, 0x5c, 0xcf, 0x28, 0xed, 0xe9, 0x05,
			0x3d, 0xd3, 0xec, 0x56, 0x46, 0x09, 0xb0, 0x2b,
			0x34, 0x40, 0x2b, 0x64, 0xec, 0xf0, 0x13, 0xb4,
			0x07, 0x52, 0xa2, 0xf2, 0x89, 0x45, 0xe3, 0x7b,
			0x93, 0x39, 0x17, 0x6a, 0x6d, 0x86, 0x49, 0x58,
			0x99, 0xd4, 0xaf, 0xfb, 0xec, 0xc8, 0xd1, 0x81,
			0x2d, 0x53, 0x6f, 0xc1, 0xc4, 0x8b, 0x7f, 0x33,
			0x45, 0x24, 0x1d, 0x5d, 0x5d, 0x18, 0xe7, 0xf5,
			0xa2, 0x42, 0x3f, 0xf3, 0x74, 0x97, 0x90, 0x45,
			0x9c, 0xb4, 0x7d, 0x71, 0xef, 0xf5, 0x79, 0xcd,
			0xaa, 0x76, 0xd2, 0x11, 0x7b, 0x53, 0x1c, 0xcf,
			0xfb, 0x48, 0x19, 0x04, 0xe7, 0x11, 0x01, 0x9b,
			0xa5, 0x0c, 0x04, 0x4b, 0x4b, 0x89, 0x86, 0x57,
			0x1b, 0xd5, 0xc3, 0xe0, 0xc9, 0xff, 0x28, 0x56,
			0x99, 0x3c, 0x2e, 0x59, 0x53, 0x5e, 0x5b, 0x6c,
			0xb0, 0x70, 0xbd, 0x22, 0x7d, 0xcf, 0x50, 0x14,
			0x9a, 0xe2, 0x15, 0xc2, 0xf4, 0x29, 0x0f, 0x43,
			0x18, 0x72, 0x0e, 0xb3, 0x3d, 0x50, 0xd1, 0xa1,
			0x3d, 0x96, 0x1b, 0xd1, 0xd9, 0x07, 0x85, 0x1a,
			0x79, 0x4d, 0x21, 0x17, 0x99, 0xa6, 0xfc, 0xff,
			0x7d, 0xda, 0x92, 0xab, 0xf9, 0x7b, 0x22, 0xe8,
		},
	},
	/* SHA1 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0xe1, 0x29, 0x29, 0xd7, 0x30, 0xaa, 0xa1, 0xbb,
			0xdf, 0xcd, 0xe3, 0x2c, 0x1d, 0x98, 0x30, 0x7e,
			0xc4, 0x0f, 0xca, 0xf3, 0xdc, 0xa8, 0xa0, 0xd1,
			0x61, 0xfe, 0x2d, 0xb7, 0x45, 0xc9, 0x9e, 0x30,
			0xb9, 0x0e, 0x33, 0x89, 0xa9, 0x93, 0xe7, 0xe0,
			0x85, 0x96, 0x53, 0x7e, 0x47, 0xbe, 0xd0, 0x67,
			0xb2, 0x52, 0xdc, 0x2e, 0xd7, 0xa9, 0xbf, 0x0e,
			0xde, 0xe2, 0x6e, 0x37, 0x03, 0xd5, 0xaf, 0x66,
			0xdf, 0x48, 0x7e, 0xb8, 0xe9, 0x67, 0xaf, 0xc0,
			0x3c, 0x6e, 0xd6, 0xd7, 0x9b, 0xa9, 0x64, 0xb3,
			0xb2, 0x9e, 0x12, 0x3c, 0x9e, 0x89, 0x83, 0x9a,
			0xb0, 0x08, 0x24, 0x93, 0x6e, 0x18, 0x58, 0x14,
			0xdb, 0x40, 0xd2, 0x0d, 0xf9, 0x0c, 0x41, 0x82,
			0x8b, 0xbd, 0x33, 0xe0, 0x1f, 0x3f, 0x33, 0x8d,
			0x4c, 0x5b, 0x35, 0xc2, 0x39, 0x2e, 0x35, 0x9d,
			0x5e, 0xda, 0x1a, 0x72, 0x8f, 0x7b, 0xe0, 0x63,
		},
		/* S8 : 512 byte */
		{
			0x7f, 0x08, 0xe8, 0x19, 0xe3, 0xcd, 0xab, 0x1a,
			0x45, 0xc3, 0xee, 0x5c, 0xfd, 0x9c, 0x30, 0x23,
			0x07, 0xd7, 0x9b, 0x83, 0x1c, 0x7f, 0x4c, 0xdf,
			0x03, 0x78, 0xae, 0x6f, 0xfd, 0xf8, 0x4d, 0x89,
			0x00, 0xb8, 0xfe, 0x8c, 0xb8, 0x9c, 0x35, 0x36,
			0xa1, 0x21, 0x9c, 0x12, 0x14, 0x25, 0x54, 0x37,
			0xda, 0x9a, 0xc5, 0xe2, 0xed, 0x67, 0x9d, 0x56,
			0x77, 0xe3, 0x70, 0xf3, 0x01, 0xfd, 0x2c, 0x0c,
			0x9f, 0x53, 0x27, 0xea, 0x24, 0x94, 0x60, 0x83,
			0x37, 0x59, 0x5e, 0x4e, 0x22, 0xed, 0x07, 0xc8,
			0xaa, 0x4b, 0xfc, 0x16, 0x00, 0xa3, 0x68, 0xa3,
			0x49, 0x75, 0x08, 0x91, 0x1f, 0x23, 0x6c, 0x4c,
			0x1b, 0xd7, 0xc2, 0xd2, 0xae, 0xfb, 0x74, 0x42,
			0x15, 0xac, 0xb2, 0xec, 0xaa, 0xbe, 0x6f, 0xd0,
			0xc4, 0xbb, 0x32, 0x6f, 0xa9, 0x37, 0x75, 0xb4,
			0x38, 0xa8, 0xd5, 0x31, 0x43, 0x30, 0xc2, 0x43,
			0x45, 0xfd, 0xce, 0x85, 0x65, 0x72, 0xcf, 0x91,
			0x02, 0x1b, 0x6f, 0x75, 0xd4, 0x07, 0xc4, 0xc4,
			0x34, 0x14, 0x73, 0x47, 0x0f, 0xd8, 0x4c, 0x45,
			0x87, 0x03, 0x7c, 0x59, 0xe0, 0x2e, 0xb0, 0x09,
			0x28, 0x7c, 0x0e, 0x51, 0x44, 0x4e, 0x39, 0x78,
			0x76, 0xe2, 0xbf, 0x5c, 0xda, 0x80, 0xa0, 0x1b,
			0xbd, 0xd5, 0xc7, 0xa6, 0x0c, 0x60, 0x20, 0xc3,
			0x6f, 0x37, 0x99, 0xc8, 0x9f, 0xd2, 0xb6, 0xcf,
			0x5d, 0xcb, 0x94, 0xa3, 0x3c, 0x7e, 0x04, 0xed,
			0x21, 0xd6, 0xca, 0x38, 0x14, 0x47, 0x16, 0xbf,
			0xbd, 0x96, 0xb9, 0xf5, 0x10, 0xc5, 0xd7, 0x01,
			0xe5, 0x6a, 0x0b, 0xc2, 0x3d, 0x55, 0x5d, 0xc5,
			0xcc, 0x53, 0xe8, 0x08, 0x06, 0x90, 0x52, 0x04,
			0xd8, 0x67, 0xee, 0xd6, 0xac, 0xeb, 0x40, 0xbd,
			0xf6, 0x63, 0xab, 0x23, 0x83, 0xdf, 0xdd, 0x1d,
			0xb4, 0x8d, 0x6c, 0xbb, 0xe0, 0x00, 0x5f, 0x69,
			0xb2, 0x4f, 0x19, 0xf9, 0xcd, 0xc9, 0xaf, 0x34,
			0xf7, 0xe5, 0xdc, 0x18, 0xf9, 0xcf, 0x0f, 0xa3,
			0x4e, 0xb4, 0xee, 0xb6, 0x73, 0x2c, 0xc0, 0x5c,
			0x34, 0x2e, 0x33, 0xc4, 0xcf, 0x79, 0x95, 0xc6,
			0x47, 0x8e, 0xf2, 0xf6, 0x4f, 0x01, 0x8d, 0xfa,
			0x0f, 0x08, 0x5b, 0x1a, 0x12, 0xb1, 0x79, 0xc3,
			0xcb, 0x58, 0x25, 0x63, 0x31, 0x89, 0xbb, 0x4a,
			0xff, 0xc0, 0xa0, 0x01, 0xa7, 0xc6, 0xb8, 0xbf,
			0x4e, 0x4a, 0x04, 0x7b, 0x63, 0x7a, 0x06, 0x59,
			0x74, 0xad, 0xbc, 0xac, 0x3c, 0xef, 0xbb, 0x42,
			0x10, 0xac, 0xd7, 0x3a, 0xfd, 0xe1, 0x8c, 0x79,
			0xdf, 0xdb, 0x1f, 0xff, 0xdf, 0x91, 0x6b, 0x30,
			0xa8, 0x0b, 0x3f, 0xf8, 0x2d, 0x39, 0xfc, 0x4d,
			0xea, 0xec, 0xdc, 0xe0, 0x5c, 0xa4, 0xfc, 0xbb,
			0xa7, 0xd5, 0xc7, 0x19, 0x81, 0x71, 0x0a, 0xde,
			0x12, 0x69, 0x86, 0xfe, 0x48, 0xfe, 0x4b, 0x11,
			0x8f, 0x99, 0x52, 0x01, 0x56, 0x67, 0xbe, 0x3f,
			0xed, 0x47, 0x3c, 0x36, 0x2d, 0x27, 0x3e, 0xc3,
			0xd3, 0xd9, 0xd2, 0x60, 0xdc, 0x39, 0x8d, 0x89,
			0x2b, 0xeb, 0xe1, 0xc1, 0x3a, 0xf3, 0x3c, 0xfe,
			0x32, 0x3c, 0x7b, 0x13, 0x71, 0x0e, 0x16, 0x52,
			0xfb, 0x8e, 0xe5, 0xcb, 0x4c, 0x06, 0x53, 0xab,
			0x2f, 0x47, 0xd1, 0xbf, 0xdd, 0x21, 0x7e, 0x30,
			0xd0, 0x82, 0xb9, 0xa2, 0x7c, 0xc1, 0xf1, 0x78,
			0x5c, 0x98, 0x26, 0xbe, 0xb5, 0x38, 0x17, 0xb5,
			0xc7, 0x63, 0xe2, 0x17, 0xaa, 0x06, 0xc7, 0x23,
			0x19, 0xbb, 0x77, 0xb4, 0x7a, 0xb3, 0x0a, 0x0c,
			0x58, 0xd8, 0x0e, 0xec, 0x4b, 0x4b, 0xce, 0xea,
			0x94, 0x78, 0xae, 0x1a, 0x35, 0x27, 0x7c, 0x61,
			0xac, 0x42, 0xb0, 0xac, 0x4b, 0x97, 0x00, 0xd1,
			0x4e, 0x85, 0x20, 0x37, 0x08, 0xb7, 0x56, 0x80,
			0x23, 0x9a, 0x6d, 0x50, 0xbc, 0x9a, 0xaa, 0x7f,
		},
	},
	/* SHA1 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0x76, 0x6e, 0x0b, 0xd6, 0x02, 0x85, 0xd5, 0x8b,
			0xea, 0xff, 0x18, 0xf4, 0x5c, 0xdc, 0x21, 0x35,
			0xc1, 0x77, 0x32, 0x9c, 0xfb, 0xb8, 0xe0, 0x7a,
			0xeb, 0x8c, 0xba, 0x31, 0x37, 0x92, 0x2b, 0x8f,
			0x4a, 0x18, 0xa2, 0x5a, 0x9c, 0x45, 0xef, 0x2c,
			0x33, 0x97, 0xf1, 0x81, 0x6a, 0x91, 0x6e, 0x0f,
			0xf1, 0xad, 0x42, 0x56, 0x15, 0x98, 0xbc, 0x77,
			0x41, 0x9e, 0x45, 0x49, 0xc5, 0x63, 0x42, 0xaa,
			0xd7, 0xd5, 0x3c, 0x13, 0xd7, 0x5e, 0x3e, 0x70,
			0xd8, 0x5e, 0x7f, 0x2e, 0x5b, 0xd8, 0x41, 0x95,
			0x5d, 0x6c, 0xf9, 0x58, 0x95, 0x85, 0xb0, 0x6a,
			0xf7, 0xd3, 0xa9, 0x14, 0x7c, 0xfc, 0x44, 0x45,
			0x25, 0x8e, 0x22, 0x29, 0x3e, 0x39, 0x8a, 0xfc,
			0xe0, 0x3b, 0x0c, 0x21, 0xb9, 0x1b, 0x64, 0x6e,
			0x0f, 0x9a, 0xa1, 0xcb, 0x03, 0xe7, 0x86, 0xfb,
			0x13, 0x12, 0x30, 0xb9, 0xe3, 0xc3, 0xf9, 0x07,
		},
		/* S9 : 512 byte */
		{
			0xd1, 0x7c, 0x50, 0xb5, 0xb0, 0x52, 0x6a, 0x1e,
			0x41, 0x2d, 0x79, 0x3b, 0x36, 0x4a, 0x3c, 0xc6,
			0x90, 0xd4, 0x0b, 0x5a, 0xd0, 0xeb, 0xf6, 0x27,
			0xad, 0x46, 0x9e, 0xe0, 0x1a, 0xf9, 0x86, 0xc5,
			0x1f, 0xcd, 0x10, 0x45, 0x2e, 0xbd, 0x90, 0xa4,
			0x65, 0x7a, 0xea, 0x9f, 0x90, 0xda, 0x2c, 0x9a,
			0x9c, 0xa7, 0x20, 0xf0, 0x82, 0xb0, 0x12, 0xa0,
			0xbd, 0x9b, 0x7b, 0x03, 0x2e, 0x53, 0xff, 0x0c,
			0xdd, 0x62, 0x75, 0x86, 0x02, 0x57, 0x28, 0x22,
			0x7f, 0x98, 0x94, 0xda, 0x05, 0xd7, 0x4d, 0x38,
			0x13, 0x79, 0xd3, 0x23, 0x58, 0x33, 0xfd, 0xc0,
			0x28, 0xb9, 0xb3, 0x0d, 0xc7, 0x81, 0x8c, 0xa2,
			0x72, 0x08, 0x26, 0xea, 0x44, 0x04, 0xf6, 0x55,
			0xf1, 0x02, 0x98, 0xb6, 0x2f, 0xe5, 0xca, 0xe1,
			0x87, 0x41, 0x7e, 0x97, 0x73, 0xf2, 0xb2, 0xec,
			0x8a, 0xf1, 0x7b, 0xa3, 0xbc, 0x09, 0x83, 0x7d,
			0xfa, 0x77, 0x67, 0xae, 0x48, 0xe5, 0x7f, 0x4e,
			0xcf, 0x07, 0xdb, 0xe3, 0x05, 0x8d, 0x25, 0xf8,
			0xac, 0xd4, 0xee, 0x78, 0x4b, 0xae, 0x85, 0x43,
			0x5f, 0xed, 0x18, 0x29, 0x79, 0x96, 0xc6, 0xa7,
			0x8c, 0x96, 0x16, 0xa6, 0xa9, 0x20, 0xe5, 0x24,
			0xae, 0x3d, 0x61, 0xbd, 0xa1, 0x25, 0x46, 0x09,
			0xed, 0xe6, 0x81, 0xb8, 0x07, 0xa7, 0x52, 0x9d,
			0xb1, 0xd5, 0xd6, 0x4a, 0xb7, 0x42, 0x7a, 0x40,
			0x29, 0xd7, 0xf8, 0xb1, 0x31, 0xef, 0x70, 0x72,
			0xf6, 0x6a, 0x30, 0x22, 0x13, 0xed, 0x13, 0xce,
			0x45, 0x1d, 0xa0, 0xc5, 0xb9, 0x8f, 0x90, 0xe2,
			0x76, 0xd5, 0xcf, 0x67, 0x5c, 0x9f, 0xef, 0xde,
			0x45, 0x7d, 0xf9, 0x1c, 0x89, 0x7f, 0x77, 0x42,
			0x6d, 0xb9, 0x46, 0x33, 0xe3, 0x45, 0xf3, 0xea,
			0xc0, 0x0c, 0x82, 0x3b, 0x62, 0x1e, 0xc6, 0xee,
			0xd4, 0x5a, 0x0f, 0xcd, 0x72, 0x47, 0x0b, 0x4c,
			0xee, 0x83, 0x90, 0x9b, 0xea, 0xae, 0x4e, 0xb4,
			0x16, 0x36, 0xff, 0x14, 0x82, 0x2d, 0xd7, 0x44,
			0x17, 0xb0, 0x44, 0x73, 0x16, 0xab, 0x33, 0xd1,
			0xa4, 0xcc, 0x7a, 0x7b, 0xf0, 0x25, 0x3f, 0xbf,
			0x81, 0x2a, 0x17, 0xfa, 0xd5, 0xfa, 0x63, 0xeb,
			0xd4, 0x1e, 0xaa, 0xf7, 0xab, 0xaf, 0xb8, 0xf4,
			0x1b, 0x09, 0x68, 0x31, 0xde, 0xf9, 0x5c, 0xe6,
			0xec, 0xa0, 0x29, 0x11, 0x49, 0x14, 0x57, 0x3b,
			0x8e, 0xeb, 0xfa, 0x96, 0xa2, 0x96, 0xb7, 0xe6,
			0xbc, 0xc9, 0xfa, 0xd8, 0x3b, 0xdc, 0x1d, 0x6e,
			0xcf, 0x3e, 0x6b, 0x22, 0xce, 0x9f, 0x3d, 0xc8,
			0x7a, 0x68, 0xae, 0x91, 0x8b, 0x7d, 0xaf, 0x98,
			0xee, 0x9c, 0x4e, 0x4d, 0x7a, 0xda, 0x23, 0x20,
			0xf1, 0x9e, 0x96, 0x37, 0x25, 0x4c, 0x72, 0xc2,
			0x9d, 0xf2, 0x72, 0x2c, 0xd9, 0xf4, 0x72, 0xf4,
			0x21, 0xe7, 0x74, 0x19, 0x48, 0x6f, 0xce, 0x49,
			0xd5, 0xdc, 0x7f, 0x4e, 0x14, 0x22, 0x4e, 0x31,
			0x2a, 0x28, 0x76, 0x9f, 0x85, 0x3a, 0x2b, 0x2b,
			0xe7, 0x3d, 0xae, 0x99, 0xbd, 0x4a, 0xe4, 0xd4,
			0x82, 0xaf, 0x32, 0xaf, 0xac, 0xc5, 0x81, 0xa9,
			0xf3, 0x49, 0xe7, 0x4f, 0xb1, 0x72, 0x9e, 0xb4,
			0x70, 0x6a, 0xc2, 0x13, 0x49, 0x3e, 0x84, 0x5f,
			0xc3, 0x7e, 0x03, 0xe4, 0xf2, 0x34, 0xda, 0x3f,
			0xba, 0x03, 0xc6, 0x20, 0xb2, 0x34, 0x88, 0x9a,
			0x3c, 0xad, 0xa9, 0x8f, 0xe1, 0xcc, 0x0b, 0x5f,
			0x5e, 0x10, 0x8e, 0x5c, 0x03, 0x64, 0xd7, 0x19,
			0x13, 0xfa, 0x7c, 0x0b, 0x42, 0x21, 0x57, 0x88,
			0xdf, 0x40, 0x92, 0x50, 0xf3, 0xcd, 0x72, 0xe8,
			0x22, 0xfd, 0x66, 0x35, 0x1a, 0xb2, 0xd9, 0x60,
			0xb6, 0xc4, 0x3b, 0xb3, 0x6a, 0x5c, 0x8f, 0x3e,
			0x62, 0x7a, 0x3e, 0x80, 0x62, 0x80, 0x2e, 0xe5,
			0x6f, 0x8f, 0xf7, 0x93, 0x6a, 0x52, 0xcc, 0xf9,
		},
	},
};

/* 10 elements */
static mod4096_t SHA224[] = {
	/* SHA224 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0xbd, 0xa7, 0x9b, 0x56, 0x4e, 0x1e, 0xf6, 0x44,
			0xc7, 0x29, 0xad, 0xd4, 0x3c, 0xa1, 0x08, 0xbb,
			0xdb, 0x05, 0x58, 0x99, 0x16, 0x11, 0x38, 0x2c,
			0x05, 0x92, 0x13, 0xc3, 0x05, 0x78, 0xe7, 0xf6,
			0xa6, 0x2d, 0x48, 0x95, 0xc4, 0x25, 0xa7, 0xde,
			0xf3, 0xb7, 0xd4, 0xf3, 0x21, 0x2b, 0xc2, 0x8b,
			0x76, 0x14, 0x8b, 0xe7, 0x7f, 0xe0, 0xb3, 0x08,
			0x8c, 0xad, 0xb1, 0x1f, 0x56, 0x5e, 0xab, 0x45,
			0xe5, 0xee, 0x82, 0x6c, 0xb1, 0x05, 0x0e, 0x95,
			0x08, 0xc9, 0xab, 0x44, 0xaf, 0xf7, 0xf4, 0x31,
			0xcc, 0x32, 0xce, 0x41, 0x03, 0x9b, 0xa9, 0xb4,
			0xe0, 0xa1, 0x02, 0xa0, 0x75, 0x1b, 0x2c, 0x9e,
			0x44, 0x94, 0x51, 0xc3, 0x5f, 0xe5, 0xb4, 0xd2,
			0xcd, 0x0a, 0x38, 0xaf, 0x30, 0x2e, 0x26, 0x17,
			0xa5, 0x08, 0x2b, 0x8f, 0xec, 0x6a, 0xa0, 0x9a,
			0xc9, 0x17, 0x0e, 0x2f, 0xa9, 0xd6, 0xf2, 0x6a,
		},
		/* S0 : 512 byte */
		{
			0x83, 0xac, 0xbc, 0x8e, 0x1f, 0x82, 0x17, 0xb7,
			0xa2, 0xf3, 0x6b, 0x57, 0xc6, 0x24, 0x29, 0xed,
			0x7a, 0x31, 0xf7, 0x93, 0x4f, 0xb1, 0x98, 0x36,
			0x56, 0x0a, 0x6b, 0xa1, 0x15, 0x17, 0x4c, 0x4d,
			0x58, 0x7d, 0x73, 0x56, 0x4a, 0x7d, 0x0d, 0x49,
			0xdd, 0x40, 0xcf, 0xc4, 0x6c, 0x52, 0xe5, 0x9e,
			0x46, 0xde, 0xd2, 0x47, 0xd9, 0x06, 0x8a, 0xd8,
			0xed, 0xe7, 0xdc, 0x4e, 0x7a, 0xce, 0xdd, 0x6c,
			0x22, 0x31, 0x6d, 0x5e, 0x6c, 0x4e, 0xf5, 0x7e,
			0xfc, 0x98, 0x6f, 0x75, 0x23, 0x68, 0x89, 0xe7,
			0x1b, 0x95, 0x8c, 0xa5, 0xc6, 0x58, 0xa6, 0xd1,
			0xd9, 0x0b, 0xe8, 0x7e, 0xdf, 0xa9, 0x41, 0x81,
			0x6d, 0xa9, 0x25, 0x33, 0xc8, 0x5d, 0xb6, 0x7f,
			0x46, 0x26, 0x20, 0x2b, 0x7d, 0xd0, 0xf5, 0xdc,
			0x9b, 0x17, 0x93, 0xa7, 0x41, 0x86, 0x05, 0x7c,
			0xba, 0x6f, 0x27, 0x7c, 0x18, 0x97, 0x61, 0x2c,
			0x13, 0xb5, 0x13, 0x90, 0x58, 0x98, 0x87, 0x1e,
			0x27, 0x60, 0x51, 0xae, 0x29, 0x5a, 0x77, 0xf7,
			0x08, 0x71, 0x32, 0x46, 0xf7, 0x2c, 0x7f, 0x0a,
			0xfb, 0x8f, 0xfd, 0x39, 0xcd, 0x37, 0xf5, 0xa1,
			0x25, 0xe0, 0xdc, 0xd7, 0x68, 0xe3, 0x08, 0x69,
			0xf8, 0x38, 0xcd, 0xa6, 0x00, 0x58, 0x58, 0xe7,
			0xfb, 0xe1, 0xdb, 0xaa, 0xd2, 0x1d, 0x57, 0x5d,
			0x5c, 0x63, 0x76, 0xb7, 0x69, 0x8a, 0x3a, 0xfd,
			0x3a, 0x52, 0xa3, 0x49, 0xb3, 0xd8, 0x86, 0x67,
			0xe0, 0xc2, 0x97, 0x4d, 0xcd, 0x55, 0x94, 0x70,
			0xfe, 0xf4, 0x77, 0x71, 0x68, 0x48, 0x4c, 0x21,
			0x77, 0x23, 0x9d, 0xec, 0x43, 0xf2, 0xc2, 0xbd,
			0x2f, 0x56, 0x82, 0xcd, 0x89, 0x15, 0x05, 0x15,
			0x92, 0x29, 0x07, 0x16, 0x05, 0x39, 0x90, 0x10,
			0x90, 0xfd, 0xb5, 0x28, 0x0f, 0x8b, 0x10, 0x70,
			0xa5, 0x6b, 0x3b, 0xf3, 0x26, 0x6b, 0x9f, 0x90,
			0xf6, 0xe4, 0x77, 0xb1, 0x1b, 0x3d, 0xe9, 0xc7,
			0x31, 0xd7, 0x7d, 0x2e, 0xcb, 0xdf, 0x5a, 0x92,
			0x35, 0xd3, 0xe5, 0xef, 0x0a, 0xbb, 0x44, 0x57,
			0x3d, 0xaa, 0x1d, 0x24, 0xef, 0x96, 0xa6, 0x5e,
			0xa4, 0x0f, 0x0d, 0x2e, 0xe5, 0xc4, 0x11, 0x8f,
			0x35, 0x86, 0x62, 0x03, 0x5d, 0x0e, 0x9f, 0x0e,
			0xac, 0xa3, 0xc2, 0xfd, 0x02, 0x17, 0x5a, 0x26,
			0x07, 0x54, 0x91, 0x08, 0x36, 0x52, 0x1d, 0x72,
			0x99, 0x4e, 0xa6, 0xd8, 0x4f, 0x35, 0xed, 0xd7,
			0xcd, 0x0c, 0x61, 0xde, 0x94, 0x04, 0xfe, 0x6b,
			0x26, 0x47, 0xea, 0x6e, 0x6e, 0xcd, 0xfc, 0xbc,
			0xdd, 0xad, 0x28, 0x79, 0xda, 0x89, 0xb0, 0x16,
			0x1d, 0x97, 0x28, 0x62, 0xbf, 0x2c, 0x24, 0x9c,
			0xda, 0x8e, 0xf2, 0xf3, 0x6f, 0xa5, 0xee, 0xaa,
			0xef, 0x14, 0xf2, 0xd4, 0x3f, 0x29, 0xd8, 0x3a,
			0x2a, 0x63, 0x52, 0xee, 0xde, 0x58, 0x29, 0x6b,
			0xaf, 0x05, 0x05, 0xa4, 0xec, 0x5f, 0x15, 0x93,
			0x1c, 0xa4, 0xdf, 0x7e, 0xbe, 0x3d, 0x84, 0xfd,
			0x9b, 0x64, 0x62, 0xdc, 0x6e, 0xce, 0x7a, 0xbf,
			0x37, 0xec, 0x73, 0xf4, 0x04, 0x83, 0x8e, 0x29,
			0x6a, 0x88, 0x29, 0xb2, 0x7d, 0x75, 0xd2, 0xfb,
			0x43, 0xbd, 0xdc, 0x2a, 0x49, 0x62, 0xce, 0x7f,
			0x05, 0x71, 0x06, 0xd7, 0x7e, 0xed, 0xdc, 0x64,
			0xd1, 0x16, 0xb0, 0xf4, 0x96, 0x4a, 0x0c, 0x7a,
			0xbe, 0x52, 0xd5, 0x17, 0xf6, 0xe5, 0x1a, 0x99,
			0xa7, 0xa7, 0xbc, 0xc8, 0x8e, 0xbb, 0x42, 0x0e,
			0xef, 0x2e, 0x41, 0x23, 0xa1, 0xf5, 0xb6, 0x78,
			0xe8, 0x65, 0x7f, 0xc0, 0x0f, 0xc7, 0xe9, 0x7a,
			0xc6, 0xdd, 0xe1, 0x90, 0xc0, 0xf1, 0x48, 0x87,
			0x42, 0x5c, 0x30, 0xed, 0xc7, 0x7b, 0xa6, 0xf7,
			0x51, 0x2f, 0x41, 0xfb, 0x3e, 0x2c, 0x0b, 0x01,
			0x75, 0xd7, 0xb7, 0x55, 0x73, 0xe3, 0xd3, 0x32,
		},
	},
	/* SHA224 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0x6a, 0x25, 0xde, 0x67, 0x60, 0xa9, 0x62, 0xe6,
			0x01, 0xaa, 0xee, 0xd7, 0xb1, 0x52, 0x11, 0x59,
			0xa3, 0x55, 0xc2, 0x5e, 0x91, 0xd4, 0xac, 0xaf,
			0xb0, 0x80, 0xc9, 0x31, 0xe3, 0x02, 0x32, 0x66,
			0x2e, 0x9b, 0x36, 0x36, 0x6e, 0x07, 0xa9, 0xf0,
			0x48, 0xa8, 0x7e, 0x8c, 0x60, 0xe0, 0x99, 0x33,
			0xcb, 0xbf, 0xde, 0x01, 0xc6, 0xde, 0x8e, 0x09,
			0xa8, 0x8a, 0xd3, 0x99, 0x15, 0x25, 0x99, 0xc0,
			0x75, 0xca, 0xa0, 0x0e, 0xa4, 0x30, 0xcf, 0x9b,
			0xeb, 0xd1, 0xcc, 0x4a, 0x30, 0x4b, 0xc9, 0x4f,
			0x69, 0x06, 0xa1, 0xef, 0x57, 0x17, 0x23, 0x87,
			0x70, 0x64, 0xad, 0xd0, 0x1d, 0x09, 0xfe, 0x61,
			0x25, 0xfb, 0x45, 0x76, 0x69, 0xeb, 0x7d, 0xef,
			0xd3, 0x6e, 0xfd, 0x9c, 0x89, 0x5a, 0x4b, 0xb7,
			0x66, 0x73, 0x32, 0xb2, 0xaa, 0x47, 0x14, 0x34,
			0xf0, 0x56, 0x81, 0x5e, 0x61, 0x76, 0x07, 0x52,
		},
		/* S1 : 512 byte */
		{
			0x61, 0xc8, 0x07, 0xee, 0x46, 0x5f, 0xd1, 0xd8,
			0x25, 0xab, 0x5d, 0x99, 0x72, 0x4b, 0x65, 0x9b,
			0xd9, 0x58, 0x3f, 0xec, 0x75, 0xcd, 0x16, 0x45,
			0x39, 0x77, 0x2e, 0x74, 0x65, 0x2b, 0xc6, 0x1e,
			0x53, 0xe4, 0xb8, 0x10, 0x21, 0xec, 0x44, 0x69,
			0x8a, 0xe9, 0x13, 0xea, 0x0b, 0x6d, 0x23, 0x70,
			0x25, 0x8f, 0xf5, 0xeb, 0x28, 0x94, 0xe4, 0x66,
			0x1c, 0x25, 0x0c, 0xf8, 0x7d, 0x81, 0x29, 0x94,
			0x73, 0xa4, 0x6e, 0x8f, 0x93, 0x01, 0x4b, 0x11,
			0x6c, 0x9f, 0x7f, 0xa0, 0xf4, 0xb8, 0x9f, 0x56,
			0xe8, 0x9b, 0x40, 0xaa, 0xe7, 0xc4, 0x14, 0x43,
			0xc2, 0x8b, 0xee, 0x17, 0x41, 0xce, 0x58, 0x6e,
			0x6f, 0x2b, 0xcb, 0x91, 0x71, 0x54, 0x47, 0xaf,
			0x4e, 0x7d, 0xc6, 0x4a, 0xe8, 0x43, 0x17, 0x6f,
			0x37, 0x29, 0xac, 0x4e, 0xed, 0xb8, 0x8c, 0x88,
			0xe4, 0x63, 0x01, 0xa0, 0x82, 0xef, 0x65, 0x6b,
			0xba, 0xd3, 0xba, 0xe1, 0x48, 0x69, 0x71, 0x5e,
			0x31, 0x7c, 0xe0, 0xff, 0x1e, 0x6d, 0xf0, 0x02,
			0x85, 0xae, 0xbd, 0xb5, 0xea, 0x2a, 0x1d, 0x1d,
			0x56, 0x4f, 0xcc, 0x67, 0xea, 0xdb, 0x23, 0x64,
			0xb7, 0xb7, 0x3a, 0x94, 0x82, 0x92, 0x60, 0x7a,
			0x89, 0xc8, 0x21, 0x33, 0x2b, 0x19, 0xd7, 0xa4,
			0x8e, 0x0e, 0xe2, 0x9e, 0xb2, 0x3f, 0x01, 0x91,
			0x59, 0x31, 0x0e, 0xb6, 0x13, 0xf3, 0x3e, 0x4a,
			0x26, 0xc3, 0x3b, 0xa6, 0x60, 0x43, 0xcb, 0x84,
			0xef, 0xea, 0xd9, 0x79, 0xdc, 0x27, 0x73, 0xff,
			0xf3, 0x3c, 0x6e, 0x2d, 0x10, 0x5b, 0x8c, 0xc0,
			0x69, 0x2b, 0xc7, 0x06, 0xa5, 0x02, 0x36, 0x1d,
			0xc7, 0x91, 0x8a, 0xfc, 0x5e, 0x84, 0x39, 0x3a,
			0x70, 0x38, 0x80, 0xb8, 0x0e, 0xb9, 0x42, 0x4d,
			0x66, 0x1e, 0x8b, 0xe1, 0x48, 0x27, 0xa6, 0x7f,
			0x49, 0xc3, 0xde, 0x2c, 0xf0, 0x06, 0xed, 0x26,
			0x0b, 0x9c, 0xdc, 0x52, 0x08, 0xe6, 0x23, 0xcc,
			0x3f, 0x6e, 0x4d, 0xb2, 0x5b, 0x3e, 0x9b, 0xfc,
			0x7b, 0xe6, 0xc0, 0xd7, 0xf9, 0xeb, 0x3d, 0xdd,
			0xb9, 0x18, 0x31, 0x29, 0xbb, 0x97, 0xc3, 0x39,
			0xfb, 0x75, 0x99, 0x36, 0xc9, 0x77, 0x10, 0xca,
			0x4f, 0x5e, 0x1c, 0xc5, 0x24, 0xd0, 0x3f, 0xf9,
			0xb0, 0x3a, 0x50, 0x45, 0xd1, 0x5d, 0x95, 0x1e,
			0x73, 0x9c, 0x91, 0x0e, 0xfd, 0x9e, 0xa2, 0xed,
			0xf7, 0xe4, 0x27, 0x94, 0xfa, 0xa1, 0x88, 0xbb,
			0x13, 0x79, 0x7d, 0xd4, 0x75, 0x79, 0x34, 0xb4,
			0x52, 0x95, 0x24, 0xa3, 0x28, 0x95, 0x7d, 0x65,
			0x5a, 0x85, 0xa3, 0xa8, 0x88, 0xe8, 0xd0, 0x5f,
			0x6d, 0x9d, 0xa9, 0xad, 0x45, 0x4d, 0xcf, 0xb1,
			0x9d, 0xa6, 0x4a, 0xd3, 0x5a, 0x73, 0xfa, 0x7c,
			0xf8, 0xca, 0x82, 0x7c, 0xa8, 0xa3, 0x08, 0x67,
			0x44, 0x30, 0x6e, 0xb6, 0x80, 0xac, 0xb4, 0x30,
			0xd6, 0x0f, 0xff, 0x15, 0x04, 0x94, 0xce, 0x1b,
			0x17, 0x02, 0xc1, 0x0b, 0x53, 0xdb, 0xe4, 0x9a,
			0x2a, 0xa5, 0x01, 0x91, 0x6a, 0x82, 0xc7, 0xee,
			0x72, 0xc2, 0xff, 0xf9, 0x5f, 0x06, 0x3d, 0x00,
			0xcf, 0x57, 0x2b, 0xcc, 0x94, 0xec, 0x0d, 0xfc,
			0x8d, 0x38, 0xf6, 0xcf, 0x64, 0x37, 0xa1, 0x80,
			0x2a, 0x0b, 0x78, 0xc4, 0x5a, 0xbe, 0x65, 0x4c,
			0x52, 0x7d, 0xfb, 0x97, 0x04, 0x7c, 0x9e, 0xd1,
			0xca, 0xfa, 0x37, 0xa9, 0xc5, 0x54, 0x99, 0x77,
			0x78, 0x07, 0x24, 0x75, 0xe5, 0x17, 0xf7, 0x65,
			0xe7, 0x1d, 0xf8, 0xaf, 0x22, 0x22, 0x95, 0xdc,
			0x55, 0x15, 0xfe, 0xfe, 0xe1, 0xa2, 0x83, 0xa1,
			0x71, 0x33, 0x3d, 0xf9, 0xcd, 0x37, 0xc9, 0xd3,
			0xef, 0x69, 0x7a, 0xd6, 0x1f, 0xb4, 0x21, 0xb5,
			0xa1, 0x0e, 0x36, 0x8d, 0xcb, 0x98, 0xca, 0xce,
			0xbc, 0x9a, 0x34, 0xa6, 0x42, 0x64, 0xfc, 0xf3,
		},
	},
	/* SHA224 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x73, 0x10, 0xae, 0x61, 0xcc, 0x71, 0x83, 0x3c,
			0x74, 0x0f, 0x26, 0x1d, 0xb8, 0xd3, 0x6b, 0xd6,
			0x31, 0xc2, 0xd7, 0x2e, 0x98, 0x7e, 0x0d, 0x86,
			0xda, 0x10, 0x8a, 0xbd, 0x6f, 0xe4, 0x92, 0x52,
			0xe8, 0x15, 0x6a, 0xc8, 0xc8, 0xec, 0xc8, 0x4f,
			0x95, 0xa3, 0x88, 0xa0, 0x2f, 0x6e, 0xa0, 0x29,
			0x60, 0xbc, 0x56, 0x6a, 0xb3, 0xb9, 0xb1, 0x9f,
			0x8a, 0x18, 0x2d, 0x9f, 0x6b, 0x11, 0xb2, 0x04,
			0x9b, 0xa5, 0xc7, 0x8c, 0xee, 0x29, 0x04, 0xbf,
			0xdd, 0x02, 0xc5, 0xd2, 0xd9, 0xae, 0x26, 0xcc,
			0xdf, 0x04, 0x2b, 0xe7, 0x4d, 0xbc, 0x3e, 0x39,
			0xf4, 0x31, 0xdf, 0x93, 0x6e, 0x66, 0x97, 0xaa,
			0xb1, 0x47, 0x2d, 0x74, 0xe5, 0x32, 0x1d, 0xd5,
			0x73, 0xb6, 0x46, 0x7b, 0x5f, 0x9b, 0xe3, 0x07,
			0xd7, 0x22, 0xbc, 0x6d, 0x0c, 0x8d, 0x9e, 0x9d,
			0x40, 0xe2, 0x08, 0x63, 0x20, 0xed, 0x27, 0x8b,
		},
		/* S2 : 512 byte */
		{
			0x9f, 0xd6, 0x34, 0x05, 0xda, 0xe6, 0xa2, 0x44,
			0x1b, 0xa0, 0x08, 0x46, 0x1a, 0x20, 0x85, 0xea,
			0xf0, 0x31, 0xee, 0x30, 0xeb, 0x71, 0x0f, 0x9b,
			0x35, 0xfb, 0x46, 0x9c, 0x3d, 0x20, 0x25, 0xf7,
			0x9d, 0x48, 0xf2, 0x95, 0x11, 0x0f, 0x7c, 0x34,
			0xda, 0x4a, 0xa6, 0x74, 0x59, 0xc7, 0x73, 0x2a,
			0x3e, 0x55, 0xad, 0xa0, 0x8a, 0xdb, 0x96, 0xf6,
			0x79, 0xc6, 0x36, 0xa9, 0x89, 0xaa, 0xc7, 0xd5,
			0xe9, 0x06, 0x77, 0x6f, 0xdf, 0x27, 0xe8, 0x6c,
			0x07, 0x1f, 0x25, 0x63, 0xec, 0x88, 0x8c, 0x37,
			0x3f, 0xd9, 0xd3, 0x6b, 0x88, 0xc0, 0xe1, 0x0c,
			0x2b, 0x6e, 0x17, 0xca, 0x4d, 0xe6, 0x02, 0xeb,
			0xc0, 0x76, 0xab, 0x6d, 0x85, 0x45, 0xb5, 0x6b,
			0x1e, 0x34, 0xec, 0x7f, 0xb3, 0xf8, 0x12, 0x93,
			0xd2, 0x7a, 0xad, 0x82, 0xc6, 0x1e, 0x52, 0x93,
			0x9d, 0x41, 0xee, 0x02, 0x32, 0xfb, 0x3b, 0x55,
			0x41, 0x14, 0x54, 0x8f, 0xda, 0x2b, 0xad, 0xaf,
			0xa6, 0xf0, 0x71, 0x91, 0xcc, 0xec, 0x84, 0xb6,
			0x0a, 0x4c, 0x11, 0x17, 0x55, 0x7e, 0xb4, 0x5f,
			0xcb, 0x76, 0x3b, 0xbd, 0xfa, 0xcf, 0x81, 0xb4,
			0xb9, 0xcd, 0x98, 0x0c, 0x0f, 0xc5, 0x33, 0xf1,
			0xd1, 0xaf, 0xcd, 0x6f, 0xa8, 0x07, 0xa2, 0x60,
			0x92, 0xe8, 0x70, 0x04, 0x17, 0x24, 0x92, 0xb1,
			0xf3, 0x3b, 0xa4, 0x3c, 0xcf, 0x7d, 0x3e, 0x17,
			0xc3, 0xf4, 0xbe, 0xcc, 0x74, 0x83, 0xda, 0x9c,
			0x7b, 0xc0, 0xb6, 0x6e, 0x85, 0xa3, 0xe9, 0xea,
			0x73, 0x34, 0xe8, 0xbb, 0x4e, 0x9b, 0x37, 0xe1,
			0xca, 0xe1, 0x3f, 0xca, 0xc8, 0xc2, 0x90, 0xf1,
			0x17, 0x45, 0x24, 0x05, 0x7e, 0x56, 0x77, 0x58,
			0x17, 0x36, 0x8c, 0xed, 0x13, 0x53, 0xf1, 0xbc,
			0xfd, 0x6f, 0xfd, 0xfb, 0x05, 0xed, 0x5a, 0xe8,
			0x39, 0xd7, 0xce, 0x22, 0x2a, 0xa6, 0x02, 0x88,
			0x89, 0x95, 0x17, 0xed, 0x8d, 0x32, 0x80, 0x09,
			0x0e, 0xff, 0x7b, 0x8e, 0x8a, 0x80, 0xb8, 0xe6,
			0x43, 0x65, 0x68, 0x05, 0x76, 0xc8, 0xe3, 0x8d,
			0x3b, 0xbf, 0xa8, 0x95, 0xae, 0xa7, 0x2d, 0xe3,
			0x00, 0xaf, 0x91, 0x9c, 0x7b, 0xb9, 0xfc, 0x86,
			0x7f, 0xd3, 0x12, 0x25, 0x27, 0xbc, 0x34, 0x84,
			0xc7, 0x50, 0x9b, 0x0c, 0x88, 0x9c, 0x83, 0x6e,
			0x71, 0x36, 0x65, 0x36, 0x2e, 0xa5, 0xce, 0x7e,
			0x62, 0x59, 0xb4, 0xed, 0xd3, 0x36, 0xa9, 0xd1,
			0xe7, 0x7c, 0x54, 0xda, 0x3c, 0xb2, 0x2c, 0xbb,
			0x25, 0xb4, 0xe7, 0x47, 0xb4, 0xf3, 0x7f, 0xf1,
			0xc9, 0x89, 0x62, 0x34, 0xbe, 0x8e, 0x34, 0xd0,
			0xd2, 0x53, 0x6d, 0xb4, 0xe9, 0xb7, 0x2e, 0x71,
			0x27, 0x12, 0x6e, 0x99, 0x3f, 0xf0, 0x31, 0x82,
			0xaf, 0xb0, 0xf0, 0xdb, 0xc2, 0xb4, 0xff, 0x59,
			0x0e, 0x51, 0x94, 0x93, 0x3a, 0x55, 0x6f, 0x4c,
			0x93, 0x86, 0x6a, 0xde, 0x23, 0x6c, 0xab, 0x40,
			0x5e, 0x0b, 0x5f, 0x75, 0xd4, 0xc9, 0x2c, 0xcf,
			0x8f, 0x2d, 0x0d, 0x69, 0x0b, 0xee, 0x78, 0x32,
			0x1b, 0x5d, 0x0a, 0x81, 0x3b, 0xf3, 0x82, 0xbd,
			0xa3, 0x27, 0x10, 0xfe, 0x89, 0x82, 0xf0, 0x7f,
			0xd6, 0x36, 0x2c, 0x0d, 0x47, 0xfb, 0x62, 0x3b,
			0xf5, 0x36, 0x0a, 0xbd, 0x1b, 0x1d, 0x7f, 0x32,
			0x6f, 0xf0, 0xb6, 0x5e, 0x92, 0xa7, 0x59, 0x44,
			0xc9, 0xc5, 0x58, 0x91, 0xb5, 0xad, 0x03, 0x9d,
			0xa8, 0xc6, 0x0a, 0xe0, 0x3a, 0x58, 0x4b, 0x98,
			0xf4, 0x8c, 0xaf, 0xa9, 0xbf, 0x06, 0xac, 0x70,
			0xdc, 0x54, 0xb0, 0x1c, 0x05, 0xa6, 0x15, 0xe0,
			0x1a, 0x61, 0x02, 0xb2, 0x12, 0x4c, 0xf2, 0x12,
			0xae, 0x93, 0x0b, 0xf0, 0xbe, 0x4d, 0x4b, 0xfb,
			0xad, 0x6f, 0x24, 0x92, 0x2f, 0x0b, 0x79, 0xd6,
			0x42, 0x65, 0x76, 0x3a, 0x47, 0x25, 0x92, 0xdf,
		},
	},
	/* SHA224 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0xa5, 0x26, 0xa9, 0xfe, 0x90, 0x96, 0x01, 0x3c,
			0x48, 0x59, 0x36, 0x62, 0xc6, 0x56, 0x23, 0x32,
			0xd0, 0x03, 0xf3, 0x79, 0x15, 0x2b, 0xda, 0xee,
			0x79, 0xee, 0x03, 0x5e, 0xae, 0x09, 0xc9, 0x06,
			0x04, 0x57, 0x29, 0x41, 0x82, 0x98, 0x77, 0x63,
			0x35, 0x72, 0x51, 0xe4, 0xc8, 0xb5, 0xd2, 0xc1,
			0x55, 0xe3, 0x07, 0x1f, 0x7b, 0xf0, 0xa7, 0x31,
			0xb1, 0x46, 0xfe, 0x5d, 0xd9, 0x0c, 0x2b, 0x9a,
			0x19, 0xa8, 0x8b, 0x1d, 0xe6, 0x82, 0x75, 0x34,
			0x61, 0x0b, 0x28, 0x72, 0xe7, 0xff, 0x01, 0x0c,
			0xe3, 0xe7, 0xf2, 0x83, 0xe7, 0xe1, 0x2f, 0x83,
			0xfc, 0xa3, 0x2a, 0x0c, 0x36, 0xbe, 0xc2, 0xcf,
			0x8b, 0x20, 0xb7, 0xdb, 0xe4, 0xdd, 0x62, 0x78,
			0x75, 0x2f, 0xa3, 0x53, 0x4a, 0xbb, 0x1a, 0xdc,
			0x23, 0x15, 0x9a, 0xed, 0x82, 0x87, 0xda, 0xac,
			0x02, 0x0e, 0x6e, 0xb2, 0xea, 0xa6, 0xf6, 0x6d,
		},
		/* S3 : 512 byte */
		{
			0xc7, 0x6a, 0x2c, 0xbb, 0xc4, 0x7f, 0xb1, 0x90,
			0xd4, 0x57, 0x3e, 0xaf, 0xf8, 0xba, 0x30, 0x15,
			0xad, 0xaf, 0xc5, 0x43, 0xf9, 0x62, 0x65, 0x17,
			0x80, 0x48, 0xb7, 0x6e, 0x87, 0x35, 0xf4, 0xa4,
			0x78, 0xc7, 0x64, 0xe4, 0x92, 0x51, 0x95, 0x8d,
			0x86, 0x86, 0xfc, 0x4a, 0x81, 0x4a, 0xd3, 0x17,
			0xe6, 0x20, 0x2e, 0x44, 0xe3, 0x65, 0x02, 0x78,
			0x92, 0xdb, 0x59, 0x98, 0x8a, 0xa6, 0xb3, 0xe7,
			0x00, 0xd1, 0x8d, 0xc6, 0x48, 0xa4, 0x4b, 0x01,
			0xbb, 0x26, 0x7c, 0xd3, 0xee, 0x63, 0x7a, 0x63,
			0x3e, 0xbd, 0xb6, 0xcd, 0xa7, 0x42, 0x56, 0x4d,
			0xef, 0x52, 0x98, 0x8d, 0x6e, 0x23, 0xde, 0x4f,
			0x85, 0x1a, 0xa4, 0xbc, 0xa4, 0xf2, 0x3f, 0xdf,
			0x1d, 0xe7, 0x05, 0xc8, 0x58, 0x82, 0x56, 0xa8,
			0xe1, 0x5e, 0x02, 0xb8, 0x55, 0x32, 0x1f, 0xd1,
			0xbb, 0xf5, 0x43, 0xf5, 0x60, 0xe0, 0xc6, 0x4f,
			0x3a, 0x29, 0x7d, 0x38, 0x4f, 0xcf, 0x60, 0x08,
			0xc7, 0x55, 0x73, 0x83, 0x0a, 0x5f, 0xd8, 0x26,
			0x4d, 0x9b, 0x89, 0x90, 0xd7, 0xeb, 0xa3, 0x92,
			0xd2, 0x7f, 0xcb, 0x41, 0xd7, 0x0f, 0x8b, 0xa1,
			0x8d, 0x48, 0x93, 0x8a, 0x89, 0x9a, 0x1a, 0x27,
			0xce, 0x2a, 0xa8, 0x60, 0xca, 0x7b, 0xff, 0x3c,
			0xc4, 0x12, 0xba, 0x87, 0x1b, 0x05, 0xc1, 0x46,
			0x1a, 0xac, 0x8c, 0x3a, 0xbe, 0x9c, 0x3e, 0xe5,
			0x4b, 0x44, 0xba, 0xbe, 0x09, 0x68, 0x53, 0x8c,
			0x4e, 0x32, 0x45, 0x36, 0xf1, 0x59, 0xc3, 0xa3,
			0xe1, 0x60, 0x10, 0xa9, 0x31, 0x4c, 0x74, 0x7f,
			0x1a, 0xe3, 0x25, 0xf0, 0xdf, 0xee, 0xf5, 0xc6,
			0x03, 0x01, 0x2a, 0xdc, 0x0f, 0xfc, 0x5f, 0x71,
			0x81, 0x20, 0xc2, 0x29, 0xfd, 0x46, 0x2c, 0x6c,
			0xdb, 0x9d, 0xde, 0x80, 0x22, 0xf2, 0x74, 0x28,
			0x5f, 0x86, 0xd4, 0x4b, 0xd6, 0xea, 0xfa, 0xae,
			0x48, 0x77, 0x27, 0x4a, 0xe4, 0x17, 0xef, 0xb1,
			0xbf, 0xf6, 0xff, 0x78, 0x1c, 0xa1, 0xce, 0x8e,
			0x4c, 0xa5, 0xe0, 0x87, 0x85, 0x65, 0xff, 0xe9,
			0x1c, 0x2c, 0xf2, 0x17, 0x58, 0x2d, 0x75, 0xfa,
			0x66, 0xb9, 0x94, 0x61, 0x5e, 0x7f, 0xc2, 0xee,
			0xbe, 0xa6, 0x66, 0x8e, 0x2f, 0x4b, 0x23, 0x61,
			0xc6, 0xfc, 0x3c, 0x14, 0x19, 0x6f, 0x76, 0xcf,
			0xe6, 0xbb, 0xca, 0x66, 0x78, 0xc7, 0xbd, 0x01,
			0x1e, 0xc8, 0xbb, 0xe1, 0x88, 0x1f, 0x55, 0x39,
			0x38, 0xf0, 0xa5, 0xa8, 0x59, 0x6a, 0x59, 0xee,
			0xe8, 0x86, 0x00, 0xb6, 0xce, 0xbf, 0xa2, 0xab,
			0x9e, 0x8f, 0xce, 0x27, 0x20, 0x8d, 0x0a, 0x5f,
			0x52, 0x8f, 0x03, 0x5f, 0xd9, 0xeb, 0x57, 0x8d,
			0xc6, 0xa5, 0xe4, 0xae, 0xa2, 0xcc, 0x6b, 0x1b,
			0x53, 0x76, 0xcc, 0x91, 0x42, 0x73, 0x65, 0x6f,
			0x7f, 0x26, 0x8b, 0x69, 0x5d, 0x02, 0xcc, 0xca,
			0x60, 0x75, 0xb2, 0xd9, 0x56, 0x82, 0xb7, 0x88,
			0x95, 0xa4, 0xf9, 0x96, 0x7d, 0x6a, 0x4e, 0x0a,
			0x52, 0xa8, 0x4a, 0x74, 0xf2, 0xd0, 0xdf, 0x95,
			0x38, 0x07, 0x39, 0x18, 0x7a, 0x53, 0x4e, 0x3f,
			0x7c, 0xb9, 0xbb, 0xd4, 0x33, 0x16, 0x36, 0x30,
			0x6e, 0x83, 0xb8, 0xc8, 0x75, 0x85, 0x98, 0x79,
			0x34, 0xd4, 0xea, 0x7e, 0x96, 0xed, 0x35, 0xdf,
			0x8c, 0xed, 0x39, 0x39, 0x40, 0xb2, 0xbf, 0x6a,
			0x31, 0xff, 0x1c, 0x97, 0x24, 0x09, 0xe8, 0x00,
			0x78, 0xf5, 0xde, 0xc8, 0xa5, 0x66, 0x16, 0xe1,
			0x71, 0x51, 0xc6, 0x0e, 0x5e, 0xef, 0x2d, 0xe5,
			0x7a, 0x70, 0x1e, 0x41, 0xfa, 0x0f, 0xde, 0x22,
			0xd1, 0x8f, 0xd5, 0xec, 0x5d, 0x7e, 0x77, 0x1b,
			0xf9, 0xdc, 0x12, 0x3a, 0xa8, 0x47, 0x41, 0x34,
			0x70, 0xd1, 0xcf, 0x8c, 0xf1, 0x43, 0x3f, 0xb7,
			0xf3, 0xc7, 0xfe, 0x90, 0xc3, 0x51, 0xbd, 0xd1,
		},
	},
	/* SHA224 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0x9e, 0x7f, 0x3f, 0x09, 0xfd, 0xda, 0x08, 0x5e,
			0x2e, 0xf1, 0xfe, 0x08, 0xca, 0x90, 0x0f, 0x4e,
			0xcd, 0x4b, 0xcb, 0x90, 0xe3, 0xb7, 0xc6, 0x22,
			0xe2, 0xda, 0x3e, 0x5b, 0x97, 0xdf, 0xfe, 0xe8,
			0xc5, 0x0e, 0x82, 0x66, 0x6b, 0xfc, 0x5c, 0x31,
			0x46, 0xff, 0xd7, 0x76, 0x97, 0xe5, 0xd9, 0x90,
			0x26, 0xe6, 0x0e, 0x91, 0x87, 0xd6, 0x42, 0x1e,
			0x9c, 0xa0, 0x0f, 0x81, 0x5b, 0xef, 0xb9, 0xf9,
			0xd1, 0x2e, 0x56, 0x54, 0x67, 0xb3, 0x32, 0xc0,
			0x65, 0x37, 0x71, 0xdf, 0xb4, 0x8a, 0xf6, 0x19,
			0xe8, 0x84, 0x84, 0x36, 0x7e, 0x3f, 0x23, 0x2c,
			0x61, 0x83, 0xc6, 0x35, 0xb3, 0x82, 0x2a, 0x25,
			0xcb, 0xc6, 0x01, 0xfb, 0x7a, 0x67, 0x50, 0xb6,
			0x93, 0x81, 0x16, 0x6a, 0xaa, 0xe5, 0x2e, 0x92,
			0x1b, 0x0f, 0x76, 0xa8, 0x4f, 0x59, 0x31, 0xf3,
			0x3a, 0x6e, 0xaa, 0xa1, 0x28, 0x00, 0x1b, 0x15,
		},
		/* S4 : 512 byte */
		{
			0xba, 0x3f, 0x84, 0x96, 0x0e, 0xaa, 0x90, 0xb6,
			0x80, 0x06, 0xc7, 0x4f, 0x70, 0xdf, 0xd6, 0xad,
			0x8c, 0x52, 0x93, 0x99, 0xcb, 0xc6, 0xb4, 0xf1,
			0x01, 0xb1, 0x6a, 0xb9, 0x02, 0x78, 0x96, 0x25,
			0x39, 0xdc, 0x41, 0x5f, 0x62, 0xd2, 0x6c, 0x28,
			0x99, 0xe3, 0x9d, 0x34, 0xd2, 0x77, 0x0e, 0x53,
			0x74, 0x7d, 0x81, 0x6e, 0x3f, 0xfd, 0xe2, 0xc4,
			0x82, 0x17, 0x37, 0x24, 0x0d, 0x73, 0xa4, 0x29,
			0xe8, 0x32, 0xe0, 0x46, 0x4b, 0x05, 0x09, 0x94,
			0x37, 0x70, 0x4f, 0x9c, 0x1a, 0x50, 0xe8, 0x0c,
			0x66, 0xdf, 0xcc, 0x0c, 0xef, 0x4d, 0xfc, 0x81,
			0x0e, 0xb7, 0x00, 0xd5, 0x59, 0x81, 0x8a, 0xf8,
			0x0e, 0xd6, 0xae, 0xa5, 0x40, 0xaa, 0x91, 0x7b,
			0x6a, 0x83, 0xe1, 0x63, 0xce, 0x46, 0x12, 0x98,
			0xad, 0x85, 0x3d, 0xc2, 0xe6, 0xb0, 0x12, 0x7e,
			0x16, 0xb7, 0x1a, 0xcb, 0x62, 0x78, 0x51, 0x1c,
			0xd1, 0x7b, 0x2c, 0x93, 0x20, 0xdf, 0xf8, 0xe7,
			0xf3, 0xc1, 0x8e, 0x64, 0xca, 0xbb, 0xda, 0x7e,
			0x1b, 0xd2, 0x5d, 0xc1, 0xaa, 0x49, 0x72, 0x01,
			0x7a, 0x24, 0xe0, 0xcb, 0x2c, 0x55, 0x02, 0xf2,
			0xf5, 0xdf, 0x33, 0x9a, 0x83, 0x1f, 0x33, 0x86,
			0xb5, 0x4c, 0x5c, 0x8c, 0x8d, 0x5a, 0xc7, 0x0e,
			0x17, 0xc3, 0x2c, 0x8a, 0x56, 0xdc, 0x92, 0x77,
			0x71, 0xe9, 0xcf, 0xd6, 0x34, 0x13, 0xbd, 0x8f,
			0xec, 0xc0, 0x8c, 0xd3, 0xbb, 0x06, 0xbf, 0xe5,
			0x23, 0x3c, 0x1d, 0xc8, 0x64, 0x0c, 0x4c, 0x1a,
			0x22, 0xa5, 0x91, 0x86, 0x17, 0xaf, 0x45, 0x73,
			0xd0, 0x79, 0x45, 0x2a, 0x4a, 0xc0, 0x53, 0x50,
			0x9f, 0x76, 0x3c, 0xe3, 0xe2, 0xd1, 0xb1, 0xed,
			0x3f, 0xd3, 0x7f, 0x47, 0xa1, 0x15, 0x31, 0x8c,
			0x16, 0x62, 0x38, 0x69, 0xf2, 0x99, 0xdd, 0xaa,
			0x63, 0xc1, 0x32, 0xa7, 0xb3, 0x0f, 0xe8, 0xff,
			0xe0, 0x60, 0x45, 0xef, 0xd7, 0x0e, 0xd8, 0x25,
			0xad, 0xf3, 0xec, 0x8e, 0x0a, 0x49, 0x17, 0x94,
			0x9a, 0x9b, 0x17, 0xe4, 0x1c, 0xb1, 0xe3, 0x75,
			0x67, 0x8d, 0x4b, 0xe9, 0x15, 0x12, 0x62, 0x92,
			0x0f, 0xde, 0xf1, 0xd7, 0x5a, 0x95, 0xf9, 0xe7,
			0xca, 0x07, 0x79, 0x02, 0x5b, 0xcd, 0x5b, 0x4f,
			0xcc, 0x0a, 0xae, 0xc1, 0xb5, 0x02, 0x75, 0xed,
			0x0e, 0xf5, 0x44, 0xad, 0xba, 0xbc, 0x90, 0xd3,
			0xba, 0x77, 0x4c, 0xfd, 0xe6, 0xf5, 0xe3, 0xb8,
			0x47, 0x7f, 0xce, 0x50, 0x13, 0x0f, 0x63, 0x52,
			0xf3, 0x1e, 0x47, 0x08, 0xbc, 0x14, 0x66, 0x55,
			0x67, 0x0d, 0x3c, 0x7a, 0x08, 0x12, 0xb2, 0x1d,
			0x8c, 0xb7, 0x9d, 0xa3, 0x46, 0x1c, 0x60, 0xf8,
			0x3b, 0xb8, 0x13, 0x1a, 0xb7, 0xc1, 0x32, 0x5c,
			0xce, 0x54, 0x84, 0xfd, 0xf3, 0x62, 0x72, 0x1f,
			0xde, 0x71, 0xca, 0x50, 0x45, 0x2c, 0x0f, 0x64,
			0x07, 0x50, 0x09, 0x3f, 0xcf, 0xde, 0xe4, 0x6d,
			0xb9, 0x62, 0xe8, 0x9a, 0x5e, 0x3b, 0x35, 0xa9,
			0xc8, 0xa5, 0xb5, 0x33, 0xf0, 0x2a, 0xd1, 0x93,
			0xde, 0x62, 0x21, 0x83, 0xa9, 0x11, 0x61, 0xe0,
			0x6f, 0xd7, 0x1d, 0xdf, 0x60, 0xee, 0x07, 0x98,
			0x40, 0x6a, 0x19, 0xe4, 0x64, 0xee, 0x1f, 0x50,
			0x43, 0xe8, 0xca, 0xd2, 0x6c, 0x7e, 0x01, 0xbe,
			0x6c, 0x65, 0xed, 0x3e, 0x87, 0xd4, 0xcd, 0x72,
			0x26, 0xdd, 0x37, 0x8b, 0xf1, 0x50, 0xfb, 0x15,
			0x86, 0x92, 0x69, 0xd5, 0xb1, 0xd7, 0x3d, 0x29,
			0xb9, 0x68, 0x33, 0x86, 0xd7, 0x7d, 0xa3, 0x7a,
			0x68, 0x55, 0xa7, 0x68, 0x07, 0xe1, 0xb3, 0x2e,
			0x9a, 0x02, 0xf3, 0xb4, 0xcd, 0xa2, 0xc1, 0xf8,
			0x8d, 0x0c, 0xd7, 0x5c, 0x47, 0xd9, 0xde, 0xe8,
			0x43, 0x20, 0x64, 0x3e, 0x00, 0x12, 0xaa, 0xbe,
			0x8b, 0xe9, 0x25, 0xf0, 0x93, 0xa5, 0xa0, 0x28,
		},
	},
	/* SHA224 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0x4d, 0x31, 0x79, 0xb3, 0xfe, 0xa6, 0xc9, 0xff,
			0xb1, 0x0c, 0x57, 0x4c, 0x1b, 0x15, 0xa5, 0x6f,
			0xd3, 0xcf, 0x12, 0x94, 0x61, 0x6e, 0x96, 0x24,
			0x98, 0x4c, 0xb2, 0xed, 0x27, 0x2e, 0xa7, 0x7d,
			0x6a, 0xc9, 0xe7, 0xce, 0x86, 0x85, 0xc9, 0x29,
			0xb4, 0x80, 0x80, 0x44, 0xdd, 0xd6, 0x4e, 0xae,
			0x68, 0x20, 0xf5, 0x6c, 0x50, 0x3a, 0x6f, 0xe5,
			0x79, 0x4a, 0xde, 0xfb, 0x44, 0xfd, 0xc6, 0x7b,
			0x61, 0x14, 0x7a, 0xb7, 0xe4, 0x20, 0xd4, 0x71,
			0x9b, 0x89, 0x29, 0xfb, 0x9f, 0xd5, 0x7c, 0x9d,
			0x2a, 0x27, 0xf2, 0x37, 0xa8, 0x06, 0x86, 0x27,
			0x11, 0x60, 0xfe, 0x6d, 0x75, 0xcf, 0x1e, 0x0d,
			0xd9, 0x1a, 0x1b, 0xb5, 0x41, 0xff, 0x53, 0xa0,
			0x0e, 0x30, 0x3a, 0xba, 0x8a, 0x9b, 0x98, 0x05,
			0xc2, 0xee, 0xf2, 0x3b, 0x94, 0x5b, 0x45, 0xb5,
			0x08, 0x99, 0xfb, 0x8b, 0xe4, 0x2b, 0x17, 0xfd,
		},
		/* S5 : 512 byte */
		{
			0x26, 0xdb, 0xa3, 0x45, 0xf2, 0xb1, 0xa2, 0x0d,
			0x82, 0xa6, 0xc1, 0x95, 0x76, 0xec, 0x3c, 0x2c,
			0x48, 0x11, 0xb2, 0x32, 0x34, 0xe2, 0x8d, 0x3c,
			0xc8, 0xa7, 0xac, 0x0e, 0x93, 0x0c, 0x95, 0x3c,
			0x23, 0x86, 0x0a, 0x57, 0xd5, 0xf4, 0x45, 0x7d,
			0xe6, 0xc7, 0xa3, 0x29, 0x35, 0x00, 0xa1, 0x1c,
			0xff, 0x78, 0xbc, 0x86, 0xd4, 0xea, 0x6c, 0x50,
			0xb7, 0xba, 0x00, 0x08, 0x02, 0x59, 0x52, 0x7a,
			0x9d, 0x0f, 0xef, 0xd6, 0x04, 0x8c, 0x9d, 0x85,
			0xae, 0xa4, 0xb8, 0x61, 0xd0, 0x5c, 0x4c, 0x28,
			0xde, 0xab, 0x10, 0xba, 0x26, 0x55, 0xb1, 0x77,
			0x50, 0x92, 0xc4, 0x4c, 0x8f, 0x83, 0xf9, 0xf5,
			0x8e, 0xaa, 0xdf, 0x29, 0xe8, 0x65, 0x8d, 0xf8,
			0x3a, 0xb1, 0x40, 0x57, 0xe7, 0x7d, 0x31, 0xa3,
			0x2c, 0xc1, 0x40, 0xd4, 0x77, 0x49, 0x40, 0xad,
			0x11, 0x04, 0x0b, 0x4f, 0x00, 0xf4, 0x26, 0x2a,
			0xf1, 0xde, 0x68, 0xbc, 0xde, 0x01, 0x27, 0x73,
			0x7f, 0xfe, 0xc9, 0x29, 0x15, 0xa0, 0xe5, 0x0d,
			0x04, 0xaf, 0x7e, 0xcc, 0x08, 0x38, 0x62, 0x23,
			0x5f, 0xce, 0xcd, 0xe9, 0x03, 0xd4, 0x72, 0xc5,
			0x8f, 0xcd, 0xbb, 0x8a, 0xc4, 0xd5, 0x0e, 0xf3,
			0x4a, 0xaa, 0x9e, 0xcd, 0xa0, 0x77, 0x23, 0x90,
			0xa2, 0xae, 0x47, 0xce, 0xf2, 0x45, 0x63, 0x53,
			0x1d, 0x56, 0xdf, 0x48, 0x51, 0x60, 0xfd, 0x9b,
			0xc1, 0x58, 0xb1, 0x59, 0x6f, 0x47, 0xae, 0xe8,
			0x47, 0x0d, 0x9b, 0x19, 0xe6, 0x99, 0x1d, 0xe3,
			0xc1, 0xb0, 0x22, 0xf0, 0xa9, 0xdd, 0xe7, 0x85,
			0x3d, 0xc4, 0xec, 0x9a, 0x81, 0x6e, 0xc5, 0x41,
			0x0b, 0x5b, 0x1a, 0xcb, 0xd2, 0x20, 0xef, 0x9e,
			0x44, 0x24, 0x09, 0x2b, 0xf1, 0x4c, 0xaa, 0xc0,
			0xe2, 0x71, 0x92, 0x25, 0x55, 0xb9, 0x3c, 0xb0,
			0xb0, 0xe2, 0xce, 0x07, 0xf3, 0x79, 0x95, 0xf6,
			0xda, 0x0a, 0x44, 0xcf, 0x56, 0xda, 0x39, 0x89,
			0xb0, 0x2d, 0x94, 0xfd, 0xee, 0x7d, 0x0b, 0xe1,
			0x40, 0xb8, 0xed, 0x25, 0xa9, 0xcb, 0x39, 0x00,
			0xc5, 0x01, 0xaa, 0x55, 0x67, 0x58, 0x0e, 0x95,
			0xf2, 0xe9, 0xec, 0x61, 0x1c, 0x70, 0x64, 0x48,
			0xbc, 0x86, 0x8b, 0xc7, 0xcc, 0xe3, 0x14, 0x49,
			0x67, 0x2b, 0x8e, 0xc1, 0x3e, 0x2d, 0x6b, 0x2e,
			0xbb, 0x42, 0xf1, 0x42, 0x18, 0x77, 0xee, 0x37,
			0x12, 0x48, 0xba, 0x22, 0x0f, 0xd7, 0x6d, 0x22,
			0xa6, 0x91, 0xdc, 0x1e, 0xbd, 0x84, 0xa2, 0x1f,
			0xf9, 0xd5, 0x38, 0xfa, 0xc9, 0x46, 0x1d, 0xb5,
			0xff, 0x40, 0x4c, 0xbe, 0x2b, 0xc6, 0xb2, 0x7d,
			0xbd, 0xa9, 0xd0, 0xa8, 0xb7, 0xcc, 0xce, 0xb6,
			0xaf, 0x14, 0x21, 0xa4, 0xd4, 0x60, 0x1f, 0x19,
			0x64, 0xcd, 0x17, 0x8e, 0xcb, 0x21, 0x2e, 0x55,
			0x1b, 0xf3, 0x9c, 0x42, 0xda, 0xc0, 0x15, 0xe5,
			0x96, 0xcb, 0x3e, 0x0a, 0x51, 0xbf, 0xe9, 0x65,
			0x27, 0x83, 0xcd, 0x99, 0x30, 0xaa, 0xd1, 0x30,
			0x9d, 0x22, 0x0e, 0x31, 0xaf, 0x6c, 0xbb, 0x5a,
			0x15, 0xee, 0x99, 0x83, 0xef, 0x09, 0xce, 0xd6,
			0x22, 0x51, 0x45, 0x08, 0xcc, 0x5b, 0xbf, 0x63,
			0xbf, 0xd7, 0x32, 0xe9, 0x9b, 0x1a, 0xb2, 0x35,
			0x75, 0xeb, 0xf2, 0x7f, 0x9a, 0x6d, 0x22, 0x77,
			0x58, 0xc9, 0xec, 0x48, 0x81, 0xbe, 0xa4, 0x41,
			0x59, 0xa5, 0x3c, 0xd4, 0xe1, 0x42, 0x0c, 0xbf,
			0x74, 0x3f, 0x64, 0x83, 0x03, 0x33, 0x03, 0x32,
			0x86, 0x7c, 0x10, 0xc0, 0x8c, 0x15, 0x9b, 0x68,
			0x27, 0x2f, 0xd4, 0xb1, 0xd6, 0x99, 0x67, 0xe2,
			0x79, 0xc9, 0x0d, 0x0f, 0xe1, 0x9c, 0x07, 0x02,
			0xd2, 0x66, 0x17, 0xe8, 0xab, 0x14, 0x47, 0xff,
			0x60, 0x57, 0x05, 0x13, 0xae, 0xc5, 0xe0, 0x09,
			0x60, 0x82, 0xdf, 0x66, 0x6c, 0xde, 0xcd, 0x70,
		},
	},
	/* SHA224 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0x25, 0x38, 0x8d, 0x8e, 0x79, 0x21, 0x60, 0xb2,
			0x7e, 0x07, 0x13, 0xa7, 0xac, 0x8d, 0x90, 0x48,
			0xcb, 0x6e, 0x74, 0x46, 0x13, 0x11, 0x5a, 0x62,
			0x91, 0x5e, 0x5c, 0xe0, 0xcc, 0xd1, 0xd7, 0x38,
			0xa9, 0xcc, 0x6f, 0x0e, 0x8c, 0x32, 0x53, 0x64,
			0xc0, 0x31, 0xc2, 0x0f, 0x19, 0x5f, 0xcf, 0x67,
			0xd7, 0x09, 0x61, 0x43, 0xce, 0xc7, 0x7d, 0x25,
			0xc4, 0x54, 0xc8, 0xb1, 0x1b, 0x9d, 0x47, 0x64,
			0xb0, 0x18, 0x6a, 0x82, 0x02, 0x57, 0x49, 0x50,
			0x97, 0xdd, 0x2d, 0x84, 0x9b, 0x30, 0x52, 0xfa,
			0xcb, 0x30, 0xeb, 0xa8, 0x91, 0xa6, 0x67, 0xd4,
			0x70, 0x20, 0xf3, 0x84, 0xa3, 0xfd, 0x40, 0x37,
			0x01, 0xc4, 0x86, 0xd2, 0x24, 0xba, 0xc9, 0xdd,
			0xca, 0xb3, 0x5b, 0xef, 0x7b, 0x28, 0xa2, 0x67,
			0x6d, 0x8a, 0x1b, 0x5d, 0xa4, 0xd6, 0x9f, 0xd9,
			0x5d, 0x6a, 0xe5, 0x42, 0xad, 0x17, 0x49, 0x19,
		},
		/* S6 : 512 byte */
		{
			0xa1, 0x97, 0x07, 0xef, 0xc8, 0x04, 0x2d, 0xd0,
			0xcb, 0x5d, 0x3d, 0x2c, 0xb8, 0x62, 0xa9, 0x3b,
			0xb9, 0x79, 0x9e, 0xc0, 0x3c, 0x07, 0xd1, 0x0a,
			0x1d, 0x69, 0x4b, 0xea, 0x36, 0x4b, 0x53, 0xbb,
			0x17, 0x66, 0xe4, 0x8a, 0x1f, 0x25, 0xa2, 0x2e,
			0x9f, 0x7c, 0xc9, 0xb6, 0xd1, 0x03, 0xdb, 0x0d,
			0xd1, 0x56, 0x0b, 0x71, 0x87, 0x10, 0x09, 0x4b,
			0x69, 0xff, 0x61, 0xe9, 0x43, 0x3e, 0x4a, 0xe1,
			0x03, 0x96, 0x00, 0x89, 0x5f, 0x47, 0x17, 0x50,
			0x92, 0x5b, 0x13, 0xe5, 0x9f, 0x9a, 0xa3, 0x94,
			0xf8, 0xfa, 0x6a, 0x4d, 0x75, 0x05, 0x88, 0x4e,
			0xf6, 0x02, 0x8f, 0x68, 0x7c, 0xff, 0x7a, 0xd9,
			0xd6, 0xa5, 0xbc, 0x9a, 0x78, 0x8a, 0x7a, 0x43,
			0x06, 0x7b, 0x82, 0x94, 0x96, 0xc5, 0x41, 0x80,
			0xfe, 0x0d, 0xe8, 0xd7, 0xce, 0x03, 0xed, 0xf4,
			0xd3, 0x00, 0xd0, 0x7d, 0xfa, 0x1a, 0xfe, 0x26,
			0x6d, 0x20, 0x92, 0x48, 0x39, 0xea, 0x63, 0x66,
			0x51, 0x12, 0x4d, 0x5c, 0x0a, 0xc0, 0xaa, 0x9b,
			0xd0, 0x26, 0xd0, 0x17, 0xc7, 0x13, 0x8e, 0xbf,
			0xf1, 0x7b, 0x6c, 0x9d, 0x0b, 0x15, 0x83, 0xaa,
			0x69, 0x43, 0xb4, 0xe9, 0x6c, 0xf1, 0x91, 0x2e,
			0xa9, 0x36, 0x93, 0xec, 0x9f, 0xbc, 0xf5, 0xce,
			0xe6, 0xed, 0xf0, 0x81, 0xbf, 0x7a, 0x0d, 0xaa,
			0x0a, 0xcd, 0x18, 0x5b, 0x64, 0x5c, 0x76, 0xac,
			0x2c, 0x38, 0xd2, 0x61, 0x32, 0x63, 0x1d, 0xdd,
			0x2b, 0x3e, 0xf5, 0x32, 0x6d, 0xd4, 0xe6, 0x29,
			0x88, 0x95, 0xa0, 0xe1, 0xd0, 0x96, 0xbf, 0x27,
			0x7b, 0x25, 0xca, 0x73, 0x97, 0x12, 0xc1, 0x66,
			0xcc, 0x41, 0x19, 0xe2, 0x9b, 0xea, 0x72, 0xb2,
			0xe8, 0x7b, 0x5a, 0xb0, 0xe1, 0x3b, 0x17, 0x00,
			0x07, 0xa3, 0xa4, 0x40, 0x22, 0x51, 0x64, 0xff,
			0x2f, 0xc7, 0xd1, 0xf0, 0xda, 0x65, 0x5c, 0xa1,
			0x9c, 0x8b, 0x08, 0x8d, 0x2f, 0x43, 0x80, 0x67,
			0x99, 0x2d, 0x59, 0x1d, 0x13, 0xa3, 0x5a, 0xcf,
			0x9c, 0x54, 0xfb, 0xef, 0x1e, 0xb4, 0x55, 0x28,
			0x51, 0x7d, 0x87, 0xb7, 0x6f, 0x3b, 0x18, 0x01,
			0x78, 0xff, 0x94, 0x36, 0x81, 0x98, 0xf0, 0x2d,
			0x3a, 0x0f, 0x21, 0x0d, 0xad, 0x5d, 0x4c, 0x91,
			0x12, 0x4c, 0xa5, 0xf5, 0xe7, 0x75, 0x3c, 0xf6,
			0x52, 0x45, 0x01, 0x6f, 0x51, 0xe2, 0x2d, 0xec,
			0xa3, 0x08, 0x7b, 0x98, 0xb0, 0x0e, 0xd3, 0x80,
			0x14, 0x73, 0xd3, 0x49, 0xaf, 0x29, 0x0c, 0xd9,
			0x34, 0x98, 0xb2, 0xb8, 0x9f, 0xf0, 0x6f, 0x26,
			0xa9, 0xb9, 0x2f, 0x94, 0xe2, 0xcd, 0xeb, 0x76,
			0x43, 0xea, 0xfd, 0x20, 0xff, 0xce, 0x2c, 0x43,
			0x38, 0x51, 0x51, 0x88, 0xb7, 0x3a, 0xc2, 0xda,
			0x53, 0xff, 0xa4, 0xc1, 0x1e, 0xb3, 0x72, 0x05,
			0x23, 0x22, 0x09, 0xf1, 0x3b, 0xd4, 0xf0, 0xc9,
			0xf5, 0x16, 0x94, 0xc6, 0x72, 0x2e, 0xb2, 0xfd,
			0xed, 0xd7, 0x0b, 0x4c, 0xdd, 0x4a, 0x46, 0x9e,
			0x9a, 0x16, 0x71, 0x7f, 0xa5, 0x6f, 0x36, 0x73,
			0xda, 0x3d, 0x40, 0x8a, 0x89, 0xdf, 0xf5, 0x32,
			0xdd, 0x0e, 0x4e, 0xef, 0xac, 0x97, 0x07, 0xf1,
			0x8a, 0x05, 0x86, 0x15, 0xe4, 0x55, 0x94, 0xe7,
			0x27, 0xc2, 0x58, 0x29, 0xbc, 0x42, 0x53, 0xf6,
			0xb6, 0x27, 0xae, 0xa3, 0x9d, 0x4f, 0xde, 0x65,
			0xa2, 0x4d, 0x40, 0xd5, 0x72, 0xbf, 0x8d, 0x01,
			0x34, 0x79, 0xd5, 0x6d, 0x67, 0xd1, 0xcb, 0x66,
			0xfe, 0x8b, 0x0b, 0x73, 0xe0, 0xe5, 0xb3, 0xf8,
			0xfc, 0x7e, 0x53, 0x83, 0xec, 0xc5, 0xd5, 0x51,
			0xee, 0xf6, 0xe7, 0xce, 0xce, 0x73, 0x77, 0x5c,
			0xb1, 0x11, 0xab, 0x67, 0x86, 0x1f, 0xc1, 0x12,
			0x21, 0xb0, 0x82, 0x18, 0xf7, 0x72, 0xd2, 0x9e,
			0xd0, 0x9a, 0xa4, 0xb0, 0xf3, 0x94, 0x7e, 0xd6,
		},
	},
	/* SHA224 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0x24, 0xc6, 0x8b, 0xad, 0xa4, 0x98, 0x64, 0x7a,
			0x77, 0xb0, 0xa7, 0x51, 0x53, 0x3f, 0xd0, 0x8e,
			0xaf, 0x31, 0x2f, 0x14, 0x77, 0x0d, 0x3c, 0x4a,
			0x03, 0x1d, 0x2a, 0x63, 0x4a, 0xd0, 0xbf, 0xe3,
			0x53, 0x5c, 0x62, 0x5e, 0xed, 0xa7, 0x76, 0x83,
			0xa0, 0x68, 0xe6, 0x5a, 0xe3, 0x89, 0x01, 0xca,
			0x8f, 0xd1, 0x89, 0x5e, 0x94, 0xbf, 0x06, 0x49,
			0xeb, 0xb9, 0x60, 0xa7, 0xcc, 0x66, 0xad, 0x05,
			0x20, 0x58, 0x48, 0x24, 0x39, 0xee, 0x23, 0xfa,
			0xc1, 0x78, 0x5e, 0xfa, 0xf0, 0x9d, 0x18, 0x95,
			0x02, 0xfa, 0x85, 0x01, 0xe9, 0x0c, 0x40, 0x39,
			0x3f, 0x50, 0xe3, 0x45, 0x7e, 0xa8, 0xd6, 0xbb,
			0x73, 0x01, 0x64, 0x86, 0xf0, 0x33, 0x14, 0xe6,
			0xc2, 0x28, 0x36, 0xb8, 0xe2, 0x41, 0xbc, 0xf8,
			0xf0, 0xf5, 0x4b, 0x84, 0xdc, 0xbc, 0x92, 0x22,
			0xe7, 0x2b, 0xdb, 0xc5, 0xca, 0x60, 0xdf, 0x0c,
		},
		/* S7 : 512 byte */
		{
			0xa6, 0x0b, 0x21, 0x23, 0x5a, 0xb8, 0x6e, 0x6a,
			0xfd, 0x0a, 0x72, 0xab, 0x6c, 0xc5, 0x31, 0x4c,
			0x00, 0x6c, 0xf8, 0xf0, 0x8d, 0xcf, 0xd3, 0x7e,
			0xf1, 0xf9, 0x75, 0x58, 0x48, 0x09, 0x4b, 0x63,
			0xa8, 0x24, 0xb3, 0x67, 0x9c, 0xae, 0xff, 0x18,
			0x8a, 0x34, 0x9b, 0x18, 0xf2, 0x51, 0xc8, 0x7f,
			0x36, 0x9d, 0x7d, 0xa9, 0xf1, 0x1b, 0xb9, 0xe1,
			0x9b, 0x1e, 0xda, 0xd3, 0xe3, 0xfc, 0xeb, 0x7d,
			0x93, 0x1f, 0xda, 0xd4, 0x6f, 0x54, 0xc4, 0x2f,
			0x8c, 0x06, 0xc1, 0x7d, 0xef, 0x94, 0xf9, 0x61,
			0x9a, 0x35, 0x9b, 0x01, 0x22, 0x8c, 0x8b, 0x90,
			0x18, 0xc3, 0x01, 0xea, 0x4d, 0x49, 0x7c, 0x42,
			0xe0, 0xa2, 0x8c, 0x41, 0x35, 0x95, 0xfe, 0x44,
			0x6e, 0x6c, 0xa2, 0x41, 0xd2, 0x26, 0x18, 0x73,
			0x08, 0xdb, 0xac, 0x5c, 0x25, 0x50, 0x6d, 0xc4,
			0xc0, 0x31, 0xaf, 0x3e, 0xbc, 0x00, 0xfa, 0x95,
			0x8f, 0x69, 0x79, 0x08, 0xe4, 0x1a, 0x62, 0x60,
			0xa1, 0x8c, 0x3d, 0x19, 0xc0, 0x78, 0xa7, 0x8d,
			0x80, 0x54, 0x1e, 0x54, 0x5b, 0xb3, 0xec, 0x67,
			0xbd, 0x0e, 0xb4, 0x5c, 0x66, 0xdb, 0xaf, 0xb9,
			0xf4, 0xae, 0xfe, 0x6d, 0x8c, 0xd9, 0x91, 0x61,
			0x04, 0x44, 0xf5, 0x62, 0x72, 0x32, 0x59, 0x24,
			0x09, 0x5f, 0x81, 0x54, 0xbc, 0x25, 0x1c, 0x4e,
			0x6e, 0x06, 0x73, 0xe5, 0xf2, 0x70, 0x31, 0x54,
			0x3c, 0x01, 0xdf, 0x0b, 0x39, 0xee, 0xdf, 0x49,
			0xf1, 0x86, 0x62, 0x22, 0x78, 0x58, 0x50, 0xa3,
			0x50, 0x60, 0x71, 0x24, 0xda, 0xfd, 0xe1, 0xfb,
			0x94, 0xf1, 0x96, 0x13, 0xfb, 0x0f, 0x66, 0xdf,
			0x88, 0x69, 0x12, 0x99, 0x49, 0x61, 0xdd, 0x02,
			0xb4, 0x1a, 0x27, 0xf0, 0x8b, 0x67, 0xbd, 0x37,
			0xdb, 0xd5, 0xbd, 0xd7, 0xff, 0x68, 0xe5, 0x1f,
			0xb3, 0x87, 0x6e, 0xa6, 0x05, 0xc4, 0x47, 0x24,
			0x80, 0x72, 0xe4, 0xb4, 0x10, 0xd9, 0x77, 0xe4,
			0x21, 0xde, 0xe2, 0xc5, 0x93, 0x7d, 0x9f, 0x09,
			0xcd, 0x1c, 0x6c, 0x5c, 0x68, 0x8d, 0xfc, 0x48,
			0xce, 0x44, 0xf3, 0xc0, 0xbb, 0x23, 0x86, 0x50,
			0xab, 0x5c, 0x0a, 0x7c, 0x4c, 0xd8, 0x52, 0xd9,
			0xd5, 0x10, 0x8a, 0xd8, 0x49, 0x7a, 0x1c, 0xdd,
			0x08, 0x3d, 0x80, 0x79, 0x7b, 0x5e, 0xf8, 0x7a,
			0xf6, 0x1d, 0x8d, 0x6b, 0xa0, 0xdb, 0x9c, 0x1b,
			0x61, 0x0b, 0x0d, 0x86, 0x3e, 0x9d, 0xff, 0x5a,
			0x31, 0x0b, 0x6d, 0x1f, 0xb1, 0xf7, 0xef, 0x40,
			0xb1, 0x8c, 0x07, 0x0f, 0xd2, 0x3c, 0x6c, 0x7d,
			0x11, 0xb1, 0xa6, 0xf8, 0x72, 0x92, 0x71, 0xfc,
			0x9e, 0x9d, 0x08, 0xac, 0x5f, 0x2f, 0x7d, 0x88,
			0x63, 0xe9, 0xc6, 0xf5, 0x14, 0x44, 0xb9, 0x9b,
			0xe4, 0xd2, 0x05, 0x3c, 0xa3, 0xaf, 0xf1, 0x5a,
			0x3a, 0x6d, 0x2b, 0x80, 0xa8, 0xc6, 0x82, 0xba,
			0x1c, 0xcb, 0x9f, 0xec, 0xf9, 0x7d, 0xdc, 0x7e,
			0xf1, 0x99, 0x3c, 0x38, 0xd5, 0x24, 0x4f, 0x15,
			0x91, 0x70, 0x0a, 0xc1, 0x5b, 0x85, 0x86, 0xec,
			0xf2, 0x30, 0x33, 0xa8, 0x6b, 0x90, 0xd4, 0xc6,
			0xa0, 0x86, 0x14, 0x26, 0x43, 0xca, 0xd6, 0x6b,
			0x40, 0x2c, 0x54, 0x43, 0x8f, 0xc7, 0x76, 0x3c,
			0x3a, 0xb1, 0x03, 0x24, 0xf3, 0xb5, 0x13, 0xbd,
			0x53, 0xac, 0xc2, 0x17, 0xfe, 0xbc, 0xdb, 0x57,
			0xa4, 0x96, 0xcf, 0xfe, 0x48, 0x4a, 0xe1, 0x2f,
			0x1c, 0x59, 0xf7, 0xce, 0x0e, 0x40, 0xf8, 0x0d,
			0x5f, 0x98, 0xd1, 0x26, 0x7a, 0x81, 0x64, 0xf5,
			0x70, 0xba, 0x0e, 0x84, 0x1e, 0x32, 0x19, 0x1e,
			0x9d, 0xdc, 0x01, 0x5d, 0x13, 0x42, 0xba, 0xc6,
			0x11, 0x01, 0xf2, 0x6a, 0xaa, 0x2c, 0x47, 0xdb,
			0xf9, 0xd2, 0xd6, 0x26, 0xa1, 0xaf, 0x53, 0xbc,
			0x58, 0x66, 0x5d, 0x6e, 0x0c, 0x5b, 0x62, 0x85,
		},
	},
	/* SHA224 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0x83, 0xb1, 0x58, 0xc3, 0x90, 0x6e, 0xb0, 0x4a,
			0x8c, 0x0a, 0x71, 0x87, 0x90, 0x04, 0xae, 0x94,
			0x65, 0xd6, 0x86, 0x5d, 0x52, 0x7a, 0x9e, 0x49,
			0x31, 0x2d, 0xcf, 0x1d, 0xc8, 0x73, 0x1b, 0x1d,
			0x57, 0x5b, 0xe6, 0x74, 0x6e, 0x04, 0x62, 0xd5,
			0x01, 0x60, 0x4c, 0x63, 0xbe, 0x38, 0xd6, 0xce,
			0x9e, 0x70, 0xe4, 0xbf, 0xbb, 0x8d, 0x7a, 0x77,
			0x61, 0x74, 0x75, 0xb3, 0x67, 0x74, 0x7a, 0xd2,
			0xc0, 0x87, 0x2d, 0x39, 0x4d, 0xd5, 0xa3, 0xf7,
			0xf5, 0x7a, 0x19, 0xa5, 0xf8, 0x89, 0xe6, 0x90,
			0x80, 0x51, 0xae, 0x1b, 0x7a, 0xdf, 0xd9, 0x5e,
			0xa3, 0xc4, 0x43, 0x13, 0xe6, 0x18, 0x35, 0xb3,
			0xe5, 0xbe, 0x95, 0xdc, 0xd6, 0xea, 0x59, 0xf6,
			0x90, 0xe1, 0x41, 0x16, 0xe5, 0x00, 0xc6, 0x22,
			0x33, 0x00, 0x4e, 0x37, 0x36, 0x78, 0xa1, 0x47,
			0x20, 0xa3, 0xa0, 0x07, 0xec, 0x64, 0x35, 0x08,
		},
		/* S8 : 512 byte */
		{
			0x3d, 0x2a, 0x6a, 0x4d, 0xfa, 0x22, 0x47, 0x79,
			0xf3, 0x0b, 0x26, 0x7a, 0x78, 0xc2, 0x2c, 0xa4,
			0xbf, 0x08, 0xce, 0x92, 0xe0, 0x7a, 0xd3, 0xc1,
			0xa4, 0x6e, 0xa7, 0x8b, 0xc5, 0x85, 0xaf, 0x1b,
			0x2c, 0x9c, 0x8a, 0xd2, 0x27, 0xcd, 0x0c, 0xb4,
			0x8d, 0x48, 0x3c, 0x75, 0x50, 0x0b, 0xf1, 0x4e,
			0x96, 0x5c, 0xc4, 0xb6, 0x2b, 0x1b, 0xb0, 0x0d,
			0x4b, 0x56, 0x2a, 0x8d, 0xe5, 0x40, 0x4c, 0xa0,
			0x7b, 0xe7, 0xe7, 0x9d, 0x29, 0xf1, 0x48, 0x9a,
			0x4b, 0xff, 0x67, 0x53, 0x01, 0x38, 0xcd, 0xdb,
			0x93, 0x33, 0x3c, 0xca, 0x6e, 0x9c, 0x96, 0xce,
			0x1c, 0x6c, 0x0d, 0xa8, 0x01, 0xda, 0x36, 0x08,
			0x11, 0x01, 0x40, 0x95, 0x5a, 0x96, 0xb2, 0x2b,
			0x74, 0x07, 0x06, 0x75, 0x15, 0xc8, 0xe9, 0xa1,
			0x70, 0x10, 0xe5, 0xdf, 0xf2, 0x99, 0x63, 0xb3,
			0xed, 0xad, 0x6f, 0xca, 0x1b, 0x1b, 0x12, 0xa8,
			0x8c, 0xfd, 0xb8, 0xb4, 0x2f, 0xff, 0x9c, 0x57,
			0xdb, 0x14, 0xa9, 0xe3, 0x34, 0xf5, 0xca, 0x22,
			0x04, 0xb1, 0x6a, 0xa7, 0x9c, 0xae, 0xb8, 0x17,
			0xa7, 0xed, 0x2e, 0xfe, 0x30, 0x61, 0x20, 0xf7,
			0x99, 0xe0, 0x14, 0x48, 0xa2, 0x0f, 0xd8, 0xe6,
			0xa8, 0xb7, 0x59, 0x70, 0x8c, 0x85, 0x8d, 0x29,
			0x3f, 0x0d, 0xd2, 0x3b, 0xd9, 0x4f, 0x51, 0xf4,
			0x8c, 0x1f, 0x7e, 0xdb, 0x0f, 0x1c, 0xfb, 0xa3,
			0x70, 0x4a, 0xcc, 0xa4, 0x08, 0x6b, 0x7c, 0x53,
			0xc3, 0x05, 0x1f, 0x42, 0x48, 0x93, 0xf2, 0xea,
			0x50, 0xcc, 0x51, 0x68, 0x46, 0x0e, 0xa8, 0xf4,
			0xe0, 0x11, 0x21, 0xa8, 0x68, 0xea, 0xb1, 0x53,
			0x94, 0xe0, 0x17, 0xa2, 0x23, 0xad, 0xf6, 0x8e,
			0x6d, 0xbd, 0xe8, 0x80, 0xb1, 0x47, 0x70, 0xd4,
			0xbb, 0x6a, 0x1b, 0xb1, 0xbb, 0x10, 0xb9, 0x01,
			0x46, 0x08, 0x56, 0xf9, 0x48, 0x08, 0xe4, 0xb4,
			0x27, 0x69, 0x23, 0x9c, 0xe8, 0x1e, 0x53, 0x5e,
			0x4d, 0x59, 0xdb, 0x02, 0xa6, 0x4c, 0x3e, 0x22,
			0x00, 0x7b, 0x10, 0x21, 0xf2, 0x6a, 0xce, 0x4a,
			0xbe, 0xca, 0x66, 0x6f, 0x11, 0x1c, 0x8b, 0xaa,
			0xc1, 0x84, 0x7a, 0xf8, 0x81, 0x45, 0x2a, 0xe0,
			0xc2, 0x04, 0x53, 0x75, 0x91, 0x95, 0xfe, 0x66,
			0x3f, 0x05, 0x4f, 0xb0, 0xb5, 0xcf, 0x8a, 0x38,
			0x78, 0xe3, 0xb5, 0xf3, 0xb5, 0x63, 0xe0, 0x93,
			0x86, 0xe5, 0xb1, 0xf7, 0xdb, 0xec, 0x53, 0x34,
			0x0b, 0x71, 0x33, 0x17, 0x75, 0x9b, 0x4e, 0xd2,
			0x03, 0xbe, 0x9b, 0xb3, 0x6f, 0xef, 0xaa, 0xd1,
			0x3a, 0xdd, 0x45, 0x57, 0x0c, 0x52, 0xaf, 0xce,
			0xae, 0xbb, 0x55, 0x56, 0x85, 0x1a, 0x59, 0xdd,
			0x04, 0x42, 0x0b, 0x39, 0xcd, 0x87, 0xf1, 0x87,
			0x95, 0x63, 0x98, 0x01, 0x04, 0xe3, 0xb5, 0xa2,
			0x52, 0x93, 0xbf, 0x96, 0x35, 0x2c, 0x3d, 0x18,
			0x2d, 0x66, 0x7a, 0xfb, 0x94, 0xd3, 0x00, 0x56,
			0xa6, 0x09, 0x29, 0x1e, 0xfc, 0x31, 0x2e, 0xbc,
			0xa1, 0xb5, 0x32, 0x08, 0xbd, 0x64, 0x8e, 0xbd,
			0x09, 0xda, 0x08, 0xec, 0xe6, 0x1e, 0xf5, 0x0f,
			0x58, 0x1d, 0xfd, 0x49, 0x32, 0xac, 0xc6, 0xbe,
			0xef, 0x1f, 0x6d, 0xa4, 0x68, 0x7a, 0x65, 0x92,
			0x0a, 0x47, 0x23, 0x71, 0x07, 0x4a, 0xbb, 0xc1,
			0x09, 0xd4, 0x24, 0xce, 0x4e, 0xf3, 0x9c, 0x0d,
			0x53, 0x04, 0x56, 0x1c, 0x76, 0x6e, 0x90, 0x42,
			0x07, 0x97, 0xb0, 0x0e, 0x5a, 0x6c, 0x57, 0x25,
			0x1e, 0x6d, 0x1f, 0x6e, 0x93, 0xeb, 0x99, 0xc2,
			0x07, 0x9e, 0x55, 0xa6, 0xe4, 0xe3, 0x2d, 0x8b,
			0xde, 0x20, 0xed, 0x7b, 0x0b, 0x75, 0xb7, 0x0a,
			0xee, 0x52, 0x0b, 0x72, 0xbf, 0x66, 0x7d, 0x3c,
			0xa3, 0x3e, 0xa7, 0x0c, 0x8f, 0x82, 0xb2, 0x35,
			0x8d, 0xdd, 0x61, 0x0f, 0x79, 0x8d, 0xf2, 0xa3,
		},
	},
	/* SHA224 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0xb1, 0xfb, 0xda, 0x78, 0x9a, 0x33, 0x9b, 0x41,
			0x63, 0xac, 0xf2, 0x2b, 0x0d, 0xb3, 0x0e, 0x83,
			0x44, 0x96, 0x16, 0xa0, 0x91, 0x05, 0x03, 0x68,
			0xd0, 0xbe, 0xb0, 0x16, 0xe5, 0x6a, 0x34, 0x3a,
			0xab, 0xc1, 0x6f, 0x00, 0x2d, 0xf5, 0xe3, 0xb6,
			0x19, 0x4e, 0xc5, 0x89, 0x71, 0x11, 0x54, 0x8f,
			0x84, 0xa7, 0x6a, 0x0b, 0x1d, 0xfb, 0x51, 0xec,
			0xfa, 0x24, 0xd7, 0x96, 0x7f, 0x44, 0xa5, 0xe5,
			0xac, 0xb0, 0xcc, 0x6f, 0x4d, 0x51, 0xa1, 0x0a,
			0x70, 0x4a, 0xcb, 0x8f, 0x1c, 0xda, 0x9a, 0xdd,
			0x40, 0x84, 0x9b, 0x20, 0xea, 0x6e, 0x69, 0x53,
			0xb9, 0x08, 0xc5, 0x03, 0x95, 0xed, 0xe9, 0x56,
			0x9d, 0x0e, 0x1a, 0x4d, 0x61, 0x0d, 0x7d, 0x46,
			0x52, 0xe7, 0x2b, 0xc5, 0x77, 0xd6, 0x87, 0x73,
			0x61, 0x76, 0xd0, 0x67, 0x60, 0x26, 0xf2, 0xa5,
			0xf7, 0xb1, 0xa2, 0xe4, 0x8e, 0x2e, 0xa9, 0x95,
		},
		/* S9 : 512 byte */
		{
			0xa2, 0x77, 0x64, 0x1f, 0xc4, 0x8a, 0xb3, 0x28,
			0xdf, 0x91, 0x62, 0xd3, 0x92, 0x89, 0x28, 0x0b,
			0xbc, 0x0a, 0xdb, 0x4b, 0x77, 0x0e, 0x85, 0x6e,
			0x31, 0x23, 0xe9, 0x70, 0x06, 0x3e, 0x2a, 0x2e,
			0xa6, 0xc8, 0xe1, 0xe4, 0x57, 0xc6, 0x32, 0xb2,
			0x1e, 0x63, 0xde, 0xa5, 0x43, 0x9a, 0x84, 0xb9,
			0x11, 0x59, 0x0e, 0xc3, 0x4c, 0xd9, 0x98, 0xb8,
			0x89, 0x64, 0x0a, 0x26, 0x42, 0x8e, 0x6a, 0x02,
			0xba, 0x2e, 0x72, 0xd2, 0xbc, 0x27, 0xc2, 0xe7,
			0x8e, 0xe0, 0x39, 0xf6, 0xdf, 0x22, 0xbe, 0xf6,
			0x38, 0xea, 0x1a, 0xf5, 0xa9, 0xcc, 0xb8, 0x9b,
			0xe8, 0xad, 0xa7, 0xc1, 0x2c, 0xb5, 0x62, 0x0f,
			0xef, 0x07, 0x9d, 0x24, 0xee, 0x84, 0xb0, 0x36,
			0xbe, 0x7a, 0x5f, 0xe3, 0xd1, 0x78, 0xbb, 0xdd,
			0x50, 0x2e, 0x34, 0x84, 0xab, 0x40, 0x4e, 0x31,
			0xc8, 0x9e, 0x84, 0x40, 0xa6, 0xd9, 0x2f, 0x9f,
			0xa9, 0x5d, 0xd8, 0xb4, 0x56, 0xc1, 0x9a, 0xf9,
			0x6b, 0x9d, 0xd9, 0x99, 0x26, 0x62, 0x17, 0xd0,
			0x4b, 0xbb, 0xe6, 0xa1, 0x70, 0xf8, 0xfa, 0xf1,
			0xf7, 0xa3, 0x84, 0x69, 0x4b, 0xc6, 0xf6, 0x64,
			0xf5, 0xbd, 0x21, 0xd4, 0xf4, 0x0f, 0x0c, 0x7c,
			0x32, 0x9c, 0x5e, 0xd4, 0x5e, 0x19, 0xd5, 0xc5,
			0x44, 0x55, 0x56, 0xda, 0xa2, 0xc9, 0x6e, 0x75,
			0x6f, 0x53, 0x68, 0xee, 0xfb, 0xd3, 0xa3, 0x9f,
			0x10, 0x4e, 0x3d, 0x2e, 0x1f, 0xbc, 0x72, 0x55,
			0x6f, 0x0a, 0x43, 0x8c, 0x56, 0x5a, 0xaa, 0xef,
			0x93, 0x62, 0x4f, 0xfb, 0xdc, 0xf9, 0x6c, 0x5b,
			0xd6, 0x07, 0x94, 0xc6, 0x6d, 0xa1, 0x5d, 0xe6,
			0x9a, 0xa5, 0x02, 0x49, 0x91, 0xe3, 0x56, 0x85,
			0xb2, 0x0e, 0x0a, 0xa1, 0x18, 0xfe, 0x09, 0x41,
			0x89, 0x07, 0xc7, 0x92, 0xcb, 0xdb, 0x67, 0x93,
			0xea, 0x9f, 0x04, 0xbb, 0x2a, 0x81, 0x75, 0x67,
			0x4f, 0x82, 0x9b, 0xaa, 0xd7, 0xa7, 0x93, 0xe0,
			0xce, 0xca, 0x95, 0x54, 0xc5, 0x3b, 0xcf, 0x6a,
			0x36, 0x4d, 0xca, 0x1d, 0xf3, 0xe8, 0x9b, 0xc1,
			0xb0, 0x93, 0x06, 0x7e, 0x6e, 0x75, 0x82, 0xfc,
			0xf8, 0xaa, 0x14, 0xe7, 0x9b, 0xca, 0xd5, 0x18,
			0x2a, 0x5e, 0xe4, 0x9e, 0xfe, 0xde, 0xa4, 0xa9,
			0xe4, 0x0e, 0x0a, 0x99, 0x60, 0x27, 0x70, 0xa2,
			0x80, 0x9d, 0x28, 0x4c, 0xa8, 0x03, 0xe4, 0x76,
			0xaf, 0x0c, 0x34, 0xaf, 0x4d, 0x7a, 0xc4, 0x5b,
			0xa5, 0xd3, 0x8b, 0xe6, 0xaa, 0xa1, 0x61, 0xce,
			0x07, 0x65, 0x47, 0x61, 0x87, 0xb1, 0x56, 0x6c,
			0x1d, 0x23, 0x78, 0x8c, 0xf6, 0xdf, 0x8e, 0xf4,
			0x38, 0x47, 0x81, 0xd3, 0xbf, 0x53, 0xc3, 0x36,
			0xb2, 0x92, 0x4f, 0x8f, 0x9a, 0x4f, 0xea, 0x77,
			0x1c, 0xf6, 0xf0, 0xca, 0xf4, 0xa3, 0xb5, 0x38,
			0x37, 0x1c, 0x01, 0x0c, 0x12, 0xb9, 0xdb, 0x0e,
			0x07, 0x55, 0x37, 0xff, 0x30, 0x91, 0x03, 0x10,
			0xe8, 0xd0, 0x61, 0x7b, 0x16, 0xe4, 0x0f, 0x5a,
			0x01, 0x0c, 0x26, 0x9c, 0x12, 0xc9, 0x2d, 0x16,
			0x0d, 0x01, 0xb3, 0x67, 0x40, 0x78, 0x0a, 0x75,
			0x14, 0xbf, 0x75, 0xaa, 0x31, 0xfc, 0xf7, 0xd7,
			0xc2, 0x12, 0xcd, 0x7e, 0xfb, 0x36, 0x7d, 0x05,
			0x26, 0x48, 0x58, 0x2d, 0xbf, 0xc7, 0x4b, 0xd7,
			0xb5, 0x70, 0xf2, 0x6c, 0xc4, 0x62, 0xfb, 0x8c,
			0x26, 0xd9, 0xcc, 0x46, 0x5e, 0x87, 0x95, 0x8c,
			0xa3, 0x0f, 0xf4, 0xa9, 0x16, 0xe4, 0x36, 0xcf,
			0xe6, 0x1b, 0x69, 0x00, 0xfe, 0xcf, 0x0e, 0x1f,
			0x1b, 0x4b, 0x34, 0x27, 0x82, 0xca, 0xbf, 0x97,
			0x47, 0x9d, 0x55, 0x4e, 0xe5, 0xfb, 0xec, 0x96,
			0x54, 0x9c, 0x8d, 0x0a, 0x79, 0xad, 0xec, 0x4c,
			0x33, 0x84, 0x7f, 0xef, 0x05, 0x42, 0x6f, 0xdf,
			0x45, 0xf5, 0x63, 0xfc, 0x0c, 0x21, 0xfa, 0x5a,
		},
	},
};

/* 10 elements */
static mod4096_t SHA256[] = {
	/* SHA256 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0x8b, 0x90, 0xa7, 0x9f, 0xec, 0x95, 0x5a, 0x7d,
			0xb2, 0x6b, 0xa6, 0x73, 0x7d, 0x3e, 0xd0, 0x45,
			0x56, 0x03, 0xd0, 0x9b, 0x95, 0xc4, 0x54, 0xf8,
			0x01, 0x38, 0x08, 0xeb, 0x3a, 0x97, 0x15, 0x89,
			0x54, 0xac, 0xe7, 0x1a, 0xc5, 0x2c, 0x34, 0x13,
			0x81, 0x46, 0x3a, 0x45, 0x63, 0x7b, 0x48, 0x53,
			0xcf, 0xb0, 0x7f, 0x69, 0x96, 0x82, 0xda, 0x2b,
			0x19, 0xd9, 0x6c, 0xec, 0xb2, 0xc0, 0xe9, 0x21,
			0x4f, 0x74, 0x68, 0x15, 0x60, 0xba, 0x67, 0xde,
			0xf4, 0x76, 0x10, 0x8f, 0x1e, 0x7a, 0xbb, 0xaf,
			0x9e, 0x9c, 0xb3, 0xea, 0xbd, 0x0b, 0x3d, 0x07,
			0x79, 0xe3, 0xd6, 0x1a, 0xe2, 0xfe, 0xbd, 0xfb,
			0xc4, 0x0e, 0xb1, 0xf6, 0x86, 0xe8, 0xfc, 0xc4,
			0x52, 0x23, 0xb0, 0xba, 0x77, 0xe2, 0x31, 0x41,
			0x0b, 0xee, 0xf3, 0x86, 0xd4, 0x08, 0x85, 0x73,
			0x69, 0x7d, 0xcf, 0xdd, 0x94, 0xfa, 0x98, 0xff,
		},
		/* S0 : 512 byte */
		{
			0xd0, 0x58, 0xfc, 0x5d, 0x23, 0x9b, 0x0f, 0x97,
			0x5f, 0xe1, 0x0f, 0x3c, 0x3a, 0x6f, 0x36, 0xb0,
			0xe9, 0xbe, 0x32, 0xc9, 0xbd, 0x2e, 0x5c, 0x81,
			0xd1, 0x0d, 0x38, 0xc6, 0xd4, 0x62, 0xbf, 0x08,
			0xd7, 0x0b, 0x1d, 0x8c, 0xe8, 0x5c, 0xec, 0xbe,
			0xfa, 0x18, 0xf5, 0x26, 0x72, 0xc9, 0xde, 0x7b,
			0x15, 0x2c, 0x93, 0xa8, 0x94, 0x36, 0xc0, 0xeb,
			0x5e, 0xde, 0x1a, 0xa7, 0x8b, 0x17, 0xd0, 0x7f,
			0x3e, 0xec, 0xb6, 0x28, 0xbe, 0x11, 0x7a, 0x23,
			0xb5, 0xbe, 0x3d, 0x5b, 0xdc, 0x48, 0xb6, 0xd5,
			0x0b, 0x86, 0x4f, 0x48, 0x99, 0x12, 0x9c, 0x3e,
			0x91, 0x72, 0x01, 0x96, 0x3e, 0x06, 0x3e, 0xd6,
			0xea, 0x42, 0x8c, 0xf9, 0x2b, 0x4c, 0xe3, 0x58,
			0xdc, 0xdd, 0xe3, 0x2f, 0x18, 0xe7, 0x68, 0xed,
			0xbf, 0x40, 0xc4, 0x3c, 0xac, 0xed, 0x67, 0xcd,
			0xf9, 0x7c, 0xa1, 0x13, 0x3f, 0x8b, 0xa1, 0x58,
			0xb9, 0xb0, 0x50, 0xeb, 0xff, 0x5b, 0xde, 0x22,
			0x75, 0x2b, 0x46, 0x4e, 0x57, 0x40, 0xc8, 0xc6,
			0xd9, 0x90, 0xc4, 0x4b, 0x29, 0x1d, 0xdf, 0x65,
			0x23, 0x17, 0xa3, 0x36, 0xcc, 0xef, 0x6e, 0x0e,
			0x8a, 0x5f, 0x08, 0x11, 0x31, 0x9c, 0x6d, 0x21,
			0x3f, 0xbf, 0x44, 0xba, 0x63, 0x7f, 0x8f, 0x32,
			0x34, 0x1e, 0xac, 0x52, 0x6e, 0xa9, 0x06, 0x38,
			0x37, 0x89, 0x56, 0xbb, 0xab, 0x74, 0x9d, 0x75,
			0x14, 0x8d, 0x6a, 0x40, 0x8b, 0x4f, 0x6c, 0x89,
			0x74, 0x92, 0x1b, 0xcd, 0x22, 0x74, 0xbe, 0x69,
			0x37, 0x32, 0xe1, 0xae, 0x30, 0x83, 0xd1, 0x7f,
			0xdc, 0x89, 0x65, 0x3e, 0x4c, 0x22, 0xe8, 0x46,
			0xd3, 0x2f, 0x9c, 0x18, 0x85, 0x3f, 0xf0, 0x3c,
			0x7f, 0x3c, 0xc1, 0x76, 0x17, 0x5e, 0xbc, 0xcc,
			0xeb, 0xcb, 0x76, 0x0d, 0x99, 0x5a, 0xb3, 0xf3,
			0xc5, 0x57, 0xb2, 0xb3, 0x6b, 0x63, 0x63, 0x82,
			0x23, 0x57, 0x29, 0xfc, 0x1d, 0xfb, 0xd8, 0x8f,
			0xa9, 0xc7, 0x66, 0xda, 0xa3, 0x51, 0xe0, 0xb0,
			0xa4, 0xae, 0x0d, 0x8e, 0xd6, 0x3a, 0x01, 0xa4,
			0x56, 0x40, 0x36, 0x1a, 0x0b, 0xb4, 0x05, 0x6a,
			0xae, 0x07, 0x4e, 0x8d, 0xdf, 0x09, 0x6e, 0x50,
			0xbb, 0x2f, 0x49, 0xb6, 0x4b, 0x20, 0xdf, 0xd4,
			0x75, 0xa1, 0xc5, 0xfd, 0x38, 0x60, 0xe5, 0x28,
			0x3b, 0xcf, 0x3d, 0xfa, 0xbb, 0x95, 0x2c, 0x20,
			0x1c, 0x0d, 0x7f, 0xec, 0xf7, 0x95, 0x95, 0x26,
			0x7b, 0xd4, 0x55, 0xf8, 0x25, 0xb6, 0x7f, 0x59,
			0x72, 0x38, 0x79, 0x3b, 0x06, 0x95, 0x12, 0xb6,
			0x5a, 0x5c, 0xf7, 0x9a, 0x2f, 0xa6, 0xd2, 0x68,
			0x0e, 0x80, 0x04, 0xaf, 0x7f, 0xbf, 0x65, 0xef,
			0xdc, 0xf5, 0x91, 0x2f, 0xc2, 0xd3, 0x21, 0xc7,
			0xb2, 0xbc, 0x96, 0x95, 0x48, 0xc3, 0xda, 0x96,
			0x0c, 0xec, 0xd9, 0x46, 0x31, 0xf7, 0x54, 0x56,
			0xd2, 0x34, 0xca, 0x38, 0x12, 0xa2, 0xcc, 0xad,
			0x43, 0x9a, 0x29, 0x2d, 0xfd, 0xe9, 0xfc, 0x98,
			0xc1, 0xbc, 0x3e, 0x17, 0x26, 0x2f, 0xdb, 0x0e,
			0x16, 0x71, 0x30, 0xfa, 0x89, 0x35, 0xef, 0xc8,
			0x4f, 0xab, 0x77, 0x7d, 0x75, 0x7f, 0x0d, 0xce,
			0xbc, 0x76, 0xe8, 0x5c, 0x8a, 0x49, 0x88, 0xbb,
			0xa6, 0x33, 0x48, 0xaf, 0x0c, 0x38, 0x3a, 0x37,
			0xb4, 0x34, 0xab, 0x60, 0x0f, 0x80, 0xf2, 0x95,
			0x05, 0x3e, 0x07, 0xcc, 0xcc, 0x84, 0x57, 0x5f,
			0x82, 0x92, 0x64, 0x5c, 0x2d, 0xbb, 0xf6, 0x06,
			0x16, 0xef, 0xa4, 0x73, 0x76, 0x7d, 0x47, 0x08,
			0x67, 0x10, 0xeb, 0x1f, 0xc2, 0x36, 0x2e, 0x97,
			0x84, 0xd0, 0x7c, 0x31, 0x4b, 0xa2, 0xd4, 0x8a,
			0x9e, 0xc4, 0x61, 0x6e, 0x25, 0x6c, 0xcc, 0x9d,
			0xfb, 0x66, 0xd5, 0x44, 0x4f, 0x5d, 0x0e, 0xde,
			0x21, 0xc7, 0xbc, 0xf6, 0x7c, 0x51, 0x46, 0x83,
		},
	},
	/* SHA256 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0x7c, 0x5b, 0x1a, 0xb7, 0x87, 0x92, 0xaf, 0x89,
			0x9c, 0xf3, 0xed, 0xf2, 0x37, 0xda, 0x4a, 0xde,
			0xa8, 0x2a, 0xb4, 0xd4, 0x66, 0x4a, 0x83, 0x1e,
			0xa7, 0x6d, 0xac, 0x2b, 0x2d, 0x17, 0x37, 0x31,
			0x63, 0xdd, 0xf3, 0x16, 0xa8, 0x6c, 0x68, 0x91,
			0xeb, 0xcc, 0x0d, 0x8d, 0x78, 0xe6, 0xbb, 0xce,
			0x11, 0x44, 0x62, 0xd6, 0x41, 0x19, 0xfb, 0x0a,
			0xed, 0x81, 0xbd, 0x11, 0x2c, 0x09, 0x94, 0x7d,
			0x59, 0x72, 0xcf, 0xad, 0x66, 0xb2, 0x18, 0xf1,
			0x72, 0x3c, 0xaa, 0xf0, 0x9e, 0x81, 0xbe, 0x47,
			0x1d, 0xb7, 0x46, 0x73, 0x8b, 0xd9, 0xde, 0xf1,
			0x80, 0xef, 0x00, 0xa3, 0x64, 0x8f, 0x77, 0x76,
			0x85, 0xa4, 0x16, 0x42, 0x66, 0x6e, 0xa9, 0xf2,
			0x5b, 0xca, 0x2c, 0xe4, 0x52, 0xb4, 0x3b, 0x93,
			0xf4, 0x0a, 0xca, 0x73, 0xed, 0x92, 0x16, 0x1c,
			0x87, 0x3f, 0xdb, 0xab, 0x7e, 0xb0, 0xc8, 0x65,
		},
		/* S1 : 512 byte */
		{
			0x8f, 0x8a, 0x6e, 0xc7, 0xcd, 0xf5, 0x1e, 0x4c,
			0xd4, 0xff, 0x00, 0x8e, 0x6e, 0x00, 0x86, 0xee,
			0x39, 0x25, 0xc1, 0xd1, 0xcc, 0xb7, 0xd5, 0x38,
			0x1f, 0x24, 0xed, 0x7f, 0x5b, 0xa0, 0xbb, 0x46,
			0x63, 0x21, 0xb4, 0xad, 0x6e, 0xb0, 0xe0, 0xe8,
			0x06, 0x4a, 0x73, 0xbc, 0x74, 0x8f, 0x81, 0x1b,
			0x23, 0x92, 0x28, 0x96, 0x6b, 0xc3, 0xb6, 0xed,
			0xff, 0xcc, 0x7e, 0xf6, 0x69, 0xa6, 0xef, 0xaf,
			0x80, 0xbb, 0x73, 0x91, 0x69, 0x75, 0x33, 0xca,
			0xaf, 0x68, 0xa7, 0x8b, 0x15, 0x20, 0x67, 0x47,
			0xb4, 0xc2, 0x6d, 0x76, 0xf2, 0xce, 0x7d, 0x5d,
			0xf0, 0x98, 0xd6, 0x6a, 0xd9, 0x34, 0x6a, 0xec,
			0xfa, 0xcb, 0x13, 0xb9, 0xe7, 0xea, 0x32, 0xd5,
			0xbc, 0x78, 0xba, 0x92, 0x96, 0xf9, 0x31, 0x92,
			0x69, 0xad, 0x38, 0x2e, 0x40, 0x17, 0x96, 0x0b,
			0x2e, 0x2e, 0x44, 0x8d, 0xdc, 0xb5, 0x96, 0x4f,
			0xaa, 0x51, 0x39, 0x5c, 0xd4, 0x5a, 0x07, 0x38,
			0x79, 0x28, 0x61, 0xea, 0x78, 0x79, 0x1f, 0x36,
			0xa5, 0xf3, 0x53, 0xc6, 0xa7, 0xcc, 0xbf, 0x50,
			0xb8, 0xda, 0xcd, 0xbf, 0xab, 0xc0, 0x31, 0x80,
			0x6d, 0x60, 0x35, 0x45, 0xb0, 0x77, 0xb9, 0x48,
			0xf5, 0xe6, 0xa5, 0x12, 0x69, 0xd8, 0xce, 0xd5,
			0x32, 0x57, 0xfb, 0xd2, 0xc9, 0x52, 0x21, 0x84,
			0x88, 0x8f, 0x4d, 0xca, 0x8e, 0x45, 0xc4, 0x1a,
			0xb4, 0xc0, 0x10, 0x0e, 0x0f, 0x01, 0x70, 0xa7,
			0xfd, 0x52, 0x78, 0x5e, 0x8b, 0x2d, 0x05, 0x0e,
			0x70, 0x73, 0x5b, 0xfc, 0x58, 0x16, 0x73, 0x3a,
			0xea, 0x65, 0xde, 0x86, 0xdd, 0x4f, 0x38, 0x5a,
			0x20, 0x8b, 0x80, 0x02, 0xb0, 0xe9, 0x46, 0x3d,
			0x34, 0x39, 0xc4, 0x42, 0x4e, 0x0c, 0x82, 0x8e,
			0x66, 0xed, 0x98, 0x47, 0x66, 0x37, 0xd8, 0xed,
			0x82, 0x97, 0x87, 0x70, 0x31, 0x3a, 0x7f, 0x5b,
			0x66, 0xcb, 0xa9, 0x76, 0xb9, 0x64, 0x0e, 0x2c,
			0x22, 0x80, 0x5c, 0x11, 0x32, 0xdc, 0xfc, 0xa2,
			0xdf, 0x0b, 0xfa, 0x54, 0x67, 0x9c, 0x2d, 0x5e,
			0xc6, 0xc5, 0xdd, 0x28, 0xe3, 0x43, 0x81, 0x22,
			0xd6, 0xfc, 0xcb, 0x62, 0x82, 0xb9, 0xee, 0x79,
			0x83, 0x57, 0xd4, 0xe3, 0xae, 0x6b, 0x83, 0x98,
			0xe6, 0xf2, 0xf0, 0x67, 0xb8, 0x09, 0x5b, 0x09,
			0xf1, 0xca, 0xb8, 0x86, 0xd8, 0xd4, 0xb2, 0xe3,
			0x46, 0x10, 0xa6, 0xa3, 0x6c, 0xee, 0x34, 0xb1,
			0xb0, 0xbe, 0x03, 0x4b, 0xac, 0x40, 0x0f, 0x9a,
			0x3d, 0xaa, 0x6f, 0xb4, 0x3a, 0x01, 0xa1, 0x1b,
			0x16, 0xee, 0x98, 0xc7, 0x4c, 0x55, 0xc3, 0x7b,
			0xee, 0x76, 0x5f, 0xd5, 0xd5, 0x09, 0x5b, 0x75,
			0xb5, 0xca, 0xde, 0xd7, 0x53, 0xe3, 0xd5, 0x18,
			0x73, 0xae, 0x74, 0xdf, 0x87, 0x3d, 0x2d, 0x83,
			0x7f, 0x8f, 0xc6, 0x1a, 0x27, 0xe2, 0xf7, 0x34,
			0x13, 0xee, 0x26, 0x2e, 0x21, 0x4a, 0xe2, 0x7e,
			0x80, 0xba, 0xb7, 0x8a, 0xe7, 0x79, 0xcf, 0x1e,
			0xb7, 0xc7, 0x1a, 0x46, 0x34, 0x06, 0x80, 0x40,
			0xfe, 0x37, 0xd3, 0x22, 0x50, 0xaa, 0xa0, 0xee,
			0x06, 0x1a, 0x1b, 0x9b, 0x09, 0x5e, 0xcb, 0x78,
			0xf5, 0x7b, 0xac, 0x72, 0x58, 0x57, 0x7a, 0x4e,
			0xd8, 0x6e, 0xf9, 0x80, 0x23, 0xe2, 0xa8, 0x8d,
			0x33, 0xfd, 0xc0, 0x7a, 0x2d, 0x14, 0xae, 0xe4,
			0x8f, 0x9e, 0xcf, 0x99, 0x83, 0xee, 0xd3, 0x8b,
			0x2f, 0x11, 0x26, 0x8b, 0xaf, 0x0c, 0x73, 0x82,
			0xc3, 0xb0, 0x2a, 0x8a, 0x2d, 0x41, 0x72, 0x9b,
			0x85, 0x98, 0xb8, 0xae, 0x74, 0xea, 0xc6, 0x35,
			0x30, 0xc0, 0x44, 0x20, 0xa0, 0x3d, 0xf1, 0xd9,
			0x73, 0xe5, 0x94, 0xaf, 0x5e, 0xb9, 0x52, 0x29,
			0x6a, 0xc4, 0x9c, 0xeb, 0x4b, 0x1a, 0x61, 0xa5,
			0xae, 0xd0, 0x94, 0xd7, 0x2c, 0x25, 0x1a, 0xa4,
		},
	},
	/* SHA256 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x6a, 0x41, 0xd3, 0xcb, 0x08, 0x57, 0x0b, 0xfd,
			0x2a, 0x07, 0xf6, 0x83, 0xfd, 0x9c, 0x4b, 0x8f,
			0xfc, 0x78, 0x6b, 0x49, 0xbf, 0xa1, 0xdc, 0x71,
			0x13, 0x7d, 0xfa, 0x45, 0x50, 0xda, 0xe9, 0xa5,
			0xe7, 0x31, 0xce, 0x75, 0x3b, 0x2c, 0x3f, 0x43,
			0xb2, 0x65, 0xf8, 0x29, 0x0b, 0x71, 0xcf, 0xf5,
			0xf2, 0x1d, 0xea, 0x88, 0xd1, 0xf7, 0x2f, 0x7c,
			0xea, 0x11, 0xfc, 0x0a, 0xa3, 0x01, 0x18, 0x2a,
			0x20, 0x2c, 0xea, 0xfb, 0x17, 0x45, 0xe9, 0x66,
			0xe0, 0x10, 0x54, 0xfe, 0xcc, 0x6c, 0x23, 0x2b,
			0xb8, 0x90, 0x3d, 0x88, 0xe2, 0x95, 0xeb, 0x89,
			0xfb, 0x35, 0x8d, 0x61, 0x7d, 0xd2, 0x8c, 0x23,
			0x3e, 0xb9, 0x8f, 0x27, 0x46, 0xa9, 0xf2, 0xaf,
			0xdd, 0x8f, 0x74, 0xca, 0xe9, 0x94, 0x2f, 0x79,
			0x77, 0x59, 0xce, 0x11, 0x98, 0x81, 0xfa, 0xe0,
			0x6e, 0x71, 0x15, 0x13, 0x86, 0x53, 0x2b, 0xd7,
		},
		/* S2 : 512 byte */
		{
			0xc2, 0x32, 0xd2, 0x19, 0xf6, 0x5b, 0x66, 0x3d,
			0x5f, 0x7c, 0x64, 0xbc, 0x0f, 0x0f, 0x11, 0xc5,
			0x26, 0x4f, 0xdd, 0x1e, 0x71, 0x91, 0x7b, 0xb1,
			0xb2, 0x5a, 0x37, 0xc5, 0x05, 0xc6, 0x49, 0xda,
			0x3b, 0xb8, 0xa3, 0xa1, 0x64, 0x78, 0x8a, 0x78,
			0x3a, 0xb7, 0x5f, 0x0e, 0xb1, 0xf2, 0x2e, 0xd0,
			0xf6, 0x78, 0x77, 0xfa, 0xf5, 0x9d, 0x0e, 0x89,
			0xbc, 0x73, 0x1a, 0xe4, 0xa6, 0x6b, 0x58, 0xff,
			0x14, 0x90, 0x91, 0x06, 0x6e, 0x50, 0x15, 0xe0,
			0xc4, 0x38, 0xe9, 0x8c, 0xb5, 0xb2, 0x70, 0x94,
			0x9d, 0xdb, 0xa2, 0xab, 0xdf, 0xd3, 0xf3, 0x8b,
			0x44, 0x3f, 0x0b, 0x82, 0xa3, 0xb4, 0x13, 0x2b,
			0x8b, 0x9e, 0xd6, 0x77, 0xa2, 0x65, 0x3b, 0x30,
			0xe0, 0xcf, 0x38, 0x1f, 0x87, 0xb2, 0xae, 0x33,
			0x90, 0xa8, 0x98, 0x34, 0x1b, 0x0b, 0xf2, 0x96,
			0x40, 0xa2, 0x2b, 0x04, 0xfa, 0x94, 0xb3, 0x7f,
			0xfa, 0x6d, 0xf6, 0x2c, 0x35, 0x2c, 0x8a, 0x47,
			0xb0, 0x4d, 0x35, 0xed, 0x0d, 0x6f, 0x99, 0xc0,
			0xfc, 0x9e, 0xa3, 0x40, 0xda, 0xb3, 0xa8, 0x4e,
			0x6f, 0x65, 0x36, 0x27, 0x60, 0x58, 0x67, 0xb4,
			0xb1, 0xdb, 0xd3, 0xd7, 0x0d, 0x6e, 0xa2, 0x18,
			0xe9, 0xdf, 0xf5, 0x70, 0x20, 0xef, 0x44, 0xe1,
			0xaf, 0xe6, 0xb6, 0x65, 0x96, 0x7d, 0xe0, 0x8a,
			0x44, 0x78, 0x11, 0xe9, 0x8f, 0xcf, 0xc0, 0xb6,
			0x0a, 0x51, 0x3f, 0xf1, 0x8a, 0x8e, 0x03, 0xdf,
			0x63, 0x30, 0x3e, 0xd0, 0x52, 0x45, 0xd4, 0x83,
			0x88, 0x7d, 0xa7, 0x3b, 0xe1, 0xc6, 0x32, 0x5a,
			0x56, 0xa6, 0xd0, 0x30, 0x93, 0x11, 0xed, 0x2e,
			0xfc, 0x9c, 0x4a, 0xe4, 0x6f, 0x6e, 0x1b, 0xf4,
			0xba, 0x24, 0x3b, 0xb2, 0x3b, 0x89, 0x41, 0x95,
			0x48, 0x03, 0x79, 0xb2, 0x0c, 0xe0, 0x28, 0x7f,
			0x4a, 0xa5, 0x78, 0xf1, 0xe9, 0xd8, 0xd1, 0x0a,
			0xd6, 0xcf, 0x50, 0xde, 0x3d, 0x6b, 0x2e, 0x32,
			0xc6, 0x03, 0x05, 0xb4, 0xe3, 0x3f, 0xed, 0x63,
			0xfe, 0xf8, 0xb9, 0xf0, 0x3e, 0xaa, 0x53, 0x99,
			0xe5, 0x43, 0xa2, 0x67, 0x51, 0x00, 0xa6, 0x3b,
			0x7b, 0x14, 0xb2, 0xd1, 0xeb, 0x33, 0x43, 0x55,
			0x6f, 0x84, 0xda, 0x00, 0x5b, 0x56, 0x71, 0xca,
			0x09, 0xc9, 0xcc, 0x6b, 0x2d, 0x1c, 0x49, 0x28,
			0x15, 0x2b, 0x3c, 0xc6, 0x74, 0x57, 0xb9, 0xa3,
			0x5b, 0xeb, 0x82, 0xf1, 0xfe, 0x7e, 0x9f, 0xed,
			0x17, 0x70, 0xb3, 0x43, 0xc6, 0x8a, 0x5b, 0xa1,
			0xb6, 0xd6, 0xc8, 0x39, 0x99, 0x5a, 0x8e, 0x6c,
			0x5d, 0xa6, 0x0c, 0x60, 0x39, 0x51, 0x14, 0x67,
			0xb0, 0x5a, 0x82, 0x75, 0xeb, 0x3d, 0x32, 0xf3,
			0x9a, 0x72, 0xe0, 0x0a, 0x17, 0xb2, 0xb0, 0x94,
			0xff, 0xc5, 0x99, 0x4a, 0xed, 0xbd, 0x80, 0xf1,
			0x07, 0x4e, 0x22, 0x4f, 0x84, 0xda, 0xd9, 0x91,
			0x64, 0xa4, 0x4b, 0xa9, 0x11, 0xb7, 0xd7, 0xfe,
			0xb6, 0x85, 0x41, 0x10, 0x88, 0x12, 0x94, 0x00,
			0xf2, 0xf1, 0x0a, 0xdc, 0x60, 0x66, 0x28, 0xbd,
			0xc3, 0x72, 0xb4, 0x8e, 0xb8, 0x1c, 0x4d, 0x5a,
			0xe2, 0x74, 0xfc, 0x9f, 0xa4, 0x74, 0x27, 0xc2,
			0x17, 0x16, 0x28, 0x23, 0x8e, 0xdd, 0x00, 0xf1,
			0xc8, 0x97, 0xce, 0xb0, 0x27, 0x3e, 0x4f, 0xd5,
			0x39, 0xc8, 0xb1, 0xb5, 0xc4, 0x3f, 0xf2, 0x01,
			0x50, 0xae, 0x78, 0xa2, 0x7b, 0x35, 0x99, 0x25,
			0x54, 0xcd, 0x62, 0xf8, 0x8f, 0x66, 0xfb, 0x38,
			0x65, 0x14, 0x63, 0x4f, 0xf4, 0x19, 0xdb, 0x19,
			0x2b, 0x61, 0xd7, 0xcd, 0xe1, 0x2c, 0x2a, 0x5a,
			0x08, 0x6f, 0xbf, 0xe4, 0xb2, 0xe7, 0xf4, 0x9a,
			0x20, 0x60, 0x93, 0xbe, 0x77, 0x6e, 0x3e, 0x7d,
			0x16, 0xd8, 0x87, 0x43, 0xfa, 0x48, 0x6f, 0x92,
			0x24, 0xd0, 0x4e, 0x5a, 0x8e, 0x5b, 0xc5, 0x98,
		},
	},
	/* SHA256 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0x15, 0xc2, 0x4e, 0xe3, 0xa9, 0x39, 0x8f, 0x6e,
			0x5e, 0x57, 0xc0, 0x7a, 0x9e, 0xad, 0xbf, 0xbc,
			0x58, 0x3e, 0x11, 0xf8, 0x52, 0x85, 0xd4, 0x21,
			0xc1, 0xba, 0xf4, 0xc2, 0x17, 0x52, 0xa2, 0x9c,
			0x1c, 0xc5, 0x02, 0x90, 0x40, 0x93, 0x24, 0xc0,
			0x2f, 0x89, 0x36, 0xf2, 0xa7, 0x40, 0xf8, 0x35,
			0x86, 0xe8, 0x70, 0xc6, 0x28, 0x43, 0xde, 0x34,
			0xef, 0xf4, 0xf2, 0xc3, 0x23, 0xfa, 0x3d, 0x0e,
			0xfd, 0x78, 0xe6, 0xf1, 0xfe, 0xb7, 0x9e, 0xa6,
			0x85, 0x6d, 0xd5, 0x2e, 0xa1, 0x40, 0x2e, 0xf3,
			0x24, 0x86, 0x31, 0xab, 0xf9, 0x4f, 0x41, 0x7f,
			0xb7, 0xa5, 0xcd, 0x6d, 0x75, 0x12, 0xca, 0xeb,
			0xe1, 0x63, 0x60, 0xcd, 0x8c, 0xac, 0xe3, 0x68,
			0x88, 0x8c, 0x06, 0xf8, 0x35, 0xb2, 0x4b, 0x3f,
			0x58, 0xa0, 0xbe, 0x6f, 0x6d, 0xaf, 0xe3, 0x4a,
			0x3f, 0x54, 0xed, 0x88, 0xb5, 0xa1, 0x30, 0x79,
		},
		/* S3 : 512 byte */
		{
			0xaf, 0x4e, 0x57, 0x0f, 0x2f, 0xd1, 0x01, 0xa0,
			0x40, 0x35, 0xc6, 0x1d, 0x55, 0x0b, 0x6a, 0xae,
			0x1a, 0xfd, 0x73, 0x28, 0xec, 0xaf, 0xa0, 0xf7,
			0x3d, 0x3a, 0x81, 0x27, 0x2e, 0x79, 0x1e, 0xad,
			0xdf, 0x08, 0x01, 0xc5, 0x0a, 0xa2, 0xbf, 0x18,
			0xe2, 0x03, 0x23, 0xaf, 0xf9, 0xfe, 0x8d, 0x44,
			0xd3, 0x23, 0x10, 0x58, 0x6d, 0x66, 0xf6, 0xb8,
			0x8f, 0xf8, 0x5f, 0x2a, 0x6e, 0x2c, 0xe8, 0x4f,
			0x41, 0x97, 0x9d, 0xdf, 0x34, 0x62, 0x0d, 0x09,
			0x70, 0x74, 0xd5, 0xfc, 0x8c, 0x5d, 0xdb, 0xca,
			0xe2, 0x6a, 0x2a, 0x02, 0x74, 0x41, 0xea, 0x50,
			0x8e, 0xdb, 0xfe, 0x44, 0xe9, 0xe4, 0x4d, 0x7b,
			0x10, 0x1a, 0x36, 0x00, 0x57, 0x0d, 0xa7, 0x40,
			0x9e, 0x0f, 0x62, 0x57, 0x0b, 0x4d, 0x69, 0x0b,
			0xa7, 0xae, 0x82, 0xa9, 0xaf, 0x67, 0x8d, 0x21,
			0x0d, 0x4c, 0x70, 0xb2, 0x9b, 0x65, 0x26, 0xd3,
			0x5e, 0xe1, 0xbd, 0x52, 0x8d, 0xef, 0x9f, 0x63,
			0xb2, 0xfd, 0xc6, 0x99, 0x3d, 0xea, 0x73, 0x2b,
			0x87, 0x79, 0xcc, 0x6a, 0x1d, 0xda, 0x0a, 0x89,
			0xef, 0x30, 0x02, 0x24, 0x86, 0x57, 0x1f, 0x4b,
			0x62, 0x35, 0x74, 0x36, 0x26, 0xc9, 0x9e, 0xdb,
			0xfe, 0xd1, 0x3b, 0x32, 0x12, 0x2d, 0xd5, 0x1e,
			0xc4, 0x5b, 0xed, 0xb2, 0xe8, 0x76, 0x59, 0x9e,
			0xbc, 0xa0, 0x55, 0x38, 0x53, 0xe6, 0xee, 0x32,
			0xa8, 0x18, 0xff, 0x6b, 0xac, 0x03, 0xd4, 0xb7,
			0x74, 0xee, 0x01, 0xd8, 0x79, 0xf2, 0x2b, 0x7f,
			0x1f, 0x30, 0x8b, 0x11, 0xe9, 0xb9, 0xe6, 0xe6,
			0x07, 0x17, 0xee, 0x22, 0xbf, 0xd4, 0x06, 0xa4,
			0x00, 0x5f, 0xeb, 0x48, 0x8b, 0xfd, 0x6d, 0x24,
			0xcd, 0x18, 0x03, 0x88, 0x48, 0xb5, 0xba, 0xdf,
			0x58, 0x0c, 0x9f, 0xd1, 0xdd, 0x8b, 0x93, 0x08,
			0x23, 0x33, 0x2c, 0x5e, 0x7c, 0xe8, 0x5c, 0xcd,
			0x1a, 0xa1, 0x23, 0x45, 0x48, 0x9d, 0x84, 0x3c,
			0x32, 0x59, 0x79, 0xcf, 0x82, 0x14, 0x7d, 0xae,
			0xfa, 0xae, 0x0d, 0x7b, 0xd5, 0xc2, 0x51, 0x05,
			0x7a, 0xd0, 0x53, 0x6c, 0xa4, 0x3c, 0x20, 0x21,
			0xba, 0xa4, 0x21, 0xea, 0x3f, 0xee, 0x0c, 0x03,
			0x76, 0xd5, 0x5e, 0x89, 0x13, 0x6e, 0x95, 0xc7,
			0x13, 0xb1, 0x82, 0xf8, 0x89, 0x3b, 0xfc, 0x47,
			0xf0, 0xb8, 0x6a, 0x2f, 0xda, 0x14, 0x66, 0x4c,
			0x4e, 0x30, 0x63, 0xc0, 0x5c, 0xe4, 0x3d, 0xdd,
			0x0b, 0x83, 0xa0, 0xae, 0x79, 0x2e, 0x07, 0xe3,
			0xa7, 0xad, 0x22, 0x2f, 0x99, 0x75, 0xbe, 0x8f,
			0xc5, 0x1b, 0x61, 0x0e, 0xa3, 0xb1, 0x27, 0xb5,
			0x9f, 0x7d, 0x88, 0xf4, 0x3b, 0x45, 0xb9, 0xaa,
			0xfb, 0xae, 0x10, 0xf0, 0x38, 0xf2, 0xab, 0x22,
			0xbf, 0xf0, 0x8b, 0x62, 0x29, 0x97, 0xd3, 0x4b,
			0x12, 0xd6, 0x9f, 0x57, 0x68, 0x35, 0x21, 0x41,
			0xd5, 0x56, 0x0d, 0x91, 0x5c, 0x08, 0xe2, 0x16,
			0xdb, 0xd9, 0x36, 0x14, 0x41, 0x0d, 0xb1, 0x33,
			0x5d, 0x7c, 0x03, 0xed, 0x23, 0x83, 0xea, 0x46,
			0x7a, 0xa3, 0x7c, 0x21, 0x61, 0x72, 0x4e, 0x2c,
			0x1d, 0x2f, 0x96, 0xbf, 0x08, 0x2c, 0x9f, 0xee,
			0xe5, 0x22, 0xed, 0xd8, 0x49, 0xf4, 0x9c, 0xe1,
			0x7a, 0xfd, 0xfd, 0xaa, 0x16, 0xd2, 0x60, 0x0d,
			0x70, 0x48, 0x4c, 0x26, 0xd1, 0x7d, 0x1e, 0xaf,
			0x12, 0x31, 0xaf, 0xef, 0x05, 0x6b, 0x65, 0x0c,
			0x8d, 0xdf, 0xaf, 0xdc, 0xc9, 0x3a, 0xba, 0x55,
			0x38, 0xd2, 0xb0, 0x32, 0xf4, 0x77, 0xba, 0x40,
			0x43, 0x34, 0x1a, 0x4d, 0xb3, 0xb0, 0xc6, 0xa5,
			0x7a, 0xab, 0x58, 0x65, 0x90, 0xec, 0xca, 0xa8,
			0xc3, 0x49, 0xa6, 0x86, 0x62, 0xc3, 0x0b, 0x9e,
			0xe7, 0x27, 0x42, 0x9c, 0x3a, 0xc4, 0x16, 0x41,
			0x90, 0xcf, 0x53, 0xcd, 0xdd, 0xf7, 0x8c, 0x91,
		},
	},
	/* SHA256 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0xd1, 0x00, 0x62, 0xdf, 0x2f, 0xd6, 0x49, 0x2a,
			0x8e, 0x3e, 0xc9, 0xc1, 0xbb, 0x1a, 0x4b, 0x3e,
			0x7d, 0x69, 0xde, 0x41, 0x82, 0x6e, 0xce, 0xb5,
			0x9a, 0x0f, 0x78, 0x55, 0x23, 0x7b, 0x5c, 0x6e,
			0xc4, 0x0f, 0x74, 0xe6, 0x5f, 0x7b, 0xc5, 0x7d,
			0x40, 0x20, 0x33, 0x53, 0x9b, 0xbd, 0x99, 0xab,
			0x35, 0x4b, 0x19, 0xee, 0x61, 0x25, 0xb1, 0xf3,
			0x2e, 0x6c, 0x0f, 0x7d, 0xdf, 0xaa, 0xa7, 0x05,
			0x9b, 0xb5, 0x02, 0xba, 0x66, 0xd7, 0x59, 0xc7,
			0x50, 0x2b, 0x62, 0xd4, 0xec, 0x4a, 0x6a, 0xcf,
			0x04, 0x27, 0xa4, 0xec, 0x8b, 0x38, 0xc1, 0xea,
			0xd3, 0xd7, 0xc0, 0x99, 0x01, 0xe7, 0x08, 0x99,
			0xbd, 0xba, 0xb4, 0x64, 0x2e, 0x32, 0x2e, 0x8e,
			0x25, 0x27, 0xff, 0x4c, 0x98, 0x09, 0xe5, 0x34,
			0x53, 0xc7, 0x26, 0x43, 0xea, 0xef, 0x25, 0x24,
			0xf2, 0x16, 0x33, 0xaf, 0x70, 0x7a, 0xa7, 0xb1,
		},
		/* S4 : 512 byte */
		{
			0x74, 0x80, 0x5c, 0x6b, 0xb3, 0x09, 0x98, 0xca,
			0x55, 0xea, 0x3d, 0x61, 0x40, 0x8a, 0xe9, 0x6c,
			0x4c, 0xa8, 0x72, 0x19, 0xb2, 0x2c, 0x4d, 0x32,
			0xc6, 0x53, 0xe6, 0x7d, 0x4b, 0x96, 0xf6, 0x41,
			0xce, 0x19, 0xf5, 0x74, 0x71, 0x2e, 0x18, 0xf1,
			0x7a, 0x4e, 0xe8, 0xc0, 0x22, 0xab, 0x7e, 0xde,
			0x8e, 0x0a, 0x2e, 0x97, 0x4e, 0x5d, 0xcf, 0x81,
			0xe8, 0x2e, 0xae, 0x3d, 0x3e, 0x93, 0x5a, 0xad,
			0x88, 0xf3, 0xf5, 0x1a, 0x9d, 0x7a, 0x31, 0x36,
			0xa7, 0xba, 0xf0, 0x5c, 0xbe, 0x3c, 0x30, 0xcb,
			0xcb, 0xa2, 0x8e, 0x7f, 0xfc, 0xcb, 0xae, 0x8a,
			0x60, 0x77, 0x99, 0xf8, 0xef, 0xad, 0x1a, 0xbc,
			0x1f, 0xf8, 0x76, 0xf9, 0xcb, 0x3b, 0xec, 0xc4,
			0x75, 0xfd, 0x6a, 0xc8, 0xfd, 0xda, 0xd9, 0x03,
			0x7c, 0x7c, 0xeb, 0xf5, 0xa4, 0x7b, 0x09, 0x68,
			0x8f, 0x51, 0xa5, 0x1b, 0x48, 0xf0, 0x87, 0xd6,
			0xe7, 0xdc, 0xc6, 0xda, 0xe5, 0x58, 0x4c, 0x53,
			0x5a, 0x07, 0xec, 0x00, 0x22, 0xaa, 0x31, 0x6d,
			0x7e, 0x39, 0x27, 0x25, 0x44, 0xba, 0xf1, 0x79,
			0x68, 0xa9, 0x95, 0x1e, 0xa9, 0xc2, 0x24, 0x74,
			0x76, 0x4d, 0x6e, 0x7d, 0x6a, 0x62, 0x2e, 0xf5,
			0x68, 0x3f, 0x5e, 0xe4, 0x7a, 0xf6, 0x73, 0x57,
			0x44, 0xfb, 0x59, 0x08, 0xa9, 0xcf, 0xae, 0x78,
			0x25, 0x87, 0x5a, 0xfa, 0xc7, 0x68, 0x6f, 0xfa,
			0xb3, 0xc4, 0x2b, 0xce, 0xf7, 0x3f, 0x85, 0x0f,
			0x87, 0x49, 0xc1, 0x40, 0xc6, 0x79, 0xe8, 0x09,
			0xab, 0x1e, 0xd4, 0x07, 0xae, 0xff, 0x28, 0x53,
			0xb0, 0x3a, 0xfd, 0x59, 0x4d, 0x85, 0x21, 0xbd,
			0xb5, 0x5d, 0xce, 0x91, 0xf5, 0x34, 0x00, 0x77,
			0x32, 0x2d, 0x81, 0xc9, 0x2e, 0x6e, 0xbd, 0xd8,
			0xbe, 0x0c, 0xa0, 0x01, 0xb6, 0x9b, 0x44, 0x4d,
			0xc5, 0x4f, 0xf4, 0x12, 0x0d, 0x7f, 0x1f, 0x0b,
			0xf5, 0x61, 0x60, 0x61, 0xf7, 0xe3, 0xa9, 0x9d,
			0x6a, 0xc4, 0x2a, 0x2f, 0x5e, 0x1b, 0x73, 0xd0,
			0x3c, 0x5d, 0xde, 0xe0, 0x47, 0x40, 0xd7, 0x93,
			0xae, 0x35, 0xb7, 0xfd, 0xac, 0x52, 0x15, 0x4c,
			0xd9, 0xdd, 0x0e, 0x78, 0xd1, 0xbc, 0xb5, 0x93,
			0x5b, 0xe8, 0x4c, 0x71, 0x04, 0x76, 0x3f, 0x00,
			0xc2, 0xb9, 0xa8, 0xbf, 0xee, 0x9d, 0x7f, 0x35,
			0xce, 0x2e, 0x62, 0xbe, 0x17, 0xf5, 0x6f, 0x9a,
			0xc0, 0xd3, 0x64, 0x5a, 0xbf, 0xb3, 0x79, 0xed,
			0xfe, 0x4f, 0x29, 0x5f, 0x2c, 0x5c, 0xd5, 0x64,
			0x12, 0xb8, 0x95, 0xb2, 0x04, 0x32, 0x9c, 0x18,
			0x6f, 0x36, 0x93, 0x30, 0xcb, 0x96, 0x45, 0xc5,
			0xc5, 0x0e, 0xaa, 0xb1, 0x52, 0x3d, 0x02, 0x96,
			0xfc, 0x83, 0x25, 0xb3, 0x43, 0x78, 0x98, 0xf7,
			0xd3, 0xce, 0x6e, 0x40, 0xa4, 0x87, 0xf7, 0x8a,
			0x7e, 0x32, 0xa7, 0x73, 0x23, 0xea, 0x47, 0x82,
			0x01, 0xe5, 0xe1, 0x82, 0x3a, 0xb8, 0x99, 0x18,
			0x9e, 0x96, 0xe0, 0x31, 0x93, 0x46, 0xf7, 0x26,
			0x2b, 0x52, 0x4e, 0x0f, 0xc2, 0xcb, 0x94, 0x9f,
			0x6a, 0x27, 0x5d, 0x69, 0xaf, 0xf5, 0x57, 0x40,
			0x96, 0xc5, 0xf8, 0x1c, 0x9c, 0x31, 0x29, 0xb5,
			0x3b, 0xb8, 0xa9, 0x83, 0x6e, 0x7d, 0x78, 0x8b,
			0x51, 0x51, 0x59, 0x13, 0x59, 0x0c, 0xf4, 0x22,
			0xd7, 0x66, 0x35, 0x45, 0xd1, 0x92, 0xf0, 0x37,
			0x70, 0xd9, 0xeb, 0x9c, 0x83, 0x8e, 0xbd, 0x6c,
			0x14, 0x17, 0x65, 0xa9, 0x11, 0x35, 0x47, 0x3c,
			0x89, 0x15, 0xe2, 0x91, 0x33, 0x74, 0xbf, 0x77,
			0x4d, 0x61, 0xb1, 0x14, 0x3f, 0x10, 0x75, 0x6a,
			0x0a, 0xb6, 0x1b, 0xe6, 0xc2, 0x00, 0xbf, 0x2a,
			0x70, 0x96, 0xe7, 0xbb, 0xb9, 0x63, 0x34, 0xb4,
			0x9b, 0x8d, 0xe2, 0x54, 0x4a, 0x58, 0x4e, 0xd9,
			0x88, 0xe4, 0x04, 0x91, 0x0c, 0x4c, 0x05, 0x9f,
		},
	},
	/* SHA256 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0xbe, 0x8d, 0x5a, 0x34, 0x15, 0x98, 0x4f, 0x55,
			0xee, 0x92, 0x0a, 0x68, 0x32, 0x97, 0xca, 0xf8,
			0x48, 0x40, 0x54, 0x0e, 0x2a, 0x69, 0xa1, 0x0d,
			0xbf, 0x38, 0xd4, 0xdb, 0x65, 0x5b, 0x6f, 0x41,
			0xcc, 0x90, 0xc5, 0x95, 0x0b, 0x6b, 0xb3, 0x96,
			0xed, 0x54, 0x8b, 0x97, 0x8c, 0xcf, 0x51, 0xe0,
			0x22, 0x00, 0x2a, 0x51, 0x1b, 0x6e, 0x11, 0x48,
			0x6f, 0x07, 0xcc, 0xc5, 0x2e, 0x64, 0x1e, 0xcc,
			0x5d, 0x4f, 0x3f, 0x0b, 0x00, 0x82, 0x89, 0xbd,
			0x78, 0xb2, 0x74, 0xce, 0x0f, 0xcc, 0xc4, 0x40,
			0xd2, 0x7f, 0x01, 0xcd, 0x9f, 0xfa, 0x28, 0xce,
			0x7e, 0xf7, 0xa0, 0x5b, 0x34, 0xf8, 0x6f, 0xb4,
			0x16, 0xd0, 0xae, 0xdf, 0x1c, 0x66, 0x3a, 0x93,
			0x36, 0xb6, 0xad, 0x55, 0xe3, 0x18, 0x8d, 0xe1,
			0xff, 0xc5, 0xc3, 0xcb, 0xdc, 0x97, 0x4e, 0x66,
			0x95, 0x10, 0x38, 0xe5, 0x9f, 0x9e, 0xc9, 0xc0,
		},
		/* S5 : 512 byte */
		{
			0xce, 0xc6, 0x01, 0xdd, 0x36, 0xbb, 0xf0, 0x49,
			0xcd, 0xe7, 0xb7, 0xf2, 0xba, 0xd0, 0x94, 0xaa,
			0x17, 0xe0, 0xbc, 0x88, 0x58, 0xac, 0x38, 0x3d,
			0x26, 0x41, 0x2e, 0x0e, 0x17, 0xcf, 0x97, 0x86,
			0x64, 0x1f, 0xed, 0x6e, 0x22, 0x10, 0x64, 0xeb,
			0xdd, 0x2b, 0xa5, 0x0f, 0xd2, 0x01, 0x6b, 0x0b,
			0x49, 0x6f, 0x29, 0x70, 0x1a, 0x6e, 0x5f, 0x1e,
			0xaa, 0xf7, 0xfd, 0x2e, 0x0c, 0x9e, 0x7e, 0x87,
			0xb6, 0x02, 0x73, 0xbc, 0x94, 0x52, 0x3f, 0xd8,
			0xad, 0xa4, 0xb7, 0x6f, 0x4c, 0xa8, 0xbc, 0x72,
			0x43, 0x0a, 0xf4, 0xde, 0xe3, 0x1a, 0x32, 0xc6,
			0xf0, 0xe8, 0xaf, 0x0d, 0x0e, 0x3c, 0xc5, 0xe8,
			0x70, 0x00, 0xca, 0x77, 0xda, 0x59, 0x92, 0xe4,
			0xed, 0xbf, 0xc0, 0x73, 0x00, 0x55, 0xa6, 0x51,
			0x9e, 0x62, 0xea, 0x8b, 0x1f, 0x15, 0x76, 0xda,
			0xd8, 0x6c, 0x38, 0x9a, 0xda, 0xb8, 0x2f, 0x11,
			0xd9, 0xc9, 0x9a, 0xb6, 0x8d, 0x10, 0x25, 0x52,
			0xa1, 0x59, 0x18, 0x12, 0x94, 0xf6, 0x97, 0x50,
			0xfd, 0xbb, 0xab, 0xbd, 0x95, 0x98, 0x4d, 0xb7,
			0x0e, 0x43, 0xa1, 0x91, 0xbc, 0x20, 0xa3, 0x5c,
			0x68, 0xa2, 0xa6, 0x13, 0xab, 0xda, 0x54, 0x97,
			0xe4, 0xd3, 0x1e, 0xb3, 0x66, 0x38, 0xe8, 0x45,
			0xf4, 0xd4, 0xf4, 0xc8, 0x6d, 0x11, 0x9a, 0xc0,
			0xeb, 0x87, 0x0c, 0x09, 0x54, 0x49, 0x80, 0x2e,
			0xd3, 0x13, 0x7c, 0xec, 0x01, 0x0d, 0xf5, 0xf9,
			0x38, 0x0c, 0x2e, 0x19, 0x01, 0x6b, 0x8b, 0xbc,
			0xcd, 0x11, 0x7f, 0x64, 0x63, 0xbe, 0x57, 0x12,
			0x2c, 0xe3, 0x93, 0xef, 0x66, 0x14, 0xf5, 0x8e,
			0xac, 0x82, 0xdb, 0x37, 0xff, 0xe1, 0xe7, 0xd4,
			0x71, 0x3c, 0x8c, 0xfd, 0x0d, 0x2b, 0x96, 0x28,
			0x23, 0x20, 0x7e, 0xfe, 0x73, 0xd9, 0x3b, 0x6f,
			0x5a, 0xf0, 0x12, 0x1b, 0x0e, 0x85, 0xb6, 0x2b,
			0xe6, 0x8e, 0xd4, 0x98, 0xd0, 0xf5, 0x74, 0xdc,
			0x4b, 0xef, 0x93, 0xa3, 0x33, 0x15, 0xd9, 0xa2,
			0xab, 0x0e, 0xa7, 0x01, 0x7f, 0x4e, 0x95, 0xf5,
			0x6f, 0xdd, 0x89, 0xbc, 0x85, 0xf1, 0xcb, 0xa9,
			0x66, 0x18, 0x11, 0x4a, 0xfb, 0xd3, 0xa1, 0xff,
			0xef, 0x32, 0x15, 0x9e, 0x9f, 0xe0, 0x27, 0x2c,
			0x4e, 0x07, 0xe0, 0x1c, 0x76, 0xfd, 0x23, 0x15,
			0x9a, 0x74, 0xac, 0x71, 0x58, 0x16, 0xa6, 0x45,
			0x31, 0x12, 0x0d, 0x3a, 0x77, 0x7e, 0x5b, 0xa4,
			0x8e, 0x30, 0x3b, 0x47, 0x21, 0x54, 0x0a, 0x55,
			0xed, 0x98, 0xc4, 0xf7, 0x45, 0xc4, 0xbb, 0x1d,
			0x41, 0x4c, 0x94, 0x09, 0x4e, 0x2b, 0xe2, 0x1d,
			0xb6, 0x9a, 0x76, 0xf3, 0x20, 0x6f, 0xd1, 0x5f,
			0x73, 0x0e, 0x55, 0xbe, 0x47, 0xe0, 0x66, 0x06,
			0x90, 0x9b, 0x8b, 0xd9, 0x5d, 0x63, 0xd3, 0x22,
			0x73, 0x17, 0x11, 0x51, 0x63, 0x0f, 0x5f, 0x18,
			0x4c, 0x76, 0x7b, 0x5b, 0xd4, 0x2c, 0x06, 0x1e,
			0x46, 0x1d, 0x86, 0x1a, 0x3a, 0xa6, 0xd0, 0x28,
			0x10, 0x27, 0x57, 0xbc, 0x11, 0x36, 0xb4, 0x69,
			0x4a, 0x42, 0x16, 0xdd, 0x8f, 0xb6, 0xd8, 0x74,
			0x01, 0x92, 0x20, 0x2e, 0x0f, 0xcf, 0x09, 0xd8,
			0xe7, 0x03, 0xe6, 0x82, 0x78, 0xf4, 0xf0, 0xaa,
			0xfe, 0x19, 0x96, 0x40, 0x9b, 0x78, 0x78, 0x25,
			0x34, 0x3e, 0x12, 0x03, 0x94, 0x68, 0x37, 0xc9,
			0xc2, 0x1f, 0xd6, 0x83, 0x7d, 0x81, 0xe1, 0x04,
			0xa1, 0xbe, 0x60, 0x34, 0xa3, 0x72, 0xd0, 0xde,
			0xd0, 0xfe, 0x34, 0x0d, 0x26, 0xe0, 0x7a, 0xb5,
			0x63, 0x3d, 0xa3, 0xc8, 0xe1, 0x2f, 0x9b, 0x4e,
			0xd3, 0x10, 0x93, 0x19, 0x8a, 0x55, 0x11, 0xcf,
			0x5b, 0x3c, 0x0d, 0xea, 0x04, 0xb5, 0x64, 0x29,
			0x97, 0x37, 0xbc, 0xa0, 0xb7, 0xbb, 0x94, 0xe7,
			0xe5, 0xa0, 0xab, 0x14, 0xd1, 0x23, 0x8c, 0x14,
		},
	},
	/* SHA256 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0xd7, 0xd9, 0x16, 0x06, 0xf1, 0xec, 0xf5, 0x57,
			0x6c, 0x6f, 0xd5, 0xc1, 0x52, 0x8f, 0x39, 0x88,
			0x66, 0x59, 0x0c, 0xb9, 0x12, 0xda, 0x38, 0x6a,
			0xa1, 0x85, 0x74, 0x43, 0xae, 0xd5, 0x5d, 0x3e,
			0xdc, 0x33, 0xc9, 0xaa, 0xc8, 0x19, 0x58, 0x76,
			0x3c, 0x78, 0x4c, 0xac, 0xa6, 0x57, 0x9a, 0x3c,
			0xc8, 0xbd, 0x40, 0xfb, 0xb0, 0xd2, 0xda, 0xeb,
			0xeb, 0x41, 0x70, 0xbd, 0xf6, 0xe0, 0x93, 0x94,
			0xf5, 0x93, 0xa8, 0x0c, 0xa7, 0x6e, 0x83, 0x7b,
			0x9a, 0x19, 0x38, 0x77, 0x9b, 0x79, 0x2d, 0x98,
			0x71, 0x8c, 0x74, 0x7e, 0xcb, 0x95, 0x58, 0x16,
			0x76, 0x7a, 0x36, 0x1a, 0xd3, 0x6a, 0x8f, 0xd7,
			0x89, 0xc2, 0x5a, 0x33, 0x77, 0x32, 0x9f, 0xee,
			0xed, 0x1c, 0x41, 0x28, 0x1b, 0x3c, 0x1c, 0x24,
			0xc9, 0x8e, 0x4f, 0x4b, 0x49, 0x6c, 0xdb, 0x74,
			0xaa, 0xf7, 0x6e, 0x62, 0x2f, 0xb9, 0x79, 0x8e,
		},
		/* S6 : 512 byte */
		{
			0x71, 0x80, 0x82, 0x72, 0x3f, 0x43, 0x86, 0x94,
			0x32, 0x43, 0x3c, 0x91, 0x0c, 0x2d, 0x40, 0x0a,
			0xda, 0xaa, 0xa4, 0x0b, 0x9b, 0x3b, 0x25, 0x14,
			0x80, 0xa4, 0xf0, 0x1d, 0xd0, 0x01, 0xc3, 0x1e,
			0x93, 0x16, 0x6a, 0xa8, 0xa4, 0x54, 0x76, 0xc0,
			0x0c, 0x10, 0x18, 0x8a, 0x53, 0x8f, 0x8a, 0x53,
			0x3a, 0x5b, 0x60, 0x1a, 0x28, 0x80, 0x95, 0x7a,
			0xed, 0x4c, 0x4b, 0xd6, 0x9f, 0x68, 0xc0, 0x8a,
			0x3b, 0x86, 0x52, 0x4c, 0xed, 0xb6, 0xfc, 0xb9,
			0xbb, 0x46, 0x50, 0x9b, 0xa3, 0x92, 0x49, 0x95,
			0x36, 0x71, 0xdc, 0x6d, 0x1f, 0x9f, 0x04, 0xf2,
			0x19, 0x80, 0x2b, 0xe5, 0xc1, 0x06, 0xd5, 0x19,
			0x57, 0x3e, 0x16, 0xfd, 0x59, 0x6c, 0x67, 0x34,
			0xe1, 0x73, 0xc5, 0x24, 0xab, 0x4d, 0x23, 0x04,
			0xc7, 0x3d, 0x9f, 0xb7, 0x8f, 0x34, 0x96, 0xe8,
			0xbb, 0x42, 0xf0, 0xbf, 0x0a, 0x27, 0x55, 0x52,
			0x31, 0x23, 0x96, 0xb9, 0xb8, 0x93, 0xb5, 0xfe,
			0x3b, 0x5e, 0xf1, 0xc9, 0xfa, 0x8c, 0x22, 0xd1,
			0xb9, 0x86, 0x3f, 0xf0, 0xc0, 0xb4, 0x22, 0x1a,
			0x7b, 0xa9, 0x1c, 0x18, 0x49, 0xed, 0x2d, 0x57,
			0x1e, 0x68, 0xc5, 0xe2, 0xeb, 0xf1, 0xfc, 0x89,
			0x42, 0xaf, 0xf5, 0x7b, 0x9f, 0xd7, 0x8b, 0x14,
			0x91, 0xc6, 0x15, 0x71, 0x7f, 0x1e, 0xc4, 0x2d,
			0x9e, 0xc0, 0xfe, 0x84, 0x10, 0x0b, 0xba, 0xfa,
			0x54, 0xe7, 0xd9, 0xf0, 0xee, 0x48, 0xf1, 0x8c,
			0x5f, 0xe2, 0xde, 0xc2, 0xd0, 0x2b, 0xc7, 0x89,
			0x77, 0x21, 0x75, 0xd5, 0xc6, 0xf1, 0xfc, 0xb5,
			0x65, 0x88, 0x67, 0x5f, 0x02, 0x35, 0x6e, 0x83,
			0x66, 0xcf, 0xef, 0xfb, 0x4e, 0x56, 0x0e, 0x1e,
			0x6d, 0x18, 0x67, 0xe9, 0x3f, 0x02, 0xd4, 0xe6,
			0x8c, 0xde, 0xfd, 0xac, 0xa4, 0xe1, 0x12, 0x2f,
			0xa9, 0xc9, 0xb5, 0x2b, 0x90, 0x28, 0x6a, 0xab,
			0x9c, 0x2d, 0x13, 0x57, 0x25, 0x78, 0x70, 0xaa,
			0x50, 0x7b, 0xb7, 0xf9, 0x4e, 0xec, 0xc3, 0xfb,
			0x36, 0x31, 0x9c, 0x3e, 0x7e, 0x34, 0x62, 0x0a,
			0x74, 0xc3, 0x05, 0xff, 0x1a, 0xb0, 0xca, 0xfc,
			0x01, 0x98, 0xff, 0x4e, 0x19, 0xf8, 0xc8, 0xcd,
			0x5a, 0x1c, 0x6f, 0xad, 0xed, 0x98, 0xbc, 0x21,
			0x67, 0xea, 0x92, 0xa3, 0xaf, 0x15, 0x7b, 0x9c,
			0xdb, 0x45, 0x46, 0xee, 0x3c, 0x27, 0x39, 0xb8,
			0xdc, 0xe0, 0x11, 0xd3, 0xda, 0x83, 0xe9, 0xe1,
			0x36, 0x52, 0x75, 0x5e, 0x79, 0xbf, 0xc6, 0xe2,
			0x8c, 0x45, 0x90, 0x6f, 0x32, 0x7b, 0x78, 0xa7,
			0xe7, 0xab, 0x9d, 0x24, 0x25, 0x6c, 0xd0, 0x36,
			0x28, 0x0e, 0xe4, 0xac, 0x89, 0x30, 0xa3, 0x6b,
			0xce, 0x28, 0xce, 0x8f, 0x06, 0x05, 0x10, 0xc0,
			0xd1, 0x16, 0xda, 0x89, 0x00, 0xb6, 0x06, 0x44,
			0x40, 0xf0, 0x26, 0x24, 0xea, 0x31, 0x37, 0x6b,
			0xa8, 0x7e, 0xac, 0x15, 0x62, 0xdb, 0xd6, 0x1d,
			0x15, 0x62, 0x44, 0x8f, 0x3e, 0x72, 0x06, 0x09,
			0xab, 0x61, 0x4b, 0x47, 0x72, 0x97, 0xa6, 0x8d,
			0x19, 0xd4, 0xf9, 0x75, 0xb4, 0x4b, 0xc4, 0x02,
			0xc4, 0xd8, 0x6c, 0x53, 0xf3, 0xdc, 0x76, 0x05,
			0x23, 0x53, 0xab, 0x22, 0x88, 0x79, 0xbe, 0x3b,
			0x25, 0xf7, 0xac, 0xd9, 0x3e, 0x52, 0xfa, 0xcd,
			0xfb, 0x7f, 0x29, 0x88, 0x86, 0xf0, 0x20, 0x5e,
			0xe8, 0x92, 0x5e, 0xd4, 0x59, 0x67, 0xdc, 0x12,
			0xcc, 0x24, 0x67, 0x7f, 0xae, 0x8f, 0x55, 0xf3,
			0x6a, 0x7b, 0x77, 0x15, 0x57, 0x0b, 0x3f, 0x9e,
			0x88, 0xb3, 0x0e, 0x18, 0x5f, 0xa6, 0x82, 0x68,
			0xd0, 0x66, 0x07, 0x5e, 0xee, 0x9f, 0x13, 0x25,
			0x26, 0xfb, 0x2c, 0xd5, 0x0e, 0xaa, 0x75, 0xfb,
			0xbc, 0x3e, 0x56, 0x60, 0xba, 0x03, 0x1d, 0xb9,
			0x2d, 0xc7, 0xc4, 0x54, 0x00, 0x20, 0xfa, 0x54,
		},
	},
	/* SHA256 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0x5e, 0x8e, 0x72, 0x5d, 0x0c, 0x19, 0x52, 0x7b,
			0xb3, 0x65, 0xf3, 0x49, 0x77, 0xcd, 0xd3, 0x9a,
			0xee, 0x69, 0x3f, 0xd8, 0xeb, 0x8d, 0xc7, 0xa7,
			0x94, 0x93, 0xee, 0xc5, 0xbb, 0xdb, 0x26, 0x34,
			0x6b, 0x0f, 0xfb, 0x9c, 0xc0, 0xac, 0x8e, 0xeb,
			0xfc, 0xbe, 0x45, 0x3e, 0xd8, 0xeb, 0xc1, 0x04,
			0x18, 0xb2, 0x10, 0x60, 0x62, 0xf5, 0x24, 0xd0,
			0x50, 0xb5, 0x84, 0x0d, 0xc4, 0x1e, 0x42, 0x32,
			0x7c, 0xc3, 0x2d, 0x1f, 0xe6, 0x2a, 0x47, 0x20,
			0x35, 0x8b, 0x7a, 0xca, 0xb4, 0xd5, 0x86, 0x26,
			0x5a, 0xf2, 0x3e, 0x12, 0x9f, 0x4c, 0xf4, 0xe3,
			0x8f, 0x90, 0x32, 0x4f, 0x1d, 0xb2, 0xab, 0x8b,
			0xb9, 0x31, 0x70, 0xb0, 0x25, 0x9c, 0x67, 0x63,
			0x84, 0x55, 0xfa, 0xb5, 0xb4, 0x95, 0x0c, 0x48,
			0xdd, 0xb0, 0x31, 0xb4, 0x4a, 0x9a, 0x1e, 0x28,
			0x78, 0xaa, 0x5e, 0x55, 0x6d, 0x9f, 0x47, 0x87,
		},
		/* S7 : 512 byte */
		{
			0x53, 0xc5, 0x51, 0xc7, 0x30, 0xcd, 0xa7, 0xbe,
			0x28, 0x12, 0xbb, 0x9e, 0x2e, 0x13, 0xd8, 0xc8,
			0x49, 0x83, 0x3b, 0x42, 0x1d, 0xbe, 0xfa, 0xf3,
			0xbc, 0x0a, 0xfb, 0x90, 0xf0, 0xc8, 0x0c, 0xfd,
			0x45, 0xab, 0xc3, 0x83, 0xa0, 0xf3, 0x85, 0x25,
			0xb9, 0x3d, 0x40, 0x51, 0x85, 0x1e, 0x48, 0xc6,
			0xfa, 0xae, 0xb8, 0xa9, 0x12, 0xce, 0x4b, 0x07,
			0xe4, 0xd8, 0xa9, 0xad, 0x37, 0x4d, 0x06, 0x5f,
			0x32, 0x19, 0xaa, 0x0a, 0xf0, 0xd9, 0x41, 0x1f,
			0xfe, 0xaa, 0x90, 0xd5, 0x5d, 0x22, 0x19, 0x4b,
			0x4b, 0xe2, 0x80, 0x7e, 0xb4, 0x4f, 0x89, 0x16,
			0x24, 0xb8, 0x0c, 0xa2, 0xac, 0xf0, 0x50, 0xd0,
			0x0a, 0x6d, 0xee, 0x14, 0xcc, 0x76, 0x53, 0x90,
			0x8f, 0x5a, 0x78, 0x84, 0x41, 0xb2, 0x46, 0xb8,
			0x9c, 0x7c, 0x84, 0x45, 0x3e, 0x85, 0x81, 0x91,
			0xc1, 0x95, 0x31, 0xae, 0x67, 0x40, 0x23, 0x63,
			0x67, 0x28, 0xfd, 0xcd, 0xc0, 0x05, 0xce, 0x72,
			0xd5, 0x1f, 0xdd, 0xc9, 0x54, 0xd0, 0x1b, 0x19,
			0xb9, 0xb6, 0x2a, 0x92, 0x2f, 0x25, 0x42, 0xd8,
			0x18, 0x32, 0x21, 0x6e, 0xfb, 0x8b, 0x4b, 0x0f,
			0x8d, 0x75, 0xb3, 0x87, 0x12, 0xca, 0x50, 0xb2,
			0x9c, 0x59, 0x9e, 0xa8, 0x52, 0xa5, 0x46, 0x6a,
			0x77, 0x6b, 0x47, 0xda, 0x0f, 0x16, 0xd5, 0x09,
			0x68, 0x2f, 0xaa, 0xb2, 0x17, 0x7a, 0xb1, 0x4c,
			0x46, 0xb0, 0x72, 0x30, 0x00, 0xc7, 0x0e, 0x45,
			0xb6, 0x10, 0xac, 0x76, 0x25, 0xdb, 0x6d, 0xa9,
			0x8d, 0xb2, 0xac, 0x59, 0x62, 0x33, 0xcc, 0xa2,
			0xc2, 0xa3, 0x11, 0xf2, 0xd4, 0xab, 0xfc, 0x25,
			0x6e, 0xc9, 0x3f, 0xf2, 0xb9, 0x0d, 0x97, 0xfc,
			0x19, 0x6f, 0x01, 0xf1, 0xb5, 0x12, 0x74, 0xe7,
			0x95, 0x96, 0x2c, 0x5d, 0xc1, 0x88, 0x18, 0xf4,
			0xe8, 0x88, 0xfe, 0x04, 0x50, 0xf7, 0xea, 0xb1,
			0x3e, 0x32, 0x95, 0x89, 0xc7, 0xb2, 0x60, 0xa9,
			0x97, 0x07, 0x11, 0xdd, 0xc7, 0xfd, 0x99, 0x0a,
			0xb0, 0x61, 0x17, 0x94, 0x80, 0xc0, 0x54, 0xfa,
			0xd2, 0x43, 0x3e, 0x66, 0x00, 0x9a, 0xf9, 0xfe,
			0x5b, 0x33, 0xf2, 0xc0, 0x72, 0x17, 0xca, 0x03,
			0x1f, 0x39, 0x23, 0x5f, 0x5c, 0x16, 0x70, 0x88,
			0x1b, 0x51, 0xc5, 0xe3, 0x35, 0xb5, 0xc5, 0xf7,
			0x9f, 0x24, 0x83, 0x3c, 0x1e, 0x60, 0xe6, 0xde,
			0x77, 0x85, 0x02, 0xb0, 0xce, 0xda, 0x9c, 0x88,
			0x2d, 0x83, 0x5b, 0xb2, 0xcc, 0x40, 0x63, 0x93,
			0x03, 0xd8, 0xbe, 0x61, 0x29, 0x4b, 0xf6, 0x75,
			0x22, 0x11, 0x01, 0x7e, 0x9c, 0x26, 0xfa, 0x6d,
			0xef, 0x5b, 0x95, 0x29, 0xa3, 0x1b, 0xcb, 0x34,
			0x73, 0xab, 0x3f, 0x48, 0xbc, 0x86, 0xb5, 0x36,
			0x1c, 0x79, 0x87, 0x3b, 0xc7, 0x17, 0x3f, 0x2b,
			0xcd, 0x72, 0x6a, 0x0b, 0xab, 0xcd, 0x5a, 0x92,
			0xdc, 0xfb, 0x4c, 0xaa, 0xcb, 0x9a, 0x9a, 0x5f,
			0x07, 0x07, 0xa2, 0x8b, 0x7e, 0xa8, 0x05, 0x46,
			0xf5, 0x29, 0x69, 0x01, 0x15, 0x64, 0x2b, 0xd8,
			0xe6, 0x0a, 0xfd, 0xd6, 0x52, 0xf1, 0x45, 0xf5,
			0x1c, 0xe1, 0x4b, 0xd0, 0x05, 0xde, 0x28, 0x13,
			0xd1, 0x58, 0x48, 0x50, 0xc6, 0xad, 0xe0, 0x75,
			0x28, 0x41, 0x56, 0xec, 0xc3, 0x01, 0x54, 0x1d,
			0x38, 0xaa, 0xd9, 0x30, 0x3c, 0xf9, 0x27, 0x5f,
			0x3b, 0x9e, 0xfb, 0x93, 0x73, 0x61, 0x50, 0x04,
			0x1b, 0x8c, 0x72, 0x01, 0x46, 0x4a, 0xf1, 0x4c,
			0x94, 0x88, 0xa2, 0xc1, 0x57, 0xa9, 0x52, 0xc8,
			0x58, 0xa4, 0x51, 0x28, 0x62, 0xe1, 0x06, 0x40,
			0x4f, 0xe3, 0xae, 0xf9, 0xe6, 0x1f, 0x11, 0x89,
			0x42, 0xa6, 0xf3, 0x7e, 0x1b, 0xf1, 0x19, 0xbc,
			0xe0, 0xcf, 0x36, 0x4d, 0xc9, 0x9b, 0x51, 0x30,
			0x17, 0xd2, 0x11, 0x1b, 0xaf, 0x77, 0xa4, 0x76,
		},
	},
	/* SHA256 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0x10, 0x13, 0x04, 0x71, 0xdc, 0x38, 0x3f, 0x0a,
			0x78, 0x71, 0xac, 0xb6, 0xd1, 0x87, 0x43, 0xa4,
			0x30, 0x31, 0x2f, 0x61, 0xbc, 0x89, 0x74, 0xed,
			0x41, 0xbd, 0xd5, 0x14, 0x2f, 0x07, 0xf5, 0xfb,
			0xe6, 0xdf, 0xbc, 0xfa, 0x30, 0x6f, 0xfc, 0xa9,
			0x0f, 0x47, 0x55, 0x3e, 0xba, 0x52, 0x56, 0x16,
			0xe2, 0x08, 0xe4, 0x91, 0x39, 0xe6, 0xe0, 0x23,
			0x35, 0x1b, 0x04, 0xa8, 0x11, 0x85, 0x0e, 0xf8,
			0x16, 0xdb, 0x21, 0xbc, 0x18, 0x2c, 0x68, 0x80,
			0x43, 0x88, 0xfc, 0x09, 0x18, 0xfd, 0x08, 0xe8,
			0xb2, 0xc5, 0x2b, 0x4f, 0x4e, 0xbd, 0x1d, 0x24,
			0x0a, 0x19, 0x96, 0x54, 0xed, 0x56, 0x6b, 0x6a,
			0x28, 0x71, 0xfa, 0x60, 0x9b, 0x57, 0xc9, 0xb4,
			0x9b, 0x9c, 0x6a, 0xbf, 0xf3, 0x6f, 0xa2, 0x43,
			0x28, 0xce, 0xca, 0x43, 0x02, 0xf6, 0x74, 0x17,
			0x47, 0x1e, 0x52, 0xbf, 0xca, 0x64, 0x54, 0x78,
		},
		/* S8 : 512 byte */
		{
			0x63, 0x6d, 0x07, 0xa3, 0x2c, 0x64, 0xf2, 0xe3,
			0xd3, 0xd2, 0xe2, 0x37, 0x00, 0x20, 0x6c, 0xc4,
			0xc0, 0xe8, 0x82, 0xc4, 0x15, 0x44, 0x11, 0xf2,
			0xaa, 0x54, 0x02, 0x9f, 0xd3, 0x3f, 0xf7, 0xbc,
			0x06, 0x0d, 0xb6, 0x15, 0x12, 0xa6, 0x33, 0x87,
			0x36, 0xb6, 0xc0, 0x44, 0xa2, 0x00, 0xbc, 0xe7,
			0x9b, 0x70, 0xed, 0x56, 0x38, 0x11, 0xda, 0x93,
			0x78, 0x6d, 0xa5, 0xb8, 0x70, 0x83, 0xb6, 0xdf,
			0x70, 0x38, 0xe0, 0xce, 0x49, 0xb7, 0x91, 0xe3,
			0x5c, 0xc6, 0xfe, 0x36, 0xff, 0xc5, 0x17, 0xf4,
			0xfc, 0xe9, 0xce, 0x55, 0xa2, 0xf9, 0x26, 0xd6,
			0x6f, 0x81, 0x66, 0xe5, 0x6c, 0x55, 0x2f, 0xaf,
			0x48, 0x5f, 0x0d, 0x0c, 0x24, 0x16, 0x81, 0xf9,
			0x68, 0xd6, 0x49, 0xdf, 0x92, 0x6f, 0xa8, 0x3a,
			0xbd, 0x1f, 0xe0, 0x17, 0xe1, 0x9a, 0x47, 0x0e,
			0x56, 0x2c, 0xfd, 0xb7, 0x25, 0x51, 0xb5, 0xbf,
			0xdf, 0x94, 0xd3, 0xec, 0x07, 0xc1, 0xcd, 0x95,
			0x96, 0x3d, 0x43, 0x6a, 0x2b, 0x44, 0x40, 0x29,
			0x3d, 0x95, 0xcd, 0x34, 0x06, 0x18, 0xa1, 0xf5,
			0xe2, 0x6a, 0xbf, 0x85, 0x98, 0x0f, 0xf4, 0x6f,
			0x91, 0x12, 0xa4, 0x5f, 0xc0, 0x9b, 0xba, 0xc2,
			0x86, 0xde, 0x98, 0xed, 0x82, 0x06, 0xfd, 0x52,
			0x87, 0x53, 0x50, 0xb4, 0xf5, 0x7a, 0xe7, 0x64,
			0x95, 0x2b, 0xcd, 0xb0, 0xbc, 0x22, 0xe4, 0x26,
			0x3f, 0x17, 0x3a, 0x58, 0xbe, 0xd2, 0xbf, 0x5e,
			0xda, 0xb1, 0x2a, 0x80, 0x96, 0x9b, 0x7f, 0x4d,
			0x0d, 0x32, 0x23, 0xc9, 0x79, 0x9e, 0xcc, 0xe4,
			0xd8, 0x05, 0x2a, 0x00, 0x8f, 0x48, 0x3c, 0x59,
			0xf0, 0x1e, 0xc2, 0x8b, 0x34, 0xa9, 0xf8, 0x86,
			0x43, 0x75, 0xa9, 0xd9, 0xab, 0x7d, 0x19, 0x39,
			0xc1, 0xf9, 0xba, 0xad, 0xce, 0xac, 0x17, 0x13,
			0x6c, 0x3d, 0x54, 0xed, 0xb3, 0x16, 0xa0, 0x2d,
			0x7c, 0x86, 0x02, 0xe6, 0x9f, 0xc3, 0x53, 0x37,
			0xa2, 0x49, 0x26, 0xd9, 0xcd, 0x3a, 0x63, 0xb5,
			0xb4, 0x60, 0xc6, 0x53, 0xc6, 0x2c, 0x80, 0x3a,
			0x68, 0xd0, 0xb7, 0x24, 0x4c, 0x8b, 0x54, 0x2c,
			0x75, 0x10, 0x20, 0x3b, 0x79, 0x4b, 0xa8, 0xb1,
			0x47, 0x8a, 0xad, 0x29, 0x4c, 0x22, 0xc1, 0xcb,
			0x82, 0xdb, 0xc2, 0xe1, 0xc4, 0x6d, 0xcc, 0xf1,
			0xaf, 0x4c, 0x02, 0x4e, 0xc9, 0x4b, 0x39, 0x5c,
			0x02, 0xd2, 0xd3, 0x71, 0x38, 0x6a, 0xe8, 0x7e,
			0x9d, 0x5c, 0xb0, 0xa3, 0xef, 0x29, 0xaa, 0x66,
			0x03, 0x62, 0x16, 0xe0, 0x7c, 0x57, 0x1e, 0xb1,
			0xf1, 0xed, 0x5f, 0x18, 0xa1, 0xa4, 0x30, 0x0e,
			0x9e, 0xf7, 0x16, 0x68, 0x77, 0x16, 0x3e, 0xb4,
			0x9b, 0x68, 0x0b, 0xe0, 0x06, 0x6d, 0xda, 0xee,
			0xc3, 0x47, 0x7c, 0x74, 0x62, 0xc1, 0x95, 0xc7,
			0x70, 0x7e, 0x74, 0x5f, 0x9f, 0x51, 0x13, 0x41,
			0x85, 0x1e, 0x81, 0x2a, 0x76, 0x00, 0x99, 0xa8,
			0x85, 0x90, 0xfd, 0x09, 0x22, 0x73, 0x90, 0x18,
			0xb9, 0xc0, 0xcf, 0xbf, 0x41, 0x39, 0x12, 0xc7,
			0x33, 0x7f, 0xc5, 0xa5, 0xe6, 0x17, 0xb5, 0xae,
			0xc5, 0xeb, 0x2b, 0xfc, 0xe5, 0x6b, 0x6c, 0x04,
			0x0d, 0x33, 0x7d, 0x52, 0xaf, 0x7e, 0x0c, 0x2c,
			0x34, 0xed, 0x02, 0x0e, 0x42, 0xc5, 0xa2, 0xca,
			0x80, 0x2a, 0x51, 0xf0, 0x4e, 0xc6, 0x84, 0x82,
			0x66, 0x86, 0x91, 0x1b, 0x0f, 0x81, 0x58, 0x05,
			0x20, 0xe3, 0x97, 0xca, 0xb9, 0xc5, 0x24, 0xd4,
			0xa6, 0x50, 0xbf, 0xb1, 0x21, 0x3d, 0xe0, 0xa4,
			0x25, 0xdf, 0xbb, 0xb2, 0x7c, 0x82, 0x4a, 0x91,
			0x9f, 0x9a, 0xce, 0x8e, 0x70, 0x4c, 0x57, 0x43,
			0xaf, 0xc7, 0xf0, 0xc0, 0x63, 0xb0, 0x40, 0x7a,
			0x9c, 0x78, 0xa2, 0xab, 0xd4, 0x13, 0xad, 0xac,
			0x30, 0x34, 0xd3, 0x43, 0x96, 0xb2, 0xda, 0x53,
		},
	},
	/* SHA256 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0xee, 0xc0, 0x8f, 0x01, 0xfb, 0x4c, 0x10, 0x4d,
			0xe9, 0x53, 0xca, 0xf6, 0xae, 0x00, 0x0a, 0x9f,
			0xb4, 0x2a, 0x1b, 0xfa, 0x36, 0x29, 0xc8, 0x39,
			0xee, 0xe8, 0x1c, 0xc5, 0xed, 0x3c, 0x6c, 0x33,
			0xca, 0x74, 0x98, 0x74, 0xfe, 0x32, 0x8d, 0x4b,
			0x8f, 0x79, 0x0e, 0x04, 0xec, 0x06, 0xb5, 0x64,
			0x18, 0xe4, 0x34, 0xdc, 0x0b, 0x3b, 0xe1, 0x01,
			0x56, 0x69, 0x72, 0x41, 0x05, 0x53, 0x04, 0xce,
			0xc9, 0xfd, 0x63, 0xdf, 0x57, 0x59, 0x85, 0x18,
			0x0b, 0x5d, 0x57, 0x46, 0xd1, 0x58, 0xbd, 0x50,
			0x45, 0x86, 0xd3, 0x6b, 0x19, 0xe2, 0x7a, 0x8c,
			0xb7, 0x3b, 0x10, 0x23, 0x28, 0x8a, 0x7e, 0x06,
			0x35, 0x83, 0x78, 0xab, 0xcc, 0x6a, 0x00, 0x9b,
			0xa5, 0x28, 0x2c, 0x23, 0x24, 0x9e, 0x2a, 0x4d,
			0x82, 0x39, 0x81, 0x0f, 0x28, 0xc7, 0x99, 0xc1,
			0x5b, 0x89, 0x7e, 0xd4, 0x1e, 0x89, 0xe1, 0xc2,
		},
		/* S9 : 512 byte */
		{
			0x94, 0x6c, 0xdf, 0xec, 0xd2, 0x7b, 0xb3, 0xb5,
			0x44, 0x7b, 0x66, 0x08, 0x9f, 0xb0, 0x8a, 0x17,
			0xe8, 0x40, 0x46, 0x0f, 0xb3, 0x4d, 0xfb, 0x55,
			0x27, 0x19, 0x08, 0xf8, 0x11, 0x20, 0x91, 0x93,
			0xfc, 0xbe, 0x27, 0x7f, 0x36, 0x0b, 0x0c, 0x7f,
			0x2a, 0x43, 0x0a, 0xde, 0xf9, 0x3f, 0x1b, 0x30,
			0xbd, 0x11, 0xed, 0x2c, 0xca, 0x35, 0xda, 0x0c,
			0x14, 0x44, 0x80, 0x4e, 0xb9, 0xef, 0x23, 0xb5,
			0x75, 0x3a, 0x52, 0x17, 0x1c, 0xb3, 0xab, 0x05,
			0xe0, 0x0d, 0xd8, 0x53, 0xc1, 0x80, 0x36, 0xe5,
			0xdc, 0x26, 0x8f, 0x52, 0xed, 0xb0, 0x3d, 0xe4,
			0x9a, 0xb9, 0x45, 0x2b, 0x7c, 0x99, 0xf6, 0xb3,
			0xdb, 0xb0, 0x4b, 0x0e, 0x32, 0x92, 0xff, 0x73,
			0x9e, 0x53, 0x98, 0x85, 0xa2, 0x05, 0x35, 0x23,
			0xb8, 0x3e, 0x93, 0x32, 0x6e, 0x46, 0x7c, 0x76,
			0x2c, 0x26, 0xfe, 0x4c, 0x44, 0xd6, 0xea, 0x9c,
			0x09, 0x2e, 0x75, 0x13, 0x08, 0x69, 0xcf, 0xa1,
			0x2f, 0x19, 0xcf, 0x8d, 0xd8, 0x17, 0x73, 0x59,
			0xc6, 0xca, 0xa6, 0x63, 0xf3, 0x3c, 0x43, 0xe4,
			0x4c, 0x58, 0xae, 0x0a, 0x1f, 0xb7, 0xc8, 0xfd,
			0xfa, 0x7e, 0x58, 0xeb, 0xe2, 0xc7, 0x80, 0x74,
			0x05, 0x02, 0x2a, 0x9a, 0x1b, 0x74, 0x72, 0xfd,
			0xa8, 0x64, 0x91, 0xe4, 0xf5, 0xfd, 0xcc, 0x87,
			0xd7, 0x1b, 0xf2, 0xb8, 0xcd, 0xcb, 0x6d, 0xf1,
			0x83, 0xbf, 0x71, 0x37, 0x1d, 0xf5, 0xb3, 0x48,
			0x9f, 0xba, 0xe3, 0x8d, 0x6a, 0x69, 0x7a, 0x91,
			0xdd, 0x54, 0x25, 0xd8, 0x85, 0x55, 0x35, 0x3c,
			0xa6, 0x53, 0x07, 0xeb, 0xeb, 0xf8, 0x46, 0x4d,
			0x9d, 0xcd, 0x32, 0xfd, 0xf8, 0x8e, 0xb2, 0x95,
			0x31, 0xd5, 0xa5, 0x68, 0xed, 0xea, 0x17, 0x92,
			0xda, 0x2d, 0x17, 0x15, 0x09, 0x54, 0xbd, 0xea,
			0xbf, 0xb3, 0xd0, 0xfe, 0x8f, 0x3b, 0x9b, 0xb7,
			0xca, 0xfc, 0x48, 0xd4, 0x43, 0x15, 0x26, 0xac,
			0x6e, 0x7d, 0x0b, 0xed, 0x95, 0x8f, 0x2d, 0x2e,
			0xf4, 0x3a, 0x70, 0x9b, 0xec, 0x97, 0x58, 0x39,
			0x8e, 0xb5, 0x3a, 0x37, 0x1f, 0x30, 0x0a, 0x45,
			0xb1, 0x47, 0x68, 0x0e, 0xb4, 0x92, 0x38, 0x3f,
			0x44, 0x25, 0x2d, 0xdc, 0xb6, 0x26, 0x91, 0xb1,
			0x59, 0x53, 0x2b, 0x38, 0x08, 0xc9, 0x0c, 0x09,
			0x4c, 0x69, 0xa9, 0xf7, 0x13, 0x0f, 0x4b, 0xa3,
			0x5e, 0x8a, 0x7c, 0xfb, 0x37, 0x6b, 0x2a, 0x46,
			0x01, 0x2f, 0x32, 0xb6, 0x43, 0xa3, 0xbf, 0xb6,
			0x2a, 0x69, 0x6e, 0xc9, 0x55, 0x18, 0x3b, 0x4f,
			0x85, 0x7d, 0xfd, 0xbd, 0xe8, 0x1e, 0xc1, 0x9c,
			0x0c, 0x9e, 0xbd, 0x86, 0x59, 0x62, 0x96, 0x37,
			0x22, 0x0b, 0x03, 0xe1, 0x08, 0xb8, 0xc4, 0xef,
			0xe7, 0x71, 0xf0, 0xfc, 0x86, 0x25, 0x59, 0x0d,
			0x59, 0x10, 0x10, 0x49, 0xc1, 0x76, 0xdb, 0xa6,
			0x20, 0x32, 0xf2, 0x51, 0x15, 0xe0, 0x55, 0x7d,
			0x84, 0xd9, 0x4b, 0x34, 0x3a, 0x00, 0x8e, 0x7d,
			0x42, 0x11, 0xad, 0x23, 0xfb, 0xd5, 0xd0, 0x7e,
			0xb3, 0xf5, 0x8f, 0x09, 0xee, 0x3f, 0x46, 0x05,
			0xfd, 0xbc, 0x12, 0x46, 0x77, 0xc5, 0x3f, 0x4f,
			0x3c, 0x93, 0xfa, 0xa8, 0x01, 0x13, 0x9a, 0xa8,
			0x1a, 0x12, 0x2e, 0x2c, 0xdd, 0x35, 0x21, 0x5e,
			0xfc, 0x80, 0x46, 0xf4, 0x75, 0xa1, 0x1c, 0x51,
			0xa1, 0x24, 0xc1, 0xcc, 0xe6, 0x80, 0xfb, 0xf7,
			0xd4, 0x79, 0xcc, 0xb5, 0x23, 0x66, 0x09, 0xbd,
			0xbc, 0x45, 0xec, 0xea, 0x5d, 0xc1, 0x8c, 0x28,
			0x5d, 0x18, 0x1a, 0x68, 0x66, 0x55, 0xe3, 0x2d,
			0xd4, 0x77, 0x93, 0xd8, 0x4d, 0x76, 0x45, 0xd8,
			0x9f, 0x05, 0x4e, 0x1c, 0xd8, 0xc1, 0xb0, 0x65,
			0x04, 0x91, 0x8d, 0xbd, 0xd4, 0x69, 0xd2, 0x9a,
			0x02, 0x22, 0xae, 0xe9, 0x2c, 0xf2, 0xd4, 0x4c,
		},
	},
};

/* 10 elements */
static mod4096_t SHA384[] = {
	/* SHA384 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0x72, 0x47, 0xdf, 0x01, 0x4b, 0x08, 0xf1, 0x09,
			0xb7, 0x78, 0xc8, 0x50, 0xbf, 0x07, 0x63, 0xe7,
			0x02, 0x6b, 0xf0, 0x56, 0xb5, 0x95, 0xed, 0x32,
			0xe8, 0xe3, 0x8d, 0xf2, 0x0b, 0x97, 0x5d, 0xfe,
			0xb0, 0x88, 0x27, 0x09, 0x93, 0x17, 0x16, 0xa1,
			0xd7, 0x65, 0x1c, 0x04, 0x92, 0xc3, 0x91, 0x36,
			0x8c, 0x6a, 0x1d, 0x50, 0x44, 0x6d, 0xd7, 0x1b,
			0x34, 0xeb, 0xae, 0xdd, 0x56, 0x89, 0x09, 0x73,
			0xe7, 0x1f, 0xaa, 0xa0, 0x6b, 0xb2, 0xd8, 0xa5,
			0xf1, 0xa3, 0xbd, 0x12, 0x98, 0x25, 0x0d, 0xdb,
			0x52, 0xf4, 0xe8, 0xae, 0xe8, 0x45, 0x02, 0x84,
			0x3f, 0xf8, 0x01, 0x77, 0xd9, 0x15, 0x26, 0xd3,
			0x9e, 0x76, 0x30, 0x6f, 0xdd, 0xc1, 0x7d, 0x3e,
			0x8e, 0x9a, 0x73, 0x21, 0xda, 0x16, 0x1e, 0xff,
			0xdd, 0x72, 0x1a, 0x18, 0xaa, 0x05, 0xb4, 0x96,
			0x8f, 0x73, 0x4c, 0x65, 0x9b, 0x24, 0xfe, 0x45,
		},
		/* S0 : 512 byte */
		{
			0x7f, 0x67, 0x26, 0xe4, 0xed, 0xfb, 0xc0, 0x85,
			0x33, 0x2e, 0x71, 0x71, 0xc8, 0x79, 0x95, 0x4e,
			0x35, 0xa3, 0x4c, 0x22, 0x3f, 0xbb, 0xda, 0x2c,
			0xab, 0x55, 0xbd, 0x33, 0xf8, 0xf5, 0x84, 0xe0,
			0xee, 0x72, 0x2d, 0x29, 0x4a, 0x79, 0xa3, 0x34,
			0xf3, 0x39, 0x56, 0x11, 0x0f, 0x04, 0xd8, 0x67,
			0x4e, 0x9a, 0x1f, 0x02, 0xbf, 0xd8, 0x78, 0xee,
			0x03, 0x83, 0x38, 0x1c, 0x8c, 0xd8, 0x99, 0x7b,
			0xb8, 0x3b, 0x60, 0x4c, 0x73, 0x72, 0x20, 0xae,
			0x56, 0xe7, 0x87, 0x5a, 0x3e, 0x65, 0xdc, 0xa7,
			0x7a, 0xd4, 0x75, 0x09, 0xb5, 0x6a, 0x00, 0xa8,
			0x40, 0x58, 0x23, 0x97, 0x21, 0xd7, 0xca, 0x67,
			0xeb, 0x03, 0xd2, 0xfb, 0x53, 0x8b, 0xc3, 0x74,
			0x56, 0xa5, 0x85, 0xc7, 0x49, 0x1f, 0xf6, 0x7d,
			0xfc, 0x4b, 0xed, 0x1a, 0xc0, 0x67, 0xef, 0xb6,
			0x98, 0x42, 0x46, 0xd6, 0x98, 0x37, 0xc2, 0xb7,
			0xa3, 0x4d, 0xee, 0x61, 0x25, 0xfb, 0x41, 0x71,
			0xb5, 0x73, 0x2a, 0x23, 0xa9, 0x83, 0x32, 0x51,
			0x88, 0x42, 0x80, 0xa3, 0xdf, 0x5e, 0x21, 0x9e,
			0x6f, 0x84, 0x16, 0x9d, 0x4b, 0x0b, 0x16, 0x23,
			0x6e, 0xbb, 0xcd, 0x68, 0xd3, 0xa9, 0x4a, 0xb7,
			0x37, 0x87, 0x51, 0x03, 0x0a, 0x2f, 0x06, 0x47,
			0xdd, 0x61, 0xe0, 0xb2, 0x6f, 0x07, 0x5a, 0x51,
			0x26, 0xb9, 0x2b, 0x60, 0x29, 0xb8, 0x75, 0x75,
			0x96, 0x01, 0x8e, 0x6f, 0x39, 0x0f, 0xd9, 0xde,
			0x03, 0x71, 0xa1, 0x1c, 0x0d, 0x7e, 0xc3, 0xa6,
			0x2e, 0x2c, 0xf2, 0x7f, 0xb0, 0x20, 0xc1, 0xad,
			0x98, 0x74, 0xd7, 0x26, 0x7d, 0xeb, 0xd9, 0x53,
			0xb9, 0x0d, 0x75, 0xc0, 0xbc, 0x73, 0x96, 0x46,
			0x53, 0xf8, 0xcc, 0xfb, 0x3d, 0x80, 0xe9, 0x51,
			0xe6, 0x48, 0x4d, 0x4e, 0xcd, 0x61, 0xa6, 0xa8,
			0x88, 0x5f, 0x4f, 0xea, 0x13, 0x0c, 0xff, 0xc1,
			0x2d, 0xd4, 0x05, 0x8c, 0x9a, 0x8e, 0x58, 0x60,
			0x0c, 0x5f, 0xef, 0x84, 0x58, 0xe5, 0xf1, 0x85,
			0x18, 0x6e, 0xcd, 0x08, 0x74, 0x34, 0x1b, 0x23,
			0xa1, 0xa0, 0xd2, 0x13, 0x8b, 0x48, 0x94, 0xff,
			0xd2, 0xe0, 0xe1, 0xaa, 0xc7, 0xc0, 0xa2, 0x30,
			0x23, 0xc9, 0xf6, 0x14, 0x42, 0x56, 0xfc, 0x88,
			0xd2, 0x02, 0x8e, 0xeb, 0x6e, 0x5e, 0xcc, 0x2e,
			0x67, 0x24, 0xc7, 0x89, 0xef, 0x59, 0x00, 0x82,
			0x76, 0xe2, 0x5f, 0xf2, 0x59, 0x1a, 0x4c, 0x7b,
			0x9d, 0x37, 0x9c, 0x1e, 0xe3, 0xeb, 0x45, 0x2d,
			0x8e, 0x76, 0xe9, 0xdf, 0x75, 0xbd, 0x83, 0x40,
			0x4d, 0x93, 0x34, 0x3f, 0x07, 0x77, 0xd6, 0x8a,
			0xc1, 0xb9, 0xa2, 0x9c, 0xd6, 0x00, 0x3e, 0xec,
			0xe0, 0x8c, 0xf2, 0x21, 0x3f, 0xef, 0x56, 0x69,
			0x33, 0x6a, 0x1c, 0xbb, 0x5e, 0x17, 0xaf, 0x34,
			0xe0, 0xbb, 0x22, 0x50, 0x97, 0x1b, 0x7d, 0x4f,
			0x84, 0x5c, 0x5a, 0x23, 0x23, 0x61, 0xab, 0xb6,
			0xac, 0xab, 0xce, 0x07, 0x03, 0x1d, 0x8e, 0xac,
			0xe9, 0x86, 0x1e, 0x8a, 0xe6, 0xaf, 0x3b, 0x7c,
			0xf6, 0xda, 0xd9, 0x04, 0xab, 0x65, 0xd2, 0x88,
			0xd2, 0x7b, 0x48, 0x28, 0x24, 0xcf, 0x92, 0x62,
			0xb1, 0x35, 0x27, 0xd3, 0x8e, 0x04, 0xf3, 0x02,
			0xec, 0x16, 0x55, 0x9f, 0xdf, 0x05, 0xe1, 0x98,
			0xe9, 0xb2, 0xfe, 0xa5, 0x7c, 0xd2, 0xbd, 0xf8,
			0xf6, 0x76, 0x7e, 0xc1, 0x38, 0xd6, 0xad, 0x3f,
			0xca, 0xb0, 0x88, 0x8d, 0x45, 0x61, 0xef, 0xe4,
			0x1c, 0x2a, 0x19, 0x49, 0xe8, 0x69, 0xa3, 0x7b,
			0x25, 0xb2, 0xe9, 0xc0, 0x30, 0x7d, 0x7e, 0x7d,
			0x4c, 0x82, 0x20, 0x53, 0xe8, 0x1d, 0x0a, 0x6a,
			0x92, 0x5d, 0x9c, 0xf8, 0xf1, 0xde, 0xe2, 0xd4,
			0xba, 0x78, 0x36, 0x8a, 0xbb, 0x4d, 0x8c, 0xa5,
			0xd3, 0x91, 0x64, 0xda, 0x55, 0xb4, 0x7f, 0x5a,
		},
	},
	/* SHA384 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0x3f, 0x1a, 0xd5, 0x6f, 0x30, 0x73, 0x85, 0x70,
			0x39, 0x71, 0x7e, 0xec, 0xd2, 0xbf, 0x90, 0x00,
			0xf5, 0x77, 0x3d, 0xb9, 0xcb, 0x38, 0x79, 0xcb,
			0x84, 0x79, 0xae, 0x86, 0x86, 0x1c, 0x03, 0x05,
			0xcb, 0x46, 0xde, 0xa9, 0x84, 0x6f, 0xa7, 0xc6,
			0x03, 0xb5, 0xd0, 0x9b, 0xd7, 0x76, 0x12, 0x88,
			0x39, 0x1d, 0xf9, 0x7b, 0xa7, 0x3c, 0xa2, 0xac,
			0xce, 0xdc, 0xc4, 0x8a, 0xad, 0x11, 0xed, 0xf4,
			0xf8, 0xd1, 0x0e, 0xad, 0xba, 0x05, 0x3a, 0x0a,
			0x3f, 0xe1, 0xab, 0xee, 0x31, 0x71, 0x06, 0xf8,
			0x01, 0x75, 0xdf, 0xfb, 0x88, 0x6f, 0x80, 0xaf,
			0xef, 0xfa, 0xe6, 0xa6, 0xcc, 0x5d, 0x0d, 0x81,
			0x8d, 0x60, 0x6e, 0x63, 0x1a, 0x5f, 0xc5, 0xa6,
			0xb6, 0x9c, 0x17, 0xcd, 0x26, 0xd9, 0xf4, 0xba,
			0x16, 0x27, 0xfa, 0x22, 0xb8, 0xf9, 0x9a, 0x3d,
			0xae, 0x7c, 0x1f, 0xc2, 0x2a, 0x2b, 0xeb, 0x12,
		},
		/* S1 : 512 byte */
		{
			0x1e, 0xe5, 0x00, 0x09, 0xe3, 0x2b, 0x7c, 0x44,
			0x25, 0x29, 0x99, 0x52, 0x25, 0x0a, 0x26, 0x62,
			0x2b, 0x39, 0xa4, 0xe6, 0xb0, 0x96, 0x45, 0x70,
			0x10, 0x25, 0xab, 0xaa, 0xc7, 0x27, 0x23, 0x8a,
			0x36, 0x13, 0x7f, 0x8c, 0x71, 0x9c, 0x62, 0xf5,
			0xe5, 0x4e, 0x26, 0xcb, 0x84, 0x53, 0x35, 0x99,
			0xd7, 0xac, 0xc8, 0x6e, 0xcc, 0x7d, 0xc0, 0xe7,
			0x30, 0xa0, 0xfc, 0x35, 0x23, 0x60, 0x70, 0x41,
			0xe2, 0x97, 0x50, 0x4a, 0xe2, 0x6a, 0x7b, 0x9b,
			0x84, 0x57, 0xb8, 0x4c, 0x6f, 0x47, 0xf1, 0xf2,
			0x07, 0xac, 0x20, 0x75, 0x09, 0x78, 0x4a, 0xd7,
			0x0a, 0x29, 0x61, 0xba, 0x17, 0x85, 0x45, 0x3c,
			0xa8, 0x62, 0x8c, 0xe9, 0x00, 0xfb, 0x26, 0xd3,
			0xc5, 0xe8, 0x1a, 0x62, 0x08, 0xd6, 0x94, 0x7e,
			0x79, 0xb1, 0xc0, 0x75, 0x08, 0xee, 0x13, 0x19,
			0x16, 0xaf, 0x33, 0xf0, 0x50, 0x53, 0xe4, 0xcb,
			0x7a, 0x51, 0xc2, 0xcd, 0xbb, 0xd6, 0xfd, 0x75,
			0xba, 0xda, 0x9f, 0x10, 0x05, 0xe5, 0x6f, 0xa8,
			0x5a, 0x41, 0xca, 0x78, 0x5d, 0xb7, 0x1e, 0x0b,
			0xa0, 0xee, 0xda, 0x40, 0xb5, 0x21, 0x27, 0x57,
			0xa4, 0x6b, 0x84, 0xba, 0x49, 0x7b, 0xc6, 0x62,
			0x14, 0x55, 0x64, 0x06, 0x59, 0xb5, 0xe9, 0xe2,
			0x51, 0xff, 0x02, 0x9a, 0xcf, 0xf1, 0x07, 0x88,
			0xb9, 0x38, 0x09, 0x14, 0x2f, 0x77, 0x21, 0x0c,
			0x0e, 0x6c, 0xbc, 0x51, 0xf7, 0x50, 0xdf, 0xae,
			0xde, 0x15, 0x1e, 0xcb, 0xc2, 0xcc, 0x43, 0xa0,
			0x00, 0xff, 0x4e, 0x70, 0x06, 0xb6, 0x77, 0xa9,
			0x1d, 0x41, 0x38, 0xba, 0x01, 0xac, 0xb2, 0x3e,
			0xd1, 0x86, 0xf6, 0xc0, 0x47, 0xb2, 0x68, 0x85,
			0x43, 0x06, 0xb5, 0xbc, 0x7a, 0x42, 0x87, 0x87,
			0x93, 0xbe, 0x83, 0x00, 0x60, 0xe1, 0xf5, 0x85,
			0x32, 0x9e, 0xd0, 0x0f, 0x1d, 0xdf, 0xe6, 0x9a,
			0x56, 0xb9, 0x08, 0xe7, 0x1a, 0x24, 0xb8, 0xa3,
			0xc4, 0x08, 0x5e, 0xcb, 0xcb, 0x21, 0xc0, 0x03,
			0x97, 0x99, 0x26, 0x4f, 0x4d, 0x7a, 0xac, 0x0b,
			0x14, 0x32, 0xa0, 0x24, 0x76, 0xd7, 0x64, 0x8b,
			0x2a, 0x10, 0x9f, 0xd6, 0x34, 0x24, 0xbf, 0xe3,
			0xfb, 0x0d, 0xa4, 0xa1, 0x9c, 0x30, 0x10, 0xbf,
			0xf5, 0x23, 0x0e, 0x28, 0x46, 0xcb, 0xd2, 0xd5,
			0x1f, 0x7f, 0x20, 0x66, 0x72, 0x84, 0x12, 0x52,
			0x1f, 0x68, 0x41, 0xf9, 0x72, 0x4d, 0x3e, 0x66,
			0xc5, 0xe9, 0x9a, 0x76, 0x7d, 0x9f, 0x1c, 0xd2,
			0xaa, 0x5c, 0xb6, 0xa6, 0xe8, 0xea, 0xe4, 0xb5,
			0x64, 0x4e, 0x2c, 0x16, 0x9a, 0x98, 0x23, 0x9e,
			0x8c, 0x6f, 0x4f, 0x62, 0xb8, 0x38, 0x47, 0x02,
			0x4d, 0xd6, 0x1f, 0x5a, 0x2a, 0x3f, 0x85, 0x6d,
			0x25, 0xd3, 0x5e, 0xe0, 0xa1, 0xba, 0xa3, 0x31,
			0x1c, 0x42, 0xa0, 0xf5, 0x01, 0xcf, 0x3d, 0xb2,
			0x61, 0xaf, 0x32, 0x27, 0xf4, 0x6e, 0x75, 0xa6,
			0xbf, 0x42, 0x3d, 0x6a, 0x57, 0xf8, 0x9b, 0x0e,
			0xbe, 0xc9, 0xc5, 0x7a, 0xe7, 0x17, 0xf0, 0x2a,
			0x85, 0x57, 0x8b, 0x61, 0x92, 0x75, 0xd1, 0x46,
			0x69, 0x32, 0x3c, 0xf1, 0x39, 0xca, 0x9e, 0x8c,
			0x37, 0xd6, 0x47, 0x42, 0x7f, 0xc9, 0xa9, 0x70,
			0xa5, 0xbc, 0x53, 0x3d, 0x0b, 0x19, 0x01, 0x8f,
			0x95, 0xd8, 0xc9, 0x18, 0x69, 0x45, 0xa8, 0x27,
			0xa8, 0x47, 0x0a, 0x2b, 0xbe, 0xd9, 0xc1, 0x7e,
			0x9f, 0x81, 0x41, 0xd6, 0x4d, 0xbd, 0x5e, 0x87,
			0x55, 0xff, 0xd7, 0x7f, 0x92, 0x87, 0xa1, 0x7a,
			0xa1, 0x38, 0xc9, 0xc8, 0xb7, 0x69, 0x7e, 0x95,
			0xe3, 0xba, 0x8d, 0x86, 0xcd, 0x27, 0xa3, 0xc9,
			0x89, 0xf0, 0x63, 0x21, 0xe4, 0x92, 0xd0, 0xe7,
			0xdb, 0xd1, 0xf9, 0x30, 0x8f, 0xab, 0x0b, 0xef,
			0xca, 0x15, 0x48, 0xbe, 0x50, 0xa2, 0x2b, 0x38,
		},
	},
	/* SHA384 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x6e, 0x5f, 0x71, 0xb6, 0xe9, 0xfb, 0xb5, 0xea,
			0x92, 0x74, 0x30, 0x04, 0xe2, 0xeb, 0xd3, 0x44,
			0xe8, 0xc6, 0xba, 0x63, 0x49, 0xa6, 0x54, 0x27,
			0x85, 0x99, 0xc5, 0x36, 0x7b, 0x92, 0x88, 0x28,
			0x30, 0x16, 0x59, 0xdf, 0xcc, 0x37, 0x52, 0x1b,
			0xe5, 0xc6, 0x54, 0x49, 0x00, 0xc5, 0xa8, 0x9c,
			0x55, 0x1b, 0xe6, 0x5f, 0xd3, 0x8b, 0x16, 0x8a,
			0x64, 0xe7, 0x16, 0x93, 0x72, 0x81, 0x3c, 0x86,
			0xaa, 0xe3, 0x3b, 0x58, 0x00, 0xc0, 0x4b, 0x79,
			0xf3, 0x2a, 0x84, 0xa8, 0x12, 0x04, 0x8b, 0x0b,
			0xbf, 0xbb, 0x71, 0x7f, 0x33, 0x31, 0xd8, 0x2d,
			0xc2, 0xfa, 0x58, 0x5e, 0x60, 0xcb, 0x21, 0x90,
			0x65, 0x2f, 0xe1, 0xc8, 0x8d, 0x7c, 0xe8, 0x2b,
			0x43, 0x06, 0x8a, 0xc8, 0x1f, 0x92, 0xc4, 0xb9,
			0xad, 0x0e, 0x23, 0x6a, 0x70, 0x7f, 0xe2, 0x38,
			0x26, 0x3a, 0x52, 0x3e, 0x4e, 0x57, 0x7f, 0x6b,
		},
		/* S2 : 512 byte */
		{
			0x47, 0x23, 0x6d, 0x74, 0x2e, 0x53, 0x05, 0xcc,
			0xa3, 0x1b, 0xa9, 0x4b, 0x07, 0xcd, 0x8f, 0x67,
			0x8f, 0xf6, 0xba, 0x40, 0xcf, 0xb6, 0x6d, 0x31,
			0xe5, 0xe9, 0x30, 0x5f, 0x30, 0x9f, 0x4a, 0x59,
			0x72, 0xb7, 0x0b, 0x96, 0x69, 0x4b, 0x5b, 0x65,
			0x2a, 0xfb, 0xab, 0xe8, 0x98, 0xef, 0xde, 0xbf,
			0xc8, 0x75, 0x96, 0xea, 0x55, 0x69, 0x13, 0x4d,
			0x8a, 0x6b, 0xec, 0x92, 0x98, 0xe5, 0xd2, 0xcf,
			0x71, 0xa3, 0x0f, 0x57, 0xac, 0x3f, 0x1a, 0x21,
			0x55, 0xe1, 0x63, 0xbd, 0x32, 0x1f, 0xa6, 0x73,
			0x9a, 0x22, 0x3e, 0xd3, 0x00, 0xd9, 0x2b, 0x17,
			0x7f, 0x26, 0x60, 0x98, 0x06, 0x8c, 0xfe, 0x88,
			0xa2, 0x7d, 0xed, 0x8f, 0x41, 0x0b, 0xf9, 0xeb,
			0x62, 0x0b, 0x39, 0x8e, 0x23, 0x1c, 0x9b, 0x48,
			0xb6, 0xfe, 0x0f, 0xd7, 0x5b, 0xea, 0x2d, 0x80,
			0xb3, 0xe2, 0x3d, 0x54, 0x88, 0x2c, 0x2f, 0xea,
			0x63, 0xbe, 0xfc, 0x69, 0x9b, 0xeb, 0x40, 0x4e,
			0xdc, 0x29, 0xbd, 0x6d, 0x5a, 0x52, 0x34, 0x8a,
			0x39, 0xc9, 0x8b, 0x92, 0x7c, 0xb5, 0x4a, 0xfd,
			0xcc, 0xa9, 0x9a, 0x27, 0xc9, 0x20, 0x5b, 0xa3,
			0xf9, 0x25, 0x6e, 0x16, 0x78, 0xb2, 0xe6, 0x7d,
			0x3f, 0x64, 0xf3, 0x21, 0x83, 0x69, 0x47, 0xe9,
			0x09, 0x8b, 0x1f, 0x16, 0x69, 0x4b, 0x84, 0x0c,
			0x2d, 0xd7, 0xe0, 0x97, 0x36, 0x50, 0x20, 0x8a,
			0x43, 0x33, 0x45, 0x41, 0x00, 0x1d, 0x07, 0x2f,
			0x62, 0x0d, 0x14, 0xf3, 0xae, 0x76, 0xe2, 0xd6,
			0x85, 0x1b, 0x5b, 0xbc, 0x1b, 0x15, 0x16, 0x42,
			0xb0, 0x80, 0xb9, 0xb4, 0x13, 0xa3, 0xa7, 0xb7,
			0x51, 0x61, 0x7a, 0x2e, 0x70, 0xdd, 0x44, 0x81,
			0xea, 0x19, 0xc1, 0x28, 0x96, 0x97, 0x71, 0xc9,
			0x39, 0xed, 0xa2, 0xde, 0x30, 0x51, 0x17, 0x94,
			0x02, 0x8a, 0x48, 0x4a, 0x3b, 0x6b, 0x19, 0x3b,
			0xc0, 0x36, 0x01, 0xf3, 0x5a, 0x7a, 0x00, 0xd6,
			0x4f, 0x96, 0xf4, 0x7a, 0xb9, 0xa8, 0x4a, 0x43,
			0xf7, 0x70, 0x91, 0x90, 0x13, 0xb3, 0x95, 0xff,
			0xb5, 0x47, 0x61, 0xac, 0x64, 0xd4, 0x8e, 0xd8,
			0x16, 0x09, 0xd2, 0xd2, 0xba, 0xae, 0x7c, 0xbb,
			0x7e, 0xfd, 0x19, 0xc6, 0x7d, 0xbc, 0x59, 0x56,
			0xc4, 0x63, 0x6a, 0x7a, 0xa9, 0xe1, 0x21, 0xd1,
			0x52, 0x44, 0xce, 0x08, 0xfc, 0x4f, 0x73, 0xcc,
			0x5b, 0x8e, 0x67, 0x6b, 0xec, 0xa4, 0xa8, 0xf7,
			0xf8, 0x12, 0x36, 0xc0, 0x98, 0x44, 0x61, 0xf9,
			0x8b, 0xa3, 0xfe, 0x90, 0xe2, 0xbb, 0xf8, 0xf5,
			0xcf, 0x4c, 0x65, 0x34, 0x35, 0x8f, 0x8a, 0xea,
			0x6f, 0x46, 0x13, 0x35, 0xeb, 0x59, 0xc2, 0x01,
			0xf7, 0x28, 0x36, 0xe1, 0x27, 0xda, 0x43, 0x32,
			0xd0, 0x45, 0xb8, 0xe7, 0x15, 0xe0, 0x1f, 0xc8,
			0xd6, 0xe8, 0x87, 0x31, 0x15, 0xe3, 0xa7, 0x56,
			0x08, 0xa8, 0xc5, 0x0f, 0xba, 0x13, 0x19, 0x12,
			0x1e, 0xe9, 0x34, 0x17, 0x82, 0x6c, 0x3d, 0xf3,
			0xed, 0xea, 0xea, 0xbe, 0xdc, 0xb6, 0x2c, 0x22,
			0xb7, 0x57, 0xc7, 0x08, 0xf5, 0x02, 0x18, 0x63,
			0x67, 0xca, 0x80, 0xd0, 0x15, 0xd4, 0x29, 0x32,
			0xdd, 0xe2, 0xb8, 0x89, 0xe1, 0x35, 0x01, 0x2a,
			0x7a, 0x43, 0x07, 0xd7, 0x82, 0xc0, 0x6e, 0xe7,
			0x41, 0x4e, 0x0a, 0x61, 0x88, 0xfe, 0xec, 0x3f,
			0x61, 0x6b, 0x58, 0x64, 0x7d, 0x78, 0x35, 0x58,
			0xee, 0xae, 0x8f, 0x68, 0x51, 0xb4, 0x65, 0x5f,
			0x49, 0x38, 0x76, 0x26, 0xee, 0x6b, 0x96, 0x95,
			0x75, 0x9a, 0xce, 0x1a, 0xd8, 0x89, 0xb7, 0x53,
			0xef, 0xb3, 0x0c, 0x4f, 0x8a, 0xf0, 0xa6, 0xcc,
			0x49, 0x2e, 0x02, 0xd1, 0x84, 0x3a, 0x3e, 0x09,
			0xce, 0x3a, 0xb6, 0x55, 0x4d, 0x97, 0x31, 0xe8,
			0xc1, 0xe5, 0xd4, 0x3a, 0xd3, 0xb7, 0x1e, 0x7d,
		},
	},
	/* SHA384 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0x41, 0x39, 0x73, 0x54, 0x37, 0x69, 0xee, 0xfd,
			0x27, 0x13, 0xc2, 0x4e, 0x10, 0x0c, 0xfd, 0x6c,
			0x8c, 0x56, 0x29, 0x05, 0xa4, 0xa2, 0xc6, 0xee,
			0xe8, 0x13, 0xf8, 0x0b, 0xc0, 0x45, 0x7c, 0x70,
			0xfd, 0x8e, 0xe0, 0xc5, 0xb3, 0x84, 0xf4, 0xef,
			0x4b, 0xbf, 0x7c, 0xed, 0xd4, 0xe0, 0x27, 0xff,
			0x9a, 0xd1, 0xc5, 0x0e, 0xfa, 0x4f, 0xb4, 0x48,
			0x74, 0xa6, 0x9b, 0x0b, 0x02, 0xbc, 0x5b, 0x04,
			0xa8, 0x51, 0x43, 0x98, 0x4d, 0x85, 0x84, 0x82,
			0xc9, 0x9b, 0xe4, 0xbe, 0x3f, 0xfc, 0xb6, 0xa6,
			0xad, 0x7f, 0x09, 0x5c, 0xc1, 0xe7, 0x20, 0x67,
			0xef, 0xae, 0x22, 0xa1, 0xc0, 0x01, 0x13, 0x50,
			0x6d, 0x0b, 0x03, 0x93, 0xac, 0x76, 0x85, 0x00,
			0xca, 0x2f, 0x63, 0x8b, 0xfa, 0x6a, 0x90, 0xa9,
			0xed, 0xe7, 0x5e, 0xb6, 0xa2, 0x73, 0xf1, 0x94,
			0x80, 0xb1, 0xf4, 0x96, 0xa2, 0x1a, 0x8a, 0x9c,
		},
		/* S3 : 512 byte */
		{
			0x62, 0xb3, 0xd6, 0x24, 0xaa, 0xc8, 0x78, 0x16,
			0xb5, 0x95, 0xc6, 0xda, 0x93, 0x1a, 0x6c, 0xaa,
			0xeb, 0x9a, 0x32, 0x9a, 0x96, 0x00, 0x0c, 0x8c,
			0xaf, 0xaa, 0xb7, 0xbf, 0xbf, 0xc3, 0x74, 0x07,
			0x18, 0x69, 0x2b, 0xcc, 0xbf, 0xf9, 0x55, 0xa0,
			0x35, 0x3c, 0x6e, 0xac, 0xa1, 0x09, 0x3b, 0x85,
			0x24, 0xd5, 0xf3, 0xc5, 0x9e, 0x78, 0x4d, 0xa8,
			0xd5, 0x73, 0xdd, 0x15, 0x1c, 0xe4, 0xa9, 0xdd,
			0xf5, 0xf3, 0x27, 0x92, 0x5f, 0x4d, 0xc6, 0x4e,
			0x73, 0x69, 0x29, 0xd8, 0xcd, 0x41, 0xbd, 0x20,
			0xbf, 0xad, 0x57, 0xac, 0x31, 0xb8, 0x52, 0x71,
			0xb8, 0xc5, 0xa9, 0x6e, 0x9c, 0xeb, 0x02, 0xe0,
			0xa0, 0x09, 0xb2, 0x4a, 0x2f, 0x66, 0x69, 0x11,
			0x7b, 0xd8, 0x6a, 0x55, 0x2f, 0x4d, 0xd2, 0x45,
			0x61, 0xa7, 0xf6, 0x2d, 0x89, 0x4e, 0xd9, 0x6b,
			0x48, 0x44, 0xb9, 0x33, 0xf4, 0xe0, 0x2f, 0x76,
			0xe1, 0xa6, 0xd9, 0x8b, 0x6f, 0x1d, 0x5f, 0x3d,
			0xba, 0x23, 0x1e, 0xcd, 0x7c, 0xf0, 0xac, 0x09,
			0x4d, 0x6b, 0x34, 0x28, 0x86, 0x93, 0x11, 0x77,
			0xd7, 0x6a, 0xff, 0x99, 0x00, 0x36, 0xec, 0x7d,
			0xde, 0x98, 0x8c, 0x1b, 0x20, 0xf4, 0x71, 0x23,
			0x3c, 0x9f, 0xb0, 0xa3, 0x8f, 0x41, 0x7c, 0xdd,
			0x83, 0x95, 0xc9, 0x03, 0x70, 0x64, 0x09, 0xca,
			0x96, 0xe2, 0x15, 0xc4, 0xfb, 0x22, 0x90, 0x53,
			0xc8, 0x6e, 0x24, 0x57, 0xa6, 0xc0, 0x72, 0xa1,
			0x56, 0x49, 0xac, 0x1c, 0x1e, 0xa6, 0x07, 0x5a,
			0x4b, 0xcd, 0x4d, 0xae, 0x3a, 0x1b, 0x23, 0x8a,
			0xe3, 0xd3, 0x3c, 0xf1, 0x78, 0x79, 0x2c, 0x3d,
			0x6b, 0xd5, 0x6e, 0xda, 0x2a, 0x6e, 0xc4, 0xab,
			0x71, 0x30, 0xce, 0x15, 0xe8, 0x03, 0x99, 0xf7,
			0x7b, 0xb7, 0x65, 0x71, 0x9c, 0x1d, 0xcd, 0x50,
			0xe4, 0x28, 0x2f, 0x84, 0x8b, 0xeb, 0x36, 0x59,
			0xa7, 0x76, 0x33, 0xad, 0x10, 0xbf, 0x27, 0x94,
			0x56, 0xbc, 0xde, 0x1c, 0x04, 0x0a, 0x00, 0x35,
			0x24, 0x43, 0x5b, 0xa8, 0x6a, 0x6e, 0xb3, 0xca,
			0x40, 0xcd, 0x1a, 0xd1, 0xfa, 0x71, 0x67, 0x19,
			0x3c, 0x58, 0x4c, 0x14, 0x87, 0xb6, 0x7e, 0xe9,
			0xed, 0x3f, 0x6f, 0x6c, 0x4a, 0xca, 0x56, 0x1f,
			0xa1, 0x32, 0x8c, 0x21, 0xe5, 0xc1, 0x77, 0xfc,
			0x98, 0x77, 0x40, 0x1f, 0xc7, 0x62, 0xc5, 0x2f,
			0xe4, 0x48, 0xaa, 0x06, 0xb4, 0xbc, 0x5e, 0xa6,
			0x63, 0x11, 0x40, 0xf9, 0x54, 0x3b, 0x6b, 0xc4,
			0x5b, 0x25, 0xe4, 0x19, 0x6a, 0xf5, 0xe4, 0x58,
			0x29, 0x25, 0x52, 0x17, 0xae, 0x53, 0x4b, 0xd3,
			0x6e, 0xf5, 0x2c, 0xe0, 0x9b, 0xce, 0xd4, 0x92,
			0x37, 0x68, 0xf1, 0x3c, 0x41, 0x5c, 0x3e, 0xea,
			0x41, 0xab, 0x76, 0x7e, 0x38, 0xec, 0x6c, 0x65,
			0xf4, 0x82, 0x08, 0x3d, 0x5c, 0x30, 0xdc, 0x45,
			0x33, 0x35, 0x99, 0xcf, 0x4b, 0x33, 0x2c, 0x67,
			0x5e, 0x3c, 0x5e, 0x0f, 0x16, 0xda, 0x05, 0x9c,
			0x19, 0x5b, 0xff, 0xa5, 0xf8, 0x7e, 0x1b, 0xcc,
			0xfe, 0xac, 0x1a, 0x7c, 0x58, 0x94, 0x4a, 0x72,
			0xe4, 0x1e, 0x0b, 0x1b, 0xa2, 0xb9, 0x78, 0xcb,
			0x25, 0x6d, 0x18, 0x7e, 0x90, 0x43, 0x5a, 0x57,
			0x07, 0x84, 0xbc, 0xc7, 0x68, 0x0c, 0x0f, 0x79,
			0x1d, 0x4a, 0xdf, 0x41, 0xef, 0xf3, 0xa8, 0x7a,
			0x05, 0xaa, 0x61, 0x23, 0x8d, 0x0a, 0xd4, 0xab,
			0xf7, 0x10, 0xe3, 0xfa, 0xa6, 0x1a, 0xc7, 0x69,
			0x93, 0xf7, 0x7e, 0x52, 0x5f, 0x14, 0x11, 0x2b,
			0xf0, 0xdb, 0x50, 0x8b, 0x55, 0xf0, 0xe8, 0xc8,
			0xde, 0x0d, 0x3c, 0x60, 0xd5, 0x10, 0xdb, 0xeb,
			0xce, 0x13, 0x05, 0x3f, 0x4e, 0xa2, 0x07, 0xc3,
			0xef, 0x40, 0x7a, 0xd4, 0xe3, 0x53, 0xcb, 0xce,
			0x07, 0xc7, 0x3f, 0x93, 0x2f, 0xd0, 0x69, 0x3a,
		},
	},
	/* SHA384 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0xcb, 0x72, 0xfa, 0x19, 0x08, 0x00, 0xbb, 0xd4,
			0x1a, 0x49, 0xac, 0xdd, 0xa9, 0xdc, 0x6f, 0x5e,
			0x08, 0x54, 0x87, 0xde, 0x76, 0x87, 0xbc, 0x13,
			0x9d, 0x00, 0xc0, 0xf7, 0x5f, 0xb0, 0x12, 0x1e,
			0xd2, 0xe0, 0x23, 0x46, 0x89, 0x6a, 0xb8, 0x30,
			0x16, 0x29, 0x8f, 0xae, 0xcc, 0xae, 0x45, 0xfe,
			0xca, 0x34, 0x77, 0x56, 0x35, 0xd6, 0x28, 0xc0,
			0xc4, 0x00, 0x62, 0x28, 0x82, 0x24, 0x20, 0xb3,
			0xcf, 0xae, 0x6c, 0x53, 0xa8, 0x39, 0xc5, 0x99,
			0x29, 0x03, 0xc3, 0xca, 0x50, 0x9e, 0xfe, 0x34,
			0x04, 0xe9, 0x2b, 0xc2, 0x56, 0x3f, 0x88, 0xcf,
			0x06, 0xee, 0x7b, 0x38, 0x49, 0xeb, 0x75, 0xb5,
			0xc9, 0xc4, 0x1d, 0x69, 0xee, 0xd5, 0xab, 0xb7,
			0x5c, 0xbf, 0x91, 0x58, 0xbd, 0x16, 0x61, 0xaa,
			0xc0, 0x5a, 0xeb, 0x4b, 0x61, 0x29, 0xa5, 0xe7,
			0x6c, 0xb2, 0x51, 0x4e, 0x3c, 0x6d, 0xd9, 0xca,
		},
		/* S4 : 512 byte */
		{
			0xb6, 0x1f, 0xcb, 0xd9, 0x06, 0x0a, 0xdb, 0x9b,
			0x48, 0x82, 0x35, 0x6a, 0x3a, 0x62, 0x48, 0x64,
			0x26, 0x4a, 0x21, 0x30, 0xbe, 0xba, 0xd9, 0x68,
			0x8f, 0xc2, 0xef, 0xed, 0x8e, 0x83, 0x83, 0x61,
			0x0b, 0x7e, 0x5c, 0xbe, 0x50, 0x91, 0xa6, 0x04,
			0x20, 0x0c, 0xb0, 0xd4, 0xbc, 0x90, 0xb3, 0xea,
			0x87, 0xaa, 0x5f, 0x17, 0x38, 0x93, 0x2a, 0xfa,
			0x68, 0xb8, 0x27, 0x70, 0xcc, 0xc2, 0x9f, 0x56,
			0xbb, 0x0e, 0x90, 0x58, 0x85, 0x82, 0x1d, 0xf1,
			0xf5, 0x5b, 0xba, 0x01, 0xf0, 0x3e, 0x49, 0xdd,
			0xb7, 0x76, 0xe9, 0x62, 0xe4, 0x39, 0x51, 0xb2,
			0xe4, 0xbf, 0xca, 0xa3, 0x3d, 0xfc, 0x8c, 0xfa,
			0x50, 0x53, 0x28, 0x2d, 0x60, 0xeb, 0xd2, 0x14,
			0x0a, 0x9e, 0x76, 0xa0, 0x97, 0xca, 0x64, 0x63,
			0x50, 0xfa, 0x41, 0x7b, 0xdb, 0x09, 0x98, 0x8d,
			0x8c, 0x18, 0xfb, 0x1a, 0xdd, 0x9b, 0x20, 0x4a,
			0xe6, 0x39, 0x9a, 0x78, 0x85, 0x03, 0x23, 0x35,
			0xee, 0xd4, 0x67, 0x69, 0xae, 0xe1, 0x0b, 0xf9,
			0x01, 0xe8, 0x1e, 0xe3, 0x6d, 0x96, 0xec, 0xdd,
			0x8c, 0xa6, 0x15, 0x61, 0xa1, 0xa2, 0x81, 0x73,
			0x17, 0x0a, 0xa2, 0x24, 0x62, 0xd5, 0x47, 0x78,
			0xc8, 0x00, 0xa4, 0x88, 0x35, 0xfb, 0x03, 0xd2,
			0x02, 0x9b, 0x79, 0x07, 0x15, 0xff, 0xcb, 0x1e,
			0x8a, 0x12, 0x67, 0x66, 0xe0, 0x32, 0x0a, 0xc1,
			0xe2, 0x9f, 0xe0, 0x33, 0x54, 0xa1, 0xe1, 0xe6,
			0x60, 0x5e, 0x2e, 0x32, 0xb2, 0x7d, 0x23, 0x87,
			0xd6, 0xeb, 0x35, 0xe1, 0x5b, 0x92, 0xfe, 0x86,
			0x81, 0x2b, 0x82, 0x9c, 0x24, 0x84, 0x3a, 0x57,
			0x78, 0x95, 0x18, 0x88, 0xf9, 0xb8, 0x2d, 0x62,
			0xf2, 0xe9, 0x80, 0xa8, 0xa7, 0x63, 0xe0, 0x76,
			0xdc, 0xa1, 0xef, 0xc0, 0x44, 0xaf, 0x51, 0xf1,
			0x84, 0xe8, 0xc7, 0xd4, 0xb0, 0xb2, 0xcc, 0xbb,
			0x76, 0x6e, 0xa5, 0x31, 0xd4, 0x14, 0xf7, 0xa0,
			0x1c, 0x27, 0xf7, 0x11, 0xf0, 0xba, 0x1c, 0xf8,
			0x65, 0xbb, 0x9d, 0x92, 0x3a, 0xda, 0x18, 0xee,
			0xc6, 0x8e, 0x25, 0xcd, 0x42, 0xa2, 0xfd, 0x1b,
			0x0e, 0x9a, 0x77, 0x6b, 0x5a, 0x3a, 0xc3, 0x78,
			0x05, 0x48, 0xb6, 0xc8, 0x57, 0x51, 0xc7, 0x70,
			0x4a, 0xd3, 0xdd, 0xdf, 0x74, 0xce, 0x0c, 0xfc,
			0xd5, 0x83, 0x90, 0x39, 0x23, 0xd5, 0xcf, 0x81,
			0x94, 0xb8, 0x5d, 0x98, 0x47, 0x7b, 0x88, 0x0b,
			0xbf, 0x8a, 0x12, 0xb4, 0xc9, 0x22, 0xb7, 0xbf,
			0xb9, 0x82, 0x44, 0xcb, 0x01, 0x64, 0x20, 0xf6,
			0xf6, 0x66, 0xa4, 0x06, 0x27, 0x8c, 0x69, 0x78,
			0x58, 0x69, 0xc0, 0xcd, 0x3f, 0xce, 0x37, 0xed,
			0xcb, 0xeb, 0x18, 0xb6, 0xe3, 0x2c, 0xdb, 0xb6,
			0x48, 0xbd, 0xe8, 0x3a, 0x5f, 0x34, 0x9d, 0x0a,
			0x61, 0xdf, 0x9a, 0xec, 0xf0, 0x79, 0x26, 0x05,
			0x6d, 0x9f, 0x58, 0x41, 0x8c, 0x77, 0xea, 0xcb,
			0xb5, 0x38, 0xb4, 0x72, 0x60, 0x1a, 0x03, 0xb7,
			0x6c, 0x72, 0xd5, 0xc8, 0xfb, 0x58, 0x8f, 0xf0,
			0xcc, 0xc1, 0xd9, 0xfd, 0x92, 0x6b, 0xd3, 0xf5,
			0x1e, 0x49, 0x89, 0xb1, 0xc1, 0x20, 0xd4, 0xf0,
			0xb6, 0x30, 0x04, 0x91, 0xc8, 0x1c, 0x5c, 0x10,
			0x66, 0x81, 0xc5, 0x71, 0x32, 0x9d, 0x0b, 0x6c,
			0xc5, 0x19, 0x0b, 0x35, 0x70, 0x0f, 0x39, 0x37,
			0xd2, 0x54, 0x17, 0x52, 0xaa, 0xa4, 0xa3, 0x12,
			0x32, 0xa9, 0x63, 0xda, 0x1e, 0xf4, 0xda, 0xc0,
			0x81, 0xd8, 0x88, 0x17, 0xc6, 0x34, 0x28, 0x58,
			0x24, 0xab, 0xcd, 0x84, 0xa1, 0xd7, 0xdc, 0xd8,
			0x8d, 0x0e, 0xb1, 0x03, 0x5c, 0x99, 0x53, 0x27,
			0x2b, 0x4a, 0x9e, 0xd4, 0x64, 0x96, 0x2f, 0xbf,
			0x04, 0xb2, 0x89, 0xb7, 0x33, 0x3f, 0x33, 0x45,
			0xa1, 0x78, 0xa9, 0xc2, 0xde, 0x62, 0xa6, 0xf8,
		},
	},
	/* SHA384 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0xcc, 0x29, 0x62, 0xfa, 0xdc, 0xe7, 0x9e, 0x25,
			0xff, 0xd2, 0x23, 0x01, 0x67, 0xe3, 0x59, 0x68,
			0x5a, 0xd7, 0x55, 0x29, 0x35, 0x28, 0xfd, 0xa3,
			0xb1, 0x3b, 0x6d, 0x92, 0x65, 0xdb, 0x9a, 0x28,
			0x8c, 0xe6, 0x35, 0x60, 0x09, 0x46, 0xd3, 0x8f,
			0xa8, 0x23, 0x4a, 0x3b, 0xa7, 0x57, 0x5e, 0x31,
			0x75, 0x23, 0x9d, 0xd2, 0xb9, 0x20, 0x87, 0xae,
			0x96, 0x78, 0xf2, 0x54, 0x2d, 0x55, 0xad, 0x67,
			0x64, 0x9a, 0x64, 0xf5, 0xe6, 0xd2, 0xc3, 0x05,
			0x6c, 0x65, 0xde, 0x6f, 0x3a, 0x12, 0xd1, 0xf1,
			0xec, 0x99, 0xa3, 0x7e, 0x72, 0xb8, 0x71, 0xd9,
			0x5a, 0x59, 0xc8, 0x63, 0x4d, 0x0a, 0x52, 0x32,
			0xdd, 0xad, 0xb5, 0x62, 0x7e, 0x71, 0xb8, 0xb0,
			0xd1, 0x01, 0xa8, 0x44, 0x29, 0xfb, 0xfa, 0xcf,
			0x48, 0xa3, 0x31, 0xd6, 0x69, 0xd9, 0x02, 0x4e,
			0x84, 0x48, 0xb6, 0x68, 0xcd, 0xe1, 0xd1, 0xae,
		},
		/* S5 : 512 byte */
		{
			0x23, 0x69, 0x10, 0x0f, 0x00, 0x7f, 0xb2, 0xf7,
			0xc4, 0x45, 0x11, 0x99, 0x60, 0x94, 0xe3, 0xba,
			0xf0, 0xb8, 0x33, 0xc4, 0x57, 0xa2, 0x5c, 0xca,
			0x52, 0x85, 0x2d, 0xa3, 0x51, 0x89, 0x5c, 0xc1,
			0xaa, 0x14, 0x9b, 0xa2, 0xa1, 0xc4, 0x78, 0x13,
			0x3f, 0xdb, 0xde, 0x6b, 0xdd, 0x78, 0x93, 0x4c,
			0xba, 0xc9, 0xad, 0x5f, 0x35, 0x36, 0x53, 0x7f,
			0x59, 0x2b, 0x9b, 0xe6, 0x23, 0xb3, 0x8a, 0xb9,
			0x1d, 0x72, 0xe1, 0x57, 0xe9, 0x60, 0x3e, 0x2d,
			0xae, 0x2c, 0x05, 0xcb, 0x4e, 0x24, 0xde, 0x38,
			0xdb, 0x95, 0xa6, 0xf1, 0xc7, 0x8b, 0xd2, 0x52,
			0x17, 0xac, 0xdd, 0xd0, 0x46, 0x39, 0xfe, 0xbd,
			0x08, 0xb3, 0x2e, 0x39, 0xb7, 0x26, 0x15, 0xf3,
			0x72, 0xdd, 0x8a, 0x52, 0x94, 0x22, 0x68, 0x4f,
			0x77, 0x3f, 0x27, 0x64, 0x94, 0x35, 0x82, 0xda,
			0xaa, 0x92, 0x4b, 0x26, 0x62, 0x2f, 0xd8, 0x96,
			0x41, 0x92, 0xa4, 0xea, 0x4b, 0x76, 0x66, 0x5a,
			0xd7, 0xd8, 0xca, 0x90, 0xb6, 0xf5, 0x35, 0x17,
			0x1b, 0x4d, 0xd9, 0xd6, 0x8d, 0x34, 0xfc, 0x86,
			0xb6, 0x2b, 0xe6, 0x59, 0x26, 0x30, 0x59, 0x9f,
			0x61, 0xea, 0xe9, 0x71, 0x58, 0x04, 0x00, 0x2b,
			0x70, 0x61, 0x06, 0x3f, 0x4d, 0x56, 0xa2, 0xe3,
			0x9c, 0x61, 0x8e, 0xc0, 0xcf, 0x2e, 0x1a, 0xf8,
			0xfa, 0x06, 0xba, 0xb4, 0x8a, 0xf7, 0x61, 0xe3,
			0x63, 0x74, 0x20, 0x5c, 0x40, 0xd1, 0x1d, 0xec,
			0x23, 0x9c, 0x18, 0x00, 0x54, 0xce, 0xca, 0x89,
			0x79, 0xaa, 0x19, 0xf8, 0x58, 0x40, 0x9e, 0xa3,
			0x3b, 0x8e, 0x9c, 0x9d, 0x67, 0x64, 0x7f, 0xd0,
			0x2f, 0x3c, 0x31, 0x22, 0x20, 0xc4, 0x1d, 0x40,
			0xc0, 0x86, 0xf7, 0xc4, 0xc3, 0x00, 0x9e, 0x2c,
			0x1c, 0x05, 0x8d, 0x63, 0xa7, 0x6f, 0xf5, 0xa8,
			0x85, 0xa2, 0xe9, 0x90, 0xcd, 0x5d, 0xbb, 0xc2,
			0x4a, 0x0b, 0xad, 0x29, 0xa7, 0xe4, 0x2b, 0x81,
			0x69, 0x10, 0xee, 0x6b, 0xa5, 0xd4, 0x92, 0x59,
			0x65, 0xb4, 0x5f, 0x15, 0x88, 0xfa, 0xca, 0xd7,
			0x8c, 0xe8, 0xdf, 0x85, 0x0f, 0x0e, 0x70, 0xba,
			0x83, 0xeb, 0x48, 0x5e, 0xe3, 0xdb, 0x12, 0xe6,
			0xb8, 0xe2, 0x64, 0x23, 0xb2, 0xc9, 0x2a, 0xe7,
			0x65, 0x37, 0x83, 0x7c, 0xf7, 0xde, 0xf7, 0xe5,
			0xff, 0x37, 0x89, 0xbc, 0x0d, 0x27, 0x82, 0xb9,
			0x7d, 0x4f, 0x25, 0x46, 0xa6, 0x2d, 0xdb, 0xf9,
			0x54, 0xf2, 0x87, 0xc9, 0x1f, 0xfa, 0x26, 0x00,
			0x90, 0xd3, 0xba, 0xaa, 0x50, 0xb5, 0xd4, 0xd0,
			0x4c, 0xc8, 0x6b, 0xa4, 0x7b, 0x91, 0x07, 0x68,
			0x72, 0xf3, 0x4d, 0xdb, 0x99, 0x91, 0x2b, 0xc6,
			0x42, 0x5f, 0x11, 0xc5, 0x0f, 0x2f, 0xd4, 0x38,
			0x50, 0xa6, 0xaf, 0xb7, 0x6a, 0x84, 0xa1, 0x1d,
			0xd3, 0xed, 0x68, 0x0e, 0x86, 0xa2, 0xa7, 0xce,
			0x90, 0x31, 0x78, 0x4f, 0xb5, 0xe6, 0x2a, 0x8d,
			0x8d, 0xf3, 0x42, 0x6d, 0xf3, 0xf6, 0x1d, 0xe4,
			0xb9, 0x08, 0xa7, 0x6a, 0xbc, 0xfe, 0xb0, 0x21,
			0x64, 0x58, 0x50, 0x86, 0xba, 0xc4, 0x58, 0x92,
			0x11, 0x43, 0x28, 0xe3, 0xdc, 0xff, 0x42, 0x3b,
			0x51, 0x97, 0x09, 0xaf, 0xd4, 0x9b, 0x46, 0x0d,
			0x29, 0x63, 0xba, 0x79, 0xf2, 0x0c, 0xe0, 0x0b,
			0x38, 0xee, 0xbd, 0xb4, 0xb9, 0x42, 0xc7, 0x40,
			0x95, 0xb1, 0x81, 0x33, 0x0a, 0x0a, 0xec, 0xa5,
			0xef, 0x7f, 0x66, 0x84, 0x69, 0x87, 0xa3, 0x25,
			0xde, 0xfd, 0x04, 0x12, 0xbe, 0x4b, 0x0d, 0xf2,
			0x77, 0x3d, 0xd5, 0x31, 0x45, 0x69, 0xa8, 0x58,
			0x4a, 0x1e, 0x7d, 0xd6, 0x7a, 0x64, 0x58, 0xca,
			0xc4, 0x6d, 0x29, 0x6c, 0x7f, 0xd8, 0xb8, 0xe1,
			0x55, 0xa7, 0x4d, 0x07, 0x2a, 0xa6, 0xc9, 0x77,
			0xbb, 0x5f, 0xf8, 0x65, 0xd9, 0xbc, 0xdb, 0x28,
		},
	},
	/* SHA384 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0x3f, 0xba, 0x35, 0x33, 0x5b, 0x2b, 0xd9, 0xa1,
			0x40, 0x32, 0xda, 0x98, 0xb1, 0x1b, 0xa7, 0xd5,
			0x48, 0xab, 0x43, 0x63, 0xb7, 0x53, 0xa6, 0x6a,
			0x54, 0xa8, 0x61, 0x48, 0xd5, 0x18, 0x80, 0xe2,
			0x57, 0xd5, 0xb4, 0x3c, 0xa6, 0x16, 0xb2, 0x48,
			0xb8, 0x7f, 0xd6, 0xfc, 0x7a, 0xdd, 0x0c, 0xa7,
			0xd0, 0x17, 0xf0, 0x85, 0xd6, 0x62, 0x34, 0x91,
			0x6d, 0xb3, 0x42, 0x9a, 0xf3, 0x7a, 0x23, 0x99,
			0x51, 0x8d, 0xa0, 0x42, 0x3b, 0x62, 0xa7, 0xd9,
			0xb3, 0x82, 0xf0, 0x47, 0xcf, 0x3a, 0xdf, 0x6d,
			0xbd, 0x94, 0xad, 0xb8, 0x04, 0x86, 0xc3, 0xf6,
			0x0c, 0x68, 0x66, 0x68, 0xe1, 0xab, 0x99, 0x18,
			0x34, 0xc9, 0x3f, 0x6c, 0xa3, 0x78, 0x81, 0xfc,
			0x3a, 0x23, 0x6f, 0xa1, 0x39, 0x9a, 0xea, 0xce,
			0x17, 0x09, 0xc1, 0x24, 0xc8, 0x27, 0x1a, 0x40,
			0x3c, 0xaf, 0x14, 0xd8, 0x28, 0x14, 0xaa, 0x06,
		},
		/* S6 : 512 byte */
		{
			0x8f, 0xe6, 0xe0, 0xd3, 0x7c, 0xbc, 0x21, 0x87,
			0xf7, 0xf5, 0xbd, 0xc5, 0x3e, 0x56, 0xd7, 0x1a,
			0xaa, 0x6b, 0x24, 0x6c, 0xf3, 0xb6, 0x3d, 0x88,
			0x64, 0x4a, 0x13, 0x0f, 0xe4, 0x97, 0x27, 0xa2,
			0x77, 0x98, 0x63, 0x33, 0x0b, 0x14, 0x63, 0xb3,
			0xfb, 0x58, 0x31, 0x76, 0x54, 0x7c, 0x44, 0x4b,
			0x3b, 0x39, 0x9f, 0x88, 0x0c, 0x1d, 0x6c, 0x13,
			0xc7, 0xe4, 0xf8, 0x33, 0x4e, 0xfa, 0xd2, 0x89,
			0x79, 0xcc, 0xee, 0x44, 0x0f, 0x26, 0x58, 0x02,
			0xf0, 0xc2, 0x0d, 0xac, 0x7b, 0xbb, 0xa4, 0x5c,
			0x1d, 0x0b, 0x70, 0x95, 0x92, 0x4e, 0x02, 0xa1,
			0xae, 0x5a, 0x6f, 0x2e, 0x1d, 0x08, 0x20, 0xa8,
			0x14, 0x78, 0xee, 0x11, 0xac, 0x8c, 0x6b, 0xe4,
			0xd6, 0x2f, 0x31, 0x43, 0xba, 0x4c, 0x1a, 0xfe,
			0x5a, 0x11, 0x78, 0x73, 0x4f, 0x5b, 0xdc, 0x8c,
			0x3b, 0x43, 0xfd, 0x87, 0x73, 0x97, 0x26, 0x31,
			0x66, 0x86, 0x59, 0x97, 0xf0, 0xe7, 0x98, 0xc3,
			0x12, 0x9c, 0xb3, 0x63, 0x54, 0x05, 0xb8, 0x5d,
			0xc2, 0xe2, 0xc4, 0x3b, 0x4f, 0xfc, 0x88, 0x2e,
			0x2d, 0x92, 0x84, 0x17, 0x3d, 0xe2, 0x39, 0xf6,
			0xee, 0xf8, 0x93, 0xd7, 0xda, 0x08, 0x83, 0xf8,
			0xa5, 0x52, 0xa5, 0x75, 0x5a, 0x1e, 0x9b, 0x95,
			0xa8, 0x54, 0xbd, 0xb8, 0x31, 0x88, 0x2a, 0x3f,
			0x0c, 0xa3, 0xa8, 0xd5, 0xc0, 0x0a, 0x15, 0xc2,
			0xc3, 0xdb, 0x53, 0xce, 0xa2, 0x2b, 0xba, 0x9a,
			0x5a, 0x95, 0x74, 0x9a, 0x33, 0x5b, 0x51, 0x0c,
			0x44, 0x46, 0x05, 0xbe, 0x6d, 0xf3, 0xaa, 0x27,
			0xcb, 0x07, 0x08, 0x27, 0xfa, 0xba, 0x75, 0x3c,
			0xbc, 0x97, 0xfb, 0x2e, 0x17, 0x63, 0x8f, 0xda,
			0x4a, 0x6c, 0xd3, 0x9c, 0xb5, 0x2a, 0x16, 0xb6,
			0x26, 0x5b, 0xd5, 0x4d, 0xd7, 0xd7, 0x0c, 0x7f,
			0x15, 0x7c, 0xb7, 0x5b, 0x2b, 0xb4, 0x81, 0xff,
			0x71, 0x87, 0xd4, 0xdb, 0x34, 0x11, 0x61, 0x84,
			0x5b, 0x9f, 0x9f, 0x11, 0x0e, 0xfc, 0x87, 0xd6,
			0x6b, 0xc4, 0x9f, 0x8f, 0xf8, 0xca, 0xbc, 0x92,
			0xb9, 0x31, 0xc0, 0xd3, 0xe3, 0x4e, 0x8a, 0x88,
			0x02, 0x1b, 0xc6, 0x03, 0x80, 0x34, 0x3a, 0x59,
			0xc9, 0x90, 0x0d, 0xd1, 0xb9, 0x64, 0x58, 0x60,
			0x9c, 0x4a, 0x51, 0x9a, 0xe0, 0x66, 0x25, 0x02,
			0xc5, 0x14, 0xf3, 0x6f, 0xa3, 0x40, 0x04, 0xdb,
			0x83, 0xaa, 0x94, 0xfb, 0xc9, 0x7c, 0x43, 0x2a,
			0xe3, 0x20, 0xa0, 0x6a, 0x18, 0x6a, 0xd8, 0xe8,
			0x97, 0xa3, 0x68, 0xf0, 0x18, 0x81, 0xa9, 0xfd,
			0x64, 0x18, 0x9f, 0x84, 0xae, 0x52, 0x37, 0xf2,
			0x36, 0x50, 0xab, 0xdb, 0xc2, 0xfe, 0x1c, 0xff,
			0xde, 0x4e, 0xb6, 0xd1, 0x81, 0x5f, 0x23, 0xda,
			0x83, 0x39, 0xbf, 0xeb, 0x19, 0xa3, 0xf6, 0x14,
			0xa4, 0x4c, 0x29, 0x2b, 0x74, 0x1f, 0xae, 0xdc,
			0x84, 0xd5, 0x61, 0x4c, 0xbb, 0xe1, 0xc7, 0x20,
			0x76, 0xc8, 0x8c, 0xfa, 0x8e, 0x8e, 0xfe, 0x49,
			0xb1, 0x2b, 0x17, 0x50, 0x62, 0x03, 0x06, 0x21,
			0x57, 0x94, 0xc6, 0x34, 0x6a, 0x3a, 0x78, 0x60,
			0x13, 0xeb, 0x0c, 0x9c, 0xfc, 0x47, 0x11, 0x2d,
			0x33, 0x7b, 0x87, 0xa5, 0x42, 0xe2, 0xe6, 0x80,
			0x22, 0x62, 0x5f, 0xaf, 0xcd, 0x43, 0x69, 0x5c,
			0x9e, 0x4d, 0xb6, 0xcd, 0xb3, 0xbf, 0x99, 0x29,
			0xf3, 0xef, 0x29, 0x3f, 0xb4, 0xd3, 0x4b, 0x01,
			0x3a, 0xd3, 0x38, 0x9c, 0x3e, 0x52, 0x56, 0x27,
			0x55, 0xc1, 0x1f, 0xf5, 0x59, 0xef, 0x82, 0x01,
			0x03, 0x8c, 0xca, 0x38, 0x66, 0x20, 0x45, 0x36,
			0x0a, 0x7c, 0xb0, 0x9c, 0x96, 0x23, 0x57, 0x9b,
			0x66, 0xbe, 0x70, 0x34, 0x88, 0x4a, 0x52, 0x55,
			0x08, 0xa2, 0xf3, 0xec, 0xc2, 0x92, 0x17, 0x31,
			0xfc, 0x4e, 0x82, 0x05, 0xa7, 0xdc, 0x00, 0xe3,
		},
	},
	/* SHA384 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0xc1, 0x6d, 0x33, 0xb0, 0x3f, 0x30, 0x70, 0xa8,
			0x8a, 0xf2, 0x82, 0xad, 0x09, 0x0e, 0xa5, 0x1b,
			0x75, 0xe7, 0x03, 0x87, 0xdf, 0xd9, 0xd5, 0x53,
			0xf5, 0x70, 0xcc, 0x7f, 0x81, 0xd8, 0xaf, 0xc2,
			0x32, 0x95, 0x45, 0xda, 0x9f, 0x74, 0x75, 0x20,
			0x89, 0xc2, 0xac, 0x5f, 0x60, 0xbb, 0x7f, 0xe6,
			0x09, 0x45, 0x92, 0xdc, 0xdf, 0xdf, 0xb8, 0xd4,
			0xf8, 0xd4, 0x5b, 0xa1, 0xa9, 0xb4, 0x1f, 0x6c,
			0x4c, 0x04, 0xc2, 0x04, 0x40, 0x35, 0x47, 0x74,
			0x32, 0xd3, 0x53, 0xdb, 0x9b, 0xff, 0xd7, 0x78,
			0x8a, 0x1f, 0xef, 0x05, 0xa4, 0xd4, 0x0a, 0x46,
			0x69, 0x2d, 0x4c, 0xe1, 0xb9, 0x1a, 0x30, 0x70,
			0x96, 0x23, 0x72, 0xd5, 0x2b, 0x59, 0x2c, 0xcd,
			0x0d, 0x8d, 0x3f, 0xc9, 0xc2, 0xc1, 0xf4, 0xf9,
			0x80, 0xdc, 0xe3, 0xa9, 0x36, 0xa2, 0x14, 0xd0,
			0xd1, 0xe2, 0x65, 0xe6, 0xb8, 0xf1, 0x2b, 0xf7,
		},
		/* S7 : 512 byte */
		{
			0x09, 0xb1, 0xd3, 0xd8, 0xfe, 0x2e, 0x57, 0x59,
			0x96, 0x77, 0x3b, 0x74, 0xe3, 0x48, 0x17, 0x34,
			0xc1, 0x43, 0x90, 0x72, 0x5c, 0x8b, 0x02, 0x31,
			0xf4, 0x46, 0xa7, 0xbb, 0x0f, 0xae, 0x65, 0x67,
			0x8a, 0xe9, 0x25, 0x49, 0x31, 0xe6, 0x88, 0x28,
			0x26, 0x0a, 0xd3, 0xa0, 0x20, 0x1f, 0xe9, 0x36,
			0x06, 0x40, 0x58, 0x69, 0xf9, 0xcc, 0xec, 0xa5,
			0xed, 0x7e, 0x36, 0x55, 0x95, 0xe9, 0xc8, 0x96,
			0x5a, 0x3e, 0xc1, 0x5f, 0xb7, 0xef, 0x50, 0x91,
			0xcf, 0x32, 0x70, 0xd2, 0xf8, 0x17, 0xd1, 0x6d,
			0xe2, 0x15, 0x2b, 0xb7, 0x4c, 0xc0, 0xff, 0xd1,
			0xfd, 0x75, 0x8d, 0x36, 0x78, 0x80, 0x9d, 0x06,
			0x94, 0x05, 0x4e, 0x5c, 0x7f, 0x83, 0x21, 0xf6,
			0xa2, 0x66, 0xed, 0x91, 0x53, 0x41, 0x26, 0xeb,
			0xfc, 0xeb, 0x6e, 0x10, 0x29, 0x12, 0xac, 0xb6,
			0xda, 0xbd, 0xf6, 0x11, 0x54, 0xa2, 0x38, 0x98,
			0x7d, 0x6a, 0x83, 0xc0, 0x14, 0x0f, 0x4e, 0x16,
			0xea, 0xff, 0xfe, 0xa7, 0x09, 0x1d, 0x25, 0xe5,
			0x51, 0x6e, 0x69, 0x85, 0xfc, 0x78, 0x0f, 0x46,
			0x5c, 0x21, 0x36, 0xcd, 0x29, 0x93, 0x56, 0x95,
			0xf6, 0xe0, 0x61, 0x82, 0x4e, 0xf3, 0xb0, 0x4e,
			0x3f, 0xcd, 0xc4, 0x3a, 0x64, 0x84, 0xae, 0x96,
			0xb5, 0x01, 0x3c, 0xdb, 0x8f, 0x59, 0x83, 0x17,
			0x32, 0x9c, 0x05, 0x8b, 0x91, 0xf8, 0x6d, 0x3c,
			0x0e, 0x04, 0x97, 0xac, 0xea, 0x16, 0x1e, 0xec,
			0xdb, 0x2d, 0xf7, 0x68, 0x25, 0x8f, 0x9c, 0xa4,
			0x61, 0xec, 0x9b, 0x4b, 0x60, 0x09, 0xa5, 0x57,
			0xe3, 0xd7, 0x17, 0xbe, 0x4f, 0xbd, 0x20, 0x07,
			0x88, 0xa5, 0xc7, 0x26, 0x7d, 0x05, 0x2f, 0x15,
			0x6d, 0x2a, 0x66, 0x10, 0xba, 0x27, 0x02, 0xe8,
			0x61, 0xb9, 0xba, 0x6e, 0xae, 0x7d, 0xca, 0xc2,
			0x47, 0x75, 0x8b, 0x8c, 0x8d, 0x8f, 0x3e, 0x33,
			0xde, 0x76, 0xbd, 0x56, 0xb7, 0x07, 0x02, 0x0b,
			0xcf, 0x74, 0x4e, 0xde, 0x53, 0x39, 0x23, 0xdf,
			0x96, 0x58, 0x06, 0x38, 0xc9, 0x24, 0xea, 0x59,
			0xc6, 0x0b, 0x6b, 0xd2, 0x44, 0xf0, 0x02, 0x95,
			0x30, 0x0f, 0x47, 0x79, 0x1d, 0x2e, 0xa7, 0xd3,
			0xfa, 0x7d, 0xd5, 0x80, 0x75, 0x6e, 0x91, 0x89,
			0x38, 0x5e, 0x9a, 0x71, 0xd1, 0xe1, 0xb6, 0xa1,
			0xec, 0x03, 0x29, 0xbd, 0x99, 0x3a, 0x40, 0x99,
			0xfd, 0x50, 0xf1, 0xd4, 0x56, 0x2e, 0xca, 0x6c,
			0x7e, 0xfe, 0x3c, 0xd5, 0x6d, 0xb6, 0x8e, 0xb1,
			0x4b, 0x96, 0xfa, 0x98, 0x1f, 0x9f, 0x43, 0xaf,
			0x9c, 0x8e, 0x34, 0x8c, 0x10, 0x1f, 0xd8, 0x1b,
			0x5e, 0x46, 0x81, 0x2d, 0xa7, 0xb2, 0x1e, 0x1b,
			0x99, 0x52, 0x12, 0x48, 0xbf, 0xeb, 0xdb, 0x6b,
			0xe7, 0x73, 0x38, 0x3a, 0x32, 0x6a, 0xad, 0xad,
			0xdd, 0x80, 0xe4, 0x0f, 0x46, 0xf3, 0x4f, 0x96,
			0x3a, 0xe3, 0x64, 0xaf, 0xbb, 0xa2, 0xc0, 0x4a,
			0xa9, 0x3f, 0xc3, 0x92, 0x96, 0x63, 0x55, 0xa3,
			0xd8, 0xb8, 0x1c, 0x51, 0x61, 0x86, 0xd9, 0xf2,
			0x3d, 0xd2, 0xb8, 0x4d, 0x9c, 0x5d, 0x54, 0xc1,
			0xfc, 0x67, 0x30, 0xdd, 0xdc, 0x21, 0x36, 0x01,
			0xa6, 0xfb, 0xdd, 0x7b, 0x93, 0xa2, 0x1c, 0x9b,
			0x8e, 0xcf, 0xb9, 0x7f, 0x4b, 0xc8, 0x0b, 0x4f,
			0x6c, 0x9a, 0x92, 0xce, 0xe6, 0x05, 0xaa, 0x47,
			0x16, 0xd0, 0x8f, 0x9c, 0x0f, 0xd4, 0x2c, 0xf8,
			0xcc, 0x9a, 0x83, 0x44, 0xb7, 0x92, 0x4f, 0xb9,
			0xf7, 0x75, 0xd9, 0xce, 0xed, 0xf9, 0xf8, 0xd6,
			0x51, 0x83, 0x1c, 0x62, 0x63, 0xb3, 0xf9, 0x85,
			0x2a, 0xfe, 0xa6, 0xc2, 0x7e, 0xce, 0xba, 0x78,
			0xfc, 0x72, 0x85, 0xf4, 0xdf, 0x7a, 0x67, 0xd1,
			0xea, 0x1f, 0xac, 0x3b, 0xe0, 0x7e, 0x3b, 0xfe,
			0x35, 0x1d, 0x6e, 0x2e, 0x28, 0x5d, 0xbf, 0x1f,
		},
	},
	/* SHA384 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0xb2, 0xd6, 0x12, 0x58, 0xc8, 0x1e, 0xdc, 0x56,
			0xf8, 0x51, 0x38, 0xa0, 0x40, 0x46, 0x35, 0x57,
			0x4e, 0xc0, 0x16, 0x05, 0x96, 0x38, 0x02, 0x80,
			0xa1, 0xdd, 0xfd, 0x12, 0xa8, 0xda, 0x17, 0xd6,
			0x61, 0xd7, 0xe3, 0x65, 0x19, 0xc2, 0x2b, 0xb2,
			0x03, 0xcf, 0xe2, 0x33, 0xd0, 0xbd, 0x5a, 0x42,
			0x7f, 0x5c, 0xb8, 0xc2, 0x77, 0x4c, 0xa4, 0x05,
			0xd3, 0x07, 0x45, 0xed, 0xfd, 0x21, 0x6c, 0x52,
			0x7e, 0xcf, 0x0f, 0xa4, 0x14, 0xa7, 0xfa, 0xd2,
			0x05, 0xa6, 0xc1, 0x6a, 0xb4, 0x77, 0xf9, 0x01,
			0x72, 0xf2, 0xa7, 0xd5, 0x96, 0xe4, 0xf9, 0xb1,
			0xd0, 0x0e, 0x34, 0x11, 0xba, 0x70, 0xee, 0x87,
			0xb0, 0xc7, 0x7d, 0x5d, 0xe2, 0x54, 0xaf, 0x7c,
			0xa7, 0xdf, 0xbc, 0x8c, 0x15, 0xfc, 0x78, 0x5d,
			0xce, 0x8d, 0xcd, 0x86, 0x1c, 0x88, 0x69, 0x4c,
			0x41, 0xfa, 0xb4, 0xc5, 0x08, 0x4c, 0x05, 0x3d,
		},
		/* S8 : 512 byte */
		{
			0x7b, 0x15, 0xa0, 0xec, 0xd9, 0x1f, 0x3a, 0x80,
			0x19, 0x62, 0xdd, 0x8c, 0x49, 0x5b, 0x12, 0x56,
			0x7f, 0x2f, 0x6e, 0xf4, 0xe3, 0x19, 0xbc, 0x1b,
			0x7f, 0x05, 0xc7, 0x9a, 0xd5, 0x57, 0xfb, 0x12,
			0x16, 0x47, 0xb0, 0x7a, 0x95, 0xbf, 0x0f, 0xac,
			0xf6, 0x49, 0xbe, 0x8b, 0x4f, 0x94, 0x30, 0x3d,
			0x7d, 0x19, 0x26, 0xaf, 0x9e, 0xf2, 0x6f, 0x8b,
			0xd8, 0x02, 0xd6, 0x96, 0x83, 0xfd, 0x59, 0x0e,
			0xc9, 0x21, 0x5e, 0xbd, 0x05, 0x26, 0xa4, 0xa5,
			0x08, 0x65, 0xc6, 0xbd, 0x49, 0x4c, 0x18, 0xaf,
			0x20, 0xe0, 0x84, 0x6e, 0x7b, 0xe8, 0x41, 0x9d,
			0xf1, 0x9d, 0xfe, 0x48, 0x2f, 0x26, 0xdd, 0x9d,
			0x02, 0x76, 0x60, 0x58, 0xa0, 0x86, 0x48, 0xd0,
			0xcf, 0xb2, 0x73, 0xe4, 0xff, 0xbe, 0x90, 0x4f,
			0x3b, 0xc9, 0xff, 0xc9, 0x2e, 0x74, 0xcd, 0x4b,
			0x8e, 0x26, 0x69, 0x91, 0x4d, 0x1d, 0x4d, 0x12,
			0x80, 0x2d, 0x97, 0x1c, 0x92, 0xca, 0xde, 0x5f,
			0x8e, 0x83, 0xf6, 0xbf, 0xc7, 0xb5, 0x18, 0xfe,
			0xc3, 0x93, 0x3f, 0xe2, 0x76, 0x2d, 0xd7, 0x9e,
			0xfc, 0x96, 0x56, 0xbd, 0x7f, 0x92, 0xad, 0x2c,
			0x04, 0x6d, 0x4b, 0x7d, 0x9a, 0xec, 0x08, 0x5d,
			0x4a, 0x8e, 0xd4, 0xcb, 0x40, 0xb6, 0x53, 0xdf,
			0x8d, 0x7d, 0x06, 0xb8, 0xb6, 0xf5, 0x1d, 0x77,
			0x33, 0x75, 0xfe, 0x10, 0x8b, 0xd4, 0x41, 0x1c,
			0xb7, 0x73, 0x69, 0x4c, 0x6f, 0x61, 0xbd, 0x7e,
			0x41, 0x6d, 0xe0, 0xb4, 0x02, 0x2e, 0xde, 0xdc,
			0xd2, 0xca, 0xd2, 0x91, 0x40, 0x82, 0x03, 0xe2,
			0x5a, 0x6b, 0xde, 0x32, 0xd4, 0xe5, 0xaa, 0xca,
			0x0b, 0x28, 0x98, 0xa4, 0x16, 0x39, 0x10, 0x14,
			0x79, 0x43, 0x3e, 0xd3, 0x4d, 0x80, 0xde, 0xaa,
			0xf0, 0x4d, 0x07, 0x3b, 0xbd, 0x6d, 0x0b, 0x2c,
			0x4b, 0x32, 0x0c, 0xaf, 0x5c, 0x3b, 0x7a, 0x1c,
			0xf4, 0xe0, 0x82, 0x56, 0xd1, 0xc9, 0x07, 0xe9,
			0x1a, 0x5c, 0xf2, 0x0e, 0x94, 0x61, 0xcc, 0xc3,
			0x9c, 0x97, 0x39, 0x10, 0xab, 0xd1, 0xa3, 0xc9,
			0x43, 0x7d, 0x20, 0xbb, 0xef, 0xd8, 0xd0, 0x0c,
			0x6a, 0xa5, 0xa5, 0xfe, 0xb6, 0xf7, 0x78, 0x98,
			0x6f, 0xe8, 0xa6, 0x44, 0xcb, 0x41, 0x30, 0x20,
			0xb9, 0x45, 0xa8, 0x92, 0x2d, 0xa9, 0x36, 0x2d,
			0xc1, 0x5b, 0xb7, 0x54, 0x5d, 0x21, 0xbd, 0x50,
			0x41, 0x15, 0x58, 0xb5, 0x7d, 0xcc, 0x16, 0xa1,
			0x9d, 0x19, 0x51, 0x37, 0xcc, 0x67, 0x89, 0x04,
			0x70, 0x05, 0x18, 0xbe, 0x66, 0xd0, 0x89, 0x68,
			0xe1, 0x19, 0x7d, 0xbe, 0x5d, 0x0c, 0x66, 0xe6,
			0xd0, 0x77, 0x14, 0x5c, 0xe8, 0x64, 0x1e, 0xae,
			0xad, 0x2e, 0x6e, 0xa5, 0x7b, 0x93, 0x06, 0x3c,
			0x41, 0x2c, 0x33, 0x94, 0x22, 0xdd, 0xcd, 0xd8,
			0x18, 0xcc, 0x72, 0x5a, 0x29, 0x0b, 0x5e, 0x49,
			0xa6, 0xb1, 0xa6, 0x4d, 0x83, 0x67, 0x6b, 0x52,
			0xfe, 0x61, 0x79, 0x7a, 0x34, 0x55, 0x6c, 0xbf,
			0x95, 0xf0, 0xbe, 0x37, 0x54, 0x62, 0xca, 0x72,
			0xd6, 0xf5, 0x96, 0xac, 0x47, 0x49, 0xab, 0xbb,
			0x44, 0x67, 0x79, 0xae, 0xbe, 0x51, 0xa5, 0x3b,
			0x7a, 0x34, 0xcf, 0xa0, 0xb0, 0xe4, 0x6e, 0x55,
			0x70, 0x7e, 0x1e, 0x65, 0x1d, 0x4a, 0xe7, 0x0b,
			0xcd, 0x7c, 0x6d, 0x53, 0x28, 0xcd, 0xca, 0xee,
			0xf9, 0xe8, 0x8f, 0xae, 0x9f, 0x6a, 0x9b, 0xe6,
			0x09, 0xc0, 0x64, 0xba, 0x19, 0x33, 0x29, 0xcd,
			0x82, 0xbc, 0x3d, 0xed, 0x54, 0x78, 0x99, 0xf2,
			0xe8, 0xd4, 0x26, 0xbc, 0xf1, 0x0b, 0xe8, 0xff,
			0x2d, 0x82, 0x8f, 0xb4, 0x96, 0x80, 0x05, 0xa7,
			0x31, 0x14, 0xac, 0x00, 0x1e, 0x8f, 0xb7, 0xf0,
			0x1c, 0xe3, 0x52, 0x2f, 0xd4, 0x6b, 0x56, 0x6f,
			0x1e, 0x57, 0x76, 0x53, 0x99, 0x56, 0xdf, 0xa6,
		},
	},
	/* SHA384 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0xfb, 0xb1, 0xaf, 0xfe, 0x25, 0xa7, 0xb2, 0x8f,
			0xe7, 0x42, 0x7a, 0xa6, 0x9a, 0x89, 0xcb, 0x87,
			0xbc, 0x0f, 0xb6, 0x8c, 0x94, 0x0d, 0x63, 0xd3,
			0x19, 0xb3, 0x0e, 0x34, 0x11, 0xc0, 0x50, 0x76,
			0x76, 0x51, 0xa0, 0xc2, 0x1d, 0xf6, 0x19, 0x57,
			0x9a, 0x0b, 0x7e, 0x80, 0x5e, 0x46, 0x7b, 0xc3,
			0x22, 0x8c, 0x2d, 0x5c, 0x89, 0x09, 0xe5, 0xea,
			0x00, 0x7f, 0x29, 0x67, 0xe9, 0xde, 0xfc, 0x5d,
			0xfa, 0x16, 0x60, 0x57, 0xdf, 0x29, 0x18, 0x96,
			0x27, 0x84, 0xb2, 0x06, 0x5d, 0x8f, 0xc5, 0x71,
			0x75, 0x83, 0x78, 0xf1, 0x6b, 0x7c, 0x43, 0xab,
			0x26, 0xcf, 0xfa, 0xba, 0xa2, 0x9d, 0x07, 0x6a,
			0x4e, 0xa8, 0xf6, 0xa7, 0xb3, 0x66, 0x38, 0xb5,
			0x57, 0x54, 0x20, 0x21, 0xe1, 0x8d, 0x32, 0xf7,
			0x7d, 0x9b, 0x9d, 0x35, 0x82, 0x8d, 0x06, 0x80,
			0x53, 0x49, 0x87, 0x16, 0x83, 0x24, 0xf8, 0x04,
		},
		/* S9 : 512 byte */
		{
			0x69, 0x60, 0x07, 0x8b, 0xc7, 0x80, 0x4d, 0x10,
			0x73, 0x16, 0xa6, 0xc4, 0xbe, 0x9d, 0x9f, 0xfa,
			0x5a, 0xcd, 0xed, 0x00, 0x52, 0xa2, 0xf2, 0x52,
			0xa0, 0x1d, 0xdd, 0xae, 0x1a, 0x42, 0x47, 0xe9,
			0x34, 0x15, 0x24, 0x64, 0x3f, 0xbb, 0x2e, 0xbc,
			0x93, 0x63, 0x94, 0xa8, 0x52, 0x1e, 0x97, 0x81,
			0x0f, 0xa8, 0x1d, 0x14, 0xfa, 0xd7, 0x84, 0xc5,
			0x89, 0xe7, 0x45, 0xde, 0x37, 0x24, 0xcc, 0x5c,
			0x26, 0x02, 0x0d, 0x8e, 0x21, 0xaf, 0xf4, 0xd1,
			0x06, 0xfa, 0xc2, 0x24, 0xff, 0xee, 0xc5, 0x09,
			0x4b, 0x60, 0x70, 0x75, 0x28, 0xa4, 0x6e, 0xed,
			0x90, 0x19, 0xd2, 0xdc, 0xaa, 0xee, 0x7e, 0xf1,
			0xaf, 0x2a, 0xa4, 0x14, 0xcb, 0x6f, 0x8b, 0x72,
			0xe2, 0x36, 0x57, 0x29, 0xba, 0xf8, 0x2d, 0x3d,
			0x8c, 0x12, 0x03, 0xed, 0xb3, 0xa7, 0x4c, 0x06,
			0xbb, 0xb2, 0xb5, 0x5d, 0xca, 0xb0, 0x21, 0xc1,
			0xff, 0x27, 0xac, 0x38, 0x27, 0xf8, 0x78, 0x6e,
			0x42, 0x2d, 0xe8, 0x2b, 0x35, 0xb2, 0x45, 0xef,
			0xeb, 0x0b, 0x47, 0xef, 0x55, 0xec, 0xdc, 0xc3,
			0x09, 0x4c, 0xc5, 0xf5, 0xc2, 0x1d, 0xa9, 0xcf,
			0x81, 0x63, 0x41, 0xb9, 0x0f, 0xdc, 0x36, 0xf8,
			0xcc, 0xd5, 0xfd, 0x8f, 0xf7, 0x8e, 0x4a, 0xa8,
			0xb1, 0x0a, 0x4a, 0x9b, 0x03, 0x8a, 0xc0, 0x9a,
			0xe0, 0x76, 0x1c, 0x30, 0x0f, 0xa4, 0x98, 0xb0,
			0x51, 0xae, 0x1f, 0x12, 0x9e, 0x0f, 0x0a, 0xa1,
			0xc0, 0xfe, 0xc9, 0x2c, 0x31, 0x60, 0x9a, 0x8a,
			0xc6, 0xc4, 0xea, 0x19, 0x90, 0x8c, 0x9e, 0x73,
			0x3d, 0x27, 0x28, 0x42, 0xb1, 0xa9, 0x7d, 0xcb,
			0xaf, 0xab, 0x51, 0x6d, 0x9f, 0xe1, 0x59, 0x2c,
			0xd3, 0xa8, 0xf6, 0x01, 0x80, 0xa9, 0xd1, 0x2c,
			0x03, 0x33, 0x87, 0xc3, 0x9e, 0x18, 0xed, 0xd4,
			0xfa, 0x3c, 0xf0, 0xe8, 0x5c, 0x3f, 0x02, 0x1b,
			0x12, 0xae, 0x42, 0x87, 0x18, 0xc6, 0x9b, 0xc5,
			0xa4, 0xd3, 0xd1, 0x9b, 0x7f, 0x4d, 0x5b, 0x3a,
			0xb8, 0x9c, 0xf4, 0x49, 0xc4, 0xf7, 0x08, 0xdb,
			0x6a, 0x1e, 0x01, 0x23, 0x56, 0x3d, 0x59, 0xbd,
			0xe0, 0xa9, 0x2d, 0x45, 0xea, 0xf6, 0x52, 0xa8,
			0xe4, 0xd3, 0x9c, 0x0f, 0x4d, 0x18, 0xb0, 0x8c,
			0xf2, 0xfc, 0xd2, 0x39, 0x98, 0x3d, 0xac, 0x1b,
			0x73, 0x8a, 0x05, 0xf5, 0x5c, 0x3c, 0x99, 0x2f,
			0xe8, 0xd2, 0x97, 0x08, 0x0c, 0x1d, 0xbe, 0xd1,
			0x09, 0x37, 0x8c, 0x2d, 0x5d, 0xd8, 0x5e, 0x20,
			0x1d, 0x41, 0xf7, 0x99, 0x43, 0xcf, 0xa0, 0x8a,
			0x8b, 0x07, 0x19, 0xe7, 0x2a, 0x85, 0xd4, 0xe4,
			0xa8, 0x93, 0x55, 0x44, 0x72, 0x2f, 0x1c, 0xb3,
			0xbb, 0x4f, 0xd5, 0xce, 0x6c, 0x21, 0x01, 0x0d,
			0x61, 0x97, 0xdf, 0xc2, 0x22, 0x7b, 0xd7, 0x4f,
			0x89, 0xb9, 0x3c, 0x82, 0x18, 0x6d, 0x11, 0x8a,
			0xed, 0x14, 0x2d, 0x9e, 0x18, 0x12, 0xb2, 0xb7,
			0xd2, 0x3d, 0x70, 0x01, 0xa3, 0x27, 0xb4, 0x9f,
			0xb3, 0xb3, 0x35, 0xfd, 0xcf, 0x6c, 0x6e, 0xee,
			0x5f, 0x56, 0x5f, 0x07, 0x33, 0x74, 0x80, 0xea,
			0x8f, 0x6e, 0x38, 0xde, 0xb1, 0xa9, 0x63, 0x3f,
			0x4d, 0x4f, 0xb8, 0xd3, 0x27, 0xf6, 0xc9, 0x53,
			0x72, 0xf7, 0x98, 0x6a, 0x93, 0x1c, 0x49, 0x14,
			0x8c, 0x98, 0x2f, 0xca, 0x85, 0x84, 0xec, 0xf6,
			0xe5, 0x74, 0x49, 0x45, 0x33, 0x37, 0xed, 0x4c,
			0x95, 0xa4, 0x18, 0x5c, 0x40, 0x93, 0x20, 0xfd,
			0x96, 0xfa, 0xbf, 0xd3, 0xc3, 0x7e, 0xe0, 0xfe,
			0x68, 0x6f, 0x20, 0xda, 0x2f, 0xa4, 0x51, 0xf2,
			0xf3, 0xba, 0xa2, 0x39, 0x4b, 0xef, 0x7a, 0xcb,
			0xb0, 0x6d, 0x0a, 0xdb, 0xf6, 0x0f, 0xf8, 0xba,
			0xa3, 0x7e, 0xd1, 0xb0, 0xeb, 0x3e, 0x32, 0x24,
			0x33, 0x13, 0x9b, 0xcf, 0xcd, 0x80, 0xec, 0x86,
		},
	},
};

/* 10 elements */
static mod4096_t SHA512[] = {
	/* SHA512 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0x0a, 0xa7, 0x72, 0x9d, 0x26, 0x5c, 0x74, 0x6f,
			0x66, 0xb7, 0x52, 0x46, 0xcf, 0xf9, 0x5b, 0x07,
			0x3c, 0xdf, 0x63, 0x02, 0x53, 0xc8, 0x5d, 0x2e,
			0xb7, 0x19, 0x2d, 0xb6, 0x7f, 0xe1, 0xb7, 0xb5,
			0xe8, 0xb9, 0x3c, 0x3e, 0x46, 0xef, 0xe6, 0xa0,
			0x41, 0xe3, 0xa1, 0xe8, 0x3d, 0xb0, 0x2d, 0xa7,
			0x1a, 0x0f, 0x71, 0xf5, 0x1e, 0x79, 0x13, 0xb5,
			0x2f, 0x8c, 0x45, 0x1b, 0x46, 0xb0, 0xe2, 0x6c,
			0x20, 0xfa, 0xb9, 0x05, 0x37, 0x51, 0x29, 0x9f,
			0xf0, 0xcf, 0xee, 0xd1, 0x16, 0x69, 0x3a, 0xd6,
			0x87, 0xd3, 0x07, 0x91, 0x2d, 0x9e, 0xac, 0xce,
			0x30, 0xa4, 0xaf, 0xcb, 0x6a, 0xa5, 0xdb, 0xf7,
			0x1c, 0x32, 0x92, 0xf9, 0xdf, 0xca, 0x64, 0xf4,
			0xdd, 0x44, 0xe0, 0x0a, 0x40, 0x6c, 0xa9, 0x1f,
			0xeb, 0xf1, 0xcd, 0xdf, 0x68, 0x7f, 0x53, 0x02,
			0x22, 0x28, 0x13, 0xce, 0xd5, 0x08, 0xcb, 0xde,
		},
		/* S0 : 512 byte */
		{
			0xb4, 0x7a, 0xe7, 0x22, 0x44, 0x29, 0xdb, 0x36,
			0xd6, 0x25, 0x38, 0x78, 0xbe, 0x84, 0x08, 0x7f,
			0xaf, 0x59, 0xb5, 0xff, 0x88, 0x96, 0x64, 0x9a,
			0xfb, 0x2a, 0xed, 0xa0, 0xe0, 0x2b, 0xf8, 0x43,
			0xb5, 0xc5, 0x91, 0x94, 0x5b, 0xf7, 0xd4, 0xdb,
			0xa0, 0xaa, 0x2b, 0x09, 0xa4, 0xa5, 0x2b, 0xa3,
			0x58, 0xf4, 0xd8, 0x0b, 0x5d, 0x6f, 0x97, 0xd8,
			0xf6, 0x36, 0xe0, 0x3f, 0xe9, 0x86, 0xf3, 0x7b,
			0x38, 0xa2, 0xe8, 0x29, 0x7a, 0x48, 0xbc, 0xa4,
			0xff, 0x37, 0x59, 0xd2, 0x1e, 0xa6, 0x95, 0x8a,
			0xf1, 0xea, 0x7a, 0x08, 0xef, 0xbd, 0x61, 0xc0,
			0xb8, 0x2f, 0xde, 0x2d, 0xef, 0x42, 0x64, 0x74,
			0x4b, 0x86, 0xca, 0xcf, 0x26, 0x03, 0xf5, 0xfc,
			0x48, 0x28, 0xd2, 0xdb, 0x7d, 0xa9, 0x47, 0xba,
			0x91, 0x05, 0x8f, 0x70, 0x42, 0x12, 0xbc, 0x6d,
			0x10, 0x75, 0x11, 0x2c, 0xf0, 0xdd, 0x9d, 0x89,
			0x11, 0xf8, 0xdf, 0xc6, 0xf0, 0xcf, 0x5a, 0xd3,
			0x19, 0x87, 0x09, 0x5b, 0x5d, 0x09, 0xe5, 0x20,
			0xb5, 0x62, 0xaf, 0x62, 0x23, 0xcb, 0x35, 0xda,
			0xc8, 0xc0, 0xa6, 0xa3, 0x02, 0xba, 0x9e, 0x28,
			0xf7, 0x97, 0xb5, 0x52, 0x99, 0xf3, 0x6f, 0x82,
			0xfe, 0x41, 0x69, 0x28, 0x57, 0xec, 0x7d, 0x73,
			0xbc, 0x24, 0x60, 0x4d, 0xb0, 0xac, 0xe1, 0x06,
			0xa2, 0xe6, 0x8f, 0x6d, 0x1f, 0x9f, 0x27, 0x4c,
			0xd5, 0xf5, 0x6c, 0xd8, 0x02, 0xa4, 0xf2, 0x68,
			0x3a, 0x01, 0x3d, 0x2d, 0x3f, 0x69, 0x92, 0xbf,
			0xf5, 0x78, 0x31, 0x0d, 0x81, 0xfc, 0x1a, 0xae,
			0x7a, 0x01, 0xa6, 0x1b, 0x8a, 0x9d, 0xf7, 0x7b,
			0x02, 0x0e, 0xbf, 0xd1, 0x91, 0xab, 0x74, 0x9c,
			0x2e, 0x3b, 0x85, 0x3f, 0x28, 0x4b, 0xb7, 0x6c,
			0x83, 0xb8, 0x6a, 0xd0, 0x97, 0x8f, 0x76, 0x02,
			0x66, 0xec, 0x5e, 0x49, 0x7f, 0x2d, 0xbb, 0xfd,
			0x73, 0xeb, 0xc6, 0x93, 0x14, 0x04, 0x06, 0xa0,
			0x51, 0x67, 0x65, 0xb8, 0x3e, 0x4c, 0xa7, 0xa8,
			0x20, 0xfa, 0x44, 0xb9, 0x05, 0x7a, 0xf4, 0xb7,
			0x02, 0xc0, 0x06, 0x91, 0x42, 0xe2, 0x93, 0x3b,
			0x7e, 0x96, 0x99, 0xac, 0x09, 0x91, 0x01, 0xc2,
			0xc5, 0x09, 0xb7, 0x82, 0x5d, 0x8a, 0xef, 0x59,
			0xa6, 0x43, 0x2f, 0x38, 0xa4, 0xcc, 0x80, 0xe0,
			0xb3, 0xc4, 0x03, 0x4b, 0xb9, 0x92, 0xec, 0xce,
			0x69, 0x0d, 0x2e, 0x80, 0x96, 0x27, 0xac, 0x13,
			0xa0, 0x78, 0xb4, 0xf1, 0xa0, 0x85, 0x5d, 0x6e,
			0x63, 0xb2, 0xab, 0xdf, 0xe3, 0x07, 0x65, 0x32,
			0x44, 0xaa, 0x9b, 0x23, 0x93, 0x4e, 0xc8, 0x98,
			0xcf, 0xfe, 0xc9, 0x04, 0x64, 0x54, 0x07, 0xca,
			0xe7, 0x28, 0xfa, 0x6c, 0xd3, 0x6f, 0x3f, 0x4a,
			0x27, 0xa0, 0x1b, 0xe5, 0x02, 0xdf, 0x5f, 0x5e,
			0xbd, 0x8a, 0x7f, 0x4d, 0x0d, 0x21, 0xa4, 0x90,
			0x86, 0xfc, 0xc2, 0xe4, 0x4c, 0x62, 0xad, 0x54,
			0x41, 0xad, 0x9f, 0xc7, 0x3a, 0x15, 0xd5, 0x75,
			0x9e, 0xeb, 0x13, 0x48, 0x39, 0x98, 0x79, 0x2f,
			0x12, 0x6f, 0x40, 0xab, 0xbf, 0x9b, 0x77, 0x5a,
			0xe2, 0x45, 0x45, 0xa1, 0x1c, 0xdc, 0x84, 0x2e,
			0xdd, 0x0f, 0x23, 0x95, 0x5d, 0xfe, 0x75, 0x24,
			0x81, 0xfc, 0xa3, 0x40, 0xdb, 0xd7, 0x5b, 0xb8,
			0x90, 0x2d, 0x3a, 0x73, 0x31, 0x4b, 0xa1, 0x12,
			0xe0, 0xd4, 0xb0, 0xff, 0xab, 0x3f, 0xa9, 0xa5,
			0x16, 0x97, 0xb5, 0xef, 0xe1, 0x2e, 0x53, 0xf7,
			0xe6, 0xba, 0x73, 0x22, 0xf5, 0x19, 0x41, 0x46,
			0xae, 0x2d, 0xa2, 0x69, 0x1d, 0x14, 0x88, 0x44,
			0x70, 0xa0, 0x7e, 0xf3, 0x7b, 0xd9, 0x0c, 0xb2,
			0x08, 0x2c, 0x2e, 0xcd, 0xb1, 0xd9, 0x20, 0x62,
			0xfe, 0x2b, 0x1c, 0x80, 0x8e, 0xc5, 0xc0, 0x7d,
			0xda, 0x12, 0xf6, 0x6d, 0x9a, 0xb8, 0x59, 0x83,
		},
	},
	/* SHA512 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0xe8, 0x53, 0x03, 0x0d, 0xcd, 0x5f, 0xe6, 0x8b,
			0x1c, 0x59, 0x84, 0x4a, 0x54, 0x8f, 0x85, 0xa1,
			0x84, 0xa8, 0x80, 0x1a, 0x18, 0x2d, 0x6f, 0x1a,
			0xf3, 0x48, 0x60, 0xc5, 0xbe, 0xcb, 0x6d, 0x1c,
			0x7f, 0x73, 0xed, 0x68, 0x6d, 0x6f, 0xec, 0xdc,
			0x03, 0x1c, 0xd9, 0x76, 0x53, 0x13, 0x7f, 0x26,
			0x9d, 0x65, 0x37, 0x32, 0x20, 0x25, 0x5b, 0xd9,
			0x30, 0x27, 0xfd, 0x37, 0x89, 0x08, 0xd9, 0x68,
			0xe3, 0x6e, 0x89, 0xf0, 0xc6, 0x91, 0xf8, 0x5a,
			0xa0, 0xfb, 0x19, 0x20, 0x15, 0x8c, 0xd9, 0xcb,
			0x96, 0xc0, 0x52, 0x5b, 0x35, 0x35, 0x41, 0xe9,
			0xb7, 0x67, 0xba, 0x8b, 0xeb, 0xbe, 0x1a, 0x78,
			0x3b, 0xcc, 0x3e, 0x6d, 0x81, 0xcf, 0xf2, 0x4f,
			0x1c, 0x7e, 0xac, 0x11, 0x42, 0x02, 0x7b, 0xd8,
			0x97, 0x42, 0x3a, 0x5d, 0xff, 0x63, 0x09, 0x93,
			0xb1, 0x8f, 0xfe, 0x8b, 0x8e, 0xf6, 0x17, 0x94,
		},
		/* S1 : 512 byte */
		{
			0x9c, 0x1b, 0xe5, 0xe8, 0x57, 0xdd, 0xa5, 0x8a,
			0xc5, 0x4a, 0xd9, 0xa1, 0x48, 0x03, 0x84, 0x07,
			0xeb, 0xea, 0x77, 0x86, 0x50, 0xfb, 0x48, 0xb9,
			0x34, 0x5f, 0xca, 0xc3, 0x0a, 0xc5, 0xd1, 0x7b,
			0xdd, 0xa8, 0x62, 0xbf, 0x38, 0x4c, 0x2e, 0xac,
			0xc9, 0x57, 0xa2, 0x0b, 0x52, 0x50, 0x76, 0x9a,
			0x3f, 0xe3, 0xd3, 0x2f, 0x0f, 0x1e, 0x3e, 0x27,
			0x57, 0x4b, 0x65, 0x7d, 0x35, 0x93, 0xc5, 0x10,
			0x00, 0xed, 0x36, 0x33, 0xf7, 0xfc, 0x6e, 0x2c,
			0x0e, 0xc7, 0xb4, 0xb1, 0x48, 0xcf, 0x34, 0xb2,
			0xf0, 0x1c, 0x80, 0xfe, 0xe1, 0xbf, 0xb2, 0x36,
			0x64, 0xed, 0x5e, 0x56, 0x78, 0xeb, 0xbb, 0x29,
			0x37, 0x92, 0x1e, 0x9f, 0x97, 0x64, 0x73, 0x32,
			0x55, 0x85, 0xba, 0x49, 0xa7, 0xd4, 0xaf, 0x05,
			0x5f, 0xb5, 0x8e, 0x35, 0x0a, 0x4d, 0x98, 0xa2,
			0xb9, 0xe4, 0xe0, 0x3f, 0x56, 0x1b, 0x09, 0x39,
			0x03, 0x58, 0xdf, 0x75, 0x8f, 0xcd, 0x3a, 0xf7,
			0xaa, 0xac, 0xcd, 0xbe, 0xee, 0x96, 0xa7, 0x55,
			0x28, 0x77, 0xc0, 0x89, 0x0a, 0xe4, 0x35, 0x93,
			0x4d, 0x47, 0xcd, 0xa9, 0x81, 0xad, 0xa6, 0x06,
			0xbb, 0xe7, 0x67, 0xc7, 0xf6, 0xa3, 0x90, 0x4d,
			0x71, 0x32, 0xec, 0x59, 0x06, 0xed, 0xbb, 0x3d,
			0x2f, 0x76, 0x6f, 0x96, 0x3a, 0xac, 0x2f, 0xf6,
			0xdb, 0xd4, 0x99, 0x86, 0xd3, 0xa5, 0x50, 0x76,
			0x66, 0x11, 0x03, 0x92, 0xfc, 0x84, 0xbb, 0x82,
			0xb4, 0x0a, 0xd0, 0xdf, 0x6b, 0x93, 0x60, 0x59,
			0xb9, 0x04, 0x5e, 0xb2, 0xc4, 0xa0, 0xcd, 0x4b,
			0xba, 0xc7, 0x1e, 0x35, 0xb0, 0xf7, 0x99, 0x4c,
			0x1d, 0xef, 0x66, 0x4b, 0x7a, 0xf8, 0x4b, 0xa5,
			0xe6, 0xcc, 0xfa, 0xc1, 0x42, 0x4b, 0xe3, 0x10,
			0x2e, 0x15, 0xd9, 0x85, 0x89, 0x0a, 0xe4, 0x4f,
			0xb0, 0x6b, 0x1b, 0x7f, 0x45, 0xca, 0x18, 0xf2,
			0x00, 0xc8, 0xac, 0xb9, 0xd9, 0x8b, 0xa1, 0x14,
			0xa0, 0x50, 0x91, 0x2b, 0x70, 0x3f, 0x3c, 0xf9,
			0x9e, 0xd1, 0x65, 0x5b, 0x5f, 0x63, 0x1b, 0x26,
			0xfb, 0x75, 0xd8, 0xdb, 0x6d, 0xcb, 0xe1, 0x45,
			0xd9, 0xaa, 0xdb, 0x79, 0x03, 0x21, 0x7e, 0xf2,
			0xbd, 0x97, 0xfa, 0xb5, 0xa0, 0x94, 0x8f, 0xec,
			0xd8, 0xfc, 0xd2, 0x2b, 0x36, 0xff, 0x8a, 0xc0,
			0xd0, 0x4d, 0x8f, 0xf1, 0x7d, 0x51, 0x8b, 0xc2,
			0x28, 0x6a, 0x80, 0xe3, 0x62, 0xbf, 0xde, 0xcc,
			0x41, 0xae, 0x40, 0x54, 0xed, 0x9b, 0x69, 0x4a,
			0xf8, 0x31, 0x3a, 0x2f, 0xf8, 0xd9, 0x09, 0xd5,
			0x0c, 0xf1, 0x40, 0x94, 0x2d, 0xa5, 0xe7, 0x7f,
			0x85, 0x1e, 0x98, 0x8e, 0xa5, 0xee, 0xb6, 0x5b,
			0xfe, 0x6b, 0xd7, 0x93, 0xed, 0xaa, 0x6a, 0xdd,
			0x63, 0x62, 0xdd, 0x4a, 0x62, 0xf6, 0x39, 0x5a,
			0xf8, 0x88, 0x08, 0x51, 0x44, 0xef, 0xa2, 0x53,
			0xdd, 0x2a, 0xc4, 0x38, 0x29, 0x72, 0xed, 0x40,
			0xb8, 0x1f, 0x06, 0xaa, 0x99, 0x4e, 0x4d, 0x02,
			0x00, 0x0d, 0x55, 0x8e, 0x39, 0x93, 0xb2, 0x8d,
			0x77, 0xaa, 0x66, 0x14, 0x1b, 0x7b, 0xbb, 0xbe,
			0x4a, 0x16, 0x2b, 0xb4, 0xf4, 0x9d, 0xd5, 0xd3,
			0x82, 0x7d, 0x75, 0x0c, 0xa7, 0x9a, 0x2b, 0xb9,
			0xfc, 0x35, 0x0c, 0x75, 0x78, 0x90, 0xd3, 0xe8,
			0x0b, 0x6d, 0x3d, 0xb1, 0x02, 0xff, 0xa2, 0x01,
			0xd7, 0x90, 0x06, 0xc5, 0xc7, 0x5a, 0xa8, 0x3b,
			0x71, 0x0e, 0xc3, 0x21, 0xa2, 0xee, 0x27, 0x11,
			0x45, 0x11, 0x95, 0x24, 0x8d, 0x5c, 0x3e, 0x7d,
			0x3b, 0x3a, 0xac, 0x26, 0xa6, 0x71, 0x99, 0xa4,
			0xe8, 0x59, 0x52, 0xb5, 0x90, 0x59, 0x58, 0x37,
			0x7a, 0x88, 0xb2, 0x0d, 0xa1, 0x6f, 0xda, 0x98,
			0x6d, 0x50, 0xbe, 0x86, 0x23, 0x49, 0x1e, 0x32,
			0xa0, 0x7b, 0xd1, 0x8a, 0xd8, 0x9d, 0xe4, 0xc6,
		},
	},
	/* SHA512 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x56, 0xb1, 0x1c, 0x94, 0x74, 0xa8, 0x86, 0xb0,
			0x54, 0xae, 0xc0, 0x59, 0x97, 0x71, 0x16, 0xb8,
			0x8d, 0x8e, 0xff, 0x7e, 0xfb, 0x49, 0xfe, 0x30,
			0x1b, 0x82, 0xc6, 0xc2, 0xe5, 0xe5, 0xa1, 0x68,
			0x8d, 0x60, 0x12, 0xa9, 0x4d, 0xda, 0xb9, 0xb5,
			0x64, 0x22, 0x78, 0xf8, 0x5f, 0x25, 0xd4, 0xef,
			0x97, 0x65, 0x9e, 0x3b, 0xfb, 0xa6, 0xd2, 0xf4,
			0x0f, 0xea, 0xbf, 0x3f, 0x35, 0xa0, 0x68, 0x23,
			0x2c, 0x19, 0x2b, 0x98, 0xd6, 0x3b, 0xa0, 0xe1,
			0x3e, 0x79, 0xc4, 0x1c, 0xd9, 0x03, 0x57, 0x18,
			0x0c, 0x39, 0x82, 0x66, 0xef, 0x66, 0xbd, 0xb3,
			0x51, 0xab, 0xf0, 0x55, 0xfc, 0x39, 0xd8, 0x23,
			0x39, 0xc3, 0xad, 0x8a, 0x97, 0x34, 0x02, 0xdd,
			0xe9, 0x9d, 0xec, 0xed, 0x8e, 0x71, 0x66, 0xda,
			0xf5, 0xf4, 0xf4, 0x28, 0x5f, 0xf1, 0x87, 0x15,
			0xe7, 0xab, 0xa1, 0x2a, 0xc0, 0x17, 0xb9, 0x11,
		},
		/* S2 : 512 byte */
		{
			0xc8, 0xd6, 0x6b, 0xf1, 0xf9, 0x02, 0x37, 0xd9,
			0x3b, 0x17, 0x46, 0x03, 0xc0, 0x1d, 0x79, 0x96,
			0x35, 0x7a, 0xfc, 0x51, 0xab, 0x96, 0x5e, 0x24,
			0x76, 0x64, 0xb3, 0x3c, 0xaa, 0x66, 0x2b, 0x56,
			0x7f, 0xbc, 0xed, 0x0b, 0x66, 0xae, 0x00, 0xf8,
			0x11, 0x68, 0x67, 0xb0, 0xd3, 0x66, 0x47, 0x70,
			0x50, 0x9a, 0x7a, 0x21, 0x92, 0xc5, 0xbf, 0x5c,
			0x38, 0x32, 0xd0, 0xa0, 0xa1, 0x9f, 0xb3, 0xc8,
			0x03, 0xa3, 0xcb, 0xe7, 0x93, 0x33, 0x46, 0xf9,
			0x2e, 0x9b, 0x57, 0x92, 0x65, 0x87, 0x09, 0xa9,
			0x2d, 0xbb, 0xb8, 0xd9, 0x28, 0xed, 0x31, 0x4b,
			0x5a, 0x27, 0xda, 0xee, 0x92, 0xc8, 0xd9, 0x27,
			0x90, 0x9a, 0x15, 0xb6, 0xc8, 0x05, 0xad, 0x87,
			0xaf, 0xa3, 0xa8, 0xd6, 0x66, 0xea, 0xc3, 0x49,
			0x6a, 0xd8, 0x1d, 0x32, 0x50, 0xb8, 0x25, 0x38,
			0x0e, 0x20, 0x30, 0x84, 0xee, 0xf1, 0x09, 0x8f,
			0xda, 0xab, 0xa3, 0x64, 0x59, 0x44, 0x18, 0x71,
			0x4e, 0x72, 0x76, 0x54, 0x4f, 0x24, 0x4c, 0xb7,
			0x55, 0xcf, 0x09, 0x5a, 0x0a, 0x1a, 0x08, 0x52,
			0x89, 0xfb, 0x97, 0xef, 0x86, 0x0e, 0x2d, 0xea,
			0xbd, 0x9d, 0x83, 0x22, 0xc2, 0xf2, 0x44, 0x5c,
			0xbb, 0xc6, 0xe1, 0x62, 0xbe, 0x34, 0x79, 0x98,
			0xd1, 0xfc, 0x86, 0x76, 0x7b, 0xfb, 0x4a, 0xc1,
			0x7a, 0xbf, 0x33, 0xe5, 0x4d, 0x22, 0x2c, 0x05,
			0x4a, 0x3f, 0x6b, 0x33, 0x9b, 0x01, 0xfb, 0x82,
			0x66, 0x94, 0x01, 0x6d, 0x8a, 0x50, 0xf2, 0x73,
			0x9a, 0x87, 0x74, 0x22, 0x3b, 0x4a, 0x68, 0x81,
			0x98, 0xe3, 0x71, 0x45, 0x57, 0x14, 0xf4, 0xdb,
			0xf9, 0x8c, 0x0d, 0x5e, 0xdc, 0x45, 0x86, 0x2c,
			0xcd, 0xb1, 0xae, 0xf6, 0xd8, 0x6f, 0x0e, 0xc4,
			0xc6, 0xe7, 0xac, 0xc1, 0x06, 0xbf, 0xe8, 0x56,
			0x5a, 0xb2, 0x11, 0xa7, 0xae, 0x3f, 0x80, 0x6c,
			0xe9, 0x8f, 0xcc, 0xc8, 0xb3, 0xe1, 0x5d, 0xd9,
			0x37, 0x22, 0x8f, 0xb7, 0x5a, 0xa9, 0x8a, 0x52,
			0x59, 0x7a, 0x8b, 0xff, 0xee, 0x05, 0xa4, 0xea,
			0xfc, 0x92, 0x97, 0x9a, 0x2f, 0xdc, 0x00, 0xe2,
			0x34, 0x74, 0x95, 0x7d, 0x22, 0x75, 0xf1, 0x09,
			0x40, 0xfe, 0x2f, 0xaa, 0x05, 0x64, 0xfb, 0x18,
			0x7d, 0xac, 0x74, 0xc3, 0xde, 0xdc, 0x66, 0x66,
			0xa8, 0xf1, 0x84, 0xc4, 0xb0, 0xbf, 0xa5, 0x91,
			0x98, 0xd1, 0xa4, 0xb0, 0xb4, 0x6b, 0xae, 0x75,
			0x28, 0x2c, 0x4a, 0x87, 0x06, 0xe2, 0x0f, 0x49,
			0xe6, 0x35, 0x12, 0x3c, 0x58, 0x0d, 0xac, 0xf6,
			0x9b, 0x28, 0x5a, 0x46, 0xbe, 0xbd, 0xb7, 0xdc,
			0x2e, 0x63, 0x22, 0x37, 0x32, 0xf8, 0xbd, 0x1a,
			0xa1, 0xee, 0x9e, 0xbf, 0x97, 0x9a, 0x95, 0x8d,
			0x19, 0xc4, 0x6c, 0xd8, 0xb0, 0xe9, 0x9e, 0xd3,
			0x71, 0x35, 0xb1, 0x6a, 0x6e, 0x95, 0xb8, 0xe4,
			0x78, 0xde, 0x5b, 0x57, 0x7a, 0xfa, 0x91, 0x8d,
			0x12, 0x8f, 0xa0, 0x12, 0xbf, 0x13, 0x91, 0x52,
			0x28, 0xd4, 0x88, 0x2f, 0x75, 0x47, 0xf7, 0x08,
			0x95, 0x25, 0x49, 0x44, 0x84, 0x1c, 0x75, 0xc8,
			0x5a, 0x55, 0x59, 0xac, 0x94, 0x46, 0xc7, 0x5d,
			0x6d, 0xf8, 0xf9, 0x23, 0x49, 0xf6, 0x92, 0xf7,
			0xa4, 0x19, 0x35, 0xe9, 0x6d, 0xd9, 0x00, 0x6d,
			0xae, 0xa1, 0x7c, 0x36, 0xa0, 0x6a, 0x0b, 0x01,
			0x0a, 0x48, 0xde, 0xb1, 0x84, 0xfd, 0x4f, 0xdf,
			0xcd, 0x31, 0x76, 0x2e, 0x7f, 0x1d, 0x54, 0x0e,
			0x0e, 0x04, 0x05, 0xf8, 0x5d, 0xe9, 0x5a, 0x03,
			0x9d, 0x3b, 0xd1, 0x21, 0x76, 0xa7, 0x5a, 0x0d,
			0x77, 0x8e, 0x00, 0xe5, 0x3d, 0x37, 0xfc, 0x47,
			0x58, 0x11, 0x70, 0xf5, 0xae, 0xd3, 0xbd, 0x44,
			0xdf, 0xaa, 0x2b, 0x70, 0x5e, 0x91, 0x32, 0xe9,
			0x6f, 0x13, 0xa7, 0xd3, 0xf7, 0x92, 0xea, 0xa0,
		},
	},
	/* SHA512 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0x6e, 0x5e, 0x78, 0x9c, 0x4b, 0x97, 0x96, 0x22,
			0x50, 0xff, 0x3a, 0xe8, 0x31, 0x0b, 0x52, 0x2b,
			0x03, 0x06, 0x4e, 0xb1, 0x45, 0x05, 0x3d, 0x5c,
			0x20, 0x1e, 0x32, 0xfe, 0xee, 0xd5, 0xed, 0x6f,
			0xfa, 0xd7, 0xb7, 0xdd, 0x86, 0xeb, 0x8e, 0x64,
			0x13, 0x25, 0x82, 0xde, 0xdc, 0x5c, 0x5f, 0xfd,
			0xa4, 0xdf, 0x8c, 0x97, 0xb1, 0x64, 0x33, 0x40,
			0x19, 0x41, 0xa2, 0x1e, 0x3c, 0xdf, 0xf2, 0xf9,
			0x92, 0x6b, 0xe6, 0x92, 0xa7, 0xce, 0x15, 0x36,
			0x63, 0xe0, 0x4c, 0x92, 0x8f, 0xd8, 0x2e, 0xc9,
			0x95, 0x08, 0x1d, 0xc4, 0x87, 0xc7, 0x5e, 0xca,
			0x63, 0xae, 0x77, 0x50, 0x96, 0x07, 0xdc, 0x12,
			0xbe, 0x82, 0xcb, 0x62, 0xb4, 0x2a, 0x75, 0xc0,
			0xca, 0x98, 0x5e, 0xac, 0x51, 0x66, 0x06, 0xb8,
			0x5f, 0xe7, 0xc9, 0xe1, 0xcf, 0x15, 0x04, 0x1f,
			0x88, 0xcb, 0x79, 0x3b, 0x03, 0x35, 0xf5, 0xe1,
		},
		/* S3 : 512 byte */
		{
			0xa0, 0x53, 0x68, 0x25, 0x6b, 0xa6, 0x3f, 0x31,
			0xc4, 0x31, 0x7f, 0x27, 0x68, 0xf8, 0xa5, 0xb4,
			0x90, 0xc0, 0x44, 0xb2, 0xee, 0x15, 0xfe, 0xc7,
			0x8a, 0x0a, 0x85, 0xd4, 0xf7, 0xc1, 0xa9, 0x88,
			0xa5, 0xc6, 0xe9, 0x46, 0x04, 0x0b, 0xf2, 0xd8,
			0x62, 0xe9, 0x70, 0xdf, 0xe3, 0x7b, 0x67, 0xf0,
			0x22, 0x1f, 0x85, 0xb0, 0x4b, 0x9e, 0x60, 0x72,
			0x45, 0xbd, 0x0b, 0x21, 0xce, 0x5b, 0xda, 0xf8,
			0xb1, 0x38, 0xbb, 0x45, 0x12, 0x05, 0xfc, 0x38,
			0xb6, 0x0b, 0x94, 0x28, 0x81, 0xff, 0x20, 0xce,
			0x07, 0x7f, 0x2a, 0xa0, 0x70, 0xe4, 0x8b, 0xf7,
			0x1a, 0x21, 0xce, 0x8a, 0x18, 0xc8, 0xb4, 0x8f,
			0x09, 0xbf, 0x59, 0x37, 0x22, 0x0e, 0x8f, 0x0e,
			0xd0, 0xd6, 0xcc, 0x2d, 0xdf, 0xf2, 0xcd, 0xa1,
			0x95, 0xe0, 0x41, 0xee, 0x31, 0x9b, 0x43, 0x21,
			0x08, 0x34, 0xec, 0xe6, 0xd9, 0x03, 0x02, 0x37,
			0xdf, 0x5b, 0xea, 0x8d, 0x93, 0xa8, 0x1c, 0x4d,
			0x62, 0xf7, 0xff, 0x48, 0xbc, 0x87, 0x26, 0xa3,
			0xe3, 0x2f, 0xe7, 0xb6, 0x34, 0x50, 0xd2, 0xd7,
			0xbb, 0x56, 0x4a, 0x8a, 0xc8, 0x0c, 0x89, 0xb5,
			0x88, 0x34, 0x8b, 0xeb, 0xba, 0xa4, 0x3c, 0x9a,
			0x62, 0x34, 0x1b, 0x19, 0x4b, 0x4c, 0xf3, 0xf4,
			0x4c, 0xf2, 0xd7, 0xa7, 0xdd, 0x8e, 0xea, 0xc7,
			0x7c, 0x86, 0x78, 0x94, 0xac, 0xd9, 0xc9, 0xaa,
			0x34, 0xea, 0xc0, 0x2e, 0x7e, 0x81, 0xea, 0xc0,
			0x63, 0x89, 0x33, 0xce, 0x19, 0x91, 0x3d, 0x7a,
			0x0f, 0x6a, 0xb9, 0xc8, 0x9d, 0x5f, 0xfe, 0x74,
			0xa9, 0xe7, 0x79, 0xa8, 0x0c, 0xb9, 0x48, 0x80,
			0x5b, 0x37, 0xa0, 0xf9, 0x52, 0xd3, 0xbc, 0x9b,
			0x50, 0x8a, 0xf0, 0xcf, 0x9d, 0x0c, 0x7d, 0x92,
			0x76, 0xb2, 0x93, 0x58, 0xbd, 0xe0, 0x46, 0xd6,
			0x5a, 0x66, 0x4b, 0x48, 0xaf, 0xf1, 0xd0, 0xb6,
			0x41, 0xa7, 0x89, 0xb5, 0xd5, 0xf9, 0xd0, 0xbf,
			0x66, 0x1a, 0xe6, 0x11, 0xd1, 0x3c, 0xe7, 0x15,
			0x3b, 0x84, 0x9c, 0xe0, 0x71, 0x62, 0x8e, 0x9f,
			0x13, 0xc1, 0xed, 0xb0, 0x92, 0x16, 0x9b, 0x26,
			0xf1, 0x3b, 0x3a, 0x6f, 0xb0, 0x92, 0x9f, 0xa4,
			0xa0, 0x33, 0xeb, 0xd3, 0x45, 0x65, 0x50, 0x22,
			0xcf, 0x39, 0x82, 0xab, 0x22, 0x30, 0x1a, 0x47,
			0x2b, 0x3d, 0xda, 0xda, 0x69, 0xdf, 0xf2, 0xd5,
			0x1d, 0xef, 0xdf, 0xd7, 0x2c, 0x91, 0xb4, 0xd3,
			0xdb, 0xa9, 0xd9, 0xa8, 0x80, 0x1f, 0x03, 0xcf,
			0xab, 0x64, 0xfd, 0x0e, 0x63, 0xff, 0xe2, 0x95,
			0x2b, 0xc4, 0x2e, 0xe1, 0x02, 0x53, 0xd8, 0x2d,
			0x70, 0x5e, 0x04, 0x09, 0x79, 0x54, 0xbc, 0x62,
			0x01, 0xc5, 0x7a, 0xb0, 0xd1, 0xe0, 0xc6, 0x05,
			0x91, 0x87, 0x5f, 0xe2, 0x40, 0x7c, 0xfb, 0x1d,
			0x67, 0x17, 0xde, 0x1c, 0x44, 0x2c, 0xab, 0xe2,
			0xf4, 0xe8, 0x1e, 0x93, 0x99, 0x47, 0xb4, 0x86,
			0x04, 0x4c, 0x02, 0x44, 0xf0, 0x46, 0x60, 0xda,
			0x55, 0x33, 0xfb, 0x4e, 0x8c, 0x25, 0x68, 0xbd,
			0xa7, 0xe5, 0x04, 0xff, 0x7b, 0x3a, 0x0e, 0x21,
			0x7d, 0x83, 0xa6, 0xdf, 0xa3, 0xbf, 0x23, 0x96,
			0xa1, 0xce, 0x92, 0x22, 0x5b, 0x8d, 0x83, 0x74,
			0xf2, 0xc6, 0x7f, 0x5a, 0xd9, 0x44, 0xe4, 0xcd,
			0xa7, 0x64, 0x10, 0x60, 0xd0, 0xae, 0x3e, 0x6e,
			0x45, 0x93, 0x25, 0xbf, 0xe6, 0x04, 0x7b, 0x5f,
			0x63, 0x96, 0x9d, 0xe5, 0xa0, 0x00, 0x4c, 0x5d,
			0x9a, 0x18, 0x30, 0x1f, 0x98, 0x62, 0xa9, 0x98,
			0x8e, 0x9d, 0x1a, 0x75, 0x07, 0xb3, 0xd0, 0x0e,
			0x75, 0x20, 0x71, 0x7d, 0x69, 0x3d, 0xb5, 0x54,
			0x4e, 0xdd, 0x83, 0x41, 0x41, 0x3b, 0x03, 0xe4,
			0xa6, 0x07, 0x2e, 0xfb, 0x80, 0x53, 0xe6, 0xff,
			0x0a, 0x2a, 0xad, 0x3e, 0x35, 0x1d, 0x3d, 0x04,
		},
	},
	/* SHA512 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0x80, 0x63, 0x63, 0xd4, 0xe2, 0x89, 0x4f, 0x3d,
			0x68, 0xf3, 0x5c, 0x78, 0xcd, 0xc3, 0x32, 0x63,
			0x07, 0x63, 0xa6, 0x4a, 0xc2, 0x23, 0xd5, 0xad,
			0x0c, 0x24, 0x6f, 0x44, 0x52, 0x76, 0x56, 0x41,
			0x8b, 0x7d, 0x68, 0xd5, 0xb3, 0x99, 0x98, 0xf5,
			0x7b, 0x05, 0x44, 0x5e, 0x3f, 0x00, 0xc3, 0xd7,
			0x38, 0x2b, 0x3d, 0xb6, 0xf3, 0x33, 0xbf, 0xa5,
			0x01, 0xaf, 0x8e, 0xa6, 0xf1, 0x4c, 0xbc, 0x55,
			0x7b, 0x27, 0xfa, 0x6d, 0xfe, 0x7a, 0xad, 0x8e,
			0x2d, 0x7b, 0x44, 0x3e, 0x04, 0x81, 0xc8, 0x2d,
			0x02, 0x69, 0xb4, 0xbd, 0x92, 0x38, 0x08, 0xc1,
			0xca, 0x01, 0x99, 0x85, 0x57, 0x0a, 0xab, 0x10,
			0xb8, 0x2a, 0xc1, 0xa5, 0xab, 0x75, 0xb4, 0x90,
			0xe5, 0x1c, 0x03, 0x1f, 0x06, 0x8e, 0x6a, 0xee,
			0xcc, 0xe3, 0xb6, 0x67, 0x87, 0x7b, 0xd7, 0x8a,
			0xcd, 0xe2, 0x37, 0xea, 0x6e, 0xf7, 0x4a, 0xf8,
		},
		/* S4 : 512 byte */
		{
			0x75, 0x91, 0xa4, 0xf6, 0x20, 0xdc, 0xf5, 0xa4,
			0x33, 0x10, 0xaf, 0x89, 0xf8, 0xa8, 0xd7, 0xde,
			0xd9, 0x42, 0x94, 0xd2, 0x04, 0xdc, 0xb1, 0x59,
			0xee, 0xdd, 0xcb, 0xcb, 0x1c, 0x7e, 0xa6, 0x67,
			0x51, 0x01, 0xd4, 0x08, 0x46, 0x4d, 0x0e, 0x4f,
			0x8a, 0xbf, 0x3c, 0x50, 0xa8, 0x2b, 0x3d, 0x09,
			0x24, 0x30, 0x46, 0x7b, 0xd0, 0x82, 0x92, 0xb5,
			0x7b, 0xa9, 0x87, 0x02, 0x20, 0xf5, 0x3d, 0x3b,
			0x6a, 0xc9, 0x89, 0xad, 0x5e, 0xd3, 0x24, 0x9d,
			0x23, 0x7b, 0xbc, 0x76, 0xb6, 0x47, 0x9f, 0x8c,
			0x45, 0xc2, 0xbc, 0x9c, 0x25, 0x46, 0x0c, 0x54,
			0xc9, 0x13, 0xc2, 0x42, 0xc0, 0x93, 0x22, 0x11,
			0xdc, 0x4a, 0xbf, 0xe2, 0x06, 0xdc, 0xe6, 0x30,
			0x2c, 0x03, 0x38, 0x85, 0xe3, 0x23, 0x66, 0xba,
			0xea, 0x3a, 0x6b, 0x2e, 0x15, 0x82, 0xbc, 0xcb,
			0xfb, 0xc9, 0xe4, 0x58, 0x82, 0xab, 0x58, 0x42,
			0x5b, 0x17, 0x44, 0x9a, 0x53, 0xe5, 0xe0, 0xb1,
			0x0d, 0x12, 0x87, 0xc5, 0x8a, 0xc2, 0xc4, 0x44,
			0x24, 0xf4, 0x18, 0x8b, 0xfe, 0x1d, 0xe4, 0x98,
			0xe2, 0x9b, 0xa2, 0xa1, 0xf2, 0xa8, 0xf2, 0x8e,
			0x56, 0x5b, 0x99, 0x79, 0x46, 0xfb, 0x35, 0x9b,
			0x7e, 0x32, 0x0e, 0x4d, 0xa6, 0xee, 0x64, 0xe4,
			0x85, 0x47, 0xb1, 0xb7, 0x94, 0x69, 0x19, 0xe9,
			0x1f, 0xac, 0x8f, 0x4f, 0x63, 0x1a, 0x8d, 0xe9,
			0xc1, 0x02, 0x9e, 0x87, 0xc0, 0x50, 0x3c, 0xb1,
			0x8a, 0x43, 0xa8, 0xa8, 0xf5, 0xd9, 0xc3, 0xdc,
			0x59, 0xd1, 0x0a, 0x41, 0xac, 0x65, 0xd4, 0xd9,
			0x3c, 0x1e, 0x7f, 0x5c, 0x0a, 0xc2, 0x7e, 0xa2,
			0xff, 0xec, 0xc1, 0x78, 0x81, 0xff, 0xb6, 0xd6,
			0x3e, 0x4a, 0xfe, 0x8b, 0x9c, 0x94, 0x61, 0x06,
			0x61, 0xb3, 0x0c, 0xe3, 0x4d, 0x4e, 0xc7, 0x32,
			0xb9, 0xcf, 0x88, 0x6b, 0x08, 0xf6, 0xf3, 0x8e,
			0xa9, 0xaa, 0x09, 0x41, 0x88, 0x06, 0x7b, 0x65,
			0x66, 0x57, 0xea, 0xb2, 0x2f, 0xc9, 0x0e, 0x23,
			0x4e, 0x64, 0xe3, 0x1d, 0xd0, 0x55, 0x8d, 0xe5,
			0x89, 0x1c, 0xd4, 0x9c, 0xde, 0x67, 0xc5, 0xd6,
			0xb1, 0xe3, 0x07, 0xe2, 0x3a, 0x35, 0x22, 0x6b,
			0xa0, 0x96, 0x82, 0xf1, 0x94, 0xee, 0xa0, 0x63,
			0x25, 0xc9, 0xe0, 0xa7, 0x33, 0xc9, 0x75, 0x8b,
			0xca, 0x95, 0x1c, 0xcc, 0xf5, 0x73, 0x0b, 0x71,
			0x98, 0x7e, 0xf5, 0x64, 0xf9, 0x20, 0x57, 0xa1,
			0x98, 0x5e, 0x45, 0xd3, 0xfe, 0x20, 0xbe, 0xc7,
			0xd9, 0x6a, 0xce, 0xea, 0xac, 0x03, 0x1c, 0xaa,
			0xf4, 0x50, 0x4d, 0xfc, 0xfd, 0x8c, 0x7c, 0x12,
			0xb9, 0xa5, 0x80, 0x84, 0x48, 0x51, 0xdd, 0x54,
			0xae, 0xda, 0xd3, 0x07, 0xb7, 0xa6, 0xfc, 0x79,
			0xe8, 0x7a, 0x94, 0x31, 0xfa, 0xff, 0xc9, 0x35,
			0x3e, 0x49, 0x78, 0x20, 0x69, 0x7e, 0x76, 0x31,
			0x60, 0x0c, 0x7c, 0xfa, 0xf8, 0x51, 0xf8, 0x44,
			0x98, 0x1c, 0xdd, 0x2b, 0x20, 0x48, 0xc7, 0x80,
			0x44, 0xe0, 0xaf, 0x69, 0x9d, 0x8b, 0x0b, 0x61,
			0x4c, 0xda, 0x9a, 0x30, 0x32, 0xf8, 0x34, 0x11,
			0x2f, 0x2a, 0xc1, 0x7b, 0xf8, 0x63, 0x63, 0x14,
			0xc1, 0xc2, 0xc5, 0xd2, 0xc3, 0x86, 0x06, 0x12,
			0x7f, 0x57, 0xd2, 0xff, 0xe8, 0xad, 0x7a, 0x33,
			0xcb, 0x02, 0xab, 0xae, 0x23, 0xc2, 0xeb, 0x3f,
			0x3b, 0x08, 0x52, 0x82, 0x66, 0x91, 0x72, 0x9b,
			0x16, 0xcf, 0xc6, 0xb2, 0xcc, 0x45, 0xd4, 0x17,
			0x77, 0x82, 0xed, 0x1c, 0x1d, 0xe6, 0x17, 0x68,
			0xb4, 0x36, 0xd5, 0xb8, 0xfa, 0x65, 0x01, 0xcf,
			0xa9, 0x7e, 0xc6, 0x5f, 0x54, 0x26, 0xb6, 0x3e,
			0x9e, 0x82, 0xec, 0x37, 0x80, 0xcc, 0x5f, 0xe0,
			0xce, 0xfe, 0xf8, 0xf3, 0x95, 0xa4, 0x93, 0x5d,
			0xa3, 0x9c, 0xff, 0xde, 0xd6, 0x5f, 0x9d, 0x37,
		},
	},
	/* SHA512 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0x1c, 0x8b, 0x88, 0xe3, 0x4c, 0x54, 0x69, 0xce,
			0x1c, 0x6f, 0xb2, 0x62, 0x5b, 0x02, 0x47, 0xfc,
			0xc0, 0x79, 0x30, 0x97, 0x2c, 0xc0, 0xfa, 0x9d,
			0x30, 0xb5, 0x5a, 0xfc, 0xcf, 0xdd, 0x14, 0x7d,
			0x85, 0x55, 0x1b, 0x52, 0x88, 0x0c, 0xfa, 0xed,
			0x4d, 0x30, 0x72, 0x19, 0x4e, 0xe6, 0x65, 0x43,
			0x0b, 0x3e, 0xf3, 0x18, 0x4e, 0x66, 0xe2, 0x56,
			0xb4, 0x1e, 0x43, 0xdb, 0x69, 0x6b, 0x52, 0x03,
			0x33, 0x90, 0x9f, 0x30, 0xb2, 0x3a, 0x6b, 0xaa,
			0xe6, 0xfb, 0x55, 0xaa, 0xf1, 0xf9, 0x36, 0xb1,
			0x1e, 0xe8, 0x49, 0x1e, 0x23, 0x35, 0x8f, 0x00,
			0xa3, 0x84, 0x71, 0x29, 0xf7, 0x5d, 0xa8, 0x84,
			0x2e, 0x22, 0x5a, 0xa8, 0x52, 0x45, 0x85, 0xac,
			0xed, 0xb7, 0xa1, 0x3d, 0xcc, 0xa4, 0x81, 0xef,
			0x03, 0x5d, 0x7b, 0xd1, 0x32, 0x96, 0xc8, 0x44,
			0x06, 0xd1, 0x2d, 0x0d, 0x02, 0x1a, 0xdb, 0xa8,
		},
		/* S5 : 512 byte */
		{
			0x9e, 0xb1, 0x0c, 0x1a, 0x31, 0x06, 0x6c, 0x3e,
			0xda, 0x46, 0xd2, 0xa6, 0x10, 0x2a, 0xf5, 0x23,
			0x2b, 0x92, 0xa9, 0xe1, 0x64, 0xc2, 0x79, 0x23,
			0x03, 0x45, 0x4c, 0x9c, 0x99, 0x19, 0xd2, 0xeb,
			0x34, 0xa8, 0x96, 0x01, 0x0c, 0xf2, 0x23, 0x0f,
			0x81, 0x38, 0xbc, 0xa6, 0x86, 0x9b, 0x46, 0x25,
			0xe9, 0x5c, 0x92, 0xcf, 0x29, 0x20, 0xb8, 0xc8,
			0x97, 0x74, 0x9a, 0x15, 0x86, 0xe9, 0xc4, 0xe2,
			0x4e, 0x7a, 0x97, 0x5f, 0x44, 0x96, 0x82, 0x16,
			0xb4, 0xee, 0x88, 0xa3, 0x98, 0xe4, 0x9e, 0x3a,
			0x23, 0x4d, 0xc3, 0x2f, 0x16, 0xff, 0x99, 0x8b,
			0x8c, 0xab, 0x80, 0x5e, 0xe9, 0x87, 0x6b, 0x2a,
			0xc6, 0x39, 0x0a, 0xbf, 0x53, 0xe6, 0xbc, 0xeb,
			0x2a, 0x8b, 0x1e, 0x6c, 0x91, 0xa5, 0x50, 0x58,
			0x42, 0x08, 0xb5, 0x3c, 0x23, 0xa1, 0x3a, 0x8d,
			0x57, 0x39, 0x34, 0x37, 0x40, 0xfc, 0x66, 0x7a,
			0x8c, 0x98, 0x83, 0x99, 0x3f, 0x9e, 0x77, 0xb2,
			0xbd, 0xcd, 0x81, 0x23, 0xf1, 0x38, 0x9b, 0xd7,
			0xb3, 0xec, 0x36, 0x3c, 0x5f, 0x84, 0xb5, 0xb6,
			0x5d, 0xc1, 0x83, 0x52, 0x9b, 0x19, 0xe2, 0x7c,
			0x3b, 0x77, 0xdb, 0x84, 0x68, 0xd9, 0x7a, 0xde,
			0xd8, 0xef, 0x87, 0x8a, 0x11, 0x2e, 0x0f, 0xd0,
			0x99, 0xad, 0x1a, 0x63, 0x4a, 0x3a, 0xe8, 0x49,
			0x72, 0xb9, 0x5d, 0xad, 0x1f, 0xa7, 0xc5, 0xc4,
			0x6a, 0xae, 0x1e, 0x6c, 0x5f, 0x2b, 0xf1, 0xb5,
			0x72, 0x65, 0xa2, 0xec, 0x4a, 0xba, 0x9c, 0xea,
			0x60, 0xc7, 0x38, 0x81, 0x0f, 0x62, 0x8a, 0xe0,
			0x76, 0xc3, 0x2d, 0x1d, 0x12, 0xcf, 0xac, 0x0a,
			0x27, 0x04, 0xaf, 0x40, 0x16, 0x20, 0x10, 0xac,
			0xbf, 0x81, 0x9b, 0xf2, 0x0c, 0x2a, 0x52, 0x82,
			0xd7, 0x6c, 0x05, 0xa0, 0x9e, 0xe5, 0x81, 0x2d,
			0xb5, 0x00, 0x7e, 0xe4, 0x39, 0x30, 0xde, 0xc8,
			0xd9, 0x28, 0xa9, 0x0d, 0x67, 0x5f, 0x42, 0x23,
			0xa7, 0x74, 0x91, 0xce, 0x2b, 0x39, 0x6b, 0xc0,
			0xa0, 0xd3, 0x3d, 0x35, 0x90, 0x51, 0xc9, 0x57,
			0xe1, 0x9e, 0x41, 0x1c, 0xb3, 0x57, 0x7e, 0x95,
			0x50, 0xf2, 0xe3, 0xce, 0x63, 0x79, 0x54, 0x55,
			0x12, 0x31, 0x08, 0x86, 0x58, 0xb1, 0x1b, 0x3e,
			0x92, 0x83, 0x5f, 0x44, 0xac, 0xf9, 0x2d, 0xf0,
			0xf0, 0x48, 0x1a, 0xa8, 0xb6, 0x52, 0x16, 0xc6,
			0x49, 0x20, 0x12, 0x84, 0xb8, 0x5c, 0xda, 0x54,
			0x4e, 0xb9, 0xc2, 0xf0, 0x85, 0x99, 0xf2, 0x57,
			0xad, 0xf3, 0xd3, 0xf1, 0x2f, 0xd3, 0x03, 0x2f,
			0x0a, 0xc8, 0x3c, 0xc5, 0xba, 0x94, 0x53, 0x7c,
			0x4a, 0x55, 0x95, 0x2e, 0x7d, 0xf2, 0x34, 0xe1,
			0x3e, 0x8d, 0x0c, 0x65, 0x13, 0x04, 0x6d, 0x7f,
			0xc5, 0xc7, 0xf2, 0xb2, 0xef, 0xbe, 0xcf, 0x4c,
			0x19, 0xf2, 0x07, 0x48, 0x6f, 0xdb, 0x59, 0x40,
			0xd3, 0x9b, 0x31, 0xe0, 0xdf, 0xc1, 0x71, 0x43,
			0xc7, 0x42, 0x06, 0xc5, 0x19, 0xf3, 0xe7, 0x99,
			0x60, 0x61, 0x22, 0x68, 0x86, 0x04, 0xc5, 0x8e,
			0xae, 0x05, 0xca, 0x3b, 0x5f, 0x2a, 0x7a, 0x8a,
			0x28, 0x0e, 0x71, 0x2e, 0x69, 0x68, 0xd4, 0xd2,
			0x25, 0xfa, 0xd8, 0x44, 0xdd, 0xa5, 0x55, 0x54,
			0x3d, 0x2e, 0x5f, 0xd9, 0xf8, 0x6a, 0xbe, 0xee,
			0x66, 0xa9, 0xb9, 0x8e, 0x26, 0xf9, 0x0d, 0xa6,
			0x31, 0x3d, 0xe0, 0x37, 0x6f, 0xfa, 0x47, 0x14,
			0xec, 0x1a, 0x82, 0xf7, 0x41, 0x32, 0xc1, 0xfa,
			0x60, 0xc6, 0xbc, 0x63, 0xd4, 0x8f, 0x6a, 0x31,
			0xd5, 0xbb, 0xd4, 0x9b, 0x8a, 0x66, 0xde, 0x5d,
			0xfe, 0xc9, 0x4b, 0x35, 0x4c, 0x60, 0x12, 0x3f,
			0x90, 0x63, 0xe8, 0x5a, 0xe5, 0x4e, 0x75, 0x95,
			0x71, 0x70, 0x11, 0x7f, 0x38, 0x4c, 0xb7, 0x2f,
			0xdd, 0x96, 0xc0, 0x48, 0x71, 0x22, 0x5d, 0x80,
		},
	},
	/* SHA512 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0xf7, 0x08, 0x07, 0x28, 0x54, 0x53, 0x41, 0x69,
			0x08, 0x17, 0x99, 0xb3, 0x71, 0x48, 0x03, 0xbe,
			0xb6, 0x7f, 0x84, 0x09, 0xae, 0x4d, 0x39, 0xcd,
			0xfe, 0xd0, 0x9a, 0x2f, 0x82, 0x83, 0x00, 0xb9,
			0xef, 0x5a, 0x65, 0x0b, 0xb6, 0x0f, 0x6b, 0xce,
			0x2b, 0xb6, 0xa0, 0xea, 0xe6, 0x98, 0x01, 0xf9,
			0x69, 0x3e, 0xbd, 0xfb, 0x04, 0xcc, 0x3b, 0x4b,
			0xf6, 0x91, 0x5d, 0xd2, 0xd5, 0xdb, 0x07, 0x7a,
			0x30, 0x10, 0xdb, 0xe5, 0x72, 0xfa, 0x47, 0x60,
			0x09, 0x63, 0xc2, 0x12, 0xc8, 0xe2, 0x52, 0x79,
			0x8a, 0xf4, 0x52, 0x18, 0x19, 0x4e, 0x72, 0x69,
			0x4e, 0x72, 0xff, 0x19, 0x79, 0x81, 0x65, 0x74,
			0x02, 0x50, 0xf2, 0x1f, 0x51, 0xbd, 0xdc, 0x05,
			0xec, 0x44, 0x86, 0x93, 0xdf, 0xcd, 0xff, 0x28,
			0x61, 0xc8, 0xcb, 0xc7, 0xb4, 0x7c, 0xe7, 0x10,
			0x4d, 0x9c, 0x07, 0x6e, 0x35, 0x1c, 0x22, 0x91,
		},
		/* S6 : 512 byte */
		{
			0x9a, 0x1e, 0x8c, 0x94, 0x59, 0x7a, 0x18, 0x6a,
			0xe7, 0x78, 0xeb, 0xb2, 0x47, 0x31, 0x82, 0xb3,
			0x2d, 0xbb, 0xbe, 0x57, 0x29, 0xa5, 0x74, 0xad,
			0x55, 0x34, 0x6c, 0x15, 0x6c, 0x5a, 0x37, 0x7a,
			0xc7, 0x30, 0x2e, 0x09, 0x85, 0x93, 0x68, 0x25,
			0x81, 0x39, 0xa4, 0x63, 0x12, 0xb1, 0x0a, 0xee,
			0x2c, 0xdf, 0xd5, 0x3f, 0x21, 0x3a, 0xb4, 0x2c,
			0x65, 0xff, 0x9b, 0x31, 0x7a, 0x14, 0x14, 0xa0,
			0x71, 0x88, 0x6b, 0x30, 0x66, 0xbc, 0xd3, 0x32,
			0x8a, 0xad, 0xe6, 0x0e, 0x8e, 0x6a, 0xfa, 0x5c,
			0x28, 0xd7, 0xd6, 0xed, 0x19, 0x30, 0x59, 0xc2,
			0xaf, 0x4c, 0x86, 0x17, 0xef, 0xdd, 0x8f, 0xbe,
			0xe7, 0xdd, 0x18, 0x0c, 0x92, 0x97, 0x78, 0x3f,
			0x56, 0x8f, 0x0f, 0xd1, 0x59, 0x04, 0x90, 0xd0,
			0x13, 0x2a, 0x87, 0xb2, 0x7c, 0x4b, 0x8b, 0xe1,
			0x88, 0xeb, 0x51, 0x3f, 0xbc, 0x12, 0xa4, 0x19,
			0xea, 0xc8, 0x94, 0xb2, 0x54, 0xb1, 0xb8, 0x62,
			0x25, 0x40, 0x47, 0xbf, 0x58, 0x2d, 0x2f, 0xaf,
			0x8c, 0x1c, 0xbb, 0xa9, 0x46, 0x5b, 0xe5, 0xe7,
			0x1b, 0x7a, 0xfd, 0x3e, 0x20, 0xbb, 0xbe, 0x93,
			0x3b, 0x04, 0x4b, 0x33, 0xe9, 0xf0, 0x6a, 0xa3,
			0x13, 0xbe, 0xb4, 0x55, 0x8d, 0x02, 0xff, 0xaa,
			0xfd, 0x48, 0x25, 0x22, 0x50, 0x37, 0x09, 0x8d,
			0x56, 0x7b, 0x11, 0x20, 0xc0, 0x42, 0xa4, 0xf5,
			0xdb, 0x63, 0x49, 0xd4, 0x2b, 0xf7, 0x10, 0xf0,
			0xad, 0xd3, 0xed, 0x28, 0x09, 0x1f, 0x0a, 0xfb,
			0x34, 0x69, 0x55, 0x67, 0xba, 0xa8, 0x17, 0x0a,
			0x83, 0x6f, 0xa0, 0xa9, 0x49, 0x9d, 0xe9, 0xf2,
			0x87, 0x70, 0x63, 0xb3, 0xe3, 0x4c, 0x92, 0xcb,
			0x72, 0x3e, 0x8a, 0x1b, 0x76, 0xaa, 0x8e, 0x26,
			0x56, 0x89, 0x38, 0xaa, 0x5e, 0x51, 0xe7, 0xd0,
			0xed, 0xf6, 0x66, 0x53, 0x5e, 0x52, 0x85, 0xba,
			0x1d, 0xc8, 0x3b, 0x59, 0xd3, 0xfb, 0x9a, 0x0e,
			0xe9, 0x69, 0xcf, 0x42, 0x69, 0xaa, 0x94, 0xf7,
			0x10, 0x2f, 0x52, 0x08, 0x0f, 0xf3, 0x98, 0x5a,
			0xdc, 0x62, 0xd4, 0xf6, 0x37, 0xab, 0x7c, 0xeb,
			0xf7, 0x91, 0x4e, 0x00, 0x14, 0x1b, 0x27, 0x68,
			0x72, 0x41, 0xbe, 0xfc, 0x72, 0x6e, 0x77, 0x65,
			0xbe, 0x8a, 0xac, 0x4b, 0x49, 0x02, 0xd5, 0x92,
			0xe0, 0xad, 0x05, 0xef, 0x54, 0xce, 0x2c, 0x27,
			0x8d, 0x01, 0xe7, 0xe0, 0xb8, 0xde, 0x1c, 0xc1,
			0xc4, 0x8d, 0x91, 0x29, 0x65, 0xf9, 0x0b, 0xe1,
			0xe5, 0x17, 0x0b, 0x75, 0x4a, 0x28, 0x3c, 0x57,
			0x1c, 0x88, 0x68, 0x8f, 0x2d, 0x29, 0x6c, 0x4c,
			0xb5, 0x11, 0x73, 0x72, 0x65, 0x4e, 0x82, 0xfd,
			0x9c, 0x87, 0x1c, 0xf3, 0x3b, 0xd8, 0xa0, 0x41,
			0xf4, 0x8d, 0x48, 0x47, 0x77, 0xc0, 0x23, 0xdb,
			0x71, 0xab, 0x2a, 0x0e, 0xb3, 0x36, 0xc3, 0xa7,
			0x6d, 0x11, 0x02, 0x1b, 0x77, 0x72, 0xdd, 0x97,
			0x11, 0xd0, 0x8d, 0x7e, 0x81, 0x2e, 0x9f, 0x13,
			0x73, 0xaf, 0xfa, 0xe8, 0xd5, 0xb2, 0xe2, 0xf6,
			0x38, 0x6d, 0xe3, 0xe3, 0x8f, 0x93, 0xdf, 0xa7,
			0x09, 0x4b, 0x1c, 0xd6, 0x25, 0xef, 0xde, 0xc3,
			0x83, 0x5b, 0x82, 0x99, 0x2a, 0x3f, 0x50, 0xe6,
			0x4e, 0xbd, 0x6f, 0xb0, 0xc4, 0xec, 0x41, 0x06,
			0xf0, 0x88, 0x98, 0x6f, 0x98, 0xf1, 0x16, 0x1c,
			0xd4, 0xa8, 0x15, 0x2c, 0x6b, 0x12, 0xc4, 0xee,
			0x2b, 0x3c, 0x11, 0x5f, 0x3f, 0xe8, 0x74, 0xc0,
			0xa8, 0xd7, 0x28, 0xa1, 0x8c, 0xc7, 0x61, 0x04,
			0x4c, 0x2b, 0x66, 0xaa, 0x09, 0x19, 0x3b, 0xe7,
			0x23, 0xd9, 0xd8, 0x41, 0xe7, 0x73, 0x34, 0x7c,
			0xf2, 0x34, 0x64, 0x24, 0x08, 0x9b, 0x21, 0xa1,
			0x43, 0x9a, 0x7c, 0xc3, 0xae, 0x53, 0x9c, 0xd4,
			0x5e, 0x99, 0xef, 0x6f, 0xe2, 0x85, 0xef, 0x80,
		},
	},
	/* SHA512 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0xc8, 0x10, 0xb1, 0x57, 0xfa, 0xc1, 0x0e, 0x2f,
			0xe1, 0xfa, 0xd7, 0xb4, 0xf0, 0x33, 0x37, 0xb5,
			0xc7, 0x5d, 0xe7, 0x58, 0x06, 0x6b, 0xa2, 0xbc,
			0x5f, 0x11, 0xd9, 0xdc, 0xf0, 0xcb, 0xa7, 0xb7,
			0xcf, 0x22, 0x07, 0xae, 0xe6, 0x2b, 0x45, 0x2d,
			0x01, 0x50, 0xb2, 0x8d, 0x25, 0x52, 0x9c, 0x39,
			0x24, 0x8f, 0x7d, 0x10, 0x6e, 0x81, 0x73, 0x0b,
			0x2a, 0x88, 0xf0, 0x20, 0x85, 0xe9, 0xb3, 0x66,
			0x4a, 0x16, 0xf7, 0x79, 0xb4, 0x30, 0xea, 0x20,
			0x80, 0xca, 0x60, 0x2d, 0x45, 0xf0, 0xc9, 0xa5,
			0x08, 0x67, 0x60, 0x21, 0x0b, 0x37, 0xa6, 0x74,
			0xe7, 0x67, 0x0f, 0x8e, 0xda, 0x08, 0xfb, 0xa1,
			0x62, 0x75, 0xe7, 0x82, 0x0b, 0xd8, 0xe5, 0x51,
			0x85, 0xf0, 0x49, 0x5c, 0xf9, 0x12, 0xa6, 0x41,
			0x9d, 0x6e, 0xf9, 0x53, 0x87, 0x94, 0x23, 0x3f,
			0xbf, 0x37, 0x9d, 0xf0, 0x97, 0x30, 0x67, 0xaf,
		},
		/* S7 : 512 byte */
		{
			0xa7, 0xab, 0x2b, 0x0a, 0x42, 0xa6, 0x0f, 0x30,
			0x3f, 0x11, 0xb8, 0xf1, 0x4f, 0xfc, 0xd3, 0x5e,
			0x73, 0x57, 0xdc, 0xf5, 0xe4, 0xe3, 0xe0, 0x1b,
			0xa3, 0x2b, 0x82, 0x66, 0xc6, 0xc3, 0xdd, 0x59,
			0x29, 0x8b, 0x05, 0x91, 0x42, 0xbb, 0xfb, 0x9f,
			0x18, 0xf2, 0x2b, 0xa1, 0xc6, 0x38, 0xbf, 0xf0,
			0x00, 0x29, 0x9f, 0xbf, 0xc3, 0x8a, 0xb5, 0xc5,
			0xfa, 0x61, 0x22, 0xd4, 0x76, 0xe2, 0x14, 0xbf,
			0x8b, 0x0f, 0xa9, 0x62, 0x97, 0xc1, 0xa6, 0x3e,
			0x13, 0x4e, 0x12, 0x83, 0x49, 0x0f, 0x16, 0xe3,
			0x62, 0xee, 0x88, 0x5d, 0x13, 0x5a, 0x58, 0xa4,
			0x50, 0x06, 0x7f, 0x9d, 0xc0, 0x82, 0x70, 0x06,
			0x3f, 0xf9, 0xbe, 0x62, 0x5e, 0x99, 0x03, 0x7a,
			0x56, 0x2f, 0x3a, 0x47, 0x01, 0x29, 0x88, 0x79,
			0x0c, 0x23, 0x84, 0x26, 0x95, 0xd4, 0xdb, 0x07,
			0x7b, 0x7a, 0xae, 0xc6, 0x33, 0x09, 0x4f, 0xb2,
			0x06, 0x56, 0xad, 0xec, 0x1c, 0x73, 0x3f, 0x49,
			0xe4, 0xeb, 0x12, 0x1d, 0xfe, 0x91, 0xc8, 0xec,
			0x9d, 0x0d, 0xb8, 0x0f, 0x78, 0x6f, 0x91, 0xfa,
			0xb8, 0x7d, 0x6d, 0xcf, 0x68, 0xbc, 0xa9, 0xd3,
			0xfe, 0x9b, 0xd1, 0x8d, 0x56, 0x2f, 0x37, 0xdb,
			0x5f, 0x86, 0x18, 0x50, 0x42, 0x2a, 0x64, 0xdf,
			0x5a, 0x69, 0xd7, 0xe6, 0xc7, 0xb1, 0x2d, 0x8a,
			0xd3, 0xd7, 0x60, 0x0e, 0x6c, 0xf2, 0x59, 0xcc,
			0xd3, 0x3e, 0x7e, 0x66, 0xc3, 0xb8, 0xca, 0x2e,
			0xee, 0xb1, 0xd4, 0xc0, 0x3d, 0x02, 0x64, 0x06,
			0x84, 0x37, 0xb0, 0x40, 0xc9, 0x0f, 0x2e, 0xdb,
			0xf3, 0x98, 0x2d, 0x22, 0x7c, 0xcd, 0xe8, 0x92,
			0x07, 0xf2, 0x9c, 0x8a, 0x91, 0x14, 0xa0, 0x54,
			0x4a, 0xc9, 0xe2, 0xb6, 0x49, 0x91, 0x62, 0x16,
			0x31, 0xa9, 0x14, 0x5c, 0xb6, 0xac, 0x0b, 0x63,
			0x7d, 0x49, 0x5a, 0xdd, 0x90, 0x0e, 0xb1, 0x59,
			0x25, 0x86, 0x00, 0x16, 0x7d, 0x71, 0xf2, 0x8e,
			0xe2, 0x49, 0x46, 0x6a, 0xd4, 0x39, 0x61, 0xb1,
			0xea, 0x56, 0x37, 0x43, 0xfd, 0x5c, 0x70, 0x0d,
			0x83, 0xaf, 0xda, 0x3d, 0x8c, 0xe6, 0x4c, 0x17,
			0x27, 0xc2, 0xd4, 0x56, 0x3c, 0xe9, 0xcd, 0x14,
			0x66, 0x87, 0xc2, 0xf8, 0xf8, 0xc2, 0xf4, 0x02,
			0xd9, 0x03, 0x78, 0x0c, 0xb7, 0x7e, 0x9d, 0x95,
			0x38, 0xbb, 0xef, 0x0d, 0x0c, 0x13, 0xf8, 0x60,
			0x44, 0xfc, 0x69, 0x0a, 0x18, 0x02, 0x27, 0xde,
			0x71, 0xfe, 0x42, 0x9b, 0x99, 0xff, 0x91, 0x9f,
			0x42, 0x14, 0xaa, 0xac, 0x66, 0x6f, 0x8b, 0x0b,
			0xf1, 0x89, 0x41, 0xe7, 0x5a, 0x61, 0xd6, 0xce,
			0x21, 0x44, 0x69, 0xb9, 0x75, 0x7a, 0x48, 0x56,
			0xb2, 0xda, 0x11, 0x86, 0x26, 0xad, 0xd8, 0xd2,
			0xfa, 0x4d, 0x09, 0x12, 0x46, 0x8f, 0x67, 0x96,
			0x90, 0x8c, 0x4b, 0xff, 0xc2, 0x87, 0xdb, 0x9b,
			0x14, 0xe9, 0x57, 0x19, 0x03, 0x70, 0xba, 0x72,
			0xad, 0xe1, 0x8a, 0xd1, 0xc2, 0xce, 0xcb, 0xb1,
			0xd9, 0xec, 0x77, 0xf9, 0x2b, 0xe5, 0x26, 0xca,
			0x33, 0x20, 0x85, 0x1e, 0xf5, 0x76, 0x19, 0x7f,
			0xc6, 0x97, 0x02, 0xc4, 0x5f, 0xb6, 0x59, 0x4d,
			0x4f, 0x9f, 0xdd, 0x27, 0xaa, 0x76, 0x23, 0xfd,
			0x28, 0x12, 0xfe, 0xd0, 0xe2, 0x62, 0x14, 0x11,
			0x43, 0x27, 0xf8, 0x5a, 0xb7, 0x40, 0x27, 0x2f,
			0x0a, 0xf3, 0xe4, 0xe0, 0x78, 0x85, 0xc1, 0x30,
			0xf9, 0x40, 0x6c, 0x0e, 0x39, 0xd9, 0x17, 0xba,
			0x55, 0xb4, 0xba, 0xb0, 0x01, 0x57, 0x2c, 0x0c,
			0xbe, 0xcd, 0xd6, 0xad, 0xe6, 0x94, 0x67, 0xbe,
			0xbc, 0xeb, 0x04, 0x73, 0xd3, 0xbc, 0x13, 0x79,
			0xbd, 0x8f, 0xde, 0xfc, 0x25, 0x5d, 0xaa, 0xc9,
			0xcb, 0x88, 0x46, 0x55, 0x52, 0xb2, 0x81, 0xd5,
			0x25, 0xb7, 0xc8, 0xee, 0x85, 0xac, 0xb9, 0xf8,
		},
	},
	/* SHA512 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0x43, 0x1f, 0x32, 0x38, 0xd8, 0x91, 0x88, 0x6e,
			0xc8, 0x47, 0x14, 0x01, 0xad, 0x04, 0xda, 0xf2,
			0x4e, 0x00, 0x88, 0x49, 0x69, 0x95, 0x59, 0x63,
			0xb0, 0x3b, 0x39, 0x8e, 0x11, 0xb6, 0x2a, 0xd3,
			0x3a, 0x8e, 0xe6, 0x13, 0x5d, 0xc0, 0xc6, 0x56,
			0x30, 0xb3, 0xe2, 0x5f, 0x39, 0x6d, 0x7a, 0x84,
			0x48, 0x07, 0x95, 0xbc, 0x45, 0x5e, 0xe7, 0xfa,
			0x2b, 0x0c, 0xf8, 0xb9, 0x55, 0x97, 0x83, 0xd8,
			0xf9, 0xe7, 0x1c, 0xa9, 0xf2, 0xfe, 0x16, 0x45,
			0xc2, 0xe3, 0xa4, 0x22, 0xd7, 0xdf, 0x3e, 0xe7,
			0x0d, 0xec, 0x47, 0x7e, 0x75, 0x6b, 0xec, 0x6b,
			0x55, 0x15, 0x4f, 0x5d, 0x6f, 0x34, 0xe5, 0x03,
			0x86, 0x13, 0x1c, 0x21, 0x1f, 0xb3, 0x42, 0xe3,
			0x85, 0xbf, 0xa2, 0x71, 0x0f, 0xc2, 0xf0, 0xc3,
			0xa5, 0x98, 0xe7, 0xb7, 0x81, 0x23, 0x30, 0x61,
			0x33, 0x3f, 0x86, 0xa2, 0xe8, 0xf7, 0x18, 0xfc,
		},
		/* S8 : 512 byte */
		{
			0xa3, 0x60, 0x86, 0xee, 0xf4, 0x66, 0x17, 0xb0,
			0x16, 0x48, 0xcf, 0xeb, 0x43, 0xaa, 0x67, 0x6b,
			0xa1, 0xd0, 0xf3, 0x2b, 0xdf, 0xba, 0xb1, 0x94,
			0xff, 0x17, 0xe6, 0x85, 0xc4, 0x76, 0xa4, 0x34,
			0x11, 0x1f, 0x02, 0x9d, 0x75, 0x05, 0x1e, 0x00,
			0x20, 0x6e, 0x7c, 0xd1, 0x24, 0x19, 0x73, 0x7d,
			0x68, 0x03, 0xfd, 0xcc, 0xbb, 0x2e, 0xb7, 0x6b,
			0xba, 0x32, 0x86, 0xe3, 0x34, 0xdf, 0xbd, 0xd0,
			0xf8, 0x3b, 0xab, 0xf2, 0x9a, 0x92, 0x24, 0x1d,
			0x97, 0xe1, 0x3b, 0x82, 0x05, 0xd2, 0xe8, 0xa3,
			0x3a, 0x81, 0x06, 0xc1, 0x6d, 0x9f, 0x98, 0xa6,
			0x7a, 0xb9, 0xb1, 0x32, 0xae, 0x0c, 0x82, 0x1d,
			0x79, 0xdc, 0xf8, 0x24, 0xe1, 0xe2, 0x81, 0x94,
			0x76, 0x7a, 0xb8, 0xb2, 0xfd, 0xe7, 0x08, 0x13,
			0xb6, 0x70, 0x1f, 0x29, 0xfc, 0xb1, 0x47, 0x06,
			0x75, 0x20, 0x74, 0x98, 0x21, 0x5a, 0x6f, 0x0a,
			0x46, 0x71, 0xbd, 0xdb, 0xbd, 0x1a, 0x56, 0x97,
			0xe1, 0x1c, 0xee, 0xb6, 0x13, 0x84, 0xb1, 0xe4,
			0x94, 0xea, 0xdc, 0xe1, 0xd3, 0x5e, 0x15, 0x3b,
			0x5a, 0x7e, 0xff, 0x8b, 0xe5, 0x4a, 0xeb, 0x17,
			0x21, 0x21, 0x18, 0xca, 0x81, 0x66, 0xca, 0x81,
			0xf7, 0xa3, 0x74, 0x27, 0x2c, 0xfa, 0x58, 0x0b,
			0x7c, 0x2b, 0xf8, 0xec, 0xde, 0x65, 0xb4, 0x28,
			0x60, 0xad, 0x94, 0x63, 0x31, 0x6d, 0xee, 0xc8,
			0x71, 0xad, 0xf5, 0x85, 0x18, 0x64, 0xb5, 0x38,
			0xef, 0xab, 0x89, 0x4a, 0x8c, 0x83, 0x70, 0xf0,
			0xbf, 0x3a, 0xfa, 0x9c, 0x30, 0x87, 0xcd, 0x7e,
			0x95, 0x55, 0x86, 0xd3, 0xad, 0x08, 0xb9, 0x02,
			0x78, 0x81, 0x0e, 0x7d, 0xb5, 0x7d, 0x0a, 0xa8,
			0x11, 0x14, 0x6f, 0x0a, 0xb9, 0x06, 0x98, 0xd1,
			0xac, 0x08, 0xa2, 0x53, 0x58, 0x93, 0x8f, 0xee,
			0x57, 0xe2, 0x5e, 0xa0, 0x62, 0x0b, 0xb4, 0x04,
			0x55, 0xe6, 0xcd, 0x2c, 0x68, 0x8c, 0x5d, 0xe2,
			0x0c, 0xad, 0xb5, 0x49, 0x1a, 0x19, 0x95, 0x6e,
			0xcf, 0xf1, 0xcb, 0xa2, 0x3c, 0xea, 0x6c, 0x99,
			0x17, 0x65, 0xe9, 0xf7, 0x49, 0x63, 0x5d, 0xd2,
			0x48, 0x71, 0xd2, 0x4a, 0x3a, 0x47, 0x53, 0x00,
			0xd6, 0xe0, 0x89, 0x24, 0xd0, 0x37, 0xb7, 0xaa,
			0x8d, 0xd7, 0x10, 0x46, 0xe9, 0xcb, 0xe8, 0x85,
			0x70, 0x81, 0xea, 0xfb, 0x66, 0xaa, 0xab, 0xb0,
			0x07, 0xb6, 0xae, 0x39, 0x51, 0x51, 0xbc, 0x9b,
			0xbc, 0x31, 0xcc, 0xf7, 0x2c, 0xa7, 0x56, 0xeb,
			0xa0, 0x1d, 0x48, 0xc2, 0xe2, 0x4d, 0x84, 0x22,
			0xc6, 0xec, 0x82, 0xd4, 0xdf, 0x1b, 0x95, 0x37,
			0x39, 0xc4, 0x84, 0x67, 0x41, 0xb8, 0x15, 0x68,
			0xaa, 0xa5, 0x95, 0x09, 0xe8, 0x11, 0x09, 0xb4,
			0x66, 0x5a, 0xdc, 0x1a, 0x57, 0xb4, 0x61, 0x69,
			0xef, 0x9b, 0xcd, 0x91, 0x2d, 0x3f, 0x10, 0x8e,
			0xca, 0x27, 0x53, 0x45, 0xb4, 0x5b, 0x2a, 0xa5,
			0x71, 0x35, 0x78, 0x64, 0xab, 0x12, 0x12, 0xbe,
			0x27, 0xd1, 0x39, 0x54, 0x7b, 0xbb, 0xa2, 0x6c,
			0x40, 0x96, 0x81, 0xf6, 0x32, 0xbb, 0xce, 0x55,
			0x3b, 0xe9, 0x4d, 0xe5, 0x0e, 0x8f, 0xc4, 0x7e,
			0x68, 0x5d, 0xca, 0x76, 0x25, 0x2b, 0x10, 0x94,
			0xb2, 0x78, 0x5d, 0x5a, 0x82, 0xc5, 0x33, 0x3a,
			0xef, 0xa2, 0x45, 0x23, 0xe1, 0xe8, 0x09, 0xa7,
			0x5f, 0x8c, 0x79, 0xb1, 0x82, 0xf7, 0x34, 0xbc,
			0x93, 0x59, 0xfa, 0x1d, 0xa4, 0x21, 0xc2, 0xeb,
			0xf6, 0xb1, 0x8f, 0x0c, 0xd7, 0x9e, 0xc9, 0x7a,
			0xb1, 0xcf, 0xff, 0x1b, 0xd1, 0x83, 0x3d, 0x90,
			0xb8, 0xaa, 0x48, 0x2a, 0x57, 0xa2, 0x9b, 0x6f,
			0x88, 0x6d, 0xb5, 0xd8, 0xc1, 0x41, 0x28, 0xae,
			0xdf, 0xdc, 0x17, 0xa1, 0x6a, 0x44, 0x49, 0xf5,
			0xe2, 0xb9, 0xc6, 0x12, 0x23, 0xb2, 0xa5, 0xcf,
		},
	},
	/* SHA512 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0xc2, 0xd6, 0x62, 0xd5, 0xea, 0x51, 0x56, 0x0c,
			0x4e, 0x4a, 0x3c, 0x25, 0xe1, 0x37, 0xc4, 0xdf,
			0xf5, 0x71, 0xf0, 0x09, 0xae, 0xde, 0x24, 0x45,
			0xb7, 0xcd, 0x7c, 0x0d, 0x33, 0x21, 0x61, 0xf3,
			0xf7, 0xb2, 0x5f, 0x2d, 0xf6, 0xf0, 0x31, 0x50,
			0xfc, 0xca, 0x1e, 0x5c, 0xa0, 0xce, 0x89, 0xf9,
			0x74, 0x91, 0xc3, 0x00, 0x7e, 0x51, 0x23, 0x3d,
			0xec, 0xd9, 0x59, 0x74, 0x03, 0xa5, 0xff, 0xa1,
			0x59, 0x47, 0x71, 0x84, 0x44, 0x09, 0xdf, 0x5d,
			0x92, 0xd4, 0xa0, 0xf5, 0x7a, 0x50, 0xc9, 0xdd,
			0xd3, 0x4d, 0xff, 0xfa, 0x84, 0x62, 0x89, 0x42,
			0x3c, 0xd3, 0xa9, 0xc0, 0x63, 0xb8, 0x2d, 0xde,
			0x50, 0x5c, 0x41, 0xe3, 0xbc, 0xe4, 0x87, 0xbb,
			0x76, 0x31, 0x6a, 0xf7, 0x59, 0x07, 0xaf, 0x14,
			0x7c, 0x6e, 0x4c, 0x00, 0xa8, 0x58, 0x7e, 0xda,
			0x0f, 0x85, 0x16, 0xf9, 0x3a, 0xa4, 0x13, 0x31,
		},
		/* S9 : 512 byte */
		{
			0xa0, 0x23, 0x83, 0x5c, 0x19, 0xf3, 0xb9, 0xe6,
			0x6e, 0xc2, 0xaf, 0xaf, 0x46, 0x82, 0x0c, 0xb3,
			0x4f, 0xbe, 0x19, 0x69, 0x19, 0xe2, 0x91, 0x07,
			0xa4, 0x72, 0x94, 0x66, 0x9f, 0x50, 0x69, 0x0b,
			0x69, 0x9c, 0x19, 0xf0, 0x8b, 0x85, 0xda, 0x51,
			0x5d, 0x44, 0x85, 0xf1, 0xfb, 0xc7, 0xcb, 0x95,
			0x41, 0xe4, 0x32, 0x9a, 0x2a, 0xec, 0xfe, 0x76,
			0x4f, 0x41, 0xa3, 0x41, 0x03, 0x85, 0x0b, 0x27,
			0x48, 0xf7, 0x5d, 0xe6, 0x82, 0x95, 0x45, 0x89,
			0x4e, 0x3b, 0x0d, 0x4f, 0xdc, 0xb0, 0x47, 0x01,
			0x7d, 0x30, 0xdd, 0x3a, 0xa8, 0xb4, 0x9b, 0x38,
			0xd2, 0x1d, 0x41, 0xe6, 0xd8, 0x9f, 0xef, 0x60,
			0xb6, 0x7d, 0x6e, 0x95, 0x6f, 0xf0, 0x24, 0x7e,
			0xaf, 0xf1, 0x57, 0xc6, 0xf0, 0x79, 0xdd, 0x52,
			0x0c, 0xdb, 0xf0, 0x91, 0x5d, 0xc0, 0x8d, 0xc6,
			0xc0, 0x1a, 0x78, 0xa5, 0xfa, 0x3d, 0xdb, 0x8b,
			0x8d, 0x0a, 0xcd, 0xa8, 0x61, 0x34, 0x1a, 0x00,
			0x2b, 0x73, 0x9f, 0xe2, 0x90, 0xf1, 0x4d, 0xc4,
			0x1d, 0xf9, 0x74, 0x90, 0xfc, 0x1e, 0x2d, 0x60,
			0xab, 0x21, 0x69, 0x30, 0x1d, 0xab, 0xfd, 0x21,
			0x5f, 0x03, 0x57, 0xb0, 0x83, 0x2a, 0x39, 0x0f,
			0x97, 0xea, 0x96, 0xee, 0x98, 0xf7, 0x3a, 0x53,
			0xe7, 0xcc, 0x63, 0x40, 0x21, 0x7a, 0xba, 0xfe,
			0x59, 0x48, 0xa0, 0xa8, 0x31, 0x90, 0x70, 0x84,
			0x04, 0xe0, 0x32, 0x8f, 0xa1, 0x56, 0xe3, 0x66,
			0xbf, 0xe2, 0xd4, 0xa9, 0xac, 0x75, 0x53, 0x1e,
			0x45, 0x8d, 0x91, 0xd1, 0x51, 0x78, 0x6c, 0x36,
			0xbe, 0xf7, 0xf0, 0xce, 0x2d, 0x5c, 0xe5, 0xb6,
			0xb0, 0xce, 0x4f, 0x0d, 0x64, 0x10, 0xfd, 0x89,
			0x73, 0x42, 0xd4, 0x92, 0xc1, 0xa7, 0xf8, 0x18,
			0x50, 0x13, 0xf7, 0x23, 0xd7, 0x46, 0xa6, 0x1d,
			0x2b, 0x97, 0xab, 0xa1, 0xe3, 0xd5, 0x34, 0xdd,
			0x63, 0xd0, 0xad, 0x21, 0xa3, 0xfc, 0x73, 0xd9,
			0x7e, 0x48, 0x0a, 0x7b, 0xe3, 0x67, 0xc5, 0x49,
			0x78, 0x43, 0x15, 0xae, 0xa3, 0x7c, 0x3a, 0x2f,
			0x4a, 0x31, 0x72, 0xa5, 0xb4, 0xc4, 0xc6, 0xcb,
			0x15, 0xa3, 0x8d, 0x4f, 0x2d, 0x27, 0x6c, 0x64,
			0x3e, 0xf4, 0xad, 0xe2, 0xb7, 0xca, 0x6a, 0xa3,
			0x4b, 0x24, 0x88, 0x49, 0x72, 0x0f, 0x1b, 0xfe,
			0xf5, 0xf9, 0xa1, 0x1f, 0x7d, 0xd2, 0x79, 0xae,
			0x21, 0x85, 0xe4, 0x19, 0x56, 0x91, 0x64, 0x7a,
			0xcb, 0xac, 0xe1, 0xdd, 0xd3, 0x85, 0x81, 0xdb,
			0x67, 0xd3, 0x7a, 0x83, 0x9c, 0x2d, 0x6c, 0x5b,
			0x62, 0xf0, 0xfa, 0x79, 0x8d, 0x35, 0xf9, 0x2e,
			0xed, 0xcc, 0x2c, 0xaf, 0x8a, 0x7d, 0x56, 0x23,
			0x88, 0x4f, 0x2f, 0x26, 0x31, 0xd8, 0x2b, 0x06,
			0x33, 0xa8, 0xc3, 0xa4, 0x2b, 0xaa, 0x8c, 0xcb,
			0xcc, 0x7d, 0xb0, 0x5a, 0x34, 0x27, 0x15, 0x78,
			0x8f, 0xe8, 0x8c, 0x23, 0x65, 0x23, 0x06, 0xfe,
			0x02, 0x50, 0x60, 0xd5, 0xc4, 0xce, 0x30, 0x5e,
			0x33, 0xf1, 0x2b, 0x8f, 0xec, 0x51, 0xd1, 0xd0,
			0x30, 0x0a, 0x7c, 0x00, 0x91, 0xe9, 0x16, 0x6c,
			0xc6, 0xa9, 0xe0, 0xf2, 0x4b, 0x93, 0xfc, 0x93,
			0xe5, 0xc7, 0x12, 0x19, 0x77, 0x81, 0x07, 0x2c,
			0x04, 0x0a, 0xcd, 0x5f, 0xeb, 0x83, 0xcc, 0x49,
			0x11, 0x8d, 0x50, 0xab, 0xa6, 0x9b, 0x54, 0xb3,
			0xe5, 0x11, 0x31, 0xa6, 0x4c, 0x12, 0x0a, 0x79,
			0x89, 0xf0, 0x5d, 0x7b, 0xf4, 0x54, 0x3e, 0xd3,
			0xcb, 0x0c, 0xb9, 0x04, 0x8a, 0xf3, 0x63, 0x12,
			0x5d, 0x75, 0xfa, 0x25, 0xda, 0x39, 0x26, 0x0e,
			0xab, 0xe4, 0x90, 0x04, 0xf5, 0x1b, 0xac, 0x9a,
			0x9f, 0xee, 0xb5, 0x81, 0x8b, 0x51, 0x4b, 0x09,
			0x32, 0x7c, 0xae, 0xf7, 0x48, 0x9f, 0x9c, 0x2c,
			0x24, 0x67, 0x00, 0x49, 0x47, 0xe4, 0x8f, 0xac,
		},
	},
};

typedef mod4096_t tv_t;


#include "siggen15test.c"
