/* siggen15_2_1024_testfc.c */
/*
 * Copyright (c) 2017 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

/*
 * This test vector was created based on SigGen15_186-2.txt from
 * https://csrc.nist.gov/CSRC/media/Projects/Cryptographic-Algorithm-Validation-Program/documents/dss/186-2rsatestvectors.zip
 *
 * The link to the archive file exists in the following URL:
 * https://csrc.nist.gov/projects/cryptographic-algorithm-validation-program/digital-signatures/
 */
#include "siggen15test.h"

static const unsigned short sn1[] = {
	0xc8a2, 0x0691, 0x8239, 0x4a2a, 0xb7c3, 0xf419, 0x0c15, 0x589c,
	0x56a2, 0xd4bc, 0x42dc, 0xa675, 0xb34c, 0xc950, 0xe246, 0x6304,
	0x8441, 0xe8aa, 0x593b, 0x2bc5, 0x9e19, 0x8b8c, 0x257e, 0x8821,
	0x20c6, 0x2336, 0xe5cc, 0x7450, 0x12c7, 0xffb0, 0x63ee, 0xbe53,
	0xf3c6, 0x504c, 0xba6c, 0xfe51, 0xbaa3, 0xb6d1, 0x074b, 0x2f39,
	0x8171, 0xf4b1, 0x982f, 0x4d65, 0xcaf8, 0x82ea, 0x4d56, 0xf32a,
	0xb57d, 0x0c44, 0xe6ad, 0x4e9c, 0xf57a, 0x4339, 0xeb69, 0x6240,
	0x6e35, 0x0c1b, 0x1539, 0x7183, 0xfbf1, 0xf035, 0x3c9f, 0xc991,
};

static const unsigned short se1[] = {
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000,
	0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0000, 0x0001, 0x0001,
};

static const unsigned short sd1[] = {
	0x5dfc, 0xb111, 0x072d, 0x2956, 0x5ba1, 0xdb3e, 0xc48f, 0x5764,
	0x5d9d, 0x8804, 0xed59, 0x8a4d, 0x4702, 0x68a8, 0x9067, 0xa2c9,
	0x21df, 0xf24b, 0xa2e3, 0x7a3c, 0xe834, 0x5550, 0x00dc, 0x868e,
	0xe658, 0x8b74, 0x9330, 0x3528, 0xb1b3, 0xa94f, 0x0b71, 0x730c,
	0xf1e8, 0x6fca, 0x5aee, 0xdc3a, 0xfa16, 0xf65c, 0x0189, 0xd810,
	0xddcd, 0x8104, 0x9ebb, 0xd039, 0x1868, 0xc50e, 0xdec9, 0x58b3,
	0xa2aa, 0xeff6, 0xa575, 0x897e, 0x2f20, 0xa3ab, 0x5455, 0xc1bf,
	0xa550, 0x10ac, 0x51a7, 0x799b, 0x1ff8, 0x4836, 0x44a3, 0xd425,
};

#define HAS_SHA1_TESTVECTOR

#define HAS_SHA224_TESTVECTOR

#define HAS_SHA256_TESTVECTOR

#define HAS_SHA384_TESTVECTOR

#define HAS_SHA512_TESTVECTOR

/* 10 elements */
static mod1024_t SHA1[] = {
	/* SHA1 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0xe8, 0x31, 0x27, 0x42, 0xae, 0x23, 0xc4, 0x56,
			0xef, 0x28, 0xa2, 0x31, 0x42, 0xc4, 0x49, 0x08,
			0x95, 0x83, 0x27, 0x65, 0xda, 0xdc, 0xe0, 0x2a,
			0xfe, 0x5b, 0xe5, 0xd3, 0x1b, 0x00, 0x48, 0xfb,
			0xee, 0xe2, 0xcf, 0x21, 0x8b, 0x17, 0x47, 0xad,
			0x4f, 0xd8, 0x1a, 0x2e, 0x17, 0xe1, 0x24, 0xe6,
			0xaf, 0x17, 0xc3, 0x88, 0x8e, 0x6d, 0x2d, 0x40,
			0xc0, 0x08, 0x07, 0xf4, 0x23, 0xa2, 0x33, 0xca,
			0xd6, 0x2c, 0xe9, 0xea, 0xef, 0xb7, 0x09, 0x85,
			0x6c, 0x94, 0xaf, 0x16, 0x6d, 0xba, 0x08, 0xe7,
			0xa0, 0x69, 0x65, 0xd7, 0xfc, 0x0d, 0x8e, 0x5c,
			0xb2, 0x65, 0x59, 0xc4, 0x60, 0xe4, 0x7b, 0xc0,
			0x88, 0x58, 0x9d, 0x22, 0x42, 0xc9, 0xb3, 0xe6,
			0x2d, 0xa4, 0x89, 0x6f, 0xab, 0x19, 0x9e, 0x14,
			0x4e, 0xc1, 0x36, 0xdb, 0x8d, 0x84, 0xab, 0x84,
			0xbc, 0xba, 0x04, 0xca, 0x3b, 0x90, 0xc8, 0xe5,
		},
		/* S0 : 128 byte */
		{
			0x28, 0x92, 0x8e, 0x19, 0xeb, 0x86, 0xf9, 0xc0,
			0x00, 0x70, 0xa5, 0x9e, 0xdf, 0x6b, 0xf8, 0x43,
			0x3a, 0x45, 0xdf, 0x49, 0x5c, 0xd1, 0xc7, 0x36,
			0x13, 0xc2, 0x12, 0x98, 0x40, 0xf4, 0x8c, 0x4a,
			0x2c, 0x24, 0xf1, 0x1d, 0xf7, 0x9b, 0xc5, 0xc0,
			0x78, 0x2b, 0xce, 0xdd, 0xe9, 0x7d, 0xbb, 0xb2,
			0xac, 0xc6, 0xe5, 0x12, 0xd1, 0x9f, 0x08, 0x50,
			0x27, 0xcd, 0x57, 0x50, 0x38, 0x45, 0x3d, 0x04,
			0x90, 0x54, 0x13, 0xe9, 0x47, 0xe6, 0xe1, 0xdd,
			0xdb, 0xeb, 0x35, 0x35, 0xcd, 0xb3, 0xd8, 0x97,
			0x1f, 0xe0, 0x20, 0x05, 0x06, 0x94, 0x10, 0x56,
			0xf2, 0x12, 0x43, 0x50, 0x3c, 0x83, 0xea, 0xdd,
			0xe0, 0x53, 0xed, 0x86, 0x6c, 0x0e, 0x02, 0x50,
			0xbe, 0xdd, 0xd9, 0x27, 0xa0, 0x82, 0x12, 0xaa,
			0x8a, 0xc0, 0xef, 0xd6, 0x16, 0x31, 0xef, 0x89,
			0xd8, 0xd0, 0x49, 0xef, 0xb3, 0x6b, 0xb3, 0x5f,
		},
	},
	/* SHA1 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0x4c, 0x95, 0x07, 0x3d, 0xac, 0x19, 0xd0, 0x25,
			0x6e, 0xaa, 0xdf, 0xf3, 0x50, 0x59, 0x10, 0xe4,
			0x31, 0xdd, 0x50, 0x01, 0x81, 0x36, 0xaf, 0xea,
			0xf6, 0x90, 0xb7, 0xd1, 0x80, 0x69, 0xfc, 0xc9,
			0x80, 0xf6, 0xf5, 0x41, 0x35, 0xc3, 0x0a, 0xcb,
			0x76, 0x9b, 0xee, 0x23, 0xa7, 0xa7, 0x2f, 0x6c,
			0xe6, 0xd9, 0x0c, 0xbc, 0x85, 0x8c, 0x86, 0xdb,
			0xbd, 0x64, 0xba, 0x48, 0xa0, 0x7c, 0x6d, 0x7d,
			0x50, 0xc0, 0xe9, 0x74, 0x6f, 0x97, 0x08, 0x6a,
			0xd6, 0xc6, 0x8e, 0xe3, 0x8a, 0x91, 0xbb, 0xee,
			0xeb, 0x22, 0x21, 0xaa, 0x2f, 0x2f, 0xb4, 0x09,
			0x0f, 0xd8, 0x20, 0xd4, 0xc0, 0xce, 0x5f, 0xf0,
			0x25, 0xba, 0x8a, 0xdf, 0x43, 0xdd, 0xef, 0x89,
			0xf5, 0xf3, 0x65, 0x3d, 0xe1, 0x5e, 0xdc, 0xf3,
			0xaa, 0x80, 0x38, 0xd4, 0x68, 0x69, 0x60, 0xfc,
			0x55, 0xb2, 0x91, 0x7e, 0xc8, 0xa8, 0xf9, 0xa8,
		},
		/* S1 : 128 byte */
		{
			0x53, 0xab, 0x60, 0x0a, 0x41, 0xc7, 0x13, 0x93,
			0xa2, 0x71, 0xb0, 0xf3, 0x2f, 0x52, 0x19, 0x63,
			0x08, 0x7e, 0x56, 0xeb, 0xd7, 0xad, 0x04, 0x0e,
			0x4e, 0xe8, 0xaa, 0x7c, 0x45, 0x0a, 0xd1, 0x8a,
			0xc3, 0xc6, 0xa0, 0x5d, 0x4a, 0xe8, 0x91, 0x3e,
			0x76, 0x3c, 0xfe, 0x96, 0x23, 0xbd, 0x9c, 0xb1,
			0xeb, 0x4b, 0xed, 0x1a, 0x38, 0x20, 0x05, 0x00,
			0xfa, 0x7d, 0xf3, 0xd9, 0x5d, 0xea, 0x48, 0x5f,
			0x03, 0x2a, 0x0a, 0xb0, 0xc6, 0x58, 0x96, 0x78,
			0xf9, 0xe8, 0x39, 0x1b, 0x5c, 0x2b, 0x13, 0x92,
			0x99, 0x7a, 0xc9, 0xf8, 0x2f, 0x1d, 0x16, 0x88,
			0x78, 0x91, 0x6a, 0xac, 0xe9, 0xac, 0x74, 0x55,
			0x80, 0x80, 0x56, 0xaf, 0x81, 0x55, 0x23, 0x1a,
			0x29, 0xf4, 0x29, 0x04, 0xb7, 0xab, 0x87, 0xa5,
			0xd7, 0x1e, 0xd6, 0x39, 0x5e, 0xe0, 0xa9, 0xd0,
			0x24, 0xb0, 0xca, 0x3d, 0x01, 0xfd, 0x71, 0x50,
		},
	},
	/* SHA1 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0xe0, 0x75, 0xad, 0x4b, 0x0f, 0x9b, 0x5b, 0x20,
			0x37, 0x6e, 0x46, 0x7a, 0x1a, 0x35, 0xe3, 0x08,
			0x79, 0x3b, 0xa3, 0x8e, 0xd9, 0x83, 0xd0, 0x38,
			0x87, 0xb8, 0xb8, 0x2e, 0xda, 0x63, 0x0e, 0x68,
			0xb8, 0x61, 0x8d, 0xc4, 0x5b, 0x93, 0xde, 0x55,
			0x55, 0xd7, 0xbc, 0xfe, 0xd2, 0x37, 0x56, 0x40,
			0x1e, 0x61, 0xf5, 0x51, 0x67, 0x57, 0xde, 0x6e,
			0xc3, 0x68, 0x7a, 0x71, 0x75, 0x5f, 0xb4, 0xa6,
			0x6c, 0xfa, 0xa3, 0xdb, 0x0c, 0x9e, 0x69, 0xb6,
			0x31, 0x48, 0x5b, 0x4c, 0x71, 0xc7, 0x62, 0xee,
			0xa2, 0x29, 0xa0, 0x46, 0x9c, 0x73, 0x57, 0xa4,
			0x40, 0x95, 0x07, 0x92, 0xba, 0x9c, 0xd7, 0xae,
			0x02, 0x2a, 0x36, 0xb9, 0xa9, 0x23, 0xc2, 0xeb,
			0xd2, 0xaa, 0x69, 0x89, 0x7f, 0x4c, 0xce, 0xba,
			0x0e, 0x7a, 0xee, 0x97, 0x03, 0x3d, 0x03, 0x81,
			0x07, 0x25, 0xa9, 0xb7, 0x31, 0x83, 0x3f, 0x27,
		},
		/* S2 : 128 byte */
		{
			0x64, 0x26, 0x09, 0xce, 0x08, 0x4f, 0x47, 0x92,
			0x71, 0xdf, 0x59, 0x64, 0x80, 0x25, 0x2e, 0x2f,
			0x89, 0x2b, 0x3e, 0x79, 0x82, 0xdf, 0xf9, 0x59,
			0x94, 0xc3, 0xee, 0xda, 0x78, 0x7f, 0x80, 0xf3,
			0xf6, 0x19, 0x8b, 0xbc, 0xe3, 0x3e, 0xc5, 0x51,
			0x53, 0x78, 0xd4, 0xb5, 0x71, 0xd7, 0x18, 0x60,
			0x78, 0xb7, 0x5b, 0x43, 0xae, 0xd1, 0x1d, 0x34,
			0x25, 0x47, 0x38, 0x6c, 0x56, 0x96, 0xeb, 0x37,
			0x99, 0xa0, 0xb2, 0x84, 0x75, 0xe5, 0x4c, 0xd4,
			0xca, 0x7d, 0x03, 0x6d, 0xcd, 0x8a, 0x11, 0xf5,
			0xe1, 0x08, 0x06, 0xf7, 0xd3, 0xb8, 0xcc, 0x4f,
			0xcb, 0x3e, 0x93, 0xe8, 0x57, 0xbe, 0x95, 0x83,
			0x44, 0xa3, 0x4e, 0x12, 0x68, 0x09, 0xc1, 0x5b,
			0x3d, 0x33, 0x66, 0x1c, 0xf5, 0x7b, 0xf5, 0xc3,
			0x38, 0xf0, 0x7a, 0xcc, 0xed, 0x60, 0xf1, 0x40,
			0x19, 0x33, 0x5c, 0x15, 0x2d, 0x86, 0xb3, 0xb2,
		},
	},
	/* SHA1 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0x18, 0x50, 0x01, 0x55, 0xd2, 0xe0, 0x58, 0x7d,
			0x15, 0x26, 0x98, 0xc0, 0x7b, 0xa4, 0x4d, 0x4f,
			0x04, 0xaa, 0x9a, 0x90, 0x0b, 0x77, 0xce, 0x66,
			0x78, 0xa1, 0x37, 0xb2, 0x38, 0xb7, 0x3b, 0x1a,
			0xea, 0x24, 0xa4, 0x09, 0xdb, 0x56, 0x3c, 0xf6,
			0x35, 0x20, 0x9a, 0xea, 0x73, 0x5d, 0x3b, 0x3c,
			0x18, 0xd7, 0xd5, 0x9f, 0xa1, 0x67, 0xe7, 0x60,
			0xb8, 0x5d, 0x95, 0xe8, 0xaa, 0x21, 0xb3, 0x88,
			0x1b, 0x1b, 0x20, 0x76, 0xf9, 0xd1, 0x55, 0x12,
			0xae, 0x4f, 0x3d, 0x6e, 0x9a, 0xcc, 0x48, 0x0e,
			0xc0, 0x8a, 0xbb, 0xec, 0xbf, 0xfe, 0x4a, 0xbf,
			0x05, 0x31, 0xe8, 0x7d, 0x3f, 0x66, 0xde, 0x1f,
			0x13, 0xfd, 0x1a, 0xa4, 0x12, 0x97, 0xca, 0x58,
			0xe8, 0x7b, 0x2a, 0x56, 0xd6, 0x39, 0x9a, 0x4c,
			0x63, 0x8d, 0xf4, 0x7e, 0x4e, 0x85, 0x1c, 0x0e,
			0xc6, 0xe6, 0xd9, 0x7a, 0xdd, 0xcd, 0xe3, 0x66,
		},
		/* S3 : 128 byte */
		{
			0x42, 0xf3, 0xc3, 0xc7, 0x5f, 0x65, 0xad, 0x42,
			0x05, 0x7b, 0xfa, 0xc1, 0x31, 0x03, 0x83, 0x7b,
			0xf9, 0xf8, 0x42, 0x7c, 0x6e, 0xbc, 0x22, 0xa3,
			0xad, 0xf7, 0xb8, 0xe4, 0x7a, 0x68, 0x57, 0xf1,
			0xcb, 0x17, 0xd2, 0xa5, 0x33, 0xc0, 0xa9, 0x13,
			0xdd, 0x9a, 0x8b, 0xdc, 0x17, 0x86, 0x22, 0x23,
			0x60, 0xcb, 0xd7, 0xe6, 0x4b, 0x45, 0xfc, 0xf5,
			0x4f, 0x5d, 0xa2, 0xf3, 0x42, 0x30, 0xab, 0x48,
			0x06, 0xa0, 0x87, 0xf8, 0xbe, 0x47, 0xf3, 0x5c,
			0x4e, 0x8f, 0xee, 0x2e, 0x6a, 0xa2, 0x91, 0x9a,
			0x56, 0x67, 0x9c, 0xe2, 0xa5, 0x28, 0xa4, 0x4b,
			0xf8, 0x18, 0x62, 0x0d, 0x5b, 0x00, 0xb9, 0xab,
			0x0e, 0x1c, 0x8d, 0x2d, 0x72, 0x2b, 0x53, 0xd3,
			0xa8, 0xcc, 0xa3, 0x5a, 0x99, 0x0e, 0xd2, 0x55,
			0x36, 0xea, 0x65, 0x33, 0x5e, 0x82, 0x53, 0xa5,
			0x4a, 0x68, 0xa6, 0x4a, 0x37, 0x3e, 0x0e, 0xd7,
		},
	},
	/* SHA1 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0xf7, 0xf7, 0x9f, 0x9d, 0xf2, 0x76, 0x0f, 0xc8,
			0x3c, 0x73, 0xc7, 0xcc, 0xea, 0x7e, 0xae, 0x48,
			0x2d, 0xcf, 0xa5, 0xe0, 0x2a, 0xcf, 0x05, 0xe1,
			0x05, 0xdb, 0x48, 0x28, 0x3f, 0x44, 0x06, 0x40,
			0x43, 0x9a, 0x24, 0xca, 0x3b, 0x2a, 0x48, 0x22,
			0x28, 0xc5, 0x8f, 0x3f, 0x32, 0xc3, 0x83, 0xdb,
			0x3c, 0x48, 0x47, 0xd4, 0xbc, 0xc6, 0x15, 0xd3,
			0xca, 0xc3, 0xeb, 0x2b, 0x77, 0xdd, 0x80, 0x04,
			0x5f, 0x0b, 0x7d, 0xb8, 0x82, 0x25, 0xea, 0x7d,
			0x4f, 0xa7, 0xe6, 0x45, 0x02, 0xb2, 0x9c, 0xe2,
			0x30, 0x53, 0x72, 0x6e, 0xa0, 0x08, 0x83, 0xea,
			0x5d, 0x80, 0x50, 0x25, 0x09, 0xa3, 0xb2, 0xdf,
			0x74, 0xd2, 0x14, 0x2f, 0x6e, 0x70, 0xde, 0x22,
			0xd9, 0xa1, 0x34, 0xa5, 0x02, 0x51, 0xe1, 0xa5,
			0x31, 0x79, 0x8e, 0x74, 0x7e, 0x9d, 0x38, 0x6f,
			0xe7, 0x9a, 0xe1, 0xde, 0xa0, 0x9e, 0x85, 0x1b,
		},
		/* S4 : 128 byte */
		{
			0xac, 0x2a, 0xe6, 0x6b, 0xca, 0x1e, 0xc1, 0x2a,
			0x66, 0xa2, 0x90, 0x9f, 0xe2, 0x14, 0x8a, 0x1d,
			0x49, 0x2d, 0x1e, 0xdd, 0x00, 0x06, 0x3b, 0x8b,
			0x33, 0xaf, 0x74, 0x76, 0x0d, 0xc4, 0x05, 0x67,
			0x18, 0xfd, 0x50, 0x41, 0xd4, 0xdf, 0xee, 0x12,
			0xbe, 0xc7, 0x08, 0x1a, 0xb1, 0xba, 0xb2, 0xd0,
			0xeb, 0x27, 0x12, 0xf3, 0x34, 0x50, 0x9f, 0x68,
			0x89, 0xb1, 0x9d, 0x75, 0xd1, 0xfd, 0x0f, 0xc6,
			0x1b, 0xf1, 0x29, 0x76, 0x10, 0x9c, 0x36, 0x14,
			0xc4, 0x60, 0x51, 0xe2, 0xa4, 0x01, 0xb2, 0x08,
			0x80, 0xd6, 0xe6, 0x4a, 0xd6, 0xa4, 0x7f, 0x23,
			0x93, 0x98, 0x03, 0xd1, 0x38, 0xaa, 0x0a, 0x44,
			0xbc, 0x41, 0xba, 0x63, 0x03, 0x07, 0x46, 0x62,
			0x22, 0x48, 0x77, 0x14, 0x31, 0xdf, 0xf9, 0x7e,
			0x8a, 0x85, 0x6f, 0x0b, 0x61, 0xd1, 0x14, 0xf8,
			0x13, 0x91, 0x1e, 0xe2, 0x29, 0x65, 0x51, 0x55,
		},
	},
	/* SHA1 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0x09, 0x9b, 0xf1, 0x7f, 0x16, 0xbc, 0xfd, 0x4c,
			0x19, 0xb3, 0x4f, 0xec, 0xb4, 0xb3, 0x23, 0x3c,
			0x9f, 0x9f, 0x98, 0x71, 0x8f, 0x67, 0xb3, 0x06,
			0x5d, 0x95, 0xa5, 0xf8, 0x64, 0x23, 0x51, 0x36,
			0x2b, 0x90, 0x09, 0x53, 0x44, 0x33, 0x98, 0x7f,
			0x73, 0xce, 0x86, 0xb5, 0x13, 0x66, 0x97, 0x36,
			0xb6, 0x52, 0x95, 0x35, 0x0c, 0x93, 0x4f, 0xd4,
			0x06, 0x63, 0xe2, 0x4f, 0x3a, 0x10, 0x37, 0x77,
			0x8a, 0x0b, 0xcd, 0x63, 0x00, 0x3c, 0xb9, 0x62,
			0xfd, 0x99, 0xeb, 0x33, 0x93, 0xf7, 0xb2, 0x79,
			0x2f, 0x20, 0x83, 0x69, 0x7b, 0x25, 0xf6, 0xc6,
			0x82, 0xf6, 0x11, 0x0f, 0x16, 0x2f, 0xc9, 0xf7,
			0x6e, 0x35, 0xc6, 0x15, 0x14, 0x82, 0x67, 0xdd,
			0xff, 0x3d, 0x06, 0xcf, 0xfb, 0x0e, 0x7d, 0xee,
			0x52, 0x30, 0xe8, 0x74, 0xa5, 0xc8, 0xad, 0xc4,
			0x1b, 0x75, 0xba, 0xa0, 0xbe, 0x28, 0x0e, 0x9c,
		},
		/* S5 : 128 byte */
		{
			0x3a, 0x2b, 0x75, 0x71, 0x61, 0x92, 0x72, 0xb8,
			0x1d, 0x35, 0x62, 0xa1, 0x1c, 0x64, 0x45, 0x02,
			0x89, 0x44, 0x21, 0x58, 0x3e, 0x02, 0x87, 0x9f,
			0x5a, 0x77, 0x59, 0xfb, 0x64, 0xec, 0x2a, 0xb8,
			0x10, 0x5f, 0x7d, 0x11, 0x94, 0x7c, 0x8e, 0x4b,
			0xfc, 0xa8, 0x72, 0x19, 0xe5, 0x29, 0x87, 0xaa,
			0xd3, 0xb8, 0x1c, 0xbd, 0x48, 0x31, 0x66, 0xed,
			0x78, 0x15, 0x2a, 0xf2, 0x44, 0x60, 0xc9, 0x08,
			0x87, 0x9f, 0x34, 0xc8, 0x70, 0x57, 0x31, 0x27,
			0xe3, 0x44, 0x8c, 0x8f, 0xbf, 0x43, 0x02, 0x83,
			0x50, 0xc9, 0x75, 0xbb, 0xc3, 0xa9, 0x99, 0x19,
			0x6a, 0x3e, 0x99, 0x61, 0x22, 0x8a, 0x2b, 0xb1,
			0x5b, 0x49, 0x85, 0xe9, 0x5b, 0xba, 0x97, 0x0a,
			0xc4, 0xad, 0x2a, 0xc5, 0xb4, 0x2a, 0xc5, 0x1d,
			0xbc, 0x65, 0x09, 0xef, 0xfc, 0x13, 0x39, 0x66,
			0x93, 0x98, 0x0f, 0xc8, 0x9b, 0xa4, 0x4c, 0x7b,
		},
	},
	/* SHA1 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0xfb, 0x40, 0xa7, 0x3d, 0xc8, 0x2f, 0x16, 0x7f,
			0x9c, 0x2b, 0xf9, 0x8a, 0x99, 0x1e, 0xa8, 0x2f,
			0xdb, 0x01, 0x41, 0xdb, 0xad, 0x44, 0x87, 0x1a,
			0xfd, 0x70, 0xf0, 0x5a, 0x0e, 0x0b, 0xf9, 0xf2,
			0x6d, 0xbc, 0xbd, 0x62, 0x26, 0xaf, 0xc6, 0xdc,
			0x37, 0x3b, 0x23, 0x04, 0x45, 0xc2, 0xba, 0xf5,
			0x8e, 0xd9, 0xe0, 0x84, 0x1f, 0xa9, 0x27, 0xc8,
			0x47, 0x95, 0x77, 0xda, 0x4b, 0x1e, 0x61, 0xd9,
			0x5b, 0x03, 0xaf, 0x31, 0xc5, 0xac, 0x40, 0x1d,
			0x69, 0xc8, 0x13, 0x6b, 0x6d, 0x36, 0xa1, 0x80,
			0x32, 0x21, 0x70, 0x9b, 0x86, 0x70, 0xe5, 0x5e,
			0x1b, 0x5d, 0x5a, 0x8a, 0x37, 0x63, 0x70, 0x0a,
			0xae, 0x5e, 0xa6, 0x33, 0x0e, 0xee, 0x2b, 0x4a,
			0x19, 0x1c, 0xf1, 0x46, 0x78, 0x40, 0x03, 0xd8,
			0xad, 0x22, 0x18, 0xa9, 0x4a, 0x5f, 0x68, 0xe3,
			0x60, 0x0e, 0xbe, 0xf2, 0x3b, 0xa4, 0xcf, 0x8c,
		},
		/* S6 : 128 byte */
		{
			0xb1, 0x03, 0x22, 0x60, 0x2c, 0x28, 0x4f, 0x40,
			0x79, 0xe5, 0x09, 0xfa, 0xf3, 0xf4, 0x0a, 0x3d,
			0x2a, 0xf3, 0xab, 0xef, 0x9f, 0x09, 0x17, 0x1f,
			0xdd, 0x16, 0x46, 0x9d, 0x67, 0x9b, 0xb9, 0xad,
			0xc7, 0xe2, 0xac, 0xb1, 0xad, 0xdb, 0x0b, 0xd5,
			0xb3, 0x8b, 0x5c, 0x4d, 0x98, 0x6b, 0x43, 0xc7,
			0x9b, 0x97, 0x24, 0xf6, 0x1e, 0x99, 0xb5, 0xb3,
			0x03, 0x63, 0x0b, 0x62, 0xd0, 0xd8, 0xd5, 0xf7,
			0x65, 0x77, 0xfe, 0x7e, 0xa3, 0x87, 0x71, 0x0b,
			0x43, 0x78, 0x9e, 0xe1, 0xb3, 0x5b, 0x61, 0x4b,
			0x69, 0x1f, 0x0a, 0x27, 0xb7, 0x3b, 0xaf, 0x6b,
			0xc3, 0xf2, 0x8e, 0xc2, 0x10, 0xb9, 0xd3, 0xe4,
			0xc5, 0xa2, 0x72, 0x9c, 0xc1, 0x20, 0x3b, 0x74,
			0xef, 0x70, 0xe3, 0x15, 0xcf, 0xe5, 0xd0, 0x6e,
			0x04, 0x0a, 0xee, 0x6b, 0x3d, 0x22, 0xd9, 0x1d,
			0x6e, 0x22, 0x9f, 0x69, 0x0a, 0x96, 0x6d, 0xd9,
		},
	},
	/* SHA1 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0x97, 0xe7, 0x49, 0x60, 0xdb, 0xd9, 0x81, 0xd4,
			0x6a, 0xad, 0xc0, 0x21, 0xa6, 0xcf, 0x18, 0x1d,
			0xdd, 0xe6, 0xe4, 0xcf, 0xcb, 0x4b, 0x63, 0x82,
			0x60, 0xc0, 0xa5, 0x19, 0xc4, 0x5f, 0xab, 0xa2,
			0x99, 0xd0, 0xca, 0x2e, 0x80, 0xbf, 0x50, 0xdf,
			0xde, 0x8d, 0x6a, 0x42, 0xe0, 0x46, 0x45, 0xdf,
			0xbc, 0xd4, 0x74, 0x0f, 0x3a, 0x72, 0x92, 0x0e,
			0x74, 0x63, 0x28, 0x51, 0xd9, 0xe3, 0xd0, 0x1a,
			0x78, 0x5e, 0x9b, 0x49, 0x7c, 0xe0, 0xb1, 0x75,
			0xf2, 0xcd, 0x37, 0x3b, 0xd3, 0xd2, 0x76, 0xd6,
			0x3e, 0x1b, 0x39, 0xf0, 0x05, 0xc6, 0x76, 0xb8,
			0x6b, 0x98, 0x31, 0x35, 0x2c, 0xef, 0x9e, 0xda,
			0xbe, 0xf8, 0x86, 0x5a, 0xd7, 0x22, 0xeb, 0xbe,
			0x2f, 0xd3, 0xef, 0xb4, 0x87, 0x59, 0xf2, 0x2a,
			0xea, 0x23, 0xfb, 0x1b, 0x33, 0x31, 0x59, 0xa9,
			0xcf, 0xc9, 0x8a, 0x6d, 0xc4, 0x6c, 0x5b, 0x0b,
		},
		/* S7 : 128 byte */
		{
			0x60, 0xeb, 0xc9, 0xe4, 0xe2, 0xe2, 0xb4, 0xfa,
			0x6d, 0x31, 0xc5, 0x7d, 0x0b, 0x86, 0x83, 0x5e,
			0x8d, 0x20, 0x1d, 0x21, 0xc2, 0x74, 0xcf, 0x54,
			0x52, 0xcd, 0xd7, 0xef, 0x28, 0x57, 0xdc, 0x78,
			0x0d, 0xde, 0x35, 0x26, 0xf3, 0x65, 0x8c, 0x4f,
			0x2c, 0x87, 0x10, 0xea, 0xae, 0x48, 0x70, 0xd2,
			0x75, 0x99, 0x7e, 0x5c, 0xbb, 0x26, 0x8e, 0x3b,
			0xd2, 0x51, 0xf5, 0x43, 0xb8, 0x82, 0x8f, 0xeb,
			0x85, 0xc2, 0x11, 0xc8, 0x58, 0xe4, 0x7a, 0x74,
			0xcb, 0x12, 0x2d, 0xc1, 0x7f, 0x26, 0xfe, 0x92,
			0xb4, 0xaf, 0xee, 0xcb, 0xf1, 0xe2, 0x0b, 0xea,
			0x75, 0xc7, 0x94, 0xc0, 0x48, 0x2a, 0xa6, 0x53,
			0x2e, 0x87, 0x95, 0x5d, 0xba, 0x24, 0x9f, 0x0f,
			0xa6, 0x56, 0x2b, 0xdf, 0x8f, 0x4c, 0xcd, 0x8a,
			0x63, 0xda, 0x69, 0xd1, 0xf3, 0x37, 0x52, 0x3f,
			0x65, 0x20, 0x6f, 0xb8, 0xeb, 0x16, 0x31, 0x73,
		},
	},
	/* SHA1 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0x95, 0xd0, 0x46, 0x24, 0xb9, 0x98, 0x93, 0x8d,
			0xd0, 0xa5, 0xba, 0x6d, 0x70, 0x42, 0xaa, 0x88,
			0xa2, 0x67, 0x4d, 0xad, 0x43, 0x8a, 0x0d, 0x31,
			0xab, 0xb7, 0x97, 0x9d, 0x8d, 0xe3, 0xde, 0xa4,
			0x1e, 0x7e, 0x63, 0x58, 0x7a, 0x47, 0xb5, 0x9d,
			0x43, 0x64, 0x33, 0xdd, 0x8b, 0xb2, 0x19, 0xfd,
			0xf4, 0x5a, 0xbb, 0x90, 0x15, 0xa5, 0x0b, 0x4b,
			0x20, 0x11, 0x61, 0xb9, 0xc2, 0xa4, 0x7c, 0x30,
			0x4b, 0x80, 0xc4, 0x04, 0x0f, 0xb8, 0xd1, 0xfa,
			0x0c, 0x62, 0x31, 0x00, 0xcd, 0xed, 0x66, 0x1b,
			0x8e, 0xb5, 0x2f, 0xa0, 0xa0, 0xd5, 0x09, 0xa7,
			0x0f, 0x3c, 0xf4, 0xbd, 0x83, 0x04, 0x7a, 0xd9,
			0x64, 0xff, 0xee, 0x92, 0x41, 0x92, 0xf2, 0x8e,
			0x73, 0xc6, 0x3b, 0x3e, 0xfd, 0x9c, 0x99, 0xc8,
			0xb7, 0xa1, 0x31, 0x45, 0xac, 0xc3, 0x0d, 0x2d,
			0xc0, 0x63, 0xd8, 0x0f, 0x96, 0xab, 0xe2, 0x86,
		},
		/* S8 : 128 byte */
		{
			0x85, 0x9c, 0xc4, 0xfc, 0xd1, 0xb8, 0x8c, 0xcd,
			0xa6, 0x95, 0xb1, 0x23, 0x11, 0xcf, 0x8b, 0xdc,
			0xa3, 0xb4, 0xc1, 0x35, 0xfa, 0xa1, 0x1f, 0x90,
			0x53, 0xdc, 0x10, 0xf4, 0xbf, 0x12, 0xe5, 0xf2,
			0x17, 0x9b, 0xe6, 0xab, 0x5a, 0xd9, 0x0f, 0x8d,
			0x11, 0x5f, 0x5d, 0xf7, 0x95, 0xa7, 0x73, 0x40,
			0xe2, 0x06, 0x62, 0x80, 0x9f, 0xa7, 0x32, 0xb9,
			0x25, 0x60, 0xad, 0xcf, 0xfd, 0xb0, 0xdd, 0xb7,
			0x2d, 0x33, 0x81, 0x1e, 0x94, 0xf8, 0x54, 0x33,
			0x06, 0x80, 0xf2, 0xb2, 0x38, 0x30, 0x09, 0x95,
			0xa9, 0x11, 0x3a, 0x46, 0x9a, 0xfd, 0x9e, 0x75,
			0x6f, 0x64, 0x92, 0x08, 0xd2, 0x94, 0x2f, 0xeb,
			0xff, 0xb2, 0x2e, 0x83, 0x22, 0x79, 0x06, 0x3e,
			0xc5, 0xb5, 0x7a, 0xb5, 0x42, 0xd9, 0xbb, 0xc5,
			0x6e, 0x82, 0xcd, 0xc6, 0xa0, 0x3b, 0x00, 0xd1,
			0x0d, 0x45, 0x80, 0x15, 0x75, 0xe9, 0x49, 0xe1,
		},
	},
	/* SHA1 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0x20, 0x71, 0x02, 0xf5, 0x98, 0xec, 0x28, 0x00,
			0x45, 0xbe, 0x67, 0x59, 0x2f, 0x5b, 0xba, 0x25,
			0xba, 0x2e, 0x2b, 0x56, 0xe0, 0xd2, 0x39, 0x7c,
			0xbe, 0x85, 0x7c, 0xde, 0x52, 0xda, 0x8c, 0xca,
			0x83, 0xae, 0x1e, 0x29, 0x61, 0x5c, 0x70, 0x56,
			0xaf, 0x35, 0xe8, 0x31, 0x9f, 0x2a, 0xf8, 0x6f,
			0xdc, 0xcc, 0x44, 0x34, 0xcd, 0x77, 0x07, 0xe3,
			0x19, 0xc9, 0xb2, 0x35, 0x66, 0x59, 0xd7, 0x88,
			0x67, 0xa6, 0x46, 0x7a, 0x15, 0x4e, 0x76, 0xb7,
			0x3c, 0x81, 0x26, 0x0f, 0x3a, 0xb4, 0x43, 0xcc,
			0x03, 0x9a, 0x0d, 0x42, 0x69, 0x50, 0x76, 0xa7,
			0x9b, 0xd8, 0xca, 0x25, 0xeb, 0xc8, 0x95, 0x2e,
			0xd4, 0x43, 0xc2, 0x10, 0x3b, 0x29, 0x00, 0xc9,
			0xf5, 0x8b, 0x6a, 0x1c, 0x8a, 0x62, 0x66, 0xe4,
			0x38, 0x80, 0xcd, 0xa9, 0x3b, 0xc6, 0x4d, 0x71,
			0x4c, 0x98, 0x0c, 0xd8, 0x68, 0x8e, 0x8e, 0x63,
		},
		/* S9 : 128 byte */
		{
			0x77, 0xf0, 0xf2, 0xa0, 0x48, 0x48, 0xfe, 0x90,
			0xa8, 0xeb, 0x35, 0xab, 0x5d, 0x94, 0xca, 0xe8,
			0x43, 0xdb, 0x61, 0x02, 0x4d, 0x01, 0x67, 0x28,
			0x9e, 0xea, 0x92, 0xe5, 0xd1, 0xe1, 0x0a, 0x52,
			0x6e, 0x42, 0x0f, 0x2d, 0x33, 0x4f, 0x1b, 0xf2,
			0xaa, 0x7e, 0xa4, 0xe1, 0x4a, 0x93, 0xa6, 0x8d,
			0xba, 0x60, 0xfd, 0x2e, 0xde, 0x58, 0xb7, 0x94,
			0xdc, 0xbd, 0x37, 0xdc, 0xb1, 0x96, 0x78, 0x77,
			0xd6, 0xb6, 0x7d, 0xa3, 0xfd, 0xf2, 0xc0, 0xc7,
			0x43, 0x3e, 0x47, 0x13, 0x4d, 0xde, 0x00, 0xc9,
			0xc4, 0xd4, 0x07, 0x2e, 0x43, 0x36, 0x1a, 0x76,
			0x7a, 0x52, 0x76, 0x75, 0xd8, 0xbd, 0xa7, 0xd5,
			0x92, 0x1b, 0xd4, 0x83, 0xc9, 0x55, 0x19, 0x50,
			0x73, 0x9e, 0x9b, 0x2b, 0xe0, 0x27, 0xdf, 0x30,
			0x15, 0xb6, 0x1f, 0x75, 0x1a, 0xc1, 0xd9, 0xf3,
			0x7b, 0xea, 0x32, 0x14, 0xd3, 0xc8, 0xdc, 0x96,
		},
	},
};

/* 10 elements */
static mod1024_t SHA224[] = {
	/* SHA224 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0xe5, 0x67, 0xa3, 0x9a, 0xe4, 0xe5, 0xef, 0x9b,
			0x68, 0x01, 0xea, 0x05, 0x61, 0xb7, 0x2a, 0x5d,
			0x4b, 0x5f, 0x38, 0x5f, 0x05, 0x32, 0xfc, 0x9f,
			0xe1, 0x0a, 0x75, 0x70, 0xf8, 0x69, 0xae, 0x05,
			0xc0, 0xbd, 0xed, 0xd6, 0xe0, 0xe2, 0x2d, 0x45,
			0x42, 0xe9, 0xce, 0x82, 0x6a, 0x18, 0x8c, 0xac,
			0x07, 0x31, 0xae, 0x39, 0xc8, 0xf8, 0x7f, 0x97,
			0x71, 0xef, 0x02, 0x13, 0x2e, 0x64, 0xe2, 0xfb,
			0x27, 0xad, 0xa8, 0xff, 0x54, 0xb3, 0x30, 0xdd,
			0x93, 0xad, 0x5e, 0x3e, 0xf8, 0x2e, 0x0d, 0xda,
			0x64, 0x62, 0x48, 0xe3, 0x59, 0x94, 0xbd, 0xa1,
			0x0c, 0xf4, 0x6e, 0x5a, 0xbc, 0x98, 0xaa, 0x74,
			0x43, 0xc0, 0x3c, 0xdd, 0xeb, 0x5e, 0xe2, 0xab,
			0x82, 0xd6, 0x01, 0x00, 0xb1, 0x02, 0x96, 0x31,
			0x89, 0x79, 0x70, 0x27, 0x5f, 0x11, 0x9d, 0x05,
			0xda, 0xa2, 0x22, 0x0a, 0x4a, 0x0d, 0xef, 0xba,
		},
		/* S0 : 128 byte */
		{
			0x5a, 0xa5, 0x03, 0x33, 0x81, 0xbd, 0xd0, 0xac,
			0xce, 0x33, 0x2d, 0xd3, 0x14, 0xda, 0xf0, 0x08,
			0xac, 0xaa, 0x9e, 0x83, 0x5f, 0x83, 0x29, 0x79,
			0x89, 0x1d, 0x1b, 0xda, 0x2b, 0x55, 0xd5, 0xea,
			0xe3, 0x5c, 0x47, 0x9c, 0x06, 0xca, 0xc5, 0xbf,
			0x33, 0xf4, 0x32, 0xc8, 0xc0, 0xa5, 0x54, 0x9d,
			0x1d, 0x1b, 0x29, 0xc5, 0xe2, 0x58, 0x90, 0x24,
			0xd2, 0x78, 0x00, 0xa0, 0xc2, 0x35, 0xa6, 0x15,
			0x32, 0xc2, 0x03, 0xcb, 0xc4, 0x06, 0xac, 0x6e,
			0xcf, 0x63, 0xf5, 0x2a, 0xe7, 0x71, 0xb9, 0x7c,
			0x08, 0xe4, 0xb1, 0x08, 0xec, 0x91, 0x69, 0x00,
			0xe5, 0xa1, 0x1b, 0x1d, 0x48, 0xcc, 0xa8, 0x6c,
			0xa5, 0xa5, 0xa7, 0x99, 0xed, 0x32, 0xe9, 0x9c,
			0x81, 0x5c, 0xef, 0x04, 0xcf, 0x8e, 0xb5, 0x52,
			0x23, 0xbf, 0xd4, 0xd9, 0xc3, 0x44, 0x92, 0x64,
			0xb6, 0x00, 0x61, 0xbc, 0x36, 0x84, 0xbc, 0x82,
		},
	},
	/* SHA224 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0xdb, 0x9e, 0xae, 0x66, 0xbc, 0xac, 0xb0, 0x1b,
			0xa8, 0x02, 0x9f, 0xba, 0xc5, 0xfe, 0xf1, 0x31,
			0x56, 0xaa, 0x1a, 0x76, 0x8c, 0x3d, 0xff, 0x25,
			0x9e, 0xe5, 0xae, 0x6b, 0x1c, 0xc3, 0x4c, 0x22,
			0x80, 0x37, 0x4e, 0x3d, 0x06, 0x5f, 0xed, 0x67,
			0xbd, 0xda, 0xd3, 0xe7, 0xde, 0xd2, 0x1b, 0xc7,
			0x5d, 0x39, 0x78, 0x35, 0xbf, 0x52, 0xb4, 0x29,
			0x2b, 0x74, 0x5d, 0x48, 0xc3, 0x0b, 0xbb, 0x09,
			0x35, 0xdf, 0x04, 0x16, 0x8c, 0x99, 0xd3, 0xf5,
			0x4d, 0x83, 0xd7, 0xe7, 0xbf, 0x8f, 0xc8, 0x10,
			0x8a, 0x9b, 0xa0, 0xd9, 0x83, 0xf3, 0x07, 0x93,
			0xc7, 0x17, 0x0f, 0x5d, 0x07, 0xbf, 0x21, 0x47,
			0x22, 0x1d, 0x38, 0xb7, 0xf9, 0x62, 0x4c, 0x0c,
			0x7f, 0x83, 0x12, 0x82, 0x0f, 0x3d, 0xe2, 0x55,
			0x03, 0x57, 0xf8, 0x2a, 0x82, 0x27, 0xe0, 0xaa,
			0x99, 0x57, 0x2c, 0x70, 0x8d, 0xe9, 0x68, 0x25,
		},
		/* S1 : 128 byte */
		{
			0x1e, 0xe0, 0x3b, 0xd2, 0xa0, 0x44, 0x0e, 0xf8,
			0x9c, 0x0f, 0x19, 0xfa, 0x03, 0xc8, 0x55, 0x07,
			0x4e, 0x16, 0xf6, 0xed, 0x42, 0x6c, 0xa3, 0x5b,
			0xb7, 0x63, 0x13, 0x88, 0x78, 0x96, 0x9a, 0x81,
			0x4b, 0xe2, 0xf1, 0x6f, 0x3d, 0xbb, 0xf7, 0x3d,
			0x2f, 0xad, 0x78, 0xd4, 0x2a, 0x8f, 0xf8, 0xaa,
			0xa3, 0x42, 0xf5, 0xfc, 0xb0, 0x8d, 0x2d, 0xde,
			0x74, 0x52, 0x5b, 0xa5, 0xb2, 0xc6, 0x5e, 0xf4,
			0xe3, 0x3d, 0x61, 0x77, 0x45, 0x45, 0x88, 0x6f,
			0x7f, 0x21, 0x7e, 0xb6, 0x81, 0xa5, 0x4b, 0x42,
			0xef, 0x7d, 0xf6, 0x6c, 0xe9, 0x43, 0x22, 0x70,
			0x21, 0xc1, 0xb5, 0x70, 0x9b, 0x7a, 0x55, 0x4a,
			0x64, 0x14, 0x1c, 0x1f, 0xa3, 0x60, 0x9a, 0xd2,
			0x5f, 0xa8, 0x31, 0x70, 0x68, 0x36, 0x9b, 0x89,
			0x90, 0xb8, 0x43, 0x94, 0x68, 0x81, 0x04, 0x40,
			0x66, 0x74, 0x6c, 0x12, 0x03, 0xb5, 0x7a, 0xcb,
		},
	},
	/* SHA224 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x2e, 0x22, 0x65, 0x6e, 0x48, 0x85, 0x62, 0x40,
			0xb0, 0xf6, 0x55, 0xd2, 0x02, 0x2d, 0x69, 0xac,
			0x73, 0x1a, 0x1c, 0x63, 0x15, 0x1b, 0xa3, 0x6e,
			0x57, 0x72, 0x8a, 0x35, 0xcb, 0xf6, 0x9c, 0x6b,
			0x2d, 0x08, 0x02, 0x9e, 0xa0, 0xc9, 0xaf, 0x0c,
			0x3a, 0x8b, 0xfb, 0xb9, 0xc3, 0x85, 0x8d, 0xbe,
			0xba, 0xca, 0x7f, 0xe4, 0x99, 0x7c, 0xbd, 0x71,
			0x5b, 0x3c, 0xae, 0x42, 0x3a, 0xe6, 0x17, 0x1a,
			0xbd, 0x7b, 0x3c, 0x3a, 0xf2, 0x5f, 0x48, 0x32,
			0xf8, 0xc5, 0xee, 0xf6, 0xc1, 0x67, 0x51, 0x34,
			0x16, 0x7a, 0x21, 0xe5, 0xdc, 0xe1, 0x07, 0x23,
			0x95, 0xa1, 0x44, 0x3a, 0x2a, 0x95, 0x95, 0x01,
			0xe7, 0x63, 0x59, 0x6b, 0xc7, 0xae, 0xf5, 0x59,
			0xf2, 0x8d, 0x78, 0xb2, 0x84, 0x3c, 0x7b, 0xb6,
			0x90, 0x12, 0x4c, 0x8f, 0x65, 0xb3, 0x4b, 0x29,
			0x0f, 0x87, 0x93, 0x44, 0x21, 0x6b, 0xdd, 0xc8,
		},
		/* S2 : 128 byte */
		{
			0xc3, 0xa6, 0x67, 0x96, 0xc8, 0x73, 0x19, 0xc0,
			0x1a, 0x83, 0x55, 0x10, 0x25, 0x62, 0x02, 0x97,
			0x97, 0xd6, 0x95, 0xde, 0x50, 0xb4, 0x6b, 0xfa,
			0x39, 0xc6, 0xd9, 0xa4, 0xc0, 0x64, 0x2b, 0x6d,
			0xe8, 0x02, 0x72, 0x48, 0x1b, 0x87, 0x85, 0xb0,
			0xff, 0x3b, 0xa4, 0x57, 0x1b, 0xb4, 0x5c, 0xd6,
			0xaa, 0xd9, 0xc5, 0xfc, 0x13, 0x83, 0xc7, 0x9f,
			0x6b, 0x7a, 0x13, 0xc4, 0xe6, 0x7f, 0x59, 0x4f,
			0x81, 0xa8, 0x11, 0x57, 0xde, 0x62, 0x47, 0xb8,
			0x96, 0x34, 0x5d, 0xe1, 0xae, 0xc9, 0x4a, 0xdc,
			0xcc, 0xf1, 0x26, 0x59, 0x91, 0x32, 0x84, 0x6e,
			0xa1, 0x39, 0x85, 0x85, 0x49, 0x9b, 0xe4, 0xe7,
			0x59, 0xa1, 0xf7, 0x75, 0x7d, 0x31, 0x8c, 0xcf,
			0x41, 0x37, 0x8b, 0x37, 0x42, 0x75, 0xad, 0x4c,
			0xbc, 0xd1, 0x07, 0x74, 0x76, 0xcb, 0xb0, 0x6e,
			0xa4, 0x2b, 0x36, 0x28, 0x9b, 0x0f, 0x80, 0xbc,
		},
	},
	/* SHA224 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0xcf, 0x78, 0x50, 0x02, 0x29, 0x2f, 0x1e, 0xf9,
			0x64, 0x5f, 0x2f, 0x2c, 0xdc, 0x04, 0x55, 0xb2,
			0x9d, 0x0b, 0xa1, 0xef, 0xc9, 0x1f, 0xde, 0x0a,
			0x3a, 0xc0, 0x5f, 0xa1, 0x69, 0x95, 0xfb, 0xbc,
			0x8e, 0x03, 0x55, 0x00, 0x09, 0xfe, 0x61, 0x26,
			0x01, 0x1f, 0x23, 0xab, 0x4b, 0xf7, 0xba, 0xc8,
			0x8d, 0xa4, 0x9e, 0xa4, 0xa3, 0xa1, 0x26, 0xc9,
			0x41, 0xe3, 0x4d, 0xd4, 0xda, 0x21, 0xb5, 0x71,
			0xc1, 0x73, 0xec, 0x87, 0x8f, 0x1e, 0xef, 0xb0,
			0xc4, 0xae, 0xec, 0xec, 0xb1, 0x46, 0x4c, 0x96,
			0x95, 0x35, 0xf1, 0x97, 0x06, 0xc9, 0xbc, 0xd9,
			0x92, 0x68, 0x53, 0xff, 0xa7, 0xdc, 0x21, 0x53,
			0xb0, 0xef, 0x9e, 0xff, 0x81, 0x38, 0xd0, 0x81,
			0xf5, 0x34, 0x12, 0x59, 0xd5, 0x99, 0x14, 0x04,
			0xfc, 0xe6, 0x23, 0x2e, 0xb9, 0x3f, 0xaa, 0x24,
			0x75, 0x78, 0xf9, 0x88, 0x96, 0x70, 0x46, 0x48,
		},
		/* S3 : 128 byte */
		{
			0x59, 0xed, 0x09, 0xae, 0x9c, 0xb7, 0x43, 0x14,
			0x67, 0x81, 0xa1, 0xaa, 0x58, 0xc6, 0x08, 0xfe,
			0x10, 0x7e, 0x2f, 0x6a, 0x17, 0x67, 0xd7, 0x74,
			0xa8, 0x4e, 0xeb, 0x00, 0xf8, 0x45, 0x86, 0xe4,
			0xbd, 0x0e, 0xfe, 0x7f, 0x20, 0x1b, 0xe1, 0xcf,
			0xc4, 0xc6, 0x7b, 0x73, 0x44, 0x1f, 0xde, 0x81,
			0x6b, 0x91, 0x53, 0x02, 0x70, 0xc6, 0x2a, 0x34,
			0xd8, 0x6e, 0x5c, 0x8c, 0xda, 0xd4, 0xf0, 0xd0,
			0xb9, 0x0b, 0x53, 0xae, 0xd9, 0xbd, 0x31, 0xc7,
			0xb4, 0x3e, 0x1d, 0x75, 0x17, 0xf4, 0x13, 0xc8,
			0xb1, 0x02, 0x32, 0xab, 0x7d, 0x7a, 0x27, 0x0f,
			0x25, 0x0f, 0x09, 0x81, 0x77, 0x21, 0x61, 0x1f,
			0x6c, 0xc9, 0x61, 0x37, 0xe8, 0x06, 0x7e, 0x8c,
			0xf7, 0x7a, 0x7a, 0xc2, 0x5b, 0xfa, 0x8f, 0x8f,
			0x15, 0x7c, 0x4e, 0xdd, 0xf0, 0x51, 0xda, 0x69,
			0xbe, 0x62, 0x84, 0xe8, 0x77, 0xaa, 0x43, 0x5d,
		},
	},
	/* SHA224 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0x30, 0x22, 0x19, 0xbc, 0x6c, 0x07, 0xe6, 0x28,
			0xf3, 0x52, 0xa4, 0x50, 0x91, 0x71, 0x9b, 0xb0,
			0xe6, 0x55, 0x87, 0x5f, 0xb9, 0x76, 0x91, 0x38,
			0x8f, 0x18, 0x7e, 0x37, 0x7c, 0xcd, 0x27, 0x25,
			0x3a, 0x3c, 0x29, 0x78, 0x78, 0x20, 0xdd, 0x75,
			0x74, 0x94, 0x8b, 0x97, 0xcc, 0x8d, 0xda, 0x34,
			0x92, 0xa8, 0x8b, 0x5e, 0x9e, 0x26, 0x67, 0x08,
			0x28, 0x34, 0x76, 0x69, 0x96, 0x70, 0x72, 0xa6,
			0x92, 0xaa, 0x7b, 0xab, 0x60, 0xe7, 0x0c, 0x59,
			0x6e, 0xac, 0x2e, 0xe7, 0x2d, 0x79, 0xec, 0x82,
			0x20, 0x93, 0x06, 0x37, 0x33, 0x06, 0xea, 0x13,
			0x4d, 0xeb, 0xdf, 0x89, 0x28, 0xe7, 0x84, 0x13,
			0x61, 0xf4, 0x79, 0x1c, 0xcb, 0x64, 0x5c, 0x1e,
			0x09, 0x23, 0xf7, 0x08, 0x5c, 0x3c, 0xb7, 0xe2,
			0xb9, 0x9e, 0x60, 0xb3, 0x1f, 0x21, 0x00, 0xa3,
			0x27, 0xc4, 0x22, 0x5d, 0xdd, 0x38, 0x43, 0xb7,
		},
		/* S4 : 128 byte */
		{
			0x44, 0x15, 0xec, 0x9c, 0xc2, 0x08, 0xf5, 0xc9,
			0x81, 0x52, 0x88, 0x21, 0x63, 0x7b, 0xcb, 0x0e,
			0xbb, 0xbf, 0xa0, 0x84, 0x18, 0xff, 0x93, 0x56,
			0x67, 0xef, 0xcf, 0xc0, 0x88, 0xc3, 0x94, 0x1b,
			0x97, 0x82, 0x80, 0xbb, 0x1c, 0x24, 0x67, 0x62,
			0x88, 0x35, 0x59, 0x46, 0x13, 0x34, 0xb0, 0xef,
			0x26, 0xec, 0xa0, 0xe5, 0xd4, 0xab, 0x72, 0x05,
			0x9f, 0xb5, 0xdd, 0x59, 0xaf, 0xf3, 0xce, 0x5d,
			0x0b, 0x49, 0x13, 0xe1, 0x97, 0x97, 0x56, 0x11,
			0x16, 0xce, 0x39, 0x34, 0xbf, 0x5f, 0x36, 0x2b,
			0x88, 0xfa, 0xe1, 0xbc, 0x9e, 0xbd, 0xa4, 0x47,
			0xec, 0x32, 0xff, 0x31, 0xd2, 0x6e, 0xec, 0x39,
			0x51, 0xeb, 0x87, 0xf0, 0x84, 0x12, 0xbd, 0x3c,
			0x3b, 0x7a, 0x23, 0xb4, 0x4d, 0x73, 0xc6, 0xc2,
			0xeb, 0x81, 0x68, 0x59, 0xf6, 0x7b, 0xec, 0xf8,
			0xdf, 0x11, 0x8c, 0x38, 0x2c, 0x91, 0x7d, 0x42,
		},
	},
	/* SHA224 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0x1f, 0xd6, 0x89, 0x98, 0xe4, 0xc3, 0x4d, 0xed,
			0x43, 0xd2, 0xe5, 0x1b, 0x16, 0x80, 0xa6, 0xb3,
			0xfd, 0xfd, 0xd5, 0x76, 0x03, 0x3b, 0x6a, 0x0d,
			0x7f, 0x44, 0x13, 0xfb, 0x39, 0xd2, 0x7c, 0xcf,
			0xce, 0x19, 0xce, 0x9e, 0x56, 0xe8, 0x76, 0xb0,
			0x43, 0xf0, 0x27, 0x7a, 0x8c, 0x63, 0x22, 0xcc,
			0x0c, 0xc2, 0xb5, 0x4c, 0x25, 0xf0, 0x01, 0x23,
			0xa8, 0x8a, 0x01, 0xf6, 0xf9, 0x7b, 0xc8, 0x18,
			0x6e, 0xca, 0x0a, 0xfe, 0xf9, 0xb7, 0xde, 0xf3,
			0x07, 0x07, 0xc2, 0x0c, 0xa7, 0x9c, 0xe0, 0x5f,
			0xef, 0x48, 0x90, 0x70, 0x9b, 0xe3, 0x1d, 0xec,
			0xf8, 0x9b, 0xcc, 0x9a, 0x01, 0x8c, 0x1a, 0x8e,
			0xc7, 0x91, 0x46, 0xa0, 0x16, 0x59, 0x0f, 0x1c,
			0x55, 0xbd, 0xbf, 0xbb, 0xae, 0x50, 0x68, 0xd2,
			0x6f, 0x38, 0x6e, 0xca, 0xb0, 0x3d, 0x45, 0xd0,
			0x37, 0x20, 0x7c, 0xcd, 0x98, 0x2e, 0xfb, 0x99,
		},
		/* S5 : 128 byte */
		{
			0x63, 0xb9, 0x46, 0xb4, 0xf8, 0xe6, 0x1f, 0x3d,
			0x5d, 0x09, 0xcc, 0xf3, 0x79, 0xcc, 0x8b, 0x6e,
			0x4b, 0x65, 0x07, 0x0a, 0x96, 0x14, 0x4e, 0x13,
			0xf1, 0xf6, 0x8c, 0x17, 0xf9, 0x16, 0x45, 0xdd,
			0xd7, 0x0b, 0x32, 0xa9, 0x76, 0xa5, 0x28, 0x8f,
			0x74, 0x84, 0x57, 0x3c, 0x46, 0x68, 0x7a, 0x82,
			0xeb, 0xa4, 0x9b, 0xc4, 0x21, 0x6c, 0x93, 0x28,
			0xd6, 0xce, 0x8a, 0xfe, 0xdd, 0xff, 0x5b, 0x12,
			0x3e, 0xc0, 0xf4, 0x95, 0xd9, 0x5b, 0xe8, 0x9a,
			0x69, 0x5d, 0x03, 0x78, 0x24, 0x1b, 0x00, 0x53,
			0x02, 0xfb, 0x85, 0xa3, 0x24, 0x29, 0x12, 0x1d,
			0x20, 0x09, 0x97, 0xd7, 0x59, 0xc4, 0x7f, 0x9f,
			0xdd, 0x4e, 0x29, 0xc8, 0x6d, 0xea, 0x2a, 0x7f,
			0x2d, 0xda, 0x2a, 0xec, 0x22, 0xad, 0x48, 0xf6,
			0x7f, 0x6f, 0x3f, 0xdd, 0x37, 0x91, 0xba, 0x84,
			0x1f, 0xc9, 0x71, 0x40, 0x3d, 0x86, 0xce, 0xd7,
		},
	},
	/* SHA224 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0x90, 0xd0, 0xd6, 0xa4, 0x10, 0xf8, 0xaa, 0x86,
			0x8b, 0x64, 0x12, 0x8c, 0x03, 0x9c, 0xc6, 0x98,
			0xa3, 0xb2, 0xe3, 0x67, 0x00, 0x74, 0x49, 0x69,
			0xa0, 0xe7, 0x41, 0xd0, 0xb9, 0xf1, 0x64, 0xe7,
			0x3a, 0x56, 0xb8, 0xa6, 0x0f, 0xe6, 0x1b, 0x83,
			0xa1, 0xf4, 0xe5, 0x8a, 0x9d, 0x09, 0xc9, 0xb0,
			0xf7, 0xda, 0xfc, 0x65, 0x2e, 0x78, 0x99, 0x68,
			0xd1, 0xc7, 0x45, 0xdf, 0x42, 0x31, 0x0f, 0xa7,
			0xba, 0x9b, 0x99, 0xe9, 0x8d, 0x98, 0x7d, 0xee,
			0xb0, 0xdd, 0xea, 0x34, 0x78, 0xf6, 0x8e, 0xc1,
			0xa4, 0x37, 0xb9, 0xb7, 0x9e, 0xf4, 0xbf, 0x29,
			0xfe, 0xf5, 0x9e, 0x2d, 0x50, 0x24, 0xe0, 0x32,
			0x17, 0x89, 0xc3, 0x94, 0x12, 0x78, 0x98, 0x6e,
			0xbd, 0x09, 0x69, 0xb0, 0x1b, 0xc9, 0xde, 0x71,
			0xb3, 0xaf, 0x61, 0x86, 0xd8, 0x52, 0x81, 0x53,
			0x2e, 0x58, 0x94, 0xe1, 0x6a, 0x32, 0xa1, 0x31,
		},
		/* S6 : 128 byte */
		{
			0xbc, 0xa2, 0xb3, 0x9e, 0x56, 0x21, 0xf6, 0xf2,
			0x00, 0x73, 0x77, 0x50, 0x67, 0xb0, 0xb7, 0xfc,
			0xdf, 0x3b, 0xac, 0x40, 0x24, 0x16, 0xf2, 0xc3,
			0x35, 0xb7, 0x29, 0x6e, 0xbf, 0xf5, 0x14, 0xff,
			0x21, 0xe8, 0x96, 0xe8, 0xae, 0xaa, 0x47, 0x01,
			0xce, 0x13, 0x1d, 0xb5, 0xd9, 0x32, 0xa1, 0x6d,
			0xc0, 0xce, 0xae, 0x23, 0x5d, 0x81, 0x9b, 0x37,
			0x9a, 0x42, 0xc9, 0x1d, 0x9f, 0x9e, 0x9d, 0xb2,
			0xe8, 0x65, 0x07, 0x01, 0xf2, 0xe6, 0x96, 0x10,
			0x35, 0xa8, 0x9b, 0x33, 0x94, 0xa6, 0x5f, 0xc2,
			0x82, 0x6a, 0x30, 0xe7, 0xca, 0xfd, 0x0d, 0x87,
			0x26, 0xb1, 0x89, 0x10, 0x2e, 0x9a, 0x9c, 0x8e,
			0x1a, 0xae, 0x15, 0x28, 0x08, 0xaa, 0x2d, 0x1a,
			0x6b, 0x87, 0x40, 0xb6, 0xff, 0xd8, 0x62, 0x36,
			0x6c, 0xd0, 0x71, 0x74, 0x51, 0x39, 0x29, 0xd8,
			0xb6, 0x57, 0x02, 0x04, 0x64, 0x47, 0x97, 0x7f,
		},
	},
	/* SHA224 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0x64, 0x4a, 0x3d, 0xe6, 0x10, 0x0d, 0xd8, 0xa3,
			0x6c, 0x84, 0x14, 0x46, 0xb9, 0x37, 0x6b, 0x34,
			0x5d, 0x47, 0x8a, 0x32, 0x94, 0x50, 0xa6, 0x6f,
			0x62, 0x93, 0x13, 0xc5, 0xcc, 0x55, 0x13, 0x3c,
			0x47, 0x82, 0xec, 0xd0, 0x71, 0x96, 0x3d, 0x74,
			0xff, 0xbd, 0x91, 0x56, 0xf6, 0x39, 0x35, 0x65,
			0x1f, 0x05, 0x47, 0x11, 0xb4, 0xb8, 0x10, 0x51,
			0x60, 0xab, 0x94, 0xeb, 0x67, 0x5e, 0x66, 0xf0,
			0x20, 0x18, 0xc1, 0x85, 0xac, 0xce, 0xd5, 0xe9,
			0xe2, 0x46, 0x5f, 0xb4, 0x89, 0x7d, 0x7c, 0x5d,
			0xca, 0x45, 0xbf, 0xff, 0xe4, 0x32, 0x81, 0x5d,
			0x22, 0xb7, 0x78, 0x23, 0x0b, 0x8d, 0x8c, 0x6f,
			0xd8, 0x54, 0xe5, 0x8f, 0x2b, 0x7d, 0x49, 0x0d,
			0xfd, 0x67, 0xa5, 0xe8, 0xd8, 0xc8, 0xe9, 0x25,
			0x47, 0x4f, 0x19, 0x86, 0x8c, 0x88, 0xa1, 0x9a,
			0x90, 0xff, 0xea, 0xf6, 0x20, 0xa8, 0x2a, 0xc1,
		},
		/* S7 : 128 byte */
		{
			0x48, 0x11, 0x57, 0xd6, 0x5c, 0x91, 0xa0, 0x9b,
			0x87, 0x1b, 0xc9, 0xf7, 0x04, 0xac, 0x2c, 0x31,
			0xb1, 0xe6, 0xb0, 0x97, 0xe0, 0xd4, 0x9e, 0x29,
			0x1e, 0x03, 0x7c, 0x11, 0xe7, 0x07, 0xf4, 0x8a,
			0xc1, 0x66, 0x55, 0x8e, 0x9e, 0x8c, 0x11, 0x4e,
			0xe4, 0x44, 0xc4, 0xd6, 0x49, 0xe7, 0xe2, 0xcf,
			0x73, 0x74, 0x26, 0x9b, 0x03, 0x6f, 0xea, 0x99,
			0xbb, 0x35, 0x66, 0xd1, 0x34, 0xd7, 0xae, 0x5d,
			0xd4, 0x14, 0x1f, 0x73, 0x26, 0x88, 0x47, 0x46,
			0xd3, 0x96, 0xbe, 0x32, 0x07, 0xda, 0x2d, 0x9f,
			0x56, 0x86, 0x25, 0x85, 0x99, 0x3d, 0xb4, 0x2c,
			0xe2, 0x82, 0xaa, 0x2f, 0x04, 0x8d, 0x12, 0xbc,
			0xcf, 0xf1, 0x2f, 0x14, 0xb9, 0xd1, 0xac, 0x4e,
			0x33, 0xd3, 0x71, 0xfd, 0x4c, 0x0d, 0xe1, 0x56,
			0x3c, 0x97, 0x88, 0x26, 0x46, 0x8e, 0x6a, 0x37,
			0x99, 0xf6, 0x4d, 0x39, 0x85, 0xd4, 0x68, 0x9e,
		},
	},
	/* SHA224 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0x29, 0xe9, 0xcb, 0x34, 0x82, 0xe9, 0xed, 0xa5,
			0xe6, 0xd7, 0x14, 0xa2, 0xdc, 0x73, 0x0d, 0x91,
			0xbd, 0xfb, 0x2d, 0x42, 0xb1, 0x4f, 0xe8, 0x2e,
			0xd9, 0x53, 0xa0, 0xfc, 0xce, 0x95, 0xbb, 0x4a,
			0x4b, 0x30, 0xf8, 0xa9, 0x26, 0xf0, 0x03, 0xe9,
			0x85, 0xb6, 0x47, 0x98, 0x4e, 0x1e, 0x03, 0xc5,
			0xdf, 0x60, 0xa0, 0x09, 0x00, 0x74, 0x42, 0xc9,
			0x15, 0x7a, 0x9d, 0xb5, 0x99, 0x58, 0x0a, 0x94,
			0xc7, 0xa5, 0xd9, 0x44, 0x9a, 0xc6, 0xe9, 0xfc,
			0xf8, 0x56, 0x77, 0xd5, 0x2a, 0xcc, 0x34, 0x89,
			0x90, 0x57, 0x9a, 0x7a, 0x93, 0x0d, 0x7f, 0xf6,
			0xda, 0x45, 0xeb, 0xb9, 0xc1, 0x41, 0x27, 0xb3,
			0x07, 0x11, 0x16, 0x0c, 0xc6, 0xb3, 0xc8, 0x66,
			0xa9, 0x7f, 0x4d, 0x62, 0x59, 0xc3, 0xc7, 0x65,
			0x38, 0xaa, 0xc3, 0x9c, 0x52, 0x62, 0xca, 0x3d,
			0x91, 0x7d, 0x32, 0x10, 0x30, 0x98, 0x3a, 0x2f,
		},
		/* S8 : 128 byte */
		{
			0x3c, 0x23, 0xc9, 0xe5, 0xab, 0x29, 0xb8, 0x10,
			0x2e, 0x03, 0x93, 0x29, 0x34, 0x61, 0x64, 0x81,
			0x01, 0x37, 0x89, 0xe1, 0x9d, 0xbe, 0x66, 0x04,
			0xee, 0x9b, 0xf0, 0xa6, 0x4c, 0xcb, 0x5e, 0x3b,
			0x73, 0xfe, 0x74, 0x7c, 0xe8, 0xb8, 0xf1, 0xef,
			0xce, 0x2a, 0x99, 0x07, 0xe0, 0x41, 0x99, 0xeb,
			0x21, 0x1c, 0xeb, 0x90, 0xbe, 0x28, 0x4b, 0xa7,
			0x13, 0x74, 0xa9, 0x42, 0x23, 0xa0, 0x7e, 0xa9,
			0xa7, 0x39, 0xca, 0x72, 0xfa, 0xa7, 0xca, 0x4d,
			0xa2, 0x9b, 0xc1, 0xc0, 0x44, 0xaf, 0x61, 0xe4,
			0xc5, 0x63, 0xb6, 0x66, 0x3a, 0xf2, 0x05, 0x0e,
			0x2a, 0x00, 0x04, 0xfe, 0x9e, 0x88, 0x39, 0x51,
			0x30, 0x0a, 0x25, 0xa4, 0x5f, 0xfc, 0x3b, 0xab,
			0x88, 0x84, 0xd9, 0xf0, 0xf2, 0x27, 0x9a, 0x26,
			0xcc, 0xe7, 0x14, 0x2d, 0xd6, 0x77, 0x68, 0x62,
			0xab, 0x63, 0xad, 0x63, 0x6c, 0x9a, 0xcc, 0x95,
		},
	},
	/* SHA224 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0x46, 0x7e, 0x8e, 0xa6, 0x34, 0xf7, 0x99, 0x5d,
			0xc4, 0x6c, 0x11, 0xb8, 0xab, 0x0b, 0x75, 0x08,
			0x89, 0x46, 0x81, 0xe8, 0x1c, 0x35, 0x02, 0xc3,
			0xb3, 0x35, 0xe8, 0x97, 0xe6, 0xd6, 0x9d, 0xf8,
			0x85, 0xf4, 0x95, 0x57, 0xce, 0x23, 0x27, 0x84,
			0xe3, 0x51, 0x9b, 0x72, 0x7b, 0xa6, 0x84, 0x3b,
			0xd7, 0xaf, 0x50, 0x63, 0xf8, 0xbc, 0x1d, 0x61,
			0x0f, 0x86, 0xce, 0x5b, 0x35, 0x15, 0x5e, 0x32,
			0x5c, 0xe1, 0x75, 0xbe, 0x85, 0x38, 0x39, 0x5b,
			0x34, 0xdf, 0x67, 0xa4, 0x21, 0xfc, 0xa2, 0x7e,
			0x31, 0x33, 0x0b, 0x59, 0xa4, 0x10, 0x11, 0xb2,
			0x90, 0xa5, 0x8b, 0xdc, 0x8e, 0x74, 0x04, 0x01,
			0xb3, 0x8f, 0x55, 0x64, 0xc2, 0xfd, 0x7a, 0xe8,
			0x9f, 0x60, 0x9e, 0xd6, 0x07, 0xd5, 0x78, 0xdb,
			0x7f, 0x1c, 0xda, 0x50, 0x8a, 0xf9, 0x87, 0xbe,
			0x1f, 0xd9, 0x46, 0xa2, 0x5a, 0xb9, 0x34, 0x6d,
		},
		/* S9 : 128 byte */
		{
			0x7f, 0xb4, 0x85, 0xa6, 0x2f, 0x83, 0x7a, 0x47,
			0x65, 0x30, 0x8e, 0xd0, 0x05, 0xa6, 0x1f, 0xb8,
			0x21, 0xa9, 0xdd, 0xfa, 0x6b, 0xab, 0x4e, 0x76,
			0x54, 0x83, 0x42, 0xe8, 0xc2, 0xa8, 0x53, 0xba,
			0x1a, 0xb7, 0x20, 0xcc, 0x37, 0xe3, 0xe3, 0x66,
			0x14, 0x08, 0xa3, 0x6b, 0xea, 0xd4, 0xae, 0x02,
			0x07, 0xc5, 0x05, 0xb8, 0xcd, 0x9f, 0xa6, 0x7f,
			0x0e, 0x73, 0xce, 0x29, 0xbf, 0x66, 0xd1, 0x87,
			0x3c, 0x98, 0xf9, 0xb4, 0x84, 0x9e, 0x4e, 0xd9,
			0x87, 0x27, 0xb2, 0x06, 0xea, 0xbb, 0x9f, 0xa4,
			0xcb, 0xca, 0xc1, 0xeb, 0x52, 0xe2, 0x0c, 0x3d,
			0x9b, 0x12, 0x44, 0x3d, 0xd1, 0xe7, 0x46, 0xa7,
			0x55, 0xba, 0xbe, 0xb3, 0x40, 0x87, 0x0e, 0x66,
			0x01, 0x12, 0xf2, 0xce, 0x18, 0x65, 0x39, 0xe9,
			0x43, 0xee, 0x7f, 0xb3, 0x35, 0xad, 0xbe, 0xe0,
			0x45, 0xd6, 0x6a, 0x30, 0x16, 0xe8, 0xc4, 0x39,
		},
	},
};

/* 10 elements */
static mod1024_t SHA256[] = {
	/* SHA256 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0xe5, 0x67, 0xa3, 0x9a, 0xe4, 0xe5, 0xef, 0x9b,
			0x68, 0x01, 0xea, 0x05, 0x61, 0xb7, 0x2a, 0x5d,
			0x4b, 0x5f, 0x38, 0x5f, 0x05, 0x32, 0xfc, 0x9f,
			0xe1, 0x0a, 0x75, 0x70, 0xf8, 0x69, 0xae, 0x05,
			0xc0, 0xbd, 0xed, 0xd6, 0xe0, 0xe2, 0x2d, 0x45,
			0x42, 0xe9, 0xce, 0x82, 0x6a, 0x18, 0x8c, 0xac,
			0x07, 0x31, 0xae, 0x39, 0xc8, 0xf8, 0x7f, 0x97,
			0x71, 0xef, 0x02, 0x13, 0x2e, 0x64, 0xe2, 0xfb,
			0x27, 0xad, 0xa8, 0xff, 0x54, 0xb3, 0x30, 0xdd,
			0x93, 0xad, 0x5e, 0x3e, 0xf8, 0x2e, 0x0d, 0xda,
			0x64, 0x62, 0x48, 0xe3, 0x59, 0x94, 0xbd, 0xa1,
			0x0c, 0xf4, 0x6e, 0x5a, 0xbc, 0x98, 0xaa, 0x74,
			0x43, 0xc0, 0x3c, 0xdd, 0xeb, 0x5e, 0xe2, 0xab,
			0x82, 0xd6, 0x01, 0x00, 0xb1, 0x02, 0x96, 0x31,
			0x89, 0x79, 0x70, 0x27, 0x5f, 0x11, 0x9d, 0x05,
			0xda, 0xa2, 0x22, 0x0a, 0x4a, 0x0d, 0xef, 0xba,
		},
		/* S0 : 128 byte */
		{
			0x0e, 0x7c, 0xdd, 0x12, 0x1e, 0x40, 0x32, 0x3c,
			0xa6, 0x11, 0x5d, 0x1e, 0xc6, 0xd1, 0xf9, 0x56,
			0x17, 0x38, 0x45, 0x5f, 0x0e, 0x9e, 0x1c, 0xd8,
			0x58, 0xe8, 0xb5, 0x66, 0xae, 0x2d, 0xa5, 0xe8,
			0xee, 0x63, 0xd8, 0xf1, 0x5c, 0x3c, 0xdd, 0x88,
			0x02, 0x7e, 0x13, 0x40, 0x6d, 0xb6, 0x09, 0x36,
			0x9c, 0x88, 0xca, 0x99, 0xb3, 0x4f, 0xa1, 0x56,
			0xc7, 0xee, 0x62, 0xbc, 0x5a, 0x39, 0x23, 0xbb,
			0x5a, 0x1e, 0xda, 0xbd, 0x45, 0xc1, 0xa4, 0x22,
			0xaa, 0xfc, 0xbb, 0x47, 0xe0, 0x94, 0x7f, 0x35,
			0xcf, 0xef, 0x87, 0x97, 0x0b, 0x4b, 0x71, 0x31,
			0x62, 0xb2, 0x19, 0x16, 0xca, 0xfb, 0x8c, 0x86,
			0x4a, 0x3e, 0x5b, 0x9f, 0xfc, 0x98, 0x94, 0x01,
			0xd4, 0xea, 0xe9, 0x92, 0x31, 0x2a, 0x32, 0xc5,
			0xbc, 0x88, 0xab, 0xbb, 0x45, 0xf9, 0x9a, 0xc8,
			0x85, 0xb5, 0x4d, 0x6b, 0x8e, 0x61, 0xb6, 0xec,
		},
	},
	/* SHA256 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0xdb, 0x9e, 0xae, 0x66, 0xbc, 0xac, 0xb0, 0x1b,
			0xa8, 0x02, 0x9f, 0xba, 0xc5, 0xfe, 0xf1, 0x31,
			0x56, 0xaa, 0x1a, 0x76, 0x8c, 0x3d, 0xff, 0x25,
			0x9e, 0xe5, 0xae, 0x6b, 0x1c, 0xc3, 0x4c, 0x22,
			0x80, 0x37, 0x4e, 0x3d, 0x06, 0x5f, 0xed, 0x67,
			0xbd, 0xda, 0xd3, 0xe7, 0xde, 0xd2, 0x1b, 0xc7,
			0x5d, 0x39, 0x78, 0x35, 0xbf, 0x52, 0xb4, 0x29,
			0x2b, 0x74, 0x5d, 0x48, 0xc3, 0x0b, 0xbb, 0x09,
			0x35, 0xdf, 0x04, 0x16, 0x8c, 0x99, 0xd3, 0xf5,
			0x4d, 0x83, 0xd7, 0xe7, 0xbf, 0x8f, 0xc8, 0x10,
			0x8a, 0x9b, 0xa0, 0xd9, 0x83, 0xf3, 0x07, 0x93,
			0xc7, 0x17, 0x0f, 0x5d, 0x07, 0xbf, 0x21, 0x47,
			0x22, 0x1d, 0x38, 0xb7, 0xf9, 0x62, 0x4c, 0x0c,
			0x7f, 0x83, 0x12, 0x82, 0x0f, 0x3d, 0xe2, 0x55,
			0x03, 0x57, 0xf8, 0x2a, 0x82, 0x27, 0xe0, 0xaa,
			0x99, 0x57, 0x2c, 0x70, 0x8d, 0xe9, 0x68, 0x25,
		},
		/* S1 : 128 byte */
		{
			0x5a, 0x61, 0x6f, 0xae, 0xb0, 0x06, 0x80, 0xf5,
			0xc4, 0xef, 0x63, 0x32, 0x05, 0x04, 0x0b, 0x49,
			0x7b, 0x5e, 0x5e, 0x22, 0x6e, 0x4a, 0x8f, 0x49,
			0x3b, 0x1e, 0xc2, 0xa2, 0x6f, 0xe7, 0xa0, 0x97,
			0x1f, 0xed, 0x4e, 0x1b, 0x8f, 0x18, 0x8e, 0xba,
			0x81, 0x61, 0x26, 0x6f, 0xe5, 0x58, 0xea, 0xb5,
			0x39, 0xf9, 0x03, 0xc0, 0xcd, 0x8e, 0xd1, 0x8d,
			0x21, 0xaf, 0x77, 0xe3, 0x0c, 0xb2, 0x64, 0xc6,
			0x21, 0x4a, 0x17, 0x55, 0x88, 0xed, 0x54, 0x2b,
			0x5f, 0xbd, 0xaa, 0xe9, 0x9e, 0x92, 0x40, 0x94,
			0xe8, 0xc8, 0xa2, 0x23, 0x66, 0x44, 0x1d, 0x86,
			0x61, 0x26, 0x43, 0x3a, 0xa4, 0x5c, 0xd3, 0x77,
			0x73, 0xb9, 0xdc, 0x3d, 0xc1, 0x79, 0xec, 0x8c,
			0xf5, 0x1e, 0xfb, 0x8b, 0xbd, 0xed, 0xcd, 0x22,
			0x9c, 0xd2, 0xdc, 0x23, 0x34, 0xf5, 0xbf, 0x2b,
			0x0b, 0x00, 0x37, 0x43, 0x50, 0xe6, 0x14, 0x7d,
		},
	},
	/* SHA256 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x2e, 0x22, 0x65, 0x6e, 0x48, 0x85, 0x62, 0x40,
			0xb0, 0xf6, 0x55, 0xd2, 0x02, 0x2d, 0x69, 0xac,
			0x73, 0x1a, 0x1c, 0x63, 0x15, 0x1b, 0xa3, 0x6e,
			0x57, 0x72, 0x8a, 0x35, 0xcb, 0xf6, 0x9c, 0x6b,
			0x2d, 0x08, 0x02, 0x9e, 0xa0, 0xc9, 0xaf, 0x0c,
			0x3a, 0x8b, 0xfb, 0xb9, 0xc3, 0x85, 0x8d, 0xbe,
			0xba, 0xca, 0x7f, 0xe4, 0x99, 0x7c, 0xbd, 0x71,
			0x5b, 0x3c, 0xae, 0x42, 0x3a, 0xe6, 0x17, 0x1a,
			0xbd, 0x7b, 0x3c, 0x3a, 0xf2, 0x5f, 0x48, 0x32,
			0xf8, 0xc5, 0xee, 0xf6, 0xc1, 0x67, 0x51, 0x34,
			0x16, 0x7a, 0x21, 0xe5, 0xdc, 0xe1, 0x07, 0x23,
			0x95, 0xa1, 0x44, 0x3a, 0x2a, 0x95, 0x95, 0x01,
			0xe7, 0x63, 0x59, 0x6b, 0xc7, 0xae, 0xf5, 0x59,
			0xf2, 0x8d, 0x78, 0xb2, 0x84, 0x3c, 0x7b, 0xb6,
			0x90, 0x12, 0x4c, 0x8f, 0x65, 0xb3, 0x4b, 0x29,
			0x0f, 0x87, 0x93, 0x44, 0x21, 0x6b, 0xdd, 0xc8,
		},
		/* S2 : 128 byte */
		{
			0xc2, 0x4d, 0xd3, 0x9d, 0xc4, 0x30, 0x6b, 0xb0,
			0xb0, 0x6d, 0x54, 0x80, 0x91, 0x71, 0x48, 0x20,
			0xf1, 0x4a, 0x39, 0x38, 0x43, 0x7e, 0xe7, 0x6d,
			0x53, 0x02, 0xf7, 0x00, 0x62, 0x13, 0xd3, 0x88,
			0x04, 0x1b, 0x79, 0xdb, 0x61, 0x25, 0xc3, 0x37,
			0xce, 0x41, 0xd2, 0x08, 0x6f, 0x25, 0x36, 0x02,
			0x6f, 0x0f, 0xff, 0xd2, 0x0a, 0xa9, 0xda, 0xcf,
			0xfe, 0x4e, 0x60, 0x1d, 0xfe, 0x51, 0x31, 0x07,
			0x29, 0xad, 0x05, 0x0c, 0x4f, 0x39, 0x6b, 0x02,
			0x23, 0x49, 0x2a, 0x63, 0xd9, 0xa0, 0x11, 0xf7,
			0x0c, 0xeb, 0xf8, 0x85, 0x0f, 0x61, 0xf0, 0xca,
			0x2e, 0xa9, 0x48, 0x00, 0xf5, 0x1e, 0x35, 0xbb,
			0xb0, 0x77, 0xe9, 0x9b, 0xea, 0xf1, 0x2f, 0xc2,
			0xca, 0xf1, 0x74, 0x47, 0x00, 0xeb, 0x1e, 0x02,
			0x7b, 0xbb, 0x45, 0x03, 0x08, 0xe1, 0x67, 0x43,
			0x72, 0x72, 0x43, 0x3e, 0xe7, 0xa7, 0x65, 0xfb,
		},
	},
	/* SHA256 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0xcf, 0x78, 0x50, 0x02, 0x29, 0x2f, 0x1e, 0xf9,
			0x64, 0x5f, 0x2f, 0x2c, 0xdc, 0x04, 0x55, 0xb2,
			0x9d, 0x0b, 0xa1, 0xef, 0xc9, 0x1f, 0xde, 0x0a,
			0x3a, 0xc0, 0x5f, 0xa1, 0x69, 0x95, 0xfb, 0xbc,
			0x8e, 0x03, 0x55, 0x00, 0x09, 0xfe, 0x61, 0x26,
			0x01, 0x1f, 0x23, 0xab, 0x4b, 0xf7, 0xba, 0xc8,
			0x8d, 0xa4, 0x9e, 0xa4, 0xa3, 0xa1, 0x26, 0xc9,
			0x41, 0xe3, 0x4d, 0xd4, 0xda, 0x21, 0xb5, 0x71,
			0xc1, 0x73, 0xec, 0x87, 0x8f, 0x1e, 0xef, 0xb0,
			0xc4, 0xae, 0xec, 0xec, 0xb1, 0x46, 0x4c, 0x96,
			0x95, 0x35, 0xf1, 0x97, 0x06, 0xc9, 0xbc, 0xd9,
			0x92, 0x68, 0x53, 0xff, 0xa7, 0xdc, 0x21, 0x53,
			0xb0, 0xef, 0x9e, 0xff, 0x81, 0x38, 0xd0, 0x81,
			0xf5, 0x34, 0x12, 0x59, 0xd5, 0x99, 0x14, 0x04,
			0xfc, 0xe6, 0x23, 0x2e, 0xb9, 0x3f, 0xaa, 0x24,
			0x75, 0x78, 0xf9, 0x88, 0x96, 0x70, 0x46, 0x48,
		},
		/* S3 : 128 byte */
		{
			0xb3, 0x4c, 0x8e, 0x11, 0xcb, 0xfd, 0x29, 0x27,
			0xbc, 0xbf, 0x44, 0x2a, 0x6a, 0xfb, 0xbd, 0x6e,
			0xa5, 0xc2, 0x0f, 0x5e, 0x47, 0xcf, 0x56, 0xc5,
			0x7d, 0xca, 0x93, 0xe1, 0x0b, 0xda, 0xbb, 0x20,
			0xb9, 0x58, 0x99, 0x7b, 0x21, 0x6b, 0x92, 0xd0,
			0x9f, 0xb9, 0x2c, 0xfc, 0x34, 0x45, 0x34, 0x6c,
			0xc9, 0xe6, 0x37, 0x21, 0xb7, 0x3d, 0xb0, 0x6d,
			0xc9, 0xe3, 0x67, 0x27, 0xb8, 0xda, 0xa5, 0xf1,
			0x6f, 0xa4, 0x95, 0x9c, 0x97, 0x7f, 0x6e, 0xef,
			0xcd, 0x2f, 0xc9, 0x79, 0x0c, 0x5e, 0x01, 0xf3,
			0x1f, 0x81, 0x90, 0x36, 0x0d, 0x38, 0x0f, 0x43,
			0x32, 0x52, 0x64, 0x72, 0xe3, 0x2b, 0x25, 0xe5,
			0x47, 0x81, 0xbd, 0xde, 0x98, 0x01, 0x6b, 0xc2,
			0x5e, 0xf8, 0x69, 0x7d, 0x5d, 0x73, 0xa6, 0xb7,
			0xbb, 0xa0, 0x6f, 0xd3, 0xd8, 0xb9, 0xeb, 0xe3,
			0xc6, 0x57, 0xb8, 0x6f, 0x07, 0x02, 0x8c, 0xed,
		},
	},
	/* SHA256 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0x30, 0x22, 0x19, 0xbc, 0x6c, 0x07, 0xe6, 0x28,
			0xf3, 0x52, 0xa4, 0x50, 0x91, 0x71, 0x9b, 0xb0,
			0xe6, 0x55, 0x87, 0x5f, 0xb9, 0x76, 0x91, 0x38,
			0x8f, 0x18, 0x7e, 0x37, 0x7c, 0xcd, 0x27, 0x25,
			0x3a, 0x3c, 0x29, 0x78, 0x78, 0x20, 0xdd, 0x75,
			0x74, 0x94, 0x8b, 0x97, 0xcc, 0x8d, 0xda, 0x34,
			0x92, 0xa8, 0x8b, 0x5e, 0x9e, 0x26, 0x67, 0x08,
			0x28, 0x34, 0x76, 0x69, 0x96, 0x70, 0x72, 0xa6,
			0x92, 0xaa, 0x7b, 0xab, 0x60, 0xe7, 0x0c, 0x59,
			0x6e, 0xac, 0x2e, 0xe7, 0x2d, 0x79, 0xec, 0x82,
			0x20, 0x93, 0x06, 0x37, 0x33, 0x06, 0xea, 0x13,
			0x4d, 0xeb, 0xdf, 0x89, 0x28, 0xe7, 0x84, 0x13,
			0x61, 0xf4, 0x79, 0x1c, 0xcb, 0x64, 0x5c, 0x1e,
			0x09, 0x23, 0xf7, 0x08, 0x5c, 0x3c, 0xb7, 0xe2,
			0xb9, 0x9e, 0x60, 0xb3, 0x1f, 0x21, 0x00, 0xa3,
			0x27, 0xc4, 0x22, 0x5d, 0xdd, 0x38, 0x43, 0xb7,
		},
		/* S4 : 128 byte */
		{
			0x68, 0xed, 0xf7, 0x86, 0xf1, 0x66, 0xf9, 0xaa,
			0x93, 0x28, 0x16, 0xd7, 0x0b, 0x77, 0x86, 0xe1,
			0xd8, 0x81, 0x28, 0xda, 0x0c, 0xe1, 0x21, 0x2b,
			0xba, 0x5b, 0x56, 0xfc, 0x8f, 0x8d, 0xc2, 0x1d,
			0x54, 0xb5, 0xbd, 0x44, 0x40, 0x1a, 0xf2, 0x11,
			0x9c, 0x4a, 0x4f, 0x03, 0xcd, 0xbf, 0x75, 0x61,
			0x5c, 0xf0, 0x8f, 0x37, 0x64, 0xfb, 0xa0, 0x3a,
			0xed, 0x85, 0x6e, 0xcd, 0x01, 0xca, 0xa1, 0x5f,
			0xfd, 0x6a, 0x36, 0x04, 0xa6, 0x12, 0xc8, 0x12,
			0x18, 0xf3, 0xc3, 0x95, 0xb9, 0x31, 0xfd, 0x5f,
			0xe7, 0x80, 0x31, 0xb6, 0x74, 0x36, 0x9e, 0x49,
			0xb1, 0x85, 0xde, 0x6c, 0x00, 0xb6, 0xef, 0xfd,
			0xd8, 0xd1, 0x0a, 0xab, 0x7b, 0xfc, 0x66, 0x71,
			0x90, 0x0c, 0x58, 0xc6, 0x88, 0x28, 0x50, 0xde,
			0x5a, 0x37, 0xb8, 0x7c, 0x15, 0xca, 0xab, 0x18,
			0xb8, 0xcd, 0x6b, 0xf8, 0xe5, 0x13, 0x20, 0x20,
		},
	},
	/* SHA256 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0x1f, 0xd6, 0x89, 0x98, 0xe4, 0xc3, 0x4d, 0xed,
			0x43, 0xd2, 0xe5, 0x1b, 0x16, 0x80, 0xa6, 0xb3,
			0xfd, 0xfd, 0xd5, 0x76, 0x03, 0x3b, 0x6a, 0x0d,
			0x7f, 0x44, 0x13, 0xfb, 0x39, 0xd2, 0x7c, 0xcf,
			0xce, 0x19, 0xce, 0x9e, 0x56, 0xe8, 0x76, 0xb0,
			0x43, 0xf0, 0x27, 0x7a, 0x8c, 0x63, 0x22, 0xcc,
			0x0c, 0xc2, 0xb5, 0x4c, 0x25, 0xf0, 0x01, 0x23,
			0xa8, 0x8a, 0x01, 0xf6, 0xf9, 0x7b, 0xc8, 0x18,
			0x6e, 0xca, 0x0a, 0xfe, 0xf9, 0xb7, 0xde, 0xf3,
			0x07, 0x07, 0xc2, 0x0c, 0xa7, 0x9c, 0xe0, 0x5f,
			0xef, 0x48, 0x90, 0x70, 0x9b, 0xe3, 0x1d, 0xec,
			0xf8, 0x9b, 0xcc, 0x9a, 0x01, 0x8c, 0x1a, 0x8e,
			0xc7, 0x91, 0x46, 0xa0, 0x16, 0x59, 0x0f, 0x1c,
			0x55, 0xbd, 0xbf, 0xbb, 0xae, 0x50, 0x68, 0xd2,
			0x6f, 0x38, 0x6e, 0xca, 0xb0, 0x3d, 0x45, 0xd0,
			0x37, 0x20, 0x7c, 0xcd, 0x98, 0x2e, 0xfb, 0x99,
		},
		/* S5 : 128 byte */
		{
			0x06, 0x87, 0xbc, 0xa5, 0xc5, 0x1b, 0xbd, 0xdd,
			0xdb, 0x68, 0xd2, 0x1a, 0x91, 0x2b, 0x7d, 0x3f,
			0x89, 0xa5, 0x45, 0x17, 0xb7, 0xf3, 0xc6, 0xea,
			0x5d, 0xa3, 0x86, 0xb5, 0xfb, 0x6b, 0xe8, 0xc7,
			0xa1, 0x72, 0xd6, 0x52, 0x8b, 0x2d, 0xfe, 0x9c,
			0xdb, 0xcb, 0x3f, 0x4c, 0xca, 0x10, 0x99, 0xa1,
			0x00, 0x3a, 0x69, 0x17, 0x6e, 0xb3, 0x29, 0x2a,
			0x15, 0x71, 0xb8, 0x45, 0x16, 0x93, 0xe1, 0x55,
			0x4c, 0x6e, 0xc5, 0xe9, 0xd3, 0x36, 0x77, 0x13,
			0x78, 0x71, 0x8a, 0x9e, 0x82, 0x2f, 0x02, 0xf7,
			0x83, 0xb9, 0xce, 0xe3, 0xc0, 0xf8, 0xff, 0x26,
			0x54, 0x53, 0x9c, 0x49, 0x54, 0x78, 0xba, 0x68,
			0xcc, 0xa6, 0xa6, 0x68, 0x33, 0x03, 0xa1, 0x04,
			0x23, 0x0e, 0x4d, 0xde, 0x1c, 0x35, 0xe1, 0x94,
			0xeb, 0xa1, 0xea, 0x50, 0xe7, 0xfe, 0x69, 0x0c,
			0xb1, 0x9d, 0x2b, 0x11, 0x18, 0xa8, 0x7b, 0xe8,
		},
	},
	/* SHA256 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0x90, 0xd0, 0xd6, 0xa4, 0x10, 0xf8, 0xaa, 0x86,
			0x8b, 0x64, 0x12, 0x8c, 0x03, 0x9c, 0xc6, 0x98,
			0xa3, 0xb2, 0xe3, 0x67, 0x00, 0x74, 0x49, 0x69,
			0xa0, 0xe7, 0x41, 0xd0, 0xb9, 0xf1, 0x64, 0xe7,
			0x3a, 0x56, 0xb8, 0xa6, 0x0f, 0xe6, 0x1b, 0x83,
			0xa1, 0xf4, 0xe5, 0x8a, 0x9d, 0x09, 0xc9, 0xb0,
			0xf7, 0xda, 0xfc, 0x65, 0x2e, 0x78, 0x99, 0x68,
			0xd1, 0xc7, 0x45, 0xdf, 0x42, 0x31, 0x0f, 0xa7,
			0xba, 0x9b, 0x99, 0xe9, 0x8d, 0x98, 0x7d, 0xee,
			0xb0, 0xdd, 0xea, 0x34, 0x78, 0xf6, 0x8e, 0xc1,
			0xa4, 0x37, 0xb9, 0xb7, 0x9e, 0xf4, 0xbf, 0x29,
			0xfe, 0xf5, 0x9e, 0x2d, 0x50, 0x24, 0xe0, 0x32,
			0x17, 0x89, 0xc3, 0x94, 0x12, 0x78, 0x98, 0x6e,
			0xbd, 0x09, 0x69, 0xb0, 0x1b, 0xc9, 0xde, 0x71,
			0xb3, 0xaf, 0x61, 0x86, 0xd8, 0x52, 0x81, 0x53,
			0x2e, 0x58, 0x94, 0xe1, 0x6a, 0x32, 0xa1, 0x31,
		},
		/* S6 : 128 byte */
		{
			0x13, 0x04, 0x90, 0x92, 0xc6, 0x75, 0x80, 0x9b,
			0x38, 0x64, 0x59, 0xec, 0xcb, 0xe1, 0xc5, 0x40,
			0xa1, 0xa8, 0xe9, 0xcb, 0x67, 0x4d, 0xa9, 0xd1,
			0x13, 0x49, 0xf2, 0xc5, 0x0e, 0x55, 0x93, 0x87,
			0xf0, 0x19, 0x15, 0x42, 0x16, 0x37, 0x07, 0x52,
			0xc8, 0xae, 0x0a, 0x7c, 0x4b, 0x6e, 0x03, 0x1f,
			0x41, 0x4f, 0x7a, 0x32, 0xa8, 0xeb, 0xf6, 0xf4,
			0xf8, 0x7d, 0x46, 0x92, 0xca, 0xa4, 0x4e, 0x53,
			0x55, 0x65, 0xe0, 0xc0, 0xde, 0xb9, 0x47, 0x14,
			0x11, 0x12, 0x6f, 0x6d, 0xfe, 0xd6, 0xdd, 0x86,
			0x0e, 0xdd, 0x1e, 0x97, 0x46, 0x8b, 0x7e, 0x14,
			0x4f, 0x8e, 0xa6, 0xe0, 0xb5, 0x06, 0x3d, 0x3e,
			0xbe, 0xa5, 0x98, 0xb4, 0x7e, 0xf9, 0xd6, 0x29,
			0x2a, 0xb0, 0xc0, 0x5f, 0x56, 0x21, 0xd2, 0x0b,
			0xcb, 0x38, 0x9d, 0x55, 0x0b, 0xee, 0x84, 0xd1,
			0x90, 0xd7, 0x9c, 0x7d, 0x88, 0x20, 0xa9, 0x7d,
		},
	},
	/* SHA256 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0x64, 0x4a, 0x3d, 0xe6, 0x10, 0x0d, 0xd8, 0xa3,
			0x6c, 0x84, 0x14, 0x46, 0xb9, 0x37, 0x6b, 0x34,
			0x5d, 0x47, 0x8a, 0x32, 0x94, 0x50, 0xa6, 0x6f,
			0x62, 0x93, 0x13, 0xc5, 0xcc, 0x55, 0x13, 0x3c,
			0x47, 0x82, 0xec, 0xd0, 0x71, 0x96, 0x3d, 0x74,
			0xff, 0xbd, 0x91, 0x56, 0xf6, 0x39, 0x35, 0x65,
			0x1f, 0x05, 0x47, 0x11, 0xb4, 0xb8, 0x10, 0x51,
			0x60, 0xab, 0x94, 0xeb, 0x67, 0x5e, 0x66, 0xf0,
			0x20, 0x18, 0xc1, 0x85, 0xac, 0xce, 0xd5, 0xe9,
			0xe2, 0x46, 0x5f, 0xb4, 0x89, 0x7d, 0x7c, 0x5d,
			0xca, 0x45, 0xbf, 0xff, 0xe4, 0x32, 0x81, 0x5d,
			0x22, 0xb7, 0x78, 0x23, 0x0b, 0x8d, 0x8c, 0x6f,
			0xd8, 0x54, 0xe5, 0x8f, 0x2b, 0x7d, 0x49, 0x0d,
			0xfd, 0x67, 0xa5, 0xe8, 0xd8, 0xc8, 0xe9, 0x25,
			0x47, 0x4f, 0x19, 0x86, 0x8c, 0x88, 0xa1, 0x9a,
			0x90, 0xff, 0xea, 0xf6, 0x20, 0xa8, 0x2a, 0xc1,
		},
		/* S7 : 128 byte */
		{
			0x43, 0xe6, 0x4b, 0x24, 0x5e, 0x65, 0xea, 0xda,
			0x15, 0xe6, 0x6d, 0xbc, 0xf1, 0x23, 0x32, 0xae,
			0x80, 0x63, 0x79, 0x17, 0xe6, 0x5a, 0x68, 0xf9,
			0xf9, 0xde, 0x45, 0xbe, 0x6c, 0xe1, 0x85, 0x4a,
			0x58, 0x26, 0x34, 0xc1, 0x39, 0x89, 0x2d, 0x5a,
			0xa2, 0x91, 0x87, 0x80, 0x1c, 0x7c, 0xa4, 0x7a,
			0xf5, 0xde, 0xd8, 0x5e, 0x5b, 0x0a, 0x32, 0xe8,
			0x25, 0xd7, 0x06, 0xf6, 0xeb, 0x0b, 0x2d, 0xff,
			0xa2, 0xf8, 0x0f, 0x69, 0xfe, 0xca, 0xf8, 0x7a,
			0xad, 0x09, 0x19, 0x76, 0x5c, 0xc2, 0xcb, 0x20,
			0x42, 0xe1, 0x24, 0xed, 0xa7, 0xcd, 0x15, 0x7b,
			0xed, 0xd3, 0x21, 0xcc, 0x12, 0x10, 0x0f, 0x8c,
			0xeb, 0x3b, 0x90, 0xb6, 0x8d, 0xa1, 0x3e, 0x5a,
			0xa6, 0x5c, 0x2c, 0x3e, 0xc1, 0x84, 0xc6, 0xab,
			0xbb, 0xf8, 0x6d, 0xcb, 0xff, 0xdf, 0x1c, 0xe3,
			0x6f, 0x1b, 0x55, 0x63, 0xe8, 0xb5, 0x60, 0x44,
		},
	},
	/* SHA256 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0x29, 0xe9, 0xcb, 0x34, 0x82, 0xe9, 0xed, 0xa5,
			0xe6, 0xd7, 0x14, 0xa2, 0xdc, 0x73, 0x0d, 0x91,
			0xbd, 0xfb, 0x2d, 0x42, 0xb1, 0x4f, 0xe8, 0x2e,
			0xd9, 0x53, 0xa0, 0xfc, 0xce, 0x95, 0xbb, 0x4a,
			0x4b, 0x30, 0xf8, 0xa9, 0x26, 0xf0, 0x03, 0xe9,
			0x85, 0xb6, 0x47, 0x98, 0x4e, 0x1e, 0x03, 0xc5,
			0xdf, 0x60, 0xa0, 0x09, 0x00, 0x74, 0x42, 0xc9,
			0x15, 0x7a, 0x9d, 0xb5, 0x99, 0x58, 0x0a, 0x94,
			0xc7, 0xa5, 0xd9, 0x44, 0x9a, 0xc6, 0xe9, 0xfc,
			0xf8, 0x56, 0x77, 0xd5, 0x2a, 0xcc, 0x34, 0x89,
			0x90, 0x57, 0x9a, 0x7a, 0x93, 0x0d, 0x7f, 0xf6,
			0xda, 0x45, 0xeb, 0xb9, 0xc1, 0x41, 0x27, 0xb3,
			0x07, 0x11, 0x16, 0x0c, 0xc6, 0xb3, 0xc8, 0x66,
			0xa9, 0x7f, 0x4d, 0x62, 0x59, 0xc3, 0xc7, 0x65,
			0x38, 0xaa, 0xc3, 0x9c, 0x52, 0x62, 0xca, 0x3d,
			0x91, 0x7d, 0x32, 0x10, 0x30, 0x98, 0x3a, 0x2f,
		},
		/* S8 : 128 byte */
		{
			0x57, 0xb1, 0x71, 0x57, 0xb4, 0xea, 0xaf, 0x0a,
			0x9b, 0xdc, 0xb9, 0xab, 0xe4, 0xb2, 0x99, 0xa7,
			0x28, 0xe6, 0xdf, 0x5f, 0x8e, 0x03, 0xd6, 0x88,
			0x03, 0x7d, 0x5e, 0x1e, 0xd5, 0xc9, 0xa6, 0x6c,
			0x20, 0xac, 0x73, 0x9e, 0x1c, 0x35, 0x16, 0xa4,
			0xcf, 0x78, 0x15, 0x5f, 0x52, 0xed, 0x7d, 0x05,
			0x4b, 0x5c, 0x5f, 0xed, 0x53, 0x4b, 0x80, 0xdd,
			0x3f, 0xc9, 0x2f, 0xc0, 0x92, 0x0e, 0xae, 0x69,
			0x5f, 0x2f, 0xb9, 0x62, 0x6c, 0xad, 0xa5, 0x84,
			0xd5, 0x72, 0x92, 0x7a, 0x00, 0xd6, 0x12, 0xaa,
			0x69, 0x0b, 0x7f, 0x60, 0x51, 0xdd, 0x58, 0x1c,
			0xce, 0x47, 0x48, 0xea, 0xb9, 0xb4, 0xd8, 0x86,
			0xad, 0xda, 0xa3, 0x2c, 0x4e, 0xd7, 0xd7, 0x22,
			0x18, 0x62, 0x35, 0x45, 0x56, 0xeb, 0x68, 0x19,
			0x7b, 0x05, 0xbe, 0xbd, 0xa9, 0x0c, 0x3e, 0x01,
			0xe0, 0x0e, 0x27, 0xc5, 0xef, 0x54, 0x7c, 0x83,
		},
	},
	/* SHA256 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0x46, 0x7e, 0x8e, 0xa6, 0x34, 0xf7, 0x99, 0x5d,
			0xc4, 0x6c, 0x11, 0xb8, 0xab, 0x0b, 0x75, 0x08,
			0x89, 0x46, 0x81, 0xe8, 0x1c, 0x35, 0x02, 0xc3,
			0xb3, 0x35, 0xe8, 0x97, 0xe6, 0xd6, 0x9d, 0xf8,
			0x85, 0xf4, 0x95, 0x57, 0xce, 0x23, 0x27, 0x84,
			0xe3, 0x51, 0x9b, 0x72, 0x7b, 0xa6, 0x84, 0x3b,
			0xd7, 0xaf, 0x50, 0x63, 0xf8, 0xbc, 0x1d, 0x61,
			0x0f, 0x86, 0xce, 0x5b, 0x35, 0x15, 0x5e, 0x32,
			0x5c, 0xe1, 0x75, 0xbe, 0x85, 0x38, 0x39, 0x5b,
			0x34, 0xdf, 0x67, 0xa4, 0x21, 0xfc, 0xa2, 0x7e,
			0x31, 0x33, 0x0b, 0x59, 0xa4, 0x10, 0x11, 0xb2,
			0x90, 0xa5, 0x8b, 0xdc, 0x8e, 0x74, 0x04, 0x01,
			0xb3, 0x8f, 0x55, 0x64, 0xc2, 0xfd, 0x7a, 0xe8,
			0x9f, 0x60, 0x9e, 0xd6, 0x07, 0xd5, 0x78, 0xdb,
			0x7f, 0x1c, 0xda, 0x50, 0x8a, 0xf9, 0x87, 0xbe,
			0x1f, 0xd9, 0x46, 0xa2, 0x5a, 0xb9, 0x34, 0x6d,
		},
		/* S9 : 128 byte */
		{
			0x2b, 0x1f, 0xfb, 0x37, 0x0d, 0x51, 0x8a, 0x82,
			0x64, 0x6d, 0x86, 0x82, 0x8d, 0xb1, 0xfc, 0x7e,
			0x8b, 0xfe, 0x73, 0xee, 0x87, 0x8d, 0xa1, 0x20,
			0xfa, 0x92, 0x73, 0x7c, 0x91, 0x74, 0x68, 0x89,
			0x95, 0xf2, 0x25, 0x5b, 0x29, 0xe8, 0x3b, 0x28,
			0xc2, 0x44, 0xcc, 0x56, 0x3c, 0x9b, 0x33, 0xef,
			0xd3, 0xf9, 0xf9, 0xe1, 0x63, 0x8e, 0x2c, 0x16,
			0xe2, 0x4f, 0x2e, 0xae, 0x19, 0x43, 0x56, 0x96,
			0xb2, 0xf4, 0xd1, 0xcf, 0x73, 0x06, 0x4f, 0xc1,
			0xcf, 0xcc, 0xb2, 0x27, 0x8c, 0x01, 0xf0, 0x97,
			0x9e, 0x7d, 0xe7, 0x46, 0x3b, 0xf8, 0x41, 0x7b,
			0xd6, 0x98, 0x6f, 0xbf, 0x1d, 0x34, 0xd3, 0x82,
			0xa9, 0x78, 0xce, 0x79, 0x95, 0x82, 0x44, 0x2a,
			0xfc, 0xc9, 0x2b, 0x4f, 0xe7, 0x43, 0x21, 0x6b,
			0x6f, 0x15, 0x1f, 0x6a, 0x56, 0x1d, 0x97, 0x9c,
			0xf6, 0x83, 0xca, 0xb6, 0xaf, 0x2f, 0xf4, 0xc5,
		},
	},
};

/* 10 elements */
static mod1024_t SHA384[] = {
	/* SHA384 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0xe5, 0x67, 0xa3, 0x9a, 0xe4, 0xe5, 0xef, 0x9b,
			0x68, 0x01, 0xea, 0x05, 0x61, 0xb7, 0x2a, 0x5d,
			0x4b, 0x5f, 0x38, 0x5f, 0x05, 0x32, 0xfc, 0x9f,
			0xe1, 0x0a, 0x75, 0x70, 0xf8, 0x69, 0xae, 0x05,
			0xc0, 0xbd, 0xed, 0xd6, 0xe0, 0xe2, 0x2d, 0x45,
			0x42, 0xe9, 0xce, 0x82, 0x6a, 0x18, 0x8c, 0xac,
			0x07, 0x31, 0xae, 0x39, 0xc8, 0xf8, 0x7f, 0x97,
			0x71, 0xef, 0x02, 0x13, 0x2e, 0x64, 0xe2, 0xfb,
			0x27, 0xad, 0xa8, 0xff, 0x54, 0xb3, 0x30, 0xdd,
			0x93, 0xad, 0x5e, 0x3e, 0xf8, 0x2e, 0x0d, 0xda,
			0x64, 0x62, 0x48, 0xe3, 0x59, 0x94, 0xbd, 0xa1,
			0x0c, 0xf4, 0x6e, 0x5a, 0xbc, 0x98, 0xaa, 0x74,
			0x43, 0xc0, 0x3c, 0xdd, 0xeb, 0x5e, 0xe2, 0xab,
			0x82, 0xd6, 0x01, 0x00, 0xb1, 0x02, 0x96, 0x31,
			0x89, 0x79, 0x70, 0x27, 0x5f, 0x11, 0x9d, 0x05,
			0xda, 0xa2, 0x22, 0x0a, 0x4a, 0x0d, 0xef, 0xba,
		},
		/* S0 : 128 byte */
		{
			0x16, 0x89, 0xa8, 0x52, 0x39, 0x19, 0xac, 0x77,
			0xcc, 0x99, 0x7e, 0xbc, 0x59, 0xcb, 0x90, 0x88,
			0x72, 0xd8, 0x8b, 0x28, 0x55, 0xa3, 0x09, 0xea,
			0xd2, 0x77, 0x9b, 0x88, 0x8b, 0x22, 0xb4, 0x23,
			0x2d, 0xa9, 0xb9, 0x3b, 0xb1, 0x9b, 0x32, 0xc1,
			0xdb, 0x77, 0xad, 0x73, 0x8c, 0x6e, 0x43, 0x36,
			0x1e, 0x9e, 0xb6, 0xb1, 0xa3, 0x7c, 0x49, 0xa8,
			0xf3, 0xc7, 0xc7, 0xae, 0x7e, 0x78, 0x4d, 0x19,
			0xa6, 0x21, 0x38, 0x74, 0x12, 0x93, 0xe4, 0x9b,
			0x18, 0x31, 0xc0, 0xc3, 0x61, 0x7e, 0xb4, 0x3c,
			0x56, 0x70, 0x6d, 0x83, 0x31, 0x49, 0x53, 0x47,
			0x06, 0x36, 0x44, 0x10, 0x86, 0x41, 0x9a, 0xb9,
			0xe6, 0xfd, 0x1e, 0xc4, 0xf9, 0xd5, 0xcc, 0x65,
			0x44, 0x81, 0x5d, 0x1e, 0x02, 0xed, 0x96, 0xa3,
			0xae, 0x64, 0xc6, 0x99, 0x8b, 0x2c, 0xf2, 0x38,
			0xe7, 0x9a, 0x12, 0x16, 0x43, 0x52, 0xd1, 0x2a,
		},
	},
	/* SHA384 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0xdb, 0x9e, 0xae, 0x66, 0xbc, 0xac, 0xb0, 0x1b,
			0xa8, 0x02, 0x9f, 0xba, 0xc5, 0xfe, 0xf1, 0x31,
			0x56, 0xaa, 0x1a, 0x76, 0x8c, 0x3d, 0xff, 0x25,
			0x9e, 0xe5, 0xae, 0x6b, 0x1c, 0xc3, 0x4c, 0x22,
			0x80, 0x37, 0x4e, 0x3d, 0x06, 0x5f, 0xed, 0x67,
			0xbd, 0xda, 0xd3, 0xe7, 0xde, 0xd2, 0x1b, 0xc7,
			0x5d, 0x39, 0x78, 0x35, 0xbf, 0x52, 0xb4, 0x29,
			0x2b, 0x74, 0x5d, 0x48, 0xc3, 0x0b, 0xbb, 0x09,
			0x35, 0xdf, 0x04, 0x16, 0x8c, 0x99, 0xd3, 0xf5,
			0x4d, 0x83, 0xd7, 0xe7, 0xbf, 0x8f, 0xc8, 0x10,
			0x8a, 0x9b, 0xa0, 0xd9, 0x83, 0xf3, 0x07, 0x93,
			0xc7, 0x17, 0x0f, 0x5d, 0x07, 0xbf, 0x21, 0x47,
			0x22, 0x1d, 0x38, 0xb7, 0xf9, 0x62, 0x4c, 0x0c,
			0x7f, 0x83, 0x12, 0x82, 0x0f, 0x3d, 0xe2, 0x55,
			0x03, 0x57, 0xf8, 0x2a, 0x82, 0x27, 0xe0, 0xaa,
			0x99, 0x57, 0x2c, 0x70, 0x8d, 0xe9, 0x68, 0x25,
		},
		/* S1 : 128 byte */
		{
			0x7b, 0x28, 0x6c, 0x61, 0xb6, 0x71, 0x10, 0xe2,
			0x32, 0x53, 0x3f, 0xd9, 0xd1, 0xac, 0x89, 0x0b,
			0x52, 0xce, 0xd7, 0x1c, 0x90, 0x29, 0xd3, 0x55,
			0xc3, 0xf3, 0x4c, 0x90, 0xc5, 0xfc, 0xe0, 0x68,
			0x80, 0x7e, 0x42, 0x4f, 0xe9, 0x48, 0x55, 0x4e,
			0xc3, 0x94, 0x6f, 0x8d, 0x44, 0x53, 0x59, 0x6a,
			0xcd, 0x29, 0xe2, 0xd4, 0x4b, 0x45, 0xe3, 0x46,
			0x02, 0x39, 0xc4, 0xcd, 0x92, 0x42, 0x6b, 0xbc,
			0xe7, 0xd1, 0xb0, 0xf8, 0xf9, 0x31, 0x41, 0xfd,
			0x9f, 0x2f, 0x80, 0xa5, 0x06, 0x94, 0x7b, 0xf5,
			0x01, 0x86, 0x50, 0xfa, 0x0b, 0xab, 0xdb, 0xb2,
			0xdc, 0x19, 0x39, 0x62, 0x2e, 0xa8, 0x79, 0x0a,
			0x1d, 0x16, 0x5b, 0x01, 0xc9, 0x2a, 0xd9, 0x15,
			0x41, 0xe2, 0xac, 0xeb, 0x8a, 0x77, 0x5b, 0x5c,
			0xa4, 0xbf, 0x22, 0x5b, 0x8b, 0x03, 0xd9, 0x9a,
			0x24, 0x05, 0xbd, 0xb7, 0xb2, 0xfd, 0xbd, 0xbe,
		},
	},
	/* SHA384 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x2e, 0x22, 0x65, 0x6e, 0x48, 0x85, 0x62, 0x40,
			0xb0, 0xf6, 0x55, 0xd2, 0x02, 0x2d, 0x69, 0xac,
			0x73, 0x1a, 0x1c, 0x63, 0x15, 0x1b, 0xa3, 0x6e,
			0x57, 0x72, 0x8a, 0x35, 0xcb, 0xf6, 0x9c, 0x6b,
			0x2d, 0x08, 0x02, 0x9e, 0xa0, 0xc9, 0xaf, 0x0c,
			0x3a, 0x8b, 0xfb, 0xb9, 0xc3, 0x85, 0x8d, 0xbe,
			0xba, 0xca, 0x7f, 0xe4, 0x99, 0x7c, 0xbd, 0x71,
			0x5b, 0x3c, 0xae, 0x42, 0x3a, 0xe6, 0x17, 0x1a,
			0xbd, 0x7b, 0x3c, 0x3a, 0xf2, 0x5f, 0x48, 0x32,
			0xf8, 0xc5, 0xee, 0xf6, 0xc1, 0x67, 0x51, 0x34,
			0x16, 0x7a, 0x21, 0xe5, 0xdc, 0xe1, 0x07, 0x23,
			0x95, 0xa1, 0x44, 0x3a, 0x2a, 0x95, 0x95, 0x01,
			0xe7, 0x63, 0x59, 0x6b, 0xc7, 0xae, 0xf5, 0x59,
			0xf2, 0x8d, 0x78, 0xb2, 0x84, 0x3c, 0x7b, 0xb6,
			0x90, 0x12, 0x4c, 0x8f, 0x65, 0xb3, 0x4b, 0x29,
			0x0f, 0x87, 0x93, 0x44, 0x21, 0x6b, 0xdd, 0xc8,
		},
		/* S2 : 128 byte */
		{
			0x98, 0xa7, 0xf8, 0x9d, 0xb0, 0x93, 0x2e, 0x94,
			0xcb, 0x1d, 0x38, 0xbb, 0x72, 0xa8, 0x5c, 0xe4,
			0x14, 0x37, 0x57, 0xc4, 0x49, 0xf8, 0x5c, 0xf8,
			0x91, 0x82, 0x6c, 0x74, 0x30, 0x72, 0x48, 0xaa,
			0x6a, 0x0a, 0x32, 0xc4, 0x5a, 0x10, 0xce, 0x4d,
			0x21, 0xbd, 0x88, 0x9b, 0x67, 0x16, 0xd8, 0xb3,
			0x23, 0xd1, 0x91, 0x95, 0x25, 0x71, 0x70, 0x7e,
			0xb7, 0x33, 0x13, 0xd0, 0xaf, 0x73, 0x24, 0xa5,
			0xaa, 0x49, 0x7b, 0x3d, 0x04, 0x5d, 0x44, 0x87,
			0x7d, 0x4d, 0x8a, 0xce, 0x8b, 0xf1, 0xcd, 0xf0,
			0x34, 0x5d, 0xba, 0x34, 0xd0, 0xc1, 0xa7, 0x99,
			0x35, 0xef, 0x94, 0xdc, 0xc6, 0x0d, 0xfa, 0x98,
			0x0a, 0x4b, 0x1e, 0x40, 0xb6, 0xd3, 0x01, 0x13,
			0xd5, 0x57, 0x78, 0x91, 0x18, 0xfc, 0x1e, 0xea,
			0x5f, 0x5c, 0x3f, 0x9f, 0x49, 0xac, 0xc1, 0x61,
			0x96, 0x43, 0x68, 0x5e, 0x6d, 0xde, 0x58, 0x27,
		},
	},
	/* SHA384 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0xcf, 0x78, 0x50, 0x02, 0x29, 0x2f, 0x1e, 0xf9,
			0x64, 0x5f, 0x2f, 0x2c, 0xdc, 0x04, 0x55, 0xb2,
			0x9d, 0x0b, 0xa1, 0xef, 0xc9, 0x1f, 0xde, 0x0a,
			0x3a, 0xc0, 0x5f, 0xa1, 0x69, 0x95, 0xfb, 0xbc,
			0x8e, 0x03, 0x55, 0x00, 0x09, 0xfe, 0x61, 0x26,
			0x01, 0x1f, 0x23, 0xab, 0x4b, 0xf7, 0xba, 0xc8,
			0x8d, 0xa4, 0x9e, 0xa4, 0xa3, 0xa1, 0x26, 0xc9,
			0x41, 0xe3, 0x4d, 0xd4, 0xda, 0x21, 0xb5, 0x71,
			0xc1, 0x73, 0xec, 0x87, 0x8f, 0x1e, 0xef, 0xb0,
			0xc4, 0xae, 0xec, 0xec, 0xb1, 0x46, 0x4c, 0x96,
			0x95, 0x35, 0xf1, 0x97, 0x06, 0xc9, 0xbc, 0xd9,
			0x92, 0x68, 0x53, 0xff, 0xa7, 0xdc, 0x21, 0x53,
			0xb0, 0xef, 0x9e, 0xff, 0x81, 0x38, 0xd0, 0x81,
			0xf5, 0x34, 0x12, 0x59, 0xd5, 0x99, 0x14, 0x04,
			0xfc, 0xe6, 0x23, 0x2e, 0xb9, 0x3f, 0xaa, 0x24,
			0x75, 0x78, 0xf9, 0x88, 0x96, 0x70, 0x46, 0x48,
		},
		/* S3 : 128 byte */
		{
			0x58, 0x4a, 0xbd, 0x1f, 0x5e, 0x14, 0x6b, 0x2b,
			0x01, 0xb1, 0x2f, 0x1c, 0x6e, 0xbb, 0xbd, 0x58,
			0x73, 0x98, 0x02, 0x82, 0xc6, 0x3c, 0x90, 0x38,
			0x5c, 0x2f, 0x70, 0x40, 0xb3, 0x44, 0x64, 0x35,
			0x3b, 0xce, 0x9d, 0xdd, 0x9b, 0x57, 0x3d, 0x33,
			0x4f, 0x76, 0x66, 0x09, 0x8e, 0x7a, 0x72, 0x20,
			0xa4, 0x53, 0x9f, 0x4b, 0x4e, 0xea, 0x2f, 0xe0,
			0x32, 0x0f, 0x97, 0xcd, 0xcb, 0xd6, 0xb3, 0xdb,
			0xdf, 0x76, 0xc3, 0x96, 0xa8, 0x8e, 0x8b, 0xc7,
			0x3c, 0x37, 0xa8, 0x32, 0xa8, 0x6b, 0xc1, 0x45,
			0x3a, 0xd9, 0xf9, 0xfb, 0x7d, 0xe1, 0x35, 0x44,
			0x2d, 0x27, 0x08, 0x0b, 0xb2, 0x20, 0x39, 0x17,
			0xa8, 0xce, 0xf6, 0x0b, 0xc3, 0xa5, 0x5e, 0xf7,
			0x3e, 0x2a, 0xd8, 0xce, 0x88, 0x0b, 0xb5, 0xe5,
			0xba, 0x63, 0x65, 0xa7, 0xb3, 0x54, 0xb3, 0xe6,
			0x38, 0xee, 0x09, 0x5d, 0x22, 0x19, 0x7e, 0xcd,
		},
	},
	/* SHA384 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0x30, 0x22, 0x19, 0xbc, 0x6c, 0x07, 0xe6, 0x28,
			0xf3, 0x52, 0xa4, 0x50, 0x91, 0x71, 0x9b, 0xb0,
			0xe6, 0x55, 0x87, 0x5f, 0xb9, 0x76, 0x91, 0x38,
			0x8f, 0x18, 0x7e, 0x37, 0x7c, 0xcd, 0x27, 0x25,
			0x3a, 0x3c, 0x29, 0x78, 0x78, 0x20, 0xdd, 0x75,
			0x74, 0x94, 0x8b, 0x97, 0xcc, 0x8d, 0xda, 0x34,
			0x92, 0xa8, 0x8b, 0x5e, 0x9e, 0x26, 0x67, 0x08,
			0x28, 0x34, 0x76, 0x69, 0x96, 0x70, 0x72, 0xa6,
			0x92, 0xaa, 0x7b, 0xab, 0x60, 0xe7, 0x0c, 0x59,
			0x6e, 0xac, 0x2e, 0xe7, 0x2d, 0x79, 0xec, 0x82,
			0x20, 0x93, 0x06, 0x37, 0x33, 0x06, 0xea, 0x13,
			0x4d, 0xeb, 0xdf, 0x89, 0x28, 0xe7, 0x84, 0x13,
			0x61, 0xf4, 0x79, 0x1c, 0xcb, 0x64, 0x5c, 0x1e,
			0x09, 0x23, 0xf7, 0x08, 0x5c, 0x3c, 0xb7, 0xe2,
			0xb9, 0x9e, 0x60, 0xb3, 0x1f, 0x21, 0x00, 0xa3,
			0x27, 0xc4, 0x22, 0x5d, 0xdd, 0x38, 0x43, 0xb7,
		},
		/* S4 : 128 byte */
		{
			0x76, 0x6e, 0x1a, 0xa6, 0x07, 0xe7, 0x99, 0x68,
			0xee, 0xc2, 0x37, 0x36, 0x9b, 0x4c, 0x91, 0x40,
			0xaa, 0x6b, 0xfb, 0x87, 0x14, 0xb3, 0xd9, 0x7d,
			0x05, 0x8a, 0x54, 0x64, 0x48, 0x2d, 0xed, 0x41,
			0xf5, 0x03, 0x6f, 0x85, 0x21, 0x12, 0x24, 0x2a,
			0x3e, 0xf9, 0xe9, 0x7e, 0x5c, 0x31, 0xb0, 0x26,
			0x22, 0xc7, 0x2f, 0x7f, 0x37, 0xdd, 0x47, 0xb7,
			0xec, 0xa8, 0xb9, 0x47, 0xef, 0x55, 0xd3, 0x37,
			0x5b, 0xec, 0x16, 0x18, 0xb0, 0x9f, 0xcd, 0x98,
			0x2e, 0x48, 0x84, 0x86, 0xa8, 0x85, 0x16, 0x5f,
			0x18, 0x9a, 0xa0, 0x27, 0x15, 0x10, 0x0b, 0xf4,
			0xe0, 0x9f, 0xea, 0xdf, 0xc6, 0xc2, 0x76, 0x71,
			0x45, 0x13, 0x1c, 0xa8, 0x7e, 0xa0, 0xa2, 0x0f,
			0x47, 0x5d, 0x6d, 0x44, 0xb6, 0x29, 0x84, 0x43,
			0xe9, 0xdc, 0x49, 0xfc, 0x70, 0xb3, 0x18, 0xf0,
			0x41, 0x88, 0xda, 0xf5, 0xec, 0x48, 0x7e, 0xcc,
		},
	},
	/* SHA384 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0x1f, 0xd6, 0x89, 0x98, 0xe4, 0xc3, 0x4d, 0xed,
			0x43, 0xd2, 0xe5, 0x1b, 0x16, 0x80, 0xa6, 0xb3,
			0xfd, 0xfd, 0xd5, 0x76, 0x03, 0x3b, 0x6a, 0x0d,
			0x7f, 0x44, 0x13, 0xfb, 0x39, 0xd2, 0x7c, 0xcf,
			0xce, 0x19, 0xce, 0x9e, 0x56, 0xe8, 0x76, 0xb0,
			0x43, 0xf0, 0x27, 0x7a, 0x8c, 0x63, 0x22, 0xcc,
			0x0c, 0xc2, 0xb5, 0x4c, 0x25, 0xf0, 0x01, 0x23,
			0xa8, 0x8a, 0x01, 0xf6, 0xf9, 0x7b, 0xc8, 0x18,
			0x6e, 0xca, 0x0a, 0xfe, 0xf9, 0xb7, 0xde, 0xf3,
			0x07, 0x07, 0xc2, 0x0c, 0xa7, 0x9c, 0xe0, 0x5f,
			0xef, 0x48, 0x90, 0x70, 0x9b, 0xe3, 0x1d, 0xec,
			0xf8, 0x9b, 0xcc, 0x9a, 0x01, 0x8c, 0x1a, 0x8e,
			0xc7, 0x91, 0x46, 0xa0, 0x16, 0x59, 0x0f, 0x1c,
			0x55, 0xbd, 0xbf, 0xbb, 0xae, 0x50, 0x68, 0xd2,
			0x6f, 0x38, 0x6e, 0xca, 0xb0, 0x3d, 0x45, 0xd0,
			0x37, 0x20, 0x7c, 0xcd, 0x98, 0x2e, 0xfb, 0x99,
		},
		/* S5 : 128 byte */
		{
			0x84, 0xf0, 0x34, 0x12, 0x4c, 0x82, 0x80, 0xcc,
			0xc4, 0xf9, 0x7e, 0x9a, 0x32, 0x6d, 0x57, 0xe0,
			0x3a, 0xb2, 0x8f, 0xbd, 0x84, 0x56, 0x2a, 0x5f,
			0x4b, 0xc0, 0xae, 0x66, 0xa8, 0x88, 0xb6, 0x4c,
			0x4c, 0x9a, 0xa2, 0x72, 0xef, 0x56, 0x72, 0x40,
			0xdf, 0xd6, 0x33, 0x56, 0x0a, 0x79, 0xc8, 0x39,
			0x74, 0xf6, 0x28, 0x44, 0x4d, 0xf3, 0x37, 0xde,
			0x9d, 0x4e, 0x26, 0xd1, 0x09, 0x64, 0xc7, 0x83,
			0x22, 0xde, 0x30, 0x53, 0xf0, 0x05, 0x43, 0xc0,
			0xb1, 0x35, 0xcf, 0x95, 0x2b, 0xdb, 0x23, 0xc3,
			0xbf, 0x02, 0xc8, 0xc6, 0xb8, 0xd3, 0xac, 0xc9,
			0x03, 0x3f, 0xb5, 0xcc, 0xf2, 0x66, 0x39, 0x4f,
			0x4c, 0x8c, 0xdc, 0x4a, 0x44, 0x82, 0x4d, 0x7d,
			0xd3, 0x86, 0x97, 0x60, 0x0d, 0xcb, 0x62, 0x24,
			0x7f, 0x6a, 0x1b, 0x69, 0xac, 0x14, 0xde, 0x06,
			0x2d, 0x7c, 0x66, 0xf1, 0x8f, 0xe6, 0x2d, 0x7b,
		},
	},
	/* SHA384 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0x90, 0xd0, 0xd6, 0xa4, 0x10, 0xf8, 0xaa, 0x86,
			0x8b, 0x64, 0x12, 0x8c, 0x03, 0x9c, 0xc6, 0x98,
			0xa3, 0xb2, 0xe3, 0x67, 0x00, 0x74, 0x49, 0x69,
			0xa0, 0xe7, 0x41, 0xd0, 0xb9, 0xf1, 0x64, 0xe7,
			0x3a, 0x56, 0xb8, 0xa6, 0x0f, 0xe6, 0x1b, 0x83,
			0xa1, 0xf4, 0xe5, 0x8a, 0x9d, 0x09, 0xc9, 0xb0,
			0xf7, 0xda, 0xfc, 0x65, 0x2e, 0x78, 0x99, 0x68,
			0xd1, 0xc7, 0x45, 0xdf, 0x42, 0x31, 0x0f, 0xa7,
			0xba, 0x9b, 0x99, 0xe9, 0x8d, 0x98, 0x7d, 0xee,
			0xb0, 0xdd, 0xea, 0x34, 0x78, 0xf6, 0x8e, 0xc1,
			0xa4, 0x37, 0xb9, 0xb7, 0x9e, 0xf4, 0xbf, 0x29,
			0xfe, 0xf5, 0x9e, 0x2d, 0x50, 0x24, 0xe0, 0x32,
			0x17, 0x89, 0xc3, 0x94, 0x12, 0x78, 0x98, 0x6e,
			0xbd, 0x09, 0x69, 0xb0, 0x1b, 0xc9, 0xde, 0x71,
			0xb3, 0xaf, 0x61, 0x86, 0xd8, 0x52, 0x81, 0x53,
			0x2e, 0x58, 0x94, 0xe1, 0x6a, 0x32, 0xa1, 0x31,
		},
		/* S6 : 128 byte */
		{
			0x78, 0x7b, 0x00, 0x5e, 0x95, 0xf9, 0x64, 0x2b,
			0xa2, 0x7e, 0x5f, 0x4d, 0x36, 0x52, 0x75, 0xbb,
			0xbb, 0xac, 0x72, 0xbd, 0x1c, 0xfe, 0x29, 0x2e,
			0xa7, 0x80, 0x16, 0x70, 0x0a, 0xae, 0x7a, 0xf3,
			0xc9, 0xe0, 0xe2, 0x0b, 0xcf, 0xa5, 0x4d, 0xc0,
			0x08, 0xc6, 0x5a, 0x3b, 0x91, 0xc6, 0xe1, 0x7c,
			0x67, 0xd6, 0x22, 0x90, 0x86, 0xc5, 0xa6, 0xf6,
			0x77, 0xe6, 0xde, 0xd9, 0x6b, 0x88, 0xf6, 0xe8,
			0x54, 0x1c, 0x63, 0xf6, 0x64, 0xa3, 0x8f, 0x4c,
			0xba, 0x64, 0x1a, 0xa1, 0x77, 0x17, 0x6f, 0x9a,
			0x3e, 0x45, 0xa6, 0x7f, 0xfc, 0x76, 0x08, 0xd2,
			0x4f, 0x4b, 0x44, 0x06, 0x5e, 0x09, 0xc2, 0x74,
			0x4b, 0x7b, 0xe4, 0xd3, 0x49, 0xaf, 0xb0, 0xe8,
			0xf2, 0xe7, 0xc5, 0x10, 0x68, 0x16, 0xa4, 0x6c,
			0x74, 0x5e, 0x61, 0xa5, 0x72, 0xbb, 0xd5, 0xe9,
			0x8c, 0xb6, 0xb8, 0xd3, 0x4c, 0x6c, 0x68, 0x2f,
		},
	},
	/* SHA384 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0x64, 0x4a, 0x3d, 0xe6, 0x10, 0x0d, 0xd8, 0xa3,
			0x6c, 0x84, 0x14, 0x46, 0xb9, 0x37, 0x6b, 0x34,
			0x5d, 0x47, 0x8a, 0x32, 0x94, 0x50, 0xa6, 0x6f,
			0x62, 0x93, 0x13, 0xc5, 0xcc, 0x55, 0x13, 0x3c,
			0x47, 0x82, 0xec, 0xd0, 0x71, 0x96, 0x3d, 0x74,
			0xff, 0xbd, 0x91, 0x56, 0xf6, 0x39, 0x35, 0x65,
			0x1f, 0x05, 0x47, 0x11, 0xb4, 0xb8, 0x10, 0x51,
			0x60, 0xab, 0x94, 0xeb, 0x67, 0x5e, 0x66, 0xf0,
			0x20, 0x18, 0xc1, 0x85, 0xac, 0xce, 0xd5, 0xe9,
			0xe2, 0x46, 0x5f, 0xb4, 0x89, 0x7d, 0x7c, 0x5d,
			0xca, 0x45, 0xbf, 0xff, 0xe4, 0x32, 0x81, 0x5d,
			0x22, 0xb7, 0x78, 0x23, 0x0b, 0x8d, 0x8c, 0x6f,
			0xd8, 0x54, 0xe5, 0x8f, 0x2b, 0x7d, 0x49, 0x0d,
			0xfd, 0x67, 0xa5, 0xe8, 0xd8, 0xc8, 0xe9, 0x25,
			0x47, 0x4f, 0x19, 0x86, 0x8c, 0x88, 0xa1, 0x9a,
			0x90, 0xff, 0xea, 0xf6, 0x20, 0xa8, 0x2a, 0xc1,
		},
		/* S7 : 128 byte */
		{
			0x59, 0x9b, 0x51, 0x1f, 0xc8, 0xbf, 0xca, 0xe4,
			0x40, 0x54, 0xa1, 0x6c, 0x29, 0x9b, 0x5f, 0x3c,
			0x64, 0xb7, 0x4c, 0x76, 0x0f, 0xd1, 0x82, 0x29,
			0x2b, 0x31, 0xf5, 0x22, 0x04, 0xc6, 0x27, 0xc8,
			0x53, 0xf6, 0xfa, 0x2a, 0x75, 0x32, 0x16, 0x02,
			0x6d, 0x31, 0x25, 0x74, 0xf6, 0xad, 0xde, 0x4a,
			0xe8, 0x29, 0x68, 0x0d, 0x28, 0xdc, 0x1c, 0x32,
			0x53, 0xff, 0xa9, 0x6f, 0x56, 0xd8, 0x1e, 0x1c,
			0x4b, 0x5d, 0xaf, 0xce, 0x91, 0xe8, 0x09, 0xd9,
			0xf5, 0x8d, 0xa7, 0x2a, 0x94, 0xe2, 0x9d, 0xca,
			0xd5, 0x2e, 0x67, 0x59, 0xa1, 0x6a, 0x06, 0x30,
			0x4e, 0x75, 0xe6, 0x41, 0xcc, 0xe6, 0x26, 0x15,
			0x4e, 0xde, 0xef, 0x36, 0x4c, 0x62, 0xae, 0xa5,
			0x28, 0x4b, 0xa4, 0x02, 0x70, 0xe9, 0x05, 0x0e,
			0x81, 0xcf, 0x5a, 0x7d, 0xfd, 0xa6, 0x7a, 0x70,
			0x22, 0xfb, 0x4d, 0xd5, 0x51, 0x4e, 0x31, 0x43,
		},
	},
	/* SHA384 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0x29, 0xe9, 0xcb, 0x34, 0x82, 0xe9, 0xed, 0xa5,
			0xe6, 0xd7, 0x14, 0xa2, 0xdc, 0x73, 0x0d, 0x91,
			0xbd, 0xfb, 0x2d, 0x42, 0xb1, 0x4f, 0xe8, 0x2e,
			0xd9, 0x53, 0xa0, 0xfc, 0xce, 0x95, 0xbb, 0x4a,
			0x4b, 0x30, 0xf8, 0xa9, 0x26, 0xf0, 0x03, 0xe9,
			0x85, 0xb6, 0x47, 0x98, 0x4e, 0x1e, 0x03, 0xc5,
			0xdf, 0x60, 0xa0, 0x09, 0x00, 0x74, 0x42, 0xc9,
			0x15, 0x7a, 0x9d, 0xb5, 0x99, 0x58, 0x0a, 0x94,
			0xc7, 0xa5, 0xd9, 0x44, 0x9a, 0xc6, 0xe9, 0xfc,
			0xf8, 0x56, 0x77, 0xd5, 0x2a, 0xcc, 0x34, 0x89,
			0x90, 0x57, 0x9a, 0x7a, 0x93, 0x0d, 0x7f, 0xf6,
			0xda, 0x45, 0xeb, 0xb9, 0xc1, 0x41, 0x27, 0xb3,
			0x07, 0x11, 0x16, 0x0c, 0xc6, 0xb3, 0xc8, 0x66,
			0xa9, 0x7f, 0x4d, 0x62, 0x59, 0xc3, 0xc7, 0x65,
			0x38, 0xaa, 0xc3, 0x9c, 0x52, 0x62, 0xca, 0x3d,
			0x91, 0x7d, 0x32, 0x10, 0x30, 0x98, 0x3a, 0x2f,
		},
		/* S8 : 128 byte */
		{
			0x79, 0x86, 0x38, 0xdf, 0x8c, 0xc5, 0x30, 0x62,
			0xb8, 0xa3, 0x12, 0xe9, 0x06, 0x52, 0x7c, 0xe8,
			0xe3, 0xa6, 0xb2, 0xb6, 0x45, 0x69, 0x92, 0xc5,
			0x99, 0xe8, 0x2c, 0xaf, 0xcf, 0x1d, 0xdf, 0xe6,
			0x67, 0x16, 0x74, 0x20, 0x2a, 0xa6, 0x1f, 0xcb,
			0x9e, 0x83, 0xdc, 0x1f, 0x6b, 0xf9, 0x3d, 0x2e,
			0xdb, 0x76, 0x51, 0xdc, 0xf8, 0x69, 0xfe, 0xd9,
			0x87, 0x47, 0x7c, 0x26, 0xd0, 0xa2, 0xa7, 0xa2,
			0xc9, 0x4d, 0xce, 0x4f, 0xe2, 0x98, 0x49, 0xc2,
			0x6d, 0x47, 0x7f, 0x78, 0x2a, 0x40, 0xf2, 0xce,
			0x1f, 0x40, 0x50, 0xb5, 0xbf, 0x3f, 0xe3, 0xe7,
			0x0d, 0x00, 0xef, 0xd2, 0x84, 0xdc, 0xb9, 0xf5,
			0x29, 0x90, 0x14, 0x9e, 0xc8, 0xc5, 0xb7, 0xd2,
			0x29, 0xdb, 0x84, 0x1e, 0x32, 0x74, 0x1a, 0x74,
			0xfd, 0x36, 0xe1, 0x05, 0xe6, 0x7c, 0x18, 0x14,
			0xc0, 0x8e, 0xa9, 0xaa, 0x44, 0xf2, 0xab, 0x26,
		},
	},
	/* SHA384 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0x46, 0x7e, 0x8e, 0xa6, 0x34, 0xf7, 0x99, 0x5d,
			0xc4, 0x6c, 0x11, 0xb8, 0xab, 0x0b, 0x75, 0x08,
			0x89, 0x46, 0x81, 0xe8, 0x1c, 0x35, 0x02, 0xc3,
			0xb3, 0x35, 0xe8, 0x97, 0xe6, 0xd6, 0x9d, 0xf8,
			0x85, 0xf4, 0x95, 0x57, 0xce, 0x23, 0x27, 0x84,
			0xe3, 0x51, 0x9b, 0x72, 0x7b, 0xa6, 0x84, 0x3b,
			0xd7, 0xaf, 0x50, 0x63, 0xf8, 0xbc, 0x1d, 0x61,
			0x0f, 0x86, 0xce, 0x5b, 0x35, 0x15, 0x5e, 0x32,
			0x5c, 0xe1, 0x75, 0xbe, 0x85, 0x38, 0x39, 0x5b,
			0x34, 0xdf, 0x67, 0xa4, 0x21, 0xfc, 0xa2, 0x7e,
			0x31, 0x33, 0x0b, 0x59, 0xa4, 0x10, 0x11, 0xb2,
			0x90, 0xa5, 0x8b, 0xdc, 0x8e, 0x74, 0x04, 0x01,
			0xb3, 0x8f, 0x55, 0x64, 0xc2, 0xfd, 0x7a, 0xe8,
			0x9f, 0x60, 0x9e, 0xd6, 0x07, 0xd5, 0x78, 0xdb,
			0x7f, 0x1c, 0xda, 0x50, 0x8a, 0xf9, 0x87, 0xbe,
			0x1f, 0xd9, 0x46, 0xa2, 0x5a, 0xb9, 0x34, 0x6d,
		},
		/* S9 : 128 byte */
		{
			0xc7, 0x00, 0x28, 0x35, 0x57, 0xda, 0xfd, 0x1f,
			0x88, 0x34, 0xec, 0x0b, 0x7a, 0x7e, 0xc7, 0x33,
			0x71, 0x9c, 0xb0, 0xbe, 0x6e, 0xdb, 0x19, 0xf1,
			0x06, 0x4e, 0xe9, 0x4c, 0x75, 0xb8, 0xd9, 0xd4,
			0x94, 0x58, 0xfa, 0x18, 0x4f, 0x7d, 0x6d, 0xfa,
			0x5a, 0x6d, 0xa2, 0xd2, 0xeb, 0xee, 0xf0, 0x65,
			0x0d, 0x8a, 0xf3, 0x54, 0x47, 0x82, 0x3c, 0x83,
			0xa7, 0x73, 0x78, 0x24, 0x50, 0x9a, 0x74, 0x25,
			0x33, 0x9c, 0xaf, 0x99, 0x10, 0x1c, 0xd1, 0xfe,
			0x3d, 0xb8, 0x83, 0xfb, 0x98, 0xd1, 0x72, 0x12,
			0x7e, 0x30, 0xe3, 0x8d, 0x1d, 0x6f, 0x9e, 0x36,
			0x54, 0x93, 0x7c, 0xd6, 0x8c, 0xbb, 0x4e, 0xa2,
			0x28, 0xc8, 0x16, 0x06, 0x4f, 0xa8, 0xca, 0x09,
			0x50, 0xc7, 0xe7, 0xb6, 0xad, 0x25, 0x04, 0x55,
			0x74, 0xa6, 0xa4, 0x06, 0x3b, 0x63, 0xf0, 0x74,
			0x66, 0xb2, 0xcb, 0x5d, 0x73, 0x11, 0xb7, 0xcf,
		},
	},
};

/* 10 elements */
static mod1024_t SHA512[] = {
	/* SHA512 [0] */
	{
		/* Msg0 : 128 byte */
		{
			0xe5, 0x67, 0xa3, 0x9a, 0xe4, 0xe5, 0xef, 0x9b,
			0x68, 0x01, 0xea, 0x05, 0x61, 0xb7, 0x2a, 0x5d,
			0x4b, 0x5f, 0x38, 0x5f, 0x05, 0x32, 0xfc, 0x9f,
			0xe1, 0x0a, 0x75, 0x70, 0xf8, 0x69, 0xae, 0x05,
			0xc0, 0xbd, 0xed, 0xd6, 0xe0, 0xe2, 0x2d, 0x45,
			0x42, 0xe9, 0xce, 0x82, 0x6a, 0x18, 0x8c, 0xac,
			0x07, 0x31, 0xae, 0x39, 0xc8, 0xf8, 0x7f, 0x97,
			0x71, 0xef, 0x02, 0x13, 0x2e, 0x64, 0xe2, 0xfb,
			0x27, 0xad, 0xa8, 0xff, 0x54, 0xb3, 0x30, 0xdd,
			0x93, 0xad, 0x5e, 0x3e, 0xf8, 0x2e, 0x0d, 0xda,
			0x64, 0x62, 0x48, 0xe3, 0x59, 0x94, 0xbd, 0xa1,
			0x0c, 0xf4, 0x6e, 0x5a, 0xbc, 0x98, 0xaa, 0x74,
			0x43, 0xc0, 0x3c, 0xdd, 0xeb, 0x5e, 0xe2, 0xab,
			0x82, 0xd6, 0x01, 0x00, 0xb1, 0x02, 0x96, 0x31,
			0x89, 0x79, 0x70, 0x27, 0x5f, 0x11, 0x9d, 0x05,
			0xda, 0xa2, 0x22, 0x0a, 0x4a, 0x0d, 0xef, 0xba,
		},
		/* S0 : 128 byte */
		{
			0xbf, 0x3f, 0xf2, 0xc6, 0x96, 0x75, 0xf1, 0xb8,
			0xed, 0x42, 0x10, 0x21, 0x80, 0x1f, 0xb4, 0xce,
			0x29, 0xa7, 0x57, 0xf7, 0xf8, 0x86, 0x9c, 0xe4,
			0x36, 0xd0, 0xd7, 0x5a, 0xb7, 0x49, 0xef, 0xc8,
			0xb9, 0x03, 0xd9, 0xf9, 0xcb, 0x21, 0x46, 0x86,
			0x14, 0x7f, 0x12, 0xf3, 0x33, 0x5f, 0xa9, 0x36,
			0x68, 0x9c, 0x19, 0x2f, 0x31, 0x0a, 0xe3, 0xc5,
			0xd7, 0x54, 0x93, 0xf4, 0x4b, 0x24, 0xbc, 0x1c,
			0xd3, 0x50, 0x15, 0x84, 0xaa, 0xa5, 0x00, 0x4b,
			0x65, 0xa8, 0x71, 0x6d, 0x1e, 0xda, 0x72, 0x40,
			0xad, 0x8a, 0x52, 0x9d, 0x5a, 0x0c, 0xf1, 0x69,
			0xf4, 0x05, 0x4b, 0x45, 0x0e, 0x07, 0x6e, 0xe0,
			0xd4, 0x1a, 0x00, 0x11, 0xc5, 0x57, 0xaa, 0x69,
			0xa8, 0x4a, 0x81, 0x04, 0xc9, 0x09, 0x20, 0x1d,
			0x60, 0xfe, 0x39, 0xc7, 0x9e, 0x68, 0x43, 0x47,
			0xef, 0x4d, 0x14, 0x4e, 0xa1, 0x8f, 0x7a, 0x4e,
		},
	},
	/* SHA512 [1] */
	{
		/* Msg1 : 128 byte */
		{
			0xdb, 0x9e, 0xae, 0x66, 0xbc, 0xac, 0xb0, 0x1b,
			0xa8, 0x02, 0x9f, 0xba, 0xc5, 0xfe, 0xf1, 0x31,
			0x56, 0xaa, 0x1a, 0x76, 0x8c, 0x3d, 0xff, 0x25,
			0x9e, 0xe5, 0xae, 0x6b, 0x1c, 0xc3, 0x4c, 0x22,
			0x80, 0x37, 0x4e, 0x3d, 0x06, 0x5f, 0xed, 0x67,
			0xbd, 0xda, 0xd3, 0xe7, 0xde, 0xd2, 0x1b, 0xc7,
			0x5d, 0x39, 0x78, 0x35, 0xbf, 0x52, 0xb4, 0x29,
			0x2b, 0x74, 0x5d, 0x48, 0xc3, 0x0b, 0xbb, 0x09,
			0x35, 0xdf, 0x04, 0x16, 0x8c, 0x99, 0xd3, 0xf5,
			0x4d, 0x83, 0xd7, 0xe7, 0xbf, 0x8f, 0xc8, 0x10,
			0x8a, 0x9b, 0xa0, 0xd9, 0x83, 0xf3, 0x07, 0x93,
			0xc7, 0x17, 0x0f, 0x5d, 0x07, 0xbf, 0x21, 0x47,
			0x22, 0x1d, 0x38, 0xb7, 0xf9, 0x62, 0x4c, 0x0c,
			0x7f, 0x83, 0x12, 0x82, 0x0f, 0x3d, 0xe2, 0x55,
			0x03, 0x57, 0xf8, 0x2a, 0x82, 0x27, 0xe0, 0xaa,
			0x99, 0x57, 0x2c, 0x70, 0x8d, 0xe9, 0x68, 0x25,
		},
		/* S1 : 128 byte */
		{
			0x39, 0x8c, 0xba, 0xa0, 0x7b, 0xc4, 0xb3, 0x43,
			0x2f, 0x0a, 0xd0, 0xef, 0x54, 0xb2, 0x40, 0x2c,
			0x2d, 0x25, 0xc0, 0xc4, 0x6f, 0x72, 0x29, 0xfb,
			0xfc, 0xba, 0x4c, 0x14, 0x1c, 0xd0, 0xdb, 0x76,
			0xae, 0xa4, 0x2c, 0x24, 0x36, 0x75, 0x7b, 0x16,
			0xf7, 0x39, 0x7c, 0x57, 0x0e, 0x89, 0x1c, 0xd4,
			0xe6, 0xf0, 0xa0, 0x52, 0xd0, 0xa9, 0x10, 0xc8,
			0x53, 0x7d, 0x4e, 0xff, 0xb2, 0xd0, 0x7f, 0x06,
			0x89, 0x05, 0x81, 0xdd, 0xe8, 0xba, 0xd7, 0xf7,
			0x7f, 0x87, 0x81, 0xbd, 0xc0, 0x52, 0xc0, 0xca,
			0xa5, 0x0d, 0x94, 0xd9, 0x58, 0x97, 0x50, 0x8d,
			0x68, 0x48, 0xf5, 0xe2, 0x6e, 0x59, 0x75, 0xf1,
			0xa6, 0x3c, 0x03, 0xf4, 0x5b, 0xd5, 0x47, 0x9b,
			0xac, 0x0a, 0x40, 0xa3, 0xe4, 0xa1, 0xad, 0xfe,
			0xf8, 0x47, 0xc3, 0xe9, 0xe9, 0xf7, 0x61, 0x2d,
			0x50, 0x34, 0xcc, 0x68, 0x8f, 0x01, 0x9b, 0x86,
		},
	},
	/* SHA512 [2] */
	{
		/* Msg2 : 128 byte */
		{
			0x2e, 0x22, 0x65, 0x6e, 0x48, 0x85, 0x62, 0x40,
			0xb0, 0xf6, 0x55, 0xd2, 0x02, 0x2d, 0x69, 0xac,
			0x73, 0x1a, 0x1c, 0x63, 0x15, 0x1b, 0xa3, 0x6e,
			0x57, 0x72, 0x8a, 0x35, 0xcb, 0xf6, 0x9c, 0x6b,
			0x2d, 0x08, 0x02, 0x9e, 0xa0, 0xc9, 0xaf, 0x0c,
			0x3a, 0x8b, 0xfb, 0xb9, 0xc3, 0x85, 0x8d, 0xbe,
			0xba, 0xca, 0x7f, 0xe4, 0x99, 0x7c, 0xbd, 0x71,
			0x5b, 0x3c, 0xae, 0x42, 0x3a, 0xe6, 0x17, 0x1a,
			0xbd, 0x7b, 0x3c, 0x3a, 0xf2, 0x5f, 0x48, 0x32,
			0xf8, 0xc5, 0xee, 0xf6, 0xc1, 0x67, 0x51, 0x34,
			0x16, 0x7a, 0x21, 0xe5, 0xdc, 0xe1, 0x07, 0x23,
			0x95, 0xa1, 0x44, 0x3a, 0x2a, 0x95, 0x95, 0x01,
			0xe7, 0x63, 0x59, 0x6b, 0xc7, 0xae, 0xf5, 0x59,
			0xf2, 0x8d, 0x78, 0xb2, 0x84, 0x3c, 0x7b, 0xb6,
			0x90, 0x12, 0x4c, 0x8f, 0x65, 0xb3, 0x4b, 0x29,
			0x0f, 0x87, 0x93, 0x44, 0x21, 0x6b, 0xdd, 0xc8,
		},
		/* S2 : 128 byte */
		{
			0x4a, 0xf9, 0x87, 0x6c, 0x2a, 0x52, 0x0a, 0x79,
			0x28, 0x22, 0x26, 0xd9, 0x53, 0x50, 0x5b, 0x16,
			0x6a, 0x67, 0xf3, 0x61, 0x54, 0x5a, 0x65, 0xa9,
			0x80, 0xab, 0xfe, 0x43, 0xe8, 0xd0, 0x9b, 0xce,
			0x6d, 0x89, 0x89, 0x2e, 0x55, 0x63, 0x13, 0x35,
			0x1c, 0xc3, 0x31, 0x2e, 0x21, 0x45, 0x17, 0xea,
			0x6d, 0x62, 0xc8, 0x0c, 0x73, 0x4e, 0xb2, 0x76,
			0x07, 0x4e, 0x64, 0x67, 0x5e, 0x53, 0x28, 0x9c,
			0xa4, 0x18, 0x40, 0xb4, 0xd7, 0xc9, 0xb2, 0x25,
			0x18, 0x3b, 0x81, 0x62, 0xa6, 0xf0, 0x76, 0x1b,
			0xf1, 0xa7, 0x13, 0x41, 0xb9, 0x48, 0x2c, 0xc1,
			0x5a, 0xd5, 0x16, 0xbd, 0xa0, 0x50, 0xda, 0x93,
			0x38, 0x25, 0x67, 0x99, 0x8f, 0x84, 0x45, 0x29,
			0x13, 0x1b, 0x1b, 0x56, 0x02, 0x94, 0x06, 0x0b,
			0x36, 0x17, 0x1a, 0x4e, 0x71, 0x18, 0xa5, 0x56,
			0x9b, 0xef, 0xf9, 0xbe, 0xa0, 0x78, 0x38, 0x5a,
		},
	},
	/* SHA512 [3] */
	{
		/* Msg3 : 128 byte */
		{
			0xcf, 0x78, 0x50, 0x02, 0x29, 0x2f, 0x1e, 0xf9,
			0x64, 0x5f, 0x2f, 0x2c, 0xdc, 0x04, 0x55, 0xb2,
			0x9d, 0x0b, 0xa1, 0xef, 0xc9, 0x1f, 0xde, 0x0a,
			0x3a, 0xc0, 0x5f, 0xa1, 0x69, 0x95, 0xfb, 0xbc,
			0x8e, 0x03, 0x55, 0x00, 0x09, 0xfe, 0x61, 0x26,
			0x01, 0x1f, 0x23, 0xab, 0x4b, 0xf7, 0xba, 0xc8,
			0x8d, 0xa4, 0x9e, 0xa4, 0xa3, 0xa1, 0x26, 0xc9,
			0x41, 0xe3, 0x4d, 0xd4, 0xda, 0x21, 0xb5, 0x71,
			0xc1, 0x73, 0xec, 0x87, 0x8f, 0x1e, 0xef, 0xb0,
			0xc4, 0xae, 0xec, 0xec, 0xb1, 0x46, 0x4c, 0x96,
			0x95, 0x35, 0xf1, 0x97, 0x06, 0xc9, 0xbc, 0xd9,
			0x92, 0x68, 0x53, 0xff, 0xa7, 0xdc, 0x21, 0x53,
			0xb0, 0xef, 0x9e, 0xff, 0x81, 0x38, 0xd0, 0x81,
			0xf5, 0x34, 0x12, 0x59, 0xd5, 0x99, 0x14, 0x04,
			0xfc, 0xe6, 0x23, 0x2e, 0xb9, 0x3f, 0xaa, 0x24,
			0x75, 0x78, 0xf9, 0x88, 0x96, 0x70, 0x46, 0x48,
		},
		/* S3 : 128 byte */
		{
			0x5a, 0x9b, 0x01, 0xa6, 0xd3, 0xe0, 0x8d, 0x45,
			0xd4, 0xc3, 0x59, 0x50, 0x2e, 0xc6, 0x13, 0x48,
			0x9b, 0x73, 0x3c, 0xd5, 0xa9, 0xc3, 0x85, 0xe2,
			0xd3, 0xe7, 0x75, 0x94, 0x99, 0x63, 0x41, 0xbf,
			0xe3, 0x47, 0x5e, 0x15, 0x94, 0xe6, 0x16, 0x1c,
			0x12, 0x37, 0x8f, 0xf2, 0xe2, 0x92, 0x83, 0x73,
			0xe2, 0x47, 0xab, 0x4c, 0x0a, 0xaf, 0xec, 0xff,
			0x22, 0x03, 0x39, 0xb0, 0xf0, 0x63, 0x47, 0x31,
			0x95, 0xed, 0xf9, 0x17, 0x25, 0xae, 0x5e, 0x11,
			0x47, 0x14, 0xdd, 0x75, 0xde, 0x8d, 0x2f, 0x6e,
			0xb7, 0x86, 0x59, 0x12, 0x39, 0x4c, 0x84, 0x9d,
			0x04, 0xfc, 0xdb, 0xf9, 0x79, 0x21, 0xac, 0x1c,
			0xa7, 0x9c, 0x78, 0x59, 0x84, 0x9e, 0x1b, 0xa0,
			0x9f, 0x39, 0x6c, 0x2b, 0x29, 0x1f, 0x9b, 0x1f,
			0x11, 0xa5, 0x86, 0x4b, 0xaa, 0x11, 0xf0, 0x9f,
			0x5c, 0x24, 0x16, 0x4e, 0xe2, 0x42, 0x27, 0x68,
		},
	},
	/* SHA512 [4] */
	{
		/* Msg4 : 128 byte */
		{
			0x30, 0x22, 0x19, 0xbc, 0x6c, 0x07, 0xe6, 0x28,
			0xf3, 0x52, 0xa4, 0x50, 0x91, 0x71, 0x9b, 0xb0,
			0xe6, 0x55, 0x87, 0x5f, 0xb9, 0x76, 0x91, 0x38,
			0x8f, 0x18, 0x7e, 0x37, 0x7c, 0xcd, 0x27, 0x25,
			0x3a, 0x3c, 0x29, 0x78, 0x78, 0x20, 0xdd, 0x75,
			0x74, 0x94, 0x8b, 0x97, 0xcc, 0x8d, 0xda, 0x34,
			0x92, 0xa8, 0x8b, 0x5e, 0x9e, 0x26, 0x67, 0x08,
			0x28, 0x34, 0x76, 0x69, 0x96, 0x70, 0x72, 0xa6,
			0x92, 0xaa, 0x7b, 0xab, 0x60, 0xe7, 0x0c, 0x59,
			0x6e, 0xac, 0x2e, 0xe7, 0x2d, 0x79, 0xec, 0x82,
			0x20, 0x93, 0x06, 0x37, 0x33, 0x06, 0xea, 0x13,
			0x4d, 0xeb, 0xdf, 0x89, 0x28, 0xe7, 0x84, 0x13,
			0x61, 0xf4, 0x79, 0x1c, 0xcb, 0x64, 0x5c, 0x1e,
			0x09, 0x23, 0xf7, 0x08, 0x5c, 0x3c, 0xb7, 0xe2,
			0xb9, 0x9e, 0x60, 0xb3, 0x1f, 0x21, 0x00, 0xa3,
			0x27, 0xc4, 0x22, 0x5d, 0xdd, 0x38, 0x43, 0xb7,
		},
		/* S4 : 128 byte */
		{
			0x5b, 0x5a, 0x22, 0xbf, 0x6b, 0xe4, 0x5c, 0xf3,
			0x11, 0xc7, 0x8c, 0xbc, 0xe0, 0x8d, 0x9c, 0x49,
			0x9b, 0x23, 0x99, 0x85, 0xe3, 0xce, 0x98, 0x0d,
			0xa0, 0x49, 0x74, 0xc9, 0x7d, 0x16, 0xa2, 0x1c,
			0x0c, 0x38, 0x66, 0x64, 0x60, 0x54, 0x6a, 0x9e,
			0xf1, 0x41, 0xcb, 0xbe, 0x06, 0xe4, 0x8d, 0x1a,
			0xe8, 0x93, 0x1f, 0x92, 0x47, 0xfa, 0x42, 0x11,
			0x42, 0x20, 0xbe, 0x91, 0xac, 0xf8, 0xd8, 0x6e,
			0x21, 0xee, 0x3f, 0x36, 0xbe, 0xd6, 0x44, 0xb1,
			0xc9, 0xc4, 0xe9, 0x67, 0x09, 0x9b, 0xfd, 0x19,
			0x06, 0x76, 0xe0, 0xb1, 0x29, 0x78, 0xe9, 0xb0,
			0xc9, 0x96, 0xaf, 0xa0, 0x70, 0x65, 0xb0, 0x77,
			0x01, 0xbd, 0xca, 0x9e, 0x6c, 0x0c, 0xa2, 0x78,
			0xb8, 0x88, 0x94, 0x54, 0x24, 0x57, 0xf0, 0xfc,
			0x02, 0xb9, 0x38, 0x52, 0x5a, 0xd6, 0xb5, 0x30,
			0x0f, 0xb5, 0x2c, 0x9c, 0xfc, 0x5b, 0x8d, 0xfb,
		},
	},
	/* SHA512 [5] */
	{
		/* Msg5 : 128 byte */
		{
			0x1f, 0xd6, 0x89, 0x98, 0xe4, 0xc3, 0x4d, 0xed,
			0x43, 0xd2, 0xe5, 0x1b, 0x16, 0x80, 0xa6, 0xb3,
			0xfd, 0xfd, 0xd5, 0x76, 0x03, 0x3b, 0x6a, 0x0d,
			0x7f, 0x44, 0x13, 0xfb, 0x39, 0xd2, 0x7c, 0xcf,
			0xce, 0x19, 0xce, 0x9e, 0x56, 0xe8, 0x76, 0xb0,
			0x43, 0xf0, 0x27, 0x7a, 0x8c, 0x63, 0x22, 0xcc,
			0x0c, 0xc2, 0xb5, 0x4c, 0x25, 0xf0, 0x01, 0x23,
			0xa8, 0x8a, 0x01, 0xf6, 0xf9, 0x7b, 0xc8, 0x18,
			0x6e, 0xca, 0x0a, 0xfe, 0xf9, 0xb7, 0xde, 0xf3,
			0x07, 0x07, 0xc2, 0x0c, 0xa7, 0x9c, 0xe0, 0x5f,
			0xef, 0x48, 0x90, 0x70, 0x9b, 0xe3, 0x1d, 0xec,
			0xf8, 0x9b, 0xcc, 0x9a, 0x01, 0x8c, 0x1a, 0x8e,
			0xc7, 0x91, 0x46, 0xa0, 0x16, 0x59, 0x0f, 0x1c,
			0x55, 0xbd, 0xbf, 0xbb, 0xae, 0x50, 0x68, 0xd2,
			0x6f, 0x38, 0x6e, 0xca, 0xb0, 0x3d, 0x45, 0xd0,
			0x37, 0x20, 0x7c, 0xcd, 0x98, 0x2e, 0xfb, 0x99,
		},
		/* S5 : 128 byte */
		{
			0x85, 0xae, 0x6a, 0x4f, 0x4f, 0x6d, 0x0b, 0x8d,
			0xea, 0xbc, 0xb6, 0x5f, 0xe3, 0x19, 0xa5, 0x14,
			0x03, 0x51, 0x2b, 0x74, 0x5a, 0xbe, 0xf4, 0xfd,
			0x30, 0x6a, 0x2f, 0xbe, 0x00, 0x85, 0x80, 0xda,
			0x8a, 0x1e, 0x97, 0x67, 0x13, 0x71, 0x3b, 0xcc,
			0xe3, 0xb2, 0xa5, 0x6c, 0xa2, 0x91, 0x07, 0x43,
			0xad, 0xee, 0x05, 0x8f, 0xd6, 0xe3, 0xdd, 0x0c,
			0xc0, 0x17, 0x4c, 0x15, 0xed, 0x1d, 0x8e, 0x13,
			0x55, 0x16, 0x1e, 0x92, 0x04, 0x93, 0x62, 0x1d,
			0xae, 0xf9, 0x6c, 0x3f, 0x74, 0x10, 0x5e, 0x2b,
			0x65, 0xb9, 0x62, 0x1c, 0xe7, 0xa1, 0x92, 0x4a,
			0x64, 0x9c, 0x13, 0x08, 0x3e, 0xbd, 0xff, 0x4c,
			0xd2, 0x0c, 0xa1, 0xf7, 0x25, 0x96, 0xc7, 0x63,
			0xd0, 0x3b, 0xb5, 0x39, 0xcf, 0xe4, 0x5a, 0x48,
			0xbd, 0x16, 0x15, 0x74, 0x88, 0x2e, 0xf6, 0x08,
			0x54, 0xa6, 0x0c, 0xe4, 0xcf, 0x38, 0xd1, 0xfd,
		},
	},
	/* SHA512 [6] */
	{
		/* Msg6 : 128 byte */
		{
			0x90, 0xd0, 0xd6, 0xa4, 0x10, 0xf8, 0xaa, 0x86,
			0x8b, 0x64, 0x12, 0x8c, 0x03, 0x9c, 0xc6, 0x98,
			0xa3, 0xb2, 0xe3, 0x67, 0x00, 0x74, 0x49, 0x69,
			0xa0, 0xe7, 0x41, 0xd0, 0xb9, 0xf1, 0x64, 0xe7,
			0x3a, 0x56, 0xb8, 0xa6, 0x0f, 0xe6, 0x1b, 0x83,
			0xa1, 0xf4, 0xe5, 0x8a, 0x9d, 0x09, 0xc9, 0xb0,
			0xf7, 0xda, 0xfc, 0x65, 0x2e, 0x78, 0x99, 0x68,
			0xd1, 0xc7, 0x45, 0xdf, 0x42, 0x31, 0x0f, 0xa7,
			0xba, 0x9b, 0x99, 0xe9, 0x8d, 0x98, 0x7d, 0xee,
			0xb0, 0xdd, 0xea, 0x34, 0x78, 0xf6, 0x8e, 0xc1,
			0xa4, 0x37, 0xb9, 0xb7, 0x9e, 0xf4, 0xbf, 0x29,
			0xfe, 0xf5, 0x9e, 0x2d, 0x50, 0x24, 0xe0, 0x32,
			0x17, 0x89, 0xc3, 0x94, 0x12, 0x78, 0x98, 0x6e,
			0xbd, 0x09, 0x69, 0xb0, 0x1b, 0xc9, 0xde, 0x71,
			0xb3, 0xaf, 0x61, 0x86, 0xd8, 0x52, 0x81, 0x53,
			0x2e, 0x58, 0x94, 0xe1, 0x6a, 0x32, 0xa1, 0x31,
		},
		/* S6 : 128 byte */
		{
			0x48, 0x95, 0xd5, 0xbd, 0xda, 0xf9, 0x11, 0xa9,
			0xff, 0x22, 0x73, 0x17, 0x26, 0xe4, 0xd6, 0x8a,
			0x4d, 0x54, 0x47, 0x21, 0xba, 0xeb, 0x82, 0xbe,
			0xa8, 0x44, 0x47, 0x97, 0xaa, 0xbd, 0x45, 0xf9,
			0x9f, 0x9a, 0x72, 0x47, 0x17, 0x37, 0xa5, 0xf6,
			0xdb, 0x09, 0xc9, 0x3e, 0xd0, 0x77, 0x28, 0xf4,
			0x5f, 0xde, 0x81, 0x10, 0xe1, 0xf9, 0x3d, 0x4b,
			0x63, 0xd0, 0x8d, 0x4a, 0x87, 0x39, 0x8f, 0x46,
			0x9c, 0xa5, 0xae, 0xa8, 0x26, 0x7a, 0xfe, 0x9f,
			0xc5, 0x71, 0xb7, 0xea, 0xca, 0xc0, 0x42, 0x51,
			0x66, 0xb9, 0xa0, 0x46, 0x4c, 0xfc, 0x64, 0xd7,
			0xfe, 0xe8, 0x7a, 0xed, 0x80, 0xf7, 0xc2, 0xbb,
			0x82, 0x5a, 0x03, 0x79, 0x91, 0x06, 0x07, 0x05,
			0x33, 0xa9, 0x73, 0x44, 0xc8, 0xcd, 0x63, 0xd5,
			0xbb, 0x8f, 0x67, 0x07, 0x19, 0x8f, 0xe9, 0x96,
			0x55, 0xa5, 0x89, 0x27, 0x2a, 0x0e, 0x5a, 0xf8,
		},
	},
	/* SHA512 [7] */
	{
		/* Msg7 : 128 byte */
		{
			0x64, 0x4a, 0x3d, 0xe6, 0x10, 0x0d, 0xd8, 0xa3,
			0x6c, 0x84, 0x14, 0x46, 0xb9, 0x37, 0x6b, 0x34,
			0x5d, 0x47, 0x8a, 0x32, 0x94, 0x50, 0xa6, 0x6f,
			0x62, 0x93, 0x13, 0xc5, 0xcc, 0x55, 0x13, 0x3c,
			0x47, 0x82, 0xec, 0xd0, 0x71, 0x96, 0x3d, 0x74,
			0xff, 0xbd, 0x91, 0x56, 0xf6, 0x39, 0x35, 0x65,
			0x1f, 0x05, 0x47, 0x11, 0xb4, 0xb8, 0x10, 0x51,
			0x60, 0xab, 0x94, 0xeb, 0x67, 0x5e, 0x66, 0xf0,
			0x20, 0x18, 0xc1, 0x85, 0xac, 0xce, 0xd5, 0xe9,
			0xe2, 0x46, 0x5f, 0xb4, 0x89, 0x7d, 0x7c, 0x5d,
			0xca, 0x45, 0xbf, 0xff, 0xe4, 0x32, 0x81, 0x5d,
			0x22, 0xb7, 0x78, 0x23, 0x0b, 0x8d, 0x8c, 0x6f,
			0xd8, 0x54, 0xe5, 0x8f, 0x2b, 0x7d, 0x49, 0x0d,
			0xfd, 0x67, 0xa5, 0xe8, 0xd8, 0xc8, 0xe9, 0x25,
			0x47, 0x4f, 0x19, 0x86, 0x8c, 0x88, 0xa1, 0x9a,
			0x90, 0xff, 0xea, 0xf6, 0x20, 0xa8, 0x2a, 0xc1,
		},
		/* S7 : 128 byte */
		{
			0x90, 0x00, 0xde, 0xa9, 0x92, 0x2c, 0x03, 0xff,
			0xd3, 0xa1, 0xca, 0x68, 0x9d, 0xd4, 0xb9, 0x8c,
			0x36, 0x8a, 0x12, 0x8f, 0x51, 0x53, 0xb2, 0x89,
			0x5c, 0xe9, 0x95, 0xf1, 0x77, 0x1a, 0x08, 0x16,
			0xa0, 0x9f, 0xd8, 0xb4, 0x93, 0xc4, 0x60, 0xf2,
			0x9e, 0x49, 0xe1, 0xab, 0x6a, 0x38, 0x67, 0xee,
			0xc2, 0x93, 0xf5, 0xd1, 0xfb, 0xb7, 0x7f, 0xbb,
			0x77, 0xc4, 0xa2, 0x8b, 0xb5, 0xb9, 0xe5, 0xd5,
			0xfa, 0x06, 0x55, 0x0e, 0xdf, 0x36, 0x5b, 0x94,
			0xed, 0x4d, 0xaa, 0x3a, 0xb9, 0x81, 0xa3, 0x3d,
			0x7c, 0x00, 0x15, 0xbb, 0xed, 0x25, 0x72, 0xf3,
			0xc5, 0xb0, 0x9f, 0x4f, 0x19, 0x22, 0xe5, 0x62,
			0x67, 0x32, 0xc8, 0xe6, 0xb5, 0xd3, 0x1e, 0x5f,
			0xef, 0x75, 0xd5, 0xa9, 0x02, 0x6c, 0xa5, 0x78,
			0x4d, 0xba, 0x57, 0x1a, 0xb7, 0x2e, 0x3c, 0xf7,
			0x0d, 0xf4, 0xeb, 0xfa, 0x6a, 0xd7, 0xb6, 0x9c,
		},
	},
	/* SHA512 [8] */
	{
		/* Msg8 : 128 byte */
		{
			0x29, 0xe9, 0xcb, 0x34, 0x82, 0xe9, 0xed, 0xa5,
			0xe6, 0xd7, 0x14, 0xa2, 0xdc, 0x73, 0x0d, 0x91,
			0xbd, 0xfb, 0x2d, 0x42, 0xb1, 0x4f, 0xe8, 0x2e,
			0xd9, 0x53, 0xa0, 0xfc, 0xce, 0x95, 0xbb, 0x4a,
			0x4b, 0x30, 0xf8, 0xa9, 0x26, 0xf0, 0x03, 0xe9,
			0x85, 0xb6, 0x47, 0x98, 0x4e, 0x1e, 0x03, 0xc5,
			0xdf, 0x60, 0xa0, 0x09, 0x00, 0x74, 0x42, 0xc9,
			0x15, 0x7a, 0x9d, 0xb5, 0x99, 0x58, 0x0a, 0x94,
			0xc7, 0xa5, 0xd9, 0x44, 0x9a, 0xc6, 0xe9, 0xfc,
			0xf8, 0x56, 0x77, 0xd5, 0x2a, 0xcc, 0x34, 0x89,
			0x90, 0x57, 0x9a, 0x7a, 0x93, 0x0d, 0x7f, 0xf6,
			0xda, 0x45, 0xeb, 0xb9, 0xc1, 0x41, 0x27, 0xb3,
			0x07, 0x11, 0x16, 0x0c, 0xc6, 0xb3, 0xc8, 0x66,
			0xa9, 0x7f, 0x4d, 0x62, 0x59, 0xc3, 0xc7, 0x65,
			0x38, 0xaa, 0xc3, 0x9c, 0x52, 0x62, 0xca, 0x3d,
			0x91, 0x7d, 0x32, 0x10, 0x30, 0x98, 0x3a, 0x2f,
		},
		/* S8 : 128 byte */
		{
			0x62, 0x44, 0xf1, 0x68, 0xe5, 0xf8, 0xef, 0xce,
			0xb0, 0x0f, 0x33, 0xe5, 0xf4, 0xea, 0xa8, 0x21,
			0x2b, 0x1b, 0x41, 0x6b, 0xa7, 0xf1, 0xf9, 0x59,
			0x48, 0xe4, 0x76, 0x75, 0x5d, 0x53, 0x2d, 0xde,
			0xed, 0x5b, 0xb8, 0x49, 0x02, 0xdf, 0x77, 0x50,
			0xf7, 0x75, 0xdc, 0x53, 0xa1, 0x51, 0x73, 0x7b,
			0x39, 0x92, 0x3d, 0xbf, 0xed, 0x37, 0x46, 0x16,
			0xf9, 0x44, 0xb5, 0xf7, 0x3d, 0x63, 0xe0, 0x2d,
			0x90, 0x07, 0xca, 0x1e, 0x9d, 0x33, 0x33, 0xe7,
			0x7f, 0x20, 0x0d, 0x8b, 0x02, 0x19, 0x12, 0xaf,
			0x91, 0x22, 0xf8, 0x61, 0x74, 0x0d, 0x0b, 0xdd,
			0x13, 0x00, 0x75, 0xd9, 0xf1, 0x12, 0xc9, 0x9f,
			0xd6, 0x13, 0xf9, 0xa2, 0x03, 0x67, 0x62, 0xdd,
			0x62, 0x37, 0xb5, 0xc9, 0x1b, 0x33, 0xfe, 0xcc,
			0x36, 0x63, 0x24, 0x53, 0x61, 0x95, 0xac, 0xf5,
			0x55, 0x21, 0x47, 0xad, 0xd0, 0xec, 0x1c, 0x8a,
		},
	},
	/* SHA512 [9] */
	{
		/* Msg9 : 128 byte */
		{
			0x46, 0x7e, 0x8e, 0xa6, 0x34, 0xf7, 0x99, 0x5d,
			0xc4, 0x6c, 0x11, 0xb8, 0xab, 0x0b, 0x75, 0x08,
			0x89, 0x46, 0x81, 0xe8, 0x1c, 0x35, 0x02, 0xc3,
			0xb3, 0x35, 0xe8, 0x97, 0xe6, 0xd6, 0x9d, 0xf8,
			0x85, 0xf4, 0x95, 0x57, 0xce, 0x23, 0x27, 0x84,
			0xe3, 0x51, 0x9b, 0x72, 0x7b, 0xa6, 0x84, 0x3b,
			0xd7, 0xaf, 0x50, 0x63, 0xf8, 0xbc, 0x1d, 0x61,
			0x0f, 0x86, 0xce, 0x5b, 0x35, 0x15, 0x5e, 0x32,
			0x5c, 0xe1, 0x75, 0xbe, 0x85, 0x38, 0x39, 0x5b,
			0x34, 0xdf, 0x67, 0xa4, 0x21, 0xfc, 0xa2, 0x7e,
			0x31, 0x33, 0x0b, 0x59, 0xa4, 0x10, 0x11, 0xb2,
			0x90, 0xa5, 0x8b, 0xdc, 0x8e, 0x74, 0x04, 0x01,
			0xb3, 0x8f, 0x55, 0x64, 0xc2, 0xfd, 0x7a, 0xe8,
			0x9f, 0x60, 0x9e, 0xd6, 0x07, 0xd5, 0x78, 0xdb,
			0x7f, 0x1c, 0xda, 0x50, 0x8a, 0xf9, 0x87, 0xbe,
			0x1f, 0xd9, 0x46, 0xa2, 0x5a, 0xb9, 0x34, 0x6d,
		},
		/* S9 : 128 byte */
		{
			0x5d, 0x5c, 0xf1, 0x80, 0xbd, 0xac, 0x8a, 0xe1,
			0x4a, 0x50, 0xc4, 0xe8, 0xb5, 0x7b, 0x88, 0xba,
			0x68, 0xb1, 0xc3, 0x52, 0x7d, 0x34, 0xa6, 0x3f,
			0xd3, 0x90, 0xf3, 0x3b, 0x8b, 0x02, 0x5a, 0x75,
			0xf7, 0xce, 0x82, 0x4f, 0x1e, 0x20, 0xde, 0x81,
			0x49, 0x91, 0x36, 0x4a, 0x97, 0xc2, 0xab, 0x9e,
			0x6c, 0x88, 0xcd, 0x19, 0x6a, 0xbf, 0x6c, 0x4a,
			0x0a, 0xc6, 0x60, 0x13, 0xa7, 0x3d, 0xb9, 0xe9,
			0x4a, 0x98, 0x8a, 0x46, 0x65, 0xa3, 0x1c, 0xd4,
			0xe0, 0x73, 0x18, 0x08, 0xc8, 0x8e, 0xc1, 0x45,
			0x7c, 0x48, 0x10, 0x47, 0xf3, 0xe3, 0x2b, 0x08,
			0xbf, 0xf6, 0x56, 0x0c, 0xbe, 0x85, 0xce, 0xc2,
			0x12, 0x23, 0x9d, 0xc8, 0xb2, 0x54, 0x67, 0xde,
			0xbe, 0x53, 0x6a, 0x8a, 0xb1, 0xa9, 0x51, 0xaf,
			0xcd, 0xcb, 0x2c, 0xfc, 0x41, 0x1a, 0xfd, 0x5c,
			0x18, 0x9f, 0x4b, 0x0e, 0xab, 0xee, 0xc3, 0xf7,
		},
	},
};

typedef mod1024_t tv_t;


#include "siggen15test.c"
