/* poly1305testfc.c */
/*
 * Copyright (c) 2017 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <ctype.h>

#include <aicrypto/nrg_poly1305.h>

#include "poly1305-donna/poly1305-donna.h"
#include "poly1305test.h"

/*
 * This constant is defined same value in poly1305-donna-32.h and
 * poly1305-donna-64.h
 */
#define poly1305_block_size 16

#ifdef POLY1305_DEBUG
static inline void dump(const uint8_t *buf, const size_t bytes)
{
	int i;

	for (i = 0; i < bytes; i++) {
		printf("%02x ", buf[i]);
	}
	printf("\n");
}
#endif /* POLY1305_DEBUG */

int test_Poly1305()
{
	/* Key Material: 85:d6:be:78:57:55:6d:33:7f:44:52:fe:42:d5:06:a8:01:
	 * 03:80:8a:fb:0d:b2:fd:4a:bf:f6:af:41:49:f5:1b
	 *
	 * [0..15] = r before clamping
	 * [16-31] = s
	 *
	 * r before clamping:
	 * 85:d6:be:78:57:55:6d:33:7f:44:52:fe:42:d5:06:a8
	 *
	 * s as an octet string:
	 * 01:03:80:8a:fb:0d:b2:fd:4a:bf:f6:af:41:49:f5:1b
	 */
	static const uint8_t key_material[] = {
		0x85, 0xd6, 0xbe, 0x78, 0x57, 0x55, 0x6d, 0x33,
		0x7f, 0x44, 0x52, 0xfe, 0x42, 0xd5, 0x06, 0xa8,
		0x01, 0x03, 0x80, 0x8a, 0xfb, 0x0d, 0xb2, 0xfd,
		0x4a, 0xbf, 0xf6, 0xaf, 0x41, 0x49, 0xf5, 0x1b,
	};

	/* Clamped r as a number:
	 * 806d5400e52447c036d555408bed685 */
	static const uint8_t r_clamped[] = {
		0x85, 0xd6, 0xbe, 0x08, 0x54, 0x55, 0x6d, 0x03,
		0x7c, 0x44, 0x52, 0x0e, 0x40, 0xd5, 0x06, 0x08,
	};

	/* Block #1
	 * Acc = ((Acc+Block)*r) % P = 2c88c77849d64ae9147ddeb88e69c83fc */
	static const uint8_t block1_acc[] = {
		0xfc, 0x83, 0x9c, 0xe6, 0x88, 0xeb, 0xdd, 0x47,
		0x91, 0xae, 0x64, 0x9d, 0x84, 0x77, 0x8c, 0xc8,
		0x02,
	};

	/* Block #1
	 * Acc = ((Acc+Block)*r) % P = 2d8adaf23b0337fa7cccfb4ea344b30de */
	static const uint8_t block2_acc[] = {
		0xde, 0x30, 0x4b, 0x34, 0xea, 0xb4, 0xcf, 0xcc,
		0xa7, 0x7f, 0x33, 0xb0, 0x23, 0xaf, 0xad, 0xd8,
		0x02,
	};

#if 0
	/* Last Block
	 * ((Acc + Block) * r) % P = 28d31b7caff946c77c8844335369d03a7 */
	static const uint8_t last_acc[] = {
		0xa7, 0x03, 0x9d, 0x36, 0x35, 0x43, 0x84, 0xc8,
		0x77, 0x6c, 0x94, 0xff, 0xca, 0xb7, 0x31, 0x8d,
		0x02,
	};
#endif
	/* Tag: a8:06:1d:c1:30:51:36:c6:c2:2b:8b:af:0c:01:27:a9 */
	static const uint8_t tag[] = {
		0xa8, 0x06, 0x1d, 0xc1, 0x30, 0x51, 0x36, 0xc6,
		0xc2, 0x2b, 0x8b, 0xaf, 0x0c, 0x01, 0x27, 0xa9,
	};

	/* Message to be Authenticated: */
	/*
  000  43 72 79 70 74 6f 67 72 61 70 68 69 63 20 46 6f  Cryptographic Fo
  016  72 75 6d 20 52 65 73 65 61 72 63 68 20 47 72 6f  rum Research Gro
  032  75 70                                            up
	 */
	static const uint8_t message[] = "Cryptographic Forum Research Group";

	poly1305_context ctx;
	unsigned char mac[16];
	uint8_t r[16];
	uint8_t block[24];
	size_t bytes;
	int i;

	for (i = 0; i < sizeof(mac); i++)
		mac[i] = 0;

	poly1305_init(&ctx, key_material);

	poly1305_copy_r(&ctx, r);
#ifdef POLY1305_DEBUG
	dump(r, 16);
#endif /* POLY1305_DEBUG */
	for (i = 0; i < 16; i++) {
		if (r[i] != r_clamped[i]) {
			printf("error : test Poly1305 1 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test Poly1305 1 ok -- %d\n", i + 1);
		}
	}

	/* Block #1 */
	bytes = strlen((const char *) message);
	poly1305_update(&ctx, message + 0, poly1305_block_size);

	poly1305_copy_h(&ctx, block);
#ifdef POLY1305_DEBUG
	dump(block, 17);
#endif /* POLY1305_DEBUG */
	for (i = 0; i < 17; i++) {
		if (block[i] != block1_acc[i]) {
			printf("error : test Poly1305 2 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test Poly1305 2 ok -- %d\n", i + 1);
		}
	}

	/* Block #2 */
	bytes -= poly1305_block_size;
	poly1305_update(&ctx, message + 16, poly1305_block_size);

	poly1305_copy_h(&ctx, block);
#ifdef POLY1305_DEBUG
	dump(block, 17);
#endif /* POLY1305_DEBUG */
	for (i = 0; i < 17; i++) {
		if (block[i] != block2_acc[i]) {
			printf("error : test Poly1305 3 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test Poly1305 3 ok -- %d\n", i + 1);
		}
	}

	/* Last Block */
	bytes -= poly1305_block_size;
	poly1305_update(&ctx, message + 32, 2);
	poly1305_finish(&ctx, mac);

#if 0
	poly1305_copy_h(&ctx, block);
	dump(block, 17);
	for (i = 0; i < 17; i++) {
		if (block[i] != last_acc[i]) {
			printf("error : test Poly1305 4 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test Poly1305 4 ok -- %d\n", i + 1);
		}
	}
#endif
#ifdef POLY1305_DEBUG
	dump(mac, 16);
#endif /* POLY1305_DEBUG */
	for (i = 0; i < 16; i++) {
		if (mac[i] != tag[i]) {
			printf("error : test Poly1305 5 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test Poly1305 5 ok -- %d\n", i + 1);
		}
	}

	return EXIT_SUCCESS;
}
