#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_pkcs11.h>

/* eToken PKI */
#if defined(ETOKEN_WIN)
# define P11LIB 	"etpkcs11.dll"
# define P11PIN  	"1234567890"
/* ePass1000 */
#elif defined(EPASS_WIN)
# define P11LIB 	"Ep1pk111.dll"
# define P11PIN  	"abcde"
#endif

/* if no Cryptoki library is specified, then the test will be skipped. */
#ifndef P11LIB
# define P11LIB	"NONE"
# define P11PIN	"personal-identification-number"
#endif

#define P11KEYLEN	512
#define P11LABEL	"CA-dev"
#define P11LABEL2	"CA-dev 2"

#define P12_TESTF2	"test.p12~"

int test_pkcs_p11(void)
{
	Pubkey_RSA *pub1, *pub2;
	Cert *ct1, *ct2;
	Key *key = NULL;
	Prvkey_P11RSA *p11prv;
	Pubkey_P11RSA *p11pub;
	PKCS12 *p12 = NULL;
	PKCS11 *p11 = NULL;
	P11Session *p11s = NULL;
	CK_INFO info;
	CK_SLOT_INFO slot_info;
	CK_TOKEN_INFO token_info;
	CK_SLOT_ID slot_id[8];
	CK_ULONG ulObjectCount = 8;
	unsigned char *subject, id[32], in[256], cry[256], dec[256];
	int i, slen, ok = -1, klen = P11KEYLEN;
	unsigned char digest[20], *sig = NULL;
	char *text;

	memset(&info, 0, sizeof(info));
	memset(&slot_info, 0, sizeof(slot_info));
	memset(&token_info, 0, sizeof(token_info));
	memset(&slot_id, 0, sizeof(slot_id));

	/* load pkcs11 library */
	if ((p11 = P11_init(P11LIB)) == NULL) {
		printf("test pkcs11 initialize -- error!\n");
		return -1;
	}
	printf("test pkcs11 : success to load %s -- ok\n", P11LIB);

	/* get info test */
	if (P11_get_libinfo(p11, &info)) {
		printf("test pkcs11 get lib info -- error!\n");
		goto done;
	}
	if (P11_get_slotlist(p11, TRUE, slot_id, &ulObjectCount)) {
		printf("test pkcs11 get slot list -- error!\n");
		goto done;
	}
#if 1
	if (P11_get_slotinfo(p11, slot_id[0], &slot_info)) {
		printf("test pkcs11 get slot info -- error!\n");
		goto done;
	}
	if (P11_get_tokeninfo(p11, slot_id[0], &token_info)) {
		printf("test pkcs11 get token info -- error!\n");
		goto done;
	}
	printf("test pkcs11 : get info test -- ok\n");

	/* key generation test */
	if ((p11s =
	     P11_open_session(p11, slot_id[0],
			      (CKF_SERIAL_SESSION | CKF_RW_SESSION),
			      0)) == NULL) {
		printf("test pkcs11 open session -- error!\n");
		goto done;
	}
	if (P11_login(p11s, CKU_USER, P11PIN)) {
		printf("test pkcs11 login -- error!\n");
		goto done;
	}
	/* key generation */
	if (P11_rsa_generate(p11s, P11LABEL, klen, &pub1)) {
		printf("test pkcs11 key generation -- error!\n");
		goto done;
	}
	printf("test pkcs11 : key generation test -- ok\n");

	OK_set_passwd("abcde");
	if ((p12 = P12_read_file(P12_TESTF2)) == NULL) {
		printf("test pkcs11 read p12 file -- error!\n");
		goto done;
	}
	OK_clear_passwd();

	ct1 = P12_get_usercert(p12);
	key = P12_get_privatekey(p12);
	if ((subject = P11_util_subject(ct1)) == NULL)
		goto done;
	if (P11_util_keyid(key, id))
		goto done;

	if (P11_put_cert(p11s, ct1, P11LABEL2, subject, id)) {
		printf("test pkcs11 put certificate -- error!\n");
		goto done;
	}
	if (P11_put_rsakey(p11s, key, P11LABEL2, subject, id)) {
		printf("test pkcs11 put private key -- error!\n");
		goto done;
	}
	printf("test pkcs11 : put cert & rsa (privte) key -- ok\n");

	if ((ct2 = P11_get_cert(p11s, P11LABEL2)) == NULL) {
		printf("test pkcs11 get certificate -- error!\n");
		goto done;
	}
	if ((pub2 = (Pubkey_RSA *) P11_get_rsapub(p11s, P11LABEL2)) == NULL) {
		printf("test pkcs11 get public key -- error!\n");
		goto done;
	}
	printf("test pkcs11 : get cert & rsa (public) key -- ok\n");

	if (Cert_cmp(ct1, ct2)) {
		printf("test pkcs11 cert cmp -- error!\n");
		goto done;
	}
	if (Key_pair_cmp(key, (Key *) pub2)) {
		printf("test pkcs11 key pair cmp -- error!\n");
		goto done;
	}
	printf("test pkcs11 : compare cert & rsa key -- ok\n");

	Key_free((Key *) pub1);
	Key_free((Key *) pub2);
	P12_free(p12);
	Cert_free(ct2);
	P11_logout(p11s);
	P11_close_session(p11s);
#endif

	/* key operation test */
	if (P11_get_slotlist(p11, TRUE, slot_id, &ulObjectCount)) {
		printf("test pkcs11 get slot list -- error!\n");
		goto done;
	}

	OK_set_passwd(P11PIN);
	if ((p11pub = P11_open_rsapub(p11, slot_id[0], P11LABEL, 0)) == NULL) {
		printf("test pkcs11 get rsa public key -- error!\n");
		goto done;
	}
	if ((p11prv = P11_open_rsaprv(p11, slot_id[0], P11LABEL, 0)) == NULL) {
		printf("test pkcs11 get rsa private key -- error!\n");
		goto done;
	}
	OK_clear_passwd();

	printf("test pkcs11 : open public & private key -- ok\n");

	klen = klen >> 3;
	for (i = 0; i < klen; i++)
		in[i] = i & 0xff;

	/* public key encryption test */
	printf("test pkcs11 : public key encryption ");
	for (i = 0; i < 10; i++) {
		if (P11RSApub_doCrypt(klen - i, in, cry, p11pub)) {
			printf("test pkcs11 public do Crypto -- error!\n");
			goto done;
		}
		if (P11RSAprv_doCrypt(klen, cry, dec, p11prv)) {
			printf("test pkcs11 private do Crypto -- error!\n");
			goto done;
		}
		if (memcmp(in, &dec[i], klen - i)) {
			printf("test pkcs11 public key encryption -- error!\n");
			goto done;
		}
		printf(".");
		fflush(stdout);
	}
	printf(" -- ok\n");

	/* private key signature test */
	printf("test pkcs11 : private key signature ");
	for (i = 0; i < 10; i++) {
		if (P11RSAprv_doCrypt(klen - i, in, cry, p11prv)) {
			printf("test pkcs11 private do Crypto -- error!\n");
			goto done;
		}
		if (P11RSApub_doCrypt(klen, cry, dec, p11pub)) {
			printf("test pkcs11 public do Crypto -- error!\n");
			goto done;
		}
		if (memcmp(in, &dec[i], klen - i)) {
			printf("test pkcs11 private key signature -- error!\n");
			goto done;
		}
		printf(".");
		fflush(stdout);
	}
	printf(" -- ok\n");
	Key_free((Key *) p11pub);
	P11key_free((P11Key *) p11prv);

	/* ok_signature & verify test */
	OK_set_passwd(P11PIN);
	if ((p11pub = P11_open_rsapub(p11, slot_id[0], P11LABEL, 0)) == NULL) {
		printf("test pkcs11 get rsa public key -- error!\n");
		goto done;
	}
	if ((p11prv = P11_open_rsaprv(p11, slot_id[0], P11LABEL, 0)) == NULL) {
		printf("test pkcs11 get rsa private key -- error!\n");
		goto done;
	}
	OK_clear_passwd();

	text = "test signature text";
	OK_SHA1(strlen(text), text, digest);

	printf("test pkcs11 : OK signature & verify ");
	for (i = 0; i < 5; i++) {
		if (OK_do_signature
		    ((Key *) p11prv, text, strlen(text), &sig, &slen,
		     OBJ_SIG_SHA1RSA))
			goto done;
		if (OK_do_verify((Key *) p11pub, digest, sig, OBJ_SIG_SHA1RSA))
			goto done;
		free(sig);
		printf(".");
		fflush(stdout);
	}
	printf(" -- ok\n");
	Key_free((Key *) p11pub);
	P11key_free((P11Key *) p11prv);

	ok = 0;
 done:
	if (p11)
		P11_free(p11);
	if (ok)
		printf("%s\n", OK_get_errstr());
	return 0;
}

int main(void)
{
	int rv = EXIT_SUCCESS;

	if (strcmp(P11LIB, "NONE") == 0) {
		fprintf(stderr, "no Cryptoki library specified.\n");
		return 77;
	}

	if (test_pkcs_p11())
		rv = EXIT_FAILURE;

	return rv;
}
