/* pkcstestfc.c */
/*
 * Modified by National Institute of Informatics in Japan, 2013-2018.
 *
 */

#include "aiconfig.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_sha1.h>
#include <aicrypto/ok_pkcs.h>
#include <aicrypto/ok_pkcs11.h>
#include <aicrypto/ok_tool.h>
#include <aicrypto/ok_x509.h>

#define P12_TESTF	"test.p12"
#define P12_TESTF2	"test.p12~"	/* for writing and reading */
#define P12_DSAF	"dsap.p12"
#define P12_DSAF2	"dsap.p12~"	/* for writing */
#define P7S_TESTF	"test.p7b"
#define P7S_TESTF2	"test.p7b~"	/* for writing */
#define P8_TESTF	"testp8.crtx"
#define P8_TESTF2	"testp8.crtx~"	/* for writing */
#define P8E_TESTF	"testp8e.crtx"
#define P8E_TESTF2	"testp8e.crtx~"	/* for writing */

#ifndef PATH
# define PATH	"."
#endif

/* test/getfpath.c */
char *get_fpath(char *path, char *filename);


int test_pkcs_rwfile(){
	PKCS12	*p12,*dmy;
	PKCS7	*p7s,*dm2;
	int i;
	char *fp_P12_TESTF = get_fpath(PATH, P12_TESTF);
	char *fp_P12_DSAF = get_fpath(PATH, P12_DSAF);
	char *fp_P7S_TESTF = get_fpath(PATH, P7S_TESTF);

	/* test PKCS#12 */
	for(i=0;i<11;i++){
		/* use default cipher setting.
		 * this means that cert encryption uses RC2 40bit,
		 * and key encryption uses 3DES 192bit.
		 * so actually, I can test all Pbe encryption & decryption
		 * algorithms :-)
		 */
		OK_set_passwd("abcde");
		if ((p12 = P12_read_file(fp_P12_TESTF)) == NULL){
			printf("test pkcs12 read file -- error!\n");
			return -1;
		}
		OK_set_passwd("abcde");
		if(P12_write_file(p12,P12_TESTF2)){
			printf("test pkcs12 write file -- error!\n");
			return -1;
		}
		OK_clear_passwd();

		/* test duplicate and free it! */
		dmy=P12_dup(p12);
		P12_free(p12);
		P12_free(dmy);

		if((i%5)==0)printf("test pkcs12 read & write file -- ok : %d\n",i);
	}

	/* test PKCS#12 (DSA key & cert version) */
	for(i=0;i<11;i++){
		/* use default cipher setting.
		 * this means that cert encryption uses RC2 40bit,
		 * and key encryption uses 3DES 192bit.
		 * so actually, I can test all Pbe encryption & decryption
		 * algorithms :-)
		 */
		OK_set_passwd("abcde");
		if ((p12 = P12_read_file(fp_P12_DSAF)) == NULL){
			printf("test pkcs12 (DSA) read file -- error!\n");
			return -1;
		}
		OK_set_passwd("abcde");
		if(P12_write_file(p12,P12_DSAF2)){
			printf("test pkcs12 (DSA) write file -- error!\n");
			return -1;
		}
		OK_clear_passwd();

		/* test duplicate and free it! */
		dmy=P12_dup(p12);
		P12_free(p12);
		P12_free(dmy);

		if((i%5)==0)printf("test pkcs12 (DSA) read & write file -- ok : %d\n",i);
	}

	/* test PKCS#7 (p7b) */
	for(i=0;i<11;i++){
		if((p7s = P7s_read_file(fp_P7S_TESTF)) == NULL){
			printf("test pkcs7 read file -- error!\n");
			return -1;
		}
		
		dm2=P7_dup(p7s);
		if(P7s_write_file(dm2,P7S_TESTF2)){
			printf("test pkcs7 write file -- error!\n");
			return -1;
		}

		/* test duplicate and free it! */
		P7_free(p7s);
		P7_free(dm2);

		if((i%5)==0)printf("test pkcs7b read & write file -- ok : %d\n",i);
	}
	free(fp_P12_TESTF); free(fp_P12_DSAF); free(fp_P7S_TESTF);
	return 0;
}

int test_pkcs_p7(){
	PKCS12	*p12;
	PKCS7	*p7s,*p7m,*dm2;
	unsigned char *der,*ret;
	int digest_algo;
	int i,j;

	digest_algo = OK_get_p7s_digest_algo();

	/* test PKCS#7 (p7s) */
	for(i=0;i<11;i++){
		char *txt = "This is sample txt file for PKCS#7 Signed Data...!#$%&'()==@:;testTEST :-)";

		OK_set_passwd("abcde");
		/* P12_TESTF2 would be created by test_pkcs_rwfile(). */
		if((p12=P12_read_file(P12_TESTF2))==NULL){
			printf("test pkcs7 Signed : read p12 file -- error!\n");
			return -1;
		}
		OK_clear_passwd();

		j = strlen(txt);
		if((p7s= P7s_get_signed(p12, txt, j, digest_algo))==NULL){
			printf("test pkcs7 Signed : P7s_get_signed -- error!\n");
			return -1;
		}
		if(P7s_verify_signed(p7s,NULL,0)){
			printf("test pkcs7 Signed : P7s_verify_signed (1) -- error!\n");
			return -1;
		}

		dm2= P7_dup(p7s);
		P7_free(p7s);

		der=P7_signed_toDER(dm2,NULL,&j);

		if((p7s=ASN1_read_p7s(der))==NULL){
			printf("test pkcs7 Signed : ASN1_read_p7s -- error!\n");
			return -1;
		}

		if(P7s_verify_signed(p7s,NULL,0)){
			printf("test pkcs7 Signed : P7s_verify_signed (2) -- error!\n");
			return -1;
		}

		free(der);
		P7_free(p7s);
		P7_free(dm2);
		P12_free(p12);
		if((i%5)==0)printf("test pkcs7 Signed gen & read & vfy -- ok : %d\n",i);
	}

	/* test PKCS#7 (p7m) */
	for(i=0;i<11;i++){
		char *txt = "This is sample txt file for PKCS#7 Enveloped Data...!#$%&'()==@:;testTEST :-)";
		Cert *cert;
		Key *key;

		OK_set_passwd("abcde");
		if((p12=P12_read_file(P12_TESTF2))==NULL){
			printf("test pkcs7 Signed : read p12 file -- error!\n");
			return -1;
		}
		OK_clear_passwd();

		j = strlen(txt)+1;
		if((p7m=P7m_encrypt_enveloped((PKCS7*)p12,txt,j))==NULL){
			printf("test pkcs7 Envelope : P7m_encrypt_enveloped -- error!\n");
			return -1;
		}

		dm2= P7_dup(p7m);
		P7_free(p7m);

		der=P7_envelope_toDER(dm2,NULL,&j);

		if((p7m=ASN1_read_p7env(der))==NULL){
			printf("test pkcs7 Envelope : ASN1_read_p7env -- error!\n");
			return -1;
		}

		cert=P12_get_usercert(p12);
		key=P12_get_privatekey(p12);
		if((ret=P7m_decrypt_enveloped(p7m,cert,key))==NULL){
			printf("test pkcs7 Envelope : P7m_decrypt_enveloped -- error!\n");
			return -1;
		}

		if(strcmp(txt,ret)){
			printf("test pkcs7 Envelope : encrypt & decrypt -- error!\n");
			return -1;
		}
		free(der);
		free(ret);
		P7_free(p7m);
		P7_free(dm2);
		P12_free(p12);
		if((i%5)==0)printf("test pkcs7 Envelope : encrypt & decrypt -- ok : %d\n",i);
	}
	return 0;
}

enum enc_algo {
	MD2DES = OBJ_P5_MD2DES,
	MD5DES = OBJ_P5_MD5DES
};
int p8e_rwfile(enum enc_algo);
int test_pkcs_p8(){
	Key *p8;
	int i;
	char *fp_P8_TESTF = get_fpath(PATH, P8_TESTF);

	/* read non-encrypted PKCS#8 file. */
	for(i=0;i<11;i++){
		if ((p8 = P8_read_file(fp_P8_TESTF)) == NULL){
			printf("test pkcs8 read file -- error!\n");
			return -1;
		}
		if(P8_write_file(p8,P8_TESTF2)){
			printf("test pkcs8 write file -- error!\n");
			return -1;
		}
		if((i%5)==0)printf("test pkcs8 read & write file -- ok : %d\n",i);

		Key_free(p8);
	}
	free(fp_P8_TESTF);

	OK_set_passwd("abcde");
	/* read and write encrypted PKCS#8 file. */
#ifdef HAVE_MD2
	if ((p8e_rwfile(MD2DES)) != 0)
		return -1;
#else
	printf("skipped: test for encrypted PKCS#8 using MD2\n");
#endif
	if ((p8e_rwfile(MD5DES)) != 0)
		return -1;

	return 0;
}

int p8e_rwfile(enum enc_algo eA) {
	int i;
	Key *p8;
	char *fp_P8E_TESTF = get_fpath(PATH, P8E_TESTF);

	for (i = 0; i < 12; i++) {
		if ((p8 = P8enc_read_file(fp_P8E_TESTF)) == NULL){
			printf("test pkcs8 read file -- error!\n");
			return -1;
		}

		/* there are 6 algorithms for PKCS#5. see include/ok_asn1.h.
		 * among such algorithms, there are 2 algorithms depending
		 * on MD2. see also pem/pemtestfc.c (test_pem_pkcs).
		 */
		switch (eA) {
		case MD2DES:
			OK_set_p5_cry_algo(MD2DES + (i % 6));
			break;
		case MD5DES:
			OK_set_p5_cry_algo(MD5DES + (i % 4));
			break;
		defaut:
			break;
		}

		if (P8enc_write_file(p8, P8E_TESTF2)) {
			printf("encryptionAlgorithm: %d\n", eA);
			printf("test pkcs8 write file -- error!\n");
			return -1;
		}
		Key_free(p8);
		if ((p8 = P8enc_read_file(P8E_TESTF2)) == NULL) {
			printf("encryptionAlgorithm: %d\n", eA);
			printf("test pkcs8 read file -- error!\n");
			return -1;
		}
		Key_free(p8);

		if((i%5)==0) printf("test encrypted pkcs8 read & write file -- ok : %d\n",i);

	}
	free(fp_P8E_TESTF);
	return 0;
}
