/* pkcs11.c */
/*
 * Modified by National Institute of Informatics in Japan, 2013-2016.
 *
 */
/*
 * Copyright (C) 1998-2003
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "aiconfig.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>

#ifdef HAVE_DLFCN_H
#include <dlfcn.h>
#endif

#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_sha1.h>
#include <aicrypto/ok_pkcs.h>
#include <aicrypto/ok_pkcs11.h>

/*-------------------------------------------
	initialize and load library
-------------------------------------------*/
PKCS11 *P11_init(char *libname){
	CK_C_GetFunctionList pGFL = 0;
	CK_RV rv = 0;
	PKCS11 *p11 = NULL;

	if((p11=(PKCS11*)malloc(sizeof(PKCS11)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_PKCS11,ERR_PT_PKCS11,NULL);
		return NULL;
	}
	memset(p11,0,sizeof(PKCS11));

	p11->version = 1;
	strncpy(p11->libname,libname,254);

	/* Load the library */
	if((p11->hLib=dlopen(libname,RTLD_NOW))==NULL){
		/* XXX: it is difficult to set &errno into the 4th argument. */
		OK_set_error(ERR_ST_LOADLIB,ERR_LC_PKCS11,ERR_PT_PKCS11,NULL);
		goto error;
	}
	/* Find the entry point. */
	if((pGFL=(CK_C_GetFunctionList)dlsym(p11->hLib,"C_GetFunctionList"))==NULL){
		/* XXX: ditto above */
		OK_set_error(ERR_ST_GETPROCADDR,ERR_LC_PKCS11,ERR_PT_PKCS11,NULL);
		goto error;
	}

	if((rv=pGFL(&p11->pFunc)) != CKR_OK){
		OK_set_error(ERR_ST_GETFUNCLIST,ERR_LC_PKCS11,ERR_PT_PKCS11,&rv);
		goto error;
	}
	/* Find optional entry point. */
	p11->pFunc->CA_ActivateMofN
		= (CK_CA_ActivateMofN)dlsym(p11->hLib,"CA_ActivateMofN");

	rv = p11->pFunc->C_Initialize(0);
	if((rv != CKR_OK) && (rv != CKR_CRYPTOKI_ALREADY_INITIALIZED)){
		OK_set_error(ERR_ST_P11_INIT,ERR_LC_PKCS11,ERR_PT_PKCS11,&rv);
		goto error;
	}

	return p11;
error:
	P11_free(p11);
	return NULL;
}

void P11_free(PKCS11 *p11){
	if(p11==NULL) return;
	if(p11->pFunc) p11->pFunc->C_Finalize(0);
	if(p11->hLib) dlclose(p11->hLib);
	free(p11);
}

/*-------------------------------------------
	get slot list
-------------------------------------------*/
int P11_get_libinfo(PKCS11 *p11, CK_INFO_PTR ret){
	CK_RV rv = -1;
	if((rv=p11->pFunc->C_GetInfo(ret)) != CKR_OK){
		OK_set_error(ERR_ST_P11_GETINFO,ERR_LC_PKCS11,ERR_PT_PKCS11+1,&rv);
		return -1;
	}
	return 0;
}

int P11_get_slotlist(PKCS11 *p11, CK_BBOOL active, CK_SLOT_ID_PTR ret, unsigned long *count){
	CK_RV rv = -1;
	if((rv=p11->pFunc->C_GetSlotList(active, ret, count)) != CKR_OK){
		OK_set_error(ERR_ST_P11_SLOTLIST,ERR_LC_PKCS11,ERR_PT_PKCS11+2,&rv);
		return -1;
	}
	return 0;
}

int P11_get_slotinfo(PKCS11 *p11, CK_SLOT_ID idx, CK_SLOT_INFO_PTR ret){
	CK_RV rv = -1;
	if((rv=p11->pFunc->C_GetSlotInfo(idx,ret)) != CKR_OK){
		OK_set_error(ERR_ST_P11_SLOTINFO,ERR_LC_PKCS11,ERR_PT_PKCS11+3,&rv);
		return -1;
	}
	return 0;
}

int P11_get_tokeninfo(PKCS11 *p11, CK_SLOT_ID idx, CK_TOKEN_INFO_PTR ret){
	CK_RV rv = -1;
	if((rv=p11->pFunc->C_GetTokenInfo(idx,ret)) != CKR_OK){
		OK_set_error(ERR_ST_P11_TKNINFO,ERR_LC_PKCS11,ERR_PT_PKCS11+4,&rv);
		return -1;
	}
	return 0;
}
