/* p11_tool.c */
/*
 * Modified by National Institute of Informatics in Japan, 2013-2014.
 *
 */
/*
 * Copyright (C) 1998-2003
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_sha1.h>
#include <aicrypto/ok_pkcs.h>
#include <aicrypto/ok_pkcs11.h>

/*-------------------------------------------
	get certificate serial DER
-------------------------------------------*/
unsigned char *P11_util_serialnum(Cert *cert){
	unsigned char *in,*cp;

	if((in=cert->der)==NULL) goto error;

	if((cp = ASN1_next(in))==NULL) goto error;
	if((*in!=0x30)||(*cp!=0x30)) goto error;

	if((cp = ASN1_step(in,2))==NULL) goto error;

	/* check x509 v3 certificate or not */
	if((cp[0]==0xa0)||(cp[1]==0x03))
		if((cp = ASN1_skip(cp))==NULL) goto error;

	/* serial number */
	if(cp[0] != ASN1_INTEGER){
		OK_set_error(ERR_ST_ASN_NOTINTEGER,ERR_LC_PKCS11,ERR_PT_P11TOOL,NULL);
		goto error;
	}

	return cp;
error:
	return NULL;
}

unsigned char *P11_util_issuer(Cert *cert){
	unsigned char *cp;

	/* get serial number */
	if((cp=P11_util_serialnum(cert))==NULL) goto error;
	if((cp = ASN1_next(cp))==NULL) goto error;

	/* algo id */
	if((cp = ASN1_skip(cp))==NULL) goto error;

	/* now issuer DN -- check it! */
	if(cp[0] != 0x30){
		OK_set_error(ERR_ST_ASN_NOTASN1,ERR_LC_PKCS11,ERR_PT_P11TOOL+1,NULL);
		goto error;
	}
	return cp;
error:
	return NULL;
}

unsigned char *P11_util_subject(Cert *cert){
	unsigned char *cp;

	/* get issuer DN */
	if((cp=P11_util_issuer(cert))==NULL) goto error;
	if((cp = ASN1_skip(cp))==NULL) goto error;

	/* validity */
	if((cp = ASN1_skip(cp))==NULL) goto error;

	/* now subject DN -- check it ! */
	if(cp[0] != 0x30){
		OK_set_error(ERR_ST_ASN_NOTASN1,ERR_LC_PKCS11,ERR_PT_P11TOOL+2,NULL);
		goto error;
	}
	return cp;
error:
	return NULL;
}

/* label should be more than 64 bytes */
int P11_util_label(Cert *cert, char *label){
	return get_dn_for_friendlyname(&cert->subject_dn,label);
}

/* id should be more than 20 bytes */
int P11_util_keyid(Key *key, unsigned char *id){
	unsigned char *pub=NULL;
	unsigned char old[256];
	int ok=-1,j;

	if(key==NULL) goto done;
	switch(key->key_type){
	case KEY_RSA_PUB:
		if(LN_get_num_c(((Pubkey_RSA*)key)->n,128,old)) goto done;
#if 0
		if((pub=RSApub_toDER((Pubkey_RSA*)key,NULL,&j))==NULL)
			goto done;
		break;
	case KEY_DSA_PUB:
		if((pub=DSApub_toDER((Pubkey_DSA*)key,NULL,&j))==NULL)
			goto done;
#endif
		break;
	case KEY_ECDSA_PUB:
		if((pub=ECDSApub_toDER((Pubkey_ECDSA*)key,NULL,&j))==NULL)
			goto done;
		break;
	}
#if 0
	OK_SHA1(j,pub,id); /* get hash */
#else
	OK_SHA1(128,old,id); /* get hash */
#endif
	ok = 0;
done:
	if(pub) free(pub);
	return ok;
}

