/* p11_sess.c */
/*
 * Modified by National Institute of Informatics in Japan, 2013-2015.
 *
 */
/*
 * Copyright (C) 1998-2003
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_sha1.h>
#include <aicrypto/ok_pkcs.h>
#include <aicrypto/ok_pkcs11.h>

/*-------------------------------------------
	get & clean session
-------------------------------------------*/
P11Session *P11_open_session(PKCS11 *p11, CK_SLOT_ID id, unsigned long mode, int mofn){
	P11Session *p11s = NULL;
	CK_RV rv = -1;
	
	if((p11s=(P11Session*)malloc(sizeof(P11Session)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_PKCS11,ERR_PT_P11SESS,NULL);
		return NULL;
	}
	memset(p11s,0,sizeof(P11Session));

	if((rv=p11->pFunc->C_OpenSession(id, mode, 0, 0, &p11s->sess)) != CKR_OK){
		OK_set_error(ERR_ST_P11_OPENSS,ERR_LC_PKCS11,ERR_PT_P11SESS,&rv);
		goto error;
	}
	p11s->p11     = p11;
	p11s->slot_id = id;
	p11s->ss_mode = mode;
	p11s->mofn    = mofn;

	return p11s;
error:
	P11_close_session(p11s);
	return NULL;
}

void P11_close_session(P11Session *p11s){
	if(p11s==NULL) return;
	if(p11s->login) P11_logout(p11s);
	if(p11s->sess)  p11s->p11->pFunc->C_CloseSession(p11s->sess);
	memset(p11s,0,sizeof(P11Session));
	free(p11s);
}

/*-------------------------------------------
	login & logout session
-------------------------------------------*/
int P11_login(P11Session *p11s, int uid, char *pwd){
	CK_SESSION_INFO info;
	CK_RV rv = -1;

	p11s->login = 0;
	if((rv=p11s->p11->pFunc->C_GetSessionInfo(p11s->sess,&info)) != CKR_OK){
		OK_set_error(ERR_ST_P11_GETSINFO,ERR_LC_PKCS11,ERR_PT_P11SESS+1,&rv);
		goto error;
	}
	if((rv=p11s->p11->pFunc->C_Login(p11s->sess,uid,(unsigned char*)pwd,strlen(pwd))) != CKR_OK){
		OK_set_error(ERR_ST_P11_LOGIN,ERR_LC_PKCS11,ERR_PT_P11SESS+1,&rv);
		goto error;
	}
	/* if first time login session, "M of N" is required */
	if((info.state==CKS_RO_PUBLIC_SESSION)||(info.state==CKS_RW_PUBLIC_SESSION)){
		/* if possible */
		if(P11_mofn_activate(p11s)) goto error;
	}
	p11s->login = 1;
	p11s->uid   = uid;

	return 0;
error:
	return -1;
}

void P11_logout(P11Session *p11s){
	if((p11s==NULL)||(p11s->p11==NULL)) return;
	if(p11s->login) p11s->p11->pFunc->C_Logout(p11s->sess);
	p11s->login = 0;
}

int P11_mofn_activate(P11Session *p11s){
	CK_RV rv = -1;

	if(p11s->mofn && p11s->p11->pFunc->CA_ActivateMofN){
		if((rv=p11s->p11->pFunc->CA_ActivateMofN(p11s->sess,NULL,0)) != CKR_OK){
			OK_set_error(ERR_ST_P11_ACTMOFN,ERR_LC_PKCS11,ERR_PT_P11SESS+1,&rv);
			return -1;
		}
	}
	return 0;
}

/*-------------------------------------------
	get certificate from Token
-------------------------------------------*/
Cert* P11_get_cert(P11Session *p11s, char *label)
{
	CK_OBJECT_CLASS ckClass = CKO_CERTIFICATE;
	CK_CERTIFICATE_TYPE certType = CKC_X_509;
	CK_BBOOL bTrue = TRUE;
	CK_ULONG ulObjectCount;
	CK_OBJECT_HANDLE hCert[4];
	CK_BYTE_PTR pDer = NULL;
	CK_RV rv = -1;
	Cert *ret = NULL;
	int llen = strlen(label);
		
	CK_ATTRIBUTE certTmpl[] = {
		{CKA_CLASS, &ckClass, sizeof(ckClass)},
		{CKA_CERTIFICATE_TYPE, &certType, sizeof(certType)},
		{CKA_TOKEN, &bTrue, sizeof(bTrue)},
		{CKA_LABEL, label, llen}
	};

	CK_ATTRIBUTE getCertTmpl[] = {
		{CKA_VALUE, NULL_PTR, 0}
	};

	if((rv = p11s->p11->pFunc->C_FindObjectsInit(p11s->sess, certTmpl, 4)) != CKR_OK){
		OK_set_error(ERR_ST_P11_FOBJINIT,ERR_LC_PKCS11,ERR_PT_P11SESS+2,&rv);
		goto error;
	}

	rv = p11s->p11->pFunc->C_FindObjects(p11s->sess, hCert, 1, &ulObjectCount);
	if (rv != CKR_OK || ulObjectCount == 0){
		OK_set_error(ERR_ST_P11_FOBJCT,ERR_LC_PKCS11,ERR_PT_P11SESS+2,&rv);
		goto error;
	}
	if((rv = p11s->p11->pFunc->C_FindObjectsFinal(p11s->sess)) != CKR_OK){
		OK_set_error(ERR_ST_P11_FOBJFIN,ERR_LC_PKCS11,ERR_PT_P11SESS+2,&rv);
		goto error;
	}

	/* get public key modulus */
	if((rv = p11s->p11->pFunc->C_GetAttributeValue(p11s->sess,hCert[0],getCertTmpl,1)) != CKR_OK){
		OK_set_error(ERR_ST_P11_GETATTR,ERR_LC_PKCS11,ERR_PT_P11SESS+2,&rv);
		goto error;
	}
	if((pDer=(CK_BYTE_PTR)malloc(getCertTmpl[0].ulValueLen + 2))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_PKCS11,ERR_PT_P11SESS+2,NULL);
		goto error;
	}
	getCertTmpl[0].pValue = pDer;
	if((rv = p11s->p11->pFunc->C_GetAttributeValue(p11s->sess,hCert[0],getCertTmpl,1)) != CKR_OK){
		OK_set_error(ERR_ST_P11_GETATTR,ERR_LC_PKCS11,ERR_PT_P11SESS+2,&rv);
		goto error;
	}
	if((ret=ASN1_read_cert(pDer))==NULL) goto error;

	return ret;
error:
	if(pDer) free(pDer);
	return NULL;
}

/*-------------------------------------------
	get public Key from Token
-------------------------------------------*/
Key* P11_get_rsapub(P11Session *p11s, char *label){
	CK_OBJECT_CLASS ckClass = CKO_PUBLIC_KEY;
	CK_KEY_TYPE keyType = CKK_RSA;
	CK_BBOOL bTrue = TRUE;
	/* CK_BBOOL bFalse = FALSE; *//* XXX:currently unused */
	CK_OBJECT_HANDLE hKey[4];
	CK_ULONG ulObjectCount;
	/* XXX:currently unused
	CK_MECHANISM mechanism = { CKM_RSA_X_509, NULL_PTR, 0 };
	 */
	CK_BYTE_PTR pModulus=NULL,pExpo=NULL;
	CK_RV rv = -1;
	Pubkey_RSA *pub = NULL;
	int llen = strlen(label), ok = -1;

	CK_ATTRIBUTE keyTmpl[] = {
		{CKA_CLASS, &ckClass, sizeof(ckClass)},
		{CKA_KEY_TYPE, &keyType, sizeof(keyType)},
		{CKA_TOKEN, &bTrue, sizeof(bTrue)},
		{CKA_LABEL, label, llen}
	};

	CK_ATTRIBUTE getPubTmpl[] = {
		{CKA_MODULUS, NULL_PTR, 0},
		{CKA_PUBLIC_EXPONENT, NULL_PTR,	0},
	};

	/* get public key object handle */
	if((rv = p11s->p11->pFunc->C_FindObjectsInit(p11s->sess, keyTmpl, 4)) != CKR_OK){
		OK_set_error(ERR_ST_P11_FOBJINIT,ERR_LC_PKCS11,ERR_PT_P11SESS+4,&rv);
		goto done;
	}

	rv = p11s->p11->pFunc->C_FindObjects(p11s->sess, hKey, 1, &ulObjectCount);
	if (rv != CKR_OK || ulObjectCount == 0){
		OK_set_error(ERR_ST_P11_FOBJCT,ERR_LC_PKCS11,ERR_PT_P11SESS+4,&rv);
		goto done;
	}
	if((rv = p11s->p11->pFunc->C_FindObjectsFinal(p11s->sess)) != CKR_OK){
		OK_set_error(ERR_ST_P11_FOBJFIN,ERR_LC_PKCS11,ERR_PT_P11SESS+4,&rv);
		goto done;
	}

	/* get public key modulus */
	if((rv = p11s->p11->pFunc->C_GetAttributeValue(p11s->sess,hKey[0],getPubTmpl,2)) != CKR_OK){
		OK_set_error(ERR_ST_P11_GETATTR,ERR_LC_PKCS11,ERR_PT_P11SESS+4,&rv);
		goto done;
	}
	if((pModulus=(CK_BYTE_PTR)malloc(getPubTmpl[0].ulValueLen))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_PKCS11,ERR_PT_P11SESS+4,NULL);
		goto done;
	}
	if((pExpo=(CK_BYTE_PTR)malloc(getPubTmpl[1].ulValueLen))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_PKCS11,ERR_PT_P11SESS+4,NULL);
		goto done;
	}
	getPubTmpl[0].pValue = pModulus;
	getPubTmpl[1].pValue = pExpo;

	if((rv = p11s->p11->pFunc->C_GetAttributeValue(p11s->sess,hKey[0],getPubTmpl,2))!=CKR_OK){
		OK_set_error(ERR_ST_P11_GETATTR,ERR_LC_PKCS11,ERR_PT_P11SESS+4,&rv);
		goto done;
	}

	/* get public key */
	if((pub=RSApubkey_new())==NULL) goto done;
	if(LN_set_num_c(pub->n,getPubTmpl[0].ulValueLen,pModulus)) goto done;
	if(LN_set_num_c(pub->e,getPubTmpl[1].ulValueLen,pExpo)) goto done;
	pub->size = pub->n->size * 4;

	ok = 0;
done:
	if(pExpo) free(pExpo);
	if(pModulus) free(pModulus);
	if(ok){ Key_free((Key*)pub); pub=NULL; }
	return (Key*)pub;
}

/*-------------------------------------------
	put certificate to Token
-------------------------------------------*/
int P11_put_cert(P11Session *p11s, Cert *cert, char *label, unsigned char *subject, unsigned char *id)
{
	CK_OBJECT_CLASS ckClass = CKO_CERTIFICATE;
	CK_CERTIFICATE_TYPE certType = CKC_X_509;
	CK_BBOOL bTrue = TRUE;
	CK_RV rv = -1;
	int slen,clen,llen,snlen,ilen;
	unsigned char *sn,*issuer;

	if((sn=P11_util_serialnum(cert))==NULL) goto error;
	if((issuer=P11_util_issuer(cert))==NULL) goto error;
	if(subject==NULL) 
		if((subject=P11_util_subject(cert))==NULL) goto error;

#if 0 /*
{FILE *fp; int i; if(fp=fopen("d:\\p11_cert.txt","a+")){fprintf(fp,"ok2\n");
fprintf(fp,"0x%.8x : ",subject); for(i=0;i<8;i++) fprintf(fp,"%.2x",subject[i]); fprintf(fp,"\n");
fprintf(fp,"0x%.8x : ",cert->der); for(i=0;i<8;i++) fprintf(fp,"%.2x",cert->der[i]); fprintf(fp,"\n");
fprintf(fp,"0x%.8x : ",issuer); for(i=0;i<8;i++) fprintf(fp,"%.2x",issuer[i]); fprintf(fp,"\n");
fprintf(fp,"0x%.8x : ",sn); for(i=0;i<8;i++) fprintf(fp,"%.2x",sn[i]); fprintf(fp,"\n");
fclose(fp);}} */
#endif

	if(ASN1_skip_(subject,&slen)==NULL) goto error;
	if(ASN1_skip_(cert->der,&clen)==NULL) goto error;
	if(ASN1_skip_(issuer,&ilen)==NULL) goto error;
	if(ASN1_skip_(sn,&snlen)==NULL) goto error;
	llen = strlen(label);

	{
		CK_ATTRIBUTE certTmpl[] = {
			{CKA_ID, id, 20},
			{CKA_LABEL, label, llen},
			{CKA_CLASS, &ckClass, sizeof(ckClass)},
			{CKA_TOKEN, &bTrue, sizeof(bTrue)},
			{CKA_CERTIFICATE_TYPE, &certType, sizeof(certType)},
			//{CKA_TRUSTED, &bTrue, sizeof(bTrue)},
			{CKA_SUBJECT, subject, slen},
			{CKA_ISSUER, issuer, ilen},
			{CKA_SERIAL_NUMBER, sn, snlen},
			{CKA_VALUE, cert->der, clen}
		};

		CK_OBJECT_HANDLE h;
		if((rv=p11s->p11->pFunc->C_CreateObject(p11s->sess,certTmpl,9,&h)) != CKR_OK){
			OK_set_error(ERR_ST_P11_CREOBJ,ERR_LC_PKCS11,ERR_PT_P11SESS+5,&rv);
			goto error;
		}
	}

	return 0;
error:
	return -1;
}

/*-------------------------------------------
	put private key to Token
-------------------------------------------*/
int P11_put_rsakey(P11Session *p11s, Key *key, char *label, unsigned char *subject, unsigned char *id)
{
	CK_OBJECT_CLASS priClass = CKO_PRIVATE_KEY;
	CK_OBJECT_CLASS pubClass = CKO_PUBLIC_KEY;
	CK_KEY_TYPE keyType = CKK_RSA;
	CK_BBOOL bTrue = TRUE;
	CK_BBOOL bFalse = FALSE;
	CK_RV rv = -1;
	int slen,llen;

	if(ASN1_skip_(subject,&slen)==NULL) goto error;
	llen = strlen(label);

	switch(key->key_type){
	case KEY_RSA_PRV:
		{
		CK_BYTE modulus[256];
		CK_BYTE publicExponent[256];
		CK_BYTE privateExponent[256];
		CK_BYTE prime1[256];
		CK_BYTE prime2[256];
		CK_BYTE exponent1[256];
		CK_BYTE exponent2[256];
		CK_BYTE coefficient[256] ;
		int nlen,elen,dlen,p1,p2,e1,e2,cof;
		Prvkey_RSA *rsa = (Prvkey_RSA*)key;

		keyType = CKK_RSA;
		if((nlen=LN_now_byte(rsa->n))<0) goto error;
		if((elen=LN_now_byte(rsa->e))<0) goto error;
		if((dlen=LN_now_byte(rsa->d))<0) goto error;
		if((p1=LN_now_byte(rsa->p))<0) goto error;
		if((p2=LN_now_byte(rsa->q))<0) goto error;
		if((e1=LN_now_byte(rsa->e1))<0) goto error;
		if((e2=LN_now_byte(rsa->e2))<0) goto error;
		if((cof=LN_now_byte(rsa->cof))<0) goto error;

		if(LN_get_num_c(rsa->n,nlen,modulus)) goto error;
		if(LN_get_num_c(rsa->e,elen,publicExponent)) goto error;
		if(LN_get_num_c(rsa->d,dlen,privateExponent)) goto error;
		if(LN_get_num_c(rsa->p,p1,prime1)) goto error;
		if(LN_get_num_c(rsa->q,p2,prime2)) goto error;
		if(LN_get_num_c(rsa->e1,e1,exponent1)) goto error;
		if(LN_get_num_c(rsa->e2,e2,exponent2)) goto error;
		if(LN_get_num_c(rsa->cof,cof,coefficient)) goto error;

		{
			CK_ATTRIBUTE priTemplate[] = {
				{CKA_CLASS, &priClass, sizeof(priClass)},
				{CKA_KEY_TYPE, &keyType, sizeof(keyType)},
				{CKA_TOKEN, &bTrue, sizeof(bTrue)},
				{CKA_LABEL, label, llen},
				{CKA_SUBJECT, subject, slen},
				{CKA_ID, id, 20},
				{CKA_SENSITIVE, &bTrue, sizeof(bTrue)},
				{CKA_PRIVATE, &bTrue, sizeof(bTrue)},
				{CKA_UNWRAP, &bTrue, sizeof(bTrue)},
				{CKA_DECRYPT, &bTrue, sizeof(bTrue)},
				{CKA_SIGN, &bTrue, sizeof(bTrue)},
				{CKA_SIGN_RECOVER, &bTrue, sizeof(bTrue)},
				{CKA_MODULUS, modulus, nlen},
				{CKA_PUBLIC_EXPONENT, publicExponent, elen},
				{CKA_PRIVATE_EXPONENT, privateExponent, dlen},
				{CKA_PRIME_1, prime1, p1},
				{CKA_PRIME_2, prime2, p2},
				{CKA_EXPONENT_1, exponent1, e1},
				{CKA_EXPONENT_2, exponent2, e2},
				{CKA_COEFFICIENT, coefficient, cof}
			};

			CK_OBJECT_HANDLE h;
			if((rv=p11s->p11->pFunc->C_CreateObject(p11s->sess,priTemplate,20,&h)) != CKR_OK){
				OK_set_error(ERR_ST_P11_CREOBJ,ERR_LC_PKCS11,ERR_PT_P11SESS+6,&rv);
				goto error;
			}
		}
		{
			CK_ATTRIBUTE pubTemplate[] = { 
				{CKA_CLASS, &pubClass, sizeof(pubClass)},
				{CKA_KEY_TYPE, &keyType, sizeof(keyType)},
				{CKA_TOKEN, &bTrue, sizeof(bTrue)},
				{CKA_LABEL, label, llen},
				{CKA_SUBJECT, subject, slen},
				{CKA_ID, id, 20},
				{CKA_PRIVATE, &bFalse, sizeof(bFalse)},
				{CKA_MODULUS, modulus, nlen},
				{CKA_PUBLIC_EXPONENT, publicExponent, elen}
			};
			CK_OBJECT_HANDLE h;
			if((rv=p11s->p11->pFunc->C_CreateObject(p11s->sess,pubTemplate,9,&h)) != CKR_OK){
				OK_set_error(ERR_ST_P11_CREOBJ,ERR_LC_PKCS11,ERR_PT_P11SESS+6,&rv);
				goto error;
			}
		}
		}
		break;
	case KEY_DSA_PRV:
	case KEY_ECDSA_PRV:
	default:
		OK_set_error(ERR_ST_UNSUPPORTED_ALGO,ERR_LC_PKCS11,ERR_PT_P11SESS+6,&rv);
		goto error;
	}

	return 0;
error:
	return -1;
}
