/* p11_key.c */
/*
 * Modified by National Institute of Informatics in Japan, 2013-2017.
 *
 */
/*
 * Copyright (C) 1998-2003
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_sha1.h>
#include <aicrypto/ok_tool.h>
#include <aicrypto/ok_pkcs.h>
#include <aicrypto/ok_pkcs11.h>

/*-------------------------------------------
	allocate & free pkcs11 keys 
-------------------------------------------*/
P11Key* P11key_new(PKCS11 *p11){
	P11Key *ret=NULL;

	if((ret = (P11Key*)malloc(sizeof(P11Key)))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_PKCS11,ERR_PT_P11KEY,NULL);
		goto error;
	}
	memset(ret,0,sizeof(P11Key));

	ret->p11 = p11; /* set pointer */

	return ret;
error:
	if(ret) free(ret);
	return NULL;
}

Prvkey_P11RSA* P11RSAprvkey_new(PKCS11 *p11){
	Prvkey_P11RSA *ret = NULL;

	if((ret=P11key_new(p11))==NULL) return NULL;
	ret->key_type = KEY_P11RSA_PRV;

	return ret;
}

Pubkey_P11RSA* P11RSApubkey_new(PKCS11 *p11){
	Pubkey_P11RSA *ret = NULL;

	if((ret=P11key_new(p11))==NULL) return NULL;
	ret->key_type = KEY_P11RSA_PUB;

	return ret;
}

void P11key_free(P11Key *key){
	if(key==NULL) return;
	if(key->p11s && key->p11s->login) P11_logout(key->p11s);
	if(key->p11s) P11_close_session(key->p11s);
	/* if(key->p11) P11_free(key->p11); */
	if(key->lock) OK_release_lock(key->lock);
	memset(key->pwd,0,32);
	free(key);
}

/*-------------------------------------------
	allocate & free pkcs11 keys 
-------------------------------------------*/
P11Key *P11key_dup(P11Key *org){
	P11Key *ret = NULL;

	if(org==NULL) goto error;
	if((ret=P11key_new(org->p11))==NULL) goto error;
	ret->key_type = org->key_type;
	ret->size     = org->size;
	ret->mode     = org->mode;
	ret->id       = org->id;
	ret->uid      = org->uid;
	memcpy(ret->pwd,org->pwd,32);
	memcpy(ret->label,org->label,32);

	return ret;
error:
	P11key_free(ret);
	return NULL;
}

/*-------------------------------------------
	generate key pair
-------------------------------------------*/
int P11_rsa_generate(P11Session *p11s, char *label, int bits, Pubkey_RSA **pubkey){
	return P11_rsa_generate_(p11s,label,bits,pubkey,0);
}

int P11_rsa_generate_(P11Session *p11s, char *label, int bits, Pubkey_RSA **pubkey, int tmplmode)
{
	CK_OBJECT_CLASS priClass = CKO_PRIVATE_KEY;
	CK_OBJECT_CLASS pubClass = CKO_PUBLIC_KEY;
	CK_KEY_TYPE     keyType = CKK_RSA;
  
	CK_ULONG ulModulusBits = (CK_ULONG)bits; 
	CK_BYTE pExponent[] = { 0x01, 0x00, 0x01 };

	CK_BBOOL bFalse = 0;
	CK_BBOOL bTrue = 1;
	CK_RV rv = -1;
	int ok=-1, llen=strlen(label);
	unsigned char keyid[32];

	CK_ATTRIBUTE pubTemplate[] = { 
		{CKA_CLASS,			 &pubClass,			sizeof(pubClass)},
		{CKA_LABEL,			 label,				llen},
		{CKA_KEY_TYPE,		 &keyType,			sizeof(keyType)},
		{CKA_TOKEN,			 &bTrue,			sizeof(bTrue)},
		{CKA_PRIVATE,        &bFalse,			sizeof(bFalse)},
		{CKA_MODULUS_BITS,	 &ulModulusBits,	sizeof(ulModulusBits)},
		{CKA_PUBLIC_EXPONENT,&pExponent,		sizeof(pExponent)},
		{CKA_VERIFY,		 &bTrue,			sizeof(bTrue)},
		{CKA_ENCRYPT,		 &bTrue,			sizeof(bTrue)},
	};
	CK_ATTRIBUTE priTemplate[] = {
		{CKA_CLASS,			&priClass,	sizeof(priClass)},
		{CKA_LABEL,			label,		llen},
		{CKA_KEY_TYPE,		&keyType,	sizeof(keyType)},
		{CKA_TOKEN,			&bTrue,		sizeof(bTrue)},
		{CKA_PRIVATE,		&bTrue,		sizeof(bTrue)},
		{CKA_SENSITIVE,		&bTrue,		sizeof(bTrue)},
		{CKA_EXTRACTABLE,	&bTrue,		sizeof(bTrue)},
		{CKA_SIGN,			&bTrue,		sizeof(bTrue)},
		{CKA_DECRYPT,		&bTrue,		sizeof(bTrue)},
	};
	CK_ATTRIBUTE getPubTmpl[] = {
		{CKA_MODULUS, NULL_PTR, 0},
		{CKA_PUBLIC_EXPONENT, NULL_PTR,	0},
	};
	CK_ATTRIBUTE setKeyID[] = {
		{CKA_ID, keyid, 20},
	};
	CK_BYTE_PTR pModulus=NULL,pExpo=NULL;
	CK_OBJECT_HANDLE public_key = NULL_PTR;
	CK_OBJECT_HANDLE private_key = NULL_PTR;
	CK_MECHANISM mechanism;

	int pubtmplsz = (tmplmode)?(7):(9);
	int pritmplsz = (tmplmode)?(5):(9);

	*pubkey = NULL;
	memset(&mechanism, 0, sizeof (mechanism));
	mechanism.mechanism = CKM_RSA_PKCS_KEY_PAIR_GEN;

	/* generate key pair */
	if((rv = p11s->p11->pFunc->C_GenerateKeyPair(p11s->sess,&mechanism, pubTemplate, pubtmplsz,
											priTemplate, pritmplsz, &public_key,
											&private_key))!=CKR_OK){
		OK_set_error(ERR_ST_P11_GENKEY,ERR_LC_PKCS11,ERR_PT_P11KEY+2,&rv);
		goto done;
	}

	/* get public key modulus */
	if((rv = p11s->p11->pFunc->C_GetAttributeValue(p11s->sess,public_key,getPubTmpl,2))!=CKR_OK){
		OK_set_error(ERR_ST_P11_GETATTR,ERR_LC_PKCS11,ERR_PT_P11KEY+2,&rv);
		goto done;
	}
	if((pModulus=(CK_BYTE_PTR)malloc(getPubTmpl[0].ulValueLen))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_PKCS11,ERR_PT_P11KEY+2,NULL);
		goto done;
	}
	if((pExpo=(CK_BYTE_PTR)malloc(getPubTmpl[1].ulValueLen))==NULL){
		OK_set_error(ERR_ST_MEMALLOC,ERR_LC_PKCS11,ERR_PT_P11KEY+2,NULL);
		goto done;
	}
	getPubTmpl[0].pValue = pModulus;
	getPubTmpl[1].pValue = pExpo;

	if((rv = p11s->p11->pFunc->C_GetAttributeValue(p11s->sess,public_key,getPubTmpl,2))!=CKR_OK){
		OK_set_error(ERR_ST_P11_GETATTR,ERR_LC_PKCS11,ERR_PT_P11KEY+2,&rv);
		goto done;
	}

	/* get public key */
	{
		/* LNm *n=NULL,*e=NULL; *//* XXX:unused */

		if((*pubkey=RSApubkey_new())==NULL) goto done;
		if(LN_set_num_c((*pubkey)->n,getPubTmpl[0].ulValueLen,pModulus)) goto done;
		if(LN_set_num_c((*pubkey)->e,getPubTmpl[1].ulValueLen,pExpo)) goto done;
		(*pubkey)->size = bits>>3;
	}

	/* set key id */
	if(P11_util_keyid((Key*)*pubkey,keyid)) goto done;
	
	if((rv = p11s->p11->pFunc->C_SetAttributeValue(p11s->sess,public_key,setKeyID,1))!=CKR_OK){
		OK_set_error(ERR_ST_P11_SETATTR,ERR_LC_PKCS11,ERR_PT_P11KEY+2,&rv);
		goto done;
	}
	if((rv = p11s->p11->pFunc->C_SetAttributeValue(p11s->sess,private_key,setKeyID,1))!=CKR_OK){
		OK_set_error(ERR_ST_P11_SETATTR,ERR_LC_PKCS11,ERR_PT_P11KEY+2,&rv);
		goto done;
	}

	ok = 0;
done:
	if(pExpo) free(pExpo);
	if(pModulus) free(pModulus);
	if(ok && *pubkey){ Key_free((Key*)*pubkey); *pubkey=NULL; }
	return ok;
}

/*-------------------------------------------
	encryption & decryption
-------------------------------------------*/
/**
 * Encrypt a data with a private key in a PKCS#11 device.
 *
 * @param[in] len  The length of the input buffer pointed by /from/.
 * @param[in] from The character pointer to the input.
 * @param[out] to  The character pointer to
 * @param[in] key
 * @return
 */
int P11RSAprv_doCrypt(int len, unsigned char *from, unsigned char *to,
		      Prvkey_P11RSA *key)
{
	P11Session *p11s = key->p11s;
	AILock lock = NULL;
	CK_OBJECT_CLASS ckClass = CKO_PRIVATE_KEY;
	CK_KEY_TYPE keyType = CKK_RSA;
	CK_BBOOL bTrue = TRUE;
	/* CK_BBOOL bFalse = FALSE; *//* XXX:currently unused */
	CK_OBJECT_HANDLE hKey[16];
	CK_BYTE data[RSA_KEY_BITLENGTH_MAX / 8];
	CK_BYTE out[RSA_KEY_BITLENGTH_MAX / 8];
	CK_ULONG outLen  = 0;
	CK_ULONG ulObjectCount;
	/* NRG_CKM_SIGNING defined in ok_pkcs11.h */
	CK_MECHANISM mechanism = { NRG_CKM_SIGNING, NULL_PTR, 0 };
	CK_RV rv = -1;
	char pwd[PWD_BUFLEN];
	int llen, ok = -1;
	int cache = 1;

	memset(hKey,0,sizeof(CK_OBJECT_HANDLE)*4);

	/***** start private key lock *****/
	if(key->lock){ if(OK_lock(key->lock,30000)) goto done; lock=key->lock; }

	/* check session */
	if(p11s == NULL){
		if((key->p11s=p11s=P11_open_session(key->p11,key->id,key->mode,0))==NULL)
			goto done;
		cache = 0; /* not cache mode */
	}

	if(p11s->login == 0){
		if(key->pwd[0]) memcpy(pwd,key->pwd,PWD_BUFLEN);
		else OK_get_passwd("Token Login PIN : ",pwd,0);

		if(P11_login(p11s,key->uid,key->pwd)) goto done;
	}

	/* set data */
	/*
	 * XXX: should modify the following condition appropriately.
	 * the max maybe depends on the device that we can access.
	 * key pair generation should be also considered...
	 */
	if((len > key->size)||(key->size > RSA_KEY_BITLENGTH_MAX / 8)){
		OK_set_error(ERR_ST_BADPARAM,ERR_LC_PKCS11,ERR_PT_P11KEY+3,NULL);
		return -1;
	}

	memset(data, 0, RSA_KEY_BITLENGTH_MAX / 8);
	switch(mechanism.mechanism) {
	case CKM_RSA_X_509:
		/* it is necessary that the padding has been completed
		 * until this time. the requried padding has already been
		 * done in tool/signature.c:P1_sign_digest().
		 * the data[] must be right-justified.
		 */
		memcpy(&data[key->size - len], from, len);
		len = key->size;
#ifdef PKCS11_DEBUG
		fprintf(stderr, "p11_key.c:CKM_RSA_X_509\n");
		fprintf(stderr, "ulEncyptedDataLen = %d\n", len);
		fflush(stderr);
#endif
		break;
	case CKM_RSA_PKCS:
		/* in this mechanism, we have only to compute the digest.
		 * the digest has already been computed
		 * in tool/signature.c:P1_do_sign().
		 */
		memcpy(data, from, len);
#ifdef PKCS11_DEBUG
		fprintf(stderr, "p11_key.c:CKM_RSA_PKCS\n");
		fprintf(stderr, "ulDataLen = %d\n", len);
		fflush(stderr);
#endif
		break;
	}

	/* get private key object handle */
	llen = strlen(key->label);
	{
		CK_ATTRIBUTE keyTmpl[] = {
			{CKA_CLASS, &ckClass, sizeof(ckClass)},
			{CKA_KEY_TYPE, &keyType, sizeof(keyType)},
			{CKA_TOKEN, &bTrue, sizeof(bTrue)},
			{CKA_LABEL, key->label, llen}
		};

		if((rv = p11s->p11->pFunc->C_FindObjectsInit(p11s->sess, keyTmpl, 4)) != CKR_OK){
			OK_set_error(ERR_ST_P11_FOBJINIT,ERR_LC_PKCS11,ERR_PT_P11KEY+3,&rv);
			goto done;
		}

		rv = p11s->p11->pFunc->C_FindObjects(p11s->sess, hKey, 16, &ulObjectCount);
		if (rv != CKR_OK || (ulObjectCount == 0) || (ulObjectCount > 16)){
			OK_set_error(ERR_ST_P11_FOBJCT,ERR_LC_PKCS11,ERR_PT_P11KEY+3,&rv);
			goto done;
		}
		if((rv = p11s->p11->pFunc->C_FindObjectsFinal(p11s->sess))!= CKR_OK){
			OK_set_error(ERR_ST_P11_FOBJFIN,ERR_LC_PKCS11,ERR_PT_P11KEY+3,&rv);
			goto done;
		}
	}

	/* do private key encryption */
	switch(mechanism.mechanism) {
	case CKM_RSA_X_509:
		/* initialize */
		rv = p11s->p11->pFunc->C_DecryptInit(p11s->sess, &mechanism,
						     hKey[ulObjectCount - 1]);
		if (rv != CKR_OK) {
			OK_set_error(ERR_ST_P11_DECINIT,ERR_LC_PKCS11,ERR_PT_P11KEY+3,&rv);
			goto done;
		}
		/* get signature length */
		rv = p11s->p11->pFunc->C_Decrypt(p11s->sess, data, len,
						 NULL, &outLen);
		if (rv != CKR_OK) {
			OK_set_error(ERR_ST_P11_DECRYPT,ERR_LC_PKCS11,ERR_PT_P11KEY+3,&rv);
			goto done;
		}
		/* get signature */
		rv = p11s->p11->pFunc->C_Decrypt(p11s->sess, data, len,
						 out, &outLen);
		if (rv != CKR_OK) {
			OK_set_error(ERR_ST_P11_DECRYPT,ERR_LC_PKCS11,ERR_PT_P11KEY+3,&rv);
			goto done;
		}
		break;
	case CKM_RSA_PKCS:
		rv = p11s->p11->pFunc->C_SignInit(p11s->sess, &mechanism,
						  hKey[ulObjectCount - 1]);
		if (rv != CKR_OK) {
			OK_set_error(ERR_ST_P11_SIGINIT,ERR_LC_PKCS11,ERR_PT_P11KEY+3,&rv);
			goto done;
		}

		rv = p11s->p11->pFunc->C_Sign(p11s->sess, data, len,
					      NULL, &outLen);
		if (rv != CKR_OK) {
			OK_set_error(ERR_ST_P11_SIGN,ERR_LC_PKCS11,ERR_PT_P11KEY+3,&rv);
			goto done;
		}

		rv = p11s->p11->pFunc->C_Sign(p11s->sess, data, len,
					      out, &outLen);
		if (rv != CKR_OK) {
			OK_set_error(ERR_ST_P11_SIGN,ERR_LC_PKCS11,ERR_PT_P11KEY+3,&rv);
			goto done;
		}
		break;
	}
	
	if(lock){ OK_unlock(lock); lock=NULL; }
	/***** end private key lock *****/
	memcpy(to,out,key->size);

	ok = 0;
done:
	if(lock){ OK_unlock(lock); lock=NULL; }
	memset(pwd,0,32);
	if(!cache){ /* not cache mode */
		P11_logout(key->p11s);
		P11_close_session(key->p11s); key->p11s = NULL;
	}
	return ok;
}

int P11RSApub_doCrypt(int len, unsigned char *from,
					  unsigned char *to, Pubkey_P11RSA *key)
{
	P11Session *p11s = key->p11s;
	AILock lock = NULL;
	CK_OBJECT_CLASS ckClass = CKO_PUBLIC_KEY;
	CK_KEY_TYPE keyType = CKK_RSA;
	CK_BBOOL bTrue = TRUE;
	/* CK_BBOOL bFalse = FALSE; *//* XXX:currently unused */
	CK_OBJECT_HANDLE hKey[4];
	CK_BYTE data[256];
	CK_BYTE out[256];
	CK_ULONG outLen  = 256;
	CK_ULONG ulObjectCount;
	CK_MECHANISM mechanism = { CKM_RSA_X_509, NULL_PTR, 0 };
	CK_RV rv = -1;
	char pwd[PWD_BUFLEN];
	int llen, ok = -1;
	int cache = 1;

	/***** start public key lock *****/
	if(key->lock){ if(OK_lock(key->lock,30000)) goto done; lock=key->lock; }

	/* check session */
	if(p11s == NULL){
		if((key->p11s=p11s=P11_open_session(key->p11,key->id,key->mode,0))==NULL)
			goto done;
		cache = 0; /* not cache mode */
	}
	if(p11s->login == 0){
		if(key->pwd[0]) memcpy(pwd,key->pwd,PWD_BUFLEN);
		else OK_get_passwd("Token Login PIN : ",pwd,0);

		if(P11_login(p11s,key->uid,key->pwd)) goto done;
	}

	/* set data */
	memset(data,0,256);
	if((len > key->size)||(key->size > 256)){
		OK_set_error(ERR_ST_BADPARAM,ERR_LC_PKCS11,ERR_PT_P11KEY+4,NULL);
		return -1;
	}else{
		memcpy(&data[key->size - len],from,len);
		len = key->size;
	}

	/* get public key object handle */
	llen = strlen(key->label);
	{
		CK_ATTRIBUTE keyTmpl[] = {
			{CKA_CLASS, &ckClass, sizeof(ckClass)},
			{CKA_KEY_TYPE, &keyType, sizeof(keyType)},
			{CKA_TOKEN, &bTrue, sizeof(bTrue)},
			{CKA_LABEL, key->label, llen}
		};

		if((rv = p11s->p11->pFunc->C_FindObjectsInit(p11s->sess, keyTmpl, 4)) != CKR_OK){
			OK_set_error(ERR_ST_P11_FOBJINIT,ERR_LC_PKCS11,ERR_PT_P11KEY+4,&rv);
			goto done;
		}

		rv = p11s->p11->pFunc->C_FindObjects(p11s->sess, hKey, 1, &ulObjectCount);
		if (rv != CKR_OK || ulObjectCount == 0){
			OK_set_error(ERR_ST_P11_FOBJCT,ERR_LC_PKCS11,ERR_PT_P11KEY+4,&rv);
			goto done;
		}
		if((rv = p11s->p11->pFunc->C_FindObjectsFinal(p11s->sess))!= CKR_OK){
			OK_set_error(ERR_ST_P11_FOBJFIN,ERR_LC_PKCS11,ERR_PT_P11KEY+4,&rv);
			goto done;
		}
	}

	/* do public key encryption */
	if((rv = p11s->p11->pFunc->C_EncryptInit(p11s->sess, &mechanism, hKey[0])) != CKR_OK){
		OK_set_error(ERR_ST_P11_ENCINIT,ERR_LC_PKCS11,ERR_PT_P11KEY+4,&rv);
		goto done;
	}
	/* get encrypt length */
	if((rv = p11s->p11->pFunc->C_Encrypt(p11s->sess, data, len, NULL, &outLen)) != CKR_OK){
		OK_set_error(ERR_ST_P11_ENCRYPT,ERR_LC_PKCS11,ERR_PT_P11KEY+4,&rv);
		goto done;
	}
	/* get encrypted */
	if((rv = p11s->p11->pFunc->C_Encrypt(p11s->sess, data, len, out, &outLen)) != CKR_OK){
		OK_set_error(ERR_ST_P11_ENCRYPT,ERR_LC_PKCS11,ERR_PT_P11KEY+4,&rv);
		goto done;
	}
	if(lock){ OK_unlock(lock); lock=NULL; }
	/***** end public key lock *****/
	memcpy(to,out,key->size);

	ok = 0;
done:
	if(lock){ OK_unlock(lock); lock=NULL; }
	memset(pwd,0,32);
	if(!cache){ /* not cache mode */
		P11_logout(key->p11s);
		P11_close_session(key->p11s); key->p11s = NULL;
	}
	return ok;
}

/*-------------------------------------------
	P11 RSA Public to DER
-------------------------------------------*/
unsigned char *P11RSApub_toDER(P11Key *key,unsigned char *buf,int *ret_len){
	P11Session *p11s = key->p11s;
	Pubkey_RSA *pub = NULL;
	unsigned char *ret = NULL;
	int cache = 1;

	/* check session */
	if(p11s == NULL){
		if((key->p11s=p11s=P11_open_session(key->p11,key->id,key->mode,0))==NULL)
			goto done;
		cache = 0; /* not session cache */
	}
	if((pub=(Pubkey_RSA*)P11_get_rsapub(key->p11s,key->label))==NULL) goto done;

	if((ret=RSApub_toDER(pub,buf,ret_len))==NULL) goto done;

done:
	Key_free((Key*)pub);
	if(!cache){ P11_close_session(key->p11s); key->p11s = NULL;}
	return ret;
}

/*-------------------------------------------
	P11 Open Token
-------------------------------------------*/
P11Key *p11_open_key_(PKCS11 *p11, P11Session *p11s, int key_type, int slot_id,
					  int mode, int uid, char *pin, char *label, int prv, int keeplogin, int mofn){
	CK_OBJECT_CLASS ckClass = CKO_PUBLIC_KEY;
	CK_KEY_TYPE keyType = CKK_RSA;
	CK_BBOOL bTrue = TRUE;
	CK_OBJECT_HANDLE hKey[4];
	CK_ULONG ulObjectCount;
	CK_RV rv = -1;
	P11Key *ret = NULL;
	int llen = strlen(label);

	CK_ATTRIBUTE keyTmpl[] = {
		{CKA_CLASS, &ckClass, sizeof(ckClass)},
		{CKA_KEY_TYPE, &keyType, sizeof(keyType)},
		{CKA_TOKEN, &bTrue, sizeof(bTrue)},
		{CKA_LABEL, label, llen}
	};

	CK_ATTRIBUTE getPubTmpl[] = {
		{CKA_MODULUS, NULL_PTR, 0},
	};

	/* allocate key memory */
	switch(key_type){
	case KEY_P11RSA_PUB:
		if((ret=(P11Key*)P11RSApubkey_new(p11))==NULL) goto error;
		break;
	case KEY_P11RSA_PRV:
		if((ret=(P11Key*)P11RSAprvkey_new(p11))==NULL) goto error;
		break;
	case KEY_P11DSA_PUB:
	case KEY_P11DSA_PRV:
	case KEY_P11ECDSA_PUB:
	case KEY_P11ECDSA_PRV:
	default:
		OK_set_error(ERR_ST_UNSUPPORTED_ALGO,ERR_LC_PKCS11,ERR_PT_P11KEY+5,NULL);
		goto error;
	}

	ret->id   = slot_id;
	ret->mode = mode;
	ret->uid  = uid;
	if(pin)   strncpy(ret->pwd,pin,32);
	if(label) strncpy(ret->label,label,32);
	
	if(p11s==NULL){
		if((ret->p11s=p11s=P11_open_session(ret->p11,slot_id,mode,mofn))==NULL) goto error;
	}else{
		ret->p11s = p11s; 
	}
	if(pin && prv && !p11s->login)
		if(P11_login(ret->p11s,uid,ret->pwd)) goto error;

	/* get public key object handle */
	if((rv = p11s->p11->pFunc->C_FindObjectsInit(p11s->sess, keyTmpl, 4)) != CKR_OK){
		OK_set_error(ERR_ST_P11_FOBJINIT,ERR_LC_PKCS11,ERR_PT_P11KEY+5,&rv);
		goto error;
	}

	rv = p11s->p11->pFunc->C_FindObjects(p11s->sess, hKey, 1, &ulObjectCount);
	if (rv != CKR_OK || ulObjectCount == 0){
		OK_set_error(ERR_ST_P11_FOBJCT,ERR_LC_PKCS11,ERR_PT_P11KEY+5,&rv);
		goto error;
	}
	if((rv = p11s->p11->pFunc->C_FindObjectsFinal(p11s->sess)) != CKR_OK){
		OK_set_error(ERR_ST_P11_FOBJFIN,ERR_LC_PKCS11,ERR_PT_P11KEY+5,&rv);
		goto error;
	}

	/* get public key modulus */
	if((rv = p11s->p11->pFunc->C_GetAttributeValue(p11s->sess,hKey[0],getPubTmpl,1)) != CKR_OK){
		OK_set_error(ERR_ST_P11_GETATTR,ERR_LC_PKCS11,ERR_PT_P11KEY+5,&rv);
		goto error;
	}
	ret->size = getPubTmpl[0].ulValueLen;

	if(pin && prv && !keeplogin)
		P11_logout(ret->p11s);
	if(!keeplogin){
		P11_close_session(ret->p11s); ret->p11s = NULL;
	}

	return ret;
error:
	P11key_free(ret);
	return NULL;
}

P11Key *P11_open_rsaprv(PKCS11 *p11, int slot_id, char *label, int mofn){
	P11Key *ret = NULL;
	char pwd[PWD_BUFLEN];

	OK_get_passwd("Token Login PIN : ",pwd,0);
	ret = p11_open_key_(p11,NULL,KEY_P11RSA_PRV,slot_id,(CKF_SERIAL_SESSION | CKF_RW_SESSION),CKU_USER,pwd,label,1,0,mofn);
	memset(pwd,0,PWD_BUFLEN);

	return ret;
}

P11Key *P11_open_rsapub(PKCS11 *p11, int slot_id, char *label, int mofn){
	P11Key *ret = NULL;
	char pwd[PWD_BUFLEN];

	OK_get_passwd("Token Login PIN : ",pwd,0);
	ret = p11_open_key_(p11,NULL,KEY_P11RSA_PUB,slot_id,(CKF_SERIAL_SESSION|CKF_RW_SESSION),CKU_USER,pwd,label,1,0,mofn);
	memset(pwd,0,PWD_BUFLEN);

	return ret;
}

/*-------------------------------------------
  generate simple symmentric key
-------------------------------------------*/
int P11_gen_symkey(P11Session *p11s, CK_OBJECT_HANDLE *hKey, CK_MECHANISM_TYPE mtype,
                   CK_KEY_TYPE type, CK_ULONG klen, char *label)
{
  CK_OBJECT_CLASS  SymKeyClass  = CKO_SECRET_KEY;
  CK_BBOOL         bTrue = 1, bFalse = 0;
  CK_MECHANISM     mech;
  CK_RV            rv = CKR_OK;

  /* set 3DES key template */
  CK_ATTRIBUTE SymKeyTemplate[] = {
    {CKA_CLASS,     &SymKeyClass, sizeof(SymKeyClass)},
    {CKA_KEY_TYPE,  &type,        sizeof(type) },
    {CKA_TOKEN,     &bTrue,       sizeof(CK_BBOOL)},
    {CKA_SENSITIVE, &bTrue,       sizeof(CK_BBOOL)},
    {CKA_PRIVATE,   &bTrue,       sizeof(CK_BBOOL)},
    {CKA_ENCRYPT,   &bTrue,       sizeof(CK_BBOOL)},
    {CKA_DECRYPT,   &bTrue,       sizeof(CK_BBOOL)},
    {CKA_SIGN,      &bFalse,      sizeof(CK_BBOOL)},
    {CKA_VERIFY,    &bFalse,      sizeof(CK_BBOOL)},
    {CKA_WRAP,      &bTrue,       sizeof(CK_BBOOL)},
    {CKA_UNWRAP,    &bTrue,       sizeof(CK_BBOOL)},
    {CKA_DERIVE,    &bTrue,       sizeof(CK_BBOOL)},
    {CKA_VALUE_LEN, &klen,        sizeof(CK_ULONG)},
    {CKA_LABEL,     label,        strlen(label) },  /* keep last for Luna CA3 */
  };

  mech.mechanism = mtype;
  mech.pParameter = 0;
  mech.ulParameterLen = 0;

  if((rv = p11s->p11->pFunc->C_GenerateKey(p11s->sess, (CK_MECHANISM_PTR)&mech,
					   SymKeyTemplate, 14, hKey)) != CKR_OK)
    {
      OK_set_error(ERR_ST_P11_GENKEY,ERR_LC_PKCS11,ERR_PT_P11KEY+6,&rv);
      return -1;
    }

  return 0;
}

int P11_unwrap_rsakey(P11Session *p11s, Key *key, char *label, unsigned char *subject,
                      unsigned char *id)
{
  CK_OBJECT_HANDLE hKey=0, hUnWrap=0, hPub=0;
  CK_BBOOL         bTrue=1, bFalse=0;
  CK_MECHANISM     mech;
  CK_OBJECT_CLASS  privateKey = CKO_PRIVATE_KEY,
    publicKey = CKO_PUBLIC_KEY;
  CK_KEY_TYPE      rsaType    = CKK_RSA;
  CK_RV            rv = CKR_OK;
  unsigned char *plain=NULL, *enc=NULL;
  unsigned long plen=0, elen=0, olen=0;
  char iv[12] = "12345678";
  int ok = -1;

  CK_BYTE  pExponent[] = { 0x01, 0x00, 0x01 };
  CK_BYTE  pModulus[256];

  CK_ATTRIBUTE pubKeyTemplate[] = {
    {CKA_CLASS,    &publicKey, sizeof(publicKey) },
    {CKA_KEY_TYPE, &rsaType,   sizeof(rsaType) },
    {CKA_TOKEN,    &bTrue,     sizeof(bTrue) },
    {CKA_PRIVATE,  &bFalse,    sizeof(bTrue) },
    {CKA_ENCRYPT,  &bTrue,     sizeof(bTrue) },
    {CKA_VERIFY,   &bTrue,     sizeof(bTrue) },
    {CKA_WRAP,     &bFalse,    sizeof(bTrue) },
    {CKA_MODULUS,  0,          0 },
    {CKA_PUBLIC_EXPONENT, pExponent, sizeof(pExponent)},
    {CKA_LABEL,    label,      strlen(label) } /* keep last for Luna CA3 */
  };
  CK_ATTRIBUTE priKeyTemplate[] = {
    {CKA_CLASS,       &privateKey,sizeof(privateKey) },
    {CKA_KEY_TYPE,    &rsaType,   sizeof(rsaType) },
    {CKA_TOKEN,       &bTrue,     sizeof(bTrue) },
    {CKA_SENSITIVE,   &bTrue,     sizeof(bTrue) },
    {CKA_PRIVATE,     &bTrue,     sizeof(bTrue) },
    {CKA_DECRYPT,     &bTrue,     sizeof(bTrue) },
    {CKA_SIGN,        &bTrue,     sizeof(bTrue) },
    {CKA_UNWRAP,      &bTrue,     sizeof(bTrue) },
    {CKA_EXTRACTABLE, &bFalse,    sizeof(bFalse) },
    {CKA_ID,          id,         20 },
    {CKA_SUBJECT,     subject,    strlen(subject)},
    {CKA_LABEL,       label,      strlen(label) } /* keep last for Luna CA3 */
  };

  /* generate symmentric key (3DES) for private key migration */
  if(P11_gen_symkey(p11s,&hKey,CKM_DES3_KEY_GEN,CKK_DES3,24,"aica key import")) goto done;

  /* get private key info */
  mech.mechanism = CKM_DES3_CBC;
  mech.pParameter = iv;
  mech.ulParameterLen = 8;

  if((plain=P8_toDER(key,NULL,(int*)&plen))==NULL) goto done;

  plen += 8 - (plen & 0x7); /* set padding -- plain should have enough memory */

  if((enc=(unsigned char*)malloc(plen + 128))==NULL){
    OK_set_error(ERR_ST_MEMALLOC,ERR_LC_PKCS11,ERR_PT_P11KEY+7,NULL);
    goto done;
  }

  /* do 3DER encryption */
  if((rv = p11s->p11->pFunc->C_EncryptInit(p11s->sess, &mech, hKey)) != CKR_OK){
    OK_set_error(ERR_ST_P11_ENCINIT,ERR_LC_PKCS11,ERR_PT_P11KEY+7,&rv);
    goto done;
  }

  elen = plen + 128;
  if((rv = p11s->p11->pFunc->C_EncryptUpdate(p11s->sess,plain,plen,enc,&elen)) != CKR_OK){
    OK_set_error(ERR_ST_P11_ENCRYPT,ERR_LC_PKCS11,ERR_PT_P11KEY+7,&rv);
    goto done;
  }

  if((rv = p11s->p11->pFunc->C_EncryptFinal(p11s->sess, enc+elen, &olen)) != CKR_OK){
    OK_set_error(ERR_ST_P11_ENCFIN,ERR_LC_PKCS11,ERR_PT_P11SESS+7,&rv);
    goto done;
  }
  elen += olen;

  /* set unrwap key parameter */
  mech.mechanism      = CKM_DES3_CBC;
  mech.pParameter     = (void*)"12345678"; // 8 byte IV
  mech.ulParameterLen = 8;

  if((rv = p11s->p11->pFunc->C_UnwrapKey(p11s->sess, &mech, hKey,(CK_BYTE_PTR)enc, elen, 
					 priKeyTemplate, 12, &hUnWrap)) != CKR_OK)
  {
    OK_set_error(ERR_ST_P11_UNWRAP,ERR_LC_PKCS11,ERR_PT_P11KEY+7,&rv);
    goto done;
  }

  /* set public key */
  if(LN_set_num_c(((Prvkey_RSA*)key)->n,key->size,pModulus)) goto done;

  pubKeyTemplate[7].pValue = pModulus;
  pubKeyTemplate[7].ulValueLen = key->size;

  if((rv = p11s->p11->pFunc->C_CreateObject(p11s->sess, pubKeyTemplate, 10, &hPub)) != CKR_OK){
    OK_set_error(ERR_ST_P11_CREOBJ,ERR_LC_PKCS11,ERR_PT_P11KEY+7,&rv);
    goto done;
  }

  ok = 0;
done:
  if(plain) free(plain);
  if(enc) free(enc);
  return ok;
}
