/* pemtestfc.c - test functions for PEM module. */
/*
 * Modified by National Institute of Informatics in Japan, 2013-2018.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_base64.h>
#include <aicrypto/ok_pem.h>
#include <aicrypto/ok_pkcs.h>
#include <aicrypto/ok_tool.h>
#include <aicrypto/ok_x509.h>

#define TEST_MSG	"tmp.txt~"	/* for writing */
#define TEST_CERT	"test.pem"
#define TEST_CERT0	"test.pem~"	/* for writing */
#define TEST_CRL	"test1.crl"
#define TEST_CRL0	"test1.crl~"	/* for writing */
#define TEST_CSR	"test1.p10"
#define TEST_CSR0	"test1.p10~"	/* for writing */
#define TEST_CCP	"cross.pem"	/* cross certificate pair */
#define TEST_CCP0	"cross.pem~"	/* for writing */
#define TEST_RSAKEY	"test.key"
#define TEST_RSAKEY0	"test.key~"	/* for writing */
#define TEST_DSAKEY	"dsakey.pem"
#define TEST_DSAKEY0	"dsakey.pem~"	/* for writing */
#define TEST_DSAPARAM	"dsapm.pem"
#define TEST_DSAPARAM0	"dsapm.pem~"	/* for writing */
#define TEST_ECDSAKEY	"ecdsakey_rfc5915.pem"
#define TEST_ECDSAKEY0	"ecdsakey_rfc5915.pem~"	/* for writing */
#define TEST_ECDSAPARAM	"ecdsapm.pem"
#define TEST_ECDSAPARAM0	"ecdsapm.pem~"	/* for writing */
#define TEST_PKCS7	"testp7.pem"
#define TEST_PKCS7w	"testp7.pem~"	/* for writing */
#define TEST_PKCS8	"testp8.pem"
#define TEST_PKCS8w	"testp8.pem~"	/* for writing */
#define TEST_PKCS8ENC	"testp8e.pem"
#define TEST_PKCS8ENC0	"testp8e.pem~"	/* for writing */
#ifndef PATH
# define PATH	"."
#endif

/* test/getfpath.c */
char *get_fpath(char *path, char *filename);

int test_base64()
{
	unsigned char in[512], *dec;
	char *enc;
	int i, size;

	for (i = 0; i < 512; i++)
		in[i] = i & 0xff;

	for (i = 0; i < 21; i++) {
		enc = Base64_encode(512, in, 16);
		dec = Base64_decode(enc, &size);

		if ((memcmp(in, dec, 512)) || (size != 512)) {
			printf("test base 64 encode & decode -- error!\n");
			return -1;
		}

		if ((i % 5) == 0)
			printf("test base 64 encode & decode -- ok : %d\n", i);
		free(enc);
		free(dec);
		in[i] = i + 3;
	}

	return 0;
}

int test_pem_cry()
{
	unsigned char *ivc = "abcdefgh";
	unsigned char in[512], *cry, *out;
	int i, len;

	for (i = 0; i < 512; i++)
		in[i] = i & 0xff;

	OK_set_passwd("abcde");
	for (i = 0; i < 21; i++) {
		len = 512;
		cry = PEM_msg_encrypt(in, &len, ivc, OBJ_CRYALGO_DESCBC);
		out = PEM_msg_decrypt(cry, len, ivc, OBJ_CRYALGO_DESCBC);
		if (memcmp(in, out, 512)) {
			printf("test msg encrypt & decrypt 1 -- error!\n");
			return -1;
		}
		free(cry);
		free(out);

		len = 512;
		cry = PEM_msg_encrypt(in, &len, ivc, OBJ_CRYALGO_3DESCBC);
		out = PEM_msg_decrypt(cry, len, ivc, OBJ_CRYALGO_3DESCBC);
		if (memcmp(in, out, 512)) {
			printf("test msg encrypt & decrypt 2 -- error!\n");
			return -1;
		}
		free(cry);
		free(out);

		if ((i % 5) == 0)
			printf("test msg encrypt & decrypt -- ok : %d\n", i);
		in[i] = i + 3;
	}
	OK_clear_passwd();
	return 0;
}

int test_pem_msg()
{
	char *in, *out;
	int i, j;
	off_t len;

	in = "This is pem test letter\n---!#$%&'&'()---\nEnd.";

	OK_set_passwd("abcde");
	for (i = 0; i < 11; i++) {
		j = strlen(in) + 1;
		if (PEM_write_message(in, j, TEST_MSG)) {
			printf
			    ("test pem msg file -- error in PEM_write_message\n");
			return -1;
		}
		out = PEM_read_message(TEST_MSG, &len);
		/* the following comment exists since v1.0:
		 * return value "len" is not as same size as original "in" size.
		 * because message had been encrypted with specific size of blocks (like 64bit long).
		 */
		if (strcmp(in, out)) {
			printf("test pem msg file -- error!\n");
			return -1;
		}
		free(out);
		if ((i % 5) == 0)
			printf("test pem msg file -- ok : %d\n", i);
	}
	OK_clear_passwd();
	return 0;
}

int test_pem_pki()
{
	Cert *cert, *ct2;
	Cert *req, *rq2;
	CRL *crl, *cl2;
	CertPair *ccp, *cp2;
	Prvkey_RSA *rsa;
	Prvkey_DSA *dsa;
	Prvkey_ECDSA *ecdsa;
	DSAParam *dpm;
	ECParam *E;
	char *tmp;
	char *fp_CERT = get_fpath(PATH, TEST_CERT);
	char *fp_CRL = get_fpath(PATH, TEST_CRL);
	char *fp_CSR = get_fpath(PATH, TEST_CSR);
	char *fp_CCP = get_fpath(PATH, TEST_CCP);
	char *fp_RSAKEY = get_fpath(PATH, TEST_RSAKEY);
	char *fp_DSAKEY = get_fpath(PATH, TEST_DSAKEY);
	char *fp_DSAPARAM = get_fpath(PATH, TEST_DSAPARAM);
	char *fp_ECDSAKEY = get_fpath(PATH, TEST_ECDSAKEY);
	char *fp_ECDSAPARAM = get_fpath(PATH, TEST_ECDSAPARAM);

	/* certificate */
	if ((cert = PEM_read_cert(fp_CERT)) == NULL) {
		printf("test pem read cert file -- error!\n");
		return -1;
	}
	if (PEM_write_cert(cert, TEST_CERT0)) {
		printf("test pem write cert file -- error!\n");
		return -1;
	}
	printf("test pem read & write cert file -- ok\n");

	if ((tmp = PEM_write_cert_buf(cert)) == NULL) {
		printf("test pem write cert buf -- error!\n");
		return -1;
	}
	if ((ct2 = PEM_read_cert_buf(tmp)) == NULL) {
		printf("test pem read cert buf -- error!\n");
		return -1;
	}
	if (Cert_cmp(cert, ct2)) {
		printf("test pem buffer compare -- error!\n");
		return -1;
	}
	printf("test pem read & write cert buf  -- ok\n");
	Cert_free(cert);
	Cert_free(ct2);
	free(tmp);
	free(fp_CERT);

	/* CRL */
	if ((crl = PEM_read_crl(fp_CRL)) == NULL) {
		printf("test pem read CRL file -- error!\n");
		return -1;
	}
	if (PEM_write_crl(crl, TEST_CRL0)) {
		printf("test pem write CRL file -- error!\n");
		return -1;
	}
	printf("test pem read & write CRL file -- ok\n");

	if ((tmp = PEM_write_crl_buf(crl)) == NULL) {
		printf("test pem write CRL buf -- error!\n");
		return -1;
	}
	if ((cl2 = PEM_read_crl_buf(tmp)) == NULL) {
		printf("test pem read CRL buf -- error!\n");
		return -1;
	}
	if (CRL_cmp(crl, cl2)) {
		printf("test pem buffer compare -- error!\n");
		return -1;
	}
	printf("test pem read & write CRL buf  -- ok\n");
	CRL_free(crl);
	CRL_free(cl2);
	free(tmp);
	free(fp_CRL);

	/* certificate request */
	if ((req = PEM_read_req(fp_CSR)) == NULL) {
		printf("test pem read cert req file -- error!\n");
		return -1;
	}
	if (PEM_write_req(req, TEST_CSR0)) {
		printf("test pem write cert req file -- error!\n");
		return -1;
	}
	printf("test pem read & write cert req file -- ok\n");

	if ((tmp = PEM_write_req_buf(req)) == NULL) {
		printf("test pem write cert req buf -- error!\n");
		return -1;
	}
	if ((rq2 = PEM_read_req_buf(tmp)) == NULL) {
		printf("test pem read certreq  buf -- error!\n");
		return -1;
	}
	if (Req_cmp(req, rq2)) {
		printf("test pem buffer compare -- error!\n");
		return -1;
	}
	printf("test pem read & write cert req buf  -- ok\n");
	Req_free(req);
	Req_free(rq2);
	free(tmp);
	free(fp_CSR);

	/* cross certificate pair */
	if ((ccp = PEM_read_crtp(fp_CCP)) == NULL) {
		printf("test pem read cross-cert-pair file -- error!\n");
		return -1;
	}
	if (PEM_write_crtp(ccp, TEST_CCP0)) {
		printf("test pem write cross-cert-pair file -- error!\n");
		return -1;
	}
	printf("test pem read & write cross-cert-pair file -- ok\n");

	if ((tmp = PEM_write_crtp_buf(ccp)) == NULL) {
		printf("test pem write cross-cert-pair buf -- error!\n");
		return -1;
	}
	if ((cp2 = PEM_read_crtp_buf(tmp)) == NULL) {
		printf("test pem read cross-cert-pair buf -- error!\n");
		return -1;
	}
	printf("test pem read & write cross-cert-pair buf  -- ok\n");
	CertPair_free(ccp);
	CertPair_free(cp2);
	free(tmp);		//free(fp_CPP);

	/* Private Key (RSA) */
	OK_set_passwd("abcde");
	if ((rsa = PEM_read_rsaprv(fp_RSAKEY)) == NULL) {
		printf("test pem read RSA key file -- error!\n");
		return -1;
	}
	free(fp_RSAKEY);
	if (PEM_write_rsaprv(rsa, TEST_RSAKEY0)) {
		printf("test pem write RSA key file -- error!\n");
		return -1;
	}
	Key_free((Key *) rsa);
	if ((rsa = PEM_read_rsaprv(TEST_RSAKEY0)) == NULL) {
		printf("test pem read RSA key file (2) -- error!\n");
		return -1;
	}
	Key_free((Key *) rsa);
	OK_clear_passwd();
	printf("test pem read & write RSA key file -- ok\n");

	/* Private Key (DSA) */
	OK_set_passwd("abcde");
	if ((dsa = PEM_read_dsaprv(fp_DSAKEY)) == NULL) {
		printf("test pem read DSA key file -- error!\n");
		return -1;
	}
	free(fp_DSAKEY);
	if (PEM_write_dsaprv(dsa, TEST_DSAKEY0)) {
		printf("test pem write DSA key file -- error!\n");
		return -1;
	}
	Key_free((Key *) dsa);
	if ((dsa = PEM_read_dsaprv(TEST_DSAKEY0)) == NULL) {
		printf("test pem read DSA key file (2) -- error!\n");
		return -1;
	}
	Key_free((Key *) dsa);
	OK_clear_passwd();
	printf("test pem read & write DSA key file -- ok\n");

	/* DSA Parameter */
	if ((dpm = PEM_read_dsaparam(fp_DSAPARAM)) == NULL) {
		printf("test pem read dsa param file -- error!\n");
		return -1;
	}
	free(fp_DSAPARAM);
	if (PEM_write_dsaparam(dpm, TEST_DSAPARAM0)) {
		printf("test pem write dsa param file -- error!\n");
		return -1;
	}
	DSAPm_free(dpm);
	printf("test pem read & write DSA param file -- ok\n");

	/* Private Key (ECDSA) */
	OK_set_passwd("dummy_password");
	if ((ecdsa = PEM_read_ecdsaprv(fp_ECDSAKEY)) == NULL) {
		printf("test pem read ECDSA key file -- error!\n");
		return -1;
	}
	free(fp_ECDSAKEY);
	if (PEM_write_ecdsaprv(ecdsa, TEST_ECDSAKEY0)) {
		printf("test pem write ECDSA key file -- error!\n");
		return -1;
	}
	Key_free((Key *) ecdsa);
	if ((ecdsa = PEM_read_ecdsaprv(TEST_ECDSAKEY0)) == NULL) {
		printf("test pem read ECDSA key file (2) -- error!\n");
		return -1;
	}
	Key_free((Key *) ecdsa);
	OK_clear_passwd();
	printf("test pem read & write ECDSA key file -- ok\n");

	/* ECDSA Parameter */
	if ((E = PEM_read_ecparam(fp_ECDSAPARAM)) == NULL) {
		printf("test pem read ecdsa param file -- error!\n");
		return -1;
	}
	free(fp_ECDSAPARAM);
	if (PEM_write_ecparam(E, TEST_ECDSAPARAM0)) {
		printf("test pem write ecdsa param file -- error!\n");
		return -1;
	}
	ECPm_free(E);
	printf("test pem read & write ECDSA param file -- ok\n");

	return 0;
}

int test_pem_pkcs()
{
	PKCS7 *p7;
	Key *p8;
	int i;
	char *fp_PKCS7 = get_fpath(PATH, TEST_PKCS7);
	char *fp_PKCS8 = get_fpath(PATH, TEST_PKCS8);
	char *fp_PKCS8ENC = get_fpath(PATH, TEST_PKCS8ENC);

	/* read PEM PKCS#7 signed data */
	for (i = 0; i < 11; i++) {
		if ((p7 = PEM_read_p7(fp_PKCS7)) == NULL) {
			printf("test pem pkcs7 read file -- error!\n");
			return -1;
		}
		if (PEM_write_p7(p7, TEST_PKCS7w)) {
			printf("test pem pkcs7 write file -- error!\n");
			return -1;
		}
		if ((i % 5) == 0)
			printf("test pem pkcs7 read & write file -- ok : %d\n",
			       i);

		P7_free(p7);
	}
	free(fp_PKCS7);

	/* read pem non-encrypted PKCS#8 file. */
	for (i = 0; i < 11; i++) {
		if ((p8 = PEM_read_p8(fp_PKCS8)) == NULL) {
			printf("test pem pkcs8 read file -- error!\n");
			return -1;
		}
		if (PEM_write_p8(p8, TEST_PKCS8w)) {
			printf("test pem pkcs8 write file -- error!\n");
			return -1;
		}
		if ((i % 5) == 0)
			printf("test pem pkcs8 read & write file -- ok : %d\n",
			       i);

		Key_free(p8);
	}
	free(fp_PKCS8);

	/* read pem encrypted PKCS#8 file. */
#ifndef HAVE_MD2
	printf("skipped: test for encrypted PKCS#8 using MD2\n");
#endif
	for (i = 0; i < 12; i++) {
		OK_set_passwd("abcde");
		if ((p8 = PEM_read_p8enc(fp_PKCS8ENC)) == NULL) {
			printf("test pem pkcs8 read file -- error!\n");
			return -1;
		}
#ifdef HAVE_MD2
		OK_set_p5_cry_algo(OBJ_P5_MD2DES + (i % 6));
#else
		OK_set_p5_cry_algo(OBJ_P5_MD5DES + (i % 4));
#endif

		if (PEM_write_p8enc(p8, TEST_PKCS8ENC0)) {
			printf("test pem pkcs8 write file -- error!\n");
			return -1;
		}
		OK_clear_passwd();

		if ((i % 5) == 0)
			printf
			    ("test pem encrypted pkcs8 read & write file -- ok : %d\n",
			     i);

		Key_free(p8);
	}
	free(fp_PKCS8ENC);

	return 0;
}
