/* bs128testfc.c */
/*
 * Copyright (c) 2016 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "bs128.h"

#define TEST_OK(label, n)	printf("test bs128 %s -- OK : %d\n", label, n);
#define TEST_NG(label, n)	printf("test bs128 %s -- NG : %d\n", label, n);
#define TEST_ERROR(label, n)	printf("test bs128 %s -- error! : %d\n", label, n);

static uint8_t k[]={
	0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
	0x6d, 0x6a, 0x8f, 0x94, 0x67, 0x30, 0x83, 0x08
};


int test_bs128_all(char *);
int test_bs_new(void);
int test_bs_get_len_1(void);
int test_bs_get_len_2(void);

/**
 * check replaces to zero the 10 bits from 20th bit .
 *
 * original feffe9928665731c6d6a8f9467308308
 * expected feffe9928665731c6c008f9467308308
 */
int test_bs_set_zero_1(void);

/**
 * check replaces to zero the 27 bits from 27th bit .
 *
 * original feffe9928665731c6d6a8f9467308308
 * expected feffe9800000031c6d6a8f9467308308
 */
int test_bs_set_zero_2(void);

/**
 * check replaces to zero the 128 bits from 0th bit .
 *
 * original feffe9928665731c6d6a8f9467308308
 * expected 00000000000000000000000000000000
 */
int test_bs_set_zero_3(void);
int test_bs_get_data(void);
int test_bs_set_data(void);
int test_bs_copy(void);
int test_bs_clone(void);
int test_bs_check_bit(void);
int test_bs_xor(void);
int test_bs_rshift_1(void);
int test_bs_inc32(void);
int test_bs_copy_uint64(void);

/**
 * check
 */
int test_bs128_all(char *x)
{
	return 0;
}

/**
 * check
 */
int test_bs_new(void)
{
	bs128_t *a;

	if ((a = bs_new(128)) != NULL) {
		TEST_OK("bs_new", 1);
		bs_free(a);
	} else {
		TEST_NG("bs_new", 1);
		return -1;
	}
#if 0
	if ((a = bs_new(129)) == NULL) {
		TEST_OK("bs_new", 1);
	} else {
		TEST_NG("bs_new", 1);
		bs_free(a);
		return -1;
	}

	if ((a = bs_new(0)) != NULL) {
		TEST_OK("bs_new", 1);
		bs_free(a);
	} else {
		TEST_NG("bs_new", 1);
		return -1;
	}
#endif
	return 0;
}

/**
 * check
 */
int test_bs_get_len_1(void)
{
	bs128_t *a;

	a = bs_new(128);
	if (bs_get_len(a) == 128) {
		TEST_OK("bs_get_len", 1);
		bs_free(a);
		return 0;
	} else {
		TEST_NG("bs_get_len", 1);
		bs_free(a);
		return -1;
	}
}

/**
 * check
 */
int test_bs_get_len_2(void)
{
	bs128_t *a;

	a = bs_new(60);
	if (bs_get_len(a) == 60) {
		TEST_OK("bs_get_len", 2);
		bs_free(a);
		return 0;
	} else {
		TEST_NG("bs_get_len", 2);
		bs_free(a);
		return -1;
	}
}

int test_bs_set_zero_1(void)
{
	/* bs_set_zero(a, 70, 10); */
	static uint8_t exp_t1[]={
		0xfe, 0xff, 0xe9, 0x92,  /*  0- 31 */
		0x86, 0x65, 0x73, 0x1c,  /* 32- 63 */
		0x6c, 0x00, 0x8f, 0x94,  /* 64- 95 */
		0x67, 0x30, 0x83, 0x08   /* 96-127 */
	};

	bs128_t *a;

	if ((a = bs_new(128)) == NULL) {
		TEST_ERROR("bs_set_zero", 3);
		return -1;
	}

	bs_set_data(a, k, 128);
	bs_set_zero(a, 70, 10);
	if (memcmp(a->buf, exp_t1, BS128_BYTE_LEN) == 0) {
		TEST_OK("bs_set_zero", 3);
		bs_free(a);
		return 0;
	} else {
		TEST_NG("bs_set_zero", 3);
		bs_free(a);
		return -1;
	}
}

int test_bs_set_zero_2(void)
{
	static uint8_t exp_t2[]={
		0xfe, 0xff, 0xe9, 0x80,  /*  0- 31 */
		0x00, 0x00, 0x03, 0x1c,  /* 32- 63 */
		0x6d, 0x6a, 0x8f, 0x94,  /* 64- 95 */
		0x67, 0x30, 0x83, 0x08}; /* 96-127 */

	bs128_t *a;

	if ((a = bs_new(128)) == NULL) {
		TEST_ERROR("bs_set_zero", 4);
		return -1;
	}

	/* test 2 */
	bs_set_data(a, k, 128);
	bs_set_zero(a, 27, 27);
	if (memcmp(a->buf, exp_t2, BS128_BYTE_LEN) == 0) {
		TEST_OK("bs_set_zero", 4);
		bs_free(a);
		return 0;
	} else {
		TEST_NG("bs_set_zero", 4);
		bs_free(a);
		return -1;
	}
}

int test_bs_set_zero_3(void)
{
	/* bs_set_zero(a, 0, 128); */
	static uint8_t exp_t3[]={
		0x00, 0x00, 0x00, 0x00,  /*  0- 31 */
		0x00, 0x00, 0x00, 0x00,  /* 32- 63 */
		0x00, 0x00, 0x00, 0x00,  /* 64- 95 */
		0x00, 0x00, 0x00, 0x00}; /* 96-127 */

	bs128_t *a;

	if ((a = bs_new(128)) == NULL) {
		TEST_ERROR("bs_set_zero", 5);
		return -1;
	}

	bs_set_data(a, k, 128);
	bs_set_zero(a, 0, 128);
	if (memcmp(a->buf, exp_t3, BS128_BYTE_LEN) == 0) {
		TEST_OK("bs_set_zero", 5);
		bs_free(a);
		return 0;
	} else {
		TEST_NG("bs_set_zero", 5);
		bs_free(a);
		return -1;
	}
}

int test_bs_get_data(void)
{
	static uint8_t exp1[]={
		0xfe, 0xc0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
	};

	static uint8_t exp2[]={
		0xfe, 0xff, 0xe9, 0x92, 0x86, 0x65, 0x73, 0x1c,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
	};

	int ret = 0;
	bs128_t *a;
	uint8_t dest[BS128_BYTE_LEN];

	if ((a = bs_new(128)) == NULL) {
		TEST_ERROR("bs_get_data", 1);
		return -1;
	}

	bs_set_data(a, k, 128);

	/* 1 */
	memset(dest, 0, BS128_BYTE_LEN);
	bs_get_data(a, dest, 10);
	if (memcmp(dest, exp1, BS128_BYTE_LEN) == 0) {
		TEST_OK("bs_get_data", 2);
	} else {
		TEST_NG("bs_get_data", 2);
		ret = -1;
	}

	/* 2 */
	memset(dest, 0, BS128_BYTE_LEN);
	bs_get_data(a, dest, 64);
	if (memcmp(dest, exp2, BS128_BYTE_LEN) == 0) {
		TEST_OK("bs_get_data", 3);
	} else {
		TEST_NG("bs_get_data", 3);
		ret = -1;
	}

	/* 3 */
	memset(dest, 0, BS128_BYTE_LEN);
	bs_get_data(a, dest, 128);
	if (memcmp(dest, k, BS128_BYTE_LEN) == 0) {
		TEST_OK("bs_get_data", 4);
	} else {
		TEST_NG("bs_get_data", 4);
		ret = -1;
	}

	bs_free(a);
	return ret;
}

int test_bs_set_data(void)
{
	bs128_t *a;
	return 0;
}

int test_bs_copy(void)
{
	bs128_t *from;
	bs128_t *to;
	int ret = 0;

	if ((from = bs_new(128)) == NULL) {
		TEST_ERROR("test_bs_copy", 1);
		return -1;
	}

	if ((to = bs_new(128)) == NULL) {
		TEST_ERROR("test_bs_copy", 2);
		bs_free(from);
		return -1;
	}

	bs_set_data(from, k, 128);

	bs_copy(from, to);
	if (from->len == to->len) {
		TEST_OK("test_bs_copy", 3);
	} else {
		TEST_NG("test_bs_copy", 3);
		ret = -1;
	}

	if (memcmp(from->buf, to->buf, BS128_BYTE_LEN) == 0) {
		TEST_OK("test_bs_copy", 4);
	} else {
		TEST_NG("test_bs_copy", 4);
		ret = -1;
	}

	bs_free(from);
	bs_free(to);
	return ret;
}

int test_bs_clone(void)
{
	bs128_t *from;
	bs128_t *to;
	int ret = 0;

	if ((from = bs_new(128)) == NULL) {
		TEST_ERROR("test_bs_clone", 1);
		return -1;
	}

	to = NULL;

	bs_set_data(from, k, 128);

	to = bs_clone(from);
	if (to == NULL) {
		TEST_ERROR("test_bs_clone", 2);
		bs_free(from);
		return -1;
	}

	if (from->len == to->len) {
		TEST_OK("test_bs_clone", 2);
	} else {
		TEST_NG("test_bs_clone", 2);
		ret = -1;
	}

	if (memcmp(from->buf, to->buf, BS128_BYTE_LEN) == 0) {
		TEST_OK("test_bs_clone", 3);
	} else {
		TEST_NG("test_bs_clone", 3);
		ret = -1;
	}

	bs_free(from);
	bs_free(to);
	return ret;
}

int test_bs_check_bit_1(void)
{
	bs128_t *a;
	int r = 0;

	if ((a = bs_new(128)) == NULL) {
		TEST_ERROR("test_bs_check_bit", 1);
		return -1;
	}

	bs_set_data(a, k, 128);
	if (bs_check_bit(a, 0) != 0) {
		TEST_OK("test_bs_check_bit", 2);
		r = 0;
	} else {
		TEST_NG("test_bs_check_bit", 2);
		r = -1;
	}

	bs_set_data(a, k, 128);
	if (bs_check_bit(a, 0) != 0) {
		TEST_OK("test_bs_check_bit", 2);
		r = 0;
	} else {
		TEST_NG("test_bs_check_bit", 2);
		r = -1;
	}

	bs_free(a);
	return r;
}

int test_bs_check_bit_2(void)
{
	bs128_t *a;

	if ((a = bs_new(128)) == NULL) {
		TEST_ERROR("test_bs_check_bit", 3);
		return -1;
	}

	bs_set_data(a, k, 128);

	bs_free(a);
	return 0;
}

int test_bs_xor(void)
{
	bs128_t *a;
	return 0;
}

int test_bs_rshift_1(void)
{
	/*
	 * 0xb8      0x3b      0xe9      0x92
	 * 1011 1000 0011 1011 1110 1001 1001 0010
	 * 0x86      0x65      0x73      0x1c
	 * 1000 0110 0110 0101 0111 0011 0001 1100
	 * 0x6c      0x00      0x8f      0x94
	 * 0110 1100 0000 0000 1000 1111 1001 0100
	 * 0x67      0x30      0x3b      0x78
	 * 0110 0111 0011 0000 0011 1011 0111 1000
	 */
	static uint8_t org[]={
		0xb8, 0x3b, 0xe9, 0x92,  /*  0- 31 */
		0x86, 0x65, 0x73, 0x1c,  /* 32- 63 */
		0x6c, 0x00, 0x8f, 0x94,  /* 64- 95 */
		0x67, 0x30, 0x3b, 0x78   /* 96-127 */
	};

	/*
	 * 0x5c      0x1d      0xf4      0xc9
	 * 0101 1100 0001 1101 1111 0100 1100 1001
	 * 0x43      0x32      0xb9      0x8e
	 * 0100 0011 0011 0010 1011 1001 1000 1110
	 * 0x36      0x00      0x47      0xca
	 * 0011 0110 0000 0000 0100 0111 1100 1010
	 * 0x33      0x98      0x1d      0xbc
	 * 0011 0011 1001 1000 0001 1101 1011 1100
	 */
	static uint8_t exp_t1[]={
		0x5c, 0x1d, 0xf4, 0xc9, /*  0- 31 */
		0x43, 0x32, 0xb9, 0x8e, /* 32- 63 */
		0x36, 0x00, 0x47, 0xca, /* 64- 95 */
		0x33, 0x98, 0x1d, 0xbc  /* 96-127 */
	};

	bs128_t *a;
	int r = 0;

	if ((a = bs_new(128)) == NULL) {
		TEST_ERROR("test_bs_rshift", 1);
		return -1;
	}

	bs_set_data(a, org, 128);
	bs_rshift(a, 1);
	if (memcmp(a->buf, exp_t1, 16) == 0) {
		TEST_OK("test_bs_rshift", 1);
		r = 0;
	} else {
		TEST_NG("test_bs_rshift", 1);
		r = -1;
	}

	bs_free(a);
	return 0;
}
