/* bs128.h */
/*
 * Copyright (c) 2016 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
#ifndef INCLUSION_GUARD_UUID_DCC5DE75_2E31_48C1_8EB2_ABE0DBD1A793
#define INCLUSION_GUARD_UUID_DCC5DE75_2E31_48C1_8EB2_ABE0DBD1A793

#include <stdint.h>

/* uint8_t array size */
#define BS128_BYTE_LEN	16

/**
 * 128 bit string
 */
typedef struct bs128 {
	int32_t len; /* bit length */
	uint8_t buf[BS128_BYTE_LEN];
} bs128_t;

/**
 * Allocate bs128 memory.
 *
 * @attention
 * if the object that was gotten by this function is not necessary
 * anymore, it is necessary to free by bs_free() function.
 */
extern bs128_t *bs_alloc();

/**
 * Allocate ans initialize bs128 memory.
 *
 * @attention
 * if the object that was gotten by this function is not necessary
 * anymore, it is necessary to free by bs_free() function.
 *
 * @param[in] bit bit length (0 <= bit <= 128).
 */
extern bs128_t *bs_new(int32_t bit);

/**
 * Free bs128 memory.
 *
 * @param[in] bs bs128 object.
 */
extern void bs_free(bs128_t *bs);

/**
 * return bit length.
 *
 * @param[in] bs bs128 object.
 */
extern int32_t bs_get_len(bs128_t *bs);

/**
 * Set 0 to len bits from begin-th bit.
 *
 * @param[out] bs bs128 object
 * @param[in] begin
 * @param[in] len
 */
extern void bs_set_zero(bs128_t *bs, int32_t begin, int32_t len);

/**
 * Get bit string.
 *
 * @param[in] bs bs128 object
 * @param[out] out
 * @param[in] len bit length.
 */
extern void bs_get_data(bs128_t *bs, uint8_t *out, int32_t len);

/**
 * Get bit string.
 *
 * @param[out] bs bs128 object
 * @param[in] data
 * @param[in] len bit length.
 */
extern void bs_set_data(bs128_t *bs, uint8_t *data, int32_t len);

/**
 * copy the bs128 structure.
 *
 * @param[in] from bs128 object that want to copy.
 * @param[out] to The copy destination object.
 */
extern void bs_copy(bs128_t *from, bs128_t *to);

/**
 * clone the bs128 structure.
 *
 * @attention
 * if the object that was gotten by this function is not necessary
 * anymore, it is necessary to free by bs_free() function.
 *
 * @param[in] bs bs128 structure that want to clone.
 * @returns
 * return NULL if error occurred. otherwise, return new allocated bs128
 * structure.
 */
extern bs128_t *bs_clone(bs128_t *bs);

/**
 * clone the bs128 structure.
 *
 * @param[in] bs bs128 object.
 * @param[in] bit
 * @returns
 * 0 or 1 or -1.
 */
extern int bs_check_bit(bs128_t *bs, int32_t bit);

/**
 * Calcurate XOR of a and b, set to a.
 */
extern int bs_xor(bs128_t *a, bs128_t *b);

/**
 * bit right shift.
 */
extern void bs_rshift(bs128_t *bs, int32_t bit);

extern void bs_print(bs128_t *bs);

#endif /* INCLUSION_GUARD_UUID_DCC5DE75_2E31_48C1_8EB2_ABE0DBD1A793 */
