/* large number test functions. */
/*
 * Modified by National Institute of Informatics in Japan, 2011-2016.
 *
 */
/*
 * Copyright (C) 1998-2002
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "aiconfig.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <math.h>
#include <aicrypto/large_num.h>

/* size=1 means 16 bit long */
void make_test_LNm(LNm **a1,LNm **a2,int size){
    unsigned short n1[LN_MAX];
    unsigned short n2[LN_MAX];
    int  i;
    
    do {
        n1[0]= (unsigned short)rand() * 3;
    } while (n1[0] == 0);
    for(i=1;i<size;i++)
        n1[i]= (unsigned short)rand() * 3;

    do {
        n2[0]= (unsigned short)rand();
    } while (n2[0] == 0);
    for(i=1;i<size-1;i++)
        n2[i]= (unsigned short)rand();

    *a1=LN_alloc_s(size,n1);
    *a2=LN_alloc_s(size-1,n2);
}

int test_plus_minus(void){
	LNm *a1,*a2,*pul,*min;
	unsigned short n1[4];
	uint32_t  n2[4];
	int	i,j;

	pul = LN_alloc();
	min = LN_alloc();

	for(j=32;j<LN_MAX;j+=32){
		for(i=0;i<2001;i++){
			make_test_LNm(&a1,&a2,j);

			LN_plus(a1,a2,pul);
			LN_minus(pul,a2,min);
			if(LN_cmp(a1,min)){
				printf("%d : ERROR eq = %d\n",i,LN_cmp(a1,min));
				return -1;
			}

			LN_minus(a2,a1,min);
			LN_plus(min,a1,pul);
			if(LN_cmp(a2,pul)){
				printf("%d : ERROR -+ = %d\n",i,LN_cmp(a1,min));
				return -1;
			}

			if(!(i%500))printf("+- (%d bit)-- ok:%d\n",j*16,i);
    
			LN_free(a1);
			LN_free(a2);
		}
	}

	/* other test 1 */
	n1[0]=0;
	LN_clean(min);
	a1=LN_alloc_s(1,n1);
	a2=LN_alloc_s(1,n1);
    LN_plus(a1,a2,pul);
    LN_minus(pul,a2,min);
    if(LN_cmp(a1,min)){
		printf("%s : ERROR eq = %d\n","test1",LN_cmp(a1,min));
		return -1;
    }
    LN_free(a1);    LN_free(a2);

	/* other test 2 */
	n2[0]=0xfffffffe;
	a1=LN_alloc_u32(1,n2);
	a2=LN_alloc_u32(1,n2);
    LN_plus(a1,a2,pul);
    LN_minus(pul,a2,min);

    if(LN_cmp(a1,min)){
		printf("%s : ERROR eq = %d\n","test2",LN_cmp(a1,min));
		return -1;
    }
    LN_free(a1);    LN_free(a2);

	/* other test 3 */
	n2[0]=0xffffffff; n2[1]=0xffffffff;
	a1=LN_alloc_u32(2,n2);
	a2=LN_alloc_u32(2,n2);

    LN_plus(a1,a2,pul);
    LN_minus(pul,a2,min);
    if(LN_cmp(a1,min)){
		printf("%s : ERROR eq = %d\n","test3",LN_cmp(a1,min));
		return -1;
    }
    LN_free(a1);    LN_free(a2);

	/* other test 4 */
	n2[0]=0x1000000; n2[1]=0; n2[1]=0; n2[3]=0;
	a1=LN_alloc_u32(4,n2);
	a2=LN_alloc_u32(1,n2);
    LN_minus(a1,a2,min);
    LN_plus(min,a2,pul);
    if(i=LN_cmp(a1,pul)){
		printf("test4 : ERROR eq = %d\n",i);
		return -1;
    }
    LN_free(a1);    LN_free(a2);

	/* other test 5 */
	make_test_LNm(&a1,&a2,64);
	LN_clean(a2);
	LN_plus(a1,a2,pul);
	if(i=LN_cmp(a1,pul)){
		printf("test5 : ERROR eq = %d\n",i);
		return -1;
	}
    LN_free(a1);    LN_free(a2);

	LN_free(pul);
	LN_free(min);

	printf("plus minus test .. ok\n");
	return 0;
}

int test_mul_div_mod(void){
	LNm *a1,*a2;
    LNm *b,*c,*d,*e;
	uint32_t  n2[16];
	uint32_t  a1n[]={
		0x5500f4cc,0xcabee396,0xdf404457,0x53131aea,0xc5b4ba67,0x5ac54b59,
		0x148f222e,0x30e76464,0xc38e2503,0xfec61789,0x92a35d64};
	uint32_t  a2n[]={
		0x0001394e,0x6a38470a,0xb92f5a44,0x2c744dad,
		0x89b8c531,0x67076fff};
	unsigned char in[] = {
		0x9a,0x67,0xbc,0xb5,0x37,0xf2,0x50,0x70,0xf2,0x8b,0x12,0xb5,0x60,0x9f,0xac,0x85,
		0x78,0x4a,0x36,0x1d,0x39,0x5a,0x68,0xff,0xd4,0x38,0x86,0x5a,0xe9,0x86,0x46,0xd4,
	};
	int	i,j;

	b = LN_alloc();
	c = LN_alloc();
	d = LN_alloc();
	e = LN_alloc();

	/* div mod test 1 */
	a1=LN_alloc_u32(11,a1n);
	a2=LN_alloc_u32(6,a2n);

	for(i=0;i<1001;i++){
		LN_div_mod(a1,a2,b,c);
		LN_multi_std(b,a2,d);
		LN_plus(d,c,e);

		if(LN_cmp(a1,e)){
			printf("ERROR! i=%d::%d\n",i,LN_cmp(a1,e));
			return -1;}

		if(!(i%200)) printf("/* -- ok:%d\n",i);
	}
	LN_free(a1);LN_free(a2);

    /* div mod test */
	for(j=32;j<LN_MAX;j+=32){
		for(i=0;i<2001;i++){
			make_test_LNm(&a1,&a2,j);

			LN_div_mod(a1,a2,b,c);
			LN_multi_std(b,a2,d);

			LN_plus(d,c,e);

			if(LN_cmp(a1,e)){
				printf("a1/a2 = b...c,\na2*c = d, d+c = e\n");
				LN_print(a1); LN_print(a2);

				LN_print(b); LN_print(c);
				LN_print(d); LN_print(e);

				printf("ERROR! i=%d::%d\n",i,LN_cmp(a1,e));
				return -1;
			}
		
			if(!(i%500)) printf("/* (%d bit) -- ok:%d\n",j*16,i);

			LN_free(a1);
			LN_free(a2);
		}
	}

    /* div mod test 2 */
	for(j=128;j<LN_MAX;j+=32){
		for(i=0;i<1001;i++){
			make_test_LNm(&a1,&a2,j);

			LN_multi(a1,a1,b);
			LN_div_mod(b,a1,c,d);

			if(LN_cmp(a1,c)&&d->top){
				printf("a1*a2 = b,\nb/a2 = c...d\n");
				LN_print(a1); LN_print(a2);

				LN_print(b); LN_print(c);
				LN_print(d);

				printf("ERROR! i=%d::%d\n",i,LN_cmp(a1,c));
				return -1;
			}
		
			if(!(i%200)) printf("*/ (%d bit)-- ok:%d\n",j*16,i);
			LN_free(a1);
			LN_free(a2);
		}
	}

    /* sqr test */
	for(j=32;j<LN_MAX;j+=32){
		for(i=0;i<1001;i++){
			make_test_LNm(&a1,&a2,j);

			LN_multi(a1,a1,b);
			LN_sqr(a1,c);

			if(LN_cmp(b,c)){
				printf("a1*a1 = b,\na1^2 = c\n");
				LN_print(a1); LN_print(b); LN_print(c);

				printf("ERROR! i=%d::%d\n",i,LN_cmp(b,c));
				return -1;
			}
		
			LN_free(a1);	LN_free(a2);
			if(!(i%200)) printf("x^2 (%d bit)-- ok:%d\n",j*16,i);
		}
	}

	/* other test 1 */
	n2[0]=0xffffffff; n2[1]=0xffffffff;
	n2[2]=0xffffffff; n2[3]=0xffffffff;
	a1=LN_alloc_u32(4,n2);
	a2=LN_alloc_u32(4,n2);

	n2[0]=0xffffffff; n2[1]=0xffffffff; n2[2]=0xffffffff; n2[3]=0xfffffffe; 
	n2[4]=0x00000000; n2[5]=0x00000000; n2[6]=0x00000000; n2[7]=0x00000001; 
	LN_set_num(c,8,n2);
	
    LN_multi(a1,a2,b);
	LN_print(b);

    if(LN_cmp(b,c)){
		LN_print(c);
		printf("%s : ERROR eq = %d\n","multi test 1",LN_cmp(b,c));
		return -1;
    }
    LN_free(a1);    LN_free(a2);

	/* other test 2 */
	n2[0]=0xffffffff; n2[1]=0xffffffff; n2[2]=0xffffffff; n2[3]=0xffffffff;
	a1=LN_alloc_u32(4,n2);
	n2[0]=0x00000001; n2[1]=0xffffffff; n2[2]=0x33333333; n2[3]=0xcccccccc;
	a2=LN_alloc_u32(3,n2);
	LN_div_mod(a1,a2,b,c);
    LN_free(a1);    LN_free(a2);

	/* other test 3 */
	n2[0]=0xffffffff; n2[1]=0xffffffff; n2[2]=0xffffffff; n2[3]=0xffffffff;
	a1=LN_alloc_u32(4,n2);
	LN_long_multi(a1,1,b);
    if(LN_cmp(a1,b)){
		LN_print(b);
		printf("%s : ERROR eq = %d\n","multi test 3",LN_cmp(a1,b));
		return -1;
    }
    LN_free(a1);

	/* other test 4 */
	n2[0]=0xffffffff; n2[1]=0xffffffff; n2[2]=0xffffffff; n2[3]=0xffffffff;
	a1=LN_alloc_u32(4,n2);
	a2=LN_alloc_u32(1,n2);
	LN_long_multi(a1,0xffffffff,b);
	LN_multi(a1,a2,c);
    if(LN_cmp(b,c)){
		LN_print(b);
		LN_print(c);
		printf("%s : ERROR eq = %d\n","multi test 4",LN_cmp(b,c));
		return -1;
    }
    LN_free(a1);    LN_free(a2);

	/* other test 5 */
	n2[0]=0x0000ffff; n2[1]=0xffffffff;
	a1=LN_alloc_u32(2,n2);
	LN_multi(a1,a1,b);
	LN_sqr(a1,c);
    if(LN_cmp(b,c)){
		LN_print(b);
		LN_print(c);
		printf("%s : ERROR eq = %d\n","multi test 5 (sqr)",LN_cmp(b,c));
		return -1;
    }
    LN_free(a1);

	/* other test 6 */
	n2[0]=0xffffffff; n2[1]=0xffffffff;
	a1=LN_alloc_u32(2,n2);
	LN_multi(a1,a1,b);
	LN_sqr(a1,c);
    if(LN_cmp(b,c)){
		LN_print(b);
		LN_print(c);
		printf("%s : ERROR eq = %d\n","multi test 6 (sqr)",LN_cmp(b,c));
		return -1;
    }

    LN_free(a1);

	/* other test 7 */
	LN_set_num_c(b,32,in);

	if(LN_multi(b,b,c)) return -1;
	if(LN_sqr(b,d)) return -1;
    if(LN_cmp(c,d)){
		LN_print(c);
		LN_print(d);
		printf("%s : ERROR eq = %d\n","multi test 7 (sqr bug)",LN_cmp(b,c));
		return -1;
    }else{
		printf("x^2 (sqr bug test) -- ok\n");
	}

    LN_free(b); LN_free(c); LN_free(d); LN_free(e);

    printf("div and mod test .. ok\n");
	return 0;
}

int test_karatsuba(void){
	LNm *a1,*a2;
    LNm *b,*c,*d,*e;
	int	i,j;

	/* test1 karatsuba */
	uint32_t tst0[]={
		0x00ab35ff,0xd8ba5fa8,0xe6d3a243,0x846a9730,
		0x33818f75,0x2dee6242,0x70d8c828};

	uint32_t tst1[]={
		0x00ab35ff,0xd8ba5fa8,0xe6d3a243,0x846a9730,0x33818f75,
		0x2dee6242,0x70d8c828,0x46bdcd0c,0xbb868c61,0x4c58de05,
		0xd7ffd46e,0x671f165b,0x5d016613,0x571df358,0x5584d059,
		0x0b4b8100,0x00000000,0x00000000,0x00000000,0x00000000,
		0x00000000,0x00000000};

	uint32_t tst2[]={
		0xe7a01eb2,0xe2c10f1a,0xfbdd9bb0,0x89a8a35c,0xd9e3847d,
		0x718f1100,0x00000000,0x00000000,0x00000000,0x00000000,
		0x00000000,0x00000000,0x00000000};

	unsigned char sq[] = {
		0x3f,0xfd,0x8e,0xbd,0xa2,0xc6,0xc6,0x46,0x15,0x59,0x22,0x9f,0x09,0xb9,0xbb,0x92,
		0x8d,0x19,0x8a,0xc6,0xb2,0x63,0xa4,0x2e,0xfe,0x97,0x63,0x38,0xd9,0x3a,0xd5,0xbd,
		0xb1,0x3b,0x42,0x88,0x2c,0xaf,0x2b,0x2e,0xc4,0x13,0xc5,0xfe,0x7a,0xf3,0xa1,0x41,
		0xbb,0x6c,0x79,0x3d,0x05,0x30,0x77,0xbb,0x57,0x09,0xec,0x4b,0xc4,0x28,0x4e,0x8e,
		0x33,0xb9,0x3a,0x71,0xf2,0xf7,0x2b,0x3c,0x97,0xaa,0x79,0x58,0xf1,0x93,0x7f,0x8e,
		0x52,0x57,0xd8,0x39,0x2b,0xde,0xf1,0x84,0x73,0xd7,0xec,0xae,0x2d,0xce,0xdb,0x37,
		0xbf,0xf7,0x7f,0x28,0xe9,0x2a,0x8f,0x43,0xf9,0x60,0x20,0x3d,0xd7,0x5a,0xf9,0x26,
		0x66,0x61,0xe3,0x9b,0xb4,0x77,0x8f,0x56,0x3d,0x61,0x4c,0x40,0x18,0x9f,0x87,0x6e
	};

	b = LN_alloc();
	c = LN_alloc();
	d = LN_alloc();
	e = LN_alloc();

	/* karatsuba test1 */
	LN_set_num(b,7,tst0);
    LN_multi_kara(b,b,c);	LN_multi_std(b,b,d);

	if(LN_cmp(c,d)){ LN_print(b); LN_print(c); LN_print(d);
		printf("Karatsuba test1: ERROR eq = %d\n",LN_cmp(c,d));
		return -1;
    }else
		printf("Karatsuba test1 -- ok\n",LN_cmp(c,d));

	/* karatsuba test2 */
	LN_set_num(b,22,tst1);
    LN_multi_kara(b,b,c);	LN_multi_std(b,b,d);

	if(LN_cmp(c,d)){ LN_print(b); LN_print(c); LN_print(d);
		printf("Karatsuba test2: ERROR eq = %d\n",LN_cmp(c,d));
		return -1;
    }else
		printf("Karatsuba test2 -- ok\n",LN_cmp(c,d));

	/* karatsuba test3 */
	LN_set_num(b,7,tst2);
    LN_multi_kara(b,b,c);	LN_multi_std(b,b,d);

	if(LN_cmp(c,d)){ LN_print(b); LN_print(c); LN_print(d);
		printf("Karatsuba test3: ERROR eq = %d\n",LN_cmp(c,d));
		return -1;
    }else
		printf("Karatsuba test3 -- ok\n",LN_cmp(c,d));

	/* karatsuba test 4 */
	for(j=32;j<LN_MAX;j+=32){
		for(i=0;i<1001;i++){
			make_test_LNm(&a1,&a2,j);

			LN_multi_kara(a1,a2,b);
			LN_multi_std(a1,a2,c);

		    if(LN_cmp(b,c)){
				LN_print(a1);	LN_print(b);	LN_print(c);
				printf("%s (%d): ERROR eq = %d\n","kara-multi test",i,LN_cmp(b,c));
				return -1;
			}

		    LN_multi_kara(a2,a2,b);
			LN_multi_std(a2,a2,c);
		    if(LN_cmp(b,c)){
				printf("%s (%d): ERROR eq = %d\n","kara-multi test",i,LN_cmp(b,c));
				return -1;
		    }

		    LN_multi_kara(a2,a1,b);
			LN_multi_std(a2,a1,c);
		    if(LN_cmp(b,c)){
				printf("%s (%d): ERROR eq = %d\n","kara-multi test",i,LN_cmp(b,c));
				return -1;
		    }

			if(!(i%200)) printf("karatsuba multi (%d bit) -- ok:%d\n",j*16,i);
		    LN_free(a1);    LN_free(a2);
		}
	}

	/* karatsuba test 5 */
	for(j=32;j<LN_MAX;j+=32){
		for(i=0;i<1001;i++){
			make_test_LNm(&a1,&a2,j);

		    LN_sqr_kara(a1,b);
			LN_sqr_std(a1,c);
		    if(LN_cmp(b,c)){
				LN_print(a1);	LN_print(b);	LN_print(c);
				printf("%s (a1:%d): ERROR eq = %d\n","kara-sqr test",i,LN_cmp(b,c));
				return -1;
		    }

		    LN_sqr_kara(a2,b);
			LN_sqr_std(a2,c);
		    if(LN_cmp(b,c)){
				printf("%s (a2:%d): ERROR eq = %d\n","kara-sqr test",i,LN_cmp(b,c));
				return -1;
		    }

			if(!(i%200)) printf("karatsuba sqr (%d bit) -- ok:%d\n",j*16,i);
		    LN_free(a1);    LN_free(a2);
		}
	}

	/* karatsuba sqr BUG test 6 .. 2003/12/2 */
	LN_set_num_c(b,128,sq);
	LN_sqr_std(b,c);
	LN_sqr_kara(b,d);
	
	if(LN_cmp(c,d)){
		printf("Karatsuba sqr BUG test 6: ERROR eq = %d\n",LN_cmp(c,d));
		return -1;
    }else
		printf("Karatsuba sqr BUG test 6 -- ok\n",LN_cmp(c,d));


    LN_free(b); LN_free(c); LN_free(d); LN_free(e);
    printf("karatsuba mul and sqr test .. ok\n");
	return 0;
}

int test_shift(void){
	LNm *a1,*a2;
    LNm *b,*c;
	uint32_t  n2[16];
	int	i,j,k;

	b = LN_alloc();
	c = LN_alloc();

	for(k=32;k<LN_MAX;k+=32){
		for(i=0;i<501;i++){
			make_test_LNm(&a1,&a2,k);

			for(j=0;j<=31;j++){
				LN_lshift32(a1,j,b);
				LN_rshift32(b,j,c);
			    if(LN_cmp(a1,c)){
					LN_print(a1);
					LN_print(c);
					printf("shift test (%d,%d): ERROR eq = %d\n",i,j,LN_cmp(a1,c));
					return -1;
				}
			}

			if(!(i%100)) printf("shift (%d bit) -- ok:%d\n",k*16,i);
			LN_free(a1); LN_free(a2);
		}	
	}

	/* other test 1 */
	n2[0]=0x0fffffff; n2[1]=0xfffffff0;
	a1=LN_alloc_u32(1,n2);
	a2=LN_alloc_u32(1,&n2[1]);
	LN_lshift32(a1,4,b);
    if(LN_cmp(b,a2)){
		LN_print(b);
		LN_print(a2);
		printf("%s : ERROR eq = %d\n","shift test 1",LN_cmp(b,a2));
		return -1;
    }
    LN_free(a1); LN_free(a2);

	/* other test 2 */
	n2[0]=0x0fffffff; n2[1]=0x00ffffff;
	a1=LN_alloc_u32(1,n2);
	a2=LN_alloc_u32(1,&n2[1]);
	LN_rshift32(a1,4,b);
    if(LN_cmp(b,a2)){
		LN_print(b);
		LN_print(a2);
		printf("%s : ERROR eq = %d\n","shift test 2",LN_cmp(b,a2));
		return -1;
    }
    LN_free(a1); LN_free(a2);

    LN_free(b); LN_free(c);
    printf("left and right shift test .. ok\n");
	return 0;
}

int test_long(void){
	LNm *a1,*a2;
    LNm *b,*c,*d;
	int	i,j;

	b = LN_alloc();
	c = LN_alloc();
	d = LN_alloc();

	for(j=32;j<LN_MAX;j+=32){
		for(i=0;i<2001;i++){
			uint32_t ul;
			make_test_LNm(&a1,&a2,j);

			ul = a2->num[LN_MAX-1];
			LN_copy(a1,b);
			LN_long_add(a1,ul);
			LN_long_sub(a1,ul);

		    if(LN_cmp(a1,b)){
				LN_print(a1);
				LN_print(b);
				printf("%s : ERROR eq = %d\n","long +-",LN_cmp(a1,b));
				return -1;
			}

			LN_long_multi(a1,ul,b);
			LN_long_div(b,ul,c);

		    if(LN_cmp(a1,c)){
				LN_print(a1);
				LN_print(c);
				printf("%s : ERROR eq = %d\n","long */",LN_cmp(a1,c));
				return -1;
			}

			if(!(i%500)) printf("long calc (%d bit) -- ok:%d\n",j*16,i);
			LN_free(a1); LN_free(a2);
		}	
	}

    LN_free(b); LN_free(c); LN_free(d);
    printf("long calc test .. ok\n");
	return 0;
}

int test_ext_euclid(void){
	unsigned short p1[]={	/* p-1 */
	0xcba9,0xe6ba,0xcf43,0x06f1,0xfe88,0xfaa3,0x01fb,0x08d0,
	0xd4ed,0xb47f,0xef57,0xf4d4,0xc2fc,0x045f,0x96af,0xb59d,
	0xa247,0x9769,0xcbae,0xbecb,0xf90c,0xf466,0x05f5,0xfa71,
	0xf931,0xd509,0x0e6f,0xa138,0xf300,0xc800,0xa672,0xe95e};
	unsigned short p2[]={ /* q-1 */
	0xc7f5,0x4c4f,0x4ced,0x39a6,0x54b7,0x3d40,0x74ee,0x6c85,
	0x68a0,0x6ff8,0x76cd,0x42ea,0x227c,0x5d54,0x62d1,0x30f9,
	0x725c,0x04ac,0x239a,0x1044,0x0566,0x39ab,0x2cc8,0x6709,
	0x6263,0x73f2,0x686e,0x672e,0x4306,0x7c6c,0x0f35,0x60fc};
	unsigned short inv[]={
	0x202d,0xcdfc,0xaa6b,0xe222,0x05a8,0x1439,0xe1a5,0xad5e,
	0xd71a,0xb9fb,0xdf9a,0xfa7e,0x1afd,0x164b,0x9a73,0x63e1,
	0x88fa,0xff75,0x093a,0xe5ba,0x4db1,0xb409,0xc724,0x1ff7,
	0xc5f5,0x066f,0xdd8d,0x64f6,0xeb10,0x6c93,0x9a51,0x93cc,
	0x1d64,0x96a5,0x7336,0x75f1,0x1606,0x8a1c,0xafb9,0x6709,
	0xd47d,0x5814,0xb0c5,0x09be,0x31d7,0xb1ae,0xdad5,0x4983,
	0xeb92,0xe7d2,0x57ac,0x64ea,0x06c9,0x8a37,0xe8cd,0x70b9,
	0x60bf,0x55b3,0xf5ff,0xb57c,0xdfc0,0x6808,0x85c3,0x3ac9};
	unsigned short ns[]={
	0x9f14,0x38aa,0x8f06,0x4960,0xd69e,0x9374,0x61db,0x7bed,
	0x01f1,0x26e2,0x3c08,0x9d49,0x3c23,0x03f8,0xd4b4,0xa327,
	0x4665,0xb2a4,0x8024,0xbf46,0x6034,0x8ee3,0x58ba,0xe8b0,
	0xb0ab,0x195f,0x385a,0x6c07,0x7e38,0x6b22,0xae02,0xc870,
	0x0631,0x65ab,0xb9dc,0xc2df,0x93b2,0xfdfe,0xfe20,0xeb7f,
	0x269f,0xdd20,0x3505,0x836e,0x9f45,0x736e,0x05ae,0x1763,
	0x9b05,0x91af,0x0075,0x579e,0xbf0c,0xdc66,0x677d,0xf41e,
	0xd91a,0x5e1b,0x2516,0xd59f,0xe560,0x4c2d,0xc316,0xf888};
	LNm *p,*b,*n,*q,*r;
	int i,j;

	p=LN_alloc();
	b=LN_alloc(); n=LN_alloc();
	q=LN_alloc(); r=LN_alloc();
	LN_long_set(b,0x10001);

	/* test ext_euclid -- 1 */
	/* set phai */
	LN_set_num_s(p,32,p1);
	LN_set_num_s(q,32,p2);
	LN_multi(p,q,n);	/* n=(p-1)(q-1) */

	LN_set_num_s(q,64,ns);
	if(LN_cmp(n,q)){ /* should be o.k. */
		printf("error: test_ext_euclid : multi(p,q,n)\n");
		return -1;
	}

	LN_mod_inverse(b,n,r);	/* r= b^-1 mod n */
	LN_set_num_s(q,64,inv);
	if(LN_cmp(r,q)){
		printf("error: test_ext_euclid 1\n");
		return -1;
	}
	printf("ext_euclid test 1 .. ok\n");

	/* test ext_euclid -- 2 */
	LN_init_lexp_tv();
	for(j=32;j<LN_MAX;j+=32){
		for(i=0;i<101;i++){
			/* any n is ok */
			LN_set_rand(n,j*2,(unsigned short)rand());
			LN_mod_inverse(b,n,r);	/* r= b^-1 mod n */

			LN_mul_mod(b,r,n,p);
			if((p->top!=1)||(p->num[LN_MAX-1]!=1)){
				printf("error: test_ext_euclid 2\n");
				LN_print(p);
				return -1;
			}
			
			if(!(i%20))
				printf("ext_euclid test 2 (%d bit) .. ok:%d\n",j*16,i);
		}
	}

	LN_free(p);
	LN_free(b); LN_free(n); LN_free(q); LN_free(r);
	return 0;
}

int test_exp_mod(void){
	unsigned short p1[]={	/* p */
	0xcba9,0xe6ba,0xcf43,0x06f1,0xfe88,0xfaa3,0x01fb,0x08d0,
	0xd4ed,0xb47f,0xef57,0xf4d4,0xc2fc,0x045f,0x96af,0xb59d,
	0xa247,0x9769,0xcbae,0xbecb,0xf90c,0xf466,0x05f5,0xfa71,
	0xf931,0xd509,0x0e6f,0xa138,0xf300,0xc800,0xa672,0xe95f};

	unsigned short p2[]={ /* q-1 */
	0xc7f5,0x4c4f,0x4ced,0x39a6,0x54b7,0x3d40,0x74ee,0x6c85,
	0x68a0,0x6ff8,0x76cd,0x42ea,0x227c,0x5d54,0x62d1,0x30f9,
	0x725c,0x04ac,0x239a,0x1044,0x0566,0x39ab,0x2cc8,0x6709,
	0x6263,0x73f2,0x686e,0x672e,0x4306,0x7c6c,0x0f35,0x60fd};

    LNm *a1,*a2;
    LNm *b,*c,*d,*e;
	LNmt_ctx *ctx;
	int i,j;

	b=LN_alloc();
	c=LN_alloc();
	d=LN_alloc();
	e=LN_alloc();

	/* exp_mod test -- 1 */
	LN_init_lexp_tv();
	for(j=32;j<LN_MAX;j+=32){
		for(i=0;i<11;i++){
			/* any n is ok */
		    make_test_LNm(&a1,&a2,j);
			LN_set_rand(b,j*2,(unsigned short)rand());

			LN_exp_mod(b,a2,a1,c);

			LN_long_sub(a2,1);
			LN_exp_mod(b,a2,a1,d);
			LN_mul_mod(d,b,a1,e);
			
			if(LN_cmp(c,e)){
				printf("error: test_exp_mod 1\n");
				LN_print(e);
				return -1;
			}
			if(!(i%2))
				printf("exp_mod test 1 (%d bit) .. ok:%d\n",j*16,i);
			LN_free(a1); LN_free(a2);
		}
	}

	/* exp_mod test -- 2 */
    make_test_LNm(&a1,&a2,64);

	LN_set_num_s(b,32,p1); /* set prime */
	p1[31]&=0xfffe;
	LN_set_num_s(c,32,p1); /* set prime-1 */
	LN_set_num_s(a1,4,p2);
	LN_exp_mod(a1,c,b,e);  /* a1^(p-1) mod p = 1 */
	if((e->top!=1)||(e->num[LN_MAX-1]!=1)){
		printf("error: test_exp_mod 2\n");
		LN_print(e);
		return -1;
	}
	printf("exp_mod test 2 .. ok\n");

	/* test montgomery exp_mod -- 1 */
	/* get montgomery context */
	ctx=LNmt_get_ctx(b);

	LNmt_exp_mod(ctx,a1,c,b,e);  /* a1^(p-1) mod p = 1 */
	if((e->top!=1)||(e->num[LN_MAX-1]!=1)){
		printf("error: montgomery exp_mod 1\n");
		LN_print(e);
		return -1;
	}
	printf("montgomery exp_mod test 1 .. ok\n");

	/* test montgomery exp_mod -- 2 */
	/* exp_mod test */
	printf("standard   : ");
	for(i=0;i<32;i++){
		LN_exp_mod(a1,c,b,e);
		if((e->top==1)&&(e->num[LN_MAX-1]==1)){
			printf("+"); fflush(stdout);
		}else{
			printf("error: test_exp_mod 1\n");
			LN_print(e);
			return -1;
		}
		LN_long_add(a1,1);
	}

	printf("\nmontgomery : ");
	for(i=0;i<32;i++){
		LNmt_exp_mod(ctx,a1,c,b,e);
		if((e->top==1)&&(e->num[LN_MAX-1]==1)){
			printf("+"); fflush(stdout);
		}else{
			printf("error: test_montgomery_exp_mod 1\n");
			LN_print(e);
			return -1;
		}
		LN_long_add(a1,1);
	}
	printf("\nexp_mod test 2 .. ok\n");

	LN_free(b); LN_free(c); LN_free(d); LN_free(e);
	LN_free(a1); LN_free(a2);
	LNmt_ctx_free(ctx);
	return 0;
}

int test_exp_mod2(void){
	unsigned char kd1[] = {
		0x9b,0x01,0x8b,0x87,0x4c,0xde,0xaf,0x17,0x02,0xa3,0xe8,0x35,0x6b,0x0e,0x66,0x46,
		0xb1,0x4c,0xea,0x32,0xfe,0x4a,0x7f,0x9b,0xcc,0x27,0xa6,0x08,0x95,0x4a,0xc3,0xb3,
		0x9f,0x00,0x8e,0x4a,0xc4,0xdc,0xee,0x4c,0xcb,0x2f,0xed,0xa8,0x76,0xfc,0x2b,0x6d,
		0x4b,0x03,0xa0,0x66,0x62,0x8b,0x12,0x03,0xa1,0x21,0x28,0x11,0xe1,0x90,0xf9,0xd5,
		0x55,0xe0,0x49,0xf0,0x83,0xbe,0xe4,0x81,0xce,0xd1,0x28,0x39,0x0c,0xff,0x0e,0x2a,
		0xcd,0xb9,0x00,0xd1,0xc5,0x0d,0xee,0x73,0x97,0x23,0x1d,0x3a,0x02,0x8f,0x6a,0xf8,
		0xcc,0x5e,0x8c,0xa6,0x83,0xcd,0x21,0x38,0xc4,0x7f,0x15,0x29,0x41,0x16,0x9b,0xf8,
		0x4c,0x96,0x00,0x67,0x81,0x64,0x02,0x20,0xdf,0x7a,0xa0,0x78,0x59,0x2a,0x54,0x01
	};
	unsigned char kd2[] = {
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00
	};
	unsigned char kn[] = {
		0xa6,0x64,0xa1,0x1e,0xfb,0x0b,0x2b,0x9d,0xa7,0x1e,0x15,0xcf,0x56,0xf2,0xc5,0xa2,
		0xcf,0x4a,0x6f,0x44,0xda,0x80,0xb4,0x3b,0x27,0xea,0x9b,0x7b,0x9c,0x73,0xb3,0x93,
		0xcb,0x6b,0x2e,0xa8,0x93,0xe6,0x40,0x25,0x46,0x61,0x7c,0x5e,0xd1,0x71,0x24,0x26,
		0xab,0xc9,0xe7,0xec,0x49,0xab,0xda,0xb1,0x3b,0x7a,0x98,0x8a,0xea,0xdc,0x9b,0x7b,
		0x29,0xa5,0x9c,0x12,0x50,0xbb,0xfb,0xdb,0x14,0x34,0x09,0x29,0xca,0xfc,0x06,0x67,
		0xfc,0xd0,0x18,0x5f,0x6e,0x69,0x3e,0x43,0xa4,0xa2,0x1e,0xa8,0x10,0x5a,0x76,0xb6,
		0xcd,0x88,0xff,0x75,0x2a,0x90,0x25,0x1c,0xa3,0xdd,0xfa,0x92,0xbd,0x77,0xc3,0x51,
		0x53,0x1c,0x24,0xdd,0x1e,0x54,0xe5,0xa9,0x5a,0x53,0xff,0x79,0xbf,0xc6,0xb4,0xb7
	};
	unsigned char in[] = {
		0x18,0x1c,0x79,0x9b,0x6a,0x15,0xfb,0x74,0xab,0xf3,0x40,0xf1,0x43,0x50,0x80,0x99,
		0xc3,0x22,0x99,0x40,0xbf,0x1f,0x6f,0x96,0x8a,0x73,0x50,0x4a,0x83,0x13,0x7f,0xf1,
		0x90,0x3d,0x3c,0x73,0xc7,0x4e,0x37,0xdf,0xb4,0x65,0xfe,0xf0,0x2a,0x22,0x56,0xce,
		0xe0,0x1c,0x8b,0xf9,0xca,0x04,0x36,0xa9,0x70,0x46,0x45,0x81,0x98,0xdc,0x2c,0xde,
		0x44,0x9f,0xda,0xf7,0x20,0x9b,0x45,0x51,0x9e,0xe0,0x75,0x02,0xb6,0x25,0x21,0x19,
		0xb0,0xa6,0x77,0xe8,0xe5,0x80,0xa8,0x38,0xae,0xdb,0xe1,0x12,0xed,0x09,0x85,0xa9,
		0xfc,0x7a,0xd0,0xc1,0x0a,0xa8,0xa1,0x52,0xf1,0xf4,0x13,0xb8,0xc3,0x72,0x2e,0x7e,
		0x16,0xf1,0x04,0xed,0x3b,0x5b,0x50,0x51,0x34,0x6b,0x51,0x7b,0xac,0xc1,0xc7,0xff,
	};
	unsigned char m3[] = {
		0x1d,0x4b,0x00,0xbd,0x54,0xcf,0xf0,0x3f,0xc4,0xcc,0x82,0x8d,0x8d,0xfc,0xc5,0xb9,
		0xc4,0xe9,0xf9,0x7b,0x30,0x8a,0x59,0x2d,0xbb,0x38,0xab,0x16,0x44,0x51,0x8d,0x1b,
		0x74,0x65,0xae,0xd5,0xa0,0x97,0xcb,0x2c,0xb5,0xf0,0x2e,0x5f,0x0e,0xc6,0xcc,0xab,
		0x54,0x3d,0xed,0x21,0x26,0x15,0x51,0xf1,0x67,0x70,0xd6,0xdf,0x00,0x6c,0x35,0x8a,
		0xb1,0xb8,0x09,0x4f,0x40,0xbf,0x12,0xfa,0x61,0x64,0x60,0x78,0x22,0xd9,0xbd,0xa1,
		0x87,0x0f,0x41,0x19,0x9a,0xee,0x66,0x26,0xdd,0x79,0xaf,0xf1,0x56,0xce,0x29,0x8f,
		0xe1,0x35,0x38,0x52,0xf9,0x09,0x6d,0x71,0x77,0x93,0x08,0x79,0x59,0x7f,0xe2,0x05,
		0x7a,0xd5,0xf6,0x97,0xfe,0x45,0xeb,0x1d,0x01,0xa7,0x20,0xd8,0x10,0xcc,0xb3,0x82,
		0x6c,0x87,0x7c,0x33,0x58,0xcd,0xb9,0xa9,0xdb,0x07,0xee,0x4e,0xc9,0x19,0xf1,0x8f,
		0x1d,0x27,0x5b,0x65,0x9c,0xf7,0x79,0x44,0xcd,0xee,0x50,0x69,0xdd,0x54,0x21,0x9a,
		0xf3,0xa2,0x32,0x66,0xd5,0x1a,0x01,0x94,0x5c,0x7d,0x10,0xe3,0x2e,0xc1,0x12,0x2d,
		0xbc,0x37,0x41,0xb7,0x9a,0x69,0x64,0xe8,0x4f,0xb2,0xf7,0x07,0xb5,0xb8,0x5c,0x75,
		0x32,0x9b,0xfc,0xe4,0x13,0xa8,0xba,0x81,0xd6,0xc1,0x92,0xea,0x27,0x22,0x70,0x08,
		0xc8,0x0e,0xf0,0x7c,0xe2,0x25,0xca,0x9d,0x36,0x21,0x3f,0xe1,0x62,0xe1,0x59,0x64,
		0xa9,0xfd,0xe7,0xe9,0xa9,0xd3,0xf0,0x70,0x2e,0x85,0x20,0x4f,0x9f,0x7e,0x03,0x22,
		0x07,0xe7,0x61,0x8a,0x87,0x4d,0x0c,0xa0,0x32,0xc5,0x5d,0x8e,0x73,0x04,0x66,0x64,
	};
	unsigned char m4[] = {
		0x25,0xc3,0x9e,0xea,0x3c,0x2d,0x7e,0x2f,0x04,0xe9,0xcc,0xde,0x28,0x18,0xda,0x37,
		0xf8,0xdd,0x19,0x18,0x5f,0x01,0x5c,0x18,0xff,0xff,0xff,0xff,0xdb,0x63,0x2e,0x20,
		0xbb,0x10,0x6f,0x3c,0xe9,0x52,0xdb,0x1a,0x63,0x56,0x32,0x41,0x9e,0x07,0xcf,0xaa,
		0xb6,0x0c,0x71,0xb6,0xa6,0xbf,0x7e,0xdc,0x46,0xc6,0x62,0x8c,0x57,0xcf,0xd1,0x36,
		0xf6,0x7e,0x24,0xe5,0x86,0x12,0x53,0x24,0xfb,0x68,0xa0,0xc8,0xdd,0x03,0x0a,0xa2,
		0x26,0xfe,0x99,0x66,0x5e,0x9f,0xe5,0xde,0x6a,0xde,0x67,0xb3,0xaf,0x07,0x09,0x08,
		0x81,0x7f,0x7b,0x92,0xb8,0x68,0xf7,0x06,0x46,0xf6,0x57,0x8a,0x08,0x44,0x54,0xc7,
		0x0d,0xbb,0x09,0x06,0xc3,0x4f,0x3b,0xf4,0x3e,0x60,0xf6,0x1a,0x69,0xfa,0xd8,0xbe
	};
	unsigned char m5[] = {
		0xc6,0x93,0x1e,0x80,0xb9,0x4a,0x0d,0x61,0x81,0x53,0x66,0x74,0x68,0xfc,0x70,0xba,0xb5,0xcf,0xce,0x97,
		0xce,0xc9,0x55,0x14,0xcc,0x74,0x2a,0x77,0x03,0x0e,0x7c,0x55,0xd7,0xd2,0xe9,0x0a,0xb7,0xe4,0x75,0x83,
		0xd2,0x90,0x27,0x77,0xc7,0x5a,0x52,0xd1,0x4c,0x38,0xf3,0x93,0x11,0xa3,0x06,0x00,0x4a,0xdf,0x0c,0xdf,
		0x76,0xdf,0x0d,0x99,0x59,0xf8,0x5d,0xbd,0x0d,0xb2,0xb2,0xac,0x61,0x9d,0x39,0x3e,0x24,0x62,0xc4,0x71,
		0xfd,0xcb,0x32,0x62,0xd2,0x0d,0x32,0x6d,0xe5,0x09,0xa6,0xb1,0xc3,0xcd,0x01,0x30,0xf4,0x20,0x2c,0xde,
		0x6e,0xee,0x7a,0x38,0xdd,0x7d,0xc9,0xb4,0xe7,0x46,0x44,0x4a,0x76,0x14,0x86,0x37,0x10,0x9d,0x6d,0xae,
		0xec,0x44,0x62,0x0f,0xba,0x58,0xbf,0x73
	};
	unsigned char m6[] = {
		0x80,0x70,0xc5,0xc4,0xff,0xf9,0xb6,0x20,0x28,0x9a,0x67,0x4e,0xb9,0xe6,0x94,0x2d,
		0x73,0x12,0xd7,0xc9,0x80,0xd4,0xe4,0x2b,0x0a,0xb9,0x54,0xac,0xd6,0xfc,0xae,0x5c,
		0xaf,0x2f,0xde,0x92,0x51,0x79,0x38,0x45,0xc6,0x8a,0xe3,0xb8,0xe0,0xc8,0x4c,0xef,
		0x54,0xb7,0x55,0x64,0x7f,0xf5,0x22,0xd2,0xcb,0x93,0x30,0x4c,0x42,0x53,0x2a,0xc2,
		0xdb,0x68,0xeb,0xd3,0x8b,0xf5,0xe2,0xbd,0xaf,0xa6,0x6d,0x0d,0xd0,0xf7,0xa7,0xf5,
		0x3a,0x0c,0x5a,0x17,0xbf,0x02,0xc0,0x39,0x11,0x7a,0x0e,0xf5,0xfd,0x5c,0xd7,0x95,
		0x40,0x3a,0xb1,0xfc,0x81,0x55,0x4d,0x81,0x44,0xf7,0xba,0x93,0xed,0x28,0x92,0xb6,
		0xc2,0x40,0x78,0x41,0x6e,0x5b,0x70,0x6b,0xda,0x87,0x2c,0xc7,0x56,0xed,0x50,0x80,
	};
	unsigned char m7[] = {
		0x71,0xa7,0xf6,0x38,0xec,0xd2,0x06,0x6b,0x6e,0xdc,0xcf,0x08,0xd3,0x5d,0x33,0xe9,
		0x52,0x87,0x6f,0x77,0xd3,0xe1,0x8e,0x9e,0xd7,0xe7,0x49,0xd1,0x48,0x8c,0x5d,0x09,
		0x33,0xae,0xf8,0x3e,0x7e,0xd3,0x3f,0x2d,0x6a,0x82,0x3b,0x36,0x63,0xee,0x98,0x5f,
		0x4a,0x2f,0x08,0x96,0xaa,0x5c,0xb5,0xf0,0x9c,0x5f,0xbb,0x27,0xd4,0xf4,0xde,0x93,
		0xa8,0x1c,0xc5,0x78,0xce,0xae,0x77,0xe1,0x57,0x72,0x3f,0x74,0x57,0x5b,0x3a,0x5a,
		0x43,0x09,0x55,0xfa,0x8c,0x33,0xe3,0xc1,0x4d,0x35,0x64,0xb4,0x1d,0x71,0xb4,0x22,
		0xe1,0x7b,0xf7,0x02,0xcf,0x30,0xaa,0x95,0xb7,0xc6,0xe1,0x3d,0x48,0x51,0xd0,0x25,
		0xa5,0x4b,0x6e,0x51,0xea,0xad,0xd9,0x31,0x5b,0x78,0x6b,0x91,0xff,0x2a,0x6d,0xcc
	};
	unsigned char sb[] = {
		0xcf,0x3f,0xd0,0x44,0x9e,0x1b,0xef,0x4d
	};
	unsigned char ke[] = { 0x01,0x00,0x01 };
	unsigned char cry[320],out[320],*buf;
	int i,j;

	LNm *lb,*lc,*lo,*d,*e,*n,*a,*b,*c,*f,*tmp;

	if((a=LN_alloc())==NULL) return -1;
	if((b=LN_alloc())==NULL) return -1;
	if((c=LN_alloc())==NULL) return -1;
	if((d=LN_alloc())==NULL) return -1;
	if((e=LN_alloc())==NULL) return -1;
	if((f=LN_alloc())==NULL) return -1;
	if((n=LN_alloc())==NULL) return -1;
	if((lb=LN_alloc())==NULL) return -1;
	if((lc=LN_alloc())==NULL) return -1;
	if((lo=LN_alloc())==NULL) return -1;

	/* test square bug */
	LN_set_num_c(a,8,sb);
	LN_sqr(a,b);
	LN_multi(a,a,c);
	if(LN_cmp(b,c)){
		printf("test large num square bug -- error!\n");
		return -1;
	}
	printf("LN_exp_mod (LN_sqr) bug test .. ok\n");

	/* test div mod bug */
	LN_set_num_c(lo,256,m3);
	LN_set_num_c(lc,128,m4);
	LN_set_num_c(n,128,m5);
	LN_div_mod(lo,n,b,c);
	if(LN_cmp(b,lc)){
		printf("test large num div mod bug -- error!\n");
		return -1;
	}
	printf("LN_exp_mod (LN_div_mod) bug test .. ok\n");


	/* test div mod bug 2 */
	LN_set_num_c(lc,128,m6);
	LN_set_num_c(lo,128,m7);
	LN_set_num_c(n,128,kn);
	if(LN_sqr(lc,a)) return -1;	/* tmp = a*b */
	if(LN_div_mod(a,n,b,c)) return -1;  /* ret = tmp mod n */
	if(LN_cmp(c,lo)){
		printf("test large num div mod bug 2 -- error!\n");
		return -1;
	}
	printf("LN_exp_mod (LN_div_mod) bug 2 test .. ok\n");

/*
	if(buf=get_file2buf("d:\\ssl_hssv6_buf.bin",&i)){
		FILE *fp;
		if(fp=fopen("d:\\ssl_hssv_buf.txt","w")){
			for(j=0; j<i; j++){ fprintf(fp,"0x%.2x,",buf[j]);}
			fclose(fp);
		}
	}
*/
	/* [test calc template] set num */
	LN_long_set(a,1);
	LN_set_num_c(d,128,kd1);
	LN_set_num_c(f,128,kd2);
	LN_set_num_c(e,3,ke);
	LN_set_num_c(n,128,kn);
	LN_set_num_c(lb,128,in);

	for(i=0; i<0; i++){
		if(LN_exp_mod(lb,f,n,b)) return -1;
		if(LN_mul_mod(a,b,n,c)) return -1;
		tmp=c; c=a; a=tmp; /* swap */
	}
	if(LN_exp_mod(lb,d,n,b)) return -1;
	if(LN_mul_mod(a,b,n,lc)) return -1;
	LN_get_num_c(lc,128,cry);

/*
	if(LN_cmp(lc,lo)){
		printf("test rsa signature verify -- error!\n");
		return -1;
	}
	if(LN_exp_mod(lc,e,n,lo)) return -1;
	LN_get_num_c(lo,128,out);

	if(LN_cmp(lb,lo)){
		printf("test rsa signature verify -- error!\n");
		return -1;
	}
*/

	LN_free(a); LN_free(b); LN_free(c); LN_free(d); LN_free(e); 
	LN_free(f); LN_free(n); LN_free(lb); LN_free(lc); LN_free(lo); 

	return 0;
}

int test_rsa(void){
	unsigned short sn[]={
	  0xed3e,0x206e,0x179f,0x7c27,0x6d0d,0x834c,0xf64b,0xe264,
	  0x4b9d,0x6f0d,0xbbbd,0x2956,0xdd70,0x3f64,0x7f40,0x660b,
	  0xbe87,0xcb3c,0x656a,0xd501,0xf647,0x1759,0xe62f,0xece6,
	  0x3735,0xadfa,0x037f,0x1c8c,0xee5e,0x00eb,0x9580,0x1fab
	};
	unsigned short se[]={
	  0x0001,0x0001
	};
	unsigned short sd[]={
	  0x1726,0x57cd,0xc65a,0x56a8,0x1639,0x1a55,0xd936,0xc069,
	  0x9f03,0x46c1,0xe54e,0xe908,0xc3e4,0xdf1a,0xb45c,0x1958,
	  0xddea,0x70f7,0xa236,0x9533,0xe67f,0x4aff,0xc4b3,0xa724,
	  0x1dd2,0x1572,0x1e9d,0x01ad,0x0ec0,0x58cb,0x368d,0xe311
	};
	LNm *n,*e,*d;
	LNm *a,*b,*c;
	unsigned char  in[130],cry[130],out[130];

	n=LN_alloc_s(32,sn);
	e=LN_alloc_s(2,se);
	d=LN_alloc_s(32,sd);

	strcpy(in,"This is test.");
	memset(cry,0,130);
	memset(out,0,130);

	a = LN_alloc_c(strlen(in),in);
	b = LN_alloc();
	c = LN_alloc();

	LN_exp_mod(a,e,n,b);
	LN_exp_mod(b,d,n,c);

	if(!LN_cmp(a,c))
		printf("rsa (exp_mod) test .. ok\n");
	else{
		printf("n,e,d\n");
		LN_print(n); LN_print(e); LN_print(d);
		printf("a,b,c\n");
		LN_print(a); LN_print(b); LN_print(c);
		printf("error: a != c ... rsa (exp_mod) test\n");
		return -1;
	}
	/* LN_get_num_c(out,len,to); */
	LN_free(a);	LN_free(b); LN_free(c);
	LN_free(d);	LN_free(e); LN_free(n);
	return 0;
}

int test_prime(void){
	LNm *a1;
	int i;

	for(i=32;i<LN_MAX;i+=32){
		a1=LN_alloc();
		LN_prime(i,a1,1);
		LN_print(a1);
		LN_free(a1);
	}
	printf("test prime .. end\n");
	return 0;
}

int test_sqrt(void){
	LNm *a1,*a2;
    LNm *b,*c,*p;
	int	i,j;

	b = LN_alloc();
	c = LN_alloc();
	p = LN_alloc();

	for(j=16;j<(LN_MAX/2+1);j+=16){
		for(i=0;i<101;i++){
			make_test_LNm(&a1,&a2,j);

			LN_sqrt(a1, b);
			LN_sqr(b,c);
			LN_long_add(b,1);
			LN_sqr(b,p);

			if((LN_zcmp(c,a1)>0)||(LN_zcmp(a1,p)>0)){
				printf("error: test sqrt (%d)\n",i);
				LN_print(a1);
				LN_print(b);
				LN_print(c);
				LN_print(p);
				return -1;
			}

			if(!(i%10)) printf("test sqrt (%d bit) -- ok:%d\n",j*16,i);

			LN_free(a1); LN_free(a2);
		}
	}

	LN_init_lexp_tv();
/*	for(i=0;i<101;i++){
		int j;
		make_test_LNm2(&a1,&a2);
		a1->num[LN_MAX-1] |= 0x1;

		LN_prime(32,p,0);

		LN_sqr_mod(a2,p,b);
		if((j=LN_jacobi(b,p))==-1){
			printf("error: test jacobi (%d) = %d\n",i,j);
			LN_print(b);
			LN_print(a1);
			exit(0);
		}

		if(!(i%10)) printf("test jacobi -- ok:%d\n",i);

		LN_free(a1); LN_free(a2);
	}*/

	for(j=16;j<(LN_MAX/2+1);j+=16){
		for(i=0;i<11;i++){
			LN_prime(j*2,p,0);
			make_test_LNm(&a1,&a2,j);

			LN_sqr_mod(a2,p,b);
			if(LN_mod_sqrt(b,p,c)){
				LN_free(a1); LN_free(a2);
				continue;}

			LN_minus(p,c,b);
			if(LN_cmp(a2,b)&&LN_cmp(a2,c)){
				printf("error: test square root\n");
				LN_print(b);
				LN_print(c);
				LN_print(a2);
				return -1;
			}

			if(!(i%2)) printf("test mod_sqrt (%d bit) -- ok:%d\n",j*16,i);

			LN_free(a1); LN_free(a2);
		}
	}

	LN_free(b); LN_free(c); LN_free(p);
	return 0;
}

