/*
 * large number test functions.
 */
/*
 * Copyright (C) 1998-2002
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <time.h>
#include <sys/types.h>
#ifdef __WINDOWS__
#include <sys/timeb.h>
#else
#include <sys/times.h>
#define TIMES
#endif

#include "large_num.h"

#define TEST_ITR	100000

/* The following if from times(3) man page.  It may need to be changed */
#ifndef HZ
# ifndef CLK_TCK
#  ifndef _BSD_CLK_TCK_ /* FreeBSD hack */
#   define HZ   100.0
#  else /* _BSD_CLK_TCK_ */
#   define HZ ((double)_BSD_CLK_TCK_)
#  endif
# else /* CLK_TCK */
#  define HZ ((double)CLK_TCK)
# endif
#endif

#define START   0
#define STOP    1

/* use same routine as OpenSSL in order to compare with it */
static double Time_F(int s)
{
        double ret;
#ifdef TIMES
        static struct tms tstart,tend;

        if (s == START)
	{
                times(&tstart);
                return(0);
	    }
        else
	{
                times(&tend);
                ret=((double)(tend.tms_utime-tstart.tms_utime))/HZ;
                return((ret < 1e-3)?1e-3:ret);
	    }
#else /* !times() */
        static struct timeb tstart,tend;
        long i;

        if (s == START)
	{
                ftime(&tstart);
                return(0);
	    }
        else
	{
                ftime(&tend);
                i=(long)tend.millitm-(long)tstart.millitm;
                ret=((double)(tend.time-tstart.time))+((double)i)/1000.0;
                return((ret < 0.001)?0.001:ret);
	    }
#endif
    }


void make_test_LNm(LNm **a1,LNm **a2,int num){
    unsigned short n1[256];
    unsigned short n2[256];
    int  i;
    
    for(i=0;i<num;i++)
	n1[i]= (unsigned short)rand() * 3;
    for(i=0;i<num-1;i++)
	n2[i]= (unsigned short)rand();

    n1[0] |= 0x8000;
    n2[0] |= 0x8000;
    *a1=LN_alloc_s(num,n1);
    *a2=LN_alloc_s(num-1,n2);
}

/*=========================================================
  large number calculation test
=========================================================*/
int main(argc,argv)
int  argc;
char **argv;
{
    LNm *a1,*a2;
    LNm *b,*c,*d;
	LNmt_ctx *ctx;

	/*clock_t start, finish;*/
	double tm;
    int	i,j,tsi,k[]={12,16,32,64,96,128};

    b = LN_alloc();
    c = LN_alloc();
    d = LN_alloc();

    j=0;
    do{
	printf("------------ stage %d (%4d bit) ------------\n",j,k[j]*16);
	make_test_LNm(&a1,&a2,k[j]);

	/* add speed test */
	Time_F(START);
	/*start = clock();*/
	for(i=0;i<TEST_ITR;i++){
	    LN_plus(a1,a1,b);
	}
	/*finish = clock();*/
	/*tm = (double)(finish - start) / CLOCKS_PER_SEC;*/
	tm=Time_F(STOP);
	printf("add      %4d + %4d (bit) -> %8.5fms\n",k[j]*16,k[j]*16,tm*1000.0/TEST_ITR);

	/* sub speed test */
	Time_F(START);
	for(i=0;i<TEST_ITR;i++){
	    LN_minus(a1,a2,b);
	}
	tm=Time_F(STOP);
	printf("sub      %4d - %4d (bit) -> %8.5fms\n",k[j]*16,k[j]*15,tm*1000.0/TEST_ITR);
    

	/* multi speed test */
	Time_F(START);
	for(i=0;i<TEST_ITR;i++){
	    LN_multi_std(a1,a1,b);
	}
	tm=Time_F(STOP);
	printf("mul(std) %4d x %4d (bit) -> %8.5fms\n",k[j]*16,k[j]*16,tm*1000.0/TEST_ITR);

	Time_F(START);
	for(i=0;i<TEST_ITR;i++){
	    LN_multi_kara(a1,a1,b);
	}
	tm=Time_F(STOP);
	printf("mul(kar) %4d x %4d (bit) -> %8.5fms\n",k[j]*16,k[j]*16,tm*1000.0/TEST_ITR);

	/* square speed test */
	Time_F(START);
	for(i=0;i<TEST_ITR;i++){
#ifdef __WINDOWS__
	    LN_sqr(a1,b);
#else
	    LN_sqr_std(a1,b);
#endif
	}
	tm=Time_F(STOP);
	printf("sqr(std) %4d x %4d (bit) -> %8.5fms\n",k[j]*16,k[j]*16,tm*1000.0/TEST_ITR);

	Time_F(START);
	for(i=0;i<TEST_ITR;i++){
	    LN_sqr_kara(a1,b);
	}
	tm=Time_F(STOP);
	printf("sqr(kar) %4d x %4d (bit) -> %8.5fms\n",k[j]*16,k[j]*16,tm*1000.0/TEST_ITR);

	/* div mod speed test */
	LN_multi(a1,a1,b);
	Time_F(START);
	for(i=0;i<TEST_ITR;i++){
	    LN_div_mod(b,a1,c,d);
	}
	tm=Time_F(STOP);
	printf("div_mod  %4d / %4d (bit) -> %8.5fms\n",k[j]*32,k[j]*16,tm*1000.0/TEST_ITR);

	/* exp mod speed test */
	tsi=TEST_ITR/(k[j]*k[j]);
	LN_prime(k[j]<<1,b,0);	/* use prime for base */
	LN_div_mod(a1,b,d,c);
	Time_F(START);
	for(i=0;i<tsi;i++){
	    LN_exp_mod(c,a1,b,d);	/* a1^a1 mod b */
	}
	tm=Time_F(STOP);
	printf("exp_mod(std)  %4d ^ %4d (bit) -> %8.5fms\n",k[j]*16,k[j]*16,tm*1000.0/tsi);

	/* exp mod (montgomery) speed test */
	if(j<4){
	    ctx=LNmt_get_ctx(b);
	    Time_F(START);
	    for(i=0;i<tsi;i++){
		LNmt_exp_mod(ctx,c,a1,b,d);  /* a1^(p-1) mod p = 1 */
	    }
	    tm=Time_F(STOP);
	    printf("exp_mod(mot)  %4d ^ %4d (bit) -> %8.5fms\n",k[j]*16,k[j]*16,tm*1000.0/tsi);
	    LNmt_ctx_free(ctx);
	}else{
	    printf("exp_mod(mot)  %4d ^ %4d (bit) -> buffer overflow..\n",k[j]*16,k[j]*16);
	}

	/* prime generation speed test */
	/* this one is not so accurate */
	tsi=5;
	Time_F(START);
	for(i=0;i<tsi;i++){
	    LN_prime(k[j]<<1,b,0);/* don't need to print out */
	}
	tm=Time_F(STOP);
	printf("prime generation %4d(bit) -> %8.5fms\n",k[j]*16,tm*1000.0/tsi);

	/* mod inverse speed test */
	LN_div_mod(a1,b,d,c);	
	tsi=TEST_ITR/(k[j]*4);
	Time_F(START);
	for(i=0;i<tsi;i++){
	    LN_mod_inverse(c,b,d);	/* c^(-1) mod b */
	}
	tm=Time_F(STOP);
	printf("mod inverse %4d^-1 mod %4d (bit) -> %8.5fms\n\n",k[j]*16,k[j]*16,tm*1000.0/tsi);

	LN_free(a1);
	LN_free(a2);
	j++;
    }while(j<6);

    LN_free(b);
    LN_free(c);
    LN_free(d);
}




