/* large_set.c */
/*
 * Modified by National Institute of Informatics in Japan, 2013-2014.
 *
 */
/*
 * Copyright (C) 1998-2002
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "aiconfig.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <aicrypto/large_num.h>
#include <aicrypto/ok_err.h>

/*-----------------------------------------------
  set large number (set a=*s)
-----------------------------------------------*/
int LN_set_num(LNm *a, int size, uint32_t *u32)
{
	int tp = LN_MAX - size;
	LN_UNIT *an;

	if ((size > LN_MAX) || (size < 0)) {
		OK_set_error(ERR_ST_BADPARAM, ERR_LC_LNM, ERR_PT_LNMSET, NULL);
		return -1;
	}
	an = a->num;
	memset(an, 0, sizeof(LN_UNIT) * LN_MAX);
	memcpy(&an[tp], u32, sizeof(uint32_t) * size);
	a->neg = 0;
	a->top = LN_now_top(tp, a);
	return 0;
}

/*-----------------------------------------------
  get large number (get *s=a)
-----------------------------------------------*/
int LN_get_num(LNm *a, int size, uint32_t *u32)
{
	if ((size > LN_MAX) || (size < 0)) {
		OK_set_error(ERR_ST_BADPARAM, ERR_LC_LNM, ERR_PT_LNMSET + 1,
			     NULL);
		return -1;
	}
	memcpy(u32, &(a->num[LN_MAX - size]), sizeof(uint32_t) * size);
	return 0;
}

/*-----------------------------------------------
  set large number
-----------------------------------------------*/
int LN_set_num_c(LNm *a, int byte, unsigned char *c)
{
	LN_UNIT *u;
	int i, j, b;

	u = a->num;
	b = byte;
	j = b & 0x03; /* depending upon sizeof(LN_UNIT) */
	b >>= 2;

	if ((b > LN_MAX) || (b < 0)) {
		OK_set_error(ERR_ST_BADPARAM, ERR_LC_LNM, ERR_PT_LNMSET + 2,
			     NULL);
		return -1;
	}

	memset(u, 0, sizeof(LN_UNIT) * LN_MAX);
	switch (j) { /* depending upon sizeof(LN_UNIT) */
	case 1:
		u[LN_MAX-1-b] = c[0];
		break;
	case 2:
		u[LN_MAX-1-b] = (c[0] << 8) | c[1];
		break;
	case 3:
		u[LN_MAX-1-b] = (c[0] << 16) | (c[1] << 8) | c[2];
		break;
	}

	for (i = LN_MAX - b; i < LN_MAX; i++, j += sizeof(LN_UNIT)) {
		u[i] = (c[j] << 24) | (c[j+1] << 16) | (c[j+2] << 8) | c[j+3];
	}
	a->neg = 0;
	a->top = LN_now_top(LN_MAX - 1 - b, a);
	return 0;
}

/*-----------------------------------------------
  get large number (get *s=a)
-----------------------------------------------*/
int LN_get_num_c(LNm *a, int byte, unsigned char *c)
{
	LN_UNIT *u, o;
	int i, j, b;

	if ((byte >= (LN_MAX * sizeof(LN_UNIT))) || (byte < 0)) {
		OK_set_error(ERR_ST_BADPARAM, ERR_LC_LNM, ERR_PT_LNMSET + 3,
			     NULL);
		return -1;
	}

	u = a->num;
	b = byte;
	j = byte - 1;
	b &= 0x03; /* depending upon sizeof(LN_UNIT) */
	i = LN_MAX - 1;
	while (j > b) { /* depending upon sizeof(LN_UNIT) */
		o = u[i];
		c[j-3] = (unsigned char)(o >> 24);
		c[j-2] = (unsigned char)(o >> 16);
		c[j-1] = (unsigned char)(o >> 8);
		c[j  ] = (unsigned char) o;
		i--;
		j -= sizeof(LN_UNIT);
	};

	o = u[i];
	switch (b) { /* depending upon sizeof(LN_UNIT) */
	case 1:
		c[j]   = (unsigned char) o;;
		break;
	case 2:
		c[j-1] = (unsigned char)(o >> 8);
		c[j  ] = (unsigned char) o;
		break;
	case 3:
		c[j-2] = (unsigned char)(o >> 16);
		c[j-1] = (unsigned char)(o >> 8);
		c[j  ] = (unsigned char) o;
		break;
	}
	return 0;
}

/*-----------------------------------------------
  set large number
-----------------------------------------------*/
int LN_set_num_s(LNm *a, int size, unsigned short *s)
{
	LN_UNIT *u;
	int i, j, sz, hsz;

	if ((size >= (LN_MAX * 2)) || (size < 0)) {
		OK_set_error(ERR_ST_BADPARAM, ERR_LC_LNM, ERR_PT_LNMSET + 4,
			     NULL);
		return -1;
	}

	sz = size;
	hsz = (sz + 1) >> 1;
	j = LN_MAX - hsz;
	u = a->num;
	i = sz & 0x1;

	memset(u, 0, sizeof(LN_UNIT) * LN_MAX);
	if (i) {		/* size is odd */
		u[j] = (uint32_t) * s;
		j++;
	}

	while (j < LN_MAX) {
		u[j] = (s[i] << 16) | (s[i + 1]);
		i += 2;
		j++;
	}
	a->neg = 0;
	a->top = LN_now_top(LN_MAX - hsz, a);
	return 0;
}

/*-----------------------------------------------
  clean large number (set a=0)
-----------------------------------------------*/
void LN_clean(LNm *a)
{
	memset(a->num, 0, sizeof(uint32_t) * LN_MAX);
	a->neg = 0;
	a->top = 0;
}

#if 0
/*
 * This function is very dangerous because almost functions in AiCrypto assume
 * that the size of LNm->num is fixed by LN_MAX.
 */
/*-----------------------------------------------
  size optimize
-----------------------------------------------*/
int LN_reset_size(LNm *a, int s)
{
	LN_UNIT *c;

	if ((c = (LN_UNIT *) malloc(sizeof(LN_UNIT) * s)) == NULL) {
		OK_set_error(ERR_ST_MEMALLOC, ERR_LC_LNM, ERR_PT_LNMSET + 6,
			     NULL);
		return -1;
	}
	if (a->size > s)
		memcpy(c, &(a->num[a->size - s]), sizeof(LN_UNIT) * s);
	else {
		memset(c, 0, sizeof(LN_UNIT) * s);
		memcpy(&c[s - a->size], a->num, sizeof(LN_UNIT) * a->size);
	}
	free(a->num);

	a->num = c;
	a->size = s;
	a->neg = 0;
	a->top = LN_now_top(0, a);
	return 0;
}
#endif
