/* hkdftestfc.c */
/*
 * Copyright (c) 2017 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>

#include <aicrypto/nrg_kdf.h>
#include "hkdftest.h"

#ifdef HKDF_DEBUG
static inline void dump(const uint8_t *buf, const size_t bytes)
{
	int i;

	for (i = 0; i < bytes; i++) {
		printf("%02x ", buf[i]);
	}
	printf("\n");
}
#endif /* HKDF_DEBUG */

int test_A1()
{
	/*
	 * Appendix A.1 of RFC 5869.
	 * Basic test case with SHA-256
	 */
	uint8_t ikm[22] = {
		0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
		0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
		0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	};

	uint8_t salt[13] = {
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x08, 0x09, 0x0a, 0x0b, 0x0c,
	};

	uint8_t info[10] = {
		0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
		0xf8, 0xf9,
	};

	uint8_t exp_prk[32] = {
		0x07, 0x77, 0x09, 0x36, 0x2c, 0x2e, 0x32, 0xdf,
		0x0d, 0xdc, 0x3f, 0x0d, 0xc4, 0x7b, 0xba, 0x63,
		0x90, 0xb6, 0xc7, 0x3b, 0xb5, 0x0f, 0x9c, 0x31,
		0x22, 0xec, 0x84, 0x4a, 0xd7, 0xc2, 0xb3, 0xe5,
	};

	uint8_t exp_okm[42] = {
		0x3c, 0xb2, 0x5f, 0x25, 0xfa, 0xac, 0xd5, 0x7a,
		0x90, 0x43, 0x4f, 0x64, 0xd0, 0x36, 0x2f, 0x2a,
		0x2d, 0x2d, 0x0a, 0x90, 0xcf, 0x1a, 0x5a, 0x4c,
		0x5d, 0xb0, 0x2d, 0x56, 0xec, 0xc4, 0xc5, 0xbf,
		0x34, 0x00, 0x72, 0x08, 0xd5, 0xb8, 0x87, 0x18,
		0x58, 0x65,
	};

	int32_t L = 42;

	size_t prk_len = 32;
	uint8_t prk[prk_len];
	uint8_t okm[L];
	int i;
	int ret;
	hkdf_hash_t hash;

	ret = HKDF_get_hash(OBJ_HASH_SHA256, &hash);
	if (ret != 0) {
		printf("error : test A1 1\n");
		return EXIT_FAILURE;
	} else {
		printf("test A1 1 ok\n");
	}

	ret = HKDF_Extract(&hash, salt, 13, ikm, 22, prk);
	if (ret != 0) {
		printf("error : test A1 2 0\n");
		return EXIT_FAILURE;
	} else {
		printf("test A1 2 ok -- 0\n");
	}
#ifdef HKDF_DEBUG
	dump(prk, prk_len);
#endif /* HKDF_DEBUG */
	for (i = 0; i < prk_len; i++) {
		if (prk[i] != exp_prk[i]) {
			printf("error : test A1 2 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test A1 2 ok -- %d\n", i + 1);
		}
	}

	ret = HKDF_Expand(&hash, prk, prk_len, info, 10, okm, L);
	if (ret != 0) {
		printf("error : test A1 3 0\n");
		return EXIT_FAILURE;
	} else {
		printf("test A1 3 ok -- 0\n");
	}
#ifdef HKDF_DEBUG
	dump(okm, L);
#endif /* HKDF_DEBUG */
	for (i = 0; i < L; i++) {
		if (okm[i] != exp_okm[i]) {
			printf("error : test A1 3 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test A1 3 ok -- %d\n", i + 1);
		}
	}

	memset(okm, 0, L);
	ret = HKDF(&hash, salt, 13, ikm, 22, info, 10, okm, L);
	if (ret != 0) {
		printf("error : test A1 4 0\n");
		return EXIT_FAILURE;
	} else {
		printf("test A1 4 ok -- 0\n");
	}
#ifdef HKDF_DEBUG
	dump(okm, L);
#endif /* HKDF_DEBUG */
	for (i = 0; i < L; i++) {
		if (okm[i] != exp_okm[i]) {
			printf("error : test A1 4 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test A1 4 ok -- %d\n", i + 1);
		}
	}
	return EXIT_SUCCESS;
}

int test_A2()
{
	/*
	 * Appendix A.2 of RFC 5869.
	 * Test with SHA-256 and longer inputs/outputs
	 */
	uint8_t ikm[80] = {
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
		0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
		0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
		0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
		0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
		0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
		0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
		0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
	};

	uint8_t salt[80] = {
		0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
		0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
		0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
		0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
		0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
		0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
		0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
		0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
		0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
		0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
	};

	uint8_t info[80] = {
		0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
		0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
		0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
		0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
		0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
		0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
		0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
		0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
		0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
		0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
	};

	uint8_t exp_prk[32] = {
		0x06, 0xa6, 0xb8, 0x8c, 0x58, 0x53, 0x36, 0x1a,
		0x06, 0x10, 0x4c, 0x9c, 0xeb, 0x35, 0xb4, 0x5c,
		0xef, 0x76, 0x00, 0x14, 0x90, 0x46, 0x71, 0x01,
		0x4a, 0x19, 0x3f, 0x40, 0xc1, 0x5f, 0xc2, 0x44,
	};

	uint8_t exp_okm[82] = {
		0xb1, 0x1e, 0x39, 0x8d, 0xc8, 0x03, 0x27, 0xa1,
		0xc8, 0xe7, 0xf7, 0x8c, 0x59, 0x6a, 0x49, 0x34,
		0x4f, 0x01, 0x2e, 0xda, 0x2d, 0x4e, 0xfa, 0xd8,
		0xa0, 0x50, 0xcc, 0x4c, 0x19, 0xaf, 0xa9, 0x7c,
		0x59, 0x04, 0x5a, 0x99, 0xca, 0xc7, 0x82, 0x72,
		0x71, 0xcb, 0x41, 0xc6, 0x5e, 0x59, 0x0e, 0x09,
		0xda, 0x32, 0x75, 0x60, 0x0c, 0x2f, 0x09, 0xb8,
		0x36, 0x77, 0x93, 0xa9, 0xac, 0xa3, 0xdb, 0x71,
		0xcc, 0x30, 0xc5, 0x81, 0x79, 0xec, 0x3e, 0x87,
		0xc1, 0x4c, 0x01, 0xd5, 0xc1, 0xf3, 0x43, 0x4f,
		0x1d, 0x87,
	};

	int32_t L = 82;
	size_t prk_len = 32;

	uint8_t prk[prk_len];
	uint8_t okm[L];
	int i;
	int ret;
	hkdf_hash_t hash;

	ret = HKDF_get_hash(OBJ_HASH_SHA256, &hash);
	if (ret != 0) {
		printf("error : test A2 1\n");
		return EXIT_FAILURE;
	} else {
		printf("test A2 1 ok\n");
	}

	ret = HKDF_Extract(&hash, salt, 80, ikm, 80, prk);
	if (ret != 0) {
		printf("error : test A2 2 0\n");
		return EXIT_FAILURE;
	} else {
		printf("test A2 2 ok -- 0\n");
	}

#ifdef HKDF_DEBUG
	dump(prk, prk_len);
#endif /* HKDF_DEBUG */
	for (i = 0; i < prk_len; i++) {
		if (prk[i] != exp_prk[i]) {
			printf("error : test A2 2 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test A2 2 ok -- %d\n", i + 1);
		}
	}

	ret = HKDF_Expand(&hash, prk, prk_len, info, 80, okm, L);
	if (ret != 0) {
		printf("error : test A2 3 0\n");
		return EXIT_FAILURE;
	} else {
		printf("test A2 3 ok -- 0\n");
	}
#ifdef HKDF_DEBUG
	dump(okm, L);
#endif /* HKDF_DEBUG */
	for (i = 0; i < L; i++) {
		if (okm[i] != exp_okm[i]) {
			printf("error : test A2 3 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test A2 3 ok -- %d\n", i + 1);
		}
	}

	memset(okm, 0, L);
	ret = HKDF(&hash, salt, 80, ikm, 80, info, 80, okm, L);
	if (ret != 0) {
		printf("error : test A2 4 0\n");
		return EXIT_FAILURE;
	} else {
		printf("test A2 4 ok -- 0\n");
	}
#ifdef HKDF_DEBUG
	dump(okm, L);
#endif /* HKDF_DEBUG */
	for (i = 0; i < L; i++) {
		if (okm[i] != exp_okm[i]) {
			printf("error : test A2 4 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test A2 4 ok -- %d\n", i + 1);
		}
	}

	return EXIT_SUCCESS;
}

int test_A3()
{
	/*
	 * Appendix A.3 of RFC 5869.
	 * Test with SHA-256 and zero-length salt/info
	 */
	uint8_t ikm[22] = {
		0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
		0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
		0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	};

	uint8_t salt[0];

	uint8_t info[0];

	uint8_t exp_prk[32] = {
		0x19, 0xef, 0x24, 0xa3, 0x2c, 0x71, 0x7b, 0x16,
		0x7f, 0x33, 0xa9, 0x1d, 0x6f, 0x64, 0x8b, 0xdf,
		0x96, 0x59, 0x67, 0x76, 0xaf, 0xdb, 0x63, 0x77,
		0xac, 0x43, 0x4c, 0x1c, 0x29, 0x3c, 0xcb, 0x04,
	};

	uint8_t exp_okm[42] = {
		0x8d, 0xa4, 0xe7, 0x75, 0xa5, 0x63, 0xc1, 0x8f,
		0x71, 0x5f, 0x80, 0x2a, 0x06, 0x3c, 0x5a, 0x31,
		0xb8, 0xa1, 0x1f, 0x5c, 0x5e, 0xe1, 0x87, 0x9e,
		0xc3, 0x45, 0x4e, 0x5f, 0x3c, 0x73, 0x8d, 0x2d,
		0x9d, 0x20, 0x13, 0x95, 0xfa, 0xa4, 0xb6, 0x1a,
		0x96, 0xc8,
	};

	int32_t L = 42;
	size_t prk_len = 32;

	uint8_t prk[prk_len];
	uint8_t okm[L];
	int i;
	int ret;
	hkdf_hash_t hash;

	ret = HKDF_get_hash(OBJ_HASH_SHA256, &hash);
	if (ret != 0) {
		printf("error : test A3 1\n");
		return EXIT_FAILURE;
	} else {
		printf("test A3 1 ok\n");
	}

	ret = HKDF_Extract(&hash, salt, 0, ikm, 22, prk);
	if (ret != 0) {
		printf("error : test A3 2 0\n");
		return EXIT_FAILURE;
	} else {
		printf("test A3 2 ok -- 0\n");
	}

#ifdef HKDF_DEBUG
	dump(prk, prk_len);
#endif /* HKDF_DEBUG */
	for (i = 0; i < prk_len; i++) {
		if (prk[i] != exp_prk[i]) {
			printf("error : test A3 2 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test A3 2 ok -- %d\n", i + 1);
		}
	}

	ret = HKDF_Expand(&hash, prk, prk_len, info, 0, okm, L);
	if (ret != 0) {
		printf("error : test A3 3 0\n");
		return EXIT_FAILURE;
	} else {
		printf("test A3 3 ok -- 0\n");
	}
#ifdef HKDF_DEBUG
	dump(okm, L);
#endif /* HKDF_DEBUG */
	for (i = 0; i < L; i++) {
		if (okm[i] != exp_okm[i]) {
			printf("error : test A3 3 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test A3 3 ok -- %d\n", i + 1);
		}
	}

	memset(okm, 0, L);
	ret = HKDF(&hash, salt, 0, ikm, 22, info, 0, okm, L);
	if (ret != 0) {
		printf("error : test A3 4 0\n");
		return EXIT_FAILURE;
	} else {
		printf("test A3 4 ok -- 0\n");
	}
#ifdef HKDF_DEBUG
	dump(okm, L);
#endif /* HKDF_DEBUG */
	for (i = 0; i < L; i++) {
		if (okm[i] != exp_okm[i]) {
			printf("error : test A3 4 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test A3 4 ok -- %d\n", i + 1);
		}
	}

	return EXIT_SUCCESS;
}

int test_A4()
{
	/*
	 * Appendix A.4 of RFC 5869.
	 * Basic test case with SHA-1
	 */
	/* same data as in A.1 */
	uint8_t ikm[11] = {
		0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
		0x0b, 0x0b, 0x0b,
	};

	/* same data as in A.1 */
	uint8_t salt[13] = {
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x08, 0x09, 0x0a, 0x0b, 0x0c,
	};

	/* same data as in A.1 */
	uint8_t info[10] = {
		0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
		0xf8, 0xf9,
	};

	uint8_t exp_prk[20] = {
		0x9b, 0x6c, 0x18, 0xc4, 0x32, 0xa7, 0xbf, 0x8f,
		0x0e, 0x71, 0xc8, 0xeb, 0x88, 0xf4, 0xb3, 0x0b,
		0xaa, 0x2b, 0xa2, 0x43,
	};

	uint8_t exp_okm[42] = {
		0x08, 0x5a, 0x01, 0xea, 0x1b, 0x10, 0xf3, 0x69,
		0x33, 0x06, 0x8b, 0x56, 0xef, 0xa5, 0xad, 0x81,
		0xa4, 0xf1, 0x4b, 0x82, 0x2f, 0x5b, 0x09, 0x15,
		0x68, 0xa9, 0xcd, 0xd4, 0xf1, 0x55, 0xfd, 0xa2,
		0xc2, 0x2e, 0x42, 0x24, 0x78, 0xd3, 0x05, 0xf3,
		0xf8, 0x96,
	};

	int32_t L = 42;
	size_t prk_len = 20;

	uint8_t prk[prk_len];
	uint8_t okm[L];
	int i;
	int ret;
	hkdf_hash_t hash;

	ret = HKDF_get_hash(OBJ_HASH_SHA1, &hash);
	if (ret != 0) {
		printf("error : test A4 1\n");
		return EXIT_FAILURE;
	} else {
		printf("test A4 1 ok\n");
	}

	ret = HKDF_Extract(&hash, salt, 13, ikm, 11, prk);
	if (ret != 0) {
		printf("error : test A4 2 0\n");
		return EXIT_FAILURE;
	} else {
		printf("test A4 2 ok -- 0\n");
	}
#ifdef HKDF_DEBUG
	dump(prk, prk_len);
#endif /* HKDF_DEBUG */
	for (i = 0; i < prk_len; i++) {
		if (prk[i] != exp_prk[i]) {
			printf("error : test A4 2 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test A4 2 ok -- %d\n", i + 1);
		}
	}

	ret = HKDF_Expand(&hash, prk, prk_len, info, 10, okm, L);
	if (ret != 0) {
		printf("error : test A4 3 0\n");
		return EXIT_FAILURE;
	} else {
		printf("test A4 3 ok -- 0\n");
	}
#ifdef HKDF_DEBUG
	dump(okm, L);
#endif /* HKDF_DEBUG */
	for (i = 0; i < L; i++) {
		if (okm[i] != exp_okm[i]) {
			printf("error : test A4 3 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test A4 3 ok -- %d\n", i + 1);
		}
	}

	memset(okm, 0, L);
	ret = HKDF(&hash, salt, 13, ikm, 11, info, 10, okm, L);
	if (ret != 0) {
		printf("error : test A4 4 0\n");
		return EXIT_FAILURE;
	} else {
		printf("test A4 4 ok -- 0\n");
	}
#ifdef HKDF_DEBUG
	dump(okm, L);
#endif /* HKDF_DEBUG */
	for (i = 0; i < L; i++) {
		if (okm[i] != exp_okm[i]) {
			printf("error : test A4 4 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test A4 4 ok -- %d\n", i + 1);
		}
	}

	return EXIT_SUCCESS;
}

int test_A5()
{
	/*
	 * Appendix A.5 of RFC 5869.
	 * Test with SHA-1 and longer inputs/outputs
	 */
	/* same data as in A.2 */
	uint8_t ikm[80] = {
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
		0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
		0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
		0x20, 0x21, 0x22, 0x23, 0x24, 0x25, 0x26, 0x27,
		0x28, 0x29, 0x2a, 0x2b, 0x2c, 0x2d, 0x2e, 0x2f,
		0x30, 0x31, 0x32, 0x33, 0x34, 0x35, 0x36, 0x37,
		0x38, 0x39, 0x3a, 0x3b, 0x3c, 0x3d, 0x3e, 0x3f,
		0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
		0x48, 0x49, 0x4a, 0x4b, 0x4c, 0x4d, 0x4e, 0x4f,
	};

	/* same data as in A.2 */
	uint8_t salt[80] = {
		0x60, 0x61, 0x62, 0x63, 0x64, 0x65, 0x66, 0x67,
		0x68, 0x69, 0x6a, 0x6b, 0x6c, 0x6d, 0x6e, 0x6f,
		0x70, 0x71, 0x72, 0x73, 0x74, 0x75, 0x76, 0x77,
		0x78, 0x79, 0x7a, 0x7b, 0x7c, 0x7d, 0x7e, 0x7f,
		0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
		0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
		0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
		0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
		0xa0, 0xa1, 0xa2, 0xa3, 0xa4, 0xa5, 0xa6, 0xa7,
		0xa8, 0xa9, 0xaa, 0xab, 0xac, 0xad, 0xae, 0xaf,
	};

	/* same data as in A.2 */
	uint8_t info[80] = {
		0xb0, 0xb1, 0xb2, 0xb3, 0xb4, 0xb5, 0xb6, 0xb7,
		0xb8, 0xb9, 0xba, 0xbb, 0xbc, 0xbd, 0xbe, 0xbf,
		0xc0, 0xc1, 0xc2, 0xc3, 0xc4, 0xc5, 0xc6, 0xc7,
		0xc8, 0xc9, 0xca, 0xcb, 0xcc, 0xcd, 0xce, 0xcf,
		0xd0, 0xd1, 0xd2, 0xd3, 0xd4, 0xd5, 0xd6, 0xd7,
		0xd8, 0xd9, 0xda, 0xdb, 0xdc, 0xdd, 0xde, 0xdf,
		0xe0, 0xe1, 0xe2, 0xe3, 0xe4, 0xe5, 0xe6, 0xe7,
		0xe8, 0xe9, 0xea, 0xeb, 0xec, 0xed, 0xee, 0xef,
		0xf0, 0xf1, 0xf2, 0xf3, 0xf4, 0xf5, 0xf6, 0xf7,
		0xf8, 0xf9, 0xfa, 0xfb, 0xfc, 0xfd, 0xfe, 0xff,
	};

	uint8_t exp_prk[20] = {
		0x8a, 0xda, 0xe0, 0x9a, 0x2a, 0x30, 0x70, 0x59,
		0x47, 0x8d, 0x30, 0x9b, 0x26, 0xc4, 0x11, 0x5a,
		0x22, 0x4c, 0xfa, 0xf6,
	};

	uint8_t exp_okm[82] = {
		0x0b, 0xd7, 0x70, 0xa7, 0x4d, 0x11, 0x60, 0xf7,
		0xc9, 0xf1, 0x2c, 0xd5, 0x91, 0x2a, 0x06, 0xeb,
		0xff, 0x6a, 0xdc, 0xae, 0x89, 0x9d, 0x92, 0x19,
		0x1f, 0xe4, 0x30, 0x56, 0x73, 0xba, 0x2f, 0xfe,
		0x8f, 0xa3, 0xf1, 0xa4, 0xe5, 0xad, 0x79, 0xf3,
		0xf3, 0x34, 0xb3, 0xb2, 0x02, 0xb2, 0x17, 0x3c,
		0x48, 0x6e, 0xa3, 0x7c, 0xe3, 0xd3, 0x97, 0xed,
		0x03, 0x4c, 0x7f, 0x9d, 0xfe, 0xb1, 0x5c, 0x5e,
		0x92, 0x73, 0x36, 0xd0, 0x44, 0x1f, 0x4c, 0x43,
		0x00, 0xe2, 0xcf, 0xf0, 0xd0, 0x90, 0x0b, 0x52,
		0xd3, 0xb4,
	};

	int32_t L = 82;
	size_t prk_len = 20;

	uint8_t prk[prk_len];
	uint8_t okm[L];
	int i;
	int ret;
	hkdf_hash_t hash;

	ret = HKDF_get_hash(OBJ_HASH_SHA1, &hash);
	if (ret != 0) {
		printf("error : test A5 1\n");
		return EXIT_FAILURE;
	} else {
		printf("test A5 1 ok\n");
	}

	ret = HKDF_Extract(&hash, salt, 80, ikm, 80, prk);
	if (ret != 0) {
		printf("error : test A5 2 0\n");
		return EXIT_FAILURE;
	} else {
		printf("test A5 2 ok -- 0\n");
	}
#ifdef HKDF_DEBUG
	dump(prk, prk_len);
#endif /* HKDF_DEBUG */
	for (i = 0; i < prk_len; i++) {
		if (prk[i] != exp_prk[i]) {
			printf("error : test A5 2 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test A5 2 ok -- %d\n", i + 1);
		}
	}

	ret = HKDF_Expand(&hash, prk, prk_len, info, 80, okm, L);
	if (ret != 0) {
		printf("error : test A5 3 0\n");
		return EXIT_FAILURE;
	} else {
		printf("test A5 3 ok -- 0\n");
	}
#ifdef HKDF_DEBUG
	dump(okm, L);
#endif /* HKDF_DEBUG */
	for (i = 0; i < L; i++) {
		if (okm[i] != exp_okm[i]) {
			printf("error : test A5 3 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test A5 3 ok -- %d\n", i + 1);
		}
	}

	memset(okm, 0, L);
	ret = HKDF(&hash, salt, 80, ikm, 80, info, 80, okm, L);
	if (ret != 0) {
		printf("error : test A5 4 0\n");
		return EXIT_FAILURE;
	} else {
		printf("test A5 4 ok -- 0\n");
	}
#ifdef HKDF_DEBUG
	dump(okm, L);
#endif /* HKDF_DEBUG */
	for (i = 0; i < L; i++) {
		if (okm[i] != exp_okm[i]) {
			printf("error : test A5 4 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test A5 4 ok -- %d\n", i + 1);
		}
	}

	return EXIT_SUCCESS;
}

int test_A6()
{
	/*
	 * Appendix A.6 of RFC 5869.
	 * Test with SHA-1 and zero-length salt/info
	 */
	/* same data as in A.3 */
	uint8_t ikm[22] = {
		0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
		0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
		0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
	};

	/* same data as in A.3 */
	uint8_t salt[0];

	/* same data as in A.3 */
	uint8_t info[0];

	uint8_t exp_prk[20] = {
		0xda, 0x8c, 0x8a, 0x73, 0xc7, 0xfa, 0x77, 0x28,
		0x8e, 0xc6, 0xf5, 0xe7, 0xc2, 0x97, 0x78, 0x6a,
		0xa0, 0xd3, 0x2d, 0x01,
	};

	uint8_t exp_okm[42] = {
		0x0a, 0xc1, 0xaf, 0x70, 0x02, 0xb3, 0xd7, 0x61,
		0xd1, 0xe5, 0x52, 0x98, 0xda, 0x9d, 0x05, 0x06,
		0xb9, 0xae, 0x52, 0x05, 0x72, 0x20, 0xa3, 0x06,
		0xe0, 0x7b, 0x6b, 0x87, 0xe8, 0xdf, 0x21, 0xd0,
		0xea, 0x00, 0x03, 0x3d, 0xe0, 0x39, 0x84, 0xd3,
		0x49, 0x18,
	};

	int32_t L = 42;
	size_t prk_len = 20;

	uint8_t prk[prk_len];
	uint8_t okm[L];
	int i;
	int ret;
	hkdf_hash_t hash;

	ret = HKDF_get_hash(OBJ_HASH_SHA1, &hash);
	if (ret != 0) {
		printf("error : test A6 1\n");
		return EXIT_FAILURE;
	} else {
		printf("test A6 1 ok\n");
	}

	ret = HKDF_Extract(&hash, salt, 0, ikm, 22, prk);
	if (ret != 0) {
		printf("error : test A6 2 0\n");
		return EXIT_FAILURE;
	} else {
		printf("test A6 2 ok -- 0\n");
	}
#ifdef HKDF_DEBUG
	dump(prk, prk_len);
#endif /* HKDF_DEBUG */
	for (i = 0; i < prk_len; i++) {
		if (prk[i] != exp_prk[i]) {
			printf("error : test A6 2 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test A6 2 ok -- %d\n", i + 1);
		}
	}

	ret = HKDF_Expand(&hash, prk, prk_len, info, 0, okm, L);
	if (ret != 0) {
		printf("error : test A6 3 0\n");
		return EXIT_FAILURE;
	} else {
		printf("test A6 3 ok -- 0\n");
	}
#ifdef HKDF_DEBUG
	dump(okm, L);
#endif /* HKDF_DEBUG */
	for (i = 0; i < L; i++) {
		if (okm[i] != exp_okm[i]) {
			printf("error : test A6 3 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test A6 3 ok -- %d\n", i + 1);
		}
	}

	memset(okm, 0, L);
	ret = HKDF(&hash, salt, 0, ikm, 22, info, 0, okm, L);
	if (ret != 0) {
		printf("error : test A6 4 0\n");
		return EXIT_FAILURE;
	} else {
		printf("test A6 4 ok -- 0\n");
	}
#ifdef HKDF_DEBUG
	dump(okm, L);
#endif /* HKDF_DEBUG */
	for (i = 0; i < L; i++) {
		if (okm[i] != exp_okm[i]) {
			printf("error : test A6 4 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test A6 4 ok -- %d\n", i + 1);
		}
	}

	return EXIT_SUCCESS;
}

int test_A7()
{
	/*
	 * Appendix A.7 of RFC 5869.
	 * Test with SHA-1, salt not provided (defaults to HashLen zero octets),
	 * zero-length info
	 */

	uint8_t ikm[22] = {
		0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
		0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
		0x0c, 0x0c, 0x0c, 0x0c, 0x0c, 0x0c,
	};

	//uint8_t salt[0];

	/* HashLen zero octets ? */
	uint8_t salt[20] = {
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00,
	};

	uint8_t info[0];

	uint8_t exp_prk[20] = {
		0x2a, 0xdc, 0xca, 0xda, 0x18, 0x77, 0x9e, 0x7c,
		0x20, 0x77, 0xad, 0x2e, 0xb1, 0x9d, 0x3f, 0x3e,
		0x73, 0x13, 0x85, 0xdd,
	};

	uint8_t exp_okm[42] = {
		0x2c, 0x91, 0x11, 0x72, 0x04, 0xd7, 0x45, 0xf3,
		0x50, 0x0d, 0x63, 0x6a, 0x62, 0xf6, 0x4f, 0x0a,
		0xb3, 0xba, 0xe5, 0x48, 0xaa, 0x53, 0xd4, 0x23,
		0xb0, 0xd1, 0xf2, 0x7e, 0xbb, 0xa6, 0xf5, 0xe5,
		0x67, 0x3a, 0x08, 0x1d, 0x70, 0xcc, 0xe7, 0xac,
		0xfc, 0x48,
	};

	int32_t L = 42;
	size_t prk_len = 20;

	uint8_t prk[prk_len];
	uint8_t okm[L];
	int i;
	int ret;
	hkdf_hash_t hash;

	ret = HKDF_get_hash(OBJ_HASH_SHA1, &hash);
	if (ret != 0) {
		printf("error : test A7 1\n");
		return EXIT_FAILURE;
	} else {
		printf("test A7 1 ok\n");
	}

	ret = HKDF_Extract(&hash, salt, 20, ikm, 22, prk);
	if (ret != 0) {
		printf("error : test A7 2 0\n");
		return EXIT_FAILURE;
	} else {
		printf("test A7 2 ok -- 0\n");
	}
#ifdef HKDF_DEBUG
	dump(prk, prk_len);
#endif /* HKDF_DEBUG */
	for (i = 0; i < prk_len; i++) {
		if (prk[i] != exp_prk[i]) {
			printf("error : test A7 2 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test A7 2 ok -- %d\n", i + 1);
		}
	}

	ret = HKDF_Expand(&hash, prk, prk_len, info, 0, okm, L);
	if (ret != 0) {
		printf("error : test A7 3 0\n");
		return EXIT_FAILURE;
	} else {
		printf("test A7 3 ok -- 0\n");
	}
#ifdef HKDF_DEBUG
	dump(okm, L);
#endif /* HKDF_DEBUG */
	for (i = 0; i < L; i++) {
		if (okm[i] != exp_okm[i]) {
			printf("error : test A7 3 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test A7 3 ok -- %d\n", i + 1);
		}
	}

	memset(okm, 0, L);
	ret = HKDF(&hash, salt, 20, ikm, 22, info, 0, okm, L);
	if (ret != 0) {
		printf("error : test A7 4 0\n");
		return EXIT_FAILURE;
	} else {
		printf("test A7 4 ok -- 0\n");
	}
#ifdef HKDF_DEBUG
	dump(okm, L);
#endif /* HKDF_DEBUG */
	for (i = 0; i < L; i++) {
		if (okm[i] != exp_okm[i]) {
			printf("error : test A7 4 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test A7 4 ok -- %d\n", i + 1);
		}
	}

	return EXIT_SUCCESS;
}
