/* hmac.c */
/*
 * Copyright (c) 2012-2017 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must 
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2002
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <string.h>
#include <aicrypto/ok_md5.h>
#include <aicrypto/ok_sha1.h>
#include <aicrypto/ok_sha2.h>
#include <aicrypto/nrg_sha3.h>
#include <aicrypto/ok_hmac.h>

/*-----------------------------------------------
    HMAC-MD5 function.(return 128bit char)
-----------------------------------------------*/
void HMAC_MD5(int txtlen,unsigned char *txt,
	      int keylen,unsigned char *key,unsigned char *ret){
	unsigned char k_ipad[64];
	unsigned char k_opad[64];
	unsigned char tk[16];
	MD5_CTX ctx;
	int i;

	if(keylen>64){
		OK_MD5(keylen,key,tk);
		key = tk;
		keylen = 16;
	}

	memset(k_ipad,0,64);
	memset(k_opad,0,64);
	memcpy(k_ipad,key,keylen);
	memcpy(k_opad,key,keylen);

	for(i=0; i<64;i++){
		k_ipad[i] ^= 0x36;
		k_opad[i] ^= 0x5c;
	}

	MD5Init(&ctx);
	MD5Update(&ctx,k_ipad,64);
	MD5Update(&ctx,txt,txtlen);
	MD5Final(ret,&ctx);

	MD5Init(&ctx);
	MD5Update(&ctx,k_opad,64);
	MD5Update(&ctx,ret,16);
	MD5Final(ret,&ctx);
}

/*-----------------------------------------------
    HMAC-SHA1 function.(return 160bit char)
-----------------------------------------------*/
void HMAC_SHA1(int txtlen,unsigned char *txt,
	      int keylen,unsigned char *key,unsigned char *ret){
	unsigned char k_ipad[64];
	unsigned char k_opad[64];
	unsigned char tk[20];
	SHA1_CTX ctx;
	int i;

	if(keylen>64){
		OK_SHA1(keylen,key,tk);
		key = tk;
		keylen = 20;
	}

	memset(k_ipad,0,64);
	memset(k_opad,0,64);
	memcpy(k_ipad,key,keylen);
	memcpy(k_opad,key,keylen);

	for(i=0; i<64;i++){
		k_ipad[i] ^= 0x36;
		k_opad[i] ^= 0x5c;
	}

	SHA1init(&ctx);
	SHA1update(&ctx,k_ipad,64);
	SHA1update(&ctx,txt,txtlen);
	SHA1final(ret,&ctx);

	SHA1init(&ctx);
	SHA1update(&ctx,k_opad,64);
	SHA1update(&ctx,ret,20);
	SHA1final(ret,&ctx);
}

/*-----------------------------------------------
    HMAC-SHA224 function.(return 224bit char)
-----------------------------------------------*/
void HMAC_SHA224(int txtlen,unsigned char *txt,
	      int keylen,unsigned char *key,unsigned char *ret){
	unsigned char k_ipad[SHA256_BLOCKSIZE];
	unsigned char k_opad[SHA256_BLOCKSIZE];
	unsigned char tk[SHA224_DIGESTSIZE];
	SHA256_CTX ctx;
	int i;

	if(keylen>SHA256_BLOCKSIZE){
		OK_SHA224(keylen,key,tk);
		key = tk;
		keylen = SHA224_DIGESTSIZE;
	}

	memset(k_ipad,0,SHA256_BLOCKSIZE);
	memset(k_opad,0,SHA256_BLOCKSIZE);
	memcpy(k_ipad,key,keylen);
	memcpy(k_opad,key,keylen);

	for(i=0; i<SHA256_BLOCKSIZE;i++){
		k_ipad[i] ^= 0x36;
		k_opad[i] ^= 0x5c;
	}

	SHA224init(&ctx);
	SHA224update(&ctx,k_ipad,SHA256_BLOCKSIZE);
	SHA224update(&ctx,txt,txtlen);
	SHA224final(ret,&ctx);

	SHA224init(&ctx);
	SHA224update(&ctx,k_opad,SHA256_BLOCKSIZE);
	SHA224update(&ctx,ret,SHA224_DIGESTSIZE);
	SHA224final(ret,&ctx);
}

/*-----------------------------------------------
    HMAC-SHA256 function.(return 256bit char)
-----------------------------------------------*/
void HMAC_SHA256(int txtlen,unsigned char *txt,
	      int keylen,unsigned char *key,unsigned char *ret){
	unsigned char k_ipad[SHA256_BLOCKSIZE];
	unsigned char k_opad[SHA256_BLOCKSIZE];
	unsigned char tk[SHA256_DIGESTSIZE];
	SHA256_CTX ctx;
	int i;

	if(keylen>SHA256_BLOCKSIZE){
		OK_SHA256(keylen,key,tk);
		key = tk;
		keylen = SHA256_DIGESTSIZE;
	}

	memset(k_ipad,0,SHA256_BLOCKSIZE);
	memset(k_opad,0,SHA256_BLOCKSIZE);
	memcpy(k_ipad,key,keylen);
	memcpy(k_opad,key,keylen);

	for(i=0; i<SHA256_BLOCKSIZE;i++){
		k_ipad[i] ^= 0x36;
		k_opad[i] ^= 0x5c;
	}

	SHA256init(&ctx);
	SHA256update(&ctx,k_ipad,SHA256_BLOCKSIZE);
	SHA256update(&ctx,txt,txtlen);
	SHA256final(ret,&ctx);

	SHA256init(&ctx);
	SHA256update(&ctx,k_opad,SHA256_BLOCKSIZE);
	SHA256update(&ctx,ret,SHA256_DIGESTSIZE);
	SHA256final(ret,&ctx);
}

/*-----------------------------------------------
    HMAC-SHA384 function.(return 384bit char)
-----------------------------------------------*/
void HMAC_SHA384(int txtlen,unsigned char *txt,
	      int keylen,unsigned char *key,unsigned char *ret){
	unsigned char k_ipad[SHA512_BLOCKSIZE];
	unsigned char k_opad[SHA512_BLOCKSIZE];
	unsigned char tk[SHA384_DIGESTSIZE];
	SHA512_CTX ctx;
	int i;

	if(keylen>SHA512_BLOCKSIZE){
		OK_SHA384(keylen,key,tk);
		key = tk;
		keylen = SHA384_DIGESTSIZE;
	}

	memset(k_ipad,0,SHA512_BLOCKSIZE);
	memset(k_opad,0,SHA512_BLOCKSIZE);
	memcpy(k_ipad,key,keylen);
	memcpy(k_opad,key,keylen);

	for(i=0; i<SHA512_BLOCKSIZE;i++){
		k_ipad[i] ^= 0x36;
		k_opad[i] ^= 0x5c;
	}

	SHA384init(&ctx);
	SHA384update(&ctx,k_ipad,SHA512_BLOCKSIZE);
	SHA384update(&ctx,txt,txtlen);
	SHA384final(ret,&ctx);

	SHA384init(&ctx);
	SHA384update(&ctx,k_opad,SHA512_BLOCKSIZE);
	SHA384update(&ctx,ret,SHA384_DIGESTSIZE);
	SHA384final(ret,&ctx);
}

/*-----------------------------------------------
    HMAC-SHA512 function.(return 512bit char)
-----------------------------------------------*/
void HMAC_SHA512(int txtlen,unsigned char *txt,
	      int keylen,unsigned char *key,unsigned char *ret){
	unsigned char k_ipad[SHA512_BLOCKSIZE];
	unsigned char k_opad[SHA512_BLOCKSIZE];
	unsigned char tk[SHA512_DIGESTSIZE];
	SHA512_CTX ctx;
	int i;

	if(keylen>SHA512_BLOCKSIZE){
		OK_SHA512(keylen,key,tk);
		key = tk;
		keylen = SHA512_DIGESTSIZE;
	}

	memset(k_ipad,0,SHA512_BLOCKSIZE);
	memset(k_opad,0,SHA512_BLOCKSIZE);
	memcpy(k_ipad,key,keylen);
	memcpy(k_opad,key,keylen);

	for(i=0; i<SHA512_BLOCKSIZE;i++){
		k_ipad[i] ^= 0x36;
		k_opad[i] ^= 0x5c;
	}

	SHA512init(&ctx);
	SHA512update(&ctx,k_ipad,SHA512_BLOCKSIZE);
	SHA512update(&ctx,txt,txtlen);
	SHA512final(ret,&ctx);

	SHA512init(&ctx);
	SHA512update(&ctx,k_opad,SHA512_BLOCKSIZE);
	SHA512update(&ctx,ret,SHA512_DIGESTSIZE);
	SHA512final(ret,&ctx);
}

/*-----------------------------------------------
    HMAC-SHA3_224 function.(return 224bit char)
-----------------------------------------------*/
void HMAC_SHA3_224(int txtlen,unsigned char *txt,
	      int keylen,unsigned char *key,unsigned char *ret){
	unsigned char k_ipad[SHA3_224_BLOCKSIZE];
	unsigned char k_opad[SHA3_224_BLOCKSIZE];
	unsigned char tk[SHA3_224_DIGESTSIZE];
	SHA3_CTX ctx;
	int i;
	int j;

	if(keylen>SHA3_224_BLOCKSIZE){
		NRG_SHA3_224(keylen,key,tk);
		key = tk;
		keylen = SHA3_224_DIGESTSIZE;
	}

	memset(k_ipad,0,SHA3_224_BLOCKSIZE);
	memset(k_opad,0,SHA3_224_BLOCKSIZE);
	memcpy(k_ipad,key,keylen);
	memcpy(k_opad,key,keylen);

	for(i=0; i<SHA3_224_BLOCKSIZE;i++){
		k_ipad[i] ^= 0x36;
		k_opad[i] ^= 0x5c;
	}

	SHA3_224init(&ctx);
	SHA3_224update(&ctx,k_ipad,SHA3_224_BLOCKSIZE);
	SHA3_224update(&ctx,txt,txtlen);
	SHA3_224final(ret,&ctx);

	SHA3_224init(&ctx);
	SHA3_224update(&ctx,k_opad,SHA3_224_BLOCKSIZE);
	SHA3_224update(&ctx,ret,SHA3_224_DIGESTSIZE);
	SHA3_224final(ret,&ctx);
}

/*-----------------------------------------------
    HMAC-SHA3_256 function.(return 256bit char)
-----------------------------------------------*/
void HMAC_SHA3_256(int txtlen,unsigned char *txt,
	      int keylen,unsigned char *key,unsigned char *ret){
	unsigned char k_ipad[SHA3_256_BLOCKSIZE];
	unsigned char k_opad[SHA3_256_BLOCKSIZE];
	unsigned char tk[SHA3_256_DIGESTSIZE];
	SHA3_CTX ctx;
	int i;

	if(keylen>SHA3_256_BLOCKSIZE){
		NRG_SHA3_256(keylen,key,tk);
		key = tk;
		keylen = SHA3_256_DIGESTSIZE;
	}

	memset(k_ipad,0,SHA3_256_BLOCKSIZE);
	memset(k_opad,0,SHA3_256_BLOCKSIZE);
	memcpy(k_ipad,key,keylen);
	memcpy(k_opad,key,keylen);

	for(i=0; i<SHA3_256_BLOCKSIZE;i++){
		k_ipad[i] ^= 0x36;
		k_opad[i] ^= 0x5c;
	}

	SHA3_256init(&ctx);
	SHA3_256update(&ctx,k_ipad,SHA3_256_BLOCKSIZE);
	SHA3_256update(&ctx,txt,txtlen);
	SHA3_256final(ret,&ctx);

	SHA3_256init(&ctx);
	SHA3_256update(&ctx,k_opad,SHA3_256_BLOCKSIZE);
	SHA3_256update(&ctx,ret,SHA3_256_DIGESTSIZE);
	SHA3_256final(ret,&ctx);
}

/*-----------------------------------------------
    HMAC-SHA3_384 function.(return 384bit char)
-----------------------------------------------*/
void HMAC_SHA3_384(int txtlen,unsigned char *txt,
	      int keylen,unsigned char *key,unsigned char *ret){
	unsigned char k_ipad[SHA3_384_BLOCKSIZE];
	unsigned char k_opad[SHA3_384_BLOCKSIZE];
	unsigned char tk[SHA3_384_DIGESTSIZE];
	SHA3_CTX ctx;
	int i;

	if(keylen>SHA3_384_BLOCKSIZE){
		NRG_SHA3_384(keylen,key,tk);
		key = tk;
		keylen = SHA3_384_DIGESTSIZE;
	}

	memset(k_ipad,0,SHA3_384_BLOCKSIZE);
	memset(k_opad,0,SHA3_384_BLOCKSIZE);
	memcpy(k_ipad,key,keylen);
	memcpy(k_opad,key,keylen);

	for(i=0; i<SHA3_384_BLOCKSIZE;i++){
		k_ipad[i] ^= 0x36;
		k_opad[i] ^= 0x5c;
	}

	SHA3_384init(&ctx);
	SHA3_384update(&ctx,k_ipad,SHA3_384_BLOCKSIZE);
	SHA3_384update(&ctx,txt,txtlen);
	SHA3_384final(ret,&ctx);

	SHA3_384init(&ctx);
	SHA3_384update(&ctx,k_opad,SHA3_384_BLOCKSIZE);
	SHA3_384update(&ctx,ret,SHA3_384_DIGESTSIZE);
	SHA3_384final(ret,&ctx);
}

/*-----------------------------------------------
    HMAC-SHA3_512 function.(return 512bit char)
-----------------------------------------------*/
void HMAC_SHA3_512(int txtlen,unsigned char *txt,
	      int keylen,unsigned char *key,unsigned char *ret){
	unsigned char k_ipad[SHA3_512_BLOCKSIZE];
	unsigned char k_opad[SHA3_512_BLOCKSIZE];
	unsigned char tk[SHA3_512_DIGESTSIZE];
	SHA3_CTX ctx;
	int i;

	if(keylen>SHA3_512_BLOCKSIZE){
		NRG_SHA3_512(keylen,key,tk);
		key = tk;
		keylen = SHA3_512_DIGESTSIZE;
	}

	memset(k_ipad,0,SHA3_512_BLOCKSIZE);
	memset(k_opad,0,SHA3_512_BLOCKSIZE);
	memcpy(k_ipad,key,keylen);
	memcpy(k_opad,key,keylen);

	for(i=0; i<SHA3_512_BLOCKSIZE;i++){
		k_ipad[i] ^= 0x36;
		k_opad[i] ^= 0x5c;
	}

	SHA3_512init(&ctx);
	SHA3_512update(&ctx,k_ipad,SHA3_512_BLOCKSIZE);
	SHA3_512update(&ctx,txt,txtlen);
	SHA3_512final(ret,&ctx);

	SHA3_512init(&ctx);
	SHA3_512update(&ctx,k_opad,SHA3_512_BLOCKSIZE);
	SHA3_512update(&ctx,ret,SHA3_512_DIGESTSIZE);
	SHA3_512final(ret,&ctx);
}
