/* edctestfc.c */
/*
 * Copyright (c) 2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include <string.h>
#include "edc_internal.h"

#ifdef DEBUG_EDC
void dump(const char *name, const uint8_t *buf, const size_t length) {
	size_t i;

	printf("%s: ", name)
	for (i = 0; i < length; i++) {
		printf("%.2X ", buf[i]);
		if (i != 0 && (i % 16) == 15) {
			printf("\n");
		}
	}
	printf("\n");
}
#endif

int test_X25519_derive() {
	size_t key_len = X25519_KEY_LENGTH;

	uint8_t scalar1[] = {
		0xA5, 0x46, 0xE3, 0x6B, 0xF0, 0x52, 0x7C, 0x9D,
		0x3B, 0x16, 0x15, 0x4B, 0x82, 0x46, 0x5E, 0xDD,
		0x62, 0x14, 0x4C, 0x0A, 0xC1, 0xFC, 0x5A, 0x18,
		0x50, 0x6A, 0x22, 0x44, 0xBA, 0x44, 0x9A, 0xC4
	};

	uint8_t u_coordinate1[] = {
		0xE6, 0xDB, 0x68, 0x67, 0x58, 0x30, 0x30, 0xDB,
		0x35, 0x94, 0xC1, 0xA4, 0x24, 0xB1, 0x5F, 0x7C,
		0x72, 0x66, 0x24, 0xEC, 0x26, 0xB3, 0x35, 0x3B,
		0x10, 0xA9, 0x03, 0xA6, 0xD0, 0xAB, 0x1C, 0x4C
	};

	uint8_t output1[] = {
		0xC3, 0xDA, 0x55, 0x37, 0x9D, 0xE9, 0xC6, 0x90,
		0x8E, 0x94, 0xEA, 0x4D, 0xF2, 0x8D, 0x08, 0x4F,
		0x32, 0xEC, 0xCF, 0x03, 0x49, 0x1C, 0x71, 0xF7,
		0x54, 0xB4, 0x07, 0x55, 0x77, 0xA2, 0x85, 0x52
	};

	uint8_t pubkey1[key_len];
	memset(pubkey1, 0, sizeof(pubkey1));
	if (X25519_derive(scalar1, u_coordinate1, pubkey1) < 0) {
		return -1;
	}

#ifdef DEBUG_EDC
	dump("public key1", pubkey1, key_len);
#endif

	if (memcmp(output1, pubkey1, key_len) != 0) {
		return -1;
	}

	uint8_t scalar2[] = {
		0x4b, 0x66, 0xe9, 0xd4, 0xd1, 0xb4, 0x67, 0x3c,
		0x5a, 0xd2, 0x26, 0x91, 0x95, 0x7d, 0x6a, 0xf5,
		0xc1, 0x1b, 0x64, 0x21, 0xe0, 0xea, 0x01, 0xd4,
		0x2c, 0xa4, 0x16, 0x9e, 0x79, 0x18, 0xba, 0x0d
	};

	/* incorrect u_coordinate pointed out by errata
	uint8_t u_coordinate2[] = {
		0xe5, 0x21, 0x0f, 0x12, 0x78, 0x68, 0x11, 0xd3,
		0xf4, 0xb7, 0x95, 0x9d, 0x05, 0x38, 0xae, 0x2c,
		0x31, 0xdb, 0xe7, 0x10, 0x6f, 0xc0, 0x3c, 0x3e,
		0xfc, 0x4c, 0xd5, 0x49, 0xc7, 0x15, 0xa4, 0x93
	};
	*/

	uint8_t u_coordinate2[] = {
		0xe5, 0x21, 0x0f, 0x12, 0x78, 0x68, 0x11, 0xd3,
		0xf4, 0xb7, 0x95, 0x9d, 0x05, 0x38, 0xae, 0x2c,
		0x31, 0xdb, 0xe7, 0x10, 0x6f, 0xc0, 0x3c, 0x3e,
		0xfc, 0x4c, 0xd5, 0x49, 0xc7, 0x15, 0xa4, 0x13
	};

	uint8_t output2[] = {
		0x95, 0xcb, 0xde, 0x94, 0x76, 0xe8, 0x90, 0x7d,
		0x7a, 0xad, 0xe4, 0x5c, 0xb4, 0xb8, 0x73, 0xf8,
		0x8b, 0x59, 0x5a, 0x68, 0x79, 0x9f, 0xa1, 0x52,
		0xe6, 0xf8, 0xf7, 0x64, 0x7a, 0xac, 0x79, 0x57
	};

	uint8_t pubkey2[key_len];
	memset(pubkey2, 0, sizeof(pubkey2));
	if (X25519_derive(scalar2, u_coordinate2, pubkey2) < 0) {
		return -1;
	}

#ifdef DEBUG_EDC
	dump("public key2", pubkey2, key_len);
#endif

	if (memcmp(output2, pubkey2, key_len) != 0) {
		return -1;
	}

	return 0;
}

int test_X448_derive() {
	size_t key_len = X448_KEY_LENGTH;

	uint8_t scalar1[] = {
		0x3d, 0x26, 0x2f, 0xdd, 0xf9, 0xec, 0x8e, 0x88,
		0x49, 0x52, 0x66, 0xfe, 0xa1, 0x9a, 0x34, 0xd2,
		0x88, 0x82, 0xac, 0xef, 0x04, 0x51, 0x04, 0xd0,
		0xd1, 0xaa, 0xe1, 0x21, 0x70, 0x0a, 0x77, 0x9c,
		0x98, 0x4c, 0x24, 0xf8, 0xcd, 0xd7, 0x8f, 0xbf,
		0xf4, 0x49, 0x43, 0xeb, 0xa3, 0x68, 0xf5, 0x4b,
		0x29, 0x25, 0x9a, 0x4f, 0x1c, 0x60, 0x0a, 0xd3
	};

	uint8_t u_coordinate1[] = {
		0x06, 0xfc, 0xe6, 0x40, 0xfa, 0x34, 0x87, 0xbf,
		0xda, 0x5f, 0x6c, 0xf2, 0xd5, 0x26, 0x3f, 0x8a,
		0xad, 0x88, 0x33, 0x4c, 0xbd, 0x07, 0x43, 0x7f,
		0x02, 0x0f, 0x08, 0xf9, 0x81, 0x4d, 0xc0, 0x31,
		0xdd, 0xbd, 0xc3, 0x8c, 0x19, 0xc6, 0xda, 0x25,
		0x83, 0xfa, 0x54, 0x29, 0xdb, 0x94, 0xad, 0xa1,
		0x8a, 0xa7, 0xa7, 0xfb, 0x4e, 0xf8, 0xa0, 0x86
	};

	uint8_t output1[] = {
		0xce, 0x3e, 0x4f, 0xf9, 0x5a, 0x60, 0xdc, 0x66,
		0x97, 0xda, 0x1d, 0xb1, 0xd8, 0x5e, 0x6a, 0xfb,
		0xdf, 0x79, 0xb5, 0x0a, 0x24, 0x12, 0xd7, 0x54,
		0x6d, 0x5f, 0x23, 0x9f, 0xe1, 0x4f, 0xba, 0xad,
		0xeb, 0x44, 0x5f, 0xc6, 0x6a, 0x01, 0xb0, 0x77,
		0x9d, 0x98, 0x22, 0x39, 0x61, 0x11, 0x1e, 0x21,
		0x76, 0x62, 0x82, 0xf7, 0x3d, 0xd9, 0x6b, 0x6f
	};

	uint8_t pubkey1[key_len];
	memset(pubkey1, 0, sizeof(pubkey1));
	if (X448_derive(scalar1, u_coordinate1, pubkey1) < 0) {
		return -1;
	}

#ifdef DEBUG_EDC
	dump("public key1", pubkey1, key_len);
#endif

	if (memcmp(output1, pubkey1, key_len) != 0) {
		return -1;
	}

	uint8_t scalar2[] = {
		0x20, 0x3d, 0x49, 0x44, 0x28, 0xb8, 0x39, 0x93,
		0x52, 0x66, 0x5d, 0xdc, 0xa4, 0x2f, 0x9d, 0xe8,
		0xfe, 0xf6, 0x00, 0x90, 0x8e, 0x0d, 0x46, 0x1c,
		0xb0, 0x21, 0xf8, 0xc5, 0x38, 0x34, 0x5d, 0xd7,
		0x7c, 0x3e, 0x48, 0x06, 0xe2, 0x5f, 0x46, 0xd3,
		0x31, 0x5c, 0x44, 0xe0, 0xa5, 0xb4, 0x37, 0x12,
		0x82, 0xdd, 0x2c, 0x8d, 0x5b, 0xe3, 0x09, 0x5f
	};

	uint8_t u_coordinate2[] = {
		0x0f, 0xbc, 0xc2, 0xf9, 0x93, 0xcd, 0x56, 0xd3,
		0x30, 0x5b, 0x0b, 0x7d, 0x9e, 0x55, 0xd4, 0xc1,
		0xa8, 0xfb, 0x5d, 0xbb, 0x52, 0xf8, 0xe9, 0xa1,
		0xe9, 0xb6, 0x20, 0x1b, 0x16, 0x5d, 0x01, 0x58,
		0x94, 0xe5, 0x6c, 0x4d, 0x35, 0x70, 0xbe, 0xe5,
		0x2f, 0xe2, 0x05, 0xe2, 0x8a, 0x78, 0xb9, 0x1c,
		0xdf, 0xbd, 0xe7, 0x1c, 0xe8, 0xd1, 0x57, 0xdb
	};

	uint8_t output2[] = {
		0x88, 0x4a, 0x02, 0x57, 0x62, 0x39, 0xff, 0x7a,
		0x2f, 0x2f, 0x63, 0xb2, 0xdb, 0x6a, 0x9f, 0xf3,
		0x70, 0x47, 0xac, 0x13, 0x56, 0x8e, 0x1e, 0x30,
		0xfe, 0x63, 0xc4, 0xa7, 0xad, 0x1b, 0x3e, 0xe3,
		0xa5, 0x70, 0x0d, 0xf3, 0x43, 0x21, 0xd6, 0x20,
		0x77, 0xe6, 0x36, 0x33, 0xc5, 0x75, 0xc1, 0xc9,
		0x54, 0x51, 0x4e, 0x99, 0xda, 0x7c, 0x17, 0x9d
	};

	uint8_t pubkey2[key_len];
	memset(pubkey2, 0, sizeof(pubkey2));
	if (X448_derive(scalar2, u_coordinate2, pubkey2) < 0) {
		return -1;
	}

#ifdef DEBUG_EDC
	dump("public key2", pubkey2, key_len);
#endif

	if (memcmp(output2, pubkey2, key_len) != 0) {
		return -1;
	}

	return 0;
}

int test_X25519_derive_with_iteration() {
	size_t key_len = X25519_KEY_LENGTH;

	uint8_t initial[] = {
		0x09, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
	};

	uint8_t pub1[] = {
		0x42, 0x2c, 0x8e, 0x7a, 0x62, 0x27, 0xd7, 0xbc,
		0xa1, 0x35, 0x0b, 0x3e, 0x2b, 0xb7, 0x27, 0x9f,
		0x78, 0x97, 0xb8, 0x7b, 0xb6, 0x85, 0x4b, 0x78,
		0x3c, 0x60, 0xe8, 0x03, 0x11, 0xae, 0x30, 0x79
	};

	uint8_t pub1t[] = {
		0x68, 0x4c, 0xf5, 0x9b, 0xa8, 0x33, 0x09, 0x55,
		0x28, 0x00, 0xef, 0x56, 0x6f, 0x2f, 0x4d, 0x3c,
		0x1c, 0x38, 0x87, 0xc4, 0x93, 0x60, 0xe3, 0x87,
		0x5f, 0x2e, 0xb9, 0x4d, 0x99, 0x53, 0x2c, 0x51
	};

	uint8_t pub1m[] = {
		0x7c, 0x39, 0x11, 0xe0, 0xab, 0x25, 0x86, 0xfd,
		0x86, 0x44, 0x97, 0x29, 0x7e, 0x57, 0x5e, 0x6f,
		0x3b, 0xc6, 0x01, 0xc0, 0x88, 0x3c, 0x30, 0xdf,
		0x5f, 0x4d, 0xd2, 0xd2, 0x4f, 0x66, 0x54, 0x24
	};

	uint8_t scalar[key_len];
	memcpy(scalar, initial, key_len);

	uint8_t u_coordinate[key_len];
	memcpy(u_coordinate, initial, key_len);

	uint8_t pubkey[key_len];
	memset(pubkey, 0, key_len);
	uint8_t *expected_value;
	for (int i = 0; i < 1000000; i++) {
		if (X25519_derive(scalar, u_coordinate, pubkey) < 0) {
			return -1;
		}

		memcpy(u_coordinate, scalar, key_len);
		memcpy(scalar, pubkey, key_len);

		if (i == 0 || i == 999 || i == 999999) {
			switch (i) {
			case 0:
				expected_value = pub1;
				break;

			case 999:
				expected_value = pub1t;
				break;

			case 999999:
				expected_value = pub1m;
				break;

			default:
				return -1;
			}

#ifdef DEBUG_EDC
			printf("%d th iteration\n", i + 1);
			dump("public key", pubkey, key_len);
#endif

			if (memcmp(pubkey, expected_value, key_len) != 0) {
				return -1;
			}
		}
	}

	return 0;
}

int test_X448_derive_with_iteration() {
	size_t key_len = X448_KEY_LENGTH;

	uint8_t initial[] = {
		0x05, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00
	};

	uint8_t pub1[] = {
		0x3f, 0x48, 0x2c, 0x8a, 0x9f, 0x19, 0xb0,
		0x1e, 0x6c, 0x46, 0xee, 0x97, 0x11, 0xd9,
		0xdc, 0x14, 0xfd, 0x4b, 0xf6, 0x7a, 0xf3,
		0x07, 0x65, 0xc2, 0xae, 0x2b, 0x84, 0x6a,
		0x4d, 0x23, 0xa8, 0xcd, 0x0d, 0xb8, 0x97,
		0x08, 0x62, 0x39, 0x49, 0x2c, 0xaf, 0x35,
		0x0b, 0x51, 0xf8, 0x33, 0x86, 0x8b, 0x9b,
		0xc2, 0xb3, 0xbc, 0xa9, 0xcf, 0x41, 0x13
	};

	uint8_t pub1t[] = {
		0xaa, 0x3b, 0x47, 0x49, 0xd5, 0x5b, 0x9d,
		0xaf, 0x1e, 0x5b, 0x00, 0x28, 0x88, 0x26,
		0xc4, 0x67, 0x27, 0x4c, 0xe3, 0xeb, 0xbd,
		0xd5, 0xc1, 0x7b, 0x97, 0x5e, 0x09, 0xd4,
		0xaf, 0x6c, 0x67, 0xcf, 0x10, 0xd0, 0x87,
		0x20, 0x2d, 0xb8, 0x82, 0x86, 0xe2, 0xb7,
		0x9f, 0xce, 0xea, 0x3e, 0xc3, 0x53, 0xef,
		0x54, 0xfa, 0xa2, 0x6e, 0x21, 0x9f, 0x38
	};

	uint8_t pub1m[] = {
		0x07, 0x7f, 0x45, 0x36, 0x81, 0xca, 0xca,
		0x36, 0x93, 0x19, 0x84, 0x20, 0xbb, 0xe5,
		0x15, 0xca, 0xe0, 0x00, 0x24, 0x72, 0x51,
		0x9b, 0x3e, 0x67, 0x66, 0x1a, 0x7e, 0x89,
		0xca, 0xb9, 0x46, 0x95, 0xc8, 0xf4, 0xbc,
		0xd6, 0x6e, 0x61, 0xb9, 0xb9, 0xc9, 0x46,
		0xda, 0x8d, 0x52, 0x4d, 0xe3, 0xd6, 0x9b,
		0xd9, 0xd9, 0xd6, 0x6b, 0x99, 0x7e, 0x37,
	};

	uint8_t scalar[key_len];
	memcpy(scalar, initial, key_len);

	uint8_t u_coordinate[key_len];
	memcpy(u_coordinate, initial, key_len);

	uint8_t pubkey[key_len];
	memset(pubkey, 0, key_len);
	uint8_t *expected_value;
	for (int i = 0; i < 1000000; i++) {
		if (X448_derive(scalar, u_coordinate, pubkey) < 0) {
			return -1;
		}

		memcpy(u_coordinate, scalar, key_len);
		memcpy(scalar, pubkey, key_len);

		if (i == 0 || i == 999 || i == 999999) {
			switch (i) {
			case 0:
				expected_value = pub1;
				break;

			case 999:
				expected_value = pub1t;
				break;

			case 999999:
				expected_value = pub1m;
				break;

			default:
				return -1;
			}

#ifdef DEBUG_EDC
			printf("%d th iteration\n", i + 1);
			dump("public key", pubkey, key_len);
#endif

			if (memcmp(pubkey, expected_value, key_len) != 0) {
				return -1;
			}
		}
	}

	return 0;
}

int test_X25519_Diffie_Hellman() {
	size_t key_len = X25519_KEY_LENGTH;

	uint8_t alice_prvkey[] = {
		0x77, 0x07, 0x6d, 0x0a, 0x73, 0x18, 0xa5, 0x7d,
		0x3c, 0x16, 0xc1, 0x72, 0x51, 0xb2, 0x66, 0x45,
		0xdf, 0x4c, 0x2f, 0x87, 0xeb, 0xc0, 0x99, 0x2a,
		0xb1, 0x77, 0xfb, 0xa5, 0x1d, 0xb9, 0x2c, 0x2a
	};

	uint8_t alice_pubkey[] = {
		0x85, 0x20, 0xf0, 0x09, 0x89, 0x30, 0xa7, 0x54,
		0x74, 0x8b, 0x7d, 0xdc, 0xb4, 0x3e, 0xf7, 0x5a,
		0x0d, 0xbf, 0x3a, 0x0d, 0x26, 0x38, 0x1a, 0xf4,
		0xeb, 0xa4, 0xa9, 0x8e, 0xaa, 0x9b, 0x4e, 0x6a
	};

	uint8_t bob_prvkey[] = {
		0x5d, 0xab, 0x08, 0x7e, 0x62, 0x4a, 0x8a, 0x4b,
		0x79, 0xe1, 0x7f, 0x8b, 0x83, 0x80, 0x0e, 0xe6,
		0x6f, 0x3b, 0xb1, 0x29, 0x26, 0x18, 0xb6, 0xfd,
		0x1c, 0x2f, 0x8b, 0x27, 0xff, 0x88, 0xe0, 0xeb,
	};

	uint8_t bob_pubkey[] = {
		0xde, 0x9e, 0xdb, 0x7d, 0x7b, 0x7d, 0xc1, 0xb4,
		0xd3, 0x5b, 0x61, 0xc2, 0xec, 0xe4, 0x35, 0x37,
		0x3f, 0x83, 0x43, 0xc8, 0x5b, 0x78, 0x67, 0x4d,
		0xad, 0xfc, 0x7e, 0x14, 0x6f, 0x88, 0x2b, 0x4f
	};

	uint8_t shared_secret[] = {
		0x4a, 0x5d, 0x9d, 0x5b, 0xa4, 0xce, 0x2d, 0xe1,
		0x72, 0x8e, 0x3b, 0xf4, 0x80, 0x35, 0x0f, 0x25,
		0xe0, 0x7e, 0x21, 0xc9, 0x47, 0xd1, 0x9e, 0x33,
		0x76, 0xf0, 0x9b, 0x3c, 0x1e, 0x16, 0x17, 0x42
	};

	uint8_t pubkey_a[key_len];
	memset(pubkey_a, 0 , key_len);
	if (X25519_derive(alice_prvkey, x25519_base_point, pubkey_a) < 0) {
		return -1;
	}

#ifdef DEBUG_EDC
	dump("alice's public key", pubkey_a, key_len);
#endif

	if (memcmp(pubkey_a, alice_pubkey, key_len) != 0) {
		return -1;
	}

	uint8_t secret[key_len];
	memset(secret, 0 , key_len);
	if (X25519_derive(alice_prvkey, bob_pubkey, secret) < 0) {
		return -1;
	}

#ifdef DEBUG_EDC
	dump("alice's shared secret", secret, key_len);
#endif

	if (memcmp(secret, shared_secret, key_len) != 0) {
		return -1;
	}

	uint8_t pubkey_b[key_len];
	memset(pubkey_b, 0, key_len);
	if (X25519_derive(bob_prvkey, x25519_base_point, pubkey_b) < 0) {
		return -1;
	}

#ifdef DEBUG_EDC
	dump("bob's public key", pubkey_b, key_len);
#endif

	if (memcmp(pubkey_b, bob_pubkey, key_len) != 0) {
		return -1;
	}

	memset(secret, 0 , key_len);
	if (X25519_derive(bob_prvkey, alice_pubkey, secret) < 0) {
		return -1;
	}

#ifdef DEBUG_EDC
	dump("bob's shared secret", secret, key_len);
#endif

	if (memcmp(secret, shared_secret, X25519_KEY_LENGTH) != 0) {
		return -1;
	}

	return 0;
}

int test_X448_Diffie_Hellman() {
	size_t key_len = X448_KEY_LENGTH;

	uint8_t alice_prvkey[] = {
		0x9a, 0x8f, 0x49, 0x25, 0xd1, 0x51, 0x9f, 0x57,
		0x75, 0xcf, 0x46, 0xb0, 0x4b, 0x58, 0x00, 0xd4,
		0xee, 0x9e, 0xe8, 0xba, 0xe8, 0xbc, 0x55, 0x65,
		0xd4, 0x98, 0xc2, 0x8d, 0xd9, 0xc9, 0xba, 0xf5,
		0x74, 0xa9, 0x41, 0x97, 0x44, 0x89, 0x73, 0x91,
		0x00, 0x63, 0x82, 0xa6, 0xf1, 0x27, 0xab, 0x1d,
		0x9a, 0xc2, 0xd8, 0xc0, 0xa5, 0x98, 0x72, 0x6b
	};

	uint8_t alice_pubkey[] = {
		0x9b, 0x08, 0xf7, 0xcc, 0x31, 0xb7, 0xe3, 0xe6,
		0x7d, 0x22, 0xd5, 0xae, 0xa1, 0x21, 0x07, 0x4a,
		0x27, 0x3b, 0xd2, 0xb8, 0x3d, 0xe0, 0x9c, 0x63,
		0xfa, 0xa7, 0x3d, 0x2c, 0x22, 0xc5, 0xd9, 0xbb,
		0xc8, 0x36, 0x64, 0x72, 0x41, 0xd9, 0x53, 0xd4,
		0x0c, 0x5b, 0x12, 0xda, 0x88, 0x12, 0x0d, 0x53,
		0x17, 0x7f, 0x80, 0xe5, 0x32, 0xc4, 0x1f, 0xa0
	};

	uint8_t bob_prvkey[] = {
		0x1c, 0x30, 0x6a, 0x7a, 0xc2, 0xa0, 0xe2, 0xe0,
		0x99, 0x0b, 0x29, 0x44, 0x70, 0xcb, 0xa3, 0x39,
		0xe6, 0x45, 0x37, 0x72, 0xb0, 0x75, 0x81, 0x1d,
		0x8f, 0xad, 0x0d, 0x1d, 0x69, 0x27, 0xc1, 0x20,
		0xbb, 0x5e, 0xe8, 0x97, 0x2b, 0x0d, 0x3e, 0x21,
		0x37, 0x4c, 0x9c, 0x92, 0x1b, 0x09, 0xd1, 0xb0,
		0x36, 0x6f, 0x10, 0xb6, 0x51, 0x73, 0x99, 0x2d
	};

	uint8_t bob_pubkey[] = {
		0x3e, 0xb7, 0xa8, 0x29, 0xb0, 0xcd, 0x20, 0xf5,
		0xbc, 0xfc, 0x0b, 0x59, 0x9b, 0x6f, 0xec, 0xcf,
		0x6d, 0xa4, 0x62, 0x71, 0x07, 0xbd, 0xb0, 0xd4,
		0xf3, 0x45, 0xb4, 0x30, 0x27, 0xd8, 0xb9, 0x72,
		0xfc, 0x3e, 0x34, 0xfb, 0x42, 0x32, 0xa1, 0x3c,
		0xa7, 0x06, 0xdc, 0xb5, 0x7a, 0xec, 0x3d, 0xae,
		0x07, 0xbd, 0xc1, 0xc6, 0x7b, 0xf3, 0x36, 0x09
	};

	uint8_t shared_secret[] = {
		0x07, 0xff, 0xf4, 0x18, 0x1a, 0xc6, 0xcc, 0x95,
		0xec, 0x1c, 0x16, 0xa9, 0x4a, 0x0f, 0x74, 0xd1,
		0x2d, 0xa2, 0x32, 0xce, 0x40, 0xa7, 0x75, 0x52,
		0x28, 0x1d, 0x28, 0x2b, 0xb6, 0x0c, 0x0b, 0x56,
		0xfd, 0x24, 0x64, 0xc3, 0x35, 0x54, 0x39, 0x36,
		0x52, 0x1c, 0x24, 0x40, 0x30, 0x85, 0xd5, 0x9a,
		0x44, 0x9a, 0x50, 0x37, 0x51, 0x4a, 0x87, 0x9d
	};

	uint8_t pubkey_a[key_len];
	memset(pubkey_a, 0, key_len);
	if (X448_derive(alice_prvkey, x448_base_point, pubkey_a) < 0) {
		return -1;
	}

#ifdef DEBUG_EDC
	dump("alice's public key", pubkey_a, key_len);
#endif

	if (memcmp(pubkey_a, alice_pubkey, key_len) != 0) {
		return -1;
	}

	uint8_t secret[key_len];
	memset(secret, 0, key_len);

	if (X448_derive(alice_prvkey, bob_pubkey, secret) < 0) {
		return -1;
	}

#ifdef DEBUG_EDC
	dump("alice's shared secret", secret, key_len);
#endif

	if (memcmp(secret, shared_secret, key_len) != 0) {
		return -1;
	}

	uint8_t pubkey_b[key_len];
	memset(pubkey_b, 0, key_len);
	if (X448_derive(bob_prvkey, x448_base_point, pubkey_b) < 0) {
		return -1;
	}

#ifdef DEBUG_EDC
	dump("bob's public key", , key_len);
#endif

	if (memcmp(pubkey_b, bob_pubkey, key_len) != 0) {
		return -1;
	}

	memset(secret, 0 , key_len);
	if (X448_derive(bob_prvkey, alice_pubkey, secret) < 0) {
		return -1;
	}

#ifdef DEBUG_EDC
	dump("bob's shared secret", , key_len);
#endif

	if (memcmp(secret, shared_secret, key_len) != 0) {
		return -1;
	}

	return 0;
}
