/* edc_key.c */
/*
 * Copyright (c) 2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include <stdlib.h>
#include <string.h>
#include <aicrypto/key_type.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/nrg_edc.h>
#include "edc_internal.h"

/* For random number generator */
#include <sys/time.h>
#include <aicrypto/ok_rand.h>

Pubkey_X25519 *X25519pubkey_new(void) {
	Pubkey_X25519 *key;

	if ((key = malloc(sizeof(Pubkey_X25519))) == NULL) {
		OK_set_error(ERR_ST_MEMALLOC,
			     ERR_LC_EDC, ERR_PT_EDCKEY + 0, NULL);
		return NULL;
	}
	memset(key, 0, sizeof(Pubkey_X25519));
	key->key_type = KEY_X25519_PUB;
	return key;
}

Prvkey_X25519 *X25519prvkey_new(void) {
	Prvkey_X25519 *key;

	if ((key = malloc(sizeof(Prvkey_X25519))) == NULL) {
		OK_set_error(ERR_ST_MEMALLOC,
			     ERR_LC_EDC, ERR_PT_EDCKEY + 1, NULL);
		return NULL;
	}
	memset(key, 0, sizeof(Prvkey_X25519));
	key->key_type = KEY_X25519_PRV;
	return key;
}

void X25519key_free(Key *key) {
	if (key == NULL) {
		OK_set_error(ERR_ST_NULLPOINTER,
			     ERR_LC_EDC, ERR_PT_EDCKEY + 2, NULL);
		return;
	}

	free(key);
}

int X25519prv_generate(Prvkey_X25519 *prv) {
	int ret = -1;

	if (RAND_init() < 0) {
		goto err;
	}

	if (RAND_bytes((unsigned char *)&(prv->key),
		       (int)sizeof(prv->key)) < 0) {
		goto err;
	}

	prv->size = X25519_KEY_LENGTH;

	ret = 0;

err:
	RAND_cleanup();

	return ret;
}

int X25519prv_2pub(Prvkey_X25519 *prv, Pubkey_X25519 *pub) {
	if (X25519_derive(prv->key, x25519_base_point, pub->key) < 0) {
		OK_set_error(ERR_ST_EDC_DERIVATION_FAILURE, ERR_LC_EDC,
			     ERR_PT_EDCKEY + 3, NULL);
		return -1;
	}

	pub->size = X25519_KEY_LENGTH;

	return 0;
}

Pubkey_X25519 *X25519pub_dup(Pubkey_X25519 *pub) {
	Pubkey_X25519 *new_pub;
	if ((new_pub = X25519pubkey_new()) == NULL) {
		return NULL;
	}

	memcpy(new_pub, pub, sizeof(Pubkey_X25519));

	return new_pub;
}

Prvkey_X25519 *X25519prv_dup(Prvkey_X25519 *prv) {
	Prvkey_X25519 *new_prv;
	if ((new_prv = X25519prvkey_new()) == NULL) {
		return NULL;
	}

	memcpy(new_prv, prv, sizeof(Prvkey_X25519));

	return new_prv;
}

Pubkey_X448 *X448pubkey_new(void) {
	Pubkey_X448 *key;

	if ((key = malloc(sizeof(Pubkey_X448))) == NULL) {
		OK_set_error(ERR_ST_MEMALLOC,
			     ERR_LC_EDC, ERR_PT_EDCKEY + 4, NULL);
		return NULL;
	}
	memset(key, 0, sizeof(Pubkey_X448));
	key->key_type = KEY_X448_PUB;
	return key;
}

Prvkey_X448 *X448prvkey_new(void) {
	Prvkey_X448 *key;

	if ((key = malloc(sizeof(Prvkey_X448))) == NULL) {
		OK_set_error(ERR_ST_MEMALLOC,
			     ERR_LC_EDC, ERR_PT_EDCKEY + 5, NULL);
		return NULL;
	}
	memset(key, 0, sizeof(Prvkey_X448));
	key->key_type = KEY_X448_PRV;
	return key;
}

void X448key_free(Key *key) {
	if (key == NULL) {
		OK_set_error(ERR_ST_NULLPOINTER,
			     ERR_LC_EDC, ERR_PT_EDCKEY + 6, NULL);
		return;
	}

	free(key);
}

int X448prv_generate(Prvkey_X448 *prv) {
	int ret = -1;

	if (RAND_init() < 0) {
		goto err;
	}

	if (RAND_bytes((unsigned char *)&(prv->key),
		       (int)sizeof(prv->key)) < 0) {
		goto err;
	}

	prv->size = X448_KEY_LENGTH;

	ret = 0;

err:
	RAND_cleanup();

	return ret;
}

int X448prv_2pub(Prvkey_X448 *prv, Pubkey_X448 *pub) {
	if (X448_derive(prv->key, x448_base_point, pub->key) < 0) {
		OK_set_error(ERR_ST_EDC_DERIVATION_FAILURE, ERR_LC_EDC,
			     ERR_PT_EDCKEY + 7, NULL);
		return -1;
	}

	pub->size = X448_KEY_LENGTH;

	return 0;
}

Pubkey_X448 *X448pub_dup(Pubkey_X448 *pub) {
	Pubkey_X448 *new_pub;
	if ((new_pub = X448pubkey_new()) == NULL) {
		return NULL;
	}

	memcpy(new_pub, pub, sizeof(Pubkey_X448));

	return new_pub;
}

Prvkey_X448 *X448prv_dup(Prvkey_X448 *prv) {
	Prvkey_X448 *new_prv;
	if ((new_prv = X448prvkey_new()) == NULL) {
		return NULL;
	}

	memcpy(new_prv, prv, sizeof(Prvkey_X448));

	return new_prv;
}
