/* edc.c */
/*
 * Copyright (c) 2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include <aicrypto/ok_err.h>
#include <decaf.h>
#include "edc_internal.h"

uint8_t x25519_base_point[X25519_KEY_LENGTH] = { 9 };
uint8_t x448_base_point[X448_KEY_LENGTH] = { 5 };

int X25519_derive(uint8_t *prv, uint8_t *base, uint8_t *secret) {
	decaf_error_t decaf_ret;
	if ((decaf_ret = decaf_x25519(secret, base, prv)) == DECAF_FAILURE) {
		OK_set_error(ERR_ST_EDC_DERIVATION_FAILURE, ERR_LC_EDC,
			     ERR_PT_EDC, NULL);
		return -1;
	}

	return 0;
}

int X25519_generate_shared_secret(Prvkey_X25519 *my_prv,
				  Pubkey_X25519 *peer_pub, uint8_t *secret) {
	if (X25519_derive(my_prv->key, peer_pub->key, secret) < 0) {
		return -1;
	}

	/*
	 * RFC7748 6.1.  Curve25519
	 *
	 *                         Both MAY check, without leaking extra
	 *    information about the value of K, whether K is the all-zero value and
	 *    abort if so (see below).  Alice and Bob can then use a key-derivation
	 *    function that includes K, K_A, and K_B to derive a symmetric key.
	 *
	 *    The check for the all-zero value results from the fact that the
	 *    X25519 function produces that value if it operates on an input
	 *    corresponding to a point with small order, where the order divides
	 *    the cofactor of the curve (see Section 7).  The check may be
	 *    performed by ORing all the bytes together and checking whether the
	 *    result is zero, as this eliminates standard side-channels in software
	 *    implementations.
	 */
	uint8_t result = 0;
	for (size_t i = 0; i < X25519_KEY_LENGTH; i++) {
		result |= secret[i];
	}

	if (result == 0) {
		OK_set_error(ERR_ST_EDC_ALL_ZERO_VALUE, ERR_LC_EDC,
			     ERR_PT_EDC + 1, NULL);
		return -1;
	}

	return 0;
}

int X448_derive(uint8_t *prv, uint8_t *base, uint8_t *secret) {
	decaf_error_t decaf_ret;
	if ((decaf_ret = decaf_x448(secret, base, prv)) == DECAF_FAILURE) {
		OK_set_error(ERR_ST_EDC_DERIVATION_FAILURE, ERR_LC_EDC,
			     ERR_PT_EDC + 2, NULL);
		return -1;
	}

	return 0;
}

int X448_generate_shared_secret(Prvkey_X448 *my_prv, Pubkey_X448 *peer_pub,
				uint8_t *secret) {
	if (X448_derive(my_prv->key, peer_pub->key, secret) < 0) {
		return -1;
	}

	/*
	 * RFC7748 6.2.  Curve448
	 *
	 *    As with X25519, both sides MAY check, without leaking extra
	 *    information about the value of K, whether the resulting shared K is
	 *    the all-zero value and abort if so.
	 */
	uint8_t result = 0;
	for (size_t i = 0; i < X448_KEY_LENGTH; i++) {
		result |= secret[i];
	}

	if (result == 0) {
		OK_set_error(ERR_ST_EDC_ALL_ZERO_VALUE, ERR_LC_EDC,
			     ERR_PT_EDC + 3, NULL);
		return -1;
	}

	return 0;
}
