/* ecdsatestfc.c - test functions for ECDSA module. */
/*
 * Modified by National Institute of Informatics in Japan, 2012-2018.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_ecc.h>
#include <aicrypto/ok_ecdsa.h>
#include <aicrypto/ok_tool.h>

#define TEST_ECDSAKEY		"ecdsa_rfc5915.key"
#define TEST_ECDSAKEY0		"ecdsa_rfc5915.key~"
#ifndef PATH
# define PATH	"."
#endif

/* test/getfpath.c */
char *get_fpath(char *path, char *filename);

int _test_ecdsa_signature(int hash_algo);


int test_ecdsa_signature_sha1(){
    printf("ECDSA signature with SHA1\n");
    return _test_ecdsa_signature(OBJ_HASH_SHA1);
}

int test_ecdsa_signature_sha224(){
    printf("ECDSA signature with SHA224\n");
    return _test_ecdsa_signature(OBJ_HASH_SHA224);
}

int test_ecdsa_signature_sha256(){
    printf("ECDSA signature with SHA256\n");
    return _test_ecdsa_signature(OBJ_HASH_SHA256);
}

int test_ecdsa_signature_sha384(){
    printf("ECDSA signature with SHA384\n");
    return _test_ecdsa_signature(OBJ_HASH_SHA384);
}

int test_ecdsa_signature_sha512(){
    printf("ECDSA signature with SHA512\n");
    return _test_ecdsa_signature(OBJ_HASH_SHA512);
}

int _test_ecdsa_signature(int hash_algo){
	ECParam *E;
	Prvkey_ECDSA *prv;
	Pubkey_ECDSA *pub;
	unsigned char *buf, *dgst;
	char txt[]="This is sample text for generating digest data used elliptic curve signature.\nHmmmm..\n";
	int sigtype;
	int curve;
	int	i,j,len;

	switch(hash_algo){
	case OBJ_HASH_SHA1:
		/* SHA-1 and prim256v1: accoding to naregi-ca-2.3.4 test. */
		curve=ECP_X962_prime256v1;
		break;
	case OBJ_HASH_SHA224:
		curve=ECP_secp224r1;
		break;
	case OBJ_HASH_SHA256:
		curve=ECP_X962_prime256v1;
		break;
	case OBJ_HASH_SHA384:
		curve=ECP_secp384r1;
		break;
	case OBJ_HASH_SHA512:
		curve=ECP_secp521r1;
		break;
	}

	E=ECPm_get_std_parameter(curve);

/* 	E=ECPm_gen_parameter(320); */

	sigtype = select_sigalgo(KEY_ECDSA_PUB, hash_algo);
	for(i=0;i<11;i++){
		prv=ECDSAprvkey_new();
		pub=ECDSApubkey_new();
		
		/* generating ecdsa private key */
		ECDSAprv_generate(E,prv);
		/*LN_print(prv->k);
		  ECp_print(prv->W);*/

		/* generating ecdsa signature */
		ECDSAprv_2pub(prv, pub);
		dgst=OK_do_digest(hash_algo,txt,strlen(txt)-i,NULL,&len);
		buf=ECDSA_get_signature(prv, dgst, len, &j);
#if 0
		{
			Prvkey_ECDSA *caprv;
			Cert *ct;
			if((ct=Cert_read_file("ca.cer"))==NULL){
				printf("test x509 cert read file (3) -- error!\n");
				return -1;
			}
			caprv = ASN1_read_ecdsaprv(ASN1_read_der("ecca.key"));

			ct->pubkey = (Key*)pub;
			ct->pubkey_algo = pub->key_type;

			OK_set_cert_sig_algo(sigtype);
			free(ct->signature);
			ct->signature= NULL;
			if((ct->der=Cert_toDER(ct,(Key*)caprv,NULL,&j))==NULL){
				printf("cannot create DER encoding with cert -- error!\n");
				return -1;
			}
			ASN1_write_der(ct->der,"ecca.cer");
			//ASN1_write_der(prv->der,"ecusr.key");
		}
#endif
		/* verifying ecdsa signature */
		if(ECDSA_vfy_signature(pub,dgst,len,buf)){
			printf("error : ECDSA signature test (%d)\n",i);
			return -1;
		}
		printf("ECDSA signature -- ok:%d\n",i);

		free(dgst);
		free(buf);
		ECDSAkey_free((Key*)pub);
		ECDSAkey_free((Key*)prv);
	}
	ECPm_free(E);	
	return 0;
}

int test_ecdsakey_der(){
	unsigned char *der=NULL,*der2=NULL;
	Prvkey_ECDSA *prv=NULL;
	Pubkey_ECDSA *pub=NULL;
	int i,j,err=-1;
	char *fp_ECDSAKEY = get_fpath(PATH, TEST_ECDSAKEY);

	/* read & write ECDSA key */
	for(j=0;j<11;j++){
		/* read ECDSA Privatekey DER */
		if ((der = ASN1_read_der(fp_ECDSAKEY)) == NULL) goto done;

		if((prv=ASN1_read_ecdsaprv(der))==NULL){
			printf("ECDSA : decode private key -- error!\n");
			goto done;
		}

		/* write EC Param DER */
		if((der2=ECDSAprv_toDER(prv,NULL,&i))==NULL){
			printf("ECDSA : encode private key -- error!\n");
			goto done;
		}
		if(ASN1_write_der(der2,TEST_ECDSAKEY0)) goto done;

		if((j%5)==0) printf("ECDSA : decode & encode private key  -- ok : %d\n",j);

		/* compare two parameters */
		/*
		 * NOTE: The ECDSAprv_toDER() function now follows RFC 5915.
		 *  Therefore, the following test always fails unless reading 
		 *  a key following RFC 5915.
		 */
		if(memcmp(der,der2,i)){
			printf("ECDSA : compare DER parameters der==der2 -- error!\n");
			goto done;
		}
		if((j%5)==0) printf("ECDSA : compare DER binary der==der2 -- ok : %d\n",j);

		free(der2); der2=NULL;
		ECDSAkey_free((Key*)prv); prv=NULL; der=NULL; /* it frees "der" memory */
	}

	/* compare ECDSA keys */
	for(j=0;j<11;j++){
		if((pub=ECDSApubkey_new())==NULL) goto done;
		if ((der = ASN1_read_der(fp_ECDSAKEY)) == NULL) goto done;

		if((prv=ASN1_read_ecdsaprv(der))==NULL){
			printf("ECDSA : decode private key -- error!\n");
			goto done;
		}
		ECDSAprv_2pub(prv,pub);

		if(Key_cmp((Key*)prv,(Key*)prv)){
			printf("ECDSA : compare private key -- error!\n");
			goto done;
		}
		if(Key_cmp((Key*)pub,(Key*)pub)){
			printf("ECDSA : compare public key -- error!\n");
			goto done;
		}
		if(!Key_cmp((Key*)prv,(Key*)pub)){
			printf("ECDSA : compare prv & pub key -- error!\n");
			goto done;
		}
		if(Key_pair_cmp((Key*)prv,(Key*)pub)){
			printf("ECDSA : compare prv & pub key -- error!\n");
			goto done;
		}

		if((j%5)==0) printf("ECDSA : compare prv & pub keys -- ok : %d\n",j);

		Key_free((Key*)pub);
		Key_free((Key*)prv); /* it frees "der" memory */
		prv=NULL; der=NULL;
	}

	err=0;
done:
	if(der2) free(der2);
	if((prv==NULL)&&(der)) free(der);
	ECDSAkey_free((Key*)prv); /* it frees "der" memory */
	if (fp_ECDSAKEY) free(fp_ECDSAKEY);
	return err;
}

