/*
 * ECDSA speed test functions.
 */
/*
 * Copyright (C) 1998-2002
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <time.h>
#include <sys/types.h>
#include <sys/times.h>

#include "ok_asn1.h"
#include "ok_tool.h"

#include "ok_ecc.h"
#include "ok_ecdsa.h"

#define TEST_ITR	200
#define TIMES

/* The following if from times(3) man page.  It may need to be changed */
#ifndef HZ
# ifndef CLK_TCK
#  ifndef _BSD_CLK_TCK_ /* FreeBSD hack */
#   define HZ   100.0
#  else /* _BSD_CLK_TCK_ */
#   define HZ ((double)_BSD_CLK_TCK_)
#  endif
# else /* CLK_TCK */
#  define HZ ((double)CLK_TCK)
# endif
#endif

#define START   0
#define STOP    1

static double Time_F(int s)
{
        double ret;
#ifdef TIMES
        static struct tms tstart,tend;

        if (s == START)
	{
                times(&tstart);
                return(0);
	    }
        else
	{
                times(&tend);
                ret=((double)(tend.tms_utime-tstart.tms_utime))/HZ;
                return((ret < 1e-3)?1e-3:ret);
	    }
#else /* !times() */
        static struct timeb tstart,tend;
        long i;

        if (s == START)
	{
                ftime(&tstart);
                return(0);
	    }
        else
	{
                ftime(&tend);
                i=(long)tend.millitm-(long)tstart.millitm;
                ret=((double)(tend.time-tstart.time))+((double)i)/1000.0;
                return((ret < 0.001)?0.001:ret);
	    }
#endif
    }


/*=========================================================
  large number calculation test
=========================================================*/
int main(argc,argv)
int  argc;
char **argv;
{
    ECParam *E;
    LNm *f,*c,*d;
    double tm;
    int	i,j,tsi,len,k[]={12};
    int	ec[]={ECP_TYPE_X9_62_192};
    unsigned char *dgst;
    char txt[]="This is sample text for generating digest data used elliptic curve signature.";

    Prvkey_ECDSA *prv;
    Pubkey_ECDSA *pub;


    prv = ECDSAprvkey_new();
    pub = ECDSApubkey_new();

    dgst=OK_do_digest(OBJ_HASH_SHA1,txt,strlen(txt),NULL,&len);

    c  = LN_alloc();
    d  = LN_alloc();
    f  = LN_alloc_c(len,dgst);


    j=0;
    do{
	E=ECPm_get_std_parameter(ec[j]);

	printf("ECC : y^2 = x^3 + a*x + b (mod p) (%d bit)\n",k[j]*16);
	printf("p : "); LN_print(E->p);
	printf("a : "); LN_print(E->a);
	printf("b : "); LN_print(E->b);
	printf("Base point G on E\n"); ECp_print(E->G);
	printf("Order of point G on E\n");
	printf("n : "); LN_print(E->n);
	printf("\n");

	/* private key generate speed */
	Time_F(START);
	for(i=0;i<TEST_ITR;i++){
	    ECDSAprv_generate(E,prv);
	}
	tm=Time_F(STOP);
	printf("ECDSA private key generation (%d bit) -> %8.5fms\n",k[j]*16,tm*1000.0/TEST_ITR);

	/* signing signature speed */
	/* use inside loop of generating a signature */
	Time_F(START);
	for(i=0;i<TEST_ITR;i++){
	    ECDSA_sig_in(E,prv,f,c,d);
	}
	tm=Time_F(STOP);
	printf("ECDSA signature generation (%d bit) -> %8.5fms\n",k[j]*16,tm*1000.0/TEST_ITR);

	/* verifying signature speed */
        ECDSAprv_2pub(prv, pub);
	Time_F(START);
	for(i=0;i<TEST_ITR;i++){
	    ECDSA_vfy_in(E,pub,f,c,d);
	}
	tm=Time_F(STOP);
	printf("ECDSA signature verification (%d bit) -> %8.5fms\n",k[j]*16,tm*1000.0/TEST_ITR);


	j++;
	ECPm_free(E);
    }while(j<1);

    LN_free(c);
    LN_free(d);
    LN_free(f);
    ECDSAkey_free((Key*)pub);
    ECDSAkey_free((Key*)prv);
}




