/* ecdsa_asn1.c */
/*
 * Modified by National Institute of Informatics in Japan, 2013-2018.
 *
 */
/*
 * Copyright (C) 1998-2002
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_ecc.h>
#include <aicrypto/ok_ecdsa.h>
static void conv_octet_to_bitstring(unsigned char *, int *);

/*-----------------------------------
	ECDSA Private Key to DER
------------------------------------*/
unsigned char *ECDSAprv_toDER(Prvkey_ECDSA *prv,unsigned char *buf,int *ret_len){
	unsigned char *cp,*ret;
	int	i,j;

	if(buf==NULL){
		if((i=ECDSAprv_estimate_der_size(prv))<=0)
			return NULL;

		if((ret=(unsigned char*)malloc(i))==NULL){
			OK_set_error(ERR_ST_MEMALLOC,ERR_LC_ECDSA,ERR_PT_ECDSAASN,NULL);
			return NULL;
		}
		memset(ret,0,i);
	}else{
		ret=buf;
	}

	/*
	 * The syntax for ECDSA private key follows RFC 5915.
	 *
	 * NOTE: The ECPrivateKey defined in AiCrypto 3.x or earlier is
	 *   incompatible with RFC 5915. Which specification previous
	 *   implementation follows is unknown.
	 */

	/* version	INTEGER { ecPrivkeyVer1(1) } (ecPrivkeyVer1) */
	ASN1_set_integer(prv->version,ret,&i);
	cp = ret+i;

	/* privateKey	OCTET STRING */
	if(ASN1_LNm2OS(prv->k,cp,&j)) goto error;
	cp += j;
	i += j;

	if(prv->E==NULL){
		OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_ECDSA,ERR_PT_ECDSAASN,NULL);
		goto error;
	}

	/* parameters [0] ECParameters {{ NamedCurve }} OPTIONAL */
	if(ECPm_toDER(prv->E,cp,&j)==NULL) goto error;
	ASN1_set_explicit(j, 0, cp, &j);
	cp+= j; i+=j;

	/* publicKey  [1] BIT STRING OPTIONAL  */
	if(ECPm_DER_ecpoint(prv->W,cp,&j)) goto error;
	/* NOTE: ECPoint ::= OCTET STRING */
	conv_octet_to_bitstring(cp, &j);
	ASN1_set_explicit(j, 1, cp, &j);
	i+=j;

	ASN1_set_sequence(i,ret,ret_len);

	return ret;
error:
	if(ret!=buf) free(ret);
	return NULL;
}

/*-----------------------------------
	ECDSA Private Key to DER
------------------------------------*/
unsigned char *ECDSApub_toDER(Pubkey_ECDSA *pub,unsigned char *buf,int *ret_len){
	unsigned char *cp,*ret;
	int	i,j;

	if(buf==NULL){
		if((i=ECDSApub_estimate_der_size(pub))<=0)
			return NULL;

		if((ret=(unsigned char*)malloc(i))==NULL){
			OK_set_error(ERR_ST_MEMALLOC,ERR_LC_ECDSA,ERR_PT_ECDSAASN+1,NULL);
			return NULL;
		}
		memset(ret,0,i);
	}else{
		ret=buf;
	}

	cp = ret;
	if(pub->E==NULL){
		OK_set_error(ERR_ST_NULLPOINTER,ERR_LC_ECDSA,ERR_PT_ECDSAASN+1,NULL);
		goto error;
	}
	/* ECDSA parameter */
	if(ECPm_toDER(pub->E,cp,&i)==NULL) goto error;
	cp+=i;

	/* public key W */
	if(ECPm_DER_ecpoint(pub->W,cp,&j)) goto error;
	i+=j;

	ASN1_set_sequence(i,ret,ret_len);

	return ret;
error:
	if(ret!=buf) free(ret);
	return NULL;
}


/*-----------------------------------
	estimate ECDSA param DER size
------------------------------------*/
int ECDSAprv_estimate_der_size(Prvkey_ECDSA *prv){
	int ret=32;	/* with a margin, that is a rough estimation. */

	ret += 4 + (prv->W->x->top<<2) + (prv->W->y->top<<2);
	ret += LN_now_byte(prv->k) +4;
	if(prv->E) ret += ECPm_estimate_der_size(prv->E);

	return ret;
}

int ECDSApub_estimate_der_size(Pubkey_ECDSA *pub){
	int ret=16;

	ret += 4 + (pub->W->x->top<<2) + (pub->W->y->top<<2);
	if(pub->E) ret += ECPm_estimate_der_size(pub->E);

	return ret;
}

/**
 * Convert OCTET STRING of ASN.1 to BIT STRING.
 *
 * @param[in] der a pointer to DER encoded data
 * @param[in,out] rt_len 
 */
static void conv_octet_to_bitstring(unsigned char *der, int *rt_len)
{
	int len;	/* the length of the contents */
	int mv;		/* # of the length octets */
	int mv1;	/* # of the length octets of new BIT STRING */
	int j;
	unsigned char buf[16] = {0};	/* buffer for identifier and length
					 * octets of new bit string
					 */
	unsigned char *cp;	/* pointer to contents */

	/*
	 * get the length of the contents assuming the identifier octets has
	 * one octet.
	 */
	len = ASN1_length(&der[1], &mv);

	/* set the length octets of new BIT STRING */
	(void)ASN1_set_length(1 + len, &buf[1], &mv1);

	/* shift the contents octets to the right */
	cp = &der[(1 + mv1 + 1 + len) - 1];
	for (j = 1 + mv + len - 1; j >= 1 + mv; j--) {
		*cp = der[j];
		cp--;
	}

	/* set the number of unused bits, actually 0 */
	der[1 + mv1] = 0;

	/* overwrite the identifier and length octets */
	(void)memcpy(der, buf, 1 + mv1);

	/* the total size of new BIT STRING */
	*rt_len = 1 + mv1 + 1 + len;

	/* set suitable value the identifier */
	der[0] = ASN1_BITSTRING;
}
