/* ecctestfunc.c */
/*
 * Modified by National Institute of Informatics in Japan, 2012-2017.
 *
 */

#include "aiconfig.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>

#ifdef USE_PTHREAD
#include <pthread.h>
#endif

#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_ecc.h>

#define TEST_ECPARAM	"ecparam1.der"
#define TEST_ECPARAM0	"ecparam1.der~"	/* for writing */
#ifndef PATH
# define PATH	"."
#endif

/* test/getfpath.c */
char *get_fpath(char *path, char *filename);


int test_point_addsub(ECParam *E){
	ECp	*a,*b,*c,*d,*tmp;
	int	i;

	a=ECp_dup(E->G);
	b=ECp_new();
	c=ECp_new();
	d=ECp_new();

	for(i=0;i<101;i++){
		ECp_add(E,a,a,b);	/* c = 2a */
		ECp_add(E,b,a,c);	/* c = 3a */
		ECp_sub(E,c,a,b);	/* b = 3a - a */
		ECp_sub(E,b,a,d);

		if(ECp_cmp(a,d)){	/* a should equal b */
			printf("error : EC Point add and sub (%d)\n",i);
			ECp_print(a);
			ECp_print(d);
			return -1;
		}

		if(!(i%20)) printf("EC Point +- -- ok:%d\n",i);

		tmp=a; a=c; c=tmp;	/* exchange a and c */
	}
	ECp_free(a); ECp_free(b);
	ECp_free(c); ECp_free(d);
	return 0;
}

int test_point_multi(ECParam *E){
	ECp	*a,*b,*c,*d;
	LNm *o,*x;
	int	i;

	a=ECp_dup(E->G);
	b=ECp_new();
	c=ECp_new();
	d=ECp_new();
	o=LN_clone(E->b);
	x=LN_alloc();

	/* point multiplication test 1 */
	ECp_multi(E,E->G,E->n,b);
	if(b->infinity)
		printf("EC Point * test 1 (infinity) -- ok\n");
	else{
		printf("EC Point * test 1 (infinity) -- error\n");
		return -1;
	}

	for(i=0;i<5;i++){
		ECp_multi(E,a,o,b);		/* b = k*a */
		ECp_multi_bin(E,a,o,c);	/* b = k*a */
		if(ECp_cmp(b,c)){		/* a should equal d */
			printf("error : EC Point multi bin&win (%d)\n",i);
			ECp_print(b);
			ECp_print(c);
			return -1;
		}
		printf("EC Point * bin&win -- ok:%d\n",i);
		LN_long_sub(o,1);
	}

	for(i=0;i<11;i++){
		ECp_multi(E,a,o,b);		/* b = k*a */
		LN_long_sub(o,1);
		ECp_multi(E,a,o,c);		/* c = (k-1)*a */

		ECp_sub(E,b,c,d);			/* d = b - c */
		if(ECp_cmp(d,a)){	/* a should equal d */
			printf("error : EC Point multi(win) (%d)\n",i);
			ECp_print(a);
			ECp_print(d);
			return -1;
		}

		printf("EC Point * (win) -- ok:%d\n",i);
		LN_long_sub(o,1);
	}

	/* check point multi after one cycle */
	for(i=0;i<6;i++){
		ECp_multi(E,a,o,b);		/* b = k*a */
		LN_plus(o,E->n,x);
		ECp_multi(E,a,x,c);		/* c = (k+n)*a */
		if(ECp_cmp(b,c)){		/* b should equal c */
			printf("error : EC Point multi (cycle) (%d)\n",i);
			return -1;
		}
		printf("EC Point * (cycle) -- ok:%d\n",i);
		LN_long_sub(o,1);
	}

	ECp_free(a); ECp_free(b);
	ECp_free(c); ECp_free(d);
	LN_free(o); LN_free(x);
	return 0;
}

int test_point_projective(ECParam *E){
	ECp	*a,*b,*c,*d,*e,*tmp;
	LNm *o,*x;
	int	i;

	a=ECp_dup(E->G);
	LN_long_set(a->z,1);
	b=ECp_new();
	c=ECp_new();
	d=ECp_new();
	e=ECp_new();
	o=LN_clone(E->b);
	x=LN_alloc();

	/* test 1 -- projective and normal double */
	ECp_add(E,a,a,b);
	LN_long_set(b->z,1);
	ECp_pdouble(E,a,c);
	ECp_copy(c,d);
	ECp_proj2af(E,c);
	if(ECp_cmp(b,c)){	/* b should equal c */
		LN_print(b->x);
		LN_print(c->x);
		LN_print(b->y);
		LN_print(c->y);
		LN_print(b->z);
		LN_print(c->z);
		printf("error: projective double \n");
		return -1;
	}else
		printf("test: projective double -- ok\n");

	/*  test 2 -- projective and normal double */
	ECp_add(E,a,b,c);			/* c = a + b */
	LN_long_set(c->z,1);
	ECp_padd_diffs(E,a,d,e);	/* e = a + d */
	ECp_proj2af(E,e);

	if(ECp_cmp(c,e)){	/* c should equal e */
		LN_print(c->x);
		LN_print(e->x);
		LN_print(c->y);
		LN_print(e->y);
		printf("error: projective add \n");
		return -1;
	}else
		printf("test: projective add -- ok\n");

	/*  test 3 -- projective addition and subtract */
	ECp_padd(E,a,b,c);
	ECp_psub(E,c,b,d);
	ECp_proj2af(E,d);
	if(ECp_cmp(a,d)){	/* a should equal d */
		printf("error: projective add \n");
		return -1;
	}else
		printf("test: projective addition and subtract -- ok\n");


	/*  test 4 -- projective addition and subtract */
	ECp_pdouble(E,a,b);		/* b = 2 * a */
	ECp_pdouble(E,b,c);		/* c = a + b */
	for(i=0;i<101;i++){
		ECp_padd(E,b,c,d);	/* d = b + c */
		ECp_psub(E,d,c,e);	/* e = d - c */
		ECp_proj2af(E,b);
		ECp_proj2af(E,e);

		if(ECp_cmp(b,e)){	/* a should equal d */
			printf("error : EC Projective Point add and sub (%d)\n",i);
			LN_print(b->x);
			LN_print(e->x);
			LN_print(b->y);
			LN_print(e->y);
			LN_print(b->z);
			LN_print(e->z);
			return -1;
		}

		if(!(i%20)) printf("EC Projective Point +- -- ok:%d\n",i);

		tmp=b; b=c; c=d; d=tmp;	/* rotate a,b,c */
	}

	/*  test 5 -- projective multi and normal multi */
	for(i=0;i<5;i++){
		ECp_multi(E,a,o,b);		/* b = k*a */
		LN_long_set(b->z,1);
		ECp_pmulti(E,a,o,c);	/* b = k*a */
		ECp_proj2af(E,c);
		if(ECp_cmp(b,c)){		/* a should equal d */
			printf("error : EC Projective Point multi (%d)\n",i);
			LN_print(b->x);
			LN_print(c->x);
			LN_print(b->y);
			LN_print(c->y);
			return -1;
		}
		printf("EC Projective Point * -- ok:%d\n",i);
		LN_long_sub(o,1);
	}

	/* test 6 -- projective multi infinity */
	ECp_pmulti(E,a,E->n,b);		/* b = k*a */
	ECp_proj2af(E,b);
	if(b->infinity)
		printf("test: projective * (infinity) -- ok\n");
	else{
		printf("test: projective * (infinity) -- error\n");
		return -1;
	}

	/*  test 7 -- projective multi */
	for(i=0;i<11;i++){
		ECp_pmulti(E,a,o,b);		/* b = k*a */
		LN_long_sub(o,1000);
		ECp_pmulti(E,a,o,c);		/* c = (k-1000)*a */
		ECp_psub(E,b,c,d);		/* d = b - c */

		LN_long_set(x,1000);
		ECp_pmulti(E,a,x,b);
		ECp_proj2af(E,d);
		ECp_proj2af(E,b);
		if(ECp_cmp(d,b)){	/* a should equal d */
			printf("error : EC Projective Point multi(2) (%d)\n",i);
			LN_print(b->x);
			LN_print(d->x);
			LN_print(b->y);
			LN_print(d->y);
			return -1;
		}
		printf("EC Projective Point * (2) -- ok:%d\n",i);
		LN_long_sub(o,1);
	}

	/* check point multi after one cycle */
	for(i=0;i<11;i++){
		ECp_pmulti(E,a,o,b);		/* b = k*a */
		LN_plus(o,E->n,x);
		ECp_pmulti(E,a,x,c);		/* c = (k+n)*a */

		ECp_proj2af(E,b);
		ECp_proj2af(E,c);
		if(ECp_cmp(b,c)){		/* b should equal c */
			printf("error : EC Point multi (cycle) (%d)\n",i);
			return -1;
		}
		printf("EC Projective Point * (cycle) -- ok:%d\n",i);
		LN_long_sub(o,1);
	}
	ECp_free(a); ECp_free(b);
	ECp_free(c); ECp_free(d); ECp_free(e);
	LN_free(o); LN_free(x);
	return 0;
}

int test_point_ppow2(ECParam *E){
	ECp     *a,*b,*c,*d,*e;
	LNm     *o,*x;
	int     i,j;

	a=ECp_dup(E->G);
	LN_long_set(a->z,1);
	b=ECp_new();
	c=ECp_new();
	d=ECp_new();
	e=ECp_new();
	o=LN_clone(E->b);
	x=LN_alloc();

	for(i=1;i<101;i++){
	  ECp_copy(a,b);
	  for(j=0;j<i;j++){
	    ECp_pdouble(E,b,c); ECp_copy(c,b);}

	  ECp_ppow2(E,a,i,c);
	  if(ECp_cmp(b,c)){		/* b should equal c */
	    printf("EC Point pow2 test 1 (%d) -- error!\n",i);
	    return -1;
	  }

	  if((i%20)==0)
	    printf("EC Point pow2 test 1 -- ok: %d\n",i);
	}

	ECp_free(a); ECp_free(b);
	ECp_free(c); ECp_free(d); ECp_free(e);
	LN_free(o); LN_free(x);
	return 0;
}


int test_std_parameter(){
	int i=0,j;
	ECParam *E;
	int named_curve[] = {
	    ECP_X962_prime192v1,
	    ECP_secp224r1,
	    ECP_X962_prime256v1,
	    ECP_secp384r1,
	    ECP_secp521r1,
	};
#define STD_NUM	5

	do{
		printf("----- std parameter test stage %d -----\n",i);

		if((E=ECPm_get_std_parameter(named_curve[i]))==NULL) {
		    printf("std parameter test stage %d -- NG:\n",i);
		    continue;
		}

		if(test_point_addsub(E))
		    break;

		if(test_point_multi(E))
		    break;

		if(test_point_projective(E))
		    break;

		if(test_point_ppow2(E))
		    break;

		ECPm_free(E);
		E=NULL;
		i++;
	}while(i<STD_NUM);

	if (E!=NULL) ECPm_free(E);
	return (i<STD_NUM) ? -1 : 0;
}

int test_generating_parameter(){
	int i=0,j;
	ECParam *E;
	int bit[]={160,192,224,256,384,512};
#define BIT_NUM 6

	do{
		j=i%BIT_NUM;
		printf("----- generating parameter test stage %d (%d bit)-----\n",i,bit[i]);

		if((E=ECPm_gen_parameter(bit[i]))==NULL){
			printf("EC Parameter generation error occured!!\n");
		    return -1;
		}
		/*
		 * XXX: should be confirm the return value
		 * of ECPm_verify_parameter().
		 */
		if((j = ECPm_verify_parameter(E)) != 0){
			printf("verification error occured!! code=%d\n",j);
			ECPm_free(E);
			continue;
		}else
			printf("elliptic curve verification ok!!\n");

		if(test_point_addsub(E))
		    return -1;

		if(test_point_multi(E))
		    return -1;

		if(test_point_projective(E))
		    return -1;

		if(test_point_ppow2(E))
		    return -1;

		ECPm_free(E);
		i++;
	}while(i<BIT_NUM);
	return 0;
}

int test_ecc_der(){
	unsigned char *der=NULL,*der2=NULL;
	ECParam *E=NULL;
	int i,j,err=-1;
	char *fp_ECPARAM = get_fpath(PATH, TEST_ECPARAM);

	/* test encode and decode parameter */
	for(j=0;j<11;j++){
		/* read EC Param DER */
		if ((der = ASN1_read_der(fp_ECPARAM)) == NULL) goto done;

		if((E=ASN1_read_ecparam(der))==NULL){
			printf("decode elliptic curve parameter -- error!\n");
			goto done;
		}
		if((j%5)==0) printf("decode elliptic curve parameter -- ok : %d\n",j);

		/* write EC Param DER */
		if((der2=ECPm_toDER(E,NULL,&i))==NULL){
			printf("encode elliptic curve parameter file -- error!\n");
			goto done;
		}
		if(ASN1_write_der(der2,TEST_ECPARAM0)) goto done;
	
		if((j%5)==0) printf("encode elliptic curve parameter -- ok : %d\n",j);

		/* compare two parameters */
		if(memcmp(der,der2,i)){
			printf("compare DER parameters : der == der2 -- error!\n");
			goto done;
		}
		if((j%5)==0) printf("compare DER parameters : der == der2 -- ok : %d\n",j);

		free(der2); der2=NULL;
		ECPm_free(E);  E=NULL; der=NULL; /* it frees "der" memory */
	}

	/* add and sub test with read parameter */
	if ((der = ASN1_read_der(fp_ECPARAM)) == NULL) goto done;
	if((E=ASN1_read_ecparam(der))==NULL){
		printf("decode elliptic curve parameter -- error!\n");
		goto done;
	}

	if(test_point_addsub(E)) goto done;
	printf("test_point_addsub with DER param -- ok : %d\n", 0); /* FIXME */

	err=0;
done:
	if(der2) free(der2);
	ECPm_free(E); /* it frees "der" memory */
	if (fp_ECPARAM) free(fp_ECPARAM);
	return err;
}

#if 0
void test_ec(){
	uint32_t lp[]={
		0xc51458a3,0x92c5cb3b,0xeaf3337d,0x18933ed4,
		0xc0a73dab,0x8b3601d5};		
	uint32_t ln[]={	
		0x000006c4,0x680d1469,0x5e1ff242,0x4dc11246,
		0x19586074,0xe02b9bdd};
	uint32_t lk = 0x001d1f69;
	uint32_t la[]={
		0x1d2f1b59,0xe0796100,0x051dd806,0x9e391f1e,
		0xe7dc96dd,0x550eec57};
	uint32_t lb[]={
		0xade17c38,0x671d2edb,0x0d8f88d8,0xb9b29d62,
		0x95d798ef,0x93a9fcd2};
	ECParam *E;
	ECp *g,*a,*b;
	LNm *vz,*tmp;
	int i=0;

	E = ECPm_new();
	a=ECp_new();
	b=ECp_new();
	g=E->G;
	vz=E->buf[2];tmp=E->buf[3];

	LN_set_num(E->p,6,lp);
	LN_set_num(E->n,6,ln);
	LN_long_set(E->h,lk);
	LN_set_num(E->a,6,la);
	LN_set_num(E->b,6,lb);
	E->psize = LN_now_bit(E->p);
	E->nsize = LN_now_bit(E->n);

	do{
		a->infinity = 0;
		g->infinity = 0;
		do{
			LN_set_rand(vz,E->psize>>3,(unsigned short)rand());
			LN_div_mod(vz,E->p,tmp,a->x);

			if(ECp_x2y(E,a->x,a->y,0))
				continue;

			ECp_multi(E,a,E->h,g);

			if(g->infinity)
				continue;

			break;
		}while(1);

		ECp_multi(E,g,E->n,a);

		if(a->infinity)	break;

		printf("now -- %d\n",i);
		i++;
	}while(1);
}
#endif
