/*
 * ECC speed test functions.
 */
/*
 * Copyright (C) 1998-2002
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <time.h>
#include <sys/types.h>
#ifdef __WINDOWS__
#include <sys/timeb.h>
#else
#include <sys/times.h>
#define TIMES
#endif

#include "ok_ecc.h"

#define TEST_ITR	2000

/* The following if from times(3) man page.  It may need to be changed */
#ifndef HZ
# ifndef CLK_TCK
#  ifndef _BSD_CLK_TCK_ /* FreeBSD hack */
#   define HZ   100.0
#  else /* _BSD_CLK_TCK_ */
#   define HZ ((double)_BSD_CLK_TCK_)
#  endif
# else /* CLK_TCK */
#  define HZ ((double)CLK_TCK)
# endif
#endif

#define START   0
#define STOP    1

static double Time_F(int s)
{
        double ret;
#ifdef TIMES
        static struct tms tstart,tend;

        if (s == START)
	{
                times(&tstart);
                return(0);
	    }
        else
	{
                times(&tend);
                ret=((double)(tend.tms_utime-tstart.tms_utime))/HZ;
                return((ret < 1e-3)?1e-3:ret);
	    }
#else /* !times() */
        static struct timeb tstart,tend;
        long i;

        if (s == START)
	{
                ftime(&tstart);
                return(0);
	    }
        else
	{
                ftime(&tend);
                i=(long)tend.millitm-(long)tstart.millitm;
                ret=((double)(tend.time-tstart.time))+((double)i)/1000.0;
                return((ret < 0.001)?0.001:ret);
	    }
#endif
    }


/*=========================================================
  large number calculation test
=========================================================*/
int main(argc,argv)
int  argc;
char **argv;
{
    ECParam *E;
    ECp     *a,*b,*c,*d;
    LNm	    *o;
    double tm;
    int	i,j,tsi,l,m,k[]={12,16};
    int	ec[]={ECP_TYPE_X9_62_192,ECP_TYPE_X9_62_256};

    a=ECp_new();
    b=ECp_new();
    c=ECp_new();
    d=ECp_new();
    o=LN_alloc();

    j=0;
    do{
	E=ECPm_get_std_parameter(ec[j]);
	ECp_copy(E->G,a);
	LN_copy(E->b,o);

	printf("ECC : y^2 = x^3 + a*x + b (mod p) (%d bit)\n",k[j]*16);
	printf("p : "); LN_print(E->p);
	printf("a : "); LN_print(E->a);
	printf("b : "); LN_print(E->b);
	printf("Base point G on E\n"); ECp_print(E->G);
	printf("Order of point G on E\n");
	printf("n : "); LN_print(E->n);
	printf("\n");

	/* standard add speed test */
	Time_F(START);
	for(i=0;i<TEST_ITR;i++){
	    ECp_add(E,a,a,b);
	}
	tm=Time_F(STOP);
	printf("add      A + B (%d bit) -> %8.5fms\n",k[j]*16,tm*1000.0/TEST_ITR);

	/* sub speed test */
	Time_F(START);
	for(i=0;i<TEST_ITR;i++){
	    ECp_sub(E,b,a,c);
	}
	tm=Time_F(STOP);
	printf("sub      A - B (%d bit) -> %8.5fms\n",k[j]*16,tm*1000.0/TEST_ITR);

	/* multi speed test */
	tsi = TEST_ITR/(k[j]*10);
	Time_F(START);
	for(i=0;i<tsi;i++){
	    ECp_multi(E,a,o,b);
	}
	tm=Time_F(STOP);
	printf("multi    k * A (%d bit) -> %8.5fms\n",k[j]*16,tm*1000.0/tsi);

	/* projective calculation test */
	LN_long_set(a->z,1);
	ECp_pdouble(E,a,b);

	/* proj double speed test */
	Time_F(START);
	for(i=0;i<TEST_ITR;i++){
	    ECp_pdouble(E,b,c);
	}
	tm=Time_F(STOP);
	printf("projective double   A + A (%d bit) -> %8.5fms\n",k[j]*16,tm*1000.0/TEST_ITR);

	/* proj add speed test */
	Time_F(START);
	for(i=0;i<TEST_ITR;i++){
	    ECp_padd(E,b,b,c);
	}
	tm=Time_F(STOP);
	printf("projective add      A + B (%d bit) -> %8.5fms\n",k[j]*16,tm*1000.0/TEST_ITR);

	/* proj sub speed test */
	Time_F(START);
	for(i=0;i<TEST_ITR;i++){
	    ECp_psub(E,c,b,d);
	}
	tm=Time_F(STOP);
	printf("projective sub      A - B (%d bit) -> %8.5fms\n",k[j]*16,tm*1000.0/TEST_ITR);

	/* proj multi speed test */
	tsi = TEST_ITR/(k[j]*10);
	Time_F(START);
	for(i=0;i<tsi;i++){
	    ECp_pmulti(E,b,o,d);
	}
	tm=Time_F(STOP);
	printf("projective multi    k * A (%d bit) -> %8.5fms\n",k[j]*16,tm*1000.0/tsi);

	/* compare between pdouble & ppow2 */
	tsi = TEST_ITR/(k[j]*10); m=100;
	Time_F(START);
	for(i=0;i<tsi;i++){
	    ECp_ppow2(E,b,m,c);
	}
	tm=Time_F(STOP);
	printf("projective pow2     2^(%d) * A (%d bit) -> %8.5fms\n",m,k[j]*16,tm*1000.0/tsi);
	Time_F(START);
	for(i=0;i<tsi;i++){
	    for(l=0;l<m;l++) ECp_pdouble(E,b,c);
	}
	tm=Time_F(STOP);
	printf("projective double   2^(%d) * A (%d bit) -> %8.5fms\n",m,k[j]*16,tm*1000.0/tsi);


	j++;
	ECPm_free(E);
    }while(j<2);

    ECp_free(a); ECp_free(b);
    ECp_free(c); ECp_free(d);
}




