/* eccptestfc.c */
/*
 * Copyright (C) 1998-2000
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "aiconfig.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef USE_PTHREAD
#include <pthread.h>
#endif

#include "ok_ecc.h"

#ifdef USE_PTHREAD
#include "eccptestfc.h"

void separate_k2(LNm *k, LNm *rk, int itr, int *pow){
    int i,j,l,m,n,ktop,kbit;
    ULONG *kn;

    ktop=k->top;
    kbit=LN_now_bit(k);
    kn  =k->num;

    j = ((kbit)*748)/1000;
    if(itr==0){
	int a,b;

	*pow = j;

	a = kbit-j;
	b = kbit&0x1f;
	
	rk->top = l =((a-1)/32)+1;
	n = (a-b)&0x1f;
	b = (b==0)?(32):(b);
	if((a%32)<=b) n=(n==0)?(32):(n);

	for(i=l,m=0;i>0;i--,m++){
	    if((a%32)<=b){
		if(i==l)
		    rk->num[LN_MAX-i] =  kn[LN_MAX-(ktop-m)]>>(32-n);
		else
		    rk->num[LN_MAX-i] = (kn[LN_MAX-(ktop-m)-1]<<n)|(kn[LN_MAX-(ktop-m)]>>(32-n));
	    }else{
		    rk->num[LN_MAX-i] = (kn[LN_MAX-(ktop-m)]<<n)|(kn[LN_MAX-(ktop-m)+1]>>(32-n));
	    }
	}
    }else{
	*pow = 0;
	rk->top = l = ((j-1)/32)+1;
	memcpy(&rk->num[LN_MAX-l],&kn[LN_MAX-l],sizeof(ULONG)*l);
	rk->num[LN_MAX-l]&=0xffffffff>>(32-(j&0x1f));
    }
}

void init_pth_ECp_pmul(pth_ecpm **pe,ECParam *E, ECp *A, LNm *k){
    int	i,j,tp;

    for(i=tp=0;i<NUM_THREADS;i++){
	pe[i]->E   = ECPm_dup(E);
	pe[i]->A   = ECp_dup(A);
	pe[i]->k   = LN_alloc();
	pe[i]->ret = ECp_new();

	/* calculate length */
	separate_k2(k,pe[i]->k,i,&pe[i]->pow);
    }
}

void thread_ECp_pmulti(pth_ecpm **pe, ECp *ret){
    pthread_t threads[NUM_THREADS];
    ECParam *E;
    int	i;

    for(i=0;i<NUM_THREADS;i++){
	pthread_create(&threads[i],NULL,(void*)ecp_multi_in,(void*)(pth_ecpm*)pe[i]);
    }    

    for(i=0;i<NUM_THREADS;i++){
	pthread_join(threads[i],NULL);
    }

    E = pe[0]->E;
    ECp_padd(E,pe[0]->ret,pe[1]->ret,ret);
}

void free_pth_ECp_pmul(pth_ecpm **pe){
    int i;
    for(i=0;i<NUM_THREADS;i++){
        ECPm_free(pe[i]->E);
        ECp_free(pe[i]->ret);
        ECp_free(pe[i]->A);
	LN_free(pe[i]->k);
    }
}

/*-----------------------------------------------------------------------*/

pth_ecpm **pth_ecpm_array_new(){
  int i;
  pth_ecpm **ret;

  ret = (pth_ecpm**)MALLOC(sizeof(pth_ecpm*)*NUM_THREADS);
  for(i=0;i<NUM_THREADS;i++)
    ret[i] = (pth_ecpm*)MALLOC(sizeof(pth_ecpm));
  return ret;
}

void pth_ecpm_array_free(pth_ecpm **pe){
  int i;
  for(i=0;i<NUM_THREADS;i++)
    FREE(pe[i]);
  FREE(pe);
}

/*-----------------------------------------------------------------------*/
void *ecp_calc_in(pth_ecpm *pe){
    ECp_pmulti(pe->E,pe->A,pe->k,pe->ret);
}

void thread_calc_test(ECParam *E, ECp *A, ECp *B, LNm *k, int pow, ECp *ret){
    pthread_t threads[NUM_THREADS];
    pth_ecpm  pthval[NUM_THREADS];
    int i,j,tp;

    for(i=tp=0;i<NUM_THREADS;i++){
        pthval[i].E   = ECPm_dup(E);
        pthval[i].A   = ECp_dup(A);
        pthval[i].B   = B;
        pthval[i].k   = k;
        pthval[i].ret = ECp_new();
        pthval[i].pow = pow;
        pthread_create(&threads[i],NULL,(void*)ecp_calc_in,(void*)(pth_ecpm*)&pthval[i]);
    }

    for(i=0;i<NUM_THREADS;i++){
        pthread_join(threads[i],NULL);
    }

    ECp_copy(pthval[1].ret,ret);
    for(i=0;i<NUM_THREADS;i++){
        ECPm_free(pthval[i].E);
        ECp_free(pthval[i].ret);
        ECp_free(pthval[i].A);
    }
}

/*-----------------------------------------------------------------------*/

void test_point_pthmulti(ECParam *E){
	ECp     *a,*b,*c,*d,*e,*tmp;
	LNm     *o;
	int     i,j;
	pth_ecpm  **pthval;

	a=ECp_dup(E->G);
	LN_long_set(a->z,1);
	b=ECp_new();
	c=ECp_new();
	d=ECp_new();
	o=LN_clone(E->b);

        /* test 1 */
        for(i=0;i<1;i++){
            ECp_pmulti(E,a,o,c);
            thread_calc_test(E,a,b,o,(i%256)+4,d);

            if(ECp_cmp(c,d)){           /* b should equal c */
                printf("EC Point calc test 1 (thread) -- error:%d\n",i);
                ECp_print(c);
                ECp_print(d);
                exit(0);
            }

            if((i%5)==0) printf("EC Point calc test 1 (thread) -- ok:%d\n",i);
            LN_long_sub(o,3);
        }

        /* test 2 -- check speed of ecp_multi_in */
	pthval = pth_ecpm_array_new();
	for(i=0;i<1;i++){
	    ECp_pmulti(E,a,o,b);		/* b = k*a */

	    init_pth_ECp_pmul(pthval,E,a,o);
	    thread_ECp_pmulti(pthval,c);
	    free_pth_ECp_pmul(pthval);

	    ECp_proj2af(E,b);
	    ECp_proj2af(E,c);
	    if(ECp_cmp(b,c)){		/* b should equal c */
		printf("EC Point multi test 2 (thread) -- error:%d\n",i);
		ECp_print(b);
		ECp_print(c);
		exit(0);
	    }

	    if((i%20)==0)
		printf("EC Point multi test 2 (thread) -- ok:%d\n",i);

	    LN_long_sub(o,3);
	}
	pth_ecpm_array_free(pthval);

	ECp_free(a); ECp_free(b); ECp_free(c); ECp_free(d);
	LN_free(o);
}
#endif /* USE_PTHREAD */
