/* eccptest.c */
/*
 * Copyright (C) 1998-2000
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <pthread.h>

#ifdef __SUNOS__
# include <thread.h>
#endif

#include <time.h>
#include <sys/types.h>
#include <sys/timeb.h>

#include "ok_ecc.h"
#include "eccptestfc.h"

/* The following if from times(3) man page.  It may need to be changed */
#ifndef HZ
# ifndef CLK_TCK
#  ifndef _BSD_CLK_TCK_ /* FreeBSD hack */
#   define HZ   100.0
#  else /* _BSD_CLK_TCK_ */
#   define HZ ((double)_BSD_CLK_TCK_)
#  endif
# else /* CLK_TCK */
#  define HZ ((double)CLK_TCK)
# endif
#endif

#define START   0
#define STOP    1

#define TEST_ITR        2000


/* thread function */
void *pth_test(void *arg);

/* global value */
pthread_t threads[NUM_THREADS];
pthread_mutex_t lock;
int num=2;

/*-----------------------------------
  time ...
-----------------------------------*/
static double Time_F(int s)
{
  double ret;
  static struct timeb tstart,tend;
  long i;

  if (s == START)
  {
      ftime(&tstart);
      return(0);
  }
  else
  {
      ftime(&tend);
      i=(long)tend.millitm-(long)tstart.millitm;
      ret=((double)(tend.time-tstart.time))+((double)i)/1000.0;
      return((ret < 0.001)?0.001:ret);
  }
}

/*---------------------------------------
  used in pthread_ECp_multi
---------------------------------------*/
void *ecp_multi_in(pth_ecpm *pe){
    int i,j,tsi;
    double tm;

#ifdef TES1
    tsi = TEST_ITR/32;
    Time_F(START);
    for(i=0;i<tsi;i++){
#endif
      ECp_pmulti(pe->E,pe->A,pe->k,pe->ret);
      if(pe->pow)
	ECp_ppow2(pe->E,pe->ret,pe->pow,pe->ret);
#ifdef TES1
    }
    tm=Time_F(STOP);
    printf("ecp_multi_in:%8.5f:%d\n",tm*1000.0/tsi,pe->pow);
#endif
    pthread_exit((void*)0);
}

/*---------------------------------------
  POSIX thread
---------------------------------------*/
void *pth_test(void *arg){
	ECParam *E;

	E=ECPm_get_std_parameter(ECP_TYPE_X9_62_192);

	test_point_addsub(E);

	test_point_multi(E);

	test_point_projective(E);

	test_point_ppow2(E);

	ECPm_free(E);

	test_generating_parameter();

	pthread_exit((void*)0);
}

void check_speed_pthmulti(){
    ECParam *E;
    ECp     *a,*b,*c,*d;
    LNm     *o;
    double tm;
    int i,j,tsi,l,m,k[]={12,16};
    int ec[]={ECP_TYPE_X9_62_192,ECP_TYPE_X9_62_256};
    pth_ecpm  **pthval;

    a=ECp_new();
    b=ECp_new();
    c=ECp_new();
    d=ECp_new();
    o=LN_alloc();

    j=0;
    do{
        E=ECPm_get_std_parameter(ec[j]);
        ECp_copy(E->G,a);
        LN_copy(E->b,o);

        /* projective calculation test */
        LN_long_set(a->z,1);
        ECp_pdouble(E,a,b);

	/* proj mutli speed test */
        tsi = TEST_ITR/(k[j]*5);
        Time_F(START);
        for(i=0;i<tsi;i++){
            ECp_pmulti(E,b,o,d);
        }
        tm=Time_F(STOP);
        printf("projective multi    k * A (%d bit) -> %8.5fms\n",k[j]*16,tm*1000.0/tsi);

        /* pthread proj multi speed test */
	pthval = pth_ecpm_array_new();
	init_pth_ECp_pmul(pthval,E,b,o);	
        tsi = TEST_ITR/(k[j]*5);
        Time_F(START);
        for(i=0;i<tsi;i++){
	  thread_ECp_pmulti(pthval,d);
        }
        tm=Time_F(STOP);
	free_pth_ECp_pmul(pthval);
	pth_ecpm_array_free(pthval);

        printf("pthread projective multi    k * A (%d bit) -> %8.5fms\n",k[j]*16,tm*1000.0/tsi);


        j++;
        ECPm_free(E);
    }while(j<2);

    ECp_free(a); ECp_free(b);
    ECp_free(c); ECp_free(d);
    LN_free(o);
}


#undef NUM_THREADS
#define NUM_THREADS 4

/*---------------------------------------
  test main
---------------------------------------*/
int main(int argc,char **argv){
    ECParam *E;
    int i,j;

#ifdef __SUNOS__
    thr_setconcurrency(2);
#endif

    /* -------- */
    printf("thread pmulti test routine -- start\n");
    E=ECPm_get_std_parameter(ECP_TYPE_X9_62_192);
    test_point_pthmulti(E);
    ECPm_free(E);

    E=ECPm_get_std_parameter(ECP_TYPE_X9_62_256);
    test_point_pthmulti(E);
    ECPm_free(E);
    printf("thread pmulti test routine -- end\n");

    /* -------- */
    printf("thread pmulti speed test routine -- start\n");
    check_speed_pthmulti();
    printf("thread pmulti speed test routine -- end\n");

    /* -------- */
    printf("start test -- pthread multi running.\n");
    pthread_mutex_init(&lock,NULL);
    for(i=0;i<NUM_THREADS;i++){
	pthread_create(&threads[i],NULL,pth_test,(void*)i);
    }
    printf("\n");
    
    for(i=0;i<NUM_THREADS;i++){
	pthread_join(threads[i],NULL);
    }
    printf("end test -- pthread multi running.\n");
    pthread_exit((void*)0);
}


