/* ecc_stdprm.c */
/*
 * Copyright (c) 2012-2014 National Institute of Informatics in Japan, 
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must 
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2002
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_ecc.h>

/* NIST P-192
 * 
 * ANSI X9.62-1998:prime192v1
 */
static unsigned char ts_p192v1[]={
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xfe,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff};
static unsigned char ts_n192v1[]={
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0x99,0xde,0xf8,0x36,
	0x14,0x6b,0xc9,0xb1,0xb4,0xd2,0x28,0x31};
static unsigned char ts_a192v1[]={
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xfe,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xfc};
static unsigned char ts_b192v1[]={
	0x64,0x21,0x05,0x19,0xe5,0x9c,0x80,0xe7,
	0x0f,0xa7,0xe9,0xab,0x72,0x24,0x30,0x49,
	0xfe,0xb8,0xde,0xec,0xc1,0x46,0xb9,0xb1};
static unsigned char ts_gp192v1[]={
	0x03,
	0x18,0x8d,0xa8,0x0e,0xb0,0x30,0x90,0xf6,
	0x7c,0xbf,0x20,0xeb,0x43,0xa1,0x88,0x00,
	0xf4,0xff,0x0a,0xfd,0x82,0xff,0x10,0x12};

/* NIST P-224
 * 
 * SEC 2: Recommended Elliptic Curve Domain Parameters
 *        2.6.2 Recommended Parameters secp224r1
 */
static unsigned char ts_p224r1[]={
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x01};
static unsigned char ts_n224r1[]={
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0x16,0xa2,
	0xe0,0xb8,0xf0,0x3e,0x13,0xdd,0x29,0x45,
	0x5c,0x5c,0x2a,0x3d};
static unsigned char ts_a224r1[]={
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xfe,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xfe};
static unsigned char ts_b224r1[]={
	0xb4,0x05,0x0a,0x85,0x0c,0x04,0xb3,0xab,
	0xf5,0x41,0x32,0x56,0x50,0x44,0xb0,0xb7,
	0xd7,0xbf,0xd8,0xba,0x27,0x0b,0x39,0x43,
	0x23,0x55,0xff,0xb4};
static unsigned char ts_gp224r1[]={
	0x02,
	0xb7,0x0e,0x0c,0xbd,0x6b,0xb4,0xbf,0x7f,
	0x32,0x13,0x90,0xb9,0x4a,0x03,0xc1,0xd3,
	0x56,0xc2,0x11,0x22,0x34,0x32,0x80,0xd6,
	0x11,0x5c,0x1d,0x21};

/* ANSI X9.62-1998:prime239v1 */
static unsigned char ts_p239v1[]={
	0x7f,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0x7f,0xff,0xff,0xff,
	0xff,0xff,0x80,0x00,0x00,0x00,0x00,0x00,
	0x7f,0xff,0xff,0xff,0xff,0xff};
static unsigned char ts_n239v1[]={
	0x7f,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0x7f,0xff,0xff,0x9e,
	0x5e,0x9a,0x9f,0x5d,0x90,0x71,0xfb,0xd1,
	0x52,0x26,0x88,0x90,0x9d,0x0b};
static unsigned char ts_a239v1[]={
	0x7f,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0x7f,0xff,0xff,0xff,
	0xff,0xff,0x80,0x00,0x00,0x00,0x00,0x00,
	0x7f,0xff,0xff,0xff,0xff,0xfc};
static unsigned char ts_b239v1[]={
	0x6b,0x01,0x6c,0x3b,0xdc,0xf1,0x89,0x41,
	0xd0,0xd6,0x54,0x92,0x14,0x75,0xca,0x71,
	0xa9,0xdb,0x2f,0xb2,0x7d,0x1d,0x37,0x79,
	0x61,0x85,0xc2,0x94,0x2c,0x0a};
static unsigned char ts_gp239v1[]={
	0x02,
	0x0f,0xfa,0x96,0x3c,0xdc,0xa8,0x81,0x6c,
	0xcc,0x33,0xb8,0x64,0x2b,0xed,0xf9,0x05,
	0xc3,0xd3,0x58,0x57,0x3d,0x3f,0x27,0xfb,
	0xbd,0x3b,0x3c,0xb9,0xaa,0xaf};

/* NIST P-256
 * 
 * ANSI X9.62-1998:prime256v1
 */
static unsigned char ts_p256v1[]={
	0xff,0xff,0xff,0xff,0x00,0x00,0x00,0x01,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff};
static unsigned char ts_n256v1[]={
	0xff,0xff,0xff,0xff,0x00,0x00,0x00,0x00,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xbc,0xe6,0xfa,0xad,0xa7,0x17,0x9e,0x84,
	0xf3,0xb9,0xca,0xc2,0xfc,0x63,0x25,0x51};
static unsigned char ts_a256v1[]={
	0xff,0xff,0xff,0xff,0x00,0x00,0x00,0x01,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xfc};
static unsigned char ts_b256v1[]={
	0x5a,0xc6,0x35,0xd8,0xaa,0x3a,0x93,0xe7,
	0xb3,0xeb,0xbd,0x55,0x76,0x98,0x86,0xbc,
	0x65,0x1d,0x06,0xb0,0xcc,0x53,0xb0,0xf6,
	0x3b,0xce,0x3c,0x3e,0x27,0xd2,0x60,0x4b};
static unsigned char ts_gp256v1[]={
	0x03,
	0x6b,0x17,0xd1,0xf2,0xe1,0x2c,0x42,0x47,
	0xf8,0xbc,0xe6,0xe5,0x63,0xa4,0x40,0xf2,
	0x77,0x03,0x7d,0x81,0x2d,0xeb,0x33,0xa0,
	0xf4,0xa1,0x39,0x45,0xd8,0x98,0xc2,0x96};

/* NIST P-384
 * 
 * SEC 2: Recommended Elliptic Curve Domain Parameters
 *        2.8.1 Recommended Parameters secp384r1
 */
static unsigned char ts_p384r1[]={
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xfe,
	0xff,0xff,0xff,0xff,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0xff,0xff,0xff,0xff};
static unsigned char ts_n384r1[]={
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xc7,0x63,0x4d,0x81,0xf4,0x37,0x2d,0xdf,
	0x58,0x1a,0x0d,0xb2,0x48,0xb0,0xa7,0x7a,
	0xec,0xec,0x19,0x6a,0xcc,0xc5,0x29,0x73};
static unsigned char ts_a384r1[]={
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xfe,
	0xff,0xff,0xff,0xff,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0xff,0xff,0xff,0xfc};
static unsigned char ts_b384r1[]={
	0xb3,0x31,0x2f,0xa7,0xe2,0x3e,0xe7,0xe4,
	0x98,0x8e,0x05,0x6b,0xe3,0xf8,0x2d,0x19,
	0x18,0x1d,0x9c,0x6e,0xfe,0x81,0x41,0x12,
	0x03,0x14,0x08,0x8f,0x50,0x13,0x87,0x5a,
	0xc6,0x56,0x39,0x8d,0x8a,0x2e,0xd1,0x9d,
	0x2a,0x85,0xc8,0xed,0xd3,0xec,0x2a,0xef};
static unsigned char ts_gp384r1[]={
	0x03,
	0xaa,0x87,0xca,0x22,0xbe,0x8b,0x05,0x37,
	0x8e,0xb1,0xc7,0x1e,0xf3,0x20,0xad,0x74,
	0x6e,0x1d,0x3b,0x62,0x8b,0xa7,0x9b,0x98,
	0x59,0xf7,0x41,0xe0,0x82,0x54,0x2a,0x38,
	0x55,0x02,0xf2,0x5d,0xbf,0x55,0x29,0x6c,
	0x3a,0x54,0x5e,0x38,0x72,0x76,0x0a,0xb7};

/* NIST P-521
 * 
 * SEC 2: Recommended Elliptic Curve Domain Parameters
 *        2.9.1 Recommended Parameters secp521r1
 */
static unsigned char ts_p521r1[]={
	0x01,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff};
static unsigned char ts_n521r1[]={
	0x01,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xfa,0x51,0x86,0x87,0x83,0xbf,0x2f,
	0x96,0x6b,0x7f,0xcc,0x01,0x48,0xf7,0x09,
	0xa5,0xd0,0x3b,0xb5,0xc9,0xb8,0x89,0x9c,
	0x47,0xae,0xbb,0x6f,0xb7,0x1e,0x91,0x38,
	0x64,0x09};
static unsigned char ts_a521r1[]={
	0x01,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0xff,0xfc};
static unsigned char ts_b521r1[]={
	0x00,0x51,0x95,0x3e,0xb9,0x61,0x8e,0x1c,
	0x9a,0x1f,0x92,0x9a,0x21,0xa0,0xb6,0x85,
	0x40,0xee,0xa2,0xda,0x72,0x5b,0x99,0xb3,
	0x15,0xf3,0xb8,0xb4,0x89,0x91,0x8e,0xf1,
	0x09,0xe1,0x56,0x19,0x39,0x51,0xec,0x7e,
	0x93,0x7b,0x16,0x52,0xc0,0xbd,0x3b,0xb1,
	0xbf,0x07,0x35,0x73,0xdf,0x88,0x3d,0x2c,
	0x34,0xf1,0xef,0x45,0x1f,0xd4,0x6b,0x50,
	0x3f,0x00};
static unsigned char ts_gp521r1[]={
	0x02,
	0x00,0xc6,0x85,0x8e,0x06,0xb7,0x04,0x04,
	0xe9,0xcd,0x9e,0x3e,0xcb,0x66,0x23,0x95,
	0xb4,0x42,0x9c,0x64,0x81,0x39,0x05,0x3f,
	0xb5,0x21,0xf8,0x28,0xaf,0x60,0x6b,0x4d,
	0x3d,0xba,0xa1,0x4b,0x5e,0x77,0xef,0xe7,
	0x59,0x28,0xfe,0x1d,0xc1,0x27,0xa2,0xff,
	0xa8,0xde,0x33,0x48,0xb3,0xc1,0x85,0x6a,
	0x42,0x9b,0xf9,0x7e,0x7e,0x31,0xc2,0xe5,
	0xbd,0x66};

int ECPm_set_std_parameter(ECParam *E,int type){
	ECp *tp;

	switch(type){
	case ECP_X962_prime192v1:
		LN_set_num_c(E->a,24,ts_a192v1);
		LN_set_num_c(E->b,24,ts_b192v1);
		LN_set_num_c(E->p,24,ts_p192v1);
		LN_set_num_c(E->n,24,ts_n192v1);
		LN_long_set(E->h,1);

		if((tp=ECp_OS2P(E,ts_gp192v1,25))==NULL) goto error;
		E->type = OBJ_X962_prime192v1;
		break;
	case ECP_secp224r1:
		LN_set_num_c(E->a,28,ts_a224r1);
		LN_set_num_c(E->b,28,ts_b224r1);
		LN_set_num_c(E->p,28,ts_p224r1);
		LN_set_num_c(E->n,28,ts_n224r1);
		LN_long_set(E->h,1);

		if((tp=ECp_OS2P(E,ts_gp224r1,29))==NULL) goto error;
		E->type = OBJ_SECP224R1;
		break;
	case ECP_X962_prime239v1:
		LN_set_num_c(E->a,30,ts_a239v1);
		LN_set_num_c(E->b,30,ts_b239v1);
		LN_set_num_c(E->p,30,ts_p239v1);
		LN_set_num_c(E->n,30,ts_n239v1);
		LN_long_set(E->h,1);

		if((tp=ECp_OS2P(E,ts_gp239v1,31))==NULL) goto error;
		E->type = OBJ_X962_prime239v1;
		break;
	case ECP_X962_prime256v1:
		LN_set_num_c(E->a,32,ts_a256v1);
		LN_set_num_c(E->b,32,ts_b256v1);
		LN_set_num_c(E->p,32,ts_p256v1);
		LN_set_num_c(E->n,32,ts_n256v1);
		LN_long_set(E->h,1);

		if((tp=ECp_OS2P(E,ts_gp256v1,33))==NULL) goto error;
		E->type = OBJ_X962_prime256v1;
		break;
	case ECP_secp384r1:
		LN_set_num_c(E->a,48,ts_a384r1);
		LN_set_num_c(E->b,48,ts_b384r1);
		LN_set_num_c(E->p,48,ts_p384r1);
		LN_set_num_c(E->n,48,ts_n384r1);
		LN_long_set(E->h,1);

		if((tp=ECp_OS2P(E,ts_gp384r1,49))==NULL) goto error;
		E->type = OBJ_SECP384R1;
		break;
	case ECP_secp521r1:
		LN_set_num_c(E->a,66,ts_a521r1);
		LN_set_num_c(E->b,66,ts_b521r1);
		LN_set_num_c(E->p,66,ts_p521r1);
		LN_set_num_c(E->n,66,ts_n521r1);
		LN_long_set(E->h,1);

		if((tp=ECp_OS2P(E,ts_gp521r1,67))==NULL) goto error;
		E->type = OBJ_SECP521R1;
		break;
	default:
		OK_set_error(ERR_ST_BADPARAM,ERR_LC_ECC,ERR_PT_ECCSTD,NULL);
		goto error;
	}
	ECp_copy(tp, E->G);
	ECp_free(tp);
	/* LN_long_set(E->G->z,1); */

	E->psize   = LN_now_bit(E->p);
	E->nsize   = LN_now_bit(E->n);
	E->version = 1;
	E->curve_type = type;
	return 0;
error:
	return -1;
}

ECParam *ECPm_get_std_parameter(int type){
	ECParam *ret;
	int err=-1;

	if((ret=ECPm_new())==NULL) goto done;
	err=ECPm_set_std_parameter(ret,type);
	
done:
	if(err&&ret){ECPm_free(ret);ret=NULL;}
	return ret;
}

/*-----------------------------------------
  get NIST recommended elliptic curves.
  (see FIPS 186-3 Appendix D.1)
-----------------------------------------*/
int ECPm_get_recommended_elliptic_curve(int size){
	if (size<224) {
		return ECP_X962_prime192v1;
	}
	if (size<256) {
		return ECP_secp224r1;
	}
	if (size<384) {
		return ECP_X962_prime256v1;
	}
	if (size<512) {
		return ECP_secp384r1;
	}
	return ECP_secp521r1;
}
