/* dsatestfc.c - test functions for DSA module. */
/*
 * Modified by National Institute of Informatics in Japan, 2012-2014.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef USE_PTHREAD
#include <pthread.h>
#endif

#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_dsa.h>
#include <aicrypto/ok_tool.h>

#define TEST_DSAPARAM	"dsapm.der"
#define TEST_DSAPARAM0	"dsapm.der~"	/* for writing */
#define TEST_DSAKEY	"dsakey.der"
#define TEST_DSAKEY0	"dsakey.der~"	/* for writing */
#define TEST_DSACERT	"pkix-ex1.ber"
#ifndef PATH
# define PATH	"."
#endif

/* test/getfpath.c */
char *get_fpath(char *path, char* filename);


int test_dsapm_der(){
	unsigned char *der=NULL,*der2=NULL;
	DSAParam *pm=NULL;
	int i,j,err=-1;
	char *fp_DSAPARAM = get_fpath(PATH, TEST_DSAPARAM);

	/* decode and encode DSA Param DER */
	for(j=0;j<11;j++){
		/* read DSA Param DER */
		if ((der = ASN1_read_der(fp_DSAPARAM)) == NULL) {
			printf("DSA : read file (1) -- error!\n");
			goto done;
		}

		if((pm=ASN1_read_dsaparam(der,0))==NULL){
			printf("DSA : decode parameter -- error!\n");
			goto done;
		}

		/* write DSA Param DER */
		if((der2=DSAPm_toDER(pm,NULL,&i,0))==NULL){
			printf("DSA : encode parameter -- error!\n");
			goto done;
		}
		if(ASN1_write_der(der2,TEST_DSAPARAM0)) goto done;

		if((j%5)==0) printf("DSA : decode & encode parameter    -- ok : %d\n",j);
	
		/* compare two parameters */
		if(memcmp(der,der2,i)){
			printf("DSA : compare DER parameters der==der2 -- error!\n");
			goto done;
		}
		if((j%5)==0) printf("DSA : compare DER binary der==der2 -- ok : %d\n",j);

		free(der2); der2=NULL;
		DSAPm_free(pm); pm=NULL; der=NULL; /* it frees "der" memory */
	}

	err=0;
done:
	if (fp_DSAPARAM) free(fp_DSAPARAM);
	if(der2) free(der2);
	if((pm==NULL)&&(der)) free(der);
	DSAPm_free(pm); /* it frees "der" memory */
	return err;
}

int test_dsakey_der(){
	unsigned char *der=NULL,*der2=NULL;
	Prvkey_DSA *prv=NULL;
	Pubkey_DSA *pub=NULL;
	int i,j,err=-1;
	char *fp_DSAKEY = get_fpath(PATH, TEST_DSAKEY);

	/* read & write DSA key */
	for(j=0;j<11;j++){
		/* read DSA Privatekey DER */
		if ((der = ASN1_read_der(fp_DSAKEY)) == NULL) goto done;

		if((prv=ASN1_read_dsaprv(der))==NULL){
			printf("DSA : decode private key -- error!\n");
			goto done;
		}

		/* write EC Param DER */
		if((der2=DSAprv_toDER(prv,NULL,&i))==NULL){
			printf("DSA : encode private key -- error!\n");
			goto done;
		}
		if(ASN1_write_der(der2,TEST_DSAKEY0)) goto done;

		if((j%5)==0) printf("DSA : decode & encode private key  -- ok : %d\n",j);
	
		/* compare two parameters */
		if(memcmp(der,der2,i)){
			printf("DSA : compare DER parameters der==der2 -- error!\n");
			goto done;
		}
		if((j%5)==0) printf("DSA : compare DER binary der==der2 -- ok : %d\n",j);

		free(der2); der2=NULL;
		DSAkey_free((Key*)prv); prv=NULL; der=NULL; /* it frees "der" memory */
	}

	/* compare DSA keys */
	for(j=0;j<11;j++){
		if((pub=DSApubkey_new())==NULL) goto done;
		if ((der = ASN1_read_der(fp_DSAKEY)) == NULL) goto done;

		if((prv=ASN1_read_dsaprv(der))==NULL){
			printf("DSA : decode private key -- error!\n");
			goto done;
		}
		DSAprv_2pub(prv,pub);

		if(Key_cmp((Key*)prv,(Key*)prv)){
			printf("DSA : compare private key -- error!\n");
			goto done;
		}
		if(Key_cmp((Key*)pub,(Key*)pub)){
			printf("DSA : compare public key -- error!\n");
			goto done;
		}
		if(!Key_cmp((Key*)prv,(Key*)pub)){
			printf("DSA : compare prv & pub key -- error!\n");
			goto done;
		}
		if(Key_pair_cmp((Key*)prv,(Key*)pub)){
			printf("DSA : compare prv & pub key -- error!\n");
			goto done;
		}

		if((j%5)==0) printf("DSA : compare prv & pub keys -- ok : %d\n",j);

		Key_free((Key*)pub);
		Key_free((Key*)prv); /* it frees "der" memory */
		prv=NULL; der=NULL;
	}
	err=0;
done:
	if (fp_DSAKEY) free(fp_DSAKEY);
	if(der2) free(der2);
	if((prv==NULL)&&(der)) free(der);
	DSAkey_free((Key*)prv); /* it frees "der" memory */
	return err;
}

int test_dsa_sig(){
	char *txt = "This is sample text."; /* this should be smaller than 20 byte :-) */
	unsigned char *der=NULL,*sig=NULL;
	Prvkey_DSA *prv=NULL;
	Pubkey_DSA *pub=NULL;
	DSAParam *pm=NULL;
	int i,j,err=-1;
	char *fp_DSAKEY = get_fpath(PATH, TEST_DSAKEY);
	char *fp_DSAPARAM = get_fpath(PATH, TEST_DSAPARAM);

	/* read DSA private key & check signature */
	for(j=0;j<11;j++){
		/* read DSA Privatekey DER */
		if ((der = ASN1_read_der(fp_DSAKEY)) == NULL) goto done;

		if((prv=ASN1_read_dsaprv(der))==NULL){
			printf("DSA : decode private key -- error!\n");
			goto done;
		}

		/* get DSA signature */
		if((sig=DSA_get_signature(prv,txt,strlen(txt),&i))==NULL){
			goto done;
		}
		/* verify DSA signature */
		if((pub=DSApubkey_new())==NULL) goto done;
		if(DSAprv_2pub(prv,pub)) goto done;

		i=DSA_vfy_signature(pub,txt,strlen(txt),sig);

		if(i<0){
			printf("cannot verify DSA signature -- error! : %d\n",j);
			goto done;
		}
		if(i>0){
			printf("Bad DSA signature -- verification error! : %d\n",j);
			goto done;
		}

		printf("DSA signature : generate & verify -- ok:%d\n",j);

		DSAkey_free((Key*)prv); prv=NULL;
		DSAkey_free((Key*)pub); pub=NULL;
		free(sig); sig=NULL; der=NULL;
	}

	/* generate DSA private key & check signaturte */
	for(j=0;j<11;j++){
		/* read DSA Param DER */
		if ((der = ASN1_read_der(fp_DSAPARAM)) == NULL) goto done;

		if((pm=ASN1_read_dsaparam(der,0))==NULL){
			printf("DSA : decode parameter -- error!\n");
			goto done;
		}

		/* generate new private key */
		if((prv=DSAprvkey_new())==NULL) goto done;
		if(DSAprv_generate(pm,prv)) goto done;
		DSAPm_free(pm); pm=NULL;

		/* get DSA signature */
		if((sig=DSA_get_signature(prv,txt,strlen(txt),&i))==NULL) goto done;

		/* verify DSA signature */
		if((pub=DSApubkey_new())==NULL) goto done;
		if(DSAprv_2pub(prv,pub)) goto done;

		i=DSA_vfy_signature(pub,txt,strlen(txt),sig);

		if(i<0){
			printf("cannot verify DSA signature (2) -- error! : %d\n",j);
			goto done;
		}
		if(i>0){
			printf("Bad DSA signature (2) -- verification error! : %d\n",j);
			goto done;
		}

		printf("DSA private key generation -- ok:%d\n",j);

		DSAkey_free((Key*)prv); prv=NULL;
		DSAkey_free((Key*)pub); pub=NULL;
		free(sig); sig=NULL; der=NULL;
	}

	err=0;
done:
	if (fp_DSAKEY) free(fp_DSAKEY);
	if (fp_DSAPARAM) free(fp_DSAPARAM);
	if(sig) free(sig);
	if((prv==NULL)&&(der)) free(der);
	DSAkey_free((Key*)prv);
	DSAkey_free((Key*)pub);
	return err;
}

int test_dsa_gen(){
	char *txt = "This is sample text."; /* this should be smaller than 20 byte :-) */
	unsigned char *sig=NULL;
	Prvkey_DSA *prv=NULL;
	Pubkey_DSA *pub=NULL;
	DSAParam *pm;
	int i=0,j,k,err=-1;
	int bit[] = {1024, 2048, 2048, 3072};
	const int number_tests = sizeof(bit) / sizeof(bit[0]);
	int hash[]={OBJ_HASH_SHA1,OBJ_HASH_SHA224,OBJ_HASH_SHA256,OBJ_HASH_SHA256};
	int testbit,testhash;

	do{
		testbit = bit[i % number_tests];
		testhash = hash[i % number_tests];

		printf("----- generating DSA parameter test stage %d (%d bit)-----\n",i,testbit);

		/*OK_set_sign_digest_algo(testhash);*/
		for(k=0;k<2;k++){
			if((pm=DSAPm_gen_parameter(testbit,testhash))==NULL){
				printf("DSA Parameter generation error occured!!\n");
				goto done;
			}

			/* generate new private key */
			if((prv=DSAprvkey_new())==NULL) goto done;
			if(DSAprv_generate(pm,prv)) goto done;

			/* get DSA signature */
			if((sig=DSA_get_signature(prv,txt,strlen(txt),&j))==NULL) goto done;

			/* verify DSA signature */
			if((pub=DSApubkey_new())==NULL) goto done;
			if(DSAprv_2pub(prv,pub)) goto done;

			j=DSA_vfy_signature(pub,txt,strlen(txt),sig);

			if(j<0){
				printf("cannot verify DSA signature (2) -- error! : %d\n",j);
				goto done;
			}
			if(j>0){
				printf("Bad DSA signature (2) -- verification error! : %d\n",j);
				goto done;
			}
			printf("\nDSA Parameter generation -- signature test ok!!\n");

			DSAPm_free(pm);
			DSAkey_free((Key*)prv);
			DSAkey_free((Key*)pub);
			free(sig); sig=NULL;
			pm=NULL; prv=NULL; pub=NULL;
		}

		i++;
	} while (i < number_tests);

	err=0;
done:
	if(sig) free(sig);
	DSAPm_free(pm);
	DSAkey_free((Key*)prv);
	DSAkey_free((Key*)pub);
	return err;
}
