/* dsa_gen.c */
/*
 * Copyright (c) 2012-2017 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2002
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include "aiconfig.h"

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_rand.h>
#include <aicrypto/ok_tool.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_dsa.h>

#define DSA_MAX_DIGESTSIZE	SHA256_DIGESTSIZE
#define DSA_MAX_KEYSIZE_BITLENGTH	3072
#define PRIME_MAX	2000	/* defined in large_prime.h */
extern const unsigned int prime[];	/* ditto */

int iter = 64;	/* Miller-Rabin iterations. the minimum for L=3072 is 64 */
int outlen;	/* bit length of the output block from Hash() */
int hs;	/* byte length of the output block */
int n;	/* variable in step 3 (FIPS 186-4, A.1.1.2) */
int bx;	/* variable 'b' in step 4 (ditto) */

static int check_selection_appropriateness(int size, int hash_algo, int *itr);
int dsapm_gen_q(DSAParam *pm, int digest_algo, unsigned char *seed,
		int seedlen, LNm *n1, LNm *a, LNm *b);
int dsapm_gen_p(DSAParam *pm, int digest_algo, int size,
		const unsigned char *seed, int seedlen,
		LNm *n1, LNm *a, LNm *b);

/**
 * DSA Param generator
 *	size: private key size (1024 or 2048 or 3072)
 *	hash_algo: OBJ_HASH_SHA1/OBJ_HASH_SHA224/OBJ_HASH_SHA256
 *
 *	size (L) | hash_algo (N)
 *	---------+--------------
 *	1024     | OBJ_HASH_SHA1   (160bit)
 *	2048     | OBJ_HASH_SHA224 (224bit)
 *	2048     | OBJ_HASH_SHA256 (256bit)
 *	3072     | OBJ_HASH_SHA256 (256bit)
 */
DSAParam *DSAPm_gen_parameter(int size /* bits */, int hash_algo)
{
	int i, err = -1;
	unsigned char seed[DSA_MAX_DIGESTSIZE];	/* domain_parameter_seed */
	DSAParam *ret = NULL;
	LNm *n1 = NULL, *a = NULL, *b = NULL;
	int seedlen;	/* in bytes */

	if (hash_algo == OBJ_HASH_SHA384 ||
	    hash_algo == OBJ_HASH_SHA512 ||
	    hash_algo == OBJ_HASH_SHA3_384 ||
	    hash_algo == OBJ_HASH_SHA3_512) {
		OK_set_error(ERR_ST_UNSUPPORTED_ALGO, ERR_LC_DSA, ERR_PT_DSAGEN, NULL);
		goto done;
	}

	/*
	 * step 1 in FIPS 186-4, A.1.1.2
	 * check that the (L,N) pair is in the list of acceptable (L,N pairs)
	 *
	 * note that not N but 'outlen' is checked here.
	 */
	if (check_selection_appropriateness(size, hash_algo, &iter) == 0) {
		OK_set_error(ERR_ST_BADPARAM, ERR_LC_DSA, ERR_PT_DSAGEN, NULL);
		goto done;
	}

	hs = hash_size(hash_algo);
	outlen = hs << 3;	/* in bits */

	if ((a = LN_alloc()) == NULL) goto done;
	if ((b = LN_alloc()) == NULL) goto done;
	if ((n1 = LN_alloc()) == NULL) goto done;
	if ((ret = DSAPm_new()) == NULL) goto done;

	/*
	 * step 2
	 * check the condition for 'seedlen'
	 *
	 * in FIPS 186-4, A.1.1.2, seedlen is an input.
	 * since we choose seedlen as follows, the condition
	 * seedlen >= N is automatically satisfied (seedlen = outlen >= N).
	 */
	seedlen = hs;	/* in bytes */

	/*
	 * step 3
	 * compute n = CEIL(L / outlen) - 1
	 *
	 * for example, if L = 1024, outlen = 160, then
	 * n = CEIL(1024 / 160) - 1 = 7 - 1 = 6.
	 * if L = 2048, outlen = 224, then
	 * n = CEIL(2048 / 224) - 1 = 10 - 1 = 9.
	 */
	if (size % outlen == 0) n = size / outlen - 1;
	else n = size / outlen;

	/*
	 * step 4
	 * compute b = L - 1 - (n * outlen)
	 *
	 * for example, if L = 1024, outlen = 160, then
	 * b = 1024 - 1 - (6 * 160) = 63.
	 * if L = 2048, outlen = 224, then
	 * b = 2048 - 1 - (9 * 224) = 31.
	 *
	 *   L   | outlen | n  | b
	 * ------+--------+----+----
	 *  1024 | 160    | 6  | 63
	 *  2048 | 224    | 9  | 31
	 *  2048 | 256    | 7  | 255
	 *  3072 | 256    | 11 | 255
	 */
	bx = size - 1 - (n * outlen);

	do {
	/* phase 1 : get q */
		i = dsapm_gen_q(ret, hash_algo, seed, seedlen, n1, a, b);
		if (i < 0) goto done;

	/* phase 2 : get p */
		i = dsapm_gen_p(ret, hash_algo, size, seed, seedlen, n1, a, b);
		if (i < 0) goto done;
	} while (i);

	/* phase 3 : get g */
	/*
	 * step 1 in FIPS 186-4, A.2.1
	 * compute e = (p-1) / q
	 */
	LN_copy(ret->p, n1);
	LN_long_sub(n1, 1);
	LN_div_mod(n1, ret->q, a, b);	/* set e to a */

	for (;;) {
	/*
	 * step 2
	 * set h = (any integer) satisfying 1 < h < (p - 1)
	 */
		if (RAND_bytes(seed, 4))	/* XXX: why 4 bytes? */
			goto done;
		if (LN_set_num_c(b, 4, seed))	/* set h to b */
			goto done;
	/*
	 * step 3
	 * compute g = h^e mod p
	 */
		LN_exp_mod(b, a, ret->p, ret->g);
	/*
	 * step 4
	 * if (g = 1), then go step 2
	 * step 5
	 * return g
	 */
		if (LN_now_bit(ret->g) > 1)
			break;
	}

	ret->version = 1;	/* XXX: what is this for? */
	if ((ret->der = DSAPm_toDER(ret, NULL, &i, 0)) == NULL)
		goto done;

	err = 0;
done:
	LN_free(a);
	LN_free(b);
	LN_free(n1);
	if (ret && err) {
		DSAPm_free(ret);
		ret = NULL;
	}
	return ret;
}

/**
 * check the size of L and outlen, and set the Miller-Rabin iterations.
 * 
 * 'L' is the bit length of 'p'.
 * 'outlen' is the bit length of the output block of the selected hash function,
 * where outlen SHALL be equal to or greater than 'N', that is, outlen >= N.
 * (from FIPS 186-4, A.1.1.2)
 *
 * if the condition is not satisfied, then return INVALID (0)
 */
static int check_selection_appropriateness(int size, int hash_algo, int *itr)
{
	int csa_hs, len;
	csa_hs = hash_size(hash_algo);
	len = csa_hs << 3;	/* in bits */

	switch (size) {
	case 1024:	/* (L,N) = (1024,160) */
		*itr = 40;	/* see Table C.1 in FIPS 186-4 */
		return (len >= 160);
	case 2048:	/* (L,N) = (2048,224 or 256) */
		*itr = 56;	/* ditto */
		return (len >= 224);
	case 3072:	/* (L,N) = (3072,256) */
		*itr = 64;	/* ditto */
		return (len >= 256);
	default:
		return 0;
	}
}

/**
 * generate the desired prime 'q'
 *
 * 'q' is a prime divisor of (p-1), where 2^(N-1) < q < 2^N,
 * and 'N' is the bit length of 'q'.
 * 'N' takes one of the following values: 160, 224, or 256 (20, 28 or 32 byte)
 */
int dsapm_gen_q(DSAParam *pm, int digest_algo, unsigned char *seed,
		int seedlen /* in bytes */, LNm *n1, LNm *a, LNm *b)
{
	unsigned char buf[DSA_MAX_DIGESTSIZE];
	unsigned char *cp;
	uint32_t md;
	int ret_len;
	int res = -1;
	int i, j = 0;

	do {
		/*
		 * step 5 in A.1.1.2
		 * get an arbitrary sequence of 'seedlen' bits as
		 * the domain_parameter_seed.
		 */
		if (RAND_bytes(seed, seedlen)) goto error;

		/*
		 * step 6 (1/2)
		 * compute Hash(domain_parameter_seed)
		 */
		cp = OK_do_digest(digest_algo, seed, seedlen, buf, &ret_len);
		/*
		 * step 6 (2/2)
		 * compute U = Hash(domain_parameter_seed) mod 2^(N-1)
		 *
		 * the following should be modified if seedlen > N.
		 */
		buf[0] &= 0x7f;	/* & 0111 1111 */

		/*
		 * step 7 (1/2)
		 * compute 2^(N-1) + U + 1 - (U mod 2)
		 *
		 * actually, make it an N-bit odd number.
		 */
		buf[0] |= 0x80;
		if ((buf[seedlen - 1] & 0x01) == 0x00) {
			buf[seedlen - 1] |= 0x01;
		}
		/*
		 * step 7 (2/2)
		 * set q = 2^(N-1) + U + 1 - (U mod 2)
		 *
		 * note that (seedlen<<3) = outlen = N
		 * in current implementation.
		 */
		if (LN_set_num_c(pm->q, seedlen, buf)) goto error;

		/* print out the process */
		if (!(++j % 5)) {
			printf(".");
			fflush(stdout);
		}

		/*
		 * step 8 
		 * test whether or not q is prime as specified in Appendix C.3.
		 *
		 * do easy test first
		 */
		for (i = 0; i < PRIME_MAX; i++) {
			if (LN_long_mod(pm->q, prime[i], &md))
				goto error;
			if (md == 0)
				break;
		}
		if (md == 0) continue;

		/*
		 * Miller-Rabin test
		 * 'iter' must be set appropriate value by
		 * check_selection_appropriateness().
		 */
		res = _LN_miller_rabin(pm->q, iter, PROGBAR_ON, n1, a, b);
		if (res < 0) goto error;

		/*
		 * step 9
		 * if q is not prime, go to step 5
		 *
		 * _LN_miller_rabin() will return 1 when q is composite.
		 */
	} while (res);

	return 0;
error:
	return -1;
}

/**
 * generate the desired prime 'p'
 *
 * 'p' is a prime modulus, where 2^(L-1) < p < 2^L,
 * and 'L' is the bit length of 'p'.
 */
int dsapm_gen_p(DSAParam *pm, int digest_algo, int size,
		const unsigned char *seed, int seedlen,
		LNm *n1, LNm *a, LNm *b)
{
	int offset, counter;
	int i, j;
	unsigned char tmp[DSA_MAX_DIGESTSIZE];
	unsigned char t;	/* least significant 'element' */
	int cr;
	unsigned char *cp;
	/*
	 * buffer for W and X
	 * the maximum size of the buffer is (n + 1) * DSA_MAX_DIGESTSIZE.
	 * the maximum of n is 11 when (L, outlen) = (3072, 256).
	 */
	unsigned char buf[12 * DSA_MAX_DIGESTSIZE];
	int mse = hs - ((bx + 1) >> 3);	/* most significant element of buf[] */
	int ret_len;
	uint32_t md;
	int res;

	/*
	 * step 10 in FIPS 186-4, A.1.1.2
	 * set offset = 1
	 */
	offset = 1;

	/*
	 * copying seed[] to another buffer
	 * this is FIPS-compatible? (comment since v1.0)
	 *
	 * maybe, YES, because domain_parameter_seed must not be changed
	 * in the generation of 'p'
	 */
	memcpy(tmp, seed, seedlen);

	/*
	 * step 11
	 * for counter = 0 to (4L-1) do 
	 */
	for (counter = 0; counter < size * 4; counter++) {
		/*
		 * step 11.1
		 * for j = 0 to n do
		 */
		for (j = 0; j < n + 1; j++) {
		/*
		 * step 11.1 (1/3)
		 * compute domain_parameter_seed + offset + j
		 */
			t = tmp[seedlen - 1];
			/* carry the one to next place if needed */
			cr = (t > (tmp[seedlen - 1] += offset + j)) ? (1) : (0);
			for (i = seedlen - 2; i >= 0; i--)
				if ((tmp[i] += cr) != 0)
					break;
		/*
		 * step 11.1 (2/3)
		 * compute (domain_parameter_seed + offset + j)
		 *         mod 2^(seedlen<<3)
		 * 
		 * we do not need anything because tmp[0] is always
		 * the most significant element.
		 * XXX: is this right?
		 */

		/*
		 * step 11.1 (3/3)
		 * compute V_{j} = Hash((domain_parameter_seed + offset + j)
		 *                      mod 2^(seedlen<<3))
		 *
		 * step 11.2 (that can be included in this loop)
		 * compute W = V_{0} + (V_{1} * 2^(outlen)) + ... +
		 *           ((V_{n} mod 2^b) * 2^(n*outlen))
		 *
		 * note that buf[] is not W itself but
		 * W' = V_{0} + ... + (V_{n} * 2^(n*outlen))
		 */
			cp = OK_do_digest(digest_algo, tmp, seedlen,
					  &buf[hs * (n - j)], &ret_len);
		}
		/*
		 * step 11.2 (continued)
		 * compute V_{n} mod 2^b
		 *
		 * to do so, find the most siginicant element of buf[].
		 * it is just buf[hs - ((b + 1) >> 3)].
		 */
		
		/*
		 * step 11.3
		 * set X = W + 2^(L-1)
		 */
		buf[mse] |= 0x80;
		/*
		 * create the LNm object 'n1' from X, i.e., buf[].
		 */
		if (LN_set_num_c(n1, (size >> 3), &buf[mse]) < 0)
			goto error;

		/*
		 * step 11.4
		 * compute c = X mod 2q
		 *
		 * set 2q to pm->p
		 */
		if (LN_lshift32(pm->q, 1, pm->p) < 0)
			goto error;
		/*
		 * set (X mod 2q) to b
		 */
		if (LN_div_mod(n1, pm->p, a, b) < 0)
			goto error;

		/*
		 * step 11.5
		 * compute p = X - (c - 1) = (X - c) + 1
		 */
		if (LN_minus(n1, b, pm->p) < 0)
			goto error;
		if (LN_long_add(pm->p, 1) < 0)
			goto error;

		/* print out the process */
		if (!(counter % 5)) {
			printf("+");
			fflush(stdout);
		}

		/*
		 * step 11.6
		 * check the bit length of p
		 */
		if (LN_now_bit(pm->p) < size)
			goto STEP_NINE;

		/*
		 * step 11.7
		 * test whether or not p is prime as specified in Appendix C.3
		 *
		 * do easy test first
		 */
		for (i = 0; i < PRIME_MAX; i++) {
			if (LN_long_mod(pm->p, prime[i], &md))
				goto error;
			if (md == 0)
				break;
		}
		if (md == 0) goto STEP_NINE;
		/*
		 * Miller-Rabin test
		 * 'iter' must be set appropriate value by
		 * check_selection_appropriateness().
		 */
		res = _LN_miller_rabin(pm->p, iter, PROGBAR_ON, n1, a, b);
		if (res < 0) goto error;

		/*
		 * step 11.8
		 * if 'p' is determined to be prime, return VALID (0)
		 */
		if (res == 0)
			return 0;

		/*
		 * step 11.9
		 * set offset = offset + n + 1
		 */
	STEP_NINE:
		offset += n + 1;
	}

	/*
	 * step 12
	 * go to step 5.
	 */
	return 1;
error:
	return -1;
}
