/* cmptestfc.c */
/*
 * Modified by National Institute of Informatics in Japan, 2013-2017.
 *
 */
/*
 * Copyright (C) 1998-2002
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_io.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_asn1.h>
#include <aicrypto/ok_cmp.h>
#include <aicrypto/ok_pem.h>
#include <aicrypto/ok_rand.h>
#include <aicrypto/ok_tool.h>

#define TEST_CERT	"test.cer"
#define TEST_CRL	"test.crl"
#define TEST_CSR	"test.p10"
#ifndef PATH
# define PATH	"."
#endif

/* test/getfpath.c */
char *get_fpath(char *path, char *filename);


int set_dummy_message(PKIMessage *pkim);

int set_dummy_head(PKIHeader *pkih);

int do_body_der_test(PKIBody *body);
int set_dummy_bd_certreqmsg(PKIBD_CertReqMsg *bd);
int set_dummy_bd_certrspmsg(PKIBD_CertRepMsg *bd);
int set_dummy_bd_pkcs10(PKIBD_CertAnn *csr);
int set_dummy_bd_popoch(PKIBD_PopoCH *bd);
int set_dummy_bd_popors(PKIBD_PopoRS *bd);
int set_dummy_bd_keyrecrsp(PKIBD_RecRep *bd);
int set_dummy_bd_revreq(PKIBD_RevReq *bd);
int set_dummy_bd_revreq_in(PKIBD_RevReq *bd);
int set_dummy_bd_revrsp(PKIBD_RevRep *bd);
int set_dummy_bd_keyupd(PKIBD_KeyUpDAnn *bd);
int set_dummy_bd_certann(PKIBD_CertAnn *bd);
int set_dummy_bd_revann(PKIBD_RevAnn *bd);
int set_dummy_bd_crlann(PKIBD_CRLAnn *bd);
int set_dummy_bd_nested(PKIBD_Nested *bd);
int set_dummy_bd_genmsg(PKIBD_GenMsg *bd);
int set_dummy_bd_errmsg(PKIBD_ErrMsg *bd);

int set_dummy_certreq(PKIBD_CertReqMsg *bd,int popchoice);
POfP *set_dummy_pop(int popchoice,int sigauth);
POPOSigningKey *set_dummy_popsig(int sa);
PKMACValue *set_dummy_pkmacv();
CertResponse *set_dummy_certrsp();
int set_dummy_challenge(Challenge *ch);
CertTemplate *set_dummy_certtmpl();
CertifiedKeyPair *set_dummy_crtkeypair(int choice,int action);
EncryptedValue *set_dummy_encryval();
PKIPubInfo *set_dummy_pubinfo(int action);
PKIStatusInfo *set_dummy_pkistatinfo(int stat,char *text,int finfo);
InfoTAV *set_dummy_infotype1();
CertList *set_dummy_certlist();
int set_dummy_certid(CertId *id);


/*-----------------------------------------
  CMP Message test 
-----------------------------------------*/
int test_cmp_message(){
	PKIMessage *pkim=NULL,*pkim2=NULL;
	unsigned char *uc=NULL,*uc2=NULL;
	int i,j,err=-1;

	/* read & write message DER binary */
	if((pkim=PKImsg_new())==NULL) goto done;
	if(set_dummy_message(pkim)) goto done;
	if((uc=PKImsg_toDER(pkim,NULL,&j))==NULL) goto done;

	if((pkim2=ASN1_read_pkimsg(uc))==NULL) goto done;
	if((uc2=PKImsg_toDER(pkim2,NULL,&i))==NULL) goto done;

	if(j!=i) goto done;
	if(memcmp(uc,uc2,i)) goto done;
	printf("CMP message : read & write DER -- ok\n");

	err=0;
done:
	if(uc)  free(uc);
	if(uc2) free(uc2);
	PKImsg_free(pkim);
	PKImsg_free(pkim2);
	if(err){printf("CMP message : error !!\n");}
	return err;
}

int set_dummy_message(PKIMessage *pkim){
	if((pkim->header=PKIhead_new())==NULL) return -1;
	if(set_dummy_head(pkim->header)) return -1;

	if((pkim->body=PKIbody_new(0))==NULL) return -1;
	if(set_dummy_bd_certreqmsg((PKIBD_CertReqMsg*)pkim->body)) return -1;

	if((pkim->protection=(unsigned char*)malloc(64))==NULL) return -1;
	RAND_bytes(pkim->protection,64);
	pkim->plen = 64;

	if((pkim->extraCerts=set_dummy_certlist())==NULL) return -1;

	return 0;
}

/*-----------------------------------------
  CMP Header test 
-----------------------------------------*/
int test_cmp_header(){
	PKIHeader *pkih=NULL,*pkih2=NULL;
	unsigned char *uc=NULL,*uc2=NULL;
	int i,j,err=-1;

	RAND_init();

	/* header alloc & free */
	if((pkih=PKIhead_new())==NULL) goto done;
	if(set_dummy_head(pkih)) goto done;
	PKIhead_free(pkih);
	printf("CMP header : memory allocation & free -- ok\n");

	/* read & write header DER binary */
	if((pkih=PKIhead_new())==NULL) goto done;
	if(set_dummy_head(pkih)) goto done;
	if((uc=PKIhead_toDER(pkih,NULL,&j))==NULL) goto done;

	if((pkih2=ASN1_read_pkihead(uc))==NULL) goto done;
	if((uc2=PKIhead_toDER(pkih2,NULL,&i))==NULL) goto done;

	if(j!=i) goto done;
	if(memcmp(uc,uc2,i)) goto done;
	printf("CMP header : read & write DER -- ok\n");

	err=0;
done:
	if(uc)  free(uc);
	if(uc2) free(uc2);
	PKIhead_free(pkih);
	PKIhead_free(pkih2);
	if(err){printf("CMP header : error !!\n");}
	return err;
}

int set_dummy_head(PKIHeader *pkih){
	Cert *ct;
	time_t t;
	struct tm *stm;
	int i,err=-1;
	char *fp_TEST_CERT = get_fpath(PATH, TEST_CERT);

	if ((ct = Cert_read_file(fp_TEST_CERT)) == NULL) goto done;

	pkih->pvno = 1;

	Cert_dncopy(&ct->issuer_dn, &pkih->sender);
	Cert_dncopy(&ct->subject_dn,&pkih->recipient);

	time(&t);
	stm=(struct tm*)gmtime(&t);
	memcpy(&pkih->messageTime,stm,sizeof(struct tm));
	pkih->protectionAlg = OBJ_CRYPT_RSA;

	pkih->skid_len = 8;
	pkih->rkid_len = 8;
	if((pkih->senderKID=malloc(pkih->skid_len))==NULL) goto done;
	if((pkih->recipKID =malloc(pkih->rkid_len))==NULL) goto done;
	RAND_bytes(pkih->senderKID,pkih->skid_len);
	RAND_bytes(pkih->recipKID, pkih->rkid_len);
	
	pkih->trid_len = 8;
	if((pkih->transactionID=malloc(pkih->trid_len))==NULL) goto done;
	RAND_bytes(pkih->transactionID, pkih->trid_len);

	pkih->snon_len = 8;
	pkih->rnon_len = 8;
	if((pkih->senderNonce=malloc(pkih->snon_len))==NULL) goto done;
	if((pkih->recipNonce =malloc(pkih->rnon_len))==NULL) goto done;
	RAND_bytes(pkih->senderNonce,pkih->snon_len);
	RAND_bytes(pkih->recipNonce, pkih->rnon_len);

	for(i=0;i<4;i++){
		if ((pkih->freeText[i] = strdup("sample freeText\n")) == NULL)
			goto done;
	}
	if((pkih->generalInfo=set_dummy_infotype1())==NULL) goto done;

	err=0;
done:
	if (fp_TEST_CERT) free(fp_TEST_CERT);
	Cert_free(ct);
	return err;
}

/*-----------------------------------------
  CMP Body test 
-----------------------------------------*/
int test_cmp_body(){
	PKIBody *body=NULL;
	int i,err=-1;
	int *funcs[24]={
		(int*)set_dummy_bd_certreqmsg,
		(int*)set_dummy_bd_certrspmsg,
		(int*)set_dummy_bd_certreqmsg,
		(int*)set_dummy_bd_certrspmsg,
		(int*)set_dummy_bd_pkcs10,
		(int*)set_dummy_bd_popoch,
		(int*)set_dummy_bd_popors,
		(int*)set_dummy_bd_certreqmsg,
		(int*)set_dummy_bd_certrspmsg,
		(int*)set_dummy_bd_certreqmsg,
		(int*)set_dummy_bd_keyrecrsp,
		(int*)set_dummy_bd_revreq,
		(int*)set_dummy_bd_revrsp,
		(int*)set_dummy_bd_certreqmsg,
		(int*)set_dummy_bd_certrspmsg,
		(int*)set_dummy_bd_keyupd,
		(int*)set_dummy_bd_certann,
		(int*)set_dummy_bd_revann,
		(int*)set_dummy_bd_crlann,
		0, /* Confirm */
		(int*)set_dummy_bd_nested, /* nested */
		(int*)set_dummy_bd_genmsg,
		(int*)set_dummy_bd_genmsg,
		(int*)set_dummy_bd_errmsg,
	};

	for(i=0;i<24;i++){
		printf("CMP body (%d) : read & write DER ",i);

		if((body=PKIbody_new(i))==NULL) goto done;
		if(funcs[i]){
			if(((int (*)(PKIBody*))funcs[i])(body))
				goto done;
		}
		if(do_body_der_test(body)) goto done;
		PKIbody_free(body);

		printf("-- ok\n");
	}

	err=0;
done:
	if(err){printf("-- test error (%d)\n",i);}
	return err;
}

int do_body_der_test(PKIBody *body){
	unsigned char *uc=NULL,*uc2=NULL;
	PKIBody *body2=NULL;
	int i,j,err=-1;

	if((uc   =PKIbody_toDER(body,NULL,&i))==NULL) goto done;
	if((body2=ASN1_read_pkibody(uc))==NULL) goto done;
	if((uc2  =PKIbody_toDER(body2,NULL,&j))==NULL) goto done;
#if 1
	*uc=0x30; *uc2=0x30;
	ASN1_write_der(uc,"uc.der");
	ASN1_write_der(uc2,"uc2.der");
#endif
	if(i!=j) goto done;
	if(memcmp(uc,uc2,i)) goto done;

	err=0;
done:
	printf("-- DER size : uc = %d, uc2 = %d ",i,j);
	if(uc)  free(uc);
	if(uc2) free(uc2);
	PKIbody_free(body2);
	return err;
}

/*--------------------------------*/
int set_dummy_bd_certreqmsg(PKIBD_CertReqMsg *bd){
	PKIBD_CertReqMsg *hd;

	if(set_dummy_certreq(bd,-1)) return -1;

	if((bd->next=(PKIBD_CertReqMsg*)PKIbody_new(PKIBD_CERT_REQ))==NULL) return -1;
	hd = bd->next;
	if(set_dummy_certreq(hd,0)) return -1;

	if((hd->next=(PKIBD_CertReqMsg*)PKIbody_new(PKIBD_CERT_REQ))==NULL) return -1;
	hd = hd->next;
	if(set_dummy_certreq(hd,1)) return -1;

	return 0;
}

int set_dummy_bd_pkcs10(PKIBD_CertAnn *csr){
	char *fp_TEST_CSR = get_fpath(PATH, TEST_CSR);

	if ((csr->cert = Req_read_file(fp_TEST_CSR)) == NULL) return -1;

	if (fp_TEST_CSR) free(fp_TEST_CSR);
	return 0;
}

int set_dummy_bd_popoch(PKIBD_PopoCH *bd){
	bd->num = 2;
	if(set_dummy_challenge(&bd->chall[0])) return -1;
	if(set_dummy_challenge(&bd->chall[1])) return -1;
	return 0;
}

int set_dummy_bd_popors(PKIBD_PopoRS *bd){
	int i;

	bd->num = 4;
	for(i=0;i<bd->num;i++)
		bd->content[i] = 0x44332211;
	return 0;
}

int set_dummy_bd_certrspmsg(PKIBD_CertRepMsg *bd){
	if((bd->caPubs=set_dummy_certlist())==NULL) return -1;
	if((bd->response=set_dummy_certrsp())==NULL) return -1;
	if((bd->response->next=set_dummy_certrsp())==NULL) return -1;
	return 0;
}

int set_dummy_bd_keyrecrsp(PKIBD_RecRep *bd){
	CertifiedKeyPair *ckp;
	char *fp_TEST_CERT = get_fpath(PATH, TEST_CERT);

	if((bd->status=set_dummy_pkistatinfo(PKISTAT_REJECTION,"dummy error message",
		PKIFL_BADALG|PKIFL_BADBADTIME|PKIFL_WRONGAUTH|PKIFL_MISSTMSTP))==NULL)
		goto error;

	if ((bd->newSigCert = Cert_read_file(fp_TEST_CERT)) == NULL) goto error;

	if((bd->caCerts=set_dummy_certlist())==NULL) goto error;

	/* SEQ OF CertifiedKeyPair */
	if((ckp=set_dummy_crtkeypair(1,1))==NULL)
		goto error;
	bd->keyPairHist = ckp;
	if((ckp=set_dummy_crtkeypair(0,1))==NULL)
		goto error;
	ckp->next = bd->keyPairHist;
	bd->keyPairHist = ckp;
	if((ckp=set_dummy_crtkeypair(0,0))==NULL)
		goto error;
	ckp->next = bd->keyPairHist;
	bd->keyPairHist = ckp;

	return 0;
error:
	if (fp_TEST_CERT) free(fp_TEST_CERT);
	return -1;
}

int set_dummy_bd_revreq(PKIBD_RevReq *bd){
	if(set_dummy_bd_revreq_in(bd)) return -1;

	if((bd->next=(PKIBD_RevReq*)PKIbody_new(PKIBD_RVOC_REQ))==NULL) return -1;
	if(set_dummy_bd_revreq_in(bd->next)) return -1;
	return 0;
}

int set_dummy_bd_revreq_in(PKIBD_RevReq *bd){
	time_t t;
	struct tm *stm;

	time(&t);
	stm=(struct tm*)gmtime(&t);
	memcpy(&bd->badSinceDate,stm,sizeof(struct tm));

	bd->revocationReason[1] = 0x80;
	if((bd->certDetails=set_dummy_certtmpl())==NULL) return -1;
	if((bd->crlEntryDetails=Extnew_keyusage(0x30))==NULL) return -1;

	return 0;
}

int set_dummy_bd_revrsp(PKIBD_RevRep *bd){
	PKIStatusInfo *stat;
	CertId *id;
	char *fp_TEST_CRL = get_fpath(PATH, TEST_CRL);

	if((stat=set_dummy_pkistatinfo(PKISTAT_REJECTION,"dummy error message",
		PKIFL_BADALG|PKIFL_BADBADTIME|PKIFL_WRONGAUTH|PKIFL_MISSTMSTP))==NULL)
		goto error;
	if((stat->next=set_dummy_pkistatinfo(PKISTAT_REJECTION,"dummy error message",
		PKIFL_BADALG|PKIFL_BADBADTIME|PKIFL_WRONGAUTH|PKIFL_MISSTMSTP))==NULL)
		goto error;
	bd->status = stat;

	if((id=CMP_certid_new())==NULL) goto error;
	if(set_dummy_certid(id)) goto error;
	if((id->next=CMP_certid_new())==NULL) goto error;
	if(set_dummy_certid(id->next)) goto error;
	bd->revCerts = id;

	if ((bd->crl = CRL_read_file(fp_TEST_CRL)) == NULL) goto error;

	return 0;
error:
	if (fp_TEST_CRL) free(fp_TEST_CRL);
	return -1;
}

int set_dummy_bd_keyupd(PKIBD_KeyUpDAnn *bd){
	char *fp_TEST_CERT = get_fpath(PATH, TEST_CERT);

	if ((bd->newWithNew = Cert_read_file(fp_TEST_CERT)) == NULL) return -1;
	if ((bd->newWithOld = Cert_read_file(fp_TEST_CERT)) == NULL) return -1;
	if ((bd->oldWithNew = Cert_read_file(fp_TEST_CERT)) == NULL) return -1;

	free(fp_TEST_CERT);
	return 0;
}

int set_dummy_bd_certann(PKIBD_CertAnn *bd){
	char *fp_TEST_CERT = get_fpath(PATH, TEST_CERT);

	if ((bd->cert = Cert_read_file(fp_TEST_CERT)) == NULL) return -1;

	free(fp_TEST_CERT);
	return 0;
}

int set_dummy_bd_revann(PKIBD_RevAnn *bd){
	time_t t;
	struct tm *stm;

	bd->status=PKISTAT_REJECTION;
	if(set_dummy_certid(&bd->certId)) return -1;

	time(&t);
	stm=(struct tm*)gmtime(&t);
	memcpy(&bd->willBeRevokedAt,stm,sizeof(struct tm));
	memcpy(&bd->badSinceData,stm,sizeof(struct tm));

	/* oops, crlDetails should not have this extension.
	 * but this one is just dummy extension for testing ;|
	 */
	if((bd->crlDetails=Extnew_basic_cons(0,0))==NULL) return -1;

	return 0;
}

int set_dummy_bd_crlann(PKIBD_CRLAnn *bd){
	char *fp_TEST_CRL = get_fpath(PATH, TEST_CRL);

	if ((bd->crl = CRL_read_file(fp_TEST_CRL)) == NULL) return -1;

	free(fp_TEST_CRL);
	return 0;
}

int set_dummy_bd_nested(PKIBD_Nested *bd){
	if((bd->msg=PKImsg_new())==NULL) return -1;
	if(set_dummy_message(bd->msg)) return -1;
	return 0;
}

int set_dummy_bd_genmsg(PKIBD_GenMsg *bd){
	if((bd->content=set_dummy_infotype1())==NULL) return -1;
	return 0;
}

int set_dummy_bd_errmsg(PKIBD_ErrMsg *bd){
	/* set "no error" */
	OK_clear_error();
	bd->errorCode = OK_get_error();
	if ((bd->errorDetails[0] = strdup(OK_get_errstr())) == NULL)
		return -1;

	if((bd->status=set_dummy_pkistatinfo(PKISTAT_REJECTION,"dummy error message",
		PKIFL_BADALG|PKIFL_BADBADTIME|PKIFL_WRONGAUTH|PKIFL_MISSTMSTP))==NULL)
		return -1;

	return 0;
}

/*--------------------------------------
  Other support functions.
--------------------------------------*/
int set_dummy_certreq(PKIBD_CertReqMsg *bd,int popchoice){
	AttrTAV *att;

	bd->certReq.certReqId = 0x44332211;
	if((bd->certReq.certTemplate=set_dummy_certtmpl())==NULL)
		goto error;
	if((bd->certReq.controls=Extnew_keyusage(0x30))==NULL)
		goto error;

	/* POP is OPTIONAL */
	if((popchoice>=0)&&(popchoice<4)){
		if((bd->pop = set_dummy_pop(popchoice,popchoice))==NULL)
			goto error;
	}
	/* (Attribute) OPTIONAL */
	if((att=Extnew_moj_timelimit(12))==NULL) goto error;
	if((att->next=Extnew_moj_suspcode(OBJ_HASH_SHA1,"abcdefgh",8))==NULL) goto error;
	bd->regInfo = att;

	return 0;
error:
	return -1;
}

POfP *set_dummy_pop(int popchoice,int sigauth){
	POfP *ret;

	if((ret=CMP_pofp_new())==NULL) goto error;
	ret->choice = popchoice;

	switch(popchoice){
	case 0: /* raVerified [0] NULL */
		break;
	case 1:
		if((ret->signature=set_dummy_popsig(sigauth))==NULL)
			goto error;
		break;
	case 2: /* keyEncipherment [2] POPPrivKey */
	case 3: /* keyAgreement [3] POPOPrivKey */
		/* Oops, in RFC2511, POPPrivKey ::= CHOICE {
		 * and modules are defined as implicit tags...
		 * i don't know how to operate it...
		 */
		break;
	}
	return ret;
error:
	CMP_pofp_free(ret);
	return NULL;
}

POPOSigningKey *set_dummy_popsig(int sa){
	POPOSigningKey *ret = NULL;
	Cert *ct;
	int err=-1;
	char *fp_TEST_CERT = get_fpath(PATH, TEST_CERT);

	if ((ct = Cert_read_file(fp_TEST_CERT)) == NULL) goto done;

	if((ret=CMP_poposign_new())==NULL) goto done;

	ret->algo_id = OBJ_SIG_SHA1RSA;
	ret->sig_len = 128;
	if((ret->signature=malloc(ret->sig_len))==NULL) goto done;
	RAND_bytes(ret->signature,ret->sig_len);

	switch(sa){
	case 0:
		ret->poposkInput.option=1;
		Cert_dncopy(&ct->subject_dn,&ret->poposkInput.sender);
		break;
	case 1:
		ret->poposkInput.option=1;
		if((ret->poposkInput.publicKeyMac=set_dummy_pkmacv())==NULL)
			goto done;
		break;
	default: /* not used -- it's just OPTIONAL */
		break;
	}
	if((ret->poposkInput.publicKey=Key_dup(ct->pubkey))==NULL)
		goto done;

	err=0;
done:
	Cert_free(ct);
	if(err){ CMP_poposign_free(ret); ret=NULL; }
	return ret;
}

PKMACValue *set_dummy_pkmacv(){
	PKMACValue *ret;

	if((ret=CMP_pkmacv_new())==NULL) goto error;

	ret->algId = OBJ_CRYALGO_RC2CBC;
	ret->vlen = 32;
	if((ret->value=malloc(ret->vlen))==NULL) goto error;
	RAND_bytes(ret->value,ret->vlen);

	return ret;
error:
	CMP_pkmacv_free(ret);
	return NULL;
}

CertResponse *set_dummy_certrsp(){
	CertResponse *ret;
	
	if((ret=CMP_certrsp_new())==NULL) goto error;

	ret->certReqID = 0x44332211;
	if((ret->status=set_dummy_pkistatinfo(PKISTAT_REJECTION,"dummy error message",
		PKIFL_BADALG|PKIFL_BADBADTIME|PKIFL_WRONGAUTH|PKIFL_MISSTMSTP))==NULL)
		goto error;
	if((ret->certifiedKeyPair=set_dummy_crtkeypair(1,1))==NULL)
		goto error;
	
	ret->rsp_len = 32;
	if((ret->rspInfo=malloc(ret->rsp_len))==NULL) goto error;
	RAND_bytes(ret->rspInfo,ret->rsp_len);

	return ret;
error:
	CMP_certrsp_free_all(ret);
	return NULL;
}

int set_dummy_challenge(Challenge *ch){
	ch->owf = OBJ_CRYPT_RSA;
	ch->ch_len = 16;
	ch->wit_len= 16;
	if((ch->challenge=malloc(ch->ch_len))==NULL) return -1;
	if((ch->witness  =malloc(ch->wit_len))==NULL) return -1;
	RAND_bytes(ch->challenge,ch->ch_len);
	RAND_bytes(ch->witness,ch->wit_len);
	return 0;
}

InfoTAV *set_dummy_infotype1(){
	InfoTAV *ret = NULL;
	unsigned char buf[64];
	char txtid[32] = "1.3.6.1.5.5.7.4.1";
	Cert *ct;
	char *fp_TEST_CERT = get_fpath(PATH, TEST_CERT);

	if ((ct = Cert_read_file(fp_TEST_CERT)) == NULL) goto error;

	if(str2objid(txtid,buf,64)<0) goto error;

	if((ret=CMP_infotype_new(ASN1_dup(buf),ct))==NULL)
		goto error;

	if((ret->next=Extnew_moj_genpres(2,OBJ_CRYALGO_3DESCBC,OBJ_CRYPT_RSA,OBJ_HASH_SHA1))==NULL)
		goto error;

	return ret;
error:
	if (fp_TEST_CERT) free(fp_TEST_CERT);
	if (ret) CMP_infotype_free(ret);
	return NULL;
}


PKIStatusInfo *set_dummy_pkistatinfo(int stat,char *text,int finfo){
	PKIStatusInfo *ret;

	if((ret=PKI_statinfo_new(stat))==NULL) goto error;
	if ((ret->freeText[0] = strdup(text)) == NULL) goto error;
	ret->failInfo[0] = (unsigned char)(finfo>>8);
	ret->failInfo[1] = (unsigned char)finfo;

	return ret;
error:
	PKI_statinfo_free_all(ret);
	return NULL;
}

CertifiedKeyPair *set_dummy_crtkeypair(int choice,int action){
	CertifiedKeyPair *ret;
	char *fp_TEST_CERT = get_fpath(PATH, TEST_CERT);

	if((ret=CMP_ctkeypair_new())==NULL) goto error;
	switch(choice){
	case 1:
		if((ret->certOrEncCert.encCert=set_dummy_encryval())==NULL)
			goto error;
		break;
	default:
		if ((ret->certOrEncCert.cert = Cert_read_file(fp_TEST_CERT)) == NULL)
			goto error;
		break;
	}

	if((ret->privateKey=set_dummy_encryval())==NULL) goto error;
	if((ret->publicationInfo=set_dummy_pubinfo(action))==NULL) goto error;

	return ret;
error:
	if (fp_TEST_CERT) free(fp_TEST_CERT);
	CMP_ctkeypair_free_all(ret);
	return NULL;
}

EncryptedValue *set_dummy_encryval(){
	EncryptedValue *ret;
	
	if((ret=CMP_encval_new())==NULL) goto error;
	ret->intendedAlg = OBJ_CRYPT_RSA;
	ret->keyAlg = OBJ_CRYALGO_RC2CBC;
	ret->symmAlg = OBJ_CRYALGO_3DESCBC;

	ret->esymm_len = 24;
	ret->hint_len = 16;
	ret->enc_len  = 128;
	if((ret->enc_symmkey=malloc(ret->esymm_len))==NULL) goto error;
	if((ret->valueHint  =malloc(ret->hint_len))==NULL) goto error;
	if((ret->encValue   =malloc(ret->enc_len))==NULL) goto error;
	RAND_bytes(ret->enc_symmkey,ret->esymm_len);
	RAND_bytes(ret->valueHint,ret->hint_len);
	RAND_bytes(ret->encValue,ret->enc_len);
	if((ret->symmKey=(Key*)DES3key_new_c(ret->esymm_len,ret->enc_symmkey))==NULL) goto error;
	DES3_set_iv((Key_3DES*)ret->symmKey,"abcdefghi");

	return ret;
error:
	CMP_encval_free(ret);
	return NULL;
}


PKIPubInfo *set_dummy_pubinfo(int action){
	PKIPubInfo *ret;
	Cert *ct;
	int err=-1;
	char *fp_TEST_CERT = get_fpath(PATH, TEST_CERT);

	if ((ct = Cert_read_file(fp_TEST_CERT)) == NULL) goto done;
	
	if((ret=CMP_pubinfo_new())==NULL) goto done;

	switch(action){
	case 1:  /* pleasePublish */
		ret->pubInfo[0].pubMethod = 0;
		ret->pubInfo[1].pubMethod = 1;
		if((ret->pubInfo[1].pubLocation=ExtGN_set_dn(&ct->issuer_dn))==NULL)
			goto done;

		ret->pubInfo[2].pubMethod = 2;
		if((ret->pubInfo[2].pubLocation=ExtGN_set_str("http://sample.org/a.cer",6))==NULL)
			goto done;
		break;

	default: /* dontPublish */
		break;
	}

	err=0;
done:
	if (fp_TEST_CERT) free(fp_TEST_CERT);
	Cert_free(ct);
	if(err){ CMP_pubinfo_free(ret); ret=NULL; }
	return ret;
}


CertTemplate *set_dummy_certtmpl(){
	CertTemplate *ret;
	Cert *ct;
	int err=-1;
	unsigned char tp[32] = {
		0x02,0x07,0x07,0x1c,0x17,0x03,0x2b,0xc4,0x67};
	char *fp_TEST_CERT = get_fpath(PATH, TEST_CERT);

	if ((ct = Cert_read_file(fp_TEST_CERT)) == NULL) goto done;

	if((ret=CMP_certtmpl_new())==NULL) goto done;
	ret->version = ct->version;
	if((ret->snum_der = ASN1_dup(tp))==NULL) goto done;
	ret->serialNumber = ct->serialNumber;
	ret->signingAlg = ct->signature_algo;
	Cert_dncopy(&ct->issuer_dn, &ret->issuer);
	Cert_dncopy(&ct->subject_dn,&ret->subject);
	memcpy(&ret->validity.notBefore,&ct->time.notBefore,sizeof(struct tm));
	memcpy(&ret->validity.notAfter, &ct->time.notAfter, sizeof(struct tm));

	if((ret->publicKey=Key_dup(ct->pubkey))==NULL) goto done;
	if((ret->ext=CertExt_dup_all(ct->ext))==NULL) goto done;

	err=0;
done:
	if (fp_TEST_CERT) free(fp_TEST_CERT);
	Cert_free(ct);
	if(err){CMP_certtmpl_free(ret); ret=NULL;}
	return ret;
}

CertList *set_dummy_certlist(){
	CertList *ret,*hd;
	char *fp_TEST_CERT = get_fpath(PATH, TEST_CERT);

	if((ret=Certlist_new())==NULL) goto error;
	if ((ret->cert = Cert_read_file(fp_TEST_CERT)) == NULL) goto error;
	if((hd =Certlist_new())==NULL) goto error;
	if((hd->cert=Cert_read_file(fp_TEST_CERT))==NULL) goto error;
	ret->next=hd;

	return ret;
error:
	if (fp_TEST_CERT) free(fp_TEST_CERT);
	Certlist_free_all(ret);
	return NULL;
}

int set_dummy_certid(CertId *id){
	Cert *ct;
	char *fp_TEST_CERT = get_fpath(PATH, TEST_CERT);

	if ((ct = Cert_read_file(fp_TEST_CERT)) == NULL) return -1;

	Cert_dncopy(&ct->issuer_dn, &id->issuer);
	id->serialNumber = 0x332211;

	free(fp_TEST_CERT);
	Cert_free(ct);
	return 0;
}
