/* chachatestfc.c */
/*
 * Copyright (c) 2017 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <ctype.h>

#include <aicrypto/nrg_chacha.h>
#include "djb/chacha.h"
#include "chachatest.h"

#ifdef CHACHA_DEBUG
static inline void dump(const uint8_t *buf, const size_t bytes)
{
	int i;

	for (i = 0; i < bytes; i++) {
		printf("%02x ", buf[i]);
	}
	printf("\n");
}
#endif /* CHACHA_DEBUG */

int test_QUARTERROUND()
{
	static const uint32_t ans[4] = {
		0xea2a92f4,
		0xcb1cf8ce,
		0x4581472e,
		0x5881c4bb,
	};

	uint32_t x[4];
	int i;

	x[0] = 0x11111111;
	x[1] = 0x01020304;
	x[2] = 0x9b8d6f43;
	x[3] = 0x01234567;

	QUARTERROUND(0,1,2,3);

	for (i = 0; i < 4; i++) {
		if (x[i] != ans[i]) {
			printf("error : test QUARTERROUND %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test QUARTERROUND ok -- %d\n", i + 1);
		}
	}

	return EXIT_SUCCESS;
}

int test_QUARTERROUND_on_the_ChaCha_State()
{
	static const uint32_t input[] = {
		0x879531e0, 0xc5ecf37d, 0x516461b1, 0xc9a62f8a,
		0x44c20ef3, 0x3390af7f, 0xd9fc690b, 0x2a5f714c,
		0x53372767, 0xb00a5631, 0x974c541a, 0x359e9963,
		0x5c971061, 0x3d631689, 0x2098d9d6, 0x91dbd320,
	};
	static const uint32_t ans[] = {
		0x879531e0, 0xc5ecf37d, 0xbdb886dc, 0xc9a62f8a,
		0x44c20ef3, 0x3390af7f, 0xd9fc690b, 0xcfacafd2,
		0xe46bea80, 0xb00a5631, 0x974c541a, 0x359e9963,
		0x5c971061, 0xccc07c79, 0x2098d9d6, 0x91dbd320,
	};

	uint32_t x[CHACHA20_STATE_ELEMENTS];
	int i;

	memcpy(x, input, sizeof(uint32_t) * CHACHA20_STATE_ELEMENTS);

	QUARTERROUND(2,7,8,13);

	for (i = 0; i < CHACHA20_STATE_ELEMENTS; i++) {
		if (x[i] != ans[i]) {
			printf("error : test QUARTERROUND_on_the_ChaCha_State %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test QUARTERROUND_on_the_ChaCha_State ok -- %d\n", i + 1);
		}
	}

	return EXIT_SUCCESS;
}

int test_ChaCha20_Block_Function()
{
	/* Key = 00:01:02:03:04:05:06:07:08:09:0a:0b:0c:0d:0e:0f:10:11:12:13:
	 * 14:15:16:17:18:19:1a:1b:1c:1d:1e:1f. */
	static const uint8_t key[] = {
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
		0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
		0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
	};

	/* Nonce = (00:00:00:09:00:00:00:4a:00:00:00:00) */
	static const uint8_t nonce[] = {
		0x00, 0x00, 0x00, 0x09, 0x00, 0x00, 0x00, 0x4a,
		0x00, 0x00, 0x00, 0x00,
	};

	/* ChaCha state with the key setup. */
	static const uint32_t key_setup[] = {
		0x61707865, 0x3320646e, 0x79622d32, 0x6b206574,
		0x03020100, 0x07060504, 0x0b0a0908, 0x0f0e0d0c,
		0x13121110, 0x17161514, 0x1b1a1918, 0x1f1e1d1c,
		0x00000001, 0x09000000, 0x4a000000, 0x00000000,
	};
#if 0
	/* ChaCha state after 20 rounds */
	static const uint32_t after20rounds[] = {
		0x837778ab, 0xe238d763, 0xa67ae21e, 0x5950bb2f,
		0xc4f2d0c7, 0xfc62bb2f, 0x8fa018fc, 0x3f5ec7b7,
		0x335271c2, 0xf29489f3, 0xeabda8fc, 0x82e46ebd,
		0xd19c12b4, 0xb04e16de, 0x9e83d0cb, 0x4e3c50a2,
	};
#endif
	/* ChaCha state at the end of the ChaCha20 operation */
	static const uint32_t endofChaCha20[] = {
		0xe4e7f110, 0x15593bd1, 0x1fdd0f50, 0xc47120a3,
		0xc7f4d1c7, 0x0368c033, 0x9aaa2204, 0x4e6cd4c3,
		0x466482d2, 0x09aa9f07, 0x05d7c214, 0xa2028bd9,
		0xd19c12b5, 0xb94e16de, 0xe883d0cb, 0x4e3c50a2,
	};

	ChaCha_state state;
	int i;

	ChaCha20_set_key(&state, key);
	ChaCha20_set_iv(&state, nonce);
	ChaCha20_set_blockcount(&state, 1);

	for (i = 0; i < CHACHA20_STATE_ELEMENTS; i++) {
		if (state.input[i] != key_setup[i]) {
			printf("error : test ChaCha20_Block_Function 1 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test ChaCha20_Block_Function 1 ok -- %d\n", i + 1);
		}
	}

	uint32_t block[CHACHA20_STATE_ELEMENTS];
	ChaCha20_block(&state, block);

	for (i = 0; i < CHACHA20_STATE_ELEMENTS; i++) {
		if (block[i] != endofChaCha20[i]) {
			printf("error : test ChaCha20_Block_Function 2 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test ChaCha20_Block_Function 2 ok -- %d\n", i + 1);
		}
	}

	return EXIT_SUCCESS;
}

int test_ChaCha20_Cipher()
{
	/* Key = 00:01:02:03:04:05:06:07:08:09:0a:0b:0c:0d:0e:0f:10:11:12:13:
	 * 14:15:16:17:18:19:1a:1b:1c:1d:1e:1f. */
	static const uint8_t key[] = {
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
		0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
		0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
	};

	/* Nonce = (00:00:00:00:00:00:00:4a:00:00:00:00) */
	static const uint8_t nonce[] = {
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4a,
		0x00, 0x00, 0x00, 0x00,
	};

	/* Plaintext Sunscreen:
	 * 114 byte length: */
	static const uint8_t plaintext[] = "Ladies and Gentlemen of the class of '99: If I could offer you only one tip for the future, sunscreen would be it.";

	/* First block setup: */
	static const uint32_t first_setup[] = {
		0x61707865, 0x3320646e, 0x79622d32, 0x6b206574,
		0x03020100, 0x07060504, 0x0b0a0908, 0x0f0e0d0c,
		0x13121110, 0x17161514, 0x1b1a1918, 0x1f1e1d1c,
		0x00000001, 0x00000000, 0x4a000000, 0x00000000,
	};
	/* First block after block operation: */
	static const uint32_t first_block[] = {
		0xf3514f22, 0xe1d91b40, 0x6f27de2f, 0xed1d63b8,
		0x821f138c, 0xe2062c3d, 0xecca4f7e, 0x78cff39e,
		0xa30a3b8a, 0x920a6072, 0xcd7479b5, 0x34932bed,
		0x40ba4c79, 0xcd343ec6, 0x4c2c21ea, 0xb7417df0,
	};

	/* Second block setup: */
	static const uint32_t second_setup[] = {
		0x61707865, 0x3320646e, 0x79622d32, 0x6b206574,
		0x03020100, 0x07060504, 0x0b0a0908, 0x0f0e0d0c,
		0x13121110, 0x17161514, 0x1b1a1918, 0x1f1e1d1c,
		0x00000002, 0x00000000, 0x4a000000, 0x00000000,
	};

	/* Second block after block operation: */
	static const uint32_t second_block[] = {
		0x9f74a669, 0x410f633f, 0x28feca22, 0x7ec44dec,
		0x6d34d426, 0x738cb970, 0x3ac5e9f3, 0x45590cc4,
		0xda6e8b39, 0x892c831a, 0xcdea67c1, 0x2b7e1d90,
		0x037463f3, 0xa11a2073, 0xe8bcfb88, 0xedc49139,
	};

	/* Ciphertext Sunscreen:
	 * 000  6e 2e 35 9a 25 68 f9 80 41 ba 07 28 dd 0d 69 81
	 * 016  e9 7e 7a ec 1d 43 60 c2 0a 27 af cc fd 9f ae 0b
	 * 032  f9 1b 65 c5 52 47 33 ab 8f 59 3d ab cd 62 b3 57
	 * 048  16 39 d6 24 e6 51 52 ab 8f 53 0c 35 9f 08 61 d8
	 * 064  07 ca 0d bf 50 0d 6a 61 56 a3 8e 08 8a 22 b6 5e
	 * 080  52 bc 51 4d 16 cc f8 06 81 8c e9 1a b7 79 37 36
	 * 096  5a f9 0b bf 74 a3 5b e6 b4 0b 8e ed f2 78 5e 42
	 * 112  87 4d
	 */
	static const unsigned char result[] = {
		0x6e, 0x2e, 0x35, 0x9a, 0x25, 0x68, 0xf9, 0x80,
		0x41, 0xba, 0x07, 0x28, 0xdd, 0x0d, 0x69, 0x81,
		0xe9, 0x7e, 0x7a, 0xec, 0x1d, 0x43, 0x60, 0xc2,
		0x0a, 0x27, 0xaf, 0xcc, 0xfd, 0x9f, 0xae, 0x0b,
		0xf9, 0x1b, 0x65, 0xc5, 0x52, 0x47, 0x33, 0xab,
		0x8f, 0x59, 0x3d, 0xab, 0xcd, 0x62, 0xb3, 0x57,
		0x16, 0x39, 0xd6, 0x24, 0xe6, 0x51, 0x52, 0xab,
		0x8f, 0x53, 0x0c, 0x35, 0x9f, 0x08, 0x61, 0xd8,
		0x07, 0xca, 0x0d, 0xbf, 0x50, 0x0d, 0x6a, 0x61,
		0x56, 0xa3, 0x8e, 0x08, 0x8a, 0x22, 0xb6, 0x5e,
		0x52, 0xbc, 0x51, 0x4d, 0x16, 0xcc, 0xf8, 0x06,
		0x81, 0x8c, 0xe9, 0x1a, 0xb7, 0x79, 0x37, 0x36,
		0x5a, 0xf9, 0x0b, 0xbf, 0x74, 0xa3, 0x5b, 0xe6,
		0xb4, 0x0b, 0x8e, 0xed, 0xf2, 0x78, 0x5e, 0x42,
		0x87, 0x4d,
	};

	ChaCha_state state;
	uint8_t serialized[CHACHA20_BLOCK_BYTES];
	int i;
	size_t len;
	unsigned char *ciphertext = NULL;
	uint32_t bytes;
	unsigned char *c;
	const uint8_t *m;

	len = strlen((const char *) plaintext);
#ifdef CHACHA_DEBUG
	dump(plaintext, len);
#endif /* CHACHA_DEBUG */

	ciphertext = (unsigned char *) malloc(len);
	memset(ciphertext, 0, len);

	bytes = len;
	c = ciphertext;
	m = plaintext;

	ChaCha20_set_key(&state, key);
	ChaCha20_set_iv(&state, nonce);
	ChaCha20_set_blockcount(&state, 1);

#ifdef CHACHA_DEBUG
	ChaCha20_serialize(state.input, serialized);
	dump(serialized, CHACHA20_BLOCK_BYTES);
#endif /* CHACHA_DEBUG */
	for (i = 0; i < CHACHA20_STATE_ELEMENTS; i++) {
		if (state.input[i] != first_setup[i]) {
			printf("error : test ChaCha20_Cipher 1 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test ChaCha20_Cipher 1 ok -- %d\n", i + 1);
		}
	}

	uint32_t block[CHACHA20_STATE_ELEMENTS];
	ChaCha20_block(&state, block);
	ChaCha20_serialize(block, serialized);
#ifdef CHACHA_DEBUG
	dump(serialized, CHACHA20_BLOCK_BYTES);
#endif /* CHACHA_DEBUG */
	for (i = 0; i < CHACHA20_STATE_ELEMENTS; i++) {
		if (U8TO32_LITTLE(serialized + i * 4) != first_block[i]) {
			printf("error : test ChaCha20_Cipher 2 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test ChaCha20_Cipher 2 ok -- %d\n", i + 1);
		}
	}

	if (bytes > CHACHA20_BLOCK_BYTES) {
		for (i = 0; i < CHACHA20_BLOCK_BYTES; i++) {
			c[i] = m[i] ^ serialized[i];
		}
	}
#ifdef CHACHA_DEBUG
	dump(c, CHACHA20_BLOCK_BYTES);
#endif /* CHACHA_DEBUG */

	bytes -= CHACHA20_BLOCK_BYTES;
	c += CHACHA20_BLOCK_BYTES;
	m += CHACHA20_BLOCK_BYTES;

	ChaCha20_set_key(&state, key);
	ChaCha20_set_iv(&state, nonce);
	ChaCha20_set_blockcount(&state, 2);

#ifdef CHACHA_DEBUG
	ChaCha20_serialize(state.input, serialized);
	dump(serialized, CHACHA20_BLOCK_BYTES);
#endif /* CHACHA_DEBUG */
	for (i = 0; i < CHACHA20_STATE_ELEMENTS; i++) {
		if (state.input[i] !=  second_setup[i]) {
			printf("error : test ChaCha20_Cipher 3 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test ChaCha20_Cipher 3 ok -- %d\n", i + 1);
		}
	}

	ChaCha20_block(&state, block);
	ChaCha20_serialize(block, serialized);
#ifdef CHACHA_DEBUG
	dump(serialized, CHACHA20_BLOCK_BYTES);
#endif /* CHACHA_DEBUG */
	for (i = 0; i < CHACHA20_STATE_ELEMENTS; i++) {
		if (U8TO32_LITTLE(serialized + i * 4) != second_block[i]) {
			printf("error : test ChaCha20_Cipher 4 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test ChaCha20_Cipher 4 ok -- %d\n", i + 1);
		}
	}

	if (bytes <= CHACHA20_BLOCK_BYTES) {
		for (i = 0; i < bytes; i++) {
			c[i] = m[i] ^ serialized[i];
		}
	}
#ifdef CHACHA_DEBUG
	dump(c, bytes);
	dump(ciphertext, len);
#endif /* CHACHA_DEBUG */

	for (i = 0; i < len; i++) {
		if (ciphertext[i] != result[i]) {
			printf("error : test ChaCha20_Cipher 5 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test ChaCha20_Cipher 5 ok -- %d\n", i + 1);
		}
	}

	free(ciphertext);

	return EXIT_SUCCESS;
}

int test_ChaCha20_Encrypt()
{
	/* Key = 00:01:02:03:04:05:06:07:08:09:0a:0b:0c:0d:0e:0f:10:11:12:13:
	 * 14:15:16:17:18:19:1a:1b:1c:1d:1e:1f. */
	static const uint8_t key[] = {
		0x00, 0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07,
		0x08, 0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f,
		0x10, 0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17,
		0x18, 0x19, 0x1a, 0x1b, 0x1c, 0x1d, 0x1e, 0x1f,
	};

	/* Nonce = (00:00:00:00:00:00:00:4a:00:00:00:00) */
	static const uint8_t nonce[] = {
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4a,
		0x00, 0x00, 0x00, 0x00,
	};

	/* Plaintext Sunscreen:
	 * 114 byte length: */
	static const uint8_t plaintext[] = "Ladies and Gentlemen of the class of '99: If I could offer you only one tip for the future, sunscreen would be it.";

	/* Ciphertext Sunscreen:
	 * 000  6e 2e 35 9a 25 68 f9 80 41 ba 07 28 dd 0d 69 81
	 * 016  e9 7e 7a ec 1d 43 60 c2 0a 27 af cc fd 9f ae 0b
	 * 032  f9 1b 65 c5 52 47 33 ab 8f 59 3d ab cd 62 b3 57
	 * 048  16 39 d6 24 e6 51 52 ab 8f 53 0c 35 9f 08 61 d8
	 * 064  07 ca 0d bf 50 0d 6a 61 56 a3 8e 08 8a 22 b6 5e
	 * 080  52 bc 51 4d 16 cc f8 06 81 8c e9 1a b7 79 37 36
	 * 096  5a f9 0b bf 74 a3 5b e6 b4 0b 8e ed f2 78 5e 42
	 * 112  87 4d
	 */
	static const unsigned char result[] = {
		0x6e, 0x2e, 0x35, 0x9a, 0x25, 0x68, 0xf9, 0x80,
		0x41, 0xba, 0x07, 0x28, 0xdd, 0x0d, 0x69, 0x81,
		0xe9, 0x7e, 0x7a, 0xec, 0x1d, 0x43, 0x60, 0xc2,
		0x0a, 0x27, 0xaf, 0xcc, 0xfd, 0x9f, 0xae, 0x0b,
		0xf9, 0x1b, 0x65, 0xc5, 0x52, 0x47, 0x33, 0xab,
		0x8f, 0x59, 0x3d, 0xab, 0xcd, 0x62, 0xb3, 0x57,
		0x16, 0x39, 0xd6, 0x24, 0xe6, 0x51, 0x52, 0xab,
		0x8f, 0x53, 0x0c, 0x35, 0x9f, 0x08, 0x61, 0xd8,
		0x07, 0xca, 0x0d, 0xbf, 0x50, 0x0d, 0x6a, 0x61,
		0x56, 0xa3, 0x8e, 0x08, 0x8a, 0x22, 0xb6, 0x5e,
		0x52, 0xbc, 0x51, 0x4d, 0x16, 0xcc, 0xf8, 0x06,
		0x81, 0x8c, 0xe9, 0x1a, 0xb7, 0x79, 0x37, 0x36,
		0x5a, 0xf9, 0x0b, 0xbf, 0x74, 0xa3, 0x5b, 0xe6,
		0xb4, 0x0b, 0x8e, 0xed, 0xf2, 0x78, 0x5e, 0x42,
		0x87, 0x4d,
	};

	ChaCha_state state;
	int i;
	size_t len;
	uint8_t *ciphertext = NULL;

	len = strlen((const char *) plaintext);

	ciphertext = (uint8_t *) malloc(len);
	memset(ciphertext, 0, len);

	ChaCha20_set_key(&state, key);
	ChaCha20_set_iv(&state, nonce);
	ChaCha20_encrypt(&state, plaintext, ciphertext, len);

#ifdef CHACHA_DEBUG
	dump(ciphertext, len);
#endif /* CHACHA_DEBUG */

	for (i = 0; i < len; i++) {
		if (ciphertext[i] != result[i]) {
			printf("error : test ChaCha20_Encrypt 1 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test ChaCha20_Encrypt 1 ok -- %d\n", i + 1);
		}
	}

	free(ciphertext);

	return EXIT_SUCCESS;
}

