/* chacha20_poly1305testfc.c */
/*
 * Copyright (c) 2017 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <string.h>
#include <ctype.h>

#include <aicrypto/nrg_chacha.h>
#include "chacha20_poly1305test.h"

#ifdef CHACHA_DEBUG
static inline void dump(const uint8_t *buf, const size_t bytes)
{
	int i;

	for (i = 0; i < bytes; i++) {
		printf("%02x ", buf[i]);
	}
	printf("\n");
}
#endif /* CHACHA_DEBUG */

int test_Key_Generation()
{
	/* Key:
	 * 000  80 81 82 83 84 85 86 87 88 89 8a 8b 8c 8d 8e 8f
	 * 016  90 91 92 93 94 95 96 97 98 99 9a 9b 9c 9d 9e 9f */
	static const uint8_t key[] = {
		0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
		0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
		0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
		0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
	};

	/* Nonce:
	 * 000  00 00 00 00 00 01 02 03 04 05 06 07 */
	static const uint8_t nonce[] = {
		0x00, 0x00, 0x00, 0x00, 0x00, 0x01, 0x02, 0x03,
		0x04, 0x05, 0x06, 0x07,
	};
#if 0
	/* The ChaCha state setup with key, nonce, and block counter zero */
	static const uint32_t setup_state[] = {
		0x61707865, 0x3320646e, 0x79622d32, 0x6b206574,
		0x83828180, 0x87868584, 0x8b8a8988, 0x8f8e8d8c,
		0x93929190, 0x97969594, 0x9b9a9998, 0x9f9e9d9c,
		0x00000000, 0x00000000, 0x03020100, 0x07060504,
	};

	/* The ChaCha state after 20 rounds */
	static const uint32_t after20rounds_state[] = {
		0x8ba0d58a, 0xcc815f90, 0x27405081, 0x7194b24a,
		0x37b633a8, 0xa50dfde3, 0xe2b8db08, 0x46a6d1fd,
		0x7da03782, 0x9183a233, 0x148ad271, 0xb46773d1,
		0x3cc1875a, 0x8607def1, 0xca5c3086, 0x7085eb87,
	};
#endif

	/* Output bytes:
	 * 000  8a d5 a0 8b 90 5f 81 cc 81 50 40 27 4a b2 94 71
	 * 016  a8 33 b6 37 e3 fd 0d a5 08 db b8 e2 fd d1 a6 46
	 *
	 * And that output is also the 32-byte one-time key used for Poly1305 */
	static const uint8_t output[] = {
		0x8a, 0xd5, 0xa0, 0x8b, 0x90, 0x5f, 0x81, 0xcc,
		0x81, 0x50, 0x40, 0x27, 0x4a, 0xb2, 0x94, 0x71,
		0xa8, 0x33, 0xb6, 0x37, 0xe3, 0xfd, 0x0d, 0xa5,
		0x08, 0xdb, 0xb8, 0xe2, 0xfd, 0xd1, 0xa6, 0x46,
	};

	uint8_t mac[32];
	int i;

	chacha20_poly1305_key_gen(key, nonce, mac);

#ifdef CHACHA_DEBUG
	dump(mac, 16);
#endif /* CHACHA_DEBUG */
	for (i = 0; i < 16; i++) {
		if (mac[i] != output[i]) {
			printf("error : test Key_Generation 1 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test Key_Generation 1 ok -- %d\n", i + 1);
		}
	}

	return EXIT_SUCCESS;
}

int test_AEAD_Construction()
{
	/* Plaintext: */
	static const uint8_t plaintext[] = "Ladies and Gentlemen of the class of '99: If I could offer you only one tip for the future, sunscreen would be it.";

	/* AAD:
	 * 000  50 51 52 53 c0 c1 c2 c3 c4 c5 c6 c7 */
	static const uint8_t aad[] = {
		0x50, 0x51, 0x52, 0x53, 0xc0, 0xc1, 0xc2, 0xc3,
		0xc4, 0xc5, 0xc6, 0xc7,
	};

	/* Key:
	 * 000  80 81 82 83 84 85 86 87 88 89 8a 8b 8c 8d 8e 8f
	 * 016  90 91 92 93 94 95 96 97 98 99 9a 9b 9c 9d 9e 9f */
	static const uint8_t key[] = {
		0x80, 0x81, 0x82, 0x83, 0x84, 0x85, 0x86, 0x87,
		0x88, 0x89, 0x8a, 0x8b, 0x8c, 0x8d, 0x8e, 0x8f,
		0x90, 0x91, 0x92, 0x93, 0x94, 0x95, 0x96, 0x97,
		0x98, 0x99, 0x9a, 0x9b, 0x9c, 0x9d, 0x9e, 0x9f,
	};

	/* IV:
	 * 000  40 41 42 43 44 45 46 47 */
	static const uint8_t iv[] = {
		0x40, 0x41, 0x42, 0x43, 0x44, 0x45, 0x46, 0x47,
	};

	/* 32-bit fixed-common part:
	 * 000  07 00 00 00 */
	static const uint8_t constant[] = {
		0x07, 0x00, 0x00, 0x00,
	};

#if 0
	/* Setup for generating Poly1305 one-time key (sender id=7): */
	static const uint32_t setup_state[] = {
		0x61707865, 0x3320646e, 0x79622d32, 0x6b206574,
		0x83828180, 0x87868584, 0x8b8a8988, 0x8f8e8d8c,
		0x93929190, 0x97969594, 0x9b9a9998, 0x9f9e9d9c,
		0x00000000, 0x00000007, 0x43424140, 0x47464544,
	};

	/* After generating Poly1305 one-time key: */
	static const uint32_t aftergenerating_state[] = {
		0x252bac7b, 0xaf47b42d, 0x557ab609, 0x8455e9a4,
		0x73d6e10a, 0xebd97510, 0x7875932a, 0xff53d53e,
		0xdecc7ea2, 0xb44ddbad, 0xe49c17d1, 0xd8430bc9,
		0x8c94b7bc, 0x8b7d4b4b, 0x3927f67d, 0x1669a432,
	};

	/* Poly1305 Key:
	 * 000  7b ac 2b 25 2d b4 47 af 09 b6 7a 55 a4 e9 55 84
	 * 016  0a e1 d6 73 10 75 d9 eb 2a 93 75 78 3e d5 53 ff */
	static const uint8_t poly1305_key[] = {
		0x7b, 0xac, 0x2b, 0x25, 0x2d, 0xb4, 0x47, 0xaf,
		0x09, 0xb6, 0x7a, 0x55, 0xa4, 0xe9, 0x55, 0x84,
		0x0a, 0xe1, 0xd6, 0x73, 0x10, 0x75, 0xd9, 0xeb,
		0x2a, 0x93, 0x75, 0x78, 0x3e, 0xd5, 0x53, 0xff,
	};

	/* keystream bytes:
	 * 9f:7b:e9:5d:01:fd:40:ba:15:e2:8f:fb:36:81:0a:ae:
	 * c1:c0:88:3f:09:01:6e:de:dd:8a:d0:87:55:82:03:a5:
	 * 4e:9e:cb:38:ac:8e:5e:2b:b8:da:b2:0f:fa:db:52:e8:
	 * 75:04:b2:6e:be:69:6d:4f:60:a4:85:cf:11:b8:1b:59:
	 * fc:b1:c4:5f:42:19:ee:ac:ec:6a:de:c3:4e:66:69:78:
	 * 8e:db:41:c4:9c:a3:01:e1:27:e0:ac:ab:3b:44:b9:cf:
	 * 5c:86:bb:95:e0:6b:0d:f2:90:1a:b6:45:e4:ab:e6:22:
	 * 15:38 */
	static const uint8_t keystream[] = {
		0x9f, 0x7b, 0xe9, 0x5d, 0x01, 0xfd, 0x40, 0xba,
		0x15, 0xe2, 0x8f, 0xfb, 0x36, 0x81, 0x0a, 0xae,
		0xc1, 0xc0, 0x88, 0x3f, 0x09, 0x01, 0x6e, 0xde,
		0xdd, 0x8a, 0xd0, 0x87, 0x55, 0x82, 0x03, 0xa5,
		0x4e, 0x9e, 0xcb, 0x38, 0xac, 0x8e, 0x5e, 0x2b,
		0xb8, 0xda, 0xb2, 0x0f, 0xfa, 0xdb, 0x52, 0xe8,
		0x75, 0x04, 0xb2, 0x6e, 0xbe, 0x69, 0x6d, 0x4f,
		0x60, 0xa4, 0x85, 0xcf, 0x11, 0xb8, 0x1b, 0x59,
		0xfc, 0xb1, 0xc4, 0x5f, 0x42, 0x19, 0xee, 0xac,
		0xec, 0x6a, 0xde, 0xc3, 0x4e, 0x66, 0x69, 0x78,
		0x8e, 0xdb, 0x41, 0xc4, 0x9c, 0xa3, 0x01, 0xe1,
		0x27, 0xe0, 0xac, 0xab, 0x3b, 0x44, 0xb9, 0xcf,
		0x5c, 0x86, 0xbb, 0x95, 0xe0, 0x6b, 0x0d, 0xf2,
		0x90, 0x1a, 0xb6, 0x45, 0xe4, 0xab, 0xe6, 0x22,
		0x15, 0x38,
	};
#endif

	/* Ciphertext:
	 * 000  d3 1a 8d 34 64 8e 60 db 7b 86 af bc 53 ef 7e c2
	 * 016  a4 ad ed 51 29 6e 08 fe a9 e2 b5 a7 36 ee 62 d6
	 * 032  3d be a4 5e 8c a9 67 12 82 fa fb 69 da 92 72 8b
	 * 048  1a 71 de 0a 9e 06 0b 29 05 d6 a5 b6 7e cd 3b 36
	 * 064  92 dd bd 7f 2d 77 8b 8c 98 03 ae e3 28 09 1b 58
	 * 080  fa b3 24 e4 fa d6 75 94 55 85 80 8b 48 31 d7 bc
	 * 096  3f f4 de f0 8e 4b 7a 9d e5 76 d2 65 86 ce c6 4b
	 * 112  61 16 */
	static const uint8_t ciphertext[] = {
		0xd3, 0x1a, 0x8d, 0x34, 0x64, 0x8e, 0x60, 0xdb,
		0x7b, 0x86, 0xaf, 0xbc, 0x53, 0xef, 0x7e, 0xc2,
		0xa4, 0xad, 0xed, 0x51, 0x29, 0x6e, 0x08, 0xfe,
		0xa9, 0xe2, 0xb5, 0xa7, 0x36, 0xee, 0x62, 0xd6,
		0x3d, 0xbe, 0xa4, 0x5e, 0x8c, 0xa9, 0x67, 0x12,
		0x82, 0xfa, 0xfb, 0x69, 0xda, 0x92, 0x72, 0x8b,
		0x1a, 0x71, 0xde, 0x0a, 0x9e, 0x06, 0x0b, 0x29,
		0x05, 0xd6, 0xa5, 0xb6, 0x7e, 0xcd, 0x3b, 0x36,
		0x92, 0xdd, 0xbd, 0x7f, 0x2d, 0x77, 0x8b, 0x8c,
		0x98, 0x03, 0xae, 0xe3, 0x28, 0x09, 0x1b, 0x58,
		0xfa, 0xb3, 0x24, 0xe4, 0xfa, 0xd6, 0x75, 0x94,
		0x55, 0x85, 0x80, 0x8b, 0x48, 0x31, 0xd7, 0xbc,
		0x3f, 0xf4, 0xde, 0xf0, 0x8e, 0x4b, 0x7a, 0x9d,
		0xe5, 0x76, 0xd2, 0x65, 0x86, 0xce, 0xc6, 0x4b,
		0x61, 0x16,
	};
#if 0
	/* AEAD Construction for Poly1305:
	 * 000  50 51 52 53 c0 c1 c2 c3 c4 c5 c6 c7 00 00 00 00
	 * 016  d3 1a 8d 34 64 8e 60 db 7b 86 af bc 53 ef 7e c2
	 * 032  a4 ad ed 51 29 6e 08 fe a9 e2 b5 a7 36 ee 62 d6
	 * 048  3d be a4 5e 8c a9 67 12 82 fa fb 69 da 92 72 8b
	 * 064  1a 71 de 0a 9e 06 0b 29 05 d6 a5 b6 7e cd 3b 36
	 * 080  92 dd bd 7f 2d 77 8b 8c 98 03 ae e3 28 09 1b 58
	 * 096  fa b3 24 e4 fa d6 75 94 55 85 80 8b 48 31 d7 bc
	 * 112  3f f4 de f0 8e 4b 7a 9d e5 76 d2 65 86 ce c6 4b
	 * 128  61 16 00 00 00 00 00 00 00 00 00 00 00 00 00 00
	 * 144  0c 00 00 00 00 00 00 00 72 00 00 00 00 00 00 00
	 *
	 * Note the four zero bytes in line 000 and the 14 zero bytes
	 * in line 128 */
	static const uint8_t AEAD_construction[] = {
		0x50, 0x51, 0x52, 0x53, 0xc0, 0xc1, 0xc2, 0xc3,
		0xc4, 0xc5, 0xc6, 0xc7, 0x00, 0x00, 0x00, 0x00,
		0xd3, 0x1a, 0x8d, 0x34, 0x64, 0x8e, 0x60, 0xdb,
		0x7b, 0x86, 0xaf, 0xbc, 0x53, 0xef, 0x7e, 0xc2,
		0xa4, 0xad, 0xed, 0x51, 0x29, 0x6e, 0x08, 0xfe,
		0xa9, 0xe2, 0xb5, 0xa7, 0x36, 0xee, 0x62, 0xd6,
		0x3d, 0xbe, 0xa4, 0x5e, 0x8c, 0xa9, 0x67, 0x12,
		0x82, 0xfa, 0xfb, 0x69, 0xda, 0x92, 0x72, 0x8b,
		0x1a, 0x71, 0xde, 0x0a, 0x9e, 0x06, 0x0b, 0x29,
		0x05, 0xd6, 0xa5, 0xb6, 0x7e, 0xcd, 0x3b, 0x36,
		0x92, 0xdd, 0xbd, 0x7f, 0x2d, 0x77, 0x8b, 0x8c,
		0x98, 0x03, 0xae, 0xe3, 0x28, 0x09, 0x1b, 0x58,
		0xfa, 0xb3, 0x24, 0xe4, 0xfa, 0xd6, 0x75, 0x94,
		0x55, 0x85, 0x80, 0x8b, 0x48, 0x31, 0xd7, 0xbc,
		0x3f, 0xf4, 0xde, 0xf0, 0x8e, 0x4b, 0x7a, 0x9d,
		0xe5, 0x76, 0xd2, 0x65, 0x86, 0xce, 0xc6, 0x4b,
		0x61, 0x16, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x72, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	};
#endif
	/* Tag: 1a:e1:0b:59:4f:09:e2:6a:7e:90:2e:cb:d0:60:06:91 */
	static const uint8_t tag[] = {
		0x1a, 0xe1, 0x0b, 0x59, 0x4f, 0x09, 0xe2, 0x6a,
		0x7e, 0x90, 0x2e, 0xcb, 0xd0, 0x60, 0x06, 0x91,
	};

	uint8_t nonce[12];
	size_t len;
	uint8_t *output;
	uint8_t mac[16];
	int i;

	/* nonce = constant | iv */
	memcpy(nonce+0, constant, 4);
	memcpy(nonce+4, iv, 8);

	/* A ciphertext is same length as the plaintext. */
	len = strlen((const char *) plaintext);
	output = (uint8_t *) malloc(len);
	memset(output, 0, len);

	chacha20_poly1305_encrypt(aad, sizeof(aad), key, nonce, plaintext, len,
				  output, mac);
#ifdef CHACHA_DEBUG
	dump(output, len);
#endif /* CHACHA_DEBUG */
	for (i = 0; i < len; i++) {
		if (output[i] != ciphertext[i]) {
			printf("error : test AEAD_Construction 1 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test AEAD_Construction 1 ok -- %d\n", i + 1);
		}
	}
#ifdef CHACHA_DEBUG
	dump(mac, 16);
#endif /* CHACHA_DEBUG */
	for (i = 0; i < 16; i++) {
		if (mac[i] != tag[i]) {
			printf("error : test AEAD_Construction 2 %d\n", i + 1);
			return EXIT_FAILURE;
		} else {
			printf("test AEAD_Construction 2 ok -- %d\n", i + 1);
		}
	}

	free(output);

	return EXIT_SUCCESS;
}

