/* chacha20_poly1305.c */
/*
 * Copyright (c) 2017-2019 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include <stdio.h>

#include <string.h>
#include <stdlib.h>
#include <aicrypto/nrg_chacha.h>
#include <aicrypto/nrg_poly1305.h>
#include "djb/ecrypt-sync.h"

/**
 * Calculate the length of padding.
 *
 * Calculate the padding size to be an integer multiple of 16 bytes.
 *
 * @param[in] len data length
 * @returns
 * length of padding.
 */
static inline size_t pad16len(const size_t len);

/**
 * Transforms a ChaCha state by running multiple quarter-rounds.
 *
 * Padded data which is an integral multiple of 16 bytes is stored in output.
 * For details of implementation, see section 2.8.1 of RFC 7539.
 *
 * @param[in] input Input data.
 * @param[in] len Input data length.
 * @param[out] output padded input data. A sufficiently long buffer to store
 * padded data is necessary.
 *
 * @returns
 * length of padded data.
 */
static inline size_t pad16(const uint8_t *input, size_t len, uint8_t *output);


void chacha20_poly1305_key_gen(const uint8_t *key, const uint8_t *nonce,
			       uint8_t *mac)
{
	ChaCha_state state;
	int i;
	uint32_t block[CHACHA20_STATE_ELEMENTS];

	/* counter = 0 */
	ChaCha20_set_key(&state, key);
	ChaCha20_set_iv(&state, nonce);
	ChaCha20_set_blockcount(&state, 0);

	/* block = chacha20_block(key,counter,nonce) */
	ChaCha20_block(&state, block);

	/* return block[0..31] */
	for (i = 0; i < 8; i++) {
		U32TO8_LITTLE(mac + 4 * i, block[i]);
	}
}

static inline size_t pad16len(const size_t len)
{
	return ((len % 16) == 0) ? 0 : 16 - (len % 16);
}

static inline size_t pad16(const uint8_t *input, size_t len, uint8_t *output)
{
	size_t padlen = pad16len(len);

	memcpy(output, input, len);
	if (padlen != 0) {
		memset(output + len, 0, padlen);
	}

	return len + padlen;
}

/*
  draft-ietf-tls-tls13-21 5.2.  Record Payload Protection
      AEADEncrypted =
          AEAD-Encrypt(write_key, nonce, plaintext)
*/
void chacha20_poly1305_encrypt(const uint8_t *aad, const size_t aadlen,
			       const uint8_t *key, const uint8_t *nonce,
			       const uint8_t *plaintext, const size_t datalen,
			       uint8_t *ciphertext, uint8_t *tag)
{
	uint8_t otk[32];
	ChaCha_state state;

	uint8_t *mac_data;
	size_t mac_data_len = 0;
	size_t written = 0;

	chacha20_poly1305_key_gen(key, nonce, otk);

	/* ciphertext = chacha20_encrypt(key, 1, nonce, plaintext) */
	ChaCha20_set_key(&state, key);
	ChaCha20_set_iv(&state, nonce);
	ChaCha20_encrypt(&state, plaintext, ciphertext, datalen);

	/*
	 * mac_data = aad | pad16(aad)
	 * mac_data |= ciphertext | pad16(ciphertext)
	 * mac_data |= num_to_4_le_bytes(aad.length)
	 * mac_data |= num_to_4_le_bytes(ciphertext.length)
	 */
	mac_data_len += aadlen + pad16len(aadlen); /* aad length */
	mac_data_len += datalen + pad16len(datalen); /* ciphertext length */
	mac_data_len += sizeof(uint64_t); /* aad.length buffer size */
	mac_data_len += sizeof(uint64_t); /* ciphertext.length buffer size */

	mac_data = (uint8_t *) malloc(mac_data_len);

	written = pad16(aad, aadlen, mac_data + 0);
	written += pad16(ciphertext, datalen, mac_data + written);
	/* The length of the additional data in octets (as a 64-bit
	 * little-endian integer). */
	*((uint64_t *)(mac_data + written)) = U64TO64_LITTLE((uint64_t) aadlen);
	written += sizeof(uint64_t);
	/* The length of the ciphertext in octets (as a 64-bit little-
	 * endian integer). */
	*((uint64_t *)(mac_data + written)) = U64TO64_LITTLE((uint64_t) datalen);

	/* tag = poly1305_mac(mac_data, otk) */
	poly1305_mac(mac_data, mac_data_len, otk, tag);
}


/*
 * RFC8446 5.2.  Record Payload Protection
 *
 *       plaintext of encrypted_record =
 *           AEAD-Decrypt(peer_write_key, nonce,
 *                        additional_data, AEADEncrypted)
 */
void chacha20_poly1305_decrypt(const uint8_t *aad, const size_t aadlen,
			       const uint8_t *key, const uint8_t *nonce,
			       const uint8_t *ciphertext, const size_t datalen,
			       uint8_t *plaintext, uint8_t *tag)
{
	uint8_t otk[32];
	ChaCha_state state;

	uint8_t *mac_data;
	size_t mac_data_len = 0;
	size_t written = 0;

	chacha20_poly1305_key_gen(key, nonce, otk);

	/*
	 * mac_data = aad | pad16(aad)
	 * mac_data |= ciphertext | pad16(ciphertext)
	 * mac_data |= num_to_4_le_bytes(aad.length)
	 * mac_data |= num_to_4_le_bytes(ciphertext.length)
	 */
	mac_data_len += aadlen + pad16len(aadlen); /* aad length */
	mac_data_len += datalen + pad16len(datalen); /* ciphertext length */
	mac_data_len += sizeof(uint64_t); /* aad.length buffer size */
	mac_data_len += sizeof(uint64_t); /* ciphertext.length buffer size */

	mac_data = (uint8_t *) malloc(mac_data_len);

	written = pad16(aad, aadlen, mac_data + 0);
	written += pad16(ciphertext, datalen, mac_data + written);
	/* The length of the additional data in octets (as a 64-bit
	 * little-endian integer). */
	*((uint64_t *)(mac_data + written)) = U64TO64_LITTLE((uint64_t) aadlen);
	written += sizeof(uint64_t);
	/* The length of the ciphertext in octets (as a 64-bit little-
	 * endian integer). */
	*((uint64_t *)(mac_data + written)) = U64TO64_LITTLE((uint64_t) datalen);

	/* tag = poly1305_mac(mac_data, otk) */
	poly1305_mac(mac_data, mac_data_len, otk, tag);

	/* ciphertext = chacha20_encrypt(key, 1, nonce, plaintext) */
	ChaCha20_set_key(&state, key);
	ChaCha20_set_iv(&state, nonce);
	ChaCha20_encrypt(&state, ciphertext, plaintext, datalen);
}
