/* chacha.c */
/*
 * Copyright (c) 2017 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

#include <stdio.h>
#include <string.h>
#include <aicrypto/nrg_chacha.h>

#include "djb/ecrypt-sync.h"
#include "djb/chacha.h"

/**
 * Transforms a ChaCha state by running multiple quarter-rounds.
 *
 * Runs 20 rounds, alternating between "column rounds" and  "diagonal rounds".
 *
 * @param[in,out] state ChaCha_state block (16 elements of 32-bit unsigned
 * integers).  returns an updated block.
 *
 * @ingroup chacha
 */
static void inner_block(uint32_t *state);

void ChaCha20_serialize(uint32_t *input, uint8_t *serialized)
{
	int i;

	for (i = 0; i < CHACHA20_STATE_ELEMENTS; i++) {
		U32TO8_LITTLE(serialized + 4 * i, input[i]);
	}
}

void ChaCha20_set_key(ChaCha_state *state, const unsigned char *key)
{
	ECRYPT_ctx ctx;
	memcpy(ctx.input, state->input,
	       sizeof(uint32_t) * CHACHA20_STATE_ELEMENTS);
	ECRYPT_keysetup(&ctx, key, 256, 0 /* unused */);
	memcpy(state->input, ctx.input,
	       sizeof(uint32_t) * CHACHA20_STATE_ELEMENTS);
	state->input[12] = 0;
}

void ChaCha20_set_blockcount(ChaCha_state *state, uint32_t count)
{
	state->input[12] = count;
}

void ChaCha20_set_iv(ChaCha_state *state, const unsigned char *ivc)
{
	state->input[13] = U8TO32_LITTLE(ivc + 0);
	state->input[14] = U8TO32_LITTLE(ivc + 4);
	state->input[15] = U8TO32_LITTLE(ivc + 8);
}

static void inner_block(uint32_t *state)
{
	int i;
	uint32_t x[CHACHA20_STATE_ELEMENTS];

	memcpy(x, state, sizeof(uint32_t) * CHACHA20_STATE_ELEMENTS);

	for (i = 0; i < 10; i++) {
		QUARTERROUND( 0, 4,  8,	12);
		QUARTERROUND( 1, 5,  9,	13);
		QUARTERROUND( 2, 6, 10,	14);
		QUARTERROUND( 3, 7, 11,	15);
		QUARTERROUND( 0, 5, 10,	15);
		QUARTERROUND( 1, 6, 11,	12);
		QUARTERROUND( 2, 7,  8,	13);
		QUARTERROUND( 3, 4,  9,	14);
	}

	memcpy(state, x, sizeof(uint32_t) * CHACHA20_STATE_ELEMENTS);
}

void ChaCha20_block(ChaCha_state *state, uint32_t *block)
{
	int i;

	memcpy(block, state->input, sizeof(uint32_t) * CHACHA20_STATE_ELEMENTS);
	inner_block(block);
	for (i = 0; i < CHACHA20_STATE_ELEMENTS; i++) {
		block[i] = PLUS(state->input[i], block[i]);
	}
}

void ChaCha20_encrypt(ChaCha_state *state, const uint8_t *plaintext,
		      uint8_t *ciphertext, uint32_t len)
{
	int i;
	uint32_t counter;
	uint32_t bytes;
	const uint8_t *m;
	uint8_t *c;
	uint8_t serialized[CHACHA20_BLOCK_BYTES];
	uint32_t block[CHACHA20_STATE_ELEMENTS];

	bytes = len;
	m = plaintext;
	c = ciphertext;

	for (counter = 1; ; counter++) {
		ChaCha20_set_blockcount(state, counter);
		ChaCha20_block(state, block);
		ChaCha20_serialize(block, serialized);

		if (bytes <= CHACHA20_BLOCK_BYTES) {
			for (i = 0; i < bytes; i++) {
				c[i] = m[i] ^ serialized[i];
			}
			break;
		} else {
			for (i = 0; i < CHACHA20_BLOCK_BYTES; i++) {
				c[i] = m[i] ^ serialized[i];
			}
		}

		bytes -= CHACHA20_BLOCK_BYTES;
		m += CHACHA20_BLOCK_BYTES;
		c += CHACHA20_BLOCK_BYTES;
	}
}
