/* asn1_extip.c */
/*
 * Modified by National Institute of Informatics in Japan, 2014.
 *
 */
/*
 * Copyright (C) 1998-2004
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_x509.h>
#include <aicrypto/ok_x509ext.h>
#include <aicrypto/ok_asn1.h>

/*-----------------------------------------
  Get IPAddrFamily
-----------------------------------------*/
int asn1_bitstr2ip(unsigned char *in, unsigned char *addr, int *prefix){
	int len= ASN1_tlen(in);
		
	if(*in != ASN1_BITSTRING){
		OK_set_error(ERR_ST_ASN_NOTBITSTR,ERR_LC_ASN1_,ERR_PT_ASN1EXTIP,NULL);
		goto error;
	}
	if((len<=0)||(len>17)){
		OK_set_error(ERR_ST_ASN_INVALIDLENGTH,ERR_LC_ASN1_,ERR_PT_ASN1EXTIP,NULL);
		goto error;
	}
	if(in[2]>7){
		OK_set_error(ERR_ST_ASN_INVALIDLENGTH,ERR_LC_ASN1_,ERR_PT_ASN1EXTIP,NULL);
		goto error;
	}
	memcpy(addr,&in[3],len-1);
	*prefix = ((len-1)<<3) - in[2];

	return 0;
error:
	return -1;	
}

AddrOrRange *asn1_get_addrrange(unsigned char *in){
	unsigned char *cp;
	int  i,j,len;
	AddrOrRange *hd,*ret=NULL;

	len= ASN1_tlen(in);
	in = ASN1_next(in);

	for(i=0;i<len;){
		/* base */
		if(ret){
			if((hd->next=ExtAddrRang_new())==NULL) goto error;
			hd = hd->next;
		}else{
			if((ret=hd=ExtAddrRang_new())==NULL) goto error;
		}

		/* addressesOrRange */
		switch(*in){
		case ASN1_BITSTRING:
			if(asn1_bitstr2ip(in,hd->min_addr,&hd->min_prefix)) goto error;
			hd->type = EXT_ADRG_IPADDR;
			break;
		case ASN1_SEQUENCE|ASN1_T_STRUCTURED:
			cp = ASN1_next(in);
			if(asn1_bitstr2ip(cp,hd->min_addr,&hd->min_prefix)) goto error;
			cp = ASN1_next(cp);
			if(asn1_bitstr2ip(cp,hd->max_addr,&hd->max_prefix)) goto error;
			hd->type = EXT_ADRG_RANGE;
			break;
		default:
			OK_set_error(ERR_ST_BADFORMAT,ERR_LC_ASN1_,ERR_PT_ASN1EXTIP+1,NULL);
			goto error;
		}

		if((in=ASN1_skip_(in,&j))==NULL) goto error;
		i+=j;
	}
	return ret;
error:
	ExtAddrRang_free_all(ret);
	return NULL;
}

IPAddrFamily *asn1_get_addrfam(unsigned char *in){
	unsigned char *cp;
	int  i,j,len;
	IPAddrFamily *hd,*ret=NULL;

	len= ASN1_tlen(in);
	in = ASN1_next(in);

	for(i=0;i<len;){
		/* base */
		if(ret){
			if((hd->next=ExtAddrFam_new())==NULL) goto error;
			hd = hd->next;
		}else{
			if((ret=hd=ExtAddrFam_new())==NULL) goto error;
		}
		cp = ASN1_next(in);

		/* addressFamily */
		if(*cp != ASN1_OCTETSTRING){
			OK_set_error(ERR_ST_ASN_NOTOCTETSTR,ERR_LC_ASN1_,ERR_PT_ASN1EXTIP+2,NULL);
			goto error;
		}
		switch(cp[1]){
		case 3:
			hd->safi = cp[4]; /* no break */
		case 2:
			hd->afi = (cp[2]<<8)|cp[3];
			break;
		default:
			OK_set_error(ERR_ST_ASN_INVALIDLENGTH,ERR_LC_ASN1_,ERR_PT_ASN1EXTIP+2,NULL);
			goto error;
		}

		/* IPAddressChoice */
		if((cp=ASN1_skip(cp))==NULL) goto error;
		if(*cp == ASN1_NULL){
			/* inherit -- nothing to do */
		}else{
			if((hd->addr=asn1_get_addrrange(cp))==NULL) goto error;
		}

		if((in=ASN1_skip_(in,&j))==NULL) goto error;
		i+=j;
	}
	return ret;
error:
	ExtAddrFam_free_all(ret);
	return NULL;
}

/*-----------------------------------------
  Get ASIdOrRange
-----------------------------------------*/
ASIdOrRange *asn1_get_asidrange(unsigned char *in){
	unsigned char *cp;
	int  i,j,len;
	ASIdOrRange *hd,*ret=NULL;

	in = ASN1_next(in);

	if(*in==ASN1_NULL){
		/* inherit */
		if((ret=ExtASIdRang_new())==NULL) goto error;
		ret->type = EXT_ASRG_INHERIT;
		return ret;
	}

	/* asIdsOrRanges */
	len= ASN1_tlen(in);
	in = ASN1_next(in);

	for(i=0;i<len;){
		/* base */
		if(ret){
			if((hd->next=ExtASIdRang_new())==NULL) goto error;
			hd = hd->next;
		}else{
			if((ret=hd=ExtASIdRang_new())==NULL) goto error;
		}

		/* ASIdOrRange */
		switch(*in){
		case ASN1_INTEGER:
			if((hd->min=ASN1_integer(in,&j))<0) goto error;
			hd->type = EXT_ASRG_ASID;
			break;
		case ASN1_SEQUENCE|ASN1_T_STRUCTURED:
			cp = ASN1_next(in);
			if((hd->min=ASN1_integer(cp,&j))<0) goto error;
			cp = ASN1_next(cp);
			if((hd->max=ASN1_integer(cp,&j))<0) goto error;
			hd->type = EXT_ASRG_RANGE;
			break;
		default:
			OK_set_error(ERR_ST_BADFORMAT,ERR_LC_ASN1_,ERR_PT_ASN1EXTIP+3,NULL);
			goto error;
		}

		if((in=ASN1_skip_(in,&j))==NULL) goto error;
		i+=j;
	}
	return ret;
error:
	ExtASIdRang_free_all(ret);
	return NULL;
}

/*-----------------------------------------
  Extension IP Address Delegation
-----------------------------------------*/
CertExt *ASN1_ext_ipaddr(unsigned char* in){
	CE_IPAddrDlg *ret=NULL;

	if((ret=(CE_IPAddrDlg*)CertExt_new(OBJ_PKIX_IDPE_IPADDR))==NULL)
		goto error;

	if((ret->der=ASN1_dup(in))==NULL) goto error;

	if((ret->family=asn1_get_addrfam(in))==NULL) goto error;

	return (CertExt*)ret;
error:
	CertExt_free((CertExt*)ret);
	return NULL;
}

/*-----------------------------------------
  Extension ASId
-----------------------------------------*/
CertExt *ASN1_ext_asid(unsigned char* in){
	CE_ASId *ret=NULL;

	if((ret=(CE_ASId*)CertExt_new(OBJ_PKIX_IDPE_ASID))==NULL)
		goto error;

	if((ret->der=ASN1_dup(in))==NULL) goto error;

	/* ASIdentifiers */
	in = ASN1_next(in);

	if(*in == 0xa0){ /* asnum [0] */
		if((ret->asnum=asn1_get_asidrange(in))==NULL) goto error;
		if((in=ASN1_skip(in))==NULL) goto error;
	}
	if(*in == 0xa1){ /* rdi [1] */
		if((ret->rdi=asn1_get_asidrange(in))==NULL) goto error;
	}

	return (CertExt*)ret;
error:
	CertExt_free((CertExt*)ret);
	return NULL;
}
