/* asn1_ecdsa.c */
/*
 * Modified by National Institute of Informatics in Japan, 2013-2018.
 *
 */
/*
 * Copyright (C) 1998-2002
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <aicrypto/key_type.h>
#include <aicrypto/ok_err.h>
#include <aicrypto/ok_ecc.h>
#include <aicrypto/ok_ecdsa.h>
#include <aicrypto/ok_asn1.h>

static uint8_t *conv_bit_to_octetstring(unsigned char *);

/*-----------------------------------------
  ASN.1 to struct Prvkey_ECDSA
-----------------------------------------*/
/**
 * Read an EC private key from a DER-encoded data.
 *
 * @note
 * AiCrypto 3.3 or earlier has EC private key structure incompatible
 * with RFC 5915. This function tries to read an input data assuming that it
 * is compatible with RFC 5915. If the trial failed, then the function
 * assumes that the data follows the obsolete syntax.
 *
 * @par
 * The ECPrivateKey defined in RFC 5915 is as follows:
 * @code{.asn1}
 * ECPrivateKey ::= SEQUENCE {
 *   version        INTEGER { ecPrivkeyVer1(1) } (ecPrivkeyVer1),
 *   privateKey     OCTET STRING,
 *   parameters [0] ECParameters {{ NamedCurve }} OPTIONAL,
 *   publicKey  [1] BIT STRING OPTIONAL
 * }
 * @endcode
 * While the obsolete structure in AiCrypto 3.3 or earlier is as follows:
 * @code{.asn1}
 * ECprivateKey ::= SEQUENCE {
 *   version        INTEGER { ecPrivkeyVer1(1) } (ecPrivkeyVer1),
 *   parameters     ECParameters {{ NamedCurve }},
 *   publicKey      OCTET STRING,
 *   privateKey     INTEGER
 * }
 * @endcode
 * It is unknown which specification defines this structure.
 */
Prvkey_ECDSA *ASN1_read_ecdsaprv(unsigned char *der){
	Prvkey_ECDSA *ret;
	unsigned char *cp;
	uint8_t *tmp;	/* a buffer for conversion from BIT to OCTET */
	int i;

	if(der == NULL) return NULL;
	if(*der != (ASN1_T_STRUCTURED | ASN1_SEQUENCE)){
		OK_set_error(ERR_ST_BADFORMAT,ERR_LC_ASN1_,ERR_PT_ASN1ECDSA,NULL);
		return NULL;
	}

	if((ret=ECDSAprvkey_new())==NULL) goto error;
	/* The DER-encoded input data must be unchanged */
	ret->der = der;
	cp = ASN1_next(der);

	/* version */
	if((ret->version=ASN1_integer(cp,&i)) != 1){
		OK_set_error(ERR_ST_UNSUPPORTED_VER,ERR_LC_ASN1_,ERR_PT_ASN1ECDSA,NULL);
		goto error;
	}
	cp += i;

	/* RFC 5915 compatible */
	/* privateKey     OCTET STRING, */
	if (ASN1_OCTETSTRING != cp[0])
		goto incompatible;
	if (ASN1_OS2LNm(cp, ret->k, &i))
		goto error;
	cp += i;

	/*   parameters [0] ECParameters {{ NamedCurve }} OPTIONAL, */
	cp = ASN1_next(cp);
	if ((ret->E = ASN1_read_ecparam(cp)) == NULL)
		goto error;
	if ((ret->E->der = ASN1_dup(cp)) == NULL)
		goto error;
	if ((cp = ASN1_skip(cp)) == NULL)
		goto error;
	ret->size = ((ret->E->psize - 1) >> 3) + 1;
	
	/*   publicKey  [1] BIT STRING OPTIONAL */
	cp = ASN1_next(cp);
	if (cp[0] != ASN1_BITSTRING)
		goto error;
	if (ret->W)
		ECp_free(ret->W);
	/*
	 * To call the ASN1_get_ecpoint() we must convert the publicKey type
	 * from BIT STRING to OCTET STRING.
	 */
	tmp = conv_bit_to_octetstring(cp);
	if ((ret->W = ASN1_get_ecpoint(tmp, ret->E)) == NULL)
		goto error;
	free(tmp);

	return ret;

incompatible:	/* with RFC 5915, followed obsolete structure */
	/* ECDSA parameter */
	if((ret->E=ASN1_read_ecparam(cp))==NULL) goto error;
	if((ret->E->der=ASN1_dup(cp))==NULL) goto error;
	if((cp = ASN1_skip(cp))==NULL) goto error;
	ret->size   =((ret->E->psize-1)>>3)+1;

	/* public key -- W */
	if(ret->W) ECp_free(ret->W);
	if((ret->W=ASN1_get_ecpoint(cp,ret->E))==NULL) goto error;
	if((cp = ASN1_skip(cp))==NULL) goto error;

	/* base integer -- k */
	if(ASN1_int2LNm(cp,ret->k,&i)) goto error;

	return ret;
error:
	ECDSAkey_free((Key*)ret);
	return NULL;
}

/*-----------------------------------------
  ASN.1 to struct Pubkey_ECDSA
-----------------------------------------*/
Pubkey_ECDSA *ASN1_read_ecdsapub(unsigned char *der){
	Pubkey_ECDSA *ret;
	unsigned char *cp;

	if(der == NULL) return NULL;
	if(*der != 0x30){
		OK_set_error(ERR_ST_BADFORMAT,ERR_LC_ASN1_,ERR_PT_ASN1ECDSA+1,NULL);
		return NULL;
	}

	if((ret=ECDSApubkey_new())==NULL) goto error;
	cp = ASN1_next(der);

	/* ECDSA parameter */
	if((ret->E=ASN1_read_ecparam(cp))==NULL) goto error;
	if((ret->E->der=ASN1_dup(cp))==NULL) goto error;
	if((cp = ASN1_skip(cp))==NULL) goto error;
	ret->size   =((ret->E->psize-1)>>3)+1;

	/* public key -- W */
	if(ret->W) ECp_free(ret->W);
	if((ret->W=ASN1_get_ecpoint(cp,ret->E))==NULL) goto error;
	if((cp = ASN1_skip(cp))==NULL) goto error;

	return ret;
error:
	ECDSAkey_free((Key*)ret);
	return NULL;
}

/**
 * Convert BIT STRING of ASN.1 to OCTET STRING.
 *
 * @param[in] der a pointer to DER-encoded data.
 * @return
 */
static uint8_t *conv_bit_to_octetstring(unsigned char *der)
{
	int len;	/* the length of the contents */
	int mv;		/* # of the length octets */
	int mv1;	/* # of the length octets of new OCTET STRING */
	int j;
	unsigned char buf[16] = {0};	/* buffer for identifier and length
					 * octets for new OCTET STRING
					 */
	unsigned char *cp;	/* pointer to contents */
	uint8_t *rt;	/* pointer to new OCTET STRING */

	len = ASN1_length(&der[1], &mv);

	/* set the length octets of new OCTET STRING */
	(void)ASN1_set_length(len - 1, &buf[1], &mv1);
	
	/* allocate a buffer for new OCTET STRING */
	rt = (uint8_t *)malloc(1 + mv1 + len - 1);

	/* shift the contents octets to the left */
	cp = &rt[1 + mv1];
	for (j = 1 + mv + 1; j < 1 + mv + len; j++) {
		*cp = der[j];
		cp++;
	}

	/* overwrite the identifier and length octets */
	(void)memcpy(rt, buf, 1 + mv1);

	/* set suitable value the identifier */
	rt[0] = ASN1_OCTETSTRING;

	return rt;
}
