/* ok_tool.h */
/*
 * Copyright (c) 2012-2017 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2002
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#ifndef INCLUSION_GUARD_UUID_28B0F358_40EF_41C3_9697_CEA36905FBF3
#define INCLUSION_GUARD_UUID_28B0F358_40EF_41C3_9697_CEA36905FBF3

#ifdef HAVE_SYNCH_H
#include <synch.h>
#endif

#include <aicrypto/ok_pkcs.h>
#include <aicrypto/ok_md2.h>
#include <aicrypto/nrg_md4.h>
#include <aicrypto/ok_md5.h>
#include <aicrypto/ok_sha1.h>
#include <aicrypto/ok_sha2.h>
#include <aicrypto/nrg_sha3.h>

#ifdef  __cplusplus
extern "C" {
#endif

/* pass.c */
void OK_set_prompt(char *prom);
void OK_set_passwd(char *pwd);
void OK_clear_passwd();

void OK_get_localpass(char *ret);
/**
 * Get password via a controlling terminal.
 * XXX: the argument "ret" is unsigned char...
 * see "struct decrypt_info" in <aicrypto/ok_pkcs12.h>
 *
 * if TRIAL_MAX-th trial failed, the OK_get_passwd() does not change "ret".
 */
#define M_VRFY		1		/* verify input password */
#define M_NOVRFY	0
#define M_NO_CHECK_LEN	(1 << 1)	/* not impose minimal length */
#define M_DECRYPT	M_NO_CHECK_LEN	/* alias for decryption */
/*
 *            VERIFY(1) NOVERIFY(0)
 * ENCRYPT(0)    01        00
 * DECRYPT(1)    1.        1.
 */
void OK_get_passwd(char *prompt, unsigned char *ret, int mode);
void OK_get_passphrase(char *prompt, unsigned char *ret, int mode);
void OK_get_password_p12(char *prompt,Dec_Info *dif,int mode);

#define PWD_BUFLEN	64

void as2uni(char *in,unsigned char *ret);
void uni2as(unsigned char *in,char *ret);

/* digest.c */
/**
 * Compute message digest with signature algorithm aioid.
 *
 * This function compute the message digest with signature algorithm aioid.
 *
 * @param[in] digest_algo specifies a signature algorithm aioid.
 * @param[in] data an input message to a signature algorithm.
 * @param[in] data_len is the length of an input message.
 * @param[out] ret returns a message digest if ret is not NULL.
 * @param[out] ret_len is the length of a message digest.
 * @returns pointer to a message digest or NULL if error.
 * @attention
 * You must free the pointer returned by this function if ret is NULL.
 *
 */
unsigned char *OK_do_digest(int digest_algo,
			    unsigned char *data, unsigned int data_len,
			    unsigned char *ret, int *ret_len);

/* hash.c */
/** Compute a hash algorithm */
typedef void (*HASHCompute)(unsigned int len, unsigned char *in,
			    unsigned char *ret);

/** Initialize the context of a hash algorithm */
typedef void (*HASHInit)(void *ctx);

/** Update the context of a hash algorithm */
typedef void (*HASHUpdate)(void *ctx, unsigned char *in, unsigned int len);

/** Finalize the context of a hash algorithm */
typedef void (*HASHFinal)(unsigned char *ret, void *ctx);

/** Finalize the context of a hash algorithm */
typedef void (*HASHFinal2)(unsigned char *ret, int len, void *ctx);

/**
 * Hash algorithm object
 *
 * This struct defines a hash algorithm
 */
typedef struct hash_algorithm {
	char		*name;		/**< uppercase name of a hash algorithm */
	int		aioid;		/**< a hash algorithm aioid */
	int		hash_size;	/**< size, in bytes, of hash value */
	HASHCompute	hash_compute;	/**< Compute a hash algorithm */
	HASHInit	hash_init;	/**< Initialize the context of a hash algorithm */
	HASHUpdate	hash_update;	/**< Update the context of a hash algorithm */
	HASHFinal	hash_final;	/**< Finalize the context of a hash algorithm */
	HASHFinal2	hash_final2;	/**< Finalize the context of a hash algorithm */
} HASHAlgorithm;

/**
 * Get the Hash algorithm object by hash algorithm name.
 *
 * This function returns the pointer to a hash algorithm object
 * by hash algorithm name
 *
 * @param[in] hash_name specifies a hash algorithm name.
 * @retruns Hash algorithm object or NULL if nou found.
 */
HASHAlgorithm *gethashalgobyname(const char *hash_name);

/**
 * Get the hash algorithm object by hash algorithm aioid.
 *
 * This function returns the pointer to a hash algorithm object
 * by aioid.
 *
 * @param[in] hash_algo specifies a hash algorithm aioid.
 * @retruns Hash algorithm object or NULL if nou found.
 * @attention
 * The hash_algo is defined in the include file "ok_asn1.h".
 * The currently supported hash algorithms are:
 *   - OBJ_HASH_MD2
 *   - OBJ_HASH_MD5
 *   - OBJ_HASH_SHA1
 *   - OBJ_HASH_SHA224
 *   - OBJ_HASH_SHA256
 *   - OBJ_HASH_SHA384
 *   - OBJ_HASH_SHA512
 *   - OBJ_HASH_SHA512224
 *   - OBJ_HASH_SHA512256
 */
HASHAlgorithm *gethashalgobyaioid(const int hash_algo);

/**
 * Get the size of a hash value
 *
 * This function returns the size of a hash value.
 *
 * @param[in] hash_algo specifies a hash algorithm aioid.
 * @returns the size of a hash value or -1 if not found.
 * @attention
 * The hash_algo is defined in the include file "ok_asn1.h".
 * The currently supported hash algorithms are:
 *   - OBJ_HASH_MD2
 *   - OBJ_HASH_MD5
 *   - OBJ_HASH_SHA1
 *   - OBJ_HASH_SHA224
 *   - OBJ_HASH_SHA256
 *   - OBJ_HASH_SHA384
 *   - OBJ_HASH_SHA512
 *   - OBJ_HASH_SHA512224
 *   - OBJ_HASH_SHA512256
 */
int hash_size(const int hash_algo);

/**
 * Get the aioid of the hash algorithm
 *
 * This function returns the hash algorithm id defined in the include file
 * "ok_asn1.h"
 *
 * @param[in] hash_name specifies a hash algorithm name.
 * @returns the hash algorithm id or -1 if not found.
 */
int get_hash_algo(const char *hash_name);

/* sigalgo.c */
int select_sigalgo(int key_type, int hash_algo);
int obj_sig2hash(int sig_oid);
int get_keytype(char *arg);

/* signature.c */
/**
 * Get signature from data (only called by test programs)
 *
 * @param[in] prv private key
 * @param[in] data data to be signed
 * @param[in] data_len octet length of data
 * @param[out] signature buffer of store signature
 * @param[out] sig_len buffer of store signature octet lenth
 * @param[in] sig_algo specifies a signature algorithm id (ex.OBJ_SIG_SHA1RSA).
 * @param[in] params set if required for the signature algorithm.
 * set NULL if no parameter.
 *
 * @returns
 * return 0 when successfull.
 * return -1 when an internal error occurred.
 */
int NRG_do_signature(Key *prv, unsigned char *data, int data_len,
		     unsigned char **signature,int *sig_len, int sig_algo,
		     void *params);
int OK_do_signature(Key *prv, unsigned char *data, int data_len, unsigned char **signature,int *sig_len, int sig_algo);
/**
 * Verify signature from data
 *
 * @param[in] pub public key
 * @param[in] digest message digest to be verified
 * @param[in] sig signature
 * @param[in] sig_algo specifies a signature algorithm id (ex.OBJ_SIG_SHA1RSA).
 * @param[in] params set if required for the signature algorithm.
 * set NULL if no parameter.
 *
 * @returns
 * return 0 when successfull.
 * return -1 when an internal error occurred.
 */
int NRG_do_verify(Key *pub, unsigned char *digest, unsigned char *sig,
		  int sig_algo, void *params);
int OK_do_verify(Key *pub, unsigned char *digest, unsigned char *sig, int sig_algo);
unsigned char *OK_do_sign(Key *key,unsigned char *data,int data_len,unsigned char *ret);
unsigned char *P1_do_sign(Key *prv,unsigned char *data,int *ret_len);
unsigned char *P1_sign_digest(Key *key,unsigned char *digest,int dig_size,int dig_type);
unsigned char *P1_pad2digest(unsigned char *dec,int *dig_algo,int keylen);

/* defalgo.c */
void OK_set_p12_cb_cry_algo(int algo);
void OK_set_p12_kb_cry_algo(int algo);
void OK_set_p7s_digest_algo(int algo);
void OK_set_p7env_cry_algo(int algo);
void OK_set_p7env_passwd_len(int bit_len);
void OK_set_p5_cry_algo(int algo);
void OK_set_pem_cry_algo(int algo);
void OK_set_sign_digest_algo(int algo);
void OK_set_cert_sig_algo(int algo);
void OK_set_crl_sig_algo(int algo);
void OK_set_ext_flag(int locate, unsigned char flag);
void OK_add_ext_flag(int locate, unsigned char flag);
void OK_del_ext_flag(int locate, unsigned char flag);
void OK_set_time_dertag(unsigned char tag);
void OK_set_sbjdn_conv_flag(unsigned char flag);

int OK_get_p12_cb_cry_algo();
int OK_get_p12_kb_cry_algo();
int OK_get_p7s_digest_algo();
int OK_get_p7env_cry_algo();
int OK_get_p7env_passwd_len();
int OK_get_p5_cry_algo();
int OK_get_pem_cry_algo();
int OK_get_sign_digest_algo();
int OK_get_cert_sig_algo();
int OK_get_crl_sig_algo();
unsigned char OK_get_ext_flag(int locate);

/* lock.c */
#ifdef HAVE_MUTEX_INIT
typedef struct mt_buf{
    mutex_t mt;
}MTBuf;
#endif

typedef struct ai_lock{
    int   fd;
    int   locked;
#ifdef HAVE_MUTEX_INIT
    MTBuf *mbuf;
#endif
} *AILock;

AILock OK_init_lock(char *name);
int OK_release_lock(AILock al);
int OK_lock(AILock al, int msec);
int OK_unlock(AILock al);

/* aicrypto extention flags */
#define AC_EXTF_LC_LNM		0
#define AC_EXTF_LC_ASN1		1
#define AC_EXTF_LC_X509		2
#define AC_EXTF_LC_KEY		3

#define AC_EXTF_ST_MOJCO	0x1

#define AC_MAX_DIGESTSIZE	DEFAULT_SHAKE256_DIGESTSIZE

#ifdef  __cplusplus
}
#endif

#endif /* INCLUSION_GUARD_UUID_28B0F358_40EF_41C3_9697_CEA36905FBF3 */
