/* ok_sha2.h */
/*
 * Copyright (c) 2012-2015 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */

/**
 * @file ok_sha2.h
 * This file defines functions, structures and macros to compute SHA-2 hash.
 **/

/**
 * @defgroup sha2 SHA-2
 * This module provides an API for SHA-2 hash.
 *
 * For the specifications of SHA-2, see FIPS 180-3.
 * @{
 */

#ifndef INCLUSION_GUARD_UUID_85068F16_F8B3_4337_9820_F302A699FDC2
#define INCLUSION_GUARD_UUID_85068F16_F8B3_4337_9820_F302A699FDC2

#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

/** Block Size of SHA-224 and SHA-256 in bytes. */
#define SHA256_BLOCKSIZE	64
/** Block Size of SHA-384 and SHA-512 in bytes. */
#define SHA512_BLOCKSIZE	128
/** Message Digest Size of SHA-224 in bytes. */
#define SHA224_DIGESTSIZE	28
/** Message Digest Size of SHA-256 in bytes. */
#define SHA256_DIGESTSIZE	32
/** Message Digest Size of SHA-384 in bytes. */
#define SHA384_DIGESTSIZE	48
/** Message Digest Size of SHA-512 in bytes. */
#define SHA512_DIGESTSIZE	64
/** Message Digest Size of SHA-512/224 in bytes. */
#define SHA512224_DIGESTSIZE	28
/** Message Digest Size of SHA-512/256 in bytes. */
#define SHA512256_DIGESTSIZE	32

/**
 * SHA-2 context object for SHA-224 and SHA-256.
 *
 * This structure has the context information of SHA-2.
 */
typedef struct sha256_context {
	uint32_t H[8];		/**< hash value */

	unsigned char dat[SHA256_BLOCKSIZE];	/**< 512-bits block */
	uint32_t len[2];	/**< total length of message in bits
				 * (LSB first)
				 */
	unsigned int mod;	/**< mod = (input length) % SHA256_BLOCKSIZE */
} SHA256_CTX;

/**
 * SHA-2 context object for SHA-384 and SHA-512.
 *
 * This structure has the context information of SHA-2.
 */
typedef struct sha512_context {
	uint64_t H[8];		/**< hash value */

	unsigned char dat[SHA512_BLOCKSIZE];	/**< 1024-bits block */
	uint64_t len[2];	/**< total length of message in bits
				 * (LSB first)
				 */
	unsigned int mod;	/**< mod = (input length) % SHA512_BLOCKSIZE */
} SHA512_CTX;

/* char in (size=len byte) -sha2-> ret (size=224bit) */
/**
 * Compute SHA-224.
 *
 * This function computes a SHA-224 hash, using the following steps:
 * -# Initialize a SHA-224 context by using SHA224init().
 * -# Update a SHA-224 context by using SHA224update().
 * -# Finalize a SHA-224 context by using SHA224final().
 *
 * For more information, see Section 6.3 of FIPS 180-3.
 *
 * @param[in] len length of message in bytes.
 * @param[in] in message.
 * @param[out] ret returns a hash in 28bytes.
 * @attention
 * The @a ret buffer must have space for 28bytes of return.
 */
void OK_SHA224(unsigned int len, unsigned char *in, unsigned char *ret);

/**
 * Initialize a SHA-224 context.
 *
 * This function sets a context to the initial hash value.
 *
 * @param[in,out] ctx context object to be initialized.
 */
void SHA224init(SHA256_CTX *ctx);

/**
 * Update a SHA-224 context.
 *
 * This function updates a context by input message.
 *
 * @param[in,out] ctx context object to be updated.
 * @param[in] in message.
 * @param[in] len length of message in bytes.
 */
void SHA224update(SHA256_CTX *ctx, unsigned char *in, unsigned int len);

/**
 * Finalize a SHA-224 context.
 *
 * This function determines a hash value.
 *
 * @param[out] ret returns a hash in 28bytes.
 * @param[in] ctx context object to be finalized.
 * @attention
 * The @a ret buffer must have space for 28bytes of return.
 */
void SHA224final(unsigned char *ret, SHA256_CTX *ctx);

/* char in (size=len byte) -sha2-> ret (size=256bit) */
/**
 * Compute SHA-256.
 *
 * This function computes a SHA-256 hash, using the following steps:
 * -# Initialize a SHA-256 context by using SHA256init().
 * -# Update a SHA-256 context by using SHA256update().
 * -# Finalize a SHA-256 context by using SHA256final().
 *
 * For more information, see Section 6.2 of FIPS 180-3.
 *
 * @param[in] len length of message in bytes.
 * @param[in] in message.
 * @param[out] ret returns a hash in 32bytes.
 * @attention
 * The @a ret buffer must have space for 32bytes of return.
 */
void OK_SHA256(unsigned int len, unsigned char *in, unsigned char *ret);

/**
 * Initialize a SHA-256 context.
 *
 * This function sets a context to the initial hash value.
 *
 * @param[in,out] ctx context object to be initialized.
 */
void SHA256init(SHA256_CTX *ctx);

/**
 * Update a SHA-256 context.
 *
 * This function updates a context by input message.
 *
 * @param[in,out] ctx context object to be updated.
 * @param[in] in message.
 * @param[in] len length of message in bytes.
 */
void SHA256update(SHA256_CTX *ctx, unsigned char *in, unsigned int len);

/**
 * Finalize a SHA-256 context.
 *
 * This function determines a hash value.
 *
 * @param[out] ret returns a hash in 32bytes.
 * @param[in] ctx context object to be finalized.
 * @attention
 * The @a ret buffer must have space for 32bytes of return.
 */
void SHA256final(unsigned char *ret, SHA256_CTX *ctx);

/* char in (size=len byte) -sha2-> ret (size=384bit) */
/**
 * Compute SHA-384.
 *
 * This function computes a SHA-384 hash, using the following steps:
 * -# Initialize a SHA-384 context by using SHA384init().
 * -# Update a SHA-384 context by using SHA384update().
 * -# Finalize a SHA-384 context by using SHA384final().
 *
 * For more information, see Section 6.5 of FIPS 180-3.
 *
 * @param[in] len length of message in bytes.
 * @param[in] in message.
 * @param[out] ret returns a hash in 48bytes.
 * @attention
 * The @a ret buffer must have space for 48bytes of return.
 */
void OK_SHA384(unsigned int len, unsigned char *in, unsigned char *ret);

/**
 * Initialize a SHA-384 context.
 *
 * This function sets a context to the initial hash value.
 *
 * @param[in,out] ctx context object to be initialized.
 */
void SHA384init(SHA512_CTX *ctx);

/**
 * Update a SHA-384 context.
 *
 * This function updates a context by input message.
 *
 * @param[in,out] ctx context object to be updated.
 * @param[in] in message.
 * @param[in] len length of message in bytes.
 */
void SHA384update(SHA512_CTX *ctx, unsigned char *in, unsigned int len);

/**
 * Finalize a SHA-384 context.
 *
 * This function determines a hash value.
 *
 * @param[out] ret returns a hash in 48bytes.
 * @param[in] ctx context object to be finalized.
 * @attention
 * The @a ret buffer must have space for 48bytes of return.
 */
void SHA384final(unsigned char *ret, SHA512_CTX *ctx);

/* char in (size=len byte) -sha2-> ret (size=512bit) */
/**
 * Compute SHA-512.
 *
 * This function computes a SHA-512 hash, using the following steps:
 * -# Initialize a SHA-512 context by using SHA512init().
 * -# Update a SHA-512 context by using SHA512update().
 * -# Finalize a SHA-512 context by using SHA512final().
 *
 * For more information, see Section 6.4 of FIPS 180-3.
 *
 * @param[in] len length of message in bytes.
 * @param[in] in message.
 * @param[out] ret returns a hash in 64bytes.
 * @attention
 * The @a ret buffer must have space for 64bytes of return.
 */
void OK_SHA512(unsigned int len, unsigned char *in, unsigned char *ret);

/**
 * Initialize a SHA-512 context.
 *
 * This function sets a context to the initial hash value.
 *
 * @param[in,out] ctx context object to be initialized.
 */
void SHA512init(SHA512_CTX *ctx);

/**
 * Update a SHA-512 context.
 *
 * This function updates a context by input message.
 *
 * @param[in,out] ctx context object to be updated.
 * @param[in] in message.
 * @param[in] len length of message in bytes.
 */
void SHA512update(SHA512_CTX *ctx, unsigned char *in, unsigned int len);

/**
 * Finalize a SHA-512 context.
 *
 * This function determines a hash value.
 *
 * @param[out] ret returns a hash in 64bytes.
 * @param[in] ctx context object to be finalized.
 * @attention
 * The @a ret buffer must have space for 64bytes of return.
 */
void SHA512final(unsigned char *ret, SHA512_CTX *ctx);

/* char in (size=len byte) -sha2-> ret (size=224bit) */
void OK_SHA512224(unsigned int len, unsigned char *in, unsigned char *ret);

void SHA512224init(SHA512_CTX *ctx);
void SHA512224update(SHA512_CTX *ctx, unsigned char *in, unsigned int len);
void SHA512224final(unsigned char *ret, SHA512_CTX *ctx);

/* char in (size=len byte) -sha2-> ret (size=256bit) */
void OK_SHA512256(unsigned int len, unsigned char *in, unsigned char *ret);

void SHA512256init(SHA512_CTX *ctx);
void SHA512256update(SHA512_CTX *ctx, unsigned char *in, unsigned int len);
void SHA512256final(unsigned char *ret, SHA512_CTX *ctx);

#ifdef __cplusplus
}
#endif

#endif /* INCLUSION_GUARD_UUID_85068F16_F8B3_4337_9820_F302A699FDC2 */
/** @} */
