/* ok_sha1.h */
/*
 * Modified by National Institute of Informatics in Japan, 2013-2015.
 *
 */
/*
 * Copyright (C) 1998-2002
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

/**
 * @file ok_sha1.h
 * This file defines functions, structures to compute the message digest
 * in SHA-1.
 **/

/**
 * @defgroup sha1 SHA-1
 * This module provides an API for computing the message digest in SHA-1.
 *
 * For the specifications of SHA-1, see RFC 3174 or FIPS PUB 180-4.
 *
 */

#ifndef INCLUSION_GUARD_UUID_ECC8039F_BBAE_48CB_BD0C_8D3A2B29D74D
#define INCLUSION_GUARD_UUID_ECC8039F_BBAE_48CB_BD0C_8D3A2B29D74D

#include <stdint.h>

#ifdef  __cplusplus
extern "C" {
#endif

/** SHA-1 block size in bytes (512 bits) defined in RFC 3174, Sec. 2 */
#define SHA1_BLOCKSIZE	64

/**
 * === DRAFT ===
 * SHA-1 context structure.
 */
typedef struct sha1_context {
	uint32_t H[5];	/**< hash value (five 32-bit words) */

	uint8_t dat[SHA1_BLOCKSIZE];	/**< 512-bit message block M(i) */
	uint32_t len[2];	/**< length of message less than 2^64 bits
				 * (LSB first)
				 */
	unsigned int mod;	/**< mod = (input length) % SHA1_BLOCKSIZE */
} SHA1_CTX;

/**
 * === DRAFT ===
 * Compute the message digest of a message.
 *
 * This function computes the message digest of an input message.
 * The message digest is a 160-bit output. 
 * This function sequentically calls three functions, SHA1init(),
 * SHA1update(), and SHA1final(). Note that the SHA1update is called
 * only once. Consequently, the input is not a part of the message
 * but should be the message itself.
 *
 * @param[in] len length of message in bytes.
 * @param[in] in pointer to message.
 * @param[out] ret returns a 160-bit output (20 bytes).
 */
void OK_SHA1(unsigned int len, unsigned char *in, unsigned char *ret);

/**
 * === DRAFT ===
 * Initialize a SHA1_CTX.
 *
 * This function sets the members of a SHA-1 context structure
 * to each initial value.
 *
 * @param[in,out] ctx pointer to a SHA-1 context structure.
 */
void SHA1init(SHA1_CTX *ctx);
/**
 * === DRAFT ===
 * Compute the intermediate hash H[].
 *
 * This function computes the intermediate hash from the message blocks
 * M(1), M(2), ..., M(j) where @j is given by @len / SHA1_BLOCKSIZE.
 * If j = 0 this function only updates the members of the SHA-1 context
 * structure @ctx.
 *
 * @param[in,out] ctx pointer to the SHA-1 context structure.
 * @param[in] in pointer message.
 * @param[in] len length of message in bytes.
 */
void SHA1update(SHA1_CTX *ctx, unsigned char *in, unsigned int len);
/**
 * === DRAFT ===
 * Finalize the SHA-1 context structure.
 *
 * This function computes the message digest from the SHA-1 context structure.
 *
 * @param[out] ret returns a hash value in 20 bytes.
 * @param[in] ctx pointer to the SHA-1 context structure.
 */
void SHA1final(unsigned char *ret, SHA1_CTX *ctx);

#ifdef  __cplusplus
}
#endif

#endif /* INCLUSION_GUARD_UUID_ECC8039F_BBAE_48CB_BD0C_8D3A2B29D74D */
