/* ok_pkcs11.h */
/*
 * Modified by National Institute of Informatics in Japan, 2013-2017.
 *
 */
/*
 * Copyright (C) 1998-2003
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 * 
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

#ifndef INCLUSION_GUARD_UUID_B6D08409_5E91_433F_A2A7_56A49D17EBE9
#define INCLUSION_GUARD_UUID_B6D08409_5E91_433F_A2A7_56A49D17EBE9

#ifdef __WINDOWS__
/* windef.h has ULONG definition */
# undef ULONG

/* just dummy for including "wincrypt.h" */
# ifndef _WIN32_WINNT
#  define _WIN32_WINNT 0x0500
# endif

# ifndef STRICT
#  define STRICT
# endif

# ifndef _WINSOCKAPI_ /* to prevent including winsock.h */
#  define _WINSOCKAPI_
# endif
# include <windows.h>

#endif

/*
 * pkcs11.h -- RSA Security Inc. PKCS #11 Cryptographic Token Interface (Cryptoki)
 */
#include <aicrypto/pkcs11.h>

/*
 * default mechanism for signing, user-controllable macro
 */
#define CKM_SIGNING_DEFAULT	CKM_RSA_PKCS
#ifndef NRG_CKM_SIGNING
#define NRG_CKM_SIGNING	CKM_SIGNING_DEFAULT
#endif

#include <aicrypto/ok_rsa.h>
#include <aicrypto/ok_x509.h>
#include <aicrypto/ok_tool.h>

#ifdef  __cplusplus
extern "C" {
#endif

typedef struct pkcs11_context{
	int version;
	char libname[256];

#ifdef __WINDOWS__
	HINSTANCE hLib;
#else
	void *hLib;
#endif
	CK_FUNCTION_LIST_PTR pFunc;

}PKCS11;

typedef struct pkcs11_session{
	PKCS11 *p11;

	CK_SESSION_HANDLE sess;

	int slot_id;
	int ss_mode;
	int login;
	int uid;
	int mofn;

}P11Session;


typedef struct key_pair_pkcs11{
	int key_type; /* key identifier */
	int size;

	PKCS11 *p11;
	P11Session *p11s;

	CK_SLOT_ID id;
	unsigned long mode;

	int uid;
	char pwd[PWD_BUFLEN];
	char label[32];

	AILock lock;

}P11Key, Pubkey_P11RSA, Prvkey_P11RSA;


/* pkcs11.c */
PKCS11 *P11_init(char *libname);
void P11_free(PKCS11 *p11);

int P11_get_libinfo(PKCS11 *p11, CK_INFO_PTR ret);
int P11_get_slotinfo(PKCS11 *p11, CK_SLOT_ID idx, CK_SLOT_INFO_PTR ret);
int P11_get_tokeninfo(PKCS11 *p11, CK_SLOT_ID idx, CK_TOKEN_INFO_PTR ret);
int P11_get_slotlist(PKCS11 *p11, CK_BBOOL active, CK_SLOT_ID_PTR ret, unsigned long *count);


/* p11_sess.c */
P11Session *P11_open_session(PKCS11 *p11, CK_SLOT_ID id, unsigned long mode, int mofn);
void P11_close_session(P11Session *p11s);

int P11_login(P11Session *p11s, int uid, char *pwd);
void P11_logout(P11Session *p11s);
int P11_mofn_activate(P11Session *p11s);

int P11_rsa_generate(P11Session *p11s, char *label, int bits, Pubkey_RSA **pubkey);
int P11_rsa_generate_(P11Session *p11s, char *label, int bits, Pubkey_RSA **pubkey, int tmplmode);

Cert* P11_get_cert(P11Session *p11s, char *label);
Key* P11_get_rsapub(P11Session *p11s, char *label);
int P11_put_cert(P11Session *p11s, Cert *cert, char *label, unsigned char *subject, unsigned char *id);
int P11_put_rsakey(P11Session *p11s, Key *key, char *label, unsigned char *subject, unsigned char *id);


/* p11_tool.c */
int P11_util_label(Cert *cert, char *label);
int P11_util_keyid(Key *key, unsigned char *id);
unsigned char *P11_util_serialnum(Cert *cert);
unsigned char *P11_util_issuer(Cert *cert);
unsigned char *P11_util_subject(Cert *cert);


/* p11_key.c */
P11Key* P11key_new(PKCS11 *p11);
void P11key_free(P11Key *key);
P11Key *P11key_dup(P11Key *org);

Prvkey_P11RSA* P11RSAprvkey_new(PKCS11 *p11);
Pubkey_P11RSA* P11RSApubkey_new(PKCS11 *p11);

int P11RSApub_doCrypt(int len, unsigned char *from, unsigned char *to, Pubkey_P11RSA *key);
int P11RSAprv_doCrypt(int len, unsigned char *from, unsigned char *to, Prvkey_P11RSA *key);
unsigned char *P11RSApub_toDER(P11Key *pub,unsigned char *buf,int *ret_len);

P11Key *p11_open_key_(PKCS11 *p11, P11Session *p11s, int key_type, int slot_id, int mode, int uid, char *pin, char *label, int prv, int keeplogin, int mofn);
Prvkey_P11RSA *P11_open_rsaprv(PKCS11 *p11, int slot_id, char *label,int mofn);
Pubkey_P11RSA *P11_open_rsapub(PKCS11 *p11, int slot_id, char *label,int mofn);

int P11_gen_symkey(P11Session *p11s, CK_OBJECT_HANDLE *hKey, CK_MECHANISM_TYPE mtype,CK_KEY_TYPE type, CK_ULONG klen, char *label);
int P11_unwrap_rsakey(P11Session *p11s, Key *key, char *label, unsigned char *subject,unsigned char *id);



#ifdef  __cplusplus
}
#endif

#endif	/* INCLUSION_GUARD_UUID_B6D08409_5E91_433F_A2A7_56A49D17EBE9 */
