/* ok_hmac.h */
/*
 * Copyright (c) 2012-2017 National Institute of Informatics in Japan,
 * All rights reserved.
 *
 * This file or a portion of this file is licensed under the terms of
 * the NAREGI Public License, found at http://www.naregi.org/download/index.html.
 * If you redistribute this file, with or without modifications, you must
 * include this notice in the file.
 */
/*
 * Copyright (C) 1998-2002
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

/**
 * @file ok_hmac.h
 * This file defines functions, structures to compute HMAC.
 */

/**
 * @defgroup hmac HMAC
 * This module provides an API for HMAC message authentication code.
 *
 * For the specifications of HMAC, see RFC 2104.
 */

#ifndef INCLUSION_GUARD_UUID_B476DD87_A0CE_44CB_BD37_6A226297DC58
#define INCLUSION_GUARD_UUID_B476DD87_A0CE_44CB_BD37_6A226297DC58

#ifdef __cplusplus
extern "C" {
#endif

typedef struct hmac_context {
	unsigned char k_ipad[64];
	unsigned char k_opad[64];
} HMAC_CTX;

/* hmac.c */
/**
 * Compute HMAC-MD5.
 *
 * This function computes a MAC using the MD5 hash function.
 *
 * @param[in] txtlen length of message in bytes.
 * @param[in] txt message.
 * @param[in] keylen length of authentication key in bytes.
 * @param[in] key authentication key.
 * @param[out] ret returns a computed MAC in 16bytes.
 * @attention
 * The @a ret buffer must have space for 16bytes of return.
 * @ingroup hmac
 */
void HMAC_MD5(int txtlen, unsigned char *txt,
	      int keylen, unsigned char *key, unsigned char *ret);

/**
 * Compute HMAC-SHA1.
 *
 * This function computes a MAC using the SHA-1 hash function.
 *
 * @param[in] txtlen length of message in bytes.
 * @param[in] txt message.
 * @param[in] keylen length of authentication key in bytes.
 * @param[in] key authentication key.
 * @param[out] ret returns a computed MAC in 20bytes.
 * @attention
 * The @a ret buffer must have space for 20bytes of return.
 * @ingroup hmac
 */
void HMAC_SHA1(int txtlen, unsigned char *txt,
	       int keylen, unsigned char *key, unsigned char *ret);

/**
 * Compute HMAC-SHA224.
 *
 * This function computes a MAC using the SHA-224 hash function.
 *
 * @param[in] txtlen length of message in bytes.
 * @param[in] txt message.
 * @param[in] keylen length of authentication key in bytes.
 * @param[in] key authentication key.
 * @param[out] ret returns a computed MAC in 28bytes.
 * @attention
 * The @a ret buffer must have space for 28bytes of return.
 * @ingroup hmac
 */
void HMAC_SHA224(int txtlen, unsigned char *txt,
		 int keylen, unsigned char *key, unsigned char *ret);

/**
 * Compute HMAC-SHA256.
 *
 * This function computes a MAC using the SHA-256 hash function.
 *
 * @param[in] txtlen length of message in bytes.
 * @param[in] txt message.
 * @param[in] keylen length of authentication key in bytes.
 * @param[in] key authentication key.
 * @param[out] ret returns a computed MAC in 32bytes.
 * @attention
 * The @a ret buffer must have space for 32bytes of return.
 * @ingroup hmac
 */
void HMAC_SHA256(int txtlen, unsigned char *txt,
		 int keylen, unsigned char *key, unsigned char *ret);

/**
 * Compute HMAC-SHA384.
 *
 * This function computes a MAC using the SHA-384 hash function.
 *
 * @param[in] txtlen length of message in bytes.
 * @param[in] txt message.
 * @param[in] keylen length of authentication key in bytes.
 * @param[in] key authentication key.
 * @param[out] ret returns a computed MAC in 48bytes.
 * @attention
 * The @a ret buffer must have space for 48bytes of return.
 * @ingroup hmac
 */
void HMAC_SHA384(int txtlen, unsigned char *txt,
		 int keylen, unsigned char *key, unsigned char *ret);

/**
 * Compute HMAC-SHA512.
 *
 * This function computes a MAC using the SHA-512 hash function.
 *
 * @param[in] txtlen length of message in bytes.
 * @param[in] txt message.
 * @param[in] keylen length of authentication key in bytes.
 * @param[in] key authentication key.
 * @param[out] ret returns a computed MAC in 64bytes.
 * @attention
 * The @a ret buffer must have space for 64bytes of return.
 * @ingroup hmac
 */
void HMAC_SHA512(int txtlen, unsigned char *txt,
		 int keylen, unsigned char *key, unsigned char *ret);

/**
 * Compute HMAC-SHA3-224.
 *
 * This function computes a MAC using the SHA3-224 hash function.
 *
 * @param[in] txtlen length of message in bytes.
 * @param[in] txt message.
 * @param[in] keylen length of authentication key in bytes.
 * @param[in] key authentication key.
 * @param[out] ret returns a computed MAC in 28bytes.
 * @attention
 * The @a ret buffer must have space for 28bytes of return.
 * @ingroup hmac
 */
void HMAC_SHA3_224(int txtlen, unsigned char *txt,
		   int keylen, unsigned char *key, unsigned char *ret);

/**
 * Compute HMAC-SHA3-256.
 *
 * This function computes a MAC using the SHA3-256 hash function.
 *
 * @param[in] txtlen length of message in bytes.
 * @param[in] txt message.
 * @param[in] keylen length of authentication key in bytes.
 * @param[in] key authentication key.
 * @param[out] ret returns a computed MAC in 32bytes.
 * @attention
 * The @a ret buffer must have space for 32bytes of return.
 * @ingroup hmac
 */
void HMAC_SHA3_256(int txtlen, unsigned char *txt,
		   int keylen, unsigned char *key, unsigned char *ret);

/**
 * Compute HMAC-SHA3-384.
 *
 * This function computes a MAC using the SHA3-384 hash function.
 *
 * @param[in] txtlen length of message in bytes.
 * @param[in] txt message.
 * @param[in] keylen length of authentication key in bytes.
 * @param[in] key authentication key.
 * @param[out] ret returns a computed MAC in 48bytes.
 * @attention
 * The @a ret buffer must have space for 48bytes of return.
 * @ingroup hmac
 */
void HMAC_SHA3_384(int txtlen, unsigned char *txt,
		   int keylen, unsigned char *key, unsigned char *ret);

/**
 * Compute HMAC-SHA3-512.
 *
 * This function computes a MAC using the SHA3-512 hash function.
 *
 * @param[in] txtlen length of message in bytes.
 * @param[in] txt message.
 * @param[in] keylen length of authentication key in bytes.
 * @param[in] key authentication key.
 * @param[out] ret returns a computed MAC in 64bytes.
 * @attention
 * The @a ret buffer must have space for 64bytes of return.
 * @ingroup hmac
 */
void HMAC_SHA3_512(int txtlen, unsigned char *txt,
		   int keylen, unsigned char *key, unsigned char *ret);

#ifdef __cplusplus
}
#endif

#endif /* INCLUSION_GUARD_UUID_B476DD87_A0CE_44CB_BD37_6A226297DC58 */
