/* ok_aes.h */
/*
 * Modified by National Institute of Informatics in Japan, 2013-2016.
 *
 */
/*
 * Copyright (C) 1998-2002
 * Akira Iwata & Takuto Okuno
 * Akira Iwata Laboratory,
 * Nagoya Institute of Technology in Japan.
 *
 * All rights reserved.
 *
 * This software is written by Takuto Okuno(usapato@anet.ne.jp)
 * And if you want to contact us, send an email to Kimitake Wakayama
 * (wakayama@elcom.nitech.ac.jp)
 *
 * Redistribution and use in source and binary forms, with or without modification,
 * are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * 3. All advertising materials mentioning features or use of this software must
 *    display the following acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *    Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 * 4. Redistributions of any form whatsoever must retain the following
 *    acknowledgment:
 *    "This product includes software developed by Akira Iwata Laboratory,
 *     Nagoya Institute of Technology in Japan (http://mars.elcom.nitech.ac.jp/)."
 *
 *   THIS SOFTWARE IS PROVIDED "AS IS" WITHOUT EXPRESS OR IMPLIED WARRANTY.
 *   AKIRA IWATA LABORATORY DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS
 *   SOFTWARE, INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS,
 *   IN NO EVENT SHALL AKIRA IWATA LABORATORY BE LIABLE FOR ANY SPECIAL,
 *   INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING
 *   FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT,
 *   NEGLIGENCE OR OTHER TORTUOUS ACTION, ARISING OUT OF OR IN CONNECTION
 *   WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 *
 */

/**
 * @file ok_aes.h
 * This file defines functions and structures to AES encryption.
 */

/**
 * @defgroup aes AES
 * This module provides an API for AES block cipher.
 *
 * For the specifications of AES, see FIPS 197.
 */

/**
 * @defgroup aes_internal AES Internal
 * The internal symbols of the AES module.
 * SHOULD NOT use the symbols.
 * @ingroup aes
 */

#ifndef INCLUSION_GUARD_UUID_D28A3684_92B0_48CA_83D7_FC153F8F3298
#define INCLUSION_GUARD_UUID_D28A3684_92B0_48CA_83D7_FC153F8F3298

#ifdef __cplusplus
extern "C" {
#endif

/**
 * AES key object composion.
 *
 * This structure has a set of information about AES key.
 * @ingroup aes
 */
typedef struct crypt_AES_key {
	int key_type;	/**< key identifier */
	int size;	/**< key size in bytes */

	int bsize;	/**< block size in bytes */
	unsigned char kbox[512];	/**< round key -- kbox[4][128] */

	unsigned char iv[32];		/**< initialization vector */
	unsigned char oiv[32];		/**< unused */
} Key_AES;

/* for modes */
#include <aicrypto/nrg_modes.h>

/* aes.c */
/**
 * AES encryption on a block.
 *
 * This function performs the AES encryption on a block.
 * For more information, see Section 5.1 of FIPS 197.
 *
 * @param[in] kbox round key.
 * @param[in,out] in pointer to block. returns an encrypted block.
 * @param[in] round number of the rounds.
 * @param[in] nb block size in bytes.
 * @ingroup aes_internal
 */
void AES_encrypt(unsigned char *kbox, unsigned char *in, int round, int nb);

/**
 * AES encryption with a 128-bit block size.
 *
 * This function performs the AES encryption with a 128-bit block size.
 *
 * @param[in] key round key.
 * @param[in,out] in pointer to block. returns an encrypted block.
 * @ingroup aes_internal
 */
void AES_encrypt_128(Key *key, unsigned char *in);

/**
 * AES decryption on a block.
 *
 * This function performs the AES decryption on a block.
 * For more information, see Section 5.3 of FIPS 197.
 *
 * @param[in] kbox round key.
 * @param[in,out] in pointer to block. returns a decrypted block.
 * @param[in] round number of the rounds.
 * @param[in] nb block size in bytes.
 * @ingroup aes_internal
 */
void AES_decrypt(unsigned char *kbox, unsigned char *in, int round, int nb);

/**
 * SubBytes and ShiftRows Transformation.
 *
 * This function performs the transformation of a state array, using
 * the following steps:
 * - SubBytes: Each byte in the state array is replaced using S-box
 *   table.
 *   For more information, see Section 5.1.1 of FIPS 197.
 * - ShiftRows: Cyclically shifting the last three rows of the State
 *   array by different offsets.
 *   For more information, see Section 5.1.2 of FIPS 197.
 *
 * @param[in,out] in state array. returns an updated state array.
 * @param[in] nb block size in bytes.
 * @ingroup aes_internal
 */
void ByteSub_ShiftRow(unsigned char *in, int nb);

/**
 * InvShiftRows and InvSubBytes Transformation.
 *
 * This function performs the transformation of a state array, using
 * the following steps:
 * - InvShiftRows: Inverse transformation of the ShiftRows.
 *   For more information, see Section 5.3.1 of FIPS 197.
 * - InvSubBytes: Inverse transformation of the SubBytes.
 *   For more information, see Section 5.3.2 of FIPS 197.
 *
 * @param[in,out] in state array. returns an updated state array.
 * @param[in] nb block size in bytes.
 * @ingroup aes_internal
 */
void InversShiftRow_ByteSub(unsigned char *in, int nb);

/**
 * MixColumns and AddRoundKey Transformation.
 *
 * This function performs the transformation of a state array, using
 * the following steps:
 * - MixColumns: Mixing column of the state array by bit operation of 4 bytes.
 *   For more information, see Section 5.1.3 of FIPS 197.
 * - AddRoundKey: See also AddRoundKey()
 *
 * @param[in] kbox round key.
 * @param[in,out] in state array. returns an updated state array.
 * @param[in] nb block size in bytes.
 * @param[in] n number of the current round.
 * @ingroup aes_internal
 */
void MixColumn_AddRoundKey(unsigned char *kbox, unsigned char *in,
			   int nb, int n);

/**
 * AddRoundKey and InvMixColumns Transformation.
 *
 * This function performs the transformation of a state array, using
 * the following steps:
 * - AddRoundKey: See also AddRoundKey()
 * - InvMixColumns: Inverse transformation of the MixColumns.
 *   For more information, see Section 5.3.3 of FIPS 197.
 *
 * @param[in] kbox round key.
 * @param[in,out] in state array. returns an updated state array.
 * @param[in] nb block size in bytes.
 * @param[in] n number of the current round.
 * @ingroup aes_internal
 */
void AddRoundKey_InversMixColumn(unsigned char *kbox, unsigned char *in,
				 int nb, int n);

/**
 * AddRoundKey Transformation.
 *
 * This function performs the transformation of a state array with
 * round key using XOR operation.
 * For more information, see Section 5.1.4 of FIPS 197.
 *
 * @param[in] kbox round key.
 * @param[in,out] in state array. returns an updated state array.
 * @param[in] type block size in bytes.
 * @param[in] n number of the current round.
 * @ingroup aes_internal
 */
void AddRoundKey(unsigned char *kbox, unsigned char *in, int type, int n);

/* aes_key.c */
/**
 * Allocate a Key_AES object.
 *
 * This function allocates and initializes a Key_AES object.
 *
 * @returns newly allocated Key_AES object or NULL if error.
 * @par Error state:
 * If the error is caused by memory allocation, this function
 * sets the error state to ERR_ST_MEMALLOC.
 * @attention
 * If the allocated object is no longer used, free the object by
 * AESkey_free().
 * @ingroup aes
 */
Key_AES *AESkey_new_();

/**
 * Allocate a Key_AES object with key parameters.
 *
 * This function allocates and initializes a Key_AES object by
 * AESkey_new_() and then, set key parameters by AESkey_set() on that
 * object.
 *
 * @param[in] key_size size of cipher key in bytes.
 * @param[in] key cipher key.
 * @param[in] block_size block size in bytes.
 * @returns newly allocated Key_AES object or NULL if error.
 * @attention
 * If the allocated object is no longer used, free the object by
 * AESkey_free().
 * @ingroup aes
 */
Key_AES *AESkey_new(int key_size, unsigned char *key, int block_size);

/**
 * Set key parameters to the Key_AES object.
 *
 * This function sets the following parameters of @a ak.
 * - Key_AES::size
 * - Key_AES::bsize
 * - Key_AES::kbox
 *
 * @param[in,out] ak Key_AES object. returns an updated Key_AES object.
 * @param[in] key_size size of cipher key in bytes.
 *        16,24,32 byte can be specified, otherwise they are rounded.
 * @param[in] key cipher key.
 * @param[in] block_size block size in bytes.
 *        16,24,32 byte can be specified, otherwise they are rounded.
 * @ingroup aes
 */
void AESkey_set(Key_AES *ak, int key_size, unsigned char *key, int block_size);

/**
 * Set an initialization vector.
 *
 * This function sets a @a key to initialization vector.
 *
 * The initialization vector should be chosen randomly.
 *
 * @param[in,out] key Key_AES object. returns an updated Key_AES object.
 * @param[in] ivc initialization vector.
 * @attention
 * The initialization vector should be chosen randomly.
 * @ingroup aes
 */
void AES_set_iv(Key_AES *key, unsigned char *ivc);

/**
 * Key Expansion.
 *
 * This function generates a round key. The round key is derived from
 * the cipher key using key schedule.
 * For more information, see Section 5.2 of FIPS 197.
 *
 * @param[in] ak Key_AES object.
 * @param[in] key cipher key.
 * @param[out] kbox returns a generated round key.
 * @ingroup aes_internal
 */
void aes_key_schedule(Key_AES *ak, unsigned char *key, unsigned char *kbox);

/**
 * Duplicate a Key_AES object.
 *
 * This function duplicates a Key_AES object from @a org.
 *
 * @param[in] org original Key_AES object.
 * @returns newly duplicated Key_AES object or NULL if error.
 * @par Error state:
 * - If the error is caused by null pointer, this function
 * sets the error state to ERR_ST_NULLPOINTER.
 * - If the error is caused by memory allocation, this function
 * sets the error state to ERR_ST_MEMALLOC.
 * @attention
 * If the duplicated object is no longer used, free the object by
 * AESkey_free().
 * @ingroup aes
 */
Key_AES *AESkey_dup(Key_AES *org);

/**
 * Free a Key_AES object.
 *
 * This function frees a Key_AES object.
 *
 * @param[in] key Key_AES object to be free.
 * @ingroup aes
 */
void AESkey_free(Key_AES *key);

/**
 * S-box.
 *
 * This macro defines a Non-linear substitution table.
 * For more information, see Section 5.1.1 of FIPS 197.
 *
 * @ingroup aes_internal
 */
#define SubByte(in) aeSbox[(in)]

/* aes_mode.c */
/**
 * ECB-mode AES encryption.
 *
 * This function performs the AES encryption using the ECB (electric
 * codebook) mode.
 *
 * @param[in] key Key_AES object.
 * @param[in] byte size of plaintext in bytes.
 * @param[in] in plaintext.
 * @param[out] ret returns a ciphertext.
 * @ingroup aes
 */
void AES_ecb_encrypt(Key_AES *key, int byte, unsigned char *in,
		     unsigned char *ret);

/**
 * ECB-mode AES decryption.
 *
 * This function performs the AES decryption using the ECB (electric
 * codebook) mode.
 *
 * @param[in] key Key_AES object.
 * @param[in] byte size of ciphertext in bytes.
 * @param[in] in ciphertext.
 * @param[out] ret returns a plaintext.
 * @ingroup aes
 */
void AES_ecb_decrypt(Key_AES *key, int byte, unsigned char *in,
		     unsigned char *ret);

/**
 * CBC-mode AES encryption.
 *
 * This function performs the AES encryption using the CBC
 * (cipher-block chaining) mode.
 * When using this mode, the Key_AES object must set the
 * initialization vector by AES_set_iv().
 *
 * @param[in] key Key_AES object.
 * @param[in] byte size of plaintext in bytes.
 * @param[in] in plaintext.
 * @param[out] ret returns a ciphertext.
 * @ingroup aes
 */
void AES_cbc_encrypt(Key_AES *key, int byte, unsigned char *in,
		     unsigned char *ret);

/**
 * CBC-mode AES decryption.
 *
 * This function performs the AES decryption using the CBC
 * (cipher-block chaining) mode.
 * When using this mode, the Key_AES object must set the
 * initialization vector by AES_set_iv().
 *
 * @param[in] key Key_AES object.
 * @param[in] byte size of ciphertext in bytes.
 * @param[in] in ciphertext.
 * @param[out] ret returns a plaintext.
 * @ingroup aes
 */
void AES_cbc_decrypt(Key_AES *key, int byte, unsigned char *in,
		     unsigned char *ret);

/**
 * GCM-mode AES encryption.
 *
 * This function performs the AES encryption using the GCM
 * (Galois Counter Mode) mode.
 * When using this mode, the Key_AES object must set the
 * initialization vector by AES_set_iv().
 *
 * @param[in] p gcm_param object.
 * @param[in] byte size of plaintext in bytes.
 * @param[in] in plaintext.
 * @param[out] ret returns a ciphertext.
 *
 * @returns
 * return 0 when successfull.
 * return -1 when an internal error occurred
 * (such as a memory allocation failure).
 *
 * @ingroup aes
 */
int AES_gcm_encrypt(gcm_param_t *p, int32_t byte, const unsigned char *in,
		    unsigned char *ret);

/**
 * GCM-mode AES decryption.
 *
 * This function performs the AES decryption using the GCM
 * (Galois Counter Mode) mode.
 * When using this mode, the Key_AES object must set the
 * initialization vector by AES_set_iv().
 *
 * @param[in] p gcm_param object.
 * @param[in] byte size of ciphertext in bytes.
 * @param[in] in ciphertext.
 * @param[out] ret returns a plaintext.
 *
 * @returns
 * return 0 when successfull.
 * return -1 when an internal error occurred.
 * (such as a memory allocation failure).
 * return -2 when FAIL.
 *
 * @ingroup aes
 */
int AES_gcm_decrypt(gcm_param_t *p, int32_t byte, const unsigned char *in,
		    unsigned char *ret);

#ifdef __cplusplus
}
#endif

#endif /* INCLUSION_GUARD_UUID_D28A3684_92B0_48CA_83D7_FC153F8F3298 */
